/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
/// @file
///
/// @page Overview_Common Overview: Common Component
///
/// The Common component provides common functionality used between all
/// components.
///
/// See OpenIPC_CommonComponent for details of the Common functions.
///
//////////////////////////////////////////////////////////////////////////////

#pragma once

#if defined(HOST_WINDOWS)
#   if defined(COMMON_EXPORTS)
        #       define EXPORT_COMMON_API __declspec(dllexport)
#   else
        #       define EXPORT_COMMON_API __declspec(dllimport)
#   endif
#else
    #   define EXPORT_COMMON_API __attribute__ ((visibility("default")))
#endif

#include <Components/Common/CommonTypes.h>
#include <Components/Common/CommonErrors.h>

#include <Foundation/Error/Error.h>
#include <Foundation/Types.h>

///
/// @brief Represents the Common component.
struct OpenIPC_CommonComponent
{
public:
    ///
    /// @brief Gets the path that the OpenIPC is installed in
    OpenIPC_CommonGetPath_Function GetPath;

    ///
    /// @brief Gets the client ID that the current thread belongs to
    OpenIPC_CommonGetClientId_Function GetClientId;

    ///
    /// @brief Sets the client ID that the current thread belongs to
    OpenIPC_CommonSetClientId_Function SetClientId;

    ///
    /// @brief Registers a new client.
    OpenIPC_CommonRegisterNewClient_Function RegisterNewClient;

    ///
    /// @brief Notifies any subscribers that the client is closed
    OpenIPC_CommonCloseClient_Function CloseClient;

    ///
    /// @brief Gets the number of registered clients.
    OpenIPC_CommonGetNumRegisteredClients_Function GetNumRegisteredClients;

    ///
    /// @brief Gets the list of registered clients.
    OpenIPC_CommonGetRegisteredClients_Function GetRegisteredClients;

    ///
    /// @brief Gets whether the client is on the same machine as the server
    OpenIPC_CommonIsClientLocal_Function IsClientLocal;

    ///
    /// @brief Adds a callback function to be called whenever a client
    /// is connected/disconnected.
    OpenIPC_CommonRegisterClientCallback_Function RegisterClientCallback;

    ///
    /// @brief Removes a callback function
    OpenIPC_CommonUnregisterClientCallback_Function UnregisterClientCallback;

    ///
    /// @brief Initializes logging based on the logging XML configuration file
    OpenIPC_CommonInitializeLogging_Function InitializeLogging;

    ///
    /// @brief Uninitializes logging
    OpenIPC_CommonUninitializeLogging_Function UninitializeLogging;

    ///
    /// @brief Gets the location of the OpenIPC home directory
    OpenIPC_CommonGetOpenIpcHomeDirectory GetOpenIpcHomeDirectory;

    ///
    /// @brief Gets the server configuration
    OpenIPC_CommonGetServerConfiguration GetServerConfiguration;

    ///
    /// @brief Flush all pending log messages and begin writing to new log
    ///        files.
    OpenIPC_CommonFlushLogs FlushLogs;

    ///
    /// @brief Flush all pending log messages, delete all non-archived log
    ///        files, and begin writing to new log files.
    OpenIPC_CommonClearLogs ClearLogs;

    ///
    /// @brief Flush all pending log messages, archive all non-archived log
    ///        files, and begin writing to new log files.
    OpenIPC_CommonArchiveLogs ArchiveLogs;

    ///
    /// @brief If the current logging configuration is different than the
    ///        configuration on disk, then update the configuration.
    OpenIPC_CommonSaveLoggingConfiguration SaveLoggingConfiguration;

    ///
    /// @brief Creates a file that will be left behind if we don't shut down cleanly,
    ///        and checks if any such files are left from previous runs
    OpenIPC_CommonInitializeProcessKilledDetector InitializeProcessKilledDetector;

    ///
    /// @brief Deletes the file created by OpenIPC_CommonInitializeProcessKilledDetector,
    ///        which means that this run shut down cleanly.
    OpenIPC_CommonUninitializeProcessKilledDetector UninitializeProcessKilledDetector;
};

extern "C"
{
    ///
    /// @brief Gets the Common component.
    ///
    /// @param[out] component
    ///     A pointer to a structure of function pointers representing the component.
    ///
    /// @return (OpenIPC_Error): A code indicating success or failure.
    /// @retval OpenIPC_Error_No_Error
    ///     Indicates success.
    /// @retval OpenIPC_Error_Null_Pointer
    ///     A parameter is NULL.
    ///
    EXPORT_COMMON_API OpenIPC_Error OpenIPC_GetCommon(OpenIPC_CommonComponent** component);
};
