/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
///  @file
///
///  @brief Contains types used throughout the Common component
///
//////////////////////////////////////////////////////////////////////////////

#pragma once

#include <Foundation/Types.h>
#include <Foundation/Error/ErrorTypes.h>

enum OpenIPC_CommonClientEventType
{
    OpenIPC_Common_ClientEvent_UNKNOWN = 0,
    OpenIPC_Common_ClientEvent_Connected,
    OpenIPC_Common_ClientEvent_Disconnected
};

/// @brief A direct representation of the server settings (usually defined in ServerProcess.xml)
typedef struct
{
    bool GuiEnabled;
    uint32_t GuiMaxLogMessagesCount;

    bool AllowExternalConnections;
    uint16_t StartPort;
    uint16_t EndPort;

    bool ShutdownWhenNoClientsAreConnected;
} OpenIPC_CommonServerConfiguration;

#define MAX_CLIENT_IDENTIFIER_STRING_LENGTH (256)

typedef struct
{
    int32_t ClientId;
    bool IsLocal;
    char ClientIdentifier[MAX_CLIENT_IDENTIFIER_STRING_LENGTH];
} OpenIPC_CommonClientDescriptor;

/// @brief This method provides a callback used when clients connect/disconnect.
///
/// @param[in] client
///     The descriptor for the client that performed the operation.
/// @param[in] eventType
///     The type of client event.
/// @param[in] data
///     The user-provided data that was supplied when the user of this API registered for this event.
typedef void (* OpenIPC_CommonClientCallback_Function)(const OpenIPC_CommonClientDescriptor* client, OpenIPC_CommonClientEventType eventType, void* data);

///
/// @brief Gets the path that the OpenIPC is installed in.
///
/// @param[in, out] pathSize
///     Going in, the maximum number of characters that can be written to the buffer.
///     Going out, the number of characters that are required for the full path.
/// @param[out] path
///     The buffer to write the path to.
///
/// @return (OpenIPC_Error): A code indicating success or failure.
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
/// @retval OpenIPC_Error_Null_Pointer
///     The @a path or @a pathSize parameter is NULL.
/// @retval OpenIPC_Error_Bad_Argument
///     The given buffer is too small. The @a pathSize parameter will contain the needed size.
typedef OpenIPC_Error (* OpenIPC_CommonGetPath_Function)(
    IN OUT size_t* pathSize,
    OUT char*      path);

///
/// @brief Gets the client ID that the current thread belongs to.
/// If the thread was not created by the server it gets a client ID
/// of -1. -1 is also the client ID used by in-process clients.
///
/// @param[out] clientId
///     The client ID.
///
/// @return (OpenIPC_Error): A code indicating success or failure.
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
/// @retval OpenIPC_Error_Null_Pointer
///     The @a clientId parameter is NULL.
typedef OpenIPC_Error (* OpenIPC_CommonGetClientId_Function)(
    OUT int32_t* clientId);

///
/// @brief Sets the client ID that the current thread belongs to.
/// The Server assigns the current thread to a client ID by calling
/// this method.
///
/// @param[in] clientId
///     The client ID.
///
/// @return (OpenIPC_Error): A code indicating success or failure.
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
typedef OpenIPC_Error (* OpenIPC_CommonSetClientId_Function)(
    IN int32_t clientId);

///
/// @brief Registers a new client.
///
/// @param[in] clientId
///     The client's unique ID.
/// @param[in] clientIdentifier
///     The identifier string for this client.
/// @param[in] isClientLocal
///     Whether or not the client is local.
///
/// @return (OpenIPC_Error): A code indicating success or failure.
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
/// @retval OpenIPC_Error_Null_Pointer
///     The @a 'clientIdentifier' parameter is NULL.
typedef OpenIPC_Error (* OpenIPC_CommonRegisterNewClient_Function)(
    IN int32_t      clientId,
    IN const char*  clientIdentifier,
    IN OpenIPC_Bool isClientLocal);

///
/// @brief Notifies any subscribers that the client is closed.
///
/// @param[in] clientId
///     The client ID.
///
/// @return (OpenIPC_Error): A code indicating success or failure.
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
typedef OpenIPC_Error (* OpenIPC_CommonCloseClient_Function)(
    IN int32_t clientId);

///
/// @brief Gets the number of registered clients.
///
/// @param[out] numClients
///     The number of registered clients.
///
/// @return (OpenIPC_Error): A code indicating success or failure.
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
/// @retval OpenIPC_Error_Null_Pointer
///     The @a numClients parameter is NULL.
typedef OpenIPC_Error (* OpenIPC_CommonGetNumRegisteredClients_Function)(
    OUT uint32_t* numClients);

///
/// @brief Gets the list of registered clients.
///
/// @param[in] maxNumClients
///     The maximum number of registered clients to return.
/// @param[out] numClients
///     The number of clients returned.
/// @param[out] clients
///     The list of clients.
///
/// @return (OpenIPC_Error): A code indicating success or failure.
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
/// @retval OpenIPC_Error_Null_Pointer
///     The @a clients or @a numClients parameter is NULL.
typedef OpenIPC_Error (* OpenIPC_CommonGetRegisteredClients_Function)(
    IN uint32_t                         maxNumClients,
    OUT uint32_t*                       numClients,
    OUT OpenIPC_CommonClientDescriptor* clients);

///
/// @brief Gets whether the client is on the same machine as the server.
///
/// @param[in] clientId
///     The client ID.
///
/// @param[out] isLocal
///     True if the client is on the same machine as the server
///
/// @return (OpenIPC_Error): A code indicating success or failure.
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
typedef OpenIPC_Error (* OpenIPC_CommonIsClientLocal_Function)(
    IN int32_t        clientId,
    OUT OpenIPC_Bool* isLocal);

///
/// @brief Adds a callback function to be called whenever a client
/// is connected/disconnected.
///
/// @param[in] callback
///     The function to call.
///
/// @return (OpenIPC_Error): A code indicating success or failure.
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
typedef OpenIPC_Error (* OpenIPC_CommonRegisterClientCallback_Function)(
    IN OpenIPC_CommonClientCallback_Function callback,
    IN void*                                 data);

///
/// @brief Removes a callback function.
///
/// @param[in] callback
///     The callback function.
///
/// @return (OpenIPC_Error): A code indicating success or failure.
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
typedef OpenIPC_Error (* OpenIPC_CommonUnregisterClientCallback_Function)(
    IN OpenIPC_CommonClientCallback_Function callback,
    IN void*                                 data);

///
/// @brief Initializes logging based on the logging XML configuration file.
///
/// @return (OpenIPC_Error): A code indicating success or failure.
/// @param[in] suffix
///     The the logging file suffix (null for default)
/// @param[in] persistLoggingPreset
///     Specifies if logging preset changes should be persisted to the logging xml file
///
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
typedef OpenIPC_Error (* OpenIPC_CommonInitializeLogging_Function)(
    IN const char* suffix,
    IN bool        persistLoggingPreset);

///
/// @brief Uninitializes logging
///
/// @return (OpenIPC_Error): A code indicating success or failure.
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
typedef OpenIPC_Error (* OpenIPC_CommonUninitializeLogging_Function)(
    void);

///
/// @brief Gets the location of the OpenIPC home directory.
///
/// Examples:
///     - C:\\Users\\bob\\.OpenIPC
///     - /home/Bob/.OpenIPC
///
/// @param[out]    directory A pointer to a buffer where the directory path can be stored
/// @param[in,out] size      Going in: the size of the buffer. Going out: The size of the directory path
///
/// @return (OpenIPC_Error): A code indicating success or failure
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
/// @retval OpenIPC_Error_Bad_Argument
///     Indicates that the given buffer size is too small. @a size will contain the required size.
/// @retval OpenIPC_Error_Null_Pointer
///     Indicates that one of the parameters is NULL.
typedef OpenIPC_Error (* OpenIPC_CommonGetOpenIpcHomeDirectory)(
    OUT char*      directory,
    IN OUT size_t* size);

///
/// @brief Gets the server configuration.
///
/// This is mostly the configuration parameters specified in the ServerProcess.xml file
///
/// @param[out] serverConfiguration A pointer to a struct that will be filled with the server configuration
///
/// @return (OpenIPC_Error): A code indicating success or failure
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
/// @retval OpenIPC_Error_Null_Pointer
///     The give @a serverConfiguration parameter is NULL.
typedef OpenIPC_Error (* OpenIPC_CommonGetServerConfiguration)(
    OUT OpenIPC_CommonServerConfiguration* serverConfiguration
    );

///
/// @brief Flush all pending log messages and begin writing to new log
///        files.
///
/// @return (OpenIPC_Error): A code indicating success or failure
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
typedef OpenIPC_Error (* OpenIPC_CommonFlushLogs)();

///
/// @brief Flush all pending log messages, delete all non-archived log
///        files, and begin writing to new log files.
///
/// @return (OpenIPC_Error): A code indicating success or failure
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
typedef OpenIPC_Error (* OpenIPC_CommonClearLogs)();

///
/// @brief Flush all pending log messages, archive all non-archived log
///        files, and begin writing to new log files.
///
/// @return (OpenIPC_Error): A code indicating success or failure
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
typedef OpenIPC_Error (* OpenIPC_CommonArchiveLogs)();

///
/// @brief If the current logging configuration is different than the
///        configuration on disk, then update the configuration.
///
/// @return (OpenIPC_Error): A code indicating success or failure
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
typedef OpenIPC_Error (* OpenIPC_CommonSaveLoggingConfiguration)();

///
/// @brief Creates a file that will be left behind if we don't shut down cleanly,
///        and checks if any such files are left from previous runs
///
/// @param[out]    found Will be set to true if we found a file left behind
///                      from a previous instance that wasn't shut down.
///
/// @return (OpenIPC_Error): A code indicating success or failure
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
/// @retval OpenIPC_Error_Null_Pointer
///     Indicates that one of the parameters is NULL.
typedef OpenIPC_Error (* OpenIPC_CommonInitializeProcessKilledDetector)(
    OUT OpenIPC_Bool* found);

///
/// @brief Deletes the file created by OpenIPC_CommonInitializeProcessKilledDetector,
///        which means that this run shut down cleanly.
///
/// @return (OpenIPC_Error): A code indicating success or failure
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
typedef OpenIPC_Error (* OpenIPC_CommonUninitializeProcessKilledDetector)();
