﻿/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
///  @file
///
///  @brief Implementation of functions for creating, destroying, and
///         manipulating BitData objects exposed as C-style functions.
///
//////////////////////////////////////////////////////////////////////////////
#pragma once

#include "BitDataTypes.h"
#include <Foundation/Types.h>
#include <Foundation/IndexList/IndexList.h>

#if defined(HOST_WINDOWS)
    #if defined(BITDATA_EXPORTS)
        #define EXPORT_BITDATA_API __declspec(dllexport)
    #elif defined(BITDATA_IMPORTS)
        #define EXPORT_BITDATA_API __declspec(dllimport)
    #else
        #define EXPORT_BITDATA_API
    #endif
#else
    #if defined(BITDATA_EXPORTS) || defined(BITDATA_IMPORTS)
        #define EXPORT_BITDATA_API __attribute__ ((visibility("default")))
    #else
        #define EXPORT_BITDATA_API
    #endif
#endif

extern "C"
{
    EXPORT_BITDATA_API bool BitData_GetBit(
        const BitData* bd,
        uint64_t	offset
    );

    EXPORT_BITDATA_API void BitData_SetBit(
        const BitData* bd,
        uint64_t	offset,
        bool bit
    );

    EXPORT_BITDATA_API uint32_t BitData_GetUInt32(
        const BitData*	bd,
        uint64_t			offset
    );

    EXPORT_BITDATA_API uint64_t BitData_GetUInt64(
        const BitData*	bd,
        uint64_t			offset
    );

    EXPORT_BITDATA_API uint64_t BitData_GetUpToUInt64(
        const BitData* const	src,
        const uint64_t			start,
        const uint32_t			size
    );

    EXPORT_BITDATA_API void BitData_GetBuffer(
        const BitData*	bd,
        uint64_t			offset,
        uint8_t*			bytearray,
        uint32_t			bytearraysize
    );

    EXPORT_BITDATA_API void BitData_ReadIndices(
        const BitData*	    bd,
        const IndexList*    indices,
        BitData*            value
    );

    EXPORT_BITDATA_API void BitData_WriteIndices(
        BitData*            bd,
        const IndexList*    indices,
        const BitData*      value
    );

    EXPORT_BITDATA_API void BitData_SetUInt32(
        BitData*		bd,
        uint64_t			offset,
        uint32_t			value
    );

    EXPORT_BITDATA_API void BitData_SetUInt64(
        BitData*		bd,
        uint64_t			offset,
        uint64_t			value
    );

    EXPORT_BITDATA_API void BitData_SetUpToUInt64(
        BitData* const	bd,
        const uint64_t	start,
        const uint32_t	size,
        const uint64_t	value
    );

    EXPORT_BITDATA_API void BitData_SetBuffer(
        BitData*		bd,
        uint64_t			offset,
        const uint8_t*	bytearray,
        uint32_t			bytearraysize
    );

    EXPORT_BITDATA_API void BitData_FindFirst(
        const BitData*	bd,
        uint64_t			start,
        uint64_t			size,
        uint64_t*			poffset
    );

    EXPORT_BITDATA_API void BitData_FindLast(
        const BitData*	bd,
        uint64_t			start,
        uint64_t			size,
        uint64_t*			poffset
    );

    EXPORT_BITDATA_API void BitData_PopCount(
        const BitData*	bd,
        uint64_t			start,
        uint64_t			size,
        uint64_t*			pbits
    );

    EXPORT_BITDATA_API void BitData_Parity(
        const BitData*	bd,
        uint64_t			start,
        uint64_t			size,
        bool*			ppar
    );

    EXPORT_BITDATA_API void BitData_IsValue(
        const BitData*	bd,
        uint64_t			start,
        uint64_t			size,
        uint64_t			value,
        bool*			psame
    );

    EXPORT_BITDATA_API void BitData_XorValue(
        BitData*	bd,
        uint64_t		start,
        uint64_t		size,
        uint64_t		value
    );

    EXPORT_BITDATA_API void BitData_InitValue(
        BitData*	bd,
        uint64_t		start,
        uint64_t		size,
        uint64_t		value
    );

    EXPORT_BITDATA_API void BitData_Randomize(
        BitData*	bd,
        uint64_t		start,
        uint64_t		size
    );

    EXPORT_BITDATA_API void BitData_ShiftLeft(
        BitData*	bd,
        uint64_t		start,
        uint64_t		size,
        uint64_t		count
    );

    EXPORT_BITDATA_API void BitData_ShiftRight(
        BitData*	bd,
        uint64_t		start,
        uint64_t		size,
        uint64_t		count
    );

    EXPORT_BITDATA_API void BitData_ReverseEndianess(
        BitData*	bd,
        uint64_t		startByte,
        uint64_t		numElements,
        uint32_t		byteWidth
    );

    EXPORT_BITDATA_API void BitData_Reverse(
        BitData*	bd,
        uint64_t		start,
        uint64_t		size
    );

    EXPORT_BITDATA_API void BitData_Copy(
        const BitData*	src,
        uint64_t			srcStart,
        BitData*		dst,
        uint64_t			dstStart,
        uint64_t			count
    );

    EXPORT_BITDATA_API void BitData_Compare(
        const BitData*	bd1,
        uint64_t			start1,
        const BitData*	bd2,
        uint64_t			start2,
        uint64_t			count,
        int32_t*			presult
    );

    EXPORT_BITDATA_API void BitData_BitwiseOr(
        BitData*		dest,
        uint64_t			deststart,
        const BitData*	src1,
        uint64_t			src1start,
        const BitData*	src2,
        uint64_t			src2start,
        uint64_t			size
    );

    EXPORT_BITDATA_API void BitData_BitwiseAnd(
        BitData*		dest,
        uint64_t			deststart,
        const BitData*	src1,
        uint64_t			src1start,
        const BitData*	src2,
        uint64_t			src2start,
        uint64_t			size
    );

    EXPORT_BITDATA_API void BitData_BitwiseXor(
        BitData*		dest,
        uint64_t			deststart,
        const BitData*	src1,
        uint64_t			src1start,
        const BitData*	src2,
        uint64_t			src2start,
        uint64_t			size
    );

    EXPORT_BITDATA_API void BitData_BitwiseSelect(
        BitData*		dest,
        uint64_t			deststart,
        const BitData*	src1,
        uint64_t			src1start,
        const BitData*	src2,
        uint64_t			src2start,
        const BitData*	mask,
        uint64_t			maskstart,
        uint64_t			size
    );

    EXPORT_BITDATA_API void BitData_IsIdentical(
        const BitData*	bd1,
        const BitData*	bd2,
        bool*			identical
    );

    EXPORT_BITDATA_API void BitData_ToBinary(
        const BitData*	bd,
        uint64_t			start,
        uint64_t			size,
        char*			    strbuffer,
        uint32_t		    buffersize
    );

    EXPORT_BITDATA_API void BitData_ToHex(
        const BitData*	bd,
        uint64_t			start,
        uint64_t			size,
        char*			    strbuffer,
        uint32_t	    	buffersize
    );

    EXPORT_BITDATA_API void BitData_ToDecimal(
        const BitData*	bd,
        uint64_t			start,
        uint64_t			size,
        char*			strbuffer,
        uint32_t			buffersize
    );

    EXPORT_BITDATA_API void BitData_ToString(
        const BitData*	bd,
        uint64_t		start,
        uint64_t		size,
        char*		strbuffer,
        uint32_t		buffersize
    );

    EXPORT_BITDATA_API void BitData_FromString(
        BitData*	bd,
        uint64_t		start,
        uint64_t		size,
        const char*	strbuffer
    );

    EXPORT_BITDATA_API void BitData_InsertUint64(
        BitData*	bd,
        uint64_t	start,
        uint64_t	size,
        uint64_t	value
    );

    EXPORT_BITDATA_API void BitData_InsertBitData(
        BitData*	bd,
        uint64_t		start,
        const BitData*	bdSrc,
        uint64_t		startSrc,
        uint64_t		size
    );

    EXPORT_BITDATA_API void BitData_Append(
        BitData*		bd,
        const BitData*	bdSrc
    );

    EXPORT_BITDATA_API void BitData_Delete(
        BitData*	bd,
        uint64_t		start,
        uint64_t		size
    );

    EXPORT_BITDATA_API void BitData_Resize(
        BitData*	bd,
        uint64_t		size
    );

    EXPORT_BITDATA_API uint32_t BitData_GenerateECC(
        const BitData*	bd,
        uint64_t			start,
        uint64_t			size,
        uint32_t*			ao4EccMasks
    );

    EXPORT_BITDATA_API void BitData_InitCrcDef(
        OpenIPC_CrcDef*	crcDef,
        uint32_t	o4Width,
        uint32_t	o4Polynomial,
        uint32_t	o4InitialValue,
        bool	bReflectIn,
        bool	bReflectOut,
        uint32_t	o4XorOut
    );

    EXPORT_BITDATA_API uint32_t BitData_GenerateCRC(
        const BitData*	bd,
        uint64_t			start,
        uint64_t			size,
        const OpenIPC_CrcDef*	crcDef
    );

    EXPORT_BITDATA_API BitData BitData_CreateLocalFromBuffer(
        uint64_t size,
        uint64_t capacity,
        void* buffer
    );

    EXPORT_BITDATA_API BitData* BitData_CreateManaged(
        uint64_t size
    );

    EXPORT_BITDATA_API BitData* BitData_CreateManagedFromBitData(
        const BitData*	src
    );

    EXPORT_BITDATA_API BitData* BitData_CreateManagedFromString(
        const char* buffer
    );

    EXPORT_BITDATA_API BitData* BitData_CreateManagedFromUInt64(
        uint64_t size,
        uint64_t value
    );

    EXPORT_BITDATA_API BitData* BitData_CreateManagedFromBuffer(
        uint64_t size,
        const void* buffer
    );

    EXPORT_BITDATA_API void BitData_FreeManaged(
        BitData* bd
    );
};
