/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
/// @file
///
//////////////////////////////////////////////////////////////////////////////
#pragma once

#include "Error.h"

#include <string>
#include <memory>
#include <sstream>

#include "ErrorMacros.h"

///
/// @brief A self-registering listener of all posted errors on the current thread.
class OpenIPC_ErrorContext
{
private:
    OpenIPC_ErrorToken _token;
public:
    OpenIPC_ErrorContext()
    {
        _token = OpenIPC_ErrorEnterContex();
    }

    OpenIPC_ErrorContext(const OpenIPC_ErrorContext& other) = delete;
    OpenIPC_ErrorContext(OpenIPC_ErrorContext&& other) noexcept = delete;
    OpenIPC_ErrorContext& operator=(const OpenIPC_ErrorContext& other) = delete;
    OpenIPC_ErrorContext& operator=(OpenIPC_ErrorContext&& other) noexcept = delete;

    ~OpenIPC_ErrorContext()
    {
        OpenIPC_ErrorExitContex(_token);
    }

    ///
    /// @brief Gets the full message containing all errors that occurred during
    /// lifetime of the context.
    std::string GetErrorMessage() const
    {
        uint32_t size = 0;
        OpenIPC_ErrorGetErrorMessageSize(_token, &size);

        std::string text;
        text.resize(size);

        OpenIPC_ErrorGetErrorMessage(_token, const_cast<char*>(text.data()), size);

        return text.substr(0, size - 1);
    }

    ///
    /// @brief Gets the full message containing all errors that occurred during
    /// lifetime of the context.
    std::string GetLastErrorMessage() const
    {
        uint32_t size = 0;
        OpenIPC_ErrorGetLastErrorMessageSize(_token, &size);

        std::string text;
        text.resize(size);

        OpenIPC_ErrorGetLastErrorMessage(_token, const_cast<char*>(text.data()), size);

        return text.substr(0, size - 1);
    }

    ///
    /// @brief Gets the error code of the last error to be reported
    OpenIPC_Error GetLastErrorCode() const
    {
        return OpenIPC_ErrorGetErrorCode(_token);
    }

    ///
    /// @brief Indicates whether a particular error occurred in the list of
    ///        stored errors maintained internally.
    ///
    /// @param errorCode
    ///     The OpenIPC error code to check if it has occurred.
    OpenIPC_Bool ErrorOccurred(OpenIPC_Error errorCode) const
    {
        return OpenIPC_ErrorOccurred(_token, errorCode);
    }

};
