/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
#pragma once

#include <cassert>
#include <cstdio>
#include <iostream>
#include <sstream>
#include <vector>
#include <deque>
#include <algorithm>

struct IndexList;

///
/// @brief An ordered list of indices.
template <typename T>
class OpenIPC_IndexListT
{
public:

    ///
    /// @brief An iterator over an index list.
    class Iterator
    {
        friend class OpenIPC_IndexListT;
    public:

        ///
        /// @brief Constructs an invalid iterator
        Iterator();

        ///
        /// @brief Deferences the iterator to the index it represents.
        const T& operator*() const;

        ///
        /// @brief Deferences the iterator to the index it represents.
        const T* operator->() const;

        ///
        /// @brief Moves to the next index in the index list.
        Iterator& operator++();

        bool operator==(const Iterator& other) const;
        bool operator!=(const Iterator& other) const;

    private:
        Iterator(const OpenIPC_IndexListT& indexList, size_t position);

        T _index;
        size_t _position;
        const OpenIPC_IndexListT* _indexList;
    };

    ///
    /// @brief Creates an empty index list.
    OpenIPC_IndexListT();

    ///
    /// @brief Creates an index list from a string.
    OpenIPC_IndexListT(const std::string& string);

    ///
    /// @brief Creates a continuous index list from the given start index to
    ///        the end index.
    ///
    /// @param start The start index.
    /// @param end The end index (inclusive).
    OpenIPC_IndexListT(T start, T end);

    ///
    /// @brief Creates a continuous index list from zero of the given length.
    ///
    /// @param length The length.
    OpenIPC_IndexListT(size_t length);

    ///
    /// @brief Returns a pointer to the underlying C struct.
    IndexList* GetCStruct();

    ///
    /// @brief Returns a pointer to the underlying C struct.
    const IndexList* GetCStruct() const;

    ///
    /// @brief Appends a new index to the end of the index list.
    ///
    /// @param index The index to append.
    void Append(T index);

    ///
    /// @brief Appends a contiguous range of indices to the end of the index
    ///        list.
    ///
    /// @note The index range may be increasing or decreasing.
    ///
    /// @param start The start of the index range to append.
    /// @param end The end of the index range to append.
    void AppendRange(T start, T end);

    ///
    /// @brief Applies an offset to each index in the index list.
    ///
    /// @param offset The offset to apply to each index.
    void ApplyOffset(T offset);

    ///
    /// @brief Slices the index by another index list.
    ///
    /// @param slice The index list used to slice.
    ///
    /// @returns The sliced index list.
    OpenIPC_IndexListT Slice(const OpenIPC_IndexListT& slice) const;

    ///
    /// @brief Resizes the index list to a new length.
    ///
    /// @param length The new length of the index list.
    void Resize(size_t length);

    ///
    /// @brief Gets the index at the specified position.
    ///
    /// @param position The position of the index to get.
    ///
    /// @returns The index at the specified position; -1 if no index exists
    ///          at the specified position.
    T GetIndex(size_t position) const;

    ///
    /// @brief Gets maximum index in the index list.
    ///
    /// @returns The maximum index; 0 if the index list is empty.
    T GetMaxIndex() const;

    ///
    /// @brief Gets minimum index in the index list.
    ///
    /// @returns The minimum index; 0 if the index list is empty.
    T GetMinIndex() const;

    ///
    /// @brief Returns the number of indices in the index list.
    size_t GetLength() const;

    ///
    /// @brief Returns whether the indices list is empty.
    bool IsEmpty() const;

    ///
    /// @brief Attempts to extract the offset and length of the index list.
    ///
    /// @returns True if the index list is a simple offset and length; false
    ///          otherwise.
    bool ExtractOffsetAndLength(T& offset, size_t& length) const;

    ///
    /// @brief Returns a string representing the index list.
    std::string ToString() const;

    ///
    /// @brief Returns an iterator at the beginning of the index list.
    Iterator begin() const;

    ///
    /// @brief Returns an iterator at the end of the index list.
    Iterator end() const;

private:
    static T Distance(T v1, T v2)
    {
        return (v1 > v2) ? (v1 - v2) : (v2 - v1);
    }

    // A sequence of indices with the index list (contiguous or non-contiguous)
    class Sequence
    {
    public:

        // Constructs an empty non-contiguous sequence
        Sequence();

        // Constructs a contiguous sequence for a range of indices
        Sequence(T start, T end);

        // Attempts to append an index to the sequence, returning true if the
        // index was appended or false if it was not
        bool Append(T index);

        // Gets the index at the specified position
        T GetIndex(size_t position) const;

        // Gets the length of the sequence
        size_t GetLength() const;

        // Returns whether the sequence is contiguous
        bool IsContiguous() const;

        // Returns whether the sequence is increasing
        bool IsIncreasing() const;

        // Applies an offset to each index in the sequence
        void ApplyOffset(T offset);

    private:
        bool _contiguous;
        bool _increasing;
        std::vector<T> _indices;
    };

    std::vector<Sequence> _sequences;
    size_t _length;
};

template <typename T>
OpenIPC_IndexListT<T>::Iterator::Iterator() :
    _index(T()),
    _position(0),
    _indexList(nullptr)
{
}

template <typename T>
const T& OpenIPC_IndexListT<T>::Iterator::operator*() const
{
    assert(_indexList);
    assert(_position <= _indexList->GetLength());
    return _index;
}

template <typename T>
const T* OpenIPC_IndexListT<T>::Iterator::operator->() const
{
    assert(_indexList);
    assert(_position <= _indexList->GetLength());
    return &_index;
}

template <typename T>
typename OpenIPC_IndexListT<T>::Iterator& OpenIPC_IndexListT<T>::Iterator::operator++()
{
    assert(_indexList);

    ++_position;
    if (_position < _indexList->GetLength())
    {
        _index = _indexList->GetIndex(_position);
    }

    return *this;
}

template <typename T>
bool OpenIPC_IndexListT<T>::Iterator::operator==(const Iterator& other) const
{
    return _position == other._position && _indexList == other._indexList;
}

template <typename T>
bool OpenIPC_IndexListT<T>::Iterator::operator!=(const Iterator& other) const
{
    return !(*this == other);
}

template <typename T>
OpenIPC_IndexListT<T>::Iterator::Iterator(const OpenIPC_IndexListT& indexList, size_t position) :
    _index(T()),
    _position(position),
    _indexList(&indexList)
{
    if (position < indexList.GetLength())
    {
        _index = indexList.GetIndex(position);
    }
}

template <typename T>
OpenIPC_IndexListT<T>::OpenIPC_IndexListT() :
    _length(0)
{
}

template <typename T>
OpenIPC_IndexListT<T>::OpenIPC_IndexListT(const std::string& string) :
    _length(0)
{
    // Tokenize the string
    std::stringstream ss(string);
    std::deque<std::string> tokens;

	{
		std::string token;
		while (std::getline(ss, token, ','))
		{
			tokens.push_front(token);
		}
	}

    for (const std::string& token : tokens)
    {
        int first = 0;
        int second = 0;
        int i;
#ifdef _MSC_VER
  #define SSCANF sscanf_s
  #else
  #define SSCANF sscanf
#endif
        i = SSCANF(token.c_str(), "%i:%i", &first, &second);
#undef SSCANF
        if (i == 1)
        {
            Append(first);
        }
        else if (i == 2)
        {
            AppendRange(second, first);
        }
    }
}

template <typename T>
OpenIPC_IndexListT<T>::OpenIPC_IndexListT(T start, T end) :
    _length(0)
{
    AppendRange(start, end);
}

template <typename T>
OpenIPC_IndexListT<T>::OpenIPC_IndexListT(size_t length) :
    _length(0)
{
    AppendRange(0, length - 1);
}

template <typename T>
IndexList* OpenIPC_IndexListT<T>::GetCStruct()
{
    return reinterpret_cast<IndexList*>(this);
}

template <typename T>
const IndexList* OpenIPC_IndexListT<T>::GetCStruct() const
{
    return reinterpret_cast<const IndexList*>(this);
}

template <typename T>
void OpenIPC_IndexListT<T>::Append(T index)
{
    _length += 1;

    // If this is the first index being added
    if (_sequences.empty())
    {
        // Add a non-contiguous sequence
        _sequences.push_back(Sequence());
    }

    // Try to append the index to the last sequence
    if (!_sequences.back().Append(index))
    {
        // Could not extend the last sequence, so create a new non-contiguous
        // sequence and append the index
        _sequences.push_back(Sequence());
        _sequences.back().Append(index);
    }
}

template <typename T>
void OpenIPC_IndexListT<T>::AppendRange(T start, T end)
{
    // Add a contiguous sequence for the range
    _sequences.push_back(Sequence(start, end));

    // Increment the length of the index list
    _length += _sequences.back().GetLength();
}

template <typename T>
void OpenIPC_IndexListT<T>::ApplyOffset(T offset)
{
    for (Sequence& sequence : _sequences)
    {
        sequence.ApplyOffset(offset);
    }
}

template <typename T>
OpenIPC_IndexListT<T> OpenIPC_IndexListT<T>::Slice(const OpenIPC_IndexListT& slice) const
{
    OpenIPC_IndexListT result;
    for (T index : slice)
    {
        T slicedIndex = GetIndex(static_cast<size_t>(index));
        result.Append(slicedIndex);
    }
    return result;
}

template <typename T>
void OpenIPC_IndexListT<T>::Resize(size_t length)
{
    OpenIPC_IndexListT result;
    for (size_t i = 0; i < length; ++i)
    {
        T index = GetIndex(i);
        result.Append(index);
    }

    *this = result;
}

template <typename T>
T OpenIPC_IndexListT<T>::GetIndex(size_t position) const
{
    T index = T(-1);

    // Find the sequence that the index at the specified position exists in
    size_t offset = 0;
    for (const Sequence& sequence : _sequences)
    {
        size_t length = sequence.GetLength();

        // If the index exists in this sequence
        size_t relativePosition = position - offset;
        if (position - offset < length)
        {
            // Get the index
            index = sequence.GetIndex(relativePosition);
            break;
        }

        offset += length;
    }

    return index;
}

template <typename T>
T OpenIPC_IndexListT<T>::GetMaxIndex() const
{
    bool indexSet = false;
    T index = 0;

    // Find the sequence that the index at the specified position exists in
    for (const Sequence& sequence : _sequences)
    {
        size_t length = sequence.GetLength();
        if (length > 0)
        {
            if (sequence.IsContiguous())
            {
                T newIndex;
                if (sequence.IsIncreasing())
                {
                    newIndex = sequence.GetIndex(length - 1);
                }
                else
                {
                    newIndex = sequence.GetIndex(0);
                }

                if (!indexSet)
                {
                    indexSet = true;
                    index = newIndex;
                }
                else
                {
                    index = std::max(index, newIndex);
                }
            }
            else
            {
                for (size_t position = 0; position < length; ++position)
                {
                    T newIndex = sequence.GetIndex(position);

                    if (!indexSet)
                    {
                        indexSet = true;
                        index = newIndex;
                    }
                    else
                    {
                        index = std::max(index, newIndex);
                    }
                }
            }
        }
    }

    return index;
}

template <typename T>
T OpenIPC_IndexListT<T>::GetMinIndex() const
{
    bool indexSet = false;
    T index = 0;

    // Find the sequence that the index at the specified position exists in
    for (const Sequence& sequence : _sequences)
    {
        size_t length = sequence.GetLength();
        if (length > 0)
        {
            if (sequence.IsContiguous())
            {
                T newIndex;

                if (sequence.IsIncreasing())
                {
                    newIndex = sequence.GetIndex(0);
                }
                else
                {
                    newIndex = sequence.GetIndex(length - 1);
                }

                if (!indexSet)
                {
                    indexSet = true;
                    index = newIndex;
                }
                else
                {
                    index = std::min(index, newIndex);
                }
            }
            else
            {
                for (size_t position = 0; position < length; ++position)
                {
                    T newIndex = sequence.GetIndex(position);

                    if (!indexSet)
                    {
                        indexSet = true;
                        index = newIndex;
                    }
                    else
                    {
                        index = std::min(index, newIndex);
                    }
                }
            }
        }
    }

    return index;
}

template <typename T>
size_t OpenIPC_IndexListT<T>::GetLength() const
{
    return _length;
}

template <typename T>
bool OpenIPC_IndexListT<T>::IsEmpty() const
{
    return _length == 0;
}


template <typename T>
bool OpenIPC_IndexListT<T>::ExtractOffsetAndLength(T& offset, size_t& length) const
{
    bool result = false;

    offset = T();
    length = 0;

    if (_sequences.size() == 1)
    {
        const Sequence& sequence = _sequences[0];
        if (sequence.GetLength() == 1 || (sequence.IsContiguous() && sequence.IsIncreasing()))
        {
            result = true;
            if (sequence.GetLength() > 0)
            {
                offset = sequence.GetIndex(0);
                length = sequence.GetLength();
            }
        }
    }

    return result;
}

template <typename T>
std::string OpenIPC_IndexListT<T>::ToString() const
{
    std::stringstream ss;
    for (const Sequence& sequence : _sequences)
    {
        if (sequence.GetLength() > 0) {
            if (sequence.IsContiguous()) {
                // Get the start/end of the contiguous sequence
                std::string start = std::to_string(sequence.GetIndex(0));
                std::reverse(start.begin(), start.end());
                std::string end = std::to_string(sequence.GetIndex(sequence.GetLength() - 1));
                std::reverse(end.begin(), end.end());

                // Append the sequence to the string
                ss << start << ":" << end << ",";
            }
            else {
                for (size_t i = 0; i < sequence.GetLength(); ++i) {
                    auto sindex = sequence.GetIndex(i);
                    if (sindex != T(-1)) {
                        std::string index = std::to_string(sindex);
                        std::reverse(index.begin(), index.end());

                        // Append the index to the string
                        ss << index << ",";
                    }
                }
            }
        }
    }

    std::string string = ss.str();
    if (string.length() > 0)
    {
        // Reverse the string and remove the extraneous comma
        std::reverse(string.begin(), string.end());
        string.erase(string.begin());
    }

    return string;
}

template <typename T>
typename OpenIPC_IndexListT<T>::Iterator OpenIPC_IndexListT<T>::begin() const
{
    return Iterator(*this, 0);
}

template <typename T>
typename OpenIPC_IndexListT<T>::Iterator OpenIPC_IndexListT<T>::end() const
{
    return Iterator(*this, _length);
}

template <typename T>
OpenIPC_IndexListT<T>::Sequence::Sequence() :
    _contiguous(false),
    _increasing(false)
{
}

template <typename T>
OpenIPC_IndexListT<T>::Sequence::Sequence(T start, T end) :
    _contiguous(true),
    _increasing(end >= start),
    _indices(2, 0)
{
    // Store the start index in the first element of the indices vector
    _indices[0] = start;

    // Store the length in the second element of the indices vector
    _indices[1] = Distance(start, end) + 1;
}

template <typename T>
bool OpenIPC_IndexListT<T>::Sequence::Append(T index)
{
    bool result = false;

    // If the sequence is non-contiguous
    if (!_contiguous)
    {
        // See if this range can transmutate into a contiguous range
        if (_indices.size() == 1 && Distance(_indices[0], index)  == 1)
        {
            // Change into a contiguous range
            _contiguous = true;
            _increasing = index > _indices[0];
            _indices.push_back(2); // Set the length to 2
            result = true;
        }
        else
        {
            // Append the index to the indices vector
            _indices.push_back(index);
            result = true;
        }
    }

    // If the sequence is contiguous and can be extended to include the index
    else if (( _increasing && _indices[0] + _indices[1] == index) ||
             (!_increasing && _indices[0] - _indices[1] == index))
    {
        // Extend the range to include the index
        _indices[1] += 1;
        result = true;
    }

    return result;
}

template <typename T>
T OpenIPC_IndexListT<T>::Sequence::GetIndex(size_t position) const
{
    T index = T(-1);

    if (position < GetLength())
    {
        // If the sequence is contiguous
        if (_contiguous)
        {
            // Interpolate the index based on the specified position in the range
            index = _increasing ? _indices[0] + position : _indices[0] - position;
        }

        // If the sequence is non-contiguous
        else
        {
            // Get the index at the specified position
            index = _indices[position];
        }
    }

    return index;
}

template <typename T>
size_t OpenIPC_IndexListT<T>::Sequence::GetLength() const
{
    if (!_contiguous)
    {
        return _indices.size();
    }
    else
    {
        return static_cast<size_t>(_indices[1]);
    }
}

template <typename T>
bool OpenIPC_IndexListT<T>::Sequence::IsContiguous() const
{
    return _contiguous;
}

template <typename T>
bool OpenIPC_IndexListT<T>::Sequence::IsIncreasing() const
{
    return _increasing;
}

template <typename T>
void OpenIPC_IndexListT<T>::Sequence::ApplyOffset(T offset)
{
    if (!_contiguous)
    {
        for (T& index : _indices)
        {
            index += offset;
        }
    }
    else
    {
        _indices[0] += offset;
    }
}

typedef OpenIPC_IndexListT<uint8_t> OpenIPC_IndexListUInt8;
typedef OpenIPC_IndexListT<uint16_t> OpenIPC_IndexListUInt16;
typedef OpenIPC_IndexListT<uint32_t> OpenIPC_IndexListUInt32;
typedef OpenIPC_IndexListT<uint64_t> OpenIPC_IndexListUInt64;

typedef OpenIPC_IndexListUInt64 OpenIPC_IndexList;
