/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////

#pragma once

#include "Logging.h"
#include "LoggingMacros.h"

#include <sstream>
#include <iomanip>
#include <vector>
#include <algorithm>
#include <functional>
#include <thread>

#ifdef HOST_LINUX
    #define __pragma(x)
#endif

class OpenIPC_Logger;

class OpenIPC_LoggerListener
{
public:
    virtual void ReceiveMessage(OpenIPC_Logger* logger, std::thread::id threadId, OpenIPC_LoggingSeverityLevel severity, const std::string& message) = 0;
    virtual void LogSeverityLevelChanged(OpenIPC_Logger* logger, OpenIPC_LoggingSeverityLevel severity) = 0;
};

class OpenIPC_Logger
{
public:
    ///
    /// @brief Returns an instance of a logger
    static OpenIPC_Logger* GetLogger(OpenIPC_LoggingLoggerId loggerId)
    {
        OpenIPC_LoggingToken token = OpenIPC_LoggingGetLogger(loggerId);
        return new OpenIPC_Logger(token, loggerId);
    }

    static const char* GetLoggerName(OpenIPC_LoggingLoggerId id)
    {
        const char* ret = nullptr;

        switch (id)
        {
            #define OPENIPC_LOGGER_LIST_NEXT
            #define OPENIPC_LOGGER_ID(NAME) case OpenIPC_LoggingLoggerId_ ## NAME: \
                ret = #NAME; \
                break; \

            OPENIPC_LOGGER_IDS

            #undef OPENIPC_LOGGER_ID
            #undef OPENIPC_LIST_NEXT
        case OpenIPC_LoggingLoggerId_Unknown:
            ret = "Unknown";
            break;
        }

        return ret;
    }

    template <typename TFunc>
    void Log(OpenIPC_LoggingSeverityLevel severity, const TFunc& fn, int indent = 0)
    {
        // Only log if the severity level is high enough.
        if (IsLogEnabled(severity) && !IsLoggingSuppressed())
        {
            std::stringstream ss;
            fn(ss);

            DoLog(severity, ss.str().c_str(), indent);
        }
    }

    bool IsLogEnabled(OpenIPC_LoggingSeverityLevel severity)
    {
        return OpenIPC_LoggingIsLogEnabled(_token, severity);
    }

    bool IsLoggingSuppressed()
    {
        return OpenIPC_LoggingIsLoggingSuppressed();
    }

    OpenIPC_LoggingSeverityLevel GetLogSeverityLevel()
    {
        return OpenIPC_LoggingGetLogSeverityLevel(_token);
    }

    void SetLogSeverityLevel(OpenIPC_LoggingSeverityLevel value)
    {
        OpenIPC_LoggingSetLogSeverityLevel(_token, value);
    }

    void GetSinkTargets(char(*sinkTargets)[OpenIPC_LOGGING_MAX_SINK_TARGET_NAME_LENGTH], uint32_t maxSinkTargets, uint32_t* numSinkTargets)
    {
        OpenIPC_LoggingGetLogSinkTargets(_token, sinkTargets, maxSinkTargets, numSinkTargets);
    }

    void SetSinkTargets(const char** sinkTargets, uint32_t numSinkTargets)
    {
        OpenIPC_LoggingSetLogSinkTargets(_token, sinkTargets, numSinkTargets);
    }

    OpenIPC_LoggingLoggerId GetId()
    {
        return _loggerId;
    }

    void AddListener(OpenIPC_LoggerListener* listener)
    {
        if (_listeners.size() == 0)
        {
            OpenIPC_LoggingAddListener(_token, reinterpret_cast<void*>(this), LoggingReceiveMessageCallback, LoggingSeverityLevelChangedCallback);
        }

        auto it = std::find(_listeners.begin(), _listeners.end(), listener);

        if (it == _listeners.end())
        {
            _listeners.push_back(listener);
        }
    }

    void RemoveListener(OpenIPC_LoggerListener* listener)
    {
        auto it = std::find(_listeners.begin(), _listeners.end(), listener);

        if (it != _listeners.end())
        {
            _listeners.erase(it);
        }

        if (_listeners.size() == 0)
        {
            OpenIPC_LoggingRemoveListener(_token, reinterpret_cast<void*>(this), LoggingReceiveMessageCallback, LoggingSeverityLevelChangedCallback);
        }
    }

private:
    OpenIPC_LoggingToken    _token;
    OpenIPC_LoggingLoggerId _loggerId;

    std::vector<OpenIPC_LoggerListener * > _listeners;

    OpenIPC_Logger(OpenIPC_LoggingToken token, OpenIPC_LoggingLoggerId loggerId) :
        _token(token),
        _loggerId(loggerId)
    {
    }

    void DoLog(OpenIPC_LoggingSeverityLevel severity, const char* message, int indent)
    {
        OpenIPC_LoggingLog(_token, severity, message, indent);
    }

    static void LoggingReceiveMessageCallback(void* data, OpenIPC_LoggingSeverityLevel severity, const char* message)
    {
        OpenIPC_Logger* _this = reinterpret_cast<OpenIPC_Logger*>(data);

        for (OpenIPC_LoggerListener* listener : _this->_listeners)
        {
            listener->ReceiveMessage(_this, std::this_thread::get_id(), severity, message);
        }
    }

    static void LoggingSeverityLevelChangedCallback(void* data, OpenIPC_LoggingSeverityLevel severity)
    {
        OpenIPC_Logger* _this = reinterpret_cast<OpenIPC_Logger*>(data);

        for (OpenIPC_LoggerListener* listener : _this->_listeners)
        {
            listener->LogSeverityLevelChanged(_this, severity);
        }
    }

};

class LogScope
{
    OpenIPC_Logger* _logger;
    OpenIPC_LoggingSeverityLevel _severity;
    const std::function<void(std::stringstream&)> _exit;
public:
    
    LogScope(OpenIPC_Logger * logger,
             OpenIPC_LoggingSeverityLevel severity,
             const std::function<void(std::stringstream&)>& entry,
             const std::function<void(std::stringstream&)> exit) :
        _logger(logger),
        _severity(severity),
        _exit(std::move(exit))
    {
        _logger->Log(_severity, entry, 1);
    }

    ~LogScope()
    {
        _logger->Log(_severity, _exit, -1);
    }

    LogScope(const LogScope &other)     = delete;
    LogScope(LogScope&& other) noexcept = delete;
    LogScope& operator=(const LogScope &other) = delete;
    LogScope& operator=(LogScope&& other)noexcept = delete;
};