# INTEL CONFIDENTIAL
# Copyright (c) 2017, Intel Corporation. All Rights Reserved.
# The source code contained or described herein and all documents related to the
# source code ("Material") are owned by Intel Corporation or its suppliers or
# licensors. Title to the Material remains with Intel Corporation or its
# suppliers and licensors. The Material contains trade secrets and proprietary
# and confidential information of Intel or its suppliers and licensors. The
# Material is protected by worldwide copyright and trade secret laws and treaty
# provisions. No part of the Material may be used, copied, reproduced, modified,
# published, uploaded, posted, transmitted, distributed, or disclosed in any way
# without Intel's prior express written permission.
# No license under any patent, copyright, trade secret or other intellectual
# property right is granted to or conferred upon you by disclosure or delivery
# of the Materials, either expressly, by implication, inducement, estoppel or
# otherwise. Any license under such intellectual property rights must be express
# and approved by Intel in writing.

import os
import sys
import argparse
import contextlib
import subprocess

IS_TEAMCITY_BUILD = os.environ.get('TEAMCITY_VERSION') is not None


class BuildException(Exception):
    pass


def main():
    try:
        parser = argparse.ArgumentParser(description='Build the ASD Open IPC plugin.')
        parser.add_argument('-c', '--configure_only', action='store_true',
                            help='just run the CMake configure step.')
        parser.add_argument('-b', '--build_only', action='store_true',
                            help='just build the plugin (no tests).')
        args = parser.parse_args()
        do_build(args)
    except BuildException:
        print 'Build Failed!'
        return 1
    print 'Build Succeeded!'
    return 0


def do_build(args):
    #setup_build_number()
    root_directory, binaries_directory, source_directory, working_directory = get_directories()
    create_build_workspace(working_directory)
    compile_project(args.configure_only, source_directory, working_directory)

def setup_build_number():
    build_number = get_build_number()
    report_build_number_to_teamcity(build_number)
    update_plugin_version(build_number)


def get_build_number():
    cmd = "git describe --tags --long".split()
    git_output = subprocess.check_output(cmd).rstrip().split('-')
    tag = git_output[0]
    commits_since_tag = int(git_output[1])
    short_sha1 = git_output[2][1:]
    if commits_since_tag == 0:
        return tag
    return short_sha1


def update_plugin_version(version):
    if IS_TEAMCITY_BUILD:
        file_to_search = os.path.join('OpenIPC_ProbePlugin', 'OpenIPC', 'Source', 'Components', 'Probe',
                                      'ASD_JTAG_Static', 'ProbePluginASD_JTAG.hpp')
        # Read in the file
        with open(file_to_search, 'r') as file_object:
            file_data = file_object.read()

        replace_string = '#define PROBE_PLUGIN_VERSION "' + version + '-"'

        # Replace the target string
        file_data = file_data.replace('#define PROBE_PLUGIN_VERSION "dev-"', replace_string)

        # Write the file out again
        with open(file_to_search, 'wb') as file_object:
            file_object.write(file_data)


def report_build_number_to_teamcity(build_number):
    if IS_TEAMCITY_BUILD:
        print "##teamcity[buildNumber '%s']" % build_number


def get_directories():
    root_directory = os.path.join(os.path.dirname(os.path.realpath(__file__)), '..')
    root_directory = os.path.abspath(root_directory)
    print 'Root Directory: ' + root_directory
    working_directory = os.path.join(root_directory, "workspace")
    print 'Build Directory: ' + working_directory
    source_directory = os.path.join(root_directory, "OpenIPC_ProbePlugin")
    print 'Source Directory: ' + source_directory
    binaries_directory = os.path.join(working_directory, 'build', 'PluginSDK', 'Bin')
    print 'Binaries Directory: ' + binaries_directory
    return root_directory, binaries_directory, source_directory, working_directory


def create_build_workspace(working_directory):
    if not os.path.exists(working_directory):
        os.makedirs(working_directory)


def compile_project(configure_only, source_directory, working_directory):
    cmake_configure(source_directory, working_directory)
    if not configure_only:
        cmake_build(working_directory)


def cmake_configure(source_directory, working_directory):
    if sys.platform == "win32":
        command = ['cmake', '-DCMAKE_BUILD_TYPE=RELEASE', '-DDEFINE_ROBOT=ON', '-G',
                   'Visual Studio 14 2015 Win64', source_directory]
    else:
        command = ['cmake', '-DCMAKE_BUILD_TYPE=RELEASE', '-DDEFINE_ROBOT=ON', source_directory]
    with change_directory(working_directory):
        command_line(command)


def cmake_build(working_directory):
    print 'Building Binaries...'
    if sys.platform == 'win32':
        command = ['cmake', '--build', '.', '--config', 'Release', '--', '/maxcpucount:8']
    else:
        command = ['make']
    with change_directory(working_directory):
        command_line(command)


def run_all_unit_tests(tests_directory, working_directory):
    print 'Running Unit Tests...'
    exception = None
    results_file = os.path.abspath(os.path.join(working_directory,
                                                "ASD_Unit_Tests_x64.xml"))
    try:
        run_unit_tests('./ASD_Unit_Tests_x64', tests_directory, results_file)
        print 'Unit Tests "ASD_Unit_Tests_x64" passed'
    except BuildException as err:
        print 'Unit Tests "ASD_Unit_Tests_x64" failed!'
        exception = err
    print "##teamcity[importData type='gtest' path='" + results_file + "']"

    if exception is not None:
        raise exception


def run_unit_tests(executable_file, tests_directory, results_file):
    if IS_TEAMCITY_BUILD:
        command = [executable_file, '--gtest_output=xml:' + results_file]
    else:
        command = [executable_file]
    with change_directory(tests_directory):
        command_line(command)

def run_code_coverage(working_directory):
    if sys.platform != 'win32':  # sorry, no windows support yet.
        print 'Running Code Coverage...'
        command = ['make', 'ASD_Unit_Tests_Coverage']
        with change_directory(working_directory):
            command_line(command)
        if IS_TEAMCITY_BUILD:
            coverage_directory = os.path.join(working_directory, 'OpenIPC',
                                              'Tests', 'ASD_Unit_Tests',
                                              'ASD_Unit_Tests_Coverage')
            print "##teamcity[publishArtifacts '{0}=>coverage.zip']".format(coverage_directory)
            report_coverage_to_teamcity(os.path.join(coverage_directory,
                                                     'index.html'))

@contextlib.contextmanager
def change_directory(new_dir):
    current_dir = os.getcwd()
    os.chdir(new_dir)
    try:
        yield
    finally:
        os.chdir(current_dir)

def command_line(command):
    print 'Current Working Directory: ' + os.getcwd()
    print 'Executing Command: ' + ' '.join(command)
    process = subprocess.Popen(command)
    process.communicate()
    if process.returncode != 0:
        raise BuildException(process.returncode, command)

def report_coverage_to_teamcity(coverage_html_file):
    # install this on build agents with: 'pip install lxml'
    from lxml import html
    with open(coverage_html_file) as file:
        tree = html.fromstring(file.read())
        metrics = tree.xpath('//td[@class="headerCovTableEntry"]/text()')
        base_str = "##teamcity[buildStatisticValue key='{0}' value='{1}']"
        print base_str.format('CodeCoverageAbsLCovered', metrics[0])
        print base_str.format('CodeCoverageAbsLTotal', metrics[1])

def report_artifacts_to_teamcity(root_directory, binaries_directory):
    if IS_TEAMCITY_BUILD:
        print "##teamcity[publishArtifacts '{0}=>artifacts.zip']".format(binaries_directory)
        print "##teamcity[publishArtifacts '{0}=>config.zip']".format(os.path.join(root_directory, 'config'))


if __name__ == "__main__":
    sys.exit(main())
