import sys
import os
import io
import shutil
import glob
import subprocess
import time
import stat
from subprocess import Popen, PIPE, STDOUT
from inspect import getframeinfo, stack
import utils  # internal utilities file
import json

from build_utils import logger


def remove_line_with_substring_from_multiline_string(line_with_substring_to_remove, multiline_string):
	text = '\n'.join(line for line in multiline_string.split('\n') if line_with_substring_to_remove not in line)
	return text


def run_cmd_blocking(cmd):
	caller = getframeinfo(stack()[1][0])
	logger.info("Called from :%s::%s %d" % (caller.filename.split("\\")[-1], caller.function, caller.lineno))
	logger.info(cmd)

	# response = subprocess.run(shlex.quote(cmd), capture_output = True)
	response = subprocess.run(cmd, capture_output=True, shell=True)

	stderr = response.stderr.decode("utf8")
	stderr = remove_line_with_substring_from_multiline_string("Picked up _JAVA_OPTIONS: -Djava.net.preferIPv4Stack=true",
															stderr)
	stderr = remove_line_with_substring_from_multiline_string("error(s)", stderr)

	if stderr:
		# logger.error("stderr: ")
		logger.error("stderr: " + stderr)

	if response.stdout:
		# logger.info("stdout: ")
		logger.info("stdout: " + response.stdout.decode("utf8"))

	return response.stdout.decode("utf8")


def remove_file(filename):
	try:
		os.chmod(filename, stat.S_IWRITE)
		os.remove(filename)
	except OSError as e:
		if e.errno == 2:
			# if file not exists
			pass
		else:
			Error(e, True)
		# pass


def Error(s, kill = False):
	logger.error(str(s))
	if kill:
		# os.system("pause")
		sys.exit(1)


def copy_shared_files(shared_dir_path, output_shared_dir_path):
	if not os.path.exists(output_shared_dir_path):
		os.makedirs(output_shared_dir_path)

	src_files = os.listdir(shared_dir_path)
	for file_name in src_files:
		full_file_name = os.path.join(shared_dir_path, file_name)
		if os.path.isfile(full_file_name):
			shutil.copy2(full_file_name, output_shared_dir_path)


class FlushFile(object):
	"""Write-only flushing wrapper for file-type objects."""
	def __init__(self, f):
		self.f = f

	def write(self, x):
		self.f.write(x)
		self.f.flush()


class make_arc(object):
	# def __init__(self, developmentPath, projectName, configuration_name, outputBinaryName, outputBinarySizeStr, kwBuild):
	def __init__(self, development_path, project_name, configuration_name, kw_build):
		self.development_path = os.path.abspath(development_path)
		self.project_name = project_name
		self.configuration_name 	= configuration_name
		# self.outputBinaryName = outputBinaryName
		# self.outputBinarySizeStr = outputBinarySizeStr

		self.auto_build_path 		= os.path.join(self.development_path, "fw_src", "AutoBuild")
		self.metaware_path 			= os.path.join(self.development_path, "fw_src", "firmware", "builds", "metaware")
		self.output_objects_path 	= os.path.join(self.development_path, "fw_src", "output", configuration_name, "objects")
		self.output_images_dir_path = os.path.join(self.development_path, "fw_src", "output", configuration_name, "images")
		self.klockwork_dir 			= os.path.join(self.development_path, "fw_src", "output", "klockwork")

		self.flavour_path = os.path.join(self.metaware_path, configuration_name)

		# self.build_tcf_lcf_path = self.output_objects_path
		self.dot_path = "."
		self.make_product_family = []
		self.make_product = ""
		self.make_phytype = ""
		self.make_rftype = ""
		self.make_corename = ""

		logger.info("configuration_name.lower() :" + configuration_name.lower())
		logger.info("os.getenv(\"chd_build_projectdir\") :" + str(os.getenv("chd_build_projectdir")))

		if "debug" in configuration_name.lower() and os.getenv("chd_build_projectdir"):
			kw_build = "false"

		# self.kwrun = kwBuild # false/kwBuild
		self.kw = kw(configuration_name, self.klockwork_dir, self.output_objects_path, self.auto_build_path, kw_build)
	# kw_build : {false/"kwBuild"}

	def generate_make_file(self):

		# copy makefile to current directory
		remove_file(os.path.join(self.output_objects_path, "makefile"))
		shutil.copy2(os.path.join(self.metaware_path, "makefile"), os.path.join(self.output_objects_path, "makefile"))

		remove_file(os.path.join(self.output_objects_path, "flavour.mk"))
		shutil.copy2(os.path.join(self.flavour_path, "flavour.mk"), os.path.join(self.output_objects_path, "flavour.mk"))

		# logger.info("project_name: " + self.project_name)

		# get FLAGS from makefile
		with open('flavour.mk', 'r') as f:
			for line in f:
				if "PRODUCT :=" in line:
					self.make_product = line.split(":=")[1].strip()
					logger.info("make_product: " + self.make_product)
				if "PRODUCT_FAMILY :=" in line:
					self.make_product_family = line.split(":=")[1].strip().split(",")
					self.make_product_family = list(map(str.strip, self.make_product_family))
					logger.info("make_product_family: " + str(", ".join(self.make_product_family)))
				if "PHYTYPE :=" in line:
					self.make_phytype = line.split(":=")[1].strip()
					logger.info("make_phytype: " + self.make_phytype)
				if "RFTYPE :=" in line:
					self.make_rftype = line.split(":=")[1].strip()
					logger.info("make_rftype: " + self.make_rftype)
				if "OUT_NAME :=" in line:
					self.make_corename = line.split(":=")[1].strip()
					logger.info("make_corename: " + self.make_corename)

		if self.make_product_family == [] or self.make_product == "" or self.make_phytype == "" or \
			self.make_rftype == "" or self.make_corename == "":
			Error("some flag/s are missing, make file must contain $(RFTYPE), $(PHYTYPE), $(PRODUCT) !!", True)

		# copy lower_mac_includes.txt / upper_mac_includes.txt file to current directory
		remove_file(os.path.join(self.output_objects_path, "includes.txt"))
		lines = []
		for product_family_item in self.make_product_family:
			inc_filename = os.path.join(self.metaware_path, self.make_corename + "_" + product_family_item + "_includes.txt")
			if os.path.exists(inc_filename):
				with open(inc_filename, 'r') as fh:
					lines += fh.readlines()
			inc_inc_filename = os.path.join(self.metaware_path, self.make_corename + "_" + product_family_item +
											"_includes_increment.txt")
			if os.path.exists(inc_inc_filename):
				with open(inc_inc_filename, 'r') as fh:
					lines += fh.readlines()

		if self.make_product not in self.make_product_family:
			with open(os.path.join(self.metaware_path, self.make_corename + "_" + self.make_product +
													"_includes_increment.txt"), 'r') as fh:
				lines += fh.readlines()

		with open(os.path.join(self.output_objects_path, "includes.txt"), 'w') as fh:
			fh.writelines(lines)

		# copy lower_mac_sources.txt / upper_mac_sources.txt file to current directory
		remove_file(os.path.join(self.output_objects_path, "sources.mk"))
		lines = []
		for product_family_item in self.make_product_family:
			src_filename = os.path.join(self.metaware_path, self.make_corename + "_" + product_family_item + "_sources.mk")
			if os.path.exists(src_filename):
				with open(src_filename, 'r') as fh:
					lines += fh.readlines()
			src_inc_filename = os.path.join(self.metaware_path, self.make_corename + "_" + product_family_item +
											"_sources_increment.mk")
			if os.path.exists(src_inc_filename):
				with open(src_inc_filename, 'r') as fh:
					lines += fh.readlines()

		if self.make_product not in self.make_product_family:
			with open(os.path.join(self.metaware_path, self.make_corename + "_" + self.make_product +
													"_sources_increment.mk"), 'r') as fh:
				lines += fh.readlines()

		with open(os.path.join(self.output_objects_path, "sources.mk"), 'w') as fh:
			fh.writelines(lines)

		############################################################
		# includes.txt - replace all macros of the includes.txt file
		############################################################
		replacements = {'$(PHYTYPE)': self.make_phytype, '$(RFTYPE)': self.make_rftype}
		lines = []
		with open('includes.txt') as infile:
			for line in infile:
				for src, target in replacements.items():
					line = line.replace(src, target)
				if '$(PRODUCT)' in line:
					line_option = line.replace('$(PRODUCT)', self.make_product)
					# logger.info("line option: " + line_option)
					# logger.info("line option: " + str(line_option.split("\"")))
					# logger.info("line option: " + line_option.split("\"")[1])
					# logger.info("line option: " + os.path.join(self.output_objects_path, line_option.split("\"")[1]))
					# logger.info("line option: " + os.path.abspath(os.path.join(self.output_objects_path,
					# line_option.split("\"")[1])))
					line_as_path = os.path.abspath(os.path.join(self.output_objects_path, line_option.split("\"")[1]))
					if os.path.exists(line_as_path):
						line = line_option
					else:
						exist_flag = False
						# iterate through the family list backwards until a file exists
						for product_family_item in reversed(self.make_product_family):
							line_option = line.replace('$(PRODUCT)', product_family_item)
							line_as_path = os.path.abspath(os.path.join(self.output_objects_path, line_option.split("\"")[1]))
							if os.path.exists(line_as_path):
								line = line_option
								exist_flag = True
								break
						if not exist_flag:
							Error("no file for line : " + line, True)
				lines.append(line)
		with open('includes.txt', 'w') as outfile:
			outfile.writelines(lines)

		############################################################
		# sources.mk - replace all macros of the sources.mk file
		############################################################
		replacements = {'$(PHYTYPE)':self.make_phytype, '$(RFTYPE)':self.make_rftype}
		lines = []
		with open('sources.mk') as infile:
			for line in infile:
				for src, target in replacements.items():
					line = line.replace(src, target)
				if '$(PRODUCT)' in line:
					line_option = line.replace('$(PRODUCT)', self.make_product)
					# logger.info("line option: " + line_option)
					# logger.info("line option: " + str(line_option.split(" ")))
					# logger.info("line option: " + line_option.split(" ")[0])
					# logger.info("line option: " + line_option.split(" ")[0].replace('$(PROJECT)/', ""))
					# logger.info("line option: " + os.path.join(self.development_path,
					# line_option.split(" ")[0].replace('$(PROJECT)/', "")))
					# logger.info("line option: " + os.path.abspath(os.path.join(self.development_path,
					# line_option.split(" ")[0].replace('$(PROJECT)/', ""))))
					line_as_path = os.path.abspath(os.path.join(self.development_path, "fw_src",
																line_option.split(" ")[0].replace('$(PROJECT)/', "")))
					if os.path.exists(line_as_path):
						line = line_option
					else:
						exist_flag = False
						# iterate through the family list backwards until a file exists
						for product_family_item in reversed(self.make_product_family):
							line_option = line.replace('$(PRODUCT)', product_family_item)
							line_as_path = os.path.abspath(os.path.join(self.development_path, "fw_src",
																		line_option.split(" ")[0].replace('$(PROJECT)/', "")))
							if os.path.exists(line_as_path):
								line = line_option
								exist_flag = True
								break
						if not exist_flag:
							Error("no file for line : " + line, True)
				lines.append(line)
		with open('sources.mk', 'w') as outfile:
			outfile.writelines(lines)

		######################
		# get list of c/s files
		######################
		with open(os.path.join(self.output_objects_path, "sources.mk"), 'r') as fh:
			lines = fh.readlines()

		path_file_names = []
		suffix = [".c", ".s"]
		for line in lines:
			if any(s in line for s in suffix):
				path_file_names.append(line.rstrip("\n").rstrip("\\").strip())
		# print(path_file_names)

		#############################
		# create list of object files
		#############################
		object_file_names = []
		for line in path_file_names:
			object_file_names.append(os.path.basename(line).replace(".c", ".o").replace(".s", ".o"))
		# print(object_file_names)

		#########################################################################################
		# create argfile for linker - list of all objects and where it's located: ./output/file.o
		#########################################################################################
		argfile_lines = [self.dot_path.replace("\\", "/") + "/" + object_file_name + "\n" for object_file_name in
						object_file_names]

		auto_gen_line_1 = "################################################################################\n"
		auto_gen_line_2 = "#      Automatically generated by build_arc.py \n"
		auto_gen_line_3 = "################################################################################\n"

		with open('argfile.txt', 'w') as argfile_fh:
			argfile_fh.write(auto_gen_line_1)
			argfile_fh.write(auto_gen_line_2)
			argfile_fh.write(auto_gen_line_3)
			argfile_fh.write("\n\n")
			argfile_fh.writelines(argfile_lines)

		####################
		# create subdir.mk
		####################
		def build_subdir_line(object_file_name, path_file_name):
			line = "./" + object_file_name + ": " + path_file_name.replace("\\", "/") + "\n"
			line += "	ccac -c $(COMPILER_FLAGS) \"$<\" -o \"$@\"" + "\n"
			line += "\t@echo 'Finished building: $<'\n"
			return line

		subdir_lines = [build_subdir_line(x, y) for x, y in zip(object_file_names, path_file_names)]
		with open('subdir.mk', 'w') as subdir_fh:
			subdir_fh.write(auto_gen_line_1)
			subdir_fh.write(auto_gen_line_2)
			subdir_fh.write(auto_gen_line_3)
			subdir_fh.write("\n\n")
			subdir_fh.writelines(subdir_lines)

	def make(self, cmd):
		kw_command = self.kw.get_command(cmd)

		if sys.platform == "win32":
			build_command = kw_command + 'gmake ' + cmd
		else:
			build_command = kw_command + 'make ' + cmd
		logger.info("build_command: " + build_command) # build_command: gmake -s -j4 all

		# Replace stdout with an automatically flushing version
		sys.stdout = FlushFile(sys.__stdout__)

		if sys.platform == "win32":
			proc = Popen(build_command, stdout=PIPE, stderr = STDOUT, universal_newlines = True)
		else:
			proc = Popen(build_command, stdout=PIPE, stderr = STDOUT, universal_newlines = True, shell = True)

		while True :
			line = proc.stdout.readline()
			# if line == '':# and proc.poll() != None:
			if line == '' and proc.poll() is not None:
				break
			if line:
				print(line.strip())

		return_code = proc.poll()

		sys.stdout = sys.__stdout__

		# logger.info("return_code: " + str(return_code))

		if return_code != 0:
			# Error(return_code, True);
			raise ValueError('build_arc.py make failed')

	def build(self, build_option):

		# change working directory to C:\work\FW_6.0.1\firmware\builds\metaware\lower_mac\
		# fpga_arc_lower_mac_cpu_wave600_ap_dummy_phy_wrx500_none
		cwd = os.getcwd()
		logger.info("current working directoy:" + cwd)
		os.chdir(self.output_objects_path)
		logger.info("current build directoy:" + self.output_objects_path)

		# Generate subdir.mk file needed for compilation
		self.generate_make_file()

		# If asked for rebuild, first clean
		if build_option == "cleanAll":
			self.make("clean")
			self.kw.prep("clean")
		else:
			self.kw.prep()

		# on TeamCity servers (chd_build servers) we can utilize 24 cores, o/w we will attempt to use max 4
		if ("chd_build_projectdir" in os.environ.keys()):
			num_cores = "24"
		else:
			num_cores = "4"

		# import multiprocessing
		# num_cores = multiprocessing.cpu_count()
		# logger.info("num_cores :" + str(num_cores))

		self.make("-s -j" + str(num_cores) + " all")

		self.validate_output()

		os.chdir(cwd)
		
	def check_file_exists(self, file_name):
		if not os.path.exists(file_name):
			Error("file :" + file_name + " not exists, aborting...", kill = True)

	def validate_output(self):
		# assuming cwd is the current build dir
		# at this folder, there should be "objects" folder
		# we expect the following file: lower_mac.elf, MT_Build_Config.h, lower_mac.hex, 
		# lower_mac.map, flavour.mk, sources.mk, subdir.mk,  includes.txt, argfile.txt

		# temp = vars(self)
		# for item in temp:
			# print (item , ' : ' , temp[item])

		self.check_file_exists("MT_Build_Config.h")
		self.check_file_exists("argfile.txt")
		self.check_file_exists("includes.txt")
		self.check_file_exists("subdir.mk")
		self.check_file_exists("sources.mk")
		self.check_file_exists("flavour.mk")
		self.check_file_exists(self.make_corename + ".elf")
		self.check_file_exists(self.make_corename + ".hex")
		self.check_file_exists(self.make_corename + ".map")

	def post_process_build(self):
		#######################
		# build post processing
		#######################

		logger.info("---------- start of metaware arc build post processing ----------")

		# maxMemConsumptionSize = int(self.outputBinarySizeStr)

		###########################
		# prepare directories names
		###########################
		build_output_source_dir_path 		= self.output_objects_path
		self.build_output_source_dir_path 	= build_output_source_dir_path
		# output_images_dir_path 				= os.path.join(self.development_path, 
		# "fw_src", "output", self.configuration_name, "images")
		shared_dir_path						= os.path.join(self.development_path, "shared_header")
		output_shared_dir_path				= os.path.join(self.output_images_dir_path, "shared")

		######################################
		# remove all files on the output folder
		######################################
		utils.RemoveFilesAndFoldersInFolder(self.output_images_dir_path)

		self.cwd = os.getcwd()
		#################
		# create lst file
		#################

		os.chdir(build_output_source_dir_path)
		if sys.platform == "win32":
			elf2lst_command = "elfdumparc.exe -T " + self.project_name + ".elf > " + self.project_name + ".lst"
		else:
			elf2lst_command = "elfdumparc -T " + self.project_name + ".elf > " + self.project_name + ".lst"

		run_cmd_blocking(elf2lst_command)

		##########################################################
		# create verilog ascii files (hex files for HW simulation)
		##########################################################
		if "debug" in self.configuration_name.lower():

			os.chdir(build_output_source_dir_path)
			if sys.platform == "win32":
				elf2vlod_command = "elf2hex.exe -n64 -V " + self.project_name + ".elf -o " + self.project_name + ".vlod"
			else:
				elf2vlod_command = "elf2hex -n64 -V " + self.project_name + ".elf -o " + self.project_name + ".vlod"

			run_cmd_blocking(elf2vlod_command)

		#################
		# run klocwork
		#################

		# if "release" in self.configuration_name:
		self.kw.run(build_output_source_dir_path)

		os.chdir(self.output_images_dir_path) # C:\work\FW_wave600_2\output\fpga_arc_upper_mac_cpu_wave500b_ap_dummy_phy_wrx500_none\images

		# ######################################
		# #remove all files on the output folder
		# ######################################
		# utils.RemoveFilesAndFoldersInFolder(output_images_dir_path)

		#########################################
		# copy build output files to output folder
		#########################################
		# logger.info("copy from build_output_source_dir_path:"+build_output_source_dir_path + 
		# "to output_images_dir_path:"+output_images_dir_path)
		remove_file(os.path.join(self.output_images_dir_path, "contr"))
		shutil.copy2(os.path.join(build_output_source_dir_path, self.project_name + ".elf"),
					os.path.join(self.output_images_dir_path, "contr"))
		remove_file(os.path.join(self.output_images_dir_path, "bin.bin"))
		shutil.copy2(os.path.join(build_output_source_dir_path, self.project_name + ".hex"),
					os.path.join(self.output_images_dir_path, "bin.bin"))
		remove_file(os.path.join(self.output_images_dir_path, "map.map"))
		shutil.copy2(os.path.join(build_output_source_dir_path, self.project_name + ".map"), 
					os.path.join(self.output_images_dir_path, "map.map"))

		###################
		#copy shared files
		###################
		copy_shared_files(shared_dir_path, output_shared_dir_path)

		##################
		# check memory consumption size
		##################
		# open map file and get the last address of the ram, 
		# there is a section at the end of the ram that is called ".end_of_ram"
		# lines on the map file:
		# .end_of_ram
		#     bss    0004d7f0  0004d7ef  00000000

		os.chdir(self.output_images_dir_path)  # C:\work\FW_wave600_2\output\fpga_arc_upper_mac_cpu_wave500b_ap_dummy_phy_wrx500_none\images

		map_path_file_name = os.path.join(build_output_source_dir_path, self.project_name + ".map")
		if not os.path.exists(map_path_file_name):
			Error("map file not exists !", True)

		start_of_iram_address = None
		end_of_ram_address = None
		with open(map_path_file_name , 'r') as infile:
			for line in infile:
				line = line.strip()
				if ".mpu_start_of_ram" == line:
					start_of_iram_address = next(infile).split()[1]
					continue
				if ".end_of_ram" == line:
					end_of_ram_address = next(infile).split()[1]
					break

		if end_of_ram_address is None:
			Error("error .end_of_ram section not found !", True)

		start_of_iram_int = int(start_of_iram_address, 16)
		end_of_ram_address_int = int(end_of_ram_address, 16)

		memory_consumption_size = end_of_ram_address_int - start_of_iram_int
		logger.info("os.getenv(\"chd_build_projectdir\") :" + str(os.getenv("chd_build_projectdir")))

		logger.info("image size is: " + str(memory_consumption_size) + " Bytes")

		import MapfileParser
		MapfileParser.main(os.path.join(self.development_path, "fw_src")) # C:\work\VersionBuilding\FW_6.1.0\fw_src

		################
		# create elfdump for enum parsing
		################
		elfdumpac_program = "c:/ARC/MetaWare/arc/bin/elfdumpac.exe"
		if not os.path.exists(elfdumpac_program):
			Error("elfdumpac.exe not found in :" + str(elfdumpac_program), True)

		elf_file_path = os.path.join(build_output_source_dir_path, self.project_name + ".elf")
		out_file = os.path.join(build_output_source_dir_path, self.project_name + ".txt")
		cmd = elfdumpac_program + " -G " + elf_file_path + " > " + out_file
		os.chdir(build_output_source_dir_path)
		run_cmd_blocking(cmd)

		# os.chdir(self.auto_build_path)
		elf_out_file_parser_script = os.path.join(self.auto_build_path, "elfOutFileParser.py")
		cmd = "python " + elf_out_file_parser_script + " " + out_file
		run_cmd_blocking(cmd)

		logger.info("---------- end of metaware arc build post processing ----------")


class kw(object):
	def __init__(self, configuration_name, kw_dir, build_path, auto_build_path, kw_build = ""):
		logger.info("kw_build :"+str(kw_build))
		self.kw_build = kw_build # "false"/"kwBuild"
		self.configuration_name = configuration_name
		self.kw_dir_report_path = kw_dir
		self.kwcheck_list_report_file_name = self.configuration_name + "_kwcheck_list_report.txt"
		self.kwcheck_run_report_file_name = self.configuration_name + "_kwcheck_run_report.txt"
		self.kwcheck_run_report_summary_file_name = self.configuration_name + "_kwcheck_run_report_summary.json"

		self.kw_app_path = ""
		self.kwcheck_cmd = ""
		self.kw_result_summary = {}
		# self.coreName = self.configuration_name.split("_")[1]
		if kw_build == "false":
			self.kw_build = ""
		self.auto_build_path = auto_build_path
		self.output_objects_path = build_path
		if (self.kw_build != "") and (self.validate_version() == False):
			Error("requested to run klockwork but failed on validation", kill = True)
			self.kw_build = ""

		self.kw_info_types = ["(1:Critical)", "(2:Error)", "(3:Warning)", "(4:Review)"]
		# self.kw_info_types_text = ["(1:Critical)", "(2:Error)", "(3:Warning)", "(4:Review)"]

	def get_command(self, makeCmd):
		kw_command = ""

		if self.kw_build == "kwBuild":
			kw_command = os.path.join(self.kw_app_path, "kwshell.exe") + " "

		# don't use kw when cmd is clean
		if makeCmd == "clean":
			kw_command = ""

		logger.info("kw_command :" + str(kw_command))
		return kw_command

	def validate_version(self):
		current_ver = ["Klocwork 2017.3", "Klocwork 2018.3"]
		kw_path = "C:/Klocwork/Klocwork 12.3 Command Line/bin"
		logger.info("checking path :" + kw_path)
		if not os.path.exists(kw_path):
			logger.info("path not exists :" + kw_path)
			kw_path = "C:/Klocwork/Command Line 18.3/bin"
			logger.info("checking path :" + kw_path)
			if not os.path.exists(kw_path):
				logger.error("path not exists :" + kw_path)
				logger.error("klockwork not exists in server, validation failed !")
				logger.error("please install Klocwork \"Command line Windows\" from: http://klocwork.intel.com/download/#KWClient")
				return False

		self.kw_app_path = kw_path
		self.kwcheck_cmd = os.path.join(self.kw_app_path, "kwcheck.exe")

		logger.info("using kwcheck from :" + self.kwcheck_cmd)

		if not os.path.exists(self.kwcheck_cmd):
			logger.error("kwcheck executable not found")
			self.kw_app_path = ""
			self.kwcheck_cmd = ""
			logger.error("kwcheck executable not found")
			return False

		cmd = [self.kwcheck_cmd, "--version"]
		content = run_cmd_blocking(cmd)

		# check klockwork is installed, if not return error
		is_installed = False
		for ver in current_ver:
			if ver in str(content):
				is_installed = True
				break

		if not is_installed:
			logger.error("supported klockwork versions :" + str(current_ver))
			logger.error("please install Klocwork \"Command line Windows\" from: http://klocwork.intel.com/download/#KWClient")
			return False

		# check if checker file exists
		checker_file = "KW12-3_SDL_Recommended_Checker_Set.pconf"
		if not os.path.exists(os.path.join(self.auto_build_path, checker_file)):
			logger.error("Error, missing checker file: " + checker_file + " at: " + self.auto_build_path)
			return False

		# check if override file exists
		override_file = "kw_override.h"
		if not os.path.exists(os.path.join(self.auto_build_path, checker_file)):
			logger.error(("Error, missing override file: " + override_file + " at: " + self.auto_build_path))
			return False

		return True

	def prep(self, op = None):
		if self.kw_build == "":
			return

		logger.info("Klockwork, prep, op: " + str(op) + " build_path: " + self.output_objects_path)

		os.chdir(self.output_objects_path)

		if op == "clean":
			utils.remove_folder(".kwps")
			utils.remove_folder(".kwlp")

		if not os.path.exists(".kwps"):
			self.create_project()

	def create_project(self):
		if self.kw_build == "":
			return

		os.chdir(self.output_objects_path)
		cmd = [self.kwcheck_cmd, "create", "--license-host", "klocwork02p.elic.intel.com", "--license-port", "7500"]
		run_cmd_blocking(cmd)

		cmd = [self.kwcheck_cmd, "import", os.path.join(self.auto_build_path, "KW12-3_SDL_Recommended_Checker_Set.pconf")]
		run_cmd_blocking(cmd)

		cmd = [self.kwcheck_cmd, "import", os.path.join(self.auto_build_path, "kw_override.h")]
		run_cmd_blocking(cmd)

	def run(self, build_output_source_dir_path):
		if self.kw_build == "":
			return

		# change working directory
		os.chdir(build_output_source_dir_path)  # C:\work\FW_wave600_2\output\fpga_arc_upper_mac_cpu_wave500b_ap_dummy_phy_wrx500_none\objects

		if self.kw_build == "kwBuild":
			kwcheck_run_report_file_path_name = os.path.join(build_output_source_dir_path, self.kwcheck_run_report_file_name)
			cmd = [self.kwcheck_cmd, "run", ">", kwcheck_run_report_file_path_name]
			run_cmd_blocking(cmd)

			shutil.copy2(kwcheck_run_report_file_path_name, os.path.join(self.kw_dir_report_path, self.kwcheck_run_report_file_name))

			kwcheck_list_report_file_path_name = os.path.join(build_output_source_dir_path, self.kwcheck_list_report_file_name)

			cmd = [self.kwcheck_cmd, "list", "-F", "detailed", ">", kwcheck_list_report_file_path_name]
			run_cmd_blocking(cmd)

			shutil.copy2(kwcheck_list_report_file_path_name, os.path.join(self.kw_dir_report_path,
																		self.kwcheck_list_report_file_name))

			self.extract_and_save_result_summary(kwcheck_list_report_file_path_name, self.kwcheck_run_report_summary_file_name)

			shutil.copy2(self.kwcheck_run_report_summary_file_name, os.path.join(self.kw_dir_report_path,
																				self.kwcheck_run_report_summary_file_name))

			self.validate_results()

	# post process methods
	def extract_and_save_result_summary(self, file, summary_file_name):
		logger.info("parsing stats kw from file :" + str(file))

		# search for (1:Critical), (2:Error), (3:Warning), (4:Review)

		self.kw_result_summary[self.kw_info_types[0]] = 0
		self.kw_result_summary[self.kw_info_types[1]] = 0
		self.kw_result_summary[self.kw_info_types[2]] = 0
		self.kw_result_summary[self.kw_info_types[3]] = 0

		with open(file, 'r') as f:
			for line in f:
				if self.kw_info_types[0] in line:
					self.kw_result_summary[self.kw_info_types[0]] += 1
					continue
				if self.kw_info_types[1] in line:
					self.kw_result_summary[self.kw_info_types[1]] += 1
					continue
				if self.kw_info_types[2] in line:
					self.kw_result_summary[self.kw_info_types[2]] += 1
					continue
				if self.kw_info_types[3] in line:
					self.kw_result_summary[self.kw_info_types[3]] += 1
					continue

		logger.info("----- klockwork results for build: " + self.configuration_name)
		logger.info(self.kw_result_summary)
		logger.info("----- klockwork results full info at :" + file)

		# save results

		with open(summary_file_name, 'w') as fp:
			json.dump(self.configuration_name + " " + str(self.kw_result_summary) + "\n", fp)

	def validate_results(self):

		val = self.kw_result_summary.get(self.kw_info_types[0], 0)
		if val > 100:
			Error("Kw report type exceeds its limit, " + self.kw_info_types[0] + " " + str(val), kill=True)
		val = self.kw_result_summary.get(self.kw_info_types[1], 0)
		if val > 100:
			Error("Kw report type exceeds its limit, " + self.kw_info_types[1] + " " + str(val), kill=True)
		val = self.kw_result_summary.get(self.kw_info_types[2], 0)
		if val > 100:
			Error("Kw report type exceeds its limit, " + self.kw_info_types[2] + " " + str(val), kill=True)
		val = self.kw_result_summary.get(self.kw_info_types[3], 0)
		if val > 100:
			Error("Kw report type exceeds its limit, " + self.kw_info_types[3] + " " + str(val), kill=True)


def main(development_path, project_name, configuration_name, build_option, kw_build):

	# for arg in sys.argv:
		# print(arg)
	# exit(0)
	
	# input list
	# ------------
	# build_arc.py
	# C:\ARC\MetaWare
	# C:/work/FW_2
	# lower_mac
	# fpga_arc_lower_mac_cpu_wave500b_ap_dummy_phy_wrx500_none
	# Wave500B_LowerMacCpuArc.bin
	# 163840
	
	# thisFileName 			= sys.argv[0] #build_arc.py
	# development_path 		= sys.argv[1] #C:/work/FW_2
	# project_name 			= sys.argv[2] #lower_mac
	# configuration_name 		= sys.argv[3] #fpga_arc_lower_mac_cpu_wave500b_ap_dummy_phy_wrx500_none
	# outputBinaryName 		= sys.argv[4] #Wave500B_LowerMacCpuArc.bin
	# outputBinarySizeStr 	= sys.argv[5] #163840
	# build_option			 	= sys.argv[6] #cleanAll/buildAll
	# kw_build				 	= sys.argv[7] #kwBuild

	logger.info(" ")	
	logger.info("---------- start of metaware arc build ----------")	

	start_time = time.time()
	
	# make_arc_p = make_arc(development_path, project_name, configuration_name, outputBinaryName,
	# outputBinarySizeStr, kw_build)
	make_arc_p = make_arc(development_path, project_name, configuration_name, kw_build)

	make_arc_p.build(build_option)

	make_arc_p.post_process_build()

	delta_time = time.time() - start_time
		
	m, s = divmod(delta_time, 60)
	h, m = divmod(m, 60)

	logger.info("--- FW Build Finished (took %d:%02d:%02d) ---" % (h, m, s))
	
	# sys.exit(0)


if __name__ == '__main__':
	main(sys.argv[1:])