import sys
import os
import io
import shutil
import glob
import subprocess
import time
import stat
import configparser
import filecmp
import datetime
import getpass
import utils # internal utilities file
import platform
from build_utils import logger


def remove_line_with_substring_from_multiline_string(line_with_substring_to_remove, multiline_string):
	text = '\n'.join(line for line in multiline_string.split('\n') if line_with_substring_to_remove not in line)
	return text


def run_cmd_blocking(cmd, out = False):
	logger.info(cmd)
	response = subprocess.run(cmd, capture_output = True)
	stderr = response.stderr.decode("utf8")
	stderr = remove_line_with_substring_from_multiline_string("Picked up _JAVA_OPTIONS: -Djava.net.preferIPv4Stack=true",
															stderr)
	stderr = remove_line_with_substring_from_multiline_string("error(s)", stderr)

	if stderr:
		logger.error("stderr: " + stderr)

	if response.stdout.strip():
		logger.info("stdout: " + response.stdout.decode("utf8"))
	
	if out is True:
		return response.stdout.decode("utf8")
	
	
def remove_file(filename):
	logger.info("removing file: " + filename)
	try:
		os.chmod(filename, stat.S_IWRITE)
		os.remove(filename)
	except OSError as e:
		if e.errno == 2:
			# if file not exists
			pass
		else:
			Error(e, True)
		# pass


def Error(s, kill=False):
	logger.error(str(s))
	if kill:
		exit(1)


class FlushFile(object):
	"""Write-only flushing wrapper for file-type objects."""
	def __init__(self, f):
		self.f = f

	def write(self, x):
		self.f.write(x)
		self.f.flush()
		

workSpace = os.getenv('METAWARE_ROOT')

ini_section = 'VARS'
ini_section_txt = '[VARS]'


class build_all(object):
	def __init__(self, conf_number, to_rebuild_all, to_kw_build, xml_file, from_gui):
		self.conf_number 	= int(conf_number) # -1
		self.to_rebuild_all 	= bool(int(to_rebuild_all)) # 0
		self.to_kw_build 		= bool(int(to_kw_build)) # 0
		self.xml_file 		= xml_file # ""
		self.from_gui 		= from_gui # True, False

		# assuming mt_buld.py located at: C:\..\FW_6.0.8\fw_src\AutoBuild
		self.autoBuild_path_name = os.path.abspath(os.curdir)
		fw_src_path_name = os.path.dirname(os.path.abspath(os.curdir))
		self.developmentPath = os.path.dirname(fw_src_path_name)

		logger.info("self.conf_number :"+str(self.conf_number))
		logger.info("self.to_rebuild_all :"+str(self.to_rebuild_all))
		logger.info("self.to_kw_build :"+str(self.to_kw_build))
		logger.info("self.xml_file :"+str(self.xml_file))
		logger.info("self.from_gui :"+str(self.from_gui))
		logger.info("self.developmentPath :"+str(self.developmentPath))
		logger.info(platform.system())
		
		if "Windows" in platform.system():
			self.confOutputFile = "C:/TEMP/MT_Batch_Build_Conf_Output.txt"
			self.confOutputFile_temp = "C:/TEMP/MT_Batch_Build_Conf_Output_temp.txt"
		else:
			self.confOutputFile = "MT_Batch_Build_Conf_Output.txt"
			self.confOutputFile_temp = "MT_Batch_Build_Conf_Output_temp.txt"

		self.options = {}
		self.configuration_name = ""
		self.release_name = ""
		self.fw_src_path = ""
		self.firmware_path = ""
		self.outputMain = ""
		self.release_notes = ""
		self.sw_version = ""
		self.outputDir = ""
		self.klockwork_dir = ""
		self.objectsDir = ""
		self.imagesDir 	= ""
		self.dependDir 	= ""

	def init(self):

		logger.info("workSpace: "+ workSpace)
		if not os.path.exists(workSpace):
			Error("arc metaware not installed !", kill=True)

		# add [VARS] as section to the conf file if not exists
		# with open(self.confOutputFile, 'r') as f:
		# 	first_line = f.readline()
		# with open(self.confOutputFile, 'r') as f:
		# 	f_lines = f.read()
		# if ini_section_txt not in first_line:
		# 	logger.info(ini_section_txt + " not in first line")
		# 	with open(self.confOutputFile_temp, 'w') as f2:
		# 		f2.write(ini_section_txt + '\n')
		# 		f2.write(f_lines)
		# 	remove_file(self.confOutputFile)
		# 	os.rename(self.confOutputFile_temp, self.confOutputFile)
		
		if not os.path.exists(self.confOutputFile):
			Error("file not exists: " + self.confOutputFile, kill=True)
		
		config = configparser.ConfigParser()
		# config.has_option(section, option)¶
		config.read(self.confOutputFile)

		if not config.has_section(ini_section):
			Error("has no section " + ini_section, kill=True)
			
		self.options = dict(config.items(ini_section))
		logger.info(self.options)
		
		# validate params
		validate_params = ["conf"]
		for param in validate_params:
			if param not in self.options.keys():
				Error(param + " not specified", kill=True)

		self.configuration_name = self.options['conf']
		
		logger.info("before from gui: "+str(self.from_gui) + " self.to_kw_build: " + str(self.to_kw_build) +
					" self.to_rebuild_all: " + str(self.to_rebuild_all))
		if self.from_gui:
			if 'kwbuild' in self.options.keys():
				self.to_kw_build = bool(self.options['kwbuild'])

			if 'rebuild' in self.options.keys():
				self.to_rebuild_all = bool(self.options['rebuild'])
		
		logger.info("after from gui: "+str(self.from_gui) + " self.to_kw_build: " + str(self.to_kw_build) +
					" self.to_rebuild_all: " + str(self.to_rebuild_all))

		# todo: rename self.core to soemthing common like configuration etc.
		# todo: change to general case without the fpga exception.
		try:
			core_arr = self.configuration_name.split("_")
			if "fpga" in core_arr[0].lower():
				self.core = "_".join(core_arr[2:4])
			else:
				self.core = "_".join(core_arr[1:3])
		except:
			Error("cpu not specified", kill = True)

		# if 'wave600' in self.configuration_name:
		# 	logger.info("mt_build, --- arc arc arc arc arc arc arc arc ---")
		# else:
		# 	Error("mt_build, --- wave500 not supported ---", True)

	def	prepare_directory(self):
		self.fw_src_path = os.path.join(self.developmentPath, "fw_src")
		self.firmware_path = os.path.join(self.fw_src_path, "firmware")
		self.outputMain = os.path.join(self.fw_src_path, "output")
		self.release_notes = os.path.join(self.fw_src_path, "ReleaseNotes", "release-notes.txt")
		self.sw_version = os.path.join(self.firmware_path, "System", "api", "System_FwVersion.h")
		self.outputDir = os.path.join(self.outputMain, self.configuration_name)
		self.klockwork_dir = os.path.join(self.outputMain, "klockwork")
		self.objectsDir = os.path.join(self.outputDir, "objects")
		self.imagesDir = os.path.join(self.outputDir, "images")
		self.dependDir = os.path.join(self.objectsDir, "depend")

		logger.info("mt_build, developDir is :" + str(self.developmentPath))

		os.makedirs(self.outputMain, exist_ok=True)  # succeeds even if directory exists.
		logger.info("mt_build, Compiling configuration :" + self.configuration_name)

		if self.to_rebuild_all:
			logger.info("mt_build, Deleting output directory because -rebuild was specified")
			utils.remove_folder(self.outputDir)

		os.makedirs(self.outputDir, exist_ok=True)  # succeeds even if directory exists.
		os.makedirs(self.objectsDir, exist_ok=True)  # succeeds even if directory exists.
		os.makedirs(self.dependDir, exist_ok=True)  # succeeds even if directory exists.
		os.makedirs(self.imagesDir, exist_ok=True)  # succeeds even if directory exists.

	def pre_build(self):
		prebuild_option = 'prebuild'
		if prebuild_option in self.options.keys():
			logger.info("pre_build:")
			logger.info(str(eval(self.options[prebuild_option])))

	def copy_build_config_file(self):
		objects_config_file = os.path.join(self.objectsDir, "MT_Build_Config.h")
		logger.info("objects_config_file :" + objects_config_file)
		new_config_file = os.path.join(self.fw_src_path, "sys", "common", "MT_Build_Config.h")
		logger.info("new_config_file :" + new_config_file)

		if os.path.exists(objects_config_file):
			if not filecmp.cmp(new_config_file, objects_config_file):
				logger.info("mt_build, MT_Build_Config file in the objects directory is wrong. Rewriting.")
				shutil.copy2(new_config_file, objects_config_file)
			else:
				logger.info("mt_build, MT_Build_Config.h is up to date.")
				# If the file in the objects dir is OK, overwrite the new one in the mips
				# directory, so it'll have the same timestamp for GHS as in previous build.
				shutil.copy2(objects_config_file, new_config_file)
		else:
			logger.info("mt_build, Copying MT_Build_Config file to objects directory...")
			shutil.copy2(new_config_file, objects_config_file)
				
	def copy_file_to_object_dir(self, filename_path):
		if not os.path.exists(filename_path):
			Error("filename_path not exists: " + filename_path, kill=True)
		logger.info("mt_build, Copying " + filename_path + " to " + self.objectsDir)
		try:
			shutil.copy2(filename_path, self.objectsDir)
		except:
			Error("Cannot copy file to Objects dir", kill=True)
		
	def copy_system_fw_version_template(self):
		# sw_version = os.path.join(self.firmware_path, "System", "api", "System_FwVersion.h")
		sw_version_template = os.path.join(self.firmware_path, "System", "api", "System_FwVersion.h.template")
		if not os.path.exists(self.sw_version):
			try:
				shutil.copy2(sw_version_template, self.sw_version)
			except:
				Error("Copy new System_FwVersion.h failed", kill=True)
	
	def get_branch_name(self):
		self.release_name = run_cmd_blocking("git rev-parse --abbrev-ref HEAD", out=True)
		self.release_name = self.release_name.strip() + "_"
		self.release_name = ""
		# logger.info("self.release_name :" + self.release_name)

	def update_fw_version_string(self):
		if self.from_gui is False:
			return
		release_name_date_time = self.release_name + datetime.datetime.now().strftime("%Y%m%d_%H%M%S")

		i = 0
		with open(self.sw_version, 'r') as fh:
			lines = fh.readlines()

		ver_str = "#define MT_WL_SW_VERSION "
		# if ver_str not exists override with template
		if ver_str not in lines:
			self.copy_system_fw_version_template()

		line1 = ""
		for line in lines:
			if ver_str in line:
				branch_name = line.split(ver_str)[1].strip().strip("\"")

				if "_#_" not in branch_name:
					branch_name = "_#_0"
				
				branch_name_list = branch_name.split("_#_")

				try:
					compile_count = int(branch_name_list[1])
				except:
					compile_count = 0
					
				compile_count = compile_count + 1
				
				branch_name_new = release_name_date_time + "_" + getpass.getuser() + "_#_" + str(compile_count)
				logger.info("branch_name_new :" + branch_name_new)
				line1 = ver_str + "\"" + branch_name_new + "\""
				# logger.info("new line :" + line)
				break
			i += 1

		lines[i] = line1 + "\n"

		with open(self.sw_version, 'w') as fh:
			fh.writelines(lines)
			fh.write("\n")

	def run_new_event_prep_script(self):
		logger.info("mt_build, Running new_event_prep perl script")
		logger.info(os.getcwd())
		cmd = "perl new_event_prep.pl " + self.developmentPath.replace("\\", "/") + " 1 FW"
		run_cmd_blocking(cmd)

	def arc_compile(self):

		if self.to_rebuild_all:
			build = "cleanAll"
		else:
			build = "buildAll"
		
		logger.info("build :" + build)
		
		kw_build = "false"
		if self.to_kw_build:
			kw_build = "kwBuild"
			
		import build_arc
		try:
			build_arc.main(self.developmentPath, self.core, self.configuration_name, build, kw_build)
		except Exception as e:
			# logger.info("")
			import traceback
			logger.error(traceback.format_exc())
			Error("build_arc Error: " + str(e), kill = True)

	def tlog(self):
		logger.info(self.core)
		logger.info(os.getcwd())

		# cmd = [self.kwcheck_cmd, "create", "--license-host", "klocwork02p.elic.intel.com", "--license-port", "7500"]

		if "debug" in self.configuration_name.lower():
			cmd = ["perl", os.path.join(self.autoBuild_path_name, "mt_tlog.pl"), self.configuration_name,
				self.outputDir.replace("\\", "/"), self.developmentPath.replace("\\", "/"), self.core]
			run_cmd_blocking(cmd)
			# run_cmd_blocking("perl " + os.path.join(self.autoBuild_path_name, "mt_tlog.pl") + " " +
		# self.configuration_name + " " + self.outputDir.replace("\\","/") + " " +
		# self.developmentPath.replace("\\","/") + " " + self.core)
		else:
			logger.info("skipping tlog process for :" + self.configuration_name)


def main(from_gui_arg, xml_file_arg, to_rebuild_all_arg, conf_number_arg, to_kw_build_arg):
	logger.info("---------- start of mt_build.py ----------")	

	start_time = time.time()
	
	build_all_p = build_all(conf_number_arg, to_rebuild_all_arg, to_kw_build_arg, xml_file_arg, from_gui_arg)
	build_all_p.init()
	build_all_p.prepare_directory()
	build_all_p.pre_build()
	build_all_p.copy_build_config_file()
	build_all_p.copy_system_fw_version_template()
	build_all_p.get_branch_name()
	build_all_p.update_fw_version_string()
	build_all_p.run_new_event_prep_script()
	build_all_p.arc_compile()
	build_all_p.tlog()

	delta_time = time.time() - start_time
		
	m, s = divmod(delta_time, 60)
	h, m = divmod(m, 60)

	logger.info("--- build_all Finished (took %d:%02d:%02d) ---" % (h, m, s))


if __name__ == '__main__':
	
	from_gui1 = False
	xml_file1 = ""
	to_rebuild_all1 = False
	conf_number1 = -1
	to_kw_build1 = False
	
	for arg in sys.argv:
		logger.info(arg)
		if arg == "-fromgui":
			from_gui1 = True
			continue
		if "-xml=" in arg:
			xml_file1 = arg.split('=')[1]
			continue
		if "-rebuild" in arg:
			to_rebuild_all1 = True
			continue
		if "-conf=" in arg:
			conf_number1 = int(arg.split('=')[1])
			continue
		if "-toKwBuild" in arg:
			to_kw_build1 = True
			continue

	main(from_gui1, xml_file1, to_rebuild_all1, conf_number1, to_kw_build1)
