/***********************************************************************************
 File:			Protocol_AirTimeAndDuration.c
 Module:		Protocol
 Purpose: 		802.11 Protocol related 
 Description:	Air Time and Duration calculation according to standard
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "ErrorHandler_Api.h"
#include "Protocol_AirTimeAndDuration.h"
#include "protocol.h"
#include "loggerAPI.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_802_11_PROTOCOL
#define LOG_LOCAL_FID 2


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition						
/----------------------------------------------------------------------------------*/

#ifdef ENET_INC_ARCH_WAVE600

/* The value below were calculated and provided by system */ 
const uint16 protocol11axCba256BitAirTimeByRate[] =
{
	96, 72, 60, 48, 40, 36, 32, 32
};

const uint16 protocol11axMba256BitAirTimeByRate[] =
{
	52, 42, 36, 32, 28, 28, 24, 24
};

const uint16 protocol11axMbaPerTidAirTimeByRate[] =
{
	48, 32, 24, 16, 12, 8, 6, 6 
};

#ifdef ENET_INC_ARCH_WAVE600D2

const uint16 protocol11axSuExtAckAirTimeRu106[MAX_LDPC_OPTIONS][MAX_DCM_OPTIONS] =
/*	
	ldpc/dcm	DCM0,	DCM1
*/
{
	{/*0*/		182, 	238},
	{/*1*/		198, 	242}
								
};

const uint16 protocol11axSuExtAckAirTimeRu242[MAX_LDPC_OPTIONS][MAX_DCM_OPTIONS] =
/*	
	ldpc/dcm	DCM0,	DCM1
*/
{
	/*0*/		{170, 	190},
	/*1*/		{170, 	190}
};

const uint16 protocol11axSuExtBaAirTimeRu106[MAX_LDPC_OPTIONS][MAX_DCM_OPTIONS] =
/*	
	ldpc/dcm	DCM0,	DCM1
*/
{
	
	/*0*/		{238,	330},
	/*1*/		{238, 	330}
								
};

const uint16 protocol11axSuExtBaAirTimeRu242[MAX_LDPC_OPTIONS][MAX_DCM_OPTIONS] =
/*	
	ldpc/dcm	DCM0,	DCM1
*/
{
	/*0*/		{182, 	222},
	/*1*/		{182, 	222}
};

#endif //ENET_INC_ARCH_WAVE600D2
#endif //ENET_INC_ARCH_WAVE600


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
static const uint16 protocol11agBlockAckAirTimeByRate[] =
{
	ACK_DURATION_11A((LM_PHY_11A_DATARATE_6 /2),SIZEOF_BLOCK_ACK_FRM),
	ACK_DURATION_11A((LM_PHY_11A_DATARATE_9 /2),SIZEOF_BLOCK_ACK_FRM),
	ACK_DURATION_11A((LM_PHY_11A_DATARATE_12/2),SIZEOF_BLOCK_ACK_FRM),
	ACK_DURATION_11A((LM_PHY_11A_DATARATE_18/2),SIZEOF_BLOCK_ACK_FRM),
	ACK_DURATION_11A((LM_PHY_11A_DATARATE_24/2),SIZEOF_BLOCK_ACK_FRM),
	ACK_DURATION_11A((LM_PHY_11A_DATARATE_36/2),SIZEOF_BLOCK_ACK_FRM),
	ACK_DURATION_11A((LM_PHY_11A_DATARATE_48/2),SIZEOF_BLOCK_ACK_FRM),
	ACK_DURATION_11A((LM_PHY_11A_DATARATE_54/2),SIZEOF_BLOCK_ACK_FRM) 
};


static const uint16 protocol11agAckAirTimeByRate[] =
{
	/* 11a */
	ACK_DURATION_11A((LM_PHY_11A_DATARATE_6/2),SIZEOF_ACK_FRM),
	ACK_DURATION_11A((LM_PHY_11A_DATARATE_9/2),SIZEOF_ACK_FRM),
	ACK_DURATION_11A((LM_PHY_11A_DATARATE_12/2),SIZEOF_ACK_FRM),
	ACK_DURATION_11A((LM_PHY_11A_DATARATE_18/2),SIZEOF_ACK_FRM),
	ACK_DURATION_11A((LM_PHY_11A_DATARATE_24/2),SIZEOF_ACK_FRM),
	ACK_DURATION_11A((LM_PHY_11A_DATARATE_36/2),SIZEOF_ACK_FRM),
	ACK_DURATION_11A((LM_PHY_11A_DATARATE_48/2),SIZEOF_ACK_FRM),
	ACK_DURATION_11A((LM_PHY_11A_DATARATE_54/2),SIZEOF_ACK_FRM),
};

static const uint16 protocol11bAckAirTimeByRate[] =
{
	0, /* Place Holder - There is no rate 0 in 11b*/
	/* 11b short preamble */
	ACK_DURATION_11B_SHORT(LM_PHY_11B_DATARATE_2_SHORT  ), 
	ACK_DURATION_11B_SHORT(LM_PHY_11B_DATARATE_5_5_SHORT), 
	ACK_DURATION_11B_SHORT(LM_PHY_11B_DATARATE_11_SHORT ), 
	/* 11b long preamble */
	ACK_DURATION_11B_LONG(LM_PHY_11B_DATARATE_1         ),
	ACK_DURATION_11B_LONG(LM_PHY_11B_DATARATE_2         ),
	ACK_DURATION_11B_LONG(LM_PHY_11B_DATARATE_5_5       ),
	ACK_DURATION_11B_LONG(LM_PHY_11B_DATARATE_11        ) 
};

static const uint16 protocol11agRtsAirTimeByRate[] =
{
	
	DURATION_11A(SIZEOF_RTS_FRM,(LM_PHY_11A_DATARATE_6/2)),
	DURATION_11A(SIZEOF_RTS_FRM,(LM_PHY_11A_DATARATE_9/2)),
	DURATION_11A(SIZEOF_RTS_FRM,(LM_PHY_11A_DATARATE_12/2)),
	DURATION_11A(SIZEOF_RTS_FRM,(LM_PHY_11A_DATARATE_18/2)),
	DURATION_11A(SIZEOF_RTS_FRM,(LM_PHY_11A_DATARATE_24/2)),
	DURATION_11A(SIZEOF_RTS_FRM,(LM_PHY_11A_DATARATE_36/2)),
	DURATION_11A(SIZEOF_RTS_FRM,(LM_PHY_11A_DATARATE_48/2)),
	DURATION_11A(SIZEOF_RTS_FRM,(LM_PHY_11A_DATARATE_54/2))
};

static const uint16 protocol11bRtsAirTimeByRate[] =
{
	0, /* Place Holder - There is no rate 0 in 11b*/
	/* 11b short preamble */
	DURATION_11B_SHORT(SIZEOF_RTS_FRM,LM_PHY_11B_DATARATE_2_SHORT  ), 
	DURATION_11B_SHORT(SIZEOF_RTS_FRM,LM_PHY_11B_DATARATE_5_5_SHORT), 
	DURATION_11B_SHORT(SIZEOF_RTS_FRM,LM_PHY_11B_DATARATE_11_SHORT ), 
	/* 11b long preamble */
	DURATION_11B_LONG(SIZEOF_RTS_FRM,LM_PHY_11B_DATARATE_1         ),
	DURATION_11B_LONG(SIZEOF_RTS_FRM,LM_PHY_11B_DATARATE_2         ),
	DURATION_11B_LONG(SIZEOF_RTS_FRM,LM_PHY_11B_DATARATE_5_5       ),
	DURATION_11B_LONG(SIZEOF_RTS_FRM,LM_PHY_11B_DATARATE_11        ) 
};


static const uint16 protocol11agNdpAirTimeByRate[] =
{
	DURATION_11A(SIZEOF_NDP_FRM,(LM_PHY_11A_DATARATE_6/2)),
	DURATION_11A(SIZEOF_NDP_FRM,(LM_PHY_11A_DATARATE_9/2)),
	DURATION_11A(SIZEOF_NDP_FRM,(LM_PHY_11A_DATARATE_12/2)),
	DURATION_11A(SIZEOF_NDP_FRM,(LM_PHY_11A_DATARATE_18/2)),
	DURATION_11A(SIZEOF_NDP_FRM,(LM_PHY_11A_DATARATE_24/2)),
	DURATION_11A(SIZEOF_NDP_FRM,(LM_PHY_11A_DATARATE_36/2)),
	DURATION_11A(SIZEOF_NDP_FRM,(LM_PHY_11A_DATARATE_48/2)),
	DURATION_11A(SIZEOF_NDP_FRM,(LM_PHY_11A_DATARATE_54/2))
};

static const uint16 protocol11bNdpAirTimeByRate[] =
{
	0, /* Place Holder - There is no rate 0 in 11b*/
	/* 11b short preamble */
	DURATION_11A(SIZEOF_NDP_FRM,LM_PHY_11B_DATARATE_2_SHORT  ), 
	DURATION_11A(SIZEOF_NDP_FRM,LM_PHY_11B_DATARATE_5_5_SHORT), 
	DURATION_11A(SIZEOF_NDP_FRM,LM_PHY_11B_DATARATE_11_SHORT ), 
	DURATION_11A(SIZEOF_NDP_FRM,LM_PHY_11B_DATARATE_1         ),
	DURATION_11A(SIZEOF_NDP_FRM,LM_PHY_11B_DATARATE_2         ),
	DURATION_11A(SIZEOF_NDP_FRM,LM_PHY_11B_DATARATE_5_5       ),
	DURATION_11A(SIZEOF_NDP_FRM,LM_PHY_11B_DATARATE_11        ) 
};

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

Protocol_GetAckCts11bAirTime

Description:
------------

Input:
-----
	rate
	
Returns:
--------
	The Ack/Cts duration in usec when transmitting in 11b
	
**********************************************************************************/
uint16 Protocol_GetAckCts11bAirTime(uint8 rate)
{
	uint16 retVal;

	DEBUG_ASSERT(rate < (LM_PHY_11B_MAX_RATES +1));

	retVal = protocol11bAckAirTimeByRate[rate];

	return retVal;	
}

/**********************************************************************************

Protocol_GetAckCts11agAirTime

Description:
------------

Input:
-----
	rate
	
	
Returns:
--------
	The Ack/Cts duration in usec when transmitting in 11A/G
	
**********************************************************************************/
uint16 Protocol_GetAckCts11agAirTime(uint8 rate)
{
	
	uint16 retVal;
	
	DEBUG_ASSERT(rate < LM_PHY_11A_MAX_RATES);

	retVal = protocol11agAckAirTimeByRate[rate];

	return retVal;
		
}

/**********************************************************************************

Protocol_GetRtsCfEnd11bAirTime

Description:
------------

Input:
-----
	rate
	
Returns:
--------
	The RTS duration in usec when transmitting in 11B
	
**********************************************************************************/
uint16 Protocol_GetRtsCfEnd11bAirTime(uint8 rate)
{
	uint16 retVal;

	DEBUG_ASSERT(rate < (LM_PHY_11B_MAX_RATES +1));

	retVal = protocol11bRtsAirTimeByRate[rate];

	return retVal;
}



/**********************************************************************************

Protocol_GetRtsCfEnd11agAirTime

Description:
------------

Input:
-----
	rate
	
Returns:
--------
	The Ack/Cts duration in usec when transmitting in 11A/G
	
**********************************************************************************/
uint16 Protocol_GetRtsCfEnd11agAirTime(uint8 rate)
{
	uint16 retVal;

	DEBUG_ASSERT(rate < LM_PHY_11A_MAX_RATES);

	retVal = protocol11agRtsAirTimeByRate[rate];

	return retVal;
}


/**********************************************************************************

Protocol_GetBaAirTime

Description:
------------

Input:
-----
	rate
	
Returns:
--------
	The BA duration in usec when transmitting in 11A/G (BA must be transmitted in A/G)
	
**********************************************************************************/
uint16 Protocol_GetBaAirTime(uint8 rate)
{
	uint16 retVal;

	DEBUG_ASSERT(rate < LM_PHY_11A_MAX_RATES);
	
	retVal= protocol11agBlockAckAirTimeByRate[rate];

	return retVal;
}

/**********************************************************************************

Protocol_GetNdp11bAirTime

Description:
------------
	
Input:
-----
	rate -  NDP rate
	
Returns:
--------
	The  NDP air time in usec when transmitting in 11B
	
**********************************************************************************/
uint16 Protocol_GetNdp11bAirTime(uint8 rate)
{
	uint16 retVal;

	DEBUG_ASSERT(rate < (LM_PHY_11B_MAX_RATES +1));

	retVal= protocol11bNdpAirTimeByRate[rate];

	return retVal;
}

/**********************************************************************************

Protocol_GetNdp11agAirTime

Description:
------------
	
Input:
-----
	rate - the NDP rate
	
Returns:
--------
	The Qos NDP air time in usec when transmitting in 11A/G
	
**********************************************************************************/
uint16 Protocol_GetNdp11agAirTime(uint8 rate)
{
	
	uint16 retVal;

	DEBUG_ASSERT(rate < LM_PHY_11A_MAX_RATES);

	retVal= protocol11agNdpAirTimeByRate[rate];

	return retVal;
}


