/***********************************************************************************
 File:			AgerEmulator.h
 Module:		Ager 
 Purpose: 	    Emulate the HW Ager - discard old PDs
 Description:	This file contains all the definitions and structures of the Ager 
 				emulator module which is responsible to discard old PDs and discard
 				PDs from specific priorities (on single check mode)
************************************************************************************/
#ifndef AGER_EMULATOR_H_
#define AGER_EMULATOR_H_

//---------------------------------------------------------------------------------
//						Includes						
//---------------------------------------------------------------------------------
#include "System_GlobalDefinitions.h"
#include "System_Configuration.h"
#include "OSAL_Kmsg.h"
#include "PacketDescriptor.h"
#include "protocol.h"
#include "Ager_Api.h"

#ifdef USE_AGER_EMULATOR

//---------------------------------------------------------------------------------
//						Defines						
//--------------------------------------------------------------------------------- 
#define AGER_EMULATOR_STATION_AND_VAP_BITMAP_SIZE ((CONVERT_BIT_INDEX_TO_WORDS(HW_NUM_OF_STATIONS + HW_NUM_OF_VAPS - 1)) + 1)
#define AGER_EMULATOR_FIRST_PART_OF_BITMAP 0
#define AGER_EMULATOR_SECOND_PART_OF_BITMAP 1
#define AGER_EMULATOR_FIRST_TID_IN_AC 0
#define AGER_EMULATOR_SECOND_TID_IN_AC 1
#define AGER_EMULATOR_PD_TYPE_SEND_TO_UMAC 0
#define AGER_EMULATOR_PD_TYPE_SEND_TO_LIBERATOR 1
#define AGER_EMULATOR_MAX_SIZE_OF_WINDOW 64 //TBD should behere?
#define AGER_EMULATOR_WINDOW_SIZE_MASK 0x3F //TBD should behere?
#define AGER_EMULATOR_BITMAP_MASK 0x1F //TBD should behere?
#define AGER_EMULATOR_WORD_SIZE_TO_ADD_TO_BIT_INDEX 32
#define AGER_EMULATOR_REORDERING_BITMAP_SIZE ((CONVERT_BIT_INDEX_TO_WORDS(AGER_EMULATOR_MAX_SIZE_OF_WINDOW - 1)) + 1)


//---------------------------------------------------------------------------------
//						Macros						
//---------------------------------------------------------------------------------
#define AGER_EMULATOR_CONVERT_10_MILISECONDS_TO_MICROSECONDS(x) ((x) * 10000)
#define AGER_EMULATOR_MASK_LOWER_PART_OF_WORD(index) (0xFFFFFFFF << (index))
#define AGER_EMULATOR_MASK_UPPER_PART_OF_WORD(index) ((0x1 << index) - 1)

//---------------------------------------------------------------------------------
//						Data Type Definition					
//---------------------------------------------------------------------------------
typedef enum
{
	AGER_EMULATOR_STATE_IDLE = 0,
	AGER_EMULATOR_STATE_PERIODIC_MODE,
	AGER_EMULATOR_STATE_SINGLE_MODE,
	AGER_EMULATOR_NUM_OF_STATES,
	AGER_EMULATOR_MAX_NUM_OF_STATES = 0xFF,	
} AgerEmulatorState_e;

typedef struct AgerEmulatorPeriodicCheckMessageParameters
{
	uint8 acTtlCriteria[ACCESS_CATEGORY_NUM];
	uint16 minimumPdsToStartCheckForAc[ACCESS_CATEGORY_NUM];

	uint8 agerPeriod;
	uint8 reserved[3];
} AgerEmulatorPeriodicCheckMessageParameters_t;

typedef struct AgerEmulatorSingleCheckMessageParameters
{
	uint8 acTtlCriteria[ACCESS_CATEGORY_NUM];
	uint16 minimumPdsToStartCheckForAc[ACCESS_CATEGORY_NUM];

    uint8 forceLock;
#ifdef ENET_INC_ARCH_WAVE600
	uint8 reserved;
	uint16 stationOrVapIndexToStart;
	uint32 maximumPdsToDiscard;
#else
	uint8 stationOrVapIndexToStart;
	uint16 maximumPdsToDiscard;
	uint16 reserved;
#endif //ENET_INC_ARCH_WAVE600
	uint8 isStartingQueueVap;
	uint8 isMaximumPdsToDiscardEnable; 
	
} AgerEmulatorSingleCheckMessageParameters_t;


typedef struct AgerEmulatorParameters_s
{     
	TxPd_t *reorderingPacketDescriptorsArray[AGER_EMULATOR_MAX_SIZE_OF_WINDOW];
	uint32 reorderingPacketDescriptorsBitmap[AGER_EMULATOR_REORDERING_BITMAP_SIZE];
	uint32 stationAndVapConnectedBitmap[AGER_EMULATOR_STATION_AND_VAP_BITMAP_SIZE];
	uint32 temporaryStationAndVapConnectedBitmap[AGER_EMULATOR_STATION_AND_VAP_BITMAP_SIZE];
	uint32 stationAndVapStatusBitmap[AGER_EMULATOR_STATION_AND_VAP_BITMAP_SIZE];
	
	uint16 pdLimitThresholds[ACCESS_CATEGORY_NUM];

	uint8 ttlCriteria[ACCESS_CATEGORY_NUM];
	
	uint32 numberOfDiscardedPdsHandledToLiberator;
	uint32 numberOfDiscardedPdsHandledToUmac;
	uint32 lastTimePeriodicCheckWasActivated;
	uint32 numberOfQueuesSkippedDueToLockFailure;
	
	uint16 numberOfStopsRequested;
	uint16 numberOfStopsProcessed;
	
	uint16 agerEmulatorPeriod;
	uint16 numberOfDiscardedPds;

	uint16 maximumPdsDiscardedPerClient;
	uint16 maximumPdsToDiscardLimit;

    uint8 currentTid;
    uint8 clientWithMaximumPdsDiscardedStationOrVapIndex;
	
    uint8 clientWithMaximumPdsDiscardedTidIndex;
    uint8 isLastQueueCheckedVap;
    uint8 lastQueueCheckedStationOrVapIndex;
	uint8 lastQueueCheckedTidIndex;

	uint8 isMaximumPdsToDiscardLimitEnabled;
	AgerEmulatorState_e state;

	uint8 currentAc;
	uint8 tidInAc; /* 0 or 1 */
	uint8 currentPartOfBitmap;
	uint8 startingLocationInBitmap;

	uint8 currentLocationInBitmap;
	uint8 startingLocationInWords;	
    uint8 isForceLock;
	uint8 isClientWithMaximumPdsDiscardedVap;
	
	uint8 lastLocationInBitmap;
	uint8 currentStationOrVapIndex;
	uint8 isSingleCheckDone;
	uint8 isStartedSingleCheck;

	StaId stationOrVapIndexToStart;
	uint8 bitIndexInWord;
	uint8 isCurrentQueueVap;
#ifndef ENET_INC_ARCH_WAVE600
	uint8 reserved;
#endif //ENET_INC_ARCH_WAVE600
}AgerEmulatorParameters_t;



//---------------------------------------------------------------------------------
//						Public Function Declaration				
//---------------------------------------------------------------------------------
void AgerEmulator_TaskEntry(K_MSG *agerEmulatorMessage);
void AgerEmulator_ConfigureAllAndActivatePeriodicCheck(AgerPeriodicCheckParameters_t  *agerCheckParameters);
void AgerEmulator_ReconfigurePartialPeriodicCheck(AgerPeriodicCheckParameters_t  *agerCheckParameters);
void AgerEmulator_StopPeriodicCheck(void);
void AgerEmulator_StartSingleCheck(AgerSingleCheckParameters_t  *agerCheckParameters, bool restrictedAc);
void AgerEmulator_GetLastCheckParameters (AgerLastCheckParametes_t *lastCheckParameters); 
void AgerEmulator_GetLastCheckCounters (AgerLastCheckCounters_t *lastCheckCounters); 
void AgerEmulator_IncreasePriority(void);
void AgerEmulator_Initialize(void);

#endif //#ifdef USE_AGER_EMULATOR

#endif /* AGER_EMULATOR_H_ */

