/***********************************************************************************
 File:			AgerEmulator.c
 Module:		Ager 
 Purpose: 	    Emulate the HW Ager - discard old PDs
 Description:   This file is the implementation of the Ager emulator module which is 
 			    responsible to discard old PDs and discard PDs from specific 
 			    priorities (on single check mode)
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"	
#include "AgerEmulator.h"
#include "queue_utility.h"
#include "loggerAPI.h"
#include "ErrorHandler_Api.h"
#include "stringLibApi.h"
#include "BSSmanager_API.h"
#include "Pac_Api.h"
#include "Locker_Api.h"
#include "HostInterfaceAcc_Api.h"
#include "StaDatabase_Api.h"
#include "Utils_Api.h"
#include "tx_application_Api.h"
#include "tx_application_priorities.h"
#include "Statistics_Api.h"
#include "HwQManager_API.h"
#include "PacketDescriptor.h"
#include "ShramPacketDescriptors.h"
#include "StaDatabase_Api.h"
#include "PsManager_Api.h"
#include "StatisticsManager_api.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_AGER_EMULATOR
#define LOG_LOCAL_FID 1



#ifdef USE_AGER_EMULATOR


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/
#define	AGER_EMULATOR_PERIODIC_MAX_DATA_PDS_TO_DISCARD	0x3FFF

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
/* Messages functions */
static void agerEmulatorHandleConfigureAllAndActivatePeriodicMessage(K_MSG* agerEmulatorMessage);
static void agerEmulatorHandleConfigurePartiallyAndActivatePeriodicMessage(K_MSG* agerEmulatorMessage);
static void agerEmulatorStopPeriodicCheck(K_MSG* agerEmulatorMessage);
static void agerEmulatorStartSingleCheck(K_MSG* agerEmulatorMessage);
static void agerEmulatorAddStation(K_MSG* agerEmulatorMessage);
static void agerEmulatorRemoveStation(K_MSG* agerEmulatorMessage);
static void agerEmulatorAddVap(K_MSG* agerEmulatorMessage);
static void agerEmulatorRemoveVap(K_MSG* agerEmulatorMessage);
static void agerEmulatorHandlePeriodicTimer(K_MSG* agerEmulatorMessage);
static void agerEmulatorHandleLockConfimration(K_MSG* agerEmulatorMessage);
#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)	
static void agerEmulatorHandleTid7LockConfimration(K_MSG* agerEmulatorMessage);
#endif



/* General functions */
static void agerEmulatorConfigurePartialAndActivatePeriodicCheck(AgerEmulatorPeriodicCheckMessageParameters_t *periodicCheckMessageParameters);
static void agerEmulatorSendConfirmToStationManager(StaId stationIndex);
static void agerEmulatorPerformSingleCheckInForceMode(uint8 updateParameters);
static void agerEmulatorPerformSingleCheckInTryMode(void);

#ifdef ENET_INC_ARCH_WAVE600
static uint16 agerEmulatorProcessQueue(StaId stationOrVapIndex, uint8 tid, uint8 isVap, uint32 numberOfDataPdsOfStationAc);
#else
static uint16 agerEmulatorProcessQueue(StaId stationOrVapIndex, uint8 tid, uint8 isVap, uint16 maxDataPdsToDiscard);
#endif //ENET_INC_ARCH_WAVE600




/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
static AgerEmulatorParameters_t AgerEmulatorParameters;

static const FunctionEntry_t afpTaskTable[TASK_AGER_EMULATOR_END - TASK_AGER_EMULATOR_START]=
{
	{agerEmulatorHandleConfigureAllAndActivatePeriodicMessage,	       DOUBLE_CHECK_MSG_TYPE(AGER_EMULATOR_CONFIGURE_ALL_AND_ACTIVATE_PERIODIC_CHECK)}, 
	{agerEmulatorHandleConfigurePartiallyAndActivatePeriodicMessage,   DOUBLE_CHECK_MSG_TYPE(AGER_EMULATOR_CONFIGURE_PARTIALLY_AND_ACTIVATE_PERIODIC_CHECK)}, 
	{agerEmulatorStopPeriodicCheck,					                   DOUBLE_CHECK_MSG_TYPE(AGER_EMULATOR_STOP_PERIODIC_CHECK)}, 
	{agerEmulatorStartSingleCheck,					                   DOUBLE_CHECK_MSG_TYPE(AGER_EMULATOR_START_SINGLE_CHECK)},
	{agerEmulatorAddStation,					           	           DOUBLE_CHECK_MSG_TYPE(AGER_EMULATOR_ADD_STATION)},
	{agerEmulatorRemoveStation,							               DOUBLE_CHECK_MSG_TYPE(AGER_EMULATOR_REMOVE_STATION)},
	{agerEmulatorAddVap,					           	               DOUBLE_CHECK_MSG_TYPE(AGER_EMULATOR_ADD_VAP)},
	{agerEmulatorRemoveVap,							                   DOUBLE_CHECK_MSG_TYPE(AGER_EMULATOR_REMOVE_VAP)},
	{agerEmulatorHandlePeriodicTimer,						           DOUBLE_CHECK_MSG_TYPE(AGER_EMULATOR_PERIODIC_CHECK_TIMER)},
	{agerEmulatorHandleLockConfimration, 					           DOUBLE_CHECK_MSG_TYPE(AGER_EMULATOR_LOCK_CONFIRMATION)},
#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)	
	{agerEmulatorHandleTid7LockConfimration,						   DOUBLE_CHECK_MSG_TYPE(AGER_EMULATOR_TID7_LOCK_CONFIRMATION)},
#endif
};


static const uint8 AgerEmulatorNextAcTable[ACCESS_CATEGORY_NUM] =
{
 	ACCESS_CATEGORY_VI             , /* ACCESS_CATEGORY_BE */
	ACCESS_CATEGORY_BE             , /* ACCESS_CATEGORY_BK */
	ACCESS_CATEGORY_VO             , /* ACCESS_CATEGORY_VI */
	ACCESS_CATEGORY_NUM         , /* ACCESS_CATEGORY_VO */
};		

static const uint8 AgerEmulatorQueuesPerAcTable[ACCESS_CATEGORY_NUM][NUMBER_OF_TIDS_PER_AC] =
{
 	{IEEE802_1D_BE_1, IEEE802_1D_BE_2}, /* ACCESS_CATEGORY_BE */
	{IEEE802_1D_BK_1, IEEE802_1D_BK_2}, /* ACCESS_CATEGORY_BK */
	{IEEE802_1D_VI_1, IEEE802_1D_VI_2}, /* ACCESS_CATEGORY_VI */
	{IEEE802_1D_VO_1, IEEE802_1D_VO_2}, /* ACCESS_CATEGORY_VO */
};	

#if !defined (ENET_INC_ARCH_WAVE600)
static const uint8 AgerEmulatorManagementPdTypeTable[PD_TYPE_LAST] =
{
 	FALSE, /* PD_TYPE_DATA */
	FALSE, /* PD_TYPE_DATA_RESERVED */
	TRUE,  /* PD_TYPE_NDP */ /*Handle NDP as management as no reordering is possible for it*/
	TRUE,  /* PD_TYPE_ADDBA_REQ */
    FALSE, /* PD_TYPE_BAR */
	TRUE,  /* PD_TYPE_CH_SWITCH_ANN */
	TRUE,  /* PD_TYPE_MANAGEMENT_UNENC_RESERVED */
	TRUE,  /* PD_TYPE_MANAGEMENT_ENC_RESERVED */
	FALSE, /* PD_TYPE_CONTROL_UNENC_RESERVED */
	TRUE,  /* PD_TYPE_MANAGEMENT_UNENC */
	TRUE,  /* PD_TYPE_MANAGEMENT_UNENC_FRAG */
	TRUE,  /* PD_TYPE_MANAGEMENT_ENC */
	FALSE, /* PD_TYPE_CONTROL_UNENC */
	TRUE,  /* PD_TYPE_PROBE_RES */
	TRUE,  /* PD_TYPE_MANAGEMENT_ENC_BIP */
	FALSE, /* PD_TYPE_BEACON */
};	
#endif //defined (ENET_INC_ARCH_WAVE600)

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/
/////////////DEBUG START////////////////////////////////////////////////////
//#define AAA_AGER_EMULATOR_DEBUG
#if defined (AAA_AGER_EMULATOR_DEBUG)
#define NUM_OF_WMM_STA_DEBUG 	(2)
#define NUMBER_OF_TIDS_PER_STA 	(IEEE802_1D_NA)
uint32 AgerEmulatorDebugStaTid[NUM_OF_WMM_STA_DEBUG][NUMBER_OF_TIDS_PER_STA] =
	{
		{0, 0, 0, 0, 0, 0, 0, 0,}, /* WMM_STA_0 */
		{0, 0, 0, 0, 0, 0, 0, 0,}, /* WMM_STA_1 */
	};	
uint32 AgerEmulatorDebugStaAc[NUM_OF_WMM_STA_DEBUG][ACCESS_CATEGORY_NUM] =
	{
		{0, 0, 0, 0,}, /* WMM_STA_0 */
		{0, 0, 0, 0,}, /* WMM_STA_1 */
	};	
#endif
/////////////DEBUG END////////////////////////////////////////////////////

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

agerEmulatorHandleConfigureAllAndActivatePeriodicMessage 


Description:
------------
Handles the event that the Qos sent a message to configure all periodic check 
parameters and to activate it


Input: 
-----	
agerEmulatorMessage - the message that contains all the parameters 
	
		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
static void agerEmulatorHandleConfigureAllAndActivatePeriodicMessage(K_MSG* agerEmulatorMessage)
{
	AgerEmulatorPeriodicCheckMessageParameters_t *periodicCheckMessageParameters = NULL;

	DEBUG_ASSERT(AgerEmulatorParameters.state != AGER_EMULATOR_STATE_SINGLE_MODE);
	AgerEmulatorParameters.numberOfStopsProcessed++;
	//("agerEmulatorHandleConfigureAllAndActivatePeriodicMessage numberOfStopsProcessed = %d , numberOfStopsRequested = %d", AgerEmulatorParameters.numberOfStopsProcessed, AgerEmulatorParameters.numberOfStopsRequested);
	if(AgerEmulatorParameters.numberOfStopsProcessed == AgerEmulatorParameters.numberOfStopsRequested)
	{
		periodicCheckMessageParameters = ((AgerEmulatorPeriodicCheckMessageParameters_t *)agerEmulatorMessage->abData);
		memcpy(AgerEmulatorParameters.pdLimitThresholds, periodicCheckMessageParameters->minimumPdsToStartCheckForAc,sizeof(AgerEmulatorParameters.pdLimitThresholds)); 
        agerEmulatorConfigurePartialAndActivatePeriodicCheck(periodicCheckMessageParameters);
	}
}

/**********************************************************************************

agerEmulatorHandleConfigurePartiallyAndActivatePeriodicMessage 


Description:
------------
Handles the event that the Qos sent a message to configure part of the  periodic 
check parameters and to activate it


Input: 
-----	
agerEmulatorMessage - the message that contains all the parameters
	
		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
static void agerEmulatorHandleConfigurePartiallyAndActivatePeriodicMessage(K_MSG* agerEmulatorMessage)
{
	AgerEmulatorPeriodicCheckMessageParameters_t *periodicCheckMessageParameters = NULL;

	DEBUG_ASSERT(AgerEmulatorParameters.state != AGER_EMULATOR_STATE_SINGLE_MODE);
	AgerEmulatorParameters.numberOfStopsProcessed++;
	//("agerEmulatorHandleConfigurePartiallyAndActivatePeriodicMessage numberOfStopsProcessed = %d , numberOfStopsRequested = %d", AgerEmulatorParameters.numberOfStopsProcessed, AgerEmulatorParameters.numberOfStopsRequested);
	if(AgerEmulatorParameters.numberOfStopsProcessed == AgerEmulatorParameters.numberOfStopsRequested)
	{
		periodicCheckMessageParameters = ((AgerEmulatorPeriodicCheckMessageParameters_t *)agerEmulatorMessage->abData);
		agerEmulatorConfigurePartialAndActivatePeriodicCheck(periodicCheckMessageParameters);
	}
}

/**********************************************************************************

agerEmulatorStopPeriodicCheck 


Description:
------------
Handles the event that the Qos sent a message to stop the periodic check


Input: 
-----	
agerEmulatorMessage 
	
		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
static void agerEmulatorStopPeriodicCheck(K_MSG* agerEmulatorMessage)
{
	DEBUG_ASSERT(AgerEmulatorParameters.state != AGER_EMULATOR_STATE_SINGLE_MODE);
	AgerEmulatorParameters.state = AGER_EMULATOR_STATE_IDLE;
	AgerEmulatorParameters.numberOfStopsProcessed++;
	//("agerEmulatorStopPeriodicCheck numberOfStopsProcessed = %d ", AgerEmulatorParameters.numberOfStopsProcessed);
	OSAL_RESET_TIMER_EXPLICIT(AGER_EMULATOR_PERIODIC_CHECK_TIMER, TASK_AGER_EMULATOR);
}

/**********************************************************************************

agerEmulatorStartSingleCheck 


Description:
------------
Handles the event that the Qos sent a message to start the single check

Input: 
-----	
agerEmulatorMessage - the message that contains all the parameters 
	
		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
static void agerEmulatorStartSingleCheck(K_MSG* agerEmulatorMessage)
{
	AgerEmulatorSingleCheckMessageParameters_t *singleCheckMessageParameters = NULL;

	ASSERT(AgerEmulatorParameters.state != AGER_EMULATOR_STATE_SINGLE_MODE);
	AgerEmulatorParameters.numberOfStopsProcessed++;
	DEBUG_ASSERT(AgerEmulatorParameters.numberOfStopsProcessed == AgerEmulatorParameters.numberOfStopsRequested);

	AgerEmulatorParameters.isSingleCheckDone = FALSE;
	AgerEmulatorParameters.isStartedSingleCheck = TRUE;
	AgerEmulatorParameters.state = AGER_EMULATOR_STATE_SINGLE_MODE;
	OSAL_RESET_TIMER_EXPLICIT(AGER_EMULATOR_PERIODIC_CHECK_TIMER, TASK_AGER_EMULATOR);
    //("agerEmulatorStartSingleCheck ");
	//(0, 0, AgerEmulatorSingleCheckMessageParameters_t, singleCheckMessageParameters);
	
	singleCheckMessageParameters = ((AgerEmulatorSingleCheckMessageParameters_t *)agerEmulatorMessage->abData);
   
    /* Clear counters and status first, These counter are clreared only in single check. In periodic check they are
       accumulated */ 
	memset32(AgerEmulatorParameters.stationAndVapStatusBitmap, 0, AGER_EMULATOR_STATION_AND_VAP_BITMAP_SIZE);
	AgerEmulatorParameters.numberOfDiscardedPdsHandledToLiberator = 0;
	AgerEmulatorParameters.numberOfDiscardedPdsHandledToUmac = 0;
	AgerEmulatorParameters.numberOfQueuesSkippedDueToLockFailure = 0;
	AgerEmulatorParameters.clientWithMaximumPdsDiscardedStationOrVapIndex = 0;
	AgerEmulatorParameters.clientWithMaximumPdsDiscardedTidIndex = 0;
	AgerEmulatorParameters.maximumPdsDiscardedPerClient = 0;
	AgerEmulatorParameters.isClientWithMaximumPdsDiscardedVap = FALSE;
	AgerEmulatorParameters.isLastQueueCheckedVap = FALSE;

	/* Copy parameters */
	memcpy(AgerEmulatorParameters.ttlCriteria, singleCheckMessageParameters->acTtlCriteria, sizeof(AgerEmulatorParameters.ttlCriteria));
	memcpy(AgerEmulatorParameters.pdLimitThresholds, singleCheckMessageParameters->minimumPdsToStartCheckForAc,sizeof(AgerEmulatorParameters.pdLimitThresholds)); 
	
	AgerEmulatorParameters.isForceLock = singleCheckMessageParameters->forceLock;
	AgerEmulatorParameters.stationOrVapIndexToStart = singleCheckMessageParameters->stationOrVapIndexToStart;
	if(singleCheckMessageParameters->isStartingQueueVap)
	{
		AgerEmulatorParameters.stationOrVapIndexToStart += HW_NUM_OF_STATIONS;
		
	}

    /* It is assumed that if limit is enabled the maximum is != 0 , becasue in force lock there will be at least one 
	   trial to lock */
	AgerEmulatorParameters.isMaximumPdsToDiscardLimitEnabled = singleCheckMessageParameters->isMaximumPdsToDiscardEnable;
	AgerEmulatorParameters.maximumPdsToDiscardLimit = singleCheckMessageParameters->maximumPdsToDiscard;

//	ILOG0_D("[agerEmulatorStartSingleCheck], AgerEmulatorParameters.isForceLock = %d", AgerEmulatorParameters.isForceLock);
	if(AgerEmulatorParameters.isForceLock)
	{
		/* In case of force lock - the scan is paused for each lock request - the thread probably will exit from this context */
		AgerEmulatorParameters.currentAc = ACCESS_CATEGORY_BK;
		AgerEmulatorParameters.numberOfDiscardedPds = 0;
		AgerEmulatorParameters.currentPartOfBitmap = AGER_EMULATOR_FIRST_PART_OF_BITMAP;
		AgerEmulatorParameters.tidInAc = AGER_EMULATOR_FIRST_TID_IN_AC; 
   	    AgerEmulatorParameters.startingLocationInBitmap = CONVERT_BIT_INDEX_TO_WORDS(AgerEmulatorParameters.stationOrVapIndexToStart);
	    AgerEmulatorParameters.bitIndexInWord = AgerEmulatorParameters.stationOrVapIndexToStart & AGER_EMULATOR_BITMAP_MASK; 
		agerEmulatorPerformSingleCheckInForceMode(TRUE);
	}
	else
	{
		agerEmulatorPerformSingleCheckInTryMode();
		AgerEmulatorParameters.state = AGER_EMULATOR_STATE_IDLE;
		AgerEmulatorParameters.isSingleCheckDone = TRUE;
		/* The single check is done - return to normal priority */

		TxApp_ChangeThreadPriority(TASK_AGER_EMULATOR, TX_APP_PRIORITY_AGER_EMULATOR);

		OSAL_SEND_NO_DATA_MESSAGE(QOS_SINGLE_CHECK_DONE, TASK_QOS, VAP_ID_DO_NOT_CARE);	
	}
	
}

/**********************************************************************************

agerEmulatorAddStation 


Description:
------------
Handles the event of adding station


Input: 
-----	
agerEmulatorMessage - the pointer to the message that contains the station id
	
		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
static void agerEmulatorAddStation(K_MSG* agerEmulatorMessage)
{
	StaId stationIndex = 0;
	BssManagerStaManagerReq_t *staManagerReq = NULL;
	K_MSG *originalAddStationMessage = NULL;
	UMI_STA_ADD *addStationMessageParameters = NULL;

	staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(agerEmulatorMessage);
	originalAddStationMessage = staManagerReq->psMsg;
	addStationMessageParameters = (UMI_STA_ADD *)pK_MSG_DATA(originalAddStationMessage);

	stationIndex = addStationMessageParameters->u16SID;  
    Utils_SetBitInBitmap(AgerEmulatorParameters.stationAndVapConnectedBitmap, stationIndex);
	
	agerEmulatorSendConfirmToStationManager(stationIndex);
}

/**********************************************************************************

agerEmulatorRemoveStation 


Description:
------------
Handles the event of removing station


Input: 
-----	
agerEmulatorMessage - the pointer to the message that contains the station id
	
		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
static void agerEmulatorRemoveStation(K_MSG* agerEmulatorMessage)
{
	StaId stationIndex = 0;
	BssManagerStaManagerReq_t *staManagerReq = NULL;
	K_MSG *originalRemoveStationMessage = NULL;
	UMI_STOP_TRAFFIC *stopTrafficMessageParameters = NULL;

	staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(agerEmulatorMessage);
	originalRemoveStationMessage = staManagerReq->psMsg;
	stopTrafficMessageParameters = (UMI_STOP_TRAFFIC *)pK_MSG_DATA(originalRemoveStationMessage);
  
	stationIndex = stopTrafficMessageParameters->u16SID;
	Utils_ZeroBitInBitmap(AgerEmulatorParameters.stationAndVapConnectedBitmap, stationIndex);
	agerEmulatorSendConfirmToStationManager(stationIndex);

}

/**********************************************************************************

agerEmulatorAddVap 


Description:
------------
Handles the event of adding VAP


Input: 
-----	
agerEmulatorMessage - the pointer to the message that contains the VAP id
	
		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
static void agerEmulatorAddVap(K_MSG* agerEmulatorMessage)
{
	uint8 vapIndex = 0;
	UMI_SET_WMM_PARAMETERS *postVapActivationMessage = NULL;
	K_MSG *vapMessage = NULL;
	BSS_MANAGER_CONFIRM_EVENT *confirmEvent = NULL; 

	vapMessage = (K_MSG *) (*((uint32 *)pK_MSG_DATA(agerEmulatorMessage)));
	postVapActivationMessage = (UMI_SET_WMM_PARAMETERS *) pK_MSG_DATA(vapMessage);
 
	vapIndex = postVapActivationMessage->vapId;

	Utils_SetBitInBitmap(AgerEmulatorParameters.stationAndVapConnectedBitmap, (vapIndex + HW_NUM_OF_STATIONS));

	confirmEvent = (BSS_MANAGER_CONFIRM_EVENT*)(&(agerEmulatorMessage->abData));
	confirmEvent->vapId = vapIndex;
	confirmEvent->eventIndex = VAP_MANAGER_POST_VAP_ACTIVATION;
	confirmEvent->clientId = BSS_MANAGER_VAP_MANAGER_AGER_CLIENT;
	OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, agerEmulatorMessage, vapIndex);
}


/**********************************************************************************

 agerEmulatorRemoveVap 


Description:
------------
Handles the event of removing VAP


Input: 
-----	
agerEmulatorMessage - the pointer to the message that contains the VAP id
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void  agerEmulatorRemoveVap(K_MSG* agerEmulatorMessage)
{
	uint8 vapIndex = 0;
	UMI_STOP_VAP_TRAFFIC *stopTrafficMessage = NULL;
	BSS_MANAGER_CONFIRM_EVENT* confirmEvent = NULL;
	K_MSG *vapMessage = NULL;

	vapMessage = (K_MSG *) (*((uint32 *)pK_MSG_DATA(agerEmulatorMessage)));
	stopTrafficMessage = (UMI_STOP_VAP_TRAFFIC *) pK_MSG_DATA(vapMessage);
	vapIndex = stopTrafficMessage->vapId;

	Utils_ZeroBitInBitmap(AgerEmulatorParameters.stationAndVapConnectedBitmap, (vapIndex + HW_NUM_OF_STATIONS));
	
	confirmEvent = (BSS_MANAGER_CONFIRM_EVENT *)(&(agerEmulatorMessage->abData));
	confirmEvent->vapId = vapIndex;
	confirmEvent->eventIndex = VAP_MANAGER_STOP_TRAFFIC;
	confirmEvent->clientId = BSS_MANAGER_VAP_MANAGER_AGER_CLIENT;
	OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, agerEmulatorMessage, vapIndex);	
	
}

/**********************************************************************************

agerEmulatorHandlePeriodicTimer 


Description:
------------
This function performs periodic scan if needed


Input: 
-----	 
agerEmulatorMessage

		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
static void agerEmulatorHandlePeriodicTimer(K_MSG* agerEmulatorMessage)
{
	uint8 currentAc = ACCESS_CATEGORY_BK;
	uint32 currentTime = 0;
	uint32 neededTimeGap = 0;
	uint8 locationInBitmap = 0;
	StaId stationOrVapIndex = 0;
	uint32 tempBitmap = 0;
	uint16 numberOfDataPdsOnAcQueues = 0;
	uint8 firstTid = 0;
	uint8 secondTid = 0;
	uint16 pdsLimitThreshold = 0;
	uint8 isVap = FALSE;
	TxQueueType_e queueType = HW_TX_Q_TYPE_STA_TID;
	TxSelectorLockStatus_e lockStatus = TX_SELECTOR_LOCK_STATUS_IDLE;
	uint16 maxDataPdsToDiscard = AGER_EMULATOR_PERIODIC_MAX_DATA_PDS_TO_DISCARD;
		
	if(AgerEmulatorParameters.state == AGER_EMULATOR_STATE_PERIODIC_MODE)
	{
		currentTime = GET_TSF_TIMER_LOW(); // TBD CDB - is it OK that ager emulator use only the TSF of band 0? 
		neededTimeGap = AGER_EMULATOR_CONVERT_10_MILISECONDS_TO_MICROSECONDS(AgerEmulatorParameters.agerEmulatorPeriod); 
		if((currentTime - AgerEmulatorParameters.lastTimePeriodicCheckWasActivated) >= neededTimeGap)
		{
			while(currentAc < ACCESS_CATEGORY_NUM) 
			{
				/* The copy here is done since that the current AC field in the Ager emulator parameters 
				   is used later in the queue process */
				AgerEmulatorParameters.currentAc = currentAc;

				/* Calculating these parameters only once for the AC */
				firstTid = AgerEmulatorQueuesPerAcTable[currentAc][AGER_EMULATOR_FIRST_TID_IN_AC];
				secondTid = AgerEmulatorQueuesPerAcTable[currentAc][AGER_EMULATOR_SECOND_TID_IN_AC];	
				pdsLimitThreshold = AgerEmulatorParameters.pdLimitThresholds[currentAc];
				
 				for(locationInBitmap = 0; locationInBitmap < AGER_EMULATOR_STATION_AND_VAP_BITMAP_SIZE; locationInBitmap++)
 				{
					if(AgerEmulatorParameters.numberOfStopsRequested != AgerEmulatorParameters.numberOfStopsProcessed)
					{
						/* If a new message that requires to stop the periodic check was received - stop the process */
						/* This check here is meant for the case that the bitmap is empty - stop processing becasue new
						   message was arrived */
						return;
					}
					
					tempBitmap = AgerEmulatorParameters.stationAndVapConnectedBitmap[locationInBitmap]; 
					while(tempBitmap)
					{		
						if(AgerEmulatorParameters.numberOfStopsRequested != AgerEmulatorParameters.numberOfStopsProcessed)
						{
							/* If a new message that requires to stop the periodic check was received - stop the process 
							   before starting to process the queue */
							return;
						}
						
						stationOrVapIndex = Utils_FindLastSetAndClear(&tempBitmap, CONVERT_BYTES_TO_BIT_INDEX(sizeof(tempBitmap)));
						stationOrVapIndex += CONVERT_WORDS_TO_BIT_INDEX(locationInBitmap);
						if(HW_NUM_OF_STATIONS <= stationOrVapIndex)
						{
							isVap = TRUE;
							queueType = HW_TX_Q_TYPE_VAP_TID;
							stationOrVapIndex -= HW_NUM_OF_STATIONS;
							numberOfDataPdsOnAcQueues = pHostIfCounters->qosTxVap[stationOrVapIndex][currentAc];
						}
						else
						{
							isVap = FALSE;
							queueType = HW_TX_Q_TYPE_STA_TID;

							numberOfDataPdsOnAcQueues = pHostIfCounters->qosTxSta[stationOrVapIndex][firstTid] + pHostIfCounters->qosTxSta[stationOrVapIndex][secondTid];
						}


						/* Instead of 'for' loop for 2 iterations the ietrations are written sequentially */
						if(pdsLimitThreshold <= numberOfDataPdsOnAcQueues)
						{
							/* Process both Tids */				
							/* The next block appears twice instead of function in order to not return value and check if the function should stop -
					  		   additional if for each queue that is processed */
							lockStatus = TxSelector_TryLockRequest(stationOrVapIndex, firstTid, queueType);
							if(TX_SELECTOR_LOCK_STATUS_LOCKED == lockStatus)
							{
								agerEmulatorProcessQueue(stationOrVapIndex, firstTid, isVap, maxDataPdsToDiscard);
							}
							else
							{
								AgerEmulatorParameters.numberOfQueuesSkippedDueToLockFailure ++;
							}

                            if(AgerEmulatorParameters.numberOfStopsRequested != AgerEmulatorParameters.numberOfStopsProcessed)
							{
								/* If a new message that requires to stop the periodic check was received - stop the process 
							   	   before starting to process the queue */
								return;
							}
							lockStatus = TxSelector_TryLockRequest(stationOrVapIndex, secondTid, queueType);
							if(TX_SELECTOR_LOCK_STATUS_LOCKED == lockStatus)
							{
								agerEmulatorProcessQueue(stationOrVapIndex, secondTid, isVap, maxDataPdsToDiscard);
							}
							else
							{
								AgerEmulatorParameters.numberOfQueuesSkippedDueToLockFailure ++;
							}
						}					
 					}     
				}
				
				/* Move to next AC */
				currentAc = AgerEmulatorNextAcTable[currentAc];
			}
			/*Update the last time periodc check was activated*/			
			AgerEmulatorParameters.lastTimePeriodicCheckWasActivated = GET_TSF_TIMER_LOW();
			/* Set timer again */
			OSAL_SET_TIMER_EXPLICIT(AGER_EMULATOR_PERIODIC_CHECK_TIMER, AgerEmulatorParameters.agerEmulatorPeriod, TASK_AGER_EMULATOR);
		}
		else
		{
			//ILOG0_V("agerEmulatorHandlePeriodicTimer >>> time is not passed");
			OSAL_SET_TIMER_EXPLICIT(AGER_EMULATOR_PERIODIC_CHECK_TIMER, AgerEmulatorParameters.agerEmulatorPeriod, TASK_AGER_EMULATOR);
		}
	}
	else
	{
//		ILOG0_V("agerEmulatorHandlePeriodicTimer >>> Not performing check");
	}
}


/**********************************************************************************

agerEmulatorHandleLockConfimration 


Description:
------------
This function handles lock confirmation


Input: 
-----	 
agerEmulatorMessage -  a pointer to the message that contains the station and TID

		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
static void agerEmulatorHandleLockConfimration(K_MSG* agerEmulatorMessage)
{
	LockReqCb_t* lockReqCb;
	uint8 tid = 0;
	RequesterLockParams_t lockRequesterParams;
	TxQueueType_e queueType = HW_TX_Q_TYPE_STA_TID;
	uint16 numberOfDiscardedPds = 0;
	uint16 maxPdsToDiscard = 0;
	
	lockReqCb = ((LockReqCb_t *)agerEmulatorMessage->abData);

	/*Locking the queue may fail if we are in the middle of the stop traffic procedure where TX manager disables the queue
	do not process the queue and continue*/
	if (lockReqCb->lockStatus == TX_SELECTOR_LOCK_STATUS_LOCKED)
	{
		maxPdsToDiscard = AgerEmulatorParameters.maximumPdsToDiscardLimit-AgerEmulatorParameters.numberOfDiscardedPds;
       	//ILOG0_DDDD("[agerEmulatorHandleLockConfimration1],maxPdsToDiscard = %d, AgerEmulatorParameters.numberOfDataPdsOnStationAc = %d, AgerEmulatorParameters.maximumPdsToDiscardLimit = %d, AgerEmulatorParameters.numberOfDiscardedPds = %d",maxPdsToDiscard ,AgerEmulatorParameters.numberOfDataPdsOnStationAc, AgerEmulatorParameters.maximumPdsToDiscardLimit, AgerEmulatorParameters.numberOfDiscardedPds);
       	//ILOG0_DDD("[agerEmulatorHandleLockConfimration], AgerEmulatorParameters.currentStationOrVapIndex = %d, AgerEmulatorParameters.currentAc = %d, AgerEmulatorParameters.currentTid = %d",AgerEmulatorParameters.currentStationOrVapIndex, AgerEmulatorParameters.currentAc, AgerEmulatorParameters.currentTid);
	   	numberOfDiscardedPds = agerEmulatorProcessQueue(AgerEmulatorParameters.currentStationOrVapIndex, AgerEmulatorParameters.currentTid, AgerEmulatorParameters.isCurrentQueueVap, maxPdsToDiscard);
	}

	//ILOG0_DDDD("[agerEmulatorHandleLockConfimration2], AgerEmulatorParameters.currentStationOrVapIndex = %d, AgerEmulatorParameters.currentAc = %d, AgerEmulatorParameters.currentTid = %d, numberOfDiscardedPds %d",AgerEmulatorParameters.currentStationOrVapIndex, AgerEmulatorParameters.currentAc, AgerEmulatorParameters.currentTid, numberOfDiscardedPds);

#if defined (AAA_AGER_EMULATOR_DEBUG)
	AgerEmulatorDebugStaTid[AgerEmulatorParameters.currentStationOrVapIndex][AgerEmulatorParameters.currentTid] += AgerEmulatorParameters.numberOfDiscardedPds;
	AgerEmulatorDebugStaAc[AgerEmulatorParameters.currentStationOrVapIndex][AgerEmulatorParameters.currentAc] += AgerEmulatorParameters.numberOfDiscardedPds;
#endif
	
	AgerEmulatorParameters.numberOfDiscardedPds += numberOfDiscardedPds; 
	if((AgerEmulatorParameters.isMaximumPdsToDiscardLimitEnabled) && (AgerEmulatorParameters.numberOfDiscardedPds >= AgerEmulatorParameters.maximumPdsToDiscardLimit))
	{
		/* Save the last queue station/VAP index and the TID */
		AgerEmulatorParameters.lastQueueCheckedTidIndex = AgerEmulatorQueuesPerAcTable[AgerEmulatorParameters.currentAc][AgerEmulatorParameters.tidInAc];
		AgerEmulatorParameters.lastQueueCheckedStationOrVapIndex = AgerEmulatorParameters.currentStationOrVapIndex;
		/* There are 2 checks for the index if it is VAP index - since the flag isLastQueueCheckedVap should not be set unless the scan is over */
		AgerEmulatorParameters.isLastQueueCheckedVap = AgerEmulatorParameters.isCurrentQueueVap;

		AgerEmulatorParameters.state = AGER_EMULATOR_STATE_IDLE;
		AgerEmulatorParameters.isSingleCheckDone = TRUE;
		/* The single check is done - return to normal priority */
		TxApp_ChangeThreadPriority(TASK_AGER_EMULATOR, TX_APP_PRIORITY_AGER_EMULATOR);

	    OSAL_SEND_NO_DATA_MESSAGE(QOS_SINGLE_CHECK_DONE, TASK_QOS, VAP_ID_DO_NOT_CARE);	
		return;
	}

    /* Process the second TID if possible */
	/*If we failed to lock the first queue no need to try and lock the second one as STA is being removed*/
	if ((lockReqCb->lockStatus == TX_SELECTOR_LOCK_STATUS_LOCKED) &&
	    (AgerEmulatorParameters.tidInAc == AGER_EMULATOR_FIRST_TID_IN_AC))

	{
		/* Try the next tid of the same AC in the same station/VAP */
		tid = AgerEmulatorQueuesPerAcTable[AgerEmulatorParameters.currentAc][AGER_EMULATOR_SECOND_TID_IN_AC];
		if(AgerEmulatorParameters.isCurrentQueueVap)
		{
			queueType = HW_TX_Q_TYPE_VAP_TID;
		}

		memset(&lockRequesterParams, 0, sizeof(RequesterLockParams_t));
#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)	
		/*If we want to lock a STA TID, except TID7, send a lock request for TID7 first - so we can remove any retransmitted
		ADDBA REQ from queue so queue will be unlocked by TS manager*/
		if ((AgerEmulatorParameters.isCurrentQueueVap == FALSE) && (tid != IEEE802_1D_VO_2))
		{
			lockRequesterParams.returnMsg = AGER_EMULATOR_TID7_LOCK_CONFIRMATION;
			lockRequesterParams.returnTask = TASK_AGER_EMULATOR;
			/*Periodic Check and Single check may request to Lock TID 7 for separate TIDs, pass it as a context*/
			lockRequesterParams.callerContext = (void *)tid;
			Locker_LockSingleQueue(queueType,AgerEmulatorParameters.currentStationOrVapIndex,IEEE802_1D_VO_2,&lockRequesterParams);
		}
#endif
		lockRequesterParams.returnMsg = AGER_EMULATOR_LOCK_CONFIRMATION;
		lockRequesterParams.returnTask = TASK_AGER_EMULATOR;

		AgerEmulatorParameters.currentTid = tid;
		AgerEmulatorParameters.tidInAc = AGER_EMULATOR_SECOND_TID_IN_AC;
			
		Locker_LockSingleQueue(queueType, AgerEmulatorParameters.currentStationOrVapIndex, tid, &lockRequesterParams);
		return;
	}

	/* Single check process continues */
	agerEmulatorPerformSingleCheckInForceMode(FALSE);	
}


/**********************************************************************************

agerEmulatorHandleTid7LockConfimration 


Description:
------------
This function handles lock confirmation for TID7


Input: 
-----	 
agerEmulatorMessage -  a pointer to the message that contains the station and TID

		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)	
static void agerEmulatorHandleTid7LockConfimration(K_MSG* agerEmulatorMessage)
{
	LockReqCb_t* lockReqCb = NULL;
	TxPd_t *currentPacketDescriptor = NULL;
	TxPd_t *lastPacketDescriptorToDiscard = NULL;
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
	uint32 dramPointer = 0;	/*DRAM Pointer in PD is still adjusted to HW*/
	AddbaRequestFrame_t *addbaRequestFrame = NULL;
	StaId stationIndex = 0;
	uint8 tid = 0;

	lockReqCb = ((LockReqCb_t *)agerEmulatorMessage->abData);
	if (lockReqCb->lockStatus != TX_SELECTOR_LOCK_STATUS_LOCKED)
	{
		/*This could happen if the queue was disabled as part of the stop traffic procedure.*/
		return;
	}
	stationIndex = lockReqCb->stationOrVapNum;
	memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
	hwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_STA_TID;
	hwQueueManagerRequestParams.primaryAddr = stationIndex;
	hwQueueManagerRequestParams.secondaryAddr = IEEE802_1D_VO_2;
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	HwQManager_PeekHeadPacket(&hwQueueManagerRequestParams);
	currentPacketDescriptor = (TxPd_t *)hwQueueManagerRequestParams.pHeadDesc;
	/*The PS Manager may add NDP to the head of queue*/
	if (((TxPd_t *)NULL_PD != currentPacketDescriptor) && 
		 ((PD_TYPE_NDP == currentPacketDescriptor->PdType) && 
		  (HW_Q_MANAGER_DONE_LIST_PS_SETTINGS_HW == currentPacketDescriptor->PdSource)))
	{
		/*Skip this PD*/
		currentPacketDescriptor = (TxPd_t *)GET_NEXT_PD(currentPacketDescriptor);
	}
	/*Check if first packet is retransmitted ADDBA REQ*/
	if (((TxPd_t *)NULL_PD != currentPacketDescriptor) && 
	  	 (currentPacketDescriptor->PdType == PD_TYPE_ADDBA_REQ) &&
	  	 (currentPacketDescriptor->Retransmit))
	{
		lastPacketDescriptorToDiscard = currentPacketDescriptor;
	}
	/*Did we find a retransmitted ADDBA REQ at head of queue?*/
	if(lastPacketDescriptorToDiscard != NULL)
	{
		/*Adjust pointer*/
		dramPointer = CONVERT_DMA_SHRAM_ADDR_TO_WLAN_SHRAM_ADDR(currentPacketDescriptor->DramPointer);  
		/*Cast to ADDBA REQ*/
		addbaRequestFrame = (AddbaRequestFrame_t *)frame_getPayloadPointerFromExistingManagementFrame((MANAGEMENT_BASIC_FRAME_HEADER *)dramPointer;
		/*Get TID from ADDBA REQ payload*/
		tid = addbaRequestFrame->addbaRequestPayload.u16BlockAckParameterSet.BlockAckTid;
		/*Is this the TID we wan't to lock? - TID was passed as the context because single check may interrupt periodic check*/
		if (tid == (uint32)lockReqCb->callerContext)
		{
			/*remove the ADDBA REQ*/
			hwQueueManagerRequestParams.pTailDesc = lastPacketDescriptorToDiscard;
			HwQManager_PopListFromHead(&hwQueueManagerRequestParams);
			/*Send to liberator - it will make its way to TS manager*/			
			hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
			hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_DONE_LIST_LIBERATOR;
			HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams);
		}
	}
	/*Unlock TID7*/
	Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID, stationIndex, (0x1 << IEEE802_1D_VO_2));
}
#endif

/**********************************************************************************

#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)	
#endif				
agerEmulatorConfigurePartialAndActivatePeriodicCheck 


Description:
------------
This function configure the part of the Ager emulator periodic check parameters
and activates the timer. It is meant to create unity between full configuration of 
periodic check parameters and partial configuration of periodic check parameters.

Input: 
-----	 
periodicCheckMessageParameters - a pointer to the message parameters that contain 
the needed parameters.
		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
static void agerEmulatorConfigurePartialAndActivatePeriodicCheck(AgerEmulatorPeriodicCheckMessageParameters_t *periodicCheckMessageParameters)
{
	/* Must be 32 because the value might be bigger than 2 ^ 16 */
	uint32 tempAgerPeriod = 0;
		
	memcpy(AgerEmulatorParameters.ttlCriteria, periodicCheckMessageParameters->acTtlCriteria, sizeof(AgerEmulatorParameters.ttlCriteria));

	/* The ager period in the message is in TTL units - need to be converted to timer units */
	tempAgerPeriod = HOST_INTERFACE_ACCELERATOR_CONVERT_TTL_UNITS_TO_TIME_UNITS(periodicCheckMessageParameters->agerPeriod);
	AgerEmulatorParameters.agerEmulatorPeriod = CONVERT_TIME_UNITS_TO_10_MILISECONDS(tempAgerPeriod);

	AgerEmulatorParameters.state = AGER_EMULATOR_STATE_PERIODIC_MODE;
	AgerEmulatorParameters.lastTimePeriodicCheckWasActivated = GET_TSF_TIMER_LOW(); // TBD CDB - is it OK that ager emulator use only the TSF of band 0? 
	OSAL_RESET_TIMER_EXPLICIT(AGER_EMULATOR_PERIODIC_CHECK_TIMER, TASK_AGER_EMULATOR);

	/* The ager period is already in 10 miliseconds units */
	OSAL_SET_TIMER_EXPLICIT(AGER_EMULATOR_PERIODIC_CHECK_TIMER, AgerEmulatorParameters.agerEmulatorPeriod, TASK_AGER_EMULATOR);
}

/**********************************************************************************

agerEmulatorSendConfirmToStationManager 


Description:
------------
Sends Confirmation message to STA manager

Input: 
-----
clientId -  the id that the Qos got when registered
stationIndex - the index of the station
	
		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
static void agerEmulatorSendConfirmToStationManager(StaId stationIndex)
{
	K_MSG *pMsg;
	BssManagerStaManagerCfm_t *confirmMessage;

	/*Allocate message*/
	pMsg = OSAL_GET_MESSAGE(sizeof(BssManagerStaManagerCfm_t));
	confirmMessage = (BssManagerStaManagerCfm_t*) pK_MSG_DATA(pMsg);
	/*Set Client ID to the registered ID*/
	confirmMessage->clientId = BSS_MANAGER_STA_MANAGER_AGER_CLIENT;
	/*Set STA ID*/
	confirmMessage->sid = stationIndex;
	/*Send confirmation message*/
	ILOG2_DD("agerEmulatorSendConfirmToStationManager, Client %d, stationIndex %d", confirmMessage->clientId, stationIndex);
	OSAL_SEND_MESSAGE(BSS_MANAGER_STA_MANAGER_REG_CFM, TASK_BSS_MANAGER, pMsg, VAP_ID_DO_NOT_CARE);
}

/**********************************************************************************

agerEmulatorPerformSingleCheckInForceMode 


Description:
------------
This function performs single check in force mode - lock request each queue.


Input: 
-----	 
updateParameters - indicates if there is a need to update parameters	
		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/

static void agerEmulatorPerformSingleCheckInForceMode(uint8 updateParameters)
{	
	StaId stationOrVapIndex = 0;
	uint16 numberOfDataPdsOnAcQueues = 0;

	RequesterLockParams_t lockRequesterParams;
	TxQueueType_e queueType = HW_TX_Q_TYPE_STA_TID;

	memset(&lockRequesterParams, 0, sizeof(RequesterLockParams_t));

	//("agerEmulatorPerformSingleCheckInForceMode updateParameters = %d", updateParameters);

//	ILOG0_DD("[agerEmulatorPerformSingleCheckInForceMode], AgerEmulatorParameters.currentAc = %d, AgerEmulatorParameters.currentPartOfBitmap = 0x%x", AgerEmulatorParameters.currentAc, AgerEmulatorParameters.currentPartOfBitmap);
	
	while(AgerEmulatorParameters.currentAc < ACCESS_CATEGORY_NUM) 
	{
		if(updateParameters)
		{
			if(AGER_EMULATOR_FIRST_PART_OF_BITMAP == AgerEmulatorParameters.currentPartOfBitmap)
			{
				/* Process first part of the bitmap - copy from word of starting index (included) */	
		   		memcpy32(&AgerEmulatorParameters.temporaryStationAndVapConnectedBitmap[AgerEmulatorParameters.startingLocationInBitmap], 
					     &AgerEmulatorParameters.stationAndVapConnectedBitmap[AgerEmulatorParameters.startingLocationInBitmap], 
					     (AGER_EMULATOR_STATION_AND_VAP_BITMAP_SIZE - AgerEmulatorParameters.startingLocationInBitmap));
		    	AgerEmulatorParameters.temporaryStationAndVapConnectedBitmap[AgerEmulatorParameters.startingLocationInBitmap] &= AGER_EMULATOR_MASK_LOWER_PART_OF_WORD(AgerEmulatorParameters.bitIndexInWord);
				AgerEmulatorParameters.currentLocationInBitmap = AgerEmulatorParameters.startingLocationInBitmap;
				AgerEmulatorParameters.lastLocationInBitmap = AGER_EMULATOR_STATION_AND_VAP_BITMAP_SIZE;
			}
			else
			{
				/* Process second part of the bitmap - copy to word of starting index (included), this is why + 1 to size */
		   		memcpy32(AgerEmulatorParameters.temporaryStationAndVapConnectedBitmap, AgerEmulatorParameters.stationAndVapConnectedBitmap, (AgerEmulatorParameters.startingLocationInBitmap + 1));
		    	AgerEmulatorParameters.temporaryStationAndVapConnectedBitmap[AgerEmulatorParameters.startingLocationInBitmap] &= AGER_EMULATOR_MASK_UPPER_PART_OF_WORD(AgerEmulatorParameters.bitIndexInWord);
				AgerEmulatorParameters.currentLocationInBitmap = 0;
				AgerEmulatorParameters.lastLocationInBitmap = AgerEmulatorParameters.startingLocationInBitmap + 1;
			}
			//("agerEmulatorPerformSingleCheckInForceMode currentPartOfBitmap = %d, currentLocationInBitmap = %d, lastLocationInBitmap = %d", AgerEmulatorParameters.currentPartOfBitmap, AgerEmulatorParameters.currentLocationInBitmap, AgerEmulatorParameters.lastLocationInBitmap);
		}	
//		ILOG0_DD("[agerEmulatorPerformSingleCheckInForceMode], AgerEmulatorParameters.currentLocationInBitmap = %d, AgerEmulatorParameters.lastLocationInBitmap = %d", AgerEmulatorParameters.currentLocationInBitmap, AgerEmulatorParameters.lastLocationInBitmap);
		while(AgerEmulatorParameters.currentLocationInBitmap < AgerEmulatorParameters.lastLocationInBitmap)
 		{
			while(AgerEmulatorParameters.temporaryStationAndVapConnectedBitmap[AgerEmulatorParameters.currentLocationInBitmap])
			{
				stationOrVapIndex = Utils_FindLastSetAndClear(&AgerEmulatorParameters.temporaryStationAndVapConnectedBitmap[AgerEmulatorParameters.currentLocationInBitmap],
																CONVERT_BYTES_TO_BIT_INDEX(sizeof(AgerEmulatorParameters.temporaryStationAndVapConnectedBitmap[AgerEmulatorParameters.currentLocationInBitmap])));
				stationOrVapIndex += CONVERT_WORDS_TO_BIT_INDEX(AgerEmulatorParameters.currentLocationInBitmap);
				//ILOG0_D("[agerEmulatorPerformSingleCheckInForceMode], stationOrVapIndex = %d", stationOrVapIndex);

				if(HW_NUM_OF_STATIONS <= stationOrVapIndex)
				{
					AgerEmulatorParameters.isCurrentQueueVap = TRUE;
					stationOrVapIndex -= HW_NUM_OF_STATIONS;
					queueType = HW_TX_Q_TYPE_VAP_TID;
					numberOfDataPdsOnAcQueues = pHostIfCounters->qosTxVap[stationOrVapIndex][AgerEmulatorParameters.currentAc];
				}
				else
				{
					AgerEmulatorParameters.isCurrentQueueVap = FALSE;
					queueType = HW_TX_Q_TYPE_STA_TID;

					numberOfDataPdsOnAcQueues = pHostIfCounters->qosTxSta[stationOrVapIndex][AgerEmulatorQueuesPerAcTable[AgerEmulatorParameters.currentAc][AGER_EMULATOR_FIRST_TID_IN_AC]] + pHostIfCounters->qosTxSta[stationOrVapIndex][AgerEmulatorQueuesPerAcTable[AgerEmulatorParameters.currentAc][AGER_EMULATOR_SECOND_TID_IN_AC]];

				}
				//ILOG0_DDDDD("[agerEmulatorPerformSingleCheckInForceMode], isCurrentQueueVap = %d, stationOrVapIndex = %d, numberOfDataPdsOnAcQueues = %d, pdLimitThresholds = %d, currentAc = %d", AgerEmulatorParameters.isCurrentQueueVap, stationOrVapIndex, numberOfDataPdsOnAcQueues, AgerEmulatorParameters.pdLimitThresholds[AgerEmulatorParameters.currentAc], AgerEmulatorParameters.currentAc);
				AgerEmulatorParameters.currentStationOrVapIndex = stationOrVapIndex;
		
                if(AgerEmulatorParameters.pdLimitThresholds[AgerEmulatorParameters.currentAc] <= numberOfDataPdsOnAcQueues)
                {
					/* Instead of 'for' loop for 2 iterations the ietrations are written sequentially */
					AgerEmulatorParameters.tidInAc = AGER_EMULATOR_FIRST_TID_IN_AC;
					
					//("agerEmulatorPerformSingleCheckInForceMode LOCK tid = %d", AgerEmulatorQueuesPerAcTable[AgerEmulatorParameters.currentAc][AGER_EMULATOR_FIRST_TID_IN_AC]);
					AgerEmulatorParameters.currentTid = AgerEmulatorQueuesPerAcTable[AgerEmulatorParameters.currentAc][AGER_EMULATOR_FIRST_TID_IN_AC];
#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)	
					/*If we want to lock a STA TID, except TID7, send a lock request for TID7 first - so we can remove any retransmitted
					ADDBA REQ from queue so queue will be unlocked by TS manager*/
					if ((AgerEmulatorParameters.isCurrentQueueVap == FALSE) && (AgerEmulatorParameters.currentTid != IEEE802_1D_VO_2))
					{
						lockRequesterParams.returnMsg = AGER_EMULATOR_TID7_LOCK_CONFIRMATION;
						lockRequesterParams.returnTask = TASK_AGER_EMULATOR;
						/*Periodic Check and Single check may request to Lock TID 7 for separate TIDs, pass it as a context*/
						lockRequesterParams.callerContext = (void *)AgerEmulatorParameters.currentTid;
						Locker_LockSingleQueue(queueType,stationOrVapIndex,IEEE802_1D_VO_2,&lockRequesterParams);
					}
#endif
					lockRequesterParams.returnMsg = AGER_EMULATOR_LOCK_CONFIRMATION;
					lockRequesterParams.returnTask = TASK_AGER_EMULATOR;
                    //ILOG0_D("agerEmulatorPerformSingleCheckInForceMode: LOCKED AC: %d",AgerEmulatorParameters.currentAc);
					Locker_LockSingleQueue(queueType, stationOrVapIndex, AgerEmulatorQueuesPerAcTable[AgerEmulatorParameters.currentAc][AGER_EMULATOR_FIRST_TID_IN_AC], &lockRequesterParams);
					return;
				}
			}
			AgerEmulatorParameters.currentLocationInBitmap ++;
 		}
		updateParameters = TRUE;
		if(AGER_EMULATOR_FIRST_PART_OF_BITMAP == AgerEmulatorParameters.currentPartOfBitmap)
		{
			AgerEmulatorParameters.currentPartOfBitmap = AGER_EMULATOR_SECOND_PART_OF_BITMAP;
		}
		else
		{
			AgerEmulatorParameters.currentPartOfBitmap = AGER_EMULATOR_FIRST_PART_OF_BITMAP;
			AgerEmulatorParameters.currentAc = AgerEmulatorNextAcTable[AgerEmulatorParameters.currentAc];
		}	
	}

    /* All the queues were processed - the last queue is before the starting index */
	AgerEmulatorParameters.lastQueueCheckedTidIndex = AgerEmulatorQueuesPerAcTable[ACCESS_CATEGORY_VO][AGER_EMULATOR_SECOND_TID_IN_AC];
    if(AgerEmulatorParameters.stationOrVapIndexToStart == 0)
    {
		AgerEmulatorParameters.lastQueueCheckedStationOrVapIndex = HW_NUM_OF_VAPS - 1;
		AgerEmulatorParameters.isLastQueueCheckedVap = TRUE;
    }
	else
	{
		AgerEmulatorParameters.lastQueueCheckedStationOrVapIndex = AgerEmulatorParameters.stationOrVapIndexToStart - 1;
		AgerEmulatorParameters.isLastQueueCheckedVap = FALSE;
		if(HW_NUM_OF_STATIONS <= AgerEmulatorParameters.lastQueueCheckedStationOrVapIndex)
		{
			AgerEmulatorParameters.lastQueueCheckedStationOrVapIndex -= HW_NUM_OF_STATIONS;
			AgerEmulatorParameters.isLastQueueCheckedVap = TRUE;
		}
	}
	
	/* The process is ended - all the queue were processed */
	AgerEmulatorParameters.state = AGER_EMULATOR_STATE_IDLE;
	AgerEmulatorParameters.isSingleCheckDone = TRUE;
	
	//ILOG0_V("[agerEmulatorPerformSingleCheckInForceMode] END");
	
	/* The single check is done - return to normal priority */
	TxApp_ChangeThreadPriority(TASK_AGER_EMULATOR, TX_APP_PRIORITY_AGER_EMULATOR);

	OSAL_SEND_NO_DATA_MESSAGE(QOS_SINGLE_CHECK_DONE, TASK_QOS, VAP_ID_DO_NOT_CARE);	
}


/**********************************************************************************

agerEmulatorPerformSingleCheckInTryMode 


Description:
------------
This function performs single check in try mode - 'try lock' each queue.


Input: 
-----	 
	
		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
static void agerEmulatorPerformSingleCheckInTryMode(void)
{	
	uint8 currentAc = ACCESS_CATEGORY_BK;
	uint8 locationInBitmap = 0;
	uint8 bitIndexInWord = 0;
	StaId stationOrVapIndex = 0;
	uint8 tempLocationInBitmap = 0;
    uint16 totalNumberOfDiscardedPds = 0;
	uint16 numberOfDataPdsOnAcQueues = 0;
	uint8 currentPartOfBitmap = AGER_EMULATOR_FIRST_PART_OF_BITMAP;
	uint8 lastLocationInBitmap = 0;
	uint8 firstTid = 0;
	uint8 secondTid = 0;
	uint16 pdsLimitThreshold = 0;
	uint8 isVap = FALSE;
	TxQueueType_e queueType = HW_TX_Q_TYPE_STA_TID;
	TxSelectorLockStatus_e lockStatus = TX_SELECTOR_LOCK_STATUS_IDLE;
	uint16 maxPdsToDiscard;
			
	locationInBitmap = CONVERT_BIT_INDEX_TO_WORDS(AgerEmulatorParameters.stationOrVapIndexToStart);
	bitIndexInWord = AgerEmulatorParameters.stationOrVapIndexToStart & AGER_EMULATOR_BITMAP_MASK; 

	//("agerEmulatorPerformSingleCheckInTryMode ");		
	while(currentAc < ACCESS_CATEGORY_NUM) 
	{	
		/* The copy here is done since that the current AC field in the Ager emulator parameters 
		   is used later in the queue process */
		AgerEmulatorParameters.currentAc = currentAc;

		/* Calculating these parameters only once for the AC */
		firstTid = AgerEmulatorQueuesPerAcTable[currentAc][AGER_EMULATOR_FIRST_TID_IN_AC];
        secondTid = AgerEmulatorQueuesPerAcTable[currentAc][AGER_EMULATOR_SECOND_TID_IN_AC];
		pdsLimitThreshold = AgerEmulatorParameters.pdLimitThresholds[currentAc];
			
		if(AGER_EMULATOR_FIRST_PART_OF_BITMAP == currentPartOfBitmap)
		{
			/* Process first part of the bitmap - copy from word of starting index (included) */	
   			memcpy32(&AgerEmulatorParameters.temporaryStationAndVapConnectedBitmap[locationInBitmap], 
			         &AgerEmulatorParameters.stationAndVapConnectedBitmap[locationInBitmap], 
			         (AGER_EMULATOR_STATION_AND_VAP_BITMAP_SIZE - locationInBitmap));
			AgerEmulatorParameters.temporaryStationAndVapConnectedBitmap[locationInBitmap] &= AGER_EMULATOR_MASK_LOWER_PART_OF_WORD(bitIndexInWord);
			tempLocationInBitmap = locationInBitmap;
			lastLocationInBitmap = AGER_EMULATOR_STATION_AND_VAP_BITMAP_SIZE;
		}
		else
		{
			/* Process second part of the bitmap - copy to word of starting index (included), this is why + 1 to size */
   			memcpy32(AgerEmulatorParameters.temporaryStationAndVapConnectedBitmap, AgerEmulatorParameters.stationAndVapConnectedBitmap, (locationInBitmap + 1));
    		AgerEmulatorParameters.temporaryStationAndVapConnectedBitmap[locationInBitmap] &= AGER_EMULATOR_MASK_UPPER_PART_OF_WORD(bitIndexInWord);
			tempLocationInBitmap = 0;
			lastLocationInBitmap = locationInBitmap + 1;
		}
		//("agerEmulatorPerformSingleCheckInTryMode currentPartOfBitmap = %d , tempLocationInBitmap = %d , lastLocationInBitmap = %d ", currentPartOfBitmap, tempLocationInBitmap, lastLocationInBitmap);	
		while(tempLocationInBitmap < lastLocationInBitmap)
 		{
			while(AgerEmulatorParameters.temporaryStationAndVapConnectedBitmap[tempLocationInBitmap])
			{
				stationOrVapIndex = Utils_FindLastSetAndClear(&AgerEmulatorParameters.temporaryStationAndVapConnectedBitmap[tempLocationInBitmap], 
															CONVERT_BYTES_TO_BIT_INDEX(sizeof(AgerEmulatorParameters.temporaryStationAndVapConnectedBitmap[tempLocationInBitmap])));
				stationOrVapIndex += CONVERT_WORDS_TO_BIT_INDEX(tempLocationInBitmap);
	
				if(HW_NUM_OF_STATIONS <= stationOrVapIndex)
				{
					isVap = TRUE;
					queueType = HW_TX_Q_TYPE_VAP_TID;
					stationOrVapIndex -= HW_NUM_OF_STATIONS;
					numberOfDataPdsOnAcQueues = pHostIfCounters->qosTxVap[stationOrVapIndex][AgerEmulatorParameters.currentAc];
				}
				else
				{
					isVap = FALSE;
					queueType = HW_TX_Q_TYPE_STA_TID;

					numberOfDataPdsOnAcQueues = pHostIfCounters->qosTxSta[stationOrVapIndex][firstTid] + pHostIfCounters->qosTxSta[stationOrVapIndex][secondTid];
				}
				//("agerEmulatorPerformSingleCheckInTryMode isVap = %d , stationOrVapIndex = %d , numberOfDataPdsOnAcQueues = %d ", isVap, stationOrVapIndex, numberOfDataPdsOnAcQueues);	


				/* Instead of 'for' loop for 2 iterations the ietrations are written sequentially */

				if(pdsLimitThreshold <= numberOfDataPdsOnAcQueues)
				{
					/* The next block appears twice instead of function in order to not return value and check if the function should stop -
					   additional if for each queue that is processed */
					lockStatus = TxSelector_TryLockRequest(stationOrVapIndex, firstTid, queueType);
					//("agerEmulatorPerformSingleCheckInTryMode data is in queue LOCK_STATUS = %d  ", lockStatus);
					if(TX_SELECTOR_LOCK_STATUS_LOCKED == lockStatus)
					{
						maxPdsToDiscard = AgerEmulatorParameters.maximumPdsToDiscardLimit-totalNumberOfDiscardedPds;
						totalNumberOfDiscardedPds += agerEmulatorProcessQueue(stationOrVapIndex, firstTid, isVap, maxPdsToDiscard);
						if(AgerEmulatorParameters.isMaximumPdsToDiscardLimitEnabled && (totalNumberOfDiscardedPds >= AgerEmulatorParameters.maximumPdsToDiscardLimit))
	                	{
							/* In case the number of discarded PDs has reached the number of PDs needed to be discarded stop the process */
							/* Save the last queue - index and TID */
							AgerEmulatorParameters.lastQueueCheckedTidIndex = firstTid;
							AgerEmulatorParameters.lastQueueCheckedStationOrVapIndex = stationOrVapIndex;								
							AgerEmulatorParameters.isLastQueueCheckedVap = isVap;
				
							AgerEmulatorParameters.state = AGER_EMULATOR_STATE_IDLE;
							AgerEmulatorParameters.isSingleCheckDone = TRUE;
							return;
							//("agerEmulatorPerformSingleCheckInTryMode FINISHED  totalNumberOfDiscardedPds = %d  ", totalNumberOfDiscardedPds);
						}
					}
					else
					{
						AgerEmulatorParameters.numberOfQueuesSkippedDueToLockFailure ++;
					}
					lockStatus = TxSelector_TryLockRequest(stationOrVapIndex, secondTid, queueType);
					//("agerEmulatorPerformSingleCheckInTryMode  LOCK_STATUS = %d  ", lockStatus);
					if(TX_SELECTOR_LOCK_STATUS_LOCKED == lockStatus)
					{
						maxPdsToDiscard = AgerEmulatorParameters.maximumPdsToDiscardLimit-totalNumberOfDiscardedPds;
						totalNumberOfDiscardedPds += agerEmulatorProcessQueue(stationOrVapIndex, secondTid, isVap, maxPdsToDiscard);
						//("agerEmulatorPerformSingleCheckInTryMode totalNumberOfDiscardedPds = %d  ", totalNumberOfDiscardedPds);
						if(AgerEmulatorParameters.isMaximumPdsToDiscardLimitEnabled && (totalNumberOfDiscardedPds >= AgerEmulatorParameters.maximumPdsToDiscardLimit))
	                	{
							/* In case the number of discarded PDs has reached the number of PDs needed to be discarded stop the process */
							/* Save the last queue - index and TID */
							AgerEmulatorParameters.lastQueueCheckedTidIndex = secondTid;
							AgerEmulatorParameters.lastQueueCheckedStationOrVapIndex = stationOrVapIndex;
							AgerEmulatorParameters.isLastQueueCheckedVap = isVap;
									
							AgerEmulatorParameters.state = AGER_EMULATOR_STATE_IDLE;
							AgerEmulatorParameters.isSingleCheckDone = TRUE;
								//("agerEmulatorPerformSingleCheckInTryMode FINISHED  totalNumberOfDiscardedPds = %d  ", totalNumberOfDiscardedPds);
							return;
                		}
					}
					else
					{
						AgerEmulatorParameters.numberOfQueuesSkippedDueToLockFailure ++;
					}
				}				
			}
			tempLocationInBitmap ++;
 		}

		if(AGER_EMULATOR_FIRST_PART_OF_BITMAP == currentPartOfBitmap)
		{
			currentPartOfBitmap = AGER_EMULATOR_SECOND_PART_OF_BITMAP;
		}
		else
		{
			currentPartOfBitmap = AGER_EMULATOR_FIRST_PART_OF_BITMAP;
			/* Move to next AC */
			currentAc = AgerEmulatorNextAcTable[currentAc];
		}	
	}

	/* All the queues were processed - the last queue checked is before the starting index */
	AgerEmulatorParameters.lastQueueCheckedTidIndex = AgerEmulatorQueuesPerAcTable[ACCESS_CATEGORY_VO][AGER_EMULATOR_SECOND_TID_IN_AC];
    if(AgerEmulatorParameters.stationOrVapIndexToStart == 0)
    {
		AgerEmulatorParameters.lastQueueCheckedStationOrVapIndex = HW_NUM_OF_VAPS - 1;
		AgerEmulatorParameters.isLastQueueCheckedVap = TRUE;
    }
	else
	{
		AgerEmulatorParameters.lastQueueCheckedStationOrVapIndex = AgerEmulatorParameters.stationOrVapIndexToStart - 1;
		AgerEmulatorParameters.isLastQueueCheckedVap = FALSE;
		if(HW_NUM_OF_STATIONS <= AgerEmulatorParameters.lastQueueCheckedStationOrVapIndex)
		{
			AgerEmulatorParameters.lastQueueCheckedStationOrVapIndex -= HW_NUM_OF_STATIONS;
			AgerEmulatorParameters.isLastQueueCheckedVap = TRUE;
		}
	}
	
	/* The process is ended - all the queue were processed */
	AgerEmulatorParameters.state = AGER_EMULATOR_STATE_IDLE;
	AgerEmulatorParameters.isSingleCheckDone = TRUE;
}

/**********************************************************************************

agerEmulatorProcessQueue 


Description:
------------
This function scans a queue in periodic check. The Queue is already locked


Input: 
-----	 
stationOrVapIndex - the index of the station or VAP
tid - the TID of the queue
isVap - indicates if this is a VAP queue
maxDataPdsToDiscard - maximum data PDs to discard

		
Output:
-------


Returns:
--------
The number of discarded PDs (all types)

	
**********************************************************************************/
static uint16 agerEmulatorProcessQueue(StaId stationOrVapIndex, uint8 tid, uint8 isVap, uint16 maxDataPdsToDiscard)
{	
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
	TxQueueType_e queueType = HW_TX_Q_TYPE_STA_TID; 
	TxPd_t *currentPacketDescriptor = NULL;
	TxPd_t *amsduTail = NULL;
	TxPd_t *popListTail = NULL;
	TxPd_t *umacListHead = NULL;
	TxPd_t *umacListTail = NULL;
	TxPd_t *notDiscardedListHead = NULL;
	TxPd_t *notDiscardedListTail = NULL;
	TxPd_t *liberatorListHead = NULL;
	TxPd_t *liberatorListTail = NULL;
	uint8 currentTtl = 0;
	uint8 ttlGap = 0;
	uint32 currentTime = 0;
	uint8 ttlCriteria = 0;
	uint16 numberOfDiscardedPds = 0;
	uint16 numberOfDiscardedDataPds = 0;
	uint8 locationInBitmap = 0;
	uint8 locationInWord = 0;
	uint16 startingSequenceNumber = 0;
	uint8 locationInArray = 0;
	uint8 isManagement = FALSE;
	uint8 pdIndex = 0;
	uint8 statisticsVapId = 0;
	uint8 numberOfPdsInAmsdu = 0;
	uint32 tempReorderingBitmap[AGER_EMULATOR_REORDERING_BITMAP_SIZE];
	TxPd_t *tempReorderingDescriptor = NULL;

	//ILOG0_DDDD("[agerEmulatorProcessQueue], stationOrVapIndex = %d, tid = %d, isVap = %d, numberOfDataPdsOfStationAc = %d", stationOrVapIndex, tid, isVap, numberOfDataPdsOfStationAc);

	memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));

	currentTime = GET_TSF_TIMER_LOW(); // TBD CDB - is it OK that ager emulator use only the TSF of band 0? 
	currentTtl = HOST_INTERFACE_ACCELERATOR_CONVERT_USEC_TO_TTL_UNITS(currentTime);
	/* The TTL is 7 bits long */
	currentTtl &= HOST_INTERFACE_TTL_MASK;
	ttlCriteria = AgerEmulatorParameters.ttlCriteria[AgerEmulatorParameters.currentAc];
	
	if(isVap)
	{
		queueType = HW_TX_Q_TYPE_VAP_TID;	
	}
	else
	{	
		/* The sequence number is used only in case of station since there should be no PDs with retransmission bit set
		   on VAP queues */
		startingSequenceNumber = StaDB_GetBaWinSsn(stationOrVapIndex, tid);
	}
	
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
	hwQueueManagerRequestParams.dplIndex = queueType;
	hwQueueManagerRequestParams.primaryAddr = stationOrVapIndex;
	hwQueueManagerRequestParams.secondaryAddr = tid;
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;

	HwQManager_PeekHeadPacket(&hwQueueManagerRequestParams);
	currentPacketDescriptor = (TxPd_t *)hwQueueManagerRequestParams.pHeadDesc;

    //("agerEmulatorProcessQueue isVap = %d, pHeadDesc = %x , pTailDesc = %x , ttlCriteria = %d , currentTtl = %d ", isVap, hwQueueManagerRequestParams.pHeadDesc, hwQueueManagerRequestParams.pTailDesc, ttlCriteria, currentTtl);
    /* Reset not discarded PDs bitmap */
	AgerEmulatorParameters.reorderingPacketDescriptorsBitmap[AGER_EMULATOR_FIRST_PART_OF_BITMAP] = 0;
	AgerEmulatorParameters.reorderingPacketDescriptorsBitmap[AGER_EMULATOR_SECOND_PART_OF_BITMAP] = 0;

    /* A BAR is a special case: it is always put at the head of the queue so its TTL might be newer and its retransmission bit
       might be cleared even if the next PD retranmsission bit is set */
    /* This code emulated the HW Ager behaviour - if the TTL is too new the processing of the queue stops */
 	while(((TxPd_t *)NULL_PD != currentPacketDescriptor) && ((PD_TYPE_BAR == currentPacketDescriptor->pdType)))
	{
		ttlGap = currentTtl - currentPacketDescriptor->ttlCount;
		/* In case of wrap around the last bit is ignored */
		ttlGap &= HOST_INTERFACE_TTL_MASK;
		ILOG2_DD("agerEmulatorProcessQueue BAR TtlCount = %d , , retransmission = %d", currentPacketDescriptor->ttlCount, currentPacketDescriptor->retransmissionIndication);
		if(ttlGap < ttlCriteria)
		{
			/*Make sure we don't handle any more PDs which may have an "older" TSF by setting descriptor to NULL*/
			currentPacketDescriptor = (TxPd_t *)NULL_PD;
		}
		else
		{
			/* The BAR or NDP PS is too old - discard the PD */
			popListTail = currentPacketDescriptor;
			numberOfDiscardedPds ++;
			if(currentPacketDescriptor->retransmissionIndication)
			{
				if(NULL == umacListHead)
				{
					/* Fisrt PD in the list */
					umacListHead = currentPacketDescriptor;
					umacListTail = currentPacketDescriptor;
				}
				else
				{
					/* The list might be not empty  as there may be both BAR and NDP PS at the head of the queue and both of them may be dropped */
					umacListTail->nextPdPointer = SET_NEXT_PD(currentPacketDescriptor);
					umacListTail = currentPacketDescriptor;	
				}
			}
			else
			{
				if(NULL == liberatorListHead)
				{
					/* Fisrt PD in the list */
					liberatorListHead = currentPacketDescriptor;
					liberatorListTail = currentPacketDescriptor;
				}
				else
				{
					/* The list might be not empty  as there may be both BAR and NDP PS at the head of the queue and both of them may be dropped */
					liberatorListTail->nextPdPointer = SET_NEXT_PD(currentPacketDescriptor);
					liberatorListTail = currentPacketDescriptor;
				}	
			}
			currentPacketDescriptor->status = PD_STATUS_DISCARDED_BY_AGER;
			currentPacketDescriptor = (TxPd_t *)GET_NEXT_PD(currentPacketDescriptor);
		}
	}

	/* A management PD is a special case if the retransmission bit is set - since that the data PDs are organized by their
	   sequence number and the management sequence number is taken from a different source it should be handled seperately */
	if((TxPd_t *)NULL_PD != currentPacketDescriptor)
	{
		isManagement = AgerEmulatorManagementPdTypeTable[currentPacketDescriptor->pdType];
	}
	if(((TxPd_t *)NULL_PD != currentPacketDescriptor) && (TRUE == isManagement) && (isVap == FALSE) && (currentPacketDescriptor->retransmissionIndication))
	{
		ttlGap = currentTtl - currentPacketDescriptor->ttlCount;
		/* In case of wrap around the last bit is ignored */
		ttlGap &= HOST_INTERFACE_TTL_MASK;
        ILOG2_D("agerEmulatorProcessQueue MANAGEMENT  TtlCount = %d ", currentPacketDescriptor->ttlCount);
		/* If the PD is not old enough the function continues (and will stop in both 'whiles' . There is no special treatmnet for
		   this case here since there might be already something in the list if a BAR was discarded */
		if(ttlGap < ttlCriteria)
		{
			/*Make sure we don't handle data which may have an "older" TSF by setting descriptor to NULL*/
			currentPacketDescriptor = (TxPd_t *)NULL_PD;
		}
		else
		{
			popListTail = currentPacketDescriptor;
			/* The PD is too old - discard the PD */
			numberOfDiscardedPds ++;
			if(NULL == umacListHead)
			{
				/* Fisrt PD in the list */
				umacListHead = currentPacketDescriptor;
				umacListTail = currentPacketDescriptor;
			}
			else
			{
				/* The list might be not empty if a BAR or PS NDP was discarded before this PD */
				umacListTail->nextPdPointer = SET_NEXT_PD(currentPacketDescriptor);
				umacListTail = currentPacketDescriptor;	
			}
			currentPacketDescriptor->status = PD_STATUS_DISCARDED_BY_AGER;
			/*Move to next descriptor*/
			currentPacketDescriptor = (TxPd_t *)GET_NEXT_PD(currentPacketDescriptor);
		}
	}   

	/*In Gen4 it is possible that a multicast packet will be in the head of the queue with the retransmit bit set - e.g. packet not pushed to HW due to Qswitch*/
	/*Check retransmit only for STA queues*/
	if (isVap == FALSE)
	{
	    /* First process all the PDs with retransmission bit set - in AR10 their order is mixed, so in order to unify the code for wave500 
	       and AR10 the list of retransmission PDs is fully scan despite the fact that in wave500 their order is by theior arrival time */
		while(((TxPd_t *)NULL_PD != currentPacketDescriptor) && (currentPacketDescriptor->retransmissionIndication))
		{
			ILOG2_V("agerEmulatorProcessQueue RETRANSMIT PDs ");
			/* If the PD is the head of A-MSDU save the A-MSDU tail */
			if(((TxPd_t *)currentPacketDescriptor)->Aggregation)
			{
				amsduTail = (TxPd_t *)CONVERT_OFFSET_TO_PD(currentPacketDescriptor->aMsduTailPointer);
			}
			else
			{
				amsduTail = currentPacketDescriptor; 
			}
			
			popListTail = amsduTail;
			/* PDs with retransmission bit set at this point should be only data */
			locationInArray = currentPacketDescriptor->sn - startingSequenceNumber;
			locationInArray &= AGER_EMULATOR_WINDOW_SIZE_MASK;

			AgerEmulatorParameters.reorderingPacketDescriptorsArray[locationInArray] = currentPacketDescriptor;
	        locationInBitmap = CONVERT_BIT_INDEX_TO_WORDS(locationInArray);
			locationInWord = locationInArray & AGER_EMULATOR_BITMAP_MASK; 

			AgerEmulatorParameters.reorderingPacketDescriptorsBitmap[locationInBitmap] |= 0x1 << locationInWord;
			ILOG9_DDD("agerEmulatorProcessQueue RETRANSMIT NOT DISCARD locationInArray = %d, locationInBitmap = %d, locationInWord = %d ", locationInArray, locationInBitmap, locationInWord);
			currentPacketDescriptor = (TxPd_t *)GET_NEXT_PD(amsduTail);
		}
	}

	/*Now search reordered queue for aged out packets*/
	tempReorderingBitmap[AGER_EMULATOR_FIRST_PART_OF_BITMAP] = AgerEmulatorParameters.reorderingPacketDescriptorsBitmap[AGER_EMULATOR_FIRST_PART_OF_BITMAP];
	tempReorderingBitmap[AGER_EMULATOR_SECOND_PART_OF_BITMAP] = AgerEmulatorParameters.reorderingPacketDescriptorsBitmap[AGER_EMULATOR_SECOND_PART_OF_BITMAP];
	/*Search first part of the bitmap*/
	while(tempReorderingBitmap[AGER_EMULATOR_FIRST_PART_OF_BITMAP])
	{
		pdIndex = Utils_FindLastSetAndClear(&tempReorderingBitmap[AGER_EMULATOR_FIRST_PART_OF_BITMAP],
										    CONVERT_BYTES_TO_BIT_INDEX(sizeof(tempReorderingBitmap[AGER_EMULATOR_FIRST_PART_OF_BITMAP])));
		tempReorderingDescriptor = AgerEmulatorParameters.reorderingPacketDescriptorsArray[pdIndex];
		ttlGap = currentTtl - tempReorderingDescriptor->ttlCount;
		/* In case of wrap around the last bit is ignored */
		ttlGap &= HOST_INTERFACE_TTL_MASK;
		ILOG9_DDDD("agerEmulatorProcessQueue RETRANSMIT PDs current = %x, amsdu tail = %x , ttlCount = %d , numberOfPdsInAmsdu %d ", tempReorderingDescriptor, amsduTail, tempReorderingDescriptor->ttlCount, numberOfPdsInAmsdu);
		if(ttlGap >= ttlCriteria)
		{
			if(((TxPd_t *)tempReorderingDescriptor)->aggregationIndication)
			{
				amsduTail = (TxPd_t *)CONVERT_OFFSET_TO_PD(tempReorderingDescriptor->aggregationIndication);
				numberOfPdsInAmsdu = tempReorderingDescriptor->pdCounter;
			}
			else
			{
				amsduTail = tempReorderingDescriptor; 
				numberOfPdsInAmsdu = 1;
			}
			/*clear bit from reordering bitmap as this PD does not need to be pushed back*/
			Utils_ZeroBitInBitmap(&AgerEmulatorParameters.reorderingPacketDescriptorsBitmap[AGER_EMULATOR_FIRST_PART_OF_BITMAP], pdIndex);
			ILOG9_DD("agerEmulatorProcessQueue RETRANSMIT DISCARD umacListHead = %x, umacListTail = %x ", umacListHead, umacListTail);
			/* Discard PD - too old */
			numberOfDiscardedPds += numberOfPdsInAmsdu;
			if (tempReorderingDescriptor->pdSource == HW_Q_MANAGER_DONE_LIST_HOST_DATA)
			{
				numberOfDiscardedDataPds += numberOfPdsInAmsdu;
			}
			AgerEmulatorParameters.numberOfDiscardedPdsHandledToUmac += numberOfPdsInAmsdu;
			
			if(NULL == umacListHead)
			{
				/* Fisrt PD in the list */
				umacListHead = tempReorderingDescriptor;
				umacListTail = amsduTail;
			}
			else
			{
				umacListTail->nextPdPointer = SET_NEXT_PD(currentPacketDescriptor);
				umacListTail = amsduTail;	
			}
			currentPacketDescriptor->status = PD_STATUS_DISCARDED_BY_AGER;
		}
		else
		{
			/*we found one PD that has not aged, stop the search*/
			tempReorderingBitmap[AGER_EMULATOR_FIRST_PART_OF_BITMAP] = 0;
			tempReorderingBitmap[AGER_EMULATOR_SECOND_PART_OF_BITMAP] = 0;
			/*Make sure we don't handle data which may have an "older" TSF by setting descriptor to NULL*/
			currentPacketDescriptor = (TxPd_t *)NULL_PD;
		}
	}
	/*Search second part of the bitmap*/
	while(tempReorderingBitmap[AGER_EMULATOR_SECOND_PART_OF_BITMAP])
	{
		pdIndex = Utils_FindLastSetAndClear(&tempReorderingBitmap[AGER_EMULATOR_SECOND_PART_OF_BITMAP],
										    CONVERT_BYTES_TO_BIT_INDEX(sizeof(tempReorderingBitmap[AGER_EMULATOR_SECOND_PART_OF_BITMAP])));
		tempReorderingDescriptor = AgerEmulatorParameters.reorderingPacketDescriptorsArray[pdIndex + AGER_EMULATOR_WORD_SIZE_TO_ADD_TO_BIT_INDEX];
		ttlGap = currentTtl - tempReorderingDescriptor->ttlCount;
		/* In case of wrap around the last bit is ignored */
		ttlGap &= HOST_INTERFACE_TTL_MASK;
		ILOG9_DDDD("agerEmulatorProcessQueue RETRANSMIT PDs current = %x, amsdu tail = %x , ttlCount = %d , numberOfPdsInAmsdu %d ", tempReorderingDescriptor, amsduTail, tempReorderingDescriptor->ttlCount, numberOfPdsInAmsdu);
		if(ttlGap >= ttlCriteria)
		{
			if(((TxPd_t *)tempReorderingDescriptor)->Aggregation)
			{
				amsduTail = (TxPd_t *)CONVERT_OFFSET_TO_PD(tempReorderingDescriptor->aggregationIndication);
				numberOfPdsInAmsdu = tempReorderingDescriptor->pdCounter;
			}
			else
			{
				amsduTail = tempReorderingDescriptor; 
				numberOfPdsInAmsdu = 1;
			}
			/*clear bit from reordering bitmap as this PD does not need to be pushed back*/
			Utils_ZeroBitInBitmap(&AgerEmulatorParameters.reorderingPacketDescriptorsBitmap[AGER_EMULATOR_SECOND_PART_OF_BITMAP], pdIndex);
			ILOG9_DD("agerEmulatorProcessQueue RETRANSMIT DISCARD umacListHead = %x, umacListTail = %x ", umacListHead, umacListTail);
			/* Discard PD - too old */
			numberOfDiscardedPds += numberOfPdsInAmsdu;
			if (tempReorderingDescriptor->pdSource == HW_Q_MANAGER_DONE_LIST_HOST_DATA)
			{
				numberOfDiscardedDataPds += numberOfPdsInAmsdu;
			}
			AgerEmulatorParameters.numberOfDiscardedPdsHandledToUmac += numberOfPdsInAmsdu;
			
			if(NULL == umacListHead)
			{
				/* Fisrt PD in the list */
				umacListHead = tempReorderingDescriptor;
				umacListTail = amsduTail;
			}
			else
			{
				umacListTail->nextPdPointer = SET_NEXT_PD(tempReorderingDescriptor);
				umacListTail = amsduTail;	
			}
			tempReorderingDescriptor->status = PD_STATUS_DISCARDED_BY_AGER;
		}
		else
		{
			/*we found one PD that has not aged, stop the search*/
			tempReorderingBitmap[AGER_EMULATOR_SECOND_PART_OF_BITMAP] = 0;
			/*Make sure we don't handle data which may have an "older" TSF by setting descriptor to NULL*/
		currentPacketDescriptor = (TxPd_t *)GET_NEXT_PD(amsduTail);
		}
	}

	/* If a PD with retransmission bit set was not old enough (was not discarded) the PDs with retransmission bit cleared
	   dont have to be checked since they are newer than this PD */
	while(((TxPd_t *)NULL_PD != currentPacketDescriptor) && (numberOfDiscardedDataPds < maxDataPdsToDiscard))
	{
		ttlGap = currentTtl - currentPacketDescriptor->ttlCount;
		/* In case of wrap around the last bit is ignored */
		ttlGap &= HOST_INTERFACE_TTL_MASK;
		ILOG2_D("agerEmulatorProcessQueue NOT RETRANSMIT PDs TtlCount = %d", currentPacketDescriptor->ttlCount);
		if(ttlGap < ttlCriteria)
		{
			/* A new PD was found - no need to check the rest of the list */
			break;
		}
		else
		{
			popListTail = currentPacketDescriptor;
			numberOfDiscardedPds ++;
			if (currentPacketDescriptor->pdSource == HW_Q_MANAGER_DONE_LIST_HOST_DATA)
			{
				numberOfDiscardedDataPds ++;
			}
			AgerEmulatorParameters.numberOfDiscardedPdsHandledToLiberator ++;
             
			/* Currently all PDs with retransmission bit cleared are handled to liberator. In the HW Ager there is a possibility
			   to send a PD to the liberator or to UMAC according to the PD type. This option is currently not implemented since it 
			   is not used (the HW Ager is configured to send all PDs to the liberator )*/

			ILOG2_DD("agerEmulatorProcessQueue NOT RETRANSMIT DISCARD liberatorListHead = %x, liberatorListTail = %x", liberatorListHead, liberatorListTail);
			if(NULL == liberatorListHead)
			{
				/* Fisrt PD in the list */
				liberatorListHead = currentPacketDescriptor;
				liberatorListTail = currentPacketDescriptor;
			}
			else
			{
				liberatorListTail->nextPdPointer = SET_NEXT_PD(currentPacketDescriptor);
				liberatorListTail = currentPacketDescriptor;
			}	
		}
		currentPacketDescriptor->status = PD_STATUS_DISCARDED_BY_AGER;
		currentPacketDescriptor = (TxPd_t *)GET_NEXT_PD(currentPacketDescriptor);
	}
   
	if(0 != numberOfDiscardedPds)
	{
		/* If one of the list is not empty it singnifies that PDs were discarded and they should be popped out from the queue */
		hwQueueManagerRequestParams.pTailDesc = popListTail;
		
		HwQManager_PopListFromHead(&hwQueueManagerRequestParams);

		ILOG2_DDD("agerEmulatorProcessQueue pHeadDesc = %x , pTailDesc = %x , numberOfDiscardedPds = %d ", hwQueueManagerRequestParams.pHeadDesc, hwQueueManagerRequestParams.pTailDesc, numberOfDiscardedPds);

		if(AgerEmulatorParameters.maximumPdsDiscardedPerClient < numberOfDiscardedPds)
		{
			AgerEmulatorParameters.maximumPdsDiscardedPerClient = numberOfDiscardedPds;
			AgerEmulatorParameters.clientWithMaximumPdsDiscardedStationOrVapIndex = stationOrVapIndex;
			AgerEmulatorParameters.clientWithMaximumPdsDiscardedTidIndex = tid;
			AgerEmulatorParameters.isClientWithMaximumPdsDiscardedVap = isVap;
		}	

        /* Organize the PDs that will be retrun to the queue by thier sequence number - the Qos will be identical
           in wave500 and AR10, it will update the sequence number from the first PD in the queue */
        while(AgerEmulatorParameters.reorderingPacketDescriptorsBitmap[AGER_EMULATOR_FIRST_PART_OF_BITMAP])
        {
			pdIndex = Utils_FindLastSetAndClear(&AgerEmulatorParameters.reorderingPacketDescriptorsBitmap[AGER_EMULATOR_FIRST_PART_OF_BITMAP], 
												CONVERT_BYTES_TO_BIT_INDEX(sizeof(AgerEmulatorParameters.reorderingPacketDescriptorsBitmap[AGER_EMULATOR_FIRST_PART_OF_BITMAP])));
			currentPacketDescriptor = AgerEmulatorParameters.reorderingPacketDescriptorsArray[pdIndex];
			
			if(currentPacketDescriptor->aggregationIndication)
			{
				amsduTail = (TxPd_t *)CONVERT_OFFSET_TO_PD(currentPacketDescriptor->aMsduTailPointer);
			}
			else
			{
				amsduTail = currentPacketDescriptor; 
			}
	
			if(NULL == notDiscardedListHead)
			{
				/* Fisrt PD in the list */
				notDiscardedListHead = currentPacketDescriptor;
			}
			else
			{
				notDiscardedListTail->nextPdPointer = SET_NEXT_PD(currentPacketDescriptor);			
			}
			notDiscardedListTail = amsduTail;
			ILOG2_DDDD("agerEmulatorProcessQueue AGER_EMULATOR_FIRST_PART_OF_BITMAP pdIndex = %d , currentPacketDescriptor = %x , amsduTail = %x , notDiscardedListHead = %x", pdIndex, currentPacketDescriptor, amsduTail, notDiscardedListHead);
			ILOG2_D("agerEmulatorProcessQueue AGER_EMULATOR_FIRST_PART_OF_BITMAP notDiscardedListTail = %x", notDiscardedListTail);
        }
		while(AgerEmulatorParameters.reorderingPacketDescriptorsBitmap[AGER_EMULATOR_SECOND_PART_OF_BITMAP])
        {
			pdIndex = Utils_FindLastSetAndClear(&AgerEmulatorParameters.reorderingPacketDescriptorsBitmap[AGER_EMULATOR_SECOND_PART_OF_BITMAP], 
												CONVERT_BYTES_TO_BIT_INDEX(sizeof(AgerEmulatorParameters.reorderingPacketDescriptorsBitmap[AGER_EMULATOR_SECOND_PART_OF_BITMAP])));
			pdIndex += AGER_EMULATOR_WORD_SIZE_TO_ADD_TO_BIT_INDEX;  
			currentPacketDescriptor = AgerEmulatorParameters.reorderingPacketDescriptorsArray[pdIndex];

			if(currentPacketDescriptor->aggregationIndication)
			{
				amsduTail = (TxPd_t *)CONVERT_OFFSET_TO_PD(currentPacketDescriptor->aMsduTailPointer);
			}
			else
			{
				amsduTail = currentPacketDescriptor; 
			}
	
			if(NULL == notDiscardedListHead)
			{
				/* Fisrt PD in the list */
				notDiscardedListHead = currentPacketDescriptor;;
			}
			else
			{
				notDiscardedListTail->nextPdPointer = SET_NEXT_PD(currentPacketDescriptor);			
			}
			notDiscardedListTail = amsduTail;
			ILOG2_DDDD("agerEmulatorProcessQueue AGER_EMULATOR_FIRST_PART_OF_BITMAP pdIndex = %d , currentPacketDescriptor = %x , amsduTail = %x , notDiscardedListHead = %x", pdIndex, currentPacketDescriptor, amsduTail, notDiscardedListHead);
			ILOG2_D("agerEmulatorProcessQueue AGER_EMULATOR_FIRST_PART_OF_BITMAP notDiscardedListTail = %x", notDiscardedListTail);
        }	
   
		if(NULL != notDiscardedListHead)
		{
			/* Return the PDs that should not be discarded */

			hwQueueManagerRequestParams.pHeadDesc = notDiscardedListHead;
			hwQueueManagerRequestParams.pTailDesc = notDiscardedListTail;
			ILOG2_DD("agerEmulatorProcessQueue NOT DISCARDED LIST notDiscardedListHead = %x, notDiscardedListTail = %x", notDiscardedListHead, notDiscardedListTail);
			HwQManager_PushPacketlistToHead(&hwQueueManagerRequestParams);

		}

		hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
		if(NULL != liberatorListHead)
		{		
			hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_DONE_LIST_LIBERATOR;
			hwQueueManagerRequestParams.pHeadDesc = liberatorListHead;
			hwQueueManagerRequestParams.pTailDesc = liberatorListTail;
			ILOG2_DD("agerEmulatorProcessQueue LIBERATOR LIST liberatorListHead = %x, liberatorListTail = %x", liberatorListHead, liberatorListTail);
			HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams);
		}

		if(NULL != umacListHead)
		{		
			hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_LOW_PR_READY_LIST_AGER;
			hwQueueManagerRequestParams.pHeadDesc = umacListHead;
			hwQueueManagerRequestParams.pTailDesc = umacListTail;
			ILOG2_DD("agerEmulatorProcessQueue UMAC LIST umacListHead = %x, umacListTail = %x", umacListHead, umacListTail);
			HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams);
		}
        else
        {
			/* If there were no discarded PDs with retry bit the queue should be unlocked */
			Locker_UnLockPerTidQueues(queueType,stationOrVapIndex,(0x1 << tid));
        }

		/* Update status bitmap */
		if(isVap)
		{
			statisticsVapId = stationOrVapIndex;
			stationOrVapIndex += HW_NUM_OF_STATIONS;	
		}
		else
		{
			StaDb_GetVapId(stationOrVapIndex, &statisticsVapId);
		}
		
		// statistics update:  PD discarded by ager to prevent it from being sent
		pHostIfCounters->agerCount[statisticsVapId]+=statisticsVapId;
		
		locationInBitmap = CONVERT_BIT_INDEX_TO_WORDS(stationOrVapIndex);
		locationInWord = stationOrVapIndex & AGER_EMULATOR_BITMAP_MASK;   
		AgerEmulatorParameters.stationAndVapStatusBitmap[locationInBitmap] |= 0x1 << locationInWord;	
	}
	else
	{
		/* No PDs were discarded - unlock the queue */
		Locker_UnLockPerTidQueues(queueType,stationOrVapIndex,(0x1 << tid));
	}
	
	return numberOfDiscardedDataPds;
}


/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

AgerEmulator_TaskEntry 


Description:
------------
the entry point of the Ager emulator task

Input: 
-----
agerEmulatorMessage - pointer to the message to handle	

	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void AgerEmulator_TaskEntry(K_MSG *agerEmulatorMessage)
{
	/* Use common task switching and Table */
	vTaskDispatcher(agerEmulatorMessage, afpTaskTable, TASK_AGER_EMULATOR_START, TASK_AGER_EMULATOR_END);
}


/**********************************************************************************

AgerEmulator_ConfigureAllAndActivatePeriodicCheck 


Description:
------------
Configure all parameters and activate Ager emulator periodic check

Input:
-----
agerCheckParameters- a pointer to the structure that contains all the periodic check 
parameters


Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void AgerEmulator_ConfigureAllAndActivatePeriodicCheck(AgerPeriodicCheckParameters_t  *agerCheckParameters)
{
	K_MSG* AgerEmulatorMessage = NULL;
	AgerEmulatorPeriodicCheckMessageParameters_t *periodicCheckMessageParameters = NULL;
	
  	AgerEmulatorParameters.numberOfStopsRequested++;
	
	AgerEmulatorMessage = OSAL_GET_MESSAGE( sizeof(AgerEmulatorPeriodicCheckMessageParameters_t));
	periodicCheckMessageParameters = ((AgerEmulatorPeriodicCheckMessageParameters_t *)AgerEmulatorMessage->abData);
	periodicCheckMessageParameters->agerPeriod = agerCheckParameters->agerPeriod;

	/* Currently the arrays in the API structure and the Ager emulator array are the same - if they are changed
	   this code should be changed too */
	memcpy(periodicCheckMessageParameters->acTtlCriteria, 
	       agerCheckParameters->generalParameters.acTtlCriteria, 
	       sizeof(periodicCheckMessageParameters->acTtlCriteria));
	memcpy(periodicCheckMessageParameters->minimumPdsToStartCheckForAc, 
		   agerCheckParameters->generalParameters.minimumPdsToStartCheckForAc, 
		   sizeof(periodicCheckMessageParameters->minimumPdsToStartCheckForAc));
	//("AgerEmulator_ConfigureAllAndActivatePeriodicCheck ");
	//(0, 0, AgerPeriodicCheckParameters_t, &agerCheckParameters);

	OSAL_SEND_MESSAGE(AGER_EMULATOR_CONFIGURE_ALL_AND_ACTIVATE_PERIODIC_CHECK, TASK_AGER_EMULATOR, AgerEmulatorMessage, VAP_ID_DO_NOT_CARE);	
}

/**********************************************************************************

AgerEmulator_ReconfigurePartialPeriodicCheck 


Description:
------------
Reconfigure only the TTL criteria and the ager period and then activate the Ager 
emulator periodic check

Input:
-----
agerCheckParameters- a pointer to the structure that contains all the periodic check 
parameters


Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void AgerEmulator_ReconfigurePartialPeriodicCheck(AgerPeriodicCheckParameters_t  *agerCheckParameters)
{
	K_MSG* AgerEmulatorMessage = NULL;
	AgerEmulatorPeriodicCheckMessageParameters_t *periodicCheckMessageParameters = NULL;
	
  	AgerEmulatorParameters.numberOfStopsRequested++;
	
	AgerEmulatorMessage = OSAL_GET_MESSAGE( sizeof(AgerEmulatorPeriodicCheckMessageParameters_t));
	periodicCheckMessageParameters = ((AgerEmulatorPeriodicCheckMessageParameters_t *)AgerEmulatorMessage->abData);
	periodicCheckMessageParameters->agerPeriod = agerCheckParameters->agerPeriod;

	/* Currently the arrays in the API structure and the Ager emulator array are the same - if they are changed
	   this code should be changed too */
	memcpy(periodicCheckMessageParameters->acTtlCriteria, 
	       agerCheckParameters->generalParameters.acTtlCriteria, 
	       sizeof(periodicCheckMessageParameters->acTtlCriteria));

	//("AgerEmulator_ReconfigurePartialPeriodicCheck ");
	//(0, 0, AgerPeriodicCheckParameters_t, &agerCheckParameters);

	OSAL_SEND_MESSAGE(AGER_EMULATOR_CONFIGURE_PARTIALLY_AND_ACTIVATE_PERIODIC_CHECK, TASK_AGER_EMULATOR, AgerEmulatorMessage, VAP_ID_DO_NOT_CARE);	
}

/**********************************************************************************

AgerEmulator_StopPeriodicCheck 


Description:
------------
Stops Ager emulator periodic check

Input:
-----


Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void AgerEmulator_StopPeriodicCheck(void)
{
  	AgerEmulatorParameters.numberOfStopsRequested++;
	//("AgerEmulator_StopPeriodicCheck numberOfStopsRequested = %d", AgerEmulator_StopPeriodicCheck);

	OSAL_SEND_NO_DATA_MESSAGE(AGER_EMULATOR_STOP_PERIODIC_CHECK, TASK_AGER_EMULATOR, VAP_ID_DO_NOT_CARE);	
}

/**********************************************************************************

AgerEmulator_StartSingleCheck 


Description:
------------
Activates Ager emulator single check

Input:
-----
agerCheckParameters - a pointer to the structure that contains all the single check 
parameters


Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void AgerEmulator_StartSingleCheck(AgerSingleCheckParameters_t  *agerCheckParameters, bool restrictedAc)
{
	K_MSG* AgerEmulatorMessage = NULL;
	AgerEmulatorSingleCheckMessageParameters_t *singleCheckMessageParameters = NULL;
	
  	AgerEmulatorParameters.numberOfStopsRequested++;
	AgerEmulatorParameters.isStartedSingleCheck = FALSE;
	
	AgerEmulatorMessage = OSAL_GET_MESSAGE( sizeof(AgerEmulatorSingleCheckMessageParameters_t));
	singleCheckMessageParameters = ((AgerEmulatorSingleCheckMessageParameters_t *)AgerEmulatorMessage->abData);
	

	/* Currently the arrays in the API structure and the Ager emulator array are the same - if they are changed
	   this code should be changed too */
	memcpy(singleCheckMessageParameters->acTtlCriteria, 
	       agerCheckParameters->generalParameters.acTtlCriteria, 
	       sizeof(singleCheckMessageParameters->acTtlCriteria));
	memcpy(singleCheckMessageParameters->minimumPdsToStartCheckForAc, 
		   agerCheckParameters->generalParameters.minimumPdsToStartCheckForAc, 
		   sizeof(singleCheckMessageParameters->minimumPdsToStartCheckForAc));

	singleCheckMessageParameters->forceLock = (restrictedAc) ? 1 : agerCheckParameters->forceLock;
	singleCheckMessageParameters->isStartingQueueVap = agerCheckParameters->isStartingFromVap;
	singleCheckMessageParameters->stationOrVapIndexToStart = agerCheckParameters->stationOrVapIndexToStart;
	singleCheckMessageParameters->isMaximumPdsToDiscardEnable = agerCheckParameters->isEnabledMaximumLimit;
	singleCheckMessageParameters->maximumPdsToDiscard = agerCheckParameters->maximumPdsToDiscard;


	//("AgerEmulator_StartSingleCheck ");
	//(0, 0, AgerSingleCheckParameters_t, &agerCheckParameters);

	OSAL_SEND_MESSAGE(AGER_EMULATOR_START_SINGLE_CHECK, TASK_AGER_EMULATOR, AgerEmulatorMessage, VAP_ID_DO_NOT_CARE);	
}

/**********************************************************************************

AgerEmulator_GetLastCheckParameters 


Description:
------------
Returns the parameters of the last Ager emulator check (for single check)
The parameters in the structure should be used only if the check is done.

Input:
-----
lastCheckParameters - pointer to the strcutre to fill

		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
void AgerEmulator_GetLastCheckParameters(AgerLastCheckParametes_t *lastCheckParameters)
{
	lastCheckParameters->tid = AgerEmulatorParameters.lastQueueCheckedTidIndex;
	lastCheckParameters->isVapQueue = AgerEmulatorParameters.isLastQueueCheckedVap;
	lastCheckParameters->index = AgerEmulatorParameters.lastQueueCheckedStationOrVapIndex;
	/* Only if both bits are set it in dicates that the check is done. 
	   There are 2 bits instead of 1 for easier debug of the module */
	lastCheckParameters->isSingleCheckDone = (AgerEmulatorParameters.isSingleCheckDone && AgerEmulatorParameters.isStartedSingleCheck);
}

/**********************************************************************************

AgerEmulator_GetLastCheckCounters 


Description:
------------
Returns the counters of the last Ager emulator check (for single check)
The parameters in the structure should be used only if the check is done.

Input:
-----
lastCheckCounters - pointer to the strcutre to fill

		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
void AgerEmulator_GetLastCheckCounters(AgerLastCheckCounters_t *lastCheckCounters)
{
	lastCheckCounters->numberOfLockFailures = AgerEmulatorParameters.numberOfQueuesSkippedDueToLockFailure;
	lastCheckCounters->numberOfPdsHandledToLiberator = AgerEmulatorParameters.numberOfDiscardedPdsHandledToLiberator;
	lastCheckCounters->numberOfPdsHandledToUmac = AgerEmulatorParameters.numberOfDiscardedPdsHandledToUmac;
}


/**********************************************************************************

AgerEmulator_IncreasePriority 


Description:
------------
This function increase the priority of the Ager emulator thread


Input:
-----


Output:
-------

	

Returns:
--------

	
**********************************************************************************/
void AgerEmulator_IncreasePriority(void)
{
	TxApp_ChangeThreadPriority(TASK_AGER_EMULATOR, TX_APP_HIGH_PRIORITY_AGER_EMULATOR);
}


/**********************************************************************************

AgerEmulator_Initialize 


Description:
------------
initialize internal structures and parameters of the Ager emulator

Input:
-----
	
		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
void AgerEmulator_Initialize(void)
{
	memset32(&AgerEmulatorParameters, 0, CONVERT_BYTES_TO_WORDS(sizeof(AgerEmulatorParameters_t)));
}

#endif //#ifdef USE_AGER_EMULATOR



