/***********************************************************************************
 File:			Atf.c
 Module:		Air Time Fairness
 Purpose: 		
 Description:	
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/

#include "loggerAPI.h"
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"	
#include "loggroups.h"
#include "HwGlobalDefinitions.h"
#include "ShramAtf.h"
#include "TxSender_AtfApi.h"
#include "Atf.h"
#include "Atf_Api.h"


#include "HostInterface_API.h"
#include "RegAccess_Api.h"
#include "TpcClbrHndlr.h"
#include "StaDatabase_Api.h"
#include "stringLibApi.h"
#include "linkAdaptation_api.h"
#include "AggregationBuilder_Api.h"
#include "PacketDescriptor.h"
#include "ShramPacketDescriptors.h"
#include "HostInterfaceAcc_Api.h"
#include "Locker_Api.h"
#include "HwQManager_API.h"
#include "TxSender_MailboxApi.h"
#include "HostInterface_API.h"
#include "Pac_Api.h"
#include "BSSmanager_API.h"
#include "StatisticsManager_api.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_ATF
#define LOG_LOCAL_FID 0
#define ATF_MANAGER_TRANSITION(x) Atf_Manager_transition(x, (AtmAlgorithmType_e) 0,(AtmWeightedType_e) 0, NULL, 0, 0)

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
#define ATF_DEBUG_MAX_NUMBER_OF_ERRORS 20


/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/ 
GRANT_SIZE findHighest(GRANT_SIZE A[HW_NUM_OF_STATIONS], uint8 n,uint8 dim2Indx);


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
static const FunctionEntry_t afpTaskTable[TASK_ATF_END - TASK_ATF_START]=
{
    {Atf_Manager_TransitionReqMsg,          DOUBLE_CHECK_MSG_TYPE(ATF_MANAGER_START_REQ)},
    {Atf_Buffering_lockCfmMsg,              DOUBLE_CHECK_MSG_TYPE(ATF_BUFFERING_LOCK_CFM)},
    {Atf_Buffering_himBufferCfmMsg,         DOUBLE_CHECK_MSG_TYPE(ATF_BUFFERING_HIM_BUFFER_CFM)},
    {Atf_Buffering_dropAllBuffered,         DOUBLE_CHECK_MSG_TYPE(ATF_BUFFERING_DROP_ALL_BUFFERED)},
    {Atf_Buffering_setTtlCriteria,          DOUBLE_CHECK_MSG_TYPE(ATF_BUFFERING_QOS_TTL_CRITERIA)},
    {Atf_Buffering_timerExpiredMsg,         DOUBLE_CHECK_MSG_TYPE(ATF_BUFFERING_TIMER_EXPIRED)},
    {Atf_Buffering_senderStopTxMsg,         DOUBLE_CHECK_MSG_TYPE(ATF_BUFFERING_SENDER_INDICATION_STOPTX)},
    {Atf_Manager_stopTrafficMsg,            DOUBLE_CHECK_MSG_TYPE(ATF_MANAGER_BSS_STOP_TRAFFIC)},
    {Atf_Manager_AddStaMsg,                 DOUBLE_CHECK_MSG_TYPE(ATF_MANAGER_BSS_ADD_STA)},        
};


/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/
AtfDb_t   AtfDb;    // ATF Manager database
AtfStationsDb_t AtfStationsDb; // ATF stations database

AtfSm_t AtfBufferingSm[ATF_STATION_STATIC_STATE_NUM_OF_STATES];						// static mode: buffering state machine
AtfBufferedList_t AtfBufferedList[HW_NUM_OF_STATIONS][NUM_OF_TID];						// static mode: buffering list head and tail
#ifdef ATF_DEBUG_LOCAL
uint8 AtfErrors[ATF_NUM_ERROR_CODES]; // when error code is returned it stored in this array
#endif //#ifdef ATF_DEBUG_LOCAL

AtfShramDbStatic_t *AtfShramDbStatic_p = (AtfShramDbStatic_t*)&AtfShramDb; // modifyied for Dynamic (per Station) Sharedram grants status for each station
AtfShramDbDynamic_t *AtfShramDbDynamic_p = (AtfShramDbDynamic_t*)&AtfShramDb; // modifyied for Dynamic (per Station AC) Sharedram grants status for each station
AtfShramDb_t *AtfShramDbGeneral_p = &AtfShramDb; // sharedram grants status for each station




/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

Atf_TaskEntry 


Description:
------------
the entry point of the Atf task

Input: 
-----
psMsg - pointer to the message to handle   

    
        
Output:
-------


Returns:
--------
    void - 
    
**********************************************************************************/
void Atf_TaskEntry(K_MSG *psMsg)
{
    /* Use common task switching and Table */
    vTaskDispatcher(psMsg, afpTaskTable, TASK_ATF_START, TASK_ATF_END);
}

/**********************************************************************************

filterManagments



Description:
------------
    
    
Input:
-----
Parameters head, tail are the queue original pointers.
Mng_head, mng_tail are modified to point the first mng and last mng in queue.
head modified to point to the first data PD.

Output:
----- 
Function return 1 if mng PD exist else 0.


**********************************************************************************/
bool filterManagmentsPd(HwQueueManagerRequestParams_t *hwQueueManagerRequestParams, HwQueueManagerRequestParams_t *hwQueueManagerRequestParamsMng)
{
    bool result = FALSE;
    TxPd_t *pd = (TxPd_t*)(hwQueueManagerRequestParams->pHeadDesc);
    
    hwQueueManagerRequestParamsMng->pTailDesc = NULL_PD;

    // 
    while ((pd != (TxPd_t *)NULL_PD) && (pd->pdType != PD_TYPE_DATA && pd->pdType != PD_TYPE_DATA_RESERVED))
    {
        hwQueueManagerRequestParamsMng->pTailDesc = pd;
        pd  = (TxPd_t *)GET_NEXT_PD(pd);
    }
    
    if (hwQueueManagerRequestParamsMng->pTailDesc == NULL_PD)
    {
        result =  FALSE;
    }
    else
    {
        hwQueueManagerRequestParamsMng->pHeadDesc = hwQueueManagerRequestParams->pHeadDesc;
        hwQueueManagerRequestParams->pHeadDesc = pd;
        
        if (pd == (TxPd_t *)NULL_PD)
            hwQueueManagerRequestParams->pTailDesc = NULL_PD;
        
        result =  TRUE;
    }

    return result;
    
}

/**********************************************************************************

filterAgedPd



Description:
------------
    
    
Input:
-----
Parameters head, tail are the queue original pointers.
aged_head, aged_tail are modified to point the head and last aged pd if aged exist.
head modified to point to the first data PD not aged.
tail modified to point to the last data PD not aged.

Output:
----- 
Function return 1 if mng PD exist else 0.


**********************************************************************************/
bool filterAgedPd(HwQueueManagerRequestParams_t *hwQueueManagerRequestParams, HwQueueManagerRequestParams_t *hwQueueManagerRequestParamsAged)
{
    bool result = FALSE;
    TxPd_t *pd;
	uint32	count = 0;

    pd = hwQueueManagerRequestParams->pHeadDesc;
    hwQueueManagerRequestParamsAged->pHeadDesc = hwQueueManagerRequestParams->pHeadDesc;
    hwQueueManagerRequestParamsAged->pTailDesc = NULL_PD;

    if (AtfDb.AgeingEnable == TRUE)
    {
        while ( (pd != (TxPd_t *)NULL_PD) &&
            (Atf_Manager_checkAgeingCondition(pd->ttlCount) == TRUE) )
        {
			/* PD is about to be dropped: set status as NACK */
			pd->status = PD_STATUS_NACK;
			
            // if AMSDU need to drop all list, it can't be broken.
            if(pd->aggregationIndication)
            {
				/* Accumulate entire Aggregation */
				count += pd->pdCounter;
				
                /* A-MSDU - discard the whole A-MSDU. There shouldnt be A-MSDU of 1 MSDU */
                pd = (TxPd_t *)CONVERT_OFFSET_TO_PD(pd->aMsduTailPointer);
            }
			else
			{
				count += 1;
			}
            hwQueueManagerRequestParamsAged->pTailDesc = pd; /*Do this here - in case of AMSDU pd is the last pd in the AMSDU*/
            pd = (TxPd_t *)GET_NEXT_PD(pd);
            result = TRUE;
        }
        // if pd is null than head shall be null,
        hwQueueManagerRequestParams->pHeadDesc = pd;

        //if tail same as aged_tail tail shall point to NULL
        if (hwQueueManagerRequestParams->pTailDesc == hwQueueManagerRequestParamsAged->pTailDesc)
        {
            hwQueueManagerRequestParams->pTailDesc =  NULL_PD;
        }

		if (count > 0)
		{	
			GeneralStatistics.dropReasonATF+=count;
		}

        // ** disconnect the tail pd of the aged list from the original list is done by HW
    }

    return result;
}

#ifdef ATF_DEBUG_GLOBAL
/**********************************************************************************

PrintGrants



Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
void printGrants(GRANT_SIZE A[HW_NUM_OF_STATIONS])
{
    uint8 i = 0;
 
    for (i = 0; i < HW_NUM_OF_STATIONS; i++)
    {
        if (A[i] != 0)
            ILOG0_DD("[printGrants] 
stationID:%d grant:%d",i,A[i]);
    }
}

#ifdef ATF_DEBUG_LOCAL
/**********************************************************************************

PrintGrants



Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
void printTxopTimeDynamic(TXOP_LIMIT_SIZE_DYNAMIC A[HW_NUM_OF_STATIONS][ACCESS_CATEGORY_NUM])
{
    uint8 i = 0;
 
    for (i = 0; i < HW_NUM_OF_STATIONS; i++)
    {
        if (A[i][0] != 0)
            ILOG0_DD("[printGrants] stationID:%d grant:%d",i,A[i][0]);
    }
}

/**********************************************************************************

printTxopTimeStatic



Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
void printTxopTimeStatic(TXOP_LIMIT_SIZE_STATIC A[HW_NUM_OF_STATIONS])
{
    uint8 i = 0;
 
    for (i = 0; i < HW_NUM_OF_STATIONS; i++)
    {
        if (A[i] != 0)
            ILOG0_DD("[printGrants] stationID:%d grant:%d",i,A[i]);
    }
}


/**********************************************************************************

printPdTypeList



Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
void printPdTypeList(HwQueueManagerRequestParams_t *hwQueueManagerRequestParams)
{
    TxPd_t *head = hwQueueManagerRequestParams->pHeadDesc;
    TxPd_t *tail = hwQueueManagerRequestParams->pTailDesc;
    
    if (head != (TxPd_t *)NULL_PD) //while (head != (PacketDescriptor_t *)NULL_PD)
    {

        ILOG0_DDD("[printPdTypeList] PdType:%d head:%x; tail:%x",(TxPd_t *)head->pdType, (uint16)CONVERT_PD_TO_OFFSET(head), (uint16)CONVERT_PD_TO_OFFSET(tail));

        head = (TxPd_t *)CONVERT_OFFSET_TO_PD(head->NextPd);
    }
}


/**********************************************************************************

PrintGrants



Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
void printPdTypeListAll(HwQueueManagerRequestParams_t *hwQueueManagerRequestParams)
{
    TxPd_t *head = hwQueueManagerRequestParams->pHeadDesc;
    while (head != (TxPd_t *)NULL_PD)
    {
#ifdef ATF_DEBUG_LOCAL        
        //ILOG0_DD("[printPdTypeListAll] PdType:%d head:%x",(PacketDescriptor_t *)head->PdType, (uint16)CONVERT_PD_TO_OFFSET(head));
#endif

        head = (TxPd_t *)CONVERT_OFFSET_TO_PD(head->NextPd);
    }
}
#endif

/**********************************************************************************

findHighest



Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
GRANT_SIZE findLowest(GRANT_SIZE A[HW_NUM_OF_STATIONS], uint8 n,uint8 dim2Indx)
{
    GRANT_SIZE min = 0;
    uint8 i = 0;
    
    min = A[0];    
    for (i = 1; i < n; i++)
    {
        if (A[i] > min)
        {
           min = A[i];
        }
    }
    return min;
}
#endif //ATF_DEBUG_LOCAL

/**********************************************************************************

findHighest



Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
GRANT_SIZE findHighest(GRANT_SIZE A[HW_NUM_OF_STATIONS], uint8 n,uint8 dim2Indx)
{
    GRANT_SIZE max = 0;
    uint8 i = 0;
    
    max = A[0];    
    for (i = 1; i < n; i++)
    {
        if (A[i] > max)
        {
           max = A[i];
        }
    }
    return max;
}

/*  Function to add an element to the stack */
void Atf_Manager_pushToStack(StaId stationId)
{
    if (AtfDb.lockStackTop == (ATF_MAX_LOCK_ELEMENTS - 1))
    {
        ASSERT(0); // stack is full, future work: istead of assert resend lock req msg
    }
    else
    {
        AtfDb.lockStackTop++;
        AtfDb.lockStack[AtfDb.lockStackTop] = stationId;
    }
    return;
}

/*  Function to delete an element from the lock stack (LIFO)*/
StaId popFromStack()
{
    StaId stationId;
    if (AtfDb.lockStackTop == ATF_INVALID_LOCK_IDX) //Stack is Empty
    {
        stationId = INVALID_STA_INDEX;
    }
    else // pop next element
    {
        stationId = AtfDb.lockStack[AtfDb.lockStackTop];
        AtfDb.lockStackTop--;
        // if AtfDb.lockStackTop == 0 then AtfDb.lockStackTop = -1 (ATF_INVALID_LOCK_IDX)
    }
    return(stationId);
}

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

Atf_Manager_transition




Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
void Atf_Manager_TransitionReqMsg(K_MSG *psMsg)
{
 
    UMI_ATF_QUOTAS*atfConfig_p = (UMI_ATF_QUOTAS *)pK_MSG_DATA(psMsg);
    // Save pointer to msg in DB
    AtfDb.psMsg = psMsg;

    // If DistType not Disable we need to disable it first, to eliminate influence of last ATM Types and stations states.
    if (AtfDb.AtmDistributionType != ATM_DISTRIBUTION_TYPE_DISABLED)
    {
        ATF_MANAGER_TRANSITION(ATM_DISTRIBUTION_TYPE_DISABLED);
    }
    
    // check that no station in ATF is active so we can set new setting, otherwise wait to be triggered again from last station set to idle.
    if (AtfDb.TotalStationsAtfActive == 0) 
    {
        // disabling is done
        AtfDb.duringDisabling = FALSE;
        
        // Move to Types according to Driver parameters.
        atfConfig_p->u8error_code = Atf_Manager_transition((AtmDistributionType_e)atfConfig_p->AtmDistributionType[0], (AtmAlgorithmType_e)atfConfig_p->u8AtmAlgorithemType, 
                                                                       (AtmWeightedType_e)atfConfig_p->u8AtmWeightedType ,atfConfig_p->stationGrant, atfConfig_p->u32interval, atfConfig_p->u32freeTime);
        // send confiramtion
        OSAL_SEND_MESSAGE(UMI_MC_MAN_ATF_QUOTAS_CFM, TASK_UM_IF_TASK, psMsg, VAP_ID_DO_NOT_CARE);
    }
    
}

/**********************************************************************************

Atf_Manager_setAgeing




Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
void Atf_Manager_setAgeing(uint32 interval)
{
    uint32 interval_us = interval*1000;
  	uint8 end_of_interval_ttl;

    if (HOST_INTERFACE_ACCELERATOR_CONVERT_USEC_TO_TTL_UNITS(interval_us) > AtfDb.AgeingTtlCriteria)
    {
        AtfDb.AgeingEnable = TRUE; 
        // calculate next interval ttl
        end_of_interval_ttl = HOST_INTERFACE_ACCELERATOR_CONVERT_USEC_TO_TTL_UNITS(GET_TSF_TIMER_LOW() + interval_us) + 1; // get ttl at interval end time,  We add 1 for a msg delay to HIM
        end_of_interval_ttl &= HOST_INTERFACE_TTL_MASK; // The TTL is 7 bits long         
        AtfDb.end_of_interval_ttl = end_of_interval_ttl;
    }
    else
    {
        AtfDb.AgeingEnable = FALSE;
    }
    
#ifdef ATF_DEBUG_LOCAL     
    ILOG0_DDDD("Atf_Manager_setAgeing AtfDb.AgeingEnable:%d, AtfDb.AgeingTtlCriteria:%d, AtfDb.end_of_interval_ttl:%d end_of_interval_ttl:%d", AtfDb.AgeingEnable, AtfDb.AgeingTtlCriteria , AtfDb.end_of_interval_ttl, end_of_interval_ttl);
#endif    
}

/**********************************************************************************

Atf_Manager_checkAgeingCondition



Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
bool Atf_Manager_checkAgeingCondition(uint8 ttlCount)
{
    uint8 ttlGap;
#ifdef ATF_DEBUG_LOCAL 
    //ILOG0_DD("AtfDb.end_of_interval_ttl %d ttlCount %d",AtfDb.end_of_interval_ttl, ttlCount);
#endif

    ttlGap = (AtfDb.end_of_interval_ttl - ttlCount);
    /* In case of wrap around the last bit is ignored */
    ttlGap &= HOST_INTERFACE_TTL_MASK;

    // if condition is TRUE that PD will be aged. since gap of time is higher than criteria
    return (ttlGap > AtfDb.AgeingTtlCriteria);
}


/**********************************************************************************

Atf_Manager_sendQosMsg




Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
void Atf_Manager_sendQosMsg(AtmDistributionType_e newAtmDistributionType)
{
    K_MSG* Atf_Qos_msg_p;
    uint8 *Atf_Qos_msg_data_p;
    
    Atf_Qos_msg_p = OSAL_GET_MESSAGE(sizeof(uint8));
    Atf_Qos_msg_data_p = (uint8*)pK_MSG_DATA(Atf_Qos_msg_p);
    *Atf_Qos_msg_data_p = (newAtmDistributionType == ATM_DISTRIBUTION_TYPE_STATIC);

    OSAL_SEND_MESSAGE(QOS_SET_IS_ATF_STATIC, TASK_QOS, Atf_Qos_msg_p, VAP_ID_DO_NOT_CARE);
}

/**********************************************************************************

Atf_Manager_sendHimMsg


Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
void Atf_Manager_sendHimMsg()
{
    K_MSG* Atf_him_msg_p;
    AtfAgeingMsg_t *Atf_him_msg_data_p;

    // allocate msg
    Atf_him_msg_p = OSAL_GET_MESSAGE(sizeof(AtfAgeingMsg_t));
    Atf_him_msg_data_p = (AtfAgeingMsg_t*)pK_MSG_DATA(Atf_him_msg_p);
    
    // fill msg data
    Atf_him_msg_data_p->AgeingEnable = AtfDb.AgeingEnable;
    Atf_him_msg_data_p->AgeingTtlCriteria = AtfDb.AgeingTtlCriteria;
    Atf_him_msg_data_p->end_of_interval_ttl = AtfDb.end_of_interval_ttl;    

    OSAL_SEND_MESSAGE(HIM_RINGS_ATF_AGEING, TASK_HIM, Atf_him_msg_p, VAP_ID_DO_NOT_CARE);
}


/**********************************************************************************

Atf_Manager_transition




Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
AtfErrorCode_e Atf_Manager_transition(AtmDistributionType_e newAtmDistributionType,
    AtmAlgorithmType_e newAtmAlgorithmType, AtmWeightedType_e newAtmWeightedType, GRANT_SIZE grants[HW_NUM_OF_STATIONS], uint32 interval, uint32 freeTime)
{
    AtfErrorCode_e checkParametersValidityResults = ATF_NO_ERRORS;

#ifdef ATF_DEBUG_GLOBAL   
    ILOG0_D("[Atf_Manager_transition] newAtmDistributionType:%d",newAtmDistributionType);
    ILOG0_DDDD("[Atf_Manager_transition] AtfDb.AtmDistributionType: %d; newAtmAlgorithmType: %d interval:%d freeTime:%d",AtfDb.AtmDistributionType, newAtmAlgorithmType, interval, freeTime);
    if (newAtmDistributionType != ATM_DISTRIBUTION_TYPE_DISABLED && newAtmAlgorithmType != ATM_ALGORITHM_TYPE_GLOBAL)
        printGrants(grants);
#endif //#ifdef ATF_DEBUG_LOCAL


    switch ( newAtmDistributionType ) 
    {
    case ATM_DISTRIBUTION_TYPE_DISABLED:
        Atf_Manager_disable();
        break;
    case ATM_DISTRIBUTION_TYPE_DYNAMIC:
        // Check configure validty according to rules and current system state
        checkParametersValidityResults = Atf_Manager_checkParametersValidityDynamic(newAtmAlgorithmType, grants, interval);

        if (checkParametersValidityResults == ATF_NO_ERRORS)
        {
            switch ( newAtmAlgorithmType ) 
            {
            case ATM_ALGORITHM_TYPE_GLOBAL:
                Atf_GrantCalc_dynamicGlobal(interval);
                break;
            case ATM_ALGORITHM_TYPE_WEIGHTED:
                Atf_GrantCalc_dynamicWeightedCommon();
                Atf_GrantCalc_dynamicWeighted(grants);
                break;
            default:
                ASSERT(0);
                break;
            }
        }
        break;
    case ATM_DISTRIBUTION_TYPE_STATIC:
        // Check configure validty according to rules and current system state
        checkParametersValidityResults = Atf_Manager_checkParametersValidityStatic(newAtmAlgorithmType, grants, interval);

        if (checkParametersValidityResults == ATF_NO_ERRORS)
        {
            switch ( newAtmAlgorithmType ) 
            {
            case ATM_ALGORITHM_TYPE_GLOBAL:
            case ATM_ALGORITHM_TYPE_WEIGHTED:  
                Atf_GrantCalc_staticWeighted(grants, interval, freeTime);
                break;
            default:
                ASSERT(0);
                break;
            }
            
            Atf_Manager_setAgeing(interval);
            Atf_Manager_sendHimMsg();
            AtfDb.interval = interval;
            
            if (AtfDb.TotalStationsAtfActive > 0)
            {
                OSAL_SET_TIMER_EXPLICIT(ATF_BUFFERING_TIMER_EXPIRED, OSAL_TIMERS_MS_TO_K_TICKS(interval), TASK_ATF);  
            }
        }
        break;
    default:
      ASSERT(0);
      break;
    }  

    // common
    if (checkParametersValidityResults == ATF_NO_ERRORS)
    {
        ILOG0_V("[Atf_Manager_transition] NO_ERRORS");
        AtfDb.AtmDistributionType = newAtmDistributionType;
        AtfShramDbGeneral_p->atfDistType = newAtmDistributionType; 
        Atf_Manager_sendQosMsg(newAtmDistributionType);
    }
    else
    {
 //       ILOG0_D("[Atf_Manager_transition] Error %d", checkParametersValidityResults);                              
#ifdef ATF_DEBUG_LOCAL
        AtfErrors[checkParametersValidityResults]++;
#endif
        /*Clear Total Stations Active as it may be increased before error is detected*/
        AtfDb.TotalStationsAtfActive = 0;
        ASSERT(0);
    }

    return checkParametersValidityResults;
}


/**********************************************************************************

Atf_Manager_ValidateStationConnected


Description:
------------
    Check that each grant has a station connected
    Check that each Total stations connected equal number of given grants 
    
Input:
-----
    

**********************************************************************************/
AtfErrorCode_e Atf_Manager_ValidateStationConnected(GRANT_SIZE grants[HW_NUM_OF_STATIONS])
{
   StaId stationId = MAX_UINT8; // need to set to -1, since Atf_Manager_GetNextActiveStation will increase 1
   uint8 TotalStationsAtfConnected = 0;
   uint8 TotalStationsAtfGrant = 0;
   
   // run over all connected station
   while(Atf_Manager_GetNextStationBitmap(&stationId, &AtfDb.AtfStationsConnected) == TRUE)   
   {
        TotalStationsAtfConnected++; // total number of connected stations by add station / stop traffic 
        if (grants[stationId]  > 0) // Station is connected by the grants list
        {
            Atf_Manager_increaseTotalStations(stationId);
        }
        else
        {
  //          ILOG0_D("[Atf_Manager_ValidateStationConnected] ATF_NOT_ALL_CONNECTED_STATIONS_RECEIVED_GRANT stationId:%d", stationId);
            return ATF_NOT_ALL_CONNECTED_STATIONS_RECEIVED_GRANT;
        }
   }

   // count the number of granted stations
   for (stationId = 0; stationId < HW_NUM_OF_STATIONS; stationId++) 
   {
        if (grants[stationId]  > 0) // Station is connected by the grants list
        {
            TotalStationsAtfGrant++;
        }
   }

   // total number of stations with a given grant need to be equal to total number of stations connected
   // if not, than it is only possible that there are more grants than connected stations 
   // because the first check for ATF_NOT_ALL_CONNECTED_STATIONS_RECEIVED_GRANT already assured that for every station connected we have a grant

   if (TotalStationsAtfConnected != TotalStationsAtfGrant)
   {
//        ILOG0_D("[Atf_Manager_ValidateStationConnected] ATF_STATION_IN_GRANT_LIST_IS_NOT_CONNECTED stationId:%d", stationId);
        return ATF_STATION_IN_GRANT_LIST_IS_NOT_CONNECTED;
   }
    
   return ATF_NO_ERRORS;

}

/**********************************************************************************

Atf_Manager_checkParametersValidityDynamic



Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
AtfErrorCode_e Atf_Manager_checkParametersValidityDynamic(AtmAlgorithmType_e newAtmAlgorithmType, GRANT_SIZE grants[HW_NUM_OF_STATIONS], uint32 interval)
{
    AtfErrorCode_e errorCode = ATF_NO_ERRORS;
    StaId stationId = MAX_UINT8; // Atf_Manager_GetNextStationBitmap will increase 1

    errorCode =  Atf_Manager_ValidateStationConnected(grants);

    if (errorCode == ATF_NO_ERRORS)
    {
        //check all stations configured to use multiple data (in DS)
        while((errorCode == ATF_NO_ERRORS) && (Atf_Manager_GetNextStationBitmap(&stationId, &AtfDb.AtfStationsActive) == TRUE))
        {
            // as long as errorCode == NO_ERRORS, check next station
            if (StaDb_getTxopMode(stationId) == UMI_TXOP_MODE_DISABLED)
            {
                errorCode = ATF_MULTIPLE_DATA_IS_DISABLED;
            }
        }
    }

   return errorCode;
}

/**********************************************************************************

Atf_Manager_checkParametersValidityStatic



Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
AtfErrorCode_e Atf_Manager_checkParametersValidityStatic(AtmAlgorithmType_e newAtmAlgorithmType, GRANT_SIZE grants[HW_NUM_OF_STATIONS], uint32 interval)
{
    AtfErrorCode_e errorCode = ATF_NO_ERRORS;
    
    errorCode =  Atf_Manager_ValidateStationConnected(grants);

    //
    if (errorCode == ATF_NO_ERRORS)
    {
        // Check interval is in range
        if ((interval < ATF_MIN_INTERVAL_TIME_MS) || (interval > ATF_MAX_INTERVAL_TIME_MS))
        {
            errorCode = ATF_INPUT_IS_INVALID;
        }
    }
    
    return errorCode;
    //return INPUT_IS_INVALID    
}

/**********************************************************************************

Atf_Manager_GetNextStationBitmap



Description:
------------
    fill the stationId for first active station
    if found clear bitmap for this station in given bitmap
    
Input:
-----

Output:
-----
    if any station in bitmap exist return TRUE, else FALSE

**********************************************************************************/
bool Atf_Manager_GetNextStationBitmap(StaId *stationId_p, AtfStationBitmap_t *AtfStations)
{
    StaId stationId = MAX_UINT8;

    // if the station id is higher than max supported
    if ((*stationId_p+1) ==  ATF_MAX_OF_SUPPORTED_STATIONS)
    {
        return FALSE;
    }
    
    // get first lsb, if bitmap is empty it return ATF_MAX_OF_SUPPORTED_STATIONS
    stationId = Utils_CountTrailZeroBitmap(AtfStations->staBitMap, *stationId_p+1, ATF_MAX_OF_SUPPORTED_STATIONS_BITMAP_SIZE);
    
    // if bitmap is not empty
    if (stationId != ATF_MAX_OF_SUPPORTED_STATIONS)
    {
        *stationId_p = stationId; // copy value to caller function
        return TRUE;
    }
    else
    {
        return FALSE;
    }
}

void Atf_Buffering_timerExpiredMsg(K_MSG *psMsg)
{
    StaId stationId = MAX_UINT8; // need to set to -1, since Atf_Manager_GetNextStationBitmap will increase 1

    // reset the end_of_interval_TSF
    if (AtfDb.AgeingEnable == TRUE)
    {
        AtfDb.end_of_interval_ttl = (HOST_INTERFACE_ACCELERATOR_CONVERT_USEC_TO_TTL_UNITS(GET_TSF_TIMER_LOW() + AtfDb.interval*1000) + 1) & HOST_INTERFACE_TTL_MASK; // We add 1 for a msg delay to HIM
        Atf_Manager_sendHimMsg();
    }
    
#ifdef ATF_DEBUG_GLOBAL     
//    ILOG0_V("[Atf_Buffering_timerExpiredMsg]");
#endif
    
    while(Atf_Manager_GetNextStationBitmap(&stationId, &AtfDb.AtfStationsActive) == TRUE)
    {        
        // unbuffer all "Buffered" stations, which mean, that are in the proccess of buffering.
        // Can be any state beside unbuffered and IDLE.
        AtfShramDbStatic_p->currGrant[stationId] = GRANT_DEFAULT; // update shared ram   
        Atf_buffering_StaSmRun(ATF_STATION_EVENT_TIMER_EXPIRED, stationId, (void*)stationId);
    }
    OSAL_SET_TIMER_EXPLICIT(ATF_BUFFERING_TIMER_EXPIRED, OSAL_TIMERS_MS_TO_K_TICKS(AtfDb.interval), TASK_ATF);
}
    
/**********************************************************************************

Atf_Manager_disable




Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
void Atf_Manager_disable(void)
{
    StaId stationId = MAX_UINT8; // need to set to -1, since Atf_Manager_GetNextStationBitmap will increase 1
    
    // indicate that disabling is in process, it shall only finish when all station are idle.
    AtfDb.duringDisabling = TRUE;
    
    // Run according to --Previous-- Atm Distribution Type
    switch ( AtfDb.AtmDistributionType ) 
    {
    case ATM_DISTRIBUTION_TYPE_DISABLED:
        // Do nothing
        break;
    case ATM_DISTRIBUTION_TYPE_DYNAMIC:
        Atf_GrantCalc_updateTxopTimeThreshold(AGGREGATION_BUILDER_TXOP_AIR_TIME_DURATION_LIMIT);
        while(Atf_Manager_GetNextStationBitmap(&stationId, &AtfDb.AtfStationsActive) == TRUE)
        {   
            // Change state to ATF_STATION_STATE_IDLE for all stations when ATF is disabled
            StaDb_SetPsduTimeLimit(stationId, MAX_PSDU_TX_TIME_LIMIT); // currently AC0 will take effect on all TID, assumig all AC are equal in driver msg
            Atf_SetStateIdle((void*)stationId);
        } 
        break; 
    case ATM_DISTRIBUTION_TYPE_STATIC:
        // stop the timer
        OSAL_RESET_TIMER_EXPLICIT(ATF_BUFFERING_TIMER_EXPIRED, TASK_ATF);
        while(Atf_Manager_GetNextStationBitmap(&stationId, &AtfDb.AtfStationsActive) == TRUE)
        {   
            /*   - station may be in wait mode, so we need to wait for it before finishing disabling of ATF.
                 - all during buffering queue should be unbuffered */
            Atf_buffering_StaSmRun(ATF_STATION_EVENT_STOP_TRAFFIC, stationId, (void*)stationId);
        }
        break;
    }
    // reset shram all grants to 0, ATF mode ATM_DISTRIBUTION_TYPE_DISABLED
    memset(AtfShramDbGeneral_p, 0, sizeof(AtfShramDb_t));
    
}


/**********************************************************************************

Atf_Buffering_setTtlCriteria



Description:
------------

    
Input:
-----
    

**********************************************************************************/
void Atf_Buffering_setTtlCriteria(K_MSG *psMsg)
{
    uint8 newTtlCriteria;
    
  	newTtlCriteria = (uint8)(*(pK_MSG_DATA(psMsg)));
    AtfDb.AgeingTtlCriteria = newTtlCriteria;
    Atf_Manager_setAgeing(AtfDb.interval);
    Atf_Manager_sendHimMsg();
}

/**********************************************************************************

Atf_Manager_decreseTotalStations



Description:
------------

    
Input:
-----
    

**********************************************************************************/
void Atf_Manager_decreseTotalStations(StaId stationId)
{
    AtfDb.TotalStationsAtfActive--;
    ATF_CLR_BIT_IN_BITMAP(AtfDb.AtfStationsActive.staBitMap, stationId); // set the station in the bitmap
    ASSERT(AtfDb.TotalStationsAtfActive >= 0);
}

/**********************************************************************************

Atf_Manager_increaseTotalStations


Description:
------------

    
Input:
-----
    

**********************************************************************************/
void Atf_Manager_increaseTotalStations(StaId stationId)
{
    AtfDb.TotalStationsAtfActive++; // increase the counter
    ATF_SET_BIT_IN_BITMAP(AtfDb.AtfStationsActive.staBitMap, stationId); // set the station in the bitmap
    ASSERT(AtfDb.TotalStationsAtfActive <= ATF_MAX_OF_SUPPORTED_STATIONS);
}

/**********************************************************************************

ATF_Manager_StaManagerSendConfirm




Description:
------------
    Stop station traffic
    
Input:
-----
    

**********************************************************************************/
static void Atf_Manager_StaManagerSendConfirm(StaId sid)
{
	K_MSG *pMsg;
	BssManagerStaManagerCfm_t *confirmMessage;

	/*Allocate message*/
	pMsg = OSAL_GET_MESSAGE(sizeof(BssManagerStaManagerCfm_t));
	confirmMessage = (BssManagerStaManagerCfm_t*) pK_MSG_DATA(pMsg);
	/*Set Client ID to the registered ID*/
	confirmMessage->clientId = BSS_MANAGER_STA_MANAGER_ATF_CLIENT;
	/*Set STA ID*/
	confirmMessage->sid = sid;
	/*Send confirmation message*/
	//ILOG0_DD("[ATF_Manager_StaManagerSendConfirm], Send Confirmation, Client %d, SID %d", clientId, sid);
	OSAL_SEND_MESSAGE(BSS_MANAGER_STA_MANAGER_REG_CFM, TASK_BSS_MANAGER, pMsg, VAP_ID_DO_NOT_CARE);
}

/**********************************************************************************

Atf_GrantCalc_updateTxopTimeThreshold



Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
void Atf_GrantCalc_updateTxopTimeThreshold(uint16 txopTimeThreshold)
{
    K_MSG *pMsg = OSAL_GET_MESSAGE(sizeof(LaTxopDurationParams_t));
	LaTxopDurationParams_t *linkAdaptationTxopTimeThreshold = (LaTxopDurationParams_t *)pK_MSG_DATA(pMsg);
    
	linkAdaptationTxopTimeThreshold->isMu = FALSE;
	linkAdaptationTxopTimeThreshold->txopTimeThreshold = AGGREGATION_BUILDER_TXOP_AIR_TIME_DURATION_LIMIT - txopTimeThreshold;
	
    OSAL_SEND_MESSAGE(LINK_ADAPTATION_TXOP_TIME_LIMIT_CONFIG_REQ, TASK_LINK_ADAPTATION, pMsg, GET_DEFAULT_VAP_FOR_MY_BAND()); // need to set the right vap
}

/**********************************************************************************

Atf_GrantCalc_dynamicGlobal



Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
void Atf_GrantCalc_dynamicGlobal(uint32 interval)
{
    uint32 globalTxopDuration;
    uint32 staTxopDuration;
    StaId stationId = MAX_UINT8; //Atf_Manager_GetNextStationBitmap will increase 1 to get to ZERO value
    uint32 interval_us = interval*1000;
    uint8 TotalStationsAtfActive;

    TotalStationsAtfActive = AtfDb.TotalStationsAtfActive;

    globalTxopDuration = MAX(MIN(interval_us/TotalStationsAtfActive, AGGREGATION_BUILDER_TXOP_AIR_TIME_DURATION_LIMIT), MAX_TRANSMISSION_TIME); //(5.4 ms < txop duration < 32 ms)

    Atf_GrantCalc_updateTxopTimeThreshold(globalTxopDuration);

    staTxopDuration = MIN(globalTxopDuration, MAX_PSDU_TX_TIME_LIMIT);
    
    while(Atf_Manager_GetNextStationBitmap(&stationId, &AtfDb.AtfStationsConnected) == TRUE)
    {   
        StaDb_SetPsduTimeLimit(stationId, staTxopDuration); // currently AC0 will take effect on all TID, assumig all AC are equal in driver msg
        // change state
        Atf_SetStationState(stationId, ATF_STATION_DYNAMIC_STATE_CONNECTED_AIR_TIME_LIMITED);
    }
}

/**********************************************************************************

Atf_GrantCalc_dynamicWeightedCommon



Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
void Atf_GrantCalc_dynamicWeightedCommon(void)
{
    Atf_GrantCalc_updateTxopTimeThreshold(AGGREGATION_BUILDER_TXOP_AIR_TIME_DURATION_LIMIT);
}

/**********************************************************************************

Atf_GrantCalc_dynamicWeighted




Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
void Atf_GrantCalc_dynamicWeighted(GRANT_SIZE grants[HW_NUM_OF_STATIONS])
{
    // find biggest station per AC with biggest grant
    uint32 maxPsduTransmissionTimeLimit = 0;
    GRANT_SIZE maximumGrantAllStations = 0;
    //StationBitmap_t AtfStationsActivelocal;
    StaId stationId = MAX_UINT8; // need to set to -1, since Atf_Manager_GetNextStationBitmap will increase 1
    uint8 ac;
    TXOP_LIMIT_SIZE_DYNAMIC sta_TXOP_limit[ACCESS_CATEGORY_NUM] = {0}; //us

    //memcpy32(&AtfStationsActivelocal, &AtfStationsActive, CONVERT_BYTES_TO_WORDS(sizeof(StationBitmap_t)));  

    maximumGrantAllStations = findHighest(grants, HW_NUM_OF_STATIONS, ATF_USED_AC);
    
#ifdef ATF_DEBUG_HIGH
    ILOG0_DD("[Atf_GrantCalc_dynamicWeighted] maxGrant %d; ac %d", maximumGrantAllStations, ATF_USED_AC);
#endif //ATF_DEBUG_HIGH

    while(Atf_Manager_GetNextStationBitmap(&stationId, &AtfDb.AtfStationsActive) == TRUE)
    {
        GRANT_SIZE grant = grants[stationId];
        if (grant > 0) // Station is connected by the grants list and was validated vs stationDb in validation section
        {
            // calculation TXOP limit per station ac
            sta_TXOP_limit[ATF_USED_AC] = (AGGREGATION_BUILDER_TXOP_AIR_TIME_DURATION_LIMIT*(grant))/(maximumGrantAllStations);

            // check calculation can't be higher than LIMIT
            //ASSERT(sta_TXOP_limit[ATF_USED_AC] <= AGGREGATION_BUILDER_TXOP_AIR_TIME_DURATION_LIMIT);

            // Determine the maxPsduTransmissionTimeLimit size
            if (sta_TXOP_limit[ATF_USED_AC] > MAX_PSDU_TX_TIME_LIMIT)
            {
                 // FUTURE WORK: Optimizations for Psdu
                 maxPsduTransmissionTimeLimit = MAX_PSDU_TX_TIME_LIMIT;
            }
            else
            {
                if (sta_TXOP_limit[ATF_USED_AC] < ATF_DYNAMIC_PSDU_DEFAULT_MINIMUM_TIME_US)
                {
                    sta_TXOP_limit[ATF_USED_AC] = ATF_DYNAMIC_PSDU_DEFAULT_MINIMUM_TIME_US; // minimum is 2.6milisec
                }
                maxPsduTransmissionTimeLimit = sta_TXOP_limit[ATF_USED_AC];
            }
            
            // set maxGrant per AC
            for (ac = 0;  ac < ACCESS_CATEGORY_NUM; ac++)
            {
                // Set the Atf shared Ram data base with Sender
                AtfShramDbDynamic_p->maxGrant[stationId][ac]= sta_TXOP_limit[ATF_USED_AC] >> ATF_DYNAMIC_PSDU_RESOLUTION; // max time in bits length is 15bits, so shift 7 not 8
                AtfShramDbDynamic_p->currGrant[stationId][ac] = GRANT_DEFAULT;
            }
            // set maxPsduTransmissionTimeLimit per TID
            StaDb_SetPsduTimeLimit(stationId, maxPsduTransmissionTimeLimit);

            // change state
            Atf_SetStationState(stationId, ATF_STATION_DYNAMIC_STATE_CONNECTED_AIR_TIME_LIMITED);
            
#ifdef ATF_DEBUG_GLOBAL
            ILOG0_DDD("[Atf_GrantCalc_dynamicWeighted] stationID:%d sta_TXOP_limit:%d AtfShramDb.maxGrant[stationId][ATF_USED_AC]:%d", stationId, sta_TXOP_limit[ATF_USED_AC], AtfShramDbDynamic_p->maxGrant[stationId][ATF_USED_AC]);
#endif           
        }
    } 
}

/**********************************************************************************

Atf_GrantCalc_staticWeightedPerStationAc



Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
void Atf_GrantCalc_staticWeighted(GRANT_SIZE grants[HW_NUM_OF_STATIONS], uint32 interval, uint32 freeTime)
{
    StaId stationId = MAX_UINT8; // need to set to -1, since Atf_Manager_GetNextStationBitmap will increase 1

    uint32 maxPsduTransmissionTimeLimit = 0;
    TXOP_LIMIT_SIZE_STATIC sta_TXOP_limit = 0;
    uint32 stationsTotalTxTime; // milisec

    //uint32 increseIntervalRatio; // milisec
    //GRANT_SIZE minimun_Grant = 0;
    //TXOP_LIMIT_SIZE_STATIC minimun_sta_TXOP_limit = 0;

    // set stations allocated tx time
    stationsTotalTxTime = (interval-freeTime);
    
    // set interval time to allow minimum 2600 microsec to station with least grant
    /*minimun_Grant = findLowest(grants);
    minimun_sta_TXOP_limit = (minimun_Grant*stationsTotalTxTime*10);
    
    if (minimun_sta_TXOP_limit < MED_TRANSMISSION_TIME)
        uint32 ratio = (minimun_sta_TXOP_limit/MED_TRANSMISSION_TIME);
        AtfDb.interval = interval;
        stationsTotalTxTime = (AtfDb.interval -freeTime);*/
   
    
    while(Atf_Manager_GetNextStationBitmap(&stationId, &AtfDb.AtfStationsActive) == TRUE)
    {
        GRANT_SIZE grant = grants[stationId];
        if (grant > 0) // Station is connected by the grants list and was validated vs stationDb in validation section
        {
            sta_TXOP_limit = ((stationsTotalTxTime * grant) / 10); // real formula: ((interval * 1000 //convert ms to us) * (grant/10000 //convert per to fraction))
            AtfShramDbStatic_p->maxGrant[stationId] = sta_TXOP_limit; // update shared ram
            AtfShramDbStatic_p->currGrant[stationId] = GRANT_DEFAULT;
            // check if PSDU limit is less then maximum, if so need to limit the aggregation.
            if (sta_TXOP_limit > MAX_PSDU_TX_TIME_LIMIT)
            {
                 // FUTURE WORK: Optimizations for Psdu
                 maxPsduTransmissionTimeLimit = MAX_PSDU_TX_TIME_LIMIT;
            }
            else
            {   
                if (sta_TXOP_limit < ATF_DYNAMIC_PSDU_DEFAULT_MINIMUM_TIME_US)
                {
                    sta_TXOP_limit = ATF_DYNAMIC_PSDU_DEFAULT_MINIMUM_TIME_US; // minimum is 2.6milisec
                }
                maxPsduTransmissionTimeLimit = sta_TXOP_limit;
            }
            // Set the Atf shared Ram data base with Sender
            StaDb_SetPsduTimeLimit(stationId, maxPsduTransmissionTimeLimit);
            // change state
            Atf_SetStationState(stationId, ATF_STATION_STATIC_STATE_CONNECTED_UNBUFFERED);
#ifdef ATF_DEBUG_GLOBAL
            ILOG0_DDD("[Atf_GrantCalc_staticWeighted] stationID:%d grant:%d AtfShramDb.maxGrant[stationId]:%d", stationId, sta_TXOP_limit, AtfShramDbStatic_p->maxGrant[stationId]);
#endif  
        }
    }
} 

/**********************************************************************************

Atf_Buffering_lockReq



Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
void Atf_Buffering_lockReqHandler(StaId stationId)
{
    RequesterLockParams_t lockRequesterParams;
	memset(&lockRequesterParams, 0, sizeof(RequesterLockParams_t));
#ifdef ATF_DEBUG_HIGH
    ILOG0_D("[Atf_Buffering_lockReqHandler] stationId:%d",stationId);
#endif
    // ATF locker is busy
    AtfDb.lockBusy = TRUE;
    
    // Lock req
    lockRequesterParams.returnMsg = ATF_BUFFERING_LOCK_CFM;
    lockRequesterParams.returnTask = TASK_ATF;
    Locker_LockAllStaQueues(HW_TX_Q_TYPE_STA_TID, stationId, &lockRequesterParams);
}


/**********************************************************************************

Atf_Buffering_lockReq



Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
void Atf_Buffering_lockReq(StaId stationId)
{   
    // if no lock of diffrent stations is pending send lock req
    if (AtfDb.lockBusy == FALSE)
    {
        // Lock req
        Atf_Buffering_lockReqHandler(stationId); 
    }
    else
    {
#ifdef ATF_DEBUG_HIGH   
        ILOG0_D("[Atf_Buffering_lockReq] pushToStack stationId:%d",stationId);
#endif
        // push lock req of this station to stack. it will be triggered in this/another station lock cfm
        Atf_Manager_pushToStack(stationId);
    }
}

/**********************************************************************************

Atf_Buffering_lockCfmHandlerMsg



Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
void Atf_Buffering_lockCfmMsg(K_MSG *psMsg)
{
    AtfStationState_e state;
    LockReqCb_t* lockRegCbMsg;
    StaId stationId;
    StaId stationIdLockNext = INVALID_STA_INDEX;
    
	lockRegCbMsg = ((LockReqCb_t *)psMsg->abData);
    stationId = lockRegCbMsg->stationOrVapNum;

    state = Atf_GetStationState(stationId);

    // locker is free again
    AtfDb.lockBusy = FALSE;

    // if a lock of another station is pending trigger (LIFO)
    stationIdLockNext = popFromStack();
    if (stationIdLockNext != INVALID_STA_INDEX)
    {
#ifdef ATF_DEBUG_GLOBAL
        ILOG0_D("[Atf_Buffering_lockCfmMsg] ATF_INVALID_LOCK_IDX stationId:%d",stationId);
#endif
        Atf_Buffering_lockReqHandler(stationIdLockNext);
    }   
        
    // if state is still "wait for lock" (by checking state here, we reduce the SM size)
    if (state == ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_LOCK)
    {
#ifdef ATF_DEBUG_LOCAL
        ILOG0_D("[Atf_Buffering_lockCfmMsg] station:%d Lock Cfm msg in wait for lock state", stationId);
#endif 
        // flash all pd in queue and filter
        Atf_Buffering_flushStationQueues((void*)stationId);
        // unlock all station TIDs
        Locker_UnLockAllStaQueues(HW_TX_Q_TYPE_STA_TID, stationId);
        // change station state to buffered    
        Atf_SetStationState(stationId, ATF_STATION_STATIC_STATE_CONNECTED_BUFFERED); 
    }
    else if (state == ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_LOCK_STOP_UNBUFFER)
    {
#ifdef ATF_DEBUG_LOCAL
        ILOG0_D("[Atf_Buffering_lockCfmMsg] station:%d During locking got timer event", stationId);
#endif    
        // unlock all station TIDs
        Locker_UnLockAllStaQueues(HW_TX_Q_TYPE_STA_TID, stationId);
        // Set state
        Atf_SetStationState(stationId, ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_HOSTINTERFACE_STOP_UNBUFFER);
        // Trigger demy-msg CFM
        Atf_Buffering_himBufferCfmHandler(stationId);      
    }
    else if (state == ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_LOCK_STOP_DROP)
    {
#ifdef ATF_DEBUG_LOCAL        
        ILOG0_D("[Atf_Buffering_lockCfmMsg] station:%d During locking got stop traffic", stationId);
#endif
        // unlock all station TIDs
        Locker_UnLockAllStaQueues(HW_TX_Q_TYPE_STA_TID, stationId);
        // Set state
        Atf_SetStationState(stationId, ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_HOSTINTERFACE_STOP_DROP);
        // Trigger demy-msg CFM
        Atf_Buffering_himBufferCfmHandler(stationId);
    }
    else
    {
        // not a valid state to be in
        ASSERT(0);
    }
    
}


/**********************************************************************************

Atf_Buffering_himBufferReq



Description:
------------
    
    
Input:
-----
    station ID

**********************************************************************************/
void Atf_Buffering_himBufferReq(void* eventInfo)
{
 
    uint32 stationId = (uint32)eventInfo;
    uint8 tidCurr = 0;
    
    K_MSG *pMsg = OSAL_GET_MESSAGE(sizeof(AtfMsg_t));
	AtfMsg_t *AtfHim = (AtfMsg_t *)pK_MSG_DATA(pMsg);
    AtfHim->stationId = stationId;
#ifdef ATF_DEBUG_LOCAL
    ILOG0_D("[Atf_Buffering_himBufferReq] station:%d ", stationId);
#endif
    // assert if head and tail of "unbuffered" station is not null (no left overs from last buffering)
    for (tidCurr = 0 ; tidCurr < NUM_OF_TID ; tidCurr++)
    {
        ASSERT(AtfBufferedList[stationId][tidCurr].head == BUFFERD_PD_NULL);
        ASSERT(AtfBufferedList[stationId][tidCurr].tail == BUFFERD_PD_NULL);
    }
    
    OSAL_SEND_MESSAGE(HIM_RINGS_ATF_BUFFER_STATION, TASK_HIM, pMsg, VAP_ID_DO_NOT_CARE);
    Atf_SetStationState(stationId, ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_HOSTINTERFACE); 
}


/**********************************************************************************

Atf_Buffering_lockCfmHandlerMsg



Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
void Atf_Buffering_himBufferCfmMsg(K_MSG *psMsg)
{ 
    StaId stationId;
    AtfMsg_t *AtfHim = ((AtfMsg_t *)psMsg->abData);
    stationId = AtfHim->stationId;
    
    Atf_Buffering_himBufferCfmHandler(stationId);

}

/**********************************************************************************

Atf_Buffering_himBufferCfmHandler



Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
void Atf_Buffering_himBufferCfmHandler(StaId stationId)
{ 
    AtfStationState_e state;
    state = Atf_GetStationState(stationId);
#ifdef ATF_DEBUG_LOCAL
    ILOG0_D("[Atf_Buffering_himBufferCfmHandler] stationId:%d",stationId);
#endif    
    // if state is still "wait for lock" (by checking state here, we reduce the SM size)
    if (state == ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_HOSTINTERFACE)
    {
        // start lock req
        Atf_Buffering_lockReq(stationId); 
        Atf_SetStationState(stationId, ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_LOCK);
    }
    else if (state == ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_HOSTINTERFACE_STOP_UNBUFFER)
    {
#ifdef ATF_DEBUG_LOCAL        
        ILOG0_D("[Atf_Buffering_himBufferCfmMsg] During locking the station:%d got timer event", stationId);
#endif
        Atf_Buffering_unbuffer((void*) stationId);
    }
    else if (state == ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_HOSTINTERFACE_STOP_DROP)
    {
#ifdef ATF_DEBUG_LOCAL        
        ILOG0_D("[Atf_Buffering_himBufferCfmMsg] During locking the station:%d got stop traffic", stationId);
#endif
        // need to drop all buffered station's queues, and move to IDLE
        Atf_Buffering_drop_buffered((void*)stationId);

        /* We reached here by: 
        stop-traffic or/and during-disabling */
        
        //This station may need to cfm to "stop traffic" msg, or to continue disabling proccess.
        if (ATF_GET_BIT_IN_BITMAP(AtfDb.AtfStationsPendingStopTraffic.staBitMap, stationId))
        {
            //ILOG0_D("STOP_DROP stationId:%d",stationId);
            // confirm station manager of stop traffic done
            Atf_Buffering_stopTrafficHandler((void*)stationId);
            // clr bit
            ATF_CLR_BIT_IN_BITMAP(AtfDb.AtfStationsPendingStopTraffic.staBitMap, stationId);
        }
        
        // We may reached here while trying to set all station to idle, 
        // if so need to check that we are last station in idle and trigger new transittion (done in transition function).
        if (AtfDb.duringDisabling == TRUE) 
        {
            //ILOG0_D("duringDisabling stationId:%d",stationId);
            Atf_Manager_TransitionReqMsg(AtfDb.psMsg);
        }
    }
    else
    {
        // not a valid state to be in
        ASSERT(0);
    }
}

/**********************************************************************************

Atf_Buffering_pushToBufferedList



Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
void Atf_Buffering_unbufferHim(StaId stationId)
{
#ifdef ATF_DEBUG_HIGH    
    ILOG0_D("[Atf_Buffering_unbufferHim], stationId %d",stationId);
#endif    
    // Set Hostinterface atf buffering bit to FALSE
    hostInterfaceRingsSetAtfBufferDisable(stationId);
        
}

/**********************************************************************************

Atf_Buffering_flushStationQueues



Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
void Atf_Buffering_flushStationQueues(void* eventInfo)
{
    uint8 tidCurr = IEEE802_1D_BE_1;
    bool result = FALSE;
    HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
    HwQueueManagerRequestParams_t hwQueueManagerRequestParamsMng;
    HwQueueManagerRequestParams_t hwQueueManagerRequestParamsAged;
    uint32 stationId = (uint32)eventInfo;
    //TxPd_t *pd;
#ifdef ATF_DEBUG_LOCAL
    ILOG0_D("[Atf_Buffering_flushStationQueues] station:%d Flush start", stationId);
#endif
    
    memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));
    hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
    hwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_STA_TID;
    hwQueueManagerRequestParams.primaryAddr = stationId;
    
    memset(&hwQueueManagerRequestParamsMng,0, sizeof(HwQueueManagerRequestParams_t));
    hwQueueManagerRequestParamsMng.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
    hwQueueManagerRequestParamsMng.dplIndex = HW_TX_Q_TYPE_STA_TID;
    hwQueueManagerRequestParamsMng.primaryAddr = stationId;

    memset(&hwQueueManagerRequestParamsAged,0, sizeof(HwQueueManagerRequestParams_t));
    hwQueueManagerRequestParamsAged.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
    hwQueueManagerRequestParamsAged.dplIndex = HW_Q_MANAGER_DONE_LIST_LIBERATOR;
    hwQueueManagerRequestParamsAged.primaryAddr = stationId;		
    
    for (tidCurr = 0 ; tidCurr < NUM_OF_TID ; tidCurr++)
    {
        //init
        hwQueueManagerRequestParams.pHeadDesc = NULL_PD;
        hwQueueManagerRequestParams.pTailDesc = NULL_PD;
        hwQueueManagerRequestParamsMng.pHeadDesc = NULL_PD;
        hwQueueManagerRequestParamsMng.pTailDesc = NULL_PD;
        hwQueueManagerRequestParamsAged.pHeadDesc = NULL_PD;
        hwQueueManagerRequestParamsAged.pTailDesc = NULL_PD;
        
        // tid is changing but not station
        hwQueueManagerRequestParams.secondaryAddr = tidCurr;
        hwQueueManagerRequestParamsMng.secondaryAddr = tidCurr;
        hwQueueManagerRequestParamsAged.secondaryAddr = tidCurr;
        
        // flush tidCurr
        HwQManager_FlushQ(&hwQueueManagerRequestParams);
        
        if (hwQueueManagerRequestParams.pHeadDesc != NULL_PD) // queue is not empty
        {
#ifdef ATF_DEBUG_HIGH
            printPdTypeListAll(&hwQueueManagerRequestParams);
#endif            

            //pd = hwQueueManagerRequestParams.pHeadDesc;
            /*
            if ((pd->PdType) != PDTYPE_DATA && (pd->PdType) != PDTYPE_DATA_RESERVED)
            { 
                HwQManager_PushPacketToHead(&hwQueueManagerRequestParams);
            }
            */
            
            // filter managements to SW list
            result = filterManagmentsPd(&hwQueueManagerRequestParams, &hwQueueManagerRequestParamsMng);
            
            // push managements from SW list back to queue
            if (result == TRUE)
            {
#ifdef ATF_DEBUG_LOCAL
                printPdTypeListAll(&hwQueueManagerRequestParams);
                ILOG0_V("[Atf_Buffering_flushStationQueues] managements exist");
                printPdTypeListAll(&hwQueueManagerRequestParamsMng);
#endif
                HwQManager_PushPacketlistToHead(&hwQueueManagerRequestParamsMng);
            }
            
            
            // drop to-be-aged of SW list (before buffring)
            result = filterAgedPd(&hwQueueManagerRequestParams, &hwQueueManagerRequestParamsAged);
           
            // push aged from SW list to liberator
            if (result == TRUE)
            {
#ifdef ATF_DEBUG_LOCAL
                ILOG0_V("[Atf_Buffering_flushStationQueues] aged exist");
                printPdTypeList(&hwQueueManagerRequestParamsAged);
#endif
				/*We may age out some retransmitted PDs - need to set Low SN according to what we buffer*/
				if((NULL_PD == hwQueueManagerRequestParams.pHeadDesc) || !((TxPd_t *)hwQueueManagerRequestParams.pHeadDesc)->retransmissionIndication)
				{
					/* In case the queue is empty of the first PD hasnt been transmitted yet - take the sequence number from the station database */
					StaDB_UpdateLowSequenceNumberFromCurrent(stationId, tidCurr);
				}
				else
				{
					/* Take the sequence number from the PD */
					StaDB_UpdateLowSequenceNumber(stationId, tidCurr, ((TxPd_t *)hwQueueManagerRequestParams.pHeadDesc)->sn);
				}				
				
                HwQManager_PushPacketListToTail(&hwQueueManagerRequestParamsAged); 
            }
            
            // push flushed list to SW buffered list head (head since hostinterface might pushed new pds to list)
            if (hwQueueManagerRequestParams.pHeadDesc != NULL_PD)
            {
                Atf_Buffering_PushListToHead(&hwQueueManagerRequestParams, stationId, tidCurr);
            }
#ifdef ATF_DEBUG_HIGH 
            ILOG0_DDDD("2[Atf_Buffering_flushStationQueues] AtfBufferedList[%d][%d].head:%x tail:%x", stationId, tidCurr, AtfBufferedList[stationId][tidCurr].head, AtfBufferedList[stationId][tidCurr].tail);
#endif
        }
    }
}

/**********************************************************************************

Atf_Buffering_unbuffer


Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
void Atf_Buffering_unbuffer(void* eventInfo)
{
	TX_INTERRUPT_SAVE_AREA;

    uint32 stationId = (uint32)eventInfo;
#ifdef ATF_DEBUG_HIGH
    ILOG0_D("[---Atf_Buffering_Unbuffer---], stationId %d",stationId);
#endif
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);	
    // return all station's buffered queues to HW queues
    Atf_Buffering_unbufferPushToQueues(stationId);
    // HIM Stop buffering
    Atf_Buffering_unbufferHim(stationId);    
    // set new state
    Atf_SetStationState(stationId, ATF_STATION_STATIC_STATE_CONNECTED_UNBUFFERED); 
    OSAL_ENABLE_INTERRUPTS(interrupt_save);
    
}

/**********************************************************************************

Atf_Buffering_unbufferDrop




Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
void Atf_Buffering_unbufferPushBuffered(StaId stationId, HwQMangerDlmNum_e dlmNum, uint8 dplIndex)
{
    TX_INTERRUPT_SAVE_AREA;
    
    uint8 tidCurr = IEEE802_1D_BE_1;
    TxPd_t *pd;
    HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
    memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));
    
    hwQueueManagerRequestParams.dlmNum = dlmNum;
    hwQueueManagerRequestParams.dplIndex = dplIndex;
    hwQueueManagerRequestParams.primaryAddr= stationId;
    
    for (tidCurr = 0 ; tidCurr < NUM_OF_TID ; tidCurr++)
    {
        //init
        hwQueueManagerRequestParams.pHeadDesc = NULL_PD;
        hwQueueManagerRequestParams.pTailDesc = NULL_PD;
        
        // when changing buffered queue we work under disable interrupt
        OSAL_DISABLE_INTERRUPTS(&interrupt_save);
        
        if (AtfBufferedList[stationId][tidCurr].head != BUFFERD_PD_NULL) // queue is not empty
        {
            // tid is changing but not station
            hwQueueManagerRequestParams.secondaryAddr = tidCurr;
            
            hwQueueManagerRequestParams.pHeadDesc = GET_FROM_BUFFER_LIST(AtfBufferedList[stationId][tidCurr].head);
            hwQueueManagerRequestParams.pTailDesc = GET_FROM_BUFFER_LIST(AtfBufferedList[stationId][tidCurr].tail);

            pd = hwQueueManagerRequestParams.pHeadDesc;
            if (dplIndex == HW_Q_MANAGER_DONE_LIST_LIBERATOR && pd->retransmissionIndication == TRUE)
            {
                /* If PD with retry bit was discarded the low sequence number should be update */
                StaDB_UpdateLowSequenceNumberFromCurrent(stationId, tidCurr);
            }
#ifdef ATF_DEBUG_HIGH
            printPdTypeList(&hwQueueManagerRequestParams);
            ILOG0_DDDD("[Atf_Buffering_unbufferPushBuffered] AtfBufferedList[%d][%d].head:%x tail:%x", stationId, tidCurr, AtfBufferedList[stationId][tidCurr].head, AtfBufferedList[stationId][tidCurr].tail);
#endif // ATF_DEBUG_LOCAL

			if (dplIndex == HW_Q_MANAGER_DONE_LIST_LIBERATOR)
			{
				GeneralStatistics.dropReasonATF += StatisticsSetPdStatusNackInList(pd);
			}

            HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams);

            AtfBufferedList[stationId][tidCurr].head = BUFFERD_PD_NULL;
            AtfBufferedList[stationId][tidCurr].tail = BUFFERD_PD_NULL;
        }
        
        OSAL_ENABLE_INTERRUPTS(interrupt_save);  
    }
}

/**********************************************************************************

Atf_Buffering_unbufferDrop


Description:
------------
    "drop" send all buffered queue to liberator
    
Input:
-----
    

**********************************************************************************/
void Atf_Buffering_unbufferDropToLiberator(StaId stationId)
{
#ifdef ATF_DEBUG_HIGH
    ILOG0_D("[Atf_Buffering_unbufferDropToLiberator] station:%d unbuffer drop start", stationId);
#endif
    Atf_Buffering_unbufferPushBuffered(stationId, HW_Q_MANAGER_TX_LISTS_DLM ,HW_Q_MANAGER_DONE_LIST_LIBERATOR);
}

/**********************************************************************************

Atf_Buffering_unbufferQueues




Description:
------------
    return all buffered pds to it original queue
    
Input:
-----
    

**********************************************************************************/
void Atf_Buffering_unbufferPushToQueues(StaId stationId)
{
#ifdef ATF_DEBUG_HIGH 
    ILOG0_D("[Atf_Buffering_unbufferPushToQueues] station:%d unbuffer queues start", stationId);
#endif

    Atf_Buffering_unbufferPushBuffered(stationId, HW_Q_MANAGER_TX_DATA_DLM ,HW_TX_Q_TYPE_STA_TID);
}

/**********************************************************************************

Atf_Buffering_pushToBufferedList



Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
void Atf_Buffering_drop_buffered(void* eventInfo)
{
    TX_INTERRUPT_SAVE_AREA;
    uint32 stationId = (uint32)eventInfo;
#ifdef ATF_DEBUG_LOCAL 
    ILOG0_D("[Atf_Buffering_drop_buffered], stationId %d",stationId);
#endif
    
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);	
    
    // Unbuffer all station's buffered queues
    Atf_Buffering_unbufferDropToLiberator(stationId);
    // HIM Stop buffering
    Atf_Buffering_unbufferHim(stationId);
    // set new state
    Atf_SetStateIdle((void*)stationId);
    
    OSAL_ENABLE_INTERRUPTS(interrupt_save);    
}

/**********************************************************************************

Atf_Manager_stopTrafficMsg




Description:
------------
    Stop station traffic
    
Input:
-----
    

**********************************************************************************/
void Atf_Manager_stopTrafficMsg(K_MSG *psMsg)
{
    StaId stationId = 0;
	BssManagerStaManagerReq_t *staManagerReq = NULL;
	K_MSG *originalRemoveStationMessage = NULL;
	UMI_STOP_TRAFFIC *stopTrafficMessageParameters = NULL;
    bool cfmImme = TRUE; // most cases confirm immediate
    
	staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(psMsg);
	originalRemoveStationMessage = staManagerReq->psMsg;
	stopTrafficMessageParameters = (UMI_STOP_TRAFFIC *)pK_MSG_DATA(originalRemoveStationMessage);
  
	stationId = stopTrafficMessageParameters->u16SID;
#ifdef ATF_DEBUG_GLOBAL 
    ILOG0_D("[Atf_Manager_stopTrafficMsg] stationId:%d", stationId);
#endif

    // eval msg only in static mode 
    if (AtfDb.AtmDistributionType == ATM_DISTRIBUTION_TYPE_STATIC)
    {
        // at wait for lock / wait for him, under stop-traffic-drop we should handle event unlock queues after cfm
        if (Atf_Buffering_isInWaitState(stationId) == TRUE)
        {
            ATF_SET_BIT_IN_BITMAP(AtfDb.AtfStationsPendingStopTraffic.staBitMap, stationId);
            cfmImme = FALSE;
        }
        
        // run stop traffic event for this station
        Atf_buffering_StaSmRun(ATF_STATION_EVENT_STOP_TRAFFIC, stationId, (void*)stationId);
    }
    
    if (cfmImme == TRUE) 
    {
        Atf_Buffering_stopTrafficHandler((void*)stationId);
    }
}

/**********************************************************************************

Atf_Buffering_AddStaMsg




Description:
------------
    Stop station traffic
    
Input:
-----
    

**********************************************************************************/
void Atf_Manager_AddStaMsg(K_MSG *psMsg)
{
    StaId stationId = 0;
	BssManagerStaManagerReq_t *staManagerReq = NULL;
	K_MSG *originalRemoveStationMessage = NULL;
	UMI_STA_ADD *stopTrafficMessageParameters = NULL;
    
	staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(psMsg);
	originalRemoveStationMessage = staManagerReq->psMsg;
	stopTrafficMessageParameters = (UMI_STA_ADD *)pK_MSG_DATA(originalRemoveStationMessage);
  
	stationId = stopTrafficMessageParameters->u16SID;
#ifdef ATF_DEBUG_GLOBAL 
    ILOG0_D("[Atf_Buffering_AddStaMsg] stationId:%d", stationId);
#endif
    // set connected station bitmap
    ATF_SET_BIT_IN_BITMAP(AtfDb.AtfStationsConnected.staBitMap, stationId); // set the station in the bitmap

    Atf_Manager_StaManagerSendConfirm(stationId);

}

/**********************************************************************************

Atf_Buffering_senderStopTxMsg



Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
void Atf_Buffering_stopTrafficHandler(void* eventInfo)
{
    uint32 stationId = (uint32)eventInfo;

    // station is no longer connected
    ATF_CLR_BIT_IN_BITMAP(AtfDb.AtfStationsConnected.staBitMap, stationId);
    
    // confirm station manager of stop traffic done
    Atf_Manager_StaManagerSendConfirm(stationId);
}



/**********************************************************************************

Atf_Buffering_senderStopTxMsg



Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
void Atf_Buffering_senderStopTxMsg(K_MSG *psMsg)
{
  	FwTxSenderMailbox0FifoOutMessage_t *pBody0;
    StaId stationId;
    pBody0 = (FwTxSenderMailbox0FifoOutMessage_t *)pK_MSG_DATA(psMsg);
    stationId = pBody0->messageValue;
#ifdef ATF_DEBUG_HIGH
    ILOG0_D("[---Atf_Buffering_senderStopTxMsg---], stationId %d",stationId);
#endif
    Atf_buffering_StaSmRun(ATF_STATION_EVENT_SENDER_INDICATION, stationId, (void*)stationId); 
}

/**********************************************************************************

Atf_Buffering_PushPacketToTail




Description:
------------
     appends a new node at the end of buffered list
    
Input:
-----
    

**********************************************************************************/
void Atf_Buffering_PushPacketToTail(TxPd_t* pDesc)
{
#ifdef ATF_DEBUG_HIGH    
    ILOG0_D("[Atf_Buffering_PushPacketToTail] pDesc:%x",(uint16)SET_TO_BUFFER_LIST(pDesc));
#endif
    ASSERT(Atf_GetStationState(pDesc->txQStaId) != ATF_STATION_STATIC_STATE_CONNECTED_UNBUFFERED);
   
    // 1. This new node is going to be the last node, so make next of it as NULL
    pDesc->nextPdPointer = BUFFERD_PD_NULL;
 
    // 2. If the Linked List is empty, than make the new node as head
    if (AtfBufferedList[pDesc->txQStaId][pDesc->txQTid].head == BUFFERD_PD_NULL)
    {
        AtfBufferedList[pDesc->txQStaId][pDesc->txQTid].head = (uint16)SET_TO_BUFFER_LIST(pDesc);
        AtfBufferedList[pDesc->txQStaId][pDesc->txQTid].tail = AtfBufferedList[pDesc->txQStaId][pDesc->txQTid].head;
    }
    else
    {
       // 3. Change the next of last node
       ((TxPd_t*)(GET_FROM_BUFFER_LIST(AtfBufferedList[pDesc->txQStaId][pDesc->txQTid].tail)))->nextPdPointer = (uint16)CONVERT_PD_TO_OFFSET(pDesc);
        
       // 4. tail point to new pd
       AtfBufferedList[pDesc->txQStaId][pDesc->txQTid].tail = (uint16)SET_TO_BUFFER_LIST(pDesc);
    }
}

/**********************************************************************************

Atf_Buffering_PushPacketToHead




Description:
------------
     appends a new node at the start of buffered list
    
Input:
-----
    

**********************************************************************************/
void Atf_Buffering_PushListToHead(HwQueueManagerRequestParams_t *hwQueueManagerRequestParams, StaId stationId, uint8 tidCurr)
{
    TX_INTERRUPT_SAVE_AREA;
    TxPd_t * pd = hwQueueManagerRequestParams->pTailDesc;

#ifdef ATF_DEBUG_HIGH
    ILOG0_V("[Atf_Buffering_PushListToHead]");
#endif

    OSAL_DISABLE_INTERRUPTS(&interrupt_save);
    
    //1. concatenation of the tail of the pushed list (pd) to the head of current buffred if null so next is null and it is ok
    pd->nextPdPointer = AtfBufferedList[stationId][tidCurr].head;
    
    //2. head is the pushed list head
    AtfBufferedList[stationId][tidCurr].head = (uint16)SET_TO_BUFFER_LIST(hwQueueManagerRequestParams->pHeadDesc);
    
    // 3. it may need to update the new tail if it is NULL, otherwise tail remain tail
    if (AtfBufferedList[stationId][tidCurr].tail == BUFFERD_PD_NULL)
        AtfBufferedList[stationId][tidCurr].tail = (uint16)SET_TO_BUFFER_LIST(hwQueueManagerRequestParams->pTailDesc);
    	
    OSAL_ENABLE_INTERRUPTS(interrupt_save);
}

/**********************************************************************************

Atf_Buffering_dropAllBuffered



Description:
------------
     Drop all buffered queue, to free up space
    
Input:
-----
    

**********************************************************************************/
void Atf_Buffering_dropAllBuffered(K_MSG *psMsg)
{
    StaId stationId = MAX_UINT8; // need to set to -1, since Atf_Manager_GetNextStationBitmap will increase 1
    
    while(Atf_Manager_GetNextStationBitmap(&stationId, &AtfDb.AtfStationsActive) == TRUE)
    {
        AtfStationState_e state = Atf_GetStationState(stationId);
        if (state == ATF_STATION_STATIC_STATE_CONNECTED_BUFFERED)
        {
            Atf_Buffering_unbufferDropToLiberator(stationId);
        }
    }
}

/**********************************************************************************
Atf_Buffering_setStateWaitHimStopUnbuffer  

Description:
------------
	
Input:
-----
	None
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void Atf_Buffering_setStateWaitHimStopUnbuffer(void* eventInfo)
{
    uint32 stationId = (uint32)eventInfo;
	Atf_SetStationState(stationId, ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_HOSTINTERFACE_STOP_UNBUFFER);
}

/**********************************************************************************
Atf_Buffering_DoNothing  

Description:
------------
	An event was received for a specific state which requires no action.
Input:
-----
	None
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void Atf_Buffering_setStateWaitHimStopDrop(void* eventInfo)
{
    uint32 stationId = (uint32)eventInfo;
	Atf_SetStationState(stationId, ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_HOSTINTERFACE_STOP_DROP);
}

/**********************************************************************************
Atf_Buffering_setStateWaitHim  

Description:
------------
	An event was received for a specific state which requires no action.
Input:
-----
	None
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void Atf_Buffering_setStateWaitHim(void* eventInfo)
{
    uint32 stationId = (uint32)eventInfo;
	Atf_SetStationState(stationId, ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_HOSTINTERFACE);
}

/**********************************************************************************
Atf_Buffering_setStateWaitHim  

Description:
------------
	An event was received for a specific state which requires no action.
Input:
-----
	None
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void Atf_Buffering_setStateWaitLockStopUnbuffer(void* eventInfo)
{
    uint32 stationId = (uint32)eventInfo;
	Atf_SetStationState(stationId, ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_LOCK_STOP_UNBUFFER);
}


/**********************************************************************************
Atf_Buffering_setStateWaitHim  

Description:
------------
	An event was received for a specific state which requires no action.
Input:
-----
	None
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void Atf_Buffering_setStateWaitLockStopDrop(void* eventInfo)
{
    uint32 stationId = (uint32)eventInfo;
	Atf_SetStationState(stationId, ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_LOCK_STOP_DROP);
}

/**********************************************************************************
Atf_Buffering_setStateWaitHim  

Description:
------------
	An event was received for a specific state which requires no action.
Input:
-----
	None
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void Atf_Buffering_setStateWaitLock(void* eventInfo)
{
    uint32 stationId = (uint32)eventInfo;
	Atf_SetStationState(stationId, ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_LOCK);
}


/**********************************************************************************
Atf_Buffering_setStateWaitHim  

Description:
------------
	An event was received for a specific state which requires no action.
Input:
-----
	None
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
bool Atf_Buffering_isInWaitState(StaId stationId)
{
    AtfStationState_e state = Atf_GetStationState(stationId);
    if (state >= ATF_STATION_STATIC_STATE_START_OF_WAIT  && state <= ATF_STATION_STATIC_STATE_END_OF_WAIT)
    {
        return TRUE;
    }
    return FALSE;
}

/**********************************************************************************
Atf_Buffering_DoNothing  

Description:
------------
	An event was received for a specific state which requires no action.
Input:
-----
	None
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void Atf_Buffering_DoNothing(void* eventInfo)
{
	// This event in this state has no impact.
}

/**********************************************************************************
Atf_Buffering_IllegalEvent  

Description:
------------
	Illegal event was received for a specific state. This is a bug.
Input:
-----
	None
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void Atf_Buffering_IllegalEvent(void* eventInfo)
{
	ASSERT(0);
}

/**********************************************************************************
beaconHandler_RunVapSm  

Description:
------------
	Run the VAP state machine Send provided event to current state.
Input:
-----
	beaconHandlerVapSmEvents_e 	event - new event
	uint8 						stationId - station index
	void* 						eventInfo - optional input
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void Atf_buffering_StaSmRun(AtfStationEvent_e event, StaId stationId, void* eventInfo)
{
	AtfStationState_e currentState;

	ASSERT(event < ATF_STATION_NUM_OF_EVENTS);

	// Get current transmission state
	currentState = Atf_GetStationState(stationId);
#ifdef ATF_DEBUG_LOCAL     
    ILOG0_DDD("[Atf_buffering_StaSmRun] stationId:%d event:%d currentState:%d",stationId, event, currentState);
#endif

	// Send the event to the state machine		
	AtfBufferingSm[currentState].handler[event](eventInfo);
}

/**********************************************************************************
Atf_Buffering_setStateIdle  

Description:
------------
	set new state to Idle
Input:
-----
	None
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void Atf_SetStateIdle(void* eventInfo)
{
    uint32 stationId = (uint32)eventInfo;

    // one less station
    Atf_Manager_decreseTotalStations(stationId);

    // will set all dynamic station per ac or static per station current/max grant to defualt
    AtfShramDbStatic_p->maxGrant[stationId] = GRANT_DEFAULT;
    AtfShramDbStatic_p->currGrant[stationId] = GRANT_DEFAULT;

  	Atf_SetStationState(stationId, ATF_STATION_STATE_IDLE);
}


/**********************************************************************************

Atf_GetStationStatus




Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
AtfStationState_e Atf_GetStationState(StaId stationId)
{
    return AtfStationsDb.state[stationId];
}

/**********************************************************************************

Atf_GetStationStatus




Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
void Atf_SetStationState(StaId stationId, AtfStationState_e state)
{
#ifdef ATF_DEBUG_HIGH   
    switch ( state ) 
    {
        case ATF_STATION_STATE_IDLE:
        {
            ILOG0_D("[Atf_SetStationState] stationId: %d, state:ATF_STATION_STATE_IDLE",stationId);
            break;
        }
        case ATF_STATION_STATIC_STATE_CONNECTED_BUFFERED:
        {
            ILOG0_D("[Atf_SetStationState] stationId: %d, state:ATF_STATION_STATIC_STATE_CONNECTED_BUFFERED",stationId);
            break;
        }
        case ATF_STATION_STATIC_STATE_CONNECTED_UNBUFFERED:
        {
            ILOG0_D("[Atf_SetStationState] stationId: %d, state:ATF_STATION_STATIC_STATE_CONNECTED_UNBUFFERED",stationId);
            break;
        }
        case ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_HOSTINTERFACE:
        {
            ILOG0_D("[Atf_SetStationState] stationId: %d, state:ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_HOSTINTERFACE",stationId);
            break;
        }
        case ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_HOSTINTERFACE_STOP_UNBUFFER:
        {
            ILOG0_D("[Atf_SetStationState] stationId: %d, state:ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_HOSTINTERFACE_STOP_UNBUFFER",stationId);
            break;
        }
        case ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_HOSTINTERFACE_STOP_DROP:
        {
            ILOG0_D("[Atf_SetStationState] stationId: %d, state:ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_HOSTINTERFACE_STOP_DROP",stationId);
            break;
        }
        case ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_LOCK:
        {
            ILOG0_D("[Atf_SetStationState] stationId: %d, state:ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_LOCK",stationId);
            break;
        }
        case ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_LOCK_STOP_UNBUFFER:
        {
            ILOG0_D("[Atf_SetStationState] stationId: %d, state:ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_LOCK_STOP_UNBUFFER",stationId);
            break;
        }
        case ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_LOCK_STOP_DROP:
        {
            ILOG0_D("[Atf_SetStationState] stationId: %d, state:ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_LOCK_STOP_DROP",stationId);
            break;
        }
    }
#else 
#ifdef ATF_DEBUG_GLOBAL
    ILOG0_DD("[Atf_SetStationState] stationId: %d, state:%d",stationId, state);
#endif  //ATF_DEBUG_GLOBAL
#endif  //ATF_DEBUG_LOCAL
    AtfStationsDb.state[stationId] = state;
}

#ifdef ATF_DEBUG_HIGH
/**********************************************************************************

Atf_GetStationStatus




Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
void Atf_DEBUG(uint32 data1, uint32 data2, uint32 data3, uint32 data4)
{
    GRANT_SIZE stationGrant[HW_NUM_OF_STATIONS] = {0}; // Percentage(0-100)*100

    stationGrant[0] = (GRANT_SIZE)(100*(0xFF & data3));
    //message.stationGrant[0] = debug1;
 
    stationGrant[1] = (GRANT_SIZE)(100*(0xFF & (data3 >> 8)));
    //message.stationGrant[1] = debug2;

    stationGrant[2] = (GRANT_SIZE)(100*(0xFF & (data3 >> 16)));
    //message.stationGrant[1] = debug3;
    
    stationGrant[3] = (GRANT_SIZE)(100*(0xFF & (data3 >> 24)));

    Atf_Manager_transition((AtmDistributionType_e)data1,(AtmAlgorithmType_e) data2,(AtmWeightedType_e) 0, stationGrant, 900, 10);
}

#endif //ATF_DEBUG_HIGH


#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif
   
/**********************************************************************************

Atf_Manager_init


Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
void Atf_Manager_init()
{
    uint8 index, tidCurr;
	
    memset(&AtfDb, 0, sizeof(AtfDb_t));
    memset(&AtfStationsDb, 0, sizeof(AtfStationsDb_t));
    memset(AtfShramDbGeneral_p, 0, sizeof(AtfShramDb_t));
#ifdef ATF_DEBUG_LOCAL
    memset(&AtfErrors, 0, sizeof(uint8)*ATF_DEBUG_MAX_NUMBER_OF_ERRORS);
#endif //#ifdef ATF_DEBUG_LOCAL
    
    for (index = 0; index < HW_NUM_OF_STATIONS; index++)
    {
        for (tidCurr = 0 ; tidCurr < NUM_OF_TID ; tidCurr++)
        {
             AtfBufferedList[index][tidCurr].head = BUFFERD_PD_NULL;
             AtfBufferedList[index][tidCurr].tail = BUFFERD_PD_NULL;
        }
    }
    for (index = 0; index < ATF_MAX_LOCK_ELEMENTS; index++)
    {
        // set lock to invalid
        AtfDb.lockStack[index] = INVALID_STA_INDEX;
    }

	AtfBufferingSm[ATF_STATION_STATE_IDLE].handler[ATF_STATION_EVENT_TIMER_EXPIRED] =			Atf_Buffering_IllegalEvent;
	AtfBufferingSm[ATF_STATION_STATE_IDLE].handler[ATF_STATION_EVENT_SENDER_INDICATION] =		Atf_Buffering_DoNothing; // station during connection can get some data before driver set the ATF, in this state max grant is 0, so this event can occuer and shall be ignored
	AtfBufferingSm[ATF_STATION_STATE_IDLE].handler[ATF_STATION_EVENT_STOP_TRAFFIC] =		    Atf_Buffering_DoNothing; 
    
	AtfBufferingSm[ATF_STATION_STATIC_STATE_CONNECTED_BUFFERED].handler[ATF_STATION_EVENT_TIMER_EXPIRED] = 		Atf_Buffering_unbuffer;
	AtfBufferingSm[ATF_STATION_STATIC_STATE_CONNECTED_BUFFERED].handler[ATF_STATION_EVENT_SENDER_INDICATION] =	Atf_Buffering_IllegalEvent; //Atf_Buffering_DoNothing (not should happen but won't harm)
	AtfBufferingSm[ATF_STATION_STATIC_STATE_CONNECTED_BUFFERED].handler[ATF_STATION_EVENT_STOP_TRAFFIC] = 		Atf_Buffering_drop_buffered;
    
	AtfBufferingSm[ATF_STATION_STATIC_STATE_CONNECTED_UNBUFFERED].handler[ATF_STATION_EVENT_TIMER_EXPIRED] = 			Atf_Buffering_DoNothing;
	AtfBufferingSm[ATF_STATION_STATIC_STATE_CONNECTED_UNBUFFERED].handler[ATF_STATION_EVENT_SENDER_INDICATION] = 		Atf_Buffering_himBufferReq;
	AtfBufferingSm[ATF_STATION_STATIC_STATE_CONNECTED_UNBUFFERED].handler[ATF_STATION_EVENT_STOP_TRAFFIC] = 			Atf_SetStateIdle;
    
	AtfBufferingSm[ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_HOSTINTERFACE].handler[ATF_STATION_EVENT_TIMER_EXPIRED] =		Atf_Buffering_setStateWaitHimStopUnbuffer; 
	AtfBufferingSm[ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_HOSTINTERFACE].handler[ATF_STATION_EVENT_SENDER_INDICATION] = 	Atf_Buffering_DoNothing;
	AtfBufferingSm[ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_HOSTINTERFACE].handler[ATF_STATION_EVENT_STOP_TRAFFIC] =			Atf_Buffering_setStateWaitHimStopDrop;
    
	AtfBufferingSm[ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_HOSTINTERFACE_STOP_UNBUFFER].handler[ATF_STATION_EVENT_TIMER_EXPIRED] =		Atf_Buffering_DoNothing; 
	AtfBufferingSm[ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_HOSTINTERFACE_STOP_UNBUFFER].handler[ATF_STATION_EVENT_SENDER_INDICATION] = 	Atf_Buffering_setStateWaitHim;
	AtfBufferingSm[ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_HOSTINTERFACE_STOP_UNBUFFER].handler[ATF_STATION_EVENT_STOP_TRAFFIC] =		Atf_Buffering_setStateWaitHimStopDrop;
    
	AtfBufferingSm[ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_HOSTINTERFACE_STOP_DROP].handler[ATF_STATION_EVENT_TIMER_EXPIRED] =		Atf_Buffering_DoNothing; 
	AtfBufferingSm[ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_HOSTINTERFACE_STOP_DROP].handler[ATF_STATION_EVENT_SENDER_INDICATION] = 	Atf_Buffering_DoNothing; // drop is more important than unbuffer, stay unchanged
	AtfBufferingSm[ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_HOSTINTERFACE_STOP_DROP].handler[ATF_STATION_EVENT_STOP_TRAFFIC] =		Atf_Buffering_DoNothing; 
    
	AtfBufferingSm[ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_LOCK].handler[ATF_STATION_EVENT_TIMER_EXPIRED] =	        Atf_Buffering_setStateWaitLockStopUnbuffer; 
	AtfBufferingSm[ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_LOCK].handler[ATF_STATION_EVENT_SENDER_INDICATION] = 	Atf_Buffering_DoNothing;
	AtfBufferingSm[ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_LOCK].handler[ATF_STATION_EVENT_STOP_TRAFFIC] =			Atf_Buffering_setStateWaitLockStopDrop; 
    
	AtfBufferingSm[ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_LOCK_STOP_UNBUFFER].handler[ATF_STATION_EVENT_TIMER_EXPIRED] =	    Atf_Buffering_DoNothing; // on the way to "Unbuffered" anyway
	AtfBufferingSm[ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_LOCK_STOP_UNBUFFER].handler[ATF_STATION_EVENT_SENDER_INDICATION] = 	Atf_Buffering_setStateWaitLock;
	AtfBufferingSm[ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_LOCK_STOP_UNBUFFER].handler[ATF_STATION_EVENT_STOP_TRAFFIC] =		Atf_Buffering_setStateWaitLockStopDrop; 

	AtfBufferingSm[ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_LOCK_STOP_DROP].handler[ATF_STATION_EVENT_TIMER_EXPIRED] =	    Atf_Buffering_DoNothing; // on the way to "Idle" anyway
	AtfBufferingSm[ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_LOCK_STOP_DROP].handler[ATF_STATION_EVENT_SENDER_INDICATION] = 	Atf_Buffering_DoNothing; // on the way to "Idle" so station is disconnecting no more buffering for it
	AtfBufferingSm[ATF_STATION_STATIC_STATE_CONNECTED_WAIT_FOR_LOCK_STOP_DROP].handler[ATF_STATION_EVENT_STOP_TRAFFIC] =		Atf_Buffering_DoNothing; 

    // set default values to databases
    AtfDb.AgeingTtlCriteria = 0xF0;
    AtfDb.AgeingEnable = FALSE;
    AtfShramDbGeneral_p->atfDistType = ATM_DISTRIBUTION_TYPE_DISABLED;
    AtfDb.TotalStationsAtfActive = 0;

    // set lock to invalid - > no elements in lock queue
    AtfDb.lockStackTop = ATF_INVALID_LOCK_IDX;
    AtfDb.lockBusy = FALSE;
     
}

#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif

