/***************************************************************************************
 File:			AutoResponse.c
 Module:	  		Auto Response (incliding Block ACK Generator)
 Purpose: 		Holds all the implementation of the Auto Response API functions
 Description:  		This file contains the implementation of the Auto Response module
				which is responsible of sending auto-reply frames (ACK/BACK/CTS/BF) 
***************************************************************************************/

/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "HwGlobalDefinitions.h"
#include "RegAccess_Api.h"
#include "AutoRespRegs.h"
#include "AutoResponse_Api.h"
#include "AutoResponse.h"
#include "ShramAutoResponseFifos.h"
#include "AutoRespErrFifo_Descriptors.h"
#include "ShramNewDeliaRecipes.h"
#include "DeliaRecipes_Api.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"
#include "ShramPreAggregator.h"


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_HW_DRIVERS
#define LOG_LOCAL_FID 	18


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/
	
AutoRespErrFifo_t* 	pAutoResErrorFifo;
AutoResponseDb_t 	AutoResponseDb;

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/


/**********************************************************************************
autoResponse_GetFifoParameters  

Description:
------------
	Get the updated information about the error fifo
	
**********************************************************************************/
void autoResponse_GetFifoParameters(AutoResponseFifoParams_t *fifoParams)
{
	RegAutoRespAutoRespErrFifoNumEntriesCount_u autoRespErrFifoNumEntriesCountReg; 


	RegAccess_Read(REG_AUTO_RESP_AUTO_RESP_ERR_FIFO_NUM_ENTRIES_COUNT, &autoRespErrFifoNumEntriesCountReg.val);
	
	fifoParams->currentIndexInFifo = AutoResponseDb.fifoReadIndex;
	fifoParams->numOfValidEntries = autoRespErrFifoNumEntriesCountReg.bitFields.autoRespErrFifoNumEntriesCount;

	autoResponse_CheckFifoError();
}

/**********************************************************************************
autoResponse_SetFifoParameters  

Description:
------------
	Set the updated information about the error fifo
	
**********************************************************************************/	
void autoResponse_SetFifoParameters(AutoResponseFifoParams_t *fifoParams)

{
	RegAutoRespAutoRespErrFifoRdEntriesNum_u autoRespErrFifoRdEntriesNumReg;


	// Decrement  Number Of Entries In Fifo 
	autoRespErrFifoRdEntriesNumReg.val = 0;
	autoRespErrFifoRdEntriesNumReg.bitFields.autoRespErrFifoRdEntriesNum = fifoParams->numOfValidEntries;
	RegAccess_Write(REG_AUTO_RESP_AUTO_RESP_ERR_FIFO_RD_ENTRIES_NUM, autoRespErrFifoRdEntriesNumReg.val);

	AutoResponseDb.fifoReadIndex = fifoParams->currentIndexInFifo;
}


/**********************************************************************************
autoResponse_CheckFifoError  

Description:
------------
	Check if there were errors on the error fifo
	
**********************************************************************************/
void autoResponse_CheckFifoError(void)
{
	RegAutoRespAutoRespErrFifoDebug_u		autoRespErrFifoDebugReg;
	RegAutoRespAutoRespErrFifoClearStrb_u 	autoRespErrFifoClearStrbReg;

	autoRespErrFifoClearStrbReg.val = 0;
	
	// Read the debug register
	RegAccess_Read(REG_AUTO_RESP_AUTO_RESP_ERR_FIFO_DEBUG,&autoRespErrFifoDebugReg.val);

	// Count the number of times the error fifo was full (nothing we can do with it)
	AutoResponseDb.fifoFullCounter += autoRespErrFifoDebugReg.bitFields.autoRespErrFifoFullDropCtr;
	ASSERT(AutoResponseDb.fifoFullCounter == 0); // While we still ramp up gen6 - we want to assert here so we can decide if we want to increase the fifo size. Later we can remove this assert.

	// Clear fifo full drop counter
	autoRespErrFifoClearStrbReg.bitFields.autoRespErrFifoClearFullDropCtrStrb = TRUE;
	RegAccess_Write(REG_AUTO_RESP_AUTO_RESP_ERR_FIFO_CLEAR_STRB,autoRespErrFifoClearStrbReg.val);
	
	// check that we didn't decrease the fifo more than we were allowed
	ASSERT(autoRespErrFifoDebugReg.bitFields.autoRespErrFifoDecrementLessThanZero == FALSE);
}


/**********************************************************************************
AutoResponse_ErrorFifoNotEmpty  

Description:
------------
	Process error FIFO. Check the cause, decrease the counter, and assert if needed.
	
**********************************************************************************/
void AutoResponse_ErrorFifoNotEmpty(void)
{
	AutoResponseFifoParams_t 	errorFifoParams;
	AutoRespErrFifo_t 			errorFifoEntry;
	uint8 						fifoIndex;

	// Get number of new entries in fifo	
	autoResponse_GetFifoParameters(&errorFifoParams);

	// Loop over all new entries
	for(fifoIndex = 0; fifoIndex < errorFifoParams.numOfValidEntries; fifoIndex++)
	{
		// Get an entry from the fifo
		errorFifoEntry = pAutoResErrorFifo[errorFifoParams.currentIndexInFifo];

		ILOG2_DD("AutoResponse_ErrorFifoNotEmpty, error type: %x, error word value: %x", errorFifoEntry.userErrorType, errorFifoEntry.word0);

		// There are 2 types of error entries - per user and per "user/tid". The type is located in the same place in the word, so it's OK to check it from the "user" error.
		switch (errorFifoEntry.userErrorType)
		{
			case AUTO_RESPONSE_ERROR_TYPE_PER_USER:
				// Check if there was multi TID error - A user has passed the num TID's that were allocated to him (by TID aggr limit info) 
				if(errorFifoEntry.multiTidErr > 0)
				{
					// During ramp up we don't expect any errors so we assert. Later, this can be replace with ignoring or counting or logging.
					// ASSERT(0);
					AutoResponseDb.multiTidErrCounter++;
					ILOG2_V("AutoResponse_ErrorFifoNotEmpty: multiTidErr");
				}
				// Check if there was BAR error - In HE, a BAR for TID that was received in the same session.
				if(errorFifoEntry.barErr == TRUE)
				{
					// During ramp up we don't expect any errors so we assert. Later, this can be replace with ignoring or counting or logging.
					FATAL("AutoResponse_barErr");
				}
#ifdef ENET_INC_ARCH_WAVE600D2
				if(errorFifoEntry.multiTidBarLengthErr == TRUE)
				{
					// Multi TID BAR length is not consistent with TID_val from RxC or num of TIDs in multi TID BAR is greater than 8
					FATAL("AutoResponse_multiTidBarLengthErr");
				}
				if(errorFifoEntry.immToNonImmErr == TRUE)
				{
					//  There is already an immediate responded frame and a non-immedaite responded frame that supposed to be registered and pushed to BA generator arrived
					FATAL("AutoResponse_immToNonImmErr");
				}
#endif
				break;
			case AUTO_RESPONSE_ERROR_TYPE_PER_TID:
				// Check how many SNs were dropped due to out of windows bounds. (Counter per user/TID.  (SSN+2^11<SN < SSN))
				if(errorFifoEntry.numSnOutOfBound > 0)
				{
					// During ramp up we don't expect any errors so we assert. Later, this can be replace with ignoring or counting or logging.
					// This assert is always relevant for WDS (the issue is in the TX of the other side)
					// Getting  BAR from STA with old SN will also trigger this assert 
					//ASSERT(0);
					AutoResponseDb.errorFifoSnOutOfBoundCounter++;
					ILOG2_D("AutoResponse_ErrorFifoNotEmpty: numSnOutOfBound  = %d", errorFifoEntry.numSnOutOfBound);
				}
				else
				{
					ILOG2_D("AutoResponse_ErrorFifoNotEmpty: numSnOutOfBound (%d) is not greater than 0 but error type per tid was set. ", errorFifoEntry.numSnOutOfBound);
					SLOG2(0, 0, AutoRespErrFifo_t, &errorFifoEntry);
				}
				break;
			default:
				ASSERT(0); //klocwork ignore, UNREACH.GEN..... All the cases in the ENUM is handled, klockwork sees this assert as unreachable code.
		}

		// Advance the index of the fifo which we will read next time (including wrap around)
		errorFifoParams.currentIndexInFifo++;
		errorFifoParams.currentIndexInFifo &= AUTO_RESPONSE_ERROR_FIFO_SIZE_MASK;
	}  

	// Set to HW the number of entries we already processed.
	autoResponse_SetFifoParameters(&errorFifoParams);
}


/**********************************************************************************
AutoResponse_EnableDisableLogger  

Description:
------------
	Enable / Disable logs from the Auto Response Module
	
**********************************************************************************/
void AutoResponse_EnableDisableLogger(bool isEnable)
{
	RegAutoRespAutoRespLogger_u autoRespLoggerReg;

	// use read-modify-write in order not to touch the other bits of the register which configure logger operation
	RegAccess_Read(REG_AUTO_RESP_AUTO_RESP_LOGGER, &autoRespLoggerReg.val);
	autoRespLoggerReg.bitFields.autoRespLoggerEn = isEnable;
	autoRespLoggerReg.bitFields.autoRespLoggerPrio = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_AUTO_RESP_AUTO_RESP_LOGGER, autoRespLoggerReg.val);
}


/**********************************************************************************
AutoResponse_ConfigureLoggerParams  

Description:
------------
	Set number of logger words, and decide if to send logs for all RX/TX or skip the middle RX
	
**********************************************************************************/
void AutoResponse_ConfigureLoggerParams(bool sendAllLogs, uint8 counterLimit)
{
	RegAutoRespAutoRespLogger_u autoRespLoggerReg;

	// use read-modify-write in order not to touch the other bits of the register
	RegAccess_Read(REG_AUTO_RESP_AUTO_RESP_LOGGER, &autoRespLoggerReg.val);
	autoRespLoggerReg.bitFields.loggerAllCfg 			= sendAllLogs; 	// 1 bit indication if to send all to logger - '1' - send all, '0' - send only rx start/end & tx
	autoRespLoggerReg.bitFields.loggerCounterLimitCfg 	= counterLimit;	// limit of number of words sent to logger before done indication 
	RegAccess_Write(REG_AUTO_RESP_AUTO_RESP_LOGGER, autoRespLoggerReg.val);
}



/**********************************************************************************
AutoResponse_Enable  

Description:
------------
	Enable/disable the auto response block
	
**********************************************************************************/
void AutoResponse_Enable(bool enable)
{
	RegAutoRespAutoRespOutofIdleEn_u autoRespOutofIdleEnReg;

	autoRespOutofIdleEnReg.bitFields.outOfIdleEn = enable;
	RegAccess_Write(REG_AUTO_RESP_AUTO_RESP_OUTOF_IDLE_EN, autoRespOutofIdleEnReg.val);
}


#if defined (ENET_INC_ARCH_WAVE600B)	
/**********************************************************************************
AutoResponse_SetVapAsStaMode

Description:
------------
	set vap bit in REG_AUTO_RESP_PER_VAP_STA_MODE_IND register
	
**********************************************************************************/
void AutoResponse_SetVapAsStaMode(uint8 vapId)
{
	RegAutoRespPerVapStaModeInd_u autoRespPerVapStaModeInd;
	
	RegAccess_Read(REG_AUTO_RESP_PER_VAP_STA_MODE_IND, &autoRespPerVapStaModeInd.val);
	autoRespPerVapStaModeInd.bitFields.perVapStaModeInd |= 1 << vapId;
	RegAccess_Write(REG_AUTO_RESP_PER_VAP_STA_MODE_IND, autoRespPerVapStaModeInd.val);
}
#endif	


/**********************************************************************************
AutoResponse_Init  

Description:
------------
	Initialize the Auto Response module
	
**********************************************************************************/
void AutoResponse_Init(void)
{	
	RegAutoRespAutoRespMiscConfig_u				autoRespMiscConfigReg;
	RegAutoRespMpduDescDlmIdx_u					mpduDescDlmInxReg; 
#ifndef ENET_INC_ARCH_WAVE600B 
	RegAutoRespVapDbTidLimitOffset_u			vapDbTidLimitOffsetReg;
#else
    RegAutoRespStaDbMuliTidEnOffset_u           staDbMuliTidEnOffsetReg;
#endif
	RegAutoRespAutoRespTcs0_u					autoRespTcs0Reg;
	RegAutoRespAutoRespTcs1_u					autoRespTcs1Reg;
	RegAutoRespAutoRespTcs2_u					autoRespTcs2Reg;
	RegAutoRespAutoRespTcs3_u					autoRespTcs3Reg;
	RegAutoRespAutoRespTcs4_u					autoRespTcs4Reg;
	RegAutoRespAutoRespTcs5_u					autoRespTcs5Reg;
	RegAutoRespAutoRespTcs6_u					autoRespTcs6Reg;
	RegAutoRespAutoRespTcs7_u					autoRespTcs7Reg;
	RegAutoRespAutoRespErrFifoBaseAddr_u		autoRespErrFifoBaseAddrReg;
	RegAutoRespAutoRespErrFifoDepthMinusOne_u	autoRespErrFifoDepthMinusOneReg;
	RegAutoRespAutoRespErrFifoEn_u 				autoRespErrFifoEnReg;
	RegAutoRespAutoRespTcrBaseAddr_u			autoRespAutoRespTcrBaseAddrReg;
	RegAutoRespAutoRespHeMuTcrBaseAddr_u		autoRespAutoRespHeMuTcrBaseAddrReg;
#ifdef ENET_INC_ARCH_WAVE600B
    RegAutoRespIntErrEn_u                       autoRespIntErrEn;
#endif

	autoRespMiscConfigReg.val 			= 0;
	mpduDescDlmInxReg.val				= 0; 
#ifndef ENET_INC_ARCH_WAVE600B 
	vapDbTidLimitOffsetReg.val 			= 0;
#else
    staDbMuliTidEnOffsetReg.val         = 0;
#endif
	autoRespTcs0Reg.val					= 0;
	autoRespTcs1Reg.val					= 0;
	autoRespTcs2Reg.val					= 0;
	autoRespTcs3Reg.val					= 0;
	autoRespTcs4Reg.val					= 0;
	autoRespTcs5Reg.val					= 0;
	autoRespTcs6Reg.val					= 0;
	autoRespTcs7Reg.val					= 0;
	autoRespErrFifoBaseAddrReg.val		= 0;
	autoRespErrFifoDepthMinusOneReg.val	= 0;
	autoRespErrFifoEnReg.val			= 0;
	autoRespAutoRespTcrBaseAddrReg.val	= 0;
	autoRespAutoRespHeMuTcrBaseAddrReg.val = 0;
#ifdef ENET_INC_ARCH_WAVE600B
    autoRespIntErrEn.val                = 0;
#endif


	// Init the local database
	memset(&AutoResponseDb, 0, sizeof(AutoResponseDb_t));

	// Misc configuration register
#ifdef ENET_INC_ARCH_WAVE600D2
	autoRespMiscConfigReg.bitFields.supportMultiTidBar 				= TRUE;
#else
	autoRespMiscConfigReg.bitFields.managementTidCounting 			= AUTO_RESPONSE_ADD_MANAGEMENT_TID_TO_MULTI_TID_COUNT_DISABLE; // don't count management TID in the TID limit quota
	autoRespMiscConfigReg.bitFields.swEnSuExtendedBitmap 			= AUTO_RESPONSE_USE_EXTENDED_BITMAP_FOR_SINGLE_USER_ENABLE; // user 8*256 bits for BA for HE SU
#endif
#ifdef WORKAROUND_FOR_HW_BUG_IN_AUTO_RESPONSE
	autoRespMiscConfigReg.bitFields.dynamicBitmapDisable 			= AUTO_RESPONSE_DYNAMIC_BITMAP_SIZE_DISABLE; // Truncate the BA bitmap size the the minimum which is necessary (size = 2^x)
#else
	autoRespMiscConfigReg.bitFields.dynamicBitmapDisable 			= AUTO_RESPONSE_DYNAMIC_BITMAP_SIZE_ENABLE; // Truncate the BA bitmap size the the minimum which is necessary (size = 2^x)
#endif


	autoRespMiscConfigReg.bitFields.barNoBaAgrAckPolicy 			= AUTO_RESPONSE_ACK_ON_BAR_DISABLE; // Don't send ACK on BAR when there is no Traffic Stream open.
	autoRespMiscConfigReg.bitFields.barNumBitsCfg 					= AUTO_RESPONSE_TAKE_WINDOW_SIZE_FROM_BAR_CONTROL;	// take BA bitmap size from "fragment" field in the BAR Controlfrom and not from BA Agreement.
	autoRespMiscConfigReg.bitFields.trafficBaAgreementRecipientEn 	= AUTO_RESPONSE_RECIPIENT_TRAFFIC_INDICATION_BITMAP_UPDATE_ENABLE; // Enable updating of Recipient traffic indication bitmap
	autoRespMiscConfigReg.bitFields.swUsrDbClr 						= AUTO_RESPONSE_FORCE_USER_DB_CLEAR_AT_NEW_RX_DISABLE; // Don't clear the user DB each time if possible
#ifdef ENET_INC_ARCH_WAVE600B 
	autoRespMiscConfigReg.bitFields.acceptFirstSnOutOfBoundsInKeepHist 	= TRUE;
 	autoRespMiscConfigReg.bitFields.heSuUseMbaInstdAck					= FALSE;
 	autoRespMiscConfigReg.bitFields.heMuUseMbaInstdAck					= TRUE;
#endif

	RegAccess_Write(REG_AUTO_RESP_AUTO_RESP_MISC_CONFIG, autoRespMiscConfigReg.val);

	mpduDescDlmInxReg.bitFields.mpduDescFreeListIdx = 0x0; 
	mpduDescDlmInxReg.bitFields.perUserTxBaseDlmListIdx = 0x50; 
	RegAccess_Write(REG_AUTO_RESP_MPDU_DESC_DLM_IDX, mpduDescDlmInxReg.val); 

	
#ifndef ENET_INC_ARCH_WAVE600B 
	// Configure the word number which contains the TID limit of the vap in bits 9:11.
	vapDbTidLimitOffsetReg.bitFields.vapDbTidLimit4BOffset = 11;	// The TID limit is in word 11 (heMultiTidAggSupport)
	RegAccess_Write(REG_AUTO_RESP_VAP_DB_TID_LIMIT_OFFSET, vapDbTidLimitOffsetReg.val);
#else
    staDbMuliTidEnOffsetReg.bitFields.staDbMultiTidEn4BOffset = 11; // The TID limit is in word 11 (heMultiTidAggSupport)
    RegAccess_Write(REG_AUTO_RESP_STA_DB_MULI_TID_EN_OFFSET, staDbMuliTidEnOffsetReg.val); 
#endif
	// Configure the TCSs for Legacy ACK -Eithr RX Phy mode is legacy or HE SU single TID and ACK reply was selected
	autoRespTcs0Reg.bitFields.autoResp0RecipePtr = DeliaRecipes_GetAutoResponseSURecipeAddress();
	RegAccess_Write(REG_AUTO_RESP_AUTO_RESP_TCS0, autoRespTcs0Reg.val);

	// Configure the TCSs for Legacy Compressed Block Ack - Either RX Phy mode is legacy or HE SU single TID and Compressed Block ACK reply was selected
	autoRespTcs1Reg.bitFields.autoResp1RecipePtr = DeliaRecipes_GetAutoResponseSURecipeAddress();
	RegAccess_Write(REG_AUTO_RESP_AUTO_RESP_TCS1, autoRespTcs1Reg.val);

	// Configure the TCSs for HE SU Multi TID. M-BA reply was selected for Single User Multi TID received
	autoRespTcs2Reg.bitFields.autoResp2RecipePtr = DeliaRecipes_GetAutoResponseSURecipeAddress();
	RegAccess_Write(REG_AUTO_RESP_AUTO_RESP_TCS2, autoRespTcs2Reg.val);

	// Configure the TCSs for HE MU Broadcast  M-BA reply was selected for Multi User received in Broadcast transmit format
	autoRespTcs3Reg.bitFields.autoResp3RecipePtr = DeliaRecipes_GetAutoResponseMUBroadcastRecipeAddress();
	RegAccess_Write(REG_AUTO_RESP_AUTO_RESP_TCS3, autoRespTcs3Reg.val);

	// Configure the TCSs for HE MU Unicast  Each User reply can be ACK/C-BA/M-BA for Multi User recieved in Unicast transmit format
	autoRespTcs4Reg.bitFields.autoResp4RecipePtr = DeliaRecipes_GetAutoResponseMUUnicastRecipeAddress();
	RegAccess_Write(REG_AUTO_RESP_AUTO_RESP_TCS4, autoRespTcs4Reg.val);

	// Configure the TCSs for CTS - Reply for RTS received
	autoRespTcs5Reg.bitFields.autoResp5RecipePtr = DeliaRecipes_GetAutoResponseSURecipeAddress();
	autoRespTcs5Reg.bitFields.autoResp5RxNavAbort = TRUE;
	autoRespTcs5Reg.bitFields.autoResp5StaticBwAbort = TRUE;
	RegAccess_Write(REG_AUTO_RESP_AUTO_RESP_TCS5, autoRespTcs5Reg.val);

#if defined (ENET_INC_ARCH_WAVE600B)
	// Configure the TCSs for BF reply
	autoRespTcs6Reg.bitFields.autoResp6RecipePtr = DeliaRecipes_GetBfReportRecipeAddress();
	autoRespTcs6Reg.bitFields.autoResp6BfAbort = TRUE;

#else 
	autoRespTcs5Reg.bitFields.autoResp5RecipePtr = DeliaRecipes_GetAutoResponseSURecipeAddress();
#endif 
	RegAccess_Write(REG_AUTO_RESP_AUTO_RESP_TCS6, autoRespTcs6Reg.val);

	// Initialize a pointer of the error fifo for this specific band;
#if defined(ENET_INC_LMAC0) 	
	pAutoResErrorFifo = &AutoResponseErrorFifo[0];
#endif
#if defined(ENET_INC_LMAC1) 	
	pAutoResErrorFifo = &AutoResponseErrorFifo_B1[0];
#endif

	// Clear the error fifo in SHRAM
	memset(pAutoResErrorFifo, 0, (sizeof(AutoRespErrFifo_t) * AUTO_RESPONSE_ERROR_FIFO_SIZE));

	// Configure base address of error fifo 
	autoRespErrFifoBaseAddrReg.bitFields.autoRespErrFifoBaseAddr = CONVERT_PHYSICAL_TO_SHRAM_OFFSET((uint32)pAutoResErrorFifo);
	RegAccess_Write(REG_AUTO_RESP_AUTO_RESP_ERR_FIFO_BASE_ADDR, autoRespErrFifoBaseAddrReg.val);

	// Configure the size of error fifo (minus 1)
	autoRespErrFifoDepthMinusOneReg.bitFields.autoRespErrFifoDepthMinusOne = AUTO_RESPONSE_ERROR_FIFO_SIZE - 1;
	RegAccess_Write(REG_AUTO_RESP_AUTO_RESP_ERR_FIFO_DEPTH_MINUS_ONE, autoRespErrFifoDepthMinusOneReg.val);

	// Configure which errors will be indicated in the error fifo. Turn them all on for the ramp up. Later we can ignore that not-interesting ones.
	autoRespErrFifoEnReg.bitFields.autoRespErrFifoMltiTidEn				= TRUE;	// multi TID error enable - a user has passed the num TID's that were allocated to him (by TID aggr limit info) 
	autoRespErrFifoEnReg.bitFields.autoRespErrFifoBarErrEn				= TRUE;	// BAR error enable - In HE, a BAR for TID that was received in the same session.
	autoRespErrFifoEnReg.bitFields.autoRespErrFifoSnOutBoundEn			= TRUE;	// NUM SN out of bounds error enable - counter per user/TID that holds the number of SNs received that were out of bounds  (SSN+2^11<SN < SSN)
#ifdef ENET_INC_ARCH_WAVE600D2	
	autoRespErrFifoEnReg.bitFields.autoRespErrFifoMultiTidBarLengthEn 	= TRUE; // AUTO RESP ERROR FIFO MULTI TID BAR LENGTH error enable
	autoRespErrFifoEnReg.bitFields.autoRespErrFifoImmToNonImmEn			= TRUE; // AUTO RESP ERROR FIFO Immediate to Non-Immediate error STA sends TX and expects immediate response and then sends data and requests non-immediate response, in this case we will not save info regarding the received packet and will not respond at all
#endif	

	RegAccess_Write(REG_AUTO_RESP_AUTO_RESP_ERR_FIFO_EN, autoRespErrFifoEnReg.val);

	// Configure the pointer (byte address, word aligned) to the first common TCR in the TX recipes memory
	autoRespAutoRespTcrBaseAddrReg.bitFields.autoRespTcrBaseAddr = (uint32)pDeliaAutoResponseTcrs;
	RegAccess_Write(REG_AUTO_RESP_AUTO_RESP_TCR_BASE_ADDR, autoRespAutoRespTcrBaseAddrReg.val);
#if defined(ENET_INC_LMAC0) 	
	autoRespAutoRespHeMuTcrBaseAddrReg.bitFields.autoRespHeMuTcrBaseAddr = (uint32)(&TxPreAggregatorGeneratedTcrsDataStructure1);
#endif
#if defined(ENET_INC_LMAC1) 	
	autoRespAutoRespHeMuTcrBaseAddrReg.bitFields.autoRespHeMuTcrBaseAddr = (uint32)(&TxPreAggregatorGeneratedTcrsDataStructure1_B1);
#endif
	RegAccess_Write(REG_AUTO_RESP_AUTO_RESP_HE_MU_TCR_BASE_ADDR,autoRespAutoRespHeMuTcrBaseAddrReg.val);

#ifdef ENET_INC_ARCH_WAVE600B
    // Enable Auto Response error interrupts
    autoRespIntErrEn.bitFields.enAutoRespErrFifoFullDropIrq              = TRUE;
    autoRespIntErrEn.bitFields.enAutoRespErrFifoDecrementLessThanZeroIrq = TRUE;
    RegAccess_Write(REG_AUTO_RESP_INT_ERR_EN, autoRespIntErrEn.val);
#endif

	// Enable the Auto Response module (it didn't work until this configuration is done)
	AutoResponse_Enable(TRUE);
}

