/***********************************************************************************
 File:			BssManager_StaManager.c
 Module:		BSS manager STA Manager
 Purpose: 		handle STA Manager events
 Description:   
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
	//TBD to check which files can be removed
#include "BSSmanager_API.h"
#include "TsManager_API.h"
#include "HostInterface_API.h"
#include "Utils_Api.h"
#include "OSAL_Kmsg.h"
#include "ErrorHandler_Api.h"
#include "HwQManager_API.h"
#include "HwGlobalDefinitions.h"
#include "ShramPacketDescriptors.h"
#include "stringLibApi.h"
#include "um_interface.h"
#include "PacketDescriptor.h"
#include "HwMemoryMap.h"
#include "StaDatabase_Api.h"
#include "BssManager_StaManager.h"
#include "loggerAPI.h"
#ifdef ENET_INC_ARCH_WAVE600
#include "ActionPoints_api.h"
#endif
#include "Pac_Api.h"
#include "OSAL_Tasks.h"
#include "OSAL_UpperMacMessages.h"
#include "ShramStationDatabase.h"
#include "RxHandler_ScratchPadApi.h"
#include "RxHandler_InitApi.h"
#include "RegAccess_Api.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_BSS_MANAGER
#define LOG_LOCAL_FID 1


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
/*The BSS Manager Station manager is trigereed by the following Driver events
STA Open
STA Set Filter
STA Stop Traffic
STA Close
It is also trigerred by module confirmation event*/

/*BSS Manager STA Manager states
There are two types of states - 
steady states
transition states - action state and post action state
transitions states - when State Machine Moves from a steady state to the transitions states it calls the modules that registered to them 
*/
typedef enum
{
	BSS_MANAGER_STA_MANAGER_CLOSED_STATE,
	BSS_MANAGER_STA_MANAGER_OPENING_STATE,
	BSS_MANAGER_STA_MANAGER_POST_OPENING_STATE,
	BSS_MANAGER_STA_MANAGER_WAIT_SET_FILTER,
	BSS_MANAGER_STA_MANAGER_SET_FILTER_STATE,
	BSS_MANAGER_STA_MANAGER_POST_SET_FILTER_STATE,
	BSS_MANAGER_STA_MANAGER_OPEN_STATE,
	BSS_MANAGER_STA_MANAGER_STOP_TRAFFIC_STATE,
	BSS_MANAGER_STA_MANAGER_POST_STOP_TRAFFIC_STATE,
	BSS_MANAGER_STA_MANAGER_WAIT_CLOSING_STATE,
	BSS_MANAGER_STA_MANAGER_CLOSING_STATE,
	BSS_MANAGER_STA_MANAGER_POST_CLOSING_STATE,
	BSS_MANAGER_STA_MANAGER_NUM_STATES
} bssManagerStaManager_state;

/*STA Database*/
typedef struct BssManagerStaDb_s
{
	bssManagerStaManager_state 	state;				/*state*/
	uint8					  	sid;				/*SID*/
	uint32						pendingBitmap;		/*pending clients for confirmation*/
	K_MSG *						psMsg;				/*The driver message we are currently handling*/
} BssManagerStaDb_t;


/*Registration Client Info*/
typedef struct BssManagerStaManagerRegClient_s
{
	uint16 taskId;
	uint16 msgId;
} BssManagerStaManagerRegClient_t;

/*Registration DB*/
typedef struct BssManagerStaManagerRegDb_s
{
	/*Number of Clients*/
	uint8				  			numClients[BSS_MANAGER_STA_NUM_REG];
	/*Client DB*/
	BssManagerStaManagerRegClient_t clientDb[BSS_MANAGER_STA_NUM_REG][BSS_MANAGER_STA_MANAGER_MAX_CLIENTS];
} BssManagerStaManagerRegDb_t;

/*State function prototype*/
typedef void (*BssManager_StaManagerFunc)(BssManagerStaDb_t *staDb);

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void BssManager_StaManagerChangeState(BssManagerStaDb_t *StaDb, bssManagerStaManager_state state);
static void BssManager_StaManagerDispatcher( BssManagerStaDb_t *StaDb, bssManagerRegTypes regType);
static void BssManager_CheckDispatcher(BssManagerStaDb_t *StaDb);
static void BssManager_StaManagerOpeningDone(BssManagerStaDb_t *StaDb);
static void BssManager_StaManagerSetFilterDone(BssManagerStaDb_t *StaDb);
static void BssManager_StaManagerStopTrafficDone(BssManagerStaDb_t *StaDb);
static void BssManager_StaManagerClosingDone(BssManagerStaDb_t *StaDb);
static void BssManager_StaManagerCheckConfirm(BssManagerStaDb_t *StaDb);
static void BssManager_StaManagerSetFilterStopTraffic(BssManagerStaDb_t *StaDb);
static void BssManager_StaManagerFatal(BssManagerStaDb_t *StaDb);

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

/*STA Open Event Handlers*/
static const BssManager_StaManagerFunc BssManager_StaManagerStaOpen[BSS_MANAGER_STA_MANAGER_NUM_STATES] =
{
	BssManager_CheckDispatcher, 						/* BSS_MANAGER_STA_MANAGER_CLOSED */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_OPENING */
 	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_POST_OPENING */
	BssManager_StaManagerFatal,		  					/* BSS_MANAGER_STA_MANAGER_WAIT_SET_FILTER */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_SET_FILTER */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_POST_SET_FILTER */
	BssManager_StaManagerFatal, 						/* BSS_MANAGER_STA_MANAGER_OPEN */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_STOP_TRAFFIC */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_POST_STOP_TRAFFIC */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_WAIT_CLOSING */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_CLOSING */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_POST_CLOSING */
};

/*STA Set Key Event Handlers*/
static const BssManager_StaManagerFunc BssManager_StaManagerSetFilter[BSS_MANAGER_STA_MANAGER_NUM_STATES] =
{
	BssManager_StaManagerFatal, 						/* BSS_MANAGER_STA_MANAGER_CLOSED */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_OPENING */
 	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_POST_OPENING */
	BssManager_CheckDispatcher,							/* BSS_MANAGER_STA_MANAGER_WAIT_SET_FILTER */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_SET_FILTER */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_POST_SET_FILTER */
	BssManager_StaManagerFatal, 						/* BSS_MANAGER_STA_MANAGER_OPEN */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_STOP_TRAFFIC */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_POST_STOP_TRAFFIC */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_WAIT_CLOSING */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_CLOSING */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_POST_CLOSING */
};

/*STA Stop Traffic Event Handlers*/
static const BssManager_StaManagerFunc BssManager_StaManagerStopTraffic[BSS_MANAGER_STA_MANAGER_NUM_STATES] =
{
	BssManager_StaManagerFatal, 						/* BSS_MANAGER_STA_MANAGER_CLOSED */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_OPENING */
 	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_POST_OPENING */
	BssManager_StaManagerSetFilterStopTraffic,		  	/* BSS_MANAGER_STA_MANAGER_WAIT_SET_FILTER */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_SET_FILTER */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_POST_SET_FILTER */
	BssManager_CheckDispatcher, 						/* BSS_MANAGER_STA_MANAGER_OPEN */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_STOP_TRAFFIC */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_POST_STOP_TRAFFIC */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_WAIT_CLOSING */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_CLOSING */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_POST_CLOSING */
};

/*STA Close Event Handlers*/
static const BssManager_StaManagerFunc BssManager_StaManagerStaClose[BSS_MANAGER_STA_MANAGER_NUM_STATES] =
{
	BssManager_StaManagerFatal, 						/* BSS_MANAGER_STA_MANAGER_CLOSED */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_OPENING */
 	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_POST_OPENING */
	BssManager_StaManagerFatal,		  					/* BSS_MANAGER_STA_MANAGER_WAIT_SET_FILTER */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_SET_FILTER */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_POST_SET_FILTER */
	BssManager_StaManagerFatal, 						/* BSS_MANAGER_STA_MANAGER_OPEN */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_STOP_TRAFFIC */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_POST_STOP_TRAFFIC */
	BssManager_CheckDispatcher,							/* BSS_MANAGER_STA_MANAGER_WAIT_CLOSING */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_CLOSING */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_POST_CLOSING */
};

/*STA Confirmation Event Handlers*/
static const BssManager_StaManagerFunc BssManager_StaManagerStaConfirm[BSS_MANAGER_STA_MANAGER_NUM_STATES] =
{
	BssManager_StaManagerFatal, 						/* BSS_MANAGER_STA_MANAGER_CLOSED */
	BssManager_StaManagerCheckConfirm,					/* BSS_MANAGER_STA_MANAGER_OPENING */
 	BssManager_StaManagerCheckConfirm,					/* BSS_MANAGER_STA_MANAGER_POST_OPENING */
	BssManager_StaManagerFatal,		  					/* BSS_MANAGER_STA_MANAGER_WAIT_SET_FILTER */
	BssManager_StaManagerCheckConfirm,					/* BSS_MANAGER_STA_MANAGER_SET_FILTER */
	BssManager_StaManagerCheckConfirm,					/* BSS_MANAGER_STA_MANAGER_POST_SET_FILTER */
	BssManager_StaManagerFatal, 						/* BSS_MANAGER_STA_MANAGER_OPEN */
	BssManager_StaManagerCheckConfirm,					/* BSS_MANAGER_STA_MANAGER_STOP_TRAFFIC */
	BssManager_StaManagerCheckConfirm,					/* BSS_MANAGER_STA_MANAGER_POST_STOP_TRAFFIC */
	BssManager_StaManagerFatal,							/* BSS_MANAGER_STA_MANAGER_WAIT_CLOSING */
	BssManager_StaManagerCheckConfirm,					/* BSS_MANAGER_STA_MANAGER_CLOSING */
	BssManager_StaManagerCheckConfirm,					/* BSS_MANAGER_STA_MANAGER_POST_CLOSING */
};

/*Handlers to be called when we are done with an external request - may be called from the appropiate steady state or its associated transition states*/
static const BssManager_StaManagerFunc BssManager_StaManagerDone[BSS_MANAGER_STA_MANAGER_NUM_STATES] = 
{
	BssManager_StaManagerOpeningDone,					/* BSS_MANAGER_STA_MANAGER_CLOSED */
	BssManager_StaManagerOpeningDone,					/* BSS_MANAGER_STA_MANAGER_OPENING */
 	BssManager_StaManagerOpeningDone,					/* BSS_MANAGER_STA_MANAGER_POST_OPENING */
	BssManager_StaManagerSetFilterDone,					/* BSS_MANAGER_STA_MANAGER_WAIT_SET_FILTER */
	BssManager_StaManagerSetFilterDone,					/* BSS_MANAGER_STA_MANAGER_SET_FILTER */
	BssManager_StaManagerSetFilterDone,					/* BSS_MANAGER_STA_MANAGER_POST_SET_FILTER */
	BssManager_StaManagerStopTrafficDone, 				/* BSS_MANAGER_STA_MANAGER_OPEN */
	BssManager_StaManagerStopTrafficDone,				/* BSS_MANAGER_STA_MANAGER_STOP_TRAFFIC */
	BssManager_StaManagerStopTrafficDone,				/* BSS_MANAGER_STA_MANAGER_POST_STOP_TRAFFIC */
	BssManager_StaManagerClosingDone,					/* BSS_MANAGER_STA_MANAGER_WAIT_CLOSING */
	BssManager_StaManagerClosingDone,					/* BSS_MANAGER_STA_MANAGER_CLOSING */
	BssManager_StaManagerClosingDone,					/* BSS_MANAGER_STA_MANAGER_POST_CLOSING */
};

/*Lookup table that associates states to their transitions states*/
static const bssManagerStaManager_state BssManager_StaManagerStateToTransitionState[BSS_MANAGER_STA_MANAGER_NUM_STATES] = 
{
	BSS_MANAGER_STA_MANAGER_OPENING_STATE,				/* BSS_MANAGER_STA_MANAGER_CLOSED */
	BSS_MANAGER_STA_MANAGER_NUM_STATES,					/* BSS_MANAGER_STA_MANAGER_OPENING_STATE */
 	BSS_MANAGER_STA_MANAGER_NUM_STATES,					/* BSS_MANAGER_STA_MANAGER_POST_OPENING */
	BSS_MANAGER_STA_MANAGER_SET_FILTER_STATE,			/* BSS_MANAGER_STA_MANAGER_WAIT_SET_FILTER */
	BSS_MANAGER_STA_MANAGER_NUM_STATES,					/* BSS_MANAGER_STA_MANAGER_SET_FILTER */
	BSS_MANAGER_STA_MANAGER_NUM_STATES,					/* BSS_MANAGER_STA_MANAGER_POST_SET_FILTER */
	BSS_MANAGER_STA_MANAGER_STOP_TRAFFIC_STATE,			/* BSS_MANAGER_STA_MANAGER_OPEN */
	BSS_MANAGER_STA_MANAGER_NUM_STATES,					/* BSS_MANAGER_STA_MANAGER_STOP_TRAFFIC */
	BSS_MANAGER_STA_MANAGER_NUM_STATES,					/* BSS_MANAGER_STA_MANAGER_POST_STOP_TRAFFIC */
	BSS_MANAGER_STA_MANAGER_CLOSING_STATE,				/* BSS_MANAGER_STA_MANAGER_WAIT_CLOSING */
	BSS_MANAGER_STA_MANAGER_NUM_STATES,					/* BSS_MANAGER_STA_MANAGER_CLOSING */
	BSS_MANAGER_STA_MANAGER_NUM_STATES,					/* BSS_MANAGER_STA_MANAGER_POST_CLOSING */
};

/*Lookup table that associates a transition state to the appropiate registration type*/
static const bssManagerRegTypes BssManager_StaManagerStateToRegType[BSS_MANAGER_STA_MANAGER_NUM_STATES] = 
{
	BSS_MANAGER_STA_NUM_REG,							/* BSS_MANAGER_STA_MANAGER_CLOSED */
	BSS_MANAGER_STA_OPEN_REG,							/* BSS_MANAGER_STA_MANAGER_OPENING */
 	BSS_MANAGER_STA_POST_OPEN_REG,						/* BSS_MANAGER_STA_MANAGER_POST_OPENING */
	BSS_MANAGER_STA_NUM_REG,							/* BSS_MANAGER_STA_MANAGER_WAIT_SET_FILTER */
	BSS_MANAGER_STA_SET_FILTER_REG,						/* BSS_MANAGER_STA_MANAGER_SET_FILTER */
	BSS_MANAGER_STA_POST_SET_FILTER_REG,				/* BSS_MANAGER_STA_MANAGER_POST_SET_FILTER */
	BSS_MANAGER_STA_NUM_REG, 							/* BSS_MANAGER_STA_MANAGER_OPEN */
	BSS_MANAGER_STA_STOP_TRAFFIC_REG,					/* BSS_MANAGER_STA_MANAGER_STOP_TRAFFIC */
	BSS_MANAGER_STA_POST_STOP_TRAFFIC_REG,				/* BSS_MANAGER_STA_MANAGER_POST_STOP_TRAFFIC */
	BSS_MANAGER_STA_NUM_REG,							/* BSS_MANAGER_STA_MANAGER_WAIT_CLOSING */
	BSS_MANAGER_STA_CLOSING_REG,						/* BSS_MANAGER_STA_MANAGER_CLOSING */
	BSS_MANAGER_STA_POST_CLOSING_REG,					/* BSS_MANAGER_STA_MANAGER_POST_CLOSING */
};

/*Lookup table that associates a transition action state to the appropiate transition post action state*/
static const bssManagerStaManager_state BssManager_StaManagerStateToPostState[BSS_MANAGER_STA_MANAGER_NUM_STATES] = 
{
	BSS_MANAGER_STA_MANAGER_NUM_STATES,					/* BSS_MANAGER_STA_MANAGER_CLOSED */
	BSS_MANAGER_STA_MANAGER_POST_OPENING_STATE,			/* BSS_MANAGER_STA_MANAGER_OPENING_STATE */
 	BSS_MANAGER_STA_MANAGER_NUM_STATES,					/* BSS_MANAGER_STA_MANAGER_POST_OPENING */
	BSS_MANAGER_STA_MANAGER_NUM_STATES,					/* BSS_MANAGER_STA_MANAGER_WAIT_SET_FILTER */
	BSS_MANAGER_STA_MANAGER_POST_SET_FILTER_STATE,		/* BSS_MANAGER_STA_MANAGER_SET_FILTER */
	BSS_MANAGER_STA_MANAGER_NUM_STATES,					/* BSS_MANAGER_STA_MANAGER_POST_SET_FILTER */
	BSS_MANAGER_STA_MANAGER_NUM_STATES,					/* BSS_MANAGER_STA_MANAGER_OPEN */
	BSS_MANAGER_STA_MANAGER_POST_STOP_TRAFFIC_STATE,	/* BSS_MANAGER_STA_MANAGER_STOP_TRAFFIC */
	BSS_MANAGER_STA_MANAGER_NUM_STATES,					/* BSS_MANAGER_STA_MANAGER_POST_STOP_TRAFFIC */
	BSS_MANAGER_STA_MANAGER_NUM_STATES,					/* BSS_MANAGER_STA_MANAGER_WAIT_CLOSING */
	BSS_MANAGER_STA_MANAGER_POST_CLOSING_STATE,			/* BSS_MANAGER_STA_MANAGER_CLOSING */
	BSS_MANAGER_STA_MANAGER_NUM_STATES,					/* BSS_MANAGER_STA_MANAGER_POST_CLOSING */
};

/*Local Databases*/
/*Registration DB*/
static BssManagerStaManagerRegDb_t BssManager_RegDb = 
{
	{//numClients[BSS_MANAGER_STA_NUM_REG];
		NUM_OF_CLIENT_TO_BSS_MANAGER_STA_OPEN_REG,
		NUM_OF_CLIENT_TO_BSS_MANAGER_STA_POST_OPEN_REG,
		NUM_OF_CLIENT_TO_BSS_MANAGER_STA_SET_FILTER_REG,
		NUM_OF_CLIENT_TO_BSS_MANAGER_STA_POST_SET_FILTER_REG,
		NUM_OF_CLIENT_TO_BSS_MANAGER_STA_STOP_TRAFFIC_REG,
		NUM_OF_CLIENT_TO_BSS_MANAGER_STA_POST_STOP_TRAFFIC_REG,
		NUM_OF_CLIENT_TO_BSS_MANAGER_STA_CLOSING_REG,
		NUM_OF_CLIENT_TO_BSS_MANAGER_STA_POST_CLOSING_REG

	},
	{//clientDb[BSS_MANAGER_STA_NUM_REG][BSS_MANAGER_STA_MANAGER_MAX_CLIENTS]
		//BSS_MANAGER_STA_OPEN_REG
		{
		{TASK_LINK_ADAPTATION,		LINK_ADAPTATION_ADD_STATION_REQ		},//BSS_MANAGER_STA_MANAGER_LA_CLIENT,
		{TASK_COC,					OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_SMPS_CLIENT,
		{TASK_TS_MANAGER,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_TS_CLIENT, 
		{TASK_STATISTICS_MANAGER,	STATISTICS_MANAGER_ADD_STA			},//BSS_MANAGER_STA_MANAGER_STAT_CLIENT,
		{TASK_QOS,					QOS_ADD_STATION						},//BSS_MANAGER_STA_MANAGER_QOS_CLIENT,
		{TASK_HIM,					OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_HIM_CLIENT,
		{TASK_TX_MANAGER,			CSA_MANAGER_STA_OPEN				},//BSS_MANAGER_STA_MANAGER_CSA_CLIENT,
#ifdef USE_AGER_EMULATOR
		{TASK_AGER_EMULATOR,		AGER_EMULATOR_ADD_STATION			},//BSS_MANAGER_STA_MANAGER_AGER_CLIENT,
#endif //USE_AGER_EMULATOR
		{TASK_ENCRYPTION,			ENCRYPTION_ADD_STA					},//BSS_MANAGER_STA_MANAGER_SEC_CLIENT,
		{TASK_TX_MANAGER,			TX_MANAGER_STA_OPEN					},//BSS_MANAGER_STA_MANAGER_TX_CLIENT,
		{TASK_TX_MANAGER,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_TXM_CLIENT,
		{TASK_RX_MANAGER,			RX_MANAGER_STA_OPEN					},//BSS_MANAGER_STA_MANAGER_RXM_CLIENT,
		{TASK_GROUP_MANAGER,		OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_GM_CLIENT
		{TASK_ATF,					ATF_MANAGER_BSS_ADD_STA 			},//BSS_MANAGER_STA_MANAGER_ATF_CLIENT
#ifdef ENET_INC_ARCH_WAVE600
		{TASK_PAC_MANAGER,			PAC_MANAGER_ADD_STATION_REQ 		},//BSS_MANAGER_STA_MANAGER_PAC_CLIENT
		{TASK_HE_GROUP_MANAGER,		OSAL_TASK_UNUSED_MSG 				},//BSS_MANAGER_STA_MANAGER_HE_GM_CLIENT
		{TASK_TWT_MANAGER,          TWT_MANAGER_ADD_STA                 },//BSS_MANAGER_STA_MANAGER_TWT_CLIENT	
#endif
		{TASK_HDK,	            	HDK_ERP_STATION_ADD					},//BSS_MANAGER_STA_MANAGER_HDK_CLIENT
		},

		//BSS_MANAGER_STA_POST_OPEN_REG
		{
		{TASK_LINK_ADAPTATION,		OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_LA_CLIENT,
		{TASK_TX_MANAGER,			SMPS_ADD_STA						},//BSS_MANAGER_STA_MANAGER_SMPS_CLIENT,
		{TASK_TS_MANAGER,			TS_MANAGER_ADD_STATION				},//BSS_MANAGER_STA_MANAGER_TS_CLIENT, 
		{TASK_STATISTICS_MANAGER,	OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_STAT_CLIENT,
		{TASK_QOS,					OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_QOS_CLIENT,
		{TASK_HIM,					HIM_STA_ADD							},//BSS_MANAGER_STA_MANAGER_HIM_CLIENT,
		{TASK_TX_MANAGER,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_CSA_CLIENT,
#ifdef USE_AGER_EMULATOR
		{TASK_AGER_EMULATOR,		OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_AGER_CLIENT,
#endif //USE_AGER_EMULATOR
		{TASK_ENCRYPTION,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_SEC_CLIENT,
		{TASK_TX_MANAGER,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_TX_CLIENT,
		{TASK_TX_MANAGER,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_TXM_CLIENT,
		{TASK_RX_MANAGER,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_RXM_CLIENT,
		{TASK_GROUP_MANAGER,		GROUP_MANAGER_ADD_STATION			},//BSS_MANAGER_STA_MANAGER_GM_CLIENT
		{TASK_ATF,	            	OSAL_TASK_UNUSED_MSG	        	},//BSS_MANAGER_STA_MANAGER_ATF_CLIENT		
#ifdef ENET_INC_ARCH_WAVE600
		{TASK_PAC_MANAGER,			OSAL_TASK_UNUSED_MSG 				},//BSS_MANAGER_STA_MANAGER_PAC_CLIENT
		{TASK_HE_GROUP_MANAGER, 	HE_GROUP_MANAGER_ADD_STATION		},//BSS_MANAGER_STA_MANAGER_HE_GM_CLIENT
		{TASK_TWT_MANAGER,          OSAL_TASK_UNUSED_MSG                },//BSS_MANAGER_STA_MANAGER_TWT_CLIENT
#endif
		{TASK_HDK,	            	OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_HDK_CLIENT
		},

		//BSS_MANAGER_STA_SET_FILTER_REG 
		{
		{TASK_LINK_ADAPTATION,		LINK_ADAPTATION_SET_FILTER			},//BSS_MANAGER_STA_MANAGER_LA_CLIENT,
		{TASK_COC,					OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_SMPS_CLIENT,
		{TASK_TS_MANAGER,			TS_MANAGER_SET_FILTER				},//BSS_MANAGER_STA_MANAGER_TS_CLIENT, 
		{TASK_STATISTICS_MANAGER,	OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_STAT_CLIENT,
		{TASK_QOS,					OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_QOS_CLIENT,
		{TASK_HIM,					OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_HIM_CLIENT,
		{TASK_TX_MANAGER,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_CSA_CLIENT,
#ifdef USE_AGER_EMULATOR
		{TASK_AGER_EMULATOR,		OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_AGER_CLIENT,
#endif //USE_AGER_EMULATOR
		{TASK_ENCRYPTION,			ENCRYPTION_SET_FILTER				},//BSS_MANAGER_STA_MANAGER_SEC_CLIENT,
		{TASK_TX_MANAGER,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_TX_CLIENT,
		{TASK_TX_MANAGER,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_TXM_CLIENT,
		{TASK_RX_MANAGER,			RX_MANAGER_STA_SET_FILTER			},//BSS_MANAGER_STA_MANAGER_RXM_CLIENT,
		{TASK_GROUP_MANAGER,		OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_GM_CLIENT
		{TASK_ATF,	            	OSAL_TASK_UNUSED_MSG	        	},//BSS_MANAGER_STA_MANAGER_ATF_CLIENT		
#ifdef ENET_INC_ARCH_WAVE600
		{TASK_PAC_MANAGER,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_PAC_CLIENT
		{TASK_HE_GROUP_MANAGER, 	OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_HE_GM_CLIENT
		{TASK_TWT_MANAGER,          TWT_MANAGER_SET_FILTER              },//BSS_MANAGER_STA_MANAGER_TWT_CLIENT
#endif
		{TASK_HDK,	            	OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_HDK_CLIENT
		},

		//BSS_MANAGER_STA_POST_SET_FILTER_REG
		{
		{TASK_LINK_ADAPTATION,		OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_LA_CLIENT,
		{TASK_COC,					OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_SMPS_CLIENT,
		{TASK_TS_MANAGER,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_TS_CLIENT, 
		{TASK_STATISTICS_MANAGER,	OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_STAT_CLIENT,
		{TASK_QOS,					OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_QOS_CLIENT,
		{TASK_HIM,					HIM_STA_SET_FILTER					},//BSS_MANAGER_STA_MANAGER_HIM_CLIENT,
		{TASK_TX_MANAGER,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_CSA_CLIENT,
#ifdef USE_AGER_EMULATOR
		{TASK_AGER_EMULATOR,		OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_AGER_CLIENT,
#endif //USE_AGER_EMULATOR
		{TASK_ENCRYPTION,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_SEC_CLIENT,
		{TASK_TX_MANAGER,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_TX_CLIENT,
		{TASK_TX_MANAGER,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_TXM_CLIENT,
		{TASK_RX_MANAGER,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_RXM_CLIENT,
		{TASK_GROUP_MANAGER,		OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_GM_CLIENT
		{TASK_ATF,	            	OSAL_TASK_UNUSED_MSG	        	},//BSS_MANAGER_STA_MANAGER_ATF_CLIENT
#ifdef ENET_INC_ARCH_WAVE600
		{TASK_PAC_MANAGER,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_PAC_CLIENT
		{TASK_HE_GROUP_MANAGER, 	OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_HE_GM_CLIENT
		{TASK_TWT_MANAGER,          OSAL_TASK_UNUSED_MSG                },//BSS_MANAGER_STA_MANAGER_TWT_CLIENT
#endif
		{TASK_HDK,	            	OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_HDK_CLIENT
		},
		
		//BSS_MANAGER_STA_STOP_TRAFFIC_REG
		{
		{TASK_LINK_ADAPTATION,		OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_LA_CLIENT,
		{TASK_TX_MANAGER,			SMPS_REMOVE_STA						},//BSS_MANAGER_STA_MANAGER_SMPS_CLIENT,
		{TASK_TS_MANAGER,			TS_MANAGER_REMOVE_STATION			},//BSS_MANAGER_STA_MANAGER_TS_CLIENT, 
		{TASK_STATISTICS_MANAGER,	OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_STAT_CLIENT,
		{TASK_QOS,					QOS_REMOVE_STATION					},//BSS_MANAGER_STA_MANAGER_QOS_CLIENT,
		{TASK_HIM,					HIM_STA_STOP_TRAFFIC				},//BSS_MANAGER_STA_MANAGER_HIM_CLIENT,
		{TASK_TX_MANAGER,			CSA_MANAGER_STA_CLOSE				},//BSS_MANAGER_STA_MANAGER_CSA_CLIENT,
#ifdef USE_AGER_EMULATOR
		{TASK_AGER_EMULATOR,		OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_AGER_CLIENT,
#endif //USE_AGER_EMULATOR
		{TASK_ENCRYPTION,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_SEC_CLIENT,
		{TASK_TX_MANAGER,			TX_MANAGER_STA_STOP_TRAFFIC			},//BSS_MANAGER_STA_MANAGER_TX_CLIENT,
		{TASK_TX_MANAGER,			MULTICAST_HANDLER_STA_STOP_TRAFFIC	},//BSS_MANAGER_STA_MANAGER_TXM_CLIENT,
		{TASK_RX_MANAGER,			RX_MANAGER_STA_STOP_TRAFFIC			},//BSS_MANAGER_STA_MANAGER_RXM_CLIENT,
		{TASK_GROUP_MANAGER,		GROUP_MANAGER_REMOVE_STATION		},//BSS_MANAGER_STA_MANAGER_GM_CLIENT,
		{TASK_ATF,	            	ATF_MANAGER_BSS_STOP_TRAFFIC		},//BSS_MANAGER_STA_MANAGER_ATF_CLIENT
#ifdef ENET_INC_ARCH_WAVE600
		{TASK_PAC_MANAGER,			PAC_MANAGER_REMOVE_STATION_REQ		},//BSS_MANAGER_STA_MANAGER_PAC_CLIENT
		{TASK_HE_GROUP_MANAGER, 	HE_GROUP_MANAGER_REMOVE_STATION		},//BSS_MANAGER_STA_MANAGER_HE_GM_CLIENT
		{TASK_TWT_MANAGER,	        TWT_MANAGER_REMOVE_STA         		},//BSS_MANAGER_STA_MANAGER_TWT_CLIENT
#endif
		{TASK_HDK,	            	OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_HDK_CLIENT
		},

		//BSS_MANAGER_STA_POST_STOP_TRAFFIC_REG
		{
		{TASK_LINK_ADAPTATION,		OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_LA_CLIENT,
		{TASK_COC,					OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_SMPS_CLIENT,
		{TASK_TS_MANAGER,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_TS_CLIENT, 
		{TASK_STATISTICS_MANAGER,	OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_STAT_CLIENT,
		{TASK_QOS,					OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_QOS_CLIENT,
		{TASK_HIM,					OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_HIM_CLIENT,
		{TASK_TX_MANAGER,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_CSA_CLIENT,
#ifdef USE_AGER_EMULATOR
		{TASK_AGER_EMULATOR,		AGER_EMULATOR_REMOVE_STATION		},//BSS_MANAGER_STA_MANAGER_AGER_CLIENT,
#endif //USE_AGER_EMULATOR
		{TASK_ENCRYPTION,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_SEC_CLIENT,
		{TASK_TX_MANAGER,			TX_MANAGER_STA_POST_STOP_TRAFFIC	},//BSS_MANAGER_STA_MANAGER_TX_CLIENT,
		{TASK_TX_MANAGER,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_TXM_CLIENT,
		{TASK_RX_MANAGER,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_RXM_CLIENT,
		{TASK_GROUP_MANAGER,		OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_GM_CLIENT
		{TASK_ATF,	            	OSAL_TASK_UNUSED_MSG	        	},//BSS_MANAGER_STA_MANAGER_ATF_CLIENT		
#ifdef ENET_INC_ARCH_WAVE600
		{TASK_PAC_MANAGER,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_PAC_CLIENT
		{TASK_HE_GROUP_MANAGER, 	OSAL_TASK_UNUSED_MSG 				},//BSS_MANAGER_STA_MANAGER_HE_GM_CLIENT
		{TASK_TWT_MANAGER,          OSAL_TASK_UNUSED_MSG                },//BSS_MANAGER_STA_MANAGER_TWT_CLIENT
#endif
		{TASK_HDK,	            	OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_HDK_CLIENT
		},

		//BSS_MANAGER_STA_CLOSING_REG
		{
		{TASK_LINK_ADAPTATION,		LINK_ADAPTATION_REMOVE_STATION_REQ	},//BSS_MANAGER_STA_MANAGER_LA_CLIENT,
		{TASK_COC,					OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_SMPS_CLIENT,
		{TASK_TS_MANAGER,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_TS_CLIENT, 
		{TASK_STATISTICS_MANAGER,	STATISTICS_MANAGER_REMOVE_STA		},//BSS_MANAGER_STA_MANAGER_STAT_CLIENT,
		{TASK_QOS,					OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_QOS_CLIENT,
		{TASK_HIM,					HIM_STA_REMOVE						},//BSS_MANAGER_STA_MANAGER_HIM_CLIENT,
		{TASK_TX_MANAGER,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_CSA_CLIENT,
#ifdef USE_AGER_EMULATOR
		{TASK_AGER_EMULATOR,		OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_AGER_CLIENT,
#endif //USE_AGER_EMULATOR
		{TASK_ENCRYPTION,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_SEC_CLIENT,
		{TASK_TX_MANAGER,			TX_MANAGER_STA_CLOSE				},//BSS_MANAGER_STA_MANAGER_TX_CLIENT,
		{TASK_TX_MANAGER,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_TXM_CLIENT,
		{TASK_RX_MANAGER,			RX_MANAGER_REMOVE_STATION			},//BSS_MANAGER_STA_MANAGER_RXM_CLIENT,
		{TASK_GROUP_MANAGER,		OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_GM_CLIENT
		{TASK_ATF,	            	OSAL_TASK_UNUSED_MSG	        	},//BSS_MANAGER_STA_MANAGER_ATF_CLIENT		
#ifdef ENET_INC_ARCH_WAVE600
		{TASK_PAC_MANAGER,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_PAC_CLIENT
		{TASK_HE_GROUP_MANAGER, 	OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_HE_GM_CLIENT
		{TASK_TWT_MANAGER,          OSAL_TASK_UNUSED_MSG                },//BSS_MANAGER_STA_MANAGER_TWT_CLIENT
#endif
		{TASK_HDK,	            	HDK_ERP_STATION_REMOVE				},//BSS_MANAGER_STA_MANAGER_HDK_CLIENT

		},

		//BSS_MANAGER_STA_POST_CLOSING_REG
		{
		{TASK_LINK_ADAPTATION,		OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_LA_CLIENT,
		{TASK_COC,					OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_SMPS_CLIENT,
		{TASK_TS_MANAGER,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_TS_CLIENT, 
		{TASK_STATISTICS_MANAGER,	OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_STAT_CLIENT,
		{TASK_QOS,					OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_QOS_CLIENT,
		{TASK_HIM,					OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_HIM_CLIENT,
		{TASK_TX_MANAGER,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_CSA_CLIENT,
#ifdef USE_AGER_EMULATOR
		{TASK_AGER_EMULATOR,		OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_AGER_CLIENT,
#endif //USE_AGER_EMULATOR
		{TASK_ENCRYPTION,			ENCRYPTION_REMOVE_STA				},//BSS_MANAGER_STA_MANAGER_SEC_CLIENT,
		{TASK_TX_MANAGER,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_TX_CLIENT,
		{TASK_TX_MANAGER,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_TXM_CLIENT,
		{TASK_RX_MANAGER,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_RXM_CLIENT,
		{TASK_GROUP_MANAGER,		OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_GM_CLIENT
		{TASK_ATF,	            	OSAL_TASK_UNUSED_MSG	        	},//BSS_MANAGER_STA_MANAGER_ATF_CLIENT		
#ifdef ENET_INC_ARCH_WAVE600
		{TASK_PAC_MANAGER,			OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_PAC_CLIENT
		{TASK_HE_GROUP_MANAGER, 	OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_HE_GM_CLIENT
		{TASK_TWT_MANAGER,          OSAL_TASK_UNUSED_MSG                },//BSS_MANAGER_STA_MANAGER_TWT_CLIENT
#endif
		{TASK_HDK,	            	OSAL_TASK_UNUSED_MSG				},//BSS_MANAGER_STA_MANAGER_HDK_CLIENT
		},		
	}
};
/*STA DB*/
static BssManagerStaDb_t BssManager_StaDb[HW_NUM_OF_STATIONS];

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

BssManager_StaManagerChangeState 


Description:
------------
Utility function called when STA state changes - logs the SID, current state and new state


Input: 
-----	
bssManagerStaManager_state - state
BssManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void BssManager_StaManagerChangeState(BssManagerStaDb_t *StaDb, bssManagerStaManager_state state)
{
	ILOG0_DDD("BSS Manager STA Manager, Station %d changed from %d to %d", StaDb->sid, StaDb->state, state);
	StaDb->state = state;
}

/**********************************************************************************

BssManager_StaManagerDispatcher 


Description:
------------
Calls clients registered with registration type
Per client- allocates message, sets contents and sends registered message ID to registered task ID


Input: 
-----	
bssManagerRegTypes - reg type
BssManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void BssManager_StaManagerDispatcher(BssManagerStaDb_t *StaDb, bssManagerRegTypes regType)
{
	uint8 i;
	K_MSG *psMsg;
	BssManagerStaManagerReq_t *staManagerReq;
	BssManagerStaManagerRegClient_t *client;
	/*Check registration type validity*/	
	DEBUG_ASSERT(regType < BSS_MANAGER_STA_NUM_REG);
	/*There must be registered clients*/
	DEBUG_ASSERT(BssManager_RegDb.numClients[regType] > 0);
	/*There must be no pending clients for this STA*/
	DEBUG_ASSERT(StaDb->pendingBitmap == 0);
	ILOG2_DDD("BSS MNG, STA MNG, Dispatch Reg %d, SID %d, num %d", regType, StaDb->sid, BssManager_RegDb.numClients[regType]);
	/*Send messages to registered clients*/
	for (i = 0; i < BSS_MANAGER_STA_MANAGER_MAX_CLIENTS; i++)
	{

		client = &BssManager_RegDb.clientDb[regType][i];
		if (client->msgId != OSAL_TASK_UNUSED_MSG)
		{
			/*Allocate message*/
			psMsg = OSAL_GET_MESSAGE(sizeof(BssManagerStaManagerReq_t));
			staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(psMsg);
			/*Set pointer to original message in registration message*/
			staManagerReq->psMsg = StaDb->psMsg;
			/*Mark this client as pending*/
			StaDb->pendingBitmap |= (1 << i);
			ILOG0_DDD("BSS MNG, STA MNG, Dispatch Client %d, MsgId %d, TaskId %d", i, client->msgId, client->taskId);
			/*Send appropiate message to task registered by this client*/
			OSAL_SEND_MESSAGE(client->msgId, client->taskId, psMsg, StaDb->psMsg->header.vapId);
		}
	}
}

/**********************************************************************************

BssManager_CheckDispatcher 


Description:
------------
When handling an event checks if there are any registered entries with the associated transition states
If there are it will change state to transition state and call registered entries
If not calls done handler of current state

Input: 
-----	
BssManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void BssManager_CheckDispatcher(BssManagerStaDb_t *StaDb)
{
	bssManagerStaManager_state state, postState;
	bssManagerRegTypes regType, posRegType;


	ILOG2_DD("BSS MNG, STA MNG, Check Dispatcher, SID %d, state %d", StaDb->sid, StaDb->state);
	/*get the transition state associated with the current state*/
	state = BssManager_StaManagerStateToTransitionState[StaDb->state];
	DEBUG_ASSERT(state < BSS_MANAGER_STA_MANAGER_NUM_STATES);
	/*Get the registration type associated with the transition state*/
	regType = BssManager_StaManagerStateToRegType[state];
	DEBUG_ASSERT(regType < BSS_MANAGER_STA_NUM_REG);
	/*get the post action state of the transition state*/
	postState = BssManager_StaManagerStateToPostState[state];
	DEBUG_ASSERT(postState < BSS_MANAGER_STA_MANAGER_NUM_STATES);
	/*Get the registration type associated with the transition post state*/
	posRegType = BssManager_StaManagerStateToRegType[postState];
	DEBUG_ASSERT(posRegType < BSS_MANAGER_STA_NUM_REG);
	/*Has anyone registered for request?*/
	if (BssManager_RegDb.numClients[regType] > 0)
	{
		/*Change State*/
		BssManager_StaManagerChangeState(StaDb, state);
		/*Call registered clients */
		BssManager_StaManagerDispatcher(StaDb, regType);
	}
	/*Has anyone registered for post request?*/
	else if (BssManager_RegDb.numClients[posRegType] > 0)
	{
		/*Change State to Post State*/
		BssManager_StaManagerChangeState(StaDb, postState);
		/*Call registered clients*/
		BssManager_StaManagerDispatcher(StaDb, posRegType);
	}
	else
	{
		BssManager_StaManagerDone[StaDb->state](StaDb);
	}
}

/**********************************************************************************

BssManager_StaManagerOpeningDone 


Description:
------------
Called when we are done with opening process - including transition states
Moves to next state according to security type

Input: 
-----	
BssManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void BssManager_StaManagerOpeningDone(BssManagerStaDb_t *StaDb)
{
	K_MSG *psMsg;
	UMI_STA_ADD* pStaAdd;
	uint8 isXFilterOpen;
	
	/*Get pointer to message*/
	psMsg = StaDb->psMsg;
	pStaAdd = (UMI_STA_ADD *)pK_MSG_DATA(psMsg);
	pStaAdd->Status = UMI_OK;
	/*Check if STA is encrypted*/
	isXFilterOpen = MTLK_BFIELD_GET(pStaAdd->u8Flags, STA_ADD_FLAGS_IS_8021X_FILTER_OPEN);
	if (isXFilterOpen)
	{
		/*Change State to Open*/
		BssManager_StaManagerChangeState(StaDb, BSS_MANAGER_STA_MANAGER_OPEN_STATE);
	}
	else
	{
		/*Change State to Wait for Set filter*/
		BssManager_StaManagerChangeState(StaDb, BSS_MANAGER_STA_MANAGER_WAIT_SET_FILTER);
	}
	/*Send confirmation to Driver*/
	ILOG2_D("BSS MNG, STA MNG, STA Open Done, SID %d", StaDb->sid);

#ifdef ENET_INC_ARCH_WAVE600

	// Debug for JIRA https://jira-chd.intel.com/browse/WLANRTSYS-6218
	// Currently use 2 action points to catch a bug in which some entity writes "0" to "maxMpduLengthLimit" in STA DB of STA 128/129
	if (pStaAdd->u16SID == 128)
	{
		ActionPoints_Set(ACTION_POINT_0, (uint32)(&StaDbHwEntries[128].common.word6) , 0x0 ,AP_TARGET_MEMORY_ADDRESS, AP_TRANSACTION_TYPE_WRITE, AP_MODE_TRIGGER_IN_RANGE, TRUE, FALSE);
		ActionPoints_Set(ACTION_POINT_1, 0x0, 0xFFFFC000 ,AP_TARGET_MEMORY_DATA, AP_TRANSACTION_TYPE_WRITE, AP_MODE_TRIGGER_IN_RANGE, FALSE, FALSE);
	}
	if (pStaAdd->u16SID == 129)
	{
		ActionPoints_Set(ACTION_POINT_2, (uint32)(&StaDbHwEntries[129].common.word6) , 0x0 ,AP_TARGET_MEMORY_ADDRESS, AP_TRANSACTION_TYPE_WRITE, AP_MODE_TRIGGER_IN_RANGE, TRUE, FALSE);
		ActionPoints_Set(ACTION_POINT_3, 0x0, 0xFFFFC000 ,AP_TARGET_MEMORY_DATA, AP_TRANSACTION_TYPE_WRITE, AP_MODE_TRIGGER_IN_RANGE, FALSE, FALSE);
	}
#endif

	
	OSAL_SEND_MESSAGE(UMI_STA_ADD_CFM, TASK_UM_IF_TASK, StaDb->psMsg, pStaAdd->u8VapIndex);
	/*Clear message pointer*/
	StaDb->psMsg = NULL;


}

/**********************************************************************************

BssManager_StaManagerSetFilterDone 


Description:
------------
Called when we are done with set key process - including transition states
Moves to open state

Input: 
-----	
BssManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void BssManager_StaManagerSetFilterDone(BssManagerStaDb_t *StaDb)
{
	K_MSG *psMsg;
	UMI_802_1X_FILTER* pSetFilter;
	
	/*Get pointer to message*/
	psMsg = StaDb->psMsg;
	pSetFilter = (UMI_802_1X_FILTER*)pK_MSG_DATA(psMsg);
	pSetFilter->u16Status = UMI_OK;
	/*Change State to Open*/
	BssManager_StaManagerChangeState(StaDb, BSS_MANAGER_STA_MANAGER_OPEN_STATE);
	/*Send confirmation to Driver*/
	ILOG2_D("BSS MNG, STA MNG, Set Filter Done, SID %d", StaDb->sid);
	OSAL_SEND_MESSAGE(UMI_MC_MAN_SET_802_1X_FILTER_CFM, TASK_UM_IF_TASK, StaDb->psMsg, StaDb->psMsg->header.vapId);
	/*Clear message pointer*/
	StaDb->psMsg = NULL;
}

/**********************************************************************************

BssManager_StaManagerStopTrafficDone 


Description:
------------
Called when we are done with stop traffic process - including transition states
Moves to wait for closing state

Input: 
-----	
BssManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void BssManager_StaManagerStopTrafficDone(BssManagerStaDb_t *StaDb)
{
	K_MSG *psMsg;
	UMI_STOP_TRAFFIC* pStopTraffic;
	
	/*Get pointer to message*/
	psMsg = StaDb->psMsg;
	pStopTraffic = (UMI_STOP_TRAFFIC*)pK_MSG_DATA(psMsg);
	pStopTraffic->u8Status = UMI_OK;
	/*Change State to Wait Closing*/
	BssManager_StaManagerChangeState(StaDb, BSS_MANAGER_STA_MANAGER_WAIT_CLOSING_STATE);
	/*Send confirmation to Driver*/
	ILOG2_D("BSS MNG, STA MNG, Stop Traffic Done, SID %d", StaDb->sid);
	OSAL_SEND_MESSAGE(UMI_MC_MAN_STOP_TRAFFIC_CFM, TASK_UM_IF_TASK, StaDb->psMsg, StaDb->psMsg->header.vapId);
	/*Clear message pointer*/
	StaDb->psMsg = NULL;
}

/**********************************************************************************

BssManager_StaManagerClosingDone 


Description:
------------
Called when we are done with closing process - including transition states
Moves to closed state

Input: 
-----	
BssManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void BssManager_StaManagerClosingDone(BssManagerStaDb_t *StaDb)
{
	K_MSG *psMsg;
	UMI_STA_REMOVE *pClose;

	StaDb_RemoveStationFromDB(StaDb->sid);
	/*Get pointer to message*/
	psMsg = StaDb->psMsg;
	pClose = (UMI_STA_REMOVE*)pK_MSG_DATA(psMsg);
	pClose->u8Status = UMI_OK;
	/*Change State to Closed*/
	BssManager_StaManagerChangeState(StaDb, BSS_MANAGER_STA_MANAGER_CLOSED_STATE);
	/*Send confirmation to Driver*/
	ILOG2_D("BSS MNG, STA MNG, STA Close Done, SID %d", StaDb->sid);

	OSAL_SEND_MESSAGE(UMI_STA_REMOVE_CFM, TASK_UM_IF_TASK, StaDb->psMsg, StaDb->psMsg->header.vapId);
	/*Clear message pointer*/
	StaDb->psMsg = NULL;
}

/**********************************************************************************

BssManager_StaManagerCheckConfirm 


Description:
------------
Called to see if we received all client confirmations and if we did move to next transition state or finish

Input: 
-----	
BssManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void BssManager_StaManagerCheckConfirm(BssManagerStaDb_t *StaDb)
{
	bssManagerStaManager_state state;
	bssManagerRegTypes regType;

	ILOG0_DD("BSS MNG, STA MNG, Check Confirm, SID %d, pending %x", StaDb->sid, StaDb->pendingBitmap);
	/*Are we done with current state*/
	if (StaDb->pendingBitmap == 0)
	{
		state = BssManager_StaManagerStateToPostState[StaDb->state]; /*Set post state*/
		/*Is there a post action state*/
		if (state < BSS_MANAGER_STA_MANAGER_NUM_STATES)
		{
			/*Does next state have a registration type? only transition states can have registered entries*/
			regType = BssManager_StaManagerStateToRegType[state];
			DEBUG_ASSERT(regType < BSS_MANAGER_STA_NUM_REG);
			/*Are there any clients registered for Post Event*/
			if (BssManager_RegDb.numClients[regType] > 0)
			{
				/*Change State to Post State*/
				BssManager_StaManagerChangeState(StaDb, state);
				/*Call registered clients for STA Post Sey Key*/
				BssManager_StaManagerDispatcher(StaDb, regType);
			}
			else
			{
				/*We are done with Message*/
				BssManager_StaManagerDone[StaDb->state](StaDb);
			}
		}
		else
		{
			/*We are done with Message*/
			BssManager_StaManagerDone[StaDb->state](StaDb);
		}
	}
}

/**********************************************************************************

BssManager_StaManagerSetFilterStopTraffic 


Description:
------------
This function is called when stop traffic was received while waiting for set filter
This wil happen if key exchange fails and STA is disconnected

Input: 
-----	
BssManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void BssManager_StaManagerSetFilterStopTraffic(BssManagerStaDb_t *StaDb)
{

	/*we "cheat" a little by changing to Open*/
 	BssManager_StaManagerChangeState(StaDb, BSS_MANAGER_STA_MANAGER_OPEN_STATE);
 	/*And then continue notmally*/
	BssManager_CheckDispatcher(StaDb);
}

/**********************************************************************************

BssManager_StaManagerFatal 


Description:
------------
Fatal handler

Input: 
-----	
BssManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void BssManager_StaManagerFatal(BssManagerStaDb_t *StaDb)
{
	UNUSED_PARAM(StaDb);
	FATAL("BSS Manager STA Manager Fatal");
}

/*---------------------------------------------------------------------------------
/						global Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

BssManager_StaManagerRegConfirm 


Description:
------------
STA Registration confirm message handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BssManager_StaManagerRegConfirm(K_MSG *pMsg)
{
	BssManagerStaManagerCfm_t *pStaCfm = (BssManagerStaManagerCfm_t *)pMsg->abData;
	StaId sid;
	BssManagerStaDb_t *staDb;

	/*Get SID from message*/
	sid = pStaCfm->sid;
	staDb = &BssManager_StaDb[sid];
	DEBUG_ASSERT(staDb->pendingBitmap != 0);
	/*Clear this client from pending clients*/
	staDb->pendingBitmap &= ~(1 << pStaCfm->clientId);
	/*Call Handler*/
	ILOG0_DDD("BSS MNG, STA MNG, Reg Confirm, SID %d, Client %d, Pending Bitmap %x", sid, pStaCfm->clientId, staDb->pendingBitmap);
	BssManager_StaManagerStaConfirm[staDb->state](staDb);
}

/**********************************************************************************

BssManager_AddSta 


Description:
------------
STA ADD message handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BssManager_AddSta(K_MSG *pMsg)
{
	UMI_STA_ADD* pStaAdd = (UMI_STA_ADD*)pMsg->abData;
	StaId sid;
	BssManagerStaDb_t *staDb;	
	uint32 rxhScpadSecondaryVapTableAddr;
	uint32 stationMask;	
	uint32 secondaryVapData;
	uint8 bandId;
	uint8 sidLocation;
	/*Get SID from message*/
	sid = pStaAdd->u16SID;
	DEBUG_ASSERT(sid < HW_NUM_OF_STATIONS);
	staDb = &BssManager_StaDb[sid];
	/*Check that no message pending*/
	DEBUG_ASSERT(staDb->psMsg == NULL);
	/*Store message in Database*/
	staDb->psMsg = pMsg;

	/*TBD - for now call this here - it fills up the STA HW DB*/
	StaDB_AddStaToDB(pStaAdd);

	//map secondary vap to station (each word holds 4 stations, and we must write to entire word because we accessing to scpad)
	sidLocation = CONVERT_BYTES_TO_BIT_INDEX(sid & FOUR_BYTE_ALIGNED_MASK);
	stationMask = MAX_UINT8 << sidLocation;
	secondaryVapData = pStaAdd->u8SecondaryVapIndex << sidLocation;

	//table entry = scpad base address + secondary vap base + sid/NUM_OF_BYTES_IN_WORD converted to bytes
	rxhScpadSecondaryVapTableAddr = B0_MAC_GENRISC_RX_SPRAM_BASE_ADDR + CONVERT_WORDS_TO_BYTES(SCPAD_ADDRESS_RX_HANDLER_SECONDARY_VAP_ID_PER_STATION_TABLE + (sid >> 0x2));
	
	for (bandId = CONFIGURATION_MANAGER_BAND_0 ; bandId < ConfigurationManager_GetNumOfActiveBands(); bandId++)
	{
		RegAccess_WriteMaskedPerBand(rxhScpadSecondaryVapTableAddr, stationMask, secondaryVapData, bandId);
	} 

	/*Call Handler*/
	ILOG2_D("BSS MNG, STA MNG, ADD STA, SID %d", sid);
	BssManager_StaManagerStaOpen[staDb->state](staDb);
}

/**********************************************************************************

BssManager_SetFilter 


Description:
------------
STA Set Filter message handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BssManager_SetFilter(K_MSG *pMsg)
{
	UMI_802_1X_FILTER *pSetFilter = (UMI_802_1X_FILTER *)pMsg->abData;
	StaId sid;
	BssManagerStaDb_t *staDb;

	/*Get SID from message*/
	sid = pSetFilter->u16SID;
	staDb = &BssManager_StaDb[sid];
	/*Check that no message pending*/
	DEBUG_ASSERT(staDb->psMsg == NULL);
	/*Store message in Database*/
	staDb->psMsg = pMsg;
	/*Call Handler*/
	ILOG2_D("BSS MNG, STA MNG, Set Filter, SID %d", sid);
	BssManager_StaManagerSetFilter[staDb->state](staDb);
}

/**********************************************************************************

BssManager_StopTraffic 


Description:
------------
STA Stop Traffic message handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BssManager_StopTraffic(K_MSG *pMsg)
{
	UMI_STOP_TRAFFIC* pStopTraffic = (UMI_STOP_TRAFFIC*)pMsg->abData;
	StaId sid;
	BssManagerStaDb_t *staDb;

	/*Get SID from message*/
	sid = pStopTraffic->u16SID;
	staDb = &BssManager_StaDb[sid];
	/*Check that no message pending*/
	DEBUG_ASSERT(staDb->psMsg == NULL);
	/*Store message in Database*/
	staDb->psMsg = pMsg;
	/*Call Handler*/
	ILOG2_D("BSS MNG, STA MNG, Stop Traffic, SID %d", sid);
	BssManager_StaManagerStopTraffic[staDb->state](staDb);
}

/**********************************************************************************

BssManager_CloseSta 


Description:
------------
STA Disconnect message handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BssManager_CloseSta(K_MSG *pMsg)
{
	UMI_STA_REMOVE* pStaClose = (UMI_STA_REMOVE*)pMsg->abData;
	StaId sid;
	BssManagerStaDb_t *staDb;

	/*Get SID from message*/
	sid = pStaClose->u16SID;
	staDb = &BssManager_StaDb[sid];
	/*Check that no message pending*/
	DEBUG_ASSERT(staDb->psMsg == NULL);
	/*Store message in Database*/
	staDb->psMsg = pMsg;
	/*Call Handler*/
	ILOG2_D("BSS MNG, STA MNG, Close STA, SID %d", sid);


#ifdef ENET_INC_ARCH_WAVE600
	// Debug for JIRA https://jira-chd.intel.com/browse/WLANRTSYS-6218
	// Currently use 2 action points to catch a bug in which some entity writes "0" to "maxMpduLengthLimit" in STA DB of STA 128/129
	if (pStaClose->u16SID == 128)
	{
		ActionPoints_Disable(ACTION_POINT_1);
		ActionPoints_Disable(ACTION_POINT_0);
	}
	if (pStaClose->u16SID == 129)
	{
		ActionPoints_Disable(ACTION_POINT_3);
		ActionPoints_Disable(ACTION_POINT_2);
	}
#endif
	
	BssManager_StaManagerStaClose[staDb->state](staDb);
}

/**********************************************************************************

BssManager_StaManagerInit 


Description:
------------
STA Manager Init


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
 
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif

void BssManager_StaManagerInit()
{
	StaId i;

	/*Init STA database*/
	for (i = 0; i < HW_NUM_OF_STATIONS; i++)
	{
		BssManager_StaDb[i].sid = i;
		BssManager_StaDb[i].state = BSS_MANAGER_STA_MANAGER_CLOSED_STATE;
		BssManager_StaDb[i].pendingBitmap = 0;
		BssManager_StaDb[i].psMsg = NULL;
	}
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif
