/***********************************************************************************
 File:			BaAnalyzer.c
 Module:		BA Analyzer
 Purpose: 		To create a driver for the BA analyer which process the BA response
 Description:	This file is the implementation  of the BA anaylzer module 
				which is responsible of processing the BA response and releasing PDs
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "BaAnalyzer_Api.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"
#include "BaAnalyzerRegs.h"
#include "RegAccess_Api.h"
#include "ShramBaAnalyzerFifos.h"
#include "stringLibApi.h"
#include "TxSequencer.h"
#ifdef ENET_INC_UMAC
#include "TsManager_API.h"
#else /* #ifdef ENET_INC_UMAC */
#include "HwQManager_API.h"
#endif /* #ifdef ENET_INC_UMAC */
#include "linkAdaptation_api.h"
/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_HW_DRIVERS 
#define LOG_LOCAL_FID 0


/* In order to have endless retries use a limit that is never reached (for statistics we allow the retry counter to reach the value of 4) */
#define BA_ANALYZER_DATA_RESPONSE_RETRY_LIMIT	(5)	
#define BA_ANALYZER_PROBE_RESPONSE_RETRY_LIMIT	(2)
#define BA_ANALYZER_TRAINING_RETRY_LIMIT		(1)

#define BA_ANALYZER_ENABLE_ALL_COUNTERS (0x3FFFFFF)



/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

#ifdef ENET_INC_UMAC
static uint8 umacStatusFifoReadPointer;
#else
BaaRateAdaptiveFifo_t 	*readBaaReportPtr = &baaFifoReport[0];
#endif

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/
	

#ifndef ENET_INC_UMAC
static bool BaAnalyzer_IsReportsFifoNotEmpty(void); 
static void baAnalyzer_GenerateRaReport(BaaRateAdaptiveFifo_t *pOldReport, 
													BaaRateAdaptiveFullFifo_t *pNewReport);

#endif 
/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/


#ifdef ENET_INC_UMAC
/**********************************************************************************

isr_BaAnalyzer_UmacStatusFifoNotEmpty 


Description:
------------
Interrupt routine that is invoked when BA analyzer UMAC status FIFO is not empty

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
ISR_VOID isr_BaAnalyzer_UmacStatusFifoNotEmpty(void)
{
	K_MSG* tsManagerMsg = NULL; 
	TsManagerBarStatusMessage_t *tsManagerBarStatusMessage = NULL;
	RegBaAnalyzerPacketsInTxStFifo_u numberOfStatusToReadRegister;
	RegBaAnalyzerUmacPacketReadReg_u statusesReadByUmacRegister;
    uint8 numberOfStatusToRead = 0;
	uint8 i = 0;
		
	numberOfStatusToReadRegister.val = 0;
	statusesReadByUmacRegister.val = 0;

    RegAccess_Read(REG_BA_ANALYZER_PACKETS_IN_TX_ST_FIFO, &numberOfStatusToReadRegister.val);	
    numberOfStatusToRead = numberOfStatusToReadRegister.bitFields.umacPacketsInTxStFifo;

	for(i = 0; i < numberOfStatusToRead; i++)
	{
		tsManagerMsg = OSAL_GET_MESSAGE(sizeof(TsManagerBarStatusMessage_t));
		tsManagerBarStatusMessage = ((TsManagerBarStatusMessage_t *)tsManagerMsg->abData);
		/* There should be only ACK or BA types */
		tsManagerBarStatusMessage->isStatusAck = (BaAnalyzerStatusFifo[umacStatusFifoReadPointer].responseType == BA_ANALYZER_RESPONSE_TYPE_ACK_RECEIVED);
		/* The pd offest is taken as is in order to save claculations in interrupt context - the task that
		   receives the message will perfrom the needed calculations */
		tsManagerBarStatusMessage->pdOffset = BaAnalyzerStatusFifo[umacStatusFifoReadPointer].headPd;
		OSAL_SEND_MESSAGE(TS_MANAGER_BAR_STATUS_RECEIVED, TASK_TS_MANAGER, tsManagerMsg, VAP_ID_DO_NOT_CARE);
		
		umacStatusFifoReadPointer++;
		/* Wrap around */
		umacStatusFifoReadPointer &= BA_ANALYZER_UMAC_STATUS_FIFO_SIZE_MASK;

	}

	/* Update the BA anaylzer about statuses that already were handled */
	/* Currently the interrupt routine reads the counter only once. if more statuses arrived during the processing 
	   of the FIFO , the FIFO wont get empty and when exiting the interrupt routine the interrupt bit will be still
	   set which mean that the interrupt routine will be invoked again. This is done in order to not use 'while' loop
	   and read the counter more than once when the common case should be that one read is enough. if in the future it
	   will be discovered that the interrupt occurs very frequently the code can be modified to 'while' loop */
	statusesReadByUmacRegister.bitFields.umacPacketRdStrb = TRUE;
	statusesReadByUmacRegister.bitFields.umacPacketToSubstract = numberOfStatusToRead;
	RegAccess_Write(REG_BA_ANALYZER_UMAC_PACKET_READ_REG, statusesReadByUmacRegister.val);
}

/**********************************************************************************

BaAnalyzer_InitializeBaaStatusFifo 


Description:
------------
initialize the UMAC status FIFO

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif
void BaAnalyzer_InitializeBaaStatusFifo(void)
{
    /* This parameter indicates where the next valid status FIFO will be */
	umacStatusFifoReadPointer = 0;
}

#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif






#else /* #ifdef ENET_INC_UMAC */

void BaAnalyzer_GetFifoEntry(BaaRateAdaptiveFifo_t* pCurrentBaaTxReport)
{
	memcpy (pCurrentBaaTxReport, readBaaReportPtr, sizeof (BaaRateAdaptiveFifo_t));
}


void BaAnalyzer_DecFifoNumEntriesCount(uint8 value)
{
	
	RegBaAnalyzerUmacPacketReadReg_u regBaAnalyzerUmacPacketReadReg;

	/*Decrement number of reports that poped out from fifo*/
	regBaAnalyzerUmacPacketReadReg.val = 0;
	regBaAnalyzerUmacPacketReadReg.bitFields.umacRtAdaptPacketToSubstract = value;
	regBaAnalyzerUmacPacketReadReg.bitFields.umacRtAdaptPacketRdStrb = 1;
	RegAccess_Write(REG_BA_ANALYZER_UMAC_PACKET_READ_REG,regBaAnalyzerUmacPacketReadReg.val);
}


void BaAnalyzer_IncrementFifoEntryPtr(void)
{
	
	if ((readBaaReportPtr + 1) > (&baaFifoReport[0] + MAX_TX_BAA_FIFO_SIZE - 1))
	{
		readBaaReportPtr = &baaFifoReport[0];
	}
	else
	{
		readBaaReportPtr++;
	}
}

/**********************************************************************************
BaAnalyzer_IsReportsFifoNotEmpty




Description:
------------

Input: 
-----
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/

static bool BaAnalyzer_IsReportsFifoNotEmpty(void)
{
	
	RegBaAnalyzerIntStatus_u regBaAnalyzerIntStatus;
	RegAccess_Read(REG_BA_ANALYZER_INT_STATUS,&regBaAnalyzerIntStatus.val);

	return((bool)regBaAnalyzerIntStatus.bitFields.rateAdaptiveFifoNotEmpty);
}


/**********************************************************************************

baAnalyzer_GenerateRaReport 


Description:
------------
converts common and user matching reports to single report for LA usage


Input:
-----
commonReport
userReport

		
Output:
-------
finalRaReport


Returns:
--------
	void - 
	
**********************************************************************************/

static void baAnalyzer_GenerateRaReport(BaaRateAdaptiveFifo_t *pOldReport, 
													BaaRateAdaptiveFullFifo_t *pNewReport)
{
	uint8 uspIndex = linkAdaptationGetVhtUspIndex(pOldReport->txqStationId,pOldReport->hwGroupDbGroup);
	
	memset(pNewReport, 0, sizeof(BaaRateAdaptiveFullFifo_t));
	pNewReport->commonReport.seqTransactionNumber = pOldReport->transactionNumber;
	if ((pOldReport->txType == TX_TYPE_IMPLICIT_BF) || (pOldReport->txType == TX_TYPE_EXPLICIT_BF))
	{
		pNewReport->userReport.phyTxBf = TRUE;
	}
	pNewReport->commonReport.cca = pOldReport->cca;
	pNewReport->commonReport.scp = pOldReport->scp;
	pNewReport->commonReport.planIndex = pOldReport->hwGroupDbGroup;
	pNewReport->commonReport.bwModifycca = pOldReport->bwModifycca;
	pNewReport->commonReport.bwModifyOther = pOldReport->bwModifyOther;
	pNewReport->userReport.primary = pOldReport->primary;
	pNewReport->userReport.tidBitmap = (1 << pOldReport->txqTid);
	pNewReport->userReport.txQStaId = pOldReport->txqStationId; 
	pNewReport->userReport.txQVapId = pOldReport->txqVapId;
	pNewReport->commonReport.txqGroupId = pOldReport->txqGroupId;
	pNewReport->userReport.timeout = pOldReport->timeout;
	pNewReport->commonReport.pdType = pOldReport->pdType;
	pNewReport->userReport.successMpduCnt = pOldReport->successMpdu;
	pNewReport->userReport.failedMpduCnt = pOldReport->failedMpdu;
	pNewReport->commonReport.firstAfterData = pOldReport->firstAfterData;
	pNewReport->userReport.otherRx = pOldReport->otherRx;
	pNewReport->commonReport.txParamOveride = pOldReport->txParamOveride;
	pNewReport->commonReport.protectionSent = pOldReport->protectionSent;
	pNewReport->userReport.totalSuccessMpduLength = pOldReport->totalSizeSuccessMpdu;
	pNewReport->userReport.totalFailMpduLength = pOldReport->totalSizeAllMpdu - pOldReport->totalSizeSuccessMpdu;
	pNewReport->commonReport.bw = pOldReport->bw;
	pNewReport->commonReport.phyMode = pOldReport->phyMode;
	pNewReport->userReport.tcrMcsNssRate = pOldReport->rate;
	pNewReport->commonReport.rfPower = pOldReport->power;
	pNewReport->userReport.reportNumber = pOldReport->reportNumber;
	pNewReport->commonReport.duration = pOldReport->psduTime;		
	if(pOldReport->reportType == BAA_REPORT_TYPE_SU)
	{
		pNewReport->commonReport.reportType = SEQ_REPORT_TYPE_SEQUENCER_SU;
		pNewReport->userReport.txUserId = INVALID_MU_USP_INDEX;
		pNewReport->userReport.altRateReasonFastProbing = pOldReport->isFastProbUsp0;
	}
	else	// BAA_REPORT_TYPE_MU
	{
		pNewReport->commonReport.reportType = SEQ_REPORT_TYPE_SEQUENCER_VHT_MU;
		pNewReport->userReport.txUserId = uspIndex;
		switch(uspIndex)
		{
			case 0:
				pNewReport->userReport.altRateReasonFastProbing = pOldReport->isFastProbUsp0;
				break;
			case 1:
				pNewReport->userReport.altRateReasonFastProbing = pOldReport->isFastProbUsp1;
				break;
			case 2:
				pNewReport->userReport.altRateReasonFastProbing = pOldReport->isFastProbUsp2;
				break;
			case 3:
				pNewReport->userReport.altRateReasonFastProbing = pOldReport->isFastProbUsp3;
				break;
			default:
				ILOG1_D("baAnalyzer_GenerateRaReport: Received unexpected USP %d", uspIndex);
				break;
		}
	}
}


/**********************************************************************************
BaAnalyzer_processBaaReportQueue



Description:
------------

Input: 
-----
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
void BaAnalyzer_processBaaReportQueue(K_MSG *psMsg)
{
	BaaRateAdaptiveFifo_t CurrentBaaTxReport;
	BaaRateAdaptiveFullFifo_t CurrentBaaTxFullReport;

	do
	{
		BaAnalyzer_GetFifoEntry(&CurrentBaaTxReport);
		/*Increment ptr must ve after the break, otherwize FW missed reports*/
		BaAnalyzer_IncrementFifoEntryPtr();
		/*Decrement one element from sequencer fifo*/
		BaAnalyzer_DecFifoNumEntriesCount(1);

		SLOG0(0, 0, BaaRateAdaptiveFifo_t, &CurrentBaaTxReport);
		baAnalyzer_GenerateRaReport(&CurrentBaaTxReport, &CurrentBaaTxFullReport);
//			ILOG3_DD("BaAnalyzer_processBaaReportQueue- MU report,  GroupIndex : %d, UspIndex: %d",stationOrGroupIndex,uspIndex);
		linkAdaptationProcessBaaReport(&CurrentBaaTxFullReport);
	}while(BaAnalyzer_IsReportsFifoNotEmpty());
}

/**********************************************************************************
BaAnalyzer_processBaaReportQueue



Description:
------------

Input: 
-----
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/

void BaAnalyzer_processBaaReportQueuePostSeq(StaId staIndex,uint16 numberOfElementInBaaFifo,bool isReportValid )
{
}


/**********************************************************************************

BaAnalyzer_Initialize 


Description:
------------
initialize internal structures and registers of the Ba analyzer module 
Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
#if defined (ENET_INC_LMAC)
#pragma ghs section text=".initialization" 
#endif
void BaAnalyzer_Initialize(void)
{
	RegBaAnalyzerDplIdx_u dplIndexesRegister;
	RegBaAnalyzerLibDplIdxInput_u txLiberatorDplIndexInputRegister;
	RegBaAnalyzerRxLibDplIdxInput_u rxLiberatorDplIndexInputRegister;
	RegBaAnalyzerRxLibDplIdxSource03_u rxLibDplIndexSrc03;
	RegBaAnalyzerRxLibDplIdxSource47_u rxLibDplIndexSrc47;
	RegBaAnalyzerLibDplIdxSource03_u	txLibDplIdxSource03;
	RegBaAnalyzerLibDplIdxSource47_u	txLibDplIdxSource47;
	RegBaAnalyzerUmacStatusStartAddr_u umacStatusFifoRegister; 
	RegBaAnalyzerIntErrorEn_u regBaAnalyzerIntErrorEn;		
	RegBaAnalyzerBaAnalyzerCountersEn_u baAnalyzerCountersEnReg;
	RegBaAnalyzerRaFifoStartAddr_u	baAnalyzerRafifoReg;
	RegBaAnalyzerIntEn_u RegBaAnalyzerIntEnReg;
	RegBaAnalyzerRetryCntPdType815_u baAnalyzerRetryCntPdType815Reg;
	RegBaAnalyzerShortRetryCntPdType815_u baAnalyzerShortRetryCntPdType815Reg;
	RegBaAnalyzerRetryCntPdType07_u baAnalyzerRetryCntPdType07Reg;
	RegBaAnalyzerShortRetryCntPdType07_u baAnalyzerShortRetryCntPdType07Reg;


	
	umacStatusFifoRegister.val = 0;	
	dplIndexesRegister.val = 0;
	txLiberatorDplIndexInputRegister.val = 0;
	rxLiberatorDplIndexInputRegister.val = 0;
	rxLibDplIndexSrc03.val = 0;
	rxLibDplIndexSrc47.val = 0;
	txLibDplIdxSource03.val = 0; 
	txLibDplIdxSource47.val = 0;
	baAnalyzerCountersEnReg.val = 0;
	baAnalyzerRafifoReg.val = 0; 
	baAnalyzerRetryCntPdType815Reg.val = 0;
	baAnalyzerShortRetryCntPdType815Reg.val = 0;
	baAnalyzerRetryCntPdType07Reg.val = 0;
	baAnalyzerShortRetryCntPdType07Reg.val = 0;
	
	dplIndexesRegister.bitFields.dplIdxSuccess = HW_Q_MANAGER_DONE_LIST_LIBERATOR;
	dplIndexesRegister.bitFields.dplIdxDiscard = HW_Q_MANAGER_DONE_LIST_DISCARDED_PACKETS;
	RegAccess_Write(REG_BA_ANALYZER_DPL_IDX, dplIndexesRegister.val);

	txLiberatorDplIndexInputRegister.bitFields.liberatorDplIdxIn = HW_Q_MANAGER_DONE_LIST_LIBERATOR;
	RegAccess_Write(REG_BA_ANALYZER_LIB_DPL_IDX_INPUT, txLiberatorDplIndexInputRegister.val);

	rxLiberatorDplIndexInputRegister.bitFields.rxLiberatorDplIdxIn = HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST;
	rxLiberatorDplIndexInputRegister.bitFields.rxLiberatorDplIdxToAdd = HW_Q_MANAGER_RX_FREE_HOST_RDS_LIST;
	RegAccess_Write(REG_BA_ANALYZER_RX_LIB_DPL_IDX_INPUT, rxLiberatorDplIndexInputRegister.val);

	rxLibDplIndexSrc03.bitFields.rxLiberatorDplIdxSource0 = HW_Q_MANAGER_RX_FREE_HOST_RDS_LIST;
	rxLibDplIndexSrc03.bitFields.rxLiberatorDplIdxSource1 = HW_Q_MANAGER_RX_FREE_FW_RDS_LIST;
    // Liberator should release RDs of source 3 (RX_DESCRIPTOR_SOURCE_FORWARD) to temp list
	rxLibDplIndexSrc03.bitFields.rxLiberatorDplIdxSource3 = HW_Q_MANAGER_RX_READY_LIST_RELEASE_FORWARD_RDS; 
	RegAccess_Write(REG_BA_ANALYZER_RX_LIB_DPL_IDX_SOURCE0_3, rxLibDplIndexSrc03.val);

	rxLibDplIndexSrc47.bitFields.rxLiberatorDplIdxSource4 = HW_Q_MANAGER_RX_FREE_MANAGEMENT_RDS_LIST;
	RegAccess_Write(REG_BA_ANALYZER_RX_LIB_DPL_IDX_SOURCE4_7, rxLibDplIndexSrc47.val);

	// configure the TX done PD source to be handled by the hostIf Acc 
#ifdef TX_DATA_CB_UM_PATH
	txLibDplIdxSource03.bitFields.liberatorDplIdxSource0 = HW_Q_MANAGER_DONE_LIST_HOST_DATA; 
#else
	txLibDplIdxSource03.bitFields.liberatorDplIdxSource0 = HW_Q_MANAGER_LOW_PR_READY_LIST_TEMP_HOST_Q; //route source0 packets to the hostif accelerator 
#endif 
	txLibDplIdxSource03.bitFields.liberatorDplIdxSource1 = 0x1; // There is no source 1. Keep initial value
	txLibDplIdxSource03.bitFields.liberatorDplIdxSource2 = HW_Q_MANAGER_DONE_LIST_MANAGEMENT_FROM_FW; 	
	txLibDplIdxSource03.bitFields.liberatorDplIdxSource3 = HW_Q_MANAGER_DONE_LIST_MANAGEMENT_FROM_HOST; 
	RegAccess_Write(REG_BA_ANALYZER_LIB_DPL_IDX_SOURCE0_3, txLibDplIdxSource03.val);

	txLibDplIdxSource47.bitFields.liberatorDplIdxSource4 = HW_Q_MANAGER_DONE_LIST_RELIABLE_MULTICAST; 	
	txLibDplIdxSource47.bitFields.liberatorDplIdxSource5 = HW_Q_MANAGER_DONE_LIST_PS_SETTINGS_HW; 	
	txLibDplIdxSource47.bitFields.liberatorDplIdxSource6 = HW_Q_MANAGER_DONE_LIST_ERROR;
	txLibDplIdxSource47.bitFields.liberatorDplIdxSource7 = HW_Q_MANAGER_DONE_LIST_ERROR;
	RegAccess_Write(REG_BA_ANALYZER_LIB_DPL_IDX_SOURCE4_7, txLibDplIdxSource47.val);

	umacStatusFifoRegister.bitFields.umacStatusStartAddr = (uint32)BaAnalyzerStatusFifo;
	/* The size in BA analyzer is + 1 to the value written by FW */
	umacStatusFifoRegister.bitFields.umacStatusPacketSize = BA_ANALYZER_MAC_STATUS_FIFO_SIZE - 1;	  
	RegAccess_Write(REG_BA_ANALYZER_UMAC_STATUS_START_ADDR, umacStatusFifoRegister.val);

	RegAccess_Read(REG_BA_ANALYZER_RETRY_CNT_PD_TYPE8_15,&baAnalyzerRetryCntPdType815Reg.val);
	/* Configure Probe Response (PD Type = 13) retry limit to 2 */
	baAnalyzerRetryCntPdType815Reg.bitFields.retryCntPdType13 = BA_ANALYZER_PROBE_RESPONSE_RETRY_LIMIT;
	/* Configure Training (PD Type = 8,) retry limit to 1 */
	baAnalyzerRetryCntPdType815Reg.bitFields.retryCntPdType8 = BA_ANALYZER_TRAINING_RETRY_LIMIT;
	RegAccess_Write(REG_BA_ANALYZER_RETRY_CNT_PD_TYPE8_15, baAnalyzerRetryCntPdType815Reg.val);

	RegAccess_Read(REG_BA_ANALYZER_SHORT_RETRY_CNT_PD_TYPE8_15,&baAnalyzerShortRetryCntPdType815Reg.val);
	/* Configure short retry Probe Response (PD Type = 13) retry limit to 2 */
	baAnalyzerShortRetryCntPdType815Reg.bitFields.shortRetryCntPdType13 = BA_ANALYZER_PROBE_RESPONSE_RETRY_LIMIT;
	/* Configure short Retry for Training (PD Type = 8) retry limit to 1 */
	baAnalyzerShortRetryCntPdType815Reg.bitFields.shortRetryCntPdType8 = BA_ANALYZER_TRAINING_RETRY_LIMIT;
	RegAccess_Write(REG_BA_ANALYZER_SHORT_RETRY_CNT_PD_TYPE8_15, baAnalyzerShortRetryCntPdType815Reg.val);

	RegAccess_Read(REG_BA_ANALYZER_RETRY_CNT_PD_TYPE0_7, &baAnalyzerRetryCntPdType07Reg.val);
	/* Configure retry Data (PD Type = 0,) retry limit to 5 */
	baAnalyzerRetryCntPdType07Reg.bitFields.retryCntPdType0 = BA_ANALYZER_DATA_RESPONSE_RETRY_LIMIT;
	/* Configure retry Data (PD Type = 1,) retry limit to 5 */
	baAnalyzerRetryCntPdType07Reg.bitFields.retryCntPdType1 = BA_ANALYZER_DATA_RESPONSE_RETRY_LIMIT;
	RegAccess_Write(REG_BA_ANALYZER_RETRY_CNT_PD_TYPE0_7, baAnalyzerRetryCntPdType07Reg.val);

	RegAccess_Read(REG_BA_ANALYZER_SHORT_RETRY_CNT_PD_TYPE0_7, &baAnalyzerShortRetryCntPdType07Reg.val);
	/* Configure short retry Data (PD Type = 0,) retry limit to 5 */
	baAnalyzerShortRetryCntPdType07Reg.bitFields.shortRetryCntPdType0 = BA_ANALYZER_DATA_RESPONSE_RETRY_LIMIT;
	/* Configure short retry Data (PD Type = 1,) retry limit to 5 */
	baAnalyzerShortRetryCntPdType07Reg.bitFields.shortRetryCntPdType1 = BA_ANALYZER_DATA_RESPONSE_RETRY_LIMIT;
	RegAccess_Write(REG_BA_ANALYZER_SHORT_RETRY_CNT_PD_TYPE0_7, baAnalyzerShortRetryCntPdType07Reg.val);



	/*Enable BA rate adaptive fifo full interrupt*/
	RegAccess_Read(REG_BA_ANALYZER_INT_ERROR_EN,&regBaAnalyzerIntErrorEn.val);
	regBaAnalyzerIntErrorEn.bitFields.rtAdaptPushWhileFullEn = 1;
	regBaAnalyzerIntErrorEn.bitFields.umacPushTxStWhileFullEn = 1;
	RegAccess_Write(REG_BA_ANALYZER_INT_ERROR_EN, regBaAnalyzerIntErrorEn.val);

	/*Enable interrupts*/
	RegAccess_Read(REG_BA_ANALYZER_INT_EN,&RegBaAnalyzerIntEnReg.val);
	RegBaAnalyzerIntEnReg.bitFields.rateAdaptiveFifoNotEmptyEn = 0x1;
	RegBaAnalyzerIntEnReg.bitFields.umacTxStNotEmptyEn = 0x1;
	RegAccess_Write(REG_BA_ANALYZER_INT_EN, RegBaAnalyzerIntEnReg.val);

	/* Enable All Counters in the BAA module*/
	baAnalyzerCountersEnReg.bitFields.baAnalyzerCountersEn = BA_ANALYZER_ENABLE_ALL_COUNTERS;
	RegAccess_Write(REG_BA_ANALYZER_BA_ANALYZER_COUNTERS_EN, baAnalyzerCountersEnReg.val);

	baAnalyzerRafifoReg.bitFields.raFifoStartAddr = CONVERT_PHYSICAL_TO_SHRAM_OFFSET(&baaFifoReport[0]); 
	baAnalyzerRafifoReg.bitFields.raFifoPacketSize =  MAX_TX_BAA_FIFO_SIZE-1; //unit size - LA report- 10 words. 0 means 1 report - 10 words. 
	RegAccess_Write(REG_BA_ANALYZER_RA_FIFO_START_ADDR,baAnalyzerRafifoReg.val);

}
#if defined (ENET_INC_LMAC)
#pragma ghs section text=default
#endif

#endif /* #ifdef ENET_INC_UMAC */

