/***********************************************************************************
 File:			BaAnalyzer.c
 Module:		BA Analyzer
 Purpose: 		To create a driver for the BA analyer which process the BA response
 Description:	This file is the implementation  of the BA anaylzer module 
				which is responsible of processing the BA response and releasing PDs
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "BaAnalyzer_Api.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"
#include "BaAnalyzerRegs.h"
#include "RegAccess_Api.h"
#include "ShramBaAnalyzerFifos.h"
#include "ShramPacketDescriptors.h"
#include "stringLibApi.h"
#include "TsManager_API.h"
#include "HwQManager_API.h"
#include "linkAdaptation_api.h"
#include "MacGeneralRegs.h"
#include "shram_man_msgs.h"
#include "TxPd_Descriptors.h"
#include "HwEventsAndErrors_Api.h"
#include "Utils_Api.h"
#ifdef BAA_RATE_ADAPTIVE_FIFO_COMMON_TRANSACTION_NUMBER_UPDATE_ISSUE_WORKAROUND
#include "Pac_Api.h"
#endif //BAA_RATE_ADAPTIVE_FIFO_COMMON_TRANSACTION_NUMBER_UPDATE_ISSUE_WORKAROUND

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_HW_DRIVERS 
#define LOG_LOCAL_FID 1

#ifdef ENET_INC_ARCH_WAVE600B
#define BA_ANALYZER_ENABLE_ALL_COUNTERS (0x7FFFFFF)
#else
#define BA_ANALYZER_ENABLE_ALL_COUNTERS (0x1FFFFFF)
#endif

#define BAA_COMMON_FIFOS_SIZE_MASK 				(BAA_COMMON_FIFO_SIZE-1)
#define BAA_USER_FIFOS_SIZE_MASK 				(BAA_USER_FIFO_SIZE-1)
#define BAA_MAX_TRANS_NUM						0xFF
#define BAA_SIZE_OF_ERROR_COUNTERS_ARRAY		8
#define BAA_PSDU_DURATION_SHIFT_FACTOR			5

#ifdef BAA_RATE_ADAPTIVE_FIFO_COMMON_TRANSACTION_NUMBER_UPDATE_ISSUE_WORKAROUND
#define BAA_MAX_DELTA_BETWEEN_USER_REPORTS_USING_SAME_COMMON_REPORT (300000) //300ms
#endif //BAA_RATE_ADAPTIVE_FIFO_COMMON_TRANSACTION_NUMBER_UPDATE_ISSUE_WORKAROUND
#ifdef ENET_INC_ARCH_WAVE600B
#define BAA_MBA_MAX_NUM_TID_EXPECTED  (0x2)
#endif



/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
// Do not change enum values, they are defined as HW values used by Ba Analyzer module

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/


static uint8 baaStatusFifoReadIndex = 0;
static uint8 baaCommonFifoReadIndex = 0;
static uint8 baaUserFifoReadIndex = 0;
static uint32 baaUserReportDropCounter = 0;
RegBaAnalyzerIntErrorStatus_u intErrorStatus;



/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/
static BaaTxStatusFifo_t *baaStatusFifoPtr;
static BaaRateAdaptiveCommonFifo_t *baaRaCommonFifoPtr;
static BaaRateAdaptiveUserFifo_t *baaRaUserFifoPtr;

static BaaRateAdaptiveCommonFifo_t baAnalyzer_lastCommonReport;
#ifdef BAA_RATE_ADAPTIVE_FIFO_COMMON_TRANSACTION_NUMBER_UPDATE_ISSUE_WORKAROUND
static uint32 baAnalyzer_lastCommonReportTimestamp;
#endif //BAA_RATE_ADAPTIVE_FIFO_COMMON_TRANSACTION_NUMBER_UPDATE_ISSUE_WORKAROUND


/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/

static void baAnalyzer_GetUserFifoEntry(BaaRateAdaptiveUserFifo_t* userFifoEntry);
static void baAnalyzer_GetCommonFifoEntry(BaaRateAdaptiveCommonFifo_t* commonFifoEntry);

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

baAnalyzer_StatusFifoNotEmptyEventHandler 


Description:
------------
Event handler that is invoked from PAC manager when BA analyzer status FIFO is not empty

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void baAnalyzer_StatusFifoNotEmptyEventHandler(void)
{
	K_MSG* tsManagerMsg = NULL; 
	TsManagerBarStatusMessage_t *tsManagerBarStatusMessage = NULL;
	RegBaAnalyzerStsReportFifoNumEntriesCount_u numOfEntriesCountReg;
	RegBaAnalyzerStsReportFifoRdEntriesNum_u statusesReadFromFifoStatus;
	RegBaAnalyzerIntEn_u eventsEnableReg;
    uint8 numberOfStatusToRead = 0;
	uint8 i = 0;
		
	numOfEntriesCountReg.val = 0;
	statusesReadFromFifoStatus.val = 0;
	eventsEnableReg.val = 0;

    RegAccess_Read(REG_BA_ANALYZER_STS_REPORT_FIFO_NUM_ENTRIES_COUNT, &numOfEntriesCountReg.val);	
    numberOfStatusToRead = numOfEntriesCountReg.bitFields.srStsReportFifoNumEntriesCount;

	for(i = 0; i < numberOfStatusToRead; i++)
	{
		ASSERT((baaStatusFifoPtr[baaStatusFifoReadIndex].pdType == PD_TYPE_BAR) ||
			(baaStatusFifoPtr[baaStatusFifoReadIndex].pdType == PD_TYPE_BEACON));
		// handle PdType = BAR
		if(baaStatusFifoPtr[baaStatusFifoReadIndex].pdType == PD_TYPE_BAR)
		{
			tsManagerMsg = OSAL_GET_MESSAGE(sizeof(TsManagerBarStatusMessage_t));
			tsManagerBarStatusMessage = ((TsManagerBarStatusMessage_t *)tsManagerMsg->abData);
			// There should be only ACK or BA types 
			tsManagerBarStatusMessage->isStatusAck = (baaStatusFifoPtr[baaStatusFifoReadIndex].respType == BA_ANALYZER_RESPONSE_TYPE_ACK_RECEIVED);
			// The pd offset is taken as is in order to save calculations in interrupt context - the task that
			//   receives the message will perfrom the needed calculations 
			tsManagerBarStatusMessage->pdOffset = baaStatusFifoPtr[baaStatusFifoReadIndex].headPd;
			OSAL_SEND_MESSAGE(TS_MANAGER_BAR_STATUS_RECEIVED, TASK_TS_MANAGER, tsManagerMsg, VAP_ID_DO_NOT_CARE);
		}
		//handle PdType = BEACON
		else if(baaStatusFifoPtr[baaStatusFifoReadIndex].pdType == PD_TYPE_BEACON)
		{
			K_MSG *kMsg_p = OSAL_GET_MESSAGE(sizeof(BeaconCsaPdMsg_t));
			BeaconCsaPdMsg_t *beaconCsaPdMsg_p = (BeaconCsaPdMsg_t *)pK_MSG_DATA(kMsg_p);	
			TxPd_t *packetDescriptor = (TxPd_t *)CONVERT_OFFSET_TO_PD(baaStatusFifoPtr[baaStatusFifoReadIndex].headPd);

			beaconCsaPdMsg_p->pd = (void *)packetDescriptor;
			OSAL_SEND_MESSAGE(PAC_MANAGER_BEACON_CSA_PD, TASK_PAC_MANAGER, kMsg_p, packetDescriptor->txQVapId);
		}
		
		baaStatusFifoReadIndex++;
		// Wrap around 
		baaStatusFifoReadIndex &= BA_ANALYZER_UMAC_STATUS_FIFO_SIZE_MASK;

	}

	// Update the BA anaylzer about statuses that already were handled 
	// Currently the interrupt routine reads the counter only once. if more statuses arrived during the processing 
	//   of the FIFO , the FIFO wont get empty and when exiting the interrupt routine the interrupt bit will be still
	//   set which mean that the interrupt routine will be invoked again. This is done in order to not use 'while' loop
	//   and read the counter more than once when the common case should be that one read is enough. if in the future it
	//   will be discovered that the interrupt occurs very frequently the code can be modified to 'while' loop 
	statusesReadFromFifoStatus.bitFields.crStsReportFifoRdEntriesNum = numberOfStatusToRead;
	RegAccess_Write(REG_BA_ANALYZER_STS_REPORT_FIFO_RD_ENTRIES_NUM, statusesReadFromFifoStatus.val);	

	//enable event - which was turned off during handling of statuses in fifo
#if defined (ENET_INC_LMAC0)	
	HwEvents_UnMaskEvent_A_Band0(HW_EVENT_BA_ANALYZER_STATUS_FIFO);
#elif defined (ENET_INC_LMAC1)
	HwEvents_UnMaskEvent_A_Band1(HW_EVENT_BA_ANALYZER_STATUS_FIFO);
#endif
}


/**********************************************************************************

isr_BaAnalyzerInterruptErrorHandler 


Description:
------------
Ba Analyzer Interrupt error handler
Possible interrupt errors:
UnexpectedStaidInt- bit 0
UnexpectedTidInt - bit 1
UnexpectedAckType0Tid15Int - bit 2
UnexpectedFragNumInt- bit 3
UnexpectedBaLengthInt - bit4
DropTxStWhileFullInt - bit 5
DropCommonRtAdaptWhileFullInt - bit6 
DropUserRtAdaptWhileFullInt - bit 7

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

ISR_VOID isr_BaAnalyzerInterruptErrorHandler(void)
{
	RegBaAnalyzerClearIntError_u clearIntError;
    RegBaAnalyzerRaUserReportFifoClearStrb_u RaUserReportFifoClearStrb;
	
	intErrorStatus.val = 0;
	clearIntError.val = 0;
    RaUserReportFifoClearStrb.val = 0;
	
	RegAccess_Read(REG_BA_ANALYZER_INT_ERROR_STATUS, &intErrorStatus.val);
#ifdef ENET_INC_ARCH_WAVE600B 
    if(intErrorStatus.bitFields.srDropFirstUserRtAdaptWhileFullInt)
    {
        baaUserReportDropCounter++;
        RaUserReportFifoClearStrb.bitFields.crRaFirstUserReportFifoClearFullDropCtrStrb = 1;
        RegAccess_Write(REG_BA_ANALYZER_RA_USER_REPORT_FIFO_CLEAR_STRB, RaUserReportFifoClearStrb.val);
		intErrorStatus.bitFields.srDropFirstUserRtAdaptWhileFullInt = 0;
    }
    if(intErrorStatus.bitFields.srDropNonFirstUserRtAdaptWhileFullInt)
    {
        baaUserReportDropCounter++;
        RaUserReportFifoClearStrb.bitFields.crRaNonFirstUserReportFifoClearFullDropCtrStrb = 1;
        RegAccess_Write(REG_BA_ANALYZER_RA_USER_REPORT_FIFO_CLEAR_STRB, RaUserReportFifoClearStrb.val);
	    intErrorStatus.bitFields.srDropNonFirstUserRtAdaptWhileFullInt = 0;
    }
#else    
	if(intErrorStatus.bitFields.srDropUserRtAdaptWhileFullInt) 
	{
		baaUserReportDropCounter++;
		clearIntError.bitFields.crClrDropUserRtAdaptWhileFullInt = 1;
		RegAccess_Write(REG_BA_ANALYZER_CLEAR_INT_ERROR, clearIntError.val);
		intErrorStatus.bitFields.srDropUserRtAdaptWhileFullInt = 0;
	}
#endif
	if(intErrorStatus.val)
	{
		FATAL("Ba Analyzer interrupt error!");
	}
	 
}

/**********************************************************************************

BaAnalyzer_InitializeBaaStatusFifo 


Description:
------------
initialize the Baa status FIFO

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BaAnalyzer_InitializeBaaStatusFifo(void)
{
	memset(&baAnalyzer_lastCommonReport, 0, sizeof(baAnalyzer_lastCommonReport)); 
	baAnalyzer_lastCommonReport.baaTransactionNumber = BAA_MAX_TRANS_NUM;

#ifdef BAA_RATE_ADAPTIVE_FIFO_COMMON_TRANSACTION_NUMBER_UPDATE_ISSUE_WORKAROUND

#if defined (ENET_INC_HW_FPGA) && defined (ENET_INC_LMAC1)
		if(ConfigurationManager_GetBandConfigurationMode() != CONFIGURATION_MODE_SINGLE_BAND)
#endif
		{
			baAnalyzer_lastCommonReportTimestamp = GET_TSF_TIMER_LOW();
		}
#endif //BAA_RATE_ADAPTIVE_FIFO_COMMON_TRANSACTION_NUMBER_UPDATE_ISSUE_WORKAROUND
	
#if defined(ENET_INC_LMAC0) 
	baaStatusFifoPtr = BaAnalyzerStatusFifo;
	baaRaCommonFifoPtr = BaAnalyzerRaCommonReportFifo;
	baaRaUserFifoPtr = BaAnalyzerRaUserReportFifo;
#endif

#if defined(ENET_INC_LMAC1) 
	baaStatusFifoPtr = BaAnalyzerStatusFifo_B1;
	baaRaCommonFifoPtr = BaAnalyzerRaCommonReportFifo_B1;
	baaRaUserFifoPtr = BaAnalyzerRaUserReportFifo_B1;
#endif

	memset(baaStatusFifoPtr, 	0, (sizeof(BaaTxStatusFifo_t) 			* BA_ANALYZER_MAC_STATUS_FIFO_SIZE));
	memset(baaRaCommonFifoPtr, 	0, (sizeof(BaaRateAdaptiveCommonFifo_t) * BAA_COMMON_FIFO_SIZE));
	memset(baaRaUserFifoPtr, 	0, (sizeof(BaaRateAdaptiveUserFifo_t) 	* BAA_USER_FIFO_SIZE));
}


/**********************************************************************************

baAnalyzer_GetUserFifoEntry 


Description:
------------
reads user report from FIFO

Input:
-----
pUserFifoEntry	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void baAnalyzer_GetUserFifoEntry(BaaRateAdaptiveUserFifo_t* pUserFifoEntry)
{
	memcpy32(pUserFifoEntry, &baaRaUserFifoPtr[baaUserFifoReadIndex], (sizeof(BaaRateAdaptiveUserFifo_t) >> 2));
	baaUserFifoReadIndex++;
	baaUserFifoReadIndex &= BAA_USER_FIFOS_SIZE_MASK;
}

/**********************************************************************************

baAnalyzer_GetCommonFifoEntry 


Description:
------------
reads common report from FIFO and decrements number fo entries in FIFO by one


Input:
-----
pCommonFifoEntry	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void baAnalyzer_GetCommonFifoEntry(BaaRateAdaptiveCommonFifo_t* pCommonFifoEntry)
{
	RegBaAnalyzerRaCommonReportFifoRdEntriesNum_u numOfReadEntries;
	
	numOfReadEntries.val = 0;
	numOfReadEntries.bitFields.crRaCommonReportFifoRdEntriesNum = 1;
	memcpy32(pCommonFifoEntry, &baaRaCommonFifoPtr[baaCommonFifoReadIndex], (sizeof(BaaRateAdaptiveCommonFifo_t) >> 2));

#ifdef BAA_RATE_ADAPTIVE_FIFO_COMMON_TRANSACTION_NUMBER_UPDATE_ISSUE_WORKAROUND
	baAnalyzer_lastCommonReportTimestamp = GET_TSF_TIMER_LOW();
#endif //BAA_RATE_ADAPTIVE_FIFO_COMMON_TRANSACTION_NUMBER_UPDATE_ISSUE_WORKAROUND

	// increment index and handle wrap-around if needed
	baaCommonFifoReadIndex++;
	baaCommonFifoReadIndex &= BAA_COMMON_FIFOS_SIZE_MASK;
	RegAccess_Write(REG_BA_ANALYZER_RA_COMMON_REPORT_FIFO_RD_ENTRIES_NUM, numOfReadEntries.val);
}

/**********************************************************************************
BaAnalyzer_processBaaReportQueue



Description:
------------

Input: 
-----
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
void BaAnalyzer_processBaaReportQueue(K_MSG *psMsg)
{
 
	BaaRateAdaptiveFullFifo_t  baaFullReport;
	
	RegBaAnalyzerRaCommonReportFifoNumEntriesCount_u numOfCommonEntriesCountReg;
	RegBaAnalyzerRaUserReportFifoNumEntriesCount_u numOfUserEntriesCountReg;
	RegBaAnalyzerRaUserReportFifoRdEntriesNum_u numOfReadEntriesReg;
	RegBaAnalyzerIntStatus_u interruptStatus;
#ifdef BAA_RATE_ADAPTIVE_FIFO_COMMON_TRANSACTION_NUMBER_UPDATE_ISSUE_WORKAROUND
	uint32 currentTimestamp;
#endif //BAA_RATE_ADAPTIVE_FIFO_COMMON_TRANSACTION_NUMBER_UPDATE_ISSUE_WORKAROUND

    uint8 numberOfCommonReports = 0;
	uint8 numberOfUserReports = 0;
	uint8 i = 0;
	UNUSED_PARAM(psMsg);		
	numOfCommonEntriesCountReg.val = 0;
	numOfUserEntriesCountReg.val = 0;
	interruptStatus.val = 0;

	RegAccess_Read(REG_BA_ANALYZER_INT_STATUS, &interruptStatus.val);

	// handle "user fifo not empty" interrupts the rest must be ignored.
	// common fifo interrupts are not enabled even if will occur - common fifo will be handled upon user fifo not empty interrupt
	DEBUG_ASSERT(interruptStatus.bitFields.srUserRateAdaptiveFifoNotEmpty == 1);
	
	
	RegAccess_Read(REG_BA_ANALYZER_RA_USER_REPORT_FIFO_NUM_ENTRIES_COUNT, &numOfUserEntriesCountReg.val);	
    numberOfUserReports = numOfUserEntriesCountReg.bitFields.srRaUserReportFifoNumEntriesCount;

	DEBUG_ASSERT(numberOfUserReports != 0);

	for(i = 0; i < numberOfUserReports; i++)
	{
	
		baAnalyzer_GetUserFifoEntry(&baaFullReport.userReport);
		//if baa seq num in stored last common report not equal to baa seq num of newly received user report 
		// the common report found in our db is not matching to the received user report and
		// we need to read it from common FIFO
		// when handling SU report there is always new common report waiting in queue that must be read by FW
		// if the received user report has the same baa trans number as previously stored commong repor
		if(baAnalyzer_lastCommonReport.baaTransactionNumber != baaFullReport.userReport.baaTransactionNumber)
		{
			RegAccess_Read(REG_BA_ANALYZER_RA_COMMON_REPORT_FIFO_NUM_ENTRIES_COUNT, &numOfCommonEntriesCountReg.val);	
    		numberOfCommonReports = numOfCommonEntriesCountReg.bitFields.srRaCommonReportFifoNumEntriesCount;
			DEBUG_ASSERT(numberOfCommonReports != 0);
				
			baAnalyzer_GetCommonFifoEntry(&baAnalyzer_lastCommonReport);
			// at this point matching common report must be in FIFO, if not ASSERT
			DEBUG_ASSERT(baAnalyzer_lastCommonReport.baaTransactionNumber == baaFullReport.userReport.baaTransactionNumber);
		}
#ifdef BAA_RATE_ADAPTIVE_FIFO_COMMON_TRANSACTION_NUMBER_UPDATE_ISSUE_WORKAROUND
		else
		{
			/* baaTransactionNumber is an 8 bit field and its value is incremented with every module activation, not only upon writes to the fifo,
				so we must verify that the time that passed from last time the current common entry was used to present time is not too long */

			currentTimestamp = GET_TSF_TIMER_LOW();
			
			if ((currentTimestamp - baAnalyzer_lastCommonReportTimestamp) > BAA_MAX_DELTA_BETWEEN_USER_REPORTS_USING_SAME_COMMON_REPORT)
			{
				RegAccess_Read(REG_BA_ANALYZER_RA_COMMON_REPORT_FIFO_NUM_ENTRIES_COUNT, &numOfCommonEntriesCountReg.val);	
				numberOfCommonReports = numOfCommonEntriesCountReg.bitFields.srRaCommonReportFifoNumEntriesCount;
				DEBUG_ASSERT(numberOfCommonReports != 0);
					
				baAnalyzer_GetCommonFifoEntry(&baAnalyzer_lastCommonReport);
				// at this point matching common report must be in FIFO, if not ASSERT
				DEBUG_ASSERT(baAnalyzer_lastCommonReport.baaTransactionNumber == baaFullReport.userReport.baaTransactionNumber);
			}
		}
#endif //BAA_RATE_ADAPTIVE_FIFO_COMMON_TRANSACTION_NUMBER_UPDATE_ISSUE_WORKAROUND
		
		// no need to copy user part - was already copied in the beginning of the loop
		memcpy32(&baaFullReport.commonReport, &baAnalyzer_lastCommonReport, (sizeof(BaaRateAdaptiveCommonFifo_t)>>2));
        linkAdaptationProcessBaaReport(&baaFullReport);		
	}
		
	// write the number of user reports extracted from FIFO - if this number is equal to the number of reports currently 
	// found in fifo this will clear interrupt line, so no need to clear the interrupt bit in REG_BA_ANALYZER_CLEAR_INT register
	numOfReadEntriesReg.val = 0;
	numOfReadEntriesReg.bitFields.crRaUserReportFifoRdEntriesNum = numberOfUserReports;
	RegAccess_Write(REG_BA_ANALYZER_RA_USER_REPORT_FIFO_RD_ENTRIES_NUM, numOfReadEntriesReg.val);
	
}

#ifdef ENET_INC_ARCH_WAVE600D2
/**********************************************************************************
BaAnalyzer_SetAidForStaMode



Description:
----------
AID configuration in BAA for M-BA Rx purpose.

**********************************************************************************/
void BaAnalyzer_SetAidForStaMode(uint16 aid)
{
	RegBaAnalyzerStaModeCfg_u baAnalyzerAidStaModeCfg;

	RegAccess_Read(REG_BA_ANALYZER_STA_MODE_CFG, &baAnalyzerAidStaModeCfg.val);
	baAnalyzerAidStaModeCfg.bitFields.crStaModeAid = aid;
	RegAccess_Write(REG_BA_ANALYZER_STA_MODE_CFG, baAnalyzerAidStaModeCfg.val);
}
#endif


/**********************************************************************************

BaAnalyzer_Initialize 


Description:
------------
initialize internal structures and registers of the Ba analyzer module 
Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BaAnalyzer_Initialize(void)
{
	RegBaAnalyzerDplIdx_u dplIndexesRegister;
	RegBaAnalyzerStsReportFifoBaseAddr_u statusFifoRegister; 
	RegBaAnalyzerStsReportFifoDepthMinusOne_u statusFifoDepth;
	RegBaAnalyzerIntErrorEn_u regBaAnalyzerIntErrorEn;	// must be configured	
	RegBaAnalyzerBaAnalyzerCountersEn_u baAnalyzerCountersEnReg; 
	RegBaAnalyzerRaCommonReportFifoBaseAddr_u commonRaReportFifoRegister;
	RegBaAnalyzerRaCommonReportFifoDepthMinusOne_u commonRaReportFifoDepth;
	RegBaAnalyzerRaUserReportFifoBaseAddr_u userRaReportFifoRegister;// for release 3.5
	RegBaAnalyzerRaUserReportFifoDepthMinusOne_u userRaReportFifoDepth;
	RegBaAnalyzerIntEn_u RegBaAnalyzerIntEnReg;
	RegBaAnalyzerTrafficIndCfg_u trafficIndCfgReg;
#ifdef ENET_INC_ARCH_WAVE600B
	RegBaAnalyzerMbaCfg_u mbaCfgReg;
#endif

	
	statusFifoRegister.val = 0; 
	statusFifoDepth.val = 0;
	commonRaReportFifoRegister.val = 0;
	commonRaReportFifoDepth.val = 0;
	userRaReportFifoRegister.val = 0;
	userRaReportFifoDepth.val = 0;
	dplIndexesRegister.val = 0;
	baAnalyzerCountersEnReg.val = 0;
	trafficIndCfgReg.val = 0;
#ifdef ENET_INC_ARCH_WAVE600B
	mbaCfgReg.val = 0;
#endif

	
	dplIndexesRegister.bitFields.crDplIdxSuccess = HW_Q_MANAGER_DONE_LIST_LIBERATOR;
	dplIndexesRegister.bitFields.crDplIdxDiscard = HW_Q_MANAGER_DONE_LIST_DISCARDED_PACKETS;
	RegAccess_Write(REG_BA_ANALYZER_DPL_IDX, dplIndexesRegister.val);

#if defined(ENET_INC_LMAC0) 
	statusFifoRegister.bitFields.crStsReportFifoBaseAddr = 				CONVERT_PHYSICAL_TO_SHRAM_OFFSET(&BaAnalyzerStatusFifo[0]);
	commonRaReportFifoRegister.bitFields.crRaCommonReportFifoBaseAddr = CONVERT_PHYSICAL_TO_SHRAM_OFFSET(&BaAnalyzerRaCommonReportFifo[0]);
	userRaReportFifoRegister.bitFields.crRaUserReportFifoBaseAddr = 	CONVERT_PHYSICAL_TO_SHRAM_OFFSET(&BaAnalyzerRaUserReportFifo);
#endif

#if defined(ENET_INC_LMAC1) 
	statusFifoRegister.bitFields.crStsReportFifoBaseAddr = 				CONVERT_PHYSICAL_TO_SHRAM_OFFSET(&BaAnalyzerStatusFifo_B1[0]);
	commonRaReportFifoRegister.bitFields.crRaCommonReportFifoBaseAddr = CONVERT_PHYSICAL_TO_SHRAM_OFFSET(&BaAnalyzerRaCommonReportFifo_B1[0]);
	userRaReportFifoRegister.bitFields.crRaUserReportFifoBaseAddr = 	CONVERT_PHYSICAL_TO_SHRAM_OFFSET(&BaAnalyzerRaUserReportFifo_B1[0]);
#endif
	RegAccess_Write(REG_BA_ANALYZER_STS_REPORT_FIFO_BASE_ADDR, statusFifoRegister.val);
	RegAccess_Write(REG_BA_ANALYZER_RA_COMMON_REPORT_FIFO_BASE_ADDR, commonRaReportFifoRegister.val);
	RegAccess_Write(REG_BA_ANALYZER_RA_USER_REPORT_FIFO_BASE_ADDR, userRaReportFifoRegister.val);
	/* The size in BA analyzer is + 1 to the value written by FW */
	statusFifoDepth.bitFields.crStsReportFifoDepthMinusOne = BA_ANALYZER_MAC_STATUS_FIFO_SIZE - 1;	  
	RegAccess_Write(REG_BA_ANALYZER_STS_REPORT_FIFO_DEPTH_MINUS_ONE, statusFifoDepth.val);
	commonRaReportFifoDepth.bitFields.crRaCommonReportFifoDepthMinusOne = BAA_COMMON_FIFO_SIZE - 1;
	RegAccess_Write(REG_BA_ANALYZER_RA_COMMON_REPORT_FIFO_DEPTH_MINUS_ONE, commonRaReportFifoDepth.val);
	userRaReportFifoDepth.bitFields.crRaUserReportFifoDepthMinusOne = BAA_USER_FIFO_SIZE - 1;
	RegAccess_Write(REG_BA_ANALYZER_RA_USER_REPORT_FIFO_DEPTH_MINUS_ONE, userRaReportFifoDepth.val);
	/*Enable BA rate adaptive fifo full interrupt*/
	
	RegAccess_Read(REG_BA_ANALYZER_INT_ERROR_EN,&regBaAnalyzerIntErrorEn.val);
#ifndef ENET_INC_ARCH_WAVE600D2
	regBaAnalyzerIntErrorEn.bitFields.crUnexpectedStaidIntEn = FALSE;
#else
	regBaAnalyzerIntErrorEn.bitFields.crTxStUnderflowIntEn = TRUE;
	regBaAnalyzerIntErrorEn.bitFields.crUnexpectedStaModeMpduLengthIntEn = FALSE; // NOTE: if turned on in the future (for Debug) need to clear via REG_BA_ANALYZER_CLEAR_INT_ERROR
	regBaAnalyzerIntErrorEn.bitFields.crUnexpectedStaModeRepeatedAidTidIntEn  = FALSE; /// NOTE: if turned on in the future (for Debug) need to clear via REG_BA_ANALYZER_CLEAR_INT_ERROR
#endif	
	regBaAnalyzerIntErrorEn.bitFields.crUnexpectedTidIntEn = FALSE;
	regBaAnalyzerIntErrorEn.bitFields.crUnexpectedAckType0Tid15IntEn = FALSE;
	regBaAnalyzerIntErrorEn.bitFields.crUnexpectedFragNumIntEn = FALSE;
	regBaAnalyzerIntErrorEn.bitFields.crUnexpectedBaLengthIntEn = FALSE;
	regBaAnalyzerIntErrorEn.bitFields.crDropTxStWhileFullIntEn = TRUE;
	regBaAnalyzerIntErrorEn.bitFields.crDropCommonRtAdaptWhileFullIntEn = TRUE;
#ifdef ENET_INC_ARCH_WAVE600B
    regBaAnalyzerIntErrorEn.bitFields.crDropFirstUserRtAdaptWhileFullIntEn = TRUE;
    regBaAnalyzerIntErrorEn.bitFields.crDropNonFirstUserRtAdaptWhileFullIntEn = TRUE;
	regBaAnalyzerIntErrorEn.bitFields.crTxStUnderflowIntEn = TRUE;
	regBaAnalyzerIntErrorEn.bitFields.crCommonRtAdaptUnderflowIntEn = TRUE;
	regBaAnalyzerIntErrorEn.bitFields.crUserRtAdaptUnderflowIntEn = TRUE;
#else
    regBaAnalyzerIntErrorEn.bitFields.crDropUserRtAdaptWhileFullIntEn = TRUE;
#endif
	RegAccess_Write(REG_BA_ANALYZER_INT_ERROR_EN, regBaAnalyzerIntErrorEn.val);

#ifdef ENET_INC_ARCH_WAVE600B
	//Initialize BA Analyzer configuration to parse BA frame up to number of expected TIDs . The rest of the fields should remain with default values.
	RegAccess_Read(REG_BA_ANALYZER_MBA_CFG,&mbaCfgReg.val);
	mbaCfgReg.bitFields.crMbaMaxNumTids = BAA_MBA_MAX_NUM_TID_EXPECTED;
	RegAccess_Write(REG_BA_ANALYZER_MBA_CFG, mbaCfgReg.val);
#endif

	/*Enable interrupts*/
	RegAccess_Read(REG_BA_ANALYZER_INT_EN,&RegBaAnalyzerIntEnReg.val);
	RegBaAnalyzerIntEnReg.bitFields.crCommonRateAdaptiveFifoNotEmptyEn = 0x0; // disable this interrupt, FW handles common report when user report arrives
	RegBaAnalyzerIntEnReg.bitFields.crUserRateAdaptiveFifoNotEmptyEn = 0x1;
	RegBaAnalyzerIntEnReg.bitFields.crTxStNotEmptyEn = 0x1;
	RegAccess_Write(REG_BA_ANALYZER_INT_EN, RegBaAnalyzerIntEnReg.val);

	/* Enable All Counters in the BAA module*/
	baAnalyzerCountersEnReg.bitFields.crBaAnalyzerCountersEn = BA_ANALYZER_ENABLE_ALL_COUNTERS;
	RegAccess_Write(REG_BA_ANALYZER_BA_ANALYZER_COUNTERS_EN, baAnalyzerCountersEnReg.val);

	// Enable traffic indicator (activity) bitmap update (reside in the RXC)
	trafficIndCfgReg.bitFields.crTrafficBaAgreementOriginatorEn = TRUE;
	RegAccess_Write(REG_BA_ANALYZER_TRAFFIC_IND_CFG, trafficIndCfgReg.val);
	
}


