/***********************************************************************************
 File:			Beacon_BlockingManager.c
 Module:			beacon Blocking Manager
 Purpose: 		handle Beacon Blocking Manager events
 Description:   
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/


#include "System_GlobalDefinitions.h"
#include "System_Configuration.h"
#include "Utils_Api.h"
#include "shram_man_msgs.h"
#include "shram_man_queues.h"
#include "stringLibApi.h"
#include "PacketDescriptor.h"
#include "TxSelector_Api.h"
#include "RegAccess_Api.h"
#include "ErrorHandler_Api.h"
#include "HwQManager_API.h"
#include "int_gen.h"
#include "lm.h"
#include "ShramPacketDescriptors.h"
#include "PacketDescriptorsPool_api.h"
#include "shramTxDesc.h"
#include "HwMemoryMap.h"
#include "OSAL_Api.h"
#include "Pac_Api.h"
#include "Locker_Api.h"
#include "BeaconHandler.h"
#include "Beacon_BlockingManager.h"
#include "BeaconHandler_api.h"
#include "PacManager_api.h"
#include "TxSender_ScratchPadApi.h"
#include "SenderInterface_Api.h"
#include "lmtimerUtility.h"
#include "loggerAPI.h"


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_BEACON_HANDLER 
#define LOG_LOCAL_FID 4


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

typedef enum
{
	BEACON_BLOCKING_MANAGER_UNBLOCKED_STATE,
	BEACON_BLOCKING_MANAGER_BLOCKED_STATE,
	BEACON_BLOCKING_MANAGER_NUM_STATES
} beaconBlockingManager_state;

typedef struct BeaconHandlerBlockingDb_s
{
	uint32	lastNumBeaconsTransmitted;
	uint32	activeBeaconsBitmap;
	uint16  initial; /* blocked to unblocked */
	uint16  iterative; /* unblocked to blocked */
	uint16  currentIterative;
	uint16	beaconInterval[HW_NUM_OF_VAPS];
	uint16	maxBeaconInterval;
	uint16	timerId;
	uint8 	numActiveBeacons;
	bool 	blockingEnabled;
	bool	timerActive;
	beaconBlockingManager_state	state;
} BeaconHandlerBlockingDb_t;

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void BeaconBlockingManagerChangeState(beaconBlockingManager_state newState);
static void BeaconBlockingManagerStartTimer(void);
static void BeaconBlockingManagerStopTimer(void);
static uint32 BeaconBlockingManager_getNumberOfBeconsTransmited(void);

void  isr_BeaconBlockingManagerTimerEvent(uint32 param);


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
static BeaconHandlerBlockingDb_t BeaconHandlerBlockingDb;

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/
void  isr_BeaconBlockingManagerTimerEvent(uint32 param)
{
	UNUSED_PARAM(param);	
	OSAL_SEND_NO_DATA_MESSAGE(PAC_MANAGER_BEACON_BLOCKING_TIMER, TASK_PAC_MANAGER, VAP_ID_DO_NOT_CARE);
	/*Invalidate timer ID - so we do not try to remove it while handling a previous message*/
	BeaconHandlerBlockingDb.timerId	= MAX_UINT16;
}

/**********************************************************************************

BeaconBlockingManagerChangeState 


Description:
------------
Start Timer


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void BeaconBlockingManagerChangeState(beaconBlockingManager_state newState)
{
	ILOG0_DD("BeaconBlockingManagerChangeState from %d to %d", BeaconHandlerBlockingDb.state, newState);
	BeaconHandlerBlockingDb.state = newState;
}

/**********************************************************************************

BeaconBlockingManagerStartTimer 


Description:
------------
Start Timer


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void BeaconBlockingManagerStartTimer(void)
{
	if (BeaconHandlerBlockingDb.timerActive == TRUE)
	{
		BeaconBlockingManagerStopTimer();
	}
	BeaconHandlerBlockingDb.timerId = TimerUtiltyAddEvent(1000 *BeaconHandlerBlockingDb.maxBeaconInterval * BeaconHandlerBlockingDb.currentIterative, isr_BeaconBlockingManagerTimerEvent, 0);
	BeaconHandlerBlockingDb.timerActive = TRUE;
}

/**********************************************************************************

BeaconBlockingManagerStopTimer 


Description:
------------
Stop Timer


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void BeaconBlockingManagerStopTimer(void)
{
	if (BeaconHandlerBlockingDb.timerId	!= MAX_UINT16)
	{
		TimerUtiltyRemoveEvent(&BeaconHandlerBlockingDb.timerId);
	}
	
	BeaconHandlerBlockingDb.timerActive = FALSE;
}

/*---------------------------------------------------------------------------------
/						global Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

BeaconBlockingManagerAddVap 


Description:
------------
VAP Add handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BeaconBlockingManagerAddVap(uint8 vapId, uint16 beaconInterval)
{
	/*Just Store Beacon Interval*/
	BeaconHandlerBlockingDb.beaconInterval[vapId] = beaconInterval;

	/*Add Beacon to Beacon Blocking Detection mechanism*/
	BeaconBlockingManagerAddBeacon(vapId);
}

/**********************************************************************************

BeaconBlockingManagerAddBeacon 


Description:
------------
Beacon Add handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BeaconBlockingManagerAddBeacon(uint8 vapId)
{
	if((BeaconHandlerBlockingDb.activeBeaconsBitmap & (1 << vapId)) != 0)
	{
		ASSERT(BeaconHandlerBlockingDb.numActiveBeacons < HW_NUM_OF_VAPS);
		/*Increment number of active beacons*/
		BeaconHandlerBlockingDb.numActiveBeacons++;
		/*Set Beacon bit in active bitmap*/
		BeaconHandlerBlockingDb.activeBeaconsBitmap |= (1 << vapId);
		/*Check if we have a new maximum Interval*/
		if (BeaconHandlerBlockingDb.maxBeaconInterval < BeaconHandlerBlockingDb.beaconInterval[vapId])
		{
			BeaconHandlerBlockingDb.maxBeaconInterval = BeaconHandlerBlockingDb.beaconInterval[vapId];
		}
		/*Is blocking enabled*/
		if (BeaconHandlerBlockingDb.blockingEnabled == TRUE)
		{
			/*Start the timer*/
			BeaconBlockingManagerStartTimer();
		}
	}
}

/**********************************************************************************

BeaconBlockingManagerRemoveBeacon 


Description:
------------
Beacon Remove handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BeaconBlockingManagerRemoveBeacon(uint8 vapId)
{
	uint8 i;
	uint16 maxInterval = 0;
	uint32 activeBitmap = 0;
	
	/*Sanity checks*/
	if((BeaconHandlerBlockingDb.activeBeaconsBitmap & (1 << vapId)) != 0)
	{
		ASSERT(BeaconHandlerBlockingDb.numActiveBeacons > 0);
		/*Decrement number of active beacons*/
		BeaconHandlerBlockingDb.numActiveBeacons--;
		/*Clear Beacon bit in active bitmap*/
		BeaconHandlerBlockingDb.activeBeaconsBitmap &= ~(1 << vapId);
		/*Any beacons left?*/
		if (BeaconHandlerBlockingDb.numActiveBeacons == 0)
		{
			/*No beacons left - was blocking enabled?*/
			if (BeaconHandlerBlockingDb.blockingEnabled == TRUE)
			{
				/*Stop the timer*/
				BeaconBlockingManagerStopTimer();
			}
		}
		else
		{
			/*Check if this was the maximum Interval*/
			if (BeaconHandlerBlockingDb.maxBeaconInterval == BeaconHandlerBlockingDb.beaconInterval[vapId])
			{
				activeBitmap = BeaconHandlerBlockingDb.activeBeaconsBitmap;
				/*Recalculate max interval*/
				while (activeBitmap)
				{
					i = Utils_FindFirstSetAndClear(&activeBitmap);
					if (BeaconHandlerBlockingDb.beaconInterval[i] >  maxInterval)
					{
						maxInterval = BeaconHandlerBlockingDb.beaconInterval[i];
					}
				}
				BeaconHandlerBlockingDb.maxBeaconInterval = maxInterval;
				/*We have a new max - is blocking enabled*/
				if (BeaconHandlerBlockingDb.blockingEnabled == TRUE)
				{
					/*Restart the timer*/
					BeaconBlockingManagerStartTimer();
				}
			}
		}
	}
}

/**********************************************************************************

BeaconBlockingManagerSet 


Description:
------------
Set handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BeaconBlockingManagerSet(K_MSG *pMsg)
{
	UMI_BeaconBlockTimerInterval_t* beaconBlockingSetMsg = (UMI_BeaconBlockTimerInterval_t*)pK_MSG_DATA(pMsg);
	uint32  numberOfBeaconsTransmited = 0;

	if (beaconBlockingSetMsg->getSetOperation == API_GET_OPERATION)
	{
		beaconBlockingSetMsg->initial = BeaconHandlerBlockingDb.initial;
		beaconBlockingSetMsg->iterative = BeaconHandlerBlockingDb.iterative;
	}
	else
	{
		if ((beaconBlockingSetMsg->initial > 0x0) || (beaconBlockingSetMsg->iterative > 0x0))
		{
			BeaconHandlerBlockingDb.blockingEnabled = TRUE;
			BeaconHandlerBlockingDb.initial = beaconBlockingSetMsg->initial;
			BeaconHandlerBlockingDb.iterative= beaconBlockingSetMsg->iterative;
			BeaconHandlerBlockingDb.currentIterative = BeaconHandlerBlockingDb.initial;
			if (BeaconHandlerBlockingDb.numActiveBeacons > 0)
			{
				BeaconBlockingManagerStartTimer();
			}
			/*Set invalid state so first sample always triggers message*/
			BeaconBlockingManagerChangeState(BEACON_BLOCKING_MANAGER_NUM_STATES);
			/*Set last number of beacons transmitted to current value*/
			numberOfBeaconsTransmited = BeaconBlockingManager_getNumberOfBeconsTransmited();
			BeaconHandlerBlockingDb.lastNumBeaconsTransmitted = numberOfBeaconsTransmited;
		}
		else
		{
			/*If Blocking is enabled, disable it*/
			if (BeaconHandlerBlockingDb.blockingEnabled == TRUE)
			{
				if (BeaconHandlerBlockingDb.numActiveBeacons > 0)
				{
					BeaconBlockingManagerStopTimer();
				}
				BeaconHandlerBlockingDb.blockingEnabled = FALSE;
			}
		}
	}
}

/**********************************************************************************

BeaconBlockingManagerStart 


Description:
------------
Start handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BeaconBlockingManagerStart(K_MSG* beaconBlockingManagerMessage)
{
	uint32 numberOfBeaconsTransmited =0;
	UNUSED_PARAM(beaconBlockingManagerMessage);	
	if (BeaconHandlerBlockingDb.blockingEnabled == TRUE)
	{
		if (BeaconHandlerBlockingDb.numActiveBeacons > 0)
		{
			BeaconBlockingManagerStartTimer();
		}
		/*Set invalid state so first sample always triggers message*/
		BeaconBlockingManagerChangeState(BEACON_BLOCKING_MANAGER_NUM_STATES);
		/*Set last number of beacons transmitted to current value*/
		numberOfBeaconsTransmited = BeaconBlockingManager_getNumberOfBeconsTransmited();
		BeaconHandlerBlockingDb.lastNumBeaconsTransmitted = numberOfBeaconsTransmited;
	}
}

/**********************************************************************************

BeaconBlockingManagerStop 


Description:
------------
Stop handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BeaconBlockingManagerStop(K_MSG* beaconBlockingManagerMessage)
{
	UNUSED_PARAM(beaconBlockingManagerMessage);	
	if (BeaconHandlerBlockingDb.blockingEnabled == TRUE)
	{
		if (BeaconHandlerBlockingDb.numActiveBeacons > 0)
		{
			BeaconBlockingManagerStopTimer();
		}
	}
}

/**********************************************************************************

BeaconBlockingManagerRestart 


Description:
------------
Restart handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BeaconBlockingManagerRestart(K_MSG* beaconBlockingManagerMessage)
{
	UNUSED_PARAM(beaconBlockingManagerMessage);	
	/*Change state so we report state to driver next time*/
	BeaconBlockingManagerChangeState(BEACON_BLOCKING_MANAGER_NUM_STATES);
}

/**********************************************************************************

BeaconBlockingManagerTimer 


Description:
------------
Timer handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

void BeaconBlockingManagerTimer(K_MSG* beaconBlockingManagerMessage)
{
	K_MSG *beaconBlockingMessage	= NULL;
	UMI_Beacon_Block_t *isBeaconBlocked = NULL;
	uint32 numberOfBeaconsTransmited= 0;
	UNUSED_PARAM(beaconBlockingManagerMessage);	
	ILOG0_V("BeaconBlockingManagerTimer");
	if (BeaconHandlerBlockingDb.timerActive == TRUE)
	{
		if (BeaconHandlerBlockingDb.numActiveBeacons > 0)
		{
			/*Read number of Beacons transmitted*/
			numberOfBeaconsTransmited = BeaconBlockingManager_getNumberOfBeconsTransmited();
#if defined (LOG_BEACON_BLOCK)
			ILOG0_DD("pScratchPadApiParams->numOfBeaconsTransmitted =%d, BeaconHandlerBlockingDb.lastNumBeaconsTransmitted =%d", numberOfBeaconsTransmited, BeaconHandlerBlockingDb.lastNumBeaconsTransmitted);
#endif
			/*Is it the same as the last time?*/
			if (numberOfBeaconsTransmited == BeaconHandlerBlockingDb.lastNumBeaconsTransmitted)
			{
				if (BeaconHandlerBlockingDb.state != BEACON_BLOCKING_MANAGER_BLOCKED_STATE)
				{
					beaconBlockingMessage = OSAL_GET_MESSAGE(sizeof(UMI_Beacon_Block_t));
					isBeaconBlocked =  ((UMI_Beacon_Block_t *)pK_MSG_DATA(beaconBlockingMessage));
					isBeaconBlocked->beaconBlock = 1;
					BeaconHandlerBlockingDb.currentIterative = BeaconHandlerBlockingDb.iterative;
					/*Notify Driver*/
					OSAL_SEND_MESSAGE(UMI_MC_MAN_BEACON_BLOCKING_IND, TASK_UM_IF_TASK, beaconBlockingMessage, GET_DEFAULT_VAP_FOR_MY_BAND());
					/*Set state to blocked*/
					BeaconBlockingManagerChangeState(BEACON_BLOCKING_MANAGER_BLOCKED_STATE);
				}
			}
			else
			{
				if (BeaconHandlerBlockingDb.state != BEACON_BLOCKING_MANAGER_UNBLOCKED_STATE)
				{
					/*Notify Driver*/
					beaconBlockingMessage = OSAL_GET_MESSAGE(sizeof(UMI_Beacon_Block_t));
					isBeaconBlocked =  ((UMI_Beacon_Block_t *)pK_MSG_DATA(beaconBlockingMessage));
					isBeaconBlocked->beaconBlock = 0;
					BeaconHandlerBlockingDb.currentIterative = BeaconHandlerBlockingDb.initial;
					OSAL_SEND_MESSAGE(UMI_MC_MAN_BEACON_BLOCKING_IND, TASK_UM_IF_TASK, beaconBlockingMessage, GET_DEFAULT_VAP_FOR_MY_BAND());
					/*Set state to unblocked*/
					BeaconBlockingManagerChangeState(BEACON_BLOCKING_MANAGER_UNBLOCKED_STATE);
				}
			}
			/*Store new value*/
			BeaconHandlerBlockingDb.lastNumBeaconsTransmitted = numberOfBeaconsTransmited;
			BeaconHandlerBlockingDb.timerActive = FALSE;
			BeaconBlockingManagerStartTimer();
		}
	}
}


static uint32 BeaconBlockingManager_getNumberOfBeconsTransmited(void)
{
	TxSender_ScratchPadApiParams_t *pScratchPadApiParams = NULL;
#ifdef ENET_INC_ARCH_WAVE600
#if defined (ENET_INC_LMAC0)
	pScratchPadApiParams = (TxSender_ScratchPadApiParams_t *)(B0_MAC_GENRISC_TX_SPRAM_BASE_ADDR + (SCPAD_ADDRESS_TX_SENDER_SCRATCHPAD_API_STRUCTURE_START << 0x2));
#endif
#if defined (ENET_INC_LMAC1)
	pScratchPadApiParams = (TxSender_ScratchPadApiParams_t *)(B1_MAC_GENRISC_TX_SPRAM_BASE_ADDR + (SCPAD_ADDRESS_TX_SENDER_SCRATCHPAD_API_STRUCTURE_START << 0x2));
#endif
#else
	pScratchPadApiParams = (TxSender_ScratchPadApiParams_t *)(MAC_GENRISC_TX_SPRAM_BASE_ADDR + (SCPAD_ADDRESS_TX_SENDER_SCRATCHPAD_API_STRUCTURE_START << 0x2));
#endif
	return pScratchPadApiParams->numOfBeaconsTransmitted;
}


/**********************************************************************************

BeaconBlockingManager_Init



Description:
------------
Initiailize Beacon Blocking MAnager


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=".initialization_start" 
#endif
void BeaconBlockingManager_Init(void)
{
	/*Initialize Beacon Blocking DB*/
	memset(&BeaconHandlerBlockingDb, 0, sizeof(BeaconHandlerBlockingDb_t));
	BeaconHandlerBlockingDb.timerId = MAX_UINT16;
}
#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=default
#endif



