/***********************************************************************************
 File:			BeaconHandler.c
 Module:			Beacon Handler
 Purpose:		Builds Beacon frames and configures Beacon HW
 Description:	This src file is the implementation of the Beacon Handler module
			which is responsible for configuring the Beacon HW and preparing Beacon frames
			for transmission.
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_GlobalDefinitions.h"
#include "System_Configuration.h"
#include "System_MainApi.h"
#include "shram_man_msgs.h"
#include "shram_man_queues.h"
#include "stringLibApi.h"
#include "VapDatabase_Api.h"
#include "PacketDescriptor.h"
#include "TxSelector_Api.h"
#include "RegAccess_Api.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"
#include "HwQManager_API.h"
#include "int_gen.h"
#include "lm.h"
#include "ShramPacketDescriptors.h"
#include "BeaconRegs.h"
#include "PacketDescriptorsPool_api.h"
#include "shramTxDesc.h"
#include "HwMemoryMap.h"
#include "Pac_Api.h"
#include "Locker_Api.h"
#include "BeaconHandler_api.h"
#include "PacManager_api.h"
#include "BeaconHandler.h"
#include "Beacon_CsaManager.h"
#include "System_MainApi.h"
#include "Beacon_BlockingManager.h"
#include "loggerAPI.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_BEACON_HANDLER 
#define LOG_LOCAL_FID 0

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/
#define BEACON_HANDLER_AGGR					(1)					// flag indicating a beacon frame in first PD of beacon "aggregation"
// mask used for determining if VAP is active
#define BEACON_HANDLER_VAP_ACTIVE_MASK      (0x0001)
/* Beacon Handler Defines */
/* Used for Activate/Deactivate Beacon HW */
#define 	BEACON_HANDLER_DISABLE  (0)
#define 	BEACON_HANDLER_ENABLE   (1)

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
/* Management struct of the LM Beacon Handler */
typedef struct BeaconHandlerManagement_s
{
	PacketDescriptorPool_t      pdPool;	// Pool of beacon packet descriptors 
	uint16	isBssLoadIe;				// vap bitmask
} BeaconHandlerManagement_t;

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
void beaconHandlerActivateHw(uint8 vapId, BeaconHwParams_t* pBeaconParams);
void beaconHandlerDeactivateHw(uint8 vapIdx);
void beaconHandlerBssLoadIeInit(void);
void beaconHandlerBssLoadIeBuild(TxPd_t *pd);

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
/* Beacon management object holds all the variables that are used for implementing the beacon handler */
static BeaconHandlerManagement_t BeaconManagement;
/*	variables used when calling HwQManger modules. We initialize them once at init and 
	only set the parameters that are changed during run time */
static HwQueueManagerRequestParams_t BeaconHwQueueManagerRequestParams;


//BeaconHandlerVapDb_t* beaconHandlerVapdb; // This was changed to pointer since it will be overlayed on the initialization section
BeaconHandlerVapDb_t beaconHandlerVapdb[HW_NUM_OF_VAPS]; 





/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

beaconHandler_ConfigureHw 

Description:
------------
	Configure the beacon HW module
	
Input:
-----
	uint8   vapId - indicates the vap we want to configure the beacon for it.
	BeaconHwParams_t - The configuration parameters  required by the Beacon HW module
		
Output:
-------
	void

Returns:
--------
	void  
	
**********************************************************************************/
void beaconHandlerActivateHw(uint8 vapId, BeaconHwParams_t* pBeaconHwParams)
{
	uint32 activateRegAddress = 0;
	uint32 offsetRegAddress = 0;
	RegBeaconVap0BeaconActivate_u beaconActivateReg;
	RegBeaconBeaconLcm_u		  beaconLcmReg; 
	RegBeaconVap0BeaconOffset_u	  beaconOffsetReg;

	beaconActivateReg.val = 0;
	beaconLcmReg.val = 0;
	beaconOffsetReg.val = 0;
	//("beaconHandlerActivateHw for vap:%d, Lcm:%d, Offset:%d, Beacon interval:%d, Dtim Interval:%d", vapId, pBeaconHwParams->beaconLCM, pBeaconHwParams->beaconOffset, pBeaconHwParams->beaconInterval, pBeaconHwParams->dtimInterval);
	beaconLcmReg.bitFields.lcm = pBeaconHwParams->beaconLCM;
	// configure LCM - least common multiplier is at register offset 0
	RegAccess_Write(REG_BEACON_BEACON_LCM, beaconLcmReg.val);
	// get address of VAP Beacon configuration registers
	activateRegAddress = REG_BEACON_VAP0_BEACON_ACTIVATE + (vapId * REG_BEACON_HANDLER_VAP_CONFIG_SIZE);
	offsetRegAddress = REG_BEACON_VAP0_BEACON_OFFSET + (vapId * REG_BEACON_HANDLER_VAP_CONFIG_SIZE);
	// set offset. The beacon offset setting register is after the activate register (Per VAP), so we add BEACON_ACTIVATE_SIZE
	// to the address
	beaconOffsetReg.bitFields.vap0BeaconOffset = pBeaconHwParams->beaconOffset;
	RegAccess_Write(offsetRegAddress, beaconOffsetReg.val);
	// enable beacon
	beaconActivateReg.bitFields.vap0BeaconEn = BEACON_HANDLER_ENABLE;
	// set beacon interval
	beaconActivateReg.bitFields.vap0BeaconInterval = pBeaconHwParams->beaconInterval;
	// set dtim interval
	beaconActivateReg.bitFields.vap0DtimInterval = pBeaconHwParams->dtimInterval;
	// activate beacon
	RegAccess_Write(activateRegAddress, beaconActivateReg.val);
}

/**********************************************************************************

beaconHandler_DeactivateHw 

Description:
------------
	Disable the beacon HW module
	
Input:
-----
	uint8   vapId - indicates the vap we want to deactivate the beacon for it.
		
Output:
-------
	void

Returns:
--------
	void  
	
**********************************************************************************/
void beaconHandlerDeactivateHw (uint8 vapId)
{
	uint32 regAddress = 0;
	RegBeaconVap0BeaconActivate_u beaconActivateReg;

	beaconActivateReg.val = 0;
	ILOG2_D("beaconHandlerDeactivateHw for vap:%d", vapId);
	// get address of VAP Beacon configuration registers
	regAddress = REG_BEACON_VAP0_BEACON_ACTIVATE + (vapId * REG_BEACON_HANDLER_VAP_CONFIG_SIZE);
	// set beacon deactivate bit. The LSB bit in the activate register is the Disable/Enable bit 
	beaconActivateReg.bitFields.vap0BeaconEn = BEACON_HANDLER_DISABLE;
	RegAccess_Write(regAddress, beaconActivateReg.val);
}

/**********************************************************************************

beaconHandler_SendTemplateWasSetIndication() 

Description:
------------
	Sends a template was set indication to UMAC
	
Input:
-----
	void
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
void beaconHandlerSendTemplateWasSetIndication(uint8 vapId)
{
	K_MSG *pMsg = OSAL_GET_MESSAGE(0);

	ILOG2_D("beaconHandlerSendTemplateWasSetIndication for vap:%d", vapId);
	pMsg->header.vapId = vapId;
	OSAL_SEND_MESSAGE(UMI_BEACON_TEMPLATE_WAS_SET_IND, TASK_UM_IF_TASK, pMsg, vapId);
}

/**********************************************************************************

beaconHandler_IsVapActive() 

Description:
------------
	return True if VAP is enabled
	
Input:
-----
	uint8 vapIdx
		
Output:
-------
	True - VAP is active
	False - VAP is inactive

Returns:
--------
	void - 
	
**********************************************************************************/
bool beaconHandlerIsVapActive(uint8 vapIdx)
{
	uint32 vapActiveValue = 0;
	RegBeaconVapActiveStatus_u beaconActivateStatusReg;

	beaconActivateStatusReg.val = 0;
	// read current value	
	RegAccess_Read(REG_BEACON_VAP_ACTIVE_STATUS, &beaconActivateStatusReg.val);
	vapActiveValue = (beaconActivateStatusReg.val >> vapIdx) & BEACON_HANDLER_VAP_ACTIVE_MASK;
	if (vapActiveValue) 
	{
		return TRUE;
	}
	else
	{
		return FALSE;
	}
}

/**********************************************************************************

beaconHandler_UpdateIEsPdList 

Description:
------------
	Creates a free template PD list and copies the Driver buffers addresses and lengths to the PDs.
	If the VAP is active
	
Input:
-----
	uint8 vapId
	BeaconHostBuffer_t*  pBuffersArray - buffers received from the Host
	uint8 numOfBuffers - Number of buffers received from the host
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void beaconHandlerUpdateIEsPdList(uint8 vapId, BeaconHostBuffer_t*  pBuffersArray)
{
	TxPd_t* pPulledPd = NULL;
	TxPd_t* pBeaconPd = NULL;
	BeaconHandlerVapDb_t *beaconHandlerVapDbEntry = &beaconHandlerVapdb[vapId];
	uint8 pdIndex = 0;

	ILOG2_D("beaconHandlerUpdateIEsPdList for vap:%d", vapId);
	pPulledPd = Pool_PullPd(&(BeaconManagement.pdPool));
	pBeaconPd = pPulledPd;
	beaconHandlerVapDbEntry->pPendingTemplateHead = pBeaconPd;
	// Host Part 1 PD (pointing to part1 of host template)	
	pBeaconPd->beaconProberesSubtype = PD_BEACON_PROBE_RESP_SUBTYPE_NORMAL;
	pBeaconPd->packetPointer = CONVERT_HOST_DDR_MEMORY_ADDR_TO_DMA_DDR_MEMORY_ADDR(pBuffersArray[BEACON_HOST_PART1_BUFFER].bufferAddress);
	pBeaconPd->aggregationIndication = BEACON_HANDLER_AGGR; 
	pBeaconPd->dataLength  = pBuffersArray[BEACON_HOST_PART1_BUFFER].bufferLength; 
	/*Set the VAP ID in the first PD of the queue - we need it for CSA support*/
	pBeaconPd->txQVapId = vapId;
	// Allocate new PD from beacon pool
	pPulledPd = Pool_PullPd(&(BeaconManagement.pdPool));
	// Set next pointer of PD to point to the new allocated PD. 
	pBeaconPd->nextPdPointer =  SET_NEXT_PD(pPulledPd);
	// Set the pdIndex field and advance the index
	pBeaconPd->beaconProberesIndex = pdIndex;
	pdIndex++;
	// TIM PD
	// advance to the new allocated PD and set its fields
	pBeaconPd = pPulledPd;
	pBeaconPd->beaconProberesSubtype = PD_BEACON_PROBE_RESP_SUBTYPE_TIM;
	pBeaconPd->dataLength = 0;
	// Set the previous PD pdIndex field
	pBeaconPd->beaconProberesIndex = pdIndex;
	// Set next PD pointer to NULL
	pBeaconPd->nextPdPointer = NEXT_PD_NULL; 
	pdIndex++;
	// Part 2 of host template - optional
	if (pBuffersArray[BEACON_HOST_PART2_BUFFER].bufferLength)
	{
		// Allocate new PD from beacon pool
		pPulledPd = Pool_PullPd(&(BeaconManagement.pdPool));
		// Set next pointer
		pBeaconPd->nextPdPointer = SET_NEXT_PD(pPulledPd);
		// advance to the new allocated PD and set its fields
		pBeaconPd =  pPulledPd;
		pBeaconPd->beaconProberesSubtype = PD_BEACON_PROBE_RESP_SUBTYPE_NORMAL;
		pBeaconPd->packetPointer = CONVERT_HOST_DDR_MEMORY_ADDR_TO_DMA_DDR_MEMORY_ADDR(pBuffersArray[BEACON_HOST_PART2_BUFFER].bufferAddress);
		pBeaconPd->dataLength  = pBuffersArray[BEACON_HOST_PART2_BUFFER].bufferLength;
		// Set the previous PD pdIndex field
		pBeaconPd->beaconProberesIndex = pdIndex;
		// Set next PD pointer to NULL
		pBeaconPd->nextPdPointer = NEXT_PD_NULL; 
		pdIndex++;
	}
	// Channel switch (IE37, IE62)
	if (BeaconCsaManagerCsaRequired())
	{
		// Allocate new PD from beacon pool
		pPulledPd = Pool_PullPd(&(BeaconManagement.pdPool));
		// Set next pointer
		pBeaconPd->nextPdPointer = SET_NEXT_PD(pPulledPd);
		// advance to the new allocated PD and set its fields
		pBeaconPd =  pPulledPd;
		pBeaconPd->beaconProberesSubtype = PD_BEACON_PROBE_RESP_SUBTYPE_CSA;
		// Set the previous PD pdIndex field
		pBeaconPd->beaconProberesIndex = pdIndex;
		// Set next PD pointer to NULL
		pBeaconPd->nextPdPointer = NEXT_PD_NULL; 
		/*Set the VAP ID in this PD so we can allocate the correct CSA IE payload*/
		pBeaconPd->txQVapId = vapId;
		/*Call Beacon CSA manager to build CSA IE*/
		BeaconCsaManagerBuildCsaIe(pBeaconPd);
		pdIndex++;
	}
	// Part 3 of host template
	// advance to the new allocated PD and set its fields
	if (pBuffersArray[BEACON_HOST_PART3_BUFFER].bufferLength)
	{
		// Allocate new PD from beacon pool
		pPulledPd = Pool_PullPd(&(BeaconManagement.pdPool));
		// Set next pointer
		pBeaconPd->nextPdPointer = SET_NEXT_PD(pPulledPd);
		pBeaconPd = pPulledPd;
		pBeaconPd->beaconProberesSubtype = PD_BEACON_PROBE_RESP_SUBTYPE_NORMAL;
		pBeaconPd->packetPointer = CONVERT_HOST_DDR_MEMORY_ADDR_TO_DMA_DDR_MEMORY_ADDR(pBuffersArray[BEACON_HOST_PART3_BUFFER].bufferAddress);
		pBeaconPd->dataLength = pBuffersArray[BEACON_HOST_PART3_BUFFER].bufferLength;
		pBeaconPd->beaconProberesIndex = pdIndex;
		// Set next PD pointer to NULL
		pBeaconPd->nextPdPointer = NEXT_PD_NULL; 
		pdIndex++;
	}
	// HS20 bss load IE (IE#11)
	if (beaconHandlerBssLoadIeIsSupported(vapId))
	{
		// Allocate new PD from beacon pool
		pPulledPd = Pool_PullPd(&(BeaconManagement.pdPool));
		// Set next pointer of CSA PD to point to the new allocated PD. 
		pBeaconPd->nextPdPointer = SET_NEXT_PD(pPulledPd);
		// advance to the new allocated PD and set its fields
		pBeaconPd =  pPulledPd;
		pBeaconPd->beaconProberesSubtype = PD_BEACON_PROBE_RESP_SUBTYPE_NORMAL;
		// Set the previous PD pdIndex field
		pBeaconPd->beaconProberesIndex = pdIndex;
		// Set next PD pointer to NULL
		pBeaconPd->nextPdPointer = NEXT_PD_NULL; 
		/*Set the VAP ID in this PD so we can allocate the correct CSA IE payload*/
		pBeaconPd->txQVapId = vapId;
		/*Call Beacon HS20 Handler to build Bss Load IE*/
		beaconHandlerBssLoadIeBuild(pBeaconPd);
		pdIndex++;
	}
	// Part 4 of host template
	if (pBuffersArray[BEACON_HOST_PART4_BUFFER].bufferLength)
	{
		// Allocate new PD from beacon pool
		pPulledPd = Pool_PullPd(&(BeaconManagement.pdPool));
		// Set next pointer of CSA PD to point to the new allocated PD. 
		pBeaconPd->nextPdPointer = SET_NEXT_PD(pPulledPd);
		pBeaconPd = pPulledPd;
		pBeaconPd->beaconProberesSubtype = PD_BEACON_PROBE_RESP_SUBTYPE_NORMAL;
		pBeaconPd->packetPointer = CONVERT_HOST_DDR_MEMORY_ADDR_TO_DMA_DDR_MEMORY_ADDR(pBuffersArray[BEACON_HOST_PART4_BUFFER].bufferAddress);
		pBeaconPd->dataLength  = pBuffersArray[BEACON_HOST_PART4_BUFFER].bufferLength;
		// Set next PD pointer to NULL
		pBeaconPd->nextPdPointer = NEXT_PD_NULL; 
		pBeaconPd->beaconProberesIndex = pdIndex;
		pdIndex++;
	}
	if ((beaconHandlerVapDbEntry->isVht == TRUE) && (BeaconCsaManagerCsaWrapperRequired() == TRUE))
	{
		// Allocate new PD from beacon pool
		pPulledPd = Pool_PullPd(&(BeaconManagement.pdPool));
		// Set next pointer of CSA PD to point to the new allocated PD. 
		pBeaconPd->nextPdPointer = SET_NEXT_PD(pPulledPd);
		// advance to the new allocated PD and set its fields
		pBeaconPd =  pPulledPd;
		pBeaconPd->beaconProberesSubtype = PD_BEACON_PROBE_RESP_SUBTYPE_NORMAL;
		// Set next PD pointer to NULL
		pBeaconPd->nextPdPointer = NEXT_PD_NULL; 
		// Set the previous PD pdIndex field
		pBeaconPd->beaconProberesIndex = pdIndex;
		/*Set the VAP ID in this PD so we can allocate the correct CSA IE payload*/
		pBeaconPd->txQVapId = vapId;
		/*Call Beacon CSA manager to build CSA IE*/
		BeaconCsaManagerBuildCsaWrapperIe(pBeaconPd);
		pdIndex++;
	}
	if (pBuffersArray[BEACON_HOST_PART5_BUFFER].bufferLength)
	{
		// Allocate new PD from beacon pool
		pPulledPd = Pool_PullPd(&(BeaconManagement.pdPool));
		// Set next pointer of CSA PD to point to the new allocated PD. 
		pBeaconPd->nextPdPointer = SET_NEXT_PD(pPulledPd);
		// Part 5 of host template
		pBeaconPd =  pPulledPd;
		pBeaconPd->beaconProberesSubtype = PD_BEACON_PROBE_RESP_SUBTYPE_NORMAL;
		pBeaconPd->packetPointer = CONVERT_HOST_DDR_MEMORY_ADDR_TO_DMA_DDR_MEMORY_ADDR(pBuffersArray[BEACON_HOST_PART5_BUFFER].bufferAddress);
		pBeaconPd->dataLength  = pBuffersArray[BEACON_HOST_PART5_BUFFER].bufferLength;
		// Set next PD pointer to NULL
		pBeaconPd->nextPdPointer = NEXT_PD_NULL; 
		pBeaconPd->beaconProberesIndex = pdIndex;
		pdIndex++;	
	}
	// set the total number of PDs in Beacon frame for the aggregator.
	beaconHandlerVapDbEntry->pPendingTemplateHead->pdCounter = pdIndex;
	beaconHandlerVapDbEntry->pPendingTemplateTail = pBeaconPd;
}

/**********************************************************************************

beaconHandlerFlushIEsPdList 

Description:
------------
	
Input:
-----
	uint8 vapId
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void beaconHandlerFlushIEsPdList(uint8 vapId)
{
	TxPd_t* pOldTemplateHeadPd = NULL;
	TxPd_t* pOldTemplateTailPd = NULL;
	uint8 pdCounter;

	//("beaconHandlerFlushIEsPdList of Vap %d",vapId);
	// flush beacon template PDs
	BeaconHwQueueManagerRequestParams.primaryAddr = vapId; 
	HwQManager_FlushQ(&BeaconHwQueueManagerRequestParams);
	pOldTemplateHeadPd = (TxPd_t *)BeaconHwQueueManagerRequestParams.pHeadDesc;
	pOldTemplateTailPd = (TxPd_t *)BeaconHwQueueManagerRequestParams.pTailDesc;
	if (pOldTemplateHeadPd != (TxPd_t *)NULL_PD)
	{
		pdCounter = ((TxPd_t*)BeaconHwQueueManagerRequestParams.pHeadDesc)->pdCounter;
		Pool_ReleasePdList(&(BeaconManagement.pdPool), (TxPd_t*)pOldTemplateHeadPd, (TxPd_t*)pOldTemplateTailPd, pdCounter);
		BeaconBlockingManagerRemoveBeacon(vapId);
	}
}
	
/**********************************************************************************

isr_beaconHandler_SwapIEsPdList 

Description:
------------
	ISR called when the locker indicates that the Beacon Queue is locked and can be updated.
	
Input:
-----
	uint8 vapId
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void beaconHandlerSwapIEsPdList(uint8 vapId)
{
	BeaconHandlerVapDb_t *beaconHandlerVapDbEntry = &beaconHandlerVapdb[vapId];
	TxPd_t* pOldTemplateHeadPd = NULL;
	TxPd_t* pOldTemplateTailPd = NULL;
	uint16 pdCounter;

	//("isr_beaconHandlerSwapIEsPdList Vap Id:%d", vapId);
	// flush old template PDs
	/*Check if template update was overwritten by CSA*/
	if (beaconHandlerVapDbEntry->csaOverwrite == FALSE)
	{
		BeaconHwQueueManagerRequestParams.primaryAddr = vapId;
		HwQManager_FlushQ(&BeaconHwQueueManagerRequestParams);
		// push new template PDs
		pOldTemplateHeadPd = (TxPd_t *)BeaconHwQueueManagerRequestParams.pHeadDesc;
		pOldTemplateTailPd = (TxPd_t *)BeaconHwQueueManagerRequestParams.pTailDesc;
	//	("isr_beaconHandlerSwapIEsPdList Vap = %d old pFreeTemplateHead:%x  old pFreeTemplateTail:%x",vapId, pOldTemplateHeadPd,pOldTemplateTailPd);
		BeaconHwQueueManagerRequestParams.pHeadDesc = beaconHandlerVapDbEntry->pPendingTemplateHead;
		BeaconHwQueueManagerRequestParams.pTailDesc = beaconHandlerVapDbEntry->pPendingTemplateTail;
		/*check if queue is active and that we have a template, could be inactive if swap- caused by CSA after VAP was added but beacon queue not enabled yet*/
		if ((beaconHandlerVapDbEntry->isBeaconQueueActive == TRUE) && (beaconHandlerVapDbEntry->pPendingTemplateHead != NULL) && (beaconHandlerVapDbEntry->pPendingTemplateHead->dataLength > 0))
		{
			HwQManager_PushPacketlistToHead(&BeaconHwQueueManagerRequestParams);
		}
		else
		{
			beaconHandlerReleasePdList(vapId);
		}
	//	("isr_beaconHandlerSwapIEsPdList Vap pFreeTemplateHead:%x pFreeTemplateTail:%x", beaconHandlerVapDbEntry->pPendingTemplateHead,beaconHandlerVapDbEntry->pPendingTemplateTail);
		// Unlock the Beacon Queue
		Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_BEACON,vapId,(0x1 <<0));
	    // Release the old template PDs and swap the pointers
		if (pOldTemplateHeadPd != (TxPd_t *)NULL_PD)
	    {
			pdCounter = ((TxPd_t*)pOldTemplateHeadPd)->pdCounter;
	    	Pool_ReleasePdList(&(BeaconManagement.pdPool), (TxPd_t*)pOldTemplateHeadPd, (TxPd_t*)pOldTemplateTailPd, pdCounter);
			BeaconBlockingManagerRemoveBeacon(vapId);
	    }
		/*check if queue is active and that we have a tenplate, could be inactive if swap- caused by CSA after VAP was added but beacon queue not enabled yet*/
		if ((beaconHandlerVapDbEntry->isBeaconQueueActive == TRUE) && (beaconHandlerVapDbEntry->pPendingTemplateHead != NULL) && (beaconHandlerVapDbEntry->pPendingTemplateHead->dataLength > 0))
		{
			BeaconBlockingManagerAddBeacon(vapId);
		}
		beaconHandlerVapDbEntry->pPendingTemplateHead = NULL;
		beaconHandlerVapDbEntry->pPendingTemplateTail = NULL;
	}
	else
	{
		// Unlock the Beacon Queue
		Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_BEACON,vapId,(0x1 <<0));
		/*and clear indication*/
		beaconHandlerVapDbEntry->csaOverwrite = FALSE;
	}
}

/**********************************************************************************

beaconHandlerReleasePdList 

Description:
------------
	this function is called to release a pending template that was never set
	
Input:
-----
	uint8 vapId
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void beaconHandlerReleasePdList(uint8 vapId)
{
	BeaconHandlerVapDb_t *beaconHandlerVapDbEntry = &beaconHandlerVapdb[vapId];
	uint8 pdCounter;

	if (beaconHandlerVapDbEntry->pPendingTemplateHead != NULL)
	{
		pdCounter= ((TxPd_t*)beaconHandlerVapDbEntry->pPendingTemplateHead)->pdCounter;
	    Pool_ReleasePdList(&(BeaconManagement.pdPool), (TxPd_t*)beaconHandlerVapDbEntry->pPendingTemplateHead, (TxPd_t*)beaconHandlerVapDbEntry->pPendingTemplateTail, pdCounter);
		beaconHandlerVapDbEntry->pPendingTemplateHead = NULL;
		beaconHandlerVapDbEntry->pPendingTemplateTail = NULL;
	}
}

/**********************************************************************************
BeaconHandler_SetVapVHT  

Description:
------------
	Sets the VAP VHT flag 
Input:
-----
	VAP ID, VHT
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void BeaconHandler_SetVapVHT(uint8 vapId, bool isVht)
{
	/*Set VHT flag*/
	beaconHandlerVapdb[vapId].isVht = isVht;
}

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

BeaconHandler_TemplateUpdate 

Description:
------------
	Handles the set beacon message sent from the UMAC
	
Input:
-----
	K_MSG* pMsg - beacon set message  received from UMAC - contains the Host template buffers
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BeaconHandler_TemplateUpdate(K_MSG *beaconHandlerMessage)
{
	BssManagerSetTemplateReq_t* pBeaconSetTemplateReq = (BssManagerSetTemplateReq_t*)pK_MSG_DATA(beaconHandlerMessage);
	UMI_BEACON_SET* beaconTemplateParams_p = (UMI_BEACON_SET*)pK_MSG_DATA(pBeaconSetTemplateReq->psMsg);
	uint8 vapId = beaconTemplateParams_p->u8vapIndex;
	BeaconHandlerVapDb_t *beaconHandlerVapDb = &beaconHandlerVapdb[vapId]; 
	RequesterLockParams_t requesterLockParams;
	uint16 offset = 0;

	/*Template Update should not be received during CSA*/
	if (BeaconCsaManagerCsaInProgress() == TRUE)
	{
		ASSERT(FALSE);
	}
	beaconHandlerVapDb->beaconBuffers[0].bufferAddress 	= beaconTemplateParams_p->u32hostAddress;
	beaconHandlerVapDb->beaconBuffers[0].bufferLength	= beaconTemplateParams_p->u16part1Len;
	offset += beaconTemplateParams_p->u16part1Len;
	beaconHandlerVapDb->beaconBuffers[1].bufferAddress	= beaconTemplateParams_p->u32hostAddress + offset;
	beaconHandlerVapDb->beaconBuffers[1].bufferLength	= beaconTemplateParams_p->u16part2Len;
	offset += beaconTemplateParams_p->u16part2Len;
	beaconHandlerVapDb->beaconBuffers[2].bufferAddress	= beaconTemplateParams_p->u32hostAddress + offset;
	beaconHandlerVapDb->beaconBuffers[2].bufferLength	= beaconTemplateParams_p->u16part3Len;
	offset += beaconTemplateParams_p->u16part3Len;
	beaconHandlerVapDb->beaconBuffers[3].bufferAddress	= beaconTemplateParams_p->u32hostAddress + offset;
	beaconHandlerVapDb->beaconBuffers[3].bufferLength	= beaconTemplateParams_p->u16part4Len;
	offset += beaconTemplateParams_p->u16part4Len;
	beaconHandlerVapDb->beaconBuffers[4].bufferAddress	= beaconTemplateParams_p->u32hostAddress + offset;
	beaconHandlerVapDb->beaconBuffers[4].bufferLength	= beaconTemplateParams_p->u16part5Len;
	BeaconCsaManagerUpdateTemplate(vapId, beaconHandlerVapDb->beaconBuffers);
	/*Free any pending template that was never set*/
	beaconHandlerReleasePdList(vapId);
	beaconHandlerBssLoadIeSetIsSupported(vapId, beaconTemplateParams_p->addBssLoadIe);
	beaconHandlerUpdateIEsPdList(vapId, beaconHandlerVapDb->beaconBuffers);
	// Send confirmation to driver. Once template is updated an indication will also be sent.
	// Set the status.
	beaconTemplateParams_p->Status = UMI_OK;
	OSAL_SEND_MESSAGE(UMI_SET_BEACON_TEMPLATE_CFM, TASK_UM_IF_TASK, pBeaconSetTemplateReq->psMsg, vapId);
	if (beaconHandlerVapdb[vapId].isBeaconQueueActive)
	{
		// VAP is active we need to lock the beacon queue before swaping the templates
		ILOG2_D("beaconHandlerUpdateIEsPdList Vap Id:%d is active so calling Locker", vapId);
		requesterLockParams.callerContext = NULL;
		requesterLockParams.returnMsg = PAC_MANAGER_BEACON_QUEUE_IS_LOCKED;
		requesterLockParams.returnTask = TASK_PAC_MANAGER;
		Locker_LockSingleQueue(HW_TX_Q_TYPE_BEACON,vapId,0,&requesterLockParams);
	}
	else // is inactive 
	{
		ILOG2_D("beaconHandlerUpdateIEsPdList Vap Id:%d is inactive sending template was set indication", vapId);
		// VAP is inactive -  Send confirm indication to driver, PDs will be queued on activate vap req 
		beaconHandlerSendTemplateWasSetIndication(vapId);
	}
}

/**********************************************************************************

BeaconHandler_BeaconActivate 

Description:
------------
	Handles the activate beacon message sent from the UMAC
	
Input:
-----
	K Msg containing the beacon parameters to configure
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BeaconHandler_AddVap(UMI_SET_BSS* setBssStructurePtr)
{	
	uint8						vapId = setBssStructurePtr->vapId;
	BeaconHwParams_t 			beaconHWparams;
	TxSelectorActionParams_t 	txSelectorActionParams;
	
	// Req 2: 4.2.2.2 - avoid adding VSTA to Beacon CSA manager */
	if (beaconHandlerVapdb[vapId].operationMode == OPERATION_MODE_NORMAL)
	{
		/* Set the intervals only once in a VAP's life cycle */
		if (beaconHandlerVapdb[vapId].isDtimAndBeaconIntervalsSet == FALSE)
		{
			beaconHandlerVapdb[vapId].isDtimAndBeaconIntervalsSet = TRUE;

			memset(&txSelectorActionParams, 0, sizeof(TxSelectorActionParams_t));
			ILOG2_D("BeaconHandler_AddVap for vap:%d", vapId);
			/* First clear the request - it might be set if the VAP was active before */
			txSelectorActionParams.queueType = HW_TX_Q_TYPE_BEACON;
			txSelectorActionParams.stationOrVapNum = vapId;
			txSelectorActionParams.action = TX_SELECTOR_RESET_POWER_SAVE_REQUEST;
			TxSelector_SetPowerSaveRequest(&txSelectorActionParams);
			// Configure Beacon HW
			beaconHWparams.dtimInterval = setBssStructurePtr->dtimInterval;
			beaconHWparams.beaconInterval = setBssStructurePtr->beaconInterval;
			beaconHWparams.beaconLCM = setBssStructurePtr->beaconInterval;
			beaconHWparams.beaconOffset = ((setBssStructurePtr->beaconInterval)/ HW_NUM_OF_VAPS)*vapId;
			beaconHandlerActivateHw(vapId, &(beaconHWparams));
			BeaconCsaManagerAddVap(vapId, setBssStructurePtr->beaconInterval);
			beaconHandlerVapdb[vapId].isVht = FALSE;
		}
	}
}

/**********************************************************************************

BeaconHandler_EnableBeaconQueue

Description:
------------
	Handles the SET WMM MSg from UM - Enable beacon queue.
	
Input:
-----
	K Msg containing the beacon parameters to configure
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BeaconHandler_SetWmmParameters(uint8 vapId)
{
	BeaconHandlerVapDb_t *beaconHandlerVapDbEntry = &beaconHandlerVapdb[vapId];
	
	if(beaconHandlerVapdb[vapId].isBeaconQueueActive != TRUE)
	{
		beaconHandlerVapDbEntry->isBeaconQueueActive = TRUE; 
		// Enable Beacon Queue 
		Locker_EnableStaQueues(HW_TX_Q_TYPE_BEACON,vapId);
		if (beaconHandlerVapDbEntry->pPendingTemplateHead != NULL)
		{
			// push new template PDs to Beacon Hw queue
			BeaconHwQueueManagerRequestParams.pHeadDesc = beaconHandlerVapDbEntry->pPendingTemplateHead;
			BeaconHwQueueManagerRequestParams.pTailDesc = beaconHandlerVapDbEntry->pPendingTemplateTail;
			BeaconHwQueueManagerRequestParams.primaryAddr = vapId;
			HwQManager_PushPacketlistToHead(&BeaconHwQueueManagerRequestParams);	
			beaconHandlerVapDbEntry->pPendingTemplateHead = NULL;
			beaconHandlerVapDbEntry->pPendingTemplateTail = NULL;
			/*Add Beacon to Beacon Blocking Detection mechanism*/
			BeaconBlockingManagerAddBeacon(vapId);
		}
	}
}

/**********************************************************************************

BeaconHandler_BeaconDeactivate 

Description:
------------
	Handles the deactivate beacon message sent from the UMAC
	
Input:
-----
	K Message containing the VAP id we want to deactivate the beacon.
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BeaconHandler_BeaconDeactivate(K_MSG *psMsg)
{
	uint8 vapId; 
	K_MSG *RemoveVapMsg;
	UMI_REMOVE_VAP* removeVapStructurePtr;
	RequesterLockParams_t requesterLockParams;
	//K_MSG *psCnfMsg = OSAL_GET_MESSAGE(sizeof(void*));
	
	RemoveVapMsg = (K_MSG *) (*((uint32 *)pK_MSG_DATA(psMsg)));
	removeVapStructurePtr = (UMI_REMOVE_VAP *)pK_MSG_DATA(RemoveVapMsg);
	//memcpy(pK_MSG_DATA(psCnfMsg), &RemoveVapMsg, sizeof(void*));
	vapId = removeVapStructurePtr->vapId;
	/* VAP is removed: clear the indication that DTIM and beacon interval are set so that the first SET_BSS after ADD_VAP will activate beacon HW  */
	beaconHandlerVapdb[vapId].isDtimAndBeaconIntervalsSet = FALSE;	
	if (beaconHandlerVapdb[vapId].isBeaconQueueActive == TRUE)
	{
		ILOG2_D("beaconHandlerDeactivateBeacon for vap:%d", vapId);
		requesterLockParams.callerContext = NULL;
		requesterLockParams.returnMsg = PAC_MANAGER_BEACON_QUEUE_IS_DISABLED;
		requesterLockParams.returnTask = TASK_PAC_MANAGER;
		Locker_DisableStaQueues(HW_TX_Q_TYPE_BEACON,vapId,&requesterLockParams);
	}
	else
	{
		/*Free any pending template that was never set*/
		beaconHandlerReleasePdList(vapId);
		/*Deactivate the beacon HW */ 
		beaconHandlerDeactivateHw(vapId);
		/*Now send confirmation to BSS manager*/
		PacManagerSendCfmToVapManagerForRemoveVap(vapId);		
	}
	BeaconCsaManagerRemoveVap(vapId);
}

/**********************************************************************************

BeaconHandler_QueueIsLocked 

Description:
------------
	
	
Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BeaconHandler_QueueIsLocked(K_MSG *beaconHandlerMessage)
{
	LockReqCb_t* lockReqCb;
	uint8 vapIndex = 0;
	
	lockReqCb = (LockReqCb_t *)beaconHandlerMessage->abData;
	/*Extract VAP Index*/
	vapIndex = lockReqCb->stationOrVapNum;
	/*Swap Templates*/
	beaconHandlerSwapIEsPdList(vapIndex);
	/*Send template was set indication to host*/
    beaconHandlerSendTemplateWasSetIndication(vapIndex);
}

/**********************************************************************************

BeaconHandler_QueueIsDisabled 

Description:
------------
	
	
Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BeaconHandler_QueueIsDisabled(K_MSG *beaconHandlerMessage)
{
	LockReqCb_t* lockReqCb;
	uint8 vapIndex = 0;
	
	lockReqCb = ((LockReqCb_t *)beaconHandlerMessage->abData);
	vapIndex = lockReqCb->stationOrVapNum;
	beaconHandlerFlushIEsPdList(vapIndex);
	/*Deactivate the beacon HW */ 
	beaconHandlerDeactivateHw(vapIndex);
	beaconHandlerVapdb[vapIndex].isBeaconQueueActive = FALSE;
	/*Now send confirmation to BSS manager*/
	PacManagerSendCfmToVapManagerForRemoveVap(vapIndex);	
}

/**********************************************************************************

BeaconHandler_BeaconCsaStart 

Description:
------------
	
	
Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BeaconHandler_BeaconCsaStart(K_MSG *beaconHandlerMessage)
{
	BeaconCsaManagerStartReq(beaconHandlerMessage);
}

/**********************************************************************************

BeaconHandler_BeaconCsaStop 

Description:
------------
	
	
Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BeaconHandler_BeaconCsaStop(K_MSG *beaconHandlerMessage)
{
	BeaconCsaManagerStopReq(beaconHandlerMessage);
}

/**********************************************************************************

BeaconHandler_BeaconCsaLock 

Description:
------------
	
	
Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BeaconHandler_BeaconCsaLock(K_MSG *beaconHandlerMessage)
{
	BeaconCsaManagerLockCfm(beaconHandlerMessage);
}

/**********************************************************************************

BeaconHandler_BeaconPd 

Description:
------------
	
	
Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BeaconHandler_BeaconCsaPd(K_MSG *beaconHandlerMessage)
{
	BeaconCsaManagerPdCfm(beaconHandlerMessage);
}


/**********************************************************************************

BeaconHandler_SetCsaOverwrite 

Description:
------------
	Called in CSA start to overwrite a pending template
	
Input:
-----
 		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BeaconHandler_SetCsaOverwrite(uint8 vapId)
{
	BeaconHandlerVapDb_t *beaconHandlerVapDbEntry = &beaconHandlerVapdb[vapId];

	if (beaconHandlerVapDbEntry->pPendingTemplateHead != NULL)
	{
		/*Release pending PDs*/
		beaconHandlerReleasePdList(vapId);
		/*If we sent a lock request, Set CSA override so we ignore lock confirmation*/
		if (beaconHandlerVapDbEntry->isBeaconQueueActive == TRUE)
		{
			beaconHandlerVapDbEntry->csaOverwrite = TRUE;
		}
	}
}


/**********************************************************************************

beaconHandlerBssLoadIeSetIsSupported 

Description:
------------
	maintain if bssLoad Ie is supported per vap
	
Input:
-----
	uint8 vapId, bool isBssLoadIe

************************************************************************************/
void beaconHandlerBssLoadIeSetIsSupported(uint8 vapId, bool isBssLoadIe)
{
	if (isBssLoadIe == TRUE)
	{
		BeaconManagement.isBssLoadIe |= (1<<vapId);
	}
	else
	{
		BeaconManagement.isBssLoadIe &= ~(1<<vapId);
	}
}

/**********************************************************************************

beaconHandlerBssLoadIeIsSupported 

Description:
------------
	returns if bssLoad Ie is supported for this vap
	
Input:
-----
	uint8 vapId
Returns:
--------
	boolean, TRUE if supported 
************************************************************************************/
bool beaconHandlerBssLoadIeIsSupported(uint8 vapId)
{
	return ((BeaconManagement.isBssLoadIe &= (1<<vapId)) != 0);
}

/**********************************************************************************

beaconHandlerBssLoadIeBuild 

Description:
------------
	builds bss load IE pd
	
Input:
-----
	TxPd_t *pd

************************************************************************************/
void beaconHandlerBssLoadIeBuild(TxPd_t *pd)
{
	DEBUG_ASSERT((BeaconManagement.isBssLoadIe & (1<<pd->txQVapId)) != 0);

	/*Set the packet pointer to the Bss Load IE payload for this VAP*/
	pd->packetPointer = CONVERT_WLAN_SHRAM_ADDR_TO_DMA_SHRAM_ADDR((uint32)&bssLoadIePayload[pd->txQVapId]);
	// payload is constantly updated
	pd->dataLength = BSS_LOAD_IE_ELEMENT_SIZE;
}


/**********************************************************************************

beaconHandlerBssLoadIeInit 

Description:
------------
	init bss load IE in shram
	
Input:
-----
	None

************************************************************************************/
#if defined (ENET_INC_LMAC)
#pragma ghs section text=".initialization_start" 
#endif
void beaconHandlerBssLoadIeInit(void)
{
	uint8 vapId = 0;

	/*Init Beacon Hs20 Bss Load IE in shram*/
	for (vapId = 0; vapId < HW_NUM_OF_VAPS; vapId++)
	{
		// header
		bssLoadIePayload[vapId].ie.u8Code = FM_EL_BSS_LOAD;
		bssLoadIePayload[vapId].ie.u8Length = BSS_LOAD_IE_PAYLOAD_SIZE;
		// payload
		memset (&bssLoadIePayload[vapId].bssLoad, 0, sizeof(BSS_LOAD_ELEMENT));
	}
}
#if defined (ENET_INC_LMAC)
#pragma ghs section text=default
#endif
/**********************************************************************************

BeaconHandler_Init 

Description:
------------
	Initializes the Beacon Handler pool (linked list of packet descriptors) and the Beacon handler 
	management structure.
	
Input:
-----
	void

************************************************************************************/
#if defined (ENET_INC_LMAC)
#pragma ghs section text=".initialization_start" 
#endif
void BeaconHandler_Init(void)
{

	//System_MainAllocInitializationMemory((uint8 **)&beaconHandlerVapdb, (sizeof(BeaconHandlerVapDb_t)*HW_NUM_OF_VAPS));
	
	Pool_InitializePdPool(&(BeaconManagement.pdPool),
				   PdBeaconPoolArray,
				   BEACON_NUM_OF_PDS_IN_POOL,
				   PD_TYPE_BEACON);

	// Initialize Hw Queue input parameter
	memset(&BeaconHwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));
	memset(beaconHandlerVapdb,0, sizeof(BeaconHandlerVapDb_t)*HW_NUM_OF_VAPS);
	
	BeaconHwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
	BeaconHwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;		
	BeaconHwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_BEACON;

	BeaconCsaManager_Init();
	
	beaconHandlerBssLoadIeInit();

	BeaconBlockingManager_Init();
}
#if defined (ENET_INC_LMAC)
#pragma ghs section text=".default" 
#endif
