/***********************************************************************************
 File:			Beacon_CsaManager.c
 Module:			beacon CSA Manager
 Purpose: 		handle Beacon CSA Manager events
 Description:   
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_GlobalDefinitions.h"
#include "System_Configuration.h"
#include "System_MainApi.h"
#include "shram_man_msgs.h"
#include "shram_man_queues.h"
#include "stringLibApi.h"
#include "PacketDescriptor.h"
#include "TxSelector_Api.h"
#include "RegAccess_Api.h"
#include "ErrorHandler_Api.h"
#include "HwQManager_API.h"
#include "int_gen.h"
#include "lm.h"
#include "ShramPacketDescriptors.h"
#include "BeaconRegs.h"
#include "PacketDescriptorsPool_api.h"
#include "shramTxDesc.h"
#include "HwMemoryMap.h"
#include "Pac_Api.h"
#include "Locker_Api.h"
#include "BeaconHandler.h"
#include "BeaconHandler_api.h"
#include "PacManager_api.h"
#include "Utils_Api.h"
#include "loggerAPI.h"


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_BEACON_HANDLER 
#define LOG_LOCAL_FID 1

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
/*Beacon CSA Manager events*/
typedef enum
{
	BEACON_CSA_MANAGER_START_REQ,
	BEACON_CSA_MANAGER_STOP_REQ,
	BEACON_CSA_MANAGER_LOCK_CFM,
	BEACON_CSA_MANAGER_PD_CFM,
	BEACON_CSA_MANAGER_NUM_EVENTS
} beaconCsaManager_ev;
	
/*Beacon CSA Manager states*/
typedef enum
{
	BEACON_CSA_MANAGER_IDLE_STATE,
	BEACON_CSA_MANAGER_START_STATE,
	BEACON_CSA_MANAGER_STOP_STATE,
	BEACON_CSA_MANAGER_NUM_STATES
} beaconCsaManager_state;

typedef struct BeaconCsaManagerDb_s
{
	uint16					activeVaps;		/*Bitmap of active VAPs*/
	uint16					pendingVaps;	/*Bitmap of pending VAPs*/
	uint16					pendingIeVaps;	/*Bitmap of VAPs that the CSA IE was not added yet*/
	BeaconCsaStartMsg_t		msg;
	beaconCsaManager_state	state;
} BeaconCsaManagerDb_t;

typedef void (*BeaconCsaManager_Func)(void *parameter);

typedef struct BeaconCsaManagerTemplateDb_s
{
	BeaconHostBuffer_t  beaconBuffers[BEACON_MAX_HOST_BUFFERS_NUM];
} BeaconCsaManagerTemplateDb_t;

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void BeaconCsaManager_ChangeState(beaconCsaManager_state state);
static void BeaconCsaManager_SetCount(uint8 vapId, uint8 count);
static void BeaconCsaManager_StartCounting(void);
static void BeaconCsaManager_StartInIdle(void *parameter);
static void BeaconCsaManager_StopInIdle(void *parameter);
static void BeaconCsaManager_StopInStart(void *parameter);
static void BeaconCsaManager_LockInStart(void *parameter);
static void BeaconCsaManager_LockInStop(void *parameter);
static void BeaconCsaManager_PdInStart(void *parameter);
static void BeaconCsaManager_PdInStop(void *parameter);
static void BeaconCsaManager_Fatal(void *parameter);

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
/*Beacon CSA Manager Start Event Handlers*/
static const BeaconCsaManager_Func BeaconCsaManager_Start[BEACON_CSA_MANAGER_NUM_STATES] =
{
	 BeaconCsaManager_StartInIdle, 				/* BEACON_CSA_MANAGER_IDLE_STATE */
	 BeaconCsaManager_Fatal,			  		/* BEACON_CSA_MANAGER_START_STATE */
 	 BeaconCsaManager_Fatal				  		/* BEACON_CSA_MANAGER_STOP_STATE */
};

/*Beacon CSA Manager Stop Event Handlers*/
static const BeaconCsaManager_Func BeaconCsaManager_Stop[BEACON_CSA_MANAGER_NUM_STATES] =
{
	 BeaconCsaManager_StopInIdle, 				/* BEACON_CSA_MANAGER_IDLE_STATE */
	 BeaconCsaManager_StopInStart,		  		/* BEACON_CSA_MANAGER_START_STATE */
 	 BeaconCsaManager_Fatal				  		/* BEACON_CSA_MANAGER_STOP_STATE */
};

/*Beacon CSA Manager Lock CFM Event Handlers*/
static const BeaconCsaManager_Func BeaconCsaManager_Lock[BEACON_CSA_MANAGER_NUM_STATES] =
{
	 BeaconCsaManager_Fatal, 					/* BEACON_CSA_MANAGER_IDLE_STATE */
	 BeaconCsaManager_LockInStart,		  		/* BEACON_CSA_MANAGER_START_STATE */
 	 BeaconCsaManager_LockInStop		  		/* BEACON_CSA_MANAGER_STOP_STATE */
};

/*Beacon CSA Manager PD CFM Event Handlers*/
static const BeaconCsaManager_Func BeaconCsaManager_Pd[BEACON_CSA_MANAGER_NUM_STATES] =
{
	 BeaconCsaManager_Fatal, 					/* BEACON_CSA_MANAGER_IDLE_STATE */
	 BeaconCsaManager_PdInStart,		  		/* BEACON_CSA_MANAGER_START_STATE */
 	 BeaconCsaManager_PdInStop		  			/* BEACON_CSA_MANAGER_STOP_STATE */
};

static BeaconCsaManagerDb_t BeaconCsaManager_Db;
static BeaconCsaManagerTemplateDb_t *BeaconCsaManager_TemplateDb;
static uint16 BeaconCsaManager_BeaconInterval[HW_NUM_OF_VAPS];

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

BeaconCsaManager_ChangeState 


Description:
------------
Utility function to change State


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void BeaconCsaManager_ChangeState(beaconCsaManager_state state)
{
 	ILOG0_DD("Beacon CSA Manager changed from %d to %d", BeaconCsaManager_Db.state, state);
	BeaconCsaManager_Db.state = state;
}

/**********************************************************************************

BeaconCsaManager_SetCount 


Description:
------------
Utility function to start/stop the Beacon CSA count


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void BeaconCsaManager_SetCount(uint8 vapdId, uint8 count)
{
	RegBeaconVap0BeaconCsa_u beaconCsaReg;
	uint32 csaRegAddress = 0;
	
	beaconCsaReg.val = 0;
	if (count)
	{
		beaconCsaReg.bitFields.vap0CsaCountLimit = count;
		beaconCsaReg.bitFields.vap0CsaEn = 1;
	}
	// get address of VAP Beacon configuration registers
	csaRegAddress = REG_BEACON_VAP0_BEACON_CSA + (vapdId * REG_BEACON_HANDLER_VAP_CONFIG_SIZE);
	RegAccess_Write(csaRegAddress, beaconCsaReg.val);
}

/**********************************************************************************

BeaconCsaManager_StartCounting 


Description:
------------
Start CSA counters for all enabled VAPs


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void BeaconCsaManager_StartCounting(void)
{
	RegBeaconBeaconCsaGo_u beaconCsaGoReg;

	beaconCsaGoReg.val = 0;
	beaconCsaGoReg.bitFields.csaBeaconCountGo = 1;
	RegAccess_Write(REG_BEACON_BEACON_CSA_GO, beaconCsaGoReg.val);
}



/**********************************************************************************

BeaconCsaManager_PdHandler 


Description:
------------
Utility function that handles Beacon PD indication


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void BeaconCsaManager_PdHandler(uint8 vapIndex)
{
	/*Flush Beacon queue*/
	beaconHandlerFlushIEsPdList(vapIndex);
	/*Stop beacon CSA count*/
	BeaconCsaManager_SetCount(vapIndex, 0);
	/*unlock it*/
	Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_BEACON,vapIndex,(0x1 << 0));
}

/**********************************************************************************

CsaManager_StartTxInIdle 


Description:
------------
Handle Start TX REQ In Idle State


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void BeaconCsaManager_StartInIdle(void *parameter)
{
	uint8 vapId = 0;
	uint16 vapBitmap = BeaconCsaManager_Db.activeVaps;
	RequesterLockParams_t lockRequesterParams;
	BeaconCsaStartMsg_t *beaconCsaStartMsg = (BeaconCsaStartMsg_t *)pK_MSG_DATA((K_MSG *)parameter);


	if(vapBitmap == 0)
	{
		OSAL_SEND_NO_DATA_MESSAGE(CHANNEL_SWITCH_MANAGER_BEACON_START_CSA_BEACON_CFM, TASK_CHANNEL_SWITCH_MANAGER, GET_DEFAULT_VAP_FOR_MY_BAND());
		return;
	}
	/*Copy the message parameters*/
	memcpy(&BeaconCsaManager_Db.msg, beaconCsaStartMsg, sizeof(BeaconCsaStartMsg_t));
	/*Change state - do it here because beacon handler calls Beacon CSA to detect if channel switch is required*/
	BeaconCsaManager_ChangeState(BEACON_CSA_MANAGER_START_STATE);
	/*Set Pending VAPs to active VAPs*/
	BeaconCsaManager_Db.pendingIeVaps = BeaconCsaManager_Db.activeVaps;
	BeaconCsaManager_Db.pendingVaps = BeaconCsaManager_Db.activeVaps;
	/*Send Lock Requests for Active VAPs*/
	memset(&lockRequesterParams, 0, sizeof(RequesterLockParams_t));
	lockRequesterParams.callerContext = NULL;
	lockRequesterParams.returnMsg = PAC_MANAGER_BEACON_CSA_LOCK;
	lockRequesterParams.returnTask = TASK_PAC_MANAGER;
	vapId = Utils_FindFirstSet(vapBitmap);
	/*Need to add CSA IE*/
	/*There could be a tenplate update pending*/
	BeaconHandler_SetCsaOverwrite(vapId);
	/*Update the template - will be swapped on Lock CFM, use stored template*/
	beaconHandlerUpdateIEsPdList(vapId, BeaconCsaManager_TemplateDb[vapId].beaconBuffers);
	/*Send Lock Request for Beacon queue*/
	Locker_LockSingleQueue(HW_TX_Q_TYPE_BEACON, vapId, 0, &lockRequesterParams);
}

/**********************************************************************************

BeaconCsaManager_StopInIdle 


Description:
------------
Handle Stop TX REQ In Idle State


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void BeaconCsaManager_StopInIdle(void *parameter)
{
	/*Just notify Channel Switch Manager that we are done*/
	OSAL_SEND_MESSAGE(CHANNEL_SWITCH_MANAGER_STOP_CSA_BEACONS_CFM, TASK_CHANNEL_SWITCH_MANAGER, parameter, VAP_ID_DO_NOT_CARE);
}

/**********************************************************************************

BeaconCsaManager_StopInStart 


Description:
------------
Handle Stop TX REQ In Start State


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void BeaconCsaManager_StopInStart(void *parameter)
{
	uint8 vapId = 0;
	uint16 vapBitmap = BeaconCsaManager_Db.pendingVaps;
	RequesterLockParams_t lockRequesterParams;
	
	/*Check that there are no queues waiting for CSA IE*/
	DEBUG_ASSERT(BeaconCsaManager_Db.pendingIeVaps == 0);
	/*Change state to Stop state*/
	BeaconCsaManager_ChangeState(BEACON_CSA_MANAGER_STOP_STATE);
	/*Set the pending IE to the pending VAPs - this is to prevent flushing the queue twice 0 if we get the PD after this*/
	BeaconCsaManager_Db.pendingIeVaps = BeaconCsaManager_Db.pendingVaps;
	/*Send Lock Requests for Pending VAPs*/
	memset(&lockRequesterParams, 0, sizeof(RequesterLockParams_t));
	lockRequesterParams.callerContext = NULL;
	lockRequesterParams.returnMsg = PAC_MANAGER_BEACON_CSA_LOCK;
	lockRequesterParams.returnTask = TASK_PAC_MANAGER;
	vapId = Utils_FindFirstSet(vapBitmap);
	/*Send Lock Request for Beacon queue*/
	Locker_LockSingleQueue(HW_TX_Q_TYPE_BEACON, vapId, 0, &lockRequesterParams);
}

/**********************************************************************************

BeaconCsaManager_LockInStart 


Description:
------------
Handle Lock CFM In Start State


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void BeaconCsaManager_LockInStart(void *parameter)
{
	LockReqCb_t* lockReqCb = (LockReqCb_t *)pK_MSG_DATA((K_MSG *)parameter);
	uint8 vapIndex = 0;
	uint8 channelCount = 0;
	RequesterLockParams_t lockRequesterParams;
	
	/*Extract VAP index from Lock CFM message*/	
	vapIndex = lockReqCb->stationOrVapNum;
	/*remove VAP from pending CSA IE*/
	BeaconCsaManager_Db.pendingIeVaps &= ~(1 << vapIndex);
	/*Channel Count is given as number of TBTT * Beacon Interval*/
	channelCount = BeaconCsaManager_Db.msg.time/BeaconCsaManager_BeaconInterval[vapIndex];
	/*Start Beacon CSA count*/
	BeaconCsaManager_SetCount(vapIndex, channelCount);
	/*Swap PD lists - this flushes the current queue and unlocks it*/
	beaconHandlerSwapIEsPdList(vapIndex);
	/*If all VAPs have been set, issue a "go" command to the CSA counter in HW and notify Channel Switch Manager*/
	if (BeaconCsaManager_Db.pendingIeVaps == 0)
	{
		// Start CSA counters for all VAPs with CSA enable.
		BeaconCsaManager_StartCounting();	
		OSAL_SEND_NO_DATA_MESSAGE(CHANNEL_SWITCH_MANAGER_BEACON_START_CSA_BEACON_CFM, TASK_CHANNEL_SWITCH_MANAGER, vapIndex);
	}
	else
	{
		/*If not all VAPs have been set lock the next one*/
		memset(&lockRequesterParams, 0, sizeof(RequesterLockParams_t));
		lockRequesterParams.callerContext = NULL;
		lockRequesterParams.returnMsg = PAC_MANAGER_BEACON_CSA_LOCK;
		lockRequesterParams.returnTask = TASK_PAC_MANAGER;
		vapIndex = Utils_FindFirstSet(BeaconCsaManager_Db.pendingIeVaps);
		/*There could be a tenplate update pending*/
		BeaconHandler_SetCsaOverwrite(vapIndex);
		/*Update the template - will be swapped on Lock CFM, use stored template*/
		beaconHandlerUpdateIEsPdList(vapIndex, BeaconCsaManager_TemplateDb[vapIndex].beaconBuffers);
		/*Send Lock Request for Beacon queue*/
		Locker_LockSingleQueue(HW_TX_Q_TYPE_BEACON, vapIndex, 0, &lockRequesterParams);
	}
}

/**********************************************************************************

BeaconCsaManager_LockInStop 


Description:
------------
Handle Lock CFM In Stop State


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void BeaconCsaManager_LockInStop(void *parameter)
{
	LockReqCb_t* lockReqCb = (LockReqCb_t *)pK_MSG_DATA((K_MSG *)parameter);
	uint8 vapIndex = 0;
	RequesterLockParams_t lockRequesterParams;

	/*Extract VAP index from Lock CFM message*/	
	vapIndex = lockReqCb->stationOrVapNum;
	/*Check we didn't already flush the queue because we got the PD after we moved to Stop State*/
	if (BeaconCsaManager_Db.pendingIeVaps & (1 << vapIndex))
	{
		/*Flush Beacon queue*/
		beaconHandlerFlushIEsPdList(vapIndex);
		/*Stop beacon CSA count*/
		BeaconCsaManager_SetCount(vapIndex, 0);
	}
	/*unlock it*/
	Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_BEACON,vapIndex,(0x1 << 0));
	/*Clear VAP from pending VAPs bitmap*/
	BeaconCsaManager_Db.pendingVaps &= ~(1 << vapIndex);
	/*Clear VAP from pending IE bitmap*/
	BeaconCsaManager_Db.pendingIeVaps &= ~(1 << vapIndex);
	/*if this was the last VAP move to Idle state*/
	if (BeaconCsaManager_Db.pendingVaps == 0)
	{
		OSAL_SEND_NO_DATA_MESSAGE(CHANNEL_SWITCH_MANAGER_STOP_CSA_BEACONS_CFM, TASK_CHANNEL_SWITCH_MANAGER, vapIndex);
		BeaconCsaManager_ChangeState(BEACON_CSA_MANAGER_IDLE_STATE);
	}
	else
	{
		/*Lock the next VAP*/
		/*Send Lock Requests for Pending VAPs*/
		memset(&lockRequesterParams, 0, sizeof(RequesterLockParams_t));
		lockRequesterParams.callerContext = NULL;
		lockRequesterParams.returnMsg = PAC_MANAGER_BEACON_CSA_LOCK;
		lockRequesterParams.returnTask = TASK_PAC_MANAGER;
		vapIndex = Utils_FindFirstSet(BeaconCsaManager_Db.pendingVaps);
		/*Send Lock Request for Beacon queue*/
		Locker_LockSingleQueue(HW_TX_Q_TYPE_BEACON, vapIndex, 0, &lockRequesterParams);
	}
}

/**********************************************************************************

BeaconCsaManager_PdInStart 


Description:
------------
Handle PD CFM In Start State


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void BeaconCsaManager_PdInStart(void *parameter)
{
	/*get Beacon PD message structure*/
	BeaconCsaPdMsg_t *beaconPdMsg = (BeaconCsaPdMsg_t *)pK_MSG_DATA((K_MSG *)parameter);
	/*Get Beacon packet Descriptor*/
	TxPd_t *pd = (TxPd_t *)beaconPdMsg->pd;
	/*Get VAP index from Packet descriptor*/
	uint8 vapIndex = pd->txQVapId;
	K_MSG *kMsg_p = OSAL_GET_MESSAGE(sizeof(LmBeaconCsaInd_t));
	LmBeaconCsaInd_t *beaconCsaInd_p = (LmBeaconCsaInd_t *)pK_MSG_DATA(kMsg_p);

	/*Handle PD*/
	BeaconCsaManager_PdHandler(vapIndex);
	/*Send Indication to Channel Manager*/
	beaconCsaInd_p->vapId = vapIndex;
	OSAL_SEND_MESSAGE(CHANNEL_SWITCH_MANAGER_CSA_BEACON_DONE_IND, TASK_CHANNEL_SWITCH_MANAGER, kMsg_p, vapIndex);
	
	/*Clear VAP from pending VAPs bitmap*/
	BeaconCsaManager_Db.pendingVaps &= ~(1 << vapIndex);
	/*if this was the last VAP move to Idle state*/
	if (BeaconCsaManager_Db.pendingVaps == 0)
	{
		BeaconCsaManager_ChangeState(BEACON_CSA_MANAGER_IDLE_STATE);
	}

}

/**********************************************************************************

BeaconCsaManager_PdInStop 


Description:
------------
Handle PD CFM In Stop State


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void BeaconCsaManager_PdInStop(void *parameter)
{
	/*get Beacon PD message structure*/
	BeaconCsaPdMsg_t *beaconPdMsg = (BeaconCsaPdMsg_t *)pK_MSG_DATA((K_MSG *)parameter);
	/*Get Beacon packet Descriptor*/
	TxPd_t *pd = (TxPd_t *)beaconPdMsg->pd;
	/*Get VAP index from Packet descriptor*/
	uint8 vapIndex = pd->txQVapId;

	/*Handle PD - Note: the VAP is not cleared here - it will be cleared when the pending lock is serviced*/
	BeaconCsaManager_PdHandler(vapIndex);
	/*Clear the VAP from the pending IE*/
	BeaconCsaManager_Db.pendingIeVaps &= ~(1 << vapIndex);

}


/**********************************************************************************

BeaconCsaManager_Fatal 


Description:
------------
handles unexpected event


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void BeaconCsaManager_Fatal(void *parameter)
{
	FATAL("Beacon CSA Manager Fatal");
}

/*---------------------------------------------------------------------------------
/						global Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

BeaconCsaManagerAddVap 


Description:
------------
VAP Add handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BeaconCsaManagerAddVap(uint8 vapId, uint16 beaconInterval)
{
	/*Since Channel Switch and ADD VAPs are separete process a VAP can be added only in Idle state*/
	DEBUG_ASSERT(BeaconCsaManager_Db.state == BEACON_CSA_MANAGER_IDLE_STATE);
	BeaconCsaManager_Db.activeVaps |= (1 << vapId);
	BeaconCsaManager_BeaconInterval[vapId] = beaconInterval;
}

/**********************************************************************************

BeaconCsaManagerRemoveVap 


Description:
------------
VAP Remove handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BeaconCsaManagerRemoveVap(uint8 vapId)
{
	/*Since Channel Switch and ADD VAPs are separete process a VAP can be added only in Idle state*/
	DEBUG_ASSERT(BeaconCsaManager_Db.state == BEACON_CSA_MANAGER_IDLE_STATE);
	BeaconCsaManager_Db.activeVaps &= ~(1 << vapId);
	/*Clear template*/
	memset(&BeaconCsaManager_TemplateDb[vapId], 0, sizeof(BeaconCsaManagerTemplateDb_t));
}

/**********************************************************************************

BeaconCsaManagerUpdateTemplate 


Description:
------------
Called when template is updated - need to store host buffers locally so we can use exisiting functions to
update the list


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BeaconCsaManagerUpdateTemplate(uint8 vapId, BeaconHostBuffer_t *beaconBuffers)
{
	memcpy(BeaconCsaManager_TemplateDb[vapId].beaconBuffers, beaconBuffers, sizeof(BeaconHostBuffer_t) * BEACON_MAX_HOST_BUFFERS_NUM);
}

/**********************************************************************************

BeaconCsaManagerStartReq 


Description:
------------
Beacon CSA Start REQ from Channel Switch Manager


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BeaconCsaManagerStartReq(K_MSG* beaconCsaManagerMessage)
{
	/*Call handler*/
	BeaconCsaManager_Start[BeaconCsaManager_Db.state]((void *)beaconCsaManagerMessage);
}

/**********************************************************************************

BeaconCsaManagerStopReq 


Description:
------------
CSA Stop TX REQ from Channel Switch Manager


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BeaconCsaManagerStopReq(K_MSG* beaconCsaManagerMessage)
{
	/*Call handler*/
	BeaconCsaManager_Stop[BeaconCsaManager_Db.state]((void *)beaconCsaManagerMessage);
}

/**********************************************************************************

BeaconCsaManagerLockCfm 


Description:
------------
Beacon CSA LOCK CFM


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BeaconCsaManagerLockCfm(K_MSG* beaconCsaManagerMessage)
{
	/*Call handler*/
	BeaconCsaManager_Lock[BeaconCsaManager_Db.state](beaconCsaManagerMessage);
}

/**********************************************************************************

BeaconCsaManagerPdCfm 


Description:
------------
Beacon CSA PD CFM


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BeaconCsaManagerPdCfm(K_MSG* beaconCsaManagerMessage)
{
	BeaconCsaManager_Pd[BeaconCsaManager_Db.state](beaconCsaManagerMessage);
}

/**********************************************************************************

BeaconCsaManagerCsaInProgress






Description:
------------
returns whether 


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
bool BeaconCsaManagerCsaInProgress(void)
{
	bool retVal = TRUE;
	
	if (BeaconCsaManager_Db.state == BEACON_CSA_MANAGER_IDLE_STATE)
	{
		retVal = FALSE;
	}
	return (retVal);
}

/**********************************************************************************

BeaconCsaManagerCsaRequired





Description:
------------
returns whether 


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
bool BeaconCsaManagerCsaRequired(void)
{
	bool retVal = FALSE;
	
	if (BeaconCsaManager_Db.state == BEACON_CSA_MANAGER_START_STATE)
	{
		retVal = TRUE;
	}
	return (retVal);
}

/**********************************************************************************

BeaconCsaManagerCsaWrapperRequired






Description:
------------
returns whether 


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
bool BeaconCsaManagerCsaWrapperRequired()
{
	bool retVal = FALSE;
	
	if ((BeaconCsaManager_Db.state == BEACON_CSA_MANAGER_START_STATE) &&
		(BeaconCsaManager_Db.msg.channelWidth != BANDWIDTH_TWENTY))
	{
		retVal = TRUE;
	}
	return (retVal);
}

/**********************************************************************************

BeaconCsaBuildCsaIe


Description:
------------
Builds the CSA IE


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BeaconCsaManagerBuildCsaIe(TxPd_t *pd)
{
	/*Build CSA frame - use CSA IE payload for this VAP*/
	CSA_IE *csaIe = &CsaIePayload[pd->txQVapId];

	/*Set the packet pointer to the CSA IE payload for this VAP*/
	pd->packetPointer = CONVERT_WLAN_SHRAM_ADDR_TO_DMA_SHRAM_ADDR((uint32)&CsaIePayload[pd->txQVapId]);
	csaIe->ie.u8Code = FM_EL_11H_CHANNEL_SWITCH_ANNOUNCEMENT;
	csaIe->ie.u8Length = sizeof(S11H_CHANNEL_SWITCH_ANNOUNCEMENT_ELEMENT);
	/*Fill in Channel parameters*/
	csaIe->csa.ChannelSwitchMode = BeaconCsaManager_Db.msg.blockTx;
	csaIe->csa.NewChannelNumber = BeaconCsaManager_Db.msg.primaryChannel;
	/*Clear the count - Sender ORs the count*/
	csaIe->csa.ChannelSwitchCount = 0;
	pd->dataLength = sizeof(CSA_IE);
}

/**********************************************************************************

BeaconCsaManagerBuildCsaWrapperIe



Description:
------------
Builds the CSA Wrapper IE


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void BeaconCsaManagerBuildCsaWrapperIe(TxPd_t *pd)
{
	/*Build CSA frame - use CSA IE payload for this VAP*/
	CSA_WRAPPER_IE *csaWrapperIe = &CsaWrapperIePayload[pd->txQVapId];

	/*Set the packet pointer to the CSA WRAPPER IE payload for this VAP*/
	pd->packetPointer = CONVERT_WLAN_SHRAM_ADDR_TO_DMA_SHRAM_ADDR((uint32)&CsaWrapperIePayload[pd->txQVapId]);
	csaWrapperIe->ie.u8Code = FM_EL_CSA_WRAPPER;
	csaWrapperIe->ie.u8Length = sizeof(WIDE_BW_IE);
	/*Fill in Channel parameters*/
	csaWrapperIe->wideBwIe.ie.u8Code = FM_EL_WIDE_BW_CH_SWITCH;
	csaWrapperIe->wideBwIe.ie.u8Length = sizeof(sHT_WIDE_BW_CHANNEL_SWITCH);
	csaWrapperIe->wideBwIe.wideBw.channelWidth = BeaconCsaManager_Db.msg.csaChannelWidth;
	csaWrapperIe->wideBwIe.wideBw.centerFreq0Idx = BeaconCsaManager_Db.msg.center0FreqIdx;
	csaWrapperIe->wideBwIe.wideBw.centerFreq1Idx = 0;
	pd->dataLength = sizeof(CSA_WRAPPER_IE);
}

/**********************************************************************************

BeaconCsaManager_Init



Description:
------------
Initiailize Beacon CSA MAnager


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
#if defined (ENET_INC_LMAC)
#pragma ghs section text=".initialization_start" 
#endif
void BeaconCsaManager_Init(void)
{
	uint8 vapId;
	
	/*Clear Database*/
	BeaconCsaManager_Db.activeVaps = 0;
	BeaconCsaManager_Db.pendingVaps = 0;
	BeaconCsaManager_Db.pendingIeVaps = 0;
	BeaconCsaManager_Db.state = BEACON_CSA_MANAGER_IDLE_STATE;
	/*Init Beacon Interval*/
	for (vapId = 0; vapId < HW_NUM_OF_VAPS; vapId++)
	{
		BeaconCsaManager_BeaconInterval[vapId] = 100;
	}
	/*Allocate memory for beacon template*/
	System_MainAllocInitializationMemory((uint8 **)&BeaconCsaManager_TemplateDb, (sizeof(BeaconCsaManagerTemplateDb_t)*HW_NUM_OF_VAPS));
	/*Clear Beacon template*/
	memset(BeaconCsaManager_TemplateDb, 0, sizeof(BeaconCsaManagerTemplateDb_t) * HW_NUM_OF_VAPS);
	/*Clear CSA IE Payloads*/
	memset(CsaIePayload, 0, sizeof(CSA_IE)*HW_NUM_OF_VAPS);
	memset(CsaWrapperIePayload, 0, sizeof(CSA_WRAPPER_IE)*HW_NUM_OF_VAPS);
}
#if defined (ENET_INC_LMAC)
#pragma ghs section text=default
#endif


