/***********************************************************************************
 File:		<RealTimeCounter.c>
 Module:		
 Purpose:	driver for the RTC (real time counter) feature of the ARC		
 Description:	
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes
/----------------------------------------------------------------------------------*/

#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "arc_reg.h"
#include "arc_intrinsics.h"
#include "RealTimeCounter_api.h"

/*---------------------------------------------------------------------------------
/						Defines
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Macros
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Variables
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Global Variables
/----------------------------------------------------------------------------------*/



/*---------------------------------------------------------------------------------
/						Static Functions Definitions
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/	

/**********************************************************************************

GetRealTimeCounterValue()

Description:
------------
Read all 64 bit of Real time counter. Read over and over until it is atomic (state machine is in arc book)
Each tick = 1 cycle (1/CPU_freq [sec])

Input:
-----

Output:
-------

Returns:
--------

**********************************************************************************/
void GetRealTimeCounterValue(uint32 *RtcLowValue, uint32 *RtcHighValue)
{
	RegRtcLow_u 	regRtcLow;
	RegRtcHigh_u 	regRtcHigh;
	volatile RegRtcControl_u regRtcControl;


	// Read the low/high values until it's atomic (low can wrap around before reading high, interrupt can occur, etc...)/
	do
	{
		regRtcLow.val = _lr(AUX_RTC_LOW);
		regRtcHigh.val = _lr(AUX_RTC_HIGH);
		regRtcControl.val = _lr(AUX_RTC_CTRL);
	} while (regRtcControl.bitFields.atomic1 == 0);

	*RtcLowValue 	= regRtcLow.bitFields.rtcLowValue;
	*RtcHighValue 	= regRtcHigh.bitFields.rtcHighValue;
}


/**********************************************************************************

GetRealTimeCounterLow()

Description:
------------
Get the LOW 32 bits of the Real Time Counter (each tick = 1 cycle)
WARNING - the low value wrap around every few seconds. Consider using "" to get both atomic.

Input:
-----

Output:
-------

Returns:
--------

**********************************************************************************/
uint32 GetRealTimeCounterLow(void)
{
	RegRtcLow_u regRtcLow;

	// Read the 32 LSB of RTC
	regRtcLow.val = _lr(AUX_RTC_LOW);

	return (regRtcLow.bitFields.rtcLowValue);
}


/**********************************************************************************

GetRealTimeCounterHigh()

Description:
------------
Get the HIGH 32 bits of the Real Time Counter (each tick = 1 cycle)
WARNING - the low value wrap around every few seconds. Consider using "" to get both atomic.

Input:
-----

Output:
-------

Returns:
--------

**********************************************************************************/
uint32 GetRealTimeCounterHigh(void)
{
	RegRtcHigh_u regRtcHigh;

	// Read the 32 LSB of RTC
	regRtcHigh.val = _lr(AUX_RTC_HIGH);

	return (regRtcHigh.bitFields.rtcHighValue);
}


/**********************************************************************************

SmartInit()

Description:
------------
Initialize RTC (real time counter) module - Turn it on and clean variables

Input:
-----

Output:
-------

Returns:
--------

**********************************************************************************/
void RealTimeCounterInit(void)
{
	RegRtcControl_u regRtcControl;

	regRtcControl.val = 0;
	
	// Clear and enable the Real Time Counter
	regRtcControl.bitFields.clear = TRUE;
	regRtcControl.bitFields.enable = TRUE;
	_sr(regRtcControl.val, AUX_RTC_CTRL);

}


