/***********************************************************************************
 File:		<actionPoints.c>
 Module:		
 Purpose:	driver for the Action Points (breakpoints + watchpoints) feature of the ARC		
 Description:	
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes
/----------------------------------------------------------------------------------*/

#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "string.h"
#include "arc_reg.h"
#include "arc_intrinsics.h"
#include "ActionPoints_api.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"
#include "ShramStationDatabase.h"


/*---------------------------------------------------------------------------------
/						Defines 					
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_DEBUG_ENVIRONMENT
#define LOG_LOCAL_FID 0

/*---------------------------------------------------------------------------------
/						Macros
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Variables
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Global Variables
/----------------------------------------------------------------------------------*/

uint8 ActionPointsActive;


/*---------------------------------------------------------------------------------
/						Static Functions Definitions
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/	


/**********************************************************************************
ActionPoints_Disable()

Description:
------------
	Disable an active action point (for which we did "ActionPoints_Set" before)
Input:
-----

Output:
-------

Returns:
--------

**********************************************************************************/
void ActionPoints_Disable(uint32 apNumber)
{
	RegApAc_u acReg;
	uint32 acRegAddress;	

	ASSERT((ActionPointsActive & (0x1 << apNumber)) != 0);

	// Mark this action point as not active.
	ActionPointsActive &= ~(0x1 << apNumber);

	// Disable the action point
	acReg.val = 0;
	acReg.bitFields.tt = AP_TRANSACTION_TYPE_DISABLED; 

	// Get the address of the register
	acRegAddress = AP_AC0 + apNumber*(AP_AC1-AP_AC0);

	// write to the register
	_sr(acReg.val, acRegAddress);
}


/**********************************************************************************
ActionPoints_Set()

Description:
------------
	Configure 1 triplet of registers in order to control 1 ActionPoint out of four.
	using "ispair" or "isquad" can link together 2/4 actionPoints.
Input:
-----

Output:
-------

Returns:
--------

**********************************************************************************/
void ActionPoints_Set(uint32 apNumber, uint32 matchValue, uint32 matchMask, ap_actionTarget_e target, ap_transactionType_e type, ap_mode_e mode, bool isPair, bool isQuad)
{
	RegApAc_u acReg;
	uint32 amvRegAddress;	
	uint32 ammRegAddress;	
	uint32 acRegAddress;	


	ASSERT(apNumber < 4); // We have only 4 action points
	ASSERT(isQuad + isPair < 2); // The valid combinations are 0/1, 1/0, 0/0. 1/1 is not allowed
	ASSERT((ActionPointsActive & (0x1 << apNumber)) == 0);

	// Mark this action point as active.
	ActionPointsActive |= (0x1 << apNumber);

	acReg.val = 0;
	acReg.bitFields.at = target;
	acReg.bitFields.tt = type; 
	acReg.bitFields.m = mode;
	acReg.bitFields.p = isPair;
	acReg.bitFields.aa = TRUE; // Raise exception
	acReg.bitFields.q = isQuad;

	amvRegAddress = AP_AMV0 + apNumber*(AP_AMV1-AP_AMV0);
	ammRegAddress = AP_AMM0 + apNumber*(AP_AMM1-AP_AMM0);
	acRegAddress = AP_AC0 + apNumber*(AP_AC1-AP_AC0);


	_sr(matchValue, amvRegAddress);
	_sr(matchMask, ammRegAddress);
	_sr(acReg.val, acRegAddress);
}


/**********************************************************************************
ActionPoints_Init()

Description:
------------

Input:
-----

Output:
-------

Returns:
--------

**********************************************************************************/
void ActionPoints_Init(void)
{
	// Mark all action points as not active (we have 4)
	ActionPointsActive = 0;
}


