/***********************************************************************************
 File:		<mpu.c>
 Module:		
 Purpose:	driver for the MPU (Memory Protection Unit) feature of the ARC		
 Description: prevent Core to access other Cores area in the unified RAM
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes
/----------------------------------------------------------------------------------*/

#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "arc_reg.h"
#include "arc_intrinsics.h"
#include "Utils_Api.h"
#include "mpu_api.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"


/*---------------------------------------------------------------------------------
/						Defines 					
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_DEBUG_ENVIRONMENT
#define LOG_LOCAL_FID 2


#define MPU_ENABLED 		1
#define RAM_START_ADDRESS	0x0
#define SIZE0_MASK			0x3
#define NUMBER_OF_MPU_REGS	16
#define DONT_CARE			0
#define MPU_REG_NUMBER_0    0
#define MPU_REG_NUMBER_1    1


#define SIZE_1K				9

#define ONE_MEGA			1024
#define SIXTY_FOUR_MEGA_BYTE (ONE_MEGA * 64)
#define SIZE_512K			512
#define ONE_KILO_BYTES		1024
#define THIRTY_TWO_BYTES_CODE					4 // according to "MPU Region Descriptor Permissions Registers, MPU_RDP0 to MPU_RDP15"
#define ONE_HUNDREAD_TWENTY_EIGHT_BYTES_CODE	6 // according to "MPU Region Descriptor Permissions Registers, MPU_RDP0 to MPU_RDP15"


#if __CCAC__
#define WEAK __attribute__((weak))
#else
#define WEAK
#endif

/*---------------------------------------------------------------------------------
/						Macros
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Variables
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Global Variables
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Externals 								
/----------------------------------------------------------------------------------*/

extern WEAK char _fmpu_start_of_ram[], _empu_end_of_ram[], _ecommon_ram[];   /* defined by the linker */

/*---------------------------------------------------------------------------------
/						Static Functions Definitions
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/	



/**********************************************************************************

mpu_configRegs()

Description:
------------
	Configure MPU registers or calculate the number of registers required to "cover" the region
	starting at address1 and ending at address2.

Input:
-----
	address1  - start address of the region
	address2  - end address of the region
	config 	  - if set, the function configures the MPU registers
			    if reset, the function calculates the number of registers required to "cover" the region
	action    - allow or disallow the region
	regNumber - registers pair to be configured (MPU_RDB0-15 and MPU_RDP0-15)

Output:
-------
	

Returns:
--------
	number of registers to be configured

**********************************************************************************/
uint32 mpu_configRegs(uint32 address1, uint32 address2, configType_e config, actionType_e action, uint32 regNumber)
{
	uint32 blockSize;
	uint32 num_of_regs=0;
	uint32 size = address2-address1;
	
	blockSize = mpu_getBiggestAlignedBlockSize(address1);

	while (size>0)
	{
		if(size >= blockSize)
		{
			// Found a block
			if (config == CONFIG)
			{
				mpu_configRegsPair(regNumber,address1,blockSize,action);
				regNumber--;
			}
			num_of_regs++;
			size -= blockSize;
			address1 += blockSize;
			blockSize = mpu_getBiggestAlignedBlockSize(address1);
		}
		else
			blockSize = blockSize>>1;
	}
	
	return num_of_regs;
	
}



/**********************************************************************************

mpu_getBiggestAlignedAddress()

Description:
------------
	Get the address in the region [startAddress:endAddress] aligned to the biggest size.
	For example:
	region [0:70] will return 0 as the biggest aligned address
	region [1:400] will return 256 as the biggest aligned address
	region [64:1024] will return 1024 as the biggest aligned address

Input:
-----
	startAddress
	endAddress 

Output:
-------

Returns:
--------
	alignedAddress
	
**********************************************************************************/

uint32 mpu_getBiggestAlignedAddress(uint32 startAddress, uint32 endAddress)
{
	uint32 candidate = 0;
	int i;
	
	for(i=10;i>=0;i--)
	{
		candidate = (endAddress>>i)<<i;
		if(candidate >= startAddress)
		{
			break;
		}
	}
	return candidate;
}

/**********************************************************************************

mpu_getBiggestAlignedBlockSize()

Description:
------------
	Search the biggest block from the allowed block sizes (see mpu_get_closest_size description for allowed sizes)
	which is aligned to the given address

Input:
-----
	address 

Output:
-------

Returns:
--------
	blockSize	
	returns 1024K (1M) in case of address is 0 or is aligned to more than 1M. 
	
**********************************************************************************/

uint32 mpu_getBiggestAlignedBlockSize(uint32 address)
{
	uint32 i;
	
	for(i=0;i<=10;i++)
	{
		if(address&0x1)
			return 1<<i;
		else
			address = address>>1;
	}	
	return ONE_MEGA;
}

/**********************************************************************************

mpu_get_closest_size()

Description:
------------
	Get the biggest allowed size that wrap the input size.
	Allowed sizes are: 1K,2K,4K,8K,16K,32K,64K,128K,256K,512K and 1024K 

Input:
-----
	size 

Output:
-------

Returns:
--------
	biggest allowed size
	
**********************************************************************************/


uint32 mpu_get_closest_size(uint32 size)
{
	uint32 i;

	for(i=0;i<=10;i++)
	{
		if(size<=(1<<i))
			break;
	}
	return 1<<i;
}

/**********************************************************************************

mpu_get_index_from_size()

Description:
------------
	Return the index of the corresponding size according to 
	Table 3-76 in "DesignWare ARCv2 ISA Programmer's Reference Manual for ARC HS Processors"
	1K-   index=9
	2K-   index=10
	4K-   index=11
	8K-   index=12
	16K-  index=13
	32K-  index=14
	64K-  index=15
	128K- index=16
	256K- index=17
	512K- index=18
	1M-   index=19

Input:
-----
	size 

Output:
-------

Returns:
--------
	index
	
**********************************************************************************/

uint32 mpu_get_index_from_size(uint32 size)
{
	size = size*ONE_KILO_BYTES;
	return (Utils_FindFirstSet(size)-1);
}

/**********************************************************************************

Mpu_configRegsPair()

Description:
------------
	Configure the pair of MPU registers (MPU_RDB and MPU_RDP)

Input:
-----
	regNumber   - registers pair to be configured (MPU_RDB0-15 and MPU_RDP0-15)
	baseAddress - start address of the region
	size   		- size of the region according to MPU Region Descriptor Permission Registers - see DesignWare ARCv2 ISA Programmer's Reference Manual for ARC HS Processors  
	action 		- allow or disallow the region

Output:
-------

Returns:
--------
	
**********************************************************************************/

void mpu_configRegsPair(uint32 regNumber, uint32 baseAddress, uint32 size, actionType_e action)
{
	RegMPU_RDB_u regMPU_RDB;
	RegMPU_RDP_u regMPU_RDP;
	uint32 regAddress;
	
	regAddress = MPU_RDB0 + (2*regNumber);
	
	// Address should be in unit of bytes
	regMPU_RDB.bitFields.baseAddress = (baseAddress*ONE_KILO_BYTES)>>5;
	regMPU_RDB.bitFields.V = 1;

	// write to register MPU_RDB#
	_sr(regMPU_RDB.val,regAddress);

	// convert size to 5 bits size
	size = mpu_get_index_from_size(size);

	regMPU_RDP.bitFields.UE = action;   
	regMPU_RDP.bitFields.UW = action; 
	regMPU_RDP.bitFields.UR = action; 
	regMPU_RDP.bitFields.KE = action; 
	regMPU_RDP.bitFields.KW = action; 
	regMPU_RDP.bitFields.KR = action; 
	regMPU_RDP.bitFields.size1 = size>>2; 		
	regMPU_RDP.bitFields.size0 = (size & SIZE0_MASK);	
	// write to register MPU_RDP#
	_sr(regMPU_RDP.val,regAddress+1); 
	
}


/**********************************************************************************

mpu_BlockNullPointerWithReg0()

Description:
------------
	Block access to the fist 32 bytes in order to get exception when accessing NULL pointer.

Input:
-----

Output:
-------

Returns:
--------
	
**********************************************************************************/
void mpu_BlockNullPointerWithReg0(void)
{
	RegMPU_RDB_u regMPU_RDB;
	RegMPU_RDP_u regMPU_RDP;
	uint32 regAddress;
	uint8 size;
	
	regAddress = MPU_RDB0;
	
	// Address should be in unit of bytes
	regMPU_RDB.bitFields.baseAddress = RAM_START_ADDRESS;
	regMPU_RDB.bitFields.V = 1;

	// write to register MPU_RDB#
	_sr(regMPU_RDB.val,regAddress);

	// Block 128 bytes which represent the bootloader section.
	size = ONE_HUNDREAD_TWENTY_EIGHT_BYTES_CODE;

	regMPU_RDP.bitFields.UE = DISALLOW_REGION;   
	regMPU_RDP.bitFields.UW = DISALLOW_REGION; 
	regMPU_RDP.bitFields.UR = DISALLOW_REGION; 
	regMPU_RDP.bitFields.KE = DISALLOW_REGION; 
	regMPU_RDP.bitFields.KW = DISALLOW_REGION; 
	regMPU_RDP.bitFields.KR = DISALLOW_REGION; 
	regMPU_RDP.bitFields.size1 = size>>2; 		
	regMPU_RDP.bitFields.size0 = (size & SIZE0_MASK);	
	// write to register MPU_RDP#
	_sr(regMPU_RDP.val,regAddress+1); 
	
}



/**********************************************************************************

mpu_checkFreeReg()

Description:
------------
	Check if a pair of MPU register is available and returns the number of the pair.
	If no pair is available return 16

Input:
-----

Output:
-------

Returns:
--------
	Pair number

**********************************************************************************/

uint32 mpu_checkFreeReg(void)
{
	RegMPU_RDB_u regRDB;
	uint32 i;

	for(i=0; i<NUMBER_OF_MPU_REGS; i++)
	{
		regRDB.val = _lr(MPU_RDB0+2*i);
		if (regRDB.bitFields.V != 1)
		{
			return i;
		}
	}
	return NUMBER_OF_MPU_REGS;
}

/**********************************************************************************

Mpu_protectRegion()

Description:
------------
	Avoid a region to be accessed by the ARC Core.	
	Note that the address should be aligned to the size.

Input:
-----
	address 	 	- start address of the region
	size		 	- size of the region
	readAccess	 	- enable/disable read access 
	writeAccess	 	- enable/disable write access
	executeAccess	- enable/disable execute access

Output:
-------

Returns: RetVal_e (SUCCESS/FAILURE)
--------
	SUCCESS: region is now protected
	FAILURE: no more MPU register is available, then the region can't be protected.

**********************************************************************************/

RetVal_e Mpu_protectRegion(uint32 address, uint32 size, actionType_e readAccess, actionType_e writeAccess, actionType_e executeAccess)
{
	RegMPUEnable_u	regMPU_EN;
	RegMPU_RDP_u	regMPU_RDP;
	uint32			regNumber;
	uint32			regAddress;

	regNumber = mpu_checkFreeReg();
	if (regNumber != NUMBER_OF_MPU_REGS)
	{
		// Disable MPU
		regMPU_EN.val = _lr(MPU_ENABLE);
		regMPU_EN.bitFields.EN = 0;
		_sr(regMPU_EN.val,MPU_ENABLE);

		mpu_configRegsPair(regNumber,address,size,DISALLOW_REGION);

		regAddress = MPU_RDB0+(2*regNumber)+1;

		regMPU_RDP.val = _lr(regAddress); 
		if(readAccess == ALLOW_REGION)
		{
			regMPU_RDP.bitFields.UR = 1;
			regMPU_RDP.bitFields.KR = 1;
		}
		if(writeAccess == ALLOW_REGION)
		{
			regMPU_RDP.bitFields.UW = 1;
			regMPU_RDP.bitFields.KW = 1;
		}
		if(executeAccess == ALLOW_REGION)
		{
			regMPU_RDP.bitFields.UE = 1;
			regMPU_RDP.bitFields.KE = 1;		
		}

		// write to register MPU_RDP#
		_sr(regMPU_RDP.val,regAddress); 
		
		// Enable MPU
		regMPU_EN.bitFields.EN = 1;
		_sr(regMPU_EN.val,MPU_ENABLE);

		return RET_VAL_SUCCESS;
	}
	else
		return RET_VAL_FAIL;
}


/**********************************************************************************

MpuInit()

Description:
------------
	Config MPU registers 

Input:
-----

Output:
-------

Returns:
--------

**********************************************************************************/
void MpuInit(void)
{

	RegMPUEnable_u regMPU_EN;


	uint32 startAddress;
	uint32 endAddress;
	uint32 alignedAddress;
	uint32 num_of_regs1=0;
	uint32 num_of_regs2=0;
	uint32 closestSize;
	uint32 commonAreaSize;
	uint32 currentRegNumber=NUMBER_OF_MPU_REGS-1; // There are 16 regs. 2 hard code: 1 reg for all common area + boot loader, and 1 for not allowing address 0. 1 optional for NULL_PD


	// Disable MPU
	regMPU_EN.val = 0;
	_sr(regMPU_EN.val,MPU_ENABLE);

	// Disallow the whole Unified RAM 
	mpu_configRegsPair(currentRegNumber,RAM_START_ADDRESS,ONE_MEGA,DISALLOW_REGION);
	currentRegNumber--;
	mpu_configRegsPair(currentRegNumber,RAM_START_ADDRESS+ONE_MEGA,SIZE_512K,DISALLOW_REGION);
	currentRegNumber--;

	// All the addresses are assured to be 1K aligned so we work in unit of kilo bytes
	startAddress =  ((uint32) _fmpu_start_of_ram)/ONE_KILO_BYTES;
	endAddress   =  ((uint32) _empu_end_of_ram)/ONE_KILO_BYTES;

	// Get the address with the biggest alignment in the region to be covered
	alignedAddress = mpu_getBiggestAlignedAddress(startAddress,endAddress);
	if (startAddress != alignedAddress)
	{ // Handle the region from startAddress to alignedAddress in 2 ways and choose the "cheapest" way in number of register

		// Calculate the required number of registers - 1rst way
		num_of_regs1 = mpu_configRegs(startAddress,alignedAddress,NO_CONFIG,DONT_CARE,DONT_CARE);

		// Calculate the number of registers - 2nd way
		// Get the biggest size fitting the region from startAddress to alignedAddress
		closestSize = mpu_get_closest_size(alignedAddress-startAddress);
		num_of_regs2 = mpu_configRegs((alignedAddress-closestSize),startAddress,NO_CONFIG,DONT_CARE,DONT_CARE) + 1;
		if(num_of_regs2 > num_of_regs1)
		{// 1rst way is better
		
			// Configure the MPU registers
			num_of_regs1 = mpu_configRegs(startAddress,alignedAddress,CONFIG,ALLOW_REGION,currentRegNumber);

		}
		else
		{// 2nd way is better
		
			// Configure the MPU registers
			mpu_configRegsPair(currentRegNumber, (alignedAddress-closestSize), closestSize, ALLOW_REGION);
			currentRegNumber--;
			num_of_regs1 = mpu_configRegs((alignedAddress-closestSize),startAddress,CONFIG,DISALLOW_REGION,currentRegNumber);

		}
		
		currentRegNumber -= num_of_regs1;
		
	}
	if (alignedAddress != endAddress)
	{// Handle the region from alignedAddress to endAddress in 2 ways and choose the "cheapest" way in number of register
	
		// Calculate the required number of registers
		num_of_regs1 = mpu_configRegs(alignedAddress,endAddress,NO_CONFIG,DONT_CARE,DONT_CARE);
		closestSize = mpu_get_closest_size(endAddress-alignedAddress);
		num_of_regs2 = mpu_configRegs(endAddress,(alignedAddress+closestSize),NO_CONFIG,DONT_CARE,DONT_CARE) + 1;
		if(num_of_regs2 > num_of_regs1)
		{
			num_of_regs1 = mpu_configRegs(alignedAddress,endAddress,CONFIG,ALLOW_REGION,currentRegNumber);
		}
		else
		{
			// Configure the MPU registers
			mpu_configRegsPair(currentRegNumber, alignedAddress, closestSize, ALLOW_REGION);
			currentRegNumber--;
			num_of_regs1 = mpu_configRegs(endAddress,(alignedAddress+closestSize),CONFIG,DISALLOW_REGION,currentRegNumber);
		}

	}	
	currentRegNumber -= num_of_regs1;

	// If there is at least 1 more spare register, we will use it to protect from accessing NULL_PD address.
	// We use area which also inclues HOST_EX_MAP_CPU_BASE_ADDR since we currently don't use it and accessing it can cause problems.

	mpu_configRegsPair(currentRegNumber,(0xa4000000/ONE_KILO_BYTES), SIXTY_FOUR_MEGA_BYTE, DISALLOW_REGION);
	currentRegNumber--;

	ASSERT(currentRegNumber > 1); // We use register 1 and register 0 hard coded earlier.

	// Allow the Common area+bootloader with 2nd highest priotity
	// All the addresses are assured to be 1K aligned so we work in unit of kilo bytes
	commonAreaSize = (((uint32)_ecommon_ram) - RAM_START_ADDRESS)/ONE_KILO_BYTES;
	mpu_configRegsPair(MPU_REG_NUMBER_1, RAM_START_ADDRESS, commonAreaSize, ALLOW_REGION);

	// don't allow address 0 since we want exception for NULL pointers. This is highest priotity
	mpu_BlockNullPointerWithReg0();




	// Enable MPU & configure default region
	regMPU_EN.bitFields.EN = MPU_ENABLED;	
	/* Allow access (read and write) for memory that is not covered by any enabled protection region descriptor 
	   Doesn't allow execution in order to avaoid the branch prediction issue */
	regMPU_EN.bitFields.KR = MPU_ENABLED;
	regMPU_EN.bitFields.KW = MPU_ENABLED;
	regMPU_EN.bitFields.UR = MPU_ENABLED;
	regMPU_EN.bitFields.UW = MPU_ENABLED;
	_sr(regMPU_EN.val, MPU_ENABLE);


}



