/***************************************************************
 File:		Bsrc.c
 Module:	BSRC
 Purpose: 	holds all the implementation of the BSRC API functions
 Description:  This module contains the implementation of the BSRC
 			 module  which is responsible for handling buffer status
 			 reports from STA in HE MU 
***************************************************************/

/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "Bsrc.h"
#include "Bsrc_Api.h"
#include "BsrcRegsRegs.h"
#include "BsrcCntMem_Descriptors.h"
#include "BsrcAgerMem_Descriptors.h"
#include "RegAccess_Api.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"
#include "Pac_Api.h"
#include "stringLibApi.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_HW_DRIVERS
#define LOG_LOCAL_FID 22



/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
#if defined (ENET_INC_LMAC0)
static void bsrcResetCountersRam(void);
static void bsrcResetAgerRam(void);
#endif
static void bsrcInvokeCommand(uint16 stationId, uint8 tIdMask, BsrcCommands_e command, RegBsrcRegsFwIndirectAccess_u* pRegBsrcRegsFwIndirectAccess);


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
static BsrcAgerMem_t* pBsrcAgerMem;
BsrcCntMem_t* pBsrcCntMem;


/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/
#if defined (BSRC_ARTIFICIAL_UPDATE_FOR_UL)
	BsrcTfInterval_t bsrcTfInterval = {FALSE, BSRC_DEFAULT_TF_INTERVAL, 0};
#endif
/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

#if defined (ENET_INC_LMAC0)
/**********************************************************************************

bsrcResetCountersRam  


Description:
------------
Reset BCR counters memory

Input:
-----
		
Output:
-------

Returns:
--------
void
	
**********************************************************************************/
static void bsrcResetCountersRam(void)
{
	/* Counter memory looks like that:
		STA0 -   |8bits mask|16bit counter TID7||16bit counter TID6|...|16bit counter TID0|
		STA1 -   |8bits mask|16bit counter TID7||16bit counter TID6|...|16bit counter TID0|
		.
		.
		.
		STA255 - |8bits mask|16bit counter TID7||16bit counter TID6|...|16bit counter TID0|
	*/
	memset32(pBsrcCntMem, BSRC_COUNTER_RESET_VALUE, CONVERT_BYTES_TO_WORDS(NUM_OF_STAS*((BSRC_NUM_OF_DATA_TIDS*BSRC_SIZE_OF_COUNTER_IN_BYTES)+BSRC_SIZE_OF_MASK_IN_BYTES)));
}

/**********************************************************************************

bsrcResetAgerRam  


Description:
------------
Reset BCR agers memory

Input:
-----
		
Output:
-------
	

Returns:
--------
void
	
**********************************************************************************/
static void bsrcResetAgerRam(void)
{
	/* Ager memory looks like that:
		STA0 -   |16bit ager TID7||16bit ager TID6|...|16bit ager TID0|
		STA1 -   |16bit ager TID7||16bit ager TID6|...|16bit ager TID0|
		.
		.
		.
		STA255 - |16bit ager TID7||16bit ager TID6|...|16bit ager TID0|
	*/
	memset32(pBsrcAgerMem, BSRC_AGER_RESET_VALUE, CONVERT_BYTES_TO_WORDS(NUM_OF_STAS*(BSRC_NUM_OF_DATA_TIDS*BSRC_SIZE_OF_AGER_IN_BYTES)));
}
#endif


/**********************************************************************************

bsrcInvokeCommand  


Description:
------------
Invoke BCR FW indirect command

Input:
-----
		
Output:
-------
	

Returns:
--------
void
	
**********************************************************************************/
static void bsrcInvokeCommand(uint16 stationId, uint8 tIdMask, BsrcCommands_e command, RegBsrcRegsFwIndirectAccess_u* pRegBsrcRegsFwIndirectAccess)
{
	RegBsrcRegsFifoStatus_u RegBsrcRegsFifoStatus;
 	uint32 timeStamp = 0;
	
	RegBsrcRegsFifoStatus.val = 0;
	
	//Prepare the command
	pRegBsrcRegsFwIndirectAccess->bitFields.fwRxTidBitmap = tIdMask;
	pRegBsrcRegsFwIndirectAccess->bitFields.fwRxStaid = stationId;
	pRegBsrcRegsFwIndirectAccess->bitFields.fwCommand = command;

	//Wait for FIFO to be not full, otherwise command will be ignored by HW
	timeStamp = GET_TSF_TIMER_LOW();
    do
    {
    	ASSERT((GET_TSF_TIMER_LOW() - timeStamp) <= BSRC_MAX_WAITING_FOR_DONE_TIMEOUT);    
		RegAccess_Read(REG_BSRC_REGS_FIFO_STATUS, &RegBsrcRegsFifoStatus.val);
    }while(RegBsrcRegsFifoStatus.bitFields.srFwFifoFull);

	//Write the command to HW
	RegAccess_Write(REG_BSRC_REGS_FW_INDIRECT_ACCESS, pRegBsrcRegsFwIndirectAccess->val);
}

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

Bsrc_GetCounter  


Description:
------------
Get counter value from BSRC memory

Input:
-----
stationId - the station id
tId
		
Output:
-------
	

Returns:
--------
void
	
**********************************************************************************/
uint16 Bsrc_GetCounter(uint16 stationId, uint8 tId)
{
	uint16* pCounter = NULL;

	pCounter = (uint16*)&(pBsrcCntMem->common[stationId]) + (tId*BSRC_SIZE_OF_COUNTER_IN_BYTES);
	
	return *pCounter;
}

/**********************************************************************************

Bsrc_SetCounterInitMode  


Description:
------------
Clear SW init bit and return control to HW

Input:
-----
stationId - the station id
eInitType - init type to set: HW or SW
		
Output:
-------
	

Returns:
--------
void
	
**********************************************************************************/
void Bsrc_SetCounterInitMode(uint16 stationId, BsrcCounterInitType_e eInitType)
{
	uint32 hwRegBaseAddress;
	uint32 maskVal;
	uint16 regOffset;
	uint16 bitLocationInWord;
	uint32 initVal;

	DEBUG_ASSERT((stationId < NUM_OF_STAS) && (eInitType < BSRC_COUNTER_INIT_INVALID));

	// Set sw init = 0 for the correct STA, first find the correct register including this STA:
	// Find regOffset - each STA is represented by a bit, 32 bits per register - devide by 32
	regOffset = BSRC_GET_SW_INIT_REGISTER_OFFSET(stationId);
	hwRegBaseAddress = REG_BSRC_REGS_BUF_STS_SW_INIT_UPDATE_31_0 + (regOffset*HW_REG_SIZE);
		
	// Find location of the STA inside the word/register:
	bitLocationInWord = BSRC_MODULO_32(stationId);
	maskVal = 0x1 << bitLocationInWord;
	initVal = eInitType << bitLocationInWord;
	
	// Write '0' to the right STA in the right register using mask to keep other STAs configuration
	RegAccess_WriteMasked(hwRegBaseAddress, maskVal, initVal);
}


/**********************************************************************************

Bsrc_Command  


Description:
------------
Invoke the 5 Bsrc commands

Input:
-----
stationId - the station id
tIdMask - specific TIDs to update (one or more)
counter - counter init value
command - one of the 5 commands to invoke
		
Output:
-------
	

Returns:
--------
void  (in case of failure there will be an assert) 
	
**********************************************************************************/
void Bsrc_Command(uint16 stationId, uint8 tIdMask, uint32 counter, BsrcCommands_e command)
{	
	RegBsrcRegsFwIndirectAccess_u RegBsrcFwIndirectAccess;
	RegBsrcRegsFwBufStsCnt10_u RegBsrcRegsFwBufStsCnt;
	uint32 hwRegBaseAddress;
	uint16 scaledCounter = 0;
	//uint16 regOffset;
	//uint8 wordLocation;
	uint8 index = 0;
	
	RegBsrcFwIndirectAccess.val = 0;
	RegBsrcRegsFwBufStsCnt.val = 0;

	DEBUG_ASSERT(stationId < NUM_OF_STAS);

	switch(command)
	{
		case BSRC_COMMAND_MASK:
		case BSRC_COMMAND_UNMASK:
		case BSRC_COMMAND_CLEAR:
		case BSRC_COMMAND_SET:
		{
			break;
		}
		case BSRC_COMMAND_INIT:
		{
			// Check if counter exeeds the maximum value
			DEBUG_ASSERT(counter <= BSRC_COUNTER_VAL_UNLIMITED);

			// counter is 64bytes quantized so less than 64 will be 0, unless we explictly write '0'
			if((counter > 0) && (counter < BSRC_MIN_COUNTER_VAL))
			{
				scaledCounter = BSRC_MIN_COUNTER_VAL;
			}
			
			// 64 Bytes Quantization
			scaledCounter = BSRC_DIVIDE_BY_64(counter);
			
			// Before setting the counter set mode to be "SW init" so HW will not override the counter
#if !defined (BSRC_ARTIFICIAL_UPDATE_FOR_UL)
			Bsrc_SetCounterInitMode(stationId, BSRC_COUNTER_INIT_SW);
#endif 
			// Writing the same counter to all TIDs, although HW supports different counter for each TID we keep it simple
			hwRegBaseAddress = REG_BSRC_REGS_FW_BUF_STS_CNT_1_0;
			RegBsrcRegsFwBufStsCnt.bitFields.crFwBufStsCnt0 = scaledCounter;
			RegBsrcRegsFwBufStsCnt.bitFields.crFwBufStsCnt1 = scaledCounter;

			for(index = 0 ; index < 4 ; index++)
			{
				RegAccess_Write(hwRegBaseAddress, RegBsrcRegsFwBufStsCnt.val);
				hwRegBaseAddress+=HW_REG_SIZE;
			}

			break;
		}
		default:
		{
			FATAL("Bsrc_Command");
 		}
	}

	bsrcInvokeCommand(stationId, tIdMask, command, &RegBsrcFwIndirectAccess);

}

#if defined (BSRC_ARTIFICIAL_UPDATE_FOR_UL)
void Bsrc_SetTfFixedInterval(uint8 tfInterval, bool isFixedInterval)
{
	BsrcCounterInitType_e initType;
	
	bsrcTfInterval.isFixedInterval = isFixedInterval;
	bsrcTfInterval.tfInterval = tfInterval;

	if (isFixedInterval == FALSE)
	{
		/* Back to default - return to HW_init mode */
		initType = BSRC_COUNTER_INIT_HW;
	}
	else
	{
		/* Set fixed value - use SW_init mode to make sure that TF interval will be accurate, and won't be affected from HW BSRC updates */			
		initType = BSRC_COUNTER_INIT_SW;
	}
	Bsrc_SetCounterInitMode(0, initType);
	Bsrc_SetCounterInitMode(1, initType);		
	Bsrc_SetCounterInitMode(2, initType);
	Bsrc_SetCounterInitMode(3, initType);
}
uint8 Bsrc_GetTfInterval(void)
{
	uint8 interval = BSRC_DEFAULT_TF_INTERVAL;

    /* If auto interval - set the default (10 ms), if fixed interval - set fixed interval */
	if (bsrcTfInterval.isFixedInterval == TRUE)
	{
		interval = bsrcTfInterval.tfInterval;
	}
	return interval;
}
uint8 Bsrc_GetTfFixedInterval(void)
{	
	return bsrcTfInterval.isFixedInterval;
}

#endif

/**********************************************************************************

Bsrc_Initialize  


Description:
------------
Initialize the Bsrc module registers and memories

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void Bsrc_Initialize()
{	
#ifdef ENET_INC_ARCH_WAVE600D2
	RegBsrcRegsGlobalReg_u RegBsrcGlobal;
#else
	RegBsrcRegsGlobal_u RegBsrcGlobal;
#endif
	RegBsrcRegsIntEn_u RegBsrcIntEn;
	RegBsrcRegsAgerCfg_u RegBsrcAgerCfg;
	RegBsrcRegsFwAgerPeriod_u RegBsrcFwAgerPeriod;
	RegBsrcRegsIntErrorEn_u RegBsrcIntErrorEn;
	RegBsrcRegsBufStsSwInitUpdate310_u RegBsrcBufStsSwInitUpdate;

	RegBsrcGlobal.val = 0;
	RegBsrcIntEn.val = 0;
	RegBsrcAgerCfg.val = 0;
	RegBsrcFwAgerPeriod.val = 0;
	RegBsrcIntErrorEn.val = 0;
	RegBsrcBufStsSwInitUpdate.val = 0x0;
	
	/* Set pointer to BSRC module memories */
	pBsrcCntMem = (BsrcCntMem_t*)(BSRC_CNT_MEM_BASE_ADDR);
	pBsrcAgerMem = (BsrcAgerMem_t*)(BSRC_AGER_MEM_BASE_ADDR);

	/* Disable FW done interrupt - interrupt for every FW command done (when enabled FW should wait for int. before sending another command) */
	RegBsrcIntEn.bitFields.crFwDoneIntEn = FALSE;
	RegAccess_Write(REG_BSRC_REGS_INT_EN, RegBsrcIntEn.val);

	/* Enable error interrupt start before rxc fifo empty */
	RegBsrcIntErrorEn.bitFields.crStartWhileRxcFifoNotEmptyIntEn = TRUE;
	RegAccess_Write(REG_BSRC_REGS_INT_ERROR_EN, RegBsrcIntErrorEn.val);

	/* Init ager mechanism - currently the mechanism is disabled */
	RegBsrcAgerCfg.bitFields.crAgerCntEn = FALSE;
	RegBsrcAgerCfg.bitFields.crAgerStrobeCnt = BSRC_AGER_DEFAULT_STROBE_COUNT;
	RegBsrcAgerCfg.bitFields.crAgerPeriod = BSRC_DEFAULT_AGER_PERIOD;
	RegAccess_Write(REG_BSRC_REGS_AGER_CFG, RegBsrcAgerCfg.val);

	/* Set default FW command ager periods */
	RegBsrcFwAgerPeriod.bitFields.crFwClrAgerPeriod = BSRC_FW_CLR_AGER_PERIOD;
	RegBsrcFwAgerPeriod.bitFields.crFwSetAgerPeriod = BSRC_FW_SET_AGER_PERIOD;
	RegAccess_Write(REG_BSRC_REGS_FW_AGER_PERIOD, RegBsrcFwAgerPeriod.val);

	/* Disable SW init option, using same struct for all STAs */
	RegAccess_Write(REG_BSRC_REGS_BUF_STS_SW_INIT_UPDATE_31_0, RegBsrcBufStsSwInitUpdate.val);
	RegAccess_Write(REG_BSRC_REGS_BUF_STS_SW_INIT_UPDATE_63_32, RegBsrcBufStsSwInitUpdate.val);
	RegAccess_Write(REG_BSRC_REGS_BUF_STS_SW_INIT_UPDATE_95_64, RegBsrcBufStsSwInitUpdate.val);
	RegAccess_Write(REG_BSRC_REGS_BUF_STS_SW_INIT_UPDATE_127_96, RegBsrcBufStsSwInitUpdate.val);
	RegAccess_Write(REG_BSRC_REGS_BUF_STS_SW_INIT_UPDATE_159_128, RegBsrcBufStsSwInitUpdate.val);
	RegAccess_Write(REG_BSRC_REGS_BUF_STS_SW_INIT_UPDATE_191_160, RegBsrcBufStsSwInitUpdate.val);
	RegAccess_Write(REG_BSRC_REGS_BUF_STS_SW_INIT_UPDATE_223_192, RegBsrcBufStsSwInitUpdate.val);
	RegAccess_Write(REG_BSRC_REGS_BUF_STS_SW_INIT_UPDATE_255_224, RegBsrcBufStsSwInitUpdate.val);

	/* Init memories - same memory is used for both bands, only band0 erases them also for single band support */
#if defined (ENET_INC_LMAC0)
	bsrcResetCountersRam();
	bsrcResetAgerRam();
#endif //ENET_INC_LMAC0

	/* Enable the BSRC HW */
	RegBsrcGlobal.bitFields.crBlockEn = TRUE;
 	RegBsrcGlobal.bitFields.crBufStsCompensationEn = FALSE;
	RegBsrcGlobal.bitFields.crMaxMpduHdrLength = BSRC_MAX_MPDU_HEADER_LENGTH;
#ifdef 	ENET_INC_ARCH_WAVE600D2
	RegAccess_Write(REG_BSRC_REGS_GLOBAL_REG, RegBsrcGlobal.val);
#else
	RegAccess_Write(REG_BSRC_REGS_GLOBAL, RegBsrcGlobal.val);
#endif

}

