/***************************************************************
 File:		Bsrc.c
 Module:	BSRC
 Purpose: 	holds all the implementation of the BSRC API functions
 Description:  This module contains the implementation of the BSRC
 			 module  which is responsible for handling buffer status
 			 reports from STA in HE MU 
***************************************************************/

/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "Bsrc.h"
#include "Bsrc_Api.h"
#include "BsrcRegsRegs.h"
#include "BsrcCntMemFw_Descriptors.h"
#include "BsrcAgerMem_Descriptors.h"
#include "RegAccess_Api.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"
#include "Pac_Api.h"
#include "stringLibApi.h"
#include "BsrcNdpfbReport_Descriptors.h"
#include "ShramBsrcFifos.h"




/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_HW_DRIVERS
#define LOG_LOCAL_FID   23
#define BSRC_SETTINGS_CLEAR_STRB 	(1)



/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/



/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
#if defined (ENET_INC_LMAC0)
static void bsrcResetCountersRam(void);
static void bsrcResetAgerRam(void);
#endif
static void bsrcInvokeCommand(uint16 stationId, uint8 tIdMask, BsrcCommands_e command, RegBsrcRegsFwIndirectAccess_u* pRegBsrcRegsFwIndirectAccess);

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
static BsrcAgerMem_t* pBsrcAgerMem = NULL;
static BsrcCntMemFw_t* pBsrcCntMemFw = NULL;
BsrcDb_t bsrcDb;
BsrcNdpfbReport_t* pBsrcNdpfReportFifo = NULL;



/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/
#if defined (BSRC_ARTIFICIAL_UPDATE_FOR_UL)
	BsrcTfInterval_t bsrcTfInterval = {FALSE, BSRC_DEFAULT_TF_INTERVAL, 0};
#endif
/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

#if defined (ENET_INC_LMAC0)
/**********************************************************************************

bsrcResetCountersRam  


Description:
------------
Reset BCR counters memory

Input:
-----
		
Output:
-------

Returns:
--------
void
	
**********************************************************************************/
static void bsrcResetCountersRam(void)
{
	memset32(pBsrcCntMemFw, BSRC_COUNTER_RESET_VALUE, CONVERT_BYTES_TO_WORDS(sizeof(BsrcCntMemFw_t)));
}

/**********************************************************************************

bsrcResetAgerRam  


Description:
------------
Reset BCR agers memory

Input:
-----
		
Output:
-------
	

Returns:
--------
void
	
**********************************************************************************/
static void bsrcResetAgerRam(void)
{
	memset32(pBsrcAgerMem, BSRC_AGER_RESET_VALUE, CONVERT_BYTES_TO_WORDS(sizeof(BsrcAgerMem_t)));
}

#endif


/**********************************************************************************

bsrcInvokeCommand  


Description:
------------
Invoke BCR FW indirect command

Input:
-----
		
Output:
-------
	

Returns:
--------
void
	
**********************************************************************************/
static void bsrcInvokeCommand(uint16 stationId, uint8 tIdMask, BsrcCommands_e command, RegBsrcRegsFwIndirectAccess_u* pRegBsrcRegsFwIndirectAccess)
{
	RegBsrcRegsFifoStatus0_u RegBsrcRegsFifoStatus;
 	uint32 timeStamp = 0;
	
	RegBsrcRegsFifoStatus.val = 0;
	
	//Prepare the command
	pRegBsrcRegsFwIndirectAccess->bitFields.fwRxTidBitmap = tIdMask;
	pRegBsrcRegsFwIndirectAccess->bitFields.fwRxStaid = stationId;
	pRegBsrcRegsFwIndirectAccess->bitFields.fwCommand = command;

	//Wait for FIFO to be not full, otherwise command will be ignored by HW
	timeStamp = GET_TSF_TIMER_LOW();
    do
    {
    	ASSERT((GET_TSF_TIMER_LOW() - timeStamp) <= BSRC_MAX_WAITING_FOR_DONE_TIMEOUT);    
		RegAccess_Read(REG_BSRC_REGS_FIFO_STATUS0, &RegBsrcRegsFifoStatus.val);
    }while(RegBsrcRegsFifoStatus.bitFields.srFwFifoFull);

	//Write the command to HW
	RegAccess_Write(REG_BSRC_REGS_FW_INDIRECT_ACCESS, pRegBsrcRegsFwIndirectAccess->val);
}

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

Bsrc_GetCounter  


Description:
------------
Get counter value from BSRC memory

Input:
-----
stationId - the station id
tId
		
Output:
-------
	

Returns:
--------
void
	
**********************************************************************************/
uint16 Bsrc_GetCounter(uint16 stationId, uint8 tId)
{
	uint16* pCounter = NULL;

	pCounter = (uint16*)&(pBsrcCntMemFw->common[stationId]) + (tId*BSRC_SIZE_OF_COUNTER_IN_BYTES);
	
	return *pCounter;
}

/**********************************************************************************

Bsrc_SetCounterInitMode  


Description:
------------
Clear SW init bit and return control to HW

Input:
-----
stationId - the station id
eInitType - init type to set: HW or SW
		
Output:
-------
	

Returns:
--------
void
	
**********************************************************************************/
void Bsrc_SetCounterInitMode(uint16 stationId, BsrcCounterInitType_e eInitType)
{
	DEBUG_ASSERT((stationId < NUM_OF_STAS) && (eInitType < BSRC_COUNTER_INIT_INVALID));

	// Write init value for given station
	pBsrcCntMemFw->common[stationId].swInit = eInitType;
}

/**********************************************************************************

Bsrc_Command  


Description:
------------
Invoke the 5 Bsrc commands

Input:
-----
stationId - the station id
tIdMask - specific TIDs to update (one or more)
counter - counter init value
command - one of the 5 commands to invoke
		
Output:
-------
	

Returns:
--------
void  (in case of failure there will be an assert) 
	
**********************************************************************************/
void Bsrc_Command(uint16 stationId, uint8 tIdMask, uint32 counter, BsrcCommands_e command)
{	
	RegBsrcRegsFwIndirectAccess_u RegBsrcFwIndirectAccess;
	RegBsrcRegsFwBufStsCnt10_u RegBsrcRegsFwBufStsCnt;
	uint32 hwRegBaseAddress;
	uint16 scaledCounter = 0;
	uint8 index = 0;
	
	RegBsrcFwIndirectAccess.val = 0;
	RegBsrcRegsFwBufStsCnt.val = 0;

	DEBUG_ASSERT(stationId < NUM_OF_STAS);

	switch(command)
	{
		case BSRC_COMMAND_MASK:
		case BSRC_COMMAND_UNMASK:
		case BSRC_COMMAND_CLEAR:
		case BSRC_COMMAND_SET:
		{
			break;
		}
		case BSRC_COMMAND_INIT:
		{
			// Check if counter exeeds the maximum value
			DEBUG_ASSERT(counter <= BSRC_COUNTER_VAL_UNLIMITED);

			// counter is 64bytes quantized so less than 64 will be 0, unless we explictly write '0'
			if((counter > 0) && (counter < BSRC_MIN_COUNTER_VAL))
			{
				scaledCounter = BSRC_MIN_COUNTER_VAL;
			}
			
			// 64 Bytes Quantization
			scaledCounter = BSRC_DIVIDE_BY_64(counter);
			
			// Before setting the counter set mode to be "SW init" so HW will not override the counter
#if !defined (BSRC_ARTIFICIAL_UPDATE_FOR_UL)
			Bsrc_SetCounterInitMode(stationId, BSRC_COUNTER_INIT_SW);
#endif 
			// Writing the same counter to all TIDs, although HW supports different counter for each TID we keep it simple
			hwRegBaseAddress = REG_BSRC_REGS_FW_BUF_STS_CNT_1_0;
			RegBsrcRegsFwBufStsCnt.bitFields.crFwBufStsCnt0 = scaledCounter;
			RegBsrcRegsFwBufStsCnt.bitFields.crFwBufStsCnt1 = scaledCounter;

			for(index = 0 ; index < 4 ; index++)
			{
				RegAccess_Write(hwRegBaseAddress, RegBsrcRegsFwBufStsCnt.val);
				hwRegBaseAddress+=HW_REG_SIZE;
			}

			break;
		}
		default:
		{
			FATAL("Bsrc_Command");
 		}
	}

	bsrcInvokeCommand(stationId, tIdMask, command, &RegBsrcFwIndirectAccess);

}

#if defined (BSRC_ARTIFICIAL_UPDATE_FOR_UL)
void Bsrc_SetTfFixedInterval(uint8 tfInterval, bool isFixedInterval)
{
	BsrcCounterInitType_e initType;
	
	bsrcTfInterval.isFixedInterval = isFixedInterval;
	bsrcTfInterval.tfInterval = tfInterval;

	if (isFixedInterval == FALSE)
	{
		/* Back to default - return to HW_init mode */
		initType = BSRC_COUNTER_INIT_HW;
	}
	else
	{
		/* Set fixed value - use SW_init mode to make sure that TF interval will be accurate, and won't be affected from HW BSRC updates */			
		initType = BSRC_COUNTER_INIT_SW;
	}
	Bsrc_SetCounterInitMode(0, initType);
	Bsrc_SetCounterInitMode(1, initType);		
	Bsrc_SetCounterInitMode(2, initType);
	Bsrc_SetCounterInitMode(3, initType);
}
uint8 Bsrc_GetTfInterval(void)
{
	uint8 interval = BSRC_DEFAULT_TF_INTERVAL;

    /* If auto interval - set the default (10 ms), if fixed interval - set fixed interval */
	if (bsrcTfInterval.isFixedInterval == TRUE)
	{
		interval = bsrcTfInterval.tfInterval;
	}
	return interval;
}
uint8 Bsrc_GetTfFixedInterval(void)
{	
	return bsrcTfInterval.isFixedInterval;
}

#endif

/**********************************************************************************

Bsrc_Initialize  


Description:
------------
Initialize the Bsrc module registers and memories
	
**********************************************************************************/
void Bsrc_Initialize()
{	

	RegBsrcRegsGlobalReg_u    RegBsrcGlobal;
	RegBsrcRegsNdpfbCfg0_u    RegBsrcNdpfbCfg0;
	RegBsrcRegsNdpfbCfg1_u    RegBsrcNdpfbCfg1;
	RegBsrcRegsIntEn_u        RegBsrcIntEn;
	RegBsrcRegsAgerCfg_u      RegBsrcAgerCfg;
	RegBsrcRegsFwAgerPeriod_u RegBsrcFwAgerPeriod;
	RegBsrcRegsIntErrorEn_u   RegBsrcIntErrorEn;
	RegBsrcRegsNdpfbReportFifoBaseAddr_u		ndpfbFifoBaseAddrRegister;
	RegBsrcRegsNdpfbReportFifoDepthMinusOne_u	ndpfbFifoDepthRegister;	
	RegBsrcRegsNdpfbReportFifoClearStrb_u 		ndpfFifoClearStrb;
	uint16 indexSta;

	RegBsrcGlobal.val 		= 0;
	RegBsrcIntEn.val 		= 0;
	RegBsrcAgerCfg.val 		= 0;
	RegBsrcFwAgerPeriod.val = 0;
	RegBsrcIntErrorEn.val 	= 0;
	ndpfbFifoBaseAddrRegister.val = 0;
	ndpfbFifoDepthRegister.val	  = 0;
	ndpfFifoClearStrb.val   = 0;
	
	/* Set pointer to BSRC module memories */
	pBsrcCntMemFw = (BsrcCntMemFw_t*)(BSRC_CNT_MEM_BASE_ADDR);
	pBsrcAgerMem = (BsrcAgerMem_t*)(BSRC_AGER_MEM_BASE_ADDR);

	/* Disable FW done interrupt - interrupt for every FW command done (when enabled FW should wait for int. before sending another command) */
	RegBsrcIntEn.bitFields.crFwDoneIntEn = FALSE;
	RegAccess_Write(REG_BSRC_REGS_INT_EN, RegBsrcIntEn.val);

	/* Enable error interrupt start before rxc fifo empty */
	RegBsrcIntErrorEn.bitFields.crStartWhileRxcFifoNotEmptyIntEn = TRUE;
	RegAccess_Write(REG_BSRC_REGS_INT_ERROR_EN, RegBsrcIntErrorEn.val);

	/* Init ager mechanism - currently the mechanism is disabled */
	RegBsrcAgerCfg.bitFields.crAgerCntEn = FALSE;
	RegBsrcAgerCfg.bitFields.crAgerStrobeCnt = BSRC_AGER_DEFAULT_STROBE_COUNT;
	RegBsrcAgerCfg.bitFields.crAgerPeriod = BSRC_DEFAULT_AGER_PERIOD;
	RegAccess_Write(REG_BSRC_REGS_AGER_CFG, RegBsrcAgerCfg.val);

	/* Set default FW command ager periods */
	RegBsrcFwAgerPeriod.bitFields.crFwClrAgerPeriod = BSRC_FW_CLR_AGER_PERIOD;
	RegBsrcFwAgerPeriod.bitFields.crFwSetAgerPeriod = BSRC_FW_SET_AGER_PERIOD;
	RegAccess_Write(REG_BSRC_REGS_FW_AGER_PERIOD, RegBsrcFwAgerPeriod.val);

	/* Disable SW init option, using same struct for all STAs */
	for (indexSta = 0; indexSta < NUM_OF_STAS; indexSta++)
	{
		pBsrcCntMemFw->common[indexSta].swInit = 0;
	}

	/* Init memories - same memory is used for both bands, only band0 erases them also for single band support */
#if defined (ENET_INC_LMAC0)
	bsrcResetCountersRam();
	bsrcResetAgerRam();
#endif //ENET_INC_LMAC0

	

	/* == init fifo BSRC == */
	// Initialize a pointer of the error fifo for this specific band;
#if defined (ENET_INC_LMAC0)
	pBsrcNdpfReportFifo = &BsrcReportFifo[0]; 
#endif
#if defined(ENET_INC_LMAC1) 
	pBsrcNdpfReportFifo = &BsrcReportFifo_B1[0];
#endif

	ndpfbFifoBaseAddrRegister.bitFields.crNdpfbReportFifoBaseAddr = CONVERT_PHYSICAL_TO_SHRAM_OFFSET((uint32)pBsrcNdpfReportFifo);
	RegAccess_Write(REG_BSRC_REGS_NDPFB_REPORT_FIFO_BASE_ADDR, ndpfbFifoBaseAddrRegister.val);

	ndpfbFifoDepthRegister.bitFields.crNdpfbReportFifoDepthMinusOne = BSRC_NDPF_FIFO_SIZE - 1;
	RegAccess_Write(REG_BSRC_REGS_NDPFB_REPORT_FIFO_DEPTH_MINUS_ONE, ndpfbFifoDepthRegister.val);
	
		
	ndpfFifoClearStrb.bitFields.crNdpfbReportFifoClearStrb = BSRC_SETTINGS_CLEAR_STRB; //Clear NDP feedback report FIFO
	ndpfFifoClearStrb.bitFields.crNdpfbReportFifoClearFullDropCtrStrb = BSRC_SETTINGS_CLEAR_STRB; //Clear NDP feedback report FIFO full drop counter.
	RegAccess_Write(REG_BSRC_REGS_NDPFB_REPORT_FIFO_CLEAR_STRB,ndpfFifoClearStrb.val);

	
	// Clear the report NFRP fifo in SHRAM
	memset(pBsrcNdpfReportFifo, 0, (sizeof(BsrcNdpfbReport_t) * BSRC_NDPF_FIFO_SIZE));

	/* enable BSRC NFRP block, Hw default is enable */
	RegAccess_Read(REG_BSRC_REGS_NDPFB_CFG0, &RegBsrcNdpfbCfg0.val);
	RegBsrcNdpfbCfg0.bitFields.crNdpfbBlockEn = TRUE;
	RegAccess_Write(REG_BSRC_REGS_NDPFB_CFG0, RegBsrcNdpfbCfg0.val);
	/* Set default NDP feedback thresholds */
	RegBsrcNdpfbCfg1.val = 0;
	RegBsrcNdpfbCfg1.bitFields.crNdpfbValBelowThresh = BSRC_NDPFB_VAL_BELOW_THRESH;
	RegBsrcNdpfbCfg1.bitFields.crNdpfbValAboveThresh = BSRC_NDPFB_VAL_ABOVE_THRESH;	
	RegAccess_Write(REG_BSRC_REGS_NDPFB_CFG1, RegBsrcNdpfbCfg1.val);


	/* Enable the BSRC HW */
	RegBsrcGlobal.bitFields.crBlockEn = TRUE;
 	RegBsrcGlobal.bitFields.crBufStsCompensationEn = FALSE;
	RegBsrcGlobal.bitFields.crMaxMpduHdrLength = BSRC_MAX_MPDU_HEADER_LENGTH;
	RegAccess_Write(REG_BSRC_REGS_GLOBAL_REG, RegBsrcGlobal.val);
	

}

/**********************************************************************************
Bsrc_UpdateStationNfrpSupport 

Description:
------------
get info from STA, if supporting the feature of NDP feedback, and update the BSRC.	 	
**********************************************************************************/
void Bsrc_UpdateStationNfrpSupport(StaId stationId, bool isNfrpSupport)
{
	pBsrcCntMemFw->common[stationId].ndpFeedbackAnticipated = isNfrpSupport;
}

/**********************************************************************************
Bsrc_ErrorHandler 

Description:
------------
	Check Bsrc Setting Hw error scenarios
	 	1) Drooped events due to fifo full
	 	2) Fifo num of entries decreased to less than zero (Synch problem between Sw and Hw)

**********************************************************************************/
void Bsrc_ErrorHandler(void)
{
	RegBsrcRegsIntErrorStatus_u intErrorStatus;
	RegBsrcRegsClearIntError_u  clearIntError;
	RegBsrcRegsNdpfbReportFifoDebug_u	fifoDebugReg;
	RegBsrcRegsNdpfbReportFifoClearStrb_u ndpfFifoClearStrb;

	intErrorStatus.val = 0;
	clearIntError.val = 0;
	fifoDebugReg.val = 0;
	ndpfFifoClearStrb.val = 0;
	

	RegAccess_Read(REG_BSRC_REGS_INT_ERROR_STATUS, &intErrorStatus.val);
	
	if(intErrorStatus.bitFields.srNdpfbReportFifoFullDropIrq) 
	{
	
	// Read the debug register
		RegAccess_Read(REG_BSRC_REGS_NDPFB_REPORT_FIFO_DEBUG,&fifoDebugReg.val);
	
		// Count the number of times the error fifo was full 
		bsrcDb.fifoFullCounter += fifoDebugReg.bitFields.srNdpfbReportFifoFullDropCtr;
		ASSERT(bsrcDb.fifoFullCounter == 0); // While we still ramp up gen6-D2 - we want to assert here so we can decide if we want to increase the fifo size. Later we can remove this assert.

		// Clear fifo full drop counter
		ndpfFifoClearStrb.bitFields.crNdpfbReportFifoClearFullDropCtrStrb = BSRC_SETTINGS_CLEAR_STRB;

		
	}
	
	if(intErrorStatus.bitFields.srNdpfbReportFifoDecrementLessThanZeroIrq) 
	{
		// check that we didn't decrease the fifo more than we were allowed
		ASSERT(fifoDebugReg.bitFields.srNdpfbReportFifoDecrementLessThanZero == FALSE);

	}
	
	RegAccess_Write(REG_BSRC_REGS_NDPFB_REPORT_FIFO_CLEAR_STRB,ndpfFifoClearStrb.val);
}


/**********************************************************************************
Bsrc_GetFifoParameters  

Description:
------------
	Get the updated information about the error fifo
	
**********************************************************************************/
void Bsrc_GetFifoParameters(BsrcFifoParams_t *fifoParams)
{
	RegBsrcRegsNdpfbReportFifoNumEntriesCount_u bsrcErrFifoNumEntriesCountReg; 


	RegAccess_Read(REG_BSRC_REGS_NDPFB_REPORT_FIFO_NUM_ENTRIES_COUNT, &bsrcErrFifoNumEntriesCountReg.val);
	
	fifoParams->currentIndexInFifo = bsrcDb.fifoReadIndex;
	fifoParams->numOfValidEntries = bsrcErrFifoNumEntriesCountReg.bitFields.srNdpfbReportFifoNumEntriesCount;

}

/**********************************************************************************
Bsrc_SetFifoParameters 

Description:
------------
	Sets the Fifo Params after FW finished processing all entries in fifo
Input: 
-----
	fifoParams
		1) Current Fifo Index
		2) Num of Entries in fifo waiting for FW processing

**********************************************************************************/	
void Bsrc_SetFifoParameters(BsrcFifoParams_t *fifoParams)

{
	RegBsrcRegsNdpfbReportFifoRdEntriesNum_u fifoRdEntriesNumReg;

	ILOG2_D("Bsrc_SetFifoParameters: fifoParams=%x",*(uint32 * )fifoParams);

	fifoRdEntriesNumReg.bitFields.crNdpfbReportFifoRdEntriesNum = fifoParams->numOfValidEntries;

	// Decrement  Number Of Entries In Fifo 
	RegAccess_Write(REG_BSRC_REGS_NDPFB_REPORT_FIFO_RD_ENTRIES_NUM,fifoRdEntriesNumReg.val);

	bsrcDb.fifoReadIndex = fifoParams->currentIndexInFifo;
}


/**********************************************************************************

Bsrc_reportFifoNotEmptyEventHandler  

Description:
------------
	
Process error FIFO. Check the cause, decrease the counter.	
	
**********************************************************************************/
void Bsrc_reportFifoNotEmptyEventHandler(void)
{
	BsrcFifoParams_t bsrcFifoParams;	
	BsrcNdpfbReport_t bsrcNdpfFifoEntry;
	
	uint8 fifoIndex;
		
	// Get number of new entries in fifo	
	Bsrc_GetFifoParameters(&bsrcFifoParams);

	// Loop over all new entries
	for(fifoIndex = 0; fifoIndex < bsrcFifoParams.numOfValidEntries; fifoIndex++)
	{
		
		// Get an entry from the fifo
		bsrcNdpfFifoEntry = pBsrcNdpfReportFifo[bsrcFifoParams.currentIndexInFifo];


		SLOG0(0, 0, BsrcNdpfbReport_t, &bsrcNdpfFifoEntry);

		// increase the index of the fifo which will be read next time (including wrap around)				
		bsrcFifoParams.currentIndexInFifo++;
		bsrcFifoParams.currentIndexInFifo &= BSRC_NDPF_FIFO_SIZE_MASK;
	}  
	
	// Set to HW the number of entries we already processed.
	Bsrc_SetFifoParameters(&bsrcFifoParams);
}

