/***********************************************************************************
 File:			ChannelSwitchManager.c
 Module:		ChannelSwitchManager
 Purpose: 		
 Description:   
************************************************************************************/

/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "OSAL_Api.h"
#include "queue_utility.h"
#include "ChannelSwitchManager_Api.h"
#include "ChannelSwitchManager.h"
#include "RxManager_Api.h"
#include "CtsManager_Api.h"
#include "ProcessManager_Api.h"
#include "ErrorHandler_Api.h"
#include "ServicesHandler_Api.h"
#include "Pauser_Api.h"
#include "loggerAPI.h"
#include "Locker_Api.h"
#include "HwQManager_API.h"
#include "Utils_Api.h"
#include "um_interface.h"
#include "init_ifmsg.h"
#include "BSSmanager_API.h"
#include "shram_man_msgs.h"
#include "CsaManager_Api.h"
#include "PacketDescriptor.h"
#include "ShramPacketDescriptors.h"
#include "frame.h"
#include "Dut_Api.h"
#include "Statistics_Api.h"
#include "StaDatabase_Api.h"
#include "PacManager_api.h"
#include "PacDur.h"
#include "Pac_Api.h"
#include "ShramPhyStatDb.h"
#include "shramTxDesc.h"
#include "CoC_Api.h"
#include "stringLibApi.h"
#ifndef ENET_INC_ARCH_WAVE600
#include "StatisticsPhyHandler.h"
#endif
#include "CommonRamStatistics.h"
#include "RegAccess_Api.h"
#include "fast_mem_psd2mips.h"
#include "Hdk_Api.h"
/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_CHANNEL_SWITCH_MANAGER
#define LOG_LOCAL_FID 0

#define CHANNEL_LOAD_TIMER_VALUE 					50 //50 mSec



/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void channelSwitchManager_ChangeState(ChannelSwitchManagerStates_e state);
static void channelSwitchManager_SetChannelRequest(K_MSG *channelSwitchManagerMessage); 
static void channelSwitchManager_TxPauseConfirm(K_MSG *channelSwitchManagerMessage); 
static void channelSwitchManager_TxPauseSpecificVapConfirm(K_MSG *channelSwitchManagerMessage); 
static void channelSwitchManager_RxPauseConfirm(K_MSG *channelSwitchManagerMessage); 
static void channelSwitchManager_ProcessStartConfirm(K_MSG *channelSwitchManagerMessage); 
static void channelSwitchManager_ProcessFinishConfirm(K_MSG *channelSwitchManagerMessage); 
static void channelSwitchManager_TxQueueLockConfirm(K_MSG *channelSwitchManagerMessage); 
static void channelSwitchManager_BeaconStartCsaBeaconConfirm(K_MSG *channelSwitchManagerMessage);
static void channelSwitchManager_CsaBeaconDoneIndication(K_MSG *channelSwitchManagerMessage);
static void channelSwitchManager_StopCsaBeaconsConfirm(K_MSG *channelSwitchManagerMessage);
static void channelSwitchManager_TimerExpired(K_MSG *channelSwitchManagerMessage);
static void channelSwitchManager_TxResumeConfirm(K_MSG *channelSwitchManagerMessage);
static void channelSwitchManager_RxResumeConfirm(K_MSG *channelSwitchManagerMessage);
static void channelSwitchManager_SetChannelConfirm(K_MSG *channelSwitchManagerMessage); 
static void channelSwitchManager_NotifyConfirm(K_MSG *channelSwitchManagerMessage); 
static void channelSwitchManager_VapManagerRegistrationEventAddVap(K_MSG *channelSwitchManagerMessage); 
static void channelSwitchManager_VapManagerRegistrationEventPostRemoveVap(K_MSG *channelSwitchManagerMessage); 
static void channelSwitchManager_CsaManagerStartTxCfm(K_MSG *channelSwitchManagerMessage); 
static void channelSwitchManager_CsaManagerStopTxCfm(K_MSG *channelSwitchManagerMessage); 
static void channelSwitchManager_ProcessInterruptedIndication(K_MSG *channelSwitchManagerMessage); 
static void channelSwitchManager_SendConfirmationToHost(void);
static void channelSwitchManager_SetRssiCfm(K_MSG *channelSwitchManagerMessage);
static void channelSwitchManager_HdkSetChannel(void);
static void channelSwitchManager_SmpsNotify(void);
#ifdef ENET_INC_ARCH_WAVE600
static void channelSwitchManager_PacNotify(void);
static void channelSwitchManager_StatsNotify(void);
#endif
static void channelSwitchManager_NotifyModules(void);
static void channelSwitchManager_SetRssiRxResumeCfm(K_MSG *channelSwitchManagerMessage);
static void channelSwitchManager_ChannelSwitchDone(void);
static void channelSwitchManager_ChannelLoadTimerExpired(K_MSG* channelSwitchManagerMessage);
static void channelSwitchManager_ChannelLoadReq(K_MSG* channelSwitchManagerMessage);
#ifdef ENET_INC_ARCH_WAVE600
static void channelSwitchManager_CalculateChannelLoadForPhyStatusReq(K_MSG* channelSwitchManagerMessage);
#endif
static void channelSwitchManager_UpdateBssLoadIeChannelUtilization(uint8 channelUtilization);
void ChannelSwitchManager_CalcActualChannel(void);
void ChannelSwitchManager_TaskEntry(K_MSG *channelSwitchManagerMessage);
void statisticsManagerResetWifiChannelStatsPerBand(uint8 channel);
uint32 channelSwitchManager_GetBusyRx(void);



/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

static const FunctionEntry_t afpTaskTable[TASK_CHANNEL_SWITCH_MANAGER_END - TASK_CHANNEL_SWITCH_MANAGER_START] =
{
	{channelSwitchManager_SetChannelRequest,							DOUBLE_CHECK_MSG_TYPE(CHANNEL_SWITCH_MANAGER_SET_CHANNEL_REQ)},
	{channelSwitchManager_TxPauseConfirm,								DOUBLE_CHECK_MSG_TYPE(CHANNEL_SWITCH_MANAGER_TX_PAUSE_CFM)},
	{channelSwitchManager_TxPauseSpecificVapConfirm,					DOUBLE_CHECK_MSG_TYPE(CHANNEL_SWITCH_MANAGER_TX_PAUSE_SPECIFIC_VAP_CFM)},
	{channelSwitchManager_RxPauseConfirm,								DOUBLE_CHECK_MSG_TYPE(CHANNEL_SWITCH_MANAGER_RX_PAUSE_CFM)},
	{channelSwitchManager_ProcessStartConfirm,							DOUBLE_CHECK_MSG_TYPE(CHANNEL_SWITCH_MANAGER_PROCESS_START_CFM)},
	{channelSwitchManager_ProcessFinishConfirm,							DOUBLE_CHECK_MSG_TYPE(CHANNEL_SWITCH_MANAGER_PROCESS_FINISH_CFM)},
	{channelSwitchManager_TxQueueLockConfirm,							DOUBLE_CHECK_MSG_TYPE(CHANNEL_SWITCH_MANAGER_TX_QUEUE_LOCK_CFM)},
	{channelSwitchManager_BeaconStartCsaBeaconConfirm,					DOUBLE_CHECK_MSG_TYPE(CHANNEL_SWITCH_MANAGER_BEACON_START_CSA_BEACON_CFM)},
	{channelSwitchManager_CsaBeaconDoneIndication,						DOUBLE_CHECK_MSG_TYPE(CHANNEL_SWITCH_MANAGER_CSA_BEACON_DONE_IND)},
	{channelSwitchManager_StopCsaBeaconsConfirm,						DOUBLE_CHECK_MSG_TYPE(CHANNEL_SWITCH_MANAGER_STOP_CSA_BEACONS_CFM)},
	{channelSwitchManager_TimerExpired,									DOUBLE_CHECK_MSG_TYPE(CHANNEL_SWITCH_MANAGER_TIMER_EXPIRED)},
	{channelSwitchManager_TxResumeConfirm,								DOUBLE_CHECK_MSG_TYPE(CHANNEL_SWITCH_MANAGER_TX_RESUME_CFM)},
	{channelSwitchManager_RxResumeConfirm,								DOUBLE_CHECK_MSG_TYPE(CHANNEL_SWITCH_MANAGER_RX_RESUME_CFM)},
	{channelSwitchManager_SetChannelConfirm,							DOUBLE_CHECK_MSG_TYPE(CHANNEL_SWITCH_MANAGER_SET_CHANNEL_CFM)},
	{channelSwitchManager_NotifyConfirm,								DOUBLE_CHECK_MSG_TYPE(CHANNEL_SWITCH_MANAGER_SET_NOTIFY_CFM)},
	{channelSwitchManager_VapManagerRegistrationEventAddVap,			DOUBLE_CHECK_MSG_TYPE(CHANNEL_SWITCH_MANAGER_VAP_MANAGER_REGISTRATION_EVENT_ADD_VAP)},
	{channelSwitchManager_VapManagerRegistrationEventPostRemoveVap,		DOUBLE_CHECK_MSG_TYPE(CHANNEL_SWITCH_MANAGER_VAP_MANAGER_REGISTRATION_EVENT_POST_REMOVE_VAP)},
	{channelSwitchManager_CsaManagerStartTxCfm,							DOUBLE_CHECK_MSG_TYPE(CHANNEL_SWITCH_MANAGER_CSA_MANAGER_START_TX_CFM)},
	{channelSwitchManager_CsaManagerStopTxCfm,							DOUBLE_CHECK_MSG_TYPE(CHANNEL_SWITCH_MANAGER_CSA_MANAGER_STOP_TX_CFM)},
	{channelSwitchManager_ProcessInterruptedIndication,					DOUBLE_CHECK_MSG_TYPE(CHANNEL_SWITCH_MANAGER_PROCESS_INTERRUPTED_IND)},
	{channelSwitchManager_SetRssiCfm,									DOUBLE_CHECK_MSG_TYPE(CHANNEL_SWITCH_MANAGER_SET_RSSI_CFM)},
	{channelSwitchManager_SetRssiRxResumeCfm,							DOUBLE_CHECK_MSG_TYPE(CHANNEL_SWITCH_MANAGER_SET_RSSI_RX_RESUME_CFM)},
	{channelSwitchManager_ChannelLoadTimerExpired,						DOUBLE_CHECK_MSG_TYPE(CHANNEL_SWITCH_MANAGER_CHANNEL_LOAD_TIMER_EXPIRED)},
	{channelSwitchManager_ChannelLoadReq,								DOUBLE_CHECK_MSG_TYPE(CHANNEL_SWITCH_MANAGER_GET_CHANNEL_LOAD)},
#ifdef ENET_INC_ARCH_WAVE600	
	{channelSwitchManager_CalculateChannelLoadForPhyStatusReq,			DOUBLE_CHECK_MSG_TYPE(CHANNEL_SWITCH_MANAGER_CALCULATE_CHANNEL_LOAD)},
#endif
};

static ChannelSwitchManagerParams_t channelSwitchManagerParams;

ChannelSwitchManagerChannelLoadDb_t channelSwitchManagerChannelLoadDb;

/*---------------------------------------------------------------------------------
/						External Definitions									
/----------------------------------------------------------------------------------*/


extern wholePhyRxStatusDb_t phyRxStatusDb;


/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

channelSwitchManager_ChangeState 

Description: Helper function to change state
------------

Input: new state
-----
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void channelSwitchManager_ChangeState(ChannelSwitchManagerStates_e state)
{
	ILOG0_DD("channelSwitchManager_ChangeState from %d, to %d", channelSwitchManagerParams.state, state);
	channelSwitchManagerParams.state = state;
}

/***********************************************************************
* statisticsManagerResetWifiChannelStatsPerBAnd
* 
* Description:
* ------------
* wifi_channelStats_t:  Reset channel statistics. Keep the channel number.
*						 
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
void statisticsManagerResetWifiChannelStatsPerBand(uint8 channel)
{
	BandId_e band = ConfigurationManager_GetMyBand();

	ChannelStatistics.channelNum[band] = channel;

	memset(&(ChannelStatistics.channelUtilizationStats[band]), 0, sizeof(chUtilization_t));

	ChannelStatistics.channelMax80211Rssi[band] = 0;
}    


/**********************************************************************************

channelSwitchManager_SetChannelRequest 

Description:
------------

Input: 
-----
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void channelSwitchManager_SetChannelRequest(K_MSG *channelSwitchManagerMessage)
{
	uint32 serviceBitmap = 0x0;
	K_MSG *pMsg = NULL;
	ProcessRequestParams_t *processRequestParams = NULL;
	UM_SET_CHAN_PARAMS *pUmSetChannelParamsLocal = NULL;
	ServiceRequestParams_t serviceRequestParams;
	uint32 serviceData = 0;


	//save a pointer to the message
	channelSwitchManagerParams.pSetChannelRequest = channelSwitchManagerMessage;
	channelSwitchManagerParams.pUmSetChannelParams = (UM_SET_CHAN_PARAMS *)pK_MSG_DATA(channelSwitchManagerParams.pSetChannelRequest);
	pUmSetChannelParamsLocal = channelSwitchManagerParams.pUmSetChannelParams;
#if defined (HDK_REL_2_0)
	channelSwitchManagerParams.pUmSetChannelParams->isRadarDetectionNeeded = FALSE; // in Gen6 Rel2.0 there is no support for Radar Detection	
	channelSwitchManagerParams.pUmSetChannelParams->isContinuousInterfererDetectionNeeded = FALSE; // in Gen6 Rel2.0 there is no support for Continuous Interferer Detection
#endif
	ChannelSwitchManager_CalcActualChannel();

	
	if (channelSwitchManagerParams.state == CHANNEL_SWITCH_MANAGER_STATE_IDLE)
	{
		//assert that we have an active vap
		DEBUG_ASSERT(channelSwitchManagerParams.activeVapsBitmap);
	}

	/* Reset channel statistic when changing channel */
	statisticsManagerResetWifiChannelStatsPerBand(channelSwitchManagerParams.pUmSetChannelParams->low_chan_num); 

	switch (channelSwitchManagerParams.state)
	{
	case CHANNEL_SWITCH_MANAGER_STATE_IDLE:
#ifdef	BEEROCK_DEBUG
		ILOG0_D("[channelSwitchManager_SetChannelRequest] CHANNEL_SWITCH_MANAGER_STATE_IDLE, channelSwitchManagerParams.pUmSetChannelParams->switch_type=%d", channelSwitchManagerParams.pUmSetChannelParams->switch_type);
#endif	//BEEROCK_DEBUG	

		//prepare request to process manager according to switch type
		switch (pUmSetChannelParamsLocal->switch_type)
		{
		case ST_NORMAL:
			//need to pause all
			serviceBitmap = (TRUE << SERVICE_ID_PAUSE_ALL);

			break;
		case ST_SCAN:
			//need to pause all and possibly send cts2self
			serviceBitmap = (TRUE << SERVICE_ID_PAUSE_ALL);
			if (pUmSetChannelParamsLocal->bg_scan == TRUE)
			{
				serviceBitmap |= (TRUE << SERVICE_ID_PAUSE_AC_TX);
				serviceBitmap |= (TRUE << SERVICE_ID_SEND_CTS_SELF_FRAME);
				serviceData = TX_PAUSE_ALL_ACS_EXCEPT_BEACON_AND_GPHP;
				serviceData |= (CTS2SELF_DURATION_MSEC_NORMAL_MODE << CTS2SELF_DURATION_MASK_SHIFT);	 					//duration of CTS2Self is also given on serviceData
			}
			break;
		case ST_RSSI:
#ifdef	BEEROCK_DEBUG			
			ILOG0_V("[channelSwitchManager_SetChannelRequest] in RSSI 0");
#endif	//BEEROCK_DEBUG	
			//bg_scan is turned on: need to pause all and send cts2self 
			if (pUmSetChannelParamsLocal->bg_scan == TRUE)
			{
				serviceBitmap  = (TRUE << SERVICE_ID_PAUSE_AC_TX); 												 //looks at serviceData to know what to send
				serviceBitmap |= (TRUE << SERVICE_ID_SEND_CTS_SELF_FRAME); 										 //send the actual CTS2Self
				serviceBitmap |= (TRUE << SERVICE_ID_PAUSE_ALL); 												 //pause all due to CTS2Self: currently is paused for only 5msec and should be configurable
				serviceData    = TX_PAUSE_ALL_ACS_EXCEPT_GPHP;	 										 		 //GPHP queue for CTS2Self should be enabled	 
				serviceData   |= (CTS2SELF_DURATION_MSEC_RSSI_MODE << CTS2SELF_DURATION_MASK_SHIFT);				 //duration of CTS2Self is also given on serviceData
			}
			else
			{
				//bg_scan is turned off: need to pause all 
				serviceBitmap = (TRUE << SERVICE_ID_PAUSE_ALL);
			}
				
			break;
		case ST_CSA:			
			if (pUmSetChannelParamsLocal->block_tx_pre == TRUE)
			{
				serviceBitmap |= (TRUE << SERVICE_ID_PAUSE_AC_TX);
				serviceData = TX_PAUSE_ALL_ACS_EXCEPT_BEACON_AND_GPHP;
			}
			else
			{
				//No pre Process service is needed
				serviceBitmap = 0x0;
			}
			break;
		default:
			DEBUG_FATAL("illegal switch type");
		}

		//move to next state
		channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_WAIT_FOR_PROCESS_START);
#ifdef	BEEROCK_DEBUG
		ILOG0_V("[channelSwitchManager_SetChannelRequest] in RSSI 1");
#endif
		//schedule process
		pMsg = OSAL_GET_MESSAGE(sizeof(ProcessRequestParams_t));
		processRequestParams = (ProcessRequestParams_t*)pK_MSG_DATA(pMsg);

		processRequestParams->processId 						= PROCESS_ID_CHANNEL_SWITCH;
		processRequestParams->startProcessMsg 					= CHANNEL_SWITCH_MANAGER_PROCESS_START_CFM;
		processRequestParams->requesterParams					= NULL;
		processRequestParams->preProcessServiceBitmap  			= serviceBitmap;
		processRequestParams->endProcessMsg						= CHANNEL_SWITCH_MANAGER_PROCESS_INTERRUPTED_IND; //this should never happen
		processRequestParams->postProcessServiceBitmap 			= 0x0; //this should never happen
		processRequestParams->returnTask						= TASK_CHANNEL_SWITCH_MANAGER; 
		processRequestParams->updateParamsBeforeFinalizing 		= TRUE;
		processRequestParams->serviceData						= serviceData;
		processRequestParams->vapId 							= channelSwitchManagerMessage->header.vapId;
		processRequestParams->dualBandProcess 					= TRUE;
		processRequestParams->processMsgHandledByCdbProcessMan 	= FALSE;

		OSAL_SEND_MESSAGE(PROCESS_MANAGER_SCHEDULE_PROCESS_REQUEST, TASK_PROCESS_MANAGER, pMsg, channelSwitchManagerMessage->header.vapId);
		break;
	case CHANNEL_SWITCH_MANAGER_STATE_SCAN:
		
		//move to next state
		channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_POST_SCAN_WAIT_FOR_TX_PAUSE);

		//send pause gplp request
		serviceRequestParams.returnTask = TASK_CHANNEL_SWITCH_MANAGER;
		serviceRequestParams.returnMsg = CHANNEL_SWITCH_MANAGER_TX_PAUSE_CFM;
		serviceRequestParams.vapIndex = channelSwitchManagerMessage->header.vapId;
		serviceRequestParams.serviceRequesterParams = NULL;
		serviceRequestParams.serviceId	= SERVICE_ID_PAUSE_AC_TX;
		serviceRequestParams.serviceData = TX_PAUSE_AC_GPLP;

		ServicesHandler_RunService(&serviceRequestParams);
		break;
		
	case CHANNEL_SWITCH_MANAGER_STATE_POST_CHANNEL_SWITCH_TX_PAUSED:

		if (pUmSetChannelParamsLocal->switch_type == ST_NORMAL)
		{
			/*If Normal - we are done*/		
			pUmSetChannelParamsLocal->result = SET_CHANNEL_RESULT_PASS;

			//move to next state
			channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_WAIT_FOR_PROCESS_FINISH);

			//request process manager to finish process & resume all tx
			pMsg = OSAL_GET_MESSAGE(sizeof(ProcessRequestParams_t));

			processRequestParams = (ProcessRequestParams_t*)pK_MSG_DATA(pMsg);

			processRequestParams->processId 				= PROCESS_ID_CHANNEL_SWITCH;
			processRequestParams->endProcessMsg				= CHANNEL_SWITCH_MANAGER_PROCESS_FINISH_CFM;
			processRequestParams->requesterParams			= NULL;
			if (channelSwitchManagerParams.staModeVapsBitmap != 0)
			{
				processRequestParams->postProcessServiceBitmap	= TRUE << SERVICE_ID_RESUME_ALL_TX;
			}
			else
			{
				processRequestParams->postProcessServiceBitmap	= TRUE << SERVICE_ID_RESUME_ALL;
			}
			processRequestParams->returnTask				= TASK_CHANNEL_SWITCH_MANAGER;
			processRequestParams->updateParamsBeforeFinalizing = TRUE;
			processRequestParams->serviceData				= 0;
			processRequestParams->vapId = channelSwitchManagerMessage->header.vapId;
			processRequestParams->dualBandProcess 					= FALSE;
			processRequestParams->processMsgHandledByCdbProcessMan 	= FALSE;

			OSAL_SEND_MESSAGE(PROCESS_MANAGER_PROCESS_EXCUTION_FINISHED, TASK_PROCESS_MANAGER, pMsg, channelSwitchManagerMessage->header.vapId);
		}
		//We are requested to do a scan - probably because a radar was detected during the silent period
		else if (pUmSetChannelParamsLocal->switch_type == ST_SCAN)
		{
			//Should not be BG scan
			DEBUG_ASSERT(pUmSetChannelParamsLocal->bg_scan == FALSE);

			// if we have a VSTA VAP, RX is not paused, need to pause it
			if (channelSwitchManagerParams.staModeVapsBitmap != 0)
			{
				//Pause Rx
				serviceRequestParams.returnTask = TASK_CHANNEL_SWITCH_MANAGER;
				serviceRequestParams.returnMsg = CHANNEL_SWITCH_MANAGER_RX_PAUSE_CFM;
				serviceRequestParams.serviceRequesterParams = NULL;
				serviceRequestParams.serviceId	= SERVICE_ID_PAUSE_RX;
				serviceRequestParams.serviceData = 0x0;
				serviceRequestParams.vapIndex = channelSwitchManagerMessage->header.vapId;
				
				ServicesHandler_RunService(&serviceRequestParams);

				//move to next state
				channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_POST_SCAN_WAIT_FOR_RX_PAUSE);
			}
			else
			{
				//We are already in Pause all - just skip to next step - locking GPLP queues
				//prepare a bitmap of all gplp queues that need to be locked
				channelSwitchManagerParams.activeSessionBitmap = channelSwitchManagerParams.activeVapsBitmap;
			
				//move to next state
				channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_PRE_SCAN_WAIT_FOR_TX_QUEUE_LOCK);

				//direct call to next state's handler
				channelSwitchManager_TxQueueLockConfirm(channelSwitchManagerMessage);
			}
		}
		else
		{
			FATAL("Channel Switch Manager: illegal type in current state");
		}
		break;
	case CHANNEL_SWITCH_MANAGER_STATE_RSSI:
		
		ASSERT(pUmSetChannelParamsLocal->switch_type == ST_NORMAL_AFTER_RSSI);
		
		//move to next state
		channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_WAIT_FOR_RSSI_RX_CFM);

		//Pause Rx
		serviceRequestParams.returnTask = TASK_CHANNEL_SWITCH_MANAGER;
		serviceRequestParams.returnMsg = CHANNEL_SWITCH_MANAGER_RX_PAUSE_CFM;
		serviceRequestParams.serviceRequesterParams = NULL;
		serviceRequestParams.serviceId	= SERVICE_ID_PAUSE_RX;
		serviceRequestParams.serviceData = 0x0;
		serviceRequestParams.vapIndex = channelSwitchManagerMessage->header.vapId;
		
		ServicesHandler_RunService(&serviceRequestParams);
		break;	
	default:
		DEBUG_FATAL("Channel Switch Manager: illegal event in current state");
	}

}

/**********************************************************************************

channelSwitchManager_ProcessStartConfirm

Description:
------------

Input: 
-----
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void channelSwitchManager_ProcessStartConfirm(K_MSG *channelSwitchManagerMessage)
{
	K_MSG* kMsg_p = NULL;
	UM_SET_CHAN_PARAMS *pUmSetChannelParamsLocal = NULL;
	ProcessManagerReturnParams_t *processMangerReturnParams = (ProcessManagerReturnParams_t*)pK_MSG_DATA(channelSwitchManagerMessage);
	BeaconCsaStartMsg_t *beaconCsaStartMsg_p = NULL;

#ifdef	BEEROCK_DEBUG	
	ILOG0_DD("[channelSwitchManager_ProcessStartConfirm] channelSwitchManagerParams.state = %d, channelSwitchManagerParams.pUmSetChannelParams->switch_type = %d", channelSwitchManagerParams.state, channelSwitchManagerParams.pUmSetChannelParams->switch_type);
#endif
	ASSERT(processMangerReturnParams->processStatus == PROCESS_STATUS_REQUEST_ACCEPTED);

	pUmSetChannelParamsLocal = channelSwitchManagerParams.pUmSetChannelParams;

	switch (channelSwitchManagerParams.state)
	{
	case CHANNEL_SWITCH_MANAGER_STATE_WAIT_FOR_PROCESS_START:
		switch (pUmSetChannelParamsLocal->switch_type)
		{
		case ST_NORMAL:
		case ST_RSSI:
#ifdef	BEEROCK_DEBUG
			ILOG0_V("[channelSwitchManager_ProcessStartConfirm] send message to HDK");
#endif
			//Set the channel
			channelSwitchManager_HdkSetChannel();
			break;						
		case ST_SCAN:
			//prepare a bitmap of all gplp queues that need to be locked
			channelSwitchManagerParams.activeSessionBitmap = channelSwitchManagerParams.activeVapsBitmap;
			
			//move to next state
			channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_PRE_SCAN_WAIT_FOR_TX_QUEUE_LOCK);

			//direct call to next state's handler
			channelSwitchManager_TxQueueLockConfirm(channelSwitchManagerMessage);
			break;
		case ST_CSA:	
			kMsg_p = OSAL_GET_MESSAGE(sizeof(BeaconCsaStartMsg_t));
			beaconCsaStartMsg_p = (BeaconCsaStartMsg_t *)pK_MSG_DATA(kMsg_p);
		
			//move to next state
			channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_PRE_CHANNEL_SWITCH_WAIT_FOR_BEACONS_UPDATE);

			//prepare parameters for beacon csa start
			beaconCsaStartMsg_p->time  = pUmSetChannelParamsLocal->chan_switch_time;
			beaconCsaStartMsg_p->primaryChannel = Utils_GetPrimaryChannel(channelSwitchManagerParams.actualLowChannel,
																					  pUmSetChannelParamsLocal->low_chan_num2,
																					  channelSwitchManagerParams.actualBW,
																					  channelSwitchManagerParams.actualPrimaryChannel);

			beaconCsaStartMsg_p->channelWidth = channelSwitchManagerParams.actualBW;
			
			beaconCsaStartMsg_p->csaChannelWidth = ChannelSwitchManager_GetChannelWidth(channelSwitchManagerParams.actualBW);

#ifdef ENET_INC_ARCH_WAVE600
			beaconCsaStartMsg_p->center0FreqIdx = ChannelSwitchManager_GetCenterFreq0Idx(channelSwitchManagerParams.actualLowChannel, channelSwitchManagerParams.actualBW, channelSwitchManagerParams.actualPrimaryChannel);
			beaconCsaStartMsg_p->center1FreqIdx = ChannelSwitchManager_GetCenterFreq1Idx(channelSwitchManagerParams.actualLowChannel, channelSwitchManagerParams.actualBW);
#else
			beaconCsaStartMsg_p->center0FreqIdx = ChannelSwitchManager_GetCenterFreq0Idx(channelSwitchManagerParams.actualLowChannel, channelSwitchManagerParams.actualBW);
#endif //ENET_INC_ARCH_WAVE600

			beaconCsaStartMsg_p->blockTx = pUmSetChannelParamsLocal->block_tx_pre;

			//send add CSA to all beacons request to Beacon Handler
			OSAL_SEND_MESSAGE(PAC_MANAGER_BEACON_CSA_START, TASK_PAC_MANAGER, kMsg_p, GET_DEFAULT_VAP_FOR_MY_BAND()); 
		
			break;
		default:
			DEBUG_FATAL("illegal switch type");
		}
		break;
	default:
		DEBUG_FATAL("Channel Switch Manager: illegal event in current state");
	}
}


/**********************************************************************************

channelSwitchManager_TxQueueLockConfirm

Description:
------------

Input: 
-----
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void channelSwitchManager_TxQueueLockConfirm(K_MSG *channelSwitchManagerMessage)
{
	uint32 vapIdx;
	RequesterLockParams_t requesterLockParams;
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
	UNUSED_PARAM(channelSwitchManagerMessage);
	ILOG2_D("state: %d, event: Tx Queue Lock Confirm", channelSwitchManagerParams.state);

	switch (channelSwitchManagerParams.state)
	{
	case CHANNEL_SWITCH_MANAGER_STATE_PRE_SCAN_WAIT_FOR_TX_QUEUE_LOCK:
	case CHANNEL_SWITCH_MANAGER_STATE_POST_SCAN_WAIT_FOR_TX_QUEUE_LOCK:
		if (channelSwitchManagerParams.activeSessionBitmap != 0x0)
		{
			vapIdx = Utils_FindFirstSet(channelSwitchManagerParams.activeSessionBitmap);
			channelSwitchManagerParams.activeSessionBitmap ^= (0x1 << vapIdx);
			
			//Send Lock Request
			requesterLockParams.callerContext = NULL;
			requesterLockParams.returnMsg = CHANNEL_SWITCH_MANAGER_TX_QUEUE_LOCK_CFM;
			requesterLockParams.returnTask = TASK_CHANNEL_SWITCH_MANAGER;
			Locker_LockSingleQueue(HW_TX_Q_TYPE_GPLP, vapIdx, 0x0, &requesterLockParams);
		}
		else
		{
			channelSwitchManagerParams.activeSessionBitmap = channelSwitchManagerParams.activeVapsBitmap;

			while (channelSwitchManagerParams.activeSessionBitmap != 0x0)
			{
				vapIdx = Utils_FindFirstSet(channelSwitchManagerParams.activeSessionBitmap);
				channelSwitchManagerParams.activeSessionBitmap ^= (0x1 << vapIdx);

				//flush vap's gplp
				hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
				hwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_GPLP;
				hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
				hwQueueManagerRequestParams.primaryAddr = vapIdx;
				hwQueueManagerRequestParams.secondaryAddr = 0x0;
				hwQueueManagerRequestParams.pHeadDesc = NULL;
				hwQueueManagerRequestParams.pTailDesc = NULL;
				HwQManager_FlushQ(&hwQueueManagerRequestParams);

				// Return PDs to liberator (flushed now - on post_scan)	
				if (hwQueueManagerRequestParams.pHeadDesc != NULL_PD)
				{
					//push flushed pd list to Liberator
					hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
					hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_DONE_LIST_LIBERATOR;
					hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
					HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams);
					// hwQueueManagerRequestParams.pHeadDesc unchanged
					// hwQueueManagerRequestParams.pTailDesc unchanged
				}

				//unlock gplp
				Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_GPLP, vapIdx, (0x1 << 0x0));
			}

			//Set the channel
			channelSwitchManager_HdkSetChannel();
		}
		break;
	default:
		DEBUG_FATAL("Channel Switch Manager: illegal event in current state");
	}
}


/**********************************************************************************

channelSwitchManager_ProcessFinishConfirm

Description:
------------

Input: 
-----
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void channelSwitchManager_ProcessFinishConfirm(K_MSG *channelSwitchManagerMessage)
{
	UNUSED_PARAM(channelSwitchManagerMessage);
#ifdef	BEEROCK_DEBUG
	ILOG0_D("[channelSwitchManager_ProcessFinishConfirm] channelSwitchManagerParams.state = %d", channelSwitchManagerParams.state);
#endif
	switch (channelSwitchManagerParams.state)
	{
	case CHANNEL_SWITCH_MANAGER_STATE_WAIT_FOR_PROCESS_FINISH:

		//send confirmation to host
		channelSwitchManager_SendConfirmationToHost();

		//move to idle state
		channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_IDLE);
		
		OSAL_SEND_NO_DATA_MESSAGE(HDK_SET_ONLINE_TIMER, TASK_HDK, ConfigurationManager_GetMyBand());
		break;
	default:
		DEBUG_FATAL("Channel Switch Manager: illegal event in current state");
	}

}


/**********************************************************************************

channelSwitchManager_SetChannelConfirm


Description:
------------

Input: 
-----
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void channelSwitchManager_SetChannelConfirm(K_MSG *channelSwitchManagerMessage)
{
	HdkSetChannelCfmParams_t *pHdkSetChannelCfmParams = NULL;
	
	//read set channel result
	pHdkSetChannelCfmParams = (HdkSetChannelCfmParams_t *)pK_MSG_DATA(channelSwitchManagerMessage);
	channelSwitchManagerParams.pUmSetChannelParams->result = UMI_OK;
	channelSwitchManagerParams.pUmSetChannelParams->antennaMask = pHdkSetChannelCfmParams->antennaMask;
	
#ifdef	BEEROCK_DEBUG	
	ILOG0_D("[channelSwitchManager_SetChannelConfirm] channelSwitchManagerParams.state = %d", channelSwitchManagerParams.state);
#endif

	//OSAL_SEND_NO_DATA_MESSAGE(DUT_CHANNEL_IS_SET, TASK_DUT, vapId);
	//dutDB.isChannelSet = TRUE;
	Dut_getDB()->isChannelSet = TRUE;

	switch (channelSwitchManagerParams.state)
	{
	case CHANNEL_SWITCH_MANAGER_STATE_WAIT_FOR_SET_CHANNEL_CFM_FROM_HDK:
		/*Notify interested modules*/
		channelSwitchManager_NotifyModules();
		break;
					
	default:
		DEBUG_FATAL("Channel Switch Manager: illegal event in current state");
	}
}

/**********************************************************************************

channelSwitchManager_NotifyConfirm



Description:
------------

Input: 
-----
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void channelSwitchManager_NotifyConfirm(K_MSG *channelSwitchManagerMessage)
{
	K_MSG *pMsg = NULL;
	ProcessRequestParams_t *processRequestParams = NULL;
	UM_SET_CHAN_PARAMS *pUmSetChannelParamsLocal = NULL;
	ServiceRequestParams_t serviceRequestParams;
	uint8 vapId = GET_DEFAULT_VAP_FOR_MY_BAND();
	RssiParams_t* pRssiParams = NULL;
	UNUSED_PARAM(channelSwitchManagerMessage);
#ifdef	BEEROCK_DEBUG	
	ILOG0_D("[channelSwitchManager_SetChannelConfirm] channelSwitchManagerParams.state = %d", channelSwitchManagerParams.state);
#endif

	pUmSetChannelParamsLocal = channelSwitchManagerParams.pUmSetChannelParams;

	switch (channelSwitchManagerParams.state)
	{
	case CHANNEL_SWITCH_MANAGER_STATE_WAIT_FOR_NOTIFY_CFM:
#ifdef	BEEROCK_DEBUG
		ILOG0_D("[channelSwitchManager_NotifyConfirm] CHANNEL_SWITCH_MANAGER_STATE_WAIT_FOR_NOTIFY_CFM, channelSwitchManagerParams.pUmSetChannelParams->switch_type = %d", pUmSetChannelParamsLocal->switch_type);
#endif
		//Increment CFM count
		channelSwitchManagerParams.notifyCfmCnt++;
		//Did all modules confirm?
		if (channelSwitchManagerParams.notifyReqCnt == channelSwitchManagerParams.notifyCfmCnt)
		{
			//clear notify counters
			channelSwitchManagerParams.notifyReqCnt = 0;
			channelSwitchManagerParams.notifyCfmCnt = 0;
			switch (pUmSetChannelParamsLocal->switch_type)
			{
			case ST_NORMAL:
				//move to next state
				channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_WAIT_FOR_PROCESS_FINISH);

				//request process manager to finish process & resume all
				pMsg = OSAL_GET_MESSAGE(sizeof(ProcessRequestParams_t));

				processRequestParams = (ProcessRequestParams_t*)pK_MSG_DATA(pMsg);

				processRequestParams->processId 				= PROCESS_ID_CHANNEL_SWITCH;
				processRequestParams->endProcessMsg				= CHANNEL_SWITCH_MANAGER_PROCESS_FINISH_CFM;
				processRequestParams->requesterParams			= NULL;
				processRequestParams->postProcessServiceBitmap 	= TRUE << SERVICE_ID_RESUME_ALL;
				processRequestParams->returnTask				= TASK_CHANNEL_SWITCH_MANAGER;
				processRequestParams->updateParamsBeforeFinalizing = TRUE;
				processRequestParams->serviceData       		= 0;
				processRequestParams->vapId = vapId;
				processRequestParams->dualBandProcess 					= FALSE;
				processRequestParams->processMsgHandledByCdbProcessMan 	= FALSE;

				OSAL_SEND_MESSAGE(PROCESS_MANAGER_PROCESS_EXCUTION_FINISHED, TASK_PROCESS_MANAGER, pMsg, vapId);
				break;
			case ST_NORMAL_AFTER_RSSI:
				//move to next state
				channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_WAIT_FOR_PROCESS_FINISH);

				// calc VapId according to SID
				pMsg = OSAL_GET_MESSAGE(sizeof(RssiParams_t));
				pRssiParams = (RssiParams_t*)pK_MSG_DATA(pMsg);
#if defined (ENET_INC_ARCH_WAVE600)
				FATAL("channelSwitchManager_NotifyConfirm"); // BEEROCK feature (rssi) is not supported in gen6 yet. There is no RXF/RXD in gen6.
#else	
				StaDb_GetVapId(pUmSetChannelParamsLocal->u16SID, &pRssiParams->vapId);
#endif
#ifdef	BEEROCK_DEBUG
				ILOG0_V("[channelSwitchManager_NotifyConfirm] in ST_NORMAL_AFTER_RSSI");
				ILOG0_DD("[channelSwitchManager_NotifyConfirm], pUmSetChannelParamsLocal->u16SID = %d, pRssiParams->vapId = %d", pUmSetChannelParamsLocal->u16SID, pRssiParams->vapId);
#endif
				OSAL_SEND_MESSAGE(PAC_MANAGER_ACTIVATE_NORMAL_REQ, TASK_PAC_MANAGER, pMsg, vapId);	
#ifdef	BEEROCK_DEBUG
				ILOG0_V("[channelSwitchManager_NotifyConfirm] in ST_NORMAL_AFTER_RSSI, before Process Manager Resume all");
#endif
				break;
			case ST_RSSI:
				//move to next state
				channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_WAIT_FOR_RSSI_SET_CFM);

				// calc MAC Address and VapId according to SID
				pMsg = OSAL_GET_MESSAGE(sizeof(RssiParams_t));
				pRssiParams = (RssiParams_t*)pK_MSG_DATA(pMsg);
#if defined (ENET_INC_ARCH_WAVE600)
				FATAL("channelSwitchManager_NotifyConfirm"); // BEEROCK feature (rssi) is not supported in gen6 yet. There is no RXF/RXD in gen6.
#else	
				StaDB_GetMacAddress(pUmSetChannelParamsLocal->u16SID, (IEEE_ADDR*)&pRssiParams->au8Addr);
				StaDb_GetVapId(pUmSetChannelParamsLocal->u16SID, &pRssiParams->vapId);
#endif

#ifdef	BEEROCK_DEBUG
				ILOG0_DD("[channelSwitchManager_NotifyConfirm], pUmSetChannelParamsLocal->u16SID = %d, pRssiParams->vapId = %d", pUmSetChannelParamsLocal->u16SID, pRssiParams->vapId);
				ILOG0_V("[channelSwitchManager_NotifyConfirm] in ST_RSSI");
#endif	
				OSAL_SEND_MESSAGE(PAC_MANAGER_ACTIVATE_RSSI_REQ, TASK_PAC_MANAGER, pMsg, vapId);	
				break;			
			case ST_CSA:
				//check if we need to keep tx paused after channel switch
				if (pUmSetChannelParamsLocal->block_tx_post == FALSE)
				{
					//move to next state
					channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_WAIT_FOR_PROCESS_FINISH);

					//request process manager to finish process & resume all tx
					pMsg = OSAL_GET_MESSAGE(sizeof(ProcessRequestParams_t));

					processRequestParams = (ProcessRequestParams_t*)pK_MSG_DATA(pMsg);

					processRequestParams->processId 				= PROCESS_ID_CHANNEL_SWITCH;
					processRequestParams->endProcessMsg				= CHANNEL_SWITCH_MANAGER_PROCESS_FINISH_CFM;
					processRequestParams->requesterParams			= NULL;
					processRequestParams->returnTask				= TASK_CHANNEL_SWITCH_MANAGER;
					processRequestParams->updateParamsBeforeFinalizing = TRUE;
					processRequestParams->serviceData				= 0;
					processRequestParams->vapId 					= vapId;
					processRequestParams->postProcessServiceBitmap	= TRUE << SERVICE_ID_RESUME_ALL;
					processRequestParams->dualBandProcess					= FALSE;
					processRequestParams->processMsgHandledByCdbProcessMan	= FALSE;

					OSAL_SEND_MESSAGE(PROCESS_MANAGER_PROCESS_EXCUTION_FINISHED, TASK_PROCESS_MANAGER, pMsg, vapId);
				}
				else if (channelSwitchManagerParams.staModeVapsBitmap != 0)
				{
					//If VSTA and TX is blocked need to resume RX
					serviceRequestParams.returnTask = TASK_CHANNEL_SWITCH_MANAGER;
					serviceRequestParams.returnMsg = CHANNEL_SWITCH_MANAGER_RX_RESUME_CFM;
					serviceRequestParams.serviceRequesterParams = NULL;
					serviceRequestParams.serviceId	= SERVICE_ID_RESUME_RX;
					serviceRequestParams.serviceData = 0x0;
					serviceRequestParams.vapIndex = vapId;

					ServicesHandler_RunService(&serviceRequestParams);
					
					//move to next state
					channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_POST_CHANNEL_SWITCH_WAIT_FOR_RX_RESUME);
				}
				else
				{
					/*Stop the Beacon Blocking Mechanism - in case new templates are set before Channel is set*/
					OSAL_SEND_NO_DATA_MESSAGE(PAC_MANAGER_BEACON_BLOCKING_STOP, TASK_PAC_MANAGER, vapId);
					
					//move to next state
					channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_POST_CHANNEL_SWITCH_TX_PAUSED);

					//send confirmation to host
					channelSwitchManager_SendConfirmationToHost();
				}
				break;
			case ST_SCAN:
				//move to next state
				channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_SCAN_WAIT_FOR_TX_RESUME);

				//resume tx on GPLP only
				serviceRequestParams.returnTask = TASK_CHANNEL_SWITCH_MANAGER;
				serviceRequestParams.returnMsg = CHANNEL_SWITCH_MANAGER_TX_RESUME_CFM;
				serviceRequestParams.vapIndex = vapId;
				serviceRequestParams.serviceRequesterParams = NULL;
				serviceRequestParams.serviceId	= SERVICE_ID_RESUME_AC_TX;
				serviceRequestParams.serviceData = TX_PAUSE_AC_GPLP; /*Resume GPLP only*/

				ServicesHandler_RunService(&serviceRequestParams);
				break;
			default:
				DEBUG_FATAL("illegal switch type");
			}
		}
		break;
					
	default:
		DEBUG_FATAL("Channel Switch Manager: illegal event in current state");
	}
	
}

/**********************************************************************************

channelSwitchManager_TxResumeConfirm


Description:
------------

Input: 
-----
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void channelSwitchManager_TxResumeConfirm(K_MSG *channelSwitchManagerMessage)
{
	ServiceRequestParams_t serviceRequestParams;
	UNUSED_PARAM(channelSwitchManagerMessage);

	ILOG2_D("state: %d, event: Tx Resume Confirm", channelSwitchManagerParams.state);

	switch (channelSwitchManagerParams.state)
	{
	case CHANNEL_SWITCH_MANAGER_STATE_SCAN_WAIT_FOR_TX_RESUME:

		//resume Rx for beacon & Probe Response only
		serviceRequestParams.returnTask = TASK_CHANNEL_SWITCH_MANAGER;
		serviceRequestParams.returnMsg = CHANNEL_SWITCH_MANAGER_RX_RESUME_CFM;
		serviceRequestParams.vapIndex = GET_DEFAULT_VAP_FOR_MY_BAND();
		serviceRequestParams.serviceRequesterParams = NULL;
		serviceRequestParams.serviceId	= SERVICE_ID_RESUME_RX;
		serviceRequestParams.serviceData = 0x0;

		ServicesHandler_RunService(&serviceRequestParams);

		//move to next state
		channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_SCAN_WAIT_FOR_RX_RESUME);
		break;
	default:
		DEBUG_FATAL("Channel Switch Manager: illegal event in current state");
	}
}


/**********************************************************************************

channelSwitchManager_RxResumeConfirm


Description:
------------

Input: 
-----
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void channelSwitchManager_RxResumeConfirm(K_MSG *channelSwitchManagerMessage)
{
	ILOG2_D("state: %d, event: Rx Resume Confirm", channelSwitchManagerParams.state);
	UNUSED_PARAM(channelSwitchManagerMessage);

	switch (channelSwitchManagerParams.state)
	{
	case CHANNEL_SWITCH_MANAGER_STATE_SCAN_WAIT_FOR_RX_RESUME:
		//send confirmation to host
		channelSwitchManager_SendConfirmationToHost();

		//move to idle state
		channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_SCAN);
		break;
	case CHANNEL_SWITCH_MANAGER_STATE_POST_CHANNEL_SWITCH_WAIT_FOR_RX_RESUME:
		//send confirmation to host
		channelSwitchManager_SendConfirmationToHost();
		
		//move to next state
		channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_POST_CHANNEL_SWITCH_TX_PAUSED);
		break;
		
	default:
		DEBUG_FATAL("Channel Switch Manager: illegal event in current state");
	}
}


/**********************************************************************************

channelSwitchManager_TxPauseConfirm

Description:
------------

Input: 
-----
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void channelSwitchManager_TxPauseConfirm(K_MSG *channelSwitchManagerMessage)
{
	ServiceRequestParams_t serviceRequestParams;

	ILOG2_D("state: %d, event: Tx Pause Confirm", channelSwitchManagerParams.state);
	UNUSED_PARAM(channelSwitchManagerMessage);

	switch (channelSwitchManagerParams.state)
	{
	case CHANNEL_SWITCH_MANAGER_STATE_POST_SCAN_WAIT_FOR_TX_PAUSE:

		//move to next state
		channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_POST_SCAN_WAIT_FOR_RX_PAUSE);

		//send pause all rx request
		serviceRequestParams.returnTask = TASK_CHANNEL_SWITCH_MANAGER;
		serviceRequestParams.returnMsg = CHANNEL_SWITCH_MANAGER_RX_PAUSE_CFM;
		serviceRequestParams.vapIndex = GET_DEFAULT_VAP_FOR_MY_BAND();
		serviceRequestParams.serviceRequesterParams = NULL;
		serviceRequestParams.serviceId	= SERVICE_ID_PAUSE_RX;
		serviceRequestParams.serviceData = 0x0;

		ServicesHandler_RunService(&serviceRequestParams);

		break;
	case CHANNEL_SWITCH_MANAGER_STATE_PRE_CHANNEL_SWITCH_WAIT_FOR_TX_PAUSE:
		//move to next state
		channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_PRE_CHANNEL_SWITCH_WAIT_FOR_RX_PAUSE);
		//send pause all rx request
		serviceRequestParams.returnTask = TASK_CHANNEL_SWITCH_MANAGER;
		serviceRequestParams.returnMsg = CHANNEL_SWITCH_MANAGER_RX_PAUSE_CFM;
		serviceRequestParams.vapIndex = GET_DEFAULT_VAP_FOR_MY_BAND();
		serviceRequestParams.serviceRequesterParams = NULL;
		serviceRequestParams.serviceId	= SERVICE_ID_PAUSE_RX;
		serviceRequestParams.serviceData = 0x0;
	
		ServicesHandler_RunService(&serviceRequestParams);
		break;
	default:
		DEBUG_FATAL("Channel Switch Manager: illegal event in current state");
	}
}


/**********************************************************************************

channelSwitchManager_TxPauseSpecificVapConfirm

Description:
------------

Input: 
-----
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void channelSwitchManager_TxPauseSpecificVapConfirm(K_MSG *channelSwitchManagerMessage)
{
	uint32 vapIdx;
	ServiceRequestParams_t serviceRequestParams;
	UNUSED_PARAM(channelSwitchManagerMessage);

	ILOG2_D("state: %d, event: TX Pause VAP Confirm", channelSwitchManagerParams.state);

	switch (channelSwitchManagerParams.state)
	{
	case CHANNEL_SWITCH_MANAGER_STATE_PRE_CHANNEL_SWITCH_WAIT_FOR_VAP_PAUSE:
		if (channelSwitchManagerParams.vapPauseBitmap)
		{
			/*Are there any pending VAPs for Pause Confirm*/
			vapIdx = Utils_FindFirstSet(channelSwitchManagerParams.vapPauseBitmap);
			channelSwitchManagerParams.vapPauseBitmap ^= (0x1 << vapIdx);
			//send pause VAP TX request
			serviceRequestParams.returnTask = TASK_CHANNEL_SWITCH_MANAGER;
			serviceRequestParams.returnMsg = CHANNEL_SWITCH_MANAGER_TX_PAUSE_SPECIFIC_VAP_CFM;
			serviceRequestParams.serviceRequesterParams = NULL;
			serviceRequestParams.serviceId	= SERVICE_ID_PAUSE_VAP_TX;
			serviceRequestParams.serviceData = vapIdx;
			serviceRequestParams.vapIndex = vapIdx;

			ServicesHandler_RunService(&serviceRequestParams);
			/*Se stau in this state*/
		}
		else if (channelSwitchManagerParams.activeSessionBitmap != 0x0)
		{
			/*Are there any pending VAPs for CSA Beacon Done Indication*/
			channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_PRE_CHANNEL_SWITCH_WAIT);
			break;
		}
		else
		{
			/*CSA Beacon Done Indication received from all VAPs*/
			//kill timer
			OSAL_RESET_TIMER_EXPLICIT(CHANNEL_SWITCH_MANAGER_TIMER_EXPIRED, TASK_CHANNEL_SWITCH_MANAGER);

			OSAL_SEND_NO_DATA_MESSAGE(CSA_MANAGER_STOP_TX, TASK_TX_MANAGER, GET_DEFAULT_VAP_FOR_MY_BAND());

			//move to next state
			channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_PRE_CHANNEL_SWITCH_WAIT_FOR_TX_PAUSE);

			//send pause all tx request
			serviceRequestParams.returnTask = TASK_CHANNEL_SWITCH_MANAGER;
			serviceRequestParams.returnMsg = CHANNEL_SWITCH_MANAGER_TX_PAUSE_CFM;
			serviceRequestParams.serviceRequesterParams = NULL;
			serviceRequestParams.serviceId	= SERVICE_ID_PAUSE_ALL_TX;
			serviceRequestParams.serviceData = 0x0;
			serviceRequestParams.vapIndex =VAP_ID_DO_NOT_CARE;

			ServicesHandler_RunService(&serviceRequestParams);
		}
		break;
	case CHANNEL_SWITCH_MANAGER_STATE_PRE_CHANNEL_SWITCH_WAIT_FOR_VAP_PAUSE_STOP:
		/*Clear Pending VAP bitmap*/
		channelSwitchManagerParams.vapPauseBitmap = 0;

		OSAL_SEND_NO_DATA_MESSAGE(CSA_MANAGER_STOP_TX, TASK_TX_MANAGER, GET_DEFAULT_VAP_FOR_MY_BAND());
		
		//move to next state
		channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_PRE_CHANNEL_SWITCH_WAIT_FOR_TX_PAUSE);
		
		//send pause all tx request
		serviceRequestParams.returnTask = TASK_CHANNEL_SWITCH_MANAGER;
		serviceRequestParams.returnMsg = CHANNEL_SWITCH_MANAGER_TX_PAUSE_CFM;
		serviceRequestParams.serviceRequesterParams = NULL;
		serviceRequestParams.serviceId	= SERVICE_ID_PAUSE_ALL_TX;
		serviceRequestParams.serviceData = 0x0;
		serviceRequestParams.vapIndex = VAP_ID_DO_NOT_CARE;
		
		ServicesHandler_RunService(&serviceRequestParams);
		break;
	default:
		DEBUG_FATAL("Channel Switch Manager: illegal event in current state");
	}
}



/**********************************************************************************

channelSwitchManager_RxPauseConfirm

Description:
------------

Input: 
-----
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void channelSwitchManager_RxPauseConfirm(K_MSG *channelSwitchManagerMessage)
{
	ILOG2_D("state: %d, event: Rx Pause Confirm", channelSwitchManagerParams.state);

	switch (channelSwitchManagerParams.state)
	{
	case CHANNEL_SWITCH_MANAGER_STATE_POST_SCAN_WAIT_FOR_RX_PAUSE:
		//prepare a bitmap of all gplp queues that need to be locked
		channelSwitchManagerParams.activeSessionBitmap = channelSwitchManagerParams.activeVapsBitmap;

		//move to next state
		channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_POST_SCAN_WAIT_FOR_TX_QUEUE_LOCK);

		//direct call to next state's handler
		channelSwitchManager_TxQueueLockConfirm(channelSwitchManagerMessage);
		break;
	case CHANNEL_SWITCH_MANAGER_STATE_PRE_CHANNEL_SWITCH_WAIT_FOR_RX_PAUSE:
		//move to next state
		channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_PRE_CHANNEL_SWITCH_WAIT_FOR_STOP_CSA_BEACONS);
		
		//send pop all Beacons request to Beacon Handler
		OSAL_SEND_NO_DATA_MESSAGE(PAC_MANAGER_BEACON_CSA_STOP, TASK_PAC_MANAGER, GET_DEFAULT_VAP_FOR_MY_BAND());	
		break;
	case CHANNEL_SWITCH_MANAGER_STATE_WAIT_FOR_RSSI_RX_CFM:
		//Set the channel
		channelSwitchManager_HdkSetChannel();
		break;
	default:
		DEBUG_FATAL("Channel Switch Manager: illegal event in current state");
	}
}


/**********************************************************************************

channelSwitchManager_TimerExpired


Description:
------------

Input: 
-----
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void channelSwitchManager_TimerExpired(K_MSG *channelSwitchManagerMessage)
{
	ServiceRequestParams_t serviceRequestParams;
	UNUSED_PARAM(channelSwitchManagerMessage);

	ILOG2_D("state: %d, event: Timer Expired", channelSwitchManagerParams.state);

	switch (channelSwitchManagerParams.state)
	{
	case CHANNEL_SWITCH_MANAGER_STATE_PRE_CHANNEL_SWITCH_WAIT:
		
		OSAL_SEND_NO_DATA_MESSAGE(CSA_MANAGER_STOP_TX, TASK_TX_MANAGER, GET_DEFAULT_VAP_FOR_MY_BAND());

		//move to next state
		channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_PRE_CHANNEL_SWITCH_WAIT_FOR_TX_PAUSE);

		//send pause all tx request
		serviceRequestParams.returnTask = TASK_CHANNEL_SWITCH_MANAGER;
		serviceRequestParams.returnMsg = CHANNEL_SWITCH_MANAGER_TX_PAUSE_CFM;
		serviceRequestParams.vapIndex = GET_DEFAULT_VAP_FOR_MY_BAND();
		serviceRequestParams.serviceRequesterParams = NULL;
		serviceRequestParams.serviceId	= SERVICE_ID_PAUSE_ALL_TX;
		serviceRequestParams.serviceData = 0x0;

		ServicesHandler_RunService(&serviceRequestParams);
		break;
	case CHANNEL_SWITCH_MANAGER_STATE_PRE_CHANNEL_SWITCH_WAIT_FOR_VAP_PAUSE:
		//change state so TX Pause VAP confirm is received we abort the rest of the VAPs
		channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_PRE_CHANNEL_SWITCH_WAIT_FOR_VAP_PAUSE_STOP);
		break;
	case CHANNEL_SWITCH_MANAGER_STATE_PRE_CHANNEL_SWITCH_WAIT_FOR_TX_PAUSE:
		/* This is an end case, in which both the beacon lock (for last beacon) & timeout event are in queue:
		Last beacon lock will move us to CHANNEL_SWITCH_MANAGER_STATE_PRE_CHANNEL_SWITCH_WAIT_FOR_TX_PAUSE state & kill timer.
		however, "timer expired" message is already in the thread's queue. 
		In this case, we should simply ignore the timer message.	*/
		break;
	default:
		DEBUG_FATAL("Channel Switch Manager: illegal event in current state");
	}
}


/**********************************************************************************

channelSwitchManager_BeaconStartCsaBeaconConfirm


Description:
------------

Input: 
-----
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void channelSwitchManager_BeaconStartCsaBeaconConfirm(K_MSG *channelSwitchManagerMessage)
{
	K_MSG* psMsg = NULL;
	CsaManagerStartTxReqParams_t *pCsaManagerStartTxRequestParams = NULL;
	UNUSED_PARAM(channelSwitchManagerMessage);

	ILOG0_D("state: %d, event: Beacon Add Csa Ie Confirm", channelSwitchManagerParams.state);

	switch (channelSwitchManagerParams.state)
	{
	case CHANNEL_SWITCH_MANAGER_STATE_PRE_CHANNEL_SWITCH_WAIT_FOR_BEACONS_UPDATE:

		//move to next state
		channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_PRE_CHANNEL_SWITCH_WAIT);

		//prepare a bitmap of all beacon queues that need to be locked
		channelSwitchManagerParams.activeSessionBitmap = channelSwitchManagerParams.activeVapsBitmap;

		//start timer
#if defined (ENET_INC_HW_FPGA)
		// Timeout on FPGA/palladium setup is too much. Need to decrease it - otherwise we get unconfirmed MSG on driver
		OSAL_SET_TIMER_EXPLICIT(CHANNEL_SWITCH_MANAGER_TIMER_EXPIRED, OSAL_TIMERS_MS_TO_K_TICKS(10), TASK_CHANNEL_SWITCH_MANAGER);
#else
		OSAL_SET_TIMER_EXPLICIT(CHANNEL_SWITCH_MANAGER_TIMER_EXPIRED, OSAL_TIMERS_MS_TO_K_TICKS(channelSwitchManagerParams.pUmSetChannelParams->chan_switch_time), TASK_CHANNEL_SWITCH_MANAGER);
#endif

		psMsg = OSAL_GET_MESSAGE(sizeof(CsaManagerStartTxReqParams_t));
		pCsaManagerStartTxRequestParams = ((CsaManagerStartTxReqParams_t *)psMsg->abData);
		pCsaManagerStartTxRequestParams->pUmSetChannelParams = channelSwitchManagerParams.pUmSetChannelParams;

		pCsaManagerStartTxRequestParams->chanSwitchTime = channelSwitchManagerParams.pUmSetChannelParams->chan_switch_time;
		pCsaManagerStartTxRequestParams->low_chan_num = channelSwitchManagerParams.actualLowChannel;
		pCsaManagerStartTxRequestParams->low_chan_num2 = channelSwitchManagerParams.pUmSetChannelParams->low_chan_num2;
		pCsaManagerStartTxRequestParams->chan_width = channelSwitchManagerParams.actualBW;
		pCsaManagerStartTxRequestParams->primary_chan_idx = channelSwitchManagerParams.actualPrimaryChannel;
		pCsaManagerStartTxRequestParams->block_tx = channelSwitchManagerParams.pUmSetChannelParams->block_tx_pre;
		pCsaManagerStartTxRequestParams->primaryChannel = Utils_GetPrimaryChannel(	pCsaManagerStartTxRequestParams->low_chan_num,
																									pCsaManagerStartTxRequestParams->low_chan_num2,
																									pCsaManagerStartTxRequestParams->chan_width,
																									pCsaManagerStartTxRequestParams->primary_chan_idx);
		pCsaManagerStartTxRequestParams->secChannelOffset = ChannelSwitchManager_GetSecondaryChannelOffset(	pCsaManagerStartTxRequestParams->low_chan_num,
																											pCsaManagerStartTxRequestParams->low_chan_num2,
																											pCsaManagerStartTxRequestParams->chan_width,
																											pCsaManagerStartTxRequestParams->primary_chan_idx);

		//notify csa manager to start sending csa action frames
		OSAL_SEND_MESSAGE(CSA_MANAGER_START_TX, TASK_TX_MANAGER, psMsg, GET_DEFAULT_VAP_FOR_MY_BAND());
		break;
	default:
		DEBUG_FATAL("Channel Switch Manager: illegal event in current state");
	}
}


/**********************************************************************************

channelSwitchManager_CsaBeaconDoneIndication


Description:
------------

Input: 
-----
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void channelSwitchManager_CsaBeaconDoneIndication(K_MSG *channelSwitchManagerMessage)
{
	ChannelSwitchManager_BeaconLockNotification_t *pBeaconLockNotification = NULL;
	uint32 vapIdx;
	ServiceRequestParams_t serviceRequestParams;

	ILOG2_D("state: %d, event: CSA Beacon Done Indication", channelSwitchManagerParams.state);

	switch (channelSwitchManagerParams.state)
	{
	case CHANNEL_SWITCH_MANAGER_STATE_PRE_CHANNEL_SWITCH_WAIT:

		pBeaconLockNotification = (ChannelSwitchManager_BeaconLockNotification_t *)pK_MSG_DATA(channelSwitchManagerMessage);
		vapIdx = pBeaconLockNotification->vapIdx;
		channelSwitchManagerParams.activeSessionBitmap ^= (0x1 << vapIdx);

		if (channelSwitchManagerParams.activeSessionBitmap != 0x0)
		{
			//send pause VAP TX request
			serviceRequestParams.returnTask = TASK_CHANNEL_SWITCH_MANAGER;
			serviceRequestParams.returnMsg = CHANNEL_SWITCH_MANAGER_TX_PAUSE_SPECIFIC_VAP_CFM;
			serviceRequestParams.vapIndex = vapIdx;
			serviceRequestParams.serviceRequesterParams = NULL;
			serviceRequestParams.serviceId	= SERVICE_ID_PAUSE_VAP_TX;
			serviceRequestParams.serviceData = vapIdx;

			ServicesHandler_RunService(&serviceRequestParams);

			channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_PRE_CHANNEL_SWITCH_WAIT_FOR_VAP_PAUSE);
		}
		else
		{
			//kill timer
			OSAL_RESET_TIMER_EXPLICIT(CHANNEL_SWITCH_MANAGER_TIMER_EXPIRED, TASK_CHANNEL_SWITCH_MANAGER);

			OSAL_SEND_NO_DATA_MESSAGE(CSA_MANAGER_STOP_TX, TASK_TX_MANAGER, GET_DEFAULT_VAP_FOR_MY_BAND());

			//move to next state
			channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_PRE_CHANNEL_SWITCH_WAIT_FOR_TX_PAUSE);

			//send pause all tx request
			serviceRequestParams.returnTask = TASK_CHANNEL_SWITCH_MANAGER;
			serviceRequestParams.returnMsg = CHANNEL_SWITCH_MANAGER_TX_PAUSE_CFM;
			serviceRequestParams.vapIndex = vapIdx;
			serviceRequestParams.serviceRequesterParams = NULL;
			serviceRequestParams.serviceId	= SERVICE_ID_PAUSE_ALL_TX;
			serviceRequestParams.serviceData = 0x0;

			ServicesHandler_RunService(&serviceRequestParams);
		}
		break;
	case CHANNEL_SWITCH_MANAGER_STATE_PRE_CHANNEL_SWITCH_WAIT_FOR_VAP_PAUSE:
	case CHANNEL_SWITCH_MANAGER_STATE_PRE_CHANNEL_SWITCH_WAIT_FOR_VAP_PAUSE_STOP:
		/*We can't send another pause command until the previous one is confirmed*/
		pBeaconLockNotification = (ChannelSwitchManager_BeaconLockNotification_t *)pK_MSG_DATA(channelSwitchManagerMessage);
		vapIdx = pBeaconLockNotification->vapIdx;
		channelSwitchManagerParams.activeSessionBitmap ^= (0x1 << vapIdx);
		channelSwitchManagerParams.vapPauseBitmap |= (0x1 << vapIdx);
		break;
	case CHANNEL_SWITCH_MANAGER_STATE_PRE_CHANNEL_SWITCH_WAIT_FOR_TX_PAUSE:
	case CHANNEL_SWITCH_MANAGER_STATE_PRE_CHANNEL_SWITCH_WAIT_FOR_STOP_CSA_BEACONS:
		pBeaconLockNotification = (ChannelSwitchManager_BeaconLockNotification_t *)pK_MSG_DATA(channelSwitchManagerMessage);
		vapIdx = pBeaconLockNotification->vapIdx;
		channelSwitchManagerParams.activeSessionBitmap ^= (0x1 << vapIdx);
		break;
	default:
		DEBUG_FATAL("Channel Switch Manager: illegal event in current state");
	}
}


/**********************************************************************************

channelSwitchManager_StopCsaBeaconsConfirm


Description:
------------

Input: 
-----
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void channelSwitchManager_StopCsaBeaconsConfirm(K_MSG *channelSwitchManagerMessage)
{
	ILOG2_D("state: %d, event: Stop CSA Beacons Confirm", channelSwitchManagerParams.state);
	UNUSED_PARAM(channelSwitchManagerMessage);

	switch (channelSwitchManagerParams.state)
	{
		case CHANNEL_SWITCH_MANAGER_STATE_PRE_CHANNEL_SWITCH_WAIT_FOR_STOP_CSA_BEACONS:			
			if ((channelSwitchManagerParams.csaManagerStartConfirm == TRUE) && (channelSwitchManagerParams.csaManagerStopConfirm == TRUE))
			{
				channelSwitchManagerParams.csaManagerStartConfirm = FALSE;
				channelSwitchManagerParams.csaManagerStopConfirm = FALSE;

				//Set the channel
				channelSwitchManager_HdkSetChannel();
			}
			else
			{
				//move to pending csa state
				channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_PRE_CHANNEL_SWITCH_WAIT_FOR_CSA_MANAGER);
			}
			break;
		default:
			DEBUG_FATAL("Channel Switch Manager: illegal event in current state");
	}
}


/**********************************************************************************

ChannelSwitchManager_TaskEntry 


Description:
------------
the entry point of the channel switch manager task

Input: 
-----
channelSwitchManagerMessage - pointer to the message to handle	
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
void ChannelSwitchManager_TaskEntry(K_MSG *channelSwitchManagerMessage)
{
	/* Use common task switching and Table */
	vTaskDispatcher(channelSwitchManagerMessage, afpTaskTable, TASK_CHANNEL_SWITCH_MANAGER_START, TASK_CHANNEL_SWITCH_MANAGER_END);
}


/**********************************************************************************

ChannelSwitchManager_Initialize 

Description:
------------
initialize the channel switch manager

Input:
-----

Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif

void ChannelSwitchManager_Initialize(void)
{
	memset(&channelSwitchManagerParams, 0x0, sizeof(channelSwitchManagerParams));
}
/**********************************************************************************

ChannelSwitchManager_PostInit

Description:
------------


Input:
-----

Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
void ChannelSwitchManager_PostInit(void)
{
	//move to idle state
	channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_IDLE);
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif


/**********************************************************************************

channelSwitchManager_VapManagerRegistrationEventAddVap

Description:
------------


Input:
-----

Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void channelSwitchManager_VapManagerRegistrationEventAddVap(K_MSG *channelSwitchManagerMessage)
{
	uint32 vapIdx;
	UMI_ADD_VAP* addVapStructurePtr = NULL;

	ILOG2_D("channelSwitchManager_VapManagerRegistrationEventAddVap, state: %d, event: Vap Manager Registration Event Add Vap", channelSwitchManagerParams.state);

	DEBUG_ASSERT(channelSwitchManagerParams.state == CHANNEL_SWITCH_MANAGER_STATE_IDLE);

	//KW_IGNORE ABV.GENERAL .. since abdata is never assigned, but used for typecast.
	addVapStructurePtr = (UMI_ADD_VAP*) EXTRACT_VAP_MANAGER_MSG(channelSwitchManagerMessage);

	vapIdx = addVapStructurePtr->vapId; 

	channelSwitchManagerParams.activeVapsBitmap |= (0x1 << vapIdx);

	if (addVapStructurePtr->operationMode == OPERATION_MODE_VSTA)
	{
		channelSwitchManagerParams.staModeVapsBitmap |= (0x1 << vapIdx);
	}

	/*return message*/
	FILL_VAP_MANAGER_CONFIRM_MSG(channelSwitchManagerMessage, vapIdx, VAP_MANAGER_ADD_VAP, BSS_MANAGER_VAP_MANAGER_CHANNEL_SWITCH_MANAGER_CLIENT);
	OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, channelSwitchManagerMessage, vapIdx);
}


/**********************************************************************************

channelSwitchManager_VapManagerRegistrationEventPostRemoveVap

Description:
------------


Input:
-----

Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void channelSwitchManager_VapManagerRegistrationEventPostRemoveVap(K_MSG *channelSwitchManagerMessage)
{
	uint32 vapIdx;
	UMI_REMOVE_VAP* removeVapStructurePtr = NULL;

	ILOG2_D("state: %d, event: Vap Manager Registration Event Post Remove Vap", channelSwitchManagerParams.state);

	DEBUG_ASSERT(channelSwitchManagerParams.state == CHANNEL_SWITCH_MANAGER_STATE_IDLE);

	//KW_IGNORE ABV.GENERAL .. since abdata is never assigned, but used for typecast.
	removeVapStructurePtr = (UMI_REMOVE_VAP *)EXTRACT_VAP_MANAGER_MSG(channelSwitchManagerMessage);

	vapIdx = removeVapStructurePtr->vapId; 

	channelSwitchManagerParams.activeVapsBitmap ^= (0x1 << vapIdx);
	channelSwitchManagerParams.staModeVapsBitmap &= (~(0x1 << vapIdx)); 

	/*return message*/
	FILL_VAP_MANAGER_CONFIRM_MSG(channelSwitchManagerMessage, vapIdx, VAP_MANAGER_REMOVE_VAP, BSS_MANAGER_VAP_MANAGER_CHANNEL_SWITCH_MANAGER_CLIENT);
	OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, channelSwitchManagerMessage, vapIdx);
}


/**********************************************************************************

channelSwitchManager_CsaManagerStartTxCfm

Description:
------------


Input:
-----

Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void channelSwitchManager_CsaManagerStartTxCfm(K_MSG *channelSwitchManagerMessage)
{
	ILOG2_D("state: %d, event: Csa Manager Start Tx Cfm", channelSwitchManagerParams.state);
	UNUSED_PARAM(channelSwitchManagerMessage);

	switch (channelSwitchManagerParams.state)
	{
	case CHANNEL_SWITCH_MANAGER_STATE_PRE_CHANNEL_SWITCH_WAIT_FOR_CSA_MANAGER:

		if (channelSwitchManagerParams.csaManagerStopConfirm == TRUE)
		{
			channelSwitchManagerParams.csaManagerStopConfirm = FALSE;

			//Set the channel
			channelSwitchManager_HdkSetChannel();
		}
		else
		{
			channelSwitchManagerParams.csaManagerStartConfirm = TRUE;
		}
		break;
	default:
		channelSwitchManagerParams.csaManagerStartConfirm = TRUE;
		break;
	}
}


/**********************************************************************************

channelSwitchManager_ProcessInterruptedIndication

Description:
------------


Input:
-----

Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void channelSwitchManager_ProcessInterruptedIndication(K_MSG *channelSwitchManagerMessage)
{
	UNUSED_PARAM(channelSwitchManagerMessage);
	DEBUG_FATAL("channelSwitchManager_ProcessInterruptedIndication");
}


/**********************************************************************************

channelSwitchManager_CsaManagerStopTxCfm

Description:
------------


Input:
-----

Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void channelSwitchManager_CsaManagerStopTxCfm(K_MSG *channelSwitchManagerMessage)
{
	ILOG2_D("state: %d, event: Csa Manager Stop Tx Cfm", channelSwitchManagerParams.state);
	UNUSED_PARAM(channelSwitchManagerMessage);
	switch (channelSwitchManagerParams.state)
	{
	case CHANNEL_SWITCH_MANAGER_STATE_PRE_CHANNEL_SWITCH_WAIT_FOR_CSA_MANAGER:

		if (channelSwitchManagerParams.csaManagerStartConfirm == TRUE)
		{
			channelSwitchManagerParams.csaManagerStartConfirm = FALSE;

			//Set the channel
			channelSwitchManager_HdkSetChannel();
		}
		else
		{
			channelSwitchManagerParams.csaManagerStopConfirm = TRUE;
		}
		break;
	default:
		channelSwitchManagerParams.csaManagerStopConfirm = TRUE;
		break;
	}
}

/**********************************************************************************

channelSwitchManager_SendConfirmationToHost

Description:
------------


Input:
-----

Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void channelSwitchManager_SendConfirmationToHost(void)
{
	uint8 	indexAnt;
	uint16 	staId;
	uint8 	vapId = GET_DEFAULT_VAP_FOR_MY_BAND();
	UM_SET_CHAN_PARAMS *pUmSetChannelParamsLocal = NULL;
	uint8 bandId = ConfigurationManager_GetMyBand();
	pUmSetChannelParamsLocal = channelSwitchManagerParams.pUmSetChannelParams;
	staId = pUmSetChannelParamsLocal->u16SID;

	if (pUmSetChannelParamsLocal->switch_type == ST_NORMAL)
	{
		OSAL_SEND_NO_DATA_MESSAGE(PAC_MANAGER_BEACON_BLOCKING_START, TASK_PAC_MANAGER, vapId);
	}

	
	//Copy RSSI metrics from SHRAM to CFM MSG
	if (pUmSetChannelParamsLocal->switch_type == ST_NORMAL_AFTER_RSSI)
	{
		for (indexAnt = 0; indexAnt < PHY_STATISTICS_MAX_RX_ANT; indexAnt++)
		{
			pUmSetChannelParamsLocal->rssi[indexAnt] = phyRxStatusDb.staPhyRxStatus[staId].rssi[indexAnt];
			/*Return the noise, it is OK to return the "global" noise because when we set the channel we reset it and
			in RSSI mode we receive packets only from the "unconnected" station*/
#ifdef ENET_INC_ARCH_WAVE600

			pUmSetChannelParamsLocal->noise[indexAnt] = phyRxStatusDb.devicePhyRxStatus[bandId].noise[indexAnt];
			pUmSetChannelParamsLocal->rf_gain[indexAnt] = phyRxStatusDb.devicePhyRxStatus[bandId].rf_gain[indexAnt];
#else

			pUmSetChannelParamsLocal->noise[indexAnt] = phyRxStatusDb.devicePhyRxStatus.noise[indexAnt];
			pUmSetChannelParamsLocal->rf_gain[indexAnt] = phyRxStatusDb.devicePhyRxStatus.rf_gain[indexAnt];
#endif
#ifdef BEEROCK_DEBUG
			ILOG0_DD("[channelSwitchManager_SendConfirmationToHost] channelSwitchManagerParams.pUmSetChannelParams->rssi[%d] = %d", indexAnt, pUmSetChannelParamsLocal->rssi[indexAnt]);
#endif
		}	
		pUmSetChannelParamsLocal->rate = phyRxStatusDb.staPhyRxStatus[staId].phyRate;
	}

	channelSwitchManager_ChannelSwitchDone();

#ifdef BEEROCK_DEBUG
	ILOG0_V("[channelSwitchManager_SendConfirmationToHost] --------------------------end]");
#endif

	if (pUmSetChannelParamsLocal->switch_type == ST_CSA)
	{
		channelSwitchManagerParams.pUmSetChannelParams->antennaMask = Hdk_GetTxAntMask();
	}

	if (pUmSetChannelParamsLocal->switch_type == ST_NORMAL)
	{
		if (channelSwitchManagerParams.pUmSetChannelParams->antennaMask == 0)
		{
			channelSwitchManagerParams.pUmSetChannelParams->antennaMask = Hdk_GetTxAntMask();
		}
	}
	//send confirmation to host
	OSAL_SEND_MESSAGE(UMI_UM_SET_CHAN_CFM, TASK_UM_IF_TASK, channelSwitchManagerParams.pSetChannelRequest, vapId);
}

/**********************************************************************************
channelSwitchManager_ChannelSwitchDone

Description:
------------
	channel switch done indication . zero stat
Input:
-----
	channel switch time
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
static void channelSwitchManager_ChannelSwitchDone(void)
{
	uint32	currTsf = 0;	
	uint8 	channelSwitchType = channelSwitchManagerParams.pUmSetChannelParams->switch_type;

#ifdef BEEROCK_DEBUG
	ILOG0_V("[channelSwitchManager_ChannelSwitchDone]");	
#endif	

	if (channelSwitchManagerChannelLoadDb.timerActive == TRUE)
	{
		OSAL_RESET_TIMER_EXPLICIT(CHANNEL_SWITCH_MANAGER_CHANNEL_LOAD_TIMER_EXPIRED, TASK_STATISTICS_MANAGER);// disable timer
	}

	if (channelSwitchType != ST_SCAN)	// while scan we dont need to use channel load timer. the driver should request it by msg.
	{
		OSAL_SET_TIMER_EXPLICIT( CHANNEL_SWITCH_MANAGER_CHANNEL_LOAD_TIMER_EXPIRED, OSAL_TIMERS_MS_TO_K_TICKS( CHANNEL_LOAD_TIMER_VALUE ), TASK_CHANNEL_SWITCH_MANAGER );	// trigger timer				
		channelSwitchManagerChannelLoadDb.timerActive = TRUE;
	}
	else
	{
		channelSwitchManagerChannelLoadDb.timerActive = FALSE;
	}

	// reset stat 
#ifndef ENET_INC_ARCH_WAVE600
	memset(&phyRxStatusDb.devicePhyRxStatus, 0 , sizeof(devicePhyRxStatusDb_t));
	PhyStatistics_resetStaPhyRxStatus();
#endif


	currTsf = Pac_TimGetTsfLow();
	Pac_TimResetQbss2Integrator();// reset channel load register
	Pac_TimResetQbssIntegrator();// reset channel load register
	Pac_DurResetTotalAirTime();
	channelSwitchManagerChannelLoadDb.totalChannelLoad = 0;
	channelSwitchManagerChannelLoadDb.totalChannelUtilization = 0;
	channelSwitchManagerChannelLoadDb.totalChannelLoadBssLoad = 0;
	channelSwitchManagerChannelLoadDb.channelLoadMesStartTime = currTsf; 
	channelSwitchManagerChannelLoadDb.channelLoadMesStartTimeBssLoad = currTsf;
}


/**********************************************************************************
channelSwitchManager_ChannelLoadTimerExpired  

Description:
------------
	read channel load, add the value to the total channel load and reset the register.
Input:
-----
	msg from Timer
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
static void channelSwitchManager_ChannelLoadTimerExpired(K_MSG* channelSwitchManagerMessage)
{
	uint32 	chLoad;
	uint32 	chLoad2;
	uint8 	totalChannelUtilization;
	uint32	currentTime;
	uint32	deltaTime;
	UNUSED_PARAM(channelSwitchManagerMessage);
	chLoad = Pac_TimReadQbssIntegrator();// read HW channel load register
	chLoad2 = Pac_TimReadQbss2Integrator();// read HW channel utilization register
	currentTime = Pac_TimGetTsfLow();
 	Pac_TimResetQbss2Integrator(); // reset HW channel load register
 	Pac_TimResetQbssIntegrator(); // reset HW channel utrilization register
	
	channelSwitchManagerChannelLoadDb.totalChannelLoad += chLoad;// add HW channel load value to total channel load value
	channelSwitchManagerChannelLoadDb.totalChannelLoadBssLoad += chLoad2;// add HW channel load value to total channel load value
	channelSwitchManagerChannelLoadDb.totalChannelUtilization += chLoad2;// accumulate total channel utilization

	// update channel load IE
	deltaTime = currentTime - channelSwitchManagerChannelLoadDb.channelLoadMesStartTimeBssLoad;
	if (channelSwitchManagerChannelLoadDb.totalChannelLoadBssLoad > deltaTime)
	{
		deltaTime = channelSwitchManagerChannelLoadDb.totalChannelLoadBssLoad;
	}
	totalChannelUtilization = (channelSwitchManagerChannelLoadDb.totalChannelLoadBssLoad * HUNDRED_PERCENT)/(deltaTime);

	ILOG2_DDDD("PhyStatistics_ChannelLoadTimerExpired, phyStatisticDb.totalChannelLoadBssLoad:%d, currentTime:%d, phyStatisticDb.channelLoadMesStartTimeBssLoad:%d, channelUtilization:%d", 
				channelSwitchManagerChannelLoadDb.totalChannelLoadBssLoad, 
				currentTime, 
				channelSwitchManagerChannelLoadDb.channelLoadMesStartTimeBssLoad,
				totalChannelUtilization);

	channelSwitchManagerChannelLoadDb.channelLoadMesStartTimeBssLoad = currentTime;
	channelSwitchManagerChannelLoadDb.totalChannelLoadBssLoad = 0;

	//update the bss load with the channel utilization
	channelSwitchManager_UpdateBssLoadIeChannelUtilization(totalChannelUtilization);

	/*If we started a scan we do not want to set the timer*/
	if (channelSwitchManagerChannelLoadDb.timerActive == TRUE)
	{
		OSAL_SET_TIMER_EXPLICIT( CHANNEL_SWITCH_MANAGER_CHANNEL_LOAD_TIMER_EXPIRED, OSAL_TIMERS_MS_TO_K_TICKS( CHANNEL_LOAD_TIMER_VALUE ), TASK_CHANNEL_SWITCH_MANAGER );	// trigger timer				
	}
}

/**********************************************************************************
channelSwitchManager_UpdateBssLoadIeChannelUtilization



Description:
------------
	calculate channel load
Input:
-----
	
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
static void channelSwitchManager_UpdateBssLoadIeChannelUtilization(uint8 channelUtilization)
{
	uint8 vapId = 0;
#ifdef ENET_INC_ARCH_WAVE600
	uint8 numOfVapsInBand = ConfigurationManager_GetNumOfVapsInMyBand();
	uint8 firstVapInBand = ConfigurationManager_GetFirstVapForMyBand();
#else
	uint8 numOfVapsInBand = HW_NUM_OF_VAPS;//ConfigurationManager_GetNumOfVapsInMyBand();
	uint8 firstVapInBand = 0;//ConfigurationManager_GetFirstVapForMyBand();
#endif
	uint8 lastVapInBand = firstVapInBand + numOfVapsInBand - 1;
	
	// todo: go over all vaps and update only active vaps
	for (vapId = firstVapInBand; vapId <= lastVapInBand; vapId++)
	{
		bssLoadIePayload[vapId].bssLoad.channelUtilization = channelUtilization;
	}
}

    
/***********************************************************************
* channelSwitchManager_GetBusyRx
* 
* Description:
* ------------
* This function gets the phy air time of all the packets decoded by phy
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
uint32 channelSwitchManager_GetBusyRx(void)
{
	uint32 phyAirTime = 0;
	RegAccess_Read(PHY_AIR_TIME, &phyAirTime);
	return(phyAirTime);
}

/**********************************************************************************
channelSwitchManager_CalcChannelLoad


Description:
------------
	calculate channel load
Input:
-----
	
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void channelSwitchManager_CalcChannelLoad(void)
{
	uint32 	chLoad;
	uint32	chLoad2;
	uint32 	currTsf; 
	uint32	rxToMeAirTime = 0;
	uint32 	deltaTime;
	uint8	totalChannelUtilization;
	uint8 	bandId = ConfigurationManager_GetMyBand();

	chUtilization_t	*chUtilization = &(ChannelStatistics.channelUtilizationStats[bandId]);

	if (channelSwitchManagerChannelLoadDb.timerActive == TRUE)
	{
		OSAL_RESET_TIMER_EXPLICIT(CHANNEL_SWITCH_MANAGER_CHANNEL_LOAD_TIMER_EXPIRED, TASK_CHANNEL_SWITCH_MANAGER);// disable timer
	}

	rxToMeAirTime = Pac_DurGetTotalAirTime();
	chLoad = Pac_TimReadQbssIntegrator(); 	// read HW channel load register;
	chLoad2 = Pac_TimReadQbss2Integrator();	// read HW channel utilization register;
	currTsf = Pac_TimGetTsfLow(); 

 	Pac_TimResetQbss2Integrator(); 	// reset HW channel utilization register
	Pac_TimResetQbssIntegrator(); 	// reset HW channel load register
	Pac_DurResetTotalAirTime();
	
	channelSwitchManagerChannelLoadDb.totalChannelLoad += chLoad;// add HW channel load value to total channel load value
	channelSwitchManagerChannelLoadDb.totalChannelLoadBssLoad += chLoad2;// accumulate BSS load
	channelSwitchManagerChannelLoadDb.totalChannelUtilization += chLoad2;// accumulate total channel utilization

	//a simply channel load protection case
	if (channelSwitchManagerChannelLoadDb.totalChannelLoad < rxToMeAirTime)
	{
		rxToMeAirTime = channelSwitchManagerChannelLoadDb.totalChannelLoad;
	}
	deltaTime = currTsf - channelSwitchManagerChannelLoadDb.channelLoadMesStartTime;
	if (channelSwitchManagerChannelLoadDb.totalChannelLoad > deltaTime)
	{
		deltaTime = channelSwitchManagerChannelLoadDb.totalChannelLoad;
	}
	
#ifdef ENET_INC_ARCH_WAVE600
	phyRxStatusDb.devicePhyRxStatus[bandId].channel_load =	(((channelSwitchManagerChannelLoadDb.totalChannelLoad - rxToMeAirTime)*HUNDRED_PERCENT)/(deltaTime));// calc channel load
#else
	phyRxStatusDb.devicePhyRxStatus.channel_load =	(((channelSwitchManagerChannelLoadDb.totalChannelLoad - rxToMeAirTime)*HUNDRED_PERCENT)/(deltaTime));// calc channel load
#endif
	//a simply channel utilization protection case
	if (channelSwitchManagerChannelLoadDb.totalChannelUtilization < channelSwitchManagerChannelLoadDb.totalChannelLoad)
	{
		channelSwitchManagerChannelLoadDb.totalChannelUtilization = channelSwitchManagerChannelLoadDb.totalChannelLoad;
	}

	if (channelSwitchManagerChannelLoadDb.totalChannelUtilization > deltaTime)
	{
		deltaTime = channelSwitchManagerChannelLoadDb.totalChannelUtilization;
	}
	// calculate total channel utilization
	totalChannelUtilization = (channelSwitchManagerChannelLoadDb.totalChannelUtilization * HUNDRED_PERCENT)/(deltaTime);
#ifdef ENET_INC_ARCH_WAVE600
	phyRxStatusDb.devicePhyRxStatus[bandId].totalChannelUtilization = totalChannelUtilization;		
#else
	phyRxStatusDb.devicePhyRxStatus.totalChannelUtilization = totalChannelUtilization;		
#endif

	/* chUtilizationBusyRx = OWN RX TIME + RX not to me (provided by phy) */
	chUtilization->chUtilizationBusyRx = channelSwitchManager_GetBusyRx();

	/* ch_utilization_busy_tx = TX TIME + RX TIME = QBSS2 - QBSS1 + RX TIME 
	   Note: QBSS1 = NAV+CCA 
	   		 QBSS2 = NAV+CCA +TX TIME*/

	/* ch_utilization_busy_self = RX TIME */
	chUtilization->chUtilizationBusySelf = rxToMeAirTime;

	/* ch_utilization_busy = QBSS2 + OWN RX TIME */
	chUtilization->chUtilizationBusy = channelSwitchManagerChannelLoadDb.totalChannelUtilization + chUtilization->chUtilizationBusySelf;

	/* ch_utilization_busy_tx = QBSS2 - QBSS1 */
	chUtilization->chUtilizationBusyTx = channelSwitchManagerChannelLoadDb.totalChannelUtilization - channelSwitchManagerChannelLoadDb.totalChannelLoad;

	/* ch_utilization_total = TX TIME + RX TIME */
	chUtilization->chUtilizationTotal = chUtilization->chUtilizationBusyTx + chUtilization->chUtilizationBusySelf;

	if (channelSwitchManagerChannelLoadDb.timerActive == TRUE)
	{
		OSAL_SET_TIMER_EXPLICIT( CHANNEL_SWITCH_MANAGER_CHANNEL_LOAD_TIMER_EXPIRED, OSAL_TIMERS_MS_TO_K_TICKS( CHANNEL_LOAD_TIMER_VALUE ), TASK_CHANNEL_SWITCH_MANAGER );	// trigger timer				
	}
	else
	{
		// If timer is not active clear BSS load
		channelSwitchManagerChannelLoadDb.totalChannelLoadBssLoad = 0;
	}

	// zero ch load calculation params
	channelSwitchManagerChannelLoadDb.channelLoadMesStartTime = currTsf;
	channelSwitchManagerChannelLoadDb.totalChannelLoad = 0;
	channelSwitchManagerChannelLoadDb.totalChannelUtilization = 0;
}
/**********************************************************************************
channelSwitchManager_ChannelLoadReq  

Description:
------------
	send the channel load value to driver
Input:
-----
	msg from Driver
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
static void channelSwitchManager_ChannelLoadReq(K_MSG* channelSwitchManagerMessage)
{
	UMI_GET_CHANNEL_LOAD_REQ* pGetChannelLoad;
	uint8 bandId = ConfigurationManager_GetMyBand();

    pGetChannelLoad = (UMI_GET_CHANNEL_LOAD_REQ *)pK_MSG_DATA(channelSwitchManagerMessage);
	/*Calculate the channel load*/
	channelSwitchManager_CalcChannelLoad();
#ifdef ENET_INC_ARCH_WAVE600
	pGetChannelLoad->channelLoad = phyRxStatusDb.devicePhyRxStatus[bandId].channel_load; 
#else
	pGetChannelLoad->channelLoad = phyRxStatusDb.devicePhyRxStatus.channel_load; 
#endif
	// send confirmarion to driver
	OSAL_SEND_MESSAGE( UMI_MC_MAN_GET_CHANNEL_LOAD_CFM, TASK_UM_IF_TASK, channelSwitchManagerMessage, channelSwitchManagerMessage->header.vapId);
}

#ifdef ENET_INC_ARCH_WAVE600	
/**********************************************************************************
channelSwitchManager_ChannelLoadReq  

Description:
------------
	send the channel load value to driver
Input:
-----
	msg from Driver
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
static void channelSwitchManager_CalculateChannelLoadForPhyStatusReq(K_MSG* channelSwitchManagerMessage)
{
	//In gen6 we call it here since it is done in LM. In gen5 we already calculated it in UM.
	channelSwitchManager_CalcChannelLoad();

	// send confirmarion to driver
	OSAL_SEND_MESSAGE(STATISTICS_MANAGER_CHANNEL_LOAD_CALCULATED, TASK_STATISTICS_MANAGER, channelSwitchManagerMessage, channelSwitchManagerMessage->header.vapId);
	
}
#endif


/**********************************************************************************

channelSwitchManager_SetRssiCfm


Description:
------------


Input:
-----

Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void channelSwitchManager_SetRssiCfm(K_MSG *channelSwitchManagerMessage)
{	
	ServiceRequestParams_t serviceRequestParams;
		
	//resume Rx
	serviceRequestParams.returnTask = TASK_CHANNEL_SWITCH_MANAGER;
	serviceRequestParams.returnMsg = CHANNEL_SWITCH_MANAGER_SET_RSSI_RX_RESUME_CFM;
	serviceRequestParams.serviceRequesterParams = NULL;
	serviceRequestParams.serviceId	= SERVICE_ID_RESUME_RX;
	serviceRequestParams.serviceData = 0x0;
	serviceRequestParams.vapIndex = channelSwitchManagerMessage->header.vapId;

	ServicesHandler_RunService(&serviceRequestParams);
}	

/**********************************************************************************

channelSwitchManager_HdkSetChannel




Description:
------------
Notidy HDK

Input:
-----

Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void channelSwitchManager_HdkSetChannel()
{
	K_MSG* psMsg = NULL;
	HdkSetChannelReqParams_t *pHdkSetChannelRequestParams = NULL;
	
	//send set channel request to hdk
	psMsg = OSAL_GET_MESSAGE(sizeof(HdkSetChannelReqParams_t));
	pHdkSetChannelRequestParams = ((HdkSetChannelReqParams_t *)psMsg->abData);
	pHdkSetChannelRequestParams->low_chan_num 							= channelSwitchManagerParams.pUmSetChannelParams->low_chan_num;
	pHdkSetChannelRequestParams->low_chan_num2 							= channelSwitchManagerParams.pUmSetChannelParams->low_chan_num2;
	pHdkSetChannelRequestParams->chan_width 							= channelSwitchManagerParams.pUmSetChannelParams->chan_width;
	pHdkSetChannelRequestParams->primary_chan_idx 						= channelSwitchManagerParams.pUmSetChannelParams->primary_chan_idx;
	pHdkSetChannelRequestParams->isContinuousInterfererDetectionNeeded 	= channelSwitchManagerParams.pUmSetChannelParams->isContinuousInterfererDetectionNeeded;
	pHdkSetChannelRequestParams->isRadarDetectionNeeded 				= channelSwitchManagerParams.pUmSetChannelParams->isRadarDetectionNeeded;;
	pHdkSetChannelRequestParams->switchType 							= channelSwitchManagerParams.pUmSetChannelParams->switch_type;
	pHdkSetChannelRequestParams->channelNotification					= channelSwitchManagerParams.pUmSetChannelParams->ChannelNotificationMode;
	pHdkSetChannelRequestParams->block_tx_post							= channelSwitchManagerParams.pUmSetChannelParams->block_tx_post;
	pHdkSetChannelRequestParams->RegulationType 						= channelSwitchManagerParams.pUmSetChannelParams->RegulationType;

	//move to next state
	channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_WAIT_FOR_SET_CHANNEL_CFM_FROM_HDK);
	//send message
	OSAL_SEND_MESSAGE(HDK_SET_CHANNEL_REQ, TASK_HDK, psMsg, GET_DEFAULT_VAP_FOR_MY_BAND());
}

/**********************************************************************************

channelSwitchManager_SmpsNotify


Description:
------------
Notidy SMPS manager

Input:
-----

Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void channelSwitchManager_SmpsNotify()
{
	K_MSG *channelMsg = NULL; // required for messaging the channel width to the SMPS
	cocChannelWidth_t *cocChannelWidth_p = NULL; // required for messaging the channel width to the SMPS

	//increment notify REQ count
	channelSwitchManagerParams.notifyReqCnt++;
	// move the channel width information to the SMPS manager by message
	channelMsg = OSAL_GET_MESSAGE(sizeof(cocChannelWidth_t));
	cocChannelWidth_p = (cocChannelWidth_t *)pK_MSG_DATA(channelMsg);
	cocChannelWidth_p->channelWidth = channelSwitchManagerParams.actualBW;
	OSAL_SEND_MESSAGE(SMPS_CHANNEL_WIDTH, TASK_TX_MANAGER, channelMsg, GET_DEFAULT_VAP_FOR_MY_BAND());
}

#ifdef ENET_INC_ARCH_WAVE600
/**********************************************************************************

channelSwitchManager_PacNotify




Description:
------------
Notidy PAC manager

Input:
-----

Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void channelSwitchManager_PacNotify()
{
	K_MSG *pacMsg_p = NULL;
	PacMngSetChannelParams_t *pacMngSetChannelInd = NULL;
	
	//increment notify REQ count
	channelSwitchManagerParams.notifyReqCnt++;
	pacMsg_p = OSAL_GET_MESSAGE(sizeof(PacMngSetChannelParams_t));
	pacMngSetChannelInd = ((PacMngSetChannelParams_t*)pacMsg_p->abData);
	pacMngSetChannelInd->bw = channelSwitchManagerParams.pUmSetChannelParams->chan_width;
	pacMngSetChannelInd->sub_band = channelSwitchManagerParams.pUmSetChannelParams->primary_chan_idx;
	OSAL_SEND_MESSAGE(PAC_MANAGER_SET_CHANNEL_IND, TASK_PAC_MANAGER, pacMsg_p, GET_DEFAULT_VAP_FOR_MY_BAND());
}

/**********************************************************************************

channelSwitchManager_StatsNotify



Description:
------------
Notidy Stats manager

Input:
-----

Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void channelSwitchManager_StatsNotify()
{
	//increment notify REQ count
	channelSwitchManagerParams.notifyReqCnt++;
	OSAL_SEND_NO_DATA_MESSAGE(STATISTICS_MANAGER_CHANNEL_SWITCH_NOTIFY, TASK_STATISTICS_MANAGER, GET_DEFAULT_VAP_FOR_MY_BAND());
}
#endif

/**********************************************************************************

channelSwitchManager_StatsNotify



Description:
------------
Notidy Stats manager

Input:
-----

Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void channelSwitchManager_NotifyModules()
{
	channelSwitchManagerParams.notifyReqCnt = 0;
	channelSwitchManagerParams.notifyCfmCnt = 0;
	/*Notify SMPS manager - in case OMN is used need to know channel width*/
	channelSwitchManager_SmpsNotify();
#ifdef ENET_INC_ARCH_WAVE600
	/*Notify PAC manager to set Band to sub band mapping*/
	channelSwitchManager_PacNotify();
	/*Notify Stats manager in order to clear PHY status*/
	channelSwitchManager_StatsNotify();
#endif
	//move to next state
	channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_WAIT_FOR_NOTIFY_CFM);
}

/**********************************************************************************

channelSwitchManager_SetRssiRxResumeCfm



Description:
------------


Input:
-----

Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void channelSwitchManager_SetRssiRxResumeCfm(K_MSG *channelSwitchManagerMessage)
{	
	UNUSED_PARAM(channelSwitchManagerMessage);
	ASSERT(channelSwitchManagerParams.state == CHANNEL_SWITCH_MANAGER_STATE_WAIT_FOR_RSSI_SET_CFM);
	
	channelSwitchManager_ChangeState(CHANNEL_SWITCH_MANAGER_STATE_RSSI);
	channelSwitchManager_SendConfirmationToHost();	
}


/**********************************************************************************

ChannelSwitchManager_GetSecondaryChannelOffset

Description:
------------


Input:
-----

Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
uint8 ChannelSwitchManager_GetSecondaryChannelOffset(uint8 low_chan_num, uint8 low_chan_num2, uint8 chan_width, uint8 primary_chan_idx)
{
	uint8 secondaryChannelOffset = SECONDARY_CHANNEL_NONE;
	UNUSED_PARAM(low_chan_num);
	UNUSED_PARAM(low_chan_num2);	
	switch (chan_width)
	{
	case BANDWIDTH_TWENTY:
		secondaryChannelOffset = SECONDARY_CHANNEL_NONE;
		break;
	case BANDWIDTH_FOURTY:
	case BANDWIDTH_EIGHTY:
	case BANDWIDTH_ONE_HUNDRED_SIXTY:
		if ((primary_chan_idx & 0x1) == 0x1)
		{
			secondaryChannelOffset = SECONDARY_CHANNEL_BELOW;
		}
		else
		{
			secondaryChannelOffset = SECONDARY_CHANNEL_ABOVE;
		}
		break;
	case BANDWIDTH_EIGHTY_EIGHTY:
		DEBUG_FATAL("Utils_GetSecondayChannelOffset: 80+80 not supported");
		break;
	default:
		DEBUG_FATAL("Utils_GetSecondayChannelOffset: Illegal rate");
	}

	return secondaryChannelOffset;
}


/**********************************************************************************

ChannelSwitchManager_GetChannelWidth


Description:
------------


Input:
-----

chan_width - as received from Driver


Output:
-------
channel Width that complies with Channel Switch Announcement (see "Table 9-252VHT Operation Information subfields" in IEEE 802.11-2016.pdf)

**********************************************************************************/
uint8 ChannelSwitchManager_GetChannelWidth(uint8 chan_width)
{
	uint8 channelWidth = 0;

	switch (chan_width)
	{
		case BANDWIDTH_TWENTY:
		case BANDWIDTH_FOURTY:
			channelWidth = 0;
			break;
		case BANDWIDTH_EIGHTY:
			channelWidth = 1;
			break;
		case BANDWIDTH_ONE_HUNDRED_SIXTY:
#ifdef ENET_INC_ARCH_WAVE600
			channelWidth = 1;
#else	
			FATAL("ChannelSwitchManager_GetChannelWidth: 160 not supported");
#endif
			break;
		case BANDWIDTH_EIGHTY_EIGHTY:
			FATAL("ChannelSwitchManager_GetChannelWidth: 80+80 not supported");
			break;
		default:
			FATAL("ChannelSwitchManager_GetChannelWidth: Illegal channel width");
			break;
	}

//	ILOG0_DD("ChannelSwitchManager_GetChannelWidth chan_width %d, channelWidth %d", chan_width, channelWidth);

	return channelWidth;
}

/**********************************************************************************

ChannelSwitchManager_GetCenterFreq0Idx


Description:
------------


Input:
-----

chan_width, low_chan_num - as received from Driver


Output:
-------
center Frequency  

**********************************************************************************/
#ifdef ENET_INC_ARCH_WAVE600
uint8 ChannelSwitchManager_GetCenterFreq0Idx(uint8 low_chan_num, uint8 chan_width, uint8 primary_chan_idx)
#else
uint8 ChannelSwitchManager_GetCenterFreq0Idx(uint8 low_chan_num, uint8 chan_width)
#endif
{
	uint8 centerFreq0Idx = 0;
	
	switch (chan_width)
	{
		case BANDWIDTH_TWENTY:
		case BANDWIDTH_FOURTY:
			centerFreq0Idx = 0;
			break;
		case BANDWIDTH_EIGHTY:
			centerFreq0Idx = low_chan_num + BW_80_OFFSET_TO_FREQ_INDEX;
			break;
		case BANDWIDTH_ONE_HUNDRED_SIXTY:
#ifdef ENET_INC_ARCH_WAVE600
			//channel center frequency index of the higher 80 MHz channel segment that contains the primary channel
			if(primary_chan_idx >= BW_160_PRIMARY_CHANNEL_HIGH_START_INDEX)
			{
				centerFreq0Idx = low_chan_num + BW_80_HIGH_OFFSET_TO_FREQ_INDEX;
			}
			else //channel center frequency index of the lower 80 MHz channel segment that contains the primary channel
			{
				centerFreq0Idx = low_chan_num + BW_80_OFFSET_TO_FREQ_INDEX;
			}
#else
			FATAL("ChannelSwitchManager_GetCenterFreq0Idx: 160 not supported");
#endif
			break;
		case BANDWIDTH_EIGHTY_EIGHTY:
			FATAL("ChannelSwitchManager_GetCenterFreq0Idx: 80+80 not supported");
			break;
		default:
			FATAL("ChannelSwitchManager_GetCenterFreq0Idx: Illegal channel width");
			break;
	}	

//	ILOG0_DDD("ChannelSwitchManager_GetCenterFreq0Idx chan_width %d, centerFreq0Idx %d primary_chan_idx %d", chan_width, centerFreq0Idx, primary_chan_idx);
	
	return centerFreq0Idx;
}

#ifdef ENET_INC_ARCH_WAVE600
/**********************************************************************************

ChannelSwitchManager_GetCenterFreq1Idx


Description:
------------


Input:
-----

chan_width, low_chan_num - as received from Driver


Output:
-------
center Frequency  

**********************************************************************************/
uint8 ChannelSwitchManager_GetCenterFreq1Idx(uint8 low_chan_num, uint8 chan_width)
{
	uint8 centerFreq1Idx = 0;
	
	switch (chan_width)
	{
		case BANDWIDTH_TWENTY:
		case BANDWIDTH_FOURTY:
		case BANDWIDTH_EIGHTY:
			centerFreq1Idx = 0;
			break;
		case BANDWIDTH_ONE_HUNDRED_SIXTY:
			centerFreq1Idx = low_chan_num + BW_160_OFFSET_TO_FREQ_INDEX;
			break;
		case BANDWIDTH_EIGHTY_EIGHTY:
			FATAL("ChannelSwitchManager_GetCenterFreq1Idx: 80+80 not supported");
			break;
		default:
			FATAL("ChannelSwitchManager_GetCenterFreq1Idx: Illegal channel width");
			break;
	}	
//	ILOG0_DD("ChannelSwitchManager_GetCenterFreq1Idx chan_width %d, centerFreq1Idx %d", chan_width, centerFreq1Idx);
	
	return centerFreq1Idx;
}
#endif //ENET_INC_ARCH_WAVE600

/**********************************************************************************
ChannelSwitchManager_CalcActualChannel


Description:
------------


Input:
-----

chan_width, low_chan_num - as received from Driver


Output:
-------
center Frequency  

**********************************************************************************/

void ChannelSwitchManager_CalcActualChannel()
{
	if (channelSwitchManagerParams.pUmSetChannelParams->ChannelNotificationMode == CHANNEL_NOTIFICATION_NORMAL)
	{
		channelSwitchManagerParams.actualBW = channelSwitchManagerParams.pUmSetChannelParams->chan_width;
		channelSwitchManagerParams.actualLowChannel = channelSwitchManagerParams.pUmSetChannelParams->low_chan_num;
		channelSwitchManagerParams.actualPrimaryChannel = channelSwitchManagerParams.pUmSetChannelParams->primary_chan_idx;
	}
	else if (channelSwitchManagerParams.pUmSetChannelParams->ChannelNotificationMode == CHANNEL_NOTIFICATION_EIGHTY)
	{
		channelSwitchManagerParams.actualBW = BANDWIDTH_EIGHTY;
		channelSwitchManagerParams.actualLowChannel = channelSwitchManagerParams.pUmSetChannelParams->low_chan_num + (16 * (channelSwitchManagerParams.pUmSetChannelParams->primary_chan_idx >> 2));
		channelSwitchManagerParams.actualPrimaryChannel = channelSwitchManagerParams.pUmSetChannelParams->primary_chan_idx & 0x3;
	}
	else if (channelSwitchManagerParams.pUmSetChannelParams->ChannelNotificationMode == CHANNEL_NOTIFICATION_FOURTY)
	{
		channelSwitchManagerParams.actualBW = BANDWIDTH_FOURTY;
		channelSwitchManagerParams.actualLowChannel = channelSwitchManagerParams.pUmSetChannelParams->low_chan_num + (8 * (channelSwitchManagerParams.pUmSetChannelParams->primary_chan_idx >> 1));
		channelSwitchManagerParams.actualPrimaryChannel = channelSwitchManagerParams.pUmSetChannelParams->primary_chan_idx & 0x1;
	}
	else if (channelSwitchManagerParams.pUmSetChannelParams->ChannelNotificationMode == CHANNEL_NOTIFICATION_TWENTY)
	{
		channelSwitchManagerParams.actualBW = BANDWIDTH_TWENTY;
		channelSwitchManagerParams.actualLowChannel = channelSwitchManagerParams.pUmSetChannelParams->low_chan_num + (4 * channelSwitchManagerParams.pUmSetChannelParams->primary_chan_idx);
		channelSwitchManagerParams.actualPrimaryChannel = 0;
	}
	else
	{
		ASSERT(0);
	}
}

