/***********************************************************************************
 File:		CoC.c
 Module:		Code Of Conduct
 Purpose:	Manages changing number of active antennas
 Description:	This source file is the implementation of CoC module
			which is responsible changing configuration of active antennas
************************************************************************************/

//---------------------------------------------------------------------------------
//						Includes						
//---------------------------------------------------------------------------------
#include "System_GlobalDefinitions.h"
#include "CoC_Api.h"
#include "CoC.h"
#include "OSAL_Api.h"
#include "ProcessManager_Api.h"
#include "CalibrationManager.h"
#include "ErrorHandler_Api.h"
#include "SmpsManager_Api.h"
#include "linkAdaptation_api.h"
#include "PacketDescriptor.h"
#include "Statistics_Api.h"
#include "BSSmanager_API.h"
#include "init_ifmsg.h"
#include "PSD.h"
#include "Utils_Api.h"
#include "queue_utility.h"
#include "loggerAPI.h"

//---------------------------------------------------------------------------------
//						Defines						
//---------------------------------------------------------------------------------

#define LOG_LOCAL_GID GLOBAL_GID_COC
#define LOG_LOCAL_FID 0


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

cocDb_t cocDb;

// CoC task table
static const FunctionEntry_t afpTaskTable[TASK_COC_END-TASK_COC_START]=
{
	{CoC_SetAntsReq,					DOUBLE_CHECK_MSG_TYPE(COC_SET_ANTENNAS_REQ)},
	{CoC_StartSetAntsProcess,			DOUBLE_CHECK_MSG_TYPE(COC_PROCESS_STARTED)},
	{CoC_FinalizeSetAntsProcess,		DOUBLE_CHECK_MSG_TYPE(COC_PROCESS_ENDED)},
	{CoC_HdkSetAntsDone,				DOUBLE_CHECK_MSG_TYPE(COC_HDK_SET_ANTS_CFM)},
	{CoC_LinkAdaptiveSetAntsDone,		DOUBLE_CHECK_MSG_TYPE(COC_LINK_ADAPTIVE_SET_ANTS_CFM)},
	{CoC_AddVap,						DOUBLE_CHECK_MSG_TYPE(COC_ADD_VAP)},
	{CoC_RemoveVap,						DOUBLE_CHECK_MSG_TYPE(COC_REMOVE_VAP)},	
	{CoC_UpdateAntConfigFromHdk, 		DOUBLE_CHECK_MSG_TYPE(COC_UPDATE_ANT_CONFIG_FROM_HDK)}, 
	{CoC_MuResetOperationCfm,			DOUBLE_CHECK_MSG_TYPE(COC_MULTI_USER_RESET_OPERATION_CFM)},
};

/*---------------------------------------------------------------------------------
/						External Definitions									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Functions Definitions									
/----------------------------------------------------------------------------------*/


/**********************************************************************************
CoC_task 

Description:
------------
	Entry function forCoC Task.
Input:
-----
	K_MSG *psMsg
Returns:
--------
	None
**********************************************************************************/
void CoC_task(K_MSG* psMsg)
{
	vTaskDispatcher(psMsg, afpTaskTable, TASK_COC_START, TASK_COC_END);
}


/**********************************************************************************
CoC_SetAntsReq  

Description:
------------
	Handle the request from driver to set antennas configuration. Trigger the process in process manager.
Input:
-----
	K_MSG* psMsg - MSG from driver
Returns:
--------
	None
**********************************************************************************/
void CoC_SetAntsReq(K_MSG* psMsg)
{
	K_MSG *processManagerMsg;
	ProcessRequestParams_t *processRequestParams;
	UMI_SET_ANTENNAS* changePowerStateMsg = (UMI_SET_ANTENNAS*)pK_MSG_DATA(psMsg);

	ASSERT(cocDb.CoCstate == COC_STATE_IDLE);
	DEBUG_ASSERT(cocDb.numOfVaps > 0);

	// Store requested antennas mask (and num) in coc database.
	cocDb.requestedTxAntsMask = changePowerStateMsg->TxAntsMask;
	cocDb.requestedRxAntsMask = changePowerStateMsg->RxAntsMask;
	cocDb.requestedTxAntsNum = Utils_GetNumAntsFromMask(cocDb.requestedTxAntsMask);
	cocDb.requestedRxAntsNum = Utils_GetNumAntsFromMask(cocDb.requestedRxAntsMask);
	
	/* Schedule Process*/	
	processManagerMsg = OSAL_GET_MESSAGE(sizeof(ProcessRequestParams_t));
	processRequestParams = (ProcessRequestParams_t*)pK_MSG_DATA(processManagerMsg);
	processRequestParams->processId 						= PROCESS_ID_COC;
	processRequestParams->startProcessMsg					= COC_PROCESS_STARTED;
	processRequestParams->endProcessMsg 					= COC_PROCESS_ENDED;
	processRequestParams->requesterParams					= psMsg; // save MSG from host here
	processRequestParams->preProcessServiceBitmap			= PROCESS_COC_PRE_SERVICES_BITMAP;
	processRequestParams->postProcessServiceBitmap			= PROCESS_COC_POST_SERVICES_BITMAP;
	processRequestParams->returnTask						= TASK_COC;
	processRequestParams->updateParamsBeforeFinalizing 		= FALSE;
	processRequestParams->dualBandProcess 					= TRUE;
	processRequestParams->processMsgHandledByCdbProcessMan 	= FALSE;
	processRequestParams->vapId 							= psMsg->header.vapId;
	// change CoC state
	cocDb.CoCstate = COC_STATE_WAIT_FOR_PROCESS_START;
	
	OSAL_SEND_MESSAGE(PROCESS_MANAGER_SCHEDULE_PROCESS_REQUEST,TASK_PROCESS_MANAGER,processManagerMsg, psMsg->header.vapId);
}
 

/**********************************************************************************
CoC_StartSetAntsProcess  

Description:
------------
	Process has started. TX and RX are paused. Send MSG to HDK to set antennas
Input:
-----
	K_MSG* psMsg - MSG from Process Manager
Returns:
--------
	None
**********************************************************************************/
void CoC_StartSetAntsProcess(K_MSG* psMsg)
{
	ProcessManagerReturnParams_t*	processMangerReturnPrams = (ProcessManagerReturnParams_t*)pK_MSG_DATA(psMsg);
	K_MSG*							pUmiMsg = (K_MSG*)processMangerReturnPrams->requesterParams;
	K_MSG*							psHdkMsg;
	cocReq_t*						pHdkCocReq;

	DEBUG_ASSERT(cocDb.CoCstate == COC_STATE_WAIT_FOR_PROCESS_START);
	DEBUG_ASSERT(processMangerReturnPrams->processStatus == PROCESS_STATUS_REQUEST_ACCEPTED);

	// change CoC state
	cocDb.CoCstate = COC_STATE_WAIT_FOR_HDK_SET_AND_CFM;

	// Send MSG to HDK to do the change in antenna configuration
	psHdkMsg = OSAL_GET_MESSAGE(sizeof(cocReq_t));
	pHdkCocReq = (cocReq_t*)pK_MSG_DATA(psHdkMsg);
	pHdkCocReq->requestedRxAntsMask = cocDb.requestedRxAntsMask;
	pHdkCocReq->requestedTxAntsMask = cocDb.requestedTxAntsMask;
	pHdkCocReq->requestedNumOfAnts = cocDb.requestedTxAntsNum;

	if (cocDb.requestedTxAntsNum > cocDb.currentTxAntsNum)
	{
		pHdkCocReq->upOrDown = COC_INCREASE_NUM_OF_ANTS;
	}
	else
	{
		pHdkCocReq->upOrDown = COC_DECREASE_NUM_OF_ANTS;
	}

	OSAL_SEND_MESSAGE(HDK_COC_SET_ANTENNAS_REQ, TASK_HDK, psHdkMsg, pUmiMsg->header.vapId);	
}


/**********************************************************************************
CoC_FinalizeSetAntsProcess  

Description:
------------
	Process has ended. TX and RX are resumed. notify SMPS Manager and send CFM to driver
Input:
-----
	K_MSG* psMsg - MSG from Process Manager
Returns:
--------
	None
**********************************************************************************/
void CoC_FinalizeSetAntsProcess(K_MSG* psMsg)
{
	ProcessManagerReturnParams_t*	processManagerReturnParams= (ProcessManagerReturnParams_t*)pK_MSG_DATA(psMsg);
	K_MSG*							pUmiMsg = (K_MSG*)processManagerReturnParams->requesterParams; // driver MSG was stored when process has started
	DEBUG_ASSERT(processManagerReturnParams->processStatus == PROCESS_STATUS_EXECUTION_COMPLETED);
	DEBUG_ASSERT((cocDb.CoCstate == COC_STATE_WAIT_FOR_PROCESS_END) || (cocDb.CoCstate == COC_STATE_HDK_IS_NOT_READY_WAIT_FOR_PROCESS_END));

	if (cocDb.CoCstate == COC_STATE_WAIT_FOR_PROCESS_END)
	{
		CoC_updateNewAntennaMask(cocDb.requestedTxAntsMask);
		CoC_sendSmps();
	
	}

	// change CoC state
	cocDb.CoCstate = COC_STATE_IDLE;
	
	// Send notification on antennas change
	CoC_updateOtherCoreOnAntChange();

	// Send confirmation to driver 
	OSAL_SEND_MESSAGE(UMI_UM_SET_ANTENNAS_CFM, TASK_UM_IF_TASK, pUmiMsg, pUmiMsg->header.vapId);
}


/**********************************************************************************
CoC_HdkSetAntsDone  

Description:
------------
	HDK has finished setting the antennas. Send MSG to link adaptive
Input:
-----
	K_MSG* psMsg - MSG from HDK
Returns:
--------
	None
**********************************************************************************/
void CoC_HdkSetAntsDone(K_MSG* psMsg)
{
	cocReq_t* pHdkCocReq = (cocReq_t*)pK_MSG_DATA(psMsg);
	K_MSG *kMsg_p  = NULL;
	LinkAdaptationFixedAntennaSelection_t *linkAdaptationAntSel_p;
	uint8 vapIndex = psMsg->header.vapId;
	/* For TLOG purpose - used to be no data message, send dummy data instead */
	K_MSG *dummyMsg = NULL;
	
	DEBUG_ASSERT(cocDb.CoCstate == COC_STATE_WAIT_FOR_HDK_SET_AND_CFM);
	
	if  (pHdkCocReq->hdkStatus == COC_STATUS_HDK_IS_READY)
	{
		
		kMsg_p = OSAL_GET_MESSAGE(sizeof(LinkAdaptationFixedAntennaSelection_t));
		linkAdaptationAntSel_p = (LinkAdaptationFixedAntennaSelection_t *)pK_MSG_DATA(kMsg_p);
		// change CoC state
		cocDb.CoCstate = COC_STATE_WAIT_FOR_LINK_ADAPTIVE_SET_AND_CFM;
	
		// Send notification to link adaptive
		linkAdaptationAntSel_p->mode 		= MODIFY_ANTENNA_MASK;
		linkAdaptationAntSel_p->vapId 		= vapIndex;
		linkAdaptationAntSel_p->stationId 	= INVALID_STA_INDEX;
		linkAdaptationAntSel_p->mask 		= cocDb.requestedTxAntsMask;
		linkAdaptationAntSel_p->retTask 	= TASK_COC;
		linkAdaptationAntSel_p->retMsg 		= COC_LINK_ADAPTIVE_SET_ANTS_CFM;

		OSAL_SEND_MESSAGE(LINK_ADAPTATION_SET_ANTENNA_SELECTION_REQ, TASK_LINK_ADAPTATION, kMsg_p, vapIndex);
	}
	else
	{
		// change CoC state
		cocDb.CoCstate = COC_STATE_HDK_IS_NOT_READY_WAIT_FOR_PROCESS_END;

		// Process has finished. Notify Process Manager
		dummyMsg = ProcessManager_GetDummyMessage();		
		OSAL_SEND_MESSAGE(PROCESS_MANAGER_PROCESS_EXCUTION_FINISHED, TASK_PROCESS_MANAGER, dummyMsg, vapIndex);
	}
}

/**********************************************************************************
CoC_AddVap	

Description:
------------
	Count number of VAPs
Input:
-----
	None
Returns:
--------
	None
**********************************************************************************/
void CoC_AddVap(K_MSG* psMsg)
{
	UMI_ADD_VAP* pAddVapMsg	= (UMI_ADD_VAP*) EXTRACT_VAP_MANAGER_MSG(psMsg);
	//KW_IGNORE ABV.GENERAL .. since abdata is never assigned, but used for typecast.

	// Count number of VAPs
	cocDb.numOfVaps++;
	if (cocDb.numOfVaps == 1)
	{
		/*Init only at the first Vap*/
		CoC_InitNumAnts();
	}

	// Send Confirmation to VAP manager
	FILL_VAP_MANAGER_CONFIRM_MSG(psMsg, pAddVapMsg->vapId, VAP_MANAGER_ADD_VAP, BSS_MANAGER_VAP_MANAGER_COC_CLIENT);
	OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, psMsg, pAddVapMsg->vapId);
}


/**********************************************************************************
CoC_RemoveVap	

Description:
------------
	Count number of VAPs
Input:
-----
	None
Returns:
--------
	None
**********************************************************************************/
void CoC_RemoveVap(K_MSG* psMsg)
{
	UMI_REMOVE_VAP* pRemoveVapMsg	= (UMI_REMOVE_VAP*) EXTRACT_VAP_MANAGER_MSG(psMsg);
	//KW_IGNORE ABV.GENERAL .. since abdata is never assigned, but used for typecast.

	DEBUG_ASSERT(cocDb.numOfVaps > 0);
	
	// count number of VAPs
	cocDb.numOfVaps--;

	// Send Confirmation to VAP manager
	FILL_VAP_MANAGER_CONFIRM_MSG(psMsg, pRemoveVapMsg->vapId, VAP_MANAGER_REMOVE_VAP, BSS_MANAGER_VAP_MANAGER_COC_CLIENT);
	OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, psMsg, pRemoveVapMsg->vapId);
}



/**********************************************************************************
CoC_LinkAdaptiveSetAntsDone  

Description:
------------
	Link adaptive has finished adjusting to new antenna configuration. Notify process manager that process
	has ended.
Input:
-----
	K_MSG* psMsg - MSG from LA
Returns:
--------
	None
**********************************************************************************/
void CoC_LinkAdaptiveSetAntsDone(K_MSG* psMsg)
{
	DEBUG_ASSERT(cocDb.CoCstate == COC_STATE_WAIT_FOR_LINK_ADAPTIVE_SET_AND_CFM);

	// change CoC state
	cocDb.CoCstate = COC_STATE_WAIT_FOR_GROUP_MANAGER_SET_AND_CFM;
	
	// Process has finished. Notify Process Manager
	OSAL_SEND_NO_DATA_MESSAGE(GROUP_MANAGER_RESET_MULTI_USER_OPERATION, TASK_GROUP_MANAGER,psMsg->header.vapId);
}


/**********************************************************************************
CoC_MuResetOperationCfm  

Description:
------------

Input:
-----

Returns:
--------
	None
**********************************************************************************/
void CoC_MuResetOperationCfm(K_MSG* psMsg)
{
	/* For TLOG purpose - used to be no data message, send dummy data instead */
	K_MSG *dummyMsg = NULL;

	DEBUG_ASSERT(cocDb.CoCstate == COC_STATE_WAIT_FOR_GROUP_MANAGER_SET_AND_CFM);

	// change CoC state
	cocDb.CoCstate = COC_STATE_WAIT_FOR_PROCESS_END;

	// Process has finished. Notify Process Manager
	dummyMsg = ProcessManager_GetDummyMessage();	
	OSAL_SEND_MESSAGE(PROCESS_MANAGER_PROCESS_EXCUTION_FINISHED, TASK_PROCESS_MANAGER, dummyMsg, psMsg->header.vapId);
}


/**********************************************************************************
CoC_getState	

Description:
------------
	Get the current CoC state
Input:
-----
	None
Returns:
--------
	cocState_e CoCstate
**********************************************************************************/
cocState_e CoC_getState(void)
{
	return cocDb.CoCstate;
}

/**********************************************************************************
CoC_getNumActiveTxAnts	

Description:
------------
	Get the current number of active TX antennas
Input:
-----
	None
Returns:
--------
	uint8 currentTxAntsNum
**********************************************************************************/
uint8 CoC_getNumActiveTxAnts(void)
{
	return cocDb.currentTxAntsNum;
}

/**********************************************************************************
CoC_getNumActiveRxAnts	

Description:
------------
	Get the current number of active RX antennas 
Input:
-----
	None
Returns:
--------
	uint8 currentRxAntsNum
**********************************************************************************/
uint8 CoC_getNumActiveRxAnts(void)
{
	return cocDb.currentRxAntsNum;
}

/**********************************************************************************
CoC_getMaskActiveTxAnts	

Description:
------------
	Get the current mask of active TX antennas
Input:
-----
	None
Returns:
--------
	uint8 currentTxAntsMask
**********************************************************************************/

uint8 CoC_getMaskActiveTxAnts(void)
{
	return cocDb.currentTxAntsMask;
}

/**********************************************************************************
CoC_getMaskActiveRxAnts	

Description:
------------
	Get the current mask of active RX antennas
Input:
-----
	None
Returns:
--------
	uint8 currentRxAntsMask
**********************************************************************************/

uint8 CoC_getMaskActiveRxAnts(void)
{
	return cocDb.currentRxAntsMask;
}


/**********************************************************************************
CoC_InitNumAnts	

Description:
------------
	Init number of TX and RX antennas
Input:
-----
	None
Returns:
--------
	None
**********************************************************************************/
void CoC_InitNumAnts(void)
{
	cocDb.currentTxAntsMask = Hdk_GetTxAntMask();
	cocDb.currentRxAntsMask = Hdk_GetRxAntMask();
	cocDb.currentTxAntsNum = Utils_GetNumAntsFromMask(cocDb.currentTxAntsMask);
	cocDb.currentRxAntsNum = Utils_GetNumAntsFromMask(cocDb.currentRxAntsMask);
	cocDb.maxRxAntsNum = cocDb.currentRxAntsNum;

	// Send notification on antennas change
	CoC_updateOtherCoreOnAntChange();
}



void CoC_SendNotificationToOtherModule(K_MSG_TYPE kmsgType, K_TASKID taskId)
{
	K_MSG*	pCocUpdateAntsNumMsg;
	cocUpdateAntMaskMsg_t* pCocAntsNum;

	// Send notification to other modules of number of current antennas
	pCocUpdateAntsNumMsg = OSAL_GET_MESSAGE(sizeof(cocUpdateAntMaskMsg_t));
	pCocAntsNum = (cocUpdateAntMaskMsg_t*)pK_MSG_DATA(pCocUpdateAntsNumMsg);		
	pCocAntsNum->currentRxAntsMask = cocDb.currentRxAntsMask;
	pCocAntsNum->currentTxAntsMask = cocDb.currentTxAntsMask;
	pCocAntsNum->currentTxAntsNum = cocDb.currentTxAntsNum;
	pCocAntsNum->band = ConfigurationManager_GetMyBand();
	
	OSAL_SEND_MESSAGE(kmsgType, taskId, pCocUpdateAntsNumMsg, GET_DEFAULT_VAP_FOR_MY_BAND());
}
void CoC_updateOtherCoreOnAntChange()
{
	CoC_SendNotificationToOtherModule(STATISTICS_MANAGER_SET_ANTENNAS_BITMAP,TASK_STATISTICS_MANAGER);
	CoC_SendNotificationToOtherModule(DUT_UPDATE_TX_RX_ANTS_FROM_COC, TASK_DUT);
	CoC_SendNotificationToOtherModule(GROUP_MANAGER_UPDATE_COC_INFO, TASK_GROUP_MANAGER);	
#if defined (ENET_INC_ARCH_WAVE600) 	
	CoC_SendNotificationToOtherModule(HE_GROUP_MANAGER_UPDATE_COC_INFO, TASK_HE_GROUP_MANAGER);
#endif
}
void CoC_UpdateAntConfigFromHdk(K_MSG* psMsg)
{
	cocUpdateAntMaskMsg_t* pCocAntsNum;

	// Send notification to DUT of number of current antennas
	pCocAntsNum = (cocUpdateAntMaskMsg_t*)pK_MSG_DATA(psMsg);		
	cocDb.currentRxAntsMask = pCocAntsNum->currentRxAntsMask; 
	cocDb.currentTxAntsMask = pCocAntsNum->currentTxAntsMask;
	cocDb.currentTxAntsNum = Utils_GetNumAntsFromMask(cocDb.currentTxAntsMask);
	cocDb.currentRxAntsNum = Utils_GetNumAntsFromMask(cocDb.currentRxAntsMask);
	/*Notify SMPS*/
	CoC_sendSmps();
	/*Send confirmation*/	
	OSAL_SEND_NO_DATA_MESSAGE(HDK_SET_ANT_CONFIG_NOTIFICATION_CFM, TASK_HDK, psMsg->header.vapId);
}

/**********************************************************************************
CoC_setAntennaMaskForNewChannel  

Description:
------------
	initialize CoC to new antenna mask of this channel
	send SMPS packets if needed
Input:
-----
	None
Returns:
--------
	None
**********************************************************************************/

void CoC_updateNewAntennaMask(uint8 antennaMask)
{
	uint8	antsNum;

	

	antsNum = Utils_GetNumAntsFromMask(antennaMask);

	cocDb.currentTxAntsMask = antennaMask;
	cocDb.currentRxAntsMask = antennaMask;
	cocDb.currentTxAntsNum	= antsNum;
	cocDb.currentRxAntsNum	= antsNum;
}

void CoC_sendSmps()
{
	K_MSG* 							smpsManagerMessage;
	SmpsManagerStartTxReqParams_t*	smpsTxReqParams;

	// Send notification to SMPS Manager
	smpsManagerMessage = OSAL_GET_MESSAGE(sizeof(SmpsManagerStartTxReqParams_t));
	smpsTxReqParams = (SmpsManagerStartTxReqParams_t *)pK_MSG_DATA(smpsManagerMessage);
	smpsTxReqParams->numOfAntennas = cocDb.currentRxAntsNum;
	smpsTxReqParams->maxRxAntsNum = cocDb.maxRxAntsNum;
	if (cocDb.currentRxAntsNum < cocDb.maxRxAntsNum)
	{
		smpsTxReqParams->mode = SMPS_MODE_STATIC;
	}
	else
	{
		smpsTxReqParams->mode = SMPS_MODE_DISABLED;
	}

	// Send message to SMPS Manager
	OSAL_SEND_MESSAGE(SMPS_TX_START_REQ, TASK_TX_MANAGER, smpsManagerMessage, GET_DEFAULT_VAP_FOR_MY_BAND());
	
}

#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif


/**********************************************************************************
CoC_Init  

Description:
------------
	Initialize the CoC module
Input:
-----
	None
Returns:
--------
	None
**********************************************************************************/
void CoC_Init(void)
{
	cocDb.CoCstate = COC_STATE_IDLE;
	cocDb.numOfVaps = 0;
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif
