/***********************************************************************************
 File:			CsaManager_StaManager.c
 Module:		CSA Manager STA Manager
 Purpose: 		handle CSA Manager STA Manager events
 Description:   
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
	//TBD to check which files can be removed
#include "BSSmanager_API.h"
#include "TxManager_Api.h"
#include "TsManager_API.h"
#include "TxSelector_Api.h"
#include "HostInterface_API.h"
#include "Utils_Api.h"
#include "ResourceManager_API.h"
#include "Locker_Api.h"
#include "OSAL_Kmsg.h"
#include "ErrorHandler_Api.h"
#include "HwQManager_API.h"
#include "HwGlobalDefinitions.h"
#include "ShramPacketDescriptors.h"
#include "stringLibApi.h"
#include "um_interface.h"
#include "PacketDescriptor.h"
#include "HwMemoryMap.h"
#include "CsaManager_Api.h"
#include "CsaManager.h"
#include "CsaManager_VapManager.h"
#include "CsaManager_StaManager.h"
#include "loggerAPI.h"


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_CSA_MANAGER 
#define LOG_LOCAL_FID 3


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/
#define CSA_MANAGER_STA_MANAGER_INVALID_LOCK_IDX	(INVALID_STA_INDEX)


/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
/*CSA Manager STA Manager events*/
typedef enum
{
	CSA_MANAGER_STA_MANAGER_ADD_STA,
	CSA_MANAGER_STA_MANAGER_REMOVE_STA,
	CSA_MANAGER_STA_MANAGER_START_TX_REQ,
	CSA_MANAGER_STA_MANAGER_STOP_TX_REQ,
	CSA_MANAGER_STA_MANAGER_TX_CFM,
	CSA_MANAGER_STA_MANAGER_PD_ALLOC,
	CSA_MANAGER_STA_MANAGER_LOCK_CFM,
	CSA_MANAGER_STA_MANAGER_NUM_EVENTS
} csaManagerStaManager_ev;
	
/*CSA Manager STA Manager states*/
typedef enum
{
	CSA_MANAGER_STA_MANAGER_IDLE_STATE,
	CSA_MANAGER_STA_MANAGER_OPEN_STATE,
	CSA_MANAGER_STA_MANAGER_WAIT_TX_CFM_STATE,
	CSA_MANAGER_STA_MANAGER_WAIT_PD_STATE,
	CSA_MANAGER_STA_MANAGER_WAIT_PD_STOP_STATE,
	CSA_MANAGER_STA_MANAGER_WAIT_PD_CLOSE_STATE,
	CSA_MANAGER_STA_MANAGER_WAIT_TX_CFM_LOCK_STATE,
	CSA_MANAGER_STA_MANAGER_WAIT_TX_CFM_CLOSE_STATE,
	CSA_MANAGER_STA_MANAGER_WAIT_GPHP_LOCK_CLOSE_STATE,
#if defined CSA_REMOVE_RETRY	
	CSA_MANAGER_STA_MANAGER_WAIT_LOCK_STATE,
	CSA_MANAGER_STA_MANAGER_WAIT_LOCK_STOP_STATE,
	CSA_MANAGER_STA_MANAGER_WAIT_LOCK_CLOSE_STATE,
#endif	
	CSA_MANAGER_STA_MANAGER_NUM_STATES
} csaManagerStaManager_state;

typedef struct CsaManagerStaManagerDb_s
{
	K_MSG						*psMsg;
	TxPd_t						*pd;
	StaId						requestId;	/*From resource manager*/
	StaId						sid;
	uint8						flags;
	csaManagerStaManager_state	state;
	uint8						vapId;
} CsaManagerStaManagerDb_t;

typedef struct CsaManagerStaManagerGlobalDb_s
{
#if defined CSA_REMOVE_RETRY	
	/*A potential problem exists when transmitting CSA
	CSA are transmitted from GPHP queue but they are still unicast management frames and therefore
	are allocated the STA management Sequence number. However,it is possible that a Management
	Frame is pending for retransmission onthe STA MNG queue and it will be retransmitted only after Channel
	Switch is over - this causes two issues: 1) out of order SEQ #, this is OK as it doesn't trigger
	duplicate detection, 2) Out of order PN if PMF is used - this causes replay detection. One replay detection
	may be acceptable but we still implement the option to remove an exisiting management frame from
	the MNG queue*/
	bool	removeRetry;
	/*For locker serialization*/
	StaId	staMngLockHead;
	StaId	staMngLockTail;
#endif	
	StaId	gphpLockHead;
	StaId	gphpLockTail;
} CsaManagerStaManagerGlobalDb_t;

typedef void (*CsaManagerStaManager_Func)(CsaManagerStaManagerDb_t *StaDb);

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
/*Utility functions*/
static void CsaManager_StaManagerChangeState(csaManagerStaManager_state state, CsaManagerStaManagerDb_t *StaDb);
static void CsaManager_StaManagerSendConfirm(CsaManagerStaManagerDb_t *StaDb);
static void CsaManager_StaManagerRemoveStaDone(CsaManagerStaManagerDb_t *StaDb);
static void CsaManager_StaManagerTxCfmDone(CsaManagerStaManagerDb_t *StaDb);
static void CsaManager_StaManagerSendCsa(CsaManagerStaManagerDb_t *StaDb);
static void CsaManager_StaManagerSendCsaUcDeauth(CsaManagerStaManagerDb_t *StaDb);
static void CsaManager_StaManagerStartTx(CsaManagerStaManagerDb_t *StaDb);
#if defined CSA_REMOVE_RETRY	
static void CsaManager_StaManagerLockStaMng(CsaManagerStaManagerDb_t *StaDb);
static void CsaManager_StaManagerHandleLockStaMng(void);
static void CsaManager_StaManagerUnlockStaMngQueue(CsaManagerStaManagerDb_t *StaDb);
static void CsaManager_StaManagerRemoveRetry(CsaManagerStaManagerDb_t *StaDb);
#endif
static void CsaManager_StaManagerLockGPHP(CsaManagerStaManagerDb_t *StaDb);
static void CsaManager_StaManagerHandleLockGPHP(void);
static bool CsaManager_StaManagerRemovePdFromGPHP(CsaManagerStaManagerDb_t *StaDb);
/*Event handlers*/
static void CsaManager_StaManagerAddStaInIdle(CsaManagerStaManagerDb_t *StaDb);
static void CsaManager_StaManagerRemoveStaInOpen(CsaManagerStaManagerDb_t *StaDb);
static void CsaManager_StaManagerRemoveStaInWaitTxCfm(CsaManagerStaManagerDb_t *StaDb);
static void CsaManager_StaManagerRemoveStaInWaitPd(CsaManagerStaManagerDb_t *StaDb);
static void CsaManager_StaManagerRemoveStaInWaitPdStop(CsaManagerStaManagerDb_t *StaDb);
#if defined CSA_REMOVE_RETRY	
static void CsaManager_StaManagerRemoveStaInWaitLock(CsaManagerStaManagerDb_t *StaDb);
static void CsaManager_StaManagerRemoveStaInWaitLockStop(CsaManagerStaManagerDb_t *StaDb);
#endif
static void CsaManager_StaManagerStartTxInOpen(CsaManagerStaManagerDb_t *StaDb);
static void CsaManager_StaManagerStopTxInOpen(CsaManagerStaManagerDb_t *StaDb);
static void CsaManager_StaManagerStopTxInWaitPd(CsaManagerStaManagerDb_t *StaDb);
#if defined CSA_REMOVE_RETRY	
static void CsaManager_StaManagerStopTxInWaitLock(CsaManagerStaManagerDb_t *StaDb);
#endif
static void CsaManager_StaManagerTxCfmInWaitTxCfm(CsaManagerStaManagerDb_t *StaDb);
static void CsaManager_StaManagerTxCfmInWaitTxCfmLock(CsaManagerStaManagerDb_t *StaDb);
static void CsaManager_StaManagerTxCfmInWaitTxCfmClose(CsaManagerStaManagerDb_t *StaDb);
static void CsaManager_StaManagerPdAllocInWaitPd(CsaManagerStaManagerDb_t *StaDb);
static void CsaManager_StaManagerPdAllocInWaitPdStop(CsaManagerStaManagerDb_t *StaDb);
static void CsaManager_StaManagerPdAllocInWaitPdClose(CsaManagerStaManagerDb_t *StaDb);
static void CsaManager_StaManagerLockCfmInTxCfmLock(CsaManagerStaManagerDb_t *StaDb);
static void CsaManager_StaManagerLockCfmInGPHPLockClose(CsaManagerStaManagerDb_t *StaDb);
#if defined CSA_REMOVE_RETRY	
static void CsaManager_StaManagerLockCfmInWaitLock(CsaManagerStaManagerDb_t *StaDb);
static void CsaManager_StaManagerLockCfmInWaitLockStop(CsaManagerStaManagerDb_t *StaDb);
static void CsaManager_StaManagerLockCfmInWaitLockClose(CsaManagerStaManagerDb_t *StaDb);
#endif
static void CsaManager_StaManagerIgnore(CsaManagerStaManagerDb_t *StaDb);
static void CsaManager_StaManagerFatal(CsaManagerStaManagerDb_t *StaDb);

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
/*CSA Manager STA Manager Add STA Handlers*/
static const CsaManagerStaManager_Func CsaManager_StaManagerAddSta[CSA_MANAGER_STA_MANAGER_NUM_STATES] =
{
	 CsaManager_StaManagerAddStaInIdle, 			/* CSA_MANAGER_STA_MANAGER_IDLE_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_OPEN_STATE */
 	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_TX_CFM_STATE */
	 CsaManager_StaManagerFatal, 					/* CSA_MANAGER_STA_MANAGER_WAIT_PD_STATE */
	 CsaManager_StaManagerFatal, 					/* CSA_MANAGER_STA_MANAGER_WAIT_PD_STOP_STATE */
	 CsaManager_StaManagerFatal, 					/* CSA_MANAGER_STA_MANAGER_WAIT_PD_CLOSE_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_TX_CFM_LOCK_STATE */
	 CsaManager_StaManagerFatal, 					/* CSA_MANAGER_STA_MANAGER_WAIT_TX_CFM_CLOSE_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_GPHP_LOCK_CLOSE_STATE */
#if defined CSA_REMOVE_RETRY	
	 CsaManager_StaManagerFatal, 					/* CSA_MANAGER_STA_MANAGER_WAIT_LOCK_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_LOCK_STOP_STATE */
	 CsaManager_StaManagerFatal 					/* CSA_MANAGER_STA_MANAGER_WAIT_LOCK_CLOSE_STATE */
#endif	 
};

/*CSA Manager STA Manager Remove STA Handlers*/
static const CsaManagerStaManager_Func CsaManager_StaManagerRemoveSta[CSA_MANAGER_STA_MANAGER_NUM_STATES] =
{
	 CsaManager_StaManagerFatal, 					/* CSA_MANAGER_STA_MANAGER_IDLE_STATE */
	 CsaManager_StaManagerRemoveStaInOpen,			/* CSA_MANAGER_STA_MANAGER_OPEN_STATE */
 	 CsaManager_StaManagerRemoveStaInWaitTxCfm,		/* CSA_MANAGER_STA_MANAGER_WAIT_TX_CFM_STATE */
	 CsaManager_StaManagerRemoveStaInWaitPd,		/* CSA_MANAGER_STA_MANAGER_WAIT_PD_STATE */
	 CsaManager_StaManagerRemoveStaInWaitPdStop,	/* CSA_MANAGER_STA_MANAGER_WAIT_PD_STOP_STATE */
	 CsaManager_StaManagerFatal,		 			/* CSA_MANAGER_STA_MANAGER_WAIT_PD_CLOSE_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_TX_CFM_LOCK_STATE */
	 CsaManager_StaManagerFatal, 					/* CSA_MANAGER_STA_MANAGER_WAIT_TX_CFM_CLOSE_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_GPHP_LOCK_CLOSE_STATE */
#if defined CSA_REMOVE_RETRY	
	 CsaManager_StaManagerRemoveStaInWaitLock,		/* CSA_MANAGER_STA_MANAGER_WAIT_LOCK_STATE */
	 CsaManager_StaManagerRemoveStaInWaitLockStop,	/* CSA_MANAGER_STA_MANAGER_WAIT_LOCK_STOP_STATE */
	 CsaManager_StaManagerFatal 					/* CSA_MANAGER_STA_MANAGER_WAIT_LOCK_CLOSE_STATE */
#endif	 
};

/*CSA Manager STA Manager Start TX REQ Handlers*/
static const CsaManagerStaManager_Func CsaManager_StaManagerStartTxReq[CSA_MANAGER_STA_MANAGER_NUM_STATES] =
{
	 CsaManager_StaManagerFatal, 					/* CSA_MANAGER_STA_MANAGER_IDLE_STATE */
	 CsaManager_StaManagerStartTxInOpen,			/* CSA_MANAGER_STA_MANAGER_OPEN_STATE */
 	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_TX_CFM_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_PD_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_PD_STOP_STATE */
	 CsaManager_StaManagerFatal,		 			/* CSA_MANAGER_STA_MANAGER_WAIT_PD_CLOSE_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_TX_CFM_LOCK_STATE */
	 CsaManager_StaManagerFatal, 					/* CSA_MANAGER_STA_MANAGER_WAIT_TX_CFM_CLOSE_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_GPHP_LOCK_CLOSE_STATE */
#if defined CSA_REMOVE_RETRY	
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_LOCK_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_LOCK_STOP_STATE */
	 CsaManager_StaManagerFatal 					/* CSA_MANAGER_STA_MANAGER_WAIT_LOCK_CLOSE_STATE */
#endif	 
};

/*CSA Manager STA Manager Stop TX REQ Handlers*/
static const CsaManagerStaManager_Func CsaManager_StaManagerStopTxReq[CSA_MANAGER_STA_MANAGER_NUM_STATES] =
{
	 CsaManager_StaManagerFatal, 					/* CSA_MANAGER_STA_MANAGER_IDLE_STATE */
	 CsaManager_StaManagerStopTxInOpen,				/* CSA_MANAGER_STA_MANAGER_OPEN_STATE */
 	 CsaManager_StaManagerIgnore,					/* CSA_MANAGER_STA_MANAGER_WAIT_TX_CFM_STATE */
	 CsaManager_StaManagerStopTxInWaitPd,			/* CSA_MANAGER_STA_MANAGER_WAIT_PD_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_PD_STOP_STATE */
	 CsaManager_StaManagerIgnore,		 			/* CSA_MANAGER_STA_MANAGER_WAIT_PD_CLOSE_STATE */
	 CsaManager_StaManagerIgnore,					/* CSA_MANAGER_STA_MANAGER_WAIT_TX_CFM_LOCK_STATE */
	 CsaManager_StaManagerIgnore, 					/* CSA_MANAGER_STA_MANAGER_WAIT_TX_CFM_CLOSE_STATE */
	 CsaManager_StaManagerIgnore,					/* CSA_MANAGER_STA_MANAGER_WAIT_GPHP_LOCK_CLOSE_STATE */
#if defined CSA_REMOVE_RETRY	
	 CsaManager_StaManagerStopTxInWaitLock,			/* CSA_MANAGER_STA_MANAGER_WAIT_LOCK_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_LOCK_STOP_STATE */
	 CsaManager_StaManagerIgnore 					/* CSA_MANAGER_STA_MANAGER_WAIT_LOCK_CLOSE_STATE */
#endif	 
};

/*CSA Manager STA Manager TX CFM Handlers*/
static const CsaManagerStaManager_Func CsaManager_StaManagerTxCfm[CSA_MANAGER_STA_MANAGER_NUM_STATES] =
{
	 CsaManager_StaManagerFatal, 					/* CSA_MANAGER_STA_MANAGER_IDLE_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_OPEN_STATE */
 	 CsaManager_StaManagerTxCfmInWaitTxCfm,			/* CSA_MANAGER_STA_MANAGER_WAIT_TX_CFM_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_PD_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_PD_STOP_STATE */
	 CsaManager_StaManagerFatal,		 			/* CSA_MANAGER_STA_MANAGER_WAIT_PD_CLOSE_STATE */
	 CsaManager_StaManagerTxCfmInWaitTxCfmLock,		/* CSA_MANAGER_STA_MANAGER_WAIT_TX_CFM_LOCK_STATE */
	 CsaManager_StaManagerTxCfmInWaitTxCfmClose, 	/* CSA_MANAGER_STA_MANAGER_WAIT_TX_CFM_CLOSE_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_GPHP_LOCK_CLOSE_STATE */
#if defined CSA_REMOVE_RETRY	
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_LOCK_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_LOCK_STOP_STATE */
	 CsaManager_StaManagerFatal 					/* CSA_MANAGER_STA_MANAGER_WAIT_LOCK_CLOSE_STATE */
#endif	 
};

/*CSA Manager STA Manager PD Alloc Handlers*/
static const CsaManagerStaManager_Func CsaManager_StaManagerPdAlloc[CSA_MANAGER_STA_MANAGER_NUM_STATES] =
{
	 CsaManager_StaManagerFatal, 					/* CSA_MANAGER_STA_MANAGER_IDLE_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_OPEN_STATE */
 	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_TX_CFM_STATE */
	 CsaManager_StaManagerPdAllocInWaitPd,			/* CSA_MANAGER_STA_MANAGER_WAIT_PD_STATE */
	 CsaManager_StaManagerPdAllocInWaitPdStop,		/* CSA_MANAGER_STA_MANAGER_WAIT_PD_STOP_STATE */
	 CsaManager_StaManagerPdAllocInWaitPdClose,		/* CSA_MANAGER_STA_MANAGER_WAIT_PD_CLOSE_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_TX_CFM_LOCK_STATE */
	 CsaManager_StaManagerFatal, 					/* CSA_MANAGER_STA_MANAGER_WAIT_TX_CFM_CLOSE_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_GPHP_LOCK_CLOSE_STATE */
#if defined CSA_REMOVE_RETRY	
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_LOCK_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_LOCK_STOP_STATE */
	 CsaManager_StaManagerFatal 					/* CSA_MANAGER_STA_MANAGER_WAIT_LOCK_CLOSE_STATE */
#endif	 
};

/*CSA Manager STA Manager GPHP Lock CFM Handlers*/
static const CsaManagerStaManager_Func CsaManager_StaManagerGphpLockCfm[CSA_MANAGER_STA_MANAGER_NUM_STATES] =
{
	 CsaManager_StaManagerFatal, 					/* CSA_MANAGER_STA_MANAGER_IDLE_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_OPEN_STATE */
 	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_TX_CFM_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_PD_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_PD_STOP_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_PD_CLOSE_STATE */
	 CsaManager_StaManagerLockCfmInTxCfmLock,		/* CSA_MANAGER_STA_MANAGER_WAIT_TX_CFM_LOCK_STATE */
	 CsaManager_StaManagerFatal, 					/* CSA_MANAGER_STA_MANAGER_WAIT_TX_CFM_CLOSE_STATE */
	 CsaManager_StaManagerLockCfmInGPHPLockClose,	/* CSA_MANAGER_STA_MANAGER_WAIT_GPHP_LOCK_CLOSE_STATE */
#if defined CSA_REMOVE_RETRY	
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_LOCK_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_LOCK_STOP_STATE */
	 CsaManager_StaManagerFatal						/* CSA_MANAGER_STA_MANAGER_WAIT_LOCK_CLOSE_STATE */
#endif	 
};

/*CSA Manager STA Manager STA Lock CFM Handlers*/
#if defined CSA_REMOVE_RETRY	
static const CsaManagerStaManager_Func CsaManager_StaManagerStaLockCfm[CSA_MANAGER_STA_MANAGER_NUM_STATES] =
{
	 CsaManager_StaManagerFatal, 					/* CSA_MANAGER_STA_MANAGER_IDLE_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_OPEN_STATE */
 	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_TX_CFM_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_PD_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_PD_STOP_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_PD_CLOSE_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_TX_CFM_LOCK_STATE */
	 CsaManager_StaManagerFatal, 					/* CSA_MANAGER_STA_MANAGER_WAIT_TX_CFM_CLOSE_STATE */
	 CsaManager_StaManagerFatal,					/* CSA_MANAGER_STA_MANAGER_WAIT_GPHP_LOCK_CLOSE_STATE */
	 CsaManager_StaManagerLockCfmInWaitLock,		/* CSA_MANAGER_STA_MANAGER_WAIT_LOCK_STATE */
	 CsaManager_StaManagerLockCfmInWaitLockStop,	/* CSA_MANAGER_STA_MANAGER_WAIT_LOCK_STOP_STATE */
	 CsaManager_StaManagerLockCfmInWaitLockClose	/* CSA_MANAGER_STA_MANAGER_WAIT_LOCK_CLOSE_STATE */
};
#endif

static CsaManagerStaManagerDb_t CsaManagerStaManager_Db[HW_NUM_OF_STATIONS];
static CsaManagerStaManagerGlobalDb_t CsaManagerStaManager_GlobalDb;

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

CsaManager_StaManagerChangeState 


Description:
------------
Utility function that changes STA state


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerChangeState(csaManagerStaManager_state state, CsaManagerStaManagerDb_t *StaDb)
{
 	ILOG0_DDD("CSA Manager STA Manager, STA %d changed from %d to %d", StaDb->sid, StaDb->state, state);
	StaDb->state = state;
}

/**********************************************************************************

CsaManager_StaManagerSendConfirm 


Description:
------------
Utility function that sends confirmation to STA Manager


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerSendConfirm(CsaManagerStaManagerDb_t *StaDb)
{
	K_MSG *pMsg;
	BssManagerStaManagerCfm_t *confirmMessage;

	/*Allocate message*/
	pMsg = OSAL_GET_MESSAGE(sizeof(BssManagerStaManagerCfm_t));
	confirmMessage = (BssManagerStaManagerCfm_t*) pK_MSG_DATA(pMsg);
	/*Set Client ID to the registered ID*/
	confirmMessage->clientId = BSS_MANAGER_STA_MANAGER_CSA_CLIENT;
	/*Set STA ID*/
	confirmMessage->sid = StaDb->sid;
	/*Send confirmation message*/
	OSAL_SEND_MESSAGE(BSS_MANAGER_STA_MANAGER_REG_CFM, TASK_BSS_MANAGER, pMsg,StaDb->vapId);
}

/**********************************************************************************

CsaManager_StaManagerRemoveStaDone 


Description:
------------
Utility function that does the necessary cleanup when STA remove is done


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerRemoveStaDone(CsaManagerStaManagerDb_t *StaDb)
{
	/*Clear message*/
	StaDb->psMsg = NULL;
	/*Set state to Idle*/
	CsaManager_StaManagerChangeState(CSA_MANAGER_STA_MANAGER_IDLE_STATE, StaDb); 
	/*Send Indication to CSA VAP Manager*/
	CsaManager_VapManagerRemoveStaInd(StaDb->vapId, StaDb->sid);
	/*Send confirmation to BSS Manager*/
	CsaManager_StaManagerSendConfirm(StaDb);
}

/**********************************************************************************

CsaManager_StaManagerTxCfmDone 


Description:
------------
Utility function that does the necessary cleanup when TX CFM is done


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerTxCfmDone(CsaManagerStaManagerDb_t *StaDb)
{
	ILOG0_V("[CSA] CsaManager_StaManagerTxCfmDone 1");
	/*Release the PD*/
	ResourceManager_ReleaseDescriptor(StaDb->pd, DESC_POOL_CSA_MANAGER);
	/*Change State to Open*/
	CsaManager_StaManagerChangeState(CSA_MANAGER_STA_MANAGER_OPEN_STATE, StaDb); 
	/*Send TX CFM to VAP manager*/
	csaManager_VapManagerTxCfm(StaDb->vapId, StaDb->sid);	
}


/**********************************************************************************
CsaManager_StaManagerSendCsaUcDeauth

Description:
------------
Utility function that builds the de auth and sends TX IND to VAP Manager

Input: 
-----	
			
Output:
-------
	
Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerSendCsaUcDeauth(CsaManagerStaManagerDb_t *StaDb)
{
	TxPd_t *pdPointer = (TxPd_t *)StaDb->pd;
	bool isStaPmf = CsaManager_StaManagerIsStaPmf(StaDb->sid);

	pdPointer->txQStaId= StaDb->sid;
	pdPointer->txQTid = MANAGEMENT_TID;
	pdPointer->mgmtFrameSubtype = MGMT_FRAME_SUBTYPE_DEAUTHENTICATION;

	/* Note: DEAUTHENTICATION does not have category nor action values. see SPEC IEEE 802.11-2016: Table 9-47 Category values */

	if (isStaPmf == FALSE)
	{
		pdPointer->pdType = PD_TYPE_MANAGEMENT_UNENC;
	}
	else
	{
		pdPointer->pdType = PD_TYPE_MANAGEMENT_ENC;
	}

	pdPointer->txQVapId = StaDb->vapId;
	pdPointer->txQGroupId = HW_TX_Q_TYPE_GPHP;
	pdPointer->mcUnicast = UNICAST;
	/*Build CSA frame - sets also len in PD*/
	csaManager_BuildCsaUcDeAuth(StaDb->pd);
	/*Send TX IND to VAP manager*/
	csaManager_VapManagerTxInd(StaDb->pd);
	/*Change State to wait for TX CFM*/
	CsaManager_StaManagerChangeState(CSA_MANAGER_STA_MANAGER_WAIT_TX_CFM_STATE, StaDb); 
}


/**********************************************************************************
CsaManager_StaManagerSendCsa 

Description:
------------
Utility function that builds the CSA and sends TX IND to VAP Manager

Input: 
-----	
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerSendCsa(CsaManagerStaManagerDb_t *StaDb)
{
	TxPd_t *pdPointer = (TxPd_t *)StaDb->pd;
	bool isStaPmf = CsaManager_StaManagerIsStaPmf(StaDb->sid);

	pdPointer->txQStaId= StaDb->sid;
	pdPointer->txQTid = MANAGEMENT_TID;
	pdPointer->mgmtFrameSubtype = MGMT_FRAME_SUBTYPE_ACTION;

	pdPointer->mgmtActionCode = SPECTRUM_MAN_CATEGORY;
	pdPointer->mgmtActionValue = SPECTRUM_MAN_CATEGORY_CHANNEL_SWITCH_ANNOUNCEMENT;

	if (isStaPmf == FALSE)
	{
		pdPointer->pdType = PD_TYPE_CH_SWITCH_ANN;
	}
	else
	{
		pdPointer->pdType = PD_TYPE_CH_SWITCH_ANN_ENC;
	}

	pdPointer->txQVapId = StaDb->vapId;
	pdPointer->txQGroupId = HW_TX_Q_TYPE_GPHP;
	pdPointer->mcUnicast = UNICAST;
	/*Build CSA frame - sets also len in PD*/
	csaManager_BuildCsa(StaDb->pd);
	/*Send TX IND to VAP manager*/
	csaManager_VapManagerTxInd(StaDb->pd);
	/*Change State to wait for TX CFM*/
	CsaManager_StaManagerChangeState(CSA_MANAGER_STA_MANAGER_WAIT_TX_CFM_STATE, StaDb); 
}


/**********************************************************************************

CsaManager_StaManagerStartTx 


Description:
------------
Utility function that starts CSA Transmission


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerStartTx(CsaManagerStaManagerDb_t *StaDb)
{
	uint8 allocationStatus = FALSE;
	RmPdRequestFillParameters_t pdRequestFillParameters;
	CsaManager_csaType_e csaType;

	memset(&pdRequestFillParameters, 0, sizeof(RmPdRequestFillParameters_t));	
	pdRequestFillParameters.context = StaDb->sid;
	pdRequestFillParameters.returnMsgType = CSA_MANAGER_PD_ALLOC;
	pdRequestFillParameters.returnTaskId = TASK_TX_MANAGER;
	allocationStatus = ResourceManager_GetDescriptorRequest(DESC_POOL_CSA_MANAGER, &pdRequestFillParameters);
	//("CsaManager_StaManagerStartTxInOpen allocation status %d ", allocationStatus);
	if(allocationStatus)
	{
		
		/*Store PD pointer*/
		StaDb->pd = pdRequestFillParameters.packetDescriptor;
		
		csaType = CsaManager_GetCurrentTxSegment(StaDb->vapId);

		if (CSA_TYPE_CSA == csaType)
		{
			/*Build and send CSA*/
			CsaManager_StaManagerSendCsa(StaDb);
		}
		else if (CSA_TYPE_UC_DEAUTH == csaType)
		{
			/*Build and send CSA*/
			CsaManager_StaManagerSendCsaUcDeauth(StaDb);
		}
		else
		{
			FATAL("CsaManager_StaManagerStartTx: WRONG csaType");
		}
	}
	else
	{
		/* PD allocation failed */		
		StaDb->requestId = pdRequestFillParameters.requestId;
		/*Change State to wait for PD*/
		CsaManager_StaManagerChangeState(CSA_MANAGER_STA_MANAGER_WAIT_PD_STATE, StaDb); 
	}
}

/**********************************************************************************

CsaManager_StaManagerLockStaMng 


Description:
------------
Utility function that locks the STA management queue in case we want to remove a retry from head of queue


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
#if defined CSA_REMOVE_RETRY	
static void CsaManager_StaManagerLockStaMng(CsaManagerStaManagerDb_t *StaDb)
{
	RequesterLockParams_t lockRequesterParams;
	uint8 isStaQos = MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_WME);

	/*Need to serialize Lock Requests - use request ID in DB, it is unused at this stage*/
	StaDb->requestId = CSA_MANAGER_STA_MANAGER_INVALID_LOCK_IDX;
	if (CsaManagerStaManager_GlobalDb.staMngLockHead == CSA_MANAGER_STA_MANAGER_INVALID_LOCK_IDX)
	{
		/*Set current entry as head*/
		CsaManagerStaManager_GlobalDb.staMngLockHead = StaDb->sid;
		/*Send lock request for STA Management queue*/
		memset(&lockRequesterParams, 0, sizeof(RequesterLockParams_t));
		/*Request SW Locker to lock TID 7*/
		lockRequesterParams.callerContext = NULL;
		lockRequesterParams.returnMsg = CSA_MANAGER_STA_MNG_LOCK;
		lockRequesterParams.returnTask = TASK_TX_MANAGER;
		if (isStaQos == TRUE)
		{
			Locker_LockSingleQueue(HW_TX_Q_TYPE_STA_TID, StaDb->sid, IEEE802_1D_VO_2, &lockRequesterParams);
		}
		else
		{
			/*For non-QoS everything is in Queue 0*/
			Locker_LockSingleQueue(HW_TX_Q_TYPE_STA_TID, StaDb->sid, IEEE802_1D_BE_1, &lockRequesterParams);
		}
	}
	else
	{
		/*Link current entry to last entry*/
		CsaManagerStaManager_Db[CsaManagerStaManager_GlobalDb.staMngLockTail].requestId = StaDb->sid;
	}
	/*Set entry as last entry*/
	CsaManagerStaManager_GlobalDb.staMngLockTail = StaDb->sid;
	/*Change State to Wait for Lock*/
	CsaManager_StaManagerChangeState(CSA_MANAGER_STA_MANAGER_WAIT_LOCK_STATE, StaDb); 
}

/**********************************************************************************

CsaManager_StaManagerHandleLockStaMng 


Description:
------------
Utility function that handles Lock STa Management


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerHandleLockStaMng()
{
	/*Get current entry*/
	StaId sid = CsaManagerStaManager_GlobalDb.staMngLockHead;
	CsaManagerStaManagerDb_t *StaDb;
	RequesterLockParams_t lockRequesterParams;
	uint8 isStaQos;

	/*Set head as next entry*/
	CsaManagerStaManager_GlobalDb.staMngLockHead = CsaManagerStaManager_Db[sid].requestId;
	/*Do we have more entries?*/
	if (CsaManagerStaManager_GlobalDb.staMngLockHead != CSA_MANAGER_STA_MANAGER_INVALID_LOCK_IDX)
	{
		/*Get entry*/
		sid = CsaManagerStaManager_GlobalDb.staMngLockHead;
		StaDb = &CsaManagerStaManager_Db[sid];
		isStaQos = MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_WME);
		/*Send lock request for STA Management queue*/
		memset(&lockRequesterParams, 0, sizeof(RequesterLockParams_t));
		/*Request SW Locker to lock TID 7*/
		lockRequesterParams.callerContext = NULL;
		lockRequesterParams.returnMsg = CSA_MANAGER_STA_MNG_LOCK;
		lockRequesterParams.returnTask = TASK_TX_MANAGER;
		if (isStaQos == TRUE)
		{
			Locker_LockSingleQueue(HW_TX_Q_TYPE_STA_TID, StaDb->sid, IEEE802_1D_VO_2, &lockRequesterParams);
		}
		else
		{
			/*For non-QoS everything is in Queue 0*/
			Locker_LockSingleQueue(HW_TX_Q_TYPE_STA_TID, StaDb->sid, IEEE802_1D_BE_1, &lockRequesterParams);
		}
	}
	else
	{
		/*Clear tail as well*/
		CsaManagerStaManager_GlobalDb.staMngLockTail = CSA_MANAGER_STA_MANAGER_INVALID_LOCK_IDX;
	}
}

/**********************************************************************************

CsaManager_StaManagerUnlockStaMngQueue 


Description:
------------
Utility function that unlocks the STA management queue in case we want to remove a retry from head of queue


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerUnlockStaMngQueue(CsaManagerStaManagerDb_t *StaDb)
{
	uint8 isStaQos = MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_WME);
	
	/*Unlock TID 7*/
	if (isStaQos == TRUE)
	{
		Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID,StaDb->sid, (0x1 << IEEE802_1D_VO_2));
	}
	else
	{
		/*In non-Qos everything goes to TID 0*/
		Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID,StaDb->sid, (0x1 << IEEE802_1D_BE_1));
	}
}

/**********************************************************************************

CsaManager_StaManagerRemoveRetry 


Description:
------------
Utility function that removes any pending management retry from head of STA management queue


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerRemoveRetry(CsaManagerStaManagerDb_t *StaDb)
{
	TxPd_t *currentPacketDescriptor = NULL;
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
	uint8 isStaQos = MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_WME);
	
	/*Peek head packet*/
	memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
	hwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_STA_TID;
	hwQueueManagerRequestParams.primaryAddr = StaDb->sid;
	if (isStaQos == TRUE)
	{
		hwQueueManagerRequestParams.secondaryAddr = IEEE802_1D_VO_2;
	}
	else
	{
		/*For non-QoS everything is on TID 0*/
		hwQueueManagerRequestParams.secondaryAddr = IEEE802_1D_BE_1;
	}
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	HwQManager_PeekHeadPacket(&hwQueueManagerRequestParams);
    currentPacketDescriptor = (TxPd_t *)hwQueueManagerRequestParams.pHeadDesc;
	/*packet is management pending for retry*/
	/*For non-QoS drop whatever is on head of queue if it is pending for retransmit*/
  	if (((TxPd_t *)NULL_PD != currentPacketDescriptor) && 
		((isStaQos == FALSE) ||
		 (currentPacketDescriptor->pdType == PD_TYPE_MANAGEMENT_UNENC) ||  
		 (currentPacketDescriptor->pdType == PD_TYPE_MANAGEMENT_ENC)) &&
		(currentPacketDescriptor->retransmissionIndication))
    {
		HwQManager_PopPacket(&hwQueueManagerRequestParams);
		hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
		hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_DONE_LIST_LIBERATOR;
		HwQManager_PushPacketToTail(&hwQueueManagerRequestParams);
    }
}
#endif

/**********************************************************************************

CsaManager_StaManagerLockGPHP 


Description:
------------
Utility function to lock GPHP


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerLockGPHP(CsaManagerStaManagerDb_t *StaDb)
{
	RequesterLockParams_t lockRequesterParams;

	/*Need to serialize Lock Requests - use request ID in DB, it is unused at this stage*/
	StaDb->requestId = CSA_MANAGER_STA_MANAGER_INVALID_LOCK_IDX;
	if (CsaManagerStaManager_GlobalDb.gphpLockHead == CSA_MANAGER_STA_MANAGER_INVALID_LOCK_IDX)
	{
		/*Set current entry as head*/
		CsaManagerStaManager_GlobalDb.gphpLockHead = StaDb->sid;
		/*Send Lock Request to GPHP*/
		memset(&lockRequesterParams, 0, sizeof(RequesterLockParams_t));
		/*Request SW Locker to Wait for Lock Stop*/
		lockRequesterParams.callerContext = (void *)StaDb;
		lockRequesterParams.returnMsg = CSA_MANAGER_STA_GPHP_LOCK;
		lockRequesterParams.returnTask = TASK_TX_MANAGER;
		Locker_LockSingleQueue(HW_TX_Q_TYPE_GPHP, StaDb->vapId, 0, &lockRequesterParams);
	}
	else
	{
		/*Link current entry to last entry*/
		CsaManagerStaManager_Db[CsaManagerStaManager_GlobalDb.gphpLockTail].requestId = StaDb->sid;
	}
	/*Set entry as last entry*/
	CsaManagerStaManager_GlobalDb.gphpLockTail = StaDb->sid;
	/*Change State to Wait for Lock*/
	CsaManager_StaManagerChangeState(CSA_MANAGER_STA_MANAGER_WAIT_TX_CFM_LOCK_STATE, StaDb); 
}

/**********************************************************************************

CsaManager_StaManagerHandleLockGPHP 


Description:
------------
Utility function that handles GPHP lock confirmations


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerHandleLockGPHP()
{
	/*Get current entry*/
	StaId sid = CsaManagerStaManager_GlobalDb.gphpLockHead;
	CsaManagerStaManagerDb_t *StaDb;
	RequesterLockParams_t lockRequesterParams;

	/*Set head as next entry*/
	CsaManagerStaManager_GlobalDb.gphpLockHead = CsaManagerStaManager_Db[sid].requestId;
	/*Do we have more entries?*/
	if (CsaManagerStaManager_GlobalDb.gphpLockHead != CSA_MANAGER_STA_MANAGER_INVALID_LOCK_IDX)
	{
		/*Get entry*/
		sid = CsaManagerStaManager_GlobalDb.gphpLockHead;
		StaDb = &CsaManagerStaManager_Db[sid];
		/*Send lock request for STA Management queue*/
		memset(&lockRequesterParams, 0, sizeof(RequesterLockParams_t));
		lockRequesterParams.callerContext = (void *)StaDb;
		lockRequesterParams.returnMsg = CSA_MANAGER_STA_GPHP_LOCK;
		lockRequesterParams.returnTask = TASK_TX_MANAGER;
		Locker_LockSingleQueue(HW_TX_Q_TYPE_GPHP, StaDb->vapId, 0, &lockRequesterParams);
	}
	else
	{
		/*Clear tail as well*/
		CsaManagerStaManager_GlobalDb.gphpLockTail= CSA_MANAGER_STA_MANAGER_INVALID_LOCK_IDX;
	}
}


/**********************************************************************************

CsaManager_StaManagerRemovePdFromGPHP 


Description:
------------
Utility function that removes s SID from GPHP queue


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static bool CsaManager_StaManagerRemovePdFromGPHP(CsaManagerStaManagerDb_t *StaDb)
{
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
	bool found = FALSE;
	TxPd_t *prevPd = (TxPd_t *)NULL_PD;
	TxPd_t *pd;

	memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	hwQueueManagerRequestParams.primaryAddr = StaDb->vapId;
	hwQueueManagerRequestParams.secondaryAddr = 0;
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
	hwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_GPHP;
	HwQManager_FlushQ(&hwQueueManagerRequestParams);
	/*Search for SID in list*/
	if (hwQueueManagerRequestParams.pHeadDesc != (TxPd_t *)NULL_PD)
	{
		pd = hwQueueManagerRequestParams.pHeadDesc;
		while (pd != (TxPd_t *)NULL_PD)
		{
			if (pd->txQStaId == StaDb->sid)
			{
				/*found it!*/
				if (prevPd != (TxPd_t *)NULL_PD)
				{
					prevPd->nextPdPointer = SET_NEXT_PD(GET_NEXT_PD(pd));
				}
				else
				{
					/*PD was head of queue*/
					hwQueueManagerRequestParams.pHeadDesc = (void *)GET_NEXT_PD(pd);
				}
				if (pd->nextPdPointer == NEXT_PD_NULL)
				{
					/*PD was tail of queue*/
					hwQueueManagerRequestParams.pTailDesc = (void *)prevPd;
				}
				/*Restore DRAM pointer*/
				pd->packetPointer = CONVERT_DMA_SHRAM_ADDR_TO_WLAN_SHRAM_ADDR(pd->packetPointer);  
				found = TRUE;
				break;
			}
			prevPd = pd;
			pd = (TxPd_t *)GET_NEXT_PD(pd);
		}
		/*Put List back to GPHP - we may have removed the only PD so check we have something in list*/
		if (hwQueueManagerRequestParams.pHeadDesc != (TxPd_t *)NULL_PD)
		{
			HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams); 
		}
	}
	return (found);
}

/**********************************************************************************

CsaManager_StaManagerAddStaInIdle 


Description:
------------
STA ADD in Idle State Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerAddStaInIdle(CsaManagerStaManagerDb_t *StaDb)
{
	K_MSG *psMsg = StaDb->psMsg;
	UMI_STA_ADD *pAddSta = (UMI_STA_ADD *)pK_MSG_DATA(psMsg);

	/*Store flags for later use*/
	StaDb->flags = pAddSta->u8Flags;
	/*Store VAP*/
	StaDb->vapId = pAddSta->u8VapIndex;
	/*Clear message*/
	StaDb->psMsg = NULL;
	/*Set state to open*/
	CsaManager_StaManagerChangeState(CSA_MANAGER_STA_MANAGER_OPEN_STATE, StaDb); 
	/*Send Indication to CSA VAP Manager*/
	CsaManager_VapManagerAddStaInd(StaDb->vapId, StaDb->sid);
	/*Send confirmation to BSS Manager*/
	CsaManager_StaManagerSendConfirm(StaDb);
}

/**********************************************************************************

CsaManager_StaManagerRemoveStaInOpen 


Description:
------------
STA Remove in Open State Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerRemoveStaInOpen(CsaManagerStaManagerDb_t *StaDb)
{
	/*We are done*/
	CsaManager_StaManagerRemoveStaDone(StaDb);
}

/**********************************************************************************

CsaManager_StaManagerRemoveStaInWaitTxCfm 


Description:
------------
STA Remove in Wait TX CFM State Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerRemoveStaInWaitTxCfm(CsaManagerStaManagerDb_t *StaDb)
{
	/*PD may be in GPHP, SEC queue or on the way back*/
	/*Try to remove it from SEC queue  found in SEC quueue*/
	if (CsaManager_VapManagerRemoveStaFromSecQueue(StaDb->vapId, StaDb->sid) == TRUE)
	{
		/*PD removed from SEC queue*/
		/*Return PD*/
		ResourceManager_ReleaseDescriptor(StaDb->pd, DESC_POOL_CSA_MANAGER);
		/*Send TX CFM to VAP manager*/
		csaManager_VapManagerTxCfm(StaDb->vapId, StaDb->sid);
		/*We are done*/
		CsaManager_StaManagerRemoveStaDone(StaDb);
	}
	else
	{
		CsaManager_StaManagerLockGPHP(StaDb);
	}
}

/**********************************************************************************

CsaManager_StaManagerRemoveStaInWaitPd 


Description:
------------
STA Remove in Wait for PD State Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerRemoveStaInWaitPd(CsaManagerStaManagerDb_t *StaDb)
{
	uint8 removeStatus = FALSE;
	
	/*Tell Resource Manager we don't need the resource anymore*/
	removeStatus = ResourceManager_RemoveRequest(StaDb->requestId, DESC_POOL_CSA_MANAGER);
	if (removeStatus)
	{
		/*Request remove succesfully*/
		/*Send TX CFM to VAP manager*/
		csaManager_VapManagerTxCfm(StaDb->vapId, StaDb->sid);
		/*We are done*/
		CsaManager_StaManagerRemoveStaDone(StaDb);
	}
	else
	{
		/*Request is on the way - wait for it - change state to wait PD Close State*/
		CsaManager_StaManagerChangeState(CSA_MANAGER_STA_MANAGER_WAIT_PD_CLOSE_STATE, StaDb); 
	}	
}

/**********************************************************************************

CsaManager_StaManagerRemoveStaInWaitPdStop 


Description:
------------
STA Remove in Wait for PD Stop Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerRemoveStaInWaitPdStop(CsaManagerStaManagerDb_t *StaDb)
{
	/*We already know PD is on the way - just move to wait PD close State*/
	CsaManager_StaManagerChangeState(CSA_MANAGER_STA_MANAGER_WAIT_PD_CLOSE_STATE, StaDb); 
}

/**********************************************************************************

CsaManager_StaManagerRemoveStaInWaitLock 


Description:
------------
STA Remove in Wait for Lock Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
#if defined CSA_REMOVE_RETRY	
static void CsaManager_StaManagerRemoveStaInWaitLock(CsaManagerStaManagerDb_t *StaDb)
{
	CsaManager_StaManagerChangeState(CSA_MANAGER_STA_MANAGER_WAIT_LOCK_CLOSE_STATE, StaDb); 
}

/**********************************************************************************

CsaManager_StaManagerRemoveStaInWaitLockStop 


Description:
------------
STA Remove in Wait for Lock Stop Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerRemoveStaInWaitLockStop(CsaManagerStaManagerDb_t *StaDb)
{
	CsaManager_StaManagerChangeState(CSA_MANAGER_STA_MANAGER_WAIT_LOCK_CLOSE_STATE, StaDb); 
}
#endif

/**********************************************************************************

CsaManager_StaManagerStartTxInOpen 


Description:
------------
TX START in Open Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerStartTxInOpen(CsaManagerStaManagerDb_t *StaDb)
{
#if defined CSA_REMOVE_RETRY	
	if (CsaManagerStaManager_GlobalDb.removeRetry == FALSE)
	{
#endif		
		CsaManager_StaManagerStartTx(StaDb);
#if defined CSA_REMOVE_RETRY	
	}
	else
	{
		CsaManager_StaManagerLockStaMng(StaDb);
	}
#endif	
}

/**********************************************************************************

CsaManager_StaManagerStopTxInOpen 


Description:
------------
TX Stop in Open Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerStopTxInOpen(CsaManagerStaManagerDb_t *StaDb)
{
	/*Just send Confirmation to VAP manager*/
	csaManager_VapManagerTxCfm(StaDb->vapId, StaDb->sid);
}

/**********************************************************************************

CsaManager_StaManagerStopTxInWaitPd 


Description:
------------
TX Stop in Wait PD Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerStopTxInWaitPd(CsaManagerStaManagerDb_t *StaDb)
{
	uint8 removeStatus = FALSE;
	
	/*Tell Resource Manager we don't need the resource anymore*/
	removeStatus = ResourceManager_RemoveRequest(StaDb->requestId, DESC_POOL_CSA_MANAGER);
	if (removeStatus)
	{
		/*Request remove succesfully*/
		/*Change State to Open*/
		CsaManager_StaManagerChangeState(CSA_MANAGER_STA_MANAGER_OPEN_STATE, StaDb); 
		/*Send TX CFM to VAP manager*/
		csaManager_VapManagerTxCfm(StaDb->vapId, StaDb->sid);		
	}
	else
	{
		/*Request is on the way - wait for it - change state to wait PD Stop State*/
		CsaManager_StaManagerChangeState(CSA_MANAGER_STA_MANAGER_WAIT_PD_STOP_STATE, StaDb); 
	}	
}

/**********************************************************************************

CsaManager_StaManagerStopTxInWaitLock 


Description:
------------
TX Stop in Wait Lock Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
#if defined CSA_REMOVE_RETRY	
static void CsaManager_StaManagerStopTxInWaitLock(CsaManagerStaManagerDb_t *StaDb)
{
	CsaManager_StaManagerChangeState(CSA_MANAGER_STA_MANAGER_WAIT_LOCK_STOP_STATE, StaDb); 
}
#endif

/**********************************************************************************

CsaManager_StaManagerTxCfmInWaitTxCfm 


Description:
------------
TX CFM in Wait TX CFM Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerTxCfmInWaitTxCfm(CsaManagerStaManagerDb_t *StaDb)
{
	/*We are done*/
	CsaManager_StaManagerTxCfmDone(StaDb);
}

/**********************************************************************************

CsaManager_StaManagerTxCfmInWaitTxCfmLock 


Description:
------------
TX CFM in Wait TX CFM Lock Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerTxCfmInWaitTxCfmLock(CsaManagerStaManagerDb_t *StaDb)
{
	/*Return PD*/
    ResourceManager_ReleaseDescriptor(StaDb->pd, DESC_POOL_CSA_MANAGER);
	/*Move to Wait GPHP Lock Close*/
	CsaManager_StaManagerChangeState(CSA_MANAGER_STA_MANAGER_WAIT_GPHP_LOCK_CLOSE_STATE, StaDb); 
}

/**********************************************************************************

CsaManager_StaManagerTxCfmInWaitTxCfmClose 


Description:
------------
TX CFM in Wait TX CFM Close Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerTxCfmInWaitTxCfmClose(CsaManagerStaManagerDb_t *StaDb)
{
    /* Release the PD */
    ResourceManager_ReleaseDescriptor(StaDb->pd, DESC_POOL_CSA_MANAGER);
	/*Send TX CFM to VAP manager*/
	csaManager_VapManagerTxCfm(StaDb->vapId, StaDb->sid);
	/*We are done*/
	CsaManager_StaManagerRemoveStaDone(StaDb);
}

/**********************************************************************************

CsaManager_StaManagerPdAllocInWaitPd 


Description:
------------
PD Alloc In Wait PD Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerPdAllocInWaitPd(CsaManagerStaManagerDb_t *StaDb)
{
	CsaManager_csaType_e csaType;
	
	csaType = CsaManager_GetCurrentTxSegment(StaDb->vapId);
	
	if (CSA_TYPE_CSA == csaType)
	{
		/*Build and send CSA*/
		CsaManager_StaManagerSendCsa(StaDb);
	}
	else if (CSA_TYPE_UC_DEAUTH == csaType)
	{
		/*Build and send CSA*/
		CsaManager_StaManagerSendCsaUcDeauth(StaDb);
	}
	else
	{
		FATAL("CsaManager_StaManagerStartTx: WRONG csaType");
	}
}

/**********************************************************************************

CsaManager_StaManagerPdAllocInWaitPdStop 


Description:
------------
PD Alloc In Wait PD Stop Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerPdAllocInWaitPdStop(CsaManagerStaManagerDb_t *StaDb)
{
	/*We are done*/
	CsaManager_StaManagerTxCfmDone(StaDb);
}

/**********************************************************************************

CsaManager_StaManagerPdAllocInWaitPdClose 


Description:
------------
PD Alloc In Wait PD Close Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerPdAllocInWaitPdClose(CsaManagerStaManagerDb_t *StaDb)
{
    /* Release the PD */
    ResourceManager_ReleaseDescriptor(StaDb->pd, DESC_POOL_CSA_MANAGER);
	/*Send TX CFM to VAP manager*/
	csaManager_VapManagerTxCfm(StaDb->vapId, StaDb->sid);
	/*We are done*/
	CsaManager_StaManagerRemoveStaDone(StaDb);
}

/**********************************************************************************

CsaManager_StaManagerLockCfmInWaitLock 


Description:
------------
Lock CFM In TX CFM Lock Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerLockCfmInTxCfmLock(CsaManagerStaManagerDb_t *StaDb)
{
	if (CsaManager_StaManagerRemovePdFromGPHP(StaDb) == TRUE)
	{
		/*PD removed*/
		/*Return PD*/
		ResourceManager_ReleaseDescriptor(StaDb->pd, DESC_POOL_CSA_MANAGER);
		/*Send TX CFM to VAP manager*/
		csaManager_VapManagerTxCfm(StaDb->vapId, StaDb->sid);
		/*We are done*/
		CsaManager_StaManagerRemoveStaDone(StaDb);
	}
	else
	{
		/*PD not found, must be on its way back - wait for it in Wait for TX CFM Close state*/
		CsaManager_StaManagerChangeState(CSA_MANAGER_STA_MANAGER_WAIT_TX_CFM_CLOSE_STATE, StaDb); 
	}
	/*Unlock GPHP*/
	Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_GPHP, StaDb->vapId, (0x1 << 0));
}

/**********************************************************************************

CsaManager_StaManagerLockCfmInGPHPLockClose 


Description:
------------
Lock CFM In GPHP Lock CLose Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerLockCfmInGPHPLockClose(CsaManagerStaManagerDb_t *StaDb)
{
	/*Unlock GPHP*/
	Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_GPHP, StaDb->vapId, (0x1 << 0));
	/*Send TX CFM to VAP manager*/
	csaManager_VapManagerTxCfm(StaDb->vapId, StaDb->sid);
	/*We are done*/
	CsaManager_StaManagerRemoveStaDone(StaDb);
}

/**********************************************************************************

CsaManager_StaManagerLockCfmInWaitLock 


Description:
------------
Lock CFM In Wait Lock Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
#if defined CSA_REMOVE_RETRY	
static void CsaManager_StaManagerLockCfmInWaitLock(CsaManagerStaManagerDb_t *StaDb)
{
	/*Need to look at the queue and remove any pending management for retry*/
	CsaManager_StaManagerRemoveRetry(StaDb);
	/*Unlock TID 7*/
	CsaManager_StaManagerUnlockStaMngQueue(StaDb);
	/*And now continue*/
	CsaManager_StaManagerStartTx(StaDb);
}

/**********************************************************************************

CsaManager_StaManagerLockCfmInWaitStop 


Description:
------------
Lock CFM In Wait Lock Stop Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerLockCfmInWaitLockStop(CsaManagerStaManagerDb_t *StaDb)
{
	CsaManager_StaManagerUnlockStaMngQueue(StaDb);
	/*Change State to Open*/
	CsaManager_StaManagerChangeState(CSA_MANAGER_STA_MANAGER_OPEN_STATE, StaDb);
	/*Send TX CFM to VAP manager*/
	csaManager_VapManagerTxCfm(StaDb->vapId, StaDb->sid);
}

/**********************************************************************************

CsaManager_StaManagerLockCfmInWaitClose 


Description:
------------
Lock CFM In Wait Lock Stop Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerLockCfmInWaitLockClose(CsaManagerStaManagerDb_t *StaDb)
{
	CsaManager_StaManagerUnlockStaMngQueue(StaDb);
	/*Send TX CFM to VAP manager*/
	csaManager_VapManagerTxCfm(StaDb->vapId, StaDb->sid);
	/*We are done*/
	CsaManager_StaManagerRemoveStaDone(StaDb);
}
#endif

/**********************************************************************************

CsaManager_StaManagerIgnore 


Description:
------------
Ignore event handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerIgnore(CsaManagerStaManagerDb_t *StaDb)
{
	UNUSED_PARAM(StaDb);	
}

/**********************************************************************************

CsaManager_StaManagerFatal 


Description:
------------
handles unexpected event


Input: 
-----	
TxManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StaManagerFatal(CsaManagerStaManagerDb_t *StaDb)
{
	UNUSED_PARAM(StaDb);	
	FATAL("Csa Manager STA Manager Fatal");
}

/*---------------------------------------------------------------------------------
/						global Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

csaManagerVapOpenInd 


Description:
------------
VAP ADD message handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void csaManagerStaOpen(K_MSG* csaManagerMessage)
{
	BssManagerStaManagerReq_t *staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(csaManagerMessage);
	K_MSG *psMsg = staManagerReq->psMsg;
	UMI_STA_ADD *pAddSta = (UMI_STA_ADD *)pK_MSG_DATA(psMsg);
	StaId sid = pAddSta->u16SID;
	CsaManagerStaManagerDb_t *StaDb = &CsaManagerStaManager_Db[sid];

	DEBUG_ASSERT(StaDb->psMsg == NULL);
	/*Store message*/
	StaDb->psMsg = psMsg;
	/*Call handler*/
	CsaManager_StaManagerAddSta[StaDb->state](StaDb);
}

/**********************************************************************************

csaManagerVapRemove 


Description:
------------
VAP Remove message handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void csaManagerStaRemove(K_MSG* csaManagerMessage)
{
	BssManagerStaManagerReq_t *staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(csaManagerMessage);
	K_MSG *psMsg = staManagerReq->psMsg;
	UMI_STOP_TRAFFIC *pRemoveSta = (UMI_STOP_TRAFFIC *)pK_MSG_DATA(psMsg);
	StaId sid = pRemoveSta->u16SID;
	CsaManagerStaManagerDb_t *StaDb = &CsaManagerStaManager_Db[sid];

	DEBUG_ASSERT(StaDb->psMsg == NULL);
	/*Store message*/
	StaDb->psMsg = psMsg;
	/*Call handler*/
	CsaManager_StaManagerRemoveSta[StaDb->state](StaDb);
}

/**********************************************************************************

CsaManager_StaManagerIsStaPmf 


Description:
------------
for a given SID returns if it is PMF


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
bool CsaManager_StaManagerIsStaPmf(StaId sid)
{
	CsaManagerStaManagerDb_t *StaDb = &CsaManagerStaManager_Db[sid];
	uint8 isStaPmf = MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_MFP);

	return (isStaPmf);
}

/**********************************************************************************

CsaManager_StaManagerGetVap 


Description:
------------
for a given SID returns its VAP


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
uint8 CsaManager_StaManagerGetVap(StaId sid)
{
	CsaManagerStaManagerDb_t *StaDb = &CsaManagerStaManager_Db[sid];
	return (StaDb->vapId);
}

/**********************************************************************************

CsaManager_StaManagerTxStart 


Description:
------------
CSA Start TX REQ from Channel Switch Manager


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void CsaManager_StaManagerTxStart(StaId sid)
{
	CsaManagerStaManagerDb_t *StaDb = &CsaManagerStaManager_Db[sid];
	
	/*Call handler*/
	CsaManager_StaManagerStartTxReq[StaDb->state](StaDb);
}

/**********************************************************************************

CsaManager_StaManagerTxStop 


Description:
------------
CSA Stop TX REQ from Channel Switch Manager


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void CsaManager_StaManagerTxStop(StaId sid)
{
	CsaManagerStaManagerDb_t *StaDb = &CsaManagerStaManager_Db[sid];
	
	/*Call handler*/
	CsaManager_StaManagerStopTxReq[StaDb->state](StaDb);
}

/**********************************************************************************

csaManagerStaGphpLockCfm






Description:
------------
GPHP Lock Confirm message handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void csaManagerStaGphpLockCfm(K_MSG* csaManagerMessage)
{
	LockReqCb_t *lockReg = (LockReqCb_t *)pK_MSG_DATA(csaManagerMessage);
	CsaManagerStaManagerDb_t *StaDb = (CsaManagerStaManagerDb_t *)lockReg->callerContext;

	/*Handle pending list*/
	CsaManager_StaManagerHandleLockGPHP();
	/*Call handler*/
	CsaManager_StaManagerGphpLockCfm[StaDb->state](StaDb);
}

/**********************************************************************************
csaManagerStaMngLockCfm

Description:
------------
STA MNG Lock Confirm message handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
#if defined CSA_REMOVE_RETRY	
void csaManagerStaMngLockCfm(K_MSG* csaManagerMessage)
{
	LockReqCb_t *lockReg = (LockReqCb_t *)pK_MSG_DATA(csaManagerMessage);
	StaId sid = lockReg->stationOrVapNum;
	CsaManagerStaManagerDb_t *StaDb = &CsaManagerStaManager_Db[sid];

	/*Handle pending list*/
	CsaManager_StaManagerHandleLockStaMng();
	/*Call handler*/
	CsaManager_StaManagerStaLockCfm[StaDb->state](StaDb);
}
#endif


/**********************************************************************************
CsaManager_StaManagerTxCfmMessageHandler

Description:
------------
TX Confirm message handler

Input: 
-----	
			
Output:
-------
	
Returns:
--------
	void - 
	
**********************************************************************************/
void CsaManager_StaManagerTxCfmMessageHandler(K_MSG* csaManagerMessage)
{
	/*From PD get SID*/
	CsaManagerPacketConfirmedMessage_t *csaMsg = (CsaManagerPacketConfirmedMessage_t *)pK_MSG_DATA(csaManagerMessage);
	TxPd_t *pd = csaMsg->pd;
	StaId sid = pd->txQStaId;
	CsaManagerStaManagerDb_t *StaDb = &CsaManagerStaManager_Db[sid];

	/*Call handler*/
	CsaManager_StaManagerTxCfm[StaDb->state](StaDb);
}

/**********************************************************************************

csaManagerPdAlloc






Description:
------------
PD Alloc Confirm message handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void csaManagerPdAlloc(K_MSG* csaManagerMessage)
{
	RmPdFreeDescResponse_t *pdMsg = (RmPdFreeDescResponse_t *)pK_MSG_DATA(csaManagerMessage);
	StaId sid = pdMsg->context;
	CsaManagerStaManagerDb_t *StaDb = &CsaManagerStaManager_Db[sid];

	/*Store PD pointer*/
	StaDb->pd = pdMsg->packetDescriptor;
	/*Release the request*/
	ResourceManager_ReleaseRequest(StaDb->requestId, DESC_POOL_CSA_MANAGER);
	/*Call handler*/
	CsaManager_StaManagerPdAlloc[StaDb->state](StaDb);
}


/**********************************************************************************

csaManager_Init



Description:
------------
Initiailize CSA MAnager VAP Manager


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void CsaManager_StaManagerInit()
{
	StaId i;
	
#if defined CSA_REMOVE_RETRY	
	CsaManagerStaManager_GlobalDb.removeRetry = FALSE;
#endif
	/*Intiialize Database*/
	for (i = 0; i < HW_NUM_OF_STATIONS; i++)
	{
		CsaManagerStaManager_Db[i].sid = i;
		CsaManagerStaManager_Db[i].state = CSA_MANAGER_STA_MANAGER_IDLE_STATE;
		CsaManagerStaManager_Db[i].pd = (TxPd_t *)NULL_PD;
		CsaManagerStaManager_Db[i].psMsg = NULL;
	}
	/*Initialize serializer indeces*/
#if defined CSA_REMOVE_RETRY	
	CsaManagerStaManager_GlobalDb.staMngLockHead = CSA_MANAGER_STA_MANAGER_INVALID_LOCK_IDX;
	CsaManagerStaManager_GlobalDb.staMngLockTail = CSA_MANAGER_STA_MANAGER_INVALID_LOCK_IDX;
#endif	
	CsaManagerStaManager_GlobalDb.gphpLockHead = CSA_MANAGER_STA_MANAGER_INVALID_LOCK_IDX;
	CsaManagerStaManager_GlobalDb.gphpLockTail = CSA_MANAGER_STA_MANAGER_INVALID_LOCK_IDX;
}



