/***********************************************************************************
 File:			CsaManager_VapManager.c
 Module:		CSA Manager VAP Manager
 Purpose: 		handle CSA Manager VAP Manager events
 Description:   
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
	//TBD to check which files can be removed
#include "VapDatabase_Api.h"
#include "BSSmanager_API.h"
#include "TxManager_Api.h"
#include "TxPacketsClassifier_API.h"
#include "TsManager_API.h"
#include "TxSelector_Api.h"
#include "HostInterface_API.h"
#include "Utils_Api.h"
#include "ResourceManager_API.h"
#include "StaDatabase_Api.h"
#include "Locker_Api.h"
#include "OSAL_Kmsg.h"
#include "ErrorHandler_Api.h"
#include "HwQManager_API.h"
#include "HwGlobalDefinitions.h"
#include "ShramPacketDescriptors.h"
#include "stringLibApi.h"
#include "um_interface.h"
#include "PacketDescriptor.h"
#include "HwMemoryMap.h"
#include "CsaManager_Api.h"
#include "CsaManager.h"
#include "CsaManager_VapManager.h"
#include "CsaManager_StaManager.h"
#include "loggerAPI.h"


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_CSA_MANAGER 
#define LOG_LOCAL_FID 5


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/
#define CSA_MANAGER_VAP_MANAGER_INVALID_LOCK_IDX 0xFF

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
	
/*CSA Manager VAP Manager states*/
typedef enum
{
	CSA_MANAGER_VAP_MANAGER_IDLE_STATE,
	CSA_MANAGER_VAP_MANAGER_WAIT_TX_IND_STATE,
	CSA_MANAGER_VAP_MANAGER_WAIT_TX_CFM_STATE,
	CSA_MANAGER_VAP_MANAGER_WAIT_LOCK_STOP_STATE,
	CSA_MANAGER_VAP_MANAGER_WAIT_STOP_CFM_STATE,
	CSA_MANAGER_VAP_MANAGER_WAIT_MC_TX_CFM_STATE,
	CSA_MANAGER_VAP_MANAGER_NUM_STATES
} CsaManagerVapManager_state_e;

typedef struct _CsaManagerVapManagerStaDb_t
{
	StaId					nextSta;
	StaId					prevSta;
	uint8					pendingPd;
} CsaManagerVapManagerStaDb_t;

typedef struct _CsaManagerVapManagerDb_t
{
	CsaManagerVapManager_state_e	state;
	uint8						flags;
	uint8						numSta;
	uint8						pendingSta;
	uint8						pendingPd;
	StaId						headSta;
	StaId						tailSta;
	StaId						confirmHead;
	StaId						confirmTail;
	/*From CSA manager perspective a VAP is added when the Set BSS command is received - as the needed flags informations
	are only present at that time. However, the Set BSS command may be sent when the VAP is already up - e.g. set slot time
	We need to differentiate between these cases. An additional state seems an overkill as all CSA events will be ignored in that state
	Just add a flag to indicate whether the VAP is already active or not*/
	uint8						active;
	uint8						next;	/*Used to link Lock requests*/
	TxPd_t						*secQueueHead;
	TxPd_t						*secQueueTail;
	TxPd_t						*pd;
} CsaManagerVapManagerDb_t;

typedef void (*CsaManagerVapManager_Func)(void *parameter);

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
/*Utility functions*/
static void CsaManager_VapManagerChangeState(CsaManagerVapManager_state_e state, uint8 vapId);
static void CsaManager_VapManagerMoveToIdle(uint8 vapId);
static void CsaManager_VapManagerStopPendingSta(uint8 vapId);
static void CsaManager_VapManagerLockGPHP(uint8 vapId);
static void CsaManager_VapManagerHandleGPHPLock(void);
static void CsaManager_VapManagerFlushGPHP(uint8 vapId);
static void Csamanager_VapManagerFlushSecQueue(uint8 vapId);
static void CsaManager_VapManagerRemoveStaFromList(uint8 vapId, StaId sid);
static void CsaManager_VapManagerRemoveStaFromConfirmList(uint8 vapId, StaId sid);
static void CsaManager_VapManagerClearStaEntry(StaId sid);
static void CsaManager_VapManagerAddStaToList(uint8 vapId, StaId sid);
static void CsaManager_VapManagerAddStaToConfirmList(uint8 vapId, StaId sid);
static void CsaManager_VapManagerMoveConfirmToList(uint8 vapId);
static bool CsaManager_VapManagerSameSec(uint8 vapId, StaId sid);
static void CsaManager_VapManagerChangeSecType(uint8 vapId);
static void CsaManager_VapManagerClearSecType(uint8 vapId);
static void CsaManager_VapManagerEnqueuePdToSecQueue(uint8 vapId, TxPd_t *pd);
static void CsaManager_VapManagerMoveSecQueueToGPHP(uint8 vapId);
static void CsaManager_VapManagerSendPacket(TxPd_t *pd);
/*Event handlers*/
static void CsaManager_VapManagerStartTxInIdle(void *parameter);
static void CsaManager_VapManagerStopTxInWaitTxInd(void *parameter);
static void CsaManager_VapManagerStopTxInWaitTxCfm(void *parameter);
static void CsaManager_VapManagerTxIndTxInWaitTxInd(void *parameter);
static void CsaManager_VapManagerTxIndInWaitTxCfm(void *parameter);
static void CsaManager_VapManagerTxCfmInWaitTxInd(void *parameter);
static void CsaManager_VapManagerTxCfmInWaitTxCfm(void *parameter);
static void CsaManager_VapManagerTxCfmInWaitLockStop(void *parameter);
static void CsaManager_VapManagerTxCfmInWaitStopCfm(void *parameter);
static void CsaManager_VapManagerLockCfmInWaitLockStop(void *parameter);
static void CsaManager_VapManagerAddStaInIdle(void *parameter);
static void CsaManager_VapManagerAddStaSendTxReq(void *parameter);
static void CsaManager_VapManagerAddStaSendTxStop(void *parameter);
static void CsaManager_VapManagerRemoveStaInList(void *parameter);
static void CsaManager_VapManagerRemoveStaInConfirm(void *parameter);
static void CsaManager_VapManagerFatal(void *parameter);
static void CsaManager_VapManagerStopTxInWaitMcTxCfm(void *parameter);
static void CsaManager_VapManagerMcTxCfmInWaitLockStop(void *parameter);
static void CsaManager_VapManagerMcTxCfmInWaitStopCfm(void *parameter);
static void CsaManager_VapManagerMcTxCfmInWaitMcTxCfm(void *parameter);

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
/*CSA Manager Vap Manager Start TX Event Handlers*/
static const CsaManagerVapManager_Func CsaManager_VapManagerStartTx[CSA_MANAGER_VAP_MANAGER_NUM_STATES] =
{
	 CsaManager_VapManagerStartTxInIdle, 		/* CSA_MANAGER_VAP_MANAGER_IDLE_STATE */
	 CsaManager_VapManagerFatal,				/* CSA_MANAGER_VAP_MANAGER_WAIT_TX_IND_STATE */
 	 CsaManager_VapManagerFatal,				/* CSA_MANAGER_VAP_MANAGER_WAIT_TX_CFM_STATE */
	 CsaManager_VapManagerFatal, 				/* CSA_MANAGER_VAP_MANAGER_WAIT_LOCK_STOP_STATE */
	 CsaManager_VapManagerFatal, 				/* CSA_MANAGER_VAP_MANAGER_WAIT_STOP_CFM_STATE */
	 CsaManager_VapManagerFatal,				/* CSA_MANAGER_VAP_MANAGER_WAIT_MC_TX_CFM_STATE */
};

/*CSA Manager Vap Manager Stop TX Event Handlers*/
static const CsaManagerVapManager_Func CsaManager_VapManagerStopTx[CSA_MANAGER_VAP_MANAGER_NUM_STATES] =
{
	 CsaManager_VapManagerFatal,				/* CSA_MANAGER_VAP_MANAGER_IDLE_STATE */
	 CsaManager_VapManagerStopTxInWaitTxInd,	/* CSA_MANAGER_VAP_MANAGER_WAIT_TX_IND_STATE */
 	 CsaManager_VapManagerStopTxInWaitTxCfm,	/* CSA_MANAGER_VAP_MANAGER_WAIT_TX_CFM_STATE */
	 CsaManager_VapManagerFatal, 				/* CSA_MANAGER_VAP_MANAGER_WAIT_LOCK_STOP_STATE */
	 CsaManager_VapManagerFatal, 				/* CSA_MANAGER_VAP_MANAGER_WAIT_STOP_CFM_STATE */
	 CsaManager_VapManagerStopTxInWaitMcTxCfm,	/* CSA_MANAGER_VAP_MANAGER_WAIT_MC_TX_CFM_STATE */
};

/*CSA Manager Vap Manager TX IND Event Handlers*/
static const CsaManagerVapManager_Func CsaManager_VapManagerTxInd[CSA_MANAGER_VAP_MANAGER_NUM_STATES] =
{
	 CsaManager_VapManagerFatal, 					/* CSA_MANAGER_VAP_MANAGER_IDLE_STATE */
	 CsaManager_VapManagerTxIndTxInWaitTxInd,		/* CSA_MANAGER_VAP_MANAGER_WAIT_TX_IND_STATE */
 	 CsaManager_VapManagerTxIndInWaitTxCfm,			/* CSA_MANAGER_VAP_MANAGER_WAIT_TX_CFM_STATE */
	 CsaManager_VapManagerFatal, 					/* CSA_MANAGER_VAP_MANAGER_WAIT_LOCK_STOP_STATE */
	 CsaManager_VapManagerFatal, 					/* CSA_MANAGER_VAP_MANAGER_WAIT_STOP_CFM_STATE */
	 CsaManager_VapManagerFatal,					/* CSA_MANAGER_VAP_MANAGER_WAIT_MC_TX_CFM_STATE */
};

/*CSA Manager Vap Manager TX CFM Event Handlers*/
static const CsaManagerVapManager_Func CsaManager_VapManagerTxCfm[CSA_MANAGER_VAP_MANAGER_NUM_STATES] =
{
	 CsaManager_VapManagerFatal, 					/* CSA_MANAGER_VAP_MANAGER_IDLE_STATE */
	 CsaManager_VapManagerTxCfmInWaitTxInd,			/* CSA_MANAGER_VAP_MANAGER_WAIT_TX_IND_STATE */
 	 CsaManager_VapManagerTxCfmInWaitTxCfm,			/* CSA_MANAGER_VAP_MANAGER_WAIT_TX_CFM_STATE */
	 CsaManager_VapManagerTxCfmInWaitLockStop, 		/* CSA_MANAGER_VAP_MANAGER_WAIT_LOCK_STOP_STATE */
	 CsaManager_VapManagerTxCfmInWaitStopCfm,		/* CSA_MANAGER_VAP_MANAGER_WAIT_STOP_CFM_STATE */
	 CsaManager_VapManagerFatal,					/* CSA_MANAGER_VAP_MANAGER_WAIT_MC_TX_CFM_STATE */
};

/*CSA Manager Vap Manager MC TX CFM Event Handlers*/
static const CsaManagerVapManager_Func CsaManager_VapManagerMcTxCfm[CSA_MANAGER_VAP_MANAGER_NUM_STATES] =
{
	 CsaManager_VapManagerFatal, 					/* CSA_MANAGER_VAP_MANAGER_IDLE_STATE */
	 CsaManager_VapManagerFatal,					/* CSA_MANAGER_VAP_MANAGER_WAIT_TX_IND_STATE */
 	 CsaManager_VapManagerFatal,					/* CSA_MANAGER_VAP_MANAGER_WAIT_TX_CFM_STATE */
	 CsaManager_VapManagerMcTxCfmInWaitLockStop,	/* CSA_MANAGER_VAP_MANAGER_WAIT_LOCK_STOP_STATE */
	 CsaManager_VapManagerMcTxCfmInWaitStopCfm,		/* CSA_MANAGER_VAP_MANAGER_WAIT_STOP_CFM_STATE */
	 CsaManager_VapManagerMcTxCfmInWaitMcTxCfm,		/* CSA_MANAGER_VAP_MANAGER_WAIT_MC_TX_CFM_STATE */
};

/*CSA Manager Vap Manager Lock CFM Event Handlers*/
static const CsaManagerVapManager_Func CsaManager_VapManagerLockCfm[CSA_MANAGER_VAP_MANAGER_NUM_STATES] =
{
	 CsaManager_VapManagerFatal, 					/* CSA_MANAGER_VAP_MANAGER_IDLE_STATE */
	 CsaManager_VapManagerFatal,					/* CSA_MANAGER_VAP_MANAGER_WAIT_TX_IND_STATE */
 	 CsaManager_VapManagerFatal,					/* CSA_MANAGER_VAP_MANAGER_WAIT_TX_CFM_STATE */
	 CsaManager_VapManagerLockCfmInWaitLockStop,	/* CSA_MANAGER_VAP_MANAGER_WAIT_LOCK_STOP_STATE */
	 CsaManager_VapManagerFatal,					/* CSA_MANAGER_VAP_MANAGER_WAIT_STOP_CFM_STATE */
	 CsaManager_VapManagerFatal,					/* CSA_MANAGER_VAP_MANAGER_WAIT_MC_TX_CFM_STATE */
};

/*CSA Manager Vap Manager Add STA Event Handlers*/
static const CsaManagerVapManager_Func CsaManager_VapManagerAddSta[CSA_MANAGER_VAP_MANAGER_NUM_STATES] =
{
	 CsaManager_VapManagerAddStaInIdle, 			/* CSA_MANAGER_VAP_MANAGER_IDLE_STATE */
	 CsaManager_VapManagerAddStaSendTxReq,			/* CSA_MANAGER_VAP_MANAGER_WAIT_TX_IND_STATE */
 	 CsaManager_VapManagerAddStaSendTxReq,			/* CSA_MANAGER_VAP_MANAGER_WAIT_TX_CFM_STATE */
	 CsaManager_VapManagerAddStaSendTxStop,			/* CSA_MANAGER_VAP_MANAGER_WAIT_LOCK_STOP_STATE */
	 CsaManager_VapManagerAddStaSendTxStop,			/* CSA_MANAGER_VAP_MANAGER_WAIT_STOP_CFM_STATE */
	 CsaManager_VapManagerAddStaInIdle,				/* CSA_MANAGER_VAP_MANAGER_WAIT_MC_TX_CFM_STATE */
};

/*CSA Manager Vap Manager Remove STA Event Handlers*/
static const CsaManagerVapManager_Func CsaManager_VapManagerRemoveSta[CSA_MANAGER_VAP_MANAGER_NUM_STATES] =
{
	 CsaManager_VapManagerRemoveStaInList, 			/* CSA_MANAGER_VAP_MANAGER_IDLE_STATE */
	 CsaManager_VapManagerRemoveStaInConfirm,		/* CSA_MANAGER_VAP_MANAGER_WAIT_TX_IND_STATE */
 	 CsaManager_VapManagerRemoveStaInConfirm,		/* CSA_MANAGER_VAP_MANAGER_WAIT_TX_CFM_STATE */
	 CsaManager_VapManagerRemoveStaInConfirm,		/* CSA_MANAGER_VAP_MANAGER_WAIT_LOCK_STOP_STATE */
	 CsaManager_VapManagerRemoveStaInConfirm,		/* CSA_MANAGER_VAP_MANAGER_WAIT_STOP_CFM_STATE */
	 CsaManager_VapManagerRemoveStaInList, 			/* CSA_MANAGER_VAP_MANAGER_WAIT_MC_TX_CFM_STATE */
};

static CsaManagerVapManagerStaDb_t CsaManagerVapManager_StaDb[HW_NUM_OF_STATIONS];
static CsaManagerVapManagerDb_t CsaManagerVapManager_Db[HW_NUM_OF_VAPS];
static uint16 CsaManagerVapManager_LockHead, CsaManagerVapManager_LockTail;

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

CsaManager_VapManagerChangeState 


Description:
------------
Utility function that changes VAP state


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerChangeState(CsaManagerVapManager_state_e state, uint8 vapId)
{
 	ILOG0_DDD("CSA Manager VAP Manager, VAP %d changed from %d to %d", vapId, CsaManagerVapManager_Db[vapId].state, state);
	CsaManagerVapManager_Db[vapId].state = state;
}

/**********************************************************************************

CsaManager_VapManagerMoveToIdle 


Description:
------------
Utility function that moves everything back to Idle


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerMoveToIdle(uint8 vapId)
{
	/*Move STAs back from confirm list to list*/
	CsaManager_VapManagerMoveConfirmToList(vapId);
	/*Make sure VAP DB security type is cleared*/
	CsaManager_VapManagerClearSecType(vapId);
	/*Change State to Idle*/
	CsaManager_VapManagerChangeState(CSA_MANAGER_VAP_MANAGER_IDLE_STATE, vapId);
	/*Send TX CFM to CSA Manager*/
	csaManager_TxCfm(vapId);	
}


/**********************************************************************************
CsaManager_VapManagerMcTxCfmMessageHandler

Description:
------------
TX Confirm message handler

Input: 
-----	
			
Output:
-------
	
Returns:
--------
	void - 
	
**********************************************************************************/
void CsaManager_VapManagerMcTxCfmMessageHandler(K_MSG* csaManagerMessage)
{
	/*From PD get VAP ID*/
	CsaManagerPacketConfirmedMessage_t *csaMsg = (CsaManagerPacketConfirmedMessage_t *)pK_MSG_DATA(csaManagerMessage);
	TxPd_t *pd = csaMsg->pd;
	uint32 vapId = pd->txQVapId;
	CsaManagerVapManagerDb_t *VapDb = &(CsaManagerVapManager_Db[vapId]);

	/*Call handler*/
	CsaManager_VapManagerMcTxCfm[VapDb->state]((void *)vapId);
}


/**********************************************************************************

CsaManager_VapManagerStopPendingSta 


Description:
------------
Utility function that stops pending STAs


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerStopPendingSta(uint8 vapId)
{
	StaId sid = CsaManagerVapManager_Db[vapId].headSta;
	StaId nextSid;

	/*Send Stop TX REQ to pending VAPs*/
	while (sid != INVALID_STA_INDEX)
	{
		nextSid = CsaManagerVapManager_StaDb[sid].nextSta;
		/*Send Stop TX REQ to pending VAP*/
		CsaManager_StaManagerTxStop(sid);
		/*Move to next*/
		sid = nextSid;
	}
}

/**********************************************************************************

CsaManager_VapManagerLockGPHP 


Description:
------------
Utility function that Locks the GPHP


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerLockGPHP(uint8 vapId)
{
	RequesterLockParams_t lockRequesterParams;

	/*need to serialize lock requests*/
	CsaManagerVapManager_Db[vapId].next = CSA_MANAGER_VAP_MANAGER_INVALID_LOCK_IDX;
	/*Do we have pending entries*/
	if (CsaManagerVapManager_LockHead == CSA_MANAGER_VAP_MANAGER_INVALID_LOCK_IDX)
	{
		/*Set as head*/
		CsaManagerVapManager_LockHead = vapId;
		/*Send Lock Request to GPHP*/
		memset(&lockRequesterParams, 0, sizeof(RequesterLockParams_t));
		/*Request SW Locker to Wait for Lock Stop*/
		lockRequesterParams.callerContext = NULL;
		lockRequesterParams.returnMsg = CSA_MANAGER_VAP_GPHP_LOCK;
		lockRequesterParams.returnTask = TASK_TX_MANAGER;
		Locker_LockSingleQueue(HW_TX_Q_TYPE_GPHP, vapId, 0, &lockRequesterParams);
	}
	else
	{
		/*Link to last*/
		CsaManagerVapManager_Db[CsaManagerVapManager_LockTail].next = vapId;
	}
	/*Set as last*/
	CsaManagerVapManager_LockTail = vapId;
}

/**********************************************************************************

CsaManager_VapManagerHandleGPHPLock 


Description:
------------
Utility function that handles pending GPHP lock requests


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerHandleGPHPLock()
{	
	uint8 vapId = CsaManagerVapManager_LockHead;
	RequesterLockParams_t lockRequesterParams;

	/*Move to next entry*/
	CsaManagerVapManager_LockHead = CsaManagerVapManager_Db[vapId].next;
	/*Do we have pending entries*/
	if (CsaManagerVapManager_LockHead != CSA_MANAGER_VAP_MANAGER_INVALID_LOCK_IDX)
	{
		vapId = CsaManagerVapManager_LockHead;
		/*Send Lock Request to GPHP*/
		memset(&lockRequesterParams, 0, sizeof(RequesterLockParams_t));
		/*Request SW Locker to Wait for Lock Stop*/
		lockRequesterParams.callerContext = NULL;
		lockRequesterParams.returnMsg = CSA_MANAGER_VAP_GPHP_LOCK;
		lockRequesterParams.returnTask = TASK_TX_MANAGER;
		Locker_LockSingleQueue(HW_TX_Q_TYPE_GPHP, vapId, 0, &lockRequesterParams);
	}
	else
	{
		/*Clear last as well*/
		CsaManagerVapManager_LockTail = CSA_MANAGER_VAP_MANAGER_INVALID_LOCK_IDX;
	}
}

/**********************************************************************************

CsaManager_VapManagerFlushGPHP 


Description:
------------
Utility function that Flushes the GPHP


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerFlushGPHP(uint8 vapId)
{
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;

	memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	hwQueueManagerRequestParams.primaryAddr = vapId;
	hwQueueManagerRequestParams.secondaryAddr = 0;
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
	hwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_GPHP;
	HwQManager_FlushQ(&hwQueueManagerRequestParams);
	/*Send to Liberator*/
	if (hwQueueManagerRequestParams.pHeadDesc != (TxPd_t *)NULL_PD)
	{
   		hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
		hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_DONE_LIST_LIBERATOR;
		/*head and tail already include the PDs*/
   		HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams); 
	}
}

/**********************************************************************************

Csamanager_VapManagerFlushSecQueue 


Description:
------------
Utility function that flushes the SEC queue


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void Csamanager_VapManagerFlushSecQueue(uint8 vapId)
{
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;

	/*Send queue to Liberator*/
	if (CsaManagerVapManager_Db[vapId].secQueueHead != (TxPd_t *)NULL_PD)
	{
		memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));
		hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
		hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
		hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_DONE_LIST_LIBERATOR;
		hwQueueManagerRequestParams.pHeadDesc = CsaManagerVapManager_Db[vapId].secQueueHead;
		hwQueueManagerRequestParams.pTailDesc = CsaManagerVapManager_Db[vapId].secQueueTail;
		CsaManagerVapManager_Db[vapId].secQueueHead = (TxPd_t *)NULL_PD;
		CsaManagerVapManager_Db[vapId].secQueueTail = (TxPd_t *)NULL_PD;
		/*head and tail already include the PDs*/
		HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams); 
	}
}

/**********************************************************************************

CsaManager_VapManagerRemoveStaFromList 


Description:
------------
Utility function that removes a STA from the list


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerRemoveStaFromList(uint8 vapId, StaId sid)
{
	StaId prevSta = CsaManagerVapManager_StaDb[sid].prevSta;
	StaId nextSta = CsaManagerVapManager_StaDb[sid].nextSta;

	if (prevSta == INVALID_STA_INDEX)
	{
		CsaManagerVapManager_Db[vapId].headSta = nextSta;
	}
	else
	{
		CsaManagerVapManager_StaDb[prevSta].nextSta = nextSta;
	}
	if (nextSta == INVALID_STA_INDEX)
	{
		CsaManagerVapManager_Db[vapId].tailSta = prevSta;
	}
	else
	{
		CsaManagerVapManager_StaDb[nextSta].prevSta = prevSta;
	}
}

/**********************************************************************************

CsaManager_VapManagerRemoveStaFromConfirmList 


Description:
------------
Utility function that removes a STA from the confirm list


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerRemoveStaFromConfirmList(uint8 vapId, StaId sid)
{
 	StaId prevSta = CsaManagerVapManager_StaDb[sid].prevSta;
	StaId nextSta = CsaManagerVapManager_StaDb[sid].nextSta;
 
	if (prevSta == INVALID_STA_INDEX)
	{
		CsaManagerVapManager_Db[vapId].confirmHead = nextSta;
	}
	else
	{
		CsaManagerVapManager_StaDb[prevSta].nextSta = nextSta;
	}
	if (nextSta == INVALID_STA_INDEX)
	{
		CsaManagerVapManager_Db[vapId].confirmTail = prevSta;
	}
	else
	{
		CsaManagerVapManager_StaDb[nextSta].prevSta = prevSta;
	}
}

/**********************************************************************************

CsaManager_ClearVapEntry 


Description:
------------
Utility function that clears a VAP entry


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerClearStaEntry(StaId sid)
{
	CsaManagerVapManager_StaDb[sid].nextSta = INVALID_STA_INDEX;
	CsaManagerVapManager_StaDb[sid].prevSta = INVALID_STA_INDEX;
}

/**********************************************************************************

CsaManager_VapManagerAddStaToList 


Description:
------------
Utility function that add a STA to the list


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerAddStaToList(uint8 vapId, StaId sid)
{
	bool isStaPmf = CsaManager_StaManagerIsStaPmf(sid);

	if (isStaPmf == FALSE)
	{
		/*If STA is PMF add to start of list*/
		if (CsaManagerVapManager_Db[vapId].headSta != INVALID_STA_INDEX)
		{
			CsaManagerVapManager_StaDb[sid].nextSta = CsaManagerVapManager_Db[vapId].headSta;
			CsaManagerVapManager_StaDb[CsaManagerVapManager_Db[vapId].headSta].prevSta = sid;
		}
		else
		{
			CsaManagerVapManager_Db[vapId].tailSta = sid;
		}
		CsaManagerVapManager_Db[vapId].headSta = sid;
	}
	else
	{
		/*If STA is PMF add to end of list*/
		if (CsaManagerVapManager_Db[vapId].headSta != INVALID_STA_INDEX)
		{
			CsaManagerVapManager_StaDb[sid].prevSta = CsaManagerVapManager_Db[vapId].tailSta;
			CsaManagerVapManager_StaDb[CsaManagerVapManager_Db[vapId].tailSta].nextSta = sid;
		}
		else
		{
			CsaManagerVapManager_Db[vapId].headSta = sid;
		}
		CsaManagerVapManager_Db[vapId].tailSta = sid;
	}
	/*Increment number of STAs*/
	CsaManagerVapManager_Db[vapId].numSta++;
}

/**********************************************************************************

CsaManager_AddVapToConfirmList 


Description:
------------
Utility function that add a VAP to the confirm list


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerAddStaToConfirmList(uint8 vapId, StaId sid)
{
	bool isStaPmf = CsaManager_StaManagerIsStaPmf(sid);

	if (isStaPmf == FALSE)
	{
		/*If STA is PMF add to start of list*/
		if (CsaManagerVapManager_Db[vapId].confirmHead != INVALID_STA_INDEX)
		{
			CsaManagerVapManager_StaDb[sid].nextSta = CsaManagerVapManager_Db[vapId].confirmHead;
			CsaManagerVapManager_StaDb[CsaManagerVapManager_Db[vapId].confirmHead].prevSta = sid;
		}
		else
		{
			CsaManagerVapManager_Db[vapId].confirmTail = sid;
		}
		CsaManagerVapManager_Db[vapId].confirmHead = sid;
	}
	else
	{
		/*If STA is PMF add to end of list*/
		if (CsaManagerVapManager_Db[vapId].confirmHead != INVALID_STA_INDEX)
		{
			CsaManagerVapManager_StaDb[sid].prevSta = CsaManagerVapManager_Db[vapId].confirmTail;
			CsaManagerVapManager_StaDb[CsaManagerVapManager_Db[vapId].confirmTail].nextSta = sid;
		}
		else
		{
			CsaManagerVapManager_Db[vapId].confirmHead = sid;
		}
		CsaManagerVapManager_Db[vapId].confirmTail= sid;
	}
}

/**********************************************************************************

CsaManager_MoveConfirmToList 


Description:
------------
Utility function that moves VAP back from the confirm list to the list


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerMoveConfirmToList(uint8 vapId)
{
	CsaManagerVapManager_Db[vapId].headSta = CsaManagerVapManager_Db[vapId].confirmHead;
	CsaManagerVapManager_Db[vapId].tailSta = CsaManagerVapManager_Db[vapId].confirmTail;
	CsaManagerVapManager_Db[vapId].confirmHead = INVALID_STA_INDEX;
	CsaManagerVapManager_Db[vapId].confirmTail = INVALID_STA_INDEX;
}

/**********************************************************************************

CsaManager_VapManagerSameSec 


Description:
------------
Utility function that checks if GPHP SEC type matches the current SID


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static bool CsaManager_VapManagerSameSec(uint8 vapId, StaId sid)
{
	uint32 isVapMngmtEnc = VapDB_IsManagementEncryptionEnabled(vapId);
	bool isStaPmf = CsaManager_StaManagerIsStaPmf(sid);

	if (((isVapMngmtEnc == FALSE) && (isStaPmf == TRUE)) ||
		((isVapMngmtEnc != FALSE) && (isStaPmf == FALSE)))
	{
		return (FALSE);
	}
	return (TRUE);
}

/**********************************************************************************

CsaManager_VapManagerChangeSecType 


Description:
------------
Utility function that changes the GPHP Encryption type


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerChangeSecType(uint8 vapId)
{
	uint32 secMngmtEnabled = VapDB_IsManagementEncryptionEnabled(vapId);

	if (secMngmtEnabled == FALSE)
	{
		/*If None move to CCMP*/
		VapDB_SetManagementEncryptionType(vapId, UMI_RSN_CIPHER_SUITE_CCMP);
		VapDB_SetManagementEncryptionEnabled(vapId, TRUE);
	}
	else
	{
		/*Move to None*/
		VapDB_DisableManagementEncryption(vapId);
		VapDB_SetManagementEncryptionEnabled(vapId, FALSE);
	}
}

/**********************************************************************************

CsaManager_VapManagerClearSecType 


Description:
------------
Utility function that clears the GPHP Encryption type


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerClearSecType(uint8 vapId)
{
	/*Move to None*/
	VapDB_SetManagementEncryptionType(vapId, UMI_RSN_CIPHER_SUITE_NONE);
	VapDB_SetManagementEncryptionEnabled(vapId, FALSE);
}

/**********************************************************************************

CsaManager_VapManagerEnqueuePdToSecQueue 


Description:
------------
Utility function that adds a PD to the SEC queue


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerEnqueuePdToSecQueue(uint8 vapId, TxPd_t *pd)
{
	pd->packetPointer = CONVERT_WLAN_SHRAM_ADDR_TO_DMA_SHRAM_ADDR(pd->packetPointer);
	if (CsaManagerVapManager_Db[vapId].secQueueHead == (TxPd_t *)NULL_PD)
	{
		CsaManagerVapManager_Db[vapId].secQueueHead = pd;
	}
	else
	{
		CsaManagerVapManager_Db[vapId].secQueueTail->nextPdPointer = SET_NEXT_PD(pd);
	}
	CsaManagerVapManager_Db[vapId].secQueueTail = pd;
}

/**********************************************************************************

CsaManager_VapManagerMoveSecQueueToGPHP 


Description:
------------
Utility function that moves teh SEC queue to GPHP


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerMoveSecQueueToGPHP(uint8 vapId)
{
	TxPd_t *pd, *nextPd;
 	StaId sid;
 
	pd = CsaManagerVapManager_Db[vapId].secQueueHead;
	while (pd != (TxPd_t *)NULL_PD)
	{
		nextPd = (TxPd_t *)GET_NEXT_PD((TxPd_t *)pd);
		/*get SID from PD*/
		sid = pd->txQStaId;
		pd->packetPointer = CONVERT_DMA_SHRAM_ADDR_TO_WLAN_SHRAM_ADDR(pd->packetPointer);
		/*Put CSA on GPHP queue*/
		CsaManager_VapManagerSendPacket(pd);
		/*Increment number of pending PDs*/
		CsaManagerVapManager_Db[vapId].pendingPd++;
		/*Mark STA that it has a pending PD*/
		CsaManagerVapManager_StaDb[sid].pendingPd = TRUE;
		/*Move to next*/
		pd = nextPd;
	}
	/*Clear SEC queue*/
	CsaManagerVapManager_Db[vapId].secQueueHead = (TxPd_t *)NULL_PD;
	CsaManagerVapManager_Db[vapId].secQueueTail = (TxPd_t *)NULL_PD;
}

/**********************************************************************************

CsaManager_VapManagerSendPacket 


Description:
------------
Utility function that sends the CSA packet


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerSendPacket(TxPd_t *pd)
{
 	StaId sid = pd->txQStaId;
	bool isStaPmf = CsaManager_StaManagerIsStaPmf(sid);
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;

	/*There is a possible race condition where STA is PMF but Key has not been set yet
	To prevent Transmission error we skip this packet - to prevent stack overflow in case we are handling more than one such STA at a time
	send the PD to Liberator so it will eventually make its way back*/
	if ((isStaPmf == FALSE) || (StaDB_IsManagementEncrypted(sid)))
	{
		/*Put CSA on GPHP queue*/
		TxPacketsClassifier_SendManagementPacketFromFw((TxPd_t *)pd);
	}
	else
	{
		pd->packetPointer = CONVERT_WLAN_SHRAM_ADDR_TO_DMA_SHRAM_ADDR(pd->packetPointer);	
		memset(&hwQueueManagerRequestParams,0,sizeof(HwQueueManagerRequestParams_t));
		hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
		hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_DONE_LIST_LIBERATOR;
		hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
		hwQueueManagerRequestParams.pHeadDesc = pd;
		/*Send to Liberator*/
		HwQManager_PushPacketToTail(&hwQueueManagerRequestParams); 
	}
}

    
/***********************************************************************
* CsaManager_VapManagerIsMcDeauthFrameEncrypted
* 
* Description:
* ------------
* Check if the VAP is using Encrytion or not
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
static bool CsaManager_VapManagerIsMcDeauthFrameEncrypted(uint8 bandId, uint32 vapId)
{
	bool isEncrypted = FALSE;
	uint32	bitmapOfVaps = CsaManager_GetCsaDeauthFrameEnc(bandId);
		
	if (bitmapOfVaps & (TRUE << vapId))
	{
		isEncrypted = TRUE;
	}
		
	return(isEncrypted);
}

/**********************************************************************************

CsaManager_VapManagerStartTxInIdle 


Description:
------------
Start TX REQ handler In Idle State


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerStartTxInIdle(void *parameter)
{
	uint32						vapId 					= (uint32)parameter;
 	StaId						sid 					= CsaManagerVapManager_Db[vapId].headSta;
 	StaId						nextSid 				= 0;
	uint8						allocationStatus 		= FALSE;
	TxPd_t 						*pdPointer 				= (TxPd_t *)NULL_PD;
	uint8 						bandId 					= ConfigurationManager_GetBandForVap(vapId);
	RmPdRequestFillParameters_t pdRequestFillParameters;

	ILOG0_V("[CSA] CsaManager_VapManagerStartTxInIdle 1");

	if (CSA_TYPE_MC_DEAUTH == CsaManager_GetCurrentTxSegment(vapId))
	{
		ILOG0_V("[CSA] CsaManager_VapManagerStartTxInIdle 2");
		memset(&pdRequestFillParameters, 0, sizeof(RmPdRequestFillParameters_t));
		pdRequestFillParameters.context = vapId;
		pdRequestFillParameters.returnMsgType = CSA_MANAGER_PD_ALLOC;
		pdRequestFillParameters.returnTaskId = TASK_TX_MANAGER;
		allocationStatus = ResourceManager_GetDescriptorRequest(DESC_POOL_MC_DEAUTH, &pdRequestFillParameters);

		//in mc deauth we assume that resources will always be available (num_of_vaps << size_of_resource_pool), so we currently don't support pending for resources
		ASSERT(TRUE == allocationStatus);
		
		/*Store PD pointer*/
		DEBUG_ASSERT((TxPd_t *)NULL_PD == CsaManagerVapManager_Db[vapId].pd);
		CsaManagerVapManager_Db[vapId].pd = pdRequestFillParameters.packetDescriptor;
		pdPointer = pdRequestFillParameters.packetDescriptor;
		
		/*Build and send M/C deauth*/
		pdPointer->txQVapId = vapId;
		pdPointer->txQTid = MANAGEMENT_TID;
		pdPointer->mgmtFrameSubtype = MGMT_FRAME_SUBTYPE_DEAUTHENTICATION;

		/* Note: DEAUTHENTICATION does not have category nor action values. see SPEC IEEE 802.11-2016: Table 9-47 Category values */
		
		if (CsaManager_VapManagerIsMcDeauthFrameEncrypted(bandId, vapId) == TRUE)
		{
			pdPointer->pdType = PD_TYPE_MANAGEMENT_ENC;
		}
		else
		{
			pdPointer->pdType = PD_TYPE_MANAGEMENT_UNENC;
		}
		pdPointer->txQGroupId = HW_TX_Q_TYPE_GPHP;
		pdPointer->mcUnicast = MULTICAST;
		
		/*Build CSA MC DEAUTH frame*/
		//Set pointer to frame
		pdPointer->packetPointer = csaManager_GetFrameLocation(bandId, vapId);
		//Set length
		pdPointer->dataLength = csaManager_GetFrameLength(bandId, vapId);		

		/*Put CSA on GPHP queue*/
		TxPacketsClassifier_SendManagementPacketFromFw(pdPointer);

		/*Change VAP State to Wait TX CFM*/
		CsaManager_VapManagerChangeState(CSA_MANAGER_VAP_MANAGER_WAIT_MC_TX_CFM_STATE, vapId);
	}
	else
	{
		ILOG0_V("[CSA] CsaManager_VapManagerStartTxInIdle 3");
		if (sid == INVALID_STA_INDEX)
		{
			ILOG0_D("[CSA] CsaManager_VapManagerStartTxInIdle 4 sid = %d", sid);
			/*If no STAs connected to VAP just send CFM*/
			csaManager_TxCfm(vapId);
			return;
		}
		/*Set State to Wait TX IND*/
		CsaManager_VapManagerChangeState(CSA_MANAGER_VAP_MANAGER_WAIT_TX_IND_STATE, vapId);
		/*Set the number of pending STAs to the current number of STAs*/
		CsaManagerVapManager_Db[vapId].pendingSta =  CsaManagerVapManager_Db[vapId].numSta;
		/*Send Start TX REQ to pending VAPs*/
		while (sid != INVALID_STA_INDEX)
		{
			ILOG0_D("[CSA] CsaManager_VapManagerStartTxInIdle 5 sid = %d", sid);
			nextSid = CsaManagerVapManager_StaDb[sid].nextSta;
			/*Send Start TX REQ to pending VAP*/
			CsaManager_StaManagerTxStart(sid);
			/*Move to next*/
			sid = nextSid;
		}
	}
}

/**********************************************************************************

CsaManager_VapManagerStopTxInWaitTxInd 


Description:
------------
Stop TX REQ handler In Wait TX IND State


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerStopTxInWaitTxInd(void *parameter)
{
	uint32 vapId = (uint32)parameter;
	
	/*Stop Pending STAs*/
	CsaManager_VapManagerStopPendingSta(vapId);
}

/**********************************************************************************

CsaManager_VapManagerStopTxInWaitTxCfm 


Description:
------------
Stop TX REQ handler In Wait TX CFM State


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerStopTxInWaitTxCfm(void *parameter)
{
	uint32 vapId = (uint32)parameter;

	/*Stop Pending STAs*/
	CsaManager_VapManagerStopPendingSta(vapId);
	/*Change VAP State to Wait Lock Stop*/
	CsaManager_VapManagerChangeState(CSA_MANAGER_VAP_MANAGER_WAIT_LOCK_STOP_STATE, vapId);
	/*Send Lock Request to GPHP*/
	CsaManager_VapManagerLockGPHP(vapId);
	/*Send SEC queue to Liberator*/
	Csamanager_VapManagerFlushSecQueue(vapId);
}


/**********************************************************************************

CsaManager_VapManagerStopTxInWaitMcTxCfm 


Description:
------------
Stop TX REQ handler In Wait MC TX CFM State


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerStopTxInWaitMcTxCfm(void *parameter)
{
	uint32 vapId = (uint32)parameter;

	/*Change VAP State to Wait Lock Stop*/
	CsaManager_VapManagerChangeState(CSA_MANAGER_VAP_MANAGER_WAIT_LOCK_STOP_STATE, vapId);
	/*Send Lock Request to GPHP*/
	CsaManager_VapManagerLockGPHP(vapId);
}

/**********************************************************************************

CsaManager_VapManagerTxIndTxInWaitTxInd 


Description:
------------
TX IND handler In Wait TX IND State


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerTxIndTxInWaitTxInd(void *parameter)
{
	TxPd_t *pd = (TxPd_t *)parameter;
	uint8 vapId = pd->txQVapId;
 	StaId sid = pd->txQStaId;

	/*Check if we need to chage SEC Type*/
	if (CsaManager_VapManagerSameSec(vapId, sid) == FALSE)
	{
		/*Change SEC type*/
		CsaManager_VapManagerChangeSecType(vapId);
	}
	/*Put CSA on GPHP queue*/
	CsaManager_VapManagerSendPacket(pd);
	/*Increment number of pending PDs*/
	CsaManagerVapManager_Db[vapId].pendingPd++;
	/*Mark STA that it has a pending PD*/
	CsaManagerVapManager_StaDb[sid].pendingPd = TRUE;
	/*Change VAP State to Wait TX CFM*/
	CsaManager_VapManagerChangeState(CSA_MANAGER_VAP_MANAGER_WAIT_TX_CFM_STATE, vapId);
}

/**********************************************************************************

CsaManager_VapManagerTxIndInWaitTxCfm 


Description:
------------
TX IND handler In Wait TX CFM State


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerTxIndInWaitTxCfm(void *parameter)
{
	TxPd_t *pd = (TxPd_t *)parameter;
	uint8 vapId = pd->txQVapId;
 	StaId sid = pd->txQStaId;

	if (CsaManager_VapManagerSameSec(vapId, sid) == TRUE)
	{
		/*Put CSA on GPHP queue*/
		CsaManager_VapManagerSendPacket(pd);
		/*Increment number of pending PDs*/
		CsaManagerVapManager_Db[vapId].pendingPd++;
		/*Mark STA that it has a pending PD*/
		CsaManagerVapManager_StaDb[sid].pendingPd = TRUE;
	}
	else
	{
		/*Add PD to SEC queue*/
		CsaManager_VapManagerEnqueuePdToSecQueue(vapId, pd);
	}
}

/**********************************************************************************

CsaManager_VapManagerTxCfmInWaitTxInd


Description:
------------
TX CFM handler In Wait TX IND State


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerTxCfmInWaitTxInd(void *parameter)
{
	uint32 sid = (uint32)parameter;
	uint8 vapId = CsaManager_StaManagerGetVap(sid);

	/*Should only happen for STAs with no pending PDs*/
	ASSERT(CsaManagerVapManager_StaDb[sid].pendingPd == FALSE);
	/*Remove STA from pending list*/
	CsaManager_VapManagerRemoveStaFromList(vapId, sid);
	CsaManager_VapManagerClearStaEntry(sid);
	/*Move STA to Confirm List*/
	CsaManager_VapManagerAddStaToConfirmList(vapId, sid);
	/*Decrement number of pending STAs*/
	CsaManagerVapManager_Db[vapId].pendingSta--;
	if (CsaManagerVapManager_Db[vapId].pendingSta == 0)
	{
		CsaManager_VapManagerMoveToIdle(vapId);
	}
}

/**********************************************************************************

CsaManager_VapManagerTxCfmInWaitTxCfm 


Description:
------------
TX CFM handler In Wait TX CFM State


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerTxCfmInWaitTxCfm(void *parameter)
{
	uint32 sid = (uint32)parameter;
	uint8 vapId = CsaManager_StaManagerGetVap(sid);

	/*Remove STA from pending list*/
	CsaManager_VapManagerRemoveStaFromList(vapId, sid);
	CsaManager_VapManagerClearStaEntry(sid);
	/*Move STA to Confirm List*/
	CsaManager_VapManagerAddStaToConfirmList(vapId, sid);
	/*Decrement number of pending PDs only if STA has a pending PD*/
	if (CsaManagerVapManager_StaDb[sid].pendingPd == TRUE)
	{
		CsaManagerVapManager_Db[vapId].pendingPd--;
		/*Mark STA that it no longer has a pending PD*/
		CsaManagerVapManager_StaDb[sid].pendingPd = FALSE;
	}
	/*Decrement number of pending STAs*/
	CsaManagerVapManager_Db[vapId].pendingSta--;
	if (CsaManagerVapManager_Db[vapId].pendingSta == 0)
	{
		CsaManager_VapManagerMoveToIdle(vapId);
	}
	else if (CsaManagerVapManager_Db[vapId].pendingPd == 0)
	{
		/*If SEC queue empty - move back to Wait For TX IND*/
		if (CsaManagerVapManager_Db[vapId].secQueueHead == (TxPd_t *)NULL_PD)
		{
			/*Change VAP State to Wait TX IND*/
			CsaManager_VapManagerChangeState(CSA_MANAGER_VAP_MANAGER_WAIT_TX_IND_STATE, vapId);
		}
		else
		{
			/*Change Security type*/
			CsaManager_VapManagerChangeSecType(vapId);
			/*Enqueue SEC queue to GPHP*/
			CsaManager_VapManagerMoveSecQueueToGPHP(vapId);
		}
	}
}


/**********************************************************************************

CsaManager_VapManagerMcTxCfmInWaitMcTxCfm


Description:
------------
TX CFM handler In Wait TX CFM State


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerMcTxCfmInWaitMcTxCfm(void *parameter)
{
	uint32 vapId = (uint32)parameter;

	/*Release the PD*/
	ResourceManager_ReleaseDescriptor(CsaManagerVapManager_Db[vapId].pd, DESC_POOL_MC_DEAUTH);
	CsaManagerVapManager_Db[vapId].pd = (TxPd_t *)NULL_PD;

	/*Change State to Idle*/
	CsaManager_VapManagerChangeState(CSA_MANAGER_VAP_MANAGER_IDLE_STATE, vapId);

	/*Send TX CFM to CSA Manager*/
	csaManager_TxCfm(vapId);	
}


/**********************************************************************************

CsaManager_VapManagerTxCfmInWaitLockStop 


Description:
------------
TX CFM handler In Wait LOCK Stop State


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerTxCfmInWaitLockStop(void *parameter)
{
	uint32 sid = (uint32)parameter;
	uint8 vapId = CsaManager_StaManagerGetVap(sid);

	/*Remove STA from pending list*/
	CsaManager_VapManagerRemoveStaFromList(vapId, sid);
	CsaManager_VapManagerClearStaEntry(sid);
	/*Move STA to Confirm List*/
	CsaManager_VapManagerAddStaToConfirmList(vapId, sid);
	/*Decrement number of pending STAs*/
	CsaManagerVapManager_Db[vapId].pendingSta--;
}


/**********************************************************************************

CsaManager_VapManagerMcTxCfmInWaitLockStop 


Description:
------------
TX CFM handler In Wait LOCK Stop State


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerMcTxCfmInWaitLockStop(void *parameter)
{
	uint32 vapId = (uint32)parameter;

	/*Release the PD*/
	ResourceManager_ReleaseDescriptor(CsaManagerVapManager_Db[vapId].pd, DESC_POOL_MC_DEAUTH);
	CsaManagerVapManager_Db[vapId].pd = (TxPd_t *)NULL_PD;
}


/**********************************************************************************

CsaManager_VapManagerTxCfmInWaitStopCfm 


Description:
------------
TX CFM handler In Wait Stop CFM State


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerTxCfmInWaitStopCfm(void *parameter)
{
	uint32 sid = (uint32)parameter;
	uint8 vapId = CsaManager_StaManagerGetVap(sid);

	ILOG0_V("[CSA] CsaManager_VapManagerTxCfmInWaitStopCfm 1");


	/*Remove STA from pending list*/
	CsaManager_VapManagerRemoveStaFromList(vapId, sid);
	CsaManager_VapManagerClearStaEntry(sid);
	/*Move STA to Confirm List*/
	CsaManager_VapManagerAddStaToConfirmList(vapId, sid);
	/*Decrement number of pending STAs*/
	CsaManagerVapManager_Db[vapId].pendingSta--;
	if (CsaManagerVapManager_Db[vapId].pendingSta == 0)
	{
		/*We are done*/
		CsaManager_VapManagerMoveToIdle(vapId);
	}
}


/**********************************************************************************

CsaManager_VapManagerMcTxCfmInWaitStopCfm 


Description:
------------
TX CFM handler In Wait Stop CFM State


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerMcTxCfmInWaitStopCfm(void *parameter)
{
	uint32 vapId = (uint32)parameter;

	/*Release the PD*/
	ResourceManager_ReleaseDescriptor(CsaManagerVapManager_Db[vapId].pd, DESC_POOL_MC_DEAUTH);
	CsaManagerVapManager_Db[vapId].pd = (TxPd_t *)NULL_PD;

	/*Change State to Idle*/
	CsaManager_VapManagerChangeState(CSA_MANAGER_VAP_MANAGER_IDLE_STATE, vapId);

	/*Send TX CFM to CSA Manager*/
	csaManager_TxCfm(vapId);	
}


/**********************************************************************************

CsaManager_VapManagerLockCfmInWaitLockStop 


Description:
------------
Lock CFM handler In Wait Lock Stop State


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerLockCfmInWaitLockStop(void *parameter)
{
	uint32 vapId = (uint32)parameter;

	/*Flush GPHP*/
	CsaManager_VapManagerFlushGPHP(vapId);
	/*Unlock GPHP*/
	Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_GPHP, vapId, (0x1 << 0));

	if (CSA_TYPE_MC_DEAUTH == CsaManager_GetCurrentTxSegment(vapId))
	{
		if (CsaManagerVapManager_Db[vapId].pd != (TxPd_t *)NULL_PD)
		{
			/*Change VAP State to Wait Stop CFM*/
			CsaManager_VapManagerChangeState(CSA_MANAGER_VAP_MANAGER_WAIT_STOP_CFM_STATE, vapId);
		}
		else
		{
			/*Change State to Idle*/
			CsaManager_VapManagerChangeState(CSA_MANAGER_VAP_MANAGER_IDLE_STATE, vapId);

			/*Send TX CFM to CSA Manager*/
			csaManager_TxCfm(vapId);			
		}
	}
	else
	{
		if (CsaManagerVapManager_Db[vapId].pendingSta != 0)
		{
			/*Change VAP State to Wait Stop CFM*/
			CsaManager_VapManagerChangeState(CSA_MANAGER_VAP_MANAGER_WAIT_STOP_CFM_STATE, vapId);
		}
		else
		{
			/*We are done*/
			CsaManager_VapManagerMoveToIdle(vapId);
		}
	}
}

/**********************************************************************************

CsaManager_VapManagerAddStaInIdle 


Description:
------------
ADD STA Handler in IDLE State


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerAddStaInIdle(void *parameter)
{
	uint32 sid = (uint32)parameter;
	uint8 vapId	= CsaManager_StaManagerGetVap(sid);

	CsaManager_VapManagerAddStaToList(vapId, sid);
}

    
/***********************************************************************
* CsaManager_VapManagerGetNumStaInVap
* 
* Description:
* ------------
* 
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* number of connected stations to this VAP
* 
************************************************************************/
uint16 CsaManager_VapManagerGetNumStaInVap(uint16 vapId)
{
	return(CsaManagerVapManager_Db[vapId].numSta);
}    
/**********************************************************************************

CsaManager_VapManagerAddStaSendTxReq 


Description:
------------
ADD STA Handler in Wait TX REQ States


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerAddStaSendTxReq(void *parameter)
{
	uint32 sid = (uint32)parameter;
	uint8 vapId	= CsaManager_StaManagerGetVap(sid);

	CsaManager_VapManagerAddStaToList(vapId, sid);
	/*Increment number of pending STAs*/
	CsaManagerVapManager_Db[vapId].pendingSta++;	
	/*Send StarT TX REQ to STA Manager*/
	CsaManager_StaManagerTxStart(sid);
}

/**********************************************************************************

CsaManager_VapManagerAddStaSendTxStop 


Description:
------------
ADD STA Handler in Wait TX STOP States


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerAddStaSendTxStop(void *parameter)
{
	uint32 sid = (uint32)parameter;
	uint8 vapId	= CsaManager_StaManagerGetVap(sid);

	CsaManager_VapManagerAddStaToList(vapId, sid);
	/*Increment number of pending STAs*/
	CsaManagerVapManager_Db[vapId].pendingSta++;	
	/*Send StarT TX STOP to STA Manager*/
	CsaManager_StaManagerTxStop(sid);
}

/**********************************************************************************

CsaManager_VapManagerRemoveStaInList 


Description:
------------
Remove STA Handler in Idle - STA is in List


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerRemoveStaInList(void *parameter)
{
	uint32 sid = (uint32)parameter;
	uint8 vapId	= CsaManager_StaManagerGetVap(sid);

	CsaManager_VapManagerRemoveStaFromList(vapId, sid);
	CsaManager_VapManagerClearStaEntry(sid);
	/*Decrement number of STAs*/
	CsaManagerVapManager_Db[vapId].numSta--;	
}

/**********************************************************************************

CsaManager_VapManagerRemoveStaInConfirm 


Description:
------------
Remove STA Handler in all states except Idle - STA is in confirm list
STA Manager will send TX CFM before sending STA remove IND


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerRemoveStaInConfirm(void *parameter)
{
	uint32 sid = (uint32)parameter;
	uint8 vapId	= CsaManager_StaManagerGetVap(sid);

	CsaManager_VapManagerRemoveStaFromConfirmList(vapId, sid);
	CsaManager_VapManagerClearStaEntry(sid);
	/*Decrement number of STAs*/
	CsaManagerVapManager_Db[vapId].numSta--;	
}

/**********************************************************************************

CsaManager_VapManagerFatal 


Description:
------------
handles unexpected event


Input: 
-----	
TxManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_VapManagerFatal(void *parameter)
{
	UNUSED_PARAM(parameter);	
	FATAL("CSA Manager VAP Manager Fatal");
}

/*---------------------------------------------------------------------------------
/						global Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

CsaManager_VapManagerIsHT 


Description:
------------
returns whether VAP is HT


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
bool CsaManager_VapManagerIsHT(uint8 vapId)
{
	bool isHT = MTLK_BFIELD_GET(CsaManagerVapManager_Db[vapId].flags, VAP_ADD_FLAGS_HT);
 	return (isHT);
}

/**********************************************************************************

CsaManager_VapManagerIsVHT 


Description:
------------
returns whether VAP is VHT


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
bool CsaManager_VapManagerIsVHT(uint8 vapId)
{
	/*Return whether VAP is VHT*/
	bool isVHT = MTLK_BFIELD_GET(CsaManagerVapManager_Db[vapId].flags, VAP_ADD_FLAGS_VHT);
 	return (isVHT);
}

/**********************************************************************************

csaManagerVapOpenInd 


Description:
------------
VAP ADD message handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void csaManagerVapOpen(K_MSG* csaManagerMessage)
{
	BSS_MANAGER_CONFIRM_EVENT* confirmEvent; 
	uint8 vapId; 
	K_MSG *vapMsg = (K_MSG *) (*((uint32 *)pK_MSG_DATA(csaManagerMessage)));
	UMI_SET_BSS* pSetBssMsg = (UMI_SET_BSS*) pK_MSG_DATA(vapMsg);
	
	vapId = pSetBssMsg->vapId; 
	/*From CSA perspective the VAP is only Added when the SET BSS command is received
	However, the Set BSS command may also be sent when VAP is already up - e.g. set slot time
	CSA manager should add the VAP only when the Set BSS is received during VAP bringup*/
	// Req 16: 4.2.14 - channel selection & switching
	if ( (CsaManagerVapManager_Db[vapId].active == FALSE) &&
		 (VAP_MODE_AP == VapDb_GetVapMode(vapId))	)
	{
		/*Store flags in VAP DB*/
		CsaManagerVapManager_Db[vapId].flags = pSetBssMsg->flags;
		/*Add VAP Indications can only be received when CSA is Idle as VAP Activate Process (in this case Set BSS process)
		will not run when a Channel Switch Process is running - just move VAP to idle and send indication to CSA manager*/
		CsaManager_VapManagerChangeState(CSA_MANAGER_VAP_MANAGER_IDLE_STATE, vapId);
		CsaManagerVapOpenInd(vapId);
		CsaManagerVapManager_Db[vapId].active = TRUE;
	}
	/*Send Confirmation to VAP manager*/
	confirmEvent = (BSS_MANAGER_CONFIRM_EVENT*)(&(csaManagerMessage->abData));
	confirmEvent->vapId = vapId;
	confirmEvent->eventIndex = VAP_MANAGER_SET_BSS; 
	confirmEvent->clientId = BSS_MANAGER_VAP_MANAGER_CSA_MANAGER_CLIENT; 
	OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, csaManagerMessage, vapId);
}

/**********************************************************************************

csaManagerVapRemove 


Description:
------------
VAP Remove message handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void csaManagerVapRemove(K_MSG* csaManagerMessage)
{
	uint8 vapId; 
	BSS_MANAGER_CONFIRM_EVENT* confirmEvent; 
	K_MSG *vapMsg = (K_MSG *) (*((uint32 *)pK_MSG_DATA(csaManagerMessage)));
	UMI_REMOVE_VAP* removeVapPtr = (UMI_REMOVE_VAP*) pK_MSG_DATA(vapMsg);
	
	vapId = removeVapPtr->vapId; 

	/*Remove VAP Indications can only be received when CSA is Idle as VAP Remove Process will not run when
	a Channel Switch Process is running - just send indication to CSA manager*/
	if (CsaManagerVapManager_Db[vapId].active == TRUE)
	{
		CsaManagerVapRemoveInd(vapId);
		/*Clear active flag*/
		CsaManagerVapManager_Db[vapId].active = FALSE;
	}
	/*Send Confirmation to VAP manager*/
	confirmEvent = (BSS_MANAGER_CONFIRM_EVENT*)(&(csaManagerMessage->abData));
	confirmEvent->vapId= vapId;
	confirmEvent->eventIndex = VAP_MANAGER_REMOVE_VAP; 
	confirmEvent->clientId = BSS_MANAGER_VAP_MANAGER_CSA_MANAGER_CLIENT; 
	OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, csaManagerMessage, vapId);
}

/**********************************************************************************

csaManagerTxStartReq 


Description:
------------
CSA Start TX REQ from Channel Switch Manager


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void CsaManager_VapManagerTxStart(uint8 vapId)
{
	/*Call handler*/
	CsaManager_VapManagerStartTx[CsaManagerVapManager_Db[vapId].state]((void *)(uint32)vapId);
}

/**********************************************************************************

CsaManager_VapManagerTxStop 


Description:
------------
CSA Stop TX REQ from Channel Switch Manager


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void CsaManager_VapManagerTxStop(uint8 vapId)
{
	/*Call handler*/
	CsaManager_VapManagerStopTx[CsaManagerVapManager_Db[vapId].state]((void *)(uint32)vapId);
}

/**********************************************************************************

csaManagerTxCfm 


Description:
------------
CSA TX CFM - from STA manager


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void csaManager_VapManagerTxCfm(uint8 vapId, StaId sid)
{
	/*Call handler*/
	CsaManager_VapManagerTxCfm[CsaManagerVapManager_Db[vapId].state]((void *)(uint32)sid);
}

/**********************************************************************************

csaManager_VapManagerTxInd 


Description:
------------
CSA TX IND - from STA manager


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void csaManager_VapManagerTxInd(TxPd_t *pd)
{
	/*Get VAP ID from PD*/
	uint8 vapId = pd->txQVapId;
	/*Call handler*/
	CsaManager_VapManagerTxInd[CsaManagerVapManager_Db[vapId].state]((void *)pd);
}

/**********************************************************************************

CsaManager_VapManagerRemoveStaFromSecQueue 


Description:
------------
Utility that removes a SID from the pending SEC queue if found


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
bool CsaManager_VapManagerRemoveStaFromSecQueue(uint8 vapId, StaId sid)
{
	TxPd_t *prevPd = (TxPd_t *)NULL_PD;
	TxPd_t *pd = CsaManagerVapManager_Db[vapId].secQueueHead;
	bool found = FALSE;

	while (pd != (TxPd_t *)NULL_PD)
	{
		if (pd->txQStaId == sid)
		{
			/*found it!*/
			if (prevPd != (TxPd_t *)NULL_PD)
			{
				prevPd->nextPdPointer = SET_NEXT_PD(GET_NEXT_PD(pd));
			}
			else
			{
				/*PD was head of queue*/
				CsaManagerVapManager_Db[vapId].secQueueHead = (TxPd_t *)GET_NEXT_PD(pd);
			}
			if (pd->nextPdPointer == NEXT_PD_NULL)
			{
				/*PD was tail of queue*/
				CsaManagerVapManager_Db[vapId].secQueueTail = prevPd;
			}
			pd->packetPointer = CONVERT_DMA_SHRAM_ADDR_TO_WLAN_SHRAM_ADDR(pd->packetPointer);
			found = TRUE;
			break;
		}
		prevPd = pd;
		pd = (TxPd_t *)GET_NEXT_PD(pd);
	}
	return (found);
}

/**********************************************************************************

CsaManagerVapManagerAddStaInd 


Description:
------------
Add STA IND - from STA manager


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void CsaManager_VapManagerAddStaInd(uint8 vapId, StaId sid)
{
	CsaManager_VapManagerAddSta[CsaManagerVapManager_Db[vapId].state]((void *)(uint32)sid);
}

/**********************************************************************************

CsaManagerVapManagerRemoveStaInd 


Description:
------------
Remove STA IND - from STA manager


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void CsaManager_VapManagerRemoveStaInd(uint8 vapId, StaId sid)
{
	CsaManager_VapManagerRemoveSta[CsaManagerVapManager_Db[vapId].state]((void *)(uint32)sid);
}

/**********************************************************************************

csaManagerLockCfm




Description:
------------
Lock COnfirm message handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void csaManagerVapGphpLockCfm(K_MSG* csaManagerMessage)
{
	LockReqCb_t *lockReg = (LockReqCb_t *)pK_MSG_DATA(csaManagerMessage);
	uint8 vapId = lockReg->stationOrVapNum;

	/*Handle pending entries*/
	CsaManager_VapManagerHandleGPHPLock();
	/*Call handler*/
	CsaManager_VapManagerLockCfm[CsaManagerVapManager_Db[vapId].state]((void *)(uint32)vapId);
}


/**********************************************************************************

csaManager_Init



Description:
------------
Initiailize CSA MAnager VAP Manager


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void CsaManager_VapManagerInit()
{
	StaId i;
	
	/*Clear Database*/
	for (i = 0; i < HW_NUM_OF_VAPS; i++)
	{
		CsaManagerVapManager_Db[i].state = CSA_MANAGER_VAP_MANAGER_IDLE_STATE;
		CsaManagerVapManager_Db[i].flags = 0;
		CsaManagerVapManager_Db[i].active = FALSE;
		CsaManagerVapManager_Db[i].numSta = 0;
		CsaManagerVapManager_Db[i].pendingSta = 0;
		CsaManagerVapManager_Db[i].pendingPd = 0;
		CsaManagerVapManager_Db[i].headSta = INVALID_STA_INDEX;
		CsaManagerVapManager_Db[i].tailSta = INVALID_STA_INDEX;
		CsaManagerVapManager_Db[i].confirmHead = INVALID_STA_INDEX;
		CsaManagerVapManager_Db[i].confirmTail = INVALID_STA_INDEX;
		CsaManagerVapManager_Db[i].secQueueHead = (TxPd_t *)NULL_PD;
		CsaManagerVapManager_Db[i].secQueueTail = (TxPd_t *)NULL_PD;
		CsaManagerVapManager_Db[i].pd = (TxPd_t *)NULL_PD;
	}
	for (i = 0; i < HW_NUM_OF_STATIONS; i++)
	{
		CsaManagerVapManager_StaDb[i].nextSta = INVALID_STA_INDEX;
		CsaManagerVapManager_StaDb[i].prevSta = INVALID_STA_INDEX;
		CsaManagerVapManager_StaDb[i].pendingPd = FALSE;
	}
	CsaManagerVapManager_LockHead = CSA_MANAGER_VAP_MANAGER_INVALID_LOCK_IDX;
	CsaManagerVapManager_LockTail = CSA_MANAGER_VAP_MANAGER_INVALID_LOCK_IDX;
}



