/***********************************************************************************
 File:			CsaManager.c
 Module:		CSA Manager
 Purpose: 		handle CSA Manager events
 Description:   
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
	//TBD to check which files can be removed
#include "BSSmanager_API.h"
#include "TxManager_Api.h"
#include "TsManager_API.h"
#include "TxSelector_Api.h"
#include "HostInterface_API.h"
#include "Utils_Api.h"
#include "ResourceManager_API.h"
#include "Locker_Api.h"
#include "OSAL_Kmsg.h"
#include "ErrorHandler_Api.h"
#include "HwQManager_API.h"
#include "HwGlobalDefinitions.h"
#include "ShramPacketDescriptors.h"
#include "stringLibApi.h"
#include "um_interface.h"
#include "PacketDescriptor.h"
#include "HwMemoryMap.h"
#include "ChannelSwitchManager_Api.h"
#include "CsaManager_Api.h"
#include "CsaManager.h"
#include "CsaManager_VapManager.h"
#include "CsaManager_StaManager.h"
#include "ConfigurationManager_api.h"
#include "loggerAPI.h"


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_CSA_MANAGER 
#define LOG_LOCAL_FID 1



/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/
#define CSA_MANAGER_INVALID_VAP			0xFF

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

/*CSA Manager states*/
typedef enum
{
	CSA_MANAGER_IDLE_STATE,
	CSA_MANAGER_WAIT_TX_CFM_STATE,
	CSA_MANAGER_WAIT_STOP_CFM_STATE,
	CSA_MANAGER_NUM_STATES
} CsaManager_state_e;

typedef struct _CsaManagerVapDb_t
{
	uint8					nextVap;
	uint8					prevVap;
} CsaManagerVapDb_t;

typedef struct _CsaManagerDb_t
{
	uint32 							txSegmentsEnabledVaps[CSA_TYPE_NUM_OF_TYPES];
	CsaManager_state_e				state;
	uint8							numVaps;
	uint8							pendingVaps;
	uint8							vapHead;
	uint8							confirmHead;
	uint8							plannedTxSegments;
	uint8							counterExpired;
	CsaManager_csaType_e			currentTxSegment;
	CsaManagerStartTxReqParams_t	msg;
} CsaManagerDb_t;

typedef void (*CsaManager_Func)(void *parameter);

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void CsaManager_ChangeState(CsaManager_state_e state, uint8 bandId);
static void CsaManager_RemoveVapFromList(uint8 vapId);
static void CsaManager_ClearVapEntry(uint8 vapId);
static void CsaManager_AddVapToList(uint8 vapId);
static void CsaManager_AddVapToConfirmList(uint8 vapId);
static void CsaManager_MoveConfirmToList(uint8 bandId);
static void CsaManager_TxCfmHandler(void *parameter);
static void CsaManager_StartTxInIdle(void *parameter);
static void CsaManager_StopTxInIdle(void *parameter);
static void CsaManager_StopTxInWaitForCfm(void *parameter);
static void CsaManager_TxCfmInWaitForCfm(void *parameter);
static void CsaManager_TxCfmInWaitForStopCfm(void *parameter);
static void CsaManager_Fatal(void *parameter);
static void CsaManager_Ignore(void *parameter);
static void CsaManager_TxStartUcDeauthTimerExpiredInWaitForCfm(void *parameter);
static void CsaManager_TxStartMcDeauthTimerExpiredInWaitForCfm(void *parameter);
static void CsaManager_TxStartSegment(uint8 bandId);
static void CsaManager_TxStopSegment(uint8 bandId);
static void CsaManagerSetVapsForDeauthFrames(uint32 csaUcDeAuthVapBitmap, uint32 csaMcDeAuthVapBitmap, uint8 bandId);

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
/*CSA Manager Start TX Event Handlers*/
static const CsaManager_Func CsaManager_StartTx[CSA_MANAGER_NUM_STATES] =
{
	 CsaManager_StartTxInIdle, 				 	/* CSA_MANAGER_IDLE_STATE */
	 CsaManager_Fatal,					  		/* CSA_MANAGER_WAIT_TX_CFM_STATE */
 	 CsaManager_Fatal					  		/* CSA_MANAGER_WAIT_STOP_CFM_STATE */
};

/*CSA Manager Stop TX Event Handlers*/
static const CsaManager_Func CsaManager_StopTx[CSA_MANAGER_NUM_STATES] =
{
	CsaManager_StopTxInIdle,				   	/* CSA_MANAGER_IDLE_STATE */
	CsaManager_StopTxInWaitForCfm,			   	/* CSA_MANAGER_WAIT_TX_CFM_STATE */
	CsaManager_Fatal						   	/* CSA_MANAGER_WAIT_STOP_CFM_STATE */
};

/*CSA Manager TX CFM Event Handler*/
static const CsaManager_Func CsaManager_TxCfm[CSA_MANAGER_NUM_STATES] =
{
	CsaManager_Fatal,							/* CSA_MANAGER_IDLE_STATE */
	CsaManager_TxCfmInWaitForCfm,			   	/* CSA_MANAGER_WAIT_TX_CFM_STATE */
	CsaManager_TxCfmInWaitForStopCfm		   	/* CSA_MANAGER_WAIT_STOP_CFM_STATE */
};

/*CSA Manager Tx Start UC Deauth Timer Event Handler*/
static const CsaManager_Func CsaManager_TxStartUcDeauthTimerExpired[CSA_MANAGER_NUM_STATES] =
{
	CsaManager_Ignore,									/* CSA_MANAGER_IDLE_STATE */
	CsaManager_TxStartUcDeauthTimerExpiredInWaitForCfm,	/* CSA_MANAGER_WAIT_TX_CFM_STATE */
	CsaManager_Ignore		   							/* CSA_MANAGER_WAIT_STOP_CFM_STATE */
};

/*CSA Manager Tx Start MC Deauth Timer Event Handler*/
static const CsaManager_Func CsaManager_TxStartMcDeauthTimerExpired[CSA_MANAGER_NUM_STATES] =
{
	CsaManager_Ignore,									/* CSA_MANAGER_IDLE_STATE */
	CsaManager_TxStartMcDeauthTimerExpiredInWaitForCfm,	/* CSA_MANAGER_WAIT_TX_CFM_STATE */
	CsaManager_Ignore		   							/* CSA_MANAGER_WAIT_STOP_CFM_STATE */
};


static CsaManagerVapDb_t CsaManager_VapDb[HW_NUM_OF_VAPS];
static CsaManagerDb_t CsaManager_Db[NUM_OF_CONFIGURATION_MANAGER_BANDS];

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

CsaManager_ChangeState 


Description:
------------
Utility function to change State


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_ChangeState(CsaManager_state_e state, uint8 bandId)
{
 	ILOG0_DD("CSA Manager changed from %d to %d", CsaManager_Db[bandId].state, state);
	CsaManager_Db[bandId].state = state;
}

/**********************************************************************************

CsaManager_RemoveVapFromList 


Description:
------------
Utility function that removes a VAP from the list


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_RemoveVapFromList(uint8 vapId)
{
	uint8 prevVap = CsaManager_VapDb[vapId].prevVap;
	uint8 nextVap = CsaManager_VapDb[vapId].nextVap;
	uint8 bandId = ConfigurationManager_GetBandForVap(vapId);

	ILOG0_DDD("[CSA] CsaManager_RemoveVapFromList vap = %d prevVap = %d nextVap = %d", vapId, prevVap, nextVap);

	if (prevVap == CSA_MANAGER_INVALID_VAP)
	{
		CsaManager_Db[bandId].vapHead = nextVap;
	}
	else
	{
		CsaManager_VapDb[prevVap].nextVap = nextVap;
	}
	if (nextVap != CSA_MANAGER_INVALID_VAP)
	{
		CsaManager_VapDb[nextVap].prevVap = prevVap;
	}
}

/**********************************************************************************

CsaManager_ClearVapEntry 


Description:
------------
Utility function that clears a VAP entry


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_ClearVapEntry(uint8 vapId)
{
	CsaManager_VapDb[vapId].nextVap = CSA_MANAGER_INVALID_VAP;
	CsaManager_VapDb[vapId].prevVap = CSA_MANAGER_INVALID_VAP;
}

/**********************************************************************************

CsaManager_AddVapToList 


Description:
------------
Utility function that add a VAP to the list


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_AddVapToList(uint8 vapId)
{
	uint8 bandId = ConfigurationManager_GetBandForVap(vapId);

	
	if (CsaManager_Db[bandId].vapHead != CSA_MANAGER_INVALID_VAP)
	{
		CsaManager_VapDb[vapId].nextVap = CsaManager_Db[bandId].vapHead;
		CsaManager_VapDb[CsaManager_Db[bandId].vapHead].prevVap = vapId;
	}
	CsaManager_Db[bandId].vapHead = vapId;
	/*Increment number of VAPs*/
	CsaManager_Db[bandId].numVaps++;
}

/**********************************************************************************

CsaManager_AddVapToConfirmList 


Description:
------------
Utility function that add a VAP to the confirm list


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_AddVapToConfirmList(uint8 vapId)
{
	uint8 bandId = ConfigurationManager_GetBandForVap(vapId);

	ILOG0_DD("[CSA] CsaManager_AddVapToConfirmList vapId = %d confirmHead = %d", vapId, CsaManager_Db[bandId].confirmHead);
	
	if (CsaManager_Db[bandId].confirmHead != CSA_MANAGER_INVALID_VAP)
	{
		CsaManager_VapDb[vapId].nextVap = CsaManager_Db[bandId].confirmHead;
		CsaManager_VapDb[CsaManager_Db[bandId].confirmHead].prevVap = vapId;
	}
	CsaManager_Db[bandId].confirmHead = vapId;
}

/**********************************************************************************

CsaManager_MoveConfirmToList 


Description:
------------
Utility function that moves VAP back from the confirm list to the list


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_MoveConfirmToList(uint8 bandId)
{
	CsaManager_Db[bandId].vapHead = CsaManager_Db[bandId].confirmHead;
	CsaManager_Db[bandId].confirmHead = CSA_MANAGER_INVALID_VAP;	
}

/**********************************************************************************

CsaManager_TxCfmHandler 


Description:
------------
TX CFM utility function


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_TxCfmHandler(void *parameter)
{
	uint32 vapId = (uint32)parameter;
	uint8 bandId = ConfigurationManager_GetBandForVap(vapId);


	/*Remove VAP from pending list*/
	CsaManager_RemoveVapFromList(vapId);
	CsaManager_ClearVapEntry(vapId);
	/*Add VAP to confirm list*/
	CsaManager_AddVapToConfirmList(vapId);
	/*If all VAPs confirmed*/
	CsaManager_Db[bandId].pendingVaps--;
	ILOG0_D("[CSA] CsaManager_TxCfmHandler pendingVaps = %d", CsaManager_Db[bandId].pendingVaps);
	if (CsaManager_Db[bandId].pendingVaps == 0)
	{
		/*Move VAPs back to pending list and clear confrim list*/
		CsaManager_MoveConfirmToList(bandId);
	}
}

/***********************************************************************

Description:
This function makes sure that we enable sending of segmentes (UC|MC DEAUTH) only
in vaps that have stations connected to. In case we attempt to send segments on VAPs
that do not have stations connected to, well, LO KEF.
 
Input:
------
None
 
Output:
-------
None
 
Returns:
--------
None
 
************************************************************************/
static void CsaManagerSetVapsForDeauthFrames(uint32 csaUcDeAuthVapBitmap, uint32 csaMcDeAuthVapBitmap, uint8 bandId)
{
	uint16 vapIndex = 0;

	/* Reset enabled VAPs */
	CsaManager_Db[bandId].txSegmentsEnabledVaps[CSA_TYPE_UC_DEAUTH] = 0;
	CsaManager_Db[bandId].txSegmentsEnabledVaps[CSA_TYPE_MC_DEAUTH] = 0;

	/* Loop on all VAPs */
	for (vapIndex = 0; vapIndex < HW_NUM_OF_VAPS; vapIndex++)
	{
		/* Check that there is at least one station connected to this VAP */
		if (CsaManager_VapManagerGetNumStaInVap(vapIndex) != 0)
		{
			/* Check if Driver requested to send UCAST DEAUTH to stations in this VAP */
			if (csaUcDeAuthVapBitmap & (TRUE << vapIndex))
			{
				CsaManager_Db[bandId].txSegmentsEnabledVaps[CSA_TYPE_UC_DEAUTH] |= (TRUE << vapIndex);
			}
			
			/* Check if Driver requested to send MCAST DEAUTH to stations in this VAP */
			if (csaMcDeAuthVapBitmap & (TRUE << vapIndex))
			{
				CsaManager_Db[bandId].txSegmentsEnabledVaps[CSA_TYPE_MC_DEAUTH] |= (TRUE << vapIndex);
			}			
		}
	}	

	ILOG0_DD("[CSA] CsaManagerSetEnabledVaps txSegmentsEnabledVaps[CSA_TYPE_UC_DEAUTH] = 0x%x txSegmentsEnabledVaps[CSA_TYPE_MC_DEAUTH] = 0x%x", 
		CsaManager_Db[bandId].txSegmentsEnabledVaps[CSA_TYPE_UC_DEAUTH], CsaManager_Db[bandId].txSegmentsEnabledVaps[CSA_TYPE_MC_DEAUTH]);	
}

/**********************************************************************************

CsaManager_StartTxInIdle 


Description:
------------
Handle Start TX REQ In Idle State


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_StartTxInIdle(void *parameter)
{
	/*Send TX Start REQ to all connected VAPs*/

	K_MSG *csaMessage = (K_MSG *)(uint32)parameter;
	uint8 vapIndex = csaMessage->header.vapId;
	uint8 bandId = ConfigurationManager_GetBandForVap(vapIndex);
	uint8 vapId = CsaManager_Db[bandId].vapHead;
	CsaManagerStartTxReqParams_t *txReqMessage = (CsaManagerStartTxReqParams_t *)pK_MSG_DATA((K_MSG *)parameter);
	uint16 startTxUcDeauthTime;
	uint16 startTxMcDeauthTime;

	ILOG0_V("[CSA] CsaManager_StartTxInIdle 1");

	if(vapId == CSA_MANAGER_INVALID_VAP)
	{
		ILOG0_V("[CSA] CsaManager_StartTxInIdle 2");
		OSAL_SEND_NO_DATA_MESSAGE(CHANNEL_SWITCH_MANAGER_CSA_MANAGER_START_TX_CFM, TASK_CHANNEL_SWITCH_MANAGER, vapIndex);
	}
	else
	{
		ILOG0_V("[CSA] CsaManager_StartTxInIdle 3");
		//initiate sequence of CSA frames:
		//0. transmit ONLY CSA frames
		//1. transmit CSA frames, followed by U/C deauthentication frames
		//2. transmit CSA frames, followed by M/C deauthentication frames
		//3. transmit CSA frames, followed by U/C & M/C deauthentication frames
		
		/* Enable VAPs that have at least one station connected to */
		CsaManagerSetVapsForDeauthFrames(txReqMessage->pUmSetChannelParams->csaUcDeAuthVapBitmap, txReqMessage->pUmSetChannelParams->csaMcDeAuthVapBitmap, bandId);

		CsaManager_Db[bandId].plannedTxSegments = (TRUE << CSA_TYPE_CSA);

		if (CsaManager_Db[bandId].txSegmentsEnabledVaps[CSA_TYPE_UC_DEAUTH] != 0)
		{			
			CsaManager_Db[bandId].plannedTxSegments |= (TRUE << CSA_TYPE_UC_DEAUTH);
			//start timer for stop tx CSA and start tx UC DEAUTH after a fraction of channel switch time (Driver configures fraction size)
			startTxUcDeauthTime = txReqMessage->pUmSetChannelParams->csaTxUcDeauthStartTime;
			ILOG0_D("[CSA] CsaManager_StartTxInIdle 4 startTxUcDeauthTime = %d", startTxUcDeauthTime);
			if (CONFIGURATION_MANAGER_BAND_0 == bandId)
			{
				OSAL_SET_TIMER_EXPLICIT(CSA_MANAGER_START_TX_UC_DEAUTH, OSAL_TIMERS_MS_TO_K_TICKS(startTxUcDeauthTime), TASK_TX_MANAGER);
			}
			else
			{
				OSAL_SET_TIMER_EXPLICIT(CSA_MANAGER_START_TX_UC_DEAUTH_B1, OSAL_TIMERS_MS_TO_K_TICKS(startTxUcDeauthTime), TASK_TX_MANAGER);
			}
		}

		if (CsaManager_Db[bandId].txSegmentsEnabledVaps[CSA_TYPE_MC_DEAUTH] != 0)
		{			
			CsaManager_Db[bandId].plannedTxSegments |= (TRUE << CSA_TYPE_MC_DEAUTH);
			//start timer for stop tx UC DEAUTH and start tx MC DEAUTH after a fraction of channel switch time (Driver configures fraction size)
			startTxMcDeauthTime = txReqMessage->pUmSetChannelParams->csaTxMcDeauthStartTime;
			ILOG0_D("[CSA] CsaManager_StartTxInIdle 5 startTxMcDeauthTime = %d", startTxMcDeauthTime);
			if (CONFIGURATION_MANAGER_BAND_0 == bandId)
			{
				OSAL_SET_TIMER_EXPLICIT(CSA_MANAGER_START_TX_MC_DEAUTH, OSAL_TIMERS_MS_TO_K_TICKS(startTxMcDeauthTime), TASK_TX_MANAGER);
			}
			else
			{
				OSAL_SET_TIMER_EXPLICIT(CSA_MANAGER_START_TX_MC_DEAUTH_B1, OSAL_TIMERS_MS_TO_K_TICKS(startTxMcDeauthTime), TASK_TX_MANAGER);
			}
		}

		CsaManager_Db[bandId].currentTxSegment = CSA_TYPE_CSA;

		/*Store channel params*/
		MEMCPY(&CsaManager_Db[bandId].msg, txReqMessage, sizeof(CsaManagerStartTxReqParams_t));

		/*Set State to Wait TX CFM*/
		CsaManager_ChangeState(CSA_MANAGER_WAIT_TX_CFM_STATE, bandId);

		CsaManager_TxStartSegment(bandId);
	}
}


/**********************************************************************************
CsaManager_GetCurrentTxSegment 

Description:
------------
return current Tx Segment

Input: 
-----	
		
Output:
-------

Returns:
--------
	currentTxSegment
**********************************************************************************/
CsaManager_csaType_e CsaManager_GetCurrentTxSegment(uint8 vapIndex)
{
	uint8 bandId = ConfigurationManager_GetBandForVap(vapIndex);
	
	return CsaManager_Db[bandId].currentTxSegment;
}

/**********************************************************************************
CsaManager_StopTxInIdle 

Description:
------------
handle Stop REQ in Idle

Input: 
-----	
		
Output:
-------

Returns:
--------
	void - 
**********************************************************************************/
static void CsaManager_StopTxInIdle(void *parameter)
{
	K_MSG *csaMessage = (K_MSG *)(uint32)parameter;

	ILOG0_V("[CSA] CsaManager_StopTxInIdle 1");
	
	/*Chanel Switch Manager always sends Stop TX REQ even if we sent Start TX CFM
	Just send Stop TX CFM to Channel Switch Manager*/
	OSAL_SEND_NO_DATA_MESSAGE(CHANNEL_SWITCH_MANAGER_CSA_MANAGER_STOP_TX_CFM, TASK_CHANNEL_SWITCH_MANAGER, csaMessage->header.vapId);
	
}

/**********************************************************************************
CsaManager_StopTxInWaitForCfm 

Description:
------------
handles Stop TX REQ in Wait for CFM state

Input: 
-----	
		
Output:
-------

Returns:
--------
	void - 
**********************************************************************************/
static void CsaManager_StopTxInWaitForCfm(void *parameter)
{
	K_MSG *csaMessage = (K_MSG *)(uint32)parameter;
	uint8 vapIndex = csaMessage->header.vapId;
	uint8 bandId = ConfigurationManager_GetBandForVap(vapIndex);

	ILOG0_V("[CSA] CsaManager_StopTxInWaitForCfm 1");


	/* We might have pending segments, so need to kill timers and mark all segments as executed (delay the marking since plannedTxSegments is checked at CsaManager_VapManagerStopTxInIdle) */
	if (CONFIGURATION_MANAGER_BAND_0 == bandId)
	{
		OSAL_RESET_TIMER_EXPLICIT(CSA_MANAGER_START_TX_UC_DEAUTH, TASK_TX_MANAGER);
		OSAL_RESET_TIMER_EXPLICIT(CSA_MANAGER_START_TX_MC_DEAUTH, TASK_TX_MANAGER);
	}
	else
	{
		OSAL_RESET_TIMER_EXPLICIT(CSA_MANAGER_START_TX_UC_DEAUTH_B1, TASK_TX_MANAGER);
		OSAL_RESET_TIMER_EXPLICIT(CSA_MANAGER_START_TX_MC_DEAUTH_B1, TASK_TX_MANAGER);
	}

	/*Set State to Wait Stop CFM*/
	CsaManager_ChangeState(CSA_MANAGER_WAIT_STOP_CFM_STATE, bandId);
	
	/*Send Stop TX REQ to pending VAPs*/
	CsaManager_TxStopSegment(bandId);

	/*  */
	CsaManager_Db[bandId].plannedTxSegments = 0x0;	
}


/**********************************************************************************

CsaManager_TxCfmInWaitForCfm 


Description:
------------
handle TX CFM in Wait For CFM


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CsaManager_TxCfmInWaitForCfm(void *parameter)
{
	uint32 vapId = (uint32)parameter;
	uint8 bandId = ConfigurationManager_GetBandForVap(vapId);
	uint32 nextTxSegment;

	ILOG0_DD("[CSA] CsaManager_TxCfmInWaitForCfm 1 pendingVaps = 0x%x current segments = 0x%x", CsaManager_Db[bandId].pendingVaps, (CsaManager_Db[bandId].currentTxSegment));
	
	CsaManager_TxCfmHandler(parameter);
	if (CsaManager_Db[bandId].pendingVaps == 0)
	{
		ILOG0_D("[CSA] CsaManager_TxCfmInWaitForCfm 2 planned segments = 0x%x", CsaManager_Db[bandId].plannedTxSegments);
		/*clear completed tx segment from planned tx segments*/
		CsaManager_Db[bandId].plannedTxSegments ^= (TRUE << (CsaManager_Db[bandId].currentTxSegment));
		
		/*Check if there are more planned tx segments*/
		if (CsaManager_Db[bandId].plannedTxSegments)
		{
			ILOG0_D("[CSA] CsaManager_TxCfmInWaitForCfm 3 planned segments = 0x%x", CsaManager_Db[bandId].plannedTxSegments);
			nextTxSegment = Utils_CountTrailingZeros(CsaManager_Db[bandId].plannedTxSegments);
			DEBUG_ASSERT(nextTxSegment < CSA_TYPE_NUM_OF_TYPES);

			if (CsaManager_Db[bandId].counterExpired & (TRUE << nextTxSegment))
			{
				ILOG0_V("[CSA] CsaManager_TxCfmInWaitForCfm 4");
				//turn off indication
				CsaManager_Db[bandId].counterExpired ^= (TRUE << nextTxSegment);
				//move to next tx segment
				CsaManager_Db[bandId].currentTxSegment = nextTxSegment;
				//start tx
				CsaManager_TxStartSegment(bandId);
				
			}
			else
			{
				ILOG0_V("[CSA] CsaManager_TxCfmInWaitForCfm 5");
				//do nothing more, just wait for timer event to start transmitting the next planned tx segment
			}
		}
		else
		{
			ILOG0_V("[CSA] CsaManager_TxCfmInWaitForCfm 6");
			/*Move back to Idle*/
			CsaManager_ChangeState(CSA_MANAGER_IDLE_STATE, bandId);
			/*Send Start TX REQ CFM to Channel Switch Manager*/
			OSAL_SEND_NO_DATA_MESSAGE(CHANNEL_SWITCH_MANAGER_CSA_MANAGER_START_TX_CFM, TASK_CHANNEL_SWITCH_MANAGER, vapId);
		}
	}
}


/**********************************************************************************
TxManager_StaManagerOpenSta 

Description:
------------
handles STA Open event

Input: 
-----	
TxManagerStaDb_t - STA DB entry
		
Output:
-------

Returns:
--------
	void - 
**********************************************************************************/
static void CsaManager_TxCfmInWaitForStopCfm(void *parameter)
{
	uint32 vapId = (uint32)parameter;
	uint8 bandId = ConfigurationManager_GetBandForVap(vapId);

	ILOG0_D("[CSA] CsaManager_TxCfmInWaitForStopCfm pendingVaps = %d", CsaManager_Db[bandId].pendingVaps);
	
	CsaManager_TxCfmHandler(parameter);
	if (CsaManager_Db[bandId].pendingVaps == 0)
	{
		/*Move back to Idle*/
		CsaManager_ChangeState(CSA_MANAGER_IDLE_STATE, bandId);
		/*We got to this state because we got a stop tx REQ before TX was confirmed. First, Send Start TX REQ CFM to Channel Switch Manager*/
		OSAL_SEND_NO_DATA_MESSAGE(CHANNEL_SWITCH_MANAGER_CSA_MANAGER_START_TX_CFM, TASK_CHANNEL_SWITCH_MANAGER, vapId);
		/*Send Stop TX REQ CFM to Channel Switch Manager*/
		OSAL_SEND_NO_DATA_MESSAGE(CHANNEL_SWITCH_MANAGER_CSA_MANAGER_STOP_TX_CFM, TASK_CHANNEL_SWITCH_MANAGER, vapId);
	}
}


/**********************************************************************************
CsaManager_Ignore 

Description:
------------
handles unexpected event that can be ignored

Input: 
-----	
			
Output:
-------

Returns:
--------
	
**********************************************************************************/
static void CsaManager_Ignore(void *parameter)
{
	UNUSED_PARAM(parameter);
	ILOG0_V("[CSA] CsaManager_Ignore");
}


/**********************************************************************************
CsaManager_TxStartUcDeauthTimerExpiredInWaitForCfm 

Description:
------------

Input: 
-----	
			
Output:
-------

Returns:
--------
	
**********************************************************************************/
static void CsaManager_TxStartUcDeauthTimerExpiredInWaitForCfm(void *parameter)
{
	uint8 bandId = (uint8)parameter;
	
	if (CsaManager_Db[bandId].pendingVaps == 0)
	{
		CsaManager_Db[bandId].currentTxSegment = CSA_TYPE_UC_DEAUTH;
		//start tx
		CsaManager_TxStartSegment(bandId);
	}
	else
	{
		CsaManager_Db[bandId].counterExpired |= (TRUE << CSA_TYPE_UC_DEAUTH);
		//stop tx
		CsaManager_TxStopSegment(bandId);
	}
}


/**********************************************************************************
CsaManager_TxStartMcDeauthTimerExpiredInWaitForCfm 

Description:
------------

Input: 
-----	
			
Output:
-------

Returns:
--------
	
**********************************************************************************/
static void CsaManager_TxStartMcDeauthTimerExpiredInWaitForCfm(void *parameter)
{
	uint8 bandId = (uint8)parameter;
	
	if (CsaManager_Db[bandId].pendingVaps == 0)
	{
		CsaManager_Db[bandId].currentTxSegment = CSA_TYPE_MC_DEAUTH;
		//start tx
		CsaManager_TxStartSegment(bandId);
	}
	else
	{
		CsaManager_Db[bandId].counterExpired |= (TRUE << CSA_TYPE_MC_DEAUTH);
		//stop tx
		CsaManager_TxStopSegment(bandId);
	}
}


/**********************************************************************************
CsaManager_TxStartSegment

Description:
------------

Input: 
-----	
			
Output:
-------

Returns:
--------
	
**********************************************************************************/
static void CsaManager_TxStartSegment(uint8 bandId)
{
	uint8 vapId;
	uint8 nextVapId;
	uint32 enabledVaps;

	ILOG0_V("[CSA] CsaManager_TxStartSegment 1");

	
	vapId = CsaManager_Db[bandId].vapHead;
		
	CsaManager_Db[bandId].pendingVaps =  CsaManager_Db[bandId].numVaps;
	enabledVaps = CsaManager_Db[bandId].txSegmentsEnabledVaps[CsaManager_Db[bandId].currentTxSegment];

	ILOG0_DDD("[CSA] CsaManager_TxStartSegment 2 vapId = %d enabledVaps = %d pendingVaps = %d", vapId, enabledVaps, CsaManager_Db[bandId].pendingVaps);
		
	/*Send Start TX REQ to all pending VAPs*/
	while (vapId != CSA_MANAGER_INVALID_VAP)
	{
		ILOG0_D("[CSA] CsaManager_TxStartSegment 3 vapId = %d", vapId);

		/*Save next VAP - if no STAs connected VAP will confirm immediately and this will remove VAP from list*/
		nextVapId = CsaManager_VapDb[vapId].nextVap;
		if (enabledVaps & (TRUE << vapId))
		{
			ILOG0_D("[CSA] CsaManager_TxStartSegment 4 vapId = %d", vapId);
			/*Send Start TX REQ to pending VAP*/
			CsaManager_VapManagerTxStart(vapId);
		}
		else
		{
			CsaManager_Db[bandId].pendingVaps--;
			/* We skip TX on this VAP. Add this VAP to the list of confirmed VAPs (immitating CsaManager_TxCfmHandler) */
			/*Remove VAP from pending list*/
			CsaManager_RemoveVapFromList(vapId);
			CsaManager_ClearVapEntry(vapId);
			/*Add VAP to confirm list*/
			CsaManager_AddVapToConfirmList(vapId);			
		}
		/*Move to next*/
		vapId = nextVapId;
	}
}


/**********************************************************************************
CsaManager_TxStopSegment

Description:
------------

Input: 
-----	
			
Output:
-------

Returns:
--------
	
**********************************************************************************/
static void CsaManager_TxStopSegment(uint8 bandId)
{
	uint8 vapId;
	uint8 nextVapId;
	uint32 enabledVaps;
	
	vapId = CsaManager_Db[bandId].vapHead;
	enabledVaps = CsaManager_Db[bandId].txSegmentsEnabledVaps[CsaManager_Db[bandId].currentTxSegment];

	ILOG0_D("[CSA] CsaManager_TxStopSegment 1 pendingVaps = %d", CsaManager_Db[bandId].pendingVaps);

	if (CsaManager_Db[bandId].pendingVaps == 0)
	{
		/*Move back to Idle*/
		CsaManager_ChangeState(CSA_MANAGER_IDLE_STATE, bandId);
		/*We got to this state because we got a stop tx REQ before TX was confirmed. First, Send Start TX REQ CFM to Channel Switch Manager*/
		OSAL_SEND_NO_DATA_MESSAGE(CHANNEL_SWITCH_MANAGER_CSA_MANAGER_START_TX_CFM, TASK_CHANNEL_SWITCH_MANAGER, vapId);
		/*Send Stop TX REQ CFM to Channel Switch Manager*/
		OSAL_SEND_NO_DATA_MESSAGE(CHANNEL_SWITCH_MANAGER_CSA_MANAGER_STOP_TX_CFM, TASK_CHANNEL_SWITCH_MANAGER, vapId);
	}
	else
	{
		while (vapId != CSA_MANAGER_INVALID_VAP)
		{
			nextVapId = CsaManager_VapDb[vapId].nextVap;
			if (enabledVaps & (TRUE << vapId))
			{
				/*Send Stop TX REQ to pending VAP*/
				CsaManager_VapManagerTxStop(vapId);
			}
			/*Move to next*/
			vapId = nextVapId;
		}
	}
}


/**********************************************************************************
CsaManager_Fatal 

Description:
------------
handles unexpected event

Input: 
-----	
			
Output:
-------

Returns:
--------
	
**********************************************************************************/
static void CsaManager_Fatal(void *parameter)
{
	UNUSED_PARAM(parameter);	
	FATAL("CSA Manager Fatal");
}


/*---------------------------------------------------------------------------------
/						global Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

csaManagerVapOpenInd 


Description:
------------
VAP ADD Indication handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void CsaManagerVapOpenInd(uint8 vapId)
{
	CsaManager_AddVapToList(vapId);
}

/**********************************************************************************

csaManagerVapRemoveInd 


Description:
------------
VAP Remove Indication handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void CsaManagerVapRemoveInd(uint8 vapId)
{
	uint8 bandId = ConfigurationManager_GetBandForVap(vapId);

	
	/*Remove VAP Indications can only be received when CSA is Idle as VAP Activate Process will not run when
	a Channel Switch Process is running - just remove the VAP from the list*/
	CsaManager_RemoveVapFromList(vapId);
	CsaManager_ClearVapEntry(vapId);
	/*Decrement number of VAPs*/
	CsaManager_Db[bandId].numVaps--;
}

/**********************************************************************************

csaManagerTxStartReq 


Description:
------------
CSA Start TX REQ from Channel Switch Manager


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void csaManagerTxStartReq(K_MSG* csaManagerMessage)
{
	uint8 bandId = ConfigurationManager_GetBandForVap(csaManagerMessage->header.vapId);
	
	/*Call handler*/
	CsaManager_StartTx[CsaManager_Db[bandId].state]((void *)csaManagerMessage);
}

/**********************************************************************************

csaManagerTxStopReq 


Description:
------------
CSA Stop TX REQ from Channel Switch Manager


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void csaManagerTxStopReq(K_MSG* csaManagerMessage)
{
	uint32 vapId = csaManagerMessage->header.vapId;
	uint8 bandId = ConfigurationManager_GetBandForVap(vapId);
	
	/*Call handler*/
	CsaManager_StopTx[CsaManager_Db[bandId].state]((void *)csaManagerMessage);
}


/**********************************************************************************
csaManagerTxCfm

Description:
------------
TX Confirm message handler

Input: 
-----	
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
void csaManagerTxCfm(K_MSG* csaManagerMessage)
{
	/*From PD get SID*/
	CsaManagerPacketConfirmedMessage_t *csaMsg = (CsaManagerPacketConfirmedMessage_t *)pK_MSG_DATA(csaManagerMessage);
	TxPd_t *pd = csaMsg->pd;

	ILOG0_V("[CSA] csaManagerTxCfm 1");

	if (pd->mcUnicast == MULTICAST)
	{
		ILOG0_V("[CSA] csaManagerTxCfm 2 MULTICAST");
		CsaManager_VapManagerMcTxCfmMessageHandler(csaManagerMessage);
	}
	else
	{
		ILOG0_V("[CSA] csaManagerTxCfm 3 UNICAST");
		CsaManager_StaManagerTxCfmMessageHandler(csaManagerMessage);
	}
}


/**********************************************************************************

csaManagerTxCfm 


Description:
------------
CSA TX CFM - from VAP manager


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void csaManager_TxCfm(uint8 vapId)
{
	uint8 bandId = ConfigurationManager_GetBandForVap(vapId);
	
	/*Call handler*/
	CsaManager_TxCfm[CsaManager_Db[bandId].state]((void *)(uint32)vapId);
}


/**********************************************************************************

csaManager_BuildCsaUcDeAuth



Description:
------------
build CSA deauthentication frame according to parameters in TX Start REQ


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void csaManager_BuildCsaUcDeAuth(TxPd_t *pd)
{
	FM_PAYLOAD_DEAUTHENTICATION *csaDeAuthFramePayload = NULL;
	uint8 len = frame_sizeOfNewManagementFrameHeader(FALSE); /*Start with header len*/

	/*Build deauth frame*/
	csaDeAuthFramePayload = (FM_PAYLOAD_DEAUTHENTICATION *)frame_getPayloadPointerForNewManagementFrame((MANAGEMENT_BASIC_FRAME_HEADER *)pd->packetPointer, FALSE);
	csaDeAuthFramePayload->reasonCode = 0x1; //stands for UNSPECIFIED_REASON code -> according to IEEE Std 802.11-2016 Table 9-45Reason codes
	len += sizeof(FM_PAYLOAD_DEAUTHENTICATION);
	pd->dataLength = len;
}

    
/***********************************************************************
* CsaManager_GetCsaDeauthFrameEnc
* 
* Description:
* ------------
* Return the bitmap describing if encryption is used or not.
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
uint32 CsaManager_GetCsaDeauthFrameEnc(uint8 bandId)
{
	return(CsaManager_Db[bandId].msg.pUmSetChannelParams->csaIsMcDeauthFrameEncrypted);
}
    
/***********************************************************************
* csaManager_GetFrameLength
* 
* Description:
* ------------
* Return the length of the MCAST DEAUTH frame as provided by Driver.
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
uint32 csaManager_GetFrameLength(uint8 bandId, uint32 vapId)
{
	return(CsaManager_Db[bandId].msg.pUmSetChannelParams->csaMcDeauthFrameLength[vapId]);
}


/**********************************************************************************

csaManager_BuildCsa 


Description:
------------
build CSA according to parameters in TX Start REQ


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void csaManager_BuildCsa(TxPd_t *pd)
{
	FM_PAYLOAD_CSA *csaFramePayload = NULL;
	TxPd_t *pdPointer = (TxPd_t *)pd;
	uint8 len = frame_sizeOfNewManagementFrameHeader(FALSE); /*Start with header len*/
	uint8 bandId = ConfigurationManager_GetBandForVap(pd->txQVapId);
	CsaManagerStartTxReqParams_t *pMsg = &CsaManager_Db[bandId].msg;

	/*Build CSA frame*/
	csaFramePayload = (FM_PAYLOAD_CSA *)frame_getPayloadPointerForNewManagementFrame((MANAGEMENT_BASIC_FRAME_HEADER *)pdPointer->packetPointer, FALSE);
	csaFramePayload->u8ActionCode = SPECTRUM_MAN_CATEGORY_CHANNEL_SWITCH_ANNOUNCEMENT;
	csaFramePayload->u8CategoryCode = SPECTRUM_MAN_CATEGORY;
	len += FM_ELEMENT_SIZE;
	csaFramePayload->csaIe.ie.u8Code = FM_EL_11H_CHANNEL_SWITCH_ANNOUNCEMENT;
	csaFramePayload->csaIe.ie.u8Length = sizeof(S11H_CHANNEL_SWITCH_ANNOUNCEMENT_ELEMENT);
	/*Fill in Channel parameters*/
	csaFramePayload->csaIe.csa.ChannelSwitchMode = CsaManager_Db[bandId].msg.block_tx;
	csaFramePayload->csaIe.csa.NewChannelNumber = pMsg->primaryChannel;
	/*Zero the count Sender will OR the count*/
	csaFramePayload->csaIe.csa.ChannelSwitchCount = 0;
	len += sizeof(CSA_IE);
	/*ADD Secondary Channel Offset IE - if VAP HT or VAP VHT*/
	if ((CsaManager_VapManagerIsHT(pdPointer->txQVapId) == TRUE) || (CsaManager_VapManagerIsVHT(pdPointer->txQVapId) == TRUE))
	{
		csaFramePayload->secIe.ie.u8Code = FM_EL_SECONDARY_CH_OFFSET;
		csaFramePayload->secIe.ie.u8Length = sizeof(sHT_SECONDARY_CHANNEL_OFFSET);
		csaFramePayload->secIe.sec.Secondary_Channel_Offset = pMsg->secChannelOffset;
		len += sizeof(SEC_IE);
	}
	/*ADD Wide BW IE - if VAP VHT and BW is 80 */
	/*This should only be added if VAP is VHT*/
	if ((CsaManager_VapManagerIsVHT(pdPointer->txQVapId) == TRUE) && (CsaManager_Db[bandId].msg.chan_width == BANDWIDTH_EIGHTY))
	{
		csaFramePayload->wideBwIe.ie.u8Code = FM_EL_WIDE_BW_CH_SWITCH;
		csaFramePayload->wideBwIe.ie.u8Length = sizeof(sHT_WIDE_BW_CHANNEL_SWITCH);
		csaFramePayload->wideBwIe.wideBw.channelWidth = csaManager_GetChannelWidth(pMsg->chan_width);
		csaFramePayload->wideBwIe.wideBw.centerFreq0Idx = csaManager_GetCenterFreq0Idx(pMsg->low_chan_num, pMsg->chan_width, pMsg->primary_chan_idx);
		csaFramePayload->wideBwIe.wideBw.centerFreq1Idx = csaManager_GetCenterFreq1Idx(pMsg->low_chan_num, pMsg->chan_width);
		len += sizeof(WIDE_BW_IE);
	}
	pdPointer->dataLength = len;
}



/**********************************************************************************

csaManager_GetCenterFreq0Idx


Description:
------------


Input:
-----

chan_width, low_chan_num - as received from Driver


Output:
-------
center Frequency  

**********************************************************************************/
uint8 csaManager_GetCenterFreq0Idx(uint8 low_chan_num, uint8 chan_width, uint8 primary_chan_idx)
{
	uint8 centerFreq0Idx = 0;
	
	switch (chan_width)
	{
		case BANDWIDTH_TWENTY:
		case BANDWIDTH_FOURTY:
			centerFreq0Idx = 0;
			break;
		case BANDWIDTH_EIGHTY:
			centerFreq0Idx = low_chan_num + BW_80_OFFSET_TO_FREQ_INDEX;
			break;
		case BANDWIDTH_ONE_HUNDRED_SIXTY:
			//channel center frequency index of the higher 80 MHz channel segment that contains the primary channel
			if(primary_chan_idx >= BW_160_PRIMARY_CHANNEL_HIGH_START_INDEX)
			{
				centerFreq0Idx = low_chan_num + BW_80_HIGH_OFFSET_TO_FREQ_INDEX;
			}
			else //channel center frequency index of the lower 80 MHz channel segment that contains the primary channel
			{
				centerFreq0Idx = low_chan_num + BW_80_OFFSET_TO_FREQ_INDEX;
			}
			break;
		case BANDWIDTH_EIGHTY_EIGHTY:
			FATAL("csaManager_GetCenterFreq0Idx: 80+80 not supported");
			break;
		default:
			FATAL("csaManager_GetCenterFreq0Idx: Illegal channel width");
			break;
	}	
	
	return centerFreq0Idx;
}


/**********************************************************************************

csaManager_GetCenterFreq1Idx


Description:
------------


Input:
-----

chan_width, low_chan_num - as received from Driver


Output:
-------
center Frequency  

**********************************************************************************/
uint8 csaManager_GetCenterFreq1Idx(uint8 low_chan_num, uint8 chan_width)
{
	uint8 centerFreq1Idx = 0;
	
	switch (chan_width)
	{
		case BANDWIDTH_TWENTY:
		case BANDWIDTH_FOURTY:
		case BANDWIDTH_EIGHTY:
			centerFreq1Idx = 0;
			break;
		case BANDWIDTH_ONE_HUNDRED_SIXTY:
			centerFreq1Idx = low_chan_num + BW_160_OFFSET_TO_FREQ_INDEX;
			break;
		case BANDWIDTH_EIGHTY_EIGHTY:
			FATAL("csaManager_GetCenterFreq1Idx: 80+80 not supported");
			break;
		default:
			FATAL("csaManager_GetCenterFreq1Idx: Illegal channel width");
			break;
	}	
	
	return centerFreq1Idx;
}

/**********************************************************************************

csaManager_GetChannelWidth

Description:
------------


Input:
-----

chan_width - as received from Driver


Output:
-------
channel Width that complies with Channel Switch Announcement (see "Table 9-252VHT Operation Information subfields" in IEEE 802.11-2016.pdf)

**********************************************************************************/
uint8 csaManager_GetChannelWidth(uint8 chan_width)
{
	uint8 channelWidth = 0;

	switch (chan_width)
	{
		case BANDWIDTH_TWENTY:
		case BANDWIDTH_FOURTY:
			channelWidth = 0;
			break;
		case BANDWIDTH_EIGHTY:
			channelWidth = 1;
			break;
		case BANDWIDTH_ONE_HUNDRED_SIXTY:
			channelWidth = 1;
			break;
		case BANDWIDTH_EIGHTY_EIGHTY:
			FATAL("csaManager_GetChannelWidth: 80+80 not supported");
			break;
		default:
			FATAL("csaManager_GetChannelWidth: Illegal channel width");
			break;
	}

	return channelWidth;
}

    
/***********************************************************************
* csaManager_GetFrameLocation
* 
* Description:
* ------------
* Return pointer the MC DEAUTH frame prepared by Driver.
* 
* Input:
* ------
* Band and VAP IDs.
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* Pointer to frame.
* 
************************************************************************/
uint32 csaManager_GetFrameLocation(uint8 bandId, uint32 vapId)
{
	return (CsaManager_Db[bandId].msg.pUmSetChannelParams->csaMcDeauthFramesLocation[vapId]);
}

/**********************************************************************************
csaManagerStartTxUcDeauthTimerExpired

Description:
------------

Input: 
-----	
		
Output:
-------

Returns:
--------
	
**********************************************************************************/
void csaManagerStartTxUcDeauthTimerExpired(K_MSG* csaManagerMessage)
{
	uint32 bandId = CONFIGURATION_MANAGER_BAND_0;

	UNUSED_PARAM(csaManagerMessage);

	ILOG0_V("[CSA] csaManagerStartTxUcDeauthTimerExpired");
	
	/*Call handler*/
	CsaManager_TxStartUcDeauthTimerExpired[CsaManager_Db[bandId].state]((void *)bandId);
}


/**********************************************************************************
csaManagerStartTxMcDeauthTimerExpired

Description:
------------

Input: 
-----	
		
Output:
-------

Returns:
--------
	
**********************************************************************************/
void csaManagerStartTxMcDeauthTimerExpired(K_MSG* csaManagerMessage)
{
	uint32 bandId = CONFIGURATION_MANAGER_BAND_0;

	UNUSED_PARAM(csaManagerMessage);

	/*Call handler*/
	CsaManager_TxStartMcDeauthTimerExpired[CsaManager_Db[bandId].state]((void *)bandId);
}


/**********************************************************************************
csaManagerStartTxUcDeauthTimerExpired_B1

Description:
------------

Input: 
-----	
		
Output:
-------

Returns:
--------
	
**********************************************************************************/
void csaManagerStartTxUcDeauthTimerExpired_B1(K_MSG* csaManagerMessage)
{
	uint32 bandId = CONFIGURATION_MANAGER_BAND_1;

	UNUSED_PARAM(csaManagerMessage);

	/*Call handler*/
	CsaManager_TxStartUcDeauthTimerExpired[CsaManager_Db[bandId].state]((void *)bandId);
}


/**********************************************************************************
csaManagerStartTxMcDeauthTimerExpired_B1

Description:
------------

Input: 
-----	
		
Output:
-------

Returns:
--------
	
**********************************************************************************/
void csaManagerStartTxMcDeauthTimerExpired_B1(K_MSG* csaManagerMessage)
{
	uint32 bandId = CONFIGURATION_MANAGER_BAND_1;

	UNUSED_PARAM(csaManagerMessage);

	/*Call handler*/
	CsaManager_TxStartMcDeauthTimerExpired[CsaManager_Db[bandId].state]((void *)bandId);
}

/**********************************************************************************

csaManager_Init



Description:
------------
Initiailize CSA MAnager


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void csaManager_Init()
{
	uint8 i;
	uint8 bandId;

	for (bandId = CONFIGURATION_MANAGER_BAND_0; bandId <= CONFIGURATION_MANAGER_BAND_1; bandId++)
	{
		/*Clear Database*/
		memset(&(CsaManager_Db[bandId]), 0x0, sizeof(CsaManagerDb_t));
		CsaManager_Db[bandId].vapHead = CSA_MANAGER_INVALID_VAP;
		CsaManager_Db[bandId].confirmHead = CSA_MANAGER_INVALID_VAP;
		CsaManager_Db[bandId].state = CSA_MANAGER_IDLE_STATE;
		CsaManager_Db[bandId].currentTxSegment = CSA_TYPE_CSA;
		CsaManager_Db[bandId].txSegmentsEnabledVaps[CSA_TYPE_CSA] = 0xFFFFFFFF; //enable all vaps for csa segment
	}

	for (i = 0; i < HW_NUM_OF_VAPS; i++)
	{
		CsaManager_VapDb[i].nextVap = CSA_MANAGER_INVALID_VAP;
		CsaManager_VapDb[i].prevVap = CSA_MANAGER_INVALID_VAP;
	}

	/*Call CSA VAP Manager Init*/
	CsaManager_VapManagerInit();
	/*Call CSA STA Manager Init*/
	CsaManager_StaManagerInit();
}


