/***********************************************************************************
 File:			CtsManager.c
 Module:		CTS Manager
 Purpose: 		handle CTS Manager events
 Description:   
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
	//TBD to check which files can be removed
#include "BSSmanager_API.h"
#include "TxManager_Api.h"
#include "TxPacketsClassifier_API.h"
#include "TsManager_API.h"
#include "TxSelector_Api.h"
#include "HostInterface_API.h"
#include "Utils_Api.h"
#include "ResourceManager_API.h"
#include "Locker_Api.h"
#include "OSAL_Kmsg.h"
#include "ErrorHandler_Api.h"
#include "HwQManager_API.h"
#include "HwGlobalDefinitions.h"
#include "ShramPacketDescriptors.h"
#include "stringLibApi.h"
#include "um_interface.h"
#include "PacketDescriptor.h"
#include "HwMemoryMap.h"
#include "CtsManager_Api.h"
#include "loggerAPI.h"


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_CTS_MANAGER 
#define LOG_LOCAL_FID 0


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/
#define CTS_MANAGER_INVALID_VAP		0xFF
#define CTS_MANAGER_TIMEOUT_IN_MS	(100)

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
/*CTS Manager events*/
typedef enum
{
	CTS_MANAGER_START_TX_EV,
	CTS_MANAGER_TX_CFM_EV,
	CTS_MANAGER_PD_ALLOC_EV,
	CTS_MANAGER_STOP_TX_EV,
	CTS_MANAGER_LOCK_CFM_EV,
	CTS_MANAGER_NUM_EVENTS
} ctsManager_ev;
	
/*CTS Manager states*/
typedef enum
{
	CTS_MANAGER_IDLE_STATE,
	CTS_MANAGER_WAIT_TX_CFM_STATE,
	CTS_MANAGER_WAIT_PD_ALLOC_STATE,
	CTS_MANAGER_WAIT_LOCK_STATE,
	CTS_MANAGER_WAIT_LOCK_STOP_STATE,
	CTS_MANAGER_WAIT_PD_STOP_STATE,
	CTS_MANAGER_NUM_STATES
} ctsManager_state;

typedef struct CtsManagerVapDb_s
{
	uint8					nextVap;
	uint8					prevVap;
	bool					active;
} CtsManagerVapDb_t;

typedef struct CtsManagerDb_s
{
	K_MSG						*psMsg;
	TxPd_t						*pd;
	CtsManagerTxReqMessage_t    txReqMsg;
	uint32						dram_pointer;	/*Store thr DRAM pointer allocated to the PD so we can restore it*/
	ctsManager_state			state;			
	uint8						vapHead;
	uint8						requestId;
	bool						timerActive;
	uint8						status;
} CtsManagerDb_t;

typedef void (*CtsManager_Func)(void);

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void CtsManager_ChangeState(ctsManager_state state);
static void CtsManager_SendConfirm(void);
static void CtsManager_SendCts(void);
static void CtsManager_StartTxInIdle(void);
static void CtsManager_StopTxInIdle(void);
static void CtsManager_StopTxInWaitTxCfm(void);
static void CtsManager_StopTxInWaitPdAlloc(void);
static void CtsManager_TxCfmInWaitTxCfm(void);
static void CtsManager_TxCfmInWaitLock(void);
static void CtsManager_PdAllocInWaitPdAlloc(void);
static void CtsManager_PdAllocInWaitPdStop(void);
static void CtsManager_LockCfmInWaitLock(void);
static void CtsManager_LockCfmInWaitLockStop(void);
static void CtsManager_Fatal(void);

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
/*CTS Manager Start TX Event Handlers*/
static const CtsManager_Func CtsManager_StartTx[CTS_MANAGER_NUM_STATES] =
{
	 CtsManager_StartTxInIdle, 				 	/* CTS_MANAGER_IDLE_STATE */
	 CtsManager_Fatal,					  		/* CTS_MANAGER_WAIT_TX_CFM_STATE */
 	 CtsManager_Fatal,					  		/* CTS_MANAGER_WAIT_PD_ALLOC_STATE */
	 CtsManager_Fatal,							/* CTS_MANAGER_WAIT_LOCK_STATE */
	 CtsManager_Fatal,							/* CTS_MANAGER_WAIT_LOCK_STOP_STATE */
	 CtsManager_Fatal							/* CTS_MANAGER_WAIT_PD_STOP_STATE */
};

/*CTS Manager Stop TX Event Handlers*/
static const CtsManager_Func CtsManager_StopTx[CTS_MANAGER_NUM_STATES] =
{
	CtsManager_StopTxInIdle,				   /* CTS_MANAGER_IDLE_STATE */
	CtsManager_StopTxInWaitTxCfm,			   /* CTS_MANAGER_WAIT_TX_CFM_STATE */
	CtsManager_StopTxInWaitPdAlloc,			   /* CTS_MANAGER_WAIT_PD_ALLOC_STATE */
	CtsManager_Fatal,						   /* CTS_MANAGER_WAIT_LOCK_STATE */
	CtsManager_Fatal,						   /* CTS_MANAGER_WAIT_LOCK_STOP_STATE */
	CtsManager_Fatal						   /* CTS_MANAGER_WAIT_PD_STOP_STATE */
};

/*CTS Manager TX CFM Event Handler*/
static const CtsManager_Func CtsManager_TxCfm[CTS_MANAGER_NUM_STATES] =
{
	CtsManager_Fatal,				   		   /* CTS_MANAGER_IDLE_STATE */
	CtsManager_TxCfmInWaitTxCfm,			   /* CTS_MANAGER_WAIT_TX_CFM_STATE */
	CtsManager_Fatal,			   			   /* CTS_MANAGER_WAIT_PD_ALLOC_STATE */
	CtsManager_TxCfmInWaitLock,				   /* CTS_MANAGER_WAIT_LOCK_STATE */
	CtsManager_Fatal,						   /* CTS_MANAGER_WAIT_LOCK_STOP_STATE */
	CtsManager_Fatal						   /* CTS_MANAGER_WAIT_PD_STOP_STATE */
};

/*CTS Manager PD Alloc Event Handler*/
static const CtsManager_Func CtsManager_PdAlloc[CTS_MANAGER_NUM_STATES] =
{
	CtsManager_Fatal,				   		   /* CTS_MANAGER_IDLE_STATE */
	CtsManager_Fatal,			   			   /* CTS_MANAGER_WAIT_TX_CFM_STATE */
	CtsManager_PdAllocInWaitPdAlloc,		   /* CTS_MANAGER_WAIT_PD_ALLOC_STATE */
	CtsManager_Fatal,				   		   /* CTS_MANAGER_WAIT_LOCK_STATE */
	CtsManager_Fatal,						   /* CTS_MANAGER_WAIT_LOCK_STOP_STATE */
	CtsManager_PdAllocInWaitPdStop			   /* CTS_MANAGER_WAIT_PD_STOP_STATE */
};

/*CTS Manager Locvk CFM Event Handler*/
static const CtsManager_Func CtsManager_LockCfm[CTS_MANAGER_NUM_STATES] =
{
	CtsManager_Fatal,				   		   /* CTS_MANAGER_IDLE_STATE */
	CtsManager_Fatal,			  			   /* CTS_MANAGER_WAIT_TX_CFM_STATE */
	CtsManager_Fatal,			   			   /* CTS_MANAGER_WAIT_PD_ALLOC_STATE */
	CtsManager_LockCfmInWaitLock,			   /* CTS_MANAGER_WAIT_LOCK_STATE */
	CtsManager_LockCfmInWaitLockStop,		   /* CTS_MANAGER_WAIT_LOCK_STOP_STATE */
	CtsManager_Fatal						   /* CTS_MANAGER_WAIT_PD_STOP_STATE */
};


static CtsManagerVapDb_t CtsManager_VapDb[HW_NUM_OF_VAPS];
static CtsManagerDb_t CtsManager_Db;

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

CtsManager_ChangeState 


Description:
------------
Utility function to change State


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CtsManager_ChangeState(ctsManager_state state)
{
 	ILOG0_DD("CTS Manager changed from %d to %d", CtsManager_Db.state, state);
	CtsManager_Db.state = state;
}

/**********************************************************************************

CtsManager_SendConfirm 


Description:
------------
Utility function that sends confirmation back


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CtsManager_SendConfirm()
{
	K_MSG *psMsg = OSAL_GET_MESSAGE(sizeof(CtsManagerTxCfmMessage_t));
	CtsManagerTxCfmMessage_t *txCfmMsg = (CtsManagerTxCfmMessage_t *)pK_MSG_DATA(psMsg);
	
 	//("CTS Manager send confirm - status - %d", CtsManager_Db.status);
	txCfmMsg->status = CtsManager_Db.status;
	OSAL_SEND_MESSAGE(CtsManager_Db.txReqMsg.msgId, CtsManager_Db.txReqMsg.taskId, psMsg, CtsManager_Db.txReqMsg.vapIndex);
}

/**********************************************************************************

CtsManager_SendCts 


Description:
------------
Utility function that builds the CTS and enqueues it


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CtsManager_SendCts()
{
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
	TxPd_t *pdPointer = (TxPd_t *)CtsManager_Db.pd;
	BaaCounters_t *pBaaCounters = (BaaCounters_t *)statisticsGetBaaCountersAddress();

	/*Fill PD*/
	pdPointer->dataLength = sizeof(CTS_FRAME_HEADER);
	pdPointer->txQStaId = 0;
	pdPointer->txQTid = 0;
	pdPointer->mgmtSubtype = PD_CONTROL_SUBTYPE_CTS;
	pdPointer->pdType = PD_TYPE_CONTROL_UNENC;
	pdPointer->txQVapId = CtsManager_Db.vapHead;
	pdPointer->txQGroupId = HW_TX_Q_TYPE_GPHP;
	pdPointer->mcUnicast = UNICAST;
#ifdef ENET_INC_ARCH_WAVE600 
	pdPointer->ctrlDuration = CtsManager_Db.txReqMsg.duration*1000; /*Duration is given in msecs*/
#else
	pdPointer->cts2SelfDuration = CtsManager_Db.txReqMsg.duration*1000; /*Duration is given in msecs*/
#endif
	pdPointer->ctrlBw =  CtsManager_Db.txReqMsg.ctrl_bw;
	pdPointer->retransmissionIndication = 0;
	pdPointer->aggregationIndication = 0;
	/*Put CTS on GPHP queue*/
	memset(&hwQueueManagerRequestParams,0,sizeof(HwQueueManagerRequestParams_t));
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
	hwQueueManagerRequestParams.secondaryAddr = 0;
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	hwQueueManagerRequestParams.primaryAddr = CtsManager_Db.vapHead;
	hwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_GPHP;  
	hwQueueManagerRequestParams.pHeadDesc = pdPointer;
	HwQManager_PushPacketToTail(&hwQueueManagerRequestParams);	
	/*Change State to wait for TX CFM*/
	CtsManager_ChangeState(CTS_MANAGER_WAIT_TX_CFM_STATE); 
	/*Update statistics*/
	pBaaCounters->fwctrlFramesSent++;
}

/**********************************************************************************

CtsManager_Done 


Description:
------------
Utility function that handles the actions needed when we are done


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CtsManager_Done()
{
	/*Restore DRAM pointer in PD*/
	CtsManager_Db.pd->packetPointer = CtsManager_Db.dram_pointer;
	/*Return PD*/
	ResourceManager_ReleaseDescriptor(CtsManager_Db.pd, DESC_POOL_CSA_MANAGER);
	/*Send Confirmation to caller*/
	CtsManager_SendConfirm();
	/*Set state to Idle*/
	CtsManager_ChangeState(CTS_MANAGER_IDLE_STATE); 
	/*Stop Timer*/
	if (CtsManager_Db.timerActive == TRUE)
	{
		CtsManager_Db.timerActive = FALSE;
		OSAL_RESET_TIMER_EXPLICIT(CTS_MANAGER_STOP_TX, TASK_TX_MANAGER);
	}
}

/**********************************************************************************

CtsManager_StartTxInIdle 


Description:
------------
Start TX In Idle State handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CtsManager_StartTxInIdle()
{
	uint8 allocationStatus = FALSE;
	RmPdRequestFillParameters_t pdRequestFillParameters;
	CtsManagerTxReqMessage_t *txReqMsg = (CtsManagerTxReqMessage_t *)pK_MSG_DATA(CtsManager_Db.psMsg);

	memset(&pdRequestFillParameters, 0, sizeof(RmPdRequestFillParameters_t));	
	pdRequestFillParameters.returnMsgType = CTS_MANAGER_PD_ALLOC;
	pdRequestFillParameters.returnTaskId = TASK_TX_MANAGER;
	allocationStatus = ResourceManager_GetDescriptorRequest(DESC_POOL_CSA_MANAGER, &pdRequestFillParameters);
	/*Store parameters*/
	memcpy(&CtsManager_Db.txReqMsg, txReqMsg, sizeof(CtsManagerTxReqMessage_t));
	//("CtsManager_StartTxInIdle allocation status %d ", allocationStatus);
	if(allocationStatus)
	{
		/*Store PD pointer*/
		CtsManager_Db.pd = pdRequestFillParameters.packetDescriptor;
		/*Store DRAM pointer - field is used for duration*/
		CtsManager_Db.dram_pointer = CtsManager_Db.pd->packetPointer;
		/*Build and send CTS*/
		CtsManager_SendCts();
	}
	else
	{
		/* PD allocation failed */		
		CtsManager_Db.requestId = pdRequestFillParameters.requestId;
		/*Change State to wait for PD*/
		CtsManager_ChangeState(CTS_MANAGER_WAIT_PD_ALLOC_STATE); 
	}
	/*Set Status to success*/
	CtsManager_Db.status = TRUE;
	/*Start timer*/	
	CtsManager_Db.timerActive = TRUE;
	OSAL_SET_TIMER_EXPLICIT(CTS_MANAGER_STOP_TX, OSAL_TIMERS_MS_TO_K_TICKS(CtsManager_Db.txReqMsg.timeout), TASK_TX_MANAGER);
}

/**********************************************************************************

CtsManager_StopTxInIdle 


Description:
------------
Stop TX In Idle State handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CtsManager_StopTxInIdle()
{
	/*it is possible that timer expires and message is sent but just before it is the TX CFM - just do nothing*/
}

/**********************************************************************************

CtsManager_StopTxInWaitTxCfm 


Description:
------------
Stop TX In Wait TX CFM State handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CtsManager_StopTxInWaitTxCfm()
{
	RequesterLockParams_t lockRequesterParams;
	
	/*PD may be in GPHP on the way back*/
	/*Send Lock Request to GPHP*/
	memset(&lockRequesterParams, 0, sizeof(RequesterLockParams_t));
	/*Request SW Locker to Wait for Lock Stop*/
	lockRequesterParams.callerContext = NULL;
	lockRequesterParams.returnMsg = CTS_MANAGER_GPHP_LOCK;
	lockRequesterParams.returnTask = TASK_TX_MANAGER;
	Locker_LockSingleQueue(HW_TX_Q_TYPE_GPHP, CtsManager_Db.vapHead, 0, &lockRequesterParams);
	/*Set state to Wait Lock*/
	CtsManager_ChangeState(CTS_MANAGER_WAIT_LOCK_STATE); 
}

/**********************************************************************************

CtsManager_StopTxInWaitPdAlloc 


Description:
------------
Stop TX Wait PD Alloc State handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CtsManager_StopTxInWaitPdAlloc()
{
	uint8 removeStatus = FALSE;
	
	/*Set Status to failure*/
	CtsManager_Db.status = FALSE;
	/*Tell Resource Manager we don't need the resource anymore*/
	
	removeStatus = ResourceManager_RemoveRequest(CtsManager_Db.requestId, DESC_POOL_CSA_MANAGER);
	
	if (removeStatus) 
	{
		/*Request remove succesfully*/
		/*Send Confirmation to caller*/
		CtsManager_SendConfirm();
		/*Set state to Idle*/
		CtsManager_ChangeState(CTS_MANAGER_IDLE_STATE); 
	}
	else
	{
		/*Request is on the way - wait for it - change state to wait PD Stop State*/
		CtsManager_ChangeState(CTS_MANAGER_WAIT_PD_STOP_STATE); 
	}	
}

/**********************************************************************************

CtsManager_TxCfmInWaitTxCfm 


Description:
------------
TX CFM In Wait TX CFM State handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CtsManager_TxCfmInWaitTxCfm()
{
	/*We are done*/
	CtsManager_Done();
}

/**********************************************************************************

CtsManager_TxCfmInWaitLock 


Description:
------------
TX CFM In Wait Lock State handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CtsManager_TxCfmInWaitLock()
{
	/*Set state to Wait Lock Stop - the PD will be released when Lock CFM is processed*/
	CtsManager_ChangeState(CTS_MANAGER_WAIT_LOCK_STOP_STATE); 
}

/**********************************************************************************

CtsManager_PdAllocInWaitPdAlloc 


Description:
------------
PD Alloc In Wait PD Alloc State handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CtsManager_PdAllocInWaitPdAlloc()
{
	/*Send CTS*/
	CtsManager_SendCts();
}

/**********************************************************************************

CtsManager_PdAllocInWaitPdStop 


Description:
------------
PD Alloc In Wait PD Stop State handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CtsManager_PdAllocInWaitPdStop()
{
	/*We are done*/
	CtsManager_Done();
}

/**********************************************************************************

CtsManager_LockCfmInWaitLock 


Description:
------------
Lock CFM In Wait Lock State handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CtsManager_LockCfmInWaitLock()
{
	/*See if PD is on GPHP - it should be the only thing on it!*/
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
	
	/*Peek head packet*/
	memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
	hwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_GPHP;
	hwQueueManagerRequestParams.primaryAddr = CtsManager_Db.vapHead;
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	HwQManager_PeekHeadPacket(&hwQueueManagerRequestParams);
  	if (hwQueueManagerRequestParams.pHeadDesc != NULL_PD)
    {
		HwQManager_PopPacket(&hwQueueManagerRequestParams);
		/*If PD is here it means it was never transmitted - Set Status to failure*/
		CtsManager_Db.status = FALSE;
		/*We are done*/
		CtsManager_Done();
    }
	else
	{
		/*packet is not on GPHP queue - must be on the way back*/
		/*Change State back to Wait for TX CFM*/
		CtsManager_ChangeState(CTS_MANAGER_WAIT_TX_CFM_STATE); 
	}
	/*Unlock GPHP*/	
	Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_GPHP, CtsManager_Db.vapHead, (0x1 << 0));
}

/**********************************************************************************

CtsManager_LockCfmInWaitLockStop 


Description:
------------
Lock CFM In Wait Lock Stop handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CtsManager_LockCfmInWaitLockStop()
{
	/*We are done*/
	CtsManager_Done();
	/*Unlock GPHP*/	
	Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_GPHP, CtsManager_Db.vapHead, (0x1 << 0));
}
 
/**********************************************************************************

CtsManager_Fatal 


Description:
------------
handles unexpected event


Input: 
-----	
 	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CtsManager_Fatal()
{
	FATAL("CTS Manager Fatal");
}

/*---------------------------------------------------------------------------------
/						global Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

ctsManagerVapOpen











Description:
------------
VAP Open messge handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void ctsManagerVapOpen(K_MSG* ctsManagerMessage)
{
	BSS_MANAGER_CONFIRM_EVENT* confirmEvent; 
	uint8 vapId; 
	K_MSG *vapMsg = (K_MSG *) (*((uint32 *)pK_MSG_DATA(ctsManagerMessage)));
	UMI_SET_WMM_PARAMETERS* pPostVapActivationMsg = (UMI_SET_WMM_PARAMETERS*) pK_MSG_DATA(vapMsg);

	vapId = pPostVapActivationMsg->vapId; 
	ASSERT(CtsManager_VapDb[vapId].active == FALSE);
	CtsManager_VapDb[vapId].active = TRUE;
	/*CTS are transmitted inside a process and a VAP can't be added or removed while another process is running
	so we don't need to handle this in CTS states - just add VAP to list*/
	if (CtsManager_Db.vapHead != CTS_MANAGER_INVALID_VAP)
	{
		CtsManager_VapDb[CtsManager_Db.vapHead].prevVap = vapId;
	}
	CtsManager_VapDb[vapId].prevVap = CTS_MANAGER_INVALID_VAP;
	CtsManager_VapDb[vapId].nextVap = CtsManager_Db.vapHead;
	CtsManager_Db.vapHead = vapId;
	/*Send Confirmation to VAP manager*/
	confirmEvent = (BSS_MANAGER_CONFIRM_EVENT*)(&(ctsManagerMessage->abData));
	confirmEvent->vapId= vapId;
	confirmEvent->eventIndex = VAP_MANAGER_POST_VAP_ACTIVATION; 
	confirmEvent->clientId = BSS_MANAGER_VAP_MANAGER_CTS_MANAGER_CLIENT; 
	OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, ctsManagerMessage, vapId);
	
}

/**********************************************************************************

ctsManagerVapRemove










Description:
------------
VAP remove messge handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void ctsManagerVapRemove(K_MSG* ctsManagerMessage)
{
	uint8 vapId; 
	BSS_MANAGER_CONFIRM_EVENT* confirmEvent; 
	K_MSG *vapMsg = (K_MSG *) (*((uint32 *)pK_MSG_DATA(ctsManagerMessage)));
	UMI_REMOVE_VAP * removeVapPtr = (UMI_REMOVE_VAP*) pK_MSG_DATA(vapMsg);

	vapId = removeVapPtr->vapId; 
	/*From CTS manager perspective a VAP is added only when WMM params are set - a VAP may be removed without rewaching that state*/
	if (CtsManager_VapDb[vapId].active == TRUE)
	{
		/*CTS are transmitted inside a process and a VAP can't be added or removed while another process is running
		so we don't need to handle this in CTS states - just remove VAP from list*/
		if (CtsManager_VapDb[vapId].prevVap == CTS_MANAGER_INVALID_VAP)
		{
			CtsManager_Db.vapHead = CtsManager_VapDb[vapId].nextVap;
		}
		else
		{
			CtsManager_VapDb[CtsManager_VapDb[vapId].prevVap].nextVap = CtsManager_VapDb[vapId].nextVap;
		}
		
		if (CtsManager_VapDb[vapId].nextVap != CTS_MANAGER_INVALID_VAP)//Marcelo
		{
			CtsManager_VapDb[CtsManager_VapDb[vapId].nextVap].prevVap = CtsManager_VapDb[vapId].prevVap;
		}
		CtsManager_VapDb[vapId].active = FALSE;
	}
	/*Send Confirmation to VAP manager*/
	confirmEvent = (BSS_MANAGER_CONFIRM_EVENT*)(&(ctsManagerMessage->abData));
	confirmEvent->vapId = vapId;
	confirmEvent->eventIndex = VAP_MANAGER_REMOVE_VAP; 
	confirmEvent->clientId = BSS_MANAGER_VAP_MANAGER_CTS_MANAGER_CLIENT; 
	OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, ctsManagerMessage, vapId);
}

/**********************************************************************************

ctsManagerStartTx









Description:
------------
CTS Start TX messge handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void ctsManagerStartTx(K_MSG* ctsManagerMessage)
{
	/*Store the message*/
	CtsManager_Db.psMsg = ctsManagerMessage;
	/*Call handler*/
	CtsManager_StartTx[CtsManager_Db.state]();
}

/**********************************************************************************

ctsManagerStopTx








Description:
------------
CTS STOP TX messge handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void ctsManagerStopTx(K_MSG* ctsManagerMessage)
{
	/*Mark timer inactive*/
	CtsManager_Db.timerActive = FALSE;
	/*Call handler*/
	CtsManager_StopTx[CtsManager_Db.state]();
}

/**********************************************************************************

ctsManagerTxCfm







Description:
------------
TX CFM messge handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void ctsManagerTxCfm(K_MSG* ctsManagerMessage)
{
	/*Call handler*/
	CtsManager_TxCfm[CtsManager_Db.state]();
}

/**********************************************************************************

ctsManagerPdAlloc






Description:
------------
PD Alloc messge handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void ctsManagerPdAlloc(K_MSG* ctsManagerMessage)
{
 	RmPdFreeDescResponse_t *pdMsg = (RmPdFreeDescResponse_t *)pK_MSG_DATA(ctsManagerMessage);

	/*Store PD pointer*/
	CtsManager_Db.pd = pdMsg->packetDescriptor;
	/*Store DRAM pointer - field is used for duration*/
	CtsManager_Db.dram_pointer = CtsManager_Db.pd->packetPointer;
	/*Release the request*/
	ResourceManager_ReleaseRequest(CtsManager_Db.requestId, DESC_POOL_CSA_MANAGER);
	/*Call handler*/
	CtsManager_PdAlloc[CtsManager_Db.state]();
}

/**********************************************************************************

ctsManagerGphpLock





Description:
------------
Lock CFM Message Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void ctsManagerGphpLock(K_MSG* ctsManagerMessage)
{
	/*Call handler*/
	CtsManager_LockCfm[CtsManager_Db.state]();
}

#if defined (ENET_INC_UMAC)
#pragma ghs section text=".initialization" 
#endif


/**********************************************************************************

ctsManager_Init




Description:
------------
Initiailize CTS MAnager


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void ctsManager_Init()
{
	uint8 i;
	
	/*Clear Database*/
	CtsManager_Db.vapHead = CTS_MANAGER_INVALID_VAP;
	CtsManager_Db.state = CTS_MANAGER_IDLE_STATE;
	CtsManager_Db.pd = (TxPd_t *)NULL_PD;
	CtsManager_Db.txReqMsg.timeout = CTS_MANAGER_TIMEOUT_IN_MS;
	CtsManager_Db.timerActive = FALSE;
	CtsManager_Db.txReqMsg.ctrl_bw = PD_CONTROL_20_MHZ;
	CtsManager_Db.txReqMsg.duration = CTS2SELF_DURATION_MSEC_NORMAL_MODE;
	for (i = 0; i < HW_NUM_OF_VAPS; i++)
	{
		CtsManager_VapDb[i].nextVap = CTS_MANAGER_INVALID_VAP;
		CtsManager_VapDb[i].prevVap = CTS_MANAGER_INVALID_VAP;
		CtsManager_VapDb[i].active = FALSE;
	}
}
#if defined (ENET_INC_UMAC)
#pragma ghs section text=default
#endif



