/***********************************************************************************
 File:			CtsManager.c
 Module:		CTS Manager
 Purpose: 		handle CTS Manager events
 Description:   
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
	//TBD to check which files can be removed
#include "BSSmanager_API.h"
#include "TxManager_Api.h"
#include "TxPacketsClassifier_API.h"
#include "TsManager_API.h"
#include "TxSelector_Api.h"
#include "HostInterface_API.h"
#include "Utils_Api.h"
#include "ResourceManager_API.h"
#include "Locker_Api.h"
#include "OSAL_Kmsg.h"
#include "ErrorHandler_Api.h"
#include "HwQManager_API.h"
#include "HwGlobalDefinitions.h"
#include "ShramPacketDescriptors.h"
#include "stringLibApi.h"
#include "um_interface.h"
#include "PacketDescriptor.h"
#include "HwMemoryMap.h"
#include "CtsManager_Api.h"
#include "ConfigurationManager_api.h"
#include "OSAL_UpperMacMessages.h"
#include "loggerAPI.h"
#include "Statistics_Descriptors.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_CTS_MANAGER 
#define LOG_LOCAL_FID 1


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/
#define CTS_MANAGER_INVALID_VAP		0xFF
#define CTS_MANAGER_TIMEOUT_IN_MS	(100)

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
/*CTS Manager events*/
typedef enum
{
	CTS_MANAGER_START_TX_EV,
	CTS_MANAGER_TX_CFM_EV,
	CTS_MANAGER_PD_ALLOC_EV,
	CTS_MANAGER_STOP_TX_EV,
	CTS_MANAGER_LOCK_CFM_EV,
	CTS_MANAGER_NUM_EVENTS
} ctsManager_ev;
	
/*CTS Manager states*/
typedef enum
{
	CTS_MANAGER_IDLE_STATE,
	CTS_MANAGER_WAIT_TX_CFM_STATE,
	CTS_MANAGER_WAIT_PD_ALLOC_STATE,
	CTS_MANAGER_WAIT_LOCK_STATE,
	CTS_MANAGER_WAIT_LOCK_STOP_STATE,
	CTS_MANAGER_WAIT_PD_STOP_STATE,
	CTS_MANAGER_NUM_STATES
} ctsManager_state;

typedef struct CtsManagerVapDb_s
{
	uint8					nextVap;
	uint8					prevVap;
	bool					active;
} CtsManagerVapDb_t;

typedef struct CtsManagerBandDb_
{
	K_MSG						*psMsg;
	TxPd_t						*pd;
	CtsManagerTxReqMessage_t    txReqMsg;
	uint32						dram_pointer;	/*Store thr DRAM pointer allocated to the PD so we can restore it*/
	uint16						timerMsgId;
	ctsManager_state			state;			
	uint8						vapHead;
	uint8						requestId;
	bool						timerActive;
	uint8						status;
} CtsManagerBandDb_t;

typedef void (*CtsManager_Func)(uint8 bandId);

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void CtsManager_ChangeState(ctsManager_state state, uint8 bandId);
static void CtsManager_SendConfirm(uint8 bandId);
static void CtsManager_SendCts(uint8 bandId);
static void CtsManager_Done(uint8 bandId);
static void CtsManager_StartTxInIdle(uint8 bandId);
static void CtsManager_StopTxInIdle(uint8 bandId);
static void CtsManager_StopTxInWaitTxCfm(uint8 bandId);
static void CtsManager_StopTxInWaitPdAlloc(uint8 bandId);
static void CtsManager_TxCfmInWaitTxCfm(uint8 bandId);
static void CtsManager_TxCfmInWaitLock(uint8 bandId);
static void CtsManager_PdAllocInWaitPdAlloc(uint8 bandId);
static void CtsManager_PdAllocInWaitPdStop(uint8 bandId);
static void CtsManager_LockCfmInWaitLock(uint8 bandId);
static void CtsManager_LockCfmInWaitLockStop(uint8 bandId);
static void CtsManager_Fatal(uint8 bandId);

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
/*CTS Manager Start TX Event Handlers*/
static const CtsManager_Func CtsManager_StartTx[CTS_MANAGER_NUM_STATES] =
{
	 CtsManager_StartTxInIdle, 				 	/* CTS_MANAGER_IDLE_STATE */
	 CtsManager_Fatal,					  		/* CTS_MANAGER_WAIT_TX_CFM_STATE */
 	 CtsManager_Fatal,					  		/* CTS_MANAGER_WAIT_PD_ALLOC_STATE */
	 CtsManager_Fatal,							/* CTS_MANAGER_WAIT_LOCK_STATE */
	 CtsManager_Fatal,							/* CTS_MANAGER_WAIT_LOCK_STOP_STATE */
	 CtsManager_Fatal							/* CTS_MANAGER_WAIT_PD_STOP_STATE */
};

/*CTS Manager Stop TX Event Handlers*/
static const CtsManager_Func CtsManager_StopTx[CTS_MANAGER_NUM_STATES] =
{
	CtsManager_StopTxInIdle,				   /* CTS_MANAGER_IDLE_STATE */
	CtsManager_StopTxInWaitTxCfm,			   /* CTS_MANAGER_WAIT_TX_CFM_STATE */
	CtsManager_StopTxInWaitPdAlloc,			   /* CTS_MANAGER_WAIT_PD_ALLOC_STATE */
	CtsManager_Fatal,						   /* CTS_MANAGER_WAIT_LOCK_STATE */
	CtsManager_Fatal,						   /* CTS_MANAGER_WAIT_LOCK_STOP_STATE */
	CtsManager_Fatal						   /* CTS_MANAGER_WAIT_PD_STOP_STATE */
};

/*CTS Manager TX CFM Event Handler*/
static const CtsManager_Func CtsManager_TxCfm[CTS_MANAGER_NUM_STATES] =
{
	CtsManager_Fatal,				   		   /* CTS_MANAGER_IDLE_STATE */
	CtsManager_TxCfmInWaitTxCfm,			   /* CTS_MANAGER_WAIT_TX_CFM_STATE */
	CtsManager_Fatal,			   			   /* CTS_MANAGER_WAIT_PD_ALLOC_STATE */
	CtsManager_TxCfmInWaitLock,				   /* CTS_MANAGER_WAIT_LOCK_STATE */
	CtsManager_Fatal,						   /* CTS_MANAGER_WAIT_LOCK_STOP_STATE */
	CtsManager_Fatal						   /* CTS_MANAGER_WAIT_PD_STOP_STATE */
};

/*CTS Manager PD Alloc Event Handler*/
static const CtsManager_Func CtsManager_PdAlloc[CTS_MANAGER_NUM_STATES] =
{
	CtsManager_Fatal,				   		   /* CTS_MANAGER_IDLE_STATE */
	CtsManager_Fatal,			   			   /* CTS_MANAGER_WAIT_TX_CFM_STATE */
	CtsManager_PdAllocInWaitPdAlloc,		   /* CTS_MANAGER_WAIT_PD_ALLOC_STATE */
	CtsManager_Fatal,				   		   /* CTS_MANAGER_WAIT_LOCK_STATE */
	CtsManager_Fatal,						   /* CTS_MANAGER_WAIT_LOCK_STOP_STATE */
	CtsManager_PdAllocInWaitPdStop			   /* CTS_MANAGER_WAIT_PD_STOP_STATE */
};

/*CTS Manager Locvk CFM Event Handler*/
static const CtsManager_Func CtsManager_LockCfm[CTS_MANAGER_NUM_STATES] =
{
	CtsManager_Fatal,				   		   /* CTS_MANAGER_IDLE_STATE */
	CtsManager_Fatal,			  			   /* CTS_MANAGER_WAIT_TX_CFM_STATE */
	CtsManager_Fatal,			   			   /* CTS_MANAGER_WAIT_PD_ALLOC_STATE */
	CtsManager_LockCfmInWaitLock,			   /* CTS_MANAGER_WAIT_LOCK_STATE */
	CtsManager_LockCfmInWaitLockStop,		   /* CTS_MANAGER_WAIT_LOCK_STOP_STATE */
	CtsManager_Fatal						   /* CTS_MANAGER_WAIT_PD_STOP_STATE */
};


static CtsManagerVapDb_t CtsManager_VapDb[HW_NUM_OF_VAPS];
static CtsManagerBandDb_t CtsManagerBand_Db[NUM_OF_CONFIGURATION_MANAGER_BANDS];


/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

CtsManager_ChangeState 


Description:
------------
Utility function to change State


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CtsManager_ChangeState(ctsManager_state state, uint8 bandId)
{
 	ILOG0_DDD("CTS Manager changed from %d to %d bandId %d", CtsManagerBand_Db[bandId].state, state, bandId);
	CtsManagerBand_Db[bandId].state = state;
}

/**********************************************************************************

CtsManager_SendConfirm 


Description:
------------
Utility function that sends confirmation back


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CtsManager_SendConfirm(uint8 bandId)
{
	K_MSG *psMsg = OSAL_GET_MESSAGE(sizeof(CtsManagerTxCfmMessage_t));
	CtsManagerTxCfmMessage_t *txCfmMsg = (CtsManagerTxCfmMessage_t *)pK_MSG_DATA(psMsg);
	
 	//("CTS Manager send confirm - status - %d", CtsManager_Db.status);
	txCfmMsg->status = CtsManagerBand_Db[bandId].status;
	OSAL_SEND_MESSAGE(CtsManagerBand_Db[bandId].txReqMsg.msgId, CtsManagerBand_Db[bandId].txReqMsg.taskId, psMsg, CtsManagerBand_Db[bandId].txReqMsg.vapIndex);
}

/**********************************************************************************

CtsManager_SendCts 


Description:
------------
Utility function that builds the CTS and enqueues it


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CtsManager_SendCts(uint8 bandId)
{
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
	TxPd_t *pdPointer = (TxPd_t *)CtsManagerBand_Db[bandId].pd;	
	BaaCounters_t *pBaaCounters = (BaaCounters_t *)statisticsGetBaaCountersAddress();

	ASSERT(CtsManagerBand_Db[bandId].vapHead != CTS_MANAGER_INVALID_VAP);
	
	/*Fill PD*/
	pdPointer->dataLength = sizeof(CTS_FRAME_HEADER);
	pdPointer->txQStaId = 0;
	pdPointer->txQTid = MANAGEMENT_TID;
	pdPointer->ctrlFrameSubtype = CTRL_FRAME_SUBTYPE_CTS_2_SELF;

	pdPointer->pdType = PD_TYPE_CONTROL_UNENC;
	pdPointer->txQVapId = CtsManagerBand_Db[bandId].vapHead;
	pdPointer->txQGroupId = HW_TX_Q_TYPE_GPHP;
	pdPointer->mcUnicast = MULTICAST;
#ifdef ENET_INC_ARCH_WAVE600 
	pdPointer->ctrlDuration = CtsManagerBand_Db[bandId].txReqMsg.duration*1000; /*Duration is given in msecs*/
#else
	pdPointer->cts2SelfDuration = CtsManagerBand_Db[bandId].txReqMsg.duration*1000; /*Duration is given in msecs*/
#endif
	pdPointer->ctrlBw =  CtsManagerBand_Db[bandId].txReqMsg.ctrl_bw;
	pdPointer->retransmissionIndication = 0;
	pdPointer->aggregationIndication = 0;
	/*Put CTS on GPHP queue*/
	memset(&hwQueueManagerRequestParams,0,sizeof(HwQueueManagerRequestParams_t));
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
	hwQueueManagerRequestParams.secondaryAddr = 0;
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	hwQueueManagerRequestParams.primaryAddr = CtsManagerBand_Db[bandId].vapHead;
	hwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_GPHP;  
	hwQueueManagerRequestParams.pHeadDesc = pdPointer;
	HwQManager_PushPacketToTail(&hwQueueManagerRequestParams);	
	/*Change State to wait for TX CFM*/
	CtsManager_ChangeState(CTS_MANAGER_WAIT_TX_CFM_STATE, bandId); 
	/*Update statistics*/
	pBaaCounters->fwctrlFramesSent[bandId]++;
}

/**********************************************************************************

CtsManager_Done 


Description:
------------
Utility function that handles the actions needed when we are done


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CtsManager_Done(uint8 bandId)
{
	/*Restore DRAM pointer in PD*/
	CtsManagerBand_Db[bandId].pd->packetPointer = CtsManagerBand_Db[bandId].dram_pointer;
	/*Return PD*/
	ResourceManager_ReleaseDescriptor(CtsManagerBand_Db[bandId].pd, DESC_POOL_CSA_MANAGER);
	/*Send Confirmation to caller*/
	CtsManager_SendConfirm(bandId);
	/*Set state to Idle*/
	CtsManager_ChangeState(CTS_MANAGER_IDLE_STATE, bandId); 
	/*Stop Timer*/
	if (CtsManagerBand_Db[bandId].timerActive == TRUE)
	{
		CtsManagerBand_Db[bandId].timerActive = FALSE;
		OSAL_RESET_TIMER_EXPLICIT(CtsManagerBand_Db[bandId].timerMsgId, TASK_TX_MANAGER);
	}
}

/**********************************************************************************

CtsManager_StartTxInIdle 


Description:
------------
Start TX In Idle State handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CtsManager_StartTxInIdle(uint8 bandId)
{
	uint8 allocationStatus = FALSE;
	RmPdRequestFillParameters_t pdRequestFillParameters;
	CtsManagerTxReqMessage_t *txReqMsg = (CtsManagerTxReqMessage_t *)pK_MSG_DATA(CtsManagerBand_Db[bandId].psMsg);
 

	memset(&pdRequestFillParameters, 0, sizeof(RmPdRequestFillParameters_t));	
	pdRequestFillParameters.returnMsgType = CTS_MANAGER_PD_ALLOC;
	pdRequestFillParameters.returnTaskId = TASK_TX_MANAGER;
	pdRequestFillParameters.context = bandId;
	
	allocationStatus = ResourceManager_GetDescriptorRequest(DESC_POOL_CSA_MANAGER, &pdRequestFillParameters);
	/*Store parameters*/
	MEMCPY(&CtsManagerBand_Db[bandId].txReqMsg, txReqMsg, sizeof(CtsManagerTxReqMessage_t));
	//("CtsManager_StartTxInIdle allocation status %d ", allocationStatus);
	if(allocationStatus)
	{
		/*Store PD pointer*/
		CtsManagerBand_Db[bandId].pd = pdRequestFillParameters.packetDescriptor;
		/*Store DRAM pointer - field is used for duration*/
		CtsManagerBand_Db[bandId].dram_pointer = CtsManagerBand_Db[bandId].pd->packetPointer;
		/*Build and send CTS*/
		CtsManager_SendCts(bandId);
	}
	else
	{
		/* PD allocation failed */		
		CtsManagerBand_Db[bandId].requestId = pdRequestFillParameters.requestId;
		/*Change State to wait for PD*/
		CtsManager_ChangeState(CTS_MANAGER_WAIT_PD_ALLOC_STATE, bandId); 
	}
	/*Set Status to success*/
	CtsManagerBand_Db[bandId].status = TRUE;
	/*Start timer*/	
	CtsManagerBand_Db[bandId].timerActive = TRUE;
	OSAL_SET_TIMER_EXPLICIT(CtsManagerBand_Db[bandId].timerMsgId, OSAL_TIMERS_MS_TO_K_TICKS(CtsManagerBand_Db[bandId].txReqMsg.timeout), TASK_TX_MANAGER);
}

/**********************************************************************************

CtsManager_StopTxInIdle 


Description:
------------
Stop TX In Idle State handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CtsManager_StopTxInIdle(uint8 bandId)
{
	/*it is possible that timer expires and message is sent but just before it is the TX CFM - just do nothing*/
	UNUSED_PARAM(bandId);	
}

/**********************************************************************************

CtsManager_StopTxInWaitTxCfm 


Description:
------------
Stop TX In Wait TX CFM State handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CtsManager_StopTxInWaitTxCfm(uint8 bandId)
{
	RequesterLockParams_t lockRequesterParams;
	
	/*PD may be in GPHP on the way back*/
	/*Send Lock Request to GPHP*/
	memset(&lockRequesterParams, 0, sizeof(RequesterLockParams_t));
	/*Request SW Locker to Wait for Lock Stop*/
	lockRequesterParams.callerContext = (void*)((uint32)bandId);
	lockRequesterParams.returnMsg = CTS_MANAGER_GPHP_LOCK;
	lockRequesterParams.returnTask = TASK_TX_MANAGER;
	Locker_LockSingleQueue(HW_TX_Q_TYPE_GPHP, CtsManagerBand_Db[bandId].vapHead, 0, &lockRequesterParams);
	/*Set state to Wait Lock*/
	CtsManager_ChangeState(CTS_MANAGER_WAIT_LOCK_STATE, bandId); 
}

/**********************************************************************************

CtsManager_StopTxInWaitPdAlloc 


Description:
------------
Stop TX Wait PD Alloc State handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CtsManager_StopTxInWaitPdAlloc(uint8 bandId)
{
	uint8 removeStatus = FALSE;
	
	/*Set Status to failure*/
	CtsManagerBand_Db[bandId].status = FALSE;
	/*Tell Resource Manager we don't need the resource anymore*/
	
	removeStatus = ResourceManager_RemoveRequest(CtsManagerBand_Db[bandId].requestId, DESC_POOL_CSA_MANAGER);
	
	if (removeStatus) 
	{
		/*Request remove succesfully*/
		/*Send Confirmation to caller*/
		CtsManager_SendConfirm(bandId);
		/*Set state to Idle*/
		CtsManager_ChangeState(CTS_MANAGER_IDLE_STATE, bandId); 
	}
	else
	{
		/*Request is on the way - wait for it - change state to wait PD Stop State*/
		CtsManager_ChangeState(CTS_MANAGER_WAIT_PD_STOP_STATE, bandId); 
	}	
}

/**********************************************************************************

CtsManager_TxCfmInWaitTxCfm 


Description:
------------
TX CFM In Wait TX CFM State handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CtsManager_TxCfmInWaitTxCfm(uint8 bandId)
{
	/*We are done*/
	CtsManager_Done(bandId);
}

/**********************************************************************************

CtsManager_TxCfmInWaitLock 


Description:
------------
TX CFM In Wait Lock State handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CtsManager_TxCfmInWaitLock(uint8 bandId)
{
	/*Set state to Wait Lock Stop - the PD will be released when Lock CFM is processed*/
	CtsManager_ChangeState(CTS_MANAGER_WAIT_LOCK_STOP_STATE, bandId); 
}

/**********************************************************************************

CtsManager_PdAllocInWaitPdAlloc 


Description:
------------
PD Alloc In Wait PD Alloc State handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CtsManager_PdAllocInWaitPdAlloc(uint8 bandId)
{
	/*Send CTS*/
	CtsManager_SendCts(bandId);
}

/**********************************************************************************

CtsManager_PdAllocInWaitPdStop 


Description:
------------
PD Alloc In Wait PD Stop State handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CtsManager_PdAllocInWaitPdStop(uint8 bandId)
{
	/*We are done*/
	CtsManager_Done(bandId);
}

/**********************************************************************************

CtsManager_LockCfmInWaitLock 


Description:
------------
Lock CFM In Wait Lock State handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CtsManager_LockCfmInWaitLock(uint8 bandId)
{
	/*See if PD is on GPHP - it should be the only thing on it!*/
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
	
	/*Peek head packet*/
	memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
	hwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_GPHP;
	hwQueueManagerRequestParams.primaryAddr = CtsManagerBand_Db[bandId].vapHead;
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	HwQManager_PeekHeadPacket(&hwQueueManagerRequestParams);
  	if (hwQueueManagerRequestParams.pHeadDesc != NULL_PD)
    {
		HwQManager_PopPacket(&hwQueueManagerRequestParams);
		/*If PD is here it means it was never transmitted - Set Status to failure*/
		CtsManagerBand_Db[bandId].status = FALSE;
		/*We are done*/
		CtsManager_Done(bandId);
    }
	else
	{
		/*packet is not on GPHP queue - must be on the way back*/
		/*Change State back to Wait for TX CFM*/
		CtsManager_ChangeState(CTS_MANAGER_WAIT_TX_CFM_STATE, bandId); 
	}
	/*Unlock GPHP*/	
	Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_GPHP, CtsManagerBand_Db[bandId].vapHead, (0x1 << 0));
}

/**********************************************************************************

CtsManager_LockCfmInWaitLockStop 


Description:
------------
Lock CFM In Wait Lock Stop handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CtsManager_LockCfmInWaitLockStop(uint8 bandId)
{
	/*We are done*/
	CtsManager_Done(bandId);
	/*Unlock GPHP*/	
	Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_GPHP, CtsManagerBand_Db[bandId].vapHead, (0x1 << 0));
}
 
/**********************************************************************************

CtsManager_Fatal 


Description:
------------
handles unexpected event


Input: 
-----	
 	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CtsManager_Fatal(uint8 bandId)
{
	UNUSED_PARAM(bandId);	
	FATAL("CTS Manager Fatal");
}

/*---------------------------------------------------------------------------------
/						global Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

ctsManagerVapOpen











Description:
------------
VAP Open messge handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void ctsManagerVapOpen(K_MSG* ctsManagerMessage)
{
	BSS_MANAGER_CONFIRM_EVENT* confirmEvent; 
	uint8 vapId; 
	K_MSG *vapMsg = (K_MSG *) (*((uint32 *)pK_MSG_DATA(ctsManagerMessage)));
	UMI_SET_WMM_PARAMETERS* pPostVapActivationMsg = (UMI_SET_WMM_PARAMETERS*) pK_MSG_DATA(vapMsg);
	uint8 bandId;

	vapId = pPostVapActivationMsg->vapId; 
	bandId = ConfigurationManager_GetBandForVap(vapId);
	ASSERT(CtsManager_VapDb[vapId].active == FALSE);
	CtsManager_VapDb[vapId].active = TRUE;
	/*CTS are transmitted inside a process and a VAP can't be added or removed while another process is running
	so we don't need to handle this in CTS states - just add VAP to list*/
	if (CtsManagerBand_Db[bandId].vapHead != CTS_MANAGER_INVALID_VAP)
	{
		CtsManager_VapDb[CtsManagerBand_Db[bandId].vapHead].prevVap = vapId;
	}
	CtsManager_VapDb[vapId].prevVap = CTS_MANAGER_INVALID_VAP;
	CtsManager_VapDb[vapId].nextVap = CtsManagerBand_Db[bandId].vapHead;
	CtsManagerBand_Db[bandId].vapHead = vapId;
	/*Send Confirmation to VAP manager*/
	confirmEvent = (BSS_MANAGER_CONFIRM_EVENT*)(&(ctsManagerMessage->abData));
	confirmEvent->vapId= vapId;
	confirmEvent->eventIndex = VAP_MANAGER_POST_VAP_ACTIVATION; 
	confirmEvent->clientId = BSS_MANAGER_VAP_MANAGER_CTS_MANAGER_CLIENT; 
	OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, ctsManagerMessage, vapId);
	
}

/**********************************************************************************

ctsManagerVapRemove










Description:
------------
VAP remove messge handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void ctsManagerVapRemove(K_MSG* ctsManagerMessage)
{
	uint8 vapId; 
	BSS_MANAGER_CONFIRM_EVENT* confirmEvent; 
	K_MSG *vapMsg = (K_MSG *) (*((uint32 *)pK_MSG_DATA(ctsManagerMessage)));
	UMI_REMOVE_VAP * removeVapPtr = (UMI_REMOVE_VAP*) pK_MSG_DATA(vapMsg);
	uint8 bandId;

	vapId = removeVapPtr->vapId; 
	bandId = ConfigurationManager_GetBandForVap(vapId);
	/*From CTS manager perspective a VAP is added only when WMM params are set - a VAP may be removed without rewaching that state*/
	if (CtsManager_VapDb[vapId].active == TRUE)
	{
		/*CTS are transmitted inside a process and a VAP can't be added or removed while another process is running
		so we don't need to handle this in CTS states - just remove VAP from list*/
		if (CtsManager_VapDb[vapId].prevVap == CTS_MANAGER_INVALID_VAP)
		{
			CtsManagerBand_Db[bandId].vapHead = CtsManager_VapDb[vapId].nextVap;
		}
		else
		{
			CtsManager_VapDb[CtsManager_VapDb[vapId].prevVap].nextVap = CtsManager_VapDb[vapId].nextVap;
		}
		
		if (CtsManager_VapDb[vapId].nextVap != CTS_MANAGER_INVALID_VAP)//Marcelo
		{
			CtsManager_VapDb[CtsManager_VapDb[vapId].nextVap].prevVap = CtsManager_VapDb[vapId].prevVap;
		}
		CtsManager_VapDb[vapId].active = FALSE;
	}
	/*Send Confirmation to VAP manager*/
	confirmEvent = (BSS_MANAGER_CONFIRM_EVENT*)(&(ctsManagerMessage->abData));
	confirmEvent->vapId = vapId;
	confirmEvent->eventIndex = VAP_MANAGER_REMOVE_VAP; 
	confirmEvent->clientId = BSS_MANAGER_VAP_MANAGER_CTS_MANAGER_CLIENT; 
	OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, ctsManagerMessage, vapId);
}

/**********************************************************************************

ctsManagerStartTx









Description:
------------
CTS Start TX messge handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void ctsManagerStartTx(K_MSG* ctsManagerMessage)
{
	uint8 vapId = ctsManagerMessage->header.vapId; 
	uint8 bandId = ConfigurationManager_GetBandForVap(vapId);

	
	/*Store the message*/
	CtsManagerBand_Db[bandId].psMsg = ctsManagerMessage;
	/*Call handler*/
	CtsManager_StartTx[CtsManagerBand_Db[bandId].state](bandId);
}

/**********************************************************************************

ctsManagerStopTx








Description:
------------
CTS STOP TX messge handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void ctsManagerStopTx(K_MSG* ctsManagerMessage)
{
	UNUSED_PARAM(ctsManagerMessage);	
	/*Mark timer inactive*/
	CtsManagerBand_Db[CONFIGURATION_MANAGER_BAND_0].timerActive = FALSE;
	/*Call handler*/
	CtsManager_StopTx[CtsManagerBand_Db[CONFIGURATION_MANAGER_BAND_0].state](CONFIGURATION_MANAGER_BAND_0);
}

void ctsManagerStopTx_B1(K_MSG* ctsManagerMessage)
{
	UNUSED_PARAM(ctsManagerMessage);	
	/*Mark timer inactive*/
	CtsManagerBand_Db[CONFIGURATION_MANAGER_BAND_1].timerActive = FALSE;
	/*Call handler*/
	CtsManager_StopTx[CtsManagerBand_Db[CONFIGURATION_MANAGER_BAND_1].state](CONFIGURATION_MANAGER_BAND_1);
}


/**********************************************************************************

ctsManagerTxCfm







Description:
------------
TX CFM messge handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void ctsManagerTxCfm(K_MSG* ctsManagerMessage)
{
	uint8 vapId = ctsManagerMessage->header.vapId; 
	uint8 bandId = ConfigurationManager_GetBandForVap(vapId);

	
	/*Call handler*/
	CtsManager_TxCfm[CtsManagerBand_Db[bandId].state](bandId);
}

/**********************************************************************************

ctsManagerPdAlloc






Description:
------------
PD Alloc messge handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void ctsManagerPdAlloc(K_MSG* ctsManagerMessage)
{
 	RmPdFreeDescResponse_t *pdMsg = (RmPdFreeDescResponse_t *)pK_MSG_DATA(ctsManagerMessage);
	uint8 bandId = pdMsg->context; // We saved the bandId on the context when we requested the PD.

	/*Store PD pointer*/
	CtsManagerBand_Db[bandId].pd = pdMsg->packetDescriptor;
	/*Store DRAM pointer - field is used for duration*/
	CtsManagerBand_Db[bandId].dram_pointer = CtsManagerBand_Db[bandId].pd->packetPointer;
	/*Release the request*/
	ResourceManager_ReleaseRequest(CtsManagerBand_Db[bandId].requestId, DESC_POOL_CSA_MANAGER);
	/*Call handler*/
	CtsManager_PdAlloc[CtsManagerBand_Db[bandId].state](bandId);
}

/**********************************************************************************

ctsManagerGphpLock





Description:
------------
Lock CFM Message Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void ctsManagerGphpLock(K_MSG* ctsManagerMessage)
{
	LockReqCb_t *lockReg = (LockReqCb_t *)pK_MSG_DATA(ctsManagerMessage);
	uint8 bandId = (uint8)lockReg->callerContext; // We saved the bandId as callerContext when we asked for the lock

	/*Call handler*/
	CtsManager_LockCfm[CtsManagerBand_Db[bandId].state](bandId);
}


/**********************************************************************************

ctsManager_Init




Description:
------------
Initiailize CTS MAnager


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void ctsManager_Init()
{
	uint8 i;
	uint8 bandId;
	
	// Clear Database per Band
	for (bandId = 0; bandId < NUM_OF_CONFIGURATION_MANAGER_BANDS; bandId++)
	{
	
		CtsManagerBand_Db[bandId].vapHead = CTS_MANAGER_INVALID_VAP;
		CtsManagerBand_Db[bandId].state = CTS_MANAGER_IDLE_STATE;
		CtsManagerBand_Db[bandId].pd = (TxPd_t *)NULL_PD;
		CtsManagerBand_Db[bandId].txReqMsg.timeout = CTS_MANAGER_TIMEOUT_IN_MS;
		CtsManagerBand_Db[bandId].timerActive = FALSE;
		CtsManagerBand_Db[bandId].txReqMsg.ctrl_bw = PD_CONTROL_20_MHZ;
		CtsManagerBand_Db[bandId].txReqMsg.duration = CTS2SELF_DURATION_MSEC_NORMAL_MODE;
	}
	
	// Each band has a different timer ID.
	CtsManagerBand_Db[CONFIGURATION_MANAGER_BAND_0].timerMsgId = CTS_MANAGER_STOP_TX;
	CtsManagerBand_Db[CONFIGURATION_MANAGER_BAND_1].timerMsgId = CTS_MANAGER_STOP_TX_B1;

	// Clean Database per VAP
	for (i = 0; i < HW_NUM_OF_VAPS; i++)
	{
		CtsManager_VapDb[i].nextVap = CTS_MANAGER_INVALID_VAP;
		CtsManager_VapDb[i].prevVap = CTS_MANAGER_INVALID_VAP;
		CtsManager_VapDb[i].active = FALSE;
	}
}



