/***************************************************************************************
 File:			Deaggregator.c
 Module:	  		De-Aggregator & Header Conversion
 Purpose: 		Holds all the implementation of the De-aggregation & Header Conversion API functions
 Description:  		This file contains the implementation of the De-aggregation & Header Conversion module
				which is responsible of parsing AMSDUs to MSDUs and doing header converion 
***************************************************************************************/

/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "HwGlobalDefinitions.h"
#include "RxMsduParserRegs.h" // Deaggregator & Header conversion registers
#include "Deaggregator.h"
#include "Deaggregator_Api.h"
#include "RegAccess_Api.h"
#include "RxDescriptor.h"
#include "RxHandler_InitApi.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_HW_DRIVERS
#define LOG_LOCAL_FID 	17


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/





/**********************************************************************************
Deaggregator_EnableDisableWpaPassThrough  

Description:
------------
	Enable / Disable WPA PT in De-aggregator & Header Conversion.
	In this mode there is no header conversion. In case size > 1500, a bit is turned on in the MSDU descriptor.
	RD Type must be configured at init since we do here read-modify-write
	
Input:
-----
	Enable/Disable
Output:
-------
	None
	
**********************************************************************************/
void Deaggregator_EnableDisableWpaPassThrough(bool isEnable)
{
	RegRxMsduParserWpaPtModeEn_u WpaPtModeEnReg;
	
	// Read-Modify-Write. Assume RD Type is already configured in the field "wpaPtRdTypeIdx" during init.	
	RegAccess_Read(REG_RX_MSDU_PARSER_WPA_PT_MODE_EN, &WpaPtModeEnReg.val);
	WpaPtModeEnReg.bitFields.wpaPtEn = isEnable;
	RegAccess_Write(REG_RX_MSDU_PARSER_WPA_PT_MODE_EN, WpaPtModeEnReg.val);
}


/**********************************************************************************
Deaggregator_EnableDisableLogger  

Description:
------------
	Enable / Disable logs from the De-aggregator & Header Conversion

Input:
-----
	Enable/Disable	
Output:
-------
	None
	
**********************************************************************************/
void Deaggregator_EnableDisableLogger(bool isEnable)
{
	RegRxMsduParserRxMsduParserLoggerCfg_u RxMsduParserLoggerCfgReg;

	RxMsduParserLoggerCfgReg.val = 0;
	RxMsduParserLoggerCfgReg.bitFields.rxMsduParserLoggerEn = isEnable;
	RxMsduParserLoggerCfgReg.bitFields.rxMsduParserLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_RX_MSDU_PARSER_RX_MSDU_PARSER_LOGGER_CFG, RxMsduParserLoggerCfgReg.val);
}


/**********************************************************************************
Deaggregator_Initialize  

Description:
------------
	initialize the De-aggregator & Header Conversion module

Input:
-----
	None
Output:
-------
	None
	
**********************************************************************************/
void Deaggregator_Init(void)
{	
	RegRxMsduParserRxMsduParserEn_u 	RxMsduParserEnReg;
	RegRxMsduParserNoMatchConvType_u	NoMatchConvTypeReg;
	RegRxMsduParserRdTypeConvEn_u		RdTypeConvEnReg;
	RegRxMsduParserMsduDescFifoStuck_u	MsduDescFifoStuckReg;
	RegRxMsduParserRxMsduParserCfg_u	RxMsduParserCfgReg;
	RegRxMsduParserMsduDescFifoCfg_u	MsduDescFifoCfgReg;


	RxMsduParserEnReg.val 		= 0;
	NoMatchConvTypeReg.val 		= 0;
	RdTypeConvEnReg.val			= 0;
	MsduDescFifoStuckReg.val	= 0;
	RxMsduParserCfgReg.val	= 0;
	MsduDescFifoCfgReg.val	= 0;


	// Configure MSDU ring size in RX Handler ScratchPad.
	MsduDescFifoCfgReg.bitFields.msduDescFifoDepthMinusOne	= MSDU_DESC_NUMBER_OF_ENTRIES - 1;
	RegAccess_Write(REG_RX_MSDU_PARSER_MSDU_DESC_FIFO_CFG, MsduDescFifoCfgReg.val);

	// Configure Max frame length and Minimum flow control bytes in case there is no coversion (used to make sure action type is decrypted already).
	// The Max frame length is checked only vs MSDUs not vs MPDU that weren't de-aggregated.
	RxMsduParserCfgReg.bitFields.maxFrameLength 	= DEAGGREGATOR_MAX_FRAME_LENGTH;
	RxMsduParserCfgReg.bitFields.flowContByteCount 	= FLOW_CONTROL_BYTE_COUNT;
	RxMsduParserCfgReg.bitFields.amsduMsduMinLength = 0x8; // keep the default value.
	RegAccess_Write(REG_RX_MSDU_PARSER_RX_MSDU_PARSER_CFG, RxMsduParserCfgReg.val);

	// Configure number of cycles to wait before setting a bit which says MSDU ring is full (can be cleared by a different bit)
	MsduDescFifoStuckReg.bitFields.msduDescFifoFullCountLimit = MSDU_RING_FULL_COUNT_LIMIT;
	RegAccess_Write(REG_RX_MSDU_PARSER_MSDU_DESC_FIFO_STUCK, MsduDescFifoStuckReg.val);

	// Set conversion On/Off per RD Type
	RdTypeConvEnReg.bitFields.rdTypeConvEn = HEADER_CONVERSION_MASK_PER_RD_TYPE;
	RegAccess_Write(REG_RX_MSDU_PARSER_RD_TYPE_CONV_EN, RdTypeConvEnReg.val);

	// Set configureable header conversion line (currently it's not needed, so it's commented out. If needed - uncomment and use the code).

	/**************************************************************************

	RegRxMsduParserConfLlc_u 			ConfLlcReg;						
	RegRxMsduParserConfOui_u 			ConfOuiReg;
	RegRxMsduParserConfType_u 			ConfTypeReg;
	RegRxMsduParserConfConvType_u 		ConfConvTypeReg;
	
	ConfLlcReg.val 			= 0;
	ConfOuiReg.val 			= 0;
	ConfTypeReg.val 		= 0;
	ConfConvTypeReg.val 	= 0;	

	ConfLlcReg.bitFields.confLlc 				= TBD;
	ConfLlcReg.bitFields.confLlcMask			= TBD;		  
	RegAccess_Write(REG_RX_MSDU_PARSER_CONF_LLC, ConfLlcReg.val);
	ConfOuiReg.bitFields.confOui 			= TBD;
	ConfOuiReg.bitFields.confOuiMask 		= TBD;
	RegAccess_Write(REG_RX_MSDU_PARSER_CONF_OUI, ConfOuiReg.val);
	ConfTypeReg.bitFields.confType 			= TBD;
	ConfTypeReg.bitFields.confTypeMask		= TBD;
	RegAccess_Write(REG_RX_MSDU_PARSER_CONF_TYPE, ConfTypeReg.val);
	ConfConvTypeReg.bitFields.confConvEn	= TRUE;
	ConfConvTypeReg.bitFields.confConvType 	= TBD;
	RegAccess_Write(REG_RX_MSDU_PARSER_CONF_CONV_TYPE, ConfConvTypeReg.val);

	***************************************************************************/

	// Configure conversion type when there is no match to any known conversion
	NoMatchConvTypeReg.bitFields.noMatchConvType = CONVERSION_TYPE_NO_CONVERSION;
	RegAccess_Write(REG_RX_MSDU_PARSER_NO_MATCH_CONV_TYPE, NoMatchConvTypeReg.val);


	// configure WPA Pass Through RD Type. Keep the feature Off. It can be turned on using a dedicated API.
	// (currently it's not needed, so it's commented out. If needed - uncomment and use the code).

	/************************************************************************
	RegRxMsduParserWpaPtModeEn_u WpaPtModeEnReg;

	WpaPtModeEnReg.val = 0;

	WpaPtModeEnReg.bitFields.wpaPtRdTypeIdx = TBD;
	RegAccess_Write(REG_RX_MSDU_PARSER_WPA_PT_MODE_EN, WpaPtModeEnReg.val);
	
	************************************************************************/

	// Turn on de-aggregation and header conversion capabilities.
	RxMsduParserEnReg.bitFields.rxMsduParserEn 	= TRUE;
	RxMsduParserEnReg.bitFields.rxMsduConvEn 	= TRUE;
	RegAccess_Write(REG_RX_MSDU_PARSER_RX_MSDU_PARSER_EN, RxMsduParserEnReg.val);
}

