/***************************************************************
 File:		DeliaRecipes.h
 Module:	  	Delia Recipes
 Purpose: 	holds the definions for construction of the Delia recipes 
 Description:  This file contains the definions of the construction of the Delia recipes module
 			and the recipes pointers table
***************************************************************/
#ifndef DELIA_RECIPES_H_
#define DELIA_RECIPES_H_
//---------------------------------------------------------------------------------
//						Includes						
//---------------------------------------------------------------------------------
#include "System_GlobalDefinitions.h"
#include "System_Configuration.h"
#include "protocol.h"
#include "frame.h"
#include "Tcr_Descriptors.h"

//---------------------------------------------------------------------------------
//						Defines						
//---------------------------------------------------------------------------------



/* Delia general values and sizes */						
#define COMMON_TCR0_LENGTH_TO_SEND        			16
#define COMMON_TCR1_LENGTH_TO_SEND         			16
#define COMMON_TCR_LENGTH_TO_SEND          			(COMMON_TCR0_LENGTH_TO_SEND + COMMON_TCR1_LENGTH_TO_SEND)
#define PER_USER_TCR_LENGTH_TO_SEND          		16
#define PER_USER_TCR_LENGTH_FIRST_4_BYTES			4
#define PER_USER_TCR_LENGTH_BYTES_1_TO_3			(PER_USER_TCR_LENGTH_TO_SEND - PER_USER_TCR_LENGTH_FIRST_4_BYTES)
#define PER_USER_TCR_OFFSET_BYTES_1_TO_3			(COMMON_TCR_LENGTH_TO_SEND + PER_USER_TCR_LENGTH_FIRST_4_BYTES)
#define COMMON_RCR_LENGTH_TO_SEND					16 // 8 bytes with actual data + 8 dummy bytes for 128bit round up
#define PER_USER_RCR_LENGTH_TO_SEND					16 // 4 bytes with actual data + 12 dummy bytes for 128bit round up


#define TCR_LENGTH_TO_SEND              			16
#define ONE_MPDU_TO_COUNTER             			1
#define BEACON_COUNTER_ONE             				1
#define MPDU_COUNTER_ZERO               			0
#define CRC_LENGTH                      			4
#define DELIMITER_LENGTH               		 		4
#define SIZE_OF_TIMESTAMP               			8 
#define BF_ENTRY_SHIFT_SIZE             			9
#define BF_INDEX_SHIFT                  			3
#define MPDU_DESCRIPTOR_SIZE            			0xC
#define ELEMENT_DESCRIPTOR_SIZE         			0xC
#define AUTO_RESPONSE_TCR_12_BITS_SHIFT				12
#ifdef WORKAROUND_FOR_BFEE_SUB_BAND_HW_BUG
#define BFEE_USER_TCR_BYTE_LENGTH_HE				11
#define BFEE_USER_TCR_BYTE_LENGTH_NONE_HE			12
#endif 
#define BFEE_USER_TCR_BYTE_LENGTH					12
#define BFEE_TXC_SECTION_A_BYTE_LENGTH				2
#define BFEE_REPORT_DURATION_BYTE_LENGTH			2
#define BFEE_TXC_SECTION_B_BYTE_LENGTH				18
#define BFEE_SEQUENCE_CONTROL_LENGTH				2
#define BFEE_ACTION_CATEGORY_LENGTH					2
#define BFEE_ACTION_MIMO_CTRL_LENGTH				2
#define BFEE_REPORT_SEQUENCE_NUMBER_BYTES_TO_MASK   0x3
#define BFEE_REPORT_SEQUENCE_NUMBER_MASK_PATTERN    0xFF
#define BFEE_REPORT_SEQUENCE_NUMBER_BYTES_TO_SAVE   0x3


/* Delia TCRs */
#define TCR3_NOT_SOUNDING_NUMOFBITS 1
#define TCR3_NOT_SOUNDING_SHIFT	    1
#define TCR3_NOT_SOUNDING_MASK 	    MASK(TCR3_NOT_SOUNDING_NUMOFBITS,TCR3_NOT_SOUNDING_SHIFT,uint32)

#define TCR3_SMOOTH_NUMOFBITS	    1
#define TCR3_SMOOTH_SHIFT		    2
#define TCR3_SMOOTH_MASK 		    MASK(TCR3_SMOOTH_NUMOFBITS,TCR3_SMOOTH_SHIFT,uint32)



#define DELIA_BA_TCR_0        0x0
#define DELIA_BA_TCR_1        0x0
#define DELIA_BA_TCR_2        0x0
#define DELIA_BA_TCR_3        (TCR3_SMOOTH_MASK | TCR3_NOT_SOUNDING_MASK)
#define DELIA_BA_TCR_4        SIZEOF_BLOCK_ACK_FRM


#define DELIA_AUTO_RESPONSE_TCR_1	0x0
#define DELIA_AUTO_RESPONSE_TCR_3	(TCR3_SMOOTH_MASK | TCR3_NOT_SOUNDING_MASK)




/* Delia events */
#define TIMER_EVENT                  0x0001
#define BFEE_LENGTH_READY            0x0002
#define BFEE_SECTION_A_READY    	 0x0004
#define BFEE_SECTION_B_READY    	 0x0008
#define BFEE_SECTION_C_READY    	 0x0010
#define DURATION_EVENT               0x0020
#define COUNTER_EVENT                0x0040
#define ALL_TCRS_IN_HOB_READY_EVENT  0x0100
#define TCR0_READY_EVENT             0x0200
#define TCR1_TO_WORD6_READY_EVENT    0x0400
#define WORD5_TO_WORD6_READY_EVENT   0x0800
#define USE_TCR_SET_2_EVENT          0x1000
#define DURATION_REG_READY_EVENT     0x2000
//#define BF_STATE_MACHINE_IDLE_EVENT  0x4000
#define AUTO_REPLY_FRAME_READY		 0x10000
#define AUTO_REPLY_TCR_LENGTH_READY	 0x20000


#define ALL_TCRS_IN_HOB_READY_EVENT_BIT_OFFSET_IN_BYTE			0x0
#define ALL_TCRS_IN_HOB_USE_TCR_SET2_EVENT_BIT_OFFSET_IN_BYTE	0x4	
#define ALL_TCRS_IN_HOB_READY_EVENT_BYTE_OFFSET_IN_REG			0x1


/* Delia masks */
#define AUTO_RESPONSE_TCR_12_BITS_MASK	  		  	0xFFF
#define AUTO_RESPONSE_TCR_16_BITS_MASK	  		  	0xFFFF
#define AUTO_RESPONSE_NOT_HE_OR_VHT_MASK	  	 	0x7

#define BFEE_16_BITS_MASK	  		 			 	0xFFFF
#define BFEE_IS_HT_MASK	  			 			 	0x0001
#define BFEE_BFR_SM_MASK	  			 			0x00000001
#define	BFEE_REDUNDENT_BYTE_LENGHT_MASK				0x3
#define	BFEE_CRC_VALID_MASK							0x1
#ifdef WORKAROUND_FOR_BFEE_SUB_BAND_HW_BUG
#define BFEE_BW_OFFSET_ON_TCR						0x4
#define BFEE_PHY_MODE_HE_BIT_NUM	  			 	0x4 //pacPhyHeSu
#define BFEE_BW_MASK								0x3 //bits
#define BFEE_ONE_BYTE_LENGTH						0x1

#endif //WORKAROUND_FOR_BFEE_SUB_BAND_HW_BUG
#define BFEE_VAP_INDX_SHIFT	  			 			0x10 
//---------------------------------------------------------------------------------
//						Delia Registers						
//----------------------------------------------------------------------------------
#define R0          0
#define R1          1
#define R2          2
#define R3          3
#define R4          4
#define R5          5
#define R6          6
#define HOB         7


#define DELIA_REGISTER_ASSERT_SENDER_VALUE				R0
#define DELIA_REGISTER_DELIAD_BASE_ADDRESS				R0
#define DELIA_REGISTER_AUTO_GENERATED_TCRS_BASE_ADDRESS	R1
#define DELIA_REGISTER_PER_USER_TCRS_TOTAL_LENGTH		R2
#define DELIA_REGISTER_TX_CTRL_WORD_ADDRESS				R6
#define DELIA_REGISTER_MPDU_HEADER_LENGTH				R2
#define DELIA_REGISTER_MPDU_COUNTER_VALUE				R1
#define DELIA_REGISTER_TF_PAYLOAD_LENGTH				R1
#define DELIA_REGISTER_TF_PADDING_LENGTH				R2
#define DELIA_REGISTER_TF_TOTAL_LENGTH_REGISTER			R3
#define DELIA_REGISTER_TF_PAYLOAD_ADDRESS				R3
#define DELIA_REGISTER_MPDU_HEADER_ADDRESS				R1
#define DELIA_REGISTER_MPDU_PAYLOAD_LENGTH				R1
#define DELIA_REGISTER_MPDU_PAYLOAD_ADDRESS				R2
#define DELIA_REGISTER_CURRENT_ELEMENT_OFFSET			R1
#define DELIA_REGISTER_BEACON_COUNTER					R2
#define DELIA_REGISTER_ELEMENT_LENGTH					R3
#define DELIA_REGISTER_REGULAR_ELEMENT_ADDRESS			R4
#define DELIA_REGISTER_VAP_INDEX						R4
#define DELIA_REGISTER_TIM_BITMAP_CTRL_LENGTH			R4
#define DELIA_REGISTER_TIM_IE_LENGTH					R4
#define DELIA_REGISTER_BITMAP_CONTROL_HOB_OFF			R5
#define DELIA_REGISTER_TIM_IE_OFFSET					R5
#define DELIA_REGISTER_AUTO_GENERATED_RCRS_BASE_ADDRESS R1
#define DELIA_REGISTER_RCR_PER_USER_INDEX				R2

#define AUTO_RESPONSE_TCR0_ADDRESS_REGISTER				R0
#define AUTO_RESPONSE_FRAME_POINTER_REGISTER			R1
#define AUTO_RESPONSE_FRAME_LENGTH_REGISTER				R2
#define AUTO_RESPONSE_PHY_MODE_REGISTER					R3
#define AUTO_RESPONSE_TCR_LEN_REGISTER					R4
#define AUTO_RESPONSE_TCR_LEN_UPDATE_REGISTER			R5
#define AUTO_RESPONSE_PER_USER_TCR_TOTAL_LENGTH			R1


#define BFEE_REPORT_TX_RECIPE_BASE_ADDRESS_REGISTER		R0 
#define BFEE_REPORT_TXC_SECTION_A_BASE_ADDRESS_REGISTER	R0 
#define BFEE_REPORT_VAP_TID_PTR_REGISTER				R0 
#define BFEE_REPORT_BFEE_STA_REGISTER					R0 
#define BFEE_REPORT_TXC_SECTION_B_BASE_ADDRESS_REGISTER	R0
#define BFEE_REPORT_TXC_SECTION_C_BASE_ADDRESS_REGISTER	R0
#define BFEE_REPORT_TXC_SECTION_CRC_REGISTER			R0
#define BFEE_REPORT_TCR_LENGTH_REGISTER					R1
#define BFEE_REPORT_SEQUENCE_NUMBER_REGISTER			R1
#define BFEE_REPORT_B_SECTION_LENGTH_REGISTER			R1
#define BFEE_REPORT_C_SECTION_LENGTH_REGISTER			R1
#ifdef WORKAROUND_FOR_BFEE_SUB_BAND_HW_BUG
#define BFEE_REPORT_BFEE_WA_REGISTER					R1
#define BFEE_ONE_BYTE_LENGTH_REGISTER					R3
#endif 
#define BFEE_REPORT_PHY_MODE_REGISTER					R2
#define BFEE_REPORT_SEQUENCE_NUMBER_ADDR_REGISTER		R2
#define BFEE_REPORT_BFEE_SM_REGISTER					R2




//---------------------------------------------------------------------------------
//						Delia offsets						
//----------------------------------------------------------------------------------

/* Delia descriptor beacon elements part */
#define BEACON_ELEMENT_GENERAL_REGISTER_OFFSET                       0x0
#define ELEMENT_LAST_INDICATION_BYTE_NUMBER                          3
#define ELEMENT_LAST_INDICATION_OFFSET	                             (ELEMENT_LAST_INDICATION_BYTE_NUMBER + BEACON_ELEMENT_GENERAL_REGISTER_OFFSET)
#define ELEMENT_TYPE_BYTE_NUMBER				                     2
#define ELEMENT_TYPE_OFFSET                                          (BEACON_ELEMENT_GENERAL_REGISTER_OFFSET + ELEMENT_TYPE_BYTE_NUMBER)
#define REGULAR_TYPE_BIT_NUMBER_IN_BYTE					             0									
#define TIMESTAMP_TYPE_BIT_NUMBER_IN_BYTE				             2
#define ELEMENT_LAST_INDICATION_LOCATION_IN_BYTE                     7

   
/* Delia BF report (auto reply) part */
#define BF_REPORT_MATRICES_EXIST_BIT_NUMBER_IN_BYTE          6
#define BF_REPORT_AFTER_NDP_BIT_NUMBER_IN_BYTE		         7
#define BF_REPORT_STATION_ID_MODULO_4_SHIFT                  30
#define BF_REPORT_BF_INDEX_SIZE_SHIFT                        3
#define BF_REPORT_BF_INDEX_SHIFT                             (BF_REPORT_STATION_ID_MODULO_4_SHIFT - BF_REPORT_BF_INDEX_SIZE_SHIFT)
#define VHT_INDICATION_BIT_NUMBER_IN_BYTE_IN_HOB             3//????
#define VALUE_TO_ADD_TO_SEQUENCE_NUMBER                      (0x1 << FM_SEQUENCE_NUMBER_SHIFT)                                  
#define BF_REPORT_SEQUENCE_NUMBER_OFFSET                     0
#define BF_REPORT_HT_CONTROL_SIZE                            4
#define BF_REPORT_HT_CONTROL_OFFSET_IN_ENTRY                 0x1FC
#define BF_REPORT_VHT_BIT_NUMBER_IN_BYTE_IN_CONTROL          4
#define BF_REPORT_ERROR_BIT_IN_BYTE_IN_CONTROL               2 
#define BF_REPORT_TOTAL_MIMO_SIZE                            3 
#define BF_REPORT_EMPTY_MIMO_VALUE                           0x007000 
#define BF_REPORT_FIRST_PART_OF_REGULAR_MIMO_SIZE            2
#define HOB_REGISTER_BF_REPORT_HTC_FIELD_SIZE				 4
#define BF_REPORT_SECOND_PART_OF_REGULAR_MIMO_SIZE           1
#define BF_REPORT_SNR_OFFSET_IN_BF_ENTRY                     0x4
#define BF_REPORT_BEAMFORMEE_MATRICES_OFFSET                 0
#define BF_REPORT_CHOOSE_ADDRESS_BIT_NUMBER_IN_BYTE			 0
#define HOB_REGISTER_ADDRESS_1_WITH_TA_BIT					 0
#define BFEE_BYTE_OFFSET_FOR_TXC_SECTION_A					 4


/* Delia Auto Response (common recipe) part */
#define AUTO_RESPONSE_DUR_TYPE_BIT_NUMBER	0x0 // bit 0 of duration type in HOB decides if it's TX or RX based duration.
#define AUTO_RESPONSE_DELIMITER_SIZE		0x4



#define BFEE_DELIMITER_SIZE					0x4


/* Delia general offsets */
#define AUTO_RESPONSE_PAYLOAD_OFFSET	(DUR_FIELD_LEN + FRAME_CONTROL_LEN) // The first 4 bytes contains the frame control and duration place holder. 

//---------------------------------------------------------------------------------
//						Macros					
//---------------------------------------------------------------------------------

#define TX_CONTROL(oo,af,f,d,i,p,s,ar,e,t,c) 													    \
	(0x53000000							| 															\
	((oo & 0x3) << 13)					| 	/*Override TCR duration*/			                    \
	((af & 0x1) << 12)					| 	/*assert on auto fill*/									\
	((f & 0x1) << 11)					| 	/*skip flow control checking*/							\
	((d & 0x1) << 10)					| 	/*descriptors caching enable*/							\
	((i & 0x1) << 9)					| 	/*enable tx end interrupt*/								\
	((p & 0x1) << 8)					| 	/*load prbs initial*/									\
	((s & 0xF) << 4)					| 	/*bus state:*/											\
	((ar & 0x1) << 3)					| 	/*auto reply frame*/									\
	((e & 0x1) << 2)					| 	/*assert tx enable*/									\
	((t & 0x1) << 1)					| 	/*latch tsf*/											\
	(c & 0x1))								/*CRC Init*/


#define DELIA_CONTROL_BUS_STATE_IDLE					     (0)
#define DELIA_CONTROL_BUS_STATE_TRANMSIT_COMMON_TCRS	     (1)
#define DELIA_CONTROL_BUS_STATE_TRANSMIT_DATA			     (3)
#define DELIA_CONTROL_BUS_STATE_TRANSMIT_PER_USER_RCRS	     (6)
#define DELIA_CONTROL_BUS_STATE_TRANSMIT_PER_USER_TCRS	     (7)
#define DELIA_CONTROL_BUS_STATE_TRANSMIT_COMMON_RCRS	     (9)

#define DELIA_CONTROL_NO_OVERRIDE                            (0x0)
#define DELIA_CONTROL_OVERRIDE_BASED_ON_TX                   (0x1)
#define DELIA_CONTROL_OVERRIDE_BASED_ON_AUTO_REPLY           (0x2)
#define DELIA_CONTROL_OVERRIDE_BASED_ON_AUTO_REPLY_BF_REPORT (0x3)

#define TCR(immediateValue)    		  					(0x50000000  | (immediateValue))
#define END_RECIPE                                      (0xF0000000)
#define NOP                                             (0x70000000)

#define WAIT(event)                                     (0x74000000 | (event))
#define WAIT_FOR_TCR0_TO_BE_READY                       WAIT(TCR0_READY_EVENT)
#define WAIT_FOR_TCR1_TO_TCR4_TO_BE_READY               WAIT(TCR1_TO_TCR4_READY_EVENT)
//#define WAIT_FOR_BF_IDLE_OR_TIMEOUT                     WAIT(TIMER_EVENT | BF_STATE_MACHINE_IDLE_EVENT)
#define WAIT_FOR_MPDU_OR_BEACON_TO_BE_READY             WAIT(COUNTER_EVENT)
#define WAIT_FOR_DURATION             					WAIT(DURATION_EVENT)
#define WAIT_BFEE_LENGTH_READY                   		WAIT(BFEE_LENGTH_READY)
#define WAIT_BFEE_SECTION_A_READY          			    WAIT(BFEE_SECTION_A_READY)
#define WAIT_BFEE_SECTION_B_READY          			    WAIT(BFEE_SECTION_B_READY)
#define WAIT_BFEE_SECTION_C_READY          			    WAIT(BFEE_SECTION_C_READY)
#define WAIT_FOR_AUTO_REPLY_FRAME_TO_BE_READY  			WAIT(AUTO_REPLY_FRAME_READY)
#define WAIT_FOR_AUTO_REPLY_TCR_LENGTH_TO_BE_READY     	WAIT(AUTO_REPLY_TCR_LENGTH_READY)

#define LOAD_REGISTER_WITH_IMMEDIATE_VALUE(registerNumber,value)                           (0x40000000 | ((registerNumber) << 24) | ((unsigned long)(value) & 0x007FFFFF))
#define LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(destinationRegister,addressRegister,offset)  (0x48000000 | ((destinationRegister) << 24) | ((addressRegister) << 20) | ((unsigned long)(offset) & 0x000FFFFF))
#define LOAD_FROM_HOB(registerNumber,offset)                                               (0x47800000 | ((registerNumber) << 20) | ((unsigned long)(offset) & 0x00000FFF))
#define STORE(valueRegister,addressRegister,bytesToStore,offset)                           (0x48800000 | ((valueRegister) << 24) | ((addressRegister) << 20) | ((bytesToStore) << 16) | ((unsigned long)(offset) & 0x0000FFFF))
#define STORE_TO_HOB(registerNumber,offset)                                                (0x4f800000 | ((registerNumber) << 20) | ((unsigned long)(offset) & 0x00000FFF))

#define TRANSMIT_FROM_HOB(numberOfBytes,offset)																	(0x07000000 |  ((numberOfBytes) << 12) | ((unsigned long)(offset) & 0x00000FFF))
#define TRANSMIT_ALIGNED_FROM_HOB(numberOfBytes,offset)															(0x0fc00000 |  ((numberOfBytes) << 12) | ((unsigned long)(offset) & 0x00000FFF))
#define TRANSMIT_ALIGNED_FROM_HOB_WITH_LENGTH_FROM_REGISTER(lengthRegister,offset)								(0x0f400000 |  ((lengthRegister) << 12) | ((unsigned long)(offset) & 0x00000FFF))
#define TRANSMIT_FROM_REGISTER(registerNumber,numberOfBytes,offset)												(0x00000000 | ((registerNumber) << 24) | ((numberOfBytes) << 12) | ((unsigned long)(offset) & 0x00000FFF))
#define TRANSMIT_FROM_REGISTER_WITH_LENGTH_FROM_REGISTER(addressRegister,numOfBytesRegister,offset)				(0x80000000 | ((addressRegister) << 24) | ((numOfBytesRegister) << 20) | ((unsigned long)(offset) & 0x00000FFF))
#define TRANSMIT_FROM_HOB_WITH_LENGTH_FROM_REGISTER(numOfBytesRegister,offset)									(0x0F000000 | ((numOfBytesRegister) << 12) | ((unsigned long)(offset) & 0x00000FFF))
#define TRANSMIT_FROM_HOB_WITH_LENGTH_FROM_REGISTER_AND_OFFSET_FROM_REGISTER(numOfBytesRegister,offsetRegister)	(0x87800000 | ((numOfBytesRegister) << 20) | (offsetRegister))
#define TRANSMIT_ALIGNED_TAIL_FROM_HOB(numberOfBytes,offset)													(0x17800000 | ((numberOfBytes) << 12) | ((unsigned long)(offset) & 0x00000FFF)) 
#define TRANSMIT_ALIGNED_TAIL_FROM_REGISTER(addressRegister,numberOfBytes,offset)								(0x10800000 | ((addressRegister) << 24) | ((numberOfBytes) << 12) | ((unsigned long)(offset) & 0x00000FFF))
#define TRANSMIT_IMMEDIATE(numberOfBytes,immediateValue)														(0x30000000 | ((numberOfBytes) << 24) | ((unsigned long)(immediateValue) & 0x00FFFFFF))
#define TRANSMIT_FROM_REGISTER_WITH_IMMEDIATE_SIZE(numberOfBytes,registerNumber)								(0x38000000 | ((numberOfBytes) << 24) | (registerNumber)) 


#define SKIP(registerNumber,falseBit,bitNumber,offset)		                            (0x60000000 | ((registerNumber) << 24) | ((falseBit) << 15) | ((bitNumber) << 12) | ((unsigned long)(offset) & 0x00000FFF))
#define SKIP_ON_HOB(falseBit,bitNumber,offset)		                                    (0x67000000 | ((falseBit) << 15) | ((bitNumber) << 12) | ((unsigned long)(offset) & 0x00000FFF))
#define SKIP_IF_EQUAL_TO_IMMEDIATE(registerNumber,falseBit,immediateValue)		        (0x60800000 | ((registerNumber) << 24) | ((falseBit) << 19) | ((unsigned long)(immediateValue) & 0x00007FFF))
#define DO_SKIP_IF_EQUAL_TO_IMMEDIATE(registerNumber,immediateValue)					SKIP_IF_EQUAL_TO_IMMEDIATE(registerNumber,FALSE,immediateValue)
#define DO_SKIP_IF_NOT_EQUAL_TO_IMMEDIATE(registerNumber,immediateValue)				SKIP_IF_EQUAL_TO_IMMEDIATE(registerNumber,TRUE,immediateValue)



#define JUMP(byteAddress)  		                                                        (0x71000000 | (((unsigned long)(byteAddress) >> 2) & 0x007FFFFF))

// registerNumber + offset must be word aligned. The HW ignore the 2 LSBs of the addition result.
#define JUMP_FROM_REGISTER(registerNumber,offset)      		                            (0x79000000 | ((registerNumber) << 20) | ((offset) & 0x000FFFFF))

#define SHIFT_LEFT(registerNumber,shift)                                                (0xa8000000 | ((registerNumber) << 24) | ((unsigned long)(shift) & 0x0000001F))
#define SHIFT_RIGHT(registerNumber,shift)                                               (0xa0000000 | ((registerNumber) << 24) | ((unsigned long)(shift) & 0x0000001F))
#define SHIFT_RIGHT_FROM_REGISTER(destinationRegister,shiftRegister)                    (0xa0800000 | ((destinationRegister) << 24) | ((shiftRegister) << 20))
#define MASK_REGISTER(registerNumber,bytesToMask,maskPattern)                           (0x90000000 | ((registerNumber) << 24) | ((bytesToMask) << 20) | ((unsigned long)(maskPattern) & 0x000000FF))///to fix with other definition
#define ADD_REGISTER(destinationRegister,registerToAdd)                                 (0xc8000000 | ((destinationRegister) << 24) | ((registerToAdd) << 20))
#define ADD_REGISTER_AND_IMMEDIATE(destinationRegister,registerToAdd,immediateValue)    (0xc8800000 | ((destinationRegister) << 24) | ((registerToAdd) << 20)|((unsigned long)(immediateValue) & 0x00000FFF))
#define COPY(destinationRegister,sourceRegister)                                        (0xc0000000 | ((destinationRegister) << 24) | ((sourceRegister) << 20))

#define ENABLE_NEW_DELIA()																(0xb0000000)

//---------------------------------------------------------------------------------
//						Data Type Definition					
//---------------------------------------------------------------------------------

typedef enum
{
	DELIA_RECIPE_0_FIRST_COMMAND = 0,
	DELIA_RECIPE_0_ENABLE_PHY = DELIA_RECIPE_0_FIRST_COMMAND,
	DELIA_RECIPE_0_LOAD_DELIA_REGISTER_ASSERT_SENDER,
	DELIA_RECIPE_0_CREATE_INTERRUPT_TO_SENDER,
	DELIA_RECIPE_0_LOAD_DELIA_REGISTER_DELIAD_BASE_ADDRESS,
	DELIA_RECIPE_0_LOAD_DELIA_REGISTER_AUTO_GENERATED_TCRS1_BASE_ADDRESS,
	DELIA_RECIPE_0_WAIT_FOR_INPUT_TO_BE_READY,
	DELIA_RECIPE_0_SKIP_IF_ALL_TCRS_READY,
	DELIA_RECIPE_0_JUMP_TO_SEND_TCRS_FROM_DELIA_DESC,
	DELIA_RECIPE_0_SKIP_IF_TCR_SET1_IS_USED,
	DELIA_RECIPE_0_LOAD_DELIA_REGISTER_AUTO_GENERATED_TCRS2_BASE_ADDRESS,
#ifdef WORKAROUND_FOR_HW_BUG_IN_TCR_N_HE_LTF_MU_OFDMA
	DELIA_RECIPE_0_AUTO_TCRS_TRANSMIT_COMMON_TCR0,
	DELIA_RECIPE_0_AUTO_TCRS_WAIT_FOR_REST_TCRS_AND_CONTROL_WORD_TO_BE_READY,
	DELIA_RECIPE_0_AUTO_TCRS_TRANSMIT_COMMON_TCR1,
#else	
	DELIA_RECIPE_0_AUTO_TCRS_TRANSMIT_COMMON,
#endif //WORKAROUND_FOR_HW_BUG_IN_TCR_N_HE_LTF_MU_OFDMA
	DELIA_RECIPE_0_AUTO_TCRS_SET_TX_CONTROL_PER_USER_TCR_STATE,
	DELIA_RECIPE_0_AUTO_TCRS_LOAD_DELIA_REGISTER_PER_USER_TCRS_TOTAL_LENGTH,
	DELIA_RECIPE_0_AUTO_TCRS_TRANSMIT_PER_USER,
	DELIA_RECIPE_0_AUTO_TCRS_WAIT_FOR_CTRL_WORD_TO_BE_READY,
	DELIA_RECIPE_0_AUTO_TCRS_JUMP_TO_CTRL_WORD_HANDLING,
	DELIA_RECIPE_0_MANUAL_TCRS_SEND_TCR0,
	DELIA_RECIPE_0_MANUAL_TCRS_WAIT_FOR_TCR1_TO_CTRL_WORD_TO_BE_READY,
	DELIA_RECIPE_0_MANUAL_TCRS_SEND_TCR1,
	DELIA_RECIPE_0_MANUAL_TCRS_SET_TX_CONTROL_PER_USER_TCR_STATE,
	DELIA_RECIPE_0_MANUAL_TCRS_LOAD_DELIA_REGISTER_PER_USER_TCRS_TOTAL_LENGTH,
	DELIA_RECIPE_0_MANUAL_TCRS_TRANSMIT_PER_USER,
	DELIA_RECIPE_0_LOAD_DELIA_REGISTER_TX_CTRL_WORD_ADDRESS,
	DELIA_RECIPE_0_LOAD_DELIA_REGISTER_MPDU_HEADER_LENGTH,
	DELIA_RECIPE_0_MASK_DELIA_REGISTER_MPDU_HEADER_LENGTH,
	DELIA_RECIPE_0_SET_TX_CONTROL_DATA_STATE,
	DELIA_RECIPE_0_SKIP_IF_OLD_DELIA,
	DELIA_RECIPE_0_JUMP_TO_NEW_DELIA_HANDLING,
	DELIA_RECIPE_0_SKIP_IF_BEACON,
	DELIA_RECIPE_0_JUMP_TO_BEACON_HANDLING,
	DELIA_RECIPE_0_SKIP_IF_TF,
	DELIA_RECIPE_0_JUMP_TO_TF_HANDLING,
	DELIA_RECIPE_0_CHECK_IF_LENGTH_ZERO,
	DELIA_RECIPE_0_LENGTH_ZERO_JUMP_TO_POST_DATA,
	DELIA_RECIPE_0_LEGACY_PACKET_LOAD_DELIA_REGISTER_MPDU_COUNTER_VALUE,
	DELIA_RECIPE_0_LEGACY_PACKET_STORE_DELIA_REGISTER_MPDU_COUNTER_VALUE_TO_HOB,
	DELIA_RECIPE_0_LEGACY_PACKET_WAIT_FOR_MPDU_TO_BE_READY,
	DELIA_RECIPE_0_LEGACY_PACKET_LOAD_DELIA_REGISTER_MPDU_HEADER_ADDRESS,
	DELIA_RECIPE_0_LEGACY_PACKET_SHIFT_LEFT_HEADER_ADDRESS,
	DELIA_RECIPE_0_LEGACY_PACKET_SHIFT_RIGHT_HEADER_ADDRESS,
	DELIA_RECIPE_0_LEGACY_PACKET_TRANSMIT_HEADER,
	DELIA_RECIPE_0_LEGACY_PACKET_LOAD_DELIA_REGISTER_MPDU_PAYLOAD_LENGTH,
	DELIA_RECIPE_0_LEGACY_PACKET_MASK_MPDU_PAYLOAD_LENGTH,
	DELIA_RECIPE_0_LEGACY_PACKET_LOAD_DELIA_REGISTER_MPDU_PAYLOAD_ADDRESS,
	DELIA_RECIPE_0_LEGACY_PACKET_SHIFT_LEFT_PAYLOAD_ADDRESS,
	DELIA_RECIPE_0_LEGACY_PACKET_SHIFT_RIGHT_PAYLOAD_ADDRESS,
	DELIA_RECIPE_0_LEGACY_PACKET_TRANSMIT_PAYLOAD,
	DELIA_RECIPE_0_LEGACY_PACKET_TRANSMIT_CRC,
	DELIA_RECIPE_0_LEGACY_JUMP_TO_POST_DATA,
	DELIA_RECIPE_0_BEACON_CALCULATE_FIRST_ELEMENT_DESCRIPTOR_OFFSET,
	DELIA_RECIPE_0_BEACON_CALCULATE_COUNTER,
	DELIA_RECIPE_0_BEACON_STORE_MPDU_COUNTER_INTO_HOB,
	DELIA_RECIPE_0_BEACON_WAIT_FOR_BEACON_TO_BE_READY,
	DELIA_RECIPE_0_BEACON_START_OF_LOOP,
	DELIA_RECIPE_0_BEACON_CALCULATE_ELEMENT_DESCRIPTOR_ADDRESS = DELIA_RECIPE_0_BEACON_START_OF_LOOP,
	DELIA_RECIPE_0_BEACON_LOAD_R5_WITH_ELEMENT_LENGTH,
	DELIA_RECIPE_0_BEACON_MASK_ELEMENT_LENGTH,
	DELIA_RECIPE_0_BEACON_SKIP_IF_REGULAR_ELEMENT,
	DELIA_RECIPE_0_BEACON_JUMP_TO_SPECIAL_ELEMENT,
	DELIA_RECIPE_0_BEACON_LOAD_R6_WITH_ELEMENT_ADDRESS,
	DELIA_RECIPE_0_BEACON_SHIFT_LEFT_ELEMENT_ADDRESS,
	DELIA_RECIPE_0_BEACON_SHIFT_RIGHT_ELEMENT_ADDRESS,
	DELIA_RECIPE_0_BEACON_TRANSMIT_ELEMENT,
	DELIA_RECIPE_0_BEACON_REGULAR_ELEMENT_JUMP_TO_CHECK_IF_LAST,
	DELIA_RECIPE_0_BEACON_SKIP_IF_TIMESTAMP_ELEMENT,
	DELIA_RECIPE_0_BEACON_JUMP_TO_TIM_ELEMENT,
	DELIA_RECIPE_0_BEACON_LOAD_R4_WITH_VAP_INDEX,
	DELIA_RECIPE_0_BEACON_STORE_VAP_INDEX_INTO_HOB,
	DELIA_RECIPE_0_BEACON_TRANSMIT_TSF_ELEMENT,
	DELIA_RECIPE_0_BEACON_TSF_ELEMENT_JUMP_TO_CHECK_IF_LAST,
	DELIA_RECIPE_0_BEACON_TRANSMIT_CONSTANT_BYTES_OF_TIM_ELEMENT,
	DELIA_RECIPE_0_BEACON_LOAD_R4_WITH_TIM_BITMAP_PARAMETERS,
	DELIA_RECIPE_0_BEACON_MASK_BITMAP_LENGTH, 
	DELIA_RECIPE_0_BEACON_SET_R5_WITH_BITMAP_CTRL_OFFSET,  //
	DELIA_RECIPE_0_BEACON_MASK_OF_BITMAP_CTRL_OFFSET,  //
	DELIA_RECIPE_0_BEACON_TRANSMIT_FROM_HOB_ACCORDING_TO_LENGTH,  //
	DELIA_RECIPE_0_BEACON_LOAD_R4_WITH_TIM_IE_PARAMETERS,
	DELIA_RECIPE_0_BEACON_COPY_R4_TO_R5_TO_SAVE_OFFSET,
	DELIA_RECIPE_0_BEACON_MASK_TIM_IE_LENGTH,
	DELIA_RECIPE_0_BEACON_SHIFT_TIM_IE_OFFSET,
	DELIA_RECIPE_0_BEACON_MASK_TIM_IE_OFFSET,
	
	DELIA_RECIPE_0_BEACON_ADD_BASE_ADDRESS_TO_BITMAP_OFFSET,
	DELIA_RECIPE_0_BEACON_TRANSMIT_TIM_ELEMENT,
	DELIA_RECIPE_0_BEACON_CHECK_IF_LAST,
	DELIA_RECIPE_0_BEACON_JUMP_TO_STRAT_OF_LOOP,
	DELIA_RECIPE_0_BEACON_TRANSMIT_CRC,
	DELIA_RECIPE_0_BEACON_JUMP_TO_POST_DATA,
	DELIA_RECIPE_0_TF_LOAD_DELIA_REGISTER_MPDU_COUNTER_VALUE,
	DELIA_RECIPE_0_TF_STORE_DELIA_REGISTER_MPDU_COUNTER_VALUE_TO_HOB,
	DELIA_RECIPE_0_TF_WAIT_FOR_MPDU_TO_BE_READY,
	DELIA_RECIPE_0_TF_LOAD_DELIA_REGISTER_PAYLOAD_LENGTH,
	DELIA_RECIPE_0_TF_MASK_PAYLOAD_LENGTH,
	DELIA_RECIPE_0_TF_LOAD_DELIA_REGISTER_PADDING_LENGTH,
	DELIA_RECIPE_0_TF_SHIFT_RIGHT_PADDING_LENGTH,
	DELIA_RECIPE_0_TF_MASK_PADDING_LENGTH,
	DELIA_RECIPE_0_TF_SKIP_ON_PHY_MODE,
	DELIA_RECIPE_0_TF_JUMP_TO_SKIP_FLOW_CONTROL,
	DELIA_RECIPE_0_TF_CALCULATE_LENGTH_HEADER_AND_CRC,
	DELIA_RECIPE_0_TF_CALCULATE_TOTAL_LENGTH,
	DELIA_RECIPE_0_TF_STORE_TOTAL_LENGTH_TO_HOB,
	DELIA_RECIPE_0_TF_TRANSMIT_DELIMITER,
	DELIA_RECIPE_0_TF_SET_TX_CONTROL_SKIP_FLOW_CONTROL,
	DELIA_RECIPE_0_TF_LOAD_DELIA_REGISTER_PAYLOAD_ADDRESS,
	DELIA_RECIPE_0_TF_SHIFT_LEFT_PAYLOAD_ADDRESS,
	DELIA_RECIPE_0_TF_SHIFT_RIGHT_PAYLOAD_ADDRESS,
	DELIA_RECIPE_0_TF_TRANSMIT_PAYLOAD,
	DELIA_RECIPE_0_TF_TRANSMIT_PADDING,
	DELIA_RECIPE_0_TF_TRANSMIT_CRC,
	DELIA_RECIPE_0_TF_JUMP_TO_POST_DATA,
	DELIA_RECIPE_0_WAIT_FOR_DURATION_REG_READY_EVENT,	
#ifdef WORKAROUND_FOR_HW_BUG_IN_DELIA_AUTOFILL
	DELIA_RECIPE_0_POST_DATA_LOAD_DELIA_REGISTER_MPDU_COUNTER_VALUE,
	DELIA_RECIPE_0_POST_DATA_STORE_DELIA_REGISTER_MPDU_COUNTER_VALUE_TO_HOB,
#endif //WORKAROUND_FOR_HW_BUG_IN_DELIA_AUTOFILL
	DELIA_RECIPE_0_NEW_DELIA,
#ifdef WORKAROUND_FOR_HW_BUG_IN_DELIA_AUTOFILL
	DELIA_RECIPE_0_POST_DATA_WAIT_FOR_MAC_PHY_IF_BUS_STATE,
#endif //WORKAROUND_FOR_HW_BUG_IN_DELIA_AUTOFILL
	DELIA_RECIPE_0_SKIP_IF_NO_RCR,
	DELIA_RECIPE_0_JUMP_TO_END_RECIPE,
	DELIA_RECIPE_0_LOAD_DELIA_REGISTER_AUTO_GENERATED_RCRS_BASE_ADDRESS,
	DELIA_RECIPE_0_AUTO_RCRS_SET_TX_CONTROL_COMMON_RCR_STATE,
	DELIA_RECIPE_0_AUTO_RCRS_TRANSMIT_COMMON,
	DELIA_RECIPE_0_AUTO_RCRS_SET_TX_CONTROL_PER_USER_RCR_STATE,
	DELIA_RECIPE_0_AUTO_RCRS_SET_RCRS_POINTER_TO_PER_USER_SECTION,
	DELIA_RECIPE_0_AUTO_RCRS_LOAD_DELIA_REGISTER_RCR_PER_USER_INDEX,
	DELIA_RECIPE_0_AUTO_RCRS_TRANSMIT_PER_USER,
	DELIA_RECIPE_0_AUTO_RCRS_INCREMENT_RCRS_POINTER_TO_NEXT_PER_USER,
	DELIA_RECIPE_0_AUTO_RCRS_INCREMENT_DELIA_REGISTER_RCR_PER_USER_INDEX,
	DELIA_RECIPE_0_AUTO_RCRS_SKIP_ON_INDEX,
	DELIA_RECIPE_0_AUTO_RCRS_JUMP_TO_TRANSMIT_PER_USER_RCR,
	DELIA_RECIPE_0_END_RECIPE,
	DELIA_RECIPE_0_RECIPE_LAST_COMMAND,
} DeliaRecipe0_e;


typedef enum
{
	DELIA_INSTRUCTION_AUTO_RESPONSE_FIRST_COMMAND = 0,
	DELIA_INSTRUCTION_AUTO_RESPONSE_START_AUTO_FRAME = DELIA_INSTRUCTION_AUTO_RESPONSE_FIRST_COMMAND,
	DELIA_INSTRUCTION_AUTO_RESPONSE_CREATE_TCR0_ADDR,
	DELIA_INSTRUCTION_AUTO_RESPONSE_SEND_COMMON_TCRS,
	DELIA_INSTRUCTION_AUTO_RESPONSE_SET_STATE_TO_PER_USER_TCR,
	DELIA_INSTRUCTION_AUTO_RESPONSE_LOAD_R3_WITH_PHY_MODE,
	DELIA_INSTRUCTION_AUTO_RESPONSE_MASK_R3_WITH_NONE_HE_VHT,
	DELIA_INSTRUCTION_AUTO_RESPONSE_WAIT_FOR_TCR_LENGTH_READY,
	DELIA_INSTRUCTION_AUTO_RESPONSE_LOAD_R4_WITH_FRAME_LEN,
	DELIA_INSTRUCTION_AUTO_RESPONSE_LOAD_R5_WITH_FRAME_LEN,
	DELIA_INSTRUCTION_AUTO_RESPONSE_SKIP_IF_NOT_HT_OR_VHT,
	DELIA_INSTRUCTION_AUTO_RESPONSE_ADD_THE_DELIMITER_SIZE,
	DELIA_INSTRUCTION_AUTO_RESPONSE_SEND_PER_USER_TCR_0_TO_31,
	DELIA_INSTRUCTION_AUTO_RESPONSE_SEND_PER_USER_TCR_32_TO_127,
	DELIA_INSTRUCTION_AUTO_RESPONSE_SET_STATE_TO_DATA,
	DELIA_INSTRUCTION_AUTO_RESPONSE_LOAD_R1_WITH_FRAME_POINTER,
	DELIA_INSTRUCTION_AUTO_RESPONSE_LOAD_R2_WITH_FRAME_LENGTH,
	DELIA_INSTRUCTION_AUTO_RESPONSE_WAIT_FOR_FRAME_READY,
	DELIA_INSTRUCTION_AUTO_RESPONSE_UPDATE_DELIMITER,
	DELIA_INSTRUCTION_AUTO_RESPONSE_SKIP_DELIMITER_IF_NOT_HT_OR_VHT,
	DELIA_INSTRUCTION_AUTO_RESPONSE_TRANSMIT_DELIMITER,
	DELIA_INSTRUCTION_AUTO_RESPONSE_INIT_CRC_AGAIN,
	DELIA_INSTRUCTION_AUTO_RESPONSE_TRANSMIT_CONTROL,
	DELIA_INSTRUCTION_AUTO_RESPONSE_WAIT_FOR_DURATION,
	DELIA_INSTRUCTION_AUTO_RESPONSE_SKIP_IF_TX_BASED_DURATION,
	DELIA_INSTRUCTION_AUTO_RESPONSE_JUMP_TO_RX_BASED_DURATION,
	DELIA_INSTRUCTION_AUTO_RESPONSE_TRANSMIT_TX_BASED_DURATION,
	DELIA_INSTRUCTION_AUTO_RESPONSE_JUMP_TO_FRAME_TRANSMISSION,
	DELIA_INSTRUCTION_AUTO_RESPONSE_TRANSMIT_RX_BASED_DURATION,
	DELIA_INSTRUCTION_AUTO_RESPONSE_TRANSMIT_HEADER_AND_FRAME,
	DELIA_INSTRUCTION_AUTO_RESPONSE_TRANSMIT_CRC,
	DELIA_INSTRUCTION_AUTO_RESPONSE_END_RECIPE,
	DELIA_INSTRUCTION_AUTO_RESPONSE_LAST_COMMAND
} DeliaRecipeAutoResponse_e;


typedef enum
{
	DELIA_INSTRUCTION_AUTO_RESPONSE_MU_UNICAST_FIRST_COMMAND = 0,
	DELIA_INSTRUCTION_AUTO_RESPONSE_MU_UNICAST_START_AUTO_FRAME = DELIA_INSTRUCTION_AUTO_RESPONSE_MU_UNICAST_FIRST_COMMAND,
	DELIA_INSTRUCTION_AUTO_RESPONSE_MU_UNICAST_CREATE_TCR0_ADDR,
	DELIA_INSTRUCTION_AUTO_RESPONSE_MU_UNICAST_SEND_COMMON_TCRS,
	DELIA_INSTRUCTION_AUTO_RESPONSE_MU_UNICAST_SET_STATE_TO_PER_USER_TCR,
	DELIA_INSTRUCTION_AUTO_RESPONSE_MU_UNICAST_WAIT_FOR_TCR_LENGTH_READY,
	DELIA_INSTRUCTION_AUTO_RESPONSE_MU_UNICAST_TCRS_LOAD_DELIA_REGISTER_PER_USER_TCRS_TOTAL_LENGTH,
	DELIA_INSTRUCTION_AUTO_RESPONSE_MU_UNICAST_TCRS_TRANSMIT_PER_USER,
	DELIA_INSTRUCTION_AUTO_RESPONSE_MU_UNICAST_SET_STATE_TO_DATA,
	DELIA_INSTRUCTION_AUTO_RESPONSE_MU_UNICAST_ENABLE_NEW_DELIA,
	DELIA_INSTRUCTION_AUTO_RESPONSE_MU_UNICAST_END_RECIPE,
	DELIA_INSTRUCTION_AUTO_RESPONSE_MU_UNICAST_LAST_COMMAND
} DeliaRecipeMUUnicastAutoResponse_e;

typedef enum
{
	DELIA_INSTRUCTION_BFEE_REPORT_FIRST_COMMAND = 0,
	DELIA_INSTRUCTION_BFEE_REPORT_ENABLE_BUS_STATE = DELIA_INSTRUCTION_BFEE_REPORT_FIRST_COMMAND,
	DELIA_INSTRUCTION_BFEE_REPORT_READ_TX_RECIPE_BASE_ADDR_FROM_HOB, 
	DELIA_INSTRUCTION_BFEE_REPORT_TRANSMIT_TCR0_TCR1,
	DELIA_INSTRUCTION_BFEE_REPORT_SET_STATE_TO_PER_USER_TCR,
	DELIA_INSTRUCTION_BFEE_REPORT_WAIT_FOR_FRAME_LENGTH_READY,
	DELIA_INSTRUCTION_BFEE_READ_TCR_FRAME_LENGHT_FROM_HOB,
	DELIA_INSTRUCTION_BFEE_READ_PHY_MODE_FROM_HOB,
	DELIA_INSTRUCTION_BFEE_MASK_R3_WITH_NONE_HE_VHT,
	DELIA_INSTRUCTION_BFEE_SKIP_IF_NOT_HE_OR_VHT, 
	DELIA_INSTRUCTION_BFEE_ADD_THE_DELIMITER_SIZE, 
	DELIA_INSTRUCTION_BFEE_SEND_PER_USER_TCR_0_TO_31,
#ifdef WORKAROUND_FOR_BFEE_SUB_BAND_HW_BUG
	DELIA_INSTRUCTION_BFEE_REPORT_SKIP_IF_HE,
	DELIA_INSTRUCTION_BFEE_REPORT_JUMP_TO_NONE_HE,
	DELIA_INSTRUCTION_BFEE_LOAD_FIFTH_BYTE_FROM_COMMON_TCR,
	DELIA_INSTRUCTION_BFEE_SHIF_BW_FIELD_TO_LSB,
	DELIA_INSTRUCTION_BFEE_MASK_BW_BITS,
	DELIA_INSTRUCTION_BFEE_CALC_TABLE_OFFSET_IN_HOB,
	DELIA_INSTRUCTION_BFEE_LOAD_REGISTER_WITH_BYTE_LENGTH,
	DELIA_INSTRUCTION_BFEE_TRANSMIT_ONE_FIFTH_BYTE_FROM_USER_TCR,
	DELIA_INSTRUCTION_BFEE_REPORT_TRANSMIT_FIRST_USER_TCR_WITH_WA,
	DELIA_INSTRUCTION_BFEE_REPORT_JUMP_TO_CHANGE_BUS_STATE,
#endif 
	DELIA_INSTRUCTION_BFEE_REPORT_TRANSMIT_FIRST_USER_TCR,
	DELIA_INSTRUCTION_BFEE_CHANGE_BUS_STATE_TO_DATA_AND_CRC,
	DELIA_INSTRUCTION_BFEE_READ_AGAIN_TCR_FRAME_LENGHT_FROM_HOB,
	DELIA_INSTRUCTION_BFEE_UPDATE_DELIMITER,
	DELIA_INSTRUCTION_BFEE_SKIP_DELIMITER_IF_NOT_HE_OR_VHT_2,
	DELIA_INSTRUCTION_BFEE_TRANSMIT_DELIMITER,
	DELIA_INSTRUCTION_BFEE_CRC_INIT,
	DELIA_INSTRUCTION_BFEE_WAIT_FOR_DURATION_READY,
	DELIA_INSTRUCTION_BFEE_WAIT_FOR_SECTION_A_READY,
	DELIA_INSTRUCTION_BFEE_REPORT_READ_TXC_SECTION_A_BASE_ADDR_FROM_HOB,
	DELIA_INSTRUCTION_BFEE_REPORT_TRANSMIT_2_BYTES_TXC_SECTION_A, 
	DELIA_INSTRUCTION_BFEE_REPORT_TRANSMIT_DURATION,
	DELIA_INSTRUCTION_INCREMENT_SECTION_A_BASE_ADDR, 
	DELIA_INSTRUCTION_BFEE_REPORT_TRANSMIT_18_BYTES_TXC_SECTION_A, 
	DELIA_INSTRUCTION_BFEE_REPORT_READ_VAP_TID_PTR_FROM_HOB, 
	DELIA_INSTRUCTION_BFEE_REPORT_READ_FIRST_WORD_FROM_VAP_DB, 
	DELIA_INSTRUCTION_BF_REPORT_SHIFT_SEQUENCE_NUMBER_WITH_FRAGEMENT_BITS,
	DELIA_INSTRUCTION_BFEE_TRANSMIT_SEQUENCE_CONTROL, 
	DELIA_INSTRUCTION_BFEE_INCREMENT_SEQUENCE_NUMBER,
	DELIA_INSTRUCTION_BFEE_WRAP_AROUND_SEQUENCE_NUMBER, 
	DELIA_INSTRUCTION_BFEE_SHIFT_RIGHT_SEQUENCE_NUMBER, 
	DELIA_INSTRUCTION_BFEE_SAVE_SEQUENCE_NUMBER,
	DELIA_INSTRUCTION_BFEE_WAIT_FOR_SECTION_B_READY, 
	DELIA_INSTRUCTION_BFEE_TRANSMIT_ACTION_CATEGORY,
	DELIA_INSTRUCTION_BFEE_TRANSMIT_MIMO_CTRL, 
	DELIA_INSTRUCTION_BFEE_READ_BFEE_PHY_MODE_FROM_HOB,
	DELIA_INSTRUCTION_BFEE_REPORT_MASK_IS_PHY_MODE_HT,
	DELIA_INSTRUCTION_BFEE_SKIP_IF_NOT_HT, 
	DELIA_INSTRUCTION_BFEE_JUMP_TO_SECTION_B, 
	DELIA_INSTRUCTION_BFEE_READ_BFEE_STA_FROM_HOB, 
	DELIA_INSTRUCTION_BFEE_SHIFT_RIGHT_VAP_INDX,
	DELIA_INSTRUCTION_BFEE_STORE_VAP_INDEX_INTO_HOB,
	DELIA_INSTRUCTION_BFEE_TRANSMIT_TSF_ELEMENT,
	DELIA_INSTRUCTION_BFEE_REPORT_READ_TXC_SECTION_B_BASE_ADDR_FROM_HOB, 
	DELIA_INSTRUCTION_BFEE_REPORT_READ_B_SECTION_LENGTH_FROM_HOB, 
	DELIA_INSTRUCTION_BFEE_REPORT_TRANSMIT_B_SECTION, 
	DELIA_INSTRUCTION_BFEE_REPORT_READ_TXC_SECTION_C_BASE_ADDR_FROM_HOB,
	DELIA_INSTRUCTION_BFEE_REPORT_READ_C_SECTION_LENGTH_FROM_HOB, 
	DELIA_INSTRUCTION_BFEE_WAIT_FOR_SECTION_C_READY,
	DELIA_INSTRUCTION_BFEE_READ_BFEE_SM,
	DELIA_INSTRUCTION_BFEE_REPORT_MASK_BFR_SUCCESS,
	DELIA_INSTRUCTION_BFEE_SKIP_IF_ERROR,
	DELIA_INSTRUCTION_BFEE_JUMP_TO_READ_SECTION_FROM_TXCB,
	DELIA_INSTRUCTION_BFEE_TRANSMIT_ERROR_VALUE,
	DELIA_INSTRUCTION_BFEE_JUMP_TO_READ_CRC,
	DELIA_INSTRUCTION_BFEE_REPORT_TRANSMIT_C_SECTION, 
	DELIA_INSTRUCTION_BFEE_READ_CRC,
	DELIA_INSTRUCTION_BFEE_SKIP_IF_CRC_VALID,
	DELIA_INSTRUCTION_BFEE_INIT_CRC_IF_NOT_VALID,
	DELIA_INSTRUCTION_BFEE_TRANSMIT_CRC,
	DELIA_INSTRUCTION_BFEE_END_RECIPE,
	DELIA_INSTRUCTION_BFEE_REPORT_LAST_COMMAND
} DeliaRecipeBfReport_e;

typedef struct _DeliaRecipes_t
{
	uint32 DeliaRecipe0[DELIA_RECIPE_0_RECIPE_LAST_COMMAND];
	uint32 DeliaRecipeSUAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_LAST_COMMAND]; //Delia Recipe for SU auto response
	uint32 DeliaRecipeMUBroadcastAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_LAST_COMMAND]; //Delia Recipe for MU Broadcast auto response
	uint32 DeliaRecipeMUUnicastAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_MU_UNICAST_LAST_COMMAND]; //Delia Recipe for MU Unicast auto response
	uint32 DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_REPORT_LAST_COMMAND]; //Delia Recipe for BF report 
} DeliaRecipes_t;

#endif /*DELIA_RECIPES_H_ */

