/***************************************************************
 File:		DeliaRecipes.c
 Module:	  	Delia Recipes
 Purpose: 	holds all the implementation of the construction of the Delia recipes
 Description:  This module contains the implementation of the construction of the Delia
 			recipes and the recipes pointers table
***************************************************************/

/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "DeliaRecipes_Api.h"
#include "DeliaRecipes.h"
#include "ShramNewDeliaRecipes.h"
#include "protocol.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"

#if defined (ENET_INC_UMAC)
#include "PhyDriver_API.h"
#endif

#ifdef DYNAMIC_RTS_ISSUE_WORKAROUND
#include "RxHandler.h"
#endif //DYNAMIC_RTS_ISSUE_WORKAROUND

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_TX_HW_ACCELERATORS
#define LOG_LOCAL_FID 2



/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void deliaConstructRecipe0(void);
static void deliaConstructLegacyPacketRecipe(void);
static void deliaConstructAggregationRecipe(void);
static void deliaConstructBeaconRecipe(void);
static void deliaConstructLegacyVhtRecipe(void);
static void deliaConstructAckRecipe(void);
static void deliaConstructBaRecipe(void);
static void deliaConstructCtsAutoResponseRecipe(void);
static void deliaConstructBfReportRecipe(void);
static void deliaInitializeRecipesLookupTable(void);
/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
static uint32* DeliaRecipesLookUpTableDefines[DELIA_RECIPE_LOOKUP_TABLE_SIZE] = 
{
	&DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_PACKET_FIRST_COMMAND],      /* data, none VHT, none aggregate */
	NULL,													           /* not used */
	&DeliaRecipe0[DELIA_INSTRUCTION_BEACON_FIRST_COMMAND],             /* beacon */
	NULL,                                                              /* not used */
	&DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_VHT_PACKET_FIRST_COMMAND],  /* data, VHT, none aggregate */
	NULL,                                                              /* not used */
	NULL,                                                              /* not used */
	NULL,                                                              /* not used */
	&DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_FIRST_COMMAND],        /* data, aggregate, none VHT */
	NULL,                                                              /* not used */
	NULL,                                                              /* not used */
	NULL,                                                              /* not used */
	&DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_FIRST_COMMAND],        /* data, aggregate, VHT */
	NULL,                                                              /* not used */
	NULL,                                                              /* not used */
	NULL,                                                              /* not used */
};

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

deliaConstructLegacyPacketRecipe  


Description:
------------
construct the legacy packet recipe

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void deliaConstructLegacyPacketRecipe()
{
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_PACKET_LOAD_R2_WITH_AUTO_FILL_START] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(AUTO_FIIL_INDICATION_REGISTER,AUTO_FILL_START_INDICATION);
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_PACKET_START_AUTO_FILL] = STORE_TO_HOB(AUTO_FIIL_INDICATION_REGISTER,HOB_REGISTER_AUTO_FILL_INDICATION_OFFSET);
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_PACKET_LOAD_R1_WITH_MPDU_COUNTER] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(DELIA_MPDU_COUNTER_REGISTER,ONE_MPDU_TO_COUNTER);
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_PACKET_STROE_MPDU_COUNTER_INTO_HOB] = STORE_TO_HOB(DELIA_MPDU_COUNTER_REGISTER,HOB_REGISTER_DELIA_COUNTER_OFFSET);		
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_PACKET_WAIT_FOR_MPDU_TO_BE_READY] = WAIT_FOR_MPDU_OR_BEACON_TO_BE_READY;		
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_PACKET_LOAD_R3_WITH_HEADER_LENGTH] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(HEADER_LENGTH_REGISTER,DELIA_DESCRIPTOR_REGISTER,HEADER_LENGTH_OFFSET);		
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_PACKET_LOAD_R4_WITH_HEADER_ADDRESS] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(HEADER_ADDRESS_REGISTER,DELIA_DESCRIPTOR_REGISTER,LEGACY_HEADER_ADDRESS_OFFSET);		
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_PACKET_TRANSMIT_HEADER] = TRANSMIT_FROM_REGISTER_WITH_LENGTH_FROM_REGISTER(HEADER_ADDRESS_REGISTER,HEADER_LENGTH_REGISTER,HEADER_TRANSMISSION_OFFSET);		
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_PACKET_LOAD_R5_WITH_PAYLOAD_LENGTH] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(PAYLOAD_LENGTH_REGISTER,DELIA_DESCRIPTOR_REGISTER,LEGACY_PAYLOAD_LENGTH_AND_ZLD_COUNT_OFFSET);		
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_PACKET_MASK_PAYLOAD_LENGTH] = MASK_REGISTER(PAYLOAD_LENGTH_REGISTER,PAYLOAD_LENGTH_BYTES_TO_MASK,PAYLOAD_LENGTH_MASK_PATTERN);		
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_PACKET_LOAD_R6_WITH_PAYLOAD_ADDRESS] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(PAYLOAD_ADDRESS_REGISTER,DELIA_DESCRIPTOR_REGISTER,LEGACY_PAYLOAD_ADDRESS_AND_LAST_BIT_OFFSET);		
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_PACKET_MASK_PAYLOAD_ADDRESS] = MASK_REGISTER(PAYLOAD_ADDRESS_REGISTER,PAYLOAD_ADDRESS_BYTES_TO_MASK,PAYLOAD_ADDRESS_MASK_PATTERN);		
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_PACKET_TRANSMIT_PAYLOAD] = TRANSMIT_FROM_REGISTER_WITH_LENGTH_FROM_REGISTER(PAYLOAD_ADDRESS_REGISTER,PAYLOAD_LENGTH_REGISTER,PAYLOAD_TRANSMISSION_OFFSET);		
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_PACKET_TRANSMIT_CRC] = TRANSMIT_ALIGNED_TAIL_FROM_HOB(CRC_LENGTH,HOB_REGISTER_CRC_OFFSET);	
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_PACKET_LOAD_R2_WITH_AUTO_FILL_END] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(AUTO_FIIL_INDICATION_REGISTER,AUTO_FILL_END_INDICATION); 	
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_PACKET_END_AUTO_FILL] = STORE_TO_HOB(AUTO_FIIL_INDICATION_REGISTER,HOB_REGISTER_AUTO_FILL_INDICATION_OFFSET);
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_PACKET_END_RECIPE] = END_RECIPE;
}

/**********************************************************************************

deliaConstructAggregationRecipe  


Description:
------------
construct the aggregation(data) recipe

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void deliaConstructAggregationRecipe()
{
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_LOAD_R3_WITH_HEADER_LENGTH] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(HEADER_LENGTH_REGISTER,DELIA_DESCRIPTOR_REGISTER,HEADER_LENGTH_OFFSET);

	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_COPY_R3_TO_R2_SAVE_START_DELIMITERS_COUNT] = COPY(START_DELIMITERS_COUNTER_REGISTER, HEADER_LENGTH_REGISTER);
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_CALCULATE_START_DELIMITERS_SHIFT_RIGHT] = SHIFT_RIGHT(START_DELIMITERS_COUNTER_REGISTER, START_DELIMITERS_COUNTER_OFFSET_IN_HEADER_LENGTH_AND_START_DELIMITERS_COUNT_REGISTER);
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_CALCULATE_START_DELIMITERS_SHIFT_LEFT] = SHIFT_LEFT(START_DELIMITERS_COUNTER_REGISTER, (SHIFT_WORDS_TO_BYTES + SHIFT_WORD_TO_8_WORDS));
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_TRANSMIT_START_DELIMITERS] = TRANSMIT_ALIGNED_FROM_HOB_WITH_LENGTH_FROM_REGISTER(START_DELIMITERS_COUNTER_REGISTER, HOB_REGISTER_BAD_DELIMITER_OFFSET);

	/* The next 2 commands are done in order to not have any operations after checking the last bit - just jump to start  of loop : the address of the first MPDU descriptor is set to be the (address of  first descriptor - descriptor  size)
	     and the counter is set to 0 so in the loop the descriptor size is added and the counter is increased by one and the first MPDU can be handled the same as all MPDUs */
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_CALCULATE_FIRST_MPDU_DESCRIPTOR_OFFSET] = ADD_REGISTER_AND_IMMEDIATE(DELIA_MPDU_DESCRIPTOR_ADDREES_REGISTER,DELIA_DESCRIPTOR_REGISTER,AGGREGATION_FIRST_MPDU_DESCRIPTOR_OFFSET);	
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_LOAD_MPDU_COUNTER_R1_WITH_ZERO] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(DELIA_MPDU_COUNTER_REGISTER,MPDU_COUNTER_ZERO);
		
	/* MPDUs loop */	
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_LOAD_R2_WITH_AUTO_FILL_START] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(AUTO_FIIL_INDICATION_REGISTER,AUTO_FILL_START_INDICATION);
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_START_AUTO_FILL] = STORE_TO_HOB(AUTO_FIIL_INDICATION_REGISTER,HOB_REGISTER_AUTO_FILL_INDICATION_OFFSET);	
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_CALCULATE_MPDU_DESCRIPTOR_ADDRESS] = ADD_REGISTER_AND_IMMEDIATE(DELIA_MPDU_DESCRIPTOR_ADDREES_REGISTER,DELIA_MPDU_DESCRIPTOR_ADDREES_REGISTER,MPDU_DESCRIPTOR_SIZE);
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_CALCULATE_MPDU_COUNTER] = ADD_REGISTER_AND_IMMEDIATE(DELIA_MPDU_COUNTER_REGISTER,DELIA_MPDU_COUNTER_REGISTER,ONE_MPDU_TO_COUNTER);			
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_STROE_MPDU_COUNTER_INTO_HOB] = STORE_TO_HOB(DELIA_MPDU_COUNTER_REGISTER,HOB_REGISTER_DELIA_COUNTER_OFFSET);	
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_WAIT_FOR_MPDU_TO_BE_READY] = WAIT_FOR_MPDU_OR_BEACON_TO_BE_READY;	
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_LOAD_R5_WITH_PAYLOAD_LENGTH_AND_ZLD] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(PAYLOAD_LENGTH_REGISTER,DELIA_MPDU_DESCRIPTOR_ADDREES_REGISTER,AGGREGATE_PAYLOAD_LENGTH_AND_ZLD_COUNT_OFFSET);
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_COPY_R5_TO_R2_SAVE_ZLD_COUNT] = COPY(ZLD_COUNTER_REGISTER,PAYLOAD_LENGTH_REGISTER);
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_MASK_PAYLOAD_LENGTH] = MASK_REGISTER(PAYLOAD_LENGTH_REGISTER,PAYLOAD_LENGTH_BYTES_TO_MASK,PAYLOAD_LENGTH_MASK_PATTERN);
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_CALCULATE_LENGTH_HEADER_AND_CRC] = ADD_REGISTER_AND_IMMEDIATE(MPDU_TOTAL_LENGTH_REGISTER,HEADER_LENGTH_REGISTER,CRC_LENGTH);	
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_CALCULATE_TOTAL_LENGTH] = ADD_REGISTER(MPDU_TOTAL_LENGTH_REGISTER,PAYLOAD_LENGTH_REGISTER);
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_STORE_TOTAL_LENGTH_TO_HOB] = STORE_TO_HOB(MPDU_TOTAL_LENGTH_REGISTER,HOB_REGISTER_DELIMITER_LENGTH_OFFSET);	
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_TRANSMIT_DELIMITER] = TRANSMIT_ALIGNED_FROM_HOB(DELIMITER_LENGTH,HOB_REGISTER_DELIMITER_RESULT_OFFSET);
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_RESET_CRC] = RESET_CRC;
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_LOAD_R4_WITH_HEADER_ADDRESS] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(HEADER_ADDRESS_REGISTER,DELIA_MPDU_DESCRIPTOR_ADDREES_REGISTER,AGGREAGTE_HEADER_ADDRESS_OFFSET);
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_TRANSMIT_HEADER] = TRANSMIT_FROM_REGISTER_WITH_LENGTH_FROM_REGISTER(HEADER_ADDRESS_REGISTER,HEADER_LENGTH_REGISTER,HEADER_TRANSMISSION_OFFSET);
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_LOAD_R6_WITH_PAYLOAD_ADDRESS] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(PAYLOAD_ADDRESS_REGISTER,DELIA_MPDU_DESCRIPTOR_ADDREES_REGISTER,AGGREGATE_PAYLOAD_ADDRESS_AND_LAST_BIT_OFFSET);
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_MASK_PAYLOAD_ADDRESS] = MASK_REGISTER(PAYLOAD_ADDRESS_REGISTER,PAYLOAD_ADDRESS_BYTES_TO_MASK,PAYLOAD_ADDRESS_MASK_PATTERN);
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_TRANSMIT_PAYLAOD] = TRANSMIT_FROM_REGISTER_WITH_LENGTH_FROM_REGISTER(PAYLOAD_ADDRESS_REGISTER,PAYLOAD_LENGTH_REGISTER,PAYLOAD_TRANSMISSION_OFFSET);
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_TRANSMIT_CRC] = TRANSMIT_ALIGNED_TAIL_FROM_HOB(CRC_LENGTH,HOB_REGISTER_CRC_OFFSET);
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_LOAD_R2_WITH_AUTO_FILL_END] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(AUTO_FIIL_INDICATION_REGISTER,AUTO_FILL_END_INDICATION);
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_END_AUTO_FILL] = STORE_TO_HOB(AUTO_FIIL_INDICATION_REGISTER,HOB_REGISTER_AUTO_FILL_INDICATION_OFFSET);	
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_CALCULATE_ZLD_SHIFT_RIGHT] = SHIFT_RIGHT(ZLD_COUNTER_REGISTER,ZLD_COUNTER_OFFSET_IN_PAYLOAD_LENGTH_AND_ZLD_COUNT_REGISTER);
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_CALCULATE_ZLD_SHIFT_LEFT] = SHIFT_LEFT(ZLD_COUNTER_REGISTER,(SHIFT_WORDS_TO_BYTES + SHIFT_WORD_TO_8_WORDS));
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_TRANSMIT_ZLD] = TRANSMIT_ALIGNED_FROM_HOB_WITH_LENGTH_FROM_REGISTER(ZLD_COUNTER_REGISTER,HOB_REGISTER_ZERO_DELIMITER_VALUE_OFFSET);
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_CHECK_IF_LAST] = SKIP(DELIA_MPDU_DESCRIPTOR_ADDREES_REGISTER,FALSE,MPDU_LAST_INDICATION_LOCATION_IN_BYTE,MPDU_LAST_INDICATION_OFFET);
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_JUMP_TO_START_OF_LOOP] = JUMP(&DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_START_OF_LOOP]);
	/*end of loop */
	
	DeliaRecipe0[DELIA_INSTRUCTION_AGGREGATION_END_RECIPE] = END_RECIPE;
}

/**********************************************************************************

deliaConstructLegacyVhtRecipe  


Description:
------------
construct the aggregation legacy VHT  recipe

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void deliaConstructLegacyVhtRecipe()
{
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_VHT_PACKET_LOAD_R1_WITH_MPDU_COUNTER] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(DELIA_MPDU_COUNTER_REGISTER,ONE_MPDU_TO_COUNTER);
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_VHT_PACKET_STROE_MPDU_COUNTER_INTO_HOB] = STORE_TO_HOB(DELIA_MPDU_COUNTER_REGISTER,HOB_REGISTER_DELIA_COUNTER_OFFSET);
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_VHT_WAIT_FOR_MPDU_TO_BE_READY] = WAIT_FOR_MPDU_OR_BEACON_TO_BE_READY;
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_VHT_LOAD_R2_WITH_AUTO_FILL_START] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(AUTO_FIIL_INDICATION_REGISTER,AUTO_FILL_START_INDICATION);
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_VHT_START_AUTO_FILL] = STORE_TO_HOB(AUTO_FIIL_INDICATION_REGISTER,HOB_REGISTER_AUTO_FILL_INDICATION_OFFSET);		
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_VHT_LOAD_R3_WITH_HEADER_LENGTH] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(HEADER_LENGTH_REGISTER,DELIA_DESCRIPTOR_REGISTER,HEADER_LENGTH_OFFSET);
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_VHT_CALCULATE_LENGTH_HEADER_AND_CRC] = ADD_REGISTER_AND_IMMEDIATE(MPDU_TOTAL_LENGTH_REGISTER,HEADER_LENGTH_REGISTER,CRC_LENGTH);	
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_VHT_PACKET_LOAD_R5_WITH_PAYLOAD_LENGTH] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(PAYLOAD_LENGTH_REGISTER,DELIA_DESCRIPTOR_REGISTER,LEGACY_PAYLOAD_LENGTH_AND_ZLD_COUNT_OFFSET);	
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_VHT_CALCULATE_TOTAL_LENGTH] = ADD_REGISTER(MPDU_TOTAL_LENGTH_REGISTER,PAYLOAD_LENGTH_REGISTER);
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_VHT_STORE_TOTAL_LENGTH_TO_HOB] = STORE_TO_HOB(MPDU_TOTAL_LENGTH_REGISTER,HOB_REGISTER_LEGACY_VHT_DELIMITER_LENGTH_OFFSET);	
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_VHT_TRANSMIT_DELIMITER] = TRANSMIT_ALIGNED_FROM_HOB(DELIMITER_LENGTH,HOB_REGISTER_DELIMITER_RESULT_OFFSET);
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_VHT_RESET_CRC] = RESET_CRC;	
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_VHT_PACKET_LOAD_R4_WITH_HEADER_ADDRESS] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(HEADER_ADDRESS_REGISTER,DELIA_DESCRIPTOR_REGISTER,LEGACY_HEADER_ADDRESS_OFFSET);
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_VHT_TRANSMIT_HEADER] = TRANSMIT_FROM_REGISTER_WITH_LENGTH_FROM_REGISTER(HEADER_ADDRESS_REGISTER,HEADER_LENGTH_REGISTER,HEADER_TRANSMISSION_OFFSET);	
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_VHT_PACKET_LOAD_R6_WITH_PAYLOAD_ADDRESS] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(PAYLOAD_ADDRESS_REGISTER,DELIA_DESCRIPTOR_REGISTER,LEGACY_PAYLOAD_ADDRESS_AND_LAST_BIT_OFFSET);		
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_VHT_PACKET_MASK_PAYLOAD_ADDRESS] = MASK_REGISTER(PAYLOAD_ADDRESS_REGISTER,PAYLOAD_ADDRESS_BYTES_TO_MASK,PAYLOAD_ADDRESS_MASK_PATTERN);		
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_VHT_PACKET_TRANSMIT_PAYLOAD] = TRANSMIT_FROM_REGISTER_WITH_LENGTH_FROM_REGISTER(PAYLOAD_ADDRESS_REGISTER,PAYLOAD_LENGTH_REGISTER,PAYLOAD_TRANSMISSION_OFFSET);		
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_VHT_PACKET_TRANSMIT_CRC] = TRANSMIT_ALIGNED_TAIL_FROM_HOB(CRC_LENGTH,HOB_REGISTER_CRC_OFFSET);	
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_VHT_LOAD_R2_WITH_AUTO_FILL_END] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(AUTO_FIIL_INDICATION_REGISTER,AUTO_FILL_END_INDICATION);
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_VHT_END_AUTO_FILL] = STORE_TO_HOB(AUTO_FIIL_INDICATION_REGISTER,HOB_REGISTER_AUTO_FILL_INDICATION_OFFSET);	
	DeliaRecipe0[DELIA_INSTRUCTION_LEGACY_VHTN_END_RECIPE] = END_RECIPE;
}

/**********************************************************************************

deliaConstructBeaconRecipe  


Description:
------------
construct the beacon recipe

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void deliaConstructBeaconRecipe()
{

	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_LOAD_R2_WITH_AUTO_FILL_START] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(AUTO_FIIL_INDICATION_REGISTER,AUTO_FILL_START_INDICATION);	
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_START_AUTO_FILL] = STORE_TO_HOB(AUTO_FIIL_INDICATION_REGISTER,HOB_REGISTER_AUTO_FILL_INDICATION_OFFSET);
	
    /* This commands is done in order to not have any operations after cheking the last bit - just jump to start  of loop : the address of the first element descriptor is set to be the (first element adress - element descriptor size)
           so in the loop the element descriptor adress is added with element descriptor size and the first element is handled as all the elements */
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_CALCULATE_FIRST_ELEMENT_DESCRIPTOR_OFFSET] = ADD_REGISTER_AND_IMMEDIATE(DELIA_ELEMENT_DESCRIPTOR_ADDRESS_REGISTER,DELIA_DESCRIPTOR_REGISTER,BEACON_FIRST_ELEMENT_DESCRIPTOR_OFFSET);
	
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_CALCULATE_COUNTER] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(DELIA_BEACON_COUNTER_REGISTER,BEACON_COUNTER_ONE); 		
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_STROE_MPDU_COUNTER_INTO_HOB] = STORE_TO_HOB(DELIA_BEACON_COUNTER_REGISTER,HOB_REGISTER_DELIA_COUNTER_OFFSET);		
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_WAIT_FOR_BEACON_TO_BE_READY] = WAIT_FOR_MPDU_OR_BEACON_TO_BE_READY;

	/* elements start of loop */
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_CALCULATE_ELEMENT_DESCRIPTOR_ADDRESS] = ADD_REGISTER_AND_IMMEDIATE(DELIA_ELEMENT_DESCRIPTOR_ADDRESS_REGISTER,DELIA_ELEMENT_DESCRIPTOR_ADDRESS_REGISTER,ELEMENT_DESCRIPTOR_SIZE);	
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_LOAD_R5_WITH_ELEMENT_LENGTH] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(ELEMENT_LENGTH_REGISTER,DELIA_ELEMENT_DESCRIPTOR_ADDRESS_REGISTER,BEACON_ELEMENT_GENERAL_REGISTER_OFFSET);		
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_MASK_ELEMENT_LENGTH] = MASK_REGISTER(ELEMENT_LENGTH_REGISTER,ELEMENT_LENGTH_BYTES_TO_MASK,ELEMENT_LENGTH_MASK_PATTERN);		

	/* regular element */
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_SKIP_IF_REGULAR_ELEMENT] = SKIP(DELIA_ELEMENT_DESCRIPTOR_ADDRESS_REGISTER,FALSE,REGULAR_TYPE_BIT_NUMBER_IN_BYTE,ELEMENT_TYPE_OFFSET);		
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_JUMP_TO_SPECIAL_ELEMENT] = JUMP(&DeliaRecipe0[DELIA_INSTRUCTION_BEACON_SKIP_IF_TIMESTAMP_ELEMENT]);	
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_LOAD_R6_WITH_ELEMENT_ADDRESS] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(REGULAR_ELEMENT_ADDRESS_REGISTER,DELIA_ELEMENT_DESCRIPTOR_ADDRESS_REGISTER,BEACON_ELEMENT_ADDRESS_OFFSET);		
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_TRANSMIT_ELEMENT] = TRANSMIT_FROM_REGISTER_WITH_LENGTH_FROM_REGISTER(REGULAR_ELEMENT_ADDRESS_REGISTER,ELEMENT_LENGTH_REGISTER,ELEMENT_TRANSMISSION_OFFSET);	
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_REGULAR_ELEMENT_JUMP_TO_CHECK_IF_LAST] = JUMP(&DeliaRecipe0[DELIA_INSTRUCTION_BEACON_CHECK_IF_LAST]);	
		
	/* Timestamp element */	
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_SKIP_IF_TIMESTAMP_ELEMENT] = SKIP(DELIA_ELEMENT_DESCRIPTOR_ADDRESS_REGISTER,FALSE,TIMESTAMP_TYPE_BIT_NUMBER_IN_BYTE,ELEMENT_TYPE_OFFSET);		
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_JUMP_TO_TIM_ELEMENT] = JUMP(&DeliaRecipe0[DELIA_INSTRUCTION_BEACON_TRANSMIT_CONSTANT_BYTES_OF_TIM_ELEMENT]);		
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_LOAD_R4_WITH_VAP_INDEX] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(VAP_INDEX_REGISTER,DELIA_ELEMENT_DESCRIPTOR_ADDRESS_REGISTER,BEACON_VAP_INDEX_OFFSET);		
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_STORE_VAP_INDEX_INTO_HOB] = STORE_TO_HOB(VAP_INDEX_REGISTER,HOB_REGISTER_VAP_INDEX_FOR_TIMESTAMP_OFFSET);		
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_TRANSMIT_TSF_ELEMENT] = TRANSMIT_FROM_HOB(SIZE_OF_TIMESTAMP,HOB_REGISTER_TIMESTAMP_RESULT_OFFSET);		
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_TSF_ELEMENT_JUMP_TO_CHECK_IF_LAST] = JUMP(&DeliaRecipe0[DELIA_INSTRUCTION_BEACON_CHECK_IF_LAST]); 

	// TIM element 
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_TRANSMIT_CONSTANT_BYTES_OF_TIM_ELEMENT] = TRANSMIT_FROM_HOB(TIM_CONSTANT_BYTES_LENGTH,HOB_REGISTER_TIM_CONSTANT_BYTES_OFFSET);
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_LOAD_R2_WITH_TIM_BITMAP_PARAMETERS] = LOAD_FROM_HOB(TIM_BITMAP_LENGTH_REGISTER,HOB_REGISTER_TIM_BITMAP_PARAMETERS_OFFSET);
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_COPY_R2_TO_R3_TO_SAVE_OFFSET] = COPY(TIM_BITMAP_OFFSET_REGISTER,TIM_BITMAP_LENGTH_REGISTER);
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_MASK_BITMAP_LENGTH] = MASK_REGISTER(TIM_BITMAP_LENGTH_REGISTER,TIM_BITMAP_LENGTH_BYTES_TO_MASK,TIM_BITMAP_LENGTH_MASK_PATTERN);
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_SHIFT_BITMAP_OFFSET] = SHIFT_RIGHT(TIM_BITMAP_OFFSET_REGISTER,TIM_BITMAP_OFFSET_SHIFT);
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_MASK_BITMAP_OFFSET] = MASK_REGISTER(TIM_BITMAP_OFFSET_REGISTER,TIM_BITMAP_OFFSET_BYTES_TO_MASK,TIM_BITMAP_OFFSET_MASK_PATTERN);
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_ADD_BASE_ADDRESS_TO_BITMAP_OFFSET] = ADD_REGISTER_AND_IMMEDIATE(TIM_BITMAP_OFFSET_REGISTER,TIM_BITMAP_OFFSET_REGISTER,HOB_REGISTER_TIM_BITMAP_START_OFFSET);
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_TRANSMIT_TIM_ELEMENT] = TRANSMIT_FROM_HOB_WITH_LENGTH_FROM_REGISTER_AND_OFFSET_FROM_REGISTER(TIM_BITMAP_LENGTH_REGISTER,TIM_BITMAP_OFFSET_REGISTER);

	/* Check if last element */	
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_CHECK_IF_LAST] = SKIP(DELIA_ELEMENT_DESCRIPTOR_ADDRESS_REGISTER,FALSE,ELEMENT_LAST_INDICATION_LOCATION_IN_BYTE,ELEMENT_LAST_INDICATION_OFFSET);
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_JUMP_TO_STRAT_OF_LOOP] = JUMP(&DeliaRecipe0[DELIA_INSTRUCTION_BEACON_START_OF_LOOP]); 
    /* end of loop */
	
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_TRANSMIT_CRC] = TRANSMIT_ALIGNED_TAIL_FROM_HOB(CRC_LENGTH,HOB_REGISTER_CRC_OFFSET);
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_LOAD_R2_WITH_AUTO_FILL_END] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(AUTO_FIIL_INDICATION_REGISTER,AUTO_FILL_END_INDICATION);
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_END_AUTO_FILL] = STORE_TO_HOB(AUTO_FIIL_INDICATION_REGISTER,HOB_REGISTER_AUTO_FILL_INDICATION_OFFSET);
	DeliaRecipe0[DELIA_INSTRUCTION_BEACON_END_RECIPE] = END_RECIPE;
}



/**********************************************************************************

deliaConstructRecipe0  


Description:
------------
construct the common recipe (recipe0)

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void deliaConstructRecipe0(void)
{
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_ENABLE_PHY] = START_RECIPE_NONE_AUTO_FRAME;
#ifdef DELIA_LOAD_IMMEDIATE_AFTER_RECOVERY_WORKAROUND
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_LOAD_R1_WITH_INTTEURPT_VALUE_TO_SENDER_DUMMY] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(SENDER_INTERRUPT_VALUE_REGISTER,SENDER_INTTEURPT_VAULE);
#endif //DELIA_LOAD_IMMEDIATE_AFTER_RECOVERY_WORKAROUND
    DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_LOAD_R1_WITH_INTTEURPT_VALUE_TO_SENDER] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(SENDER_INTERRUPT_VALUE_REGISTER,SENDER_INTTEURPT_VAULE); 	
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_CREATE_INTERRUPT_TO_SENDER] = STORE_TO_HOB(SENDER_INTERRUPT_VALUE_REGISTER,HOB_REGISTER_SENDER_INTTEURPT_OFFSET);	
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_LOAD_R0_WITH_DELIA_DESCRIPTOR] = LOAD_FROM_HOB(DELIA_DESCRIPTOR_REGISTER,HOB_REGISTER_DELIA_DESCRIPTOR_OFFSET);

	//Wait for either tcr0 to be ready in delia desc or for all tcrs to be ready in hob
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_WAIT_FOR_INPUT_TO_BE_READY] = WAIT(TCR0_READY_EVENT | ALL_TCRS_IN_HOB_READY_EVENT);
	//Check if all tcrs are ready in hob and if so skip the next line 
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_SKIP_IF_ALL_TCRS_READY] = SKIP_ON_HOB(FALSE, ALL_TCRS_IN_HOB_READY_EVENT_BIT_OFFSET_IN_BYTE, (HOB_REGISTER_DELIA_EVENTS_OFFSET + ALL_TCRS_IN_HOB_READY_EVENT_BYTE_OFFSET_IN_REG));
	//if we got tcr0 is ready in delia desc and we jump to the send tcrs from delia desc section
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_JUMP_TO_SEND_TCRS_FROM_DELIA_DESC] = JUMP(&DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_SEND_TCR0]);

	//send all tcrs from hob
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_SEND_TCR0_FROM_HOB] = TRANSMIT_FROM_HOB(TCR_LENGTH_TO_SEND, HOB_REGISTER_TCR0_OFFSET);
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_SEND_TCR1_FROM_HOB] = TRANSMIT_FROM_HOB(TCR_LENGTH_TO_SEND, HOB_REGISTER_TCR1_OFFSET);
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_SEND_TCR2_FROM_HOB] = TRANSMIT_FROM_HOB(TCR_LENGTH_TO_SEND, HOB_REGISTER_TCR2_OFFSET);
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_SEND_TCR3_FROM_HOB] = TRANSMIT_FROM_HOB(TCR_LENGTH_TO_SEND, HOB_REGISTER_TCR3_OFFSET);
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_SEND_TCR4_FROM_HOB] = TRANSMIT_FROM_HOB(TCR_LENGTH_TO_SEND, HOB_REGISTER_TCR4_OFFSET);

	//wait for word5 & word6 to be ready in delia desc
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_WAIT_FOR_WORD5_TO_WORD6_TO_BE_READY] = WAIT(WORD5_TO_WORD6_READY_EVENT);
	//jump to word5 & word6 handling
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_JUMP_TO_LOAD_R1_WITH_GENERAL_INFO] = JUMP(&DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_CHECK_IF_LENGTH_ZERO]);

	/* send TCR 0 */
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_SEND_TCR0] = TRANSMIT_FROM_REGISTER(DELIA_DESCRIPTOR_REGISTER,TCR_LENGTH_TO_SEND,TCR0_OFFSET_IN_DELIA_DESCRIPTOR);

	//Wait for tcr1 to word 6 to be ready
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_WAIT_FOR_TCR1_TO_WORD6_TO_BE_READY] = WAIT(TCR1_TO_WORD6_READY_EVENT);
	/* send TCR 1-4 */
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_SEND_TCR1] = TRANSMIT_FROM_REGISTER(DELIA_DESCRIPTOR_REGISTER,TCR_LENGTH_TO_SEND,TCR1_OFFSET_IN_DELIA_DESCRIPTOR);
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_SEND_TCR2] = TRANSMIT_FROM_REGISTER(DELIA_DESCRIPTOR_REGISTER,TCR_LENGTH_TO_SEND,TCR2_OFFSET_IN_DELIA_DESCRIPTOR);
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_SEND_TCR3] = TRANSMIT_FROM_REGISTER(DELIA_DESCRIPTOR_REGISTER,TCR_LENGTH_TO_SEND,TCR3_OFFSET_IN_DELIA_DESCRIPTOR);
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_SEND_TCR4] = TRANSMIT_FROM_REGISTER(DELIA_DESCRIPTOR_REGISTER,TCR_LENGTH_TO_SEND,TCR4_OFFSET_IN_DELIA_DESCRIPTOR);
	
	//Check if length is 0 and if so end recipe
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_CHECK_IF_LENGTH_ZERO] = SKIP_ON_HOB(TRUE, HOB_REGISTER_IS_LENGTH_ZERO_BIT, HOB_REGISTER_IS_LENGTH_ZERO_OFFSET);
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_LENGTH_ZERO_RECIPE_END] = END_RECIPE;

    /* Load general info word */
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_LOAD_R1_WITH_GENERAL_INFO] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(DELIA_DESCRIPTOR_GENERAL_INFO_REGISTER,DELIA_DESCRIPTOR_REGISTER,GENERAL_INFORMATION_OFFSET);
	
	/* BF */
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_SKIP_IF_BF_EXIST] = SKIP(DELIA_DESCRIPTOR_REGISTER,FALSE,BF_INDICATION_BIT_NUMBER_IN_BYTE,BF_INDICATION_OFFSET);
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_BF_NOT_EXIST_JUMP_TO_RECIPES_BRANCHING] = JUMP(&DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_RESET_CRC_AND_CHANGE_BUS_TO_DATA]); 

	
    DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_BF_START] = START_BF;
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_BF_COPY_GENERAL_INFO_TO_R2] = COPY(BF_STATION_ID_REGISTER,DELIA_DESCRIPTOR_GENERAL_INFO_REGISTER);
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_BF_MASK_STATION_ID] = MASK_REGISTER(BF_STATION_ID_REGISTER,BF_STATION_ID_BYTES_TO_MASK,BF_STATION_ID_MASK_PATTERN);
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_BF_LOAD_R3_WITH_BF_INDEXES_TABLE_BASE_ADDRESS] = LOAD_FROM_HOB(BF_INDEX_ENTRY_ADDRESS_REGISTER,HOB_REGISTER_BF_INDEX_TABLE_BASE_ADDRESS_OFFSET);
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_BF_CALCULATE_BF_INDEX_ENTRY_ADDRESS] = ADD_REGISTER(BF_INDEX_ENTRY_ADDRESS_REGISTER,BF_STATION_ID_REGISTER);
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_BF_LOAD_R1_WITH_TIMEOUT_VALUES] = LOAD_FROM_HOB(BF_TIMEOUT_VALUES_REGISTER,HOB_REGISTER_BF_TIMEOUT_VALUES_OFFSET); 
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_BF_SET_TIMER] = STORE_TO_HOB(BF_TIMEOUT_VALUES_REGISTER,HOB_REGISTER_DELIA_TIMER_OFFSET);
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_BF_WAIT_FOR_TIMEOUT_OR_BF_READY] = WAIT_FOR_BF_IDLE_OR_TIMEOUT;
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_BF_LOAD_R6_WITH_BF_INDEX] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(BF_INDEX_REGISTER,BF_INDEX_ENTRY_ADDRESS_REGISTER,BF_INDEX_LOAD_OFFSET);
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_BF_MASK_STATION_ID_TO_MODULO_4] =  MASK_REGISTER(BF_STATION_ID_REGISTER,BF_STATION_ID_BYTES_TO_MASK,BF_STATION_ID_MODULO_4_MASK_PATTERN);
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_BF_CACULATE_SHIFT_RIGHT_TO_INDEX] = SHIFT_LEFT(BF_STATION_ID_REGISTER,BF_INDEX_SHIFT);
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_BF_SHIFT_RIGHT_BF_INDEX] = SHIFT_RIGHT_FROM_REGISTER(BF_INDEX_REGISTER,BF_STATION_ID_REGISTER);
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_BF_MASK_BF_INDEX] = MASK_REGISTER(BF_INDEX_REGISTER,BF_INDEX_ENTRY_BYTES_TO_MASK,BF_INDEX_ENTRY_MASK_PATTERN);
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_BF_CALCULATE_BF_REPORT_OFFSET] = SHIFT_LEFT(BF_INDEX_REGISTER,BF_ENTRY_SHIFT_SIZE);
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_BF_LOAD_R4_WITH_BF_REPORT_TABLE_BASE_ADDRESS] = LOAD_FROM_HOB(BF_REPORT_ADDRESS_REGISTER,HOB_REGISTER_BF_DATA_BASE_OFFSET);
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_BF_CALCULATE_BF_REPORT_ADDRESS] = ADD_REGISTER(BF_REPORT_ADDRESS_REGISTER,BF_INDEX_REGISTER);
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_BF_LOAD_R5_WITH_BF_REPORT_LENGTH] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(BF_REPORT_LENGTH_REGISTER,BF_REPORT_ADDRESS_REGISTER,BF_REPORT_LENGTH_OFFSET);
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_BF_MASK_BF_REPORT_LENGTH] = MASK_REGISTER(BF_REPORT_LENGTH_REGISTER,BF_LENGTH_BYTES_TO_MASK,BF_LENGTH_MASK_PATTERN);
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_BF_TRANSMIT_BF_REPORT] = TRANSMIT_FROM_REGISTER_WITH_LENGTH_FROM_REGISTER(BF_REPORT_ADDRESS_REGISTER,BF_REPORT_LENGTH_REGISTER,BF_TRANSMISSION_OFFSET);
	/* BF end */

	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_RESET_CRC_AND_CHANGE_BUS_TO_DATA] = START_DATA;

	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_SHIFT_TYPE] = SHIFT_RIGHT(DELIA_DESCRIPTOR_GENERAL_INFO_REGISTER,RECIPE_TYPE_SHIFT);
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_CALCULATE_LOOKUP_TABLE_ENTRY_OFFSET] = SHIFT_LEFT(DELIA_DESCRIPTOR_GENERAL_INFO_REGISTER,RECIPE_LOOKUP_TABLE_ENTRY_SIZE_SHIFT);
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_LOAD_R2_WITH_LOOKUP_TABLE_BASE_ADDRESS] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(RECIPE_ADDRESS_REGISTER,DeliaRecipesLookupTable);
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_CALCULATE_RECIPE_ADDRESS] = ADD_REGISTER(RECIPE_ADDRESS_REGISTER,DELIA_DESCRIPTOR_GENERAL_INFO_REGISTER);
	DeliaRecipe0[DELIA_INSTRUCTION_RECIPE_0_JUMP_TO_PROPER_RECIPE] = JUMP_FROM_REGISTER(RECIPE_ADDRESS_REGISTER,LOOKUP_TABLE_ENTRY_OFFSET);

	
}

/**********************************************************************************

deliaConstructAckRecipe  


Description:
------------
construct ACK recipe

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void deliaConstructAckRecipe()
{
	DeliaRecipeAck[DELIA_INSTRUCTION_ACK_START_AUTO_FRAME] = START_RECIPE_AUTO_FRAME;
#ifdef DELIA_LOAD_IMMEDIATE_AFTER_RECOVERY_WORKAROUND
	DeliaRecipeAck[DELIA_INSTRUCTION_ACK_LOAD_TCR0_ADDR_DUMMY] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(AUTO_REPLY_TCR0_ADDR_REGISTER,&DeliaAutoResponseTcr0);
#endif //DELIA_LOAD_IMMEDIATE_AFTER_RECOVERY_WORKAROUND
	DeliaRecipeAck[DELIA_INSTRUCTION_ACK_LOAD_TCR0_ADDR] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(AUTO_REPLY_TCR0_ADDR_REGISTER,&DeliaAutoResponseTcr0);
	DeliaRecipeAck[DELIA_INSTRUCTION_ACK_LOAD_TCR2_ADDR] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(AUTO_REPLY_TCR2_ADDR_REGISTER,&DeliaAutoResponseTcr2);
	DeliaRecipeAck[DELIA_INSTRUCTION_ACK_SEND_TCR0] = TRANSMIT_FROM_REGISTER(AUTO_REPLY_TCR0_ADDR_REGISTER,TCR_LENGTH_TO_SEND,0);
	DeliaRecipeAck[DELIA_INSTRUCTION_ACK_SEND_TCR1] = TCR(DELIA_ACK_TCR_1);
	DeliaRecipeAck[DELIA_INSTRUCTION_ACK_SEND_TCR2] = TRANSMIT_FROM_REGISTER(AUTO_REPLY_TCR2_ADDR_REGISTER,TCR_LENGTH_TO_SEND,0);
	DeliaRecipeAck[DELIA_INSTRUCTION_ACK_SEND_TCR3] = TCR(DELIA_ACK_TCR_3);
	DeliaRecipeAck[DELIA_INSTRUCTION_ACK_SEND_TCR4] = TCR(DELIA_ACK_TCR_4);
	DeliaRecipeAck[DELIA_INSTRUCTION_ACK_START_DATA] = START_DATA;
	DeliaRecipeAck[DELIA_INSTRUCTION_ACK_TRANSMIT_CONTROL] = TRANSMIT_IMMEDIATE(FRAME_CONTROL_LEN,ACK_CONTROL_FIELD);
	  //*asmPtr++ = (uint32)TXR(1,HOB,HOB_PM_ADDR); TO CHECK IF TO SET PM IN OUR IMPELMENTATION
	DeliaRecipeAck[DELIA_INSTRUCTION_ACK_WAIT_FOR_DURATION] = WAIT_FOR_DURATION;
	DeliaRecipeAck[DELIA_INSTRUCTION_ACK_TRANSMIT_DURATION] = TRANSMIT_FROM_HOB(DUR_FIELD_LEN,HOB_REGISTER_GENERAL_AUTO_REPLY_DUR_OFFSET); 
	DeliaRecipeAck[DELIA_INSTRUCTION_ACK_TRANSMIT_RECIEVER_ADDRESS] = TRANSMIT_FROM_HOB(MAC_ADDRESS_LEN,HOB_REGISTER_ADDRESS_1_OFFSET);
	DeliaRecipeAck[DELIA_INSTRUCTION_ACK_TRANSMIT_CRC] = TRANSMIT_FROM_HOB(CRC_LENGTH,HOB_REGISTER_CRC_OFFSET);
	DeliaRecipeAck[DELIA_INSTRUCTION_ACK_END_RECIPE] = END_RECIPE;
}

/**********************************************************************************

deliaConstructBaRecipe  


Description:
------------
construct BA recipe

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void deliaConstructBaRecipe()
{
	DeliaRecipeBa[DELIA_INSTRUCTION_BA_START_AUTO_FRAME] = START_RECIPE_AUTO_FRAME;
#ifdef DELIA_LOAD_IMMEDIATE_AFTER_RECOVERY_WORKAROUND
	DeliaRecipeBa[DELIA_INSTRUCTION_BA_LOAD_TCR0_ADDR_DUMMY] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(AUTO_REPLY_TCR0_ADDR_REGISTER,&DeliaAutoResponseTcr0);
#endif //DELIA_LOAD_IMMEDIATE_AFTER_RECOVERY_WORKAROUND
	DeliaRecipeBa[DELIA_INSTRUCTION_BA_LOAD_TCR0_ADDR] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(AUTO_REPLY_TCR0_ADDR_REGISTER,&DeliaAutoResponseTcr0);
	DeliaRecipeBa[DELIA_INSTRUCTION_BA_LOAD_TCR2_ADDR] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(AUTO_REPLY_TCR2_ADDR_REGISTER,&DeliaAutoResponseTcr2);
	DeliaRecipeBa[DELIA_INSTRUCTION_BA_SEND_TCR0] = TRANSMIT_FROM_REGISTER(AUTO_REPLY_TCR0_ADDR_REGISTER,TCR_LENGTH_TO_SEND,0);
	DeliaRecipeBa[DELIA_INSTRUCTION_BA_SEND_TCR1] = TCR(DELIA_BA_TCR_1);
	DeliaRecipeBa[DELIA_INSTRUCTION_BA_SEND_TCR2] = TRANSMIT_FROM_REGISTER(AUTO_REPLY_TCR2_ADDR_REGISTER,TCR_LENGTH_TO_SEND,0);
	DeliaRecipeBa[DELIA_INSTRUCTION_BA_SEND_TCR3] = TCR(DELIA_BA_TCR_3);
	DeliaRecipeBa[DELIA_INSTRUCTION_BA_SEND_TCR4] = TCR(DELIA_BA_TCR_4);
	DeliaRecipeBa[DELIA_INSTRUCTION_BA_START_DATA] = START_DATA;
	DeliaRecipeBa[DELIA_INSTRUCTION_BA_TRANSMIT_CONTROL] = TRANSMIT_IMMEDIATE(FRAME_CONTROL_LEN,BA_CONTROL_FIELD);
	  //*asmPtr++ = (uint32)TXR(1,HOB,HOB_PM_ADDR); TO CHECK IF TO SET PM IN OUR IMPELMENTATION
	DeliaRecipeBa[DELIA_INSTRUCTION_BA_WAIT_FOR_DURATION] = WAIT_FOR_DURATION;
	DeliaRecipeBa[DELIA_INSTRUCTION_BA_TRANSMIT_DURATION] = TRANSMIT_FROM_HOB(DUR_FIELD_LEN,HOB_REGISTER_GENERAL_AUTO_REPLY_DUR_OFFSET); 
	DeliaRecipeBa[DELIA_INSTRUCTION_BA_TRANSMIT_RECIEVER_ADDRESS] = TRANSMIT_FROM_HOB(MAC_ADDRESS_LEN,HOB_REGISTER_ADDRESS_1_OFFSET);
	DeliaRecipeBa[DELIA_INSTRUCTION_BA_TRANSMIT_TRANSMITTER_ADDRESS] = TRANSMIT_FROM_HOB(MAC_ADDRESS_LEN,HOB_REGISTER_ADDRESS_2_OFFSET);
	DeliaRecipeBa[DELIA_INSTRUCTION_BA_TRANSMIT_BA_CONTROL_AND_BITMAP] = TRANSMIT_FROM_HOB(BA_FRAME_ONLY_DATA_SIZE,HOB_REGISTER_BA_CTRL_AND_SSN_OFFSET);
	DeliaRecipeBa[DELIA_INSTRUCTION_BA_TRANSMIT_CRC] = TRANSMIT_FROM_HOB(CRC_LENGTH,HOB_REGISTER_CRC_OFFSET);
	DeliaRecipeBa[DELIA_INSTRUCTION_BA_END_RECIPE] = END_RECIPE;
}

/**********************************************************************************

deliaConstructCtsAutoResponseRecipe  


Description:
------------
construct CTS auto response recipe

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void deliaConstructCtsAutoResponseRecipe()
{
#ifdef DYNAMIC_RTS_ISSUE_WORKAROUND

	/*

	The logic is as follows:

	If signaling bit in TA is set and we are in 11a/g phy mode, we should transmit TCR0 from reserved word (word1 of  phy metrics):
	
		1. Check if signaling bit is set in HOB TA register and continue if so. If not - jump to run current logic
		2. Read CB pointer of the start of the phy header
		3. calculate Phy metrics reserved word address (word1)
		4. calculate Phy mode word address (in phy status)
		5. Check if both 14 and 15 bits are 0 (i.e. legacy phy mode) and transmit from Phy metrics reserved word address if so (if not - jump to run current logic)
		6. continue to original code for all other TCRs
	*/
	

	//Tx Enable
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_RTS_WORKAROUND_TX_ENABLE] = START_RECIPE_BF_REPORT;

	//1. Check if signaling bit is set in HOB TA register and continue if so. If not - jump to run current logic
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_RTS_WORKAROUND_SKIP_ON_HOB_IF_SIGNALING_TA] = SKIP_ON_HOB(FALSE, HOB_REGISTER_ADDRESS_1_WITH_TA_BIT, HOB_REGISTER_ADDRESS_1_WITH_TA_BIT_OFFSET);
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_RTS_WORKAROUND_NOT_SIGNALING_JUMP_TO_ORIGINAL_CODE] = JUMP(&DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_START_AUTO_FRAME]);

	//2. Read CB pointer of the start of the phy header
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_RTS_WORKAROUND_LOAD_CB_POINTER] = LOAD_FROM_HOB(RTS_WORKAROUND_CB_POINTER_REGISTER, HOB_REGISTER_LAST_RX_CB_POINTER_OFFSET); 
	//3. calculate Phy metrics reserved word address (word1)
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_RTS_WORKAROUND_CALC_PHY_METRICS_ADDRESS] = ADD_REGISTER_AND_IMMEDIATE(RTS_WORKAROUND_TCR0_LOCATION_REGISTER, RTS_WORKAROUND_CB_POINTER_REGISTER, (RX_PHY_HEADER_FIELD_PHY_METRICES_WORD_1_WORD_OFFSET << SHIFT_WORDS_TO_BYTES));
	//4. calculate Phy mode word address (in phy status)
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_RTS_WORKAROUND_CALC_PHY_MODE_WORD_ADDRESS] = ADD_REGISTER_AND_IMMEDIATE(RTS_WORKAROUND_CB_POINTER_REGISTER, RTS_WORKAROUND_CB_POINTER_REGISTER, (RX_PHY_HEADER_FIELD_PHY_STATUS_WORD_1_WORD_OFFSET << SHIFT_WORDS_TO_BYTES));

	//5. Check if both 14 and 15 bits are 0 (i.e. legacy phy mode) and transmit from Phy metrics reserved word address if so (if not - jump to run current logic)
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_RTS_WORKAROUND_SKIP_IF_11_A_G_BIT_6_FALSE] = SKIP(RTS_WORKAROUND_CB_POINTER_REGISTER, TRUE, (RX_PHY_HEADER_PHY_STATUS_WORD_1_FIELD_PHY_MODE_START_BIT - (NUM_OF_BITS_IN_BYTE * RX_PHY_HEADER_PHY_STATUS_WORD_1_FIELD_PHY_MODE_START_BYTE)) /*0x6*/, RX_PHY_HEADER_PHY_STATUS_WORD_1_FIELD_PHY_MODE_START_BYTE);
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_RTS_WORKAROUND_NOT_11_A_G__BIT_6_JUMP_TO_ORIGINAL_CODE] = JUMP(&DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_START_AUTO_FRAME]);
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_RTS_WORKAROUND_SKIP_IF_11_A_G_BIT_7_FALSE] = SKIP(RTS_WORKAROUND_CB_POINTER_REGISTER, TRUE, (RX_PHY_HEADER_PHY_STATUS_WORD_1_FIELD_PHY_MODE_START_BIT - (NUM_OF_BITS_IN_BYTE * RX_PHY_HEADER_PHY_STATUS_WORD_1_FIELD_PHY_MODE_START_BYTE) + 0x1) /*0x7*/, RX_PHY_HEADER_PHY_STATUS_WORD_1_FIELD_PHY_MODE_START_BYTE);
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_RTS_WORKAROUND_NOT_11_A_G__BIT_7_JUMP_TO_ORIGINAL_CODE] = JUMP(&DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_START_AUTO_FRAME]);

	//transmit from the reserved word - phy has already copied the correct TCR to this location
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_RTS_WORKAROUND_TRANSMIT_FROM_CYCLIC_BUF_RESERVED_WORD] = TRANSMIT_FROM_REGISTER(RTS_WORKAROUND_TCR0_LOCATION_REGISTER, TCR_LENGTH_TO_SEND, 0);
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_RTS_WORKAROUND_START_AUTO_FRAME] = START_RECIPE_AUTO_FRAME;
	//6. continue to original code for all other TCRs
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_RTS_WORKAROUND_JUMP_TO_CONTINUE_TRANSMIT_THE_REST_TCRS] = JUMP(&DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_SEND_TCR1]);
						
#endif //DYNAMIC_RTS_ISSUE_WORKAROUND

					
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_START_AUTO_FRAME] = START_RECIPE_AUTO_FRAME;	
#ifdef DELIA_LOAD_IMMEDIATE_AFTER_RECOVERY_WORKAROUND
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_LOAD_TCR0_ADDR_DUMMY] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(AUTO_REPLY_TCR0_ADDR_REGISTER,&DeliaAutoResponseTcr0);
#endif //DELIA_LOAD_IMMEDIATE_AFTER_RECOVERY_WORKAROUND
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_LOAD_TCR0_ADDR] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(AUTO_REPLY_TCR0_ADDR_REGISTER,&DeliaAutoResponseTcr0);
	
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_SEND_TCR0] = TRANSMIT_FROM_REGISTER(AUTO_REPLY_TCR0_ADDR_REGISTER,TCR_LENGTH_TO_SEND,0);
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_SEND_TCR1] = TCR(DELIA_CTS_AUTO_RESPONSE_TCR_1);
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_LOAD_TCR2_ADDR] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(AUTO_REPLY_TCR2_ADDR_REGISTER,&DeliaAutoResponseCtsTcr2);
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_SEND_TCR2] = TRANSMIT_FROM_REGISTER(AUTO_REPLY_TCR2_ADDR_REGISTER,TCR_LENGTH_TO_SEND,0);	

#ifdef DYNAMIC_RTS_ISSUE_WORKAROUND

	/*Do 200 NOPs - 
	This is done because  as the result of the w/a, the delia works faster (TCR0 is transferred faster), what not letting the phy genrisc much time to copy the bw and dyn bits to the scrembler header.
	These nops are delaying the delia TCR3 transmittion and therefore delays locking the scrembler value by the phy hw, so genrisc will have the time to overwrite it with correct values */	
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_RTS_WORKAROUND_INITIATE_LOOP_COUNTER] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(RTS_WORKAROUND_LOOP_COUNTER_REGISTER, 0x0);
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_RTS_WORKAROUND_DO_NOPS]= NOP;
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_RTS_WORKAROUND_INCREMENT_LOOP_COUNTER] = ADD_REGISTER_AND_IMMEDIATE(RTS_WORKAROUND_LOOP_COUNTER_REGISTER, RTS_WORKAROUND_LOOP_COUNTER_REGISTER, 0x1);
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_RTS_WORKAROUND_SKIP_IF_COUNTER_LIMIT_REACHED] = SKIP_IF_EQUAL_TO_IMMEDIATE(RTS_WORKAROUND_LOOP_COUNTER_REGISTER, FALSE, RTS_WORKAROUND_LOOP_NUM_OF_NOPS_BEFORE_TCR3);
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_RTS_WORKAROUND_JUMP_TO_START_OF_THE_LOOP] = JUMP(&DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_RTS_WORKAROUND_DO_NOPS]);
	
#endif //DYNAMIC_RTS_ISSUE_WORKAROUND


	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_SEND_TCR3] = TCR(DELIA_CTS_AUTO_RESPONSE_TCR_3);		
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_SEND_TCR4] = TCR(DELIA_CTS_AUTO_RESPONSE_TCR_4);
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_START_DATA] = START_DATA;
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_TRANSMIT_CONTROL] = TRANSMIT_IMMEDIATE(FRAME_CONTROL_LEN,CTS_AUTO_RESPONSE_CONTROL_FIELD);
	  //*asmPtr++ = (uint32)TXR(1,HOB,HOB_PM_ADDR); TO CHECK IF TO SET PM IN OUR IMPELMENTATION
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_WAIT_FOR_DURATION] = WAIT_FOR_DURATION;
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_TRANSMIT_DURATION] = TRANSMIT_FROM_HOB(DUR_FIELD_LEN,HOB_REGISTER_GENERAL_AUTO_REPLY_DUR_OFFSET); 
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_TRANSMIT_RECIEVER_ADDRESS] = TRANSMIT_FROM_HOB(MAC_ADDRESS_LEN,HOB_REGISTER_ADDRESS_1_OFFSET);
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_TRANSMIT_CRC] = TRANSMIT_FROM_HOB(CRC_LENGTH,HOB_REGISTER_CRC_OFFSET);
	DeliaRecipeCtsAutoResponse[DELIA_INSTRUCTION_CTS_AUTO_RESPONSE_END_RECIPE] = END_RECIPE;
}

/**********************************************************************************

deliaConstructBfReportRecipe  


Description:
------------
construct the BF report recipe (auto reply);
Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void deliaConstructBfReportRecipe()
{
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_START_RECIPE] = START_RECIPE_BF_REPORT;
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_SEND_TCR_0_3] = TRANSMIT_FROM_HOB(DELIA_BF_REPORT_TCR_0_3_SIZE_IN_HOB,HOB_REGISTER_BF_REPORT_TCR_0_3_OFFSET);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_SKIP_IF_MATRICES_EXIST] = SKIP_ON_HOB(FALSE,BF_REPORT_MATRICES_EXIST_BIT_NUMBER_IN_BYTE,HOB_REGISTER_BF_REPORT_CONTROL_REGISTER_OFFSET);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_MATRICES_NOT_EXIST_JUMP_TO_WAIT_FOR_LENGHT] = JUMP(&DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_WAIT_FOR_LENGTH_OR_ERROR]); 

	/* Preparing BF matrices:
	   1)use station index to find the BF index - 
	   		a)read allign the word of the station(shift index right and left)
	   		b)find the entry in the word - leave only bits 0,1 of station index by shift left 30 and multiply this value by 8 with shift right 27,
	   		  then shift the index by the result to get the index and mask this byte.
	   2)use BF index to find BD entry address 
	   3)find BF entry length */
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_LOAD_R2_WITH_STATION_INDEX] = LOAD_FROM_HOB(BF_STATION_ID_REGISTER,HOB_REGISTER_BF_REPORT_STATION_ID_OFFSET); 
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_LOAD_R3_WITH_BF_INDEXES_TABLE_BASE_ADDRESS] = LOAD_FROM_HOB(BF_INDEX_ENTRY_ADDRESS_REGISTER,HOB_REGISTER_BF_INDEX_TABLE_BASE_ADDRESS_OFFSET);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_CALCULATE_BF_INDEX_ENTRY_ADDRESS] = ADD_REGISTER(BF_INDEX_ENTRY_ADDRESS_REGISTER,BF_STATION_ID_REGISTER);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_SHIFT_RIGHT_R3_FOR_WORD_ALLIGNED_ADDRESS] = SHIFT_RIGHT(BF_INDEX_ENTRY_ADDRESS_REGISTER,SHIFT_WORDS_TO_BYTES);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_SHIFT_LEFT_R3_TO_RETURN_TO_BYTES_ADDRESS] = SHIFT_LEFT(BF_INDEX_ENTRY_ADDRESS_REGISTER,SHIFT_WORDS_TO_BYTES);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_LOAD_R6_WITH_TIMEOUT_VALUES] = LOAD_FROM_HOB(BF_TIMEOUT_VALUES_REGISTER,HOB_REGISTER_BF_TIMEOUT_VALUES_OFFSET); 
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_SET_TIMER] = STORE_TO_HOB(BF_TIMEOUT_VALUES_REGISTER,HOB_REGISTER_DELIA_TIMER_OFFSET);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_WAIT_FOR_TIMEOUT_OR_BF_READY] = WAIT_FOR_BF_IDLE_OR_TIMEOUT;
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_LOAD_R6_WITH_BF_INDEX] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(BF_INDEX_REGISTER,BF_INDEX_ENTRY_ADDRESS_REGISTER,BF_INDEX_LOAD_OFFSET);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_SHIFT_LEFT_R2_TO_CALCULATE_MODULO_4] = SHIFT_LEFT(BF_STATION_ID_REGISTER,BF_REPORT_STATION_ID_MODULO_4_SHIFT);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_SHIFT_RIGHT_R2_TO_CALCULATE_SHIFT_FOR_INDEX] = SHIFT_RIGHT(BF_STATION_ID_REGISTER,BF_REPORT_BF_INDEX_SHIFT);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_SHIFT_RIGHT_BF_INDEX] = SHIFT_RIGHT_FROM_REGISTER(BF_INDEX_REGISTER,BF_STATION_ID_REGISTER);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_MASK_BF_INDEX] = MASK_REGISTER(BF_INDEX_REGISTER,BF_INDEX_ENTRY_BYTES_TO_MASK,BF_INDEX_ENTRY_MASK_PATTERN);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_CALCULATE_BF_REPORT_OFFSET] = SHIFT_LEFT(BF_INDEX_REGISTER,BF_ENTRY_SHIFT_SIZE);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_LOAD_R4_WITH_BF_REPORT_TABLE_BASE_ADDRESS] = LOAD_FROM_HOB(BF_REPORT_ADDRESS_REGISTER,HOB_REGISTER_BF_DATA_BASE_OFFSET);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_CALCULATE_BF_REPORT_ADDRESS] = ADD_REGISTER(BF_REPORT_ADDRESS_REGISTER,BF_INDEX_REGISTER);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_LOAD_R5_WITH_BF_REPORT_LENGTH] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(BF_REPORT_LENGTH_REGISTER,BF_REPORT_ADDRESS_REGISTER,BF_REPORT_LENGTH_OFFSET);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_MASK_BF_REPORT_LENGTH] = MASK_REGISTER(BF_REPORT_LENGTH_REGISTER,BF_LENGTH_BYTES_TO_MASK,BF_LENGTH_MASK_PATTERN);

	/* Wait for BF length or error to be ready
	   transmit TCR4 and then check if matrices exist and transmit thme if needed */
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_WAIT_FOR_LENGTH_OR_ERROR] = WAIT_BF_LENGTH_READY_OR_ERROR;
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_TRANSMIT_TCR4] = TRANSMIT_FROM_HOB(TCR_LENGTH_TO_SEND,HOB_REGISTER_BF_REPORT_TOTAL_LENGHT_OFFSET);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_SET_BUS_TO_BF] = START_BF;
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_SKIP_IF_MATRICES_DONT_EXIST] =  SKIP_ON_HOB(TRUE,BF_REPORT_MATRICES_EXIST_BIT_NUMBER_IN_BYTE,HOB_REGISTER_BF_REPORT_CONTROL_REGISTER_OFFSET);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_TRANSMIT_BF_MATRICES] = TRANSMIT_FROM_REGISTER_WITH_LENGTH_FROM_REGISTER(BF_REPORT_ADDRESS_REGISTER,BF_REPORT_LENGTH_REGISTER,BF_TRANSMISSION_OFFSET);

    /* Prepare VHT delimiter that will be used in case it is needed */
    DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_LOAD_R6_WITH_REPORT_FRAME_LENGTH] = LOAD_FROM_HOB(MPDU_TOTAL_LENGTH_REGISTER,HOB_REGISTER_BF_REPORT_FRAME_LENGTH_OFFSET);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_STORE_FRAME_LENGTH_TO_HOB] = STORE_TO_HOB(MPDU_TOTAL_LENGTH_REGISTER,HOB_REGISTER_LEGACY_VHT_DELIMITER_LENGTH_OFFSET);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_SKIP_IF_NOT_VHT] = SKIP_ON_HOB(TRUE,VHT_INDICATION_BIT_NUMBER_IN_BYTE_IN_HOB,HOB_REGISTER_VHT_INDICATION_OFFSET);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_TRANSMIT_VHT_DELIMITER] = TRANSMIT_FROM_HOB(DELIMITER_LENGTH,HOB_REGISTER_DELIMITER_RESULT_OFFSET);

    /* Transmit header 
	   1)Transmit 2 bytes of control from IMMEDIATE value of BF report 
	   2)Transmit duration
	   3)transmit address1 , address2 , address3
	   4)get sequence number from vap data base from TID 0 parameters */
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_START_DATA] = START_DATA;
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_TRANSMIT_CONTROL] = TRANSMIT_IMMEDIATE(FRAME_CONTROL_LEN,BF_REPORT_CONTROL_FIELD);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_WAIT_FOR_DURATION] = WAIT_FOR_DURATION;
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_SKIP_IF_NOT_AFTER_NDP] =  SKIP_ON_HOB(TRUE,BF_REPORT_AFTER_NDP_BIT_NUMBER_IN_BYTE,HOB_REGISTER_BF_REPORT_CONTROL_REGISTER_OFFSET);
	//Jump to transmit duration after PHY NDP
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_JUMP_TO_TRANSMIT_DURATION_AFTER_NDP] = JUMP(&DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_TRANSMIT_DURATION_AFTER_NDP]); 
	//Transmit duration of BF poll
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_TRANSMIT_DURATION_AFTER_BF_POLL] = TRANSMIT_FROM_HOB(DUR_FIELD_LEN,HOB_REGISTER_GENERAL_AUTO_REPLY_DUR_OFFSET); 
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_JUMP_TO_TRANSMIT_RECEIVER_ADDRESS] = JUMP(&DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_TRANSMIT_RECIEVER_ADDRESS]); 
	//Transmit duration after PHY NDP
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_TRANSMIT_DURATION_AFTER_NDP] = TRANSMIT_FROM_HOB(DUR_FIELD_LEN,HOB_REGISTER_BF_REPORT_DURATION_OFFSET);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_TRANSMIT_RECIEVER_ADDRESS] = TRANSMIT_FROM_HOB(MAC_ADDRESS_LEN,HOB_REGISTER_BF_REPORT_ADDRESS_1_OFFSET);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_TRANSMIT_TRANSMITTER_ADDRESS] = TRANSMIT_FROM_HOB(MAC_ADDRESS_LEN,HOB_REGISTER_BF_REPORT_ADDRESS_2_OFFSET);

	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_LOAD_R6_WITH_NDPA_VAP_INDEX] = LOAD_FROM_HOB(VAP_INDEX_REGISTER,HOB_REGISTER_BF_REPORT_NDPA_VAP_INDEX_ADDRESS_OFFSET);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_STORE_VAP_INDEX_INTO_HOB] = STORE_TO_HOB(VAP_INDEX_REGISTER,HOB_REGISTER_VAP_INDEX_FOR_TIMESTAMP_OFFSET);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_SKIP_CHOOSE_ADDRESS] = SKIP_ON_HOB(TRUE,BF_REPORT_CHOOSE_ADDRESS_BIT_NUMBER_IN_BYTE,HOB_REGISTER_BF_ADDRESS_3_SELECT_OFFSET);
	//Jump to Transmit address 1 as address 3
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_JUMP_TO_TRANSMIT_ADDRESS_1_AS_ADDRESS_3] = JUMP(&DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_TRANSMIT_ADDRESS_1_AS_ADDRESS_3]); 
	//Transmit address 2 as address 3
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_TRANSMIT_ADDRESS_2_AS_ADDRESS_3] = TRANSMIT_FROM_HOB(MAC_ADDRESS_LEN,HOB_REGISTER_BF_REPORT_ADDRESS_2_OFFSET); 
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_JUMP_TO_LOAD_R3_WITH_VAP_DATABASE_ADDRESS] = JUMP(&DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_LOAD_R3_WITH_VAP_DATABASE_ADDRESS]); 
	//Transmit address 1 as address 3
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_TRANSMIT_ADDRESS_1_AS_ADDRESS_3] = TRANSMIT_FROM_HOB(MAC_ADDRESS_LEN,HOB_REGISTER_BF_REPORT_ADDRESS_1_OFFSET);

	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_LOAD_R3_WITH_VAP_DATABASE_ADDRESS] = LOAD_FROM_HOB(BF_REPORT_SEQUENCE_NUMBER_ADDRESS_REGISTER,HOB_REGISTER_BF_REPORT_VAP_DATABASE_TID_0_ADDRESS_OFFSET);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_LOAD_R5_WITH_SEQUENCE_NUMBER] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(BF_REPORT_SEQUENCE_NUMBER_REGISTER,BF_REPORT_SEQUENCE_NUMBER_ADDRESS_REGISTER,BF_REPORT_SEQUENCE_NUMBER_OFFSET);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_SHIFT_SEQUENCE_NUMBER_WITH_FRAGEMENT_BITS] = SHIFT_LEFT(BF_REPORT_SEQUENCE_NUMBER_REGISTER,FM_SEQUENCE_NUMBER_SHIFT);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_TRANSMIT_SEQUENCE_NUMBER] = TRANSMIT_FROM_REGISTER_WITH_IMMEDIATE_SIZE(SEQUENCE_NUMBER_SIZE,BF_REPORT_SEQUENCE_NUMBER_REGISTER);

	/* Update the sequence number */
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_INCREMENT_SEQUENCE_NUMBER] = ADD_REGISTER_AND_IMMEDIATE(BF_REPORT_SEQUENCE_NUMBER_REGISTER,BF_REPORT_SEQUENCE_NUMBER_REGISTER,VALUE_TO_ADD_TO_SEQUENCE_NUMBER);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_WRAP_AROUND_SEQUENCE_NUMBER] = MASK_REGISTER(BF_REPORT_SEQUENCE_NUMBER_REGISTER,BF_REPORT_SEQUENCE_NUMBER_BYTES_TO_MASK,BF_REPORT_SEQUENCE_NUMBER_MASK_PATTERN);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_SHIFT_RIGHT_SEQUENCE_NUMBER] = SHIFT_RIGHT(BF_REPORT_SEQUENCE_NUMBER_REGISTER,FM_SEQUENCE_NUMBER_SHIFT);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_SAVE_SEQUENCE_NUMBER] = STORE(BF_REPORT_SEQUENCE_NUMBER_REGISTER,BF_REPORT_SEQUENCE_NUMBER_ADDRESS_REGISTER,BF_REPORT_SEQUENCE_NUMBER_BYTES_TO_SAVE,BF_REPORT_SEQUENCE_NUMBER_OFFSET);

	/* Transmit HT control */
//	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_LOAD_R0_WITH_TEMPORARY_ENTRY_ADDRESS] = LOAD_FROM_HOB(BF_TEMPORARY_ENTRY_ADDRESS_REGISTER,HOB_REGISTER_BF_REPORT_TEMPORARY_ENTRY_ADDRESS_OFFSET);
//	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_WAIT_FOR_HT_CONTROL_OR_ERROR] = WAIT_BF_HT_CONTROL_READY_OR_ERROR;
//	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_TRANSMIT_HT_CONTROL] = TRANSMIT_FROM_HOB(HOB_REGISTER_BF_REPORT_HTC_FIELD_SIZE,HOB_REGISTER_BF_REPORT_HTC_FIELD_OFFSET);

	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_SKIP_IF_VHT] = SKIP_ON_HOB(FALSE,BF_REPORT_VHT_BIT_NUMBER_IN_BYTE_IN_CONTROL,HOB_REGISTER_BF_REPORT_CONTROL_REGISTER_OFFSET);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_JUMP_TO_HT_FRAME] = JUMP(&DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_HT_TRANSMIT_ACTION_CATEGORY_AND_VALUE]); 

	/* Transmit VHT action frame (VHT category) 
	   1)Transmit action category and value
	   2)Wait for BF report ready or error
	   3)Transmit MIMO (empty MIMO) in case of an error */
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_VHT_TRANSMIT_ACTION_CATEGORY_AND_VALUE] = TRANSMIT_IMMEDIATE(ACTION_CATEGORY_AND_FIELD_SIZE,BF_REPROT_ACTION_CATEGORY_AND_VALUE_IN_VHT);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_VHT_WAIT_FOR_REPORT_READY_OR_ERROR] = WAIT_BF_REPORT_READY_OR_ERROR;
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_VHT_SKIP_IF_ERROR] = SKIP_ON_HOB(FALSE,BF_REPORT_ERROR_BIT_IN_BYTE_IN_CONTROL,HOB_REGISTER_BF_REPORT_CONTROL_REGISTER_OFFSET);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_VHT_JUMP_TO_REGULAR_MIMO] =  JUMP(&DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_VHT_TRANSMIT_FIRST_PART_OF_REGULAR_MIMO]);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_VHT_TRANSMIT_EMPTY_MIMO] =  TRANSMIT_IMMEDIATE(BF_REPORT_TOTAL_MIMO_SIZE,BF_REPORT_EMPTY_MIMO_VALUE);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_VHT_JUMP_TO_VHT_REPORT_PAYLOAD] = JUMP(&DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_VHT_LOAD_R1_WITH_NUMBER_OF_COLUMNS]);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_VHT_TRANSMIT_FIRST_PART_OF_REGULAR_MIMO] = TRANSMIT_FROM_HOB(BF_REPORT_FIRST_PART_OF_REGULAR_MIMO_SIZE,HOB_REGISTER_BF_REPORT_FIRST_PART_OF_REGULAR_MIMO_OFFSET);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_VHT_TRANSMIT_SECOND_PART_OF_REGULAR_MIMO] = TRANSMIT_FROM_HOB(BF_REPORT_SECOND_PART_OF_REGULAR_MIMO_SIZE,HOB_REGISTER_BF_REPORT_SECOND_PART_OF_REGULAR_MIMO_OFFSET);

	/* Transmit BF report VHT payload and finish
	   1)Transmit SNR
	   2)Transmit BF payload
	   3)Transmit CRC and finish recipe*/
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_VHT_LOAD_R1_WITH_NUMBER_OF_COLUMNS] = LOAD_FROM_HOB(BF_REPORT_NUMBER_OF_COLUMNS_REGISTER,HOB_REGISTER_BF_REPORT_NUMBER_OF_COLUMNS_OFFSET);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_VHT_TRANSMIT_SNR] = TRANSMIT_FROM_REGISTER_WITH_LENGTH_FROM_REGISTER(BF_TEMPORARY_ENTRY_ADDRESS_REGISTER,BF_REPORT_NUMBER_OF_COLUMNS_REGISTER,BF_REPORT_SNR_OFFSET_IN_BF_ENTRY);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_VHT_LOAD_R4_WITH_BEAMFORMEE_TX_BUFFER_BASE_ADDRESS] = LOAD_FROM_HOB(BF_REPORT_BEAMFORMEE_TX_BUFFER_ADDRESS_REGISTER,HOB_REGISTER_BF_REPORT_BEAMFORMEE_TX_BUFFER_BASE_ADDRESS_OFFSET);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_VHT_ADD_OFFSET_TO_R4] = ADD_REGISTER_AND_IMMEDIATE(BF_REPORT_BEAMFORMEE_TX_BUFFER_ADDRESS_REGISTER, BF_REPORT_BEAMFORMEE_TX_BUFFER_ADDRESS_REGISTER, 0x8);
    DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_VHT_LOAD_R2_WITH_BEAMFORMEE_MATRICES_FINAL_SIZE] = LOAD_FROM_HOB(BF_REPORT_BEAMFORMEE_FINAL_SIZE_REGISTER,HOB_REGISTER_BF_REPORT_BEAMFORMEE_MATRICES_FINAL_SIZE);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_VHT_TRANSMIT_BEAMFORMEE_MATRICES] = TRANSMIT_FROM_REGISTER_WITH_LENGTH_FROM_REGISTER(BF_REPORT_BEAMFORMEE_TX_BUFFER_ADDRESS_REGISTER,BF_REPORT_BEAMFORMEE_FINAL_SIZE_REGISTER,BF_REPORT_BEAMFORMEE_MATRICES_OFFSET);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_VHT_CRC] = TRANSMIT_FROM_HOB(CRC_LENGTH,HOB_REGISTER_CRC_OFFSET);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_VHT_END_OF_RECIPE] = END_RECIPE;

    /* Transmit HT action frame (HT category) 
	   1)Transmit action category and value 
	   2)Transmit MIMO
	   	  a)2 bytes of data
	      b)lower part of TSF
	   3)Transmit SNR
	   4)Wait for BF report ready or error 
	   5)In case of error reset the CRC and any case transmit the CRC and finish recipe */
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_HT_TRANSMIT_ACTION_CATEGORY_AND_VALUE] = TRANSMIT_IMMEDIATE(ACTION_CATEGORY_AND_FIELD_SIZE,BF_REPROT_ACTION_CATEGORY_AND_VALUE_IN_HT);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_HT_TRANSMIT_MIMO] = TRANSMIT_FROM_HOB(BF_REPORT_FIRST_PART_OF_REGULAR_MIMO_SIZE,HOB_REGISTER_BF_REPORT_FIRST_PART_OF_REGULAR_MIMO_OFFSET);

	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_HT_TRANSMIT_LOWER_TSF_BYTES] =  TRANSMIT_FROM_HOB(LOWER_TSF_SIZE,HOB_REGISTER_TIMESTAMP_RESULT_OFFSET);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_HT_LOAD_R1_WITH_NUMBER_OF_COLUMNS] = LOAD_FROM_HOB(BF_REPORT_NUMBER_OF_COLUMNS_REGISTER,HOB_REGISTER_BF_REPORT_NUMBER_OF_COLUMNS_OFFSET);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_HT_TRANSMIT_SNR] = TRANSMIT_FROM_REGISTER_WITH_LENGTH_FROM_REGISTER(BF_TEMPORARY_ENTRY_ADDRESS_REGISTER,BF_REPORT_NUMBER_OF_COLUMNS_REGISTER,BF_REPORT_SNR_OFFSET_IN_BF_ENTRY);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_HT_LOAD_R4_WITH_BEAMFORMEE_TX_BUFFER_BASE_ADDRESS] = LOAD_FROM_HOB(BF_REPORT_BEAMFORMEE_TX_BUFFER_ADDRESS_REGISTER,HOB_REGISTER_BF_REPORT_BEAMFORMEE_TX_BUFFER_BASE_ADDRESS_OFFSET);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_HT_WAIT_FOR_REPORT_READY_OR_ERROR] = WAIT_BF_REPORT_READY_OR_ERROR;//????why here?
    DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_HT_LOAD_R2_WITH_BEAMFORMEE_MATRICES_FINAL_SIZE] = LOAD_FROM_HOB(BF_REPORT_BEAMFORMEE_FINAL_SIZE_REGISTER,HOB_REGISTER_BF_REPORT_BEAMFORMEE_MATRICES_FINAL_SIZE);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_HT_TRANSMIT_BEAMFORMEE_MATRICES] = TRANSMIT_FROM_REGISTER_WITH_LENGTH_FROM_REGISTER(BF_REPORT_BEAMFORMEE_TX_BUFFER_ADDRESS_REGISTER,BF_REPORT_BEAMFORMEE_FINAL_SIZE_REGISTER,BF_REPORT_BEAMFORMEE_MATRICES_OFFSET);	
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_HT_SKIP_IF_NOT_ERROR] = SKIP_ON_HOB(TRUE,BF_REPORT_ERROR_BIT_IN_BYTE_IN_CONTROL,HOB_REGISTER_BF_REPORT_CONTROL_REGISTER_OFFSET);

	/* The next command reset the CRC in case of an error to create bad CRC */
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_HT_RESET_CRC] = RESET_CRC;
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_HT_TRANSMIT_CRC] = TRANSMIT_FROM_HOB(CRC_LENGTH,HOB_REGISTER_CRC_OFFSET);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_HT_END_OF_RECIPE] = END_RECIPE;
}


/**********************************************************************************

deliaInitializeRecipeLookupTable  


Description:
------------
build the recipes lookup table

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void deliaInitializeRecipesLookupTable()
{
	uint32 i = 0;
	
	for(i = 0; i < DELIA_RECIPE_LOOKUP_TABLE_SIZE; i++)
	{
		if(NULL != DeliaRecipesLookUpTableDefines[i])
		{
			DeliaRecipesLookupTable[i] = JUMP(DeliaRecipesLookUpTableDefines[i]);
		}
		else
		{
			DeliaRecipesLookupTable[i] = END_RECIPE;
		}
	}
}


/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

DeliaRecipes_GetRecipe0Address  


Description:
------------
return the recipes base address (recipe 0 address)

Input:
-----
	
		
Output:
-------
	

Returns:
--------
recipes base address  
	
**********************************************************************************/

uint32* DeliaRecipes_GetRecipe0Address  ()

{	
	return DeliaRecipe0;  
}

/**********************************************************************************

DeliaRecipes_GetAckRecipeAddress  


Description:
------------
return ACK recipe address

Input:
-----
	
		
Output:
-------
	

Returns:
--------
ACK recipe address  
	
**********************************************************************************/

uint32* DeliaRecipes_GetAckRecipeAddress(void)

{	
	return DeliaRecipeAck;  
}

/**********************************************************************************

DeliaRecipes_GetBaRecipeAddress  


Description:
------------
return BA recipe address

Input:
-----
	
		
Output:
-------
	

Returns:
--------
BA recipe address 
	
**********************************************************************************/

uint32* DeliaRecipes_GetBaRecipeAddress  ()

{	
	return DeliaRecipeBa;  
}

/**********************************************************************************

DeliaRecipes_GetCtsAutoResponseRecipeAddress  


Description:
------------
return CTS auto response recipe address

Input:
-----
	
		
Output:
-------
	

Returns:
--------
CTS auto response recipe address 
	
**********************************************************************************/

uint32* DeliaRecipes_GetCtsAutoResponseRecipeAddress(void)

{	
	return DeliaRecipeCtsAutoResponse;  
}

/**********************************************************************************

DeliaRecipes_GetBfReportAutoResponseRecipeAddress  


Description:
------------
return BF report auto response recipe address

Input:
-----
	
		
Output:
-------
	

Returns:
--------
BF report auto response recipe address 
	
**********************************************************************************/

uint32* DeliaRecipes_GetBfReportAutoResponseRecipeAddress(void)

{	
	return DeliaRecipeBfReport;  
}

/**********************************************************************************

DeliaRecipes_ConstructRecipes  


Description:
------------
construct the Delia recipes and the recipes pointers table

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif
void DeliaRecipes_ConstructRecipes(void)
{	
    deliaConstructRecipe0();
	deliaConstructLegacyPacketRecipe();
	deliaConstructAggregationRecipe();
	deliaConstructBeaconRecipe();
	deliaConstructLegacyVhtRecipe();
	deliaConstructAckRecipe();
	deliaConstructBaRecipe();
	deliaConstructCtsAutoResponseRecipe();
	deliaConstructBfReportRecipe();
	deliaInitializeRecipesLookupTable();
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif


	
		













