/***************************************************************
 File:		DeliaRecipes.c
 Module:	  	Delia Recipes
 Purpose: 	holds all the implementation of the construction of the Delia recipes
 Description:  This module contains the implementation of the construction of the Delia
 			recipes and the recipes pointers table
***************************************************************/

/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "DeliaRecipes_Api.h"
#include "DeliaRecipes.h"
#include "ShramNewDeliaRecipes.h"
#include "protocol.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"
#include "HobRegs.h"
#include "Utils_Api.h"
#include "Deliad_Descriptors.h"
#include "Rcr_Descriptors.h"

#if defined (ENET_INC_UMAC)
#include "PhyDriver_API.h"
#endif


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_TX_HW_ACCELERATORS
#define LOG_LOCAL_FID   11



/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/
// We need this "ifndef" since this define is used in simulation which is not part of our code.
#ifndef DELIA_RECIPES_RELATIVE_ADDRESS
#define DELIA_RECIPES_RELATIVE_ADDRESS(x)	(x)
#endif

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void deliaConstructRecipe0(void);
static void deliaConstructAutoResponseSURecipe(void);
static void deliaConstructAutoResponseMuBroadcastRecipe(void);
static void deliaConstructAutoResponseMuUnicastRecipe(void);
static void deliaConstructBfReportRecipe(void); 

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

DeliaRecipes_t* pDeliaRecipes 			= NULL;
Tcr_t* 			pDeliaAutoResponseTcrs	= NULL;
SuTcrs_t*		pMbfmTcrs				= NULL;


/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/


/**********************************************************************************

deliaConstructRecipe0  


Description:
------------
construct the common recipe (recipe0)

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void deliaConstructRecipe0(void)
{
	uint32* DeliaRecipe0 = &(pDeliaRecipes->DeliaRecipe0[DELIA_RECIPE_0_FIRST_COMMAND]);
	
	//start non-auto frame recipe
	DeliaRecipe0[DELIA_RECIPE_0_ENABLE_PHY] = TX_CONTROL(DELIA_CONTROL_OVERRIDE_BASED_ON_TX, FALSE, FALSE, TRUE, TRUE, FALSE, DELIA_CONTROL_BUS_STATE_TRANMSIT_COMMON_TCRS, FALSE, TRUE, FALSE, FALSE);

	//kickstart tx sender
	DeliaRecipe0[DELIA_RECIPE_0_LOAD_DELIA_REGISTER_ASSERT_SENDER] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(DELIA_REGISTER_ASSERT_SENDER_VALUE, 0x1);
	DeliaRecipe0[DELIA_RECIPE_0_CREATE_INTERRUPT_TO_SENDER] = STORE_TO_HOB(DELIA_REGISTER_ASSERT_SENDER_VALUE, REG_HOB_DEL2GENRISC_START_TX_INT);

	//load base address of delia data structure
	DeliaRecipe0[DELIA_RECIPE_0_LOAD_DELIA_REGISTER_DELIAD_BASE_ADDRESS] = LOAD_FROM_HOB(DELIA_REGISTER_DELIAD_BASE_ADDRESS, REG_HOB_TX_DATA_STRUCTURE_ADDR);

	//load base address of auto generated tcrs set 1
	DeliaRecipe0[DELIA_RECIPE_0_LOAD_DELIA_REGISTER_AUTO_GENERATED_TCRS1_BASE_ADDRESS] = LOAD_FROM_HOB(DELIA_REGISTER_AUTO_GENERATED_TCRS_BASE_ADDRESS, REG_HOB_TCR_BASE_ADDR1);

	//Wait for either manual tcrs ready event0 or for all auto tcrs ready event
	DeliaRecipe0[DELIA_RECIPE_0_WAIT_FOR_INPUT_TO_BE_READY] = WAIT(TCR0_READY_EVENT | ALL_TCRS_IN_HOB_READY_EVENT);
	
	//Check if auto tcrs are ready and if so skip the next line
	DeliaRecipe0[DELIA_RECIPE_0_SKIP_IF_ALL_TCRS_READY] = SKIP_ON_HOB(FALSE, ALL_TCRS_IN_HOB_READY_EVENT_BIT_OFFSET_IN_BYTE, (REG_HOB_NOP_WAIT_EVENT_VECTOR + ALL_TCRS_IN_HOB_READY_EVENT_BYTE_OFFSET_IN_REG));

	//if we got here we need to use manual tcrs and event0 is set - we jump to manual tcrs handling
	DeliaRecipe0[DELIA_RECIPE_0_JUMP_TO_SEND_TCRS_FROM_DELIA_DESC] = JUMP(DELIA_RECIPES_RELATIVE_ADDRESS(&DeliaRecipe0[DELIA_RECIPE_0_MANUAL_TCRS_SEND_TCR0]));

	//transmit auto tcrs:
	//--------------------------

	//Check if auto tcrs set1 or set2 should be used
	DeliaRecipe0[DELIA_RECIPE_0_SKIP_IF_TCR_SET1_IS_USED] = SKIP_ON_HOB(TRUE, ALL_TCRS_IN_HOB_USE_TCR_SET2_EVENT_BIT_OFFSET_IN_BYTE, (REG_HOB_NOP_WAIT_EVENT_VECTOR + ALL_TCRS_IN_HOB_READY_EVENT_BYTE_OFFSET_IN_REG));

	//load base address of auto generated tcrs set 2
	DeliaRecipe0[DELIA_RECIPE_0_LOAD_DELIA_REGISTER_AUTO_GENERATED_TCRS2_BASE_ADDRESS] = LOAD_FROM_HOB(DELIA_REGISTER_AUTO_GENERATED_TCRS_BASE_ADDRESS, REG_HOB_TCR_BASE_ADDR2);

#ifdef WORKAROUND_FOR_HW_BUG_IN_TCR_N_HE_LTF_MU_OFDMA
	
	//transmit common TCR0
	DeliaRecipe0[DELIA_RECIPE_0_AUTO_TCRS_TRANSMIT_COMMON_TCR0] = TRANSMIT_ALIGNED_TAIL_FROM_REGISTER(DELIA_REGISTER_AUTO_GENERATED_TCRS_BASE_ADDRESS, COMMON_TCR0_LENGTH_TO_SEND, 0x0);

	//wait for ctrl word to be ready in delia desc
	DeliaRecipe0[DELIA_RECIPE_0_AUTO_TCRS_WAIT_FOR_REST_TCRS_AND_CONTROL_WORD_TO_BE_READY] = WAIT(WORD5_TO_WORD6_READY_EVENT);

	//transmit common TCR1
	DeliaRecipe0[DELIA_RECIPE_0_AUTO_TCRS_TRANSMIT_COMMON_TCR1] = TRANSMIT_ALIGNED_TAIL_FROM_REGISTER(DELIA_REGISTER_AUTO_GENERATED_TCRS_BASE_ADDRESS, COMMON_TCR1_LENGTH_TO_SEND, COMMON_TCR0_LENGTH_TO_SEND);

#else
	//transmit common
	DeliaRecipe0[DELIA_RECIPE_0_AUTO_TCRS_TRANSMIT_COMMON] = TRANSMIT_ALIGNED_TAIL_FROM_REGISTER(DELIA_REGISTER_AUTO_GENERATED_TCRS_BASE_ADDRESS, COMMON_TCR_LENGTH_TO_SEND, 0x0);

#endif //WORKAROUND_FOR_HW_BUG_IN_TCR_N_HE_LTF_MU_OFDMA

	//set delia to transmit per user tcrs
	DeliaRecipe0[DELIA_RECIPE_0_AUTO_TCRS_SET_TX_CONTROL_PER_USER_TCR_STATE] = TX_CONTROL(DELIA_CONTROL_NO_OVERRIDE, FALSE, FALSE, FALSE, TRUE, FALSE, DELIA_CONTROL_BUS_STATE_TRANSMIT_PER_USER_TCRS, FALSE, TRUE, FALSE, FALSE);
	
	//load the number of total bytes of user TCRs
	DeliaRecipe0[DELIA_RECIPE_0_AUTO_TCRS_LOAD_DELIA_REGISTER_PER_USER_TCRS_TOTAL_LENGTH] = LOAD_FROM_HOB(DELIA_REGISTER_PER_USER_TCRS_TOTAL_LENGTH, REG_HOB_NUM_OF_USER_TCR_BYTES);

	//trasmit all per user tcrs
	DeliaRecipe0[DELIA_RECIPE_0_AUTO_TCRS_TRANSMIT_PER_USER] = TRANSMIT_FROM_REGISTER_WITH_LENGTH_FROM_REGISTER(DELIA_REGISTER_AUTO_GENERATED_TCRS_BASE_ADDRESS, DELIA_REGISTER_PER_USER_TCRS_TOTAL_LENGTH, COMMON_TCR_LENGTH_TO_SEND);

	//wait for ctrl word to be ready in delia desc. If we were in WORKAROUND_FOR_HW_BUG_IN_TCR_N_HE_LTF_MU_OFDMA, it will happen immediatly
	DeliaRecipe0[DELIA_RECIPE_0_AUTO_TCRS_WAIT_FOR_CTRL_WORD_TO_BE_READY] = WAIT(WORD5_TO_WORD6_READY_EVENT);

	//jump to ctrl word handling
	DeliaRecipe0[DELIA_RECIPE_0_AUTO_TCRS_JUMP_TO_CTRL_WORD_HANDLING] = JUMP(DELIA_RECIPES_RELATIVE_ADDRESS(&DeliaRecipe0[DELIA_RECIPE_0_LOAD_DELIA_REGISTER_TX_CTRL_WORD_ADDRESS]));

	//transmit tcrs from delia data structure:
	//---------------------------------------

	//transmit common tcr0
	DeliaRecipe0[DELIA_RECIPE_0_MANUAL_TCRS_SEND_TCR0] = TRANSMIT_ALIGNED_TAIL_FROM_REGISTER(DELIA_REGISTER_DELIAD_BASE_ADDRESS, COMMON_TCR0_LENGTH_TO_SEND, 0x0);

	//Wait for the rest of tcrs (and ctrl word) to be ready
	DeliaRecipe0[DELIA_RECIPE_0_MANUAL_TCRS_WAIT_FOR_TCR1_TO_CTRL_WORD_TO_BE_READY] = WAIT(TCR1_TO_WORD6_READY_EVENT);

	//transmit common tcr1
	DeliaRecipe0[DELIA_RECIPE_0_MANUAL_TCRS_SEND_TCR1] = TRANSMIT_ALIGNED_TAIL_FROM_REGISTER(DELIA_REGISTER_DELIAD_BASE_ADDRESS, COMMON_TCR1_LENGTH_TO_SEND, COMMON_TCR0_LENGTH_TO_SEND);

	//set delia to transmit per user tcrs
	DeliaRecipe0[DELIA_RECIPE_0_MANUAL_TCRS_SET_TX_CONTROL_PER_USER_TCR_STATE] = TX_CONTROL(DELIA_CONTROL_NO_OVERRIDE, FALSE, FALSE, FALSE, TRUE, FALSE, DELIA_CONTROL_BUS_STATE_TRANSMIT_PER_USER_TCRS, FALSE, TRUE, FALSE, FALSE);
	
	//load the number of total bytes of user TCRs
	DeliaRecipe0[DELIA_RECIPE_0_MANUAL_TCRS_LOAD_DELIA_REGISTER_PER_USER_TCRS_TOTAL_LENGTH] = LOAD_FROM_HOB(DELIA_REGISTER_PER_USER_TCRS_TOTAL_LENGTH, REG_HOB_NUM_OF_USER_TCR_BYTES);

	//trasmit all per user tcrs
	DeliaRecipe0[DELIA_RECIPE_0_MANUAL_TCRS_TRANSMIT_PER_USER] = TRANSMIT_FROM_REGISTER_WITH_LENGTH_FROM_REGISTER(DELIA_REGISTER_DELIAD_BASE_ADDRESS, DELIA_REGISTER_PER_USER_TCRS_TOTAL_LENGTH, COMMON_TCR_LENGTH_TO_SEND);

	//Load ctrl word:
	//--------------
	DeliaRecipe0[DELIA_RECIPE_0_LOAD_DELIA_REGISTER_TX_CTRL_WORD_ADDRESS] = ADD_REGISTER_AND_IMMEDIATE(DELIA_REGISTER_TX_CTRL_WORD_ADDRESS, DELIA_REGISTER_DELIAD_BASE_ADDRESS, OFFSETOF(Deliad_t, common) + OFFSETOF(DeliadCommon_t, word12));
	DeliaRecipe0[DELIA_RECIPE_0_LOAD_DELIA_REGISTER_MPDU_HEADER_LENGTH] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(DELIA_REGISTER_MPDU_HEADER_LENGTH, DELIA_REGISTER_TX_CTRL_WORD_ADDRESS, 0x0);
	DeliaRecipe0[DELIA_RECIPE_0_MASK_DELIA_REGISTER_MPDU_HEADER_LENGTH] = MASK_REGISTER(DELIA_REGISTER_MPDU_HEADER_LENGTH, 0x3, 0xFF);

	//start data
	DeliaRecipe0[DELIA_RECIPE_0_SET_TX_CONTROL_DATA_STATE] = TX_CONTROL(DELIA_CONTROL_NO_OVERRIDE, FALSE, FALSE, FALSE, TRUE, FALSE, DELIA_CONTROL_BUS_STATE_TRANSMIT_DATA, FALSE, TRUE, FALSE, TRUE);

	//check if new delia needs to run
	DeliaRecipe0[DELIA_RECIPE_0_SKIP_IF_OLD_DELIA] = SKIP(DELIA_REGISTER_TX_CTRL_WORD_ADDRESS, FALSE, 0x0, 0x2);

	//jump to new delia handling
	DeliaRecipe0[DELIA_RECIPE_0_JUMP_TO_NEW_DELIA_HANDLING] = JUMP(DELIA_RECIPES_RELATIVE_ADDRESS(&DeliaRecipe0[DELIA_RECIPE_0_WAIT_FOR_DURATION_REG_READY_EVENT]));

	//check if this is a beacon frame
	DeliaRecipe0[DELIA_RECIPE_0_SKIP_IF_BEACON] = SKIP(DELIA_REGISTER_TX_CTRL_WORD_ADDRESS, TRUE, 0x1, 0x2);

	//jump to beacon handling
	DeliaRecipe0[DELIA_RECIPE_0_JUMP_TO_BEACON_HANDLING] = JUMP(DELIA_RECIPES_RELATIVE_ADDRESS(&DeliaRecipe0[DELIA_RECIPE_0_BEACON_CALCULATE_FIRST_ELEMENT_DESCRIPTOR_OFFSET]));

	//check if this is a trigger frame
	DeliaRecipe0[DELIA_RECIPE_0_SKIP_IF_TF] = SKIP(DELIA_REGISTER_TX_CTRL_WORD_ADDRESS, TRUE, 0x3, 0x2);

	//jump to TF handling
	DeliaRecipe0[DELIA_RECIPE_0_JUMP_TO_TF_HANDLING] = JUMP(DELIA_RECIPES_RELATIVE_ADDRESS(&DeliaRecipe0[DELIA_RECIPE_0_TF_LOAD_DELIA_REGISTER_MPDU_COUNTER_VALUE]));

	//Check if this is phy NDP transmission (indicated by length == 0) and if so skip directly to RCR check
	DeliaRecipe0[DELIA_RECIPE_0_CHECK_IF_LENGTH_ZERO] = SKIP_ON_HOB(TRUE, 0x0, REG_HOB_PAC_PHY_TX_LENGTH_ZERO);
	DeliaRecipe0[DELIA_RECIPE_0_LENGTH_ZERO_JUMP_TO_POST_DATA] = JUMP(DELIA_RECIPES_RELATIVE_ADDRESS(&DeliaRecipe0[DELIA_RECIPE_0_SKIP_IF_NO_RCR]));

	//old delia legacy frame handling:
	//-------------------------------
	DeliaRecipe0[DELIA_RECIPE_0_LEGACY_PACKET_LOAD_DELIA_REGISTER_MPDU_COUNTER_VALUE] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(DELIA_REGISTER_MPDU_COUNTER_VALUE, 0x1);
	DeliaRecipe0[DELIA_RECIPE_0_LEGACY_PACKET_STORE_DELIA_REGISTER_MPDU_COUNTER_VALUE_TO_HOB] = STORE_TO_HOB(DELIA_REGISTER_MPDU_COUNTER_VALUE, REG_HOB_HOB_DEL_COUNTER0_EVENT);		
	DeliaRecipe0[DELIA_RECIPE_0_LEGACY_PACKET_WAIT_FOR_MPDU_TO_BE_READY] = WAIT_FOR_MPDU_OR_BEACON_TO_BE_READY;
	DeliaRecipe0[DELIA_RECIPE_0_LEGACY_PACKET_LOAD_DELIA_REGISTER_MPDU_HEADER_ADDRESS] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(DELIA_REGISTER_MPDU_HEADER_ADDRESS, DELIA_REGISTER_DELIAD_BASE_ADDRESS, OFFSETOF(Deliad_t, mpdu[0]) + OFFSETOF(DeliadMpdu_t, word1));
	DeliaRecipe0[DELIA_RECIPE_0_LEGACY_PACKET_SHIFT_LEFT_HEADER_ADDRESS] = SHIFT_LEFT(DELIA_REGISTER_MPDU_HEADER_ADDRESS, 0x7);
	DeliaRecipe0[DELIA_RECIPE_0_LEGACY_PACKET_SHIFT_RIGHT_HEADER_ADDRESS] = SHIFT_RIGHT(DELIA_REGISTER_MPDU_HEADER_ADDRESS, 0x7);
	DeliaRecipe0[DELIA_RECIPE_0_LEGACY_PACKET_TRANSMIT_HEADER] = TRANSMIT_FROM_REGISTER_WITH_LENGTH_FROM_REGISTER(DELIA_REGISTER_MPDU_HEADER_ADDRESS, DELIA_REGISTER_MPDU_HEADER_LENGTH, 0x0);		
	DeliaRecipe0[DELIA_RECIPE_0_LEGACY_PACKET_LOAD_DELIA_REGISTER_MPDU_PAYLOAD_LENGTH] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(DELIA_REGISTER_MPDU_PAYLOAD_LENGTH, DELIA_REGISTER_DELIAD_BASE_ADDRESS, OFFSETOF(Deliad_t, mpdu[0]) + OFFSETOF(DeliadMpdu_t, word0));
	DeliaRecipe0[DELIA_RECIPE_0_LEGACY_PACKET_MASK_MPDU_PAYLOAD_LENGTH] = MASK_REGISTER(DELIA_REGISTER_MPDU_PAYLOAD_LENGTH, 0x3, 0xFF);
	DeliaRecipe0[DELIA_RECIPE_0_LEGACY_PACKET_LOAD_DELIA_REGISTER_MPDU_PAYLOAD_ADDRESS] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(DELIA_REGISTER_MPDU_PAYLOAD_ADDRESS, DELIA_REGISTER_DELIAD_BASE_ADDRESS, OFFSETOF(Deliad_t, mpdu[0]) + OFFSETOF(DeliadMpdu_t, word2));
	DeliaRecipe0[DELIA_RECIPE_0_LEGACY_PACKET_SHIFT_LEFT_PAYLOAD_ADDRESS] = SHIFT_LEFT(DELIA_REGISTER_MPDU_PAYLOAD_ADDRESS, 0x7);
	DeliaRecipe0[DELIA_RECIPE_0_LEGACY_PACKET_SHIFT_RIGHT_PAYLOAD_ADDRESS] = SHIFT_RIGHT(DELIA_REGISTER_MPDU_PAYLOAD_ADDRESS, 0x7);
	DeliaRecipe0[DELIA_RECIPE_0_LEGACY_PACKET_TRANSMIT_PAYLOAD] = TRANSMIT_FROM_REGISTER_WITH_LENGTH_FROM_REGISTER(DELIA_REGISTER_MPDU_PAYLOAD_ADDRESS, DELIA_REGISTER_MPDU_PAYLOAD_LENGTH, 0x0);
	DeliaRecipe0[DELIA_RECIPE_0_LEGACY_PACKET_TRANSMIT_CRC] = TRANSMIT_FROM_HOB(CRC_LENGTH, REG_HOB_TCC_CRC);

	//jump to recipe post data
	DeliaRecipe0[DELIA_RECIPE_0_LEGACY_JUMP_TO_POST_DATA] = JUMP(DELIA_RECIPES_RELATIVE_ADDRESS(&DeliaRecipe0[DELIA_RECIPE_0_SKIP_IF_NO_RCR]));
	
	//old delia beacon delia handling:
	//-------------------------------
    /* This commands is done in order to not have any operations after checking the last bit - just jump to start of loop : the address of the first element descriptor is set to be the (first element adress - element descriptor size)
           so in the loop the element descriptor adress is added with element descriptor size and the first element is handled as all the elements */
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_CALCULATE_FIRST_ELEMENT_DESCRIPTOR_OFFSET] = ADD_REGISTER_AND_IMMEDIATE(DELIA_REGISTER_CURRENT_ELEMENT_OFFSET, DELIA_REGISTER_DELIAD_BASE_ADDRESS, OFFSETOF(Deliad_t, mpdu[0]) - sizeof(DeliadMpdu_t));
	
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_CALCULATE_COUNTER] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(DELIA_REGISTER_BEACON_COUNTER, 0x1);
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_STORE_MPDU_COUNTER_INTO_HOB] = STORE_TO_HOB(DELIA_REGISTER_BEACON_COUNTER, REG_HOB_HOB_DEL_COUNTER0_EVENT);
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_WAIT_FOR_BEACON_TO_BE_READY] = WAIT_FOR_MPDU_OR_BEACON_TO_BE_READY;

	/* elements start of loop */
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_CALCULATE_ELEMENT_DESCRIPTOR_ADDRESS] = ADD_REGISTER_AND_IMMEDIATE(DELIA_REGISTER_CURRENT_ELEMENT_OFFSET, DELIA_REGISTER_CURRENT_ELEMENT_OFFSET, sizeof(DeliadMpdu_t));	
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_LOAD_R5_WITH_ELEMENT_LENGTH] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(DELIA_REGISTER_ELEMENT_LENGTH, DELIA_REGISTER_CURRENT_ELEMENT_OFFSET, OFFSETOF(DeliadMpdu_t, word0));		
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_MASK_ELEMENT_LENGTH] = MASK_REGISTER(DELIA_REGISTER_ELEMENT_LENGTH, 0x3, 0xFF);		

	/* regular element */
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_SKIP_IF_REGULAR_ELEMENT] = SKIP(DELIA_REGISTER_CURRENT_ELEMENT_OFFSET, FALSE, REGULAR_TYPE_BIT_NUMBER_IN_BYTE, ELEMENT_TYPE_OFFSET);		
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_JUMP_TO_SPECIAL_ELEMENT] = JUMP(DELIA_RECIPES_RELATIVE_ADDRESS(&DeliaRecipe0[DELIA_RECIPE_0_BEACON_SKIP_IF_TIMESTAMP_ELEMENT]));
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_LOAD_R6_WITH_ELEMENT_ADDRESS] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(DELIA_REGISTER_REGULAR_ELEMENT_ADDRESS, DELIA_REGISTER_CURRENT_ELEMENT_OFFSET, OFFSETOF(DeliadMpdu_t, word1));		
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_SHIFT_LEFT_ELEMENT_ADDRESS] = SHIFT_LEFT(DELIA_REGISTER_REGULAR_ELEMENT_ADDRESS, 0x7);
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_SHIFT_RIGHT_ELEMENT_ADDRESS] = SHIFT_RIGHT(DELIA_REGISTER_REGULAR_ELEMENT_ADDRESS, 0x7);
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_TRANSMIT_ELEMENT] = TRANSMIT_FROM_REGISTER_WITH_LENGTH_FROM_REGISTER(DELIA_REGISTER_REGULAR_ELEMENT_ADDRESS, DELIA_REGISTER_ELEMENT_LENGTH, 0x0);	
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_REGULAR_ELEMENT_JUMP_TO_CHECK_IF_LAST] = JUMP(DELIA_RECIPES_RELATIVE_ADDRESS(&DeliaRecipe0[DELIA_RECIPE_0_BEACON_CHECK_IF_LAST]));
		
	/* Timestamp element */	
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_SKIP_IF_TIMESTAMP_ELEMENT] = SKIP(DELIA_REGISTER_CURRENT_ELEMENT_OFFSET, FALSE, TIMESTAMP_TYPE_BIT_NUMBER_IN_BYTE, ELEMENT_TYPE_OFFSET);		
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_JUMP_TO_TIM_ELEMENT] = JUMP(DELIA_RECIPES_RELATIVE_ADDRESS(&DeliaRecipe0[DELIA_RECIPE_0_BEACON_TRANSMIT_CONSTANT_BYTES_OF_TIM_ELEMENT]));
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_LOAD_R4_WITH_VAP_INDEX] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(DELIA_REGISTER_VAP_INDEX, DELIA_REGISTER_CURRENT_ELEMENT_OFFSET, OFFSETOF(DeliadMpdu_t, word2));		
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_STORE_VAP_INDEX_INTO_HOB] = STORE_TO_HOB(DELIA_REGISTER_VAP_INDEX, REG_HOB_TSF_OFFSET_INDEX);		
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_TRANSMIT_TSF_ELEMENT] = TRANSMIT_FROM_HOB(0x8, REG_HOB_TSF_FIELD_SFR5_LOW);		
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_TSF_ELEMENT_JUMP_TO_CHECK_IF_LAST] = JUMP(DELIA_RECIPES_RELATIVE_ADDRESS(&DeliaRecipe0[DELIA_RECIPE_0_BEACON_CHECK_IF_LAST]));

	// TIM element 
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_TRANSMIT_CONSTANT_BYTES_OF_TIM_ELEMENT] = TRANSMIT_FROM_HOB(0x4, REG_HOB_TIM_IE_CONTROL); // Transmit   element Id + length + DTIM count + DTIM period 
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_LOAD_R4_WITH_TIM_BITMAP_PARAMETERS] = LOAD_FROM_HOB(DELIA_REGISTER_TIM_BITMAP_CTRL_LENGTH, REG_HOB_TIM_IE_HOB_RECIPE_FIELDS1); //load bitmap ctl length to R4 
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_MASK_BITMAP_LENGTH] = MASK_REGISTER(DELIA_REGISTER_TIM_BITMAP_CTRL_LENGTH, 0x1, 0x7); //Mask 3 LSBs (which holds the length - the reset are reserved)

	DeliaRecipe0[DELIA_RECIPE_0_BEACON_SET_R5_WITH_BITMAP_CTRL_OFFSET] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(DELIA_REGISTER_BITMAP_CONTROL_HOB_OFF, REG_HOB_TIM_IE_BITMAP_CONTROL);
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_MASK_OF_BITMAP_CTRL_OFFSET] = MASK_REGISTER(DELIA_REGISTER_BITMAP_CONTROL_HOB_OFF,0x3,0xFF);  
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_TRANSMIT_FROM_HOB_ACCORDING_TO_LENGTH] = TRANSMIT_FROM_HOB_WITH_LENGTH_FROM_REGISTER_AND_OFFSET_FROM_REGISTER(DELIA_REGISTER_TIM_BITMAP_CTRL_LENGTH, DELIA_REGISTER_BITMAP_CONTROL_HOB_OFF); // Transmit from HOB bitmap CTRL offset - the legnth which we have read

	DeliaRecipe0[DELIA_RECIPE_0_BEACON_LOAD_R4_WITH_TIM_IE_PARAMETERS] = LOAD_FROM_HOB(DELIA_REGISTER_TIM_IE_LENGTH, REG_HOB_TIM_IE_HOB_RECIPE_FIELDS0); // // HOB read length  [0-7] + HOB read offset [8-15]
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_COPY_R4_TO_R5_TO_SAVE_OFFSET] = COPY(DELIA_REGISTER_TIM_IE_OFFSET, DELIA_REGISTER_TIM_IE_LENGTH); // Copy R4 to R5 
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_MASK_TIM_IE_LENGTH] = MASK_REGISTER(DELIA_REGISTER_TIM_IE_LENGTH, 0x1, 0xFF); // MASK R4  - with the first byte , holds the length 
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_SHIFT_TIM_IE_OFFSET] = SHIFT_RIGHT(DELIA_REGISTER_TIM_IE_OFFSET, 0x8); // Shift right R5 and override the length,  ond left it with the offset value   
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_MASK_TIM_IE_OFFSET] = MASK_REGISTER(DELIA_REGISTER_TIM_IE_OFFSET, 0x1, 0xFF); // MASK only the first byte and R5 has the offset value 
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_ADD_BASE_ADDRESS_TO_BITMAP_OFFSET] = ADD_REGISTER_AND_IMMEDIATE(DELIA_REGISTER_TIM_IE_OFFSET, DELIA_REGISTER_TIM_IE_OFFSET, 0xC0C); // Add fixed value to the offset define 0xC0C
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_TRANSMIT_TIM_ELEMENT] = TRANSMIT_FROM_HOB_WITH_LENGTH_FROM_REGISTER_AND_OFFSET_FROM_REGISTER(DELIA_REGISTER_TIM_IE_LENGTH, DELIA_REGISTER_TIM_IE_OFFSET);  

	/* Check if last element */	
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_CHECK_IF_LAST] = SKIP(DELIA_REGISTER_CURRENT_ELEMENT_OFFSET, FALSE, ELEMENT_LAST_INDICATION_LOCATION_IN_BYTE, ELEMENT_LAST_INDICATION_OFFSET);
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_JUMP_TO_STRAT_OF_LOOP] = JUMP(DELIA_RECIPES_RELATIVE_ADDRESS(&DeliaRecipe0[DELIA_RECIPE_0_BEACON_START_OF_LOOP]));
    /* end of loop */
	
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_TRANSMIT_CRC] = TRANSMIT_FROM_HOB(CRC_LENGTH, REG_HOB_TCC_CRC);

	//jump to recipe post data
	DeliaRecipe0[DELIA_RECIPE_0_BEACON_JUMP_TO_POST_DATA] = JUMP(DELIA_RECIPES_RELATIVE_ADDRESS(&DeliaRecipe0[DELIA_RECIPE_0_SKIP_IF_NO_RCR]));

	//TF handling:
	//-----------
	DeliaRecipe0[DELIA_RECIPE_0_TF_LOAD_DELIA_REGISTER_MPDU_COUNTER_VALUE] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(DELIA_REGISTER_MPDU_COUNTER_VALUE, 0x1);
	DeliaRecipe0[DELIA_RECIPE_0_TF_STORE_DELIA_REGISTER_MPDU_COUNTER_VALUE_TO_HOB] = STORE_TO_HOB(DELIA_REGISTER_MPDU_COUNTER_VALUE, REG_HOB_HOB_DEL_COUNTER0_EVENT);
	DeliaRecipe0[DELIA_RECIPE_0_TF_WAIT_FOR_MPDU_TO_BE_READY] = WAIT_FOR_MPDU_OR_BEACON_TO_BE_READY;
	DeliaRecipe0[DELIA_RECIPE_0_TF_LOAD_DELIA_REGISTER_PAYLOAD_LENGTH] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(DELIA_REGISTER_TF_PAYLOAD_LENGTH, DELIA_REGISTER_DELIAD_BASE_ADDRESS, OFFSETOF(Deliad_t, mpdu[0]) + OFFSETOF(DeliadMpdu_t, word0));
	DeliaRecipe0[DELIA_RECIPE_0_TF_MASK_PAYLOAD_LENGTH] = MASK_REGISTER(DELIA_REGISTER_TF_PAYLOAD_LENGTH, 0x3, 0xFF);
	DeliaRecipe0[DELIA_RECIPE_0_TF_LOAD_DELIA_REGISTER_PADDING_LENGTH] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(DELIA_REGISTER_TF_PADDING_LENGTH, DELIA_REGISTER_DELIAD_BASE_ADDRESS, OFFSETOF(Deliad_t, mpdu[0]) + OFFSETOF(DeliadMpdu_t, word0));
	DeliaRecipe0[DELIA_RECIPE_0_TF_SHIFT_RIGHT_PADDING_LENGTH] = SHIFT_RIGHT(DELIA_REGISTER_TF_PADDING_LENGTH, 0x10);
	DeliaRecipe0[DELIA_RECIPE_0_TF_MASK_PADDING_LENGTH] = MASK_REGISTER(DELIA_REGISTER_TF_PADDING_LENGTH, 0x3, 0xFF);

	//check if we need to transmit EOF Delimiter
	DeliaRecipe0[DELIA_RECIPE_0_TF_SKIP_ON_PHY_MODE] = SKIP_ON_HOB(FALSE, 0x0, (REG_HOB_PHY_MODE + 0x1));
	DeliaRecipe0[DELIA_RECIPE_0_TF_JUMP_TO_SKIP_FLOW_CONTROL] = JUMP(DELIA_RECIPES_RELATIVE_ADDRESS(&DeliaRecipe0[DELIA_RECIPE_0_TF_SET_TX_CONTROL_SKIP_FLOW_CONTROL]));

	//Transmit eof delimiter	
	DeliaRecipe0[DELIA_RECIPE_0_TF_CALCULATE_LENGTH_HEADER_AND_CRC] = ADD_REGISTER_AND_IMMEDIATE(DELIA_REGISTER_TF_TOTAL_LENGTH_REGISTER, DELIA_REGISTER_TF_PAYLOAD_LENGTH, CRC_LENGTH);	
	DeliaRecipe0[DELIA_RECIPE_0_TF_CALCULATE_TOTAL_LENGTH] = ADD_REGISTER(DELIA_REGISTER_TF_TOTAL_LENGTH_REGISTER, DELIA_REGISTER_TF_PADDING_LENGTH);
	DeliaRecipe0[DELIA_RECIPE_0_TF_STORE_TOTAL_LENGTH_TO_HOB] = STORE_TO_HOB(DELIA_REGISTER_TF_TOTAL_LENGTH_REGISTER, REG_HOB_VHT_EOF_DELIMITER_DATA_IN);	
	DeliaRecipe0[DELIA_RECIPE_0_TF_TRANSMIT_DELIMITER] = TRANSMIT_ALIGNED_FROM_HOB(0x4, REG_HOB_DELIMITER_DATA_OUT);

	DeliaRecipe0[DELIA_RECIPE_0_TF_SET_TX_CONTROL_SKIP_FLOW_CONTROL] = TX_CONTROL(DELIA_CONTROL_NO_OVERRIDE, FALSE, TRUE, FALSE, TRUE, FALSE, DELIA_CONTROL_BUS_STATE_TRANSMIT_DATA, FALSE, TRUE, FALSE, TRUE);

	DeliaRecipe0[DELIA_RECIPE_0_TF_LOAD_DELIA_REGISTER_PAYLOAD_ADDRESS] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(DELIA_REGISTER_TF_PAYLOAD_ADDRESS, DELIA_REGISTER_DELIAD_BASE_ADDRESS, OFFSETOF(Deliad_t, mpdu[0]) + OFFSETOF(DeliadMpdu_t, word2));
	DeliaRecipe0[DELIA_RECIPE_0_TF_SHIFT_LEFT_PAYLOAD_ADDRESS] = SHIFT_LEFT(DELIA_REGISTER_TF_PAYLOAD_ADDRESS, 0x7);
	DeliaRecipe0[DELIA_RECIPE_0_TF_SHIFT_RIGHT_PAYLOAD_ADDRESS] = SHIFT_RIGHT(DELIA_REGISTER_TF_PAYLOAD_ADDRESS, 0x7);
	DeliaRecipe0[DELIA_RECIPE_0_TF_TRANSMIT_PAYLOAD] = TRANSMIT_FROM_REGISTER_WITH_LENGTH_FROM_REGISTER(DELIA_REGISTER_TF_PAYLOAD_ADDRESS, DELIA_REGISTER_TF_PAYLOAD_LENGTH, 0x0);

	//transmit padding
	DeliaRecipe0[DELIA_RECIPE_0_TF_TRANSMIT_PADDING] = TRANSMIT_FROM_HOB_WITH_LENGTH_FROM_REGISTER(DELIA_REGISTER_TF_PADDING_LENGTH, REG_HOB_ALL_ONES_PADDING);

	//transmit crc
	DeliaRecipe0[DELIA_RECIPE_0_TF_TRANSMIT_CRC] = TRANSMIT_FROM_HOB(CRC_LENGTH, REG_HOB_TCC_CRC);

	//jump to recipe post data
	DeliaRecipe0[DELIA_RECIPE_0_TF_JUMP_TO_POST_DATA] = JUMP(DELIA_RECIPES_RELATIVE_ADDRESS(&DeliaRecipe0[DELIA_RECIPE_0_SKIP_IF_NO_RCR]));

	//new delia handling:
	//------------------
	DeliaRecipe0[DELIA_RECIPE_0_WAIT_FOR_DURATION_REG_READY_EVENT] = WAIT(DURATION_REG_READY_EVENT);
	
#ifdef WORKAROUND_FOR_HW_BUG_IN_DELIA_AUTOFILL
	//prepare counter in hobe before enabling new delia. Sender will write the counter event only after starting new delia
	DeliaRecipe0[DELIA_RECIPE_0_POST_DATA_LOAD_DELIA_REGISTER_MPDU_COUNTER_VALUE] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(DELIA_REGISTER_MPDU_COUNTER_VALUE, 0x1);
	DeliaRecipe0[DELIA_RECIPE_0_POST_DATA_STORE_DELIA_REGISTER_MPDU_COUNTER_VALUE_TO_HOB] = STORE_TO_HOB(DELIA_REGISTER_MPDU_COUNTER_VALUE, REG_HOB_HOB_DEL_COUNTER0_EVENT);
#endif //WORKAROUND_FOR_HW_BUG_IN_DELIA_AUTOFILL

	DeliaRecipe0[DELIA_RECIPE_0_NEW_DELIA] = ENABLE_NEW_DELIA();
	
	//recipe post data:
	//----------------

#ifdef WORKAROUND_FOR_HW_BUG_IN_DELIA_AUTOFILL
	//wait for event from sender that bus state is not data (data is finished including possible autofill)
	DeliaRecipe0[DELIA_RECIPE_0_POST_DATA_WAIT_FOR_MAC_PHY_IF_BUS_STATE] = WAIT_FOR_MPDU_OR_BEACON_TO_BE_READY;
#endif //WORKAROUND_FOR_HW_BUG_IN_DELIA_AUTOFILL

	//check if we need to transmit RCRs
	DeliaRecipe0[DELIA_RECIPE_0_SKIP_IF_NO_RCR] = SKIP(DELIA_REGISTER_TX_CTRL_WORD_ADDRESS, FALSE, 0x2, 0x2);
	DeliaRecipe0[DELIA_RECIPE_0_JUMP_TO_END_RECIPE] = JUMP(DELIA_RECIPES_RELATIVE_ADDRESS(&DeliaRecipe0[DELIA_RECIPE_0_END_RECIPE]));

	//transmit auto rcrs:
	//-------------------

	//load base address of auto generated rcrs
	DeliaRecipe0[DELIA_RECIPE_0_LOAD_DELIA_REGISTER_AUTO_GENERATED_RCRS_BASE_ADDRESS] = LOAD_FROM_HOB(DELIA_REGISTER_AUTO_GENERATED_RCRS_BASE_ADDRESS, REG_HOB_RCR_BASE_ADDR);

	//set delia to transmit common rcrs
	DeliaRecipe0[DELIA_RECIPE_0_AUTO_RCRS_SET_TX_CONTROL_COMMON_RCR_STATE] = TX_CONTROL(DELIA_CONTROL_NO_OVERRIDE, FALSE, FALSE, TRUE, TRUE, FALSE, DELIA_CONTROL_BUS_STATE_TRANSMIT_COMMON_RCRS, FALSE, TRUE, FALSE, TRUE);

	//transmit common
	DeliaRecipe0[DELIA_RECIPE_0_AUTO_RCRS_TRANSMIT_COMMON] = TRANSMIT_ALIGNED_TAIL_FROM_REGISTER(DELIA_REGISTER_AUTO_GENERATED_RCRS_BASE_ADDRESS, COMMON_RCR_LENGTH_TO_SEND, 0x0);

	//set delia to transmit per user rcrs
	DeliaRecipe0[DELIA_RECIPE_0_AUTO_RCRS_SET_TX_CONTROL_PER_USER_RCR_STATE] = TX_CONTROL(DELIA_CONTROL_NO_OVERRIDE, FALSE, FALSE, TRUE, TRUE, FALSE, DELIA_CONTROL_BUS_STATE_TRANSMIT_PER_USER_RCRS, FALSE, TRUE, FALSE, TRUE);

	//increment address reg to start of per-user RCRs
	DeliaRecipe0[DELIA_RECIPE_0_AUTO_RCRS_SET_RCRS_POINTER_TO_PER_USER_SECTION] = ADD_REGISTER_AND_IMMEDIATE(DELIA_REGISTER_AUTO_GENERATED_RCRS_BASE_ADDRESS, DELIA_REGISTER_AUTO_GENERATED_RCRS_BASE_ADDRESS, sizeof(RcrCommon_t));

	//set DELIA_REGISTER_RCR_PER_USER_INDEX to 0
	DeliaRecipe0[DELIA_RECIPE_0_AUTO_RCRS_LOAD_DELIA_REGISTER_RCR_PER_USER_INDEX] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(DELIA_REGISTER_RCR_PER_USER_INDEX, 0x0);

	//transmit per-user RCR
	DeliaRecipe0[DELIA_RECIPE_0_AUTO_RCRS_TRANSMIT_PER_USER] = TRANSMIT_ALIGNED_TAIL_FROM_REGISTER(DELIA_REGISTER_AUTO_GENERATED_RCRS_BASE_ADDRESS, PER_USER_RCR_LENGTH_TO_SEND, 0x0);

	//increment address reg to start of next per-user RCRs
	DeliaRecipe0[DELIA_RECIPE_0_AUTO_RCRS_INCREMENT_RCRS_POINTER_TO_NEXT_PER_USER] = ADD_REGISTER_AND_IMMEDIATE(DELIA_REGISTER_AUTO_GENERATED_RCRS_BASE_ADDRESS, DELIA_REGISTER_AUTO_GENERATED_RCRS_BASE_ADDRESS, sizeof(RcrUser_t));
	
	//increment index reg to next user
	DeliaRecipe0[DELIA_RECIPE_0_AUTO_RCRS_INCREMENT_DELIA_REGISTER_RCR_PER_USER_INDEX] = ADD_REGISTER_AND_IMMEDIATE(DELIA_REGISTER_RCR_PER_USER_INDEX, DELIA_REGISTER_RCR_PER_USER_INDEX, 0x1);

	//iterate while index < NUM_OF_USERS_MU_HE (36)
	DeliaRecipe0[DELIA_RECIPE_0_AUTO_RCRS_SKIP_ON_INDEX] = SKIP_IF_EQUAL_TO_IMMEDIATE(DELIA_REGISTER_RCR_PER_USER_INDEX, FALSE, NUM_OF_USERS_MU_HE);
	DeliaRecipe0[DELIA_RECIPE_0_AUTO_RCRS_JUMP_TO_TRANSMIT_PER_USER_RCR] = JUMP(DELIA_RECIPES_RELATIVE_ADDRESS(&DeliaRecipe0[DELIA_RECIPE_0_AUTO_RCRS_TRANSMIT_PER_USER]));
	
	//recipe end:
	//-----------
	
	//currently only end recipe
	DeliaRecipe0[DELIA_RECIPE_0_END_RECIPE] = END_RECIPE;

}



/**********************************************************************************
deliaConstructAutoResponseSURecipe  

Description:
------------
construct Auto Response recipe

Input:
-----
		
Output:
-------

Returns:
--------
	void - 	
**********************************************************************************/
static void deliaConstructAutoResponseSURecipe(void)
{
	uint32 *DeliaRecipeAutoResponse = &(pDeliaRecipes->DeliaRecipeSUAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_FIRST_COMMAND]);

	// Start Auto Response recipe (enable TCR parameters override mode) and set bus state to " Common TCRs"
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_START_AUTO_FRAME] = TX_CONTROL(DELIA_CONTROL_OVERRIDE_BASED_ON_AUTO_REPLY, FALSE, FALSE, FALSE, FALSE, FALSE, DELIA_CONTROL_BUS_STATE_TRANMSIT_COMMON_TCRS, TRUE, TRUE, FALSE, FALSE);

	// Load the address of Common TCRs (TCR0 + TCR1) to R0
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_CREATE_TCR0_ADDR] = LOAD_FROM_HOB(AUTO_RESPONSE_TCR0_ADDRESS_REGISTER, REG_HOB_AUTO_RESP_TCR_BASE_ADDR);

	// Transmit common TCRs (TCR0 + TCR1)
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_SEND_COMMON_TCRS] = TRANSMIT_FROM_REGISTER(AUTO_RESPONSE_TCR0_ADDRESS_REGISTER, COMMON_TCR_LENGTH_TO_SEND, 0x0);

	// Change bus state to "per user tcrs" 
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_SET_STATE_TO_PER_USER_TCR] = TX_CONTROL(DELIA_CONTROL_NO_OVERRIDE, FALSE, FALSE, FALSE, FALSE, FALSE, DELIA_CONTROL_BUS_STATE_TRANSMIT_PER_USER_TCRS, TRUE, FALSE, FALSE, FALSE);

	// Get the Phy mode from HOB
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_LOAD_R3_WITH_PHY_MODE] = LOAD_FROM_HOB(AUTO_RESPONSE_PHY_MODE_REGISTER, REG_HOB_PHY_MODE);

	// Mask the phy mode (R3) with none he/vht phy modes
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_MASK_R3_WITH_NONE_HE_VHT] = MASK_REGISTER(AUTO_RESPONSE_PHY_MODE_REGISTER,0x1,AUTO_RESPONSE_NOT_HE_OR_VHT_MASK);

	// Wait until TCR length will be ready in HOB register
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_WAIT_FOR_TCR_LENGTH_READY] = WAIT_FOR_AUTO_REPLY_TCR_LENGTH_TO_BE_READY;

	// Get the frame length from HOB
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_LOAD_R4_WITH_FRAME_LEN] = LOAD_FROM_HOB(AUTO_RESPONSE_TCR_LEN_REGISTER, REG_HOB_DEL_TCR_FRAME_LEN);

	// Get the frame length from HOB
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_LOAD_R5_WITH_FRAME_LEN] = LOAD_FROM_HOB(AUTO_RESPONSE_TCR_LEN_UPDATE_REGISTER, REG_HOB_DEL_TCR_FRAME_LEN);

	// SKIP if not HE or VHT
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_SKIP_IF_NOT_HT_OR_VHT] = DO_SKIP_IF_NOT_EQUAL_TO_IMMEDIATE(AUTO_RESPONSE_PHY_MODE_REGISTER, 0x0);

	// we are in HE or in VHT, we need to add the delimiter size to the frame size
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_ADD_THE_DELIMITER_SIZE] = ADD_REGISTER_AND_IMMEDIATE(AUTO_RESPONSE_TCR_LEN_UPDATE_REGISTER, AUTO_RESPONSE_TCR_LEN_REGISTER,AUTO_RESPONSE_DELIMITER_SIZE);
	
	// Send first 32 bits of Per User TCR (user 0). It contains the length, which was calculated to include delimiter and CRC.
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_SEND_PER_USER_TCR_0_TO_31] = TRANSMIT_FROM_REGISTER_WITH_IMMEDIATE_SIZE(PER_USER_TCR_LENGTH_FIRST_4_BYTES, AUTO_RESPONSE_TCR_LEN_UPDATE_REGISTER);
	
	// Send the rest (96 bits) of Per User TCR (user 0).
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_SEND_PER_USER_TCR_32_TO_127] = TRANSMIT_FROM_REGISTER(AUTO_RESPONSE_TCR0_ADDRESS_REGISTER, PER_USER_TCR_LENGTH_BYTES_1_TO_3, PER_USER_TCR_OFFSET_BYTES_1_TO_3);

	// Change bus state to "Data" and init CRC
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_SET_STATE_TO_DATA] = TX_CONTROL(DELIA_CONTROL_NO_OVERRIDE, FALSE, TRUE, FALSE, FALSE, FALSE, DELIA_CONTROL_BUS_STATE_TRANSMIT_DATA, FALSE, FALSE, FALSE, TRUE);

	// Get the auto response frame pointer from HOB (it contains header + payload)
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_LOAD_R1_WITH_FRAME_POINTER] = LOAD_FROM_HOB(AUTO_RESPONSE_FRAME_POINTER_REGISTER, REG_HOB_DEL_DESC_FRAME_PTR);

	// Get the frame length from HOB (doesn't include FCS, delimeter, frame control, duration which are total of 12 bytes) 
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_LOAD_R2_WITH_FRAME_LENGTH] = LOAD_FROM_HOB(AUTO_RESPONSE_FRAME_LENGTH_REGISTER, REG_HOB_DEL_DESC_FRAME_LEN);

	// Wait for the frame payload to be ready in HOB
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_WAIT_FOR_FRAME_READY] = WAIT_FOR_AUTO_REPLY_FRAME_TO_BE_READY;
	
	// Prepare VHT delimiter that will be used in case it is needed  
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_UPDATE_DELIMITER] = STORE_TO_HOB(AUTO_RESPONSE_TCR_LEN_REGISTER,REG_HOB_VHT_EOF_DELIMITER_DATA_IN);

	// Skip the delimiter transnit if we are not in he or vht
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_SKIP_DELIMITER_IF_NOT_HT_OR_VHT] = DO_SKIP_IF_NOT_EQUAL_TO_IMMEDIATE(AUTO_RESPONSE_PHY_MODE_REGISTER, 0x0);

	// Transmit delimiter
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_TRANSMIT_DELIMITER] = TRANSMIT_FROM_HOB(DELIMITER_LENGTH,REG_HOB_DELIMITER_DATA_OUT);

	//Init the CRC again (delimiter should not be part of CRC. Bus state remains the same)
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_INIT_CRC_AGAIN] = TX_CONTROL(DELIA_CONTROL_NO_OVERRIDE, FALSE, TRUE, FALSE, FALSE, FALSE, DELIA_CONTROL_BUS_STATE_TRANSMIT_DATA, FALSE, FALSE, FALSE, TRUE);

	// Transmit 2 bytes of frame control from frame pointer
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_TRANSMIT_CONTROL] = TRANSMIT_FROM_REGISTER(AUTO_RESPONSE_FRAME_POINTER_REGISTER, FRAME_CONTROL_LEN, 0);

	// Wait for duration to be ready in the HOB
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_WAIT_FOR_DURATION] = WAIT_FOR_DURATION;

	// Duration type from HOB can be: 2'b10 - Add duration based last Rx frame / 2'b11 - Add duration based TX Sequence Counter.
	// If LSB == 1 skip the next command and go to (1) to transmit TX based duration
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_SKIP_IF_TX_BASED_DURATION] = SKIP_ON_HOB(FALSE, AUTO_RESPONSE_DUR_TYPE_BIT_NUMBER, REG_HOB_DEL_DESC_DUR_TYPE);
	
	// LSB == 0 so go to (2) to transmit RX based duration
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_JUMP_TO_RX_BASED_DURATION] = JUMP(DELIA_RECIPES_RELATIVE_ADDRESS(&DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_TRANSMIT_RX_BASED_DURATION]));

	// (1) Transmit TX Based Duration from HOB
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_TRANSMIT_TX_BASED_DURATION] = TRANSMIT_FROM_HOB(DUR_FIELD_LEN, REG_HOB_TX_MPDU_DUR_FIELD); 

	// Jump to header and payload transmission
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_JUMP_TO_FRAME_TRANSMISSION] = JUMP(DELIA_RECIPES_RELATIVE_ADDRESS(&DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_TRANSMIT_HEADER_AND_FRAME]));

	// (2) Transmit RX Based Duration from HOB
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_TRANSMIT_RX_BASED_DURATION] = TRANSMIT_FROM_HOB(DUR_FIELD_LEN, REG_HOB_AUTO_REPLY_DUR); 

	// Transmit the rest of the frame header and the payload. We skip 2 bytes of Frame Control and 2 bytes of Duration which were already transmitted.
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_TRANSMIT_HEADER_AND_FRAME] = TRANSMIT_FROM_REGISTER_WITH_LENGTH_FROM_REGISTER(AUTO_RESPONSE_FRAME_POINTER_REGISTER, AUTO_RESPONSE_FRAME_LENGTH_REGISTER, AUTO_RESPONSE_PAYLOAD_OFFSET);

	// transmit CRC from HOB
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_TRANSMIT_CRC] = TRANSMIT_FROM_HOB(CRC_LENGTH, REG_HOB_TCC_CRC);

	// End of recipe
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_END_RECIPE] = END_RECIPE;
}



/**********************************************************************************
deliaConstructAutoResponseMuBroadcastRecipe  

Description:
------------
construct Auto Response recipe

Input:
-----
		
Output:
-------

Returns:
--------
	void - 	
**********************************************************************************/
static void deliaConstructAutoResponseMuBroadcastRecipe(void)
{
	uint32 *DeliaRecipeAutoResponse = &(pDeliaRecipes->DeliaRecipeMUBroadcastAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_FIRST_COMMAND]);

	// Start Auto Response recipe and set bus state to " Common TCRs"
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_START_AUTO_FRAME] = TX_CONTROL(DELIA_CONTROL_OVERRIDE_BASED_ON_TX, FALSE, FALSE, FALSE, FALSE, FALSE, DELIA_CONTROL_BUS_STATE_TRANMSIT_COMMON_TCRS, FALSE, TRUE, FALSE, FALSE);

	// Load the address of Common TCRs (TCR0 + TCR1) to R0 
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_CREATE_TCR0_ADDR] = LOAD_FROM_HOB(AUTO_RESPONSE_TCR0_ADDRESS_REGISTER, REG_HOB_AUTO_RESP_HE_MU_TCR_BASE_ADDR);

	// Transmit common TCRs (TCR0 + TCR1)
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_SEND_COMMON_TCRS] = TRANSMIT_FROM_REGISTER(AUTO_RESPONSE_TCR0_ADDRESS_REGISTER, COMMON_TCR_LENGTH_TO_SEND, 0x0);

	// Change bus state to "per user tcrs" 
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_SET_STATE_TO_PER_USER_TCR] = TX_CONTROL(DELIA_CONTROL_NO_OVERRIDE, FALSE, FALSE, FALSE, FALSE, FALSE, DELIA_CONTROL_BUS_STATE_TRANSMIT_PER_USER_TCRS, FALSE, FALSE, FALSE, FALSE);

	// Get the Phy mode from HOB
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_LOAD_R3_WITH_PHY_MODE] = LOAD_FROM_HOB(AUTO_RESPONSE_PHY_MODE_REGISTER, REG_HOB_PHY_MODE);

	// Mask the phy mode (R3) with none he/vht phy modes
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_MASK_R3_WITH_NONE_HE_VHT] = MASK_REGISTER(AUTO_RESPONSE_PHY_MODE_REGISTER, 0x1, AUTO_RESPONSE_NOT_HE_OR_VHT_MASK);

	// Wait until TCR length will be ready in HOB register
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_WAIT_FOR_TCR_LENGTH_READY] = WAIT_FOR_AUTO_REPLY_TCR_LENGTH_TO_BE_READY;

	// Get the frame length from HOB
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_LOAD_R4_WITH_FRAME_LEN] = LOAD_FROM_HOB(AUTO_RESPONSE_TCR_LEN_REGISTER, REG_HOB_DEL_TCR_FRAME_LEN);

	// Get the frame length from HOB into the tcr length update in order to add the delimiter size if its needed
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_LOAD_R5_WITH_FRAME_LEN] = LOAD_FROM_HOB(AUTO_RESPONSE_TCR_LEN_UPDATE_REGISTER, REG_HOB_DEL_TCR_FRAME_LEN);

	// SKIP if not HE or VHT
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_SKIP_IF_NOT_HT_OR_VHT] = DO_SKIP_IF_NOT_EQUAL_TO_IMMEDIATE(AUTO_RESPONSE_PHY_MODE_REGISTER, 0x0);

	// we are in HE or in VHT, we need to add the delimiter size to the frame size
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_ADD_THE_DELIMITER_SIZE] = ADD_REGISTER_AND_IMMEDIATE(AUTO_RESPONSE_TCR_LEN_UPDATE_REGISTER, AUTO_RESPONSE_TCR_LEN_REGISTER,AUTO_RESPONSE_DELIMITER_SIZE);
	
	// Send first 32 bits of Per User TCR (user 0). It contains the length, which was calculated to include delimiter and CRC.
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_SEND_PER_USER_TCR_0_TO_31] = TRANSMIT_FROM_REGISTER_WITH_IMMEDIATE_SIZE(PER_USER_TCR_LENGTH_FIRST_4_BYTES, AUTO_RESPONSE_TCR_LEN_UPDATE_REGISTER);
	
	// Send the rest (96 bits) of Per User TCR (user 0).
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_SEND_PER_USER_TCR_32_TO_127] = TRANSMIT_FROM_REGISTER(AUTO_RESPONSE_TCR0_ADDRESS_REGISTER, PER_USER_TCR_LENGTH_BYTES_1_TO_3, PER_USER_TCR_OFFSET_BYTES_1_TO_3);

	// Change bus state to "Data" and init CRC
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_SET_STATE_TO_DATA] = TX_CONTROL(DELIA_CONTROL_NO_OVERRIDE, FALSE, TRUE, FALSE, FALSE, FALSE, DELIA_CONTROL_BUS_STATE_TRANSMIT_DATA, FALSE, FALSE, FALSE, TRUE);

	// Get the auto response frame pointer from HOB (it contains header + payload)
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_LOAD_R1_WITH_FRAME_POINTER] = LOAD_FROM_HOB(AUTO_RESPONSE_FRAME_POINTER_REGISTER, REG_HOB_DEL_DESC_FRAME_PTR);

	// Get the frame length from HOB (doesn't include FCS, delimeter, frame control, duration which are total of 12 bytes) 
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_LOAD_R2_WITH_FRAME_LENGTH] = LOAD_FROM_HOB(AUTO_RESPONSE_FRAME_LENGTH_REGISTER, REG_HOB_DEL_DESC_FRAME_LEN);

	// Wait for the frame payload to be ready in HOB
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_WAIT_FOR_FRAME_READY] = WAIT_FOR_AUTO_REPLY_FRAME_TO_BE_READY;
	
	// Prepare VHT delimiter that will be used in case it is needed  
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_UPDATE_DELIMITER] = STORE_TO_HOB(AUTO_RESPONSE_TCR_LEN_REGISTER,REG_HOB_VHT_EOF_DELIMITER_DATA_IN);

	// Skip the delimiter transnit if we are not in he or vht
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_SKIP_DELIMITER_IF_NOT_HT_OR_VHT] = DO_SKIP_IF_NOT_EQUAL_TO_IMMEDIATE(AUTO_RESPONSE_PHY_MODE_REGISTER, 0x0);

	// Transmit delimiter
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_TRANSMIT_DELIMITER] = TRANSMIT_FROM_HOB(DELIMITER_LENGTH,REG_HOB_DELIMITER_DATA_OUT);

	//Init the CRC again (delimiter should not be part of CRC. Bus state remains the same)
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_INIT_CRC_AGAIN] = TX_CONTROL(DELIA_CONTROL_NO_OVERRIDE, FALSE, TRUE, FALSE, FALSE, FALSE, DELIA_CONTROL_BUS_STATE_TRANSMIT_DATA, FALSE, FALSE, FALSE, TRUE);

	// Transmit 2 bytes of frame control from frame pointer
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_TRANSMIT_CONTROL] = TRANSMIT_FROM_REGISTER(AUTO_RESPONSE_FRAME_POINTER_REGISTER, FRAME_CONTROL_LEN, 0);

	// Wait for duration to be ready in the HOB
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_WAIT_FOR_DURATION] = WAIT_FOR_DURATION;

	// Duration type from HOB can be: 2'b10 - Add duration based last Rx frame / 2'b11 - Add duration based TX Sequence Counter.
	// If LSB == 1 skip the next command and go to (1) to transmit TX based duration
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_SKIP_IF_TX_BASED_DURATION] = SKIP_ON_HOB(FALSE, AUTO_RESPONSE_DUR_TYPE_BIT_NUMBER, REG_HOB_DEL_DESC_DUR_TYPE);
	
	// LSB == 0 so go to (2) to transmit RX based duration
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_JUMP_TO_RX_BASED_DURATION] = JUMP(DELIA_RECIPES_RELATIVE_ADDRESS(&DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_TRANSMIT_RX_BASED_DURATION]));

	// (1) Transmit TX Based Duration from HOB
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_TRANSMIT_TX_BASED_DURATION] = TRANSMIT_FROM_HOB(DUR_FIELD_LEN, REG_HOB_TX_MPDU_DUR_FIELD); 

	// Jump to header and payload transmission
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_JUMP_TO_FRAME_TRANSMISSION] = JUMP(DELIA_RECIPES_RELATIVE_ADDRESS(&DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_TRANSMIT_HEADER_AND_FRAME]));

	// (2) Transmit RX Based Duration from HOB
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_TRANSMIT_RX_BASED_DURATION] = TRANSMIT_FROM_HOB(DUR_FIELD_LEN, REG_HOB_AUTO_REPLY_DUR); 

	// Transmit the rest of the frame header and the payload. We skip 2 bytes of Frame Control and 2 bytes of Duration which were already transmitted.
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_TRANSMIT_HEADER_AND_FRAME] = TRANSMIT_FROM_REGISTER_WITH_LENGTH_FROM_REGISTER(AUTO_RESPONSE_FRAME_POINTER_REGISTER, AUTO_RESPONSE_FRAME_LENGTH_REGISTER, AUTO_RESPONSE_PAYLOAD_OFFSET);

	// transmit CRC from HOB
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_TRANSMIT_CRC] = TRANSMIT_FROM_HOB(CRC_LENGTH, REG_HOB_TCC_CRC);

	// End of recipe
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_END_RECIPE] = END_RECIPE;
}



/**********************************************************************************
deliaConstructAutoResponseMuUnicastRecipe  

Description:
------------
construct Auto Response recipe

Input:
-----
		
Output:
-------

Returns:
--------
	void - 	
**********************************************************************************/
static void deliaConstructAutoResponseMuUnicastRecipe(void)
{
	uint32 *DeliaRecipeAutoResponse = &(pDeliaRecipes->DeliaRecipeMUUnicastAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_MU_UNICAST_FIRST_COMMAND]);

	// Start Auto Response recipe and set bus state to " Common TCRs"
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_MU_UNICAST_START_AUTO_FRAME] = TX_CONTROL(DELIA_CONTROL_OVERRIDE_BASED_ON_TX, FALSE, FALSE, FALSE, FALSE, FALSE, DELIA_CONTROL_BUS_STATE_TRANMSIT_COMMON_TCRS, FALSE, TRUE, FALSE, FALSE);

	// Load the address of Common TCRs (TCR0 + TCR1) to R0
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_MU_UNICAST_CREATE_TCR0_ADDR] = LOAD_FROM_HOB(AUTO_RESPONSE_TCR0_ADDRESS_REGISTER, REG_HOB_AUTO_RESP_HE_MU_TCR_BASE_ADDR);

	// Transmit common TCRs (TCR0 + TCR1)
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_MU_UNICAST_SEND_COMMON_TCRS] = TRANSMIT_FROM_REGISTER(AUTO_RESPONSE_TCR0_ADDRESS_REGISTER, COMMON_TCR_LENGTH_TO_SEND, 0x0);

	// Change bus state to "per user tcrs" 
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_MU_UNICAST_SET_STATE_TO_PER_USER_TCR] = TX_CONTROL(DELIA_CONTROL_NO_OVERRIDE, FALSE, FALSE, FALSE, FALSE, FALSE, DELIA_CONTROL_BUS_STATE_TRANSMIT_PER_USER_TCRS, FALSE, FALSE, FALSE, FALSE);

	// Wait until TCR length will be ready in HOB register
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_MU_UNICAST_WAIT_FOR_TCR_LENGTH_READY] = WAIT_FOR_AUTO_REPLY_TCR_LENGTH_TO_BE_READY;

	//load the number of total bytes of user TCRs
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_MU_UNICAST_TCRS_LOAD_DELIA_REGISTER_PER_USER_TCRS_TOTAL_LENGTH] = LOAD_FROM_HOB(AUTO_RESPONSE_PER_USER_TCR_TOTAL_LENGTH, REG_HOB_NUM_OF_USER_TCR_BYTES);

	//trasmit all per user tcrs
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_MU_UNICAST_TCRS_TRANSMIT_PER_USER] = TRANSMIT_FROM_REGISTER_WITH_LENGTH_FROM_REGISTER(AUTO_RESPONSE_TCR0_ADDRESS_REGISTER, AUTO_RESPONSE_PER_USER_TCR_TOTAL_LENGTH, COMMON_TCR_LENGTH_TO_SEND);

	// Change bus state to "Data" and init CRC
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_MU_UNICAST_SET_STATE_TO_DATA] = TX_CONTROL(DELIA_CONTROL_NO_OVERRIDE, FALSE, TRUE, TRUE, FALSE, FALSE, DELIA_CONTROL_BUS_STATE_TRANSMIT_DATA, FALSE, FALSE, FALSE, TRUE);

	// Enable new HW
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_MU_UNICAST_ENABLE_NEW_DELIA] = ENABLE_NEW_DELIA();

	// End recipe
	DeliaRecipeAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_MU_UNICAST_END_RECIPE] = END_RECIPE;
}

/**********************************************************************************
deliaConstructBfReportRecipe  

Description:
------------
construct Beemforming report recipe 

Input:
-----
		
Output:
-------

Returns:
--------
	void - 	
**********************************************************************************/
static void deliaConstructBfReportRecipe(void)
{
	uint32 *DeliaRecipeBfReport = &(pDeliaRecipes->DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_REPORT_FIRST_COMMAND]);
	
	// Start BF report recipe and set bus state to " Common TCRs"
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_REPORT_ENABLE_BUS_STATE] = TX_CONTROL(DELIA_CONTROL_OVERRIDE_BASED_ON_AUTO_REPLY_BF_REPORT, FALSE, FALSE, FALSE, FALSE, FALSE, DELIA_CONTROL_BUS_STATE_TRANMSIT_COMMON_TCRS, FALSE, TRUE, FALSE, FALSE);
	//	Read via HOB register tx_recipe_base_addr the pointer to TCRs in the TX Recipe area
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_REPORT_READ_TX_RECIPE_BASE_ADDR_FROM_HOB]  = LOAD_FROM_HOB(BFEE_REPORT_TX_RECIPE_BASE_ADDRESS_REGISTER, REG_HOB_TX_RECIPE_BASE_ADDR);
	//	Transfer common TCR0 + TCR1 from TX Recipe (total 8 DWs)
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_REPORT_TRANSMIT_TCR0_TCR1] = TRANSMIT_FROM_REGISTER(BFEE_REPORT_TX_RECIPE_BASE_ADDRESS_REGISTER,COMMON_TCR_LENGTH_TO_SEND,0x0); 
	// Change bus state to "per user tcrs" 
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_REPORT_SET_STATE_TO_PER_USER_TCR]= TX_CONTROL(DELIA_CONTROL_NO_OVERRIDE, FALSE, FALSE, FALSE, FALSE, FALSE, DELIA_CONTROL_BUS_STATE_TRANSMIT_PER_USER_TCRS, FALSE, FALSE, FALSE, FALSE);
	//	Wait for frame length ready event 
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_REPORT_WAIT_FOR_FRAME_LENGTH_READY] = WAIT(BFEE_LENGTH_READY);		
	// Get the frame length from HOB
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_READ_TCR_FRAME_LENGHT_FROM_HOB] = LOAD_FROM_HOB(BFEE_REPORT_TCR_LENGTH_REGISTER, REG_HOB_LENGTH_WITHOUT_DELIMITER);
	// Get the Phy mode from HOB
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_READ_PHY_MODE_FROM_HOB] = LOAD_FROM_HOB(BFEE_REPORT_PHY_MODE_REGISTER, REG_HOB_PHY_MODE);
	// Mask the phy mode (R3) with none he/vht phy modes
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_MASK_R3_WITH_NONE_HE_VHT] = MASK_REGISTER(BFEE_REPORT_PHY_MODE_REGISTER, 0x1, AUTO_RESPONSE_NOT_HE_OR_VHT_MASK);
	// SKIP if not HE or VHT
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_SKIP_IF_NOT_HE_OR_VHT] = DO_SKIP_IF_NOT_EQUAL_TO_IMMEDIATE(BFEE_REPORT_PHY_MODE_REGISTER, 0x0);
	// we are in HE or in VHT, we need to add the delimiter size to the frame size
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_ADD_THE_DELIMITER_SIZE] = ADD_REGISTER_AND_IMMEDIATE(BFEE_REPORT_TCR_LENGTH_REGISTER, BFEE_REPORT_TCR_LENGTH_REGISTER,BFEE_DELIMITER_SIZE);
	// Send first 32 bits of Per User TCR (user 0). It contains the length, which was calculated to include delimiter and CRC.
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_SEND_PER_USER_TCR_0_TO_31] = TRANSMIT_FROM_REGISTER_WITH_IMMEDIATE_SIZE(PER_USER_TCR_LENGTH_FIRST_4_BYTES, BFEE_REPORT_TCR_LENGTH_REGISTER);
#ifdef WORKAROUND_FOR_BFEE_SUB_BAND_HW_BUG
	//skip if not HE 
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_REPORT_SKIP_IF_HE] = SKIP_ON_HOB(FALSE, BFEE_PHY_MODE_HE_BIT_NUM, REG_HOB_PHY_MODE);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_REPORT_JUMP_TO_NONE_HE] = JUMP(DELIA_RECIPES_RELATIVE_ADDRESS(&DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_REPORT_TRANSMIT_FIRST_USER_TCR]));
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_LOAD_FIFTH_BYTE_FROM_COMMON_TCR] = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(BFEE_REPORT_BFEE_WA_REGISTER,BFEE_REPORT_TX_RECIPE_BASE_ADDRESS_REGISTER,BFEE_BW_OFFSET_ON_TCR);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_SHIF_BW_FIELD_TO_LSB] = SHIFT_RIGHT(BFEE_REPORT_BFEE_WA_REGISTER,10);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_MASK_BW_BITS] = MASK_REGISTER(BFEE_REPORT_BFEE_WA_REGISTER, 0x1, BFEE_BW_MASK);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_CALC_TABLE_OFFSET_IN_HOB] = ADD_REGISTER_AND_IMMEDIATE(BFEE_REPORT_BFEE_WA_REGISTER,BFEE_REPORT_BFEE_WA_REGISTER,REG_HOB_REG2HOB_GP0); 
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_LOAD_REGISTER_WITH_BYTE_LENGTH] = LOAD_REGISTER_WITH_IMMEDIATE_VALUE(BFEE_ONE_BYTE_LENGTH_REGISTER,BFEE_ONE_BYTE_LENGTH); 
	// Transfer from WA register a byte which replace byte 28 in TCR 
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_TRANSMIT_ONE_FIFTH_BYTE_FROM_USER_TCR] = TRANSMIT_FROM_HOB_WITH_LENGTH_FROM_REGISTER_AND_OFFSET_FROM_REGISTER(BFEE_ONE_BYTE_LENGTH_REGISTER,BFEE_REPORT_BFEE_WA_REGISTER);
	// Transfer from TX Recipe + offset 0x20 1 User TCR (11 bytes)
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_REPORT_TRANSMIT_FIRST_USER_TCR_WITH_WA] = TRANSMIT_FROM_REGISTER(BFEE_REPORT_TX_RECIPE_BASE_ADDRESS_REGISTER, BFEE_USER_TCR_BYTE_LENGTH_HE, 0x25);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_REPORT_JUMP_TO_CHANGE_BUS_STATE] = JUMP(DELIA_RECIPES_RELATIVE_ADDRESS(&DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_CHANGE_BUS_STATE_TO_DATA_AND_CRC]));
	// in case of none HE - no WA-  Transfer from TX Recipe + offset 0x20 1 User TCR (3 DWORDS)
#endif //WORKAROUND_FOR_BFEE_SUB_BAND_HW_BUG
	// Transfer from TX Recipe + offset 0x20 1 User TCR (3 DWORDS)
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_REPORT_TRANSMIT_FIRST_USER_TCR] = TRANSMIT_FROM_REGISTER(BFEE_REPORT_TX_RECIPE_BASE_ADDRESS_REGISTER, BFEE_USER_TCR_BYTE_LENGTH, 0x24);
	// Change bus state to "Data" and init CRC
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_CHANGE_BUS_STATE_TO_DATA_AND_CRC] = TX_CONTROL(DELIA_CONTROL_NO_OVERRIDE, FALSE, TRUE, FALSE, FALSE, FALSE, DELIA_CONTROL_BUS_STATE_TRANSMIT_DATA, FALSE, FALSE, FALSE, TRUE);
	// Get the frame length from HOB
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_READ_AGAIN_TCR_FRAME_LENGHT_FROM_HOB] = LOAD_FROM_HOB(BFEE_REPORT_TCR_LENGTH_REGISTER, REG_HOB_LENGTH_WITHOUT_DELIMITER);
	// Prepare VHT delimiter that will be used in case it is needed  
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_UPDATE_DELIMITER] = STORE_TO_HOB(BFEE_REPORT_TCR_LENGTH_REGISTER,REG_HOB_VHT_EOF_DELIMITER_DATA_IN);
	// Skip the delimiter transnit if we are not in he or vht
	DeliaRecipeBfReport [DELIA_INSTRUCTION_BFEE_SKIP_DELIMITER_IF_NOT_HE_OR_VHT_2] = DO_SKIP_IF_NOT_EQUAL_TO_IMMEDIATE(BFEE_REPORT_PHY_MODE_REGISTER, 0x0);	
	// Transmit delimiter
	DeliaRecipeBfReport [DELIA_INSTRUCTION_BFEE_TRANSMIT_DELIMITER] = TRANSMIT_FROM_HOB(DELIMITER_LENGTH,REG_HOB_DELIMITER_DATA_OUT);
	// Change bus state to "Data" and init CRC
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_CRC_INIT] = TX_CONTROL(DELIA_CONTROL_NO_OVERRIDE, FALSE, TRUE, FALSE, FALSE, FALSE, DELIA_CONTROL_BUS_STATE_TRANSMIT_DATA, FALSE, FALSE, FALSE, TRUE);
	//	Wait for frame duration ready event 
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_WAIT_FOR_DURATION_READY] = WAIT(DURATION_EVENT);
	// Wait for section ready event (bit [1] from MBFM to Delia) ???
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_WAIT_FOR_SECTION_A_READY] = WAIT(BFEE_SECTION_A_READY);  
	//Read section pointer from HOB register txc_section_a_base_addr
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_REPORT_READ_TXC_SECTION_A_BASE_ADDR_FROM_HOB]  = LOAD_FROM_HOB(BFEE_REPORT_TXC_SECTION_A_BASE_ADDRESS_REGISTER, REG_HOB_TXC_SECTION_A_BASE_ADDR);
	//	Transmit 2 bytes from txc_section_a_base_addr
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_REPORT_TRANSMIT_2_BYTES_TXC_SECTION_A] = TRANSMIT_FROM_REGISTER(BFEE_REPORT_TXC_SECTION_A_BASE_ADDRESS_REGISTER,BFEE_TXC_SECTION_A_BYTE_LENGTH,0x0); 
	//Transmit the calculated "Duration" from HOB register auto_reply_bf_report_dur 
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_REPORT_TRANSMIT_DURATION] = TRANSMIT_FROM_HOB(BFEE_REPORT_DURATION_BYTE_LENGTH,REG_HOB_AUTO_REPLY_BF_REPORT_DUR);
	/* Add 4 for txc_section_a_base_addr */
	DeliaRecipeBfReport[DELIA_INSTRUCTION_INCREMENT_SECTION_A_BASE_ADDR] = ADD_REGISTER_AND_IMMEDIATE(BFEE_REPORT_TXC_SECTION_A_BASE_ADDRESS_REGISTER,BFEE_REPORT_TXC_SECTION_A_BASE_ADDRESS_REGISTER,BFEE_BYTE_OFFSET_FOR_TXC_SECTION_A);
	//	Transmit 18 bytes from txc_section_a_base_addr
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_REPORT_TRANSMIT_18_BYTES_TXC_SECTION_A] = TRANSMIT_FROM_REGISTER(BFEE_REPORT_TXC_SECTION_A_BASE_ADDRESS_REGISTER,BFEE_TXC_SECTION_B_BYTE_LENGTH,0x0); 
	//	Read the VAP TID pointer from HOB register rd_bf_rpt_vap_tid0_db_addr
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_REPORT_READ_VAP_TID_PTR_FROM_HOB]  = LOAD_FROM_HOB(BFEE_REPORT_VAP_TID_PTR_REGISTER, REG_HOB_RD_BF_RPT_VAP_TID0_DB_ADDR);
	//	Read first word from VAP DB to register 
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_REPORT_READ_FIRST_WORD_FROM_VAP_DB]  = LOAD_REGISTER_WITH_VALUE_FROM_ADDRESS(BFEE_REPORT_SEQUENCE_NUMBER_REGISTER, BFEE_REPORT_VAP_TID_PTR_REGISTER,0x0);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BF_REPORT_SHIFT_SEQUENCE_NUMBER_WITH_FRAGEMENT_BITS] = SHIFT_LEFT(BFEE_REPORT_SEQUENCE_NUMBER_REGISTER,FM_SEQUENCE_NUMBER_SHIFT);
	//	Transmit Sequence Control (2 bytes) to phy	
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_TRANSMIT_SEQUENCE_CONTROL] = TRANSMIT_FROM_REGISTER_WITH_IMMEDIATE_SIZE(BFEE_SEQUENCE_CONTROL_LENGTH, BFEE_REPORT_SEQUENCE_NUMBER_REGISTER);	
	/* Update the sequence number */
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_INCREMENT_SEQUENCE_NUMBER] = ADD_REGISTER_AND_IMMEDIATE(BFEE_REPORT_SEQUENCE_NUMBER_REGISTER,BFEE_REPORT_SEQUENCE_NUMBER_REGISTER,VALUE_TO_ADD_TO_SEQUENCE_NUMBER);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_WRAP_AROUND_SEQUENCE_NUMBER] = MASK_REGISTER(BFEE_REPORT_SEQUENCE_NUMBER_REGISTER,BFEE_REPORT_SEQUENCE_NUMBER_BYTES_TO_MASK,BFEE_REPORT_SEQUENCE_NUMBER_MASK_PATTERN);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_SHIFT_RIGHT_SEQUENCE_NUMBER] = SHIFT_RIGHT(BFEE_REPORT_SEQUENCE_NUMBER_REGISTER,FM_SEQUENCE_NUMBER_SHIFT);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_SAVE_SEQUENCE_NUMBER] = STORE(BFEE_REPORT_SEQUENCE_NUMBER_REGISTER,BFEE_REPORT_VAP_TID_PTR_REGISTER,BFEE_REPORT_SEQUENCE_NUMBER_BYTES_TO_SAVE,BF_REPORT_SEQUENCE_NUMBER_OFFSET);
	/* Wait for section B ready event (bit [2] from MBFM to Delia) */
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_WAIT_FOR_SECTION_B_READY] = WAIT(BFEE_SECTION_B_READY);
	//	Transmit ACTION_CATEGORY (2 bytes) to phy	
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_TRANSMIT_ACTION_CATEGORY] = TRANSMIT_FROM_HOB(BFEE_ACTION_CATEGORY_LENGTH,REG_HOB_ACTION_AND_CATEGORY);
	//	Transmit data (2 bytes) to phy	- MIMO Control 
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_TRANSMIT_MIMO_CTRL] = TRANSMIT_FROM_HOB(BFEE_ACTION_MIMO_CTRL_LENGTH, REG_HOB_BFEE_MIMO_CONTROL_SU);	
	//	Read HOB register bfee_phy_mode field mp_phy_mode_is_ht	
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_READ_BFEE_PHY_MODE_FROM_HOB] = LOAD_FROM_HOB(BFEE_REPORT_PHY_MODE_REGISTER, REG_HOB_BFEE_PHY_MODE);
	// Mask mp_phy_mode_is_ht field  (lowest bit)
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_REPORT_MASK_IS_PHY_MODE_HT] = MASK_REGISTER(BFEE_REPORT_PHY_MODE_REGISTER, 0x1, BFEE_IS_HT_MASK);
	// SKIP if not  HT
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_SKIP_IF_NOT_HT] = DO_SKIP_IF_EQUAL_TO_IMMEDIATE(BFEE_REPORT_PHY_MODE_REGISTER, 0x1); // Skip Jump if HT 
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_JUMP_TO_SECTION_B] = JUMP(DELIA_RECIPES_RELATIVE_ADDRESS(&DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_REPORT_READ_TXC_SECTION_B_BASE_ADDR_FROM_HOB]));
	// In case of HT - Need to transmit Sounding Timestamp
	// 		 i.	Read VAP ID from HOB register bfee_sta  field rxc_vap_idx_usr0_smp
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_READ_BFEE_STA_FROM_HOB] = LOAD_FROM_HOB(BFEE_REPORT_BFEE_STA_REGISTER, REG_HOB_BFEE_STA);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_SHIFT_RIGHT_VAP_INDX] = SHIFT_RIGHT(BFEE_REPORT_BFEE_STA_REGISTER,BFEE_VAP_INDX_SHIFT);
	// 		ii.	Write it to tsf_offset_index HOB register
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_STORE_VAP_INDEX_INTO_HOB] = STORE_TO_HOB(BFEE_REPORT_BFEE_STA_REGISTER, REG_HOB_TSF_OFFSET_INDEX);		
	//		iii.	Transmit 4 bytes of Timestamp 
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_TRANSMIT_TSF_ELEMENT] = TRANSMIT_FROM_HOB(0x4, REG_HOB_TSF_FIELD_SFR5_LOW);	
	//	Section B:
	//			Read section pointer from HOB register txc_section_b_base_addr
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_REPORT_READ_TXC_SECTION_B_BASE_ADDR_FROM_HOB] = LOAD_FROM_HOB(BFEE_REPORT_TXC_SECTION_B_BASE_ADDRESS_REGISTER, REG_HOB_TXC_SECTION_B_BASE_ADDR);
	//			Read section length from HOB register txc_section_b_byte_len
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_REPORT_READ_B_SECTION_LENGTH_FROM_HOB] = LOAD_FROM_HOB(BFEE_REPORT_B_SECTION_LENGTH_REGISTER, REG_HOB_TXC_SECTION_B_BYTE_LEN);
	//			Transmit data to phy
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_REPORT_TRANSMIT_B_SECTION] = TRANSMIT_FROM_REGISTER_WITH_LENGTH_FROM_REGISTER(BFEE_REPORT_TXC_SECTION_B_BASE_ADDRESS_REGISTER, BFEE_REPORT_B_SECTION_LENGTH_REGISTER, 0x0);
	//	Section C:
	// 			Read section pointer from HOB register txc_section_c_base_addr
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_REPORT_READ_TXC_SECTION_C_BASE_ADDR_FROM_HOB]  = LOAD_FROM_HOB(BFEE_REPORT_TXC_SECTION_C_BASE_ADDRESS_REGISTER, REG_HOB_TXC_SECTION_C_BASE_ADDR);
	//			Read section length from HOB register txc_section_c_byte_len
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_REPORT_READ_C_SECTION_LENGTH_FROM_HOB]  = LOAD_FROM_HOB(BFEE_REPORT_C_SECTION_LENGTH_REGISTER, REG_HOB_TXC_SECTION_C_BYTE_LEN);
	//			Wait for section ready event 
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_WAIT_FOR_SECTION_C_READY] = WAIT(BFEE_SECTION_C_READY);
	//			Read the HOB register bfee_sm filed bfee_bfr_success
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_READ_BFEE_SM] = LOAD_FROM_HOB(BFEE_REPORT_BFEE_SM_REGISTER, REG_HOB_BFEE_SM);
	//			MASK bfee_bfr_success   -  bit 0  
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_REPORT_MASK_BFR_SUCCESS] = MASK_REGISTER(BFEE_REPORT_BFEE_SM_REGISTER, 0x1, BFEE_BFR_SM_MASK);
	// 			If the value is 0 (PHY error detected):
	//				Skip in case of no error 
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_SKIP_IF_ERROR] = DO_SKIP_IF_NOT_EQUAL_TO_IMMEDIATE(BFEE_REPORT_BFEE_SM_REGISTER, 0x1); // skip jump if error  
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_JUMP_TO_READ_SECTION_FROM_TXCB] = JUMP(DELIA_RECIPES_RELATIVE_ADDRESS(&DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_REPORT_TRANSMIT_C_SECTION]));
	//					If the value is 0 (PHY error detected):
	//							Repeatedly Read HOB register bfee_invalid_tx_data and transmit txc_section_c_byte_len bytes towards the phy
	//transmit ERROR value with original section C length 
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_TRANSMIT_ERROR_VALUE] = TRANSMIT_FROM_HOB_WITH_LENGTH_FROM_REGISTER(BFEE_REPORT_C_SECTION_LENGTH_REGISTER, REG_HOB_BFEE_INVALID_TX_DATA);
    DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_JUMP_TO_READ_CRC] = JUMP(DELIA_RECIPES_RELATIVE_ADDRESS(&DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_READ_CRC]));
	// 			phy is valid:
	//			Transmit data to phy
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_REPORT_TRANSMIT_C_SECTION] = TRANSMIT_FROM_REGISTER_WITH_LENGTH_FROM_REGISTER(BFEE_REPORT_TXC_SECTION_C_BASE_ADDRESS_REGISTER,BFEE_REPORT_C_SECTION_LENGTH_REGISTER,0x0);
	//	Read the valid CRC HOB register bfee_gen_fcs_valid. If the value is 0 clear the frame CRC
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_READ_CRC] = LOAD_FROM_HOB(BFEE_REPORT_TXC_SECTION_CRC_REGISTER, REG_HOB_BFEE_GEN_FCS_VALID);
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_SKIP_IF_CRC_VALID] = DO_SKIP_IF_EQUAL_TO_IMMEDIATE(BFEE_REPORT_TXC_SECTION_CRC_REGISTER, 0x1); // skip jump if no error  
	// Change bus state to "Data" and init CRC
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_INIT_CRC_IF_NOT_VALID] = TX_CONTROL(DELIA_CONTROL_NO_OVERRIDE, FALSE, FALSE, FALSE, FALSE, FALSE, DELIA_CONTROL_BUS_STATE_TRANSMIT_DATA, FALSE, FALSE, FALSE, TRUE);
	// transmit CRC from HOB
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_TRANSMIT_CRC] = TRANSMIT_FROM_HOB(CRC_LENGTH, REG_HOB_TCC_CRC);
	// End of recipe
	DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_END_RECIPE] = END_RECIPE;
}

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

DeliaRecipes_GetRecipe0Address  


Description:
------------
return the recipes base address (recipe 0 address)

Input:
-----
	
		
Output:
-------
	

Returns:
--------
recipes base address  
	
**********************************************************************************/
uint32 DeliaRecipes_GetRecipe0Address(void)
{	
	return (uint32)(&(pDeliaRecipes->DeliaRecipe0[DELIA_RECIPE_0_FIRST_COMMAND]));
}


/**********************************************************************************
DeliaRecipes_GetAutoResponseSURecipeAddress  


Description:
------------
return Auto Response recipe address

Input:
-----
			
Output:
------
	
Returns:
--------
Auto Response recipe address  
**********************************************************************************/
uint32 DeliaRecipes_GetAutoResponseSURecipeAddress(void)
{	
	return (uint32)(&(pDeliaRecipes->DeliaRecipeSUAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_FIRST_COMMAND]));
}
/**********************************************************************************
DeliaRecipes_GetAutoResponseMUBroadcastRecipeAddress  


Description:
------------
return Auto Response recipe address

Input:
-----
			
Output:
------
	
Returns:
--------
Auto Response recipe address  
**********************************************************************************/
uint32 DeliaRecipes_GetAutoResponseMUBroadcastRecipeAddress(void)
{	
	return (uint32)(&(pDeliaRecipes->DeliaRecipeMUBroadcastAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_FIRST_COMMAND]));
	
}
/**********************************************************************************
DeliaRecipes_GetAutoResponseMUUnicastRecipeAddress  


Description:
------------
return Auto Response recipe address

Input:
-----
			
Output:
------
	
Returns:
--------
Auto Response recipe address  
**********************************************************************************/
uint32 DeliaRecipes_GetAutoResponseMUUnicastRecipeAddress(void)
{	
	return (uint32)(&(pDeliaRecipes->DeliaRecipeMUUnicastAutoResponse[DELIA_INSTRUCTION_AUTO_RESPONSE_MU_UNICAST_FIRST_COMMAND]));
}


/**********************************************************************************

DeliaRecipes_GetBfReportAutoResponseRecipeAddress  


Description:
------------
return BF report auto response recipe address

Input:
-----
	
		
Output:
-------
	

Returns:
--------
BF report auto response recipe address 
	
**********************************************************************************/

uint32 DeliaRecipes_GetBfReportRecipeAddress(void)

{	
	return (uint32)(&(pDeliaRecipes->DeliaRecipeBfReport[DELIA_INSTRUCTION_BFEE_REPORT_FIRST_COMMAND]));
	  
}


/**********************************************************************************

DeliaRecipes_ConstructRecipes  


Description:
------------
construct the Delia recipes and the recipes pointers table

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void DeliaRecipes_ConstructRecipes(void)
{	
    deliaConstructRecipe0();
	deliaConstructAutoResponseSURecipe();
	deliaConstructAutoResponseMuBroadcastRecipe();
	deliaConstructAutoResponseMuUnicastRecipe();
	deliaConstructBfReportRecipe(); 
}


