/***********************************************************************************
 File:		ErrorHandler.c		
 Module:		Error Handler
 Purpose:		
 Description:	
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "System_Information.h"
#include "stringLibApi.h"
#include "Utils_Api.h"
#include "mt_cachedefs.h"
#include "ErrorHandler_Api.h"
#include "HostInterface_API.h"
#include "um_interface.h"
#include "init_ifmsg.h"
#include "ShramHim.h"
#include "shram_man_msgs.h"
#include "ShramErrorHandler.h"
#include "int_gen.h"
#include "enet_hrc.h"
#include "enet_pas.h"
#include "Pac_Api.h"
#include "LmHdk_API.h"
#include "PhyDriver_API.h"
#include "RegAccess_Api.h"
#include "HostInterface_API.h"
#include "PhyTestBus_API.h"
#include "HwMemoryMap.h"
#include "MacHtExtensionsRegs.h"
#include "TxHandler_Api.h"
#include "arc_intrinsics.h"
#include "arc_reg.h"
#include "smart_api.h"
#include "lm_statistics.h"
#include "HwQManager_API.h"
#include "ShramRxDescriptors.h"
#include "loggerAPI.h"
#include "Pac_Api.h"
#include "HostInterface_ScratchPadApi.h"

#if defined (ENET_INC_UMAC)
#include "MacGenriscRxRegs.h"
#include "MacGenriscTxRegs.h"
#include "MacGenriscRegs.h"
#include "HostGenRiscInterface_Api.h"
#include "HostInterface_MailboxApi.h"
#include "HostInterface_InitApi.h"
#include "HwLoggerAPI.h"
#include "ShramPacketDescriptors.h"
#endif 

#if defined (ENET_INC_LMAC)
#include "Pauser_Api.h"
#include "SenderInterface_Api.h"
#include "AutoResponse_Api.h"
#include "PreAggregator_Api.h"
#include "RxHandler_Api.h"
#endif





/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_ERROR_HANDLER 
#define LOG_LOCAL_FID 0

#define ERROR_EXCEPTION           			(0xA7)
#define ERROR_ASSERT              			(0xA8)
#define MAX_NUMBER_OF_WLAN_INTERFACES		(4)
#define WLAN_INTERFACE_ID_MASK				(0x3)
#define MAX_WAITING_TIME_FOR_LM_ASSERT  	(500)
#define NUMBER_OF_ARC_CORE_REGISTERS		(32)
#define MAX_NUMBER_OF_WLAN_INTERFACES		(4)
#define TRAP_EXCEPTION_VALUE_FOR_ASSERT		(0x3)
/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/
#define GET_WLAN_INTERFACE_ID()		(FWinterface & WLAN_INTERFACE_ID_MASK)
#define GET_ORIGIN_ID()				(1)//Origin ID *must* match its definition in AutoBuild\mt_build.pl subroutine RunNew_Event_PrepScript(): "...my $originID = 1 ..."

/*---------------------------------------------------------------------------------
/                       Data Type Definition                        
/----------------------------------------------------------------------------------*/

typedef struct _RegsDump_t
{	
	uint32 eretReg;									// Exception Return Address (0x400)
	uint32 erbtaReg;								// Exception Return Branch Target Address (0x401)
	uint32 erStatusReg;								// Exception Return Status (0x402)
	uint32 ecrReg;									// Exception Cause Register (0x403)
	uint32 efaReg;									// Exception Fault Address (0x404)
	uint32 status32Reg;								// Status Register (0xa)
	uint32 coreRegs[NUMBER_OF_ARC_CORE_REGISTERS];	// See arc_reg.h (REG_R0 / REG_R1 ...). Can be regular/shadow set. Depends on STATUS32[RB(16:18)]. R28 = SP. R31 = BLINK
	uint32 mpu_ic_Reg;								// MPU interrupt cause register (MPU_ECR) (0x420)
} RegsDump_t;

typedef struct ErrorHandlerDb_s
{
	uint32 enabledModulesHalt;
} ErrorHandlerDb_t;

#if defined (ENET_INC_UMAC)

typedef struct _RiscDump_t

{
	RegMacGenriscRxLastPcExecuted_u RegMacGenriscRxLastPcExecuted[NUM_OF_CONFIGURATION_MANAGER_BANDS];
	RegMacGenriscRxAddOutAbort_u	RegMacGenriscRxAddOutAbort[NUM_OF_CONFIGURATION_MANAGER_BANDS];
	RegMacGenriscTxLastPcExecuted_u RegMacGenriscTxLastPcExecuted[NUM_OF_CONFIGURATION_MANAGER_BANDS];
	RegMacGenriscTxAddOutAbort_u	RegMacGenriscTxAddOutAbort[NUM_OF_CONFIGURATION_MANAGER_BANDS];
	RegMacGenriscLastPcExecuted_u 	RegMacGenriscLastPcExecuted;
	RegMacGenriscAddOutAbort_u		RegMacGenriscAddOutAbort;
	RegMacGenriscIntVector_u		RegMacGenriscIntVector[NUM_OF_CONFIGURATION_MANAGER_BANDS];
}RiscDump_t;
#endif



/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void errorHandlerAssertHandlerForInitSequence(uint32 assertGroupId, uint32 assertFileId, uint32 assertLine);
static void errorHandlerAssertHandlerForSteadyState(uint32 assertGroupId, uint32 assertFileId, uint32 assertLine);
static void errorHandlerFillAssertMsg(uint32 in_uLine, uint32 groupId, uint32 fileId);
static void errorHandlerFillExceptionMsg(uint16 in_uCpuType);
#if defined (ENET_INC_UMAC)
static void errorHandlerSendErrorMsgToHost(void);
static void errorHandlerFillRiscInfo(void);

#endif

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
static ErrorHandlerDb_t errorHandlerDb;

void (*errorHandlerAssert)(uint32 assertGroupId, uint32 assertFileId, uint32 assertLine); /* pointer to assert handler */
static volatile bool   errorHandlerDebuggingFlag; 

#if defined (ENET_INC_UMAC)
FwUtils_queue_t errorHandlerAssertLmMsgQueue;
#endif 
#if defined (ENET_INC_UMAC)
volatile uint32 errorHandlerAssertLowerMacOneTime;
#endif

#if defined (ENET_INC_LMAC)
bool errorHandlerResetPhyOnAssert;
#endif

uint32 FWinterface; /* Logmacros Uses thid variable*/
bool assertOccurred = FALSE; 

extern volatile uint32 True_g;


RegsDump_t RegsDump; // Usefull information for debugging exceptions

#if defined (ENET_INC_UMAC)
RiscDump_t RiscDump;
#endif


/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

errorHandlerAssertHandlerForSteadyState 

Description:
------------
	Handler for fatal error event in FW after initialization sequence
		LM:
			1) Disable All HW
			2) Sends debug msg to UM
		UM:
			1) If fatal error is not originated from LM causes Lm to enter into assert.
			2) Sends msg To Host with all information of the error.
Input:
-----
	None	
Output:
-------
	None
Returns:
--------
	void  
**********************************************************************************/
#ifdef ENET_INC_UMAC
static void errorHandlerAssertHandlerForSteadyState(uint32 assertGroupId, uint32 assertFileId, uint32 assertLine)
{	
	volatile HostInterface_ScratchPadApiParams_t * pHostInterface_ScratchPadApiParams = (HostInterface_ScratchPadApiParams_t *) (MAC_GENRISC_HOST_SPRAM_BASE_ADDR + (SCPAD_ADDRESS_HOST_INTERFACE_SCRATCHPAD_API_STRUCTURE_START << 0x2));

	TX_INTERRUPT_SAVE_AREA;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	/* turn on the flag which signs that we've got an assert */
	if(assertOccurred == TRUE)
	{
		//Klocwork: while loop is implemented for not to exit from this loop But klocwork will give error as "INFINITE_LOOP.GLOBAL: Infinite loop" ignore it. 
		while(True_g);
//		return; /* In case we allready on ASSERT contexts (Assert within an Assert) */ 
	}

	assertOccurred = TRUE; 
	// Stop the SMART debugger recording
	SmartDisable();
	// Dump the SMART memory
	SmartDumpMemory();

	if(errorHandlerDb.enabledModulesHalt & (1 << HALT_MODULE_HOST_IF_GENRISC))
	{		
		//	HostGenRisc_Halt();
		
		//leave only Rx out and logger tasks in Host IF Genrisc
		pHostInterface_ScratchPadApiParams->enabledTasksBitmap = ((0x1 << HOST_INTERFACE_INIT_FIELD_ENABLED_TASKS_BITMAP_RX_OUT_TASK_BIT) | (0x1 << HOST_INTERFACE_INIT_FIELD_ENABLED_TASKS_BITMAP_LOGGER_TASK_BIT));
	}
		
	//read info from the riscs 
	errorHandlerFillRiscInfo();

#ifndef	ENET_INC_HW_FPGA
	memcpy32(&B0_MpduDescriptorsDump, &B0_MpduDescriptors, (sizeof(b0RxTxMpduDesc_u)>>2));
	memcpy32(&B1_MpduDescriptorsDump, &B1_MpduDescriptors, (sizeof(b1RxTxMpduDesc_u)>>2));
#endif

	if (errorHandlerTriggerCore == CORE_NUM_INVALID)
	{
		errorHandlerTriggerCore = SystemInfo_GetCoreNum();
	}

	errorHandlerFillAssertMsg(assertLine, assertGroupId, assertFileId);

	switch (errorHandlerTriggerCore)
	{
		case CORE_HOST:
			// if errorHandlerTriggerCore is CORE_HOST and i am CORE_NUM_2: trigger no one, host is responsible to trigger all cores
			break;
		case CORE_NUM_0:
			// if errorHandlerTriggerCore is CORE_NUM_0 and i am CORE_NUM_2: trigger no one, lm0 is responsible to trigger lm1
			break;
		case CORE_NUM_1:
			// if errorHandlerTriggerCore is CORE_NUM_1 and i am CORE_NUM_2: trigger no one, lm1 is responsible to trigger lm0
			break;
		case CORE_NUM_2:
			// if errorHandlerTriggerCore is CORE_NUM_2 and i am CORE_NUM_2: trigger lm0 and lm1
			vIGEN_SendToLm0(IGEN_EVT_ASSERT_TO_LM0_FROM_UM);
			vIGEN_SendToLm1(IGEN_EVT_ASSERT_TO_LM1_FROM_UM);
			break;
		default:
			ILOG0_V("errorHandlerTriggerCore not set");
			break;
	}

	ILOG0_DDD("vAPP_Assert assertGroupId=%d assertFileId=%d assertLine=%d", assertGroupId, assertFileId, assertLine);

#if defined (ANALYZE_DISABLE_INTERRUPT)
	AAA_DebugStopTestBus(); /* Stop GenRisc trace*/
#endif

	
	/* This block is a waiting for the LM to finish its own assertion */
	{
		uint32 startWaitToLMs = GET_TSF_TIMER_LOW();

		while(TRUE)
		{
			if ((ErrorHandlerLm0InAssert == TRUE) && (ErrorHandlerLm1InAssert == TRUE))
			{
				break;
			}

			/* We wait at least 10 micro seconds for the LM to finish the Assert procedure */
			if ((GET_TSF_TIMER_LOW() - startWaitToLMs) > MAX_WAITING_TIME_FOR_LM_ASSERT)
			{
				break;
			}
		}
	}

	errorHandlerSendErrorMsgToHost();

	sys_flush_caches();

	// Klocwork: while loop is implemented for not to exit from this loop But klocwork will give error as "INFINITE_LOOP.GLOBAL: Infinite loop" ignore it. 
	while( errorHandlerDebuggingFlag == TRUE )
	{
		MT_CheckBclOnException();
	}
	OSAL_ENABLE_INTERRUPTS(interrupt_save); //only for the protocol (unreachable code)
}
#endif


/**********************************************************************************

errorHandlerAssertHandlerForSteadyState 

Description:
------------
	Handler for fatal error event in FW after initialization sequence
		LM:
			1) Disable All HW
			2) Sends debug msg to UM
		UM:
			1) If fatal error is not originated from LM causes Lm to enter into assert.
			2) Sends msg To Host with all information of the error.
Input:
-----
	None	
Output:
-------
	None
Returns:
--------
	void  
**********************************************************************************/
#ifdef ENET_INC_LMAC
static void errorHandlerAssertHandlerForSteadyState(uint32 assertGroupId, uint32 assertFileId, uint32 assertLine)
{	
	TX_INTERRUPT_SAVE_AREA;
	CoreNum_e myCoreNum = SystemInfo_GetCoreNum();

	PhyTestBus_Stop();	

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	
	/* turn on the flag which signs that we've got an assert */
	if(assertOccurred == TRUE)
	{
		while(True_g);
//		return; /* In case we allready on ASSERT contexts (Assert within an Assert) */ 
	}

	assertOccurred = TRUE; 

	// Stop the SMART debugger recording
	SmartDisable();
	// Dump the SMART memory
	SmartDumpMemory();

	Pac_DeliaReadDebugInfoOnAssert();

	if (errorHandlerTriggerCore == CORE_NUM_INVALID)
	{
		errorHandlerTriggerCore = SystemInfo_GetCoreNum();
	}

	errorHandlerFillAssertMsg(assertLine, assertGroupId, assertFileId);

	switch (errorHandlerTriggerCore)
	{
		case CORE_NUM_0:
			// if errorHandlerTriggerCore is CORE_NUM_0 and i am CORE_NUM_0: trigger lm1 and um
			// if errorHandlerTriggerCore is CORE_NUM_0 and i am CORE_NUM_1: trigger no one, lm0 triggered um
			if (myCoreNum == CORE_NUM_0)
			{
				vIGEN_SendToLm1(IGEN_EVT_ASSERT_TO_LM1_FROM_LM0);
				vIGEN_SendToUm(IGEN_EVT_ASSERT_TO_UM_FROM_LM0);
			}
			break;
		case CORE_NUM_1:
			// if errorHandlerTriggerCore is CORE_NUM_1 and i am CORE_NUM_1: trigger lm0 and um
			// if errorHandlerTriggerCore is CORE_NUM_1 and i am CORE_NUM_0: trigger no one, lm1 triggered um
			if (myCoreNum == CORE_NUM_1)
			{
				vIGEN_SendToLm0(IGEN_EVT_ASSERT_TO_LM0_FROM_LM1);
				vIGEN_SendToUm(IGEN_EVT_ASSERT_TO_UM_FROM_LM1);
			}
			break;
		case CORE_NUM_2:
			// if errorHandlerTriggerCore is CORE_NUM_2 and i am CORE_NUM_0 or CORE_NUM_1: trigger no one, um is responsible to trigger lm0 and lm1
			break;
		case CORE_HOST:
			// if errorHandlerTriggerCore is CORE_HOST and i am CORE_NUM_0 or CORE_NUM_1: trigger no one, host is responsible to trigger all cores
			break;
		default:
			ILOG0_V("errorHandlerTriggerCore not set");
			break;
	}


#if defined (ENET_INC_HW_FPGA) && defined (ENET_INC_LMAC1)
	if(ConfigurationManager_GetBandConfigurationMode() != CONFIGURATION_MODE_SINGLE_BAND)
#endif
	{
		PreAggregator_Halt();

		//sender halt
		Sender_Halt();

#if defined (ANALYZE_DISABLE_INTERRUPT)
		AAA_DebugStopTestBus(); /* Stop GenRisc trace*/
#endif
		// RxH halt
		RxHandler_Halt();

		// Host IF halt - 
		// The next line is in comment since we want the logs to be sent but if the FW are corrupted,it should be uncomment.
		//HostGenRisc_Halt();
		TxPauser_ReadDebug(0xFF);

		/* Disable Phy & RF & RXC*/
		//disable RXC
		Pac_RxcDisable();//stop RX on Assert

		//Clear auto_resp_fifo_update_en register in RxC
		Pac_RxcAutoRespFifoUpdateEnable(FALSE);

		// TX Handler is in LMAC in wave600 and in UMAC in wave500
		TxHandler_DisableTxh();
	}

#ifdef ENET_INC_LMAC0
	ErrorHandlerLm0InAssert = TRUE;
#endif

#ifdef ENET_INC_LMAC1
	ErrorHandlerLm1InAssert = TRUE; 
#endif

	sys_flush_caches();

	while( errorHandlerDebuggingFlag == TRUE )
	{
		MT_CheckBclOnException();
	}
	OSAL_ENABLE_INTERRUPTS(interrupt_save); //only for the protocol (unreachable code)
}

#endif


/**********************************************************************************

errorHandlerAssertHandlerForInitSequence 

Description:
------------
	Handler for fatal error event in FW during initialization sequence

Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void  
**********************************************************************************/
static void errorHandlerAssertHandlerForInitSequence(uint32 assertGroupId, uint32 assertFileId, uint32 assertLine)
{
	UNUSED_PARAM(assertGroupId);	
	UNUSED_PARAM(assertFileId);	
	UNUSED_PARAM(assertLine);	
	
	// Stop the SMART debugger recording
	SmartDisable();

	// Dump the SMART memory
	SmartDumpMemory();
	
	// Klocwork: while loop is implemented for not to exit from this loop But klocwork will give error as "INFINITE_LOOP.GLOBAL: Infinite loop" ignore it. 
	// currently do nothing, just hang here and wait
	while( errorHandlerDebuggingFlag == TRUE )
	{
		MT_CheckBclOnException();
	}
}


/**********************************************************************************

errorHandlerFillAssertMsg 

Description:
------------
	Fill in Error structure in SHRAM will be updated with Assert Data. 

Input:
-----
	in_ppcSourceFilename - the filename of the Assert
	in_uTimeStamp - time of the Assert
Output:
-------
	None
Returns:
--------
	void  
**********************************************************************************/
static void errorHandlerFillAssertMsg(uint32 line, uint32 groupId, uint32 fileId)
{
	CoreNum_e coreNum = SystemInfo_GetCoreNum();
	
    ErrorHandler_EventDetails[coreNum].uMagicMun 			= ERROR_ASSERT;
    ErrorHandler_EventDetails[coreNum].uCauseRegOrLineNum 	= line;
	ErrorHandler_EventDetails[coreNum].uFileId 				= (uint16) fileId;
	ErrorHandler_EventDetails[coreNum].uGroupId 			= (uint16) groupId;
	ErrorHandler_EventDetails[coreNum].uOriginId 			= GET_ORIGIN_ID();
	ErrorHandler_EventDetails[coreNum].uFWinterface 		= GET_WLAN_INTERFACE_ID();
    ErrorHandler_EventDetails[coreNum].uCoreNum 			= coreNum;
	ErrorHandler_EventDetails[coreNum].uSpReg 				= RegsDump.coreRegs[REG_SP];
	ErrorHandler_EventDetails[coreNum].uEpcReg 				= 0;

#if defined ENET_INC_HW_FPGA && defined (ENET_INC_LMAC1)
	ErrorHandler_EventDetails[coreNum].uTimeStamp			= 0;
#else
	ErrorHandler_EventDetails[coreNum].uTimeStamp			= GET_TSF_TIMER_LOW();
#endif
}

/**********************************************************************************

errorHandlerFillExceptionMsg 

Description:
------------
	Fill Exception data, Error structure in SHRAM will be updete with exception data
Input:
-----
	in_uCpuType - 1 for UM 0- for LM		
Output:
-------
	None
Returns:
--------
	void  
**********************************************************************************/
static void errorHandlerFillExceptionMsg(uint16 cpuType)
{
	CoreNum_e coreNum = SystemInfo_GetCoreNum();

    ErrorHandler_EventDetails[coreNum].uMagicMun 			= ERROR_EXCEPTION;
    ErrorHandler_EventDetails[coreNum].uCoreNum 			= cpuType;
    ErrorHandler_EventDetails[coreNum].uCauseRegOrLineNum 	= RegsDump.ecrReg;
    ErrorHandler_EventDetails[coreNum].uEpcReg 				= RegsDump.efaReg;
    ErrorHandler_EventDetails[coreNum].uStatusReg 			= RegsDump.erStatusReg;
#if defined (ENET_INC_HW_FPGA) && defined (ENET_INC_LMAC1)
	if(ConfigurationManager_GetBandConfigurationMode() != CONFIGURATION_MODE_SINGLE_BAND)
#endif
	{
  		ErrorHandler_EventDetails[coreNum].uTimeStamp 			= GET_TSF_TIMER_LOW();
	}
	ErrorHandler_EventDetails[coreNum].uSpReg  				= RegsDump.coreRegs[REG_SP];
	ErrorHandler_EventDetails[coreNum].uFWinterface 		= GET_WLAN_INTERFACE_ID();
}



#if defined (ENET_INC_UMAC)
/**********************************************************************************

errorHandlerFillRiscInfo 

Description:
------------
	get relavent info from the riscs regs

Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void  
**********************************************************************************/
static void errorHandlerFillRiscInfo(void)
{
	uint8 bandId;
	
	for (bandId = CONFIGURATION_MANAGER_BAND_0 ; bandId < ConfigurationManager_GetNumOfActiveBands() ; bandId++)
	{
		RegAccess_ReadPerBand(REG_MAC_GENRISC_RX_LAST_PC_EXECUTED, &RiscDump.RegMacGenriscRxLastPcExecuted[bandId].val, bandId);	
		RegAccess_ReadPerBand(REG_MAC_GENRISC_TX_LAST_PC_EXECUTED, &RiscDump.RegMacGenriscTxLastPcExecuted[bandId].val, bandId);

		RegAccess_ReadPerBand(REG_MAC_GENRISC_RX_ADD_OUT_ABORT, &RiscDump.RegMacGenriscRxAddOutAbort[bandId].val, bandId);	
		RegAccess_ReadPerBand(REG_MAC_GENRISC_TX_ADD_OUT_ABORT, &RiscDump.RegMacGenriscTxAddOutAbort[bandId].val, bandId);

		RegAccess_ReadPerBand(REG_MAC_GENRISC_TX_INT_VECTOR, 	&RiscDump.RegMacGenriscIntVector[bandId].val, bandId);
		RegAccess_ReadPerBand(REG_MAC_GENRISC_RX_INT_VECTOR, 	&RiscDump.RegMacGenriscIntVector[bandId].val, bandId);
		RegAccess_ReadPerBand(REG_MAC_GENRISC_INT_VECTOR, 		&RiscDump.RegMacGenriscIntVector[bandId].val, bandId);
	}
	RegAccess_Read(REG_MAC_GENRISC_LAST_PC_EXECUTED, &RiscDump.RegMacGenriscLastPcExecuted.val);
	RegAccess_Read(REG_MAC_GENRISC_ADD_OUT_ABORT, &RiscDump.RegMacGenriscAddOutAbort.val);	
}
/**********************************************************************************

errorHandlerSendErrorMsgToHost 

Description:
------------
	Send error massage to driver

Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void  
**********************************************************************************/

static void errorHandlerSendErrorMsgToHost(void)
{
    K_MSG*	   psErrorMsg ;
    MAC_EVENT* psErrorData;
	CoreNum_e coreNum = errorHandlerTriggerCore;
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;

	psErrorMsg  = (K_MSG*) &asMAN_IndData[BSS_IND_COUNT + STA_IND_COUNT + RSN_IND_COUNT];
    psErrorData = (MAC_EVENT*)psErrorMsg->abData;
	
    OSAL_INIT_PERSISTENT_MSG( psErrorMsg, sizeof( UMI_IND ) );

	if (coreNum == CORE_HOST)
	{
		// If this is an assert from host, it means all cores were stopped. We can take the details (line, groupID, etc...) from the UM core.
		// We must not access "ErrorHandler_EventDetails" with index "CORE_HOST" since it doesn't exist.
		coreNum = CORE_NUM_2;
	}
	
    if(ErrorHandler_EventDetails[coreNum].uMagicMun == ERROR_ASSERT)
    {
        psErrorData->u32EventID = EVENT_APP_FATAL;
		psErrorData->u.sAppFatalEvent.FileId				= ErrorHandler_EventDetails[coreNum].uFileId;
		psErrorData->u.sAppFatalEvent.GroupId				= ErrorHandler_EventDetails[coreNum].uGroupId;
		psErrorData->u.sAppFatalEvent.OriginId				= ErrorHandler_EventDetails[coreNum].uOriginId;
		psErrorData->u.sAppFatalEvent.FWinterface 			= ErrorHandler_EventDetails[coreNum].uFWinterface;
        psErrorData->u.sAppFatalEvent.uCauseRegOrLineNum 	= ErrorHandler_EventDetails[coreNum].uCauseRegOrLineNum;
    }
    else if (ErrorHandler_EventDetails[coreNum].uMagicMun == ERROR_EXCEPTION)
    {
        psErrorData->u32EventID								= EVENT_EXCEPTION; 
        psErrorData->u.sAppFatalEvent.uEpcReg 				= ErrorHandler_EventDetails[coreNum].uEpcReg;
        psErrorData->u.sAppFatalEvent.uStatusReg 			= ErrorHandler_EventDetails[coreNum].uStatusReg;
        psErrorData->u.sAppFatalEvent.uCauseRegOrLineNum 	= ErrorHandler_EventDetails[coreNum].uCauseRegOrLineNum;
		psErrorData->u.sAppFatalEvent.FWinterface 			= ErrorHandler_EventDetails[coreNum].uFWinterface;
    }

    psErrorData->u32Timestamp 								= ErrorHandler_EventDetails[coreNum].uTimeStamp;
	psErrorData->u.sAppFatalEvent.uTimeStamp  				= ErrorHandler_EventDetails[coreNum].uTimeStamp;
    psErrorData->u.sAppFatalEvent.uCoreNum 					= ErrorHandler_EventDetails[coreNum].uCoreNum;
    psErrorMsg->header.tKMsgType 							= MC_MAN_MAC_EVENT_IND;
	psErrorMsg->header.vapId 								= 0;

	
	/*send to driver the first PD in the free PD pool, 
	driver will go over the list and compare with the full list to understand which pds doesn't returned to host. 
	For each such PD it will release the host buffer for it, before going to recovery*/
	
	memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_TX_POOL_LIST_DATA_PD;
	HwQManager_PeekHeadPacket(&hwQueueManagerRequestParams);
	
	psErrorData->u.sAppFatalEvent.uFirstFreePdOffset = 	CONVERT_PD_TO_OFFSET(hwQueueManagerRequestParams.pHeadDesc);

    vHIM_SendMsgToHost(psErrorMsg);       
}
#endif /* defined (ENET_INC_UMAC)  */



/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/	


/**********************************************************************************

ErrorHadler_CheckAssertReqFromHost

Description:
------------
<Description of the purpose of the function>

Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void  
**********************************************************************************/
void ErrorHadler_CheckAssertReqFromHost(void)
{
	CoreNum_e cpuNumber = SystemInfo_GetCoreNum();	
    uint32 localMipsCtrl = u32FwCoreControlDesc.u32FwCoreCtrl[cpuNumber];
	TX_INTERRUPT_SAVE_AREA;

    if(MTLK_BFIELD_GET((localMipsCtrl), FW_CORE_CTRL_DO_ASSERT) == TRUE)
    {
		if ((errorHandlerTriggerCore == CORE_NUM_INVALID) || (errorHandlerTriggerCore == CORE_HOST))
		{
#if defined (ENET_INC_LMAC)
			errorHandlerResetPhyOnAssert = FALSE;
#endif
			
			OSAL_DISABLE_INTERRUPTS(&interrupt_save);
			errorHandlerTriggerCore = CORE_HOST;
	        FATAL("Assert Requested by the Host");
			OSAL_ENABLE_INTERRUPTS(interrupt_save); //only for the protocol (unreachable code)
		}
    }
}

/************************************************************************************

ErrorHandler_GetCtrlStructure

Description:
------------
<Description of the purpose of the function>

Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void - 
	
**********************************************************************************/
#if defined (ENET_INC_UMAC)
uint32 ErrorHandler_GetCtrlStructure(void)
{
    return (uint32)(&u32FwCoreControlDesc.u32FwCoreCtrl[0]);
}
#endif


/**********************************************************************************

ErrorHandlerEnableModuleHalt 

Description:
------------
Enable Module halt during Fatal/Assert or exception

Input:
-----
	Required Halted Module id
	
**********************************************************************************/
void ErrorHandlerEnableModuleHalt(HaltedModules_e haltModuleId)
{
	errorHandlerDb.enabledModulesHalt |= (1 << haltModuleId);
}

/**********************************************************************************

errorHandlerAssertHandlerForInitSequence 

Description:
------------
<Description of the purpose of the function>

Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void  
	
**********************************************************************************/
void ErrorHandler_AssertHandler(uint32 assertGroupId, uint32 assertFileId, uint32 assertLine)
{
	// Trigger exception which will store all registers for later analysis.
	// TRAP_EXCEPTION_VALUE_FOR_ASSERT (0x3) will be an indication that the TRAP exception came from an ASSERT (it can support more sources in the assembler code).
	_ASM("TRAP_S 3");

	errorHandlerAssert(assertGroupId, assertFileId, assertLine);
}


/**********************************************************************************

ErrorHandler_Exception

Description:
------------
	Handle exception. Registers were dumped already by assembler code.

Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void  
**********************************************************************************/

	/* following pragma will avoid warning of "missing-noreturn" flag */   
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wmissing-noreturn"	
void ErrorHandler_Exception(void)
{
	CoreNum_e myCore;
	volatile HostInterface_ScratchPadApiParams_t *pHostInterface_ScratchPadApiParams = NULL;

	// Stop the SMART debugger recording
	SmartDisable();

	// Dump the SMART memory
	SmartDumpMemory();

	myCore = SystemInfo_GetCoreNum();

	if (errorHandlerTriggerCore == CORE_NUM_INVALID)
	{
		errorHandlerTriggerCore = myCore;
	}

	UNUSED_PARAM(pHostInterface_ScratchPadApiParams);

	switch (myCore)
	{
#if defined (ENET_INC_LMAC)
		case CORE_NUM_0:
			vIGEN_SendToUm(IGEN_EVT_ASSERT_TO_UM_FROM_LM0);
			vIGEN_SendToLm1(IGEN_EVT_ASSERT_TO_LM1_FROM_LM0);
			break;
		case CORE_NUM_1:
			vIGEN_SendToUm(IGEN_EVT_ASSERT_TO_UM_FROM_LM1);
			vIGEN_SendToLm0(IGEN_EVT_ASSERT_TO_LM0_FROM_LM1);
			break;
#endif
#if defined (ENET_INC_UMAC)
		case CORE_NUM_2:
			
			pHostInterface_ScratchPadApiParams = (HostInterface_ScratchPadApiParams_t *) (MAC_GENRISC_HOST_SPRAM_BASE_ADDR + (SCPAD_ADDRESS_HOST_INTERFACE_SCRATCHPAD_API_STRUCTURE_START << 0x2));

			vIGEN_SendToLm0(IGEN_EVT_ASSERT_TO_LM0_FROM_UM);
			vIGEN_SendToLm1(IGEN_EVT_ASSERT_TO_LM1_FROM_UM);

			//leave only Rx out and logger tasks in Host IF Genrisc

			if(errorHandlerDb.enabledModulesHalt & (1 << HALT_MODULE_HOST_IF_GENRISC))
			{				
				pHostInterface_ScratchPadApiParams->enabledTasksBitmap = ((0x1 << HOST_INTERFACE_INIT_FIELD_ENABLED_TASKS_BITMAP_RX_OUT_TASK_BIT) | (0x1 << HOST_INTERFACE_INIT_FIELD_ENABLED_TASKS_BITMAP_LOGGER_TASK_BIT));
			}
		
			break;
#endif			
		default:
			break;
	}

	
	errorHandlerFillExceptionMsg(myCore);

#if defined (ENET_INC_UMAC)
	errorHandlerSendErrorMsgToHost();
#endif

	sys_flush_caches();

	while(1) //KW IGNORE INFINITE_LOOP.LOCAL
	{
		// BCL on Exception 
		MT_CheckBclOnException();
    } 
}
#pragma clang diagnostic pop	

/**********************************************************************************

ErrorHandler_Ready

Description:
------------
<Description of the purpose of the function>

Input:
-----
	None	
Output:
-------
	None
Returns:
--------
	void  
**********************************************************************************/
void ErrorHandler_Ready(void)
{
	/* Identify cpu number */ 
 	CoreNum_e cpuNumber = SystemInfo_GetCoreNum();
	
	u32FwCoreControlDesc.u32FwCoreCtrl[cpuNumber] = 0; // init
	
#if defined (ENET_INC_UMAC)	
	errorHandlerAssertLowerMacOneTime = TRUE;
#endif
	
#if defined (ENET_INC_LMAC)
	errorHandlerResetPhyOnAssert = TRUE;
#endif

	errorHandlerAssert = errorHandlerAssertHandlerForSteadyState;
}

/**********************************************************************************

ErrorHandler_SetFwInterfaceId 

Description:
------------
<Description of the purpose of the function>

Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void  
**********************************************************************************/
void ErrorHandler_SetFwInterfaceId(uint32 fwInterface)
{
	FWinterface = fwInterface;
	ASSERT(FWinterface < MAX_NUMBER_OF_WLAN_INTERFACES);
}
/**********************************************************************************

ErrorHandler_EnableDivideByZeroException


Description:
------------
enable Divide by zero exception

Input:
-----
	None	
Output:
-------
	None
Returns:
--------
	void  
**********************************************************************************/

void ErrorHandler_EnableDivideByZeroException(void)
{
	//enable Divide by zero exception 
	uint32 regStatus32;
	regStatus32 = _lr(REG_STATUS32);
	regStatus32 |= REG_STATUS32_DZ_BIT;
	_kflag(regStatus32);
}
/**********************************************************************************

ErrorHandler_Init 

Description:
------------
<Description of the purpose of the function>

Input:
-----
	None	
Output:
-------
	None
Returns:
--------
	void  
**********************************************************************************/
void ErrorHandler_Init(void)
{
	memset(&errorHandlerDb,0,sizeof(ErrorHandlerDb_t));
	memset(&RegsDump, 0, sizeof(RegsDump_t));
	memset(&u32FwCoreControlDesc, 0, sizeof(FW_CORE_CONTROL_DESCRIPTOR));
	memset(&ErrorHandler_EventDetails, 0, MAX_NUM_OF_FW_CORES * sizeof(ErrorHandleEventDetails_t));

	errorHandlerDebuggingFlag = TRUE;
	errorHandlerTriggerCore = CORE_NUM_INVALID;

	ErrorHandlerLm0InAssert = FALSE;
	ErrorHandlerLm1InAssert = FALSE;

	errorHandlerAssert = errorHandlerAssertHandlerForInitSequence;
#if defined (ENET_INC_UMAC)
	MT_BclOnExceptionInit();
#endif
}

#if defined (ENET_INC_ARCH_WAVE600)
#ifdef ENET_INC_LMAC0

/****************************************************************************
 **
 ** NAME:           vTLM_ASSERT_TO_LM0_FROM_UM_ISR
 **
 ** PARAMETERS:     none
 **
 ** RETURN VALUES:  void
 **
 ** DESCRIPTION:    ISR routine for LM1 event IGEN_EVT_ASSERT_TO_LM1_FROM_UM.
 **					ASSERT from UM (Core 2).
 **
 ***************************************************************************/
void vTLM_ASSERT_TO_LM0_FROM_UM_ISR(void)
{
	MT_LM_STAT_INC(MT_LM_STAT_EVT_ASSERT_TO_LM0_FROM_UM);

	FATAL("error handler, core 0: core 2 triggered assert");
}

#endif // ENET_INC_LMAC0
#endif

#ifdef ENET_INC_UMAC

/****************************************************************************
 **
 ** NAME:           vTLM_ASSERT_TO_UM_FROM_LM0_ISR
 **
 ** PARAMETERS:     none
 **
 ** RETURN VALUES:  void
 **
 ** DESCRIPTION:    ISR routine for UM event IGEN_EVT_ASSERT_TO_UM_FROM_LM0.
 **					ASSERT from LM0 (Core 0).
 **
 ***************************************************************************/
void vTLM_ASSERT_TO_UM_FROM_LM0_ISR(void)
{
	MT_LM_STAT_INC(MT_LM_STAT_EVT_ASSERT_TO_UM_FROM_LM0);

	FATAL("error handler, core 2: core 0 triggered assert");
}

/****************************************************************************
 **
 ** NAME:           vTLM_ASSERT_TO_UM_FROM_LM1_ISR
 **
 ** PARAMETERS:     none
 **
 ** RETURN VALUES:  void
 **
 ** DESCRIPTION:    ISR routine for UM event IGEN_EVT_ASSERT_TO_UM_FROM_LM0.
 **					ASSERT from LM1 (Core 1).
 **
 ***************************************************************************/
void vTLM_ASSERT_TO_UM_FROM_LM1_ISR(void)
{
	MT_LM_STAT_INC(MT_LM_STAT_EVT_ASSERT_TO_UM_FROM_LM1);

	FATAL("error handler, core 2: core 1 triggered assert");
}

#endif // ENET_INC_UMAC

/****************************************************************************
 **
 ** NAME:           vTLM_ASSERT_TO_LM1_FROM_UM_ISR
 **
 ** PARAMETERS:     none
 **
 ** RETURN VALUES:  void
 **
 ** DESCRIPTION:    ISR routine for LM1 event IGEN_EVT_ASSERT_TO_LM_2_FROM_UM.
 **					ASSERT fom UM (Core 2).
 **
 ***************************************************************************/
 
#if defined (ENET_INC_ARCH_WAVE600)
#if defined(ENET_INC_LMAC0)
void vTLM_ASSERT_TO_LM0_FROM_LM1_ISR(void)
{
	MT_LM_STAT_INC(MT_LM_STAT_EVT_ASSERT_TO_LM0_FROM_LM1);

	FATAL("error handler, core 0: core 1 triggered assert");
}
#endif
#endif

#ifdef ENET_INC_LMAC1

/****************************************************************************
 **
 ** NAME:           vTLM_ASSERT_TO_LM1_FROM_UM_ISR
 **
 ** PARAMETERS:     none
 **
 ** RETURN VALUES:  void
 **
 ** DESCRIPTION:    ISR routine for LM1 event IGEN_EVT_ASSERT_TO_LM1_FROM_UM.
 **					ASSERT from UM (Core 2).
 **
 ***************************************************************************/
void vTLM_ASSERT_TO_LM1_FROM_UM_ISR(void)
{
	MT_LM_STAT_INC(MT_LM_STAT_EVT_ASSERT_TO_LM1_FROM_UM);
	
	FATAL("error handler, core 1: core 2 triggered assert");
}

/****************************************************************************
 **
 ** NAME:           vTLM_ASSERT_TO_LM1_FROM_LM0_ISR
 **
 ** PARAMETERS:     none
 **
 ** RETURN VALUES:  void
 **
 ** DESCRIPTION:    ISR routine for LM1 event IGEN_EVT_ASSERT_TO_LM1_FROM_LM0.
 **					ASSERT from LM0 (Core 0).
 **
 ***************************************************************************/
void vTLM_ASSERT_TO_LM1_FROM_LM0_ISR(void)
{
	MT_LM_STAT_INC(MT_LM_STAT_EVT_ASSERT_TO_LM1_FROM_LM0);

	FATAL("error handler, core 1: core 0 triggered assert");
}

#endif

