/***********************************************************************************
 File:		ErrorHandlerArc.s		
 Module:		Error Handler
 Purpose:		Store Arc registers on exception/assert
 Description:	
************************************************************************************/

	.extern ErrorHandler_Exception

	.file "ErrorHandlerArc.s"
	.text


/**********************************************************************************

_trapExceptionHandler

Description:
------------
	The code that handles the TRAP exceptions.
	TRAP is an instruction that can be inserted to c code using: _ASM("TRAP_S 6_bit_number");
	Currently we use _ASM("TRAP_S 3") to trigger registers dump in ASSERT.
	If TRAP is used for other purposes, make sure the given value is not 0x3.
Input:
-----
	None	
Output:
-------
	None		
Returns:
--------
	None
**********************************************************************************/	
.global	_trapExceptionHandler
.type	_trapExceptionHandler, @function
_trapExceptionHandler:
	st	 	%blink, [%sp, -20]
	jl 		_dumpArcRegs
	ld	 	%blink, [%sp, -20]
	push_s	%r0
	lr		%r0,[%ecr]
	xbfu	%r0, %r0, 0, 6
	cmp_s	%r0, 0x3
	pop_s	%r0
	bnz		ErrorHandler_Exception
	rtie




/**********************************************************************************

_exceptionHandler

Description:
------------
	The code that handles exceptions. Store registers and jump back to c code.
Input:
-----
	None	
Output:
-------
	None		
Returns:
--------
	None
**********************************************************************************/	
.global	_exceptionHandler
.type	_exceptionHandler, @function
_exceptionHandler:
	st	 	%blink, [%sp, -20]
	jl 		_dumpArcRegs
	ld	 	%blink, [%sp, -20]	
	b 		ErrorHandler_Exception




/**********************************************************************************

_dumpArcRegs

Description:
------------
	Store exception related Auxiliary registers and all 32 core registers
Input:
-----
	None	
Output:
-------
	None		
Returns:
--------
	None
**********************************************************************************/	
.global	_dumpArcRegs
.type	_dumpArcRegs, @function
_dumpArcRegs:

	// Store BLINK/R0/R1/R2 in stack so that we can change them in order to save all the registers
	push_s 	%r2
	push_s 	%blink
	push_s 	%r0 
	push_s 	%r1

	// Get the original value of blink from the moment of the exception (it was stored before the call to _dumpArcRegs)
	ld		%blink, [%sp, -4]

	// Get R0 to points at RegsDump
	mov     %r0,RegsDump 

	// Store eret auxiliary register
	lr      %r1,[%eret]
	st_s	%r1,[%r0]

	// Store erbta auxiliary register (to R0 which is the base address of RegsDump + offset to the correct struct member)
	lr		%r1,[%erbta]
	st_s	%r1,[%r0,4]

	// Store erstatus auxiliary register
	lr		%r1,[%erstatus]
	st_s	%r1,[%r0,8]

    // Store ecr auxiliary register      
	lr		%r1,[%ecr]
	st_s	%r1,[%r0,12]

	// Store efa auxiliary register 
	lr		%r1,[%efa]
	st_s	%r1,[%r0,16]

	// Store status32 auxiliary register (after exception was already done. value before exception is stored in erstatus)
	lr		%r1,[%status32]
	st_s	%r1,[%r0,20]

	// Store mpu_ecr/mpu_ic auxiliary register
	lr		%r1,[0x420]		// get mpc_ecr/mpu_ic
	add		%r0,%r0,152		// get R0 to point to RegsDump.mpu_ic_Reg - (152=128+6*4)
	st_s	%r1,[%r0]		// store mpc_ecr/mpu_ic after Core Regs
	sub		%r0,%r0,152		// restore R0 to point to RegsDump.

	// Store core registers R2-R25. Use STD ("Store double") which stores 2 consecutive registers to a given [address + offset]. offset is 8 bits singed, so can't be more than 127.
	std		%r2,[%r0,32]
 	std		%r4,[%r0,40]
	std		%r6,[%r0,48]
	std		%r8,[%r0,56]
	std		%r10,[%r0,64]
	std		%r12,[%r0,72]
	std		%r14,[%r0,80]
	std		%r16,[%r0,88]
	std		%r18,[%r0,96]
	std		%r20,[%r0,104]
	std		%r22,[%r0,112]
	std		%r24,[%r0,120]

	// Get R2 to points at RegsDump instead of R0 so that we can store R0 as well.
	mov		%r2, %r0

	// Get original values of R0/R1 and store them both
	pop_s 	%r1 
	pop_s 	%r0
	std		%r0,[%r2,24]
	
	// Get R2 to points at RegsDump.coreRegs[REG_R26] (add2 is adding (0x20<<2 = 0x80) to R2)
	add2    %r2,%r2, 0x20

	// Store core registers: R26/R27 + R30/R31
	std		%r26,[%r2,0]
	std		%r30,[%r2,16]

	// Get the BLINK value that will take us back to caller function
	pop		%blink

	// Store core registers R28(SP)/R29
	std		%r28,[%r2,8]

	// Get original values of R2
	pop		%r2

	// Jump back to caller (_exceptionHandler / _trapExceptionHandler)
	j		[%blink]


