/***********************************************************************************
 File:		ErrorHandler.c		
 Module:		Error Handler
 Purpose:		
 Description:	
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "System_Information.h"
#include "stringLibApi.h"
#include "Utils_Api.h"
#include "mt_cachedefs.h"
#include "ErrorHandler_Api.h"
#include "HostInterface_API.h"
#include "um_interface.h"
#include "init_ifmsg.h"

#include "ShramHim.h"
#include "shram_man_msgs.h"
#include "ShramErrorHandler.h"
#include "MT_osutl.h"
#include "int_gen.h"
#include "enet_hrc.h"
#include "enet_pas.h"
#include "Pac_Api.h"
#include "Ipc_Api.h"
#include "LmHdk_API.h"
#include "PhyDriver_API.h"
#include "RegAccess_Api.h"
#include "HostInterface_API.h"
#include "PhyTestBus_API.h"
#include "HwMemoryMap.h"
#include "MacHtExtensionsRegs.h"
#include "MT_MAC_HT_extensions_regs.h"
#include "lm_statistics.h"
#include "loggerAPI.h"
#include "Pac_Api.h"
#if defined (ENET_INC_UMAC)
#include "SenderInterface_Api.h"
#include "HostGenRiscInterface_Api.h"
#include "HostInterface_MailboxApi.h"
#include "HostInterface_InitApi.h"
#include "RxHandler_Api.h"
#include "HwLoggerAPI.h"
#include "TxHandler_Api.h"
#include "QManagerUmacRegs.h"
#endif
#if defined (ENET_INC_LMAC)
#include "AggregationBuilder_Api.h"
#endif 


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_ERROR_HANDLER 
#define LOG_LOCAL_FID 1

#define ERROR_EXCEPTION           			(0xA7)
#define ERROR_ASSERT              			(0xA8)
#define MAX_NUMBER_OF_WLAN_INTERFACES		(4)
#define WLAN_INTERFACE_ID_MASK				(0x3)
#define MIN_WAITING_TIME_FOR_LM_ASSERT  	(10)
#define MAX_WAITING_TIME_FOR_LM_ASSERT  	(500)
#define NUM_OF_GP_REGS_DUMPED_IN_EXCEPTION 	(16)
#define MAX_NUMBER_OF_WLAN_INTERFACES		(4)
#define WLAN_INTERFACE_ID_MASK				(0x3)
/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/
#define GET_WLAN_INTERFACE_ID()		(FWinterface & WLAN_INTERFACE_ID_MASK)
#define GET_ORIGIN_ID()				(1)//Origin ID *must* match its definition in AutoBuild\mt_build.pl subroutine RunNew_Event_PrepScript(): "...my $originID = 1 ..."

/*---------------------------------------------------------------------------------
/                       Data Type Definition                        
/----------------------------------------------------------------------------------*/

typedef struct ErrorHandlerexcRegsValues
{	
	uint32 causeReg;
	uint32 epcReg;
	uint32 srsCtlReg;
	uint32 statusReg;
} ErrorHandlerExcRegsValues_t;

typedef struct ErrorHandlerDb_s
{
	uint32 enabledModulesHalt;
	CoreNum_e triggerCore;
} ErrorHandlerDb_t;

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void errorHandlerAssertHandlerForInitSequence(uint32 assertGroupId, uint32 assertFileId, uint32 assertLine);
static void errorHandlerAssertHandlerForSteadyState(uint32 assertGroupId, uint32 assertFileId, uint32 assertLine);
static void errorHandlerFillAssertMsg(uint32 in_uLine, uint32 groupId, uint32 fileId, uint32 in_uTimeStamp, uint32 in_uSpReg);
static void errorHandlerFillExceptionMsg(uint16 in_uCpuType);
#if defined (ENET_INC_UMAC)
static void errorHandlerSendErrorMsgToHost(void);
#endif

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
static ErrorHandlerDb_t errorHandlerDb;

void (*errorHandlerAssert)(uint32 assertGroupId, uint32 assertFileId, uint32 assertLine); /* pointer to assert handler */
uint32 errorHandlerFatalSpReg;
static bool   errorHandlerDebuggingFlag; 

#if defined (ENET_INC_UMAC)
volatile uint32 errorHandlerAssertLowerMacOneTime;
uint32 errorHandlerHwQmanagerNullPushErr = 0;
#endif

#if defined (ENET_INC_LMAC)
bool errorHandlerResetPhyOnAssert;
#endif

uint32 FWinterface; /* Logmacros Uses thid variable*/
bool assertOccurred = FALSE; 

extern volatile uint32 True_g;

ErrorHandlerExcRegsValues_t errorhandlerRegsValues;

/* Hold the information about the exception that occurred  -Special Regs values and required GP Regs values */
	/*	General Purpose Registers:
		0-sp, 1-ra,  2-v0,	3-v1,  4-a0,   5-a1,  6-a2,   7-a3
		8-s0, 9-s1, 10-s2, 11-s3, 12-s4, 13-s5, 14-s6, 15-s7 */
uint32 ErrorHandlerExceptionregs[NUM_OF_GP_REGS_DUMPED_IN_EXCEPTION]; 


/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

errorHandlerAssertHandlerForSteadyState 

Description:
------------
	Handler for fatal error event in FW after initialization sequence
		LM:
			1) Disable All HW
			2) Sends debug msg to UM
		UM:
			1) If fatal error is not originated from LM causes Lm to enter into assert.
			2) Sends msg To Host with all information of the error.
Input:
-----
	None	
Output:
-------
	None
Returns:
--------
	void  
**********************************************************************************/
#ifdef ENET_INC_UMAC
static void errorHandlerAssertHandlerForSteadyState(uint32 assertGroupId, uint32 assertFileId, uint32 assertLine)
{	
	FwHostGenRiscInterfaceMailbox1FifoInMessage_t message;
	CoreNum_e myCoreNum = SystemInfo_GetCoreNum();

	TX_INTERRUPT_SAVE_AREA;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	
	/* turn on the flag which signs that we've got an assert */
	if(assertOccurred == TRUE)
	{
		while(True_g);
//		return; /* In case we allready on ASSERT contexts (Assert within an Assert) */ 
	}
		
	assertOccurred = TRUE;
	
	if (errorHandlerTriggerCore == CORE_NUM_INVALID)
	{
		errorHandlerTriggerCore = SystemInfo_GetCoreNum();
	}

	/* we take stack pointer, and add 12 bytes offset to compensate for the assert 
		 call itself and deliver pointer to the actual function where the assert occurred */
	if (errorHandlerFatalSpReg == 0)
	{
		errorHandlerFatalSpReg = __GetStackPointer() + 12; // 4 bytes due to this function (errorHandlerAssertHandlerForSteadyState) and 8 more bytes in "ErrorHandler_AssertHandler"
	}
	
	
	ILOG0_DDD("vAPP_Assert assertGroupId=%d assertFileId=%d assertLine=%d", assertGroupId, assertFileId, assertLine);

#if defined (ANALYZE_DISABLE_INTERRUPT)
	AAA_DebugStopTestBus(); /* Stop GenRisc trace*/
#endif

	// Read HW Q Manager null push error counter
	RegAccess_Read(REG_Q_MANAGER_UMAC_UMAC_NULL_PUSH_ERR, &errorHandlerHwQmanagerNullPushErr);

	// TX Handler is in LMAC in wave600 and in UMAC in wave500
	TxHandler_DisableTxh();

	//	HostGenRisc_Halt();
	Sender_Halt();

	//leave only Rx out and logger tasks in Host IF Genrisc
	message.messageId = FW_HOST_INTERFACE_MAILBOX_1_FIFO_IN_MESSAGE_ID_CHANGE_ENABLED_TASKS_BITMAP;
	message.messageValue = ((0x1 << HOST_INTERFACE_INIT_FIELD_ENABLED_TASKS_BITMAP_RX_OUT_TASK_BIT) | (0x1 << HOST_INTERFACE_INIT_FIELD_ENABLED_TASKS_BITMAP_LOGGER_TASK_BIT));

	if(errorHandlerDb.enabledModulesHalt & (1 << HALT_MODULE_HOST_IF_GENRISC))
	{
		HostGenRiscInterface_Mailbox1_Push(&message);
	}
	
	/* Stop Rx */
	RegAccess_Write(REG_MAC_HT_EXTENSIONS_DEL_RX_HALT_REQUEST,1);

	RxHandler_Halt();

	errorHandlerFillAssertMsg(assertLine, assertGroupId, assertFileId, GET_TSF_TIMER_LOW(), errorHandlerFatalSpReg);

	switch (errorHandlerTriggerCore)
	{
		case CORE_HOST:
			// if errorHandlerTriggerCore is CORE_HOST and i am CORE_NUM_2: trigger no one, host is responsible to trigger all cores
			// Patch - currently host trigger only UM, so we need to stop LM. Patch can be removed if all bits are set in "u32FwCoreControlDesc.u32FwCoreCtrl".
			if (myCoreNum == CORE_NUM_2)
			{
				vIGEN_SendToLm0(IGEN_EVT_ASSERT_TO_LM0_FROM_UM);
			}			
			break;
		case CORE_NUM_0:
			// if errorHandlerTriggerCore is CORE_NUM_0 and i am CORE_NUM_2: trigger no one
			break;
		case CORE_NUM_2:
			// if errorHandlerTriggerCore is CORE_NUM_2 and i am CORE_NUM_2: trigger lm0
			if (myCoreNum == CORE_NUM_2)
			{
				vIGEN_SendToLm0(IGEN_EVT_ASSERT_TO_LM0_FROM_UM);
			}
			break;
		default:
			FATAL("errorHandlerTriggerCore not set");
			break;
	}

	/* This block is a waiting for the	LM to finish its own assertion */
	{
		uint32 startWaitToLMs = GET_TSF_TIMER_LOW();

		while(TRUE)
		{
			if (ErrorHandlerLm0InAssert == TRUE)
			{
				break;
			}

			/* We wait at least 10 micro seconds for the LM to finish the Assert procedure */
			if ((GET_TSF_TIMER_LOW() - startWaitToLMs) > MAX_WAITING_TIME_FOR_LM_ASSERT)
			{
				break;
			}
		}
	}

	errorHandlerSendErrorMsgToHost();

	sys_flush_caches();

	while( errorHandlerDebuggingFlag == TRUE )
	{
		MT_CheckBclOnException();
	}
	OSAL_ENABLE_INTERRUPTS(interrupt_save); //only for the protocol (unreachable code)
}
#endif


/**********************************************************************************

errorHandlerAssertHandlerForSteadyState 

Description:
------------
	Handler for fatal error event in FW after initialization sequence
		LM:
			1) Disable All HW
			2) Sends debug msg to UM
		UM:
			1) If fatal error is not originated from LM causes Lm to enter into assert.
			2) Sends msg To Host with all information of the error.
Input:
-----
	None	
Output:
-------
	None
Returns:
--------
	void  
**********************************************************************************/
#ifdef ENET_INC_LMAC
static void errorHandlerAssertHandlerForSteadyState(uint32 assertGroupId, uint32 assertFileId, uint32 assertLine)
{	
	TX_INTERRUPT_SAVE_AREA;
	CoreNum_e myCoreNum = SystemInfo_GetCoreNum();

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	
	/* turn on the flag which signs that we've got an assert */
	if(assertOccurred == TRUE)
	{
		while(True_g);
//		return; /* In case we allready on ASSERT contexts (Assert within an Assert) */ 
	}

	assertOccurred = TRUE;
	
	if (errorHandlerTriggerCore == CORE_NUM_INVALID)
	{
		errorHandlerTriggerCore = SystemInfo_GetCoreNum();
	}

	AggregationBuilder_Halt();

#if defined (ANALYZE_DISABLE_INTERRUPT)
	AAA_DebugStopTestBus(); /* Stop GenRisc trace*/
#endif

	/* Disable Phy & RF & RXC*/
	if (errorHandlerResetPhyOnAssert == FALSE)
	{
		Pac_RxcDisable();//stop RX on Assert 
	}
	
	/* we take stack pointer, and add 12 bytes offset to compensate for the assert 
		 call itself and deliver pointer to the actual function where the assert occurred */
	if (errorHandlerFatalSpReg == 0)
	{
		errorHandlerFatalSpReg = __GetStackPointer() + 12; // 4 bytes due to this function (errorHandlerAssertHandlerForSteadyState) and 8 more bytes in "ErrorHandler_AssertHandler"
	}

	errorHandlerFillAssertMsg(assertLine, assertGroupId, assertFileId, GET_TSF_TIMER_LOW(), errorHandlerFatalSpReg);

	ErrorHandlerLm0InAssert = TRUE; 

	switch (errorHandlerTriggerCore)
	{
		case CORE_HOST:
			// if errorHandlerTriggerCore is CORE_HOST and i am CORE_NUM_0: trigger no one, host is responsible to trigger all cores
			break;
		case CORE_NUM_0:
			// if errorHandlerTriggerCore is CORE_NUM_0 and i am CORE_NUM_0: trigger um
			if (myCoreNum == CORE_NUM_0)
			{
				vIGEN_SendToUm(IGEN_EVT_ASSERT_TO_UM_FROM_LM0);
			}			
			break;
		case CORE_NUM_2:
			// if errorHandlerTriggerCore is CORE_NUM_2 and i am CORE_NUM_0: trigger no one
			break;
		default:
			FATAL("errorHandlerTriggerCore not set");
			break;
	}

	sys_flush_caches();

	while( errorHandlerDebuggingFlag == TRUE )
	{
		MT_CheckBclOnException();
	}
	OSAL_ENABLE_INTERRUPTS(interrupt_save); //only for the protocol (unreachable code)
}

#endif


/**********************************************************************************

errorHandlerAssertHandlerForInitSequence 

Description:
------------
	Handler for fatal error event in FW during initialization sequence

Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void  
**********************************************************************************/
static void errorHandlerAssertHandlerForInitSequence(uint32 assertGroupId, uint32 assertFileId, uint32 assertLine)
{
	//currently do nothing, just hang here and wait
	while( errorHandlerDebuggingFlag == TRUE )
	{
		MT_CheckBclOnException();
	}
}


/**********************************************************************************

errorHandlerFillAssertMsg 

Description:
------------
	Fill in Error structure in SHRAM will be updated with Assert Data. 

Input:
-----
	in_ppcSourceFilename - the filename of the Assert
	in_uTimeStamp - time of the Assert
Output:
-------
	None
Returns:
--------
	void  
**********************************************************************************/
static void errorHandlerFillAssertMsg(uint32 line, uint32 groupId, uint32 fileId, uint32 timeStamp, uint32 spReg)
{	
	CoreNum_e coreNum = SystemInfo_GetCoreNum();

    ErrorHandler_EventDetails[coreNum].uMagicMun 			= ERROR_ASSERT;
    ErrorHandler_EventDetails[coreNum].uCauseRegOrLineNum 	= line;
	ErrorHandler_EventDetails[coreNum].uFileId 				= (uint16) fileId;
	ErrorHandler_EventDetails[coreNum].uGroupId 			= (uint16) groupId;
	ErrorHandler_EventDetails[coreNum].uOriginId 			= GET_ORIGIN_ID();
	ErrorHandler_EventDetails[coreNum].uFWinterface 		= GET_WLAN_INTERFACE_ID();
    ErrorHandler_EventDetails[coreNum].uCoreNum 			= coreNum;
    ErrorHandler_EventDetails[coreNum].uTimeStamp 			= timeStamp;
	ErrorHandler_EventDetails[coreNum].uSpReg 				= spReg;
	ErrorHandler_EventDetails[coreNum].uEpcReg 				= 0;
}

/**********************************************************************************

errorHandlerFillExceptionMsg 

Description:
------------
	Fill Exception data, Error structure in SHRAM will be updete with exception data
Input:
-----
	in_uCpuType - 1 for UM 0- for LM		
Output:
-------
	None
Returns:
--------
	void  
**********************************************************************************/
static void errorHandlerFillExceptionMsg(uint16 cpuType)
{	
	CoreNum_e coreNum = SystemInfo_GetCoreNum();

    ErrorHandler_EventDetails[coreNum].uMagicMun 			= ERROR_EXCEPTION;
    ErrorHandler_EventDetails[coreNum].uCoreNum 			= cpuType;
    ErrorHandler_EventDetails[coreNum].uCauseRegOrLineNum 	= ((__GetCauseReg())>>2) & 0x1f;
    ErrorHandler_EventDetails[coreNum].uEpcReg 				= __GetEpcReg();
    ErrorHandler_EventDetails[coreNum].uStatusReg 			= __GetStatusReg();
    ErrorHandler_EventDetails[coreNum].uTimeStamp 			= GET_TSF_TIMER_LOW();
	ErrorHandler_EventDetails[coreNum].uSpReg  				= ErrorHandlerExceptionregs[0];
}


/**********************************************************************************

errorHandlerSendErrorMsgToHost 

Description:
------------
	Send error massage to driver

Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void  
**********************************************************************************/
#if defined (ENET_INC_UMAC)
static void errorHandlerSendErrorMsgToHost(void)
{
    K_MSG*	   psErrorMsg ;
    MAC_EVENT* psErrorData;
	CoreNum_e coreNum = errorHandlerTriggerCore;

	psErrorMsg  = (K_MSG*) &asMAN_IndData[BSS_IND_COUNT + STA_IND_COUNT + RSN_IND_COUNT];
    psErrorData = (MAC_EVENT*)psErrorMsg->abData;
	
    OSAL_INIT_PERSISTENT_MSG( psErrorMsg, sizeof( UMI_IND ) );

	if (coreNum == CORE_HOST)
	{
		// If this is an assert from host, it means all cores were stopped. We can take the details (line, groupID, etc...) from the UM core.
		// We must not access "ErrorHandler_EventDetails" with index "CORE_HOST" since it doesn't exist.
		coreNum = CORE_NUM_1;
	}
	
    if(ErrorHandler_EventDetails[coreNum].uMagicMun == ERROR_ASSERT)
    {
		psErrorData->u32EventID 							= EVENT_APP_FATAL;
		psErrorData->u.sAppFatalEvent.FileId				= ErrorHandler_EventDetails[coreNum].uFileId;
		psErrorData->u.sAppFatalEvent.GroupId				= ErrorHandler_EventDetails[coreNum].uGroupId;
		psErrorData->u.sAppFatalEvent.OriginId				= ErrorHandler_EventDetails[coreNum].uOriginId;
		psErrorData->u.sAppFatalEvent.FWinterface 			= ErrorHandler_EventDetails[coreNum].uFWinterface;
        psErrorData->u.sAppFatalEvent.uCauseRegOrLineNum 	= ErrorHandler_EventDetails[coreNum].uCauseRegOrLineNum;
    }
    else if (ErrorHandler_EventDetails[coreNum].uMagicMun == ERROR_EXCEPTION)
    {
		psErrorData->u32EventID								= EVENT_EXCEPTION; 
		psErrorData->u.sAppFatalEvent.uEpcReg 				= ErrorHandler_EventDetails[coreNum].uEpcReg;
		psErrorData->u.sAppFatalEvent.uStatusReg 			= ErrorHandler_EventDetails[coreNum].uStatusReg;
		psErrorData->u.sAppFatalEvent.uCauseRegOrLineNum 	= ErrorHandler_EventDetails[coreNum].uCauseRegOrLineNum;
		psErrorData->u.sAppFatalEvent.FWinterface 			= ErrorHandler_EventDetails[coreNum].uFWinterface;
    }
	
	psErrorData->u32Timestamp 					= ErrorHandler_EventDetails[coreNum].uTimeStamp;
	psErrorData->u.sAppFatalEvent.uTimeStamp  	= ErrorHandler_EventDetails[coreNum].uTimeStamp;
	psErrorData->u.sAppFatalEvent.uCoreNum 		= ErrorHandler_EventDetails[coreNum].uCoreNum;
	psErrorMsg->header.tKMsgType 				= MC_MAN_MAC_EVENT_IND;
	psErrorMsg->header.vapId 					= 0;

    vHIM_SendMsgToHost(psErrorMsg);       
}
#endif /* defined (ENET_INC_UMAC)  */



/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/	


/**********************************************************************************

ErrorHadler_CheckAssertReqFromHost

Description:
------------
<Description of the purpose of the function>

Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void  
**********************************************************************************/
void ErrorHadler_CheckAssertReqFromHost(void)
{
	CoreNum_e cpuNumber = SystemInfo_GetCoreNum();	
	uint32 localMipsCtrl = u32FwCoreControlDesc.u32FwCoreCtrl[cpuNumber];
	TX_INTERRUPT_SAVE_AREA;

	if(MTLK_BFIELD_GET((localMipsCtrl), FW_CORE_CTRL_DO_ASSERT) == TRUE)
	{
		if ((errorHandlerTriggerCore == CORE_NUM_INVALID) || (errorHandlerTriggerCore == CORE_HOST))
		{
#if defined (ENET_INC_LMAC)
			errorHandlerResetPhyOnAssert = FALSE;
#endif
			
			OSAL_DISABLE_INTERRUPTS(&interrupt_save);
			errorHandlerTriggerCore = CORE_HOST;
			FATAL("Assert Requested by the Host");
			OSAL_ENABLE_INTERRUPTS(interrupt_save); //only for the protocol (unreachable code)
		}
	}
}

/************************************************************************************

ErrorHandler_GetCtrlStructure

Description:
------------
<Description of the purpose of the function>

Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void - 
	
**********************************************************************************/
#if defined (ENET_INC_UMAC)
uint32 ErrorHandler_GetCtrlStructure(void)
{
    return (uint32)(&u32FwCoreControlDesc.u32FwCoreCtrl[0]);
}
#endif


/**********************************************************************************

ErrorHandlerEnableModuleHalt 

Description:
------------
Enable Module halt during Fatal\Assert or exception

Input:
-----
	Required Halted Module id
	
**********************************************************************************/
void ErrorHandlerEnableModuleHalt(HaltedModules_e haltModuleId)
{
	errorHandlerDb.enabledModulesHalt |= (1 << haltModuleId);
}

/**********************************************************************************

errorHandlerAssertHandlerForInitSequence 

Description:
------------
<Description of the purpose of the function>

Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void  
	
**********************************************************************************/
void ErrorHandler_AssertHandler(uint32 assertGroupId, uint32 assertFileId, uint32 assertLine)
{
	errorHandlerAssert(assertGroupId,assertFileId,assertLine);
}


/**********************************************************************************

ErrorHandler_Exception

Description:
------------
<Description of the purpose of the function>

Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void  
**********************************************************************************/
void ErrorHandler_Exception(void)
{  
    int epcReg;
	CoreNum_e myCore;

	
    epcReg = __GetEpcReg();                 // The exception point program counter 
   *((volatile uint32*)(SHARED_RAM_BASE_ADDR)) = epcReg; 
    
    // Store epcReg and causeReg for debug
    errorhandlerRegsValues.causeReg = ((__GetCauseReg())>>2) & 0x1f; // The exception cause bits ;
	errorhandlerRegsValues.epcReg = epcReg;
	errorhandlerRegsValues.statusReg = __GetStatusReg();
	errorhandlerRegsValues.srsCtlReg = __GetSrsCtlReg();

	myCore = SystemInfo_GetCoreNum();

	if (errorHandlerTriggerCore == CORE_NUM_INVALID)
	{
		errorHandlerTriggerCore = myCore;
	}

	errorHandlerFillExceptionMsg(myCore);

#if defined (ENET_INC_LMAC)
	if (myCore == CORE_NUM_0)
	{
		vIGEN_SendToUm(IGEN_EVT_ASSERT_TO_UM_FROM_LM0);
	}
#endif

#if defined (ENET_INC_UMAC)
	if (myCore == CORE_NUM_1)
	{
		vIGEN_SendToLm0(IGEN_EVT_ASSERT_TO_LM0_FROM_UM);
	}
	errorHandlerSendErrorMsgToHost();
#endif

    while(1)
	{
		// BCL on Exception 
		MT_CheckBclOnException();
    } 
}

/**********************************************************************************

ErrorHandler_Ready

Description:
------------
<Description of the purpose of the function>

Input:
-----
	None	
Output:
-------
	None
Returns:
--------
	void  
**********************************************************************************/
#if defined (ENET_INC_UMAC)
#pragma ghs section text=".initialization" 
#endif
void ErrorHandler_Ready(void)
{
	CoreNum_e cpuNumber = SystemInfo_GetCoreNum();
	
	errorHandlerFatalSpReg = 0;

	/* Identify cpu number 1 -ucpu, 0 - lcpu */ 
	u32FwCoreControlDesc.u32FwCoreCtrl[cpuNumber] = 0;
	
#if defined (ENET_INC_UMAC)	
	errorHandlerAssertLowerMacOneTime = TRUE;
#endif
	
#if defined (ENET_INC_LMAC)
	errorHandlerResetPhyOnAssert = TRUE;
#endif

	errorHandlerAssert = errorHandlerAssertHandlerForSteadyState;
}
#if defined (ENET_INC_UMAC)
#pragma ghs section text=default
#endif
/**********************************************************************************

ErrorHandler_SetFwInterfaceId 

Description:
------------
<Description of the purpose of the function>

Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void  
**********************************************************************************/
void ErrorHandler_SetFwInterfaceId(uint32 fwInterface)
{
	FWinterface = fwInterface;
	ASSERT(FWinterface < MAX_NUMBER_OF_WLAN_INTERFACES);
}

/**********************************************************************************

ErrorHandler_Init 

Description:
------------
<Description of the purpose of the function>

Input:
-----
	None	
Output:
-------
	None
Returns:
--------
	void  
**********************************************************************************/
#if defined (ENET_INC_UMAC) || defined (ENET_INC_LMAC)
#pragma ghs section text=".initialization" 
#endif

void ErrorHandler_Init(void)
{
	memset(&errorHandlerDb,0,sizeof(ErrorHandlerDb_t));
	memset(&u32FwCoreControlDesc, 0, sizeof(FW_CORE_CONTROL_DESCRIPTOR));
	memset(&ErrorHandler_EventDetails, 0, MAX_NUM_OF_FW_CORES * sizeof(ErrorHandleEventDetails_t));

	errorHandlerDebuggingFlag = TRUE;
	errorHandlerTriggerCore = CORE_NUM_INVALID;

	ErrorHandlerLm0InAssert = FALSE;
	errorHandlerAssert = errorHandlerAssertHandlerForInitSequence;
}
#if defined (ENET_INC_UMAC) || defined (ENET_INC_LMAC)
#pragma ghs section text=default
#endif


/****************************************************************************
 **
 ** NAME:           vTLM_ASSERT_TO_LM0_FROM_UM_ISR
 **
 ** PARAMETERS:     none
 **
 ** RETURN VALUES:  void
 **
 ** DESCRIPTION:    ISR routine for LM1 event IGEN_EVT_ASSERT_TO_LM1_FROM_UM.
 **					ASSERT fom UM (Core 2).
 **
 ***************************************************************************/
void vTLM_ASSERT_TO_LM0_FROM_UM_ISR(void)
{
	MT_LM_STAT_INC(MT_LM_STAT_EVT_ASSERT_TO_LM0_FROM_UM);

	FATAL("error handler, core 0: core 2 triggered assert");
}

/****************************************************************************
 **
 ** NAME:           vTLM_ASSERT_TO_UM_FROM_LM0_ISR
 **
 ** PARAMETERS:     none
 **
 ** RETURN VALUES:  void
 **
 ** DESCRIPTION:    ISR routine for UM event IGEN_EVT_ASSERT_TO_UM_FROM_LM0.
 **					ASSERT from LM0 (Core 0).
 **
 ***************************************************************************/
void vTLM_ASSERT_TO_UM_FROM_LM0_ISR(void)
{
	MT_LM_STAT_INC(MT_LM_STAT_EVT_ASSERT_TO_UM_FROM_LM0);

	FATAL("error handler, core 2: core 0 triggered assert");
}

