/***********************************************************************************
 File:			GidmManager.c
 Module:		Group ID Management Action Frame Manager
 Purpose: 		handle IDM Manager events
 Description:   
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "HwMemoryMap.h"
#include "TxManager_Api.h"
#include "ResourceManager_API.h"
#include "Locker_Api.h"
#include "OSAL_Kmsg.h"
#include "ErrorHandler_Api.h"
#include "HwQManager_API.h"
#include "PacketDescriptor.h"
#include "ShramPacketDescriptors.h"
#include "ChannelSwitchManager_Api.h"
#include "TxPacketsClassifier_API.h"
#include "PhyDriver_API.h"
#include "GroupManager_API.h"
#include "GidmManager_Api.h"
#include "GidmManager.h"
#include "loggerAPI.h"
#include "Utils_Api.h"
#include "Pac_Api.h"
#include "VapDatabase_Api.h"
#include "RegAccess_Api.h"
#include "StaDatabase_Api.h"
#include "loggerAPI.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_GIDM_MANAGER 
#define LOG_LOCAL_FID 0


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

#define GET_USER_POSITION_FOR_GROUP(UserPositionBitmap, groupIndex) 		((((UserPositionBitmap)[((groupIndex) >> 2)]) & (0x3 << (((groupIndex) & 0x3)*2))) >> (((groupIndex) & 0x3)*2))
#define GET_GROUP_MEMBERSHIP_STATUS(MembershipStatusBitmap, groupIndex)		((((MembershipStatusBitmap)[((groupIndex) >> 3)]) & (0x1 << ((groupIndex) & 0x7))) >> ((groupIndex) & 0x7))

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Debug Code									
/----------------------------------------------------------------------------------*/

// Turn on/off gidm debug by comment/uncomment:

//#define DEBUG_GIDM_MANAGER 

#if defined (DEBUG_GIDM_MANAGER)

#define AAA_GIDM_MANAGER_DEBUG_ARRAY_SIZE (128)	// Debug array size

typedef struct _aaa_gidmManager_
{
	uint32 					tsf;
	uint8 					staIndex;	
	gidmManagerStates_e 	state;
	gidmManagerEvents_e 	event;	
}	aaa_gidmManager_t;

aaa_gidmManager_t 			AAA_GIDM_MANAGER[AAA_GIDM_MANAGER_DEBUG_ARRAY_SIZE] = {0}; 	// Debug array
uint32 						AAA_GIDM_MANAGER_INDEX = 0;									// Debug array index

#endif


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

GidmManagerSm_t GidmManagerSm[GIDM_MANAGER_STATE_NUM_STATES] = 
{
	// GIDM_MANAGER_STATE_IDLE
	{
		GidmManager_StartTx, 						// GIDM_MANAGER_EVENT_START_TX_REQ
		GidmManager_IllegalEvent,					// GIDM_MANAGER_EVENT_STOP_TX_REQ
		GidmManager_IllegalEvent,					// GIDM_MANAGER_EVENT_TX_CFM
		GidmManager_IllegalEvent,					// GIDM_MANAGER_EVENT_PD_ALLOCATED
		GidmManager_IllegalEvent,					// GIDM_MANAGER_EVENT_LOCK_CONFIRMED
	},
	// GIDM_MANAGER_STATE_WAIT_TX_CFM
	{
		GidmManager_IllegalEvent, 					// GIDM_MANAGER_EVENT_START_TX_REQ
		GidmManager_StopInWaitTxCfm,				// GIDM_MANAGER_EVENT_STOP_TX_REQ
		GidmManager_TxCfm,							// GIDM_MANAGER_EVENT_TX_CFM
		GidmManager_IllegalEvent,					// GIDM_MANAGER_EVENT_PD_ALLOCATED
		GidmManager_IllegalEvent,					// GIDM_MANAGER_EVENT_LOCK_CONFIRMED
	},
	// GIDM_MANAGER_STATE_WAIT_FOR_PD
	{
		GidmManager_IllegalEvent, 					// GIDM_MANAGER_EVENT_START_TX_REQ
		GidmManager_StopInWaitForPd,				// GIDM_MANAGER_EVENT_STOP_TX_REQ
		GidmManager_IllegalEvent,					// GIDM_MANAGER_EVENT_TX_CFM
		GidmManager_PdAllocatedInWaitforPd,			// GIDM_MANAGER_EVENT_PD_ALLOCATED
		GidmManager_IllegalEvent,					// GIDM_MANAGER_EVENT_LOCK_CONFIRMED
	},
	// GIDM_MANAGER_STATE_WAIT_FOR_PD_PENDING_STOP
	{
		GidmManager_IllegalEvent, 					// GIDM_MANAGER_EVENT_START_TX_REQ
		GidmManager_IllegalEvent,					// GIDM_MANAGER_EVENT_STOP_TX_REQ
		GidmManager_IllegalEvent,					// GIDM_MANAGER_EVENT_TX_CFM
		GidmManager_PdAllocatedInPendingStop,		// GIDM_MANAGER_EVENT_PD_ALLOCATED
		GidmManager_IllegalEvent,					// GIDM_MANAGER_EVENT_LOCK_CONFIRMED
	},
	// GIDM_MANAGER_STATE_WAIT_FOR_LOCK
	{
		GidmManager_IllegalEvent, 					// GIDM_MANAGER_EVENT_START_TX_REQ
		GidmManager_IllegalEvent,					// GIDM_MANAGER_EVENT_STOP_TX_REQ
		GidmManager_TxCfmInWaitForLock,				// GIDM_MANAGER_EVENT_TX_CFM
		GidmManager_IllegalEvent,					// GIDM_MANAGER_EVENT_PD_ALLOCATED
		GidmManager_LockCfmInWaitForLock,			// GIDM_MANAGER_EVENT_LOCK_CONFIRMED
	},	
	// GIDM_MANAGER_STATE_WAIT_FOR_LOCK_PD_CONFIRMED
	{
		GidmManager_IllegalEvent, 					// GIDM_MANAGER_EVENT_START_TX_REQ
		GidmManager_IllegalEvent,					// GIDM_MANAGER_EVENT_STOP_TX_REQ
		GidmManager_IllegalEvent,					// GIDM_MANAGER_EVENT_TX_CFM
		GidmManager_IllegalEvent,					// GIDM_MANAGER_EVENT_PD_ALLOCATED
		GidmManager_LockCfmInWaitLockPdConfirmed,	// GIDM_MANAGER_EVENT_LOCK_CONFIRMED
	},	
};

// Group ID Management Action Frame Manager Database
static GidmManagerDb_t	GidmManagerDb;


/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/


void GidmManager_RunStateMachine(gidmManagerEvents_e event, StaId staIndex, void *parameter)
{
	gidmManagerStates_e currentState;

	DEBUG_ASSERT(event < GIDM_MANAGER_EVENT_NUM_EVENTS);

	// Get current transmission state
	currentState = GidmManagerDb.staDb[staIndex].state;

#if defined (DEBUG_GIDM_MANAGER)
	AAA_GIDM_MANAGER[AAA_GIDM_MANAGER_INDEX].staIndex = staIndex;	
	AAA_GIDM_MANAGER[AAA_GIDM_MANAGER_INDEX].tsf   = GET_TSF_TIMER_LOW();
	AAA_GIDM_MANAGER[AAA_GIDM_MANAGER_INDEX].state = currentState;
	AAA_GIDM_MANAGER[AAA_GIDM_MANAGER_INDEX].event = event;	
	AAA_GIDM_MANAGER_INDEX = (AAA_GIDM_MANAGER_INDEX + 1) % AAA_GIDM_MANAGER_DEBUG_ARRAY_SIZE;
#endif /* DEBUG_GIDM_MANAGER */

	ILOG0_DDD("GidmManager_RunStateMachine: Sta = %d, CurrentState = %d, event = %d", staIndex, currentState, event);

	// Send the event to the state machine		
	GidmManagerSm[currentState].handler[event](parameter);	
}

void GidmManager_ChangeState(StaId staIndex, gidmManagerStates_e state)
{
	ILOG0_DD("GidmManager_ChangeState: staIndex = %d, newState = %d", staIndex, state);

	GidmManagerDb.staDb[staIndex].state = state;
}

void GidmManager_IllegalEvent(void *parameter)
{
	UNUSED_PARAM(parameter);	
	FATAL("GidmManager_IllegalEvent");
}

void GidmManager_StartTx(void *parameter)
{
	uint8 allocationStatus = FALSE;
	RmPdRequestFillParameters_t pdRequestFillParameters;
	StaId stationIndex = 0; 

	stationIndex = *(StaId *)parameter;

	// Try to get a PD for the transmission
	memset(&pdRequestFillParameters, 0, sizeof(RmPdRequestFillParameters_t));	
	pdRequestFillParameters.returnMsgType = GIDM_MANAGER_PD_ALLOCATED;
	pdRequestFillParameters.returnTaskId = TASK_GROUP_MANAGER;
	pdRequestFillParameters.context = stationIndex;
	allocationStatus = ResourceManager_GetDescriptorRequest(DESC_POOL_MANAGEMENT_FROM_FW, &pdRequestFillParameters);
	
	if (allocationStatus == TRUE) // PD was successfully allocated
	{
		// Build the frame and send it
		gidmManager_SendGidmFrame(stationIndex, pdRequestFillParameters.packetDescriptor);

		// Change state
		GidmManager_ChangeState(stationIndex, GIDM_MANAGER_STATE_WAIT_TX_CFM);
	}
	else // there was no available PD
	{
		// Save the request ID
		GidmManagerDb.staDb[stationIndex].requestId = pdRequestFillParameters.requestId;

		// Change state
		GidmManager_ChangeState(stationIndex, GIDM_MANAGER_STATE_WAIT_FOR_PD);
	}
}


void GidmManager_StopInWaitTxCfm(void *parameter)
{
	GidmManagerStopTxReq_t *stopTxParams = (GidmManagerStopTxReq_t *)parameter;
	StaId 					staIndex = stopTxParams->staIndex;
	gidmStopReason_e		stopReason = stopTxParams->stopReason;
	GidmStaManagerDb_t*		gidmStaDbEntry = &GidmManagerDb.staDb[staIndex];
	RequesterLockParams_t 	lockRequesterParams;

	// Set flag to remember there stop was requested.
	gidmStaDbEntry->stopRequested = TRUE;

	if (stopReason == GIDM_STOP_REASON_TIMEOUT)
	{
		// Lock the queue so that we can try to find the PD and release it
		memset(&lockRequesterParams, 0, sizeof(RequesterLockParams_t));
		lockRequesterParams.callerContext = NULL;
		lockRequesterParams.returnMsg = 	GIDM_MANAGER_LOCK_CONFIRMED;
		lockRequesterParams.returnTask = 	TASK_GROUP_MANAGER;	
		Locker_LockSingleQueue(HW_TX_Q_TYPE_STA_TID, staIndex, IEEE802_1D_VO_2, &lockRequesterParams);

		// Change state
		GidmManager_ChangeState(staIndex, GIDM_MANAGER_STATE_WAIT_FOR_LOCK);
	}
	else
	{
		// In case of REMOVE STA we do nothing. We wait for PD to be confirmed.
		// TX Manager flushes the queues for us.
	}

}

void GidmManager_StopInWaitForPd(void *parameter)
{
	GidmManagerStopTxReq_t *stopTxParams = (GidmManagerStopTxReq_t *)parameter;
	StaId 					staIndex = stopTxParams->staIndex;
	GidmStaManagerDb_t*		gidmStaDbEntry = &GidmManagerDb.staDb[staIndex];
	uint8					pdAllocationRequestCancelled = FALSE;
	GroupManagerNotificationParameters_t notificationParameters;

    memset(&notificationParameters, 0, sizeof(GroupManagerNotificationParameters_t));
	

	// Set flag to remember there stop was requested.
	gidmStaDbEntry->stopRequested = TRUE;

	// Copy Stop reason
	gidmStaDbEntry->stopReason = stopTxParams->stopReason;

	// try to cancel the PD allocation pending request
	pdAllocationRequestCancelled = ResourceManager_RemoveRequest(gidmStaDbEntry->requestId, DESC_POOL_MANAGEMENT_FROM_FW);

	if (pdAllocationRequestCancelled == TRUE)
	{
		// Pd request was successfully canceled.

		gidmStaDbEntry->stopRequested = FALSE;

		// Change state to idle
		GidmManager_ChangeState(staIndex, GIDM_MANAGER_STATE_IDLE);

		/* Send confirmation to the group manager */
		notificationParameters.stationIndex = staIndex;
		notificationParameters.notificationStatus = GROUP_MANAGER_NOTIFICATION_NOT_TRANSMITTED;
		GroupManager_GidmNotificationConfirmation(&notificationParameters);
	}
	else
	{
		// PD request couldn't be canceled. It means we will get the PD soon.
		// Change state and wait for it.
		GidmManager_ChangeState(staIndex, GIDM_MANAGER_STATE_WAIT_FOR_PD_PENDING_STOP);
	}
}


void GidmManager_TxCfm(void *parameter)
{
	uint32 					staIndex = ((TxPd_t*)parameter)->txQStaId;
	GidmStaManagerDb_t*		gidmStaDbEntry = &GidmManagerDb.staDb[staIndex];
	TxPd_t*		pPd = (TxPd_t*)CONVERT_OFFSET_TO_PD(gidmStaDbEntry->pdOffset);

	GroupManagerNotificationParameters_t notificationParameters;

	memset(&notificationParameters, 0, sizeof(GroupManagerNotificationParameters_t));

    if(PD_STATUS_ACK_RECEIVED == pPd->status)
    {
		notificationParameters.notificationStatus = GROUP_MANAGER_NOTIFICATION_SUCCEEDED;
    }
	else if(!pPd->retransmissionIndication)
	{
		notificationParameters.notificationStatus = GROUP_MANAGER_NOTIFICATION_NOT_TRANSMITTED;
	}
	else
	{
		notificationParameters.notificationStatus = GROUP_MANAGER_NOTIFICATION_FAILED;
	}

	notificationParameters.stationIndex = staIndex;
	
	DEBUG_ASSERT((uint32)pPd == (uint32)parameter);

	// Release the PD
	ResourceManager_ReleaseDescriptor(pPd, DESC_POOL_MANAGEMENT_FROM_FW);
	
	gidmStaDbEntry->stopRequested = FALSE;

	// Change state
	GidmManager_ChangeState(staIndex, GIDM_MANAGER_STATE_IDLE);

	/* Send confirmation to the group manager */
	/* NOTE: make sure that below function is called only after status of the station is changed top IDLE*/
	GroupManager_GidmNotificationConfirmation(&notificationParameters);
}

void GidmManager_TxCfmInWaitForLock(void *parameter)
{
	uint32 					staIndex = ((TxPd_t*)parameter)->txQStaId;
	GidmStaManagerDb_t*		gidmStaDbEntry = &GidmManagerDb.staDb[staIndex];
	TxPd_t*		pPd = (TxPd_t*)CONVERT_OFFSET_TO_PD(gidmStaDbEntry->pdOffset);

	DEBUG_ASSERT((uint32)pPd == (uint32)parameter);

	// PD is confirmed but STOP request was issued (this is why we are pending for lock).
	// Just release the PD and wait for lock confirmation
	ASSERT(gidmStaDbEntry->stopRequested == TRUE);
	ResourceManager_ReleaseDescriptor(pPd, DESC_POOL_MANAGEMENT_FROM_FW);

	// Change state
	GidmManager_ChangeState(staIndex, GIDM_MANAGER_STATE_WAIT_FOR_LOCK_PD_CONFIRMED);
}

void GidmManager_PdAllocatedInWaitforPd(void *parameter)
{
	TxPd_t*	pPd = (TxPd_t*)parameter;
	StaId					staIndex = pPd->txQStaId; // We stored SID on the PD when it was allocated
	GidmStaManagerDb_t*		gidmStaDbEntry = &GidmManagerDb.staDb[staIndex];

	ResourceManager_ReleaseRequest(gidmStaDbEntry->requestId, DESC_POOL_MANAGEMENT_FROM_FW);

	// Build the frame and send it (payload pointer and vap index are already stored in the staDb)
	gidmManager_SendGidmFrame(staIndex, pPd);
	
	// Change state
	GidmManager_ChangeState(staIndex, GIDM_MANAGER_STATE_WAIT_TX_CFM);
}

void GidmManager_PdAllocatedInPendingStop(void *parameter)
{
	TxPd_t*		pPd = (TxPd_t*)parameter;
	StaId					staIndex = pPd->txQStaId; // We stored SID on the PD when it was allocated
	GidmStaManagerDb_t*		gidmStaDbEntry = &GidmManagerDb.staDb[staIndex];
	GroupManagerNotificationParameters_t notificationParameters;

    memset(&notificationParameters, 0, sizeof(GroupManagerNotificationParameters_t));

	// Stop is pending so we don't need to transmit. Just release the PD.
	ASSERT(gidmStaDbEntry->stopRequested == TRUE);
	ResourceManager_ReleaseDescriptor(pPd, DESC_POOL_MANAGEMENT_FROM_FW);
	ResourceManager_ReleaseRequest(gidmStaDbEntry->requestId, DESC_POOL_MANAGEMENT_FROM_FW);

	gidmStaDbEntry->stopRequested = FALSE;

	// Change state
	GidmManager_ChangeState(staIndex, GIDM_MANAGER_STATE_IDLE);	

	/* Send confirmation to the group manager */
	notificationParameters.stationIndex = staIndex;
	notificationParameters.notificationStatus = GROUP_MANAGER_NOTIFICATION_NOT_TRANSMITTED;
	GroupManager_GidmNotificationConfirmation(&notificationParameters);
}


void GidmManager_LockCfmInWaitForLock(void *parameter)
{
	LockReqCb_t* 			lockReqCb = (LockReqCb_t*)parameter;
	uint8 					staIndex = lockReqCb->stationOrVapNum;
	TxSelectorLockStatus_e 	lockStatus = lockReqCb->lockStatus;
	GidmStaManagerDb_t*		gidmStaDbEntry = &GidmManagerDb.staDb[staIndex];
	TxPd_t*		pGidmPd = (TxPd_t*)CONVERT_OFFSET_TO_PD(gidmStaDbEntry->pdOffset);
	bool					pdFound = FALSE;
	GroupManagerNotificationParameters_t notificationParameters;

    memset(&notificationParameters, 0, sizeof(GroupManagerNotificationParameters_t));

	//Locking the queue may fail, unlock only on success. In case of failure we will treat the pd as not found
	if (lockStatus == TX_SELECTOR_LOCK_STATUS_LOCKED)
	{

		// Queue is locked now. Try to find the PD.
		pdFound = GidmManager_RemovePdFromQueue(staIndex, pGidmPd);

		// Unlock the queue
		Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID, staIndex, (0x1 << IEEE802_1D_VO_2));
	}

	//Check if PD was found
	if (pdFound == TRUE)
	{
		// Release the PD we found
		ResourceManager_ReleaseDescriptor(pGidmPd, DESC_POOL_MANAGEMENT_FROM_FW);

		gidmStaDbEntry->stopRequested = FALSE;

		// Change state to idle
		GidmManager_ChangeState(staIndex, GIDM_MANAGER_STATE_IDLE);

		/* Send confirmation to the group manager */
		notificationParameters.stationIndex = staIndex;
		notificationParameters.notificationStatus = GROUP_MANAGER_NOTIFICATION_NOT_TRANSMITTED;
		GroupManager_GidmNotificationConfirmation(&notificationParameters);
	}
	else
	{
		// PD was not found. It means it is on the way back.
		// Change state and wait for it
		GidmManager_ChangeState(staIndex, GIDM_MANAGER_STATE_WAIT_TX_CFM);		
	}
}

void GidmManager_LockCfmInWaitLockPdConfirmed(void *parameter)
{
	LockReqCb_t* 			lockReqCb = (LockReqCb_t*)parameter;
	uint8 					staIndex = lockReqCb->stationOrVapNum;
	GidmStaManagerDb_t*		gidmStaDbEntry = &GidmManagerDb.staDb[staIndex];
	TxSelectorLockStatus_e 	lockStatus = lockReqCb->lockStatus;
	GroupManagerNotificationParameters_t notificationParameters;

    memset(&notificationParameters, 0, sizeof(GroupManagerNotificationParameters_t));

	//Locking the queue may fail, unlock only on success. In case of failure we will treat the pd as not found
	if (lockStatus == TX_SELECTOR_LOCK_STATUS_LOCKED)
	{
		// Unlock the queue since the PD was already confirmed before the lock occurred.
		Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID, staIndex, (0x1 << IEEE802_1D_VO_2));
	}

	gidmStaDbEntry->stopRequested = FALSE;

	// Change state to idle
	GidmManager_ChangeState(staIndex, GIDM_MANAGER_STATE_IDLE);
	
	/* Send confirmation to the group manager */
	notificationParameters.stationIndex = staIndex;
	notificationParameters.notificationStatus = GROUP_MANAGER_NOTIFICATION_NOT_TRANSMITTED;
	GroupManager_GidmNotificationConfirmation(&notificationParameters);
}


bool GidmManager_RemovePdFromQueue(StaId staIndex, TxPd_t* pGidmPd)
{
	HwQueueManagerRequestParams_t 	hwQueueManagerRequestParams;

	memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	hwQueueManagerRequestParams.primaryAddr = staIndex;
	hwQueueManagerRequestParams.secondaryAddr = IEEE802_1D_VO_2;
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
	hwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_STA_TID;
	
	return HwQManager_RemovePdFromQueue(&hwQueueManagerRequestParams, pGidmPd);
}



/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

void gidmManager_SendGidmFrame(StaId stationIndex, TxPd_t* pPd)
{
	GidmStaManagerDb_t*         gidmStaDbEntry = &GidmManagerDb.staDb[stationIndex];	
    GroupIdManagementPayload_t* gidmFramePayload = NULL;
	uint8 vapIndex = 0;

	StaDb_GetVapId(stationIndex, &vapIndex);

	// Save the PD offset
	gidmStaDbEntry->pdOffset = CONVERT_PD_TO_OFFSET(pPd);

	// Fill PD fields
	
	pPd->txQStaId= stationIndex;
#ifdef ENET_INC_ARCH_WAVE600 
	pPd->txQTid = MANAGEMENT_TID;
	pPd->mgmtFrameSubtype = MGMT_FRAME_SUBTYPE_ACTION;
#else
	pPd->txQTid = IEEE802_1D_VO_2;
	pPd->mgmtSubtype = MGMT_FRAME_SUBTYPE_ACTION;
#endif
	pPd->mgmtActionCode = VHT_CATEGORY;
	pPd->mgmtActionValue = VHT_CATEGORY_GROUP_ID_MANAGEMENT;
	pPd->pdType = PD_TYPE_MANAGEMENT_UNENC;
	pPd->txQVapId = vapIndex;
	pPd->txQGroupId = HW_TX_Q_TYPE_STA_TID;
	pPd->mcUnicast = UNICAST;
	pPd->dataLength = sizeof(GroupIdManagementPayload_t) + frame_sizeOfNewManagementFrameHeader(FALSE);

	// Build GIDM frame payload
	gidmFramePayload = (GroupIdManagementPayload_t *)frame_getPayloadPointerForNewManagementFrame((MANAGEMENT_BASIC_FRAME_HEADER *)pPd->packetPointer, FALSE);
	
	/* Use group manager function in order to fill the payload */
	GroupManager_FillGidmPacketPayload(stationIndex, gidmFramePayload);

	// Put frame on STA queue
	TxPacketsClassifier_SendManagementPacketFromFw(pPd);
}


void GidmManagerStartTxReq(StaId *stationIndex)
{
	// Run the state machine
	GidmManager_RunStateMachine(GIDM_MANAGER_EVENT_START_TX_REQ, *stationIndex, stationIndex);	
}

void GidmManagerStopTxReq(GidmManagerStopTxReq_t *stopTxParams)
{
	// Run the state machine
	GidmManager_RunStateMachine(GIDM_MANAGER_EVENT_STOP_TX_REQ, stopTxParams->staIndex, stopTxParams);	
}

void gidmManagerTxCfm(K_MSG* gidmManagerMessage)
{
	GidmManagerPacketConfirmed_t*	pPacketCfmMsg = (GidmManagerPacketConfirmed_t *)pK_MSG_DATA(gidmManagerMessage);
	TxPd_t*							packetDescriptor = pPacketCfmMsg->pd;
	StaId							staIndex = packetDescriptor->txQStaId;

	// Run the state machine
	GidmManager_RunStateMachine(GIDM_MANAGER_EVENT_TX_CFM, staIndex, packetDescriptor);	
}

void gidmManagerPdAllocated(K_MSG* gidmManagerMessage)
{
	RmPdFreeDescResponse_t *pPdAllocatedMsg = (RmPdFreeDescResponse_t *)pK_MSG_DATA(gidmManagerMessage);
	TxPd_t *packetDescriptor = pPdAllocatedMsg->packetDescriptor;
	uint32 staIndex = (uint32)pPdAllocatedMsg->context; // Sta index should have been provided as "context" to the resource manager on the pd allocation request

	// Save the staIndex on the PD (so that we can get it later)
	packetDescriptor->txQStaId = staIndex;

	// Run the state machine
	GidmManager_RunStateMachine(GIDM_MANAGER_EVENT_PD_ALLOCATED, staIndex, packetDescriptor);	
}

void gidmManagerStaLockCfm(K_MSG* gidmManagerMessage)
{
	LockReqCb_t* 	lockReqCb = (LockReqCb_t *)pK_MSG_DATA(gidmManagerMessage);
	uint8 			staIndex = lockReqCb->stationOrVapNum;

	// Run the state machine
	GidmManager_RunStateMachine(GIDM_MANAGER_EVENT_LOCK_CONFIRMED, staIndex, lockReqCb);	
}

#if defined (DEBUG_GIDM_MANAGER)
GroupIdManagementPayload_t AAA_GidmFramePayload = {0};
#endif // (DEBUG_GIDM_MANAGER)

void gidmManagerHandleRxActionFrame(GroupIdManagementPayload_t* gidmFramePayload, uint8 vapIndex)
{
	uint16 	groupIndex;
	uint16 	userPosition = 0;
	bool 	membershipStatus;
#ifdef MU_MIMO_STATIC_GROUP_SEND_AID_TO_STATION_ENABLED
	uint16 aid;
#endif //MU_MIMO_STATIC_GROUP_SEND_AID_TO_STATION_ENABLED

#if defined (DEBUG_GIDM_MANAGER)
	memcpy(&AAA_GidmFramePayload, gidmFramePayload, sizeof(GroupIdManagementPayload_t));
#endif // (DEBUG_GIDM_MANAGER)

	for (groupIndex = 1 ; groupIndex < 64 ; groupIndex++)
	{
		// Check if we are members of this group or not.
		membershipStatus = GET_GROUP_MEMBERSHIP_STATUS(&(gidmFramePayload->MemberShipStatus.MembershipStatusBitmap[0]), groupIndex);
        
		if (membershipStatus == TRUE)
		{
			// Extract the user position in this group. If we are not members of the group, the user position is don't care.
			userPosition = GET_USER_POSITION_FOR_GROUP(gidmFramePayload->UserPosition.UserPositionBitmap, groupIndex);
        	// Update the PHY
	    	PhyDrv_SetMuGroupUsp(vapIndex, groupIndex, userPosition, membershipStatus);
		}
	}
	
#ifdef MU_MIMO_STATIC_GROUP_SEND_AID_TO_STATION_ENABLED
	//read from last byte of user position array - which is never used - the aid of the wds station
	aid = gidmFramePayload->UserPosition.UserPositionBitmap[sizeof(UserPositionArray_t) - 1];
	VapDB_SetAid(vapIndex, aid);
#endif //MU_MIMO_STATIC_GROUP_SEND_AID_TO_STATION_ENABLED

	SERIAL_TRACE("MU Static Map updated",0,0,0);
	ILOG0_V("[gidmManagerHandleRxActionFrame], MU Static Map updated");
}


/**********************************************************************************
gidmManager_Init

Description:
------------
	Initialize Group ID Management Action Frame Manager

Input: 
-----	
	
Output:
-------

Returns:
--------
	 
**********************************************************************************/
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif

void gidmManager_Init(void)
{
	StaId staIndex;

	memset(&GidmManagerDb, 0, sizeof(GidmManagerDb_t));

	for (staIndex = 0 ; staIndex < HW_NUM_OF_STATIONS ; staIndex++)
	{
		GidmManagerDb.staDb[staIndex].pdOffset = NEXT_PD_NULL;
	}
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif



