/***************************************************************
 File:		  GroupDatabase.c
 Module:	  	  Group Database
 Purpose: 	 
 Description:     This module is the implementation of the Group Database module. Calls that access group database parameters 
 			  should be made through the api function located in this file. Direct access (Not through the api) to the
 			  station database parameters is not permitted.
***************************************************************/

//---------------------------------------------------------------------------------
//						Includes						
//---------------------------------------------------------------------------------
#include "GroupDatabase_Api.h"
#include "GroupDatabase_Debug.h" 
#include "ShramGroupDatabase.h"
#include "ShramStationDatabase.h"
#include "GroupDb.h"
#include "stringLibApi.h"
#include "Pac_Api.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"
#include "PacketDescriptor.h"
#include "TxSelector_Api.h"
#include "HwBeamforming_Api.h"
#include "linkAdaptation_api.h"
#include "Tcr_Descriptors.h"
#include "Locker_Api.h"
#include "AggregationBuilder_Api.h"
 





//---------------------------------------------------------------------------------
//						Defines						
//---------------------------------------------------------------------------------

#define LOG_LOCAL_GID   GLOBAL_GID_GROUP_DB
#define LOG_LOCAL_FID 0



//---------------------------------------------------------------------------------
//						Macros						
//---------------------------------------------------------------------------------


//---------------------------------------------------------------------------------
//						Data Type Definition					
//---------------------------------------------------------------------------------
	IEEE_ADDR	macAddress;   //should be moved to another location


//---------------------------------------------------------------------------------
//						Globals					
//---------------------------------------------------------------------------------

GroupDbObject_t	GroupDatabaseObj;
uint8 GroupDb_TxopEnable = TRUE;
uint8 GroupDb_MuNumOfMpdusInAmpdu = MU_NUM_OF_MPDUS_IN_A_MPDU;


//---------------------------------------------------------------------------------
//						Private Function Declaration 			
//---------------------------------------------------------------------------------

RetVal_e GroupDb_ResetGroupEntry(uint8 groupId);



//---------------------------------------------------------------------------------
//						VAP Function implementation	 - move to VAP DB module			
//--------------------------------------------------------------------------------





/**********************************************************************************
GroupDb_FillGroupParams 

Description:
------------
Fill static configuration of group params.
ALL values are set based on requirements described in SAS Final document, 
Section "MU Group Handler Module": under "New group generation" section

Input:
-----
uint8 groupId - Group index

Output:
-------

Returns:
--------
	Status_e  
	
**********************************************************************************/
void GroupDb_FillGroupParams(uint8 groupId, uint8 numOfStations, uint8* staIds, uint8 *nssPerUsp, int8 mcsPerUsp[][MAX_STATION_IN_GROUP])
{
	MuGrpDb_t* pGroupDbHwEntry = NULL;
	Tcr1_t* pTcr1 = NULL;
	Tcr3_t* pTcr3 = NULL;

	uint8 i=0;
	bool groupDynamicBwEnable = TRUE;

	uint8 actualGroupId = 0;

	/* Group id 0 is reserved : groups are 1- 16 */
	actualGroupId = groupId + 1;

	ILOG2_D("GroupDb_FillGroupParams: groupIndex is %x", groupId);
	ILOG2_DDDD("GroupDb_FillGroupParams: SIDs in group (%d, %d, %d, %d)", staIds[0], staIds[1], staIds[2], staIds[3]);
	pGroupDbHwEntry =  &(GroupDatabaseObj.pGroupDbHwEntries[groupId]);

	

	/* INITIALIZE HW DB */
	/* DW0 */

	/* INITIALIZE HW DB */
	pGroupDbHwEntry->groupId = actualGroupId; /* real group id+1, group id starts from value 1*/
	/* The following fields should be zero- SAS_
Final.doc in REQ 4943 - under "The new MU group generation (default template values) */
	pGroupDbHwEntry->ctsToSelfTxMethod = CTS_TO_SELF_TX_METHOD_NO_CTS2SELF; 
	pGroupDbHwEntry->multipleMuDataTransmission = GroupDb_TxopEnable; /* allow TXOP or not - BF sequence is done only once in the beginning of TXOP and data is sent multiple times*/
	pGroupDbHwEntry->multipleMuDataTransmissionType = GROUP_DB_DATA_TX_TYPE_TRANSMIT_FULL_BAR_DATA_SEQ; // only this mode is supported
	pGroupDbHwEntry->txPostpone = TX_POSTPONE_CTS2SELF; /*if required more time for processing of BF reports send  CTS2Self */
	for (i=0; i< numOfStations; i ++)
	{
		//Based on SAS_Final.doc [REQ 4947] (section 9.4.4.2.5.1.2):  group Dynamic BW enable setting to 1, if Dynamic BW enable is set to 1 in SU STA DB (DW0[22]) for all
		// group memberrs. Otherwise  setting to 0. 
		groupDynamicBwEnable &= StaDbHwEntries[staIds[i]].common.dynamicBwEnable;
		
	}
	pGroupDbHwEntry->dynamicBwEnable = groupDynamicBwEnable;
		
	/* DW1 */
	pGroupDbHwEntry->maxPsduLengthLimit = VHT_MAX_A_MPDU_SIZE; // based on SAS "The maximum valid value for "MU group Current PSDU max length limit"
	pGroupDbHwEntry->bfReportResponseFormat = BF_RE_FORMAT_IMMEDIATE_REPORT_EXPECTED;  // changed from value 1 , based on SAS specification section "New group generation"
	// Based on SAS_final.doc: 'The USP0/1/2/3 Immediate BA ind setting shall be according to BAR Ind value in the corresponding 
	// SU STA DB (DW1[24]) of each USP.'
	if(staIds[0]!=INVALID_STA_INDEX)
	{
		pGroupDbHwEntry->usp0UseImmediateBa = TRUE;//StaDbHwEntries[staIds[0]].barInd; 
	}
	if(staIds[1]!=INVALID_STA_INDEX)
	{
		pGroupDbHwEntry->usp1UseImmediateBa = TRUE;//StaDbHwEntries[staIds[1]].barInd; 
	}
	if(staIds[2]!=INVALID_STA_INDEX)
	{
		pGroupDbHwEntry->usp2UseImmediateBa = TRUE;//StaDbHwEntries[staIds[2]].barInd; 
	}
	if(staIds[3]!=INVALID_STA_INDEX)
	{
		pGroupDbHwEntry->usp3UseImmediateBa = TRUE; //StaDbHwEntries[staIds[3]].barInd; 
	}
	

	/* DW2 */
	pGroupDbHwEntry->groupTxopDurationLimit = 0x7FFF; // max field size in uSec 
	pGroupDbHwEntry->maxPsduTransmissionTimeLimit = (MAX_PSDU_TX_TIME_LIMIT >> 1); //max time in 2 uSec resolution 

	/* DW3 */
	pGroupDbHwEntry->usp0AMpduLimit = GroupDb_MuNumOfMpdusInAmpdu; 
	pGroupDbHwEntry->usp0MaxMpduLengthLimit = MAX_VHT_MPDU_IN_A_MPDU_LENGTH_TX; //bytes 
	pGroupDbHwEntry->usp0PpduTxMode = GROUP_DB_PPDU_TX_MODE_TWO_LEVEL_AGGR;
	
	/* DW4*/
	pGroupDbHwEntry->usp1AMpduLimit = GroupDb_MuNumOfMpdusInAmpdu;
	pGroupDbHwEntry->usp1MaxMpduLengthLimit = MAX_VHT_MPDU_IN_A_MPDU_LENGTH_TX; //bytes
	pGroupDbHwEntry->usp1PpduTxMode = GROUP_DB_PPDU_TX_MODE_TWO_LEVEL_AGGR; // init

	/* DW5 */
	pGroupDbHwEntry->usp2AMpduLimit = GroupDb_MuNumOfMpdusInAmpdu;
	pGroupDbHwEntry->usp2MaxMpduLengthLimit = MAX_VHT_MPDU_IN_A_MPDU_LENGTH_TX; //bytes	
	pGroupDbHwEntry->usp2PpduTxMode = GROUP_DB_PPDU_TX_MODE_TWO_LEVEL_AGGR;

	/* DW6 */
	pGroupDbHwEntry->usp3AMpduLimit = GroupDb_MuNumOfMpdusInAmpdu;
	pGroupDbHwEntry->usp3MaxMpduLengthLimit = MAX_VHT_MPDU_IN_A_MPDU_LENGTH_TX; //bytes
	pGroupDbHwEntry->usp3PpduTxMode = GROUP_DB_PPDU_TX_MODE_TWO_LEVEL_AGGR;


	
	
	/*****************************/
	/*			TCRs		*/
	/*****************************/

	/* DW15  TCR1[23:0] Data USP0 */
	pTcr1 = (Tcr1_t*)&(pGroupDbHwEntry->word15);
	pTcr1->partialAid = 0; //fixed value according to SAS
	pTcr1->txopPsNotAllowed = FALSE; //according to sata - Set to 0 only  if 0 in STA DB of all simultaneous TX users.
	pTcr1->groupId = actualGroupId;
	pTcr1->ndpTrainingMode = FALSE; //not training
	pTcr1->muTraining = FALSE; //not training
	pTcr1->muPhyNdp = FALSE; //not phy ndp

	/* DW16  TCR1[23:0] Data USP1 */
	pTcr1 = (Tcr1_t*)&(pGroupDbHwEntry->word16);
	pTcr1->partialAid = 0; //fixed value according to SAS
	pTcr1->txopPsNotAllowed = FALSE; //according to sata - Set to 0 only  if 0 in STA DB of all simultaneous TX users.
	pTcr1->groupId = actualGroupId;
	pTcr1->ndpTrainingMode = FALSE; //not training
	pTcr1->muTraining = FALSE; //not training
	pTcr1->muPhyNdp = FALSE; //not phy ndp



	/* DW17  TCR1[23:0] Data USP2 */
	pTcr1 = (Tcr1_t*)&(pGroupDbHwEntry->word17);
	pTcr1->partialAid = 0; //fixed value according to SAS
	pTcr1->txopPsNotAllowed = FALSE; //according to sata - Set to 0 only  if 0 in STA DB of all simultaneous TX users.
	pTcr1->groupId = actualGroupId;
	pTcr1->ndpTrainingMode = FALSE; //not training
	pTcr1->muTraining = FALSE; //not training
	pTcr1->muPhyNdp = FALSE; //not phy ndp



	/* DW18  TCR1[23:0] Data USP3 */
	pTcr1 = (Tcr1_t*)&(pGroupDbHwEntry->word18);
	pTcr1->partialAid = 0; //fixed value according to SAS
	pTcr1->txopPsNotAllowed = FALSE; //according to sata - Set to 0 only  if 0 in STA DB of all simultaneous TX users.
	pTcr1->groupId = actualGroupId;
	pTcr1->ndpTrainingMode = FALSE; //not training
	pTcr1->muTraining = FALSE; //not training
	pTcr1->muPhyNdp = FALSE; //not phy ndp

	
	/* DW19 - TCR3 fixed values */
	pTcr3 = (Tcr3_t*)&(pGroupDbHwEntry->word19);
	pTcr3->aggregate = DISABLED; // N/A for MU 
	pTcr3->notSounding = ENABLED; // specified in SAS to deafult value 1 

	/* DW20, 21, 22 - TCR3 rates */
	
	pGroupDbHwEntry->usp0Tcr3Rate20Mhz = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[0][0],nssPerUsp[0], NO_DCM);
	pGroupDbHwEntry->usp0Tcr3Rate40Mhz = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[1][0],nssPerUsp[0], NO_DCM);
	pGroupDbHwEntry->usp0Tcr3Rate80Mhz = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[2][0],nssPerUsp[0], NO_DCM);
	pGroupDbHwEntry->usp0AltRateReasonFastProbing = 0;
	pGroupDbHwEntry->usp0AltRateReasonSlowProbing = 0;

	pGroupDbHwEntry->usp1Tcr3Rate20Mhz = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[0][1],nssPerUsp[1], NO_DCM);
	pGroupDbHwEntry->usp1Tcr3Rate40Mhz = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[1][1],nssPerUsp[1], NO_DCM);
	pGroupDbHwEntry->usp1Tcr3Rate80Mhz = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[2][1],nssPerUsp[1], NO_DCM);
	pGroupDbHwEntry->usp1AltRateReasonFastProbing = 0;
	pGroupDbHwEntry->usp1AltRateReasonSlowProbing = 0;

	pGroupDbHwEntry->usp2Tcr3Rate20Mhz = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[0][2],nssPerUsp[2], NO_DCM);
	pGroupDbHwEntry->usp2Tcr3Rate40Mhz = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[1][2],nssPerUsp[2], NO_DCM);
	pGroupDbHwEntry->usp2Tcr3Rate80Mhz = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[2][2],nssPerUsp[2], NO_DCM);
	pGroupDbHwEntry->usp2AltRateReasonFastProbing = 0;
	pGroupDbHwEntry->usp2AltRateReasonSlowProbing = 0;

	pGroupDbHwEntry->usp3Tcr3Rate20Mhz = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[0][3],nssPerUsp[3], NO_DCM);
	pGroupDbHwEntry->usp3Tcr3Rate40Mhz = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[1][3],nssPerUsp[3], NO_DCM);
	pGroupDbHwEntry->usp3Tcr3Rate80Mhz = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[2][3],nssPerUsp[3], NO_DCM);
	pGroupDbHwEntry->usp3AltRateReasonFastProbing = 0;
	pGroupDbHwEntry->usp3AltRateReasonSlowProbing = 0;

	pGroupDbHwEntry->usp0WpTcr3Rate20Mhz = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[0][0],nssPerUsp[0], NO_DCM);
	pGroupDbHwEntry->usp0WpTcr3Rate40Mhz = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[1][0],nssPerUsp[0], NO_DCM);
	pGroupDbHwEntry->usp0WpTcr3Rate80Mhz = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[2][0],nssPerUsp[0], NO_DCM);
                                                                                                 
	pGroupDbHwEntry->usp1WpTcr3Rate20Mhz = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[0][1],nssPerUsp[1], NO_DCM);
	pGroupDbHwEntry->usp1WpTcr3Rate40Mhz = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[1][1],nssPerUsp[1], NO_DCM);
	pGroupDbHwEntry->usp1WpTcr3Rate80Mhz = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[2][1],nssPerUsp[1], NO_DCM);
                                                                                                    
	pGroupDbHwEntry->usp2WpTcr3Rate20Mhz = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[0][2],nssPerUsp[2], NO_DCM);
	pGroupDbHwEntry->usp2WpTcr3Rate40Mhz = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[1][2],nssPerUsp[2], NO_DCM);
	pGroupDbHwEntry->usp2WpTcr3Rate80Mhz = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[2][2],nssPerUsp[2], NO_DCM);
                                                                                                  
	pGroupDbHwEntry->usp3WpTcr3Rate20Mhz = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[0][3],nssPerUsp[3], NO_DCM);
	pGroupDbHwEntry->usp3WpTcr3Rate40Mhz = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[1][3],nssPerUsp[3], NO_DCM);
	pGroupDbHwEntry->usp3WpTcr3Rate80Mhz = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[2][3],nssPerUsp[3], NO_DCM);

}

#if defined (MU_MIMO_STATIC_GROUP_ENABLED)

void GroupDb_UpdateMuGroupTxopMode(uint8 groupId, bool txopMode)
{
	MuGrpDb_t* pGroupDbHwEntry = &(GroupDatabaseObj.pGroupDbHwEntries[groupId]);
	pGroupDbHwEntry->multipleMuDataTransmission = txopMode;
}


#endif // MU_MIMO_STATIC_GROUP_ENABLED

void GroupDb_SetTxPpduModePerUspInGroup(uint8 groupId, uint8 usp, GroupDbPpduTxMode_e ppduTxMode)
{
	MuGrpDb_t* pGroupDbHwEntry = NULL;

	pGroupDbHwEntry =  &(GroupDatabaseObj.pGroupDbHwEntries[groupId]);

	
	switch (usp)
	{
		case 0:
			pGroupDbHwEntry->usp0PpduTxMode = ppduTxMode;
			break;
		case 1:
			pGroupDbHwEntry->usp1PpduTxMode = ppduTxMode;
			break;
		case 2:
			pGroupDbHwEntry->usp2PpduTxMode = ppduTxMode;
			break;
		case 3:
			pGroupDbHwEntry->usp3PpduTxMode = ppduTxMode;
			break;
		default:
			DEBUG_FATAL("GroupDb_SetTxPpduModePerUspInGroup");
			break;
	}
}

/**********************************************************************************
GroupDb_ResetGroupEntry 

Description:
------------
Reset the Group parameters in the Database. Set the VAP entry to FREE state

Input:
-----
uint8 	groupId


Output:
-------
	
Returns:
--------
	RetVal_e  	
**********************************************************************************/
RetVal_e GroupDb_ResetGroupEntry(uint8 groupId)
{	
	MuGrpDb_t*			pGroupDbHwEntry = NULL;

	ILOG2_D("GroupDb_ResetGroupEntry: groupIndex is %X", groupId);

	pGroupDbHwEntry =  &(GroupDatabaseObj.pGroupDbHwEntries[groupId]);
		
	memset(pGroupDbHwEntry, 0 ,sizeof(MuGrpDb_t));

	return RET_VAL_SUCCESS;
}




/**********************************************************************************
GroupDB_Init

Description:
------------
Set group db object to point to the HW database in shared RAM and reset all the group DB entries

Input:
-----


Output:
-------

Returns:
--------
Status code [success/failure]

**********************************************************************************/

RetVal_e GroupDB_Init(void)
{	
	ILOG2_V("GroupDB_Init");
	
	GroupDatabaseObj.pGroupDbHwEntries = GroupDbHwEntries;
	
	memset(&GroupDbHwEntries, 0, sizeof(GroupDbHwEntries));

	return RET_VAL_SUCCESS;
}

