/***************************************************************
 File:		  GroupDatabase.c
 Module:	  	  Group Database
 Purpose: 	 
 Description:     This module is the implementation of the Group Database module. Calls that access group database parameters 
 			  should be made through the api function located in this file. Direct access (Not through the api) to the
 			  station database parameters is not permitted.
***************************************************************/

//---------------------------------------------------------------------------------
//						Includes						
//---------------------------------------------------------------------------------
#include "GroupDatabase_Api.h"
#include "GroupDatabase_Debug.h" 
#include "ShramGroupDatabase.h"
#include "ShramStationDatabase.h"
#include "GroupDb.h"
#include "stringLibApi.h"
#include "Pac_Api.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"
#include "PacketDescriptor.h"
#include "TxSelector_Api.h"
#include "HwBeamforming_Api.h"
#include "linkAdaptation_api.h"
#include "Tcr_Descriptors.h"
#include "Locker_Api.h"





//---------------------------------------------------------------------------------
//						Defines						
//---------------------------------------------------------------------------------

#define LOG_LOCAL_GID   GLOBAL_GID_GROUP_DB
#define LOG_LOCAL_FID 1



//---------------------------------------------------------------------------------
//						Macros						
//---------------------------------------------------------------------------------


//---------------------------------------------------------------------------------
//						Data Type Definition					
//---------------------------------------------------------------------------------
	IEEE_ADDR	macAddress;   //should be moved to another location


//---------------------------------------------------------------------------------
//						Globals					
//---------------------------------------------------------------------------------

GroupDbObject_t	GroupDatabaseObj;
uint8 GroupDb_TxopEnable = TRUE;
uint8 GroupDb_MuNumOfMpdusInAmpdu = MU_NUM_OF_MPDUS_IN_A_MPDU;


//---------------------------------------------------------------------------------
//						Private Function Declaration 			
//---------------------------------------------------------------------------------

RetVal_e GroupDb_ResetGroupEntry(uint8 groupId);



//---------------------------------------------------------------------------------
//						VAP Function implementation	 - move to VAP DB module			
//--------------------------------------------------------------------------------


void GroupDb_SetTxPpduModePerUspInGroup(uint8 groupId, uint8 usp, GroupDbPpduTxMode_e ppduTxMode)
{
	UNUSED_PARAM(groupId);	
	UNUSED_PARAM(usp);	
	UNUSED_PARAM(ppduTxMode);	
	// In gen6, HW takes the Tx PPDU mode from sta DB, no need to copy it to group DB
}



/**********************************************************************************
GroupDb_FillGroupParams 

Description:
------------
Fill static configuration of group params.
ALL values are set based on requirements described in SAS Final document, 
Section "MU Group Handler Module": under "New group generation" section

Input:
-----
uint8 groupId - Group index

Output:
-------

Returns:
--------
	Status_e  
	
**********************************************************************************/
void GroupDb_FillGroupParams(uint8 groupId, uint8 numOfStations, uint16* staIds, uint8 *nssPerUsp, int8 mcsPerUsp[][MAX_STATION_IN_GROUP])
{
	MuGrpDb_t* pGroupDbHwEntry = NULL;

	uint8 i=0;
	bool groupDynamicBwEnable = TRUE;

	uint8 actualGroupId = 0;

	/* Group id 0 is reserved : groups are 1- 16 */
	actualGroupId = groupId + 1;

	ILOG2_D("GroupDb_FillGroupParams: groupIndex is %x", groupId);
	ILOG2_DDDD("GroupDb_FillGroupParams: SIDs in group (%d, %d, %d, %d)", staIds[0], staIds[1], staIds[2], staIds[3]);
	pGroupDbHwEntry =  &(GroupDatabaseObj.pGroupDbHwEntries[groupId]);


	/* INITIALIZE HW DB */
	pGroupDbHwEntry->common.groupId = actualGroupId; /* real group id+1, group id starts from value 1*/
	/* The following fields should be zero- SAS_
Final.doc in REQ 4943 - under "The new MU group generation (default template values) */
	pGroupDbHwEntry->common.ctsToSelfTxMethod = CTS_TO_SELF_TX_METHOD_NO_CTS2SELF; 
	pGroupDbHwEntry->common.multipleMuDataTransmission = GroupDb_TxopEnable; /* allow TXOP or not - BF sequence is done only once in the beginning of TXOP and data is sent multiple times*/
	pGroupDbHwEntry->common.txPostpone = TX_POSTPONE_CTS2SELF; /*if required more time for processing of BF reports send  CTS2Self */
	pGroupDbHwEntry->common.durationUpdateMode = DURATION_UPDATE_MODE_FULL_TXOP; //working with full txop mode
	for (i=0; i< numOfStations; i ++)
	{
		//Based on SAS_Final.doc [REQ 4947] (section 9.4.4.2.5.1.2):  group Dynamic BW enable setting to 1, if Dynamic BW enable is set to 1 in SU STA DB (DW0[22]) for all
		// group members. Otherwise  setting to 0. 
		groupDynamicBwEnable &= (bool) StaDbHwEntries[staIds[i]].common.dynamicBwEnable; // KW_FIX_FW_G explict type casting from bitmapping to bool
		
	}
	pGroupDbHwEntry->common.dynamicBwEnable = groupDynamicBwEnable;
		
	/* DW1 */
	pGroupDbHwEntry->common.maxPsduLengthLimit = VHT_MAX_A_MPDU_SIZE; // based on SAS "The maximum valid value for "MU group Current PSDU max length limit"
	pGroupDbHwEntry->common.bfReportResponseFormat = BF_RE_FORMAT_IMMEDIATE_REPORT_EXPECTED;  // changed from value 1 , based on SAS specification section "New group generation"
	// Based on SAS_final.doc: 'The USP0/1/2/3 Immediate BA ind setting shall be according to BAR Ind value in the corresponding 
	// SU STA DB (DW1[24]) of each USP.'
	if(staIds[0]!=INVALID_STA_INDEX)
	{
		pGroupDbHwEntry->common.usp0UseImmediateBa = TRUE;
	}
	if(staIds[1]!=INVALID_STA_INDEX)
	{
		pGroupDbHwEntry->common.usp1UseImmediateBa = TRUE;
	}
	if(staIds[2]!=INVALID_STA_INDEX)
	{
		pGroupDbHwEntry->common.usp2UseImmediateBa = TRUE;
	}
	if(staIds[3]!=INVALID_STA_INDEX)
	{
		pGroupDbHwEntry->common.usp3UseImmediateBa = TRUE;
	}
	
	pGroupDbHwEntry->common.groupTxopDurationLimit = 0x7FFF; // max field size in uSec 
	
	pGroupDbHwEntry->common.maxPsduTransmissionTimeLimit = (MAX_PSDU_TX_TIME_LIMIT >> 1); //max time in 2 uSec resolution 

	pGroupDbHwEntry->common.usp0AMpduLimit = GroupDb_MuNumOfMpdusInAmpdu;  
	pGroupDbHwEntry->common.usp1AMpduLimit = GroupDb_MuNumOfMpdusInAmpdu;
	pGroupDbHwEntry->common.usp2AMpduLimit = GroupDb_MuNumOfMpdusInAmpdu;
	pGroupDbHwEntry->common.usp3AMpduLimit = GroupDb_MuNumOfMpdusInAmpdu;

	
	/*****************************/
	/*			TCRs		*/
	/*****************************/

	
	pGroupDbHwEntry->common.tcr1TxopPsNotAllowed = FALSE; //according to sata - Set to 0 only  if 0 in STA DB of all simultaneous TX users.
	pGroupDbHwEntry->common.tcr1GroupId = actualGroupId;

	pGroupDbHwEntry->common.tcr120MhzDataNotSounding = ENABLED;
	pGroupDbHwEntry->common.tcr140MhzDataNotSounding = ENABLED;
	pGroupDbHwEntry->common.tcr180MhzDataNotSounding = ENABLED;
	pGroupDbHwEntry->common.tcr1160MhzDataNotSounding = ENABLED;
	
	pGroupDbHwEntry->common.user0Tcr220MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[0][0],nssPerUsp[0], NO_DCM);
	pGroupDbHwEntry->common.user0Tcr240MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[1][0],nssPerUsp[0], NO_DCM);
	pGroupDbHwEntry->common.user0Tcr280MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[2][0],nssPerUsp[0], NO_DCM);
	pGroupDbHwEntry->common.user0Tcr2160MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[3][0],nssPerUsp[0], NO_DCM);
	
	pGroupDbHwEntry->common.user1Tcr220MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[0][1],nssPerUsp[1], NO_DCM);
	pGroupDbHwEntry->common.user1Tcr240MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[1][1],nssPerUsp[1], NO_DCM);
	pGroupDbHwEntry->common.user1Tcr280MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[2][1],nssPerUsp[1], NO_DCM);
	pGroupDbHwEntry->common.user1Tcr2160MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[3][1],nssPerUsp[1], NO_DCM);

	pGroupDbHwEntry->common.user2Tcr220MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[0][2],nssPerUsp[2], NO_DCM);
	pGroupDbHwEntry->common.user2Tcr240MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[1][2],nssPerUsp[2], NO_DCM);
	pGroupDbHwEntry->common.user2Tcr280MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[2][2],nssPerUsp[2], NO_DCM);
	pGroupDbHwEntry->common.user2Tcr2160MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[3][2],nssPerUsp[2], NO_DCM);
	
	pGroupDbHwEntry->common.user3Tcr220MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[0][3],nssPerUsp[3], NO_DCM);
	pGroupDbHwEntry->common.user3Tcr240MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[1][3],nssPerUsp[3], NO_DCM);
	pGroupDbHwEntry->common.user3Tcr280MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[2][3],nssPerUsp[3], NO_DCM);
	pGroupDbHwEntry->common.user3Tcr2160MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[3][3],nssPerUsp[3], NO_DCM);

	pGroupDbHwEntry->wp.user0Tcr220MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[0][0],nssPerUsp[0], NO_DCM);
	pGroupDbHwEntry->wp.user0Tcr240MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[1][0],nssPerUsp[0], NO_DCM);
	pGroupDbHwEntry->wp.user0Tcr280MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[2][0],nssPerUsp[0], NO_DCM);
	pGroupDbHwEntry->wp.user0Tcr2160MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[3][0],nssPerUsp[0], NO_DCM);

	pGroupDbHwEntry->wp.user1Tcr220MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[0][0],nssPerUsp[0], NO_DCM);
	pGroupDbHwEntry->wp.user1Tcr240MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[1][0],nssPerUsp[0], NO_DCM);
	pGroupDbHwEntry->wp.user1Tcr280MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[2][0],nssPerUsp[0], NO_DCM);
	pGroupDbHwEntry->wp.user1Tcr2160MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[3][0],nssPerUsp[0], NO_DCM);
	
	pGroupDbHwEntry->wp.user2Tcr220MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[0][0],nssPerUsp[0], NO_DCM);
	pGroupDbHwEntry->wp.user2Tcr240MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[1][0],nssPerUsp[0], NO_DCM);
	pGroupDbHwEntry->wp.user2Tcr280MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[2][0],nssPerUsp[0], NO_DCM);
	pGroupDbHwEntry->wp.user2Tcr2160MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[3][0],nssPerUsp[0], NO_DCM);

	pGroupDbHwEntry->wp.user3Tcr220MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[0][0],nssPerUsp[0], NO_DCM);
	pGroupDbHwEntry->wp.user3Tcr240MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[1][0],nssPerUsp[0], NO_DCM);
	pGroupDbHwEntry->wp.user3Tcr280MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[2][0],nssPerUsp[0], NO_DCM);
	pGroupDbHwEntry->wp.user3Tcr2160MhzDataPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcsPerUsp[3][0],nssPerUsp[0], NO_DCM);

}

#if defined (MU_MIMO_STATIC_GROUP_ENABLED)

void GroupDb_UpdateMuGroupTxopMode(uint8 groupId, bool txopMode)
{
	MuGrpDb_t* pGroupDbHwEntry = &(GroupDatabaseObj.pGroupDbHwEntries[groupId]);
	pGroupDbHwEntry->common.multipleMuDataTransmission = txopMode;
}


#endif // MU_MIMO_STATIC_GROUP_ENABLED


/**********************************************************************************
GroupDb_ResetGroupEntry 

Description:
------------
Reset the Group parameters in the Database. Set the VAP entry to FREE state

Input:
-----
uint8 	groupId


Output:
-------
	
Returns:
--------
	RetVal_e  	
**********************************************************************************/
RetVal_e GroupDb_ResetGroupEntry(uint8 groupId)
{	
	MuGrpDb_t*			pGroupDbHwEntry = NULL;

	ILOG2_D("GroupDb_ResetGroupEntry: groupIndex is %X", groupId);

	pGroupDbHwEntry =  &(GroupDatabaseObj.pGroupDbHwEntries[groupId]);
		
	memset(pGroupDbHwEntry, 0 ,sizeof(MuGrpDb_t));

	return RET_VAL_SUCCESS;
}




/**********************************************************************************
GroupDB_Init

Description:
------------
Set group db object to point to the HW database in shared RAM and reset all the group DB entries

Input:
-----


Output:
-------

Returns:
--------
Status code [success/failure]

**********************************************************************************/

RetVal_e GroupDB_Init(void)
{	
	ILOG2_V("GroupDB_Init");
	
	GroupDatabaseObj.pGroupDbHwEntries = GroupDbHwEntries;
	
	memset(&GroupDbHwEntries, 0, sizeof(GroupDbHwEntries));

	return RET_VAL_SUCCESS;
}

