/***********************************************************************************
 File:			GroupManager_API.h
 Module:		Group manager 
 Purpose: 		To create and delete multi user groups
 Description:   This file is the API of the Group manager which is responsible of 
 				creating and deleting multi user groups
************************************************************************************/
#ifndef GROUP_MANAGER_API_H_
#define GROUP_MANAGER_API_H_

/*---------------------------------------------------------------------------------
/						includes					
/----------------------------------------------------------------------------------*/
#include "System_GlobalDefinitions.h"
#include "SharedDbTypes.h"
#include "System_Configuration.h"
#include "HwGlobalDefinitions.h"
#include "frame.h"
#include "TxSelector_Api.h"


/*---------------------------------------------------------------------------------
/						defines				
/----------------------------------------------------------------------------------*/
#define GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP 		(4)
#define GROUP_MANAGER_MAX_NUMBER_OF_GROUPS 					(16)	/* Warning: if you modify this value please also modify it at Statistics_Descriptors.h */
#define GROUP_MANAGER_SIZE_OF_STATION_BITMAP_IN_WORDS 		(CONVERT_BIT_INDEX_TO_WORDS(HW_NUM_OF_STATIONS - 1) + 1)
#define GROUP_MANAGER_MAX_NUM_OF_STA_FOR_MU_TRAINING 		(16)


#define GROUP_MANAGER_INVALID_COMBINATION_VALUE 			(0xFFFFFFFF)
#define GROUP_MANAGER_INVALID_RATE_VALUE 					(0xFF)
#define GROUP_MANAGER_NSS_SIZE_SHIFT 						(1)
#define GROUP_MANAGER_NSS_MASK 								(0x3)


#define GROUP_MANAGER_MINIMAL_GROUP_GRADE_THRESHOLD			0


typedef enum
{
	STATISTICS_MU_GROUP_ID_INDEX,			
	STATISTICS_MU_GROUP_STATE_INDEX,		
	STATISTICS_NUM_OF_MU_GROUP_MEMBERS_INDEX,	
	/*This group of MEMBERS stata shall not be saparated, used in groupManager to access by index from STATISTICS_MU_GROUP_MEMBERS_ONE*/
	STATISTICS_MU_GROUP_MEMBERS_ONE,	
	STATISTICS_MU_GROUP_MEMBERS_TWO,
	STATISTICS_MU_GROUP_MEMBERS_TREE,
	STATISTICS_MU_GROUP_MEMBERS_FOUR,
	/**************************************/
	MU_NUM_OF_STATS, /* Warning: if you modify this value please also modify it at Statistics_Descriptors.h */
}GroupManagerStatisticTypes_e;


//#define GROUP_MANAGER_DEBUG
/*---------------------------------------------------------------------------------
/						Macros					
/----------------------------------------------------------------------------------*/

#define GROUP_MANAGER_GET_NSS_PER_USP(nssBitmap, usp) (( nssBitmap >> (usp << GROUP_MANAGER_NSS_SIZE_SHIFT)) & GROUP_MANAGER_NSS_MASK)
			
/*---------------------------------------------------------------------------------
/						Data type definitions					
/----------------------------------------------------------------------------------*/

// the following enum represents stations priority from high to low as Training manager requires
// station can be found only in single priority. station will always get the highest possible priority.
// Active station which participates in probing will be in active state
// station which once performed probing and now performs probing again will be in MID1 state
typedef enum
{
	GROUP_MANAGER_STATION_PRIORITY_HIGH = 0, // active	
	GROUP_MANAGER_STATION_PRIORITY_MID,		 // probing + notifying
	GROUP_MANAGER_STATION_PRIORITY_LOW,		 // untested - found in virtual table but didn't perform training at all / re-tested - passed training at least once
	GROUP_MANAGER_NUMBER_OF_STATION_PRIORITIES,
	GROUP_MANAGER_STATION_PRIORITY_NONE = GROUP_MANAGER_NUMBER_OF_STATION_PRIORITIES,
	GROUP_MANAGER_TOTAL_NUM_OF_PRIORITIES,
	GROUP_MANAGER_MAX_NUMBER_OF_STATION_PRIORITIES = 0xFF,	
} groupManagerStationPriorities_e;

typedef enum
{
	GROUP_MANAGER_NOTIFICATION_NOT_RECEIVED = 0, /*this state is used before GIDM sent notifcation message and after the message was sent but reply not received yet*/ 
	GROUP_MANAGER_NOTIFICATION_FAILED,
	GROUP_MANAGER_NOTIFICATION_SUCCEEDED,
	GROUP_MANAGER_NOTIFICATION_CANCELED, /* canceled by group manager*/
	GROUP_MANAGER_NOTIFICATION_NOT_TRANSMITTED, /* notifcation pushed to queue but wasn't transmitted even once*/
	GROUP_MANAGER_NOTIFICATION_TIMEOUT,
	GROUP_MANAGER_NUM_OF_NOTIFICATION_STATUS,
	GROUP_MANAGER_MAX_NUM_OF_NOTIFICATION_STATUS = 0xFF,	
} GroupManagerNotificationStatus_e;

typedef enum
{	
	GROUP_MANAGER_STATION_FIRST_STATE = 0,
	GROUP_MANAGER_STATION_STATE_IDLE = GROUP_MANAGER_STATION_FIRST_STATE,
	GROUP_MANAGER_STATION_STATE_NOT_ALLOWED,
	GROUP_MANAGER_STATION_STATE_SINGLE_USER,
	GROUP_MANAGER_STATION_STATE_NOTIFY_STATION,
	GROUP_MANAGER_STATION_STATE_PROBING,
	GROUP_MANAGER_STATION_STATE_TEMP_ACTIVE, // station can be in this state only if it was active state and now it temorary acts as primary for new probing group
	GROUP_MANAGER_STATION_STATE_ACTIVE,
	GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_PENDING,
	GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_WAITING_FOR_LINK_ADAPTATION,
	GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION,
	GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_FAILED,
	GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_SUCCEEDED,
	GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_CANCELED_GROUP_REMOVED,
	GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_CANCELED_GROUP_FAILED,
	GROUP_MANAGER_STATION_STATE_HALTING_WAITING_FOR_TRAINING_CONFIRMATION,
	GROUP_MANAGER_STATION_STATE_HALT,
	GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_LINK_ADAPTATION_AND_TRAINING_CONFIRMATION,
	GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_TRAINING_CONFIRMATION,
	GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_LINK_ADAPTATION_CONFIRMATION,
	GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_ALL_GROUP_REMOVAL,
	GROUP_MANAGER_NUM_OF_STATION_STATES,
	GROUP_MANAGER_MAX_NUM_OF_STATION_STATES = 0xFF,	
} GroupManagerStationState_e;

typedef enum
{
	GROUP_MANAGER_PROBING_GROUP_GRADE_PARAM,
	GROUP_MANAGER_MAX_SU_VAL_STATIONS_PARAM,
	GROUP_MANAGER_MIN_STATION_GRADE_IN_GROUP_PARAM,
	GROUP_MANAGER_SU_PERFORMENCE_PARAM,
	GROUP_MANAGER_MAX_PROBING_GROUPS_PARAM,
	GROUP_MANAGER_NUM_OF_CONFIG_PARAMS,
	GROUP_MANAGER_MAX_NUM_OF_CONFIG_PARAMS
} GroupManagerConfigurationParamType_e;

typedef struct _GroupCombinationTableEntry_t
{
	StaId staId;
	StaId groupCombinationSIDs[SPATIAL_STREAM_NUM]; 	//Group combination including the current station
	uint8 groupCombinationPhyRates[SPATIAL_STREAM_NUM]; //MCS for each SID in groupCombinationSIDs
	uint32 groupMark;									//Group mark for current combination
	uint8 singleNstsPhyRate; 							//MCS that could be for the staId in SU for 1 NSS
	uint8 doubleNstsPhyRate; 							//MCS that could be for the staId in SU for 2 NSS
} GroupCombinationTableEntry_t;


typedef struct _GroupCombinationTableDebug_t
{
	GroupCombinationTableEntry_t groupCombinationTable[GROUP_MANAGER_MAX_NUM_OF_STA_FOR_MU_TRAINING]; 

}GroupCombinationTableDebug_t;


typedef struct GroupManagerStationBitmap
{	
	uint32 stationBitmap[GROUP_MANAGER_SIZE_OF_STATION_BITMAP_IN_WORDS];
} GroupManagerStationBitmap_t;

typedef struct GroupManagerTrainingBitmap
{	
	GroupManagerStationBitmap_t stationPriorityBitmap[GROUP_MANAGER_NUMBER_OF_STATION_PRIORITIES];
	GroupManagerStationBitmap_t haltStationsBitmap; /* represents stations that are in halt or in halt process */
} GroupManagerTrainingBitmap_t;

typedef struct _GroupManagerTrainingResults_t
{
	GroupCombinationTableEntry_t *pGroupCombinationTable;
	Bandwidth_e resultsBw;
} GroupManagerTrainingResults_t;

typedef struct _GroupManagerStopTrainingCfm_t
{
	uint32 confirmedStaBitmap[GROUP_MANAGER_SIZE_OF_STATION_BITMAP_IN_WORDS]; //stations which "remove station" was requested for and were a part of current training
} GroupManagerStopTrainingCfm_t;

typedef struct GroupManagerAddGroupConfirmationMessage
{
	uint8 groupIndex;
	uint8 reserved[3];
}GroupManagerAddGroupConfirmationMessage_t;

typedef struct GroupManagerRemoveGroupConfirmationMessage
{
	uint8 groupIndex;
	uint8 reserved[3];
}GroupManagerRemoveGroupConfirmationMessage_t;

typedef struct GroupManagerSingleUserValidationStartConfirmationMessage
{
	StaId stationIndex;
	uint8 reserved[4-SIZE_OF_STA_ID];
}GroupManagerSingleUserValidationStartConfirmationMessage_t;

typedef struct GroupManagerSingleUserValidationEndConfirmationMessage
{
	StaId stationIndex;
	uint8 reserved[4-SIZE_OF_STA_ID];
}GroupManagerSingleUserValidationEndConfirmationMessage_t;

typedef struct GroupManagerGroupsMarksMessage
{
	uint8 groupMarks[GROUP_MANAGER_MAX_NUMBER_OF_GROUPS];
}GroupManagerGroupsMarksMessage_t;

typedef struct GroupManagerSingleUserResultsMessage
{
	uint8 groupMarks[GROUP_MANAGER_MAX_NUMBER_OF_GROUPS];
	
	StaId stationIndex;
	uint8 stationMark;
	uint8 reserved[3-SIZE_OF_STA_ID];
}GroupManagerSingleUserResultsMessage_t;

typedef struct GroupManagerProbingResultsMessage
{
	uint8 groupMarks[GROUP_MANAGER_MAX_NUMBER_OF_GROUPS]; /* contains group marks of all groups*/
	uint8 stationMuMark[GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP]; /*scores of participating members from the beginning of MAC probbing of the group*/
	uint8 stationSuMark[GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP]; /*SU performance value of participating members*/
	uint8 probingGroupIndex;
	uint8 reserved[3];
}GroupManagerProbingResultsMessage_t;

typedef struct GroupManagerStationNssHasChangedMessage
{
	StaId stationIndex;
	uint8 newNss;
	uint8 reserved[3-SIZE_OF_STA_ID];
}GroupManagerStationNssHasChangedMessage_t;

typedef struct GroupManagerGroupHasFailedMessage
{
	uint8 groupIndex;
	uint8 reserved[3];
}GroupManagerGroupHasFailedMessage_t;

typedef struct GroupManagerNotificationParameters
{
	StaId stationIndex;
	GroupManagerNotificationStatus_e notificationStatus; 
	uint8 reserved[3-SIZE_OF_STA_ID];
} GroupManagerNotificationParameters_t;

typedef struct GroupManagerSetMultiUserOperationParameters
{
	uint8 isMultiUserOperationEnabled;
	uint8 reserved[3];
} GroupManagerSetMultiUserOperationParameters_t;

typedef struct GroupManagerFixedRateParams
{
	bool 	fixedRateMuFlag;
} GroupManagerFixedRateParams_t;

/*---------------------------------------------------------------------------------
/						Public functions declarations			
/----------------------------------------------------------------------------------*/
void GroupManager_StationIsNotInTraining(StaId stationIndex); 
void GroupManager_TrainingWasStoppedConfirmation(uint32 *stationBitmap);
void GroupManager_TrainingFailed(void);
void GroupManager_TrainingResultsWereReceived(GroupManagerTrainingResults_t *trainingCombinations);
GroupManagerTrainingBitmap_t* GroupManager_GetTrainingBitmaps(void);
void GroupManager_GidmNotificationConfirmation(GroupManagerNotificationParameters_t *notificationParameters);
void GroupManager_FillGidmPacketPayload(StaId stationIndex, GroupIdManagementPayload_t *gidmPayload);
void GroupManager_PostInit(void);
void GroupManager_Initialize(void);

uint8* GroupManagerGetGroupDbAddress(void);
void GroupManager_UpdateMuGroupTxopMode(bool txopMode);
GroupManagerStationState_e GroupManager_GetStaState(StaId staId);
#ifdef ENET_INC_ARCH_WAVE600D2
uint8 GroupManager_GetStaMbssRxCtrlSupport(StaId staId);
#endif
void GroupManager_UpdateMuConfigParams(GroupManagerConfigurationParamType_e paramType, uint8 paramVal);
void GroupManager_EnableDynamicModeWithStaticMode(void);
#if defined (ENET_INC_ARCH_WAVE600)
uint8 GroupManager_GetVapforGroup(uint8 groupIndex);
#endif
uint8 groupManagerGetCoCTxAntsNum(void);
#ifndef ENET_INC_ARCH_WAVE600
void GroupManager_SetPpduTxModePerStation(StaId stationIndex);
#endif
#if defined (MU_MIMO_STATIC_GROUP_ENABLED)

#define GROUP_MANAGER_STATIC_GROUP_INITIAL_MCS	7
#define GROUP_MANAGER_NSTS_PHY_RATE_DOUBLE 		5
#define GROUP_MANAGER_NSTS_PHY_RATE_SINGLE 		5

typedef struct
{
	StaId staIds[GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP];
	uint8 nssPerUsp[GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP];	
}GroupManager_CreateStaticGroup_t;

void GroupManager_UpdateMuStaticGroupTxMode(TxSelectorGroupTxMode_e txMode);
void GroupManager_UpdateMuStaticGroupInitialMcs(mcs_e mcs);

#endif // MU_MIMO_STATIC_GROUP_ENABLED


#endif /* GROUP_MANAGER_API_H_ */

