/***********************************************************************************
 File:			GroupManager.h
 Module:		Group Manager 
 Purpose: 		To create and delete multi user groups
 Description:	This file contains all definitions and the structures of the group 
 				manager which is responsible of creating and deleting multi user 
 				groups
************************************************************************************/
#ifndef GROUP_MANAGER_H_
#define GROUP_MANAGER_H_
//---------------------------------------------------------------------------------
//						Includes						
//---------------------------------------------------------------------------------
#include "System_GlobalDefinitions.h"
#include "System_Configuration.h"
#include "GroupManager_API.h"
#include "CalendarWheel_Api.h"
#include "OSAL_Kmsg.h"
#include "VapDatabase_Api.h"


//---------------------------------------------------------------------------------
//						Defines						
//---------------------------------------------------------------------------------
#define GROUP_MANAGER_MAX_NUMBER_OF_VIRTUAL_GROUPS (32)
#define GROUP_MANAGER_INVALID_GROUP_INDEX (0xFF)
#define GROUP_MANAGER_CALENDAR_WHEEL_SIZE (32)
#define GROUP_MANAGER_CALENDAR_WHEEL_TIMER_IN_MILLISECONDS (400)
#define GROUP_MANAGER_NUMBER_OF_STATIONS_BITS_IN_WORD (32)
#define GROUP_MANAGER_NUMBER_OF_USER_POSITION_BYTES_IN_USE (5)
#define GROUP_MANAGER_USER_POSITION_INCREASE_SHIFT_SIZE (8)
#define GROUP_MANAGER_MAX_NUMBER_OF_ACTIVE_GROUPS (16)
#define GROUP_MANAGER_ALL_GROUPS_MASK (0xFFFF)


#define GROUP_MANAGER_MAX_HALT_THRESHOLD 					(4)
#define GROUP_MANAGER_MAX_HALT_PERIOD 						(GROUP_MANAGER_CALENDAR_WHEEL_SIZE - 1)
#define GROUP_MANAGER_HALT_PENALTY_FROM_PROBING 			(3)//TBD get the value from system
#define GROUP_MANAGER_HALT_PENALTY_FROM_ACTIVE 				(6)//TBD get the value from system
#define GROUP_MANAGER_SINGLE_USER_VALIDATION_TIMER 			(16) // represents 16 slots, each slot length is 400 millisecs
#define GROUP_MANAGER_SINGLE_USER_VALIDATION_RESULTS_TIMER 	(3)//TBD get the value from system
#define GROUP_MANAGER_PROBING_RESULTS_TIMER 				(12)//TBD get the value from system
#define GROUP_MANAGER_SINGLE_USER_PERFORMENCE_MARGIN 		(10)//TBD get the value from system
#define GROUP_MANAGER_PROBING_GROUP_MARK_THRESHOLD 			(15)//TBD get the value from system
#define GROUP_MANAGER_PROBING_GROUP_MARK_MARGIN				(1)//TBD get the value from system
#define GROUP_MANAGER_SU_AGGREGATION_THRESH					(8)


#define GROUP_MANAGER_PROBING_GROUP_STATION_MARK_THRESHOLD 	(1)//TBD get the value from system
#define GROUP_MANAGER_MAX_NUMBER_OF_SINGLE_USER_VALIDATION 	(0)//Actually disables SU validation until we know when it is worth enabling it
#define GROUP_MANAGER_MAX_NUMBER_OF_PROBING_GROUPS 			(5)
#define GROUP_MANAGER_VIRTUAL_GROUP_VALID_TIME_PERIOD 		((GROUP_MANAGER_CALENDAR_WHEEL_TIMER_IN_MILLISECONDS * 3) * 1000) /* Currently the valid time is 3 training cycles */
#define GROUP_MANAGER_BEST_GROUP_MARK 						(100)
#define GROUP_MANAGER_BEST_STATIC_GROUP_MARK				(GROUP_MANAGER_BEST_GROUP_MARK+1)

#define GROUP_MANAGER_USP_BITS_MASK					0x3
#define GROUP_MANAGER_USP_SIZE_SHIFT				2
#define GROUP_MANAGER_VALID_USER_POSITION_MASK		0x80
#define GROUP_MANAGER_STAID_LENGTH_MASK				0xFF

#ifdef ENET_INC_ARCH_WAVE600
#define GROUP_MANAGER_NUM_OF_SUPPORTED_BW			4 // 20Mhx, 40 MHz, 80 MHz, 160 MHz
#else 
#define GROUP_MANAGER_NUM_OF_SUPPORTED_BW			3 // 20Mhx, 40 MHz, 80 MHz
#endif
//Defines relevant for group manager statistics


//---------------------------------------------------------------------------------
//						Macros						
//---------------------------------------------------------------------------------
//

#define GROUP_MANAGER_MCS_DEGRADATION_FACTOR(mcs)			0 // TODO: in the future for each mcs should extract matching degradation factor from some configuration table
#define GROUP_MANAGER_STATIONS_BITMAP_SIZE_IN_BYTES			CONVERT_BIT_INDEX_TO_BYTES(HW_NUM_OF_STATIONS)
//---------------------------------------------------------------------------------
//						Data Type Definition					
//---------------------------------------------------------------------------------
typedef enum
{ 
	GROUP_MANAGER_STATION_EVENT_REMOVE_STATION = 0,
	GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION,
	GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP,
	GROUP_MANAGER_STATION_EVENT_TRAINING_FAILED,
	GROUP_MANAGER_STATION_EVENT_NOTIFICATION_STARTED,
	GROUP_MANAGER_STATION_EVENT_PROBING_STARTED,
	GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP,
	GROUP_MANAGER_STATION_EVENT_TIMER,
	GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_START_CONFIRMATION,
	GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_RESULTS_RECEIVED,
	GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION,
	GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION,
	GROUP_MANAGER_NUM_OF_STATION_EVENTS,
	GROUP_MANAGER_MAX_NUM_OF_STATION_EVENTS = 0xFF,	
} GroupManagerStationEvent_e;

typedef enum
{ 
	GROUP_MANAGER_STATION_EVENT_TO_STATE_FIRST = 0, 
		
	/* GROUP_MANAGER_STATION_STATE_IDLE */

	/* GROUP_MANAGER_STATION_STATE_NOT_ALLOWED */
	GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_IN_NOT_ALLOWED = GROUP_MANAGER_STATION_EVENT_TO_STATE_FIRST, /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION */

	/* GROUP_MANAGER_STATION_STATE_SINGLE_USER */
	GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_IN_SINGLE_USER,       /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION */
	GROUP_MANAGER_STATION_EVENT_TRAINING_FAILED_IN_SINGLE_USER,      /* GROUP_MANAGER_STATION_EVENT_TRAINING_FAILED */
	GROUP_MANAGER_STATION_EVENT_NOTIFICATION_STARTED_IN_SINGLE_USER, /* GROUP_MANAGER_STATION_EVENT_NOTIFICATION_STARTED */
	
	/* GROUP_MANAGER_STATION_STATE_NOTIFY_STATION */
	GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_IN_NOTIFYING_OR_PROBING ,  /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION */
	GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION_IN_NOTIFYING, /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION */
	GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_IN_NOTIFYING,                /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP */
	GROUP_MANAGER_STATION_EVENT_PROBING_STARTED_IN_NOTIFYING,             /* GROUP_MANAGER_STATION_EVENT_PROBING_STARTED */

	/* GROUP_MANAGER_STATION_STATE_PROBING */
	GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION_IN_PROBING, /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION */
	GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_IN_PROBING,                /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP */
	GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP_IN_PROBING,				/* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP */

	/* GROUP_MANAGER_STATION_STATE_TEMP_ACTIVE */
	GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_IN_ACTIVE_OR_TEMP_ACTIVE, /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION */
	GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_IN_TEMP_ACTIVE,			 /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP */
	GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP_IN_TEMP_ACTIVE,			 /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP */

	/* GROUP_MANAGER_STATION_STATE_ACTIVE */
	GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION_IN_ACTIVE, /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION */
	GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_IN_ACTIVE,				   /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP */
	GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP_IN_ACTIVE,			   /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP */
	
    /* GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_PENDING */
    GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_IN_SINGLE_USER_VALIDATION_PENDING,	           /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION */
	GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION_IN_SINGLE_USER_VALIDATION_PENDING, /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION */
	GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_IN_SINGLE_USER_VALIDATION_PENDING,				   /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP */
	GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP_IN_SINGLE_USER_VALIDATION_PENDING,			   /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP */
	GROUP_MANAGER_STATION_EVENT_TIMER_IN_SINGLE_USER_VALIDATION_PENDING,                       /* GROUP_MANAGER_STATION_EVENT_TIMER */
	
	/* GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_WAITING_FOR_LINK_ADAPTATION */
	GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_IN_SINGLE_USER_VALIDATION_WAITING_FOR_LINK_ADAPTATION,                            /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION */
	GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION_IN_SINGLE_USER_VALIDATION_WAITING_FOR_LINK_ADAPTATION,               /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION */
	GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_IN_SINGLE_USER_VALIDATION_WAITING_FOR_LINK_ADAPTATION,				                 /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP */
	GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP_IN_SINGLE_USER_VALIDATION_WAITING_FOR_LINK_ADAPTATION,			                 /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP */
	GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_START_CONFIRMATION_IN_SINGLE_USER_VALIDATION_WAITING_FOR_LINK_ADAPTATION, /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_START_CONFIRMATION */

	/* GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION */
	GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_IN_SINGLE_USER_VALIDATION,				           /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION */
	GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION_IN_SINGLE_USER_VALIDATION,	           /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION */
	GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_IN_SINGLE_USER_VALIDATION, 				           /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP */
	GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP_IN_SINGLE_USER_VALIDATION,				           /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP */
	GROUP_MANAGER_STATION_EVENT_TIMER_IN_SINGLE_USER_VALIDATION,                                   /* GROUP_MANAGER_STATION_EVENT_TIMER */
	GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_RESULTS_RECEIVED_IN_SINGLE_USER_VALIDATION, /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_RESULTS */

	/* GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_FAILED */
	GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_AFTER_SINGLE_USER_VALIDATION_ENDED,			              /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION */
	GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION_IN_SINGLE_USER_VALIDATION_ENDED,              /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION */
	GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_IN_SINGLE_USER_VALIDATION_ENDED, 			                  /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP */
	GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP_IN_SINGLE_USER_VALIDATION_ENDED,						      /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP */
	GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION_IN_SINGLE_USER_VALIDATION_FAILED, /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION */

	/* GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_SUCCEEDED */
	GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION_IN_SINGLE_USER_VALIDATION_SUCCEEDED, /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION */
	
    /* GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_CANCELED_GROUP_REMOVED */
    GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP_IN_SINGLE_USER_VALIDATION_CANCELED, 						                  /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP */
	GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION_IN_SINGLE_USER_VALIDATION_CANCELED_GROUP_REMOVED, /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION */
	
    /* GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_CANCELED_GROUP_FAILED */
	GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION_IN_SINGLE_USER_VALIDATION_CANCELED_GROUP_FAILED, /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION */

    /* GROUP_MANAGER_STATION_STATE_HALTING_WAITING_FOR_TRAINING_CONFIRMATION */
	GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_IN_HALTING_WAITING_FOR_TRAINING_CONFIRMATION,        /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION */
	GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION_IN_HALTING_WAITING_FOR_TRAINING_CONFIRMATION, /* GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION */

	/* GROUP_MANAGER_STATION_STATE_HALT */
	GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_IN_HALT,	/* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION */	
	GROUP_MANAGER_STATION_EVENT_TIMER_IN_HALT,          /* GROUP_MANAGER_STATION_EVENT_TIMER */
	
    /* GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_LINK_ADAPTATION_AND_TRAINING_CONFIRMATION */
	GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION_IN_REMOVING_WAITING_FOR_LINK_ADAPTATION_AND_TRAINING_CONFIRMATION, /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION */
	GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION_IN_REMOVING_WAITING_FOR_LINK_ADAPTATION_AND_TRAINING_CONFIRMATION,                   /* GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION */

	/* GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_TRAINING_CONFIRMATION */
	GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION_IN_REMOVING_WAITING_FOR_TRAINING_CONFIRMATION, /* GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION */

	/* GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_LINK_ADAPTATION_CONFIRMATION */
	GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION_IN_REMOVING_WAITING_FOR_LINK_ADAPTATION_CONFIRMATION, /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION */

	/* GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_ALL_GROUP_REMOVAL */
	GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_IN_REMOVING_WAITING_FOR_ALL_GROUPS_REMOVAL, /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP (due to station or not) */

	/* General Functions */
	GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,           /* ASSERT */
	GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION,           /* Ignore */
	GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_UPDATE_STATION_BITMAP, /* Only update station bitmap when group is removed */
	GROUP_MANAGER_STATION_EVENT_ADD_GROUP_UPDATE_STATION_BITMAP,    /* Only update station bitmap when group is ADDed */
	
	GROUP_MANAGER_STATION_NUM_OF_EVENTS_TO_STATES_FUNCTIONS,
	GROUP_MANAGER_STATION_MAX_NUM_OF_EVENTS_TO_STATES_FUNCTIONS = 0xFF,	
} GroupManagerStationEventToStateFunction_e;

typedef enum
{
	GROUP_MANAGER_GROUP_STATE_IDLE = 0,
	GROUP_MANAGER_GROUP_STATE_PROBING_NOTIFYING,
	GROUP_MANAGER_GROUP_STATE_PROBING_WAITING_FOR_LINK_ADAPTATION,
	GROUP_MANAGER_GROUP_STATE_PROBING,
	GROUP_MANAGER_GROUP_STATE_ACTIVE,
	GROUP_MANAGER_GROUP_STATE_REMOVING_WAITING_FOR_NOTIFICATIONS,
	GROUP_MANAGER_GROUP_STATE_REMOVING_WAITING_FOR_LOCK_AND_LINK_ADAPTATION_CONFIRMATIONS,
	GROUP_MANAGER_GROUP_STATE_REMOVING_WAITING_FOR_LOCK_CONFIRMATION,
	GROUP_MANAGER_GROUP_STATE_REMOVING_WAITING_FOR_LINK_ADAPTATION_CONFIRMATION,
	GROUP_MANAGER_NUM_OF_GROUP_STATES,
	GROUP_MANAGER_MAX_NUM_OF_GROUP_STATES = 0xFF,	
} GroupManagerGroupState_e;

typedef enum
{
	GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP = 0,
	GROUP_MANAGER_GROUP_EVENT_NOTIFICATION_HAS_BEEN_CONFIRMED,
	GROUP_MANAGER_GROUP_EVENT_TIMER,
	GROUP_MANAGER_GROUP_EVENT_ADD_GROUP_CONFIRMATION,
	GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP_CONFIRMATION,
	GROUP_MANAGER_GROUP_EVENT_PROBING_RESULTS_READY,
	GROUP_MANAGER_GROUP_EVENT_GROUP_FAILED_INDICATION,
	GROUP_MANAGER_GROUP_EVENT_LOCK_CONFIRMATION,
	GROUP_MANAGER_NUM_OF_GROUP_EVENTS,
	GROUP_MANAGER_MAX_NUM_OF_GROUP_EVENTS = 0xFF,	
} GroupManagerGroupEvent_e;

typedef enum
{ 
	GROUP_MANAGER_GROUP_EVENT_TO_STATE_FIRST = 0,               
		
	/* GROUP_MANAGER_GROUP_STATE_IDLE */

	/* GROUP_MANAGER_GROUP_STATE_PROBING_NOTIFYING */
	GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP_IN_PROBING_NOTIFYING = GROUP_MANAGER_GROUP_EVENT_TO_STATE_FIRST, /* GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP */
	GROUP_MANAGER_GROUP_EVENT_NOTIFICATION_HAS_BEEN_CONFIRMED_IN_PROBING_NOTIFYING,                         /* GROUP_MANAGER_GROUP_EVENT_NOTIFICATION_HAS_BEEN_CONFIRMED */
	GROUP_MANAGER_GROUP_EVENT_TIMER_IN_PROBING_NOTIFYING,                                                   /* GROUP_MANAGER_GROUP_EVENT_TIMER */

	/* GROUP_MANAGER_GROUP_STATE_PROBING_WAITING_FOR_LINK_ADAPTATION */
	GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP_IN_PROBING_WAITING_FOR_LINK_ADAPTATION,				 /* GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP */
	GROUP_MANAGER_GROUP_EVENT_ADD_GROUP_CONFIRMATION_IN_PROBING_WAITING_FOR_LINK_ADAPTATION,	 /* GROUP_MANAGER_GROUP_EVENT_ADD_GROUP_CONFIRMATION */

    /* GROUP_MANAGER_GROUP_STATE_PROBING */
	GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP_IN_PROBING,		    /* GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP */
	GROUP_MANAGER_GROUP_EVENT_TIMER_IN_PROBING,                 /* GROUP_MANAGER_GROUP_EVENT_TIMER */
	GROUP_MANAGER_GROUP_EVENT_PROBING_RESULTS_READY_IN_PROBING,	/* GROUP_MANAGER_GROUP_EVENT_PROBING_RESULTS_READY */

	/* GROUP_MANAGER_GROUP_STATE_ACTIVE */
	GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP_IN_ACTIVE,		     /* GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP */
	GROUP_MANAGER_GROUP_EVENT_PROBING_RESULTS_READY_IN_ACTIVE,   /* GROUP_MANAGER_GROUP_EVENT_PROBING_RESULTS_READY */
	GROUP_MANAGER_GROUP_EVENT_GROUP_FAILED_INDICATION_IN_ACTIVE, /* GROUP_MANAGER_GROUP_EVENT_GROUP_FAILED_INDICATION */

	/* GROUP_MANAGER_GROUP_STATE_REMOVING_WAITING_FOR_NOTIFICATIONS */
	GROUP_MANAGER_GROUP_EVENT_NOTIFICATION_HAS_BEEN_CONFIRMED_IN_REMOVING_WAITING_FOR_NOTIFICATIONS, /* GROUP_MANAGER_GROUP_EVENT_NOTIFICATION_HAS_BEEN_CONFIRMED */

	/* GROUP_MANAGER_GROUP_STATE_REMOVING_WAITING_FOR_LOCK_AND_LINK_ADAPTATION_CONFIRMATIONS */
	GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP_CONFIRMATION_IN_REMOVING_WAITING_FOR_LOCK_AND_LINK_ADAPTATION_CONFIRMATIONS, /* GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP_CONFIRMATION */
	GROUP_MANAGER_GROUP_EVENT_LOCK_CONFIRMATION_IN_REMOVING_WAITING_FOR_LOCK_AND_LINK_ADAPTATION_CONFIRMATIONS,         /* GROUP_MANAGER_GROUP_EVENT_LOCK_CONFIRMATION */
	
	/* GROUP_MANAGER_GROUP_STATE_REMOVING_WAITING_FOR_LOCK_CONFIRMATION */
	GROUP_MANAGER_GROUP_EVENT_LOCK_CONFIRMATION_IN_REMOVING_WAITING_FOR_LOCK_CONFIRMATION,         /* GROUP_MANAGER_GROUP_EVENT_LOCK_CONFIRMATION */
	
	/* GROUP_MANAGER_GROUP_STATE_REMOVING_WAITING_FOR_LINK_ADAPTATION_CONFIRMATION */
	GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP_CONFIRMATION_IN_REMOVING_WAITING_FOR_LINK_ADAPTATION_CONFIRMATIONS, /* GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP_CONFIRMATION */
                 
	/* General Functions */
	GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION, /* ASSERT */
	GROUP_MANAGER_GROUP_EVENT_TO_STATE_IGNORE_FUNCTION, /* Ignore */   
	
	GROUP_MANAGER_GROUP_NUM_OF_EVENTS_TO_STATES_FUNCTIONS,
	GROUP_MANAGER_GROUP_MAX_NUM_OF_EVENTS_TO_STATES_FUNCTIONS = 0xFF,	
} GroupManagerGroupEventToStateFunction_e;


typedef enum
{
	GROUP_MANAGER_GLOBAL_STATE_IDLE = 0,
	GROUP_MANAGER_GLOBAL_STATE_ACTIVE,
	GROUP_MANAGER_GLOBAL_STATE_DISABLING_WAITING_FOR_TRAINING,
	GROUP_MANAGER_GLOBAL_STATE_DISABLING_WAITING_FOR_ALL_GROUP_REMOVAL,
	GROUP_MANAGER_GLOBAL_STATE_CREATE_STATIC_GROUP_WAITING_FOR_TRAINING,
	GROUP_MANAGER_GLOBAL_STATE_CREATE_STATIC_GROUP_WAITING_FOR_ALL_GROUP_REMOVAL,
	GROUP_MANAGER_GLOBAL_STATE_CREATING_STATIC_GROUP,
	GROUP_MANAGER_NUM_OF_GLOBAL_STATES,
	GROUP_MANAGER_MAX_NUM_OF_GLOBAL_STATES = 0xFF,	
} GroupManagerGlobalState_e;

typedef enum
{
	GROUP_MANAGER_SET_OPERATION_REQUEST_PHASE_IDLE,
	GROUP_MANAGER_SET_OPERATION_REQUEST_PHASE_DISABLING,
	GROUP_MANAGER_SET_OPERATION_REQUEST_PHASE_ENABLING,
	GROUP_MANAGER_SET_OPERATION_REQUEST_PHASE_REQ_FROM_HOST,
	GROUP_MANAGER_SET_OPERATION_REQUEST_PHASE_MAX_NUM = 0xFF,
} GroupManagerSetOperationRequestPhase_e;

typedef struct GroupManagerVirtualGroupParameters
{
	uint32 timeStamp;
	StaId participatingStations[GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP];
	uint8 groupMark;
	uint8 reserved;
	/* bits 0-1 contain nss value of station found in index 0 in participatingStations array.
	   bits 2-3 contain nss value of station found in index 1 in participatingStations array.
	   etc*/
	uint8 nssBitmap;
	uint8 numberOfParticipatingStations;
	uint8 mcs[GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP];
	uint8 ratePerStationSingleUserOneNsts[GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP];
	uint8 ratePerStationSingleUserTwoNsts[GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP];
} GroupManagerVirtualGroupParameters_t;

typedef struct GroupManagerGroupParameters
{
	CalendarWheelEntry_t calendarWheelEntry;
	StaId participatingStations[GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP]; /* this array contains SIDs of 4 stations that participate in this group*/

	uint8 numberOfParticipatingStations;
	uint8 groupFailed; /* A flag that indicates if the group has failed due to notification failure or bad results */
	uint8 groupIndex;
	GroupManagerGroupState_e state;
	uint8 primaryStationsBitmap; /* Each bit represents if the station in the correlating index is primary member */
	uint8 nssBitmap;/* Each 2 bits represent the nss of the station in the correlating index */
	uint16 reserved;
	uint8 mcs[GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP]; // mcs values per station - from Training results
	GroupManagerNotificationStatus_e notificationsArray[GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP]; /* This array is only used to monitor notifications status in the notification process */  
	uint8 singleNstsPhyRates[GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP];
	uint8 doubleNstsPhyRates[GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP];
} GroupManagerGroupParameters_t;

typedef struct GroupManagerStationParameters
{
	CalendarWheelEntry_t calendarWheelEntry;
	
	/* tempPrimaryGroup contains index of the probing group to which this station currently refers. The station itself can either be in probing state or in 
	   tempactive state*/
#ifdef ENET_INC_ARCH_WAVE600
    uint32 userPositionVector; /* Used to create the group id management frame payload */ 

	uint32 virtualGroupsBitmap; // each bit represents index of virtual group the station is member of

    uint16 groupsBitmap; // includes all group types: notifying, probing, active and during remove process

    uint16 stationIndex;
	GroupManagerStationState_e state;    
    uint8 numberOfHalts;

    uint8 timeout;           
    uint8 primaryGroup; //TBD change the name to index
	uint8 tempPrimaryGroup;
	
#ifdef ENET_INC_ARCH_WAVE600D2
	uint8 mbssRxControlSupport;
	uint8 reserved[2];
#else
	uint8 reserved[3];
#endif

#else
	uint8 tempPrimaryGroup;
	GroupManagerStationState_e state;    
    uint8 numberOfHalts;
    uint8 timeout;           

	uint32 virtualGroupsBitmap; // each bit represents index of virtual group the station is member of

    uint16 groupsBitmap; // includes all group types: notifying, probing, active and during remove process
    uint8 primaryGroup; //TBD change the name to index
    uint8 stationIndex; 

    uint32 userPositionVector; /* Used to create the group id management frame payload */ 

#endif //ENET_INC_ARCH_WAVE600
} GroupManagerStationParameters_t;

typedef struct GroupManagerConfigurationParameters
{
	uint8 probingGroupGradeThreshold;
	uint8 stationGradeInProbingGroupThreshold;
	uint8 maxSingleUserValidationStations;
	uint8 singleUserPerformenceMargin;
	uint8 maxNumProbingGroups;
} GroupManagerConfigurationParameters_t;

typedef struct GroupManagerGlobalParameters_
{	
	GroupManagerTrainingBitmap_t trainingBitmaps; /* contains priority bitmaps of training and bitmap of halted / halt in process stations */ 
	uint32 waitingGroupsMarksBitmap[GROUP_MANAGER_SIZE_OF_STATION_BITMAP_IN_WORDS]; /* represents stations that their primary group was removed and are waiting for group marks to choose new primary group */
	uint32 singleUserValidationBitmap[GROUP_MANAGER_SIZE_OF_STATION_BITMAP_IN_WORDS]; /* bitmap of single user validation candidates, stations in tempActive or in process of single user validation are not in this bitmap*/
	GroupManager_CreateStaticGroup_t staticGroupParams; /* contains static group members details as received in host request*/

	CalendarWheelParameters_t  groupManagerCalendarWheel;

	uint32 virtualGroupsBitmap;
	
	uint16 activeGroupsBitmap;
	uint16 removingGroupsBitmap; /* all groups which are during ongoing remove process*/
	
	uint16 probingGroupsBitmap;
	uint16 notifyingGroupsBitmap;
	
	uint16 occupiedGroupsBitmap; /* this bitmap represents groups' indices that are in used - in order fo find a free index  for probing */
	uint16 waitingGroupsForLockBitmap;
	uint8 numberOfProbingGroups;
	uint8 numberOfOccupiedGroups;//TBD check if needed

	uint8 isLockPending;
	uint8 numberOfActiveGroups;
	uint8 numberOfRemovingGroupsInActive; // number of groups during remove process which started remove in Active state
	GroupManagerGlobalState_e globalState;

	uint8 numberOfCurrentSingleUserValidation;
	uint8 wasGroupsMarksMessageSent;
	uint8 lastSingleUserValidationStation; /* last station that was chosen for single user validation for round robin procedure */
	uint8 staticOccupiedGroupIndex; //index in  groups DB

	K_MSG *returnMsg;

	uint8 staticVirtualGroupIndex; // index in virtual groups DB
	uint8 staticGroupInitialMcs;
	bool  staticGroupOnlyMode;

	GroupManagerSetOperationRequestPhase_e setOperationRequestPhase;
	bool fixedRateFlag;
	GroupManagerConfigurationParameters_t muConfigParams;
	uint8 cocTxAntsNum;
} GroupManagerGlobalParameters_t;

typedef void (*GroupManagerStationGeneralFunc)(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
typedef void (*GroupManagerGroupGeneralFunc)(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters);

//---------------------------------------------------------------------------------
//						Public Function Declaration				
//---------------------------------------------------------------------------------
extern void GroupManager_TaskEntry(K_MSG *groupManagerMessage);

#endif /* GROUP_MANAGER_H_ */

