/***********************************************************************************
 File:			GroupManager.c
 Module:		Group Manager
 Purpose: 		To create and delete multi user groups
 Description:   This file is the implementation of the group manager which is 
 				responsible of creating and deleting multi user groups
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "BSSmanager_API.h"
#include "GroupManager_API.h"
#include "GroupManager.h"
#include "GidmManager_Api.h"
#include "TrainingManager_Api.h"
#include "Utils_Api.h"
#include "OSAL_Api.h"
#include "queue_utility.h"
#include "loggerAPI.h"
#include "ShramStationDatabase.h"
#include "TxSelector_Api.h"
#if defined (MU_MIMO_STATIC_GROUP_ENABLED)
#include "StaDatabase_Api.h"
#include "CoC_Api.h"
#endif //MU_MIMO_STATIC_GROUP_ENABLED
#include "GroupDatabase_Api.h"
#include "linkAdaptation_api.h"
#include "Locker_Api.h"
#include "Pac_Api.h"
#include "StaDatabase_Api.h"

#include "Hdk_Api.h" 
#ifndef TRAINING_WAVE600_Z0
#include "NdpaManager_Api.h"
#endif // TRAINING_WAVE600_Z0
#include "Statistics_Descriptors.h"
#include "StatisticsManager_api.h"
#include "GroupManagerPhyDriver.h"
#include "stringLibApi.h"
/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_GROUP_MANAGER 
#define LOG_LOCAL_FID 0

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
	

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
/* Message functions */
static void groupManagerAddStation(K_MSG* groupManagerMessage);
static void groupManagerRemoveStation(K_MSG* groupManagerMessage);
static void groupManagerTrainingTimer(K_MSG* groupManagerMessage);
static void groupManagerSingleUserValidationStartConfirmation(K_MSG* groupManagerMessage);
static void groupManagerSingleUserValidationResultsReceived(K_MSG* groupManagerMessage);
static void groupManagerSingleUserValidationEndConfirmation(K_MSG* groupManagerMessage);
static void groupManagerGroupsMarksReceived(K_MSG* groupManagerMessage);
static void groupManagerStationNssHasChanged(K_MSG* groupManagerMessage);
static void groupManagerAddGroupConfirmation(K_MSG* groupManagerMessage);
static void groupManagerProbingResultsReceived(K_MSG* groupManagerMessage);
static void groupManagerRemovedGroupConfirmation(K_MSG* groupManagerMessage);
static void groupManagerGroupsHasFailed(K_MSG* groupManagerMessage);
static void groupManagerGroupLockConfirmation(K_MSG* groupManagerMessage);
static void groupManagerSetMultiUserOperation(K_MSG* groupManagerMessage);
static void groupManagerResetMultiUserOperation(K_MSG* groupManagerMessage);
static void groupManagerSetMultiUserOperationReq(K_MSG* groupManagerMessage);
static void groupManagerSetMultiUserOperationCfm(void);
static void groupManagerSetMuFixedRate(K_MSG* groupManagerMessage);

/* Station section */

/* Internal events functions */
static void groupManagerStationEventRemoveGroupDueToStation(StaId stationIndex, uint8 probingGroupIndex);
static void groupManagerStationEventRemoveGroup(StaId stationIndex, uint8 probingGroupIndex);
static void groupManagerStationEventTrainingFailed(StaId stationIndex);
static void groupManagerStationEventNotificationStarted(StaId stationIndex, uint8 probingGroupIndex);
static void groupManagerStationEventProbingStarted(StaId stationIndex, uint8 probingGroupIndex);
static void groupManagerStationEventMoveToTempActive(StaId stationIndex, uint8 probingGroupIndex);
static void groupManagerStationEventActivateGroup(StaId stationIndex, GroupManagerProbingResultsMessage_t *groupManagerProbingResultsParameters);
static void groupManagerStationEventSingleUserValidationStartProcess(StaId stationIndex);

/* Remove station functions */
static void groupManagerStationEventRemoveStationInNotAllowed(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventRemoveStationInSingleUser(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventRemoveStationInNotifyingOrProbing(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventRemoveStationInActiveOrTempActive(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventRemoveStationInSingleUserValidationPending(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventRemoveStationInSuValWaitingLa(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventRemoveStationInSingleUserValidation(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventRemoveStationAfterSuValEnded(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventRemoveStationInHaltingWaitingForTrainingCfm(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventRemoveStationInHalt(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);


/* Remove group due to station functions */
static void groupManagerStationEventRemoveGroupDueToStationInNotifying(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventRemoveGroupDueToStationInProbing(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventRemoveGroupInTempActive(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventRemoveGroupDueToStationInActive(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventRemoveGroupDueToStationInSingleUserValidationPending(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventRemoveGroupDueToStationInSuValWaitingLa(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventRemoveGroupDueToStationInSingleUserValidation(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventRemoveGroupDueToStationInSingleUserValidationEnded(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventRemoveGroupInRemovingWaitingForAllGroupRemoval(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);


/* Remove group functions */
static void groupManagerStationEventRemoveGroupInNotifying(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventRemoveGroupInProbing(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventRemoveGroupInActive(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventRemoveGroupInSingleUserValidationPending(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventRemoveGroupInSuValWaitingLa(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventRemoveGroupInSingleUserValidation(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventRemoveGroupInSingleUserValidationEnded(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);


/* Training failed functions */
static void groupManagerStationEventTrainingFailedInSingleUser(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);


/* Notification started functions */
static void groupManagerStationEventNotificationStartedInSingleUser(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);


/* Probing started functions */
static void groupManagerStationEventProbingStartedInNotifying(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);


/* Activate group functions */
static void groupManagerStationEventActivateGroupInProbing(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventActivateGroupInTempActive(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventActivateGroupInActive(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventActivateGroupInSingleUserValidationPending(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventActivateGroupInSuValWaitingLa(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventActivateGroupInSingleUserValidation(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventActivateGroupInSingleUserValidationEnded(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventActivateGroupInSingleUserValidationCanceled(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);


/* Timer functions */
static void groupManagerStationEventTimerInSingleUserValidationPending(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventTimerInSingleUserValidation(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventTimerInHalt(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);


/* Single user validation start confirmation functions */
static void groupManagerStationEventSuValStartCfmInSuValWaitingLa(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);


/* Single user validation results functions */
static void groupManagerStationEventSuValResultsReceivedInSuVal(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);


/* Single user validation end confirmation fucntions */
static void groupManagerStationEventSuValEndCfmInSuValFailed(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventSuValEndCfmInSuValSucceeded(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventSuValEndCfmInSuValCanceledGroupRemoved(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventSuValEndCfmInSuValCanceledGroupFailed(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventSuValEndCfmInRemovingWaitingForTrainingAndLaCfm(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventSuValEndCfmInRemovingWaitingForLaCfm(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);


/* Training confirmation functions */
static void groupManagerStationEventTrainingCfmInHaltingWaitingForTrainingCfm(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventTrainingCfmInRemovingWaitingForTrainingAndLaCfm(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventTrainingCfmInRemovingWaitingForTrainingCfm(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);


/* General functions */
static void groupManagerStationEventToStateAssert(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventToStateIgnore(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventRemoveGroupUpdateStationBitmap(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);
static void groupManagerStationEventAddGroupUpdateStationBitmap(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters);


/* Service function */
static void groupManagerRunStationStateMachine(StaId stationIndex, GroupManagerStationEvent_e event, void *parameters);
static void groupManagerUpdateTrainingPriorityBitmaps(GroupManagerStationState_e currentState, GroupManagerStationState_e newState, StaId stationIndex);
static void groupManagerChangeStationState(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, GroupManagerStationState_e newState);
static void groupManagerSendConfirmToStationManager(StaId stationIndex);
static void groupManagerRemoveAllStationVirtualGroups(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry);
static void groupManagerRemoveStationActiveGroups(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry);
static uint8 groupManagerFindStationNssInVirtualGroup(StaId stationIndex, uint8 virtualGroupIndex);
static uint8 groupManagerFindStationNssInActiveGroup(StaId stationIndex, uint8 groupIndex);
static void groupManagerSetStationPrimaryBitInGroupsDb(StaId stationIndex, uint8 groupIndex);
static void groupManagerConfigureStationAsPrimaryInGroup(StaId stationIndex, uint8 groupIndex);
static void groupManagerClearStationPrimaryInProbingGroup(StaId stationIndex, uint8 groupIndex);
static void groupManagerClearStationPrimaryInGroup(StaId stationIndex, uint8 groupIndex);
static void groupManagerSendEndSingleUserValidationMessage(StaId stationIndex);
static void groupManagerTriggerRemoveStationInTraining(StaId stationIndex);
static void groupManagerTriggerHaltStationInTraining(StaId stationIndex);
static uint8 groupManagerChooseBestGroupFromProbingMessage(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, GroupManagerProbingResultsMessage_t *groupManagerProbingResultsParameters);
static uint8 groupManagerFindBestGroup(uint16 groupsBitmap, uint8 *groupMarks);
static void groupManagerPrimaryGroupWasRemoved(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry);
static void groupManagerRemoveGroupInActive(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters, uint8 isHaltNeeded);
static void groupManagerRemoveGroupInSingleUserValidationPending(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters, uint8 isHaltNeeded);
static void groupManagerSuValEndCfmInSuValCanceled(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters, uint8 isHaltNeeded);
static void groupManagerTryToStartNewSingleUserValidation(StaId stationIndex);
static void groupManagerSendGroupsMarksMessage(StaId stationIndex);
static void groupManagerStationExitActiveState(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, uint8 isHaltNeeded);
static void groupManagerStartNewSingleUserValidation(void);
static void groupManagerConfigureStationForSelector(StaId stationIndex, bool isPrimary, uint8 groupIndex, bool changeInStaDbRequired);
static void groupManagerConfigureNewGroupForSelector(uint8 groupIndex, StaId primaryStationIndex);
static void groupManagerTriggerStaticGroupGeneration(void);
static void groupManagerClearGroupBitInStationBitmap(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, uint8 groupIndex);
static void groupManagerSetGroupBitInStationBitmap(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, uint8 groupIndex);
static void groupManagerSetStationSuTxopAndAmpduLimit(StaId stationIndex, uint8 txopMode,uint8 numOfAmpdus);


/* Group section */

/* Internal events functions */
static void groupManagerGroupEventRemoveGroup(uint8 groupIndex, void *parameters);
static void groupManagerGroupEventStartNotifying(uint8 groupIndex, uint8 virtualGroupIndex);


/* Remove group functions */
static void groupManagerGroupEventRemoveGroupInProbingNotifying(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters);
static void groupManagerGroupEventRemoveGroupInProbingWaitingForLinkAdaptation(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters);
static void groupManagerGroupEventRemoveGroupInProbing(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters);
static void groupManagerGroupEventRemoveGroupInActive(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters);


/* Notification confirmation functions */
static void groupManagerGroupEventNotificationCfmInProbingNotifying(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters);
static void groupManagerGroupEventNotificationCfmInRemovingWaitingForNotifications(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters);


/* Timer functions */
static void groupManagerGroupEventTimerInProbingNotifying(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters);
static void groupManagerGroupEventTimerInProbing(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters);


/* Add group confirmation functions */
static void groupManagerGroupEventAddGroupCfmInProbingWaitingForLa(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters);


/* Remove group confirmation functions */
static void groupManagerGroupEventRemoveGroupCfmInRemovingWaitingForLockAndLaCfms(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters);
static void groupManagerGroupEventRemoveGroupCfmInRemovingWaitingForLaCfm(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters);


/* Probing results ready functions */
static void groupManagerGroupEventProbingResultsAreReadyInProbing(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters);
static void groupManagerGroupEventPropingResultsReadyInActive(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters);

/* Group has failed functions */
static void groupManagerGroupEventGroupHasFailedInActive(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters);


/* Lock confirmation functions */
static void groupManagerGroupEventLockCfmInRemovingWaitingForLockAndLaCfm(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters);
static void groupManagerGroupEventLockConfirmationInRemovingWaitingForLockConfirmation(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters);

/* General Functions */
static void groupManagerGroupEventToStateAssert(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters);
static void groupManagerGroupEventToStateIgnore(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters);
static void GroupManagerResetGroupMembersStatistic(uint8 groupIndex);


/* Service functions */
static void groupManagerRunGroupStateMachine(uint8 groupIndex, GroupManagerGroupEvent_e event, void *parameters);
static void groupManagerChangeGroupState(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, GroupManagerGroupState_e newState);
static void groupManagerChangeGlobalState(GroupManagerGlobalState_e newState);
static void groupManagerSendRemoveGroupMessage(uint8 groupIndex);
static void groupManagerSendAddGroupMessage(uint8 groupIndex);
static void groupManagerLockGroup(uint8 groupIndex);
static void groupManagerGroupRemoveGroupInProbingNotifying(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters);
static void groupManagerHandleNextGroupWaitingForLock(uint8 groupIndex);
static void groupManagerFinishGroupRemovalProcess(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry);
static void groupManagerFinishGlobalRemovalProcess(void);
static void groupManagerTryToFindNewGroupForProbing(bool isStaticGroup);
static void groupManagerConfigureHwGroupDbs(uint8 groupIndex);
static void groupManagerProcessTrainingResults(uint8 numOfCombinations, GroupCombinationTableEntry_t *trainingCombinations, bool isStaticGroup);



/* General section */

/* General functions */
static void groupManagerRemoveAllActiveGroups(void);
static void groupManagerRemoveAllVirtualGroups(void);
static void groupManagerRemoveVirtualGroup(uint8 virtualGroupIndex);

#if defined (MU_MIMO_STATIC_GROUP_ENABLED)
static void groupManagerCreateMuStaticGroup(K_MSG* groupManagerMessage);
static void groupManagerDeleteMuStaticGroup(K_MSG* groupManagerMessage);	                    	
#endif

#ifdef TRAINING_WAVE600_Z0
static void GroupManagerIllegalEvent(K_MSG* groupManagerMessage);
#endif
void groupManagerUpdateCoCInfo(K_MSG *pCoCInfoMsg);


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

GroupManagerStatistics_t groupDbStatistics;


static uint16 GroupManagerStasCalendarWheelSlots[GROUP_MANAGER_CALENDAR_WHEEL_SIZE]; //  db which contains stations events in calendar wheel
static uint16 GroupManagerGrpsCalendarWheelSlots[GROUP_MANAGER_CALENDAR_WHEEL_SIZE]; //  db which contains groups events in calendar wheel
static CalendarWheelParameters_t GroupManagerStationsCalendarWheel; // calendar wheel for station timer events
static CalendarWheelParameters_t GroupManagerGroupsCalendarWheel; // calendar wheel for groups timer events

static GroupManagerGlobalParameters_t GroupManagerGlobalParametersDb;
static GroupManagerStationParameters_t GroupManagerStationDb[HW_NUM_OF_STATIONS]; 
static GroupManagerGroupParameters_t   GroupManagerGroupDb[GROUP_MANAGER_MAX_NUMBER_OF_GROUPS];

static GroupManagerVirtualGroupParameters_t GroupManagerVirtualGroupDatabase[GROUP_MANAGER_MAX_NUMBER_OF_VIRTUAL_GROUPS];


static const FunctionEntry_t afpTaskTable[TASK_GROUP_MANAGER_END - TASK_GROUP_MANAGER_START] =
{
	/* Group manager main state machine messages */
	{groupManagerAddStation, 					                DOUBLE_CHECK_MSG_TYPE(GROUP_MANAGER_ADD_STATION)},
	{groupManagerRemoveStation, 					            DOUBLE_CHECK_MSG_TYPE(GROUP_MANAGER_REMOVE_STATION)},
	{groupManagerTrainingTimer, 					            DOUBLE_CHECK_MSG_TYPE(GROUP_MANAGER_TRAINING_TIMER_EXPIRED)},
	{groupManagerSingleUserValidationStartConfirmation, 		DOUBLE_CHECK_MSG_TYPE(GROUP_MANAGER_SINGLE_USER_VALIDATION_START_CONFIRMATION)},
	{groupManagerSingleUserValidationResultsReceived, 		    DOUBLE_CHECK_MSG_TYPE(GROUP_MANAGER_SINGLE_USER_VALIDATION_RESULTS)},
	{groupManagerSingleUserValidationEndConfirmation, 	        DOUBLE_CHECK_MSG_TYPE(GROUP_MANAGER_SINGLE_USER_VALIDATION_END_CONFIRMATION)},
	{groupManagerStationNssHasChanged, 	                    	DOUBLE_CHECK_MSG_TYPE(GROUP_MANAGER_STATION_NSS_HAS_CHANGED)},	
	{groupManagerAddGroupConfirmation, 				            DOUBLE_CHECK_MSG_TYPE(GROUP_MANAGER_ADD_GROUP_CONFIRMATION)},
	{groupManagerProbingResultsReceived, 			            DOUBLE_CHECK_MSG_TYPE(GROUP_MANAGER_PROBING_RESULTS)},
	{groupManagerRemovedGroupConfirmation, 			            DOUBLE_CHECK_MSG_TYPE(GROUP_MANAGER_REMOVE_GROUP_CONFIRMATION)},
	{groupManagerGroupsMarksReceived, 	                        DOUBLE_CHECK_MSG_TYPE(GROUP_MANAGER_GROUPS_MARKS)},	
	{groupManagerGroupsHasFailed, 	                        	DOUBLE_CHECK_MSG_TYPE(GROUP_MANAGER_GROUP_FAILED)},
	{groupManagerGroupLockConfirmation, 	                    DOUBLE_CHECK_MSG_TYPE(GROUP_MANAGER_GROUP_LOCK_CONFIRM)},
	{groupManagerSetMultiUserOperation, 	                    DOUBLE_CHECK_MSG_TYPE(GROUP_MANAGER_SET_MULTI_USER_OPERATION)},
	{groupManagerResetMultiUserOperation, 						DOUBLE_CHECK_MSG_TYPE(GROUP_MANAGER_RESET_MULTI_USER_OPERATION)},
	{groupManagerSetMultiUserOperationReq,						DOUBLE_CHECK_MSG_TYPE(GROUP_MANAGER_SET_MULTI_USER_OPERATION_REQ)},	
	{groupManagerSetMuFixedRate,								DOUBLE_CHECK_MSG_TYPE(GROUP_MANAGER_SET_MU_GROUP_FIXED_RATE)},
#if defined (MU_MIMO_STATIC_GROUP_ENABLED)
	{groupManagerCreateMuStaticGroup, 	                    	DOUBLE_CHECK_MSG_TYPE(GROUP_MANAGER_CREATE_MU_STATIC_GROUP)},
	{groupManagerDeleteMuStaticGroup, 	                    	DOUBLE_CHECK_MSG_TYPE(GROUP_MANAGER_DELETE_MU_STATIC_GROUP)},
#endif // MU_MIMO_STATIC_GROUP_ENABLED
	/*Group id management action frame manager messages*/
	{gidmManagerTxCfm,								            DOUBLE_CHECK_MSG_TYPE(GIDM_MANAGER_TX_CFM)}, 
	{gidmManagerPdAllocated,						            DOUBLE_CHECK_MSG_TYPE(GIDM_MANAGER_PD_ALLOCATED)},
	{gidmManagerStaLockCfm,							            DOUBLE_CHECK_MSG_TYPE(GIDM_MANAGER_LOCK_CONFIRMED)}, 
#ifndef TRAINING_WAVE600_Z0
	/*NDPA manager*/
	{NdpaManager_TxCfm,											DOUBLE_CHECK_MSG_TYPE(NDPA_MANAGER_TX_CFM)},
	{NdpaManager_QueueLockCfm,									DOUBLE_CHECK_MSG_TYPE(NDPA_MANAGER_LOCK_CONFIRMED)},
	{NdpaManager_PdAllocated,									DOUBLE_CHECK_MSG_TYPE(NDPA_MANAGER_PD_ALLOCATED)},
	{NdpaManager_LaBfReportInd,									DOUBLE_CHECK_MSG_TYPE(NDPA_MANAGER_LA_BF_REPORT_IND)},
#else
	/*NDPA manager*/
	{GroupManagerIllegalEvent,									DOUBLE_CHECK_MSG_TYPE(NDPA_MANAGER_TX_CFM)},
	{GroupManagerIllegalEvent,									DOUBLE_CHECK_MSG_TYPE(NDPA_MANAGER_LOCK_CONFIRMED)},
	{GroupManagerIllegalEvent,									DOUBLE_CHECK_MSG_TYPE(NDPA_MANAGER_PD_ALLOCATED)},
	{GroupManagerIllegalEvent,									DOUBLE_CHECK_MSG_TYPE(NDPA_MANAGER_LA_BF_REPORT_IND)},
#endif //TRAINING_WAVE600_Z0
	/*Training manager messages*/
	{TrainingManagerAddVapReq,									DOUBLE_CHECK_MSG_TYPE(TRAINING_MANAGER_SET_BSS)},
	{TrainingManagerRemoveVapReq,								DOUBLE_CHECK_MSG_TYPE(TRAINING_MANAGER_STOP_TRAFFIC)},
	{TrainingManagerLaInfoCfm,									DOUBLE_CHECK_MSG_TYPE(TRAINING_MANAGER_LA_INFO_CFM)},
	/*Update CoC info*/
	{groupManagerUpdateCoCInfo,									DOUBLE_CHECK_MSG_TYPE(GROUP_MANAGER_UPDATE_COC_INFO)},

};


static const GroupManagerStationEventToStateFunction_e GroupManagerStationStateMachine[GROUP_MANAGER_NUM_OF_STATION_STATES][GROUP_MANAGER_NUM_OF_STATION_EVENTS] = 
{
	/* GROUP_MANAGER_STATION_STATE_IDLE */
	{
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION, /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION, /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION, /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION, /* GROUP_MANAGER_STATION_EVENT_TRAINING_FAILED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION, /* GROUP_MANAGER_STATION_EVENT_NOTIFICATION_STARTED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION, /* GROUP_MANAGER_STATION_EVENT_PROBING_STARTED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION, /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION, /* GROUP_MANAGER_STATION_EVENT_TIMER */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION, /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_START_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION, /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_RESULTS_RECEIVED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION, /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION, /* GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION */
	},
	
	/* GROUP_MANAGER_STATION_STATE_NOT_ALLOWED */
	{
		GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_IN_NOT_ALLOWED, /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,      /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,      /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,      /* GROUP_MANAGER_STATION_EVENT_TRAINING_FAILED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,      /* GROUP_MANAGER_STATION_EVENT_NOTIFICATION_STARTED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,      /* GROUP_MANAGER_STATION_EVENT_PROBING_STARTED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,      /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,      /* GROUP_MANAGER_STATION_EVENT_TIMER */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,      /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_START_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,      /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_RESULTS_RECEIVED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,      /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,      /* GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION */
	},
	
	/* GROUP_MANAGER_STATION_STATE_SINGLE_USER */
	{
		GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_IN_SINGLE_USER,       /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,            /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,            /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TRAINING_FAILED_IN_SINGLE_USER,      /* GROUP_MANAGER_STATION_EVENT_TRAINING_FAILED */
		GROUP_MANAGER_STATION_EVENT_NOTIFICATION_STARTED_IN_SINGLE_USER, /* GROUP_MANAGER_STATION_EVENT_NOTIFICATION_STARTED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,            /* GROUP_MANAGER_STATION_EVENT_PROBING_STARTED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,            /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,            /* GROUP_MANAGER_STATION_EVENT_TIMER */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,            /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_START_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,            /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_RESULTS_RECEIVED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,            /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,            /* GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION */
	},
	
    /* GROUP_MANAGER_STATION_STATE_NOTIFY_STATION */
	{
		GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_IN_NOTIFYING_OR_PROBING ,  /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION */
		GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION_IN_NOTIFYING, /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION */
		GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_IN_NOTIFYING,                /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION,                 /* GROUP_MANAGER_STATION_EVENT_TRAINING_FAILED */
		GROUP_MANAGER_STATION_EVENT_ADD_GROUP_UPDATE_STATION_BITMAP,          /* GROUP_MANAGER_STATION_EVENT_NOTIFICATION_STARTED */
		GROUP_MANAGER_STATION_EVENT_PROBING_STARTED_IN_NOTIFYING,             /* GROUP_MANAGER_STATION_EVENT_PROBING_STARTED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                 /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                 /* GROUP_MANAGER_STATION_EVENT_TIMER */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                 /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_START_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                 /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_RESULTS_RECEIVED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                 /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                 /* GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION */
	},
	
    /* GROUP_MANAGER_STATION_STATE_PROBING */
	{
		GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_IN_NOTIFYING_OR_PROBING,	/* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION */
		GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION_IN_PROBING, /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION */
		GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_IN_PROBING,                /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION,               /* GROUP_MANAGER_STATION_EVENT_TRAINING_FAILED */
		GROUP_MANAGER_STATION_EVENT_ADD_GROUP_UPDATE_STATION_BITMAP,        /* GROUP_MANAGER_STATION_EVENT_NOTIFICATION_STARTED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION,               /* GROUP_MANAGER_STATION_EVENT_PROBING_STARTED */
		GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP_IN_PROBING,				/* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,               /* GROUP_MANAGER_STATION_EVENT_TIMER */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,               /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_START_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,               /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_RESULTS_RECEIVED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,               /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,               /* GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION */
	},	
	
	/* GROUP_MANAGER_STATION_STATE_TEMP_ACTIVE */
	{		
		GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_IN_ACTIVE_OR_TEMP_ACTIVE, /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION */
		GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_IN_TEMP_ACTIVE,             /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION */
		GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_IN_TEMP_ACTIVE,		     /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION,                /* GROUP_MANAGER_STATION_EVENT_TRAINING_FAILED */
		GROUP_MANAGER_STATION_EVENT_ADD_GROUP_UPDATE_STATION_BITMAP,         /* GROUP_MANAGER_STATION_EVENT_NOTIFICATION_STARTED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION,                /* GROUP_MANAGER_STATION_EVENT_PROBING_STARTED */
		GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP_IN_TEMP_ACTIVE,		     /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                /* GROUP_MANAGER_STATION_EVENT_TIMER */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_START_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_RESULTS_RECEIVED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                /* GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION */
	},
	
	/* GROUP_MANAGER_STATION_STATE_ACTIVE */
	{	
		GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_IN_ACTIVE_OR_TEMP_ACTIVE, /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION */
	    GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION_IN_ACTIVE,   /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION */
	    GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_IN_ACTIVE,				     /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION,                /* GROUP_MANAGER_STATION_EVENT_TRAINING_FAILED */
		GROUP_MANAGER_STATION_EVENT_ADD_GROUP_UPDATE_STATION_BITMAP,		 /* GROUP_MANAGER_STATION_EVENT_NOTIFICATION_STARTED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION,                /* GROUP_MANAGER_STATION_EVENT_PROBING_STARTED */
		GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP_IN_ACTIVE,			     /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                /* GROUP_MANAGER_STATION_EVENT_TIMER */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_START_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_RESULTS */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                /* GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION */
	},	

	/* GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_PENDING */
	{	
	    GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_IN_SINGLE_USER_VALIDATION_PENDING,	           /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION */
	    GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION_IN_SINGLE_USER_VALIDATION_PENDING, /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION */
	    GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_IN_SINGLE_USER_VALIDATION_PENDING,				   /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION,                                      /* GROUP_MANAGER_STATION_EVENT_TRAINING_FAILED */
		GROUP_MANAGER_STATION_EVENT_ADD_GROUP_UPDATE_STATION_BITMAP,		                       /* GROUP_MANAGER_STATION_EVENT_NOTIFICATION_STARTED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION,                                      /* GROUP_MANAGER_STATION_EVENT_PROBING_STARTED */
		GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP_IN_SINGLE_USER_VALIDATION_PENDING,			   /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TIMER_IN_SINGLE_USER_VALIDATION_PENDING,                       /* GROUP_MANAGER_STATION_EVENT_TIMER */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                      /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_START_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                      /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_RESULTS */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                      /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                      /* GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION */
	},

	/* GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_WAITING_FOR_LINK_ADAPTATION */
	{	
	    GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_IN_SINGLE_USER_VALIDATION_WAITING_FOR_LINK_ADAPTATION,                            /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION */
		GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION_IN_SINGLE_USER_VALIDATION_WAITING_FOR_LINK_ADAPTATION,               /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION */
		GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_IN_SINGLE_USER_VALIDATION_WAITING_FOR_LINK_ADAPTATION,				                 /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION,                                                                        /* GROUP_MANAGER_STATION_EVENT_TRAINING_FAILED */
		GROUP_MANAGER_STATION_EVENT_ADD_GROUP_UPDATE_STATION_BITMAP,		                                                         /* GROUP_MANAGER_STATION_EVENT_NOTIFICATION_STARTED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION,                                                                        /* GROUP_MANAGER_STATION_EVENT_PROBING_STARTED */
		GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP_IN_SINGLE_USER_VALIDATION_WAITING_FOR_LINK_ADAPTATION,			                 /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                                                        /* GROUP_MANAGER_STATION_EVENT_TIMER */
		GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_START_CONFIRMATION_IN_SINGLE_USER_VALIDATION_WAITING_FOR_LINK_ADAPTATION, /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_START_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                                                        /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_RESULTS_RECEIVED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                                                        /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                                                        /* GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION */
	},	

    /* GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION */
	{
	    GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_IN_SINGLE_USER_VALIDATION,				           /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION */
		GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION_IN_SINGLE_USER_VALIDATION,	           /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION */
		GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_IN_SINGLE_USER_VALIDATION, 				           /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION,                                          /* GROUP_MANAGER_STATION_EVENT_TRAINING_FAILED */
		GROUP_MANAGER_STATION_EVENT_ADD_GROUP_UPDATE_STATION_BITMAP, 		                           /* GROUP_MANAGER_STATION_EVENT_NOTIFICATION_STARTED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION,                                          /* GROUP_MANAGER_STATION_EVENT_PROBING_STARTED */
		GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP_IN_SINGLE_USER_VALIDATION,				           /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TIMER_IN_SINGLE_USER_VALIDATION,                                   /* GROUP_MANAGER_STATION_EVENT_TIMER */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION, 								           /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_START_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_RESULTS_RECEIVED_IN_SINGLE_USER_VALIDATION, /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_RESULTS_RECEIVED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                          /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                          /* GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION */
	},

	/* GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_FAILED */
	{	
	    GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_AFTER_SINGLE_USER_VALIDATION_ENDED,			              /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION */
		GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION_IN_SINGLE_USER_VALIDATION_ENDED,              /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION */
		GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_IN_SINGLE_USER_VALIDATION_ENDED,				              /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION,                                                 /* GROUP_MANAGER_STATION_EVENT_TRAINING_FAILED */
		GROUP_MANAGER_STATION_EVENT_ADD_GROUP_UPDATE_STATION_BITMAP,					                      /* GROUP_MANAGER_STATION_EVENT_NOTIFICATION_STARTED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION,                                                 /* GROUP_MANAGER_STATION_EVENT_PROBING_STARTED */
		GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP_IN_SINGLE_USER_VALIDATION_ENDED,						      /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                                 /* GROUP_MANAGER_STATION_EVENT_TIMER */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION, 								                  /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_START_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                                 /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_RESULTS_RECEIVED */
		GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION_IN_SINGLE_USER_VALIDATION_FAILED, /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                                 /* GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION */
	},

	/* GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_SUCCEEDED */
	{	
	    GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_AFTER_SINGLE_USER_VALIDATION_ENDED,				             /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION */
		GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION_IN_SINGLE_USER_VALIDATION_ENDED,	             /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION */
		GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_IN_SINGLE_USER_VALIDATION_ENDED, 				                 /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION,                                                    /* GROUP_MANAGER_STATION_EVENT_TRAINING_FAILED */
		GROUP_MANAGER_STATION_EVENT_ADD_GROUP_UPDATE_STATION_BITMAP, 					                         /* GROUP_MANAGER_STATION_EVENT_NOTIFICATION_STARTED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION,                                                    /* GROUP_MANAGER_STATION_EVENT_PROBING_STARTED */
		GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP_IN_SINGLE_USER_VALIDATION_ENDED,						         /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                                    /* GROUP_MANAGER_STATION_EVENT_TIMER */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION, 								                     /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_START_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION,                                                    /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_RESULTS_RECEIVED */
		GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION_IN_SINGLE_USER_VALIDATION_SUCCEEDED, /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                                    /* GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION */
	},
	
	/* GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_CANCELED_GROUP_REMOVED */
	{	
	    GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_AFTER_SINGLE_USER_VALIDATION_ENDED,			                              /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION */
	    GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_UPDATE_STATION_BITMAP,                                                       /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION */
	    GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_UPDATE_STATION_BITMAP, 				                                      /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION,                                                                 /* GROUP_MANAGER_STATION_EVENT_TRAINING_FAILED */
		GROUP_MANAGER_STATION_EVENT_ADD_GROUP_UPDATE_STATION_BITMAP, 		                                                  /* GROUP_MANAGER_STATION_EVENT_NOTIFICATION_STARTED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION,                                                                 /* GROUP_MANAGER_STATION_EVENT_PROBING_STARTED */
		GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP_IN_SINGLE_USER_VALIDATION_CANCELED, 						                  /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                                                 /* GROUP_MANAGER_STATION_EVENT_TIMER */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION, 								                                  /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_START_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION,                                                                 /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_RESULTS_RECEIVED */
		GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION_IN_SINGLE_USER_VALIDATION_CANCELED_GROUP_REMOVED, /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                                                 /* GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION */
	},

	/* GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_CANCELED_GROUP_FAILED */
	{		
	    GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_AFTER_SINGLE_USER_VALIDATION_ENDED,			                             /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION */
	    GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_UPDATE_STATION_BITMAP,                                                      /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION */
	    GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_UPDATE_STATION_BITMAP, 				                                     /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION,                                                                /* GROUP_MANAGER_STATION_EVENT_TRAINING_FAILED */
		GROUP_MANAGER_STATION_EVENT_ADD_GROUP_UPDATE_STATION_BITMAP, 		                                                 /* GROUP_MANAGER_STATION_EVENT_NOTIFICATION_STARTED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION,                                                                /* GROUP_MANAGER_STATION_EVENT_PROBING_STARTED */
		GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP_IN_SINGLE_USER_VALIDATION_CANCELED, 						                 /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                                                /* GROUP_MANAGER_STATION_EVENT_TIMER */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION, 								                                 /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_START_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION,                                                                /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_RESULTS_RECEIVED */
		GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION_IN_SINGLE_USER_VALIDATION_CANCELED_GROUP_FAILED, /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                                                /* GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION */
	},	
	
	/* GROUP_MANAGER_STATION_STATE_HALTING_WAITING_FOR_TRAINING_CONFIRMATION */
	{		
	    GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_IN_HALTING_WAITING_FOR_TRAINING_CONFIRMATION,        /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                           /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,	                                        /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                           /* GROUP_MANAGER_STATION_EVENT_TRAINING_FAILED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION, 		                                    /* GROUP_MANAGER_STATION_EVENT_NOTIFICATION_STARTED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                           /* GROUP_MANAGER_STATION_EVENT_PROBING_STARTED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION, 						                    /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                           /* GROUP_MANAGER_STATION_EVENT_TIMER */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION, 								            /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_START_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                           /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_RESULTS_RECEIVED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                           /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION_IN_HALTING_WAITING_FOR_TRAINING_CONFIRMATION, /* GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION */
	},
	
	/* GROUP_MANAGER_STATION_STATE_HALT */
	{	
	    GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_IN_HALT,	  /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION, /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION, /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION, /* GROUP_MANAGER_STATION_EVENT_TRAINING_FAILED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION, /* GROUP_MANAGER_STATION_EVENT_NOTIFICATION_STARTED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION, /* GROUP_MANAGER_STATION_EVENT_PROBING_STARTED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION, /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TIMER_IN_HALT,            /* GROUP_MANAGER_STATION_EVENT_TIMER */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION, /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_START_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION, /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_RESULTS_RECEIVED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION, /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION, /* GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION */
	},
	
	/* GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_LINK_ADAPTATION_AND_TRAINING_CONFIRMATION */
	{	
	    GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,	                                                                               /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION */
		GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_UPDATE_STATION_BITMAP,                                                                        /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION */
		GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_UPDATE_STATION_BITMAP,                                                                        /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                                                                  /* GROUP_MANAGER_STATION_EVENT_TRAINING_FAILED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION, 																				   /* GROUP_MANAGER_STATION_EVENT_NOTIFICATION_STARTED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                                                                  /* GROUP_MANAGER_STATION_EVENT_PROBING_STARTED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                                                                  /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                                                                  /* GROUP_MANAGER_STATION_EVENT_TIMER */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION,                                                                                  /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_START_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION,                                                                                  /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_RESULTS_RECEIVED */
		GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION_IN_REMOVING_WAITING_FOR_LINK_ADAPTATION_AND_TRAINING_CONFIRMATION, /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION_IN_REMOVING_WAITING_FOR_LINK_ADAPTATION_AND_TRAINING_CONFIRMATION,                   /* GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION */
	},	
	
	/* GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_TRAINING_CONFIRMATION */
	{			
	    GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,	                                         /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION */
		GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_UPDATE_STATION_BITMAP,                                  /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION */
		GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_UPDATE_STATION_BITMAP, 			                     /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                            /* GROUP_MANAGER_STATION_EVENT_TRAINING_FAILED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION, 										     /* GROUP_MANAGER_STATION_EVENT_NOTIFICATION_STARTED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                            /* GROUP_MANAGER_STATION_EVENT_PROBING_STARTED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                            /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                            /* GROUP_MANAGER_STATION_EVENT_TIMER */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                            /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_START_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                            /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_RESULTS_RECEIVED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                            /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION_IN_REMOVING_WAITING_FOR_TRAINING_CONFIRMATION, /* GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION */		
	},	
	
	/* GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_LINK_ADAPTATION_CONFIRMATION */
	{					
	    GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,	                                                                  /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION */
		GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_UPDATE_STATION_BITMAP,			                                                  /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION */
		GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_UPDATE_STATION_BITMAP,							                                  /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                                                     /* GROUP_MANAGER_STATION_EVENT_TRAINING_FAILED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION, 												                      /* GROUP_MANAGER_STATION_EVENT_NOTIFICATION_STARTED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                                                     /* GROUP_MANAGER_STATION_EVENT_PROBING_STARTED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                                                     /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                                                     /* GROUP_MANAGER_STATION_EVENT_TIMER */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION,                                                                     /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_START_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION,                                                                     /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_RESULTS_RECEIVED */
		GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION_IN_REMOVING_WAITING_FOR_LINK_ADAPTATION_CONFIRMATION, /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,	                                                                  /* GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION */		
	},	
	
	/* GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_ALL_GROUP_REMOVAL */
	{	
	    GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,	                             /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION */
	    GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_IN_REMOVING_WAITING_FOR_ALL_GROUPS_REMOVAL, /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION */
	    GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_IN_REMOVING_WAITING_FOR_ALL_GROUPS_REMOVAL, /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP */	
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                /* GROUP_MANAGER_STATION_EVENT_TRAINING_FAILED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION, 								 /* GROUP_MANAGER_STATION_EVENT_NOTIFICATION_STARTED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                /* GROUP_MANAGER_STATION_EVENT_PROBING_STARTED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                /* GROUP_MANAGER_STATION_EVENT_TIMER */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_START_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_RESULTS_RECEIVED */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,                                /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION */
		GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION,	                             /* GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION */		
	},		
};

static const GroupManagerStationGeneralFunc GroupManagerStaionEventToStateFunctionsTable[GROUP_MANAGER_STATION_NUM_OF_EVENTS_TO_STATES_FUNCTIONS] = 
{
	/* GROUP_MANAGER_STATION_STATE_IDLE */

	/* GROUP_MANAGER_STATION_STATE_NOT_ALLOWED */
	groupManagerStationEventRemoveStationInNotAllowed, /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_IN_NOT_ALLOWED */

	/* GROUP_MANAGER_STATION_STATE_SINGLE_USER */
	groupManagerStationEventRemoveStationInSingleUser,       /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_IN_SINGLE_USER */
	groupManagerStationEventTrainingFailedInSingleUser,      /* GROUP_MANAGER_STATION_EVENT_TRAINING_FAILED_IN_SINGLE_USER */
	groupManagerStationEventNotificationStartedInSingleUser, /* GROUP_MANAGER_STATION_EVENT_NOTIFICATION_STARTED_IN_SINGLE_USER */
	
	/* GROUP_MANAGER_STATION_STATE_NOTIFY_STATION */ 
	groupManagerStationEventRemoveStationInNotifyingOrProbing,  /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_IN_NOTIFYING_OR_PROBING */
	groupManagerStationEventRemoveGroupDueToStationInNotifying, /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION_IN_NOTIFYING */
	groupManagerStationEventRemoveGroupInNotifying,             /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_IN_NOTIFYING */
	groupManagerStationEventProbingStartedInNotifying,          /* GROUP_MANAGER_STATION_EVENT_PROBING_STARTED_IN_NOTIFYING */

	/* GROUP_MANAGER_STATION_STATE_PROBING */       
	groupManagerStationEventRemoveGroupDueToStationInProbing, /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION_IN_PROBING */
	groupManagerStationEventRemoveGroupInProbing,             /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_IN_PROBING */
	groupManagerStationEventActivateGroupInProbing,           /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP_IN_PROBING */

	/* GROUP_MANAGER_STATION_STATE_TEMP_ACTIVE */
	groupManagerStationEventRemoveStationInActiveOrTempActive,   /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_IN_ACTIVE_OR_TEMP_ACTIVE */
	groupManagerStationEventRemoveGroupInTempActive,             /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_IN_TEMP_ACTIVE */				           
	groupManagerStationEventActivateGroupInTempActive,           /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP_IN_TEMP_ACTIVE */

	/* GROUP_MANAGER_STATION_STATE_ACTIVE */
	groupManagerStationEventRemoveGroupDueToStationInActive, /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION_IN_ACTIVE */
	groupManagerStationEventRemoveGroupInActive,             /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_IN_ACTIVE */
	groupManagerStationEventActivateGroupInActive,			 /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP_IN_ACTIVE */
	
    /* GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_PENDING */
	groupManagerStationEventRemoveStationInSingleUserValidationPending,           /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_IN_SINGLE_USER_VALIDATION_PENDING */
	groupManagerStationEventRemoveGroupDueToStationInSingleUserValidationPending, /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION_IN_SINGLE_USER_VALIDATION_PENDING */
	groupManagerStationEventRemoveGroupInSingleUserValidationPending,             /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_IN_SINGLE_USER_VALIDATION_PENDING */
	groupManagerStationEventActivateGroupInSingleUserValidationPending,			  /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP_IN_SINGLE_USER_VALIDATION_PENDING*/
	groupManagerStationEventTimerInSingleUserValidationPending,	                  /* GROUP_MANAGER_STATION_EVENT_TIMER_IN_SINGLE_USER_VALIDATION_PENDING */
	
	/* GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_WAITING_FOR_LINK_ADAPTATION */
	groupManagerStationEventRemoveStationInSuValWaitingLa,                         /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_IN_SINGLE_USER_VALIDATION_WAITING_FOR_LINK_ADAPTATION */                         
	groupManagerStationEventRemoveGroupDueToStationInSuValWaitingLa,               /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION_IN_SINGLE_USER_VALIDATION_WAITING_FOR_LINK_ADAPTATION */
	groupManagerStationEventRemoveGroupInSuValWaitingLa,                           /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_IN_SINGLE_USER_VALIDATION_WAITING_FOR_LINK_ADAPTATION */
	groupManagerStationEventActivateGroupInSuValWaitingLa,			              /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP */
	groupManagerStationEventSuValStartCfmInSuValWaitingLa, /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_START_CONFIRMATION_IN_SINGLE_USER_VALIDATION_WAITING_FOR_LINK_ADAPTATION */

	/* GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION */
	groupManagerStationEventRemoveStationInSingleUserValidation,	                   /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_IN_SINGLE_USER_VALIDATION */
	groupManagerStationEventRemoveGroupDueToStationInSingleUserValidation,             /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION_IN_SINGLE_USER_VALIDATION */
	groupManagerStationEventRemoveGroupInSingleUserValidation,                         /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_IN_SINGLE_USER_VALIDATION */
	groupManagerStationEventActivateGroupInSingleUserValidation,		               /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP */
	groupManagerStationEventTimerInSingleUserValidation,	                           /* GROUP_MANAGER_STATION_EVENT_TIMER_IN_SINGLE_USER_VALIDATION */
	groupManagerStationEventSuValResultsReceivedInSuVal, /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_RESULTS_RECEIVED_IN_SINGLE_USER_VALIDATION */

	/* GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_FAILED */
	groupManagerStationEventRemoveStationAfterSuValEnded,                     /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_AFTER_SINGLE_USER_VALIDATION_ENDED */
	groupManagerStationEventRemoveGroupDueToStationInSingleUserValidationEnded,              /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION_IN_SINGLE_USER_VALIDATION_ENDED */
	groupManagerStationEventRemoveGroupInSingleUserValidationEnded,                          /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_IN_SINGLE_USER_VALIDATION_ENDED */
	groupManagerStationEventActivateGroupInSingleUserValidationEnded,                        /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP_IN_SINGLE_USER_VALIDATION_ENDED */
	groupManagerStationEventSuValEndCfmInSuValFailed, /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION_IN_SINGLE_USER_VALIDATION_FAILED */

	/* GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_SUCCEEDED */
	groupManagerStationEventSuValEndCfmInSuValSucceeded, /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION_IN_SINGLE_USER_VALIDATION_SUCCEEDED */
	
    /* GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_CANCELED_GROUP_REMOVED */
    groupManagerStationEventActivateGroupInSingleUserValidationCanceled,                                   /* GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP_IN_SINGLE_USER_VALIDATION_CANCELED */
	groupManagerStationEventSuValEndCfmInSuValCanceledGroupRemoved, /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION_IN_SINGLE_USER_VALIDATION_CANCELED_GROUP_REMOVED */
	
    /* GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_CANCELED_GROUP_FAILED */
	groupManagerStationEventSuValEndCfmInSuValCanceledGroupFailed, /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION_IN_SINGLE_USER_VALIDATION_CANCELED_GROUP_FAILED */

    /* GROUP_MANAGER_STATION_STATE_HALTING_WAITING_FOR_TRAINING_CONFIRMATION */
	groupManagerStationEventRemoveStationInHaltingWaitingForTrainingCfm,        /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_IN_HALTING_WAITING_FOR_TRAINING_CONFIRMATION */
	groupManagerStationEventTrainingCfmInHaltingWaitingForTrainingCfm, /* GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION_IN_HALTING_WAITING_FOR_TRAINING_CONFIRMATION  */

	/* GROUP_MANAGER_STATION_STATE_HALT */
	groupManagerStationEventRemoveStationInHalt, /* GROUP_MANAGER_STATION_EVENT_REMOVE_STATION_IN_HALT */
	groupManagerStationEventTimerInHalt,         /* GROUP_MANAGER_STATION_EVENT_TIMER_IN_HALT */
	
    /* GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_LINK_ADAPTATION_AND_TRAINING_CONFIRMATION */
	groupManagerStationEventSuValEndCfmInRemovingWaitingForTrainingAndLaCfm, /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION_IN_REMOVING_WAITING_FOR_LINK_ADAPTATION_AND_TRAINING_CONFIRMATION */
	groupManagerStationEventTrainingCfmInRemovingWaitingForTrainingAndLaCfm, /* GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION_IN_REMOVING_WAITING_FOR_LINK_ADAPTATION_AND_TRAINING_CONFIRMATION */

	/* GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_TRAINING_CONFIRMATION */
	groupManagerStationEventTrainingCfmInRemovingWaitingForTrainingCfm, /* GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION_IN_REMOVING_WAITING_FOR_TRAINING_CONFIRMATION */

	/* GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_LINK_ADAPTATION_CONFIRMATION */
	groupManagerStationEventSuValEndCfmInRemovingWaitingForLaCfm, /* GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION_IN_REMOVING_WAITING_FOR_LINK_ADAPTATION_CONFIRMATION */
	

	/* GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_ALL_GROUP_REMOVAL */
	groupManagerStationEventRemoveGroupInRemovingWaitingForAllGroupRemoval, /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_IN_REMOVING_WAITING_FOR_ALL_GROUPS_REMOVAL */

	/* General Functions */
	groupManagerStationEventToStateAssert,                  /* GROUP_MANAGER_STATION_EVENT_TO_STATE_ASSERT_FUNCTION */
	groupManagerStationEventToStateIgnore,                  /* GROUP_MANAGER_STATION_EVENT_TO_STATE_IGNORE_FUNCTION */
	groupManagerStationEventRemoveGroupUpdateStationBitmap, /* GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_UPDATE_STATION_BITMAP */
	groupManagerStationEventAddGroupUpdateStationBitmap,    /* GROUP_MANAGER_STATION_EVENT_ADD_GROUP_UPDATE_STATION_BITMAP */
};


static const GroupManagerGroupEventToStateFunction_e GroupManagerGroupStateMachine[GROUP_MANAGER_NUM_OF_GROUP_STATES][GROUP_MANAGER_NUM_OF_GROUP_EVENTS] = 
{
	/* GROUP_MANAGER_GROUP_STATE_IDLE */
	{
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION, /* GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION, /* GROUP_MANAGER_GROUP_EVENT_NOTIFICATION_HAS_BEEN_CONFIRMED */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION, /* GROUP_MANAGER_GROUP_EVENT_TIMER */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION, /* GROUP_MANAGER_GROUP_EVENT_ADD_GROUP_CONFIRMATION */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION, /* GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP_CONFIRMATION */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION, /* GROUP_MANAGER_GROUP_EVENT_PROBING_RESULTS_READY */
	    GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION, /* GROUP_MANAGER_GROUP_EVENT_GROUP_FAILED_INDICATION */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION, /* GROUP_MANAGER_GROUP_EVENT_LOCK_CONFIRMATION */	
	},

	/* GROUP_MANAGER_GROUP_STATE_PROBING_NOTIFYING */
	{	
		GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP_IN_PROBING_NOTIFYING,				    /* GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP */
		GROUP_MANAGER_GROUP_EVENT_NOTIFICATION_HAS_BEEN_CONFIRMED_IN_PROBING_NOTIFYING, /* GROUP_MANAGER_GROUP_EVENT_NOTIFICATION_HAS_BEEN_CONFIRMED */
		GROUP_MANAGER_GROUP_EVENT_TIMER_IN_PROBING_NOTIFYING,                           /* GROUP_MANAGER_GROUP_EVENT_TIMER */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,                             /* GROUP_MANAGER_GROUP_EVENT_ADD_GROUP_CONFIRMATION */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,                             /* GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP_CONFIRMATION */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,                             /* GROUP_MANAGER_GROUP_EVENT_PROBING_RESULTS_READY */
	    GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,                             /* GROUP_MANAGER_GROUP_EVENT_GROUP_FAILED_INDICATION */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,                             /* GROUP_MANAGER_GROUP_EVENT_LOCK_CONFIRMATION */	
	},
	
	/* GROUP_MANAGER_GROUP_STATE_PROBING_WAITING_FOR_LINK_ADAPTATION */
	{	
		GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP_IN_PROBING_WAITING_FOR_LINK_ADAPTATION,			 /* GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,                                      /* GROUP_MANAGER_GROUP_EVENT_NOTIFICATION_HAS_BEEN_CONFIRMED */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,                                      /* GROUP_MANAGER_GROUP_EVENT_TIMER */
		GROUP_MANAGER_GROUP_EVENT_ADD_GROUP_CONFIRMATION_IN_PROBING_WAITING_FOR_LINK_ADAPTATION, /* GROUP_MANAGER_GROUP_EVENT_ADD_GROUP_CONFIRMATION */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,                                      /* GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP_CONFIRMATION */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,                                      /* GROUP_MANAGER_GROUP_EVENT_PROBING_RESULTS_READY */
	    GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,                                      /* GROUP_MANAGER_GROUP_EVENT_GROUP_FAILED_INDICATION */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,                                      /* GROUP_MANAGER_GROUP_EVENT_LOCK_CONFIRMATION */	
	},
		
	/* GROUP_MANAGER_GROUP_STATE_PROBING */
	{	
		GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP_IN_PROBING,		    /* GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,         /* GROUP_MANAGER_GROUP_EVENT_NOTIFICATION_HAS_BEEN_CONFIRMED */
		GROUP_MANAGER_GROUP_EVENT_TIMER_IN_PROBING,                 /* GROUP_MANAGER_GROUP_EVENT_TIMER */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,         /* GROUP_MANAGER_GROUP_EVENT_ADD_GROUP_CONFIRMATION */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,         /* GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP_CONFIRMATION */
		GROUP_MANAGER_GROUP_EVENT_PROBING_RESULTS_READY_IN_PROBING,	/* GROUP_MANAGER_GROUP_EVENT_PROBING_RESULTS_READY */
	    GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,         /* GROUP_MANAGER_GROUP_EVENT_GROUP_FAILED_INDICATION */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,         /* GROUP_MANAGER_GROUP_EVENT_LOCK_CONFIRMATION */	
	},
	
	/* GROUP_MANAGER_GROUP_STATE_ACTIVE */
	{	
		GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP_IN_ACTIVE,		     /* GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,          /* GROUP_MANAGER_GROUP_EVENT_NOTIFICATION_HAS_BEEN_CONFIRMED */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,          /* GROUP_MANAGER_GROUP_EVENT_TIMER */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,          /* GROUP_MANAGER_GROUP_EVENT_ADD_GROUP_CONFIRMATION */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,          /* GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP_CONFIRMATION */
		GROUP_MANAGER_GROUP_EVENT_PROBING_RESULTS_READY_IN_ACTIVE,	/* GROUP_MANAGER_GROUP_EVENT_PROBING_RESULTS_READY */
	    GROUP_MANAGER_GROUP_EVENT_GROUP_FAILED_INDICATION_IN_ACTIVE, /* GROUP_MANAGER_GROUP_EVENT_GROUP_FAILED_INDICATION */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,          /* GROUP_MANAGER_GROUP_EVENT_LOCK_CONFIRMATION */	
	},
	
	/* GROUP_MANAGER_GROUP_STATE_REMOVING_WAITING_FOR_NOTIFICATIONS */
	{	
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_IGNORE_FUNCTION,		                                         /* GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP */
		GROUP_MANAGER_GROUP_EVENT_NOTIFICATION_HAS_BEEN_CONFIRMED_IN_REMOVING_WAITING_FOR_NOTIFICATIONS, /* GROUP_MANAGER_GROUP_EVENT_NOTIFICATION_HAS_BEEN_CONFIRMED */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,                                              /* GROUP_MANAGER_GROUP_EVENT_TIMER */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,                                              /* GROUP_MANAGER_GROUP_EVENT_ADD_GROUP_CONFIRMATION */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,                                              /* GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP_CONFIRMATION */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,	                                             /* GROUP_MANAGER_GROUP_EVENT_PROBING_RESULTS_READY */
	    GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION, 		                                     /* GROUP_MANAGER_GROUP_EVENT_GROUP_FAILED_INDICATION */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,                                              /* GROUP_MANAGER_GROUP_EVENT_LOCK_CONFIRMATION */	
	},
	
	/* GROUP_MANAGER_GROUP_STATE_REMOVING_WAITING_FOR_LOCK_AND_LINK_ADAPTATION_CONFIRMATIONS */
	{	
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_IGNORE_FUNCTION,		                                                            /* GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,                                                                 /* GROUP_MANAGER_GROUP_EVENT_NOTIFICATION_HAS_BEEN_CONFIRMED */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,                                                                 /* GROUP_MANAGER_GROUP_EVENT_TIMER */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,                                                                 /* GROUP_MANAGER_GROUP_EVENT_ADD_GROUP_CONFIRMATION */
		GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP_CONFIRMATION_IN_REMOVING_WAITING_FOR_LOCK_AND_LINK_ADAPTATION_CONFIRMATIONS, /* GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP_CONFIRMATION */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_IGNORE_FUNCTION,	                                                                /* GROUP_MANAGER_GROUP_EVENT_PROBING_RESULTS_READY */
	    GROUP_MANAGER_GROUP_EVENT_TO_STATE_IGNORE_FUNCTION, 		                                                        /* GROUP_MANAGER_GROUP_EVENT_GROUP_FAILED_INDICATION */
		GROUP_MANAGER_GROUP_EVENT_LOCK_CONFIRMATION_IN_REMOVING_WAITING_FOR_LOCK_AND_LINK_ADAPTATION_CONFIRMATIONS,         /* GROUP_MANAGER_GROUP_EVENT_LOCK_CONFIRMATION */
	},
	
	/* GROUP_MANAGER_GROUP_STATE_REMOVING_WAITING_FOR_LOCK_CONFIRMATION */
	{	
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_IGNORE_FUNCTION,		                               /* GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,                                    /* GROUP_MANAGER_GROUP_EVENT_NOTIFICATION_HAS_BEEN_CONFIRMED */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,                                    /* GROUP_MANAGER_GROUP_EVENT_TIMER */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,                                    /* GROUP_MANAGER_GROUP_EVENT_ADD_GROUP_CONFIRMATION */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,                                    /* GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP_CONFIRMATION */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,	                                   /* GROUP_MANAGER_GROUP_EVENT_PROBING_RESULTS_READY */
	    GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION, 		                           /* GROUP_MANAGER_GROUP_EVENT_GROUP_FAILED_INDICATION */
		GROUP_MANAGER_GROUP_EVENT_LOCK_CONFIRMATION_IN_REMOVING_WAITING_FOR_LOCK_CONFIRMATION, /* GROUP_MANAGER_GROUP_EVENT_LOCK_CONFIRMATION */
	},
	
	/* GROUP_MANAGER_GROUP_STATE_REMOVING_WAITING_FOR_LINK_ADAPTATION_CONFIRMATION */
	{	
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_IGNORE_FUNCTION,		                                                   /* GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,                                                        /* GROUP_MANAGER_GROUP_EVENT_NOTIFICATION_HAS_BEEN_CONFIRMED */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,                                                        /* GROUP_MANAGER_GROUP_EVENT_TIMER */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_IGNORE_FUNCTION,                                                        /* GROUP_MANAGER_GROUP_EVENT_ADD_GROUP_CONFIRMATION */
		GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP_CONFIRMATION_IN_REMOVING_WAITING_FOR_LINK_ADAPTATION_CONFIRMATIONS, /* GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP_CONFIRMATION */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_IGNORE_FUNCTION,	                                                       /* GROUP_MANAGER_GROUP_EVENT_PROBING_RESULTS_READY */
	    GROUP_MANAGER_GROUP_EVENT_TO_STATE_IGNORE_FUNCTION, 		                                               /* GROUP_MANAGER_GROUP_EVENT_GROUP_FAILED_INDICATION */
		GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION,                                                        /* GROUP_MANAGER_GROUP_EVENT_LOCK_CONFIRMATION */
	},	
};
	
static const GroupManagerGroupGeneralFunc GroupManagerGroupEventToStateFunctionsTable[GROUP_MANAGER_GROUP_NUM_OF_EVENTS_TO_STATES_FUNCTIONS] = 
{       	
	/* GROUP_MANAGER_GROUP_STATE_IDLE */

	/* GROUP_MANAGER_GROUP_STATE_PROBING_NOTIFYING */
	groupManagerGroupEventRemoveGroupInProbingNotifying,              /* GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP_IN_PROBING_NOTIFYING */
	groupManagerGroupEventNotificationCfmInProbingNotifying, /* GROUP_MANAGER_GROUP_EVENT_NOTIFICATION_HAS_BEEN_CONFIRMED_IN_PROBING_NOTIFYING */
	groupManagerGroupEventTimerInProbingNotifying,                    /* GROUP_MANAGER_GROUP_EVENT_TIMER_IN_PROBING_NOTIFYING */

	/* GROUP_MANAGER_GROUP_STATE_PROBING_WAITING_FOR_LINK_ADAPTATION */
	groupManagerGroupEventRemoveGroupInProbingWaitingForLinkAdaptation,          /* GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP_IN_PROBING_WAITING_FOR_LINK_ADAPTATION */
	groupManagerGroupEventAddGroupCfmInProbingWaitingForLa, /* GROUP_MANAGER_GROUP_EVENT_ADD_GROUP_CONFIRMATION_IN_PROBING_WAITING_FOR_LINK_ADAPTATION */

    /* GROUP_MANAGER_GROUP_STATE_PROBING */
	groupManagerGroupEventRemoveGroupInProbing,            /* GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP_IN_PROBING */
	groupManagerGroupEventTimerInProbing,                  /* GROUP_MANAGER_GROUP_EVENT_TIMER_IN_PROBING */
	groupManagerGroupEventProbingResultsAreReadyInProbing, /*GROUP_MANAGER_GROUP_EVENT_PROBING_RESULTS_READY_IN_PROBING */

	/* GROUP_MANAGER_GROUP_STATE_ACTIVE */
	groupManagerGroupEventRemoveGroupInActive,    /* GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP_IN_ACTIVE */
	groupManagerGroupEventPropingResultsReadyInActive, /* GROUP_MANAGER_GROUP_EVENT_PROBING_RESULTS_READY_IN_ACTIVE */
	groupManagerGroupEventGroupHasFailedInActive, /* GROUP_MANAGER_GROUP_EVENT_GROUP_FAILED_INDICATION_IN_ACTIVE */

	/* GROUP_MANAGER_GROUP_STATE_REMOVING_WAITING_FOR_NOTIFICATIONS */
	groupManagerGroupEventNotificationCfmInRemovingWaitingForNotifications, /* GROUP_MANAGER_GROUP_EVENT_NOTIFICATION_HAS_BEEN_CONFIRMED_IN_REMOVING_WAITING_FOR_NOTIFICATIONS */

	/* GROUP_MANAGER_GROUP_STATE_REMOVING_WAITING_FOR_LOCK_AND_LINK_ADAPTATION_CONFIRMATIONS */
	groupManagerGroupEventRemoveGroupCfmInRemovingWaitingForLockAndLaCfms, /* GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP_CONFIRMATION_IN_REMOVING_WAITING_FOR_LOCK_AND_LINK_ADAPTATION_CONFIRMATIONS*/
	groupManagerGroupEventLockCfmInRemovingWaitingForLockAndLaCfm,        /* GROUP_MANAGER_GROUP_EVENT_LOCK_CONFIRMATION_IN_REMOVING_WAITING_FOR_LOCK_AND_LINK_ADAPTATION_CONFIRMATIONS */
	
	/* GROUP_MANAGER_GROUP_STATE_REMOVING_WAITING_FOR_LOCK_CONFIRMATION */
	groupManagerGroupEventLockConfirmationInRemovingWaitingForLockConfirmation, /* GROUP_MANAGER_GROUP_EVENT_LOCK_CONFIRMATION_IN_REMOVING_WAITING_FOR_LOCK_CONFIRMATION */
	
	/* GROUP_MANAGER_GROUP_STATE_REMOVING_WAITING_FOR_LINK_ADAPTATION_CONFIRMATION */
	groupManagerGroupEventRemoveGroupCfmInRemovingWaitingForLaCfm, /* GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP_CONFIRMATION_IN_REMOVING_WAITING_FOR_LINK_ADAPTATION_CONFIRMATIONS */
                 
	/* General Functions */
	groupManagerGroupEventToStateAssert, /* GROUP_MANAGER_GROUP_EVENT_TO_STATE_ASSERT_FUNCTION */
	groupManagerGroupEventToStateIgnore, /* GROUP_MANAGER_GROUP_EVENT_TO_STATE_IGNORE_FUNCTION */   
};


static const groupManagerStationPriorities_e GroupManagerStationStateToPriorityConversionTable[GROUP_MANAGER_NUM_OF_STATION_STATES] =
{
	GROUP_MANAGER_STATION_PRIORITY_NONE,		/*GROUP_MANAGER_STATION_STATE_IDLE */
	GROUP_MANAGER_STATION_PRIORITY_NONE,		/*GROUP_MANAGER_STATION_STATE_NOT_ALLOWED */
	GROUP_MANAGER_STATION_PRIORITY_LOW, 		/*GROUP_MANAGER_STATION_STATE_SINGLE_USER */
	GROUP_MANAGER_STATION_PRIORITY_MID, 		/*GROUP_MANAGER_STATION_STATE_NOTIFY_STATION */
	GROUP_MANAGER_STATION_PRIORITY_MID, 		/*GROUP_MANAGER_STATION_STATE_PROBING */
	GROUP_MANAGER_STATION_PRIORITY_HIGH,		/*GROUP_MANAGER_STATION_STATE_TEMP_ACTIVE */
	GROUP_MANAGER_STATION_PRIORITY_HIGH,		/*GROUP_MANAGER_STATION_STATE_ACTIVE */
	GROUP_MANAGER_STATION_PRIORITY_HIGH,		/*GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_PENDING */
	GROUP_MANAGER_STATION_PRIORITY_HIGH,		/*GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_WAITING_FOR_LINK_ADAPTATION */
	GROUP_MANAGER_STATION_PRIORITY_HIGH,		/*GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION */
	GROUP_MANAGER_STATION_PRIORITY_HIGH,		/*GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_FAILED */
	GROUP_MANAGER_STATION_PRIORITY_HIGH,		/*GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_SUCCEEDED */
	GROUP_MANAGER_STATION_PRIORITY_HIGH,		/*GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_CANCELED_GROUP_REMOVED */
	GROUP_MANAGER_STATION_PRIORITY_HIGH,		/*GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_CANCELED_GROUP_FAILED */
	GROUP_MANAGER_STATION_PRIORITY_NONE,		/*GROUP_MANAGER_STATION_STATE_HALTING_WAITING_FOR_TRAINING_CONFIRMATION */
	GROUP_MANAGER_STATION_PRIORITY_NONE,		/*GROUP_MANAGER_STATION_STATE_HALT */
	GROUP_MANAGER_STATION_PRIORITY_NONE,		/*GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_LINK_ADAPTATION_AND_TRAINING_CONFIRMATION */
	GROUP_MANAGER_STATION_PRIORITY_NONE,		/*GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_TRAINING_CONFIRMATION */
	GROUP_MANAGER_STATION_PRIORITY_NONE,		/*GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_LINK_ADAPTATION_CONFIRMATION */
	GROUP_MANAGER_STATION_PRIORITY_NONE,		/*GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_ALL_GROUP_REMOVAL */
};


/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

extern uint8 GroupDb_TxopEnable;

/*---------------------------------------------------------------------------------

/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

groupManagerAddStation 


Description:
------------
add a new station to the group manager. The station is added to index of Station id 
provided in the message

Input: 
-----
groupManagerMessage - pointer to the message that contains the station id 	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerAddStation(K_MSG* groupManagerMessage)
{
	BssManagerStaManagerReq_t *stationManagerRequest = NULL;
	K_MSG *originalAddStationMessage = NULL;
	UMI_STA_ADD *addStationMessageParameters = NULL;
	StaId stationIndex = 0;
	uint8 isVht = FALSE;
#ifdef ENET_INC_ARCH_WAVE600
	uint8 isHe = FALSE;
#endif //ENET_INC_ARCH_WAVE600
	sVHT_CAPABILITIES_INFO *vhtCapabilitiesInfo = NULL;
#ifdef ENET_INC_ARCH_WAVE600D2
	HE_MAC_PHY_CAPABILITIES_INFO *heMacPhyCapabilitiesInfo = NULL;
#endif
	GroupManagerStationParameters_t *groupManagerStationDataBaseEntry = NULL;

	stationManagerRequest = (BssManagerStaManagerReq_t *)pK_MSG_DATA(groupManagerMessage);
	originalAddStationMessage = stationManagerRequest->psMsg;
	addStationMessageParameters = (UMI_STA_ADD *)pK_MSG_DATA(originalAddStationMessage);
    stationIndex = addStationMessageParameters->u16SID;
	vhtCapabilitiesInfo = (sVHT_CAPABILITIES_INFO *)&addStationMessageParameters->u32VHT_Cap_Info;
	
   	groupManagerStationDataBaseEntry = &GroupManagerStationDb[stationIndex];
   
	/* Only in idle state this message is allowed */
	ASSERT(groupManagerStationDataBaseEntry->state == GROUP_MANAGER_STATION_STATE_IDLE); 

	/* Clear fields in station entry */
	groupManagerStationDataBaseEntry->tempPrimaryGroup = GROUP_MANAGER_INVALID_GROUP_INDEX;
	groupManagerStationDataBaseEntry->numberOfHalts = 0;
    groupManagerStationDataBaseEntry->virtualGroupsBitmap = 0;
	groupManagerStationDataBaseEntry->groupsBitmap = 0;
	groupManagerStationDataBaseEntry->primaryGroup = GROUP_MANAGER_INVALID_GROUP_INDEX;
	groupManagerStationDataBaseEntry->userPositionVector = 0;
	
	isVht = MTLK_BFIELD_GET(addStationMessageParameters->u8Flags, STA_ADD_FLAGS_IS_VHT); 
#ifdef ENET_INC_ARCH_WAVE600D2
	heMacPhyCapabilitiesInfo = (HE_MAC_PHY_CAPABILITIES_INFO *)addStationMessageParameters->u8HE_Mac_Phy_Cap_Info;
	/* update station DB supporting mbssRxControl capability */
	groupManagerStationDataBaseEntry->mbssRxControlSupport = heMacPhyCapabilitiesInfo->heMacCapInfo.HE_MAC_RX_CONTROL_FRAME_TO_MULTIBSS;
#endif

#ifdef ENET_INC_ARCH_WAVE600
	isHe = MTLK_BFIELD_GET(addStationMessageParameters->u8FlagsExt, STA_ADD_FLAGS_EXT_IS_HE); 

	/* Check if station is not HE but is VHT and support multi user */
	if(!isHe && isVht && vhtCapabilitiesInfo->mu_beamformee_capable)
#else
	if(isVht && vhtCapabilitiesInfo->mu_beamformee_capable)
#endif //ENET_INC_ARCH_WAVE600
	{
		/* For newly added station - set the lowest priority bit (never was tested)*/
#ifdef GROUP_MANAGER_DEBUG		
		ILOG0_D("groupManagerAddStation: Adding station %d", stationIndex);
#endif
		groupManagerChangeStationState(&GroupManagerStationDb[stationIndex], GROUP_MANAGER_STATION_STATE_SINGLE_USER);
		TrainingManagerAddStaReq(addStationMessageParameters);
		/* Note : no need to send "add station" to LA , it is notified by BSS manager*/
	}
	else
	{
#ifdef GROUP_MANAGER_DEBUG		
		ILOG0_D("groupManagerAddStation: station id %d is not allowed", stationIndex);
#endif
		groupManagerChangeStationState(&GroupManagerStationDb[stationIndex], GROUP_MANAGER_STATION_STATE_NOT_ALLOWED);
	}

	groupManagerSendConfirmToStationManager(stationIndex);
}	

/**********************************************************************************

groupManagerRemoveStation 


Description:
------------
remove a station from the group manager 

Input: 
-----
groupManagerMessage - pointer to the message that contains the station id 	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerRemoveStation(K_MSG* groupManagerMessage)
{
	BssManagerStaManagerReq_t *stationManagerRequest = NULL;
	K_MSG *originalRemoveStationMessage = NULL;
	UMI_STOP_TRAFFIC *stopTrafficMessageParameters = NULL;
	StaId stationIndex = 0;
	
	stationManagerRequest = (BssManagerStaManagerReq_t *)pK_MSG_DATA(groupManagerMessage);
	originalRemoveStationMessage = stationManagerRequest->psMsg;
	stopTrafficMessageParameters = (UMI_STOP_TRAFFIC *)pK_MSG_DATA(originalRemoveStationMessage);
	stationIndex = stopTrafficMessageParameters->u16SID;
#ifdef GROUP_MANAGER_DEBUG		
	ILOG0_D("groupManagerRemoveStation: received on SID %d", stationIndex);
#endif
	groupManagerRunStationStateMachine(stationIndex, GROUP_MANAGER_STATION_EVENT_REMOVE_STATION, NULL);
}

/**********************************************************************************

groupManagerTrainingTimer 


Description:
------------
performs the following actions:  
1. handle calendar wheel entries event
2. trigger training process
3. clean outdated virtual groups

Input: 
-----
groupManagerMessage - not used 	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerTrainingTimer(K_MSG* groupManagerMessage)
{
	uint16 currentIndex = CALENDAR_WHEEL_NULL_ENTRY_INDEX;
	uint16 nextIndex = CALENDAR_WHEEL_NULL_ENTRY_INDEX;
	GroupManagerStationParameters_t* stationCalendarWheelEntry = NULL;
	GroupManagerGroupParameters_t* groupCalendarWheelEntry = NULL;
	uint32 currentTimeStamp = 0;
	uint32 timeGap  = 0;
	uint32 tempBitmap = 0;
	uint8 virtualGroupIndex = 0;
	UNUSED_PARAM(groupManagerMessage);
	/* handle stations timer events first */
	/* currentIndex -  represents the first entry in current slot of stations calendar wheel*/	
	currentIndex = CalendarWheel_TimerEvent(&GroupManagerStationsCalendarWheel);
	/* loop over all entries in this slot and handle all events in the list */
	while(CALENDAR_WHEEL_NULL_ENTRY_INDEX != currentIndex)
	{
#ifdef GROUP_MANAGER_DEBUG				
		ILOG0_D("groupManagerTrainingTimer: stations timer event, handle index %d", currentIndex);
#endif
		/* take first entry in slot */
		stationCalendarWheelEntry = (GroupManagerStationParameters_t *)(GroupManagerStationDb + currentIndex);
		/* save next index in entries list per slot*/
		nextIndex = stationCalendarWheelEntry->calendarWheelEntry.nextIndex;
		groupManagerRunStationStateMachine(currentIndex, GROUP_MANAGER_STATION_EVENT_TIMER, NULL);
		currentIndex = nextIndex;
	}

	/* handle groups timer events */
	/* currentIndex -  represents the first entry in current slot of groups calendar wheel*/	
	currentIndex = CalendarWheel_TimerEvent(&GroupManagerGroupsCalendarWheel);
	
	/* loop over all entries in this slot and handle all events in the list */
	while(CALENDAR_WHEEL_NULL_ENTRY_INDEX != currentIndex)
	{
#ifdef GROUP_MANAGER_DEBUG				
		ILOG0_D("groupManagerTrainingTimer: groups timer event, handle index %d", currentIndex);
#endif

		/* take first entry in slot */
		groupCalendarWheelEntry = (GroupManagerGroupParameters_t *)(GroupManagerGroupDb + currentIndex);
		/* save next index in entries list per slot*/
		nextIndex = groupCalendarWheelEntry->calendarWheelEntry.nextIndex;
		groupManagerRunGroupStateMachine(currentIndex, GROUP_MANAGER_GROUP_EVENT_TIMER, NULL);
		currentIndex = nextIndex;
	}

	/* clean outdated virtual groups*/
	currentTimeStamp = GET_TSF_TIMER_LOW();
	
	tempBitmap = GroupManagerGlobalParametersDb.virtualGroupsBitmap;
	while(tempBitmap)
	{
		virtualGroupIndex = Utils_FindFirstSetAndClear(&tempBitmap);
		timeGap = currentTimeStamp - GroupManagerVirtualGroupDatabase[virtualGroupIndex].timeStamp;
		if(GROUP_MANAGER_VIRTUAL_GROUP_VALID_TIME_PERIOD < timeGap)
		{
			GroupManagerGlobalParametersDb.virtualGroupsBitmap &= ~(0x1 << virtualGroupIndex);
		}
	}

	/* trigger training process here after all other processing were ended */
	if(GROUP_MANAGER_GLOBAL_STATE_ACTIVE == GroupManagerGlobalParametersDb.globalState)
	{
		if(FALSE == GroupManagerGlobalParametersDb.staticGroupOnlyMode)
		{
			TrainingManagerStartTrainingReq();
		}
		else
		{
			// trigger training timer - to allow static group creation even if there is no traffic and grop will be created after MAC probing timeout
			OSAL_SET_TIMER_EXPLICIT(GROUP_MANAGER_TRAINING_TIMER_EXPIRED, OSAL_TIMERS_MS_TO_K_TICKS(GROUP_MANAGER_CALENDAR_WHEEL_TIMER_IN_MILLISECONDS), TASK_GROUP_MANAGER);
		}
	}
}

/**********************************************************************************

groupManagerSingleUserValidationStartConfirmation 


Description:
------------
handle the event that single user validation start confirmation has arrived from
link adaptation

Input: 
-----
groupManagerMessage - pointer to the message that contains the station id  	
	
		
Output:
-------
	

Returns:
--------
	void - 
	 
**********************************************************************************/
static void groupManagerSingleUserValidationStartConfirmation(K_MSG* groupManagerMessage)
{
	GroupManagerSingleUserValidationStartConfirmationMessage_t *groupManagerMessageParameters = NULL;
	StaId stationIndex = 0;

	groupManagerMessageParameters = ((GroupManagerSingleUserValidationStartConfirmationMessage_t *)groupManagerMessage->abData);
	stationIndex = groupManagerMessageParameters->stationIndex;	

	groupManagerRunStationStateMachine(stationIndex, GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_START_CONFIRMATION, NULL);
}

/**********************************************************************************

groupManagerSingleUserValidationResultsReceived 


Description:
------------
handle the event that single user validation results have been received

Input: 
-----
groupManagerMessage - pointer to the message that contains the station id  	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerSingleUserValidationResultsReceived(K_MSG* groupManagerMessage)
{
	GroupManagerSingleUserResultsMessage_t *groupManagerMessageParameters = NULL;
	StaId stationIndex = 0;

	groupManagerMessageParameters = ((GroupManagerSingleUserResultsMessage_t *)groupManagerMessage->abData);
	stationIndex = groupManagerMessageParameters->stationIndex;	

	groupManagerRunStationStateMachine(stationIndex, GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_RESULTS_RECEIVED, groupManagerMessageParameters);
}

/**********************************************************************************

groupManagerSingleUserValidationEndConfirmation 


Description:
------------
handle the event that single user validation results have been received

Input: 
-----
groupManagerMessage - pointer to the message that contains the station id  	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerSingleUserValidationEndConfirmation(K_MSG* groupManagerMessage)
{
	GroupManagerSingleUserValidationEndConfirmationMessage_t *groupManagerMessageParameters = NULL; 
	StaId stationIndex = 0;

	groupManagerMessageParameters = ((GroupManagerSingleUserValidationEndConfirmationMessage_t *)groupManagerMessage->abData);
	stationIndex = groupManagerMessageParameters->stationIndex;	

	groupManagerRunStationStateMachine(stationIndex, GROUP_MANAGER_STATION_EVENT_SINGLE_USER_VALIDATION_END_CONFIRMATION, NULL);
}


/**********************************************************************************

groupManagerGroupsMarksReceived 


Description:
------------
handle the event that groups' marks message has received

Input: 
-----
groupManagerMessage - pointer to the message that contains the marks  	
	
		
Output:
-------
	

Returns:
--------
	void - 

Notes:
------
This handler is not in state machine DB of station because this event can happen only in 
two states : GROUP_MANAGER_STATION_STATE_ACTIVE and GROUP_MANAGER_STATION_STATE_TEMP_ACTIVE. 
So the event is handled directly in afpTaskTable
**********************************************************************************/
static void groupManagerGroupsMarksReceived(K_MSG* groupManagerMessage)
{
	GroupManagerGroupsMarksMessage_t *groupManagerMessageParameters = NULL;
	GroupManagerStationState_e state = GROUP_MANAGER_STATION_FIRST_STATE;  
	StaId stationIndex = 0;
	uint8 wordInBitmap = 0;
	uint32 tempBitmap = 0;
	uint8 indexOffset = 0;
	uint16 stationActiveGroupsBitmap = 0;
	uint8 bestGroupIndex = 0;

	groupManagerMessageParameters = ((GroupManagerGroupsMarksMessage_t *)groupManagerMessage->abData);
	/* go over bitmap of 128 stations - divided to 4 words*/
    for(wordInBitmap = 0; wordInBitmap < GROUP_MANAGER_SIZE_OF_STATION_BITMAP_IN_WORDS; wordInBitmap++)
    {
		/* tempBitmap - represents current word in process, contains all stations which are waiting for marks in this word*/
		tempBitmap = GroupManagerGlobalParametersDb.waitingGroupsMarksBitmap[wordInBitmap];
#ifdef GROUP_MANAGER_DEBUG				
		ILOG0_DD("groupManagerGroupsMarksReceived: word in Bitmap %d, stations waiting for group marks in this word 0x%x",
			wordInBitmap, tempBitmap);
#endif		

		/* for each station which is waiting for group marks*/
		while(tempBitmap)
		{
			/*  turn off waiting bit in tempBitmap  */
			stationIndex = Utils_FindFirstSetAndClear(&tempBitmap); // the index value is 0 - 32
					
			/* extract station index. the received index value must be below 128, because represents station entry*/
			stationIndex += indexOffset;

			state = GroupManagerStationDb[stationIndex].state;
   		    ASSERT((state == GROUP_MANAGER_STATION_STATE_ACTIVE) || (state == GROUP_MANAGER_STATION_STATE_TEMP_ACTIVE));
			/* find only active groups of this stations */
			stationActiveGroupsBitmap = GroupManagerStationDb[stationIndex].groupsBitmap & GroupManagerGlobalParametersDb.activeGroupsBitmap;
			/* among all active groups find the index of the one with the highest score*/
			bestGroupIndex = groupManagerFindBestGroup(stationActiveGroupsBitmap, groupManagerMessageParameters->groupMarks);

            /* the group is waiting for score - therefore the group has no primary,
            set the best group index as primary group for this station*/
            groupManagerConfigureStationAsPrimaryInGroup(stationIndex, bestGroupIndex);

			/* turn off waiting bit in global db*/
			Utils_ZeroBitInBitmap(GroupManagerGlobalParametersDb.waitingGroupsMarksBitmap, stationIndex);

			/* station which has now primary group can be again part of periodic user validation process*/
   		    if (state == GROUP_MANAGER_STATION_STATE_ACTIVE)
   		    {
				groupManagerTryToStartNewSingleUserValidation(stationIndex);
   		    }
		}
		/* set to zero all bits in the current word before moving to next word */
		GroupManagerGlobalParametersDb.waitingGroupsMarksBitmap[wordInBitmap] = 0;
		/*  move to first bit in next word */
		indexOffset += GROUP_MANAGER_NUMBER_OF_STATIONS_BITS_IN_WORD;
    }
}

/**********************************************************************************

groupManagerStationNssHasChanged 


Description:
------------
handle the event that the NSS of station has changed with/without changing
the BW. the event is sent by Rx Manager

Input: 
-----
groupManagerMessage - pointer to the message that contains the parameters  	
	
		
Output:
-------
	

Returns:
--------
	void - 

Notes: 
------
This handler is not in state machine DB of because this event triggers exactly 
the same flow in all states. So the event is handled directly in afpTaskTable

**********************************************************************************/
static void groupManagerStationNssHasChanged(K_MSG* groupManagerMessage)
{
	GroupManagerStationNssHasChangedMessage_t *groupManagerMessageParameters = NULL;
	uint8 virtualGroupIndex = 0;
	uint8 groupIndex = 0;
	uint32 mask = 0;
	uint8 oldNss = 0;
	uint8 newNss = 0;
	StaId stationIndex = 0;
	uint32 tempBitmap = 0;
	TrainingManagerStopStaParams_t stopStationParameters;

	memset(&stopStationParameters, 0, sizeof(stopStationParameters));

	groupManagerMessageParameters = ((GroupManagerStationNssHasChangedMessage_t *)groupManagerMessage->abData);
	stationIndex = groupManagerMessageParameters->stationIndex;
	stopStationParameters.staId = stationIndex;

	/* Remove virtual groups with higher station NSS than allowed or higher BW than allowed */
	newNss = groupManagerMessageParameters->newNss;
	tempBitmap = GroupManagerStationDb[stationIndex].virtualGroupsBitmap;

	while(tempBitmap)
	{
		virtualGroupIndex = Utils_FindFirstSetAndClear(&tempBitmap);	
		/* extract current Nss value of the virtual group*/
		oldNss = groupManagerFindStationNssInVirtualGroup(stationIndex, virtualGroupIndex);
			
		if(oldNss > newNss)
		{
			mask = ~(0x1 << virtualGroupIndex);
			ILOG2_DDD("groupManagerStationNssHasChanged: station index %d. New nss %d. virtual group removed %d",
					stationIndex, newNss, virtualGroupIndex);
			GroupManagerStationDb[stationIndex].virtualGroupsBitmap &= mask;
			GroupManagerGlobalParametersDb.virtualGroupsBitmap &= mask;	
		}
	}

	/* Remove active groups with higher station NSS than allowed */
	tempBitmap = GroupManagerStationDb[stationIndex].groupsBitmap;
	while(tempBitmap)
	{
		groupIndex = Utils_CountTrailingZeros(tempBitmap);	
		tempBitmap &= ~(0x1 << groupIndex);
		oldNss = groupManagerFindStationNssInActiveGroup(stationIndex, groupIndex);
		// if NSS changed for one of stations member in static group - the group will not be removed in this case
		if((oldNss > newNss) && (groupIndex!=GroupManagerGlobalParametersDb.staticOccupiedGroupIndex))
		{
			ILOG2_DDD("groupManagerStationNssHasChanged: station index %d. New nss %d. active group removed %d",
					stationIndex, newNss, groupIndex);
			groupManagerGroupEventRemoveGroup(groupIndex, NULL);
		}
	}
	/* Stop Training of the station - only if the station is part of the current training, Training Mng will
	abort the training.  If the training can't be stopped (NDPA was sent) Traning mng will ignore the results of training*/
    TrainingManagerStopStaReq(&stopStationParameters);
}


/**********************************************************************************

groupManagerAddGroupConfirmation 


Description:
------------
handle the event that add group confirmation was received from the link adaptation

Input: 
-----
groupManagerMessage -  pointer to the message that contains the group index 	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerAddGroupConfirmation(K_MSG* groupManagerMessage)
{
	GroupManagerAddGroupConfirmationMessage_t *groupManagerMessageParameters = NULL; 
	uint8 groupIndex = 0;
	
	groupManagerMessageParameters = ((GroupManagerAddGroupConfirmationMessage_t *)groupManagerMessage->abData);
	groupIndex = groupManagerMessageParameters->groupIndex;	
#ifdef GROUP_MANAGER_DEBUG			
	ILOG0_D("groupManagerAddGroupConfirmation - groupIdx %d", groupIndex);
#endif
	groupManagerRunGroupStateMachine(groupIndex, GROUP_MANAGER_GROUP_EVENT_ADD_GROUP_CONFIRMATION, NULL);
}

/**********************************************************************************

groupManagerProbingResultsReceived 


Description:
------------
handle the event that probing results were received from link adaptation

Input: 
-----
groupManagerMessage -  pointer to the message that contains the group index 	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerProbingResultsReceived(K_MSG* groupManagerMessage)
{
	GroupManagerProbingResultsMessage_t *groupManagerMessageParameters = NULL;
	uint8 groupIndex = 0;
	uint8 i=0;

	groupManagerMessageParameters = ((GroupManagerProbingResultsMessage_t *)groupManagerMessage->abData);
	groupIndex = groupManagerMessageParameters->probingGroupIndex;	

	// in case static group exists overwrite it's mark with highest possible mark to prevent static group removal during process of probing results
	if(GroupManagerGlobalParametersDb.staticOccupiedGroupIndex!=GROUP_MANAGER_INVALID_GROUP_INDEX)
	{
		groupManagerMessageParameters->groupMarks[GroupManagerGlobalParametersDb.staticOccupiedGroupIndex] = GROUP_MANAGER_BEST_STATIC_GROUP_MARK;
		if(groupIndex == GroupManagerGlobalParametersDb.staticOccupiedGroupIndex)
		{
			for(i=0; i<GroupManagerGroupDb[groupIndex].numberOfParticipatingStations; i++)
			{
				groupManagerMessageParameters->stationMuMark[i] = (GroupManagerGlobalParametersDb.muConfigParams.stationGradeInProbingGroupThreshold + 1);
				groupManagerMessageParameters->stationSuMark[i] = 0;
			}
		}
	}
	groupManagerRunGroupStateMachine(groupIndex, GROUP_MANAGER_GROUP_EVENT_PROBING_RESULTS_READY, groupManagerMessageParameters);
}

/**********************************************************************************

groupManagerRemovedGroupConfirmation 


Description:
------------
handle the event that remove group confirmation was received from the link adaptation

Input: 
-----
groupManagerMessage -  pointer to the message that contains the group index 	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerRemovedGroupConfirmation(K_MSG* groupManagerMessage)
{
	GroupManagerRemoveGroupConfirmationMessage_t *groupManagerMessageParameters = NULL;
	uint8 groupIndex = 0;

	groupManagerMessageParameters = ((GroupManagerRemoveGroupConfirmationMessage_t *)groupManagerMessage->abData);
	groupIndex = groupManagerMessageParameters->groupIndex;	
	// if static group removed notify user
	if(GroupManagerGlobalParametersDb.staticOccupiedGroupIndex == groupIndex)
	{
		GroupManagerGlobalParametersDb.staticOccupiedGroupIndex = GROUP_MANAGER_INVALID_GROUP_INDEX;
		SERIAL_TRACE("Static group was deleted!",0,0,0);
		ILOG0_V("[groupManagerRemovedGroupConfirmation], Static group was deleted!");
	}
#ifdef GROUP_MANAGER_DEBUG			
	ILOG0_DD("groupManagerRemovedGroupConfirmation: from LA on group index %d. group state %d", groupIndex, GroupManagerGroupDb[groupIndex].state);
#endif
	groupManagerRunGroupStateMachine(groupIndex, GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP_CONFIRMATION, NULL);
}

/**********************************************************************************

groupManagerGroupsHasFailed 


Description:
------------
handle the event that a group has failed - some failure criterions have reached threshold
This event arrives from LA when group in active state and some criterions indicate that
this grup is bad


Input: 
-----
groupManagerMessage - pointer to the message that contains the group index	
	
		
Output:
-------
	

Returns:
--------
	void - 

Notes:
------

**********************************************************************************/
static void groupManagerGroupsHasFailed(K_MSG* groupManagerMessage)
{
	GroupManagerGroupHasFailedMessage_t *groupManagerMessageParameters = NULL;
	uint8 groupIndex = 0;

	groupManagerMessageParameters = ((GroupManagerGroupHasFailedMessage_t *)groupManagerMessage->abData);
	groupIndex = groupManagerMessageParameters->groupIndex;	
#ifdef GROUP_MANAGER_DEBUG	
	ILOG0_D("groupManagerGroupsHasFailed: group index %d", groupIndex);
#endif
	if((groupIndex != GROUP_MANAGER_INVALID_GROUP_INDEX) && 
		(GroupManagerGlobalParametersDb.staticOccupiedGroupIndex != groupIndex))
	{
		groupManagerRunGroupStateMachine(groupIndex, GROUP_MANAGER_GROUP_EVENT_GROUP_FAILED_INDICATION, NULL);
	}
}

/**********************************************************************************

groupManagerGroupLockConfirmation 


Description:
------------
handle the event that lock confirmation was received 

Input: 
-----
groupManagerMessage -  pointer to the message that contains the group index 	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerGroupLockConfirmation(K_MSG* groupManagerMessage)
{
	uint8 groupIndex = 0;
	LockGroupReqCb_t* lockConfirmationParameters = NULL; 

	lockConfirmationParameters = ((LockGroupReqCb_t *)groupManagerMessage->abData);
	groupIndex = lockConfirmationParameters->groupId;	

	groupManagerRunGroupStateMachine(groupIndex, GROUP_MANAGER_GROUP_EVENT_LOCK_CONFIRMATION, NULL);
}

/**********************************************************************************

groupManager
ResetMultiUserOperation 


Description:
------------

Input: 
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerResetMultiUserOperation(K_MSG* groupManagerMessage)
{
	K_MSG *kMsg_p = NULL;
	GroupManagerSetMultiUserOperationParameters_t *message = NULL;
	
	//check set operation request phase
	switch (GroupManagerGlobalParametersDb.setOperationRequestPhase)
	{
	case GROUP_MANAGER_SET_OPERATION_REQUEST_PHASE_IDLE:
		//phase is idle - need to check global state:
		//if state is active - start set operation and set phase to disabling
		//if state is idle - send confirmation
		switch (GroupManagerGlobalParametersDb.globalState)
		{
		case GROUP_MANAGER_GLOBAL_STATE_ACTIVE:
			kMsg_p = OSAL_GET_MESSAGE(sizeof(GroupManagerSetMultiUserOperationParameters_t));
			message = (GroupManagerSetMultiUserOperationParameters_t *)pK_MSG_DATA(kMsg_p);
			message->isMultiUserOperationEnabled = FALSE;
			GroupManagerGlobalParametersDb.setOperationRequestPhase = GROUP_MANAGER_SET_OPERATION_REQUEST_PHASE_DISABLING;
			OSAL_SEND_MESSAGE(GROUP_MANAGER_SET_MULTI_USER_OPERATION, TASK_GROUP_MANAGER, kMsg_p, groupManagerMessage->header.vapId);
			break;
		case GROUP_MANAGER_GLOBAL_STATE_IDLE:
			OSAL_SEND_NO_DATA_MESSAGE(COC_MULTI_USER_RESET_OPERATION_CFM, TASK_COC, groupManagerMessage->header.vapId);
			break;
		default:
			FATAL("groupManagerResetMultiUserOperation");
		}
		break;
	case GROUP_MANAGER_SET_OPERATION_REQUEST_PHASE_DISABLING:
		//change phase to enabling and start set operation
		kMsg_p = OSAL_GET_MESSAGE(sizeof(GroupManagerSetMultiUserOperationParameters_t));
		message = (GroupManagerSetMultiUserOperationParameters_t *)pK_MSG_DATA(kMsg_p);
		message->isMultiUserOperationEnabled = TRUE;
		GroupManagerGlobalParametersDb.setOperationRequestPhase = GROUP_MANAGER_SET_OPERATION_REQUEST_PHASE_ENABLING;
		OSAL_SEND_MESSAGE(GROUP_MANAGER_SET_MULTI_USER_OPERATION, TASK_GROUP_MANAGER, kMsg_p, groupManagerMessage->header.vapId);
		break;
	case GROUP_MANAGER_SET_OPERATION_REQUEST_PHASE_ENABLING:
		GroupManagerGlobalParametersDb.setOperationRequestPhase = GROUP_MANAGER_SET_OPERATION_REQUEST_PHASE_IDLE;
		OSAL_SEND_NO_DATA_MESSAGE(COC_MULTI_USER_RESET_OPERATION_CFM, TASK_COC, groupManagerMessage->header.vapId);
		break;
	default:
		FATAL("groupManagerResetMultiUserOperation");
	}
}


/**********************************************************************************

groupManager
SetMultiUserOperationCfm


Description:
------------

Input: 
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerSetMultiUserOperationCfm(void)
{
	K_MSG *psMsg = NULL;

	//check set operation request phase
	switch (GroupManagerGlobalParametersDb.setOperationRequestPhase)
	{
	case GROUP_MANAGER_SET_OPERATION_REQUEST_PHASE_REQ_FROM_HOST:
		//request was originated directly from host, send confirmation using original message
		psMsg = GroupManagerGlobalParametersDb.returnMsg;
		GroupManagerGlobalParametersDb.setOperationRequestPhase = GROUP_MANAGER_SET_OPERATION_REQUEST_PHASE_IDLE;
		GroupManagerGlobalParametersDb.returnMsg = NULL;
		OSAL_SEND_MESSAGE(UMI_MULTI_USER_OPERATION_SETTING_CFM, TASK_UM_IF_TASK, psMsg, VAP_ID_DO_NOT_CARE);
		break;
	case GROUP_MANAGER_SET_OPERATION_REQUEST_PHASE_DISABLING:
	case GROUP_MANAGER_SET_OPERATION_REQUEST_PHASE_ENABLING:
		//request is part of a diable/enable phase, send message to resume this process
		OSAL_SEND_NO_DATA_MESSAGE(GROUP_MANAGER_RESET_MULTI_USER_OPERATION, TASK_GROUP_MANAGER, VAP_ID_DO_NOT_CARE);
		break;
	default:
		FATAL("groupManagerSetMultiUserOperationCfm");
	}
}


/**********************************************************************************

groupManagerSetMultiUserOperationReq 


Description:
------------
handle the message 'set multi user operation' 
When multi user mode is disabled stations in HALT state remain in the same state and 
periodic timer of 400 millisec will be still activated

Input: 
-----
groupManagerMessage - pointer to the message that contains the flag 'is multi user
operation enabled'
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerSetMultiUserOperationReq(K_MSG* groupManagerMessage)
{
	K_MSG *kMsg_p = NULL;
	GroupManagerSetMultiUserOperationParameters_t *message = NULL;
	UMI_MU_OPERATION_CONFIG *msgData = (UMI_MU_OPERATION_CONFIG*)pK_MSG_DATA(groupManagerMessage);


	DEBUG_ASSERT(GroupManagerGlobalParametersDb.setOperationRequestPhase == GROUP_MANAGER_SET_OPERATION_REQUEST_PHASE_IDLE);
	DEBUG_ASSERT(GroupManagerGlobalParametersDb.returnMsg == NULL);
	
	GroupManagerGlobalParametersDb.returnMsg = groupManagerMessage;
	GroupManagerGlobalParametersDb.setOperationRequestPhase = GROUP_MANAGER_SET_OPERATION_REQUEST_PHASE_REQ_FROM_HOST;

	if (msgData->getSetOperation == API_GET_OPERATION)
	{
		msgData->enableMuOperation = GroupManagerGlobalParametersDb.globalState;
		groupManagerSetMultiUserOperationCfm();
	}
	else
	{
		msgData = (UMI_MU_OPERATION_CONFIG *)pK_MSG_DATA(groupManagerMessage);
		kMsg_p = OSAL_GET_MESSAGE(sizeof(GroupManagerSetMultiUserOperationParameters_t));
		message = (GroupManagerSetMultiUserOperationParameters_t *)pK_MSG_DATA(kMsg_p);
		message->isMultiUserOperationEnabled = msgData->enableMuOperation;
		
	OSAL_SEND_MESSAGE(GROUP_MANAGER_SET_MULTI_USER_OPERATION, TASK_GROUP_MANAGER, kMsg_p, VAP_ID_DO_NOT_CARE);
	}
}


/**********************************************************************************

groupManagerSetMultiUserOperation 


Description:
------------
handle the message 'set multi user operation' 
When multi user mode is disabled stations in HALT state remain in the same state and 
periodic timer of 400 millisec will be still activated

Input: 
-----
groupManagerMessage - pointer to the message that contains the flag 'is multi user
operation enabled'
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerSetMultiUserOperation(K_MSG* groupManagerMessage)
{
	GroupManagerSetMultiUserOperationParameters_t *groupManagerMessageParameters = NULL;

	groupManagerMessageParameters = ((GroupManagerSetMultiUserOperationParameters_t *)groupManagerMessage->abData);

#ifdef GROUP_MANAGER_DEBUG				
		ILOG0_D("groupManagerSetMultiUserOperation: current MU state = %d", groupManagerMessageParameters->isMultiUserOperationEnabled);
#endif

	/* multi user mode can be enabled only if currently it is disabled*/
	if(groupManagerMessageParameters->isMultiUserOperationEnabled)
	{
		if(GROUP_MANAGER_GLOBAL_STATE_IDLE == GroupManagerGlobalParametersDb.globalState)
		{
			groupManagerChangeGlobalState(GROUP_MANAGER_GLOBAL_STATE_ACTIVE);
			// trigger timer to start training mechanism all over again
			OSAL_SET_TIMER_EXPLICIT(GROUP_MANAGER_TRAINING_TIMER_EXPIRED, OSAL_TIMERS_MS_TO_K_TICKS(GROUP_MANAGER_CALENDAR_WHEEL_TIMER_IN_MILLISECONDS), TASK_GROUP_MANAGER);
		}
		else
		{
#ifdef GROUP_MANAGER_DEBUG
		ILOG2_V("groupManagerSetMultiUserOperation: MU is already active");
#endif
		}
		groupManagerSetMultiUserOperationCfm();
		
	}
	else
	{
		if (GROUP_MANAGER_GLOBAL_STATE_ACTIVE == GroupManagerGlobalParametersDb.globalState)
		{
			groupManagerChangeGlobalState(GROUP_MANAGER_GLOBAL_STATE_DISABLING_WAITING_FOR_TRAINING);
			GroupManagerGlobalParametersDb.staticGroupOnlyMode = FALSE;
		
			groupManagerRemoveAllVirtualGroups();
			groupManagerRemoveAllActiveGroups();

			/* do not delete HALTED stations from DB, these stations are still monitored and periodic timer is still 
			  running even when multi user disabled*/
			TrainingManagerStopTrainingReq();
		}
		else
		{
			ILOG2_V("groupManagerSetMultiUserOperation: MU is already deactivated");
			groupManagerSetMultiUserOperationCfm();
		}
	}
}

/**********************************************************************************

groupManagerSetMuFixedRate

Description: sets boolean value fixedRateFlag in GroupManagerGlobalParams db. 
------------

Input: 
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerSetMuFixedRate(K_MSG* groupManagerMessage)
{
	GroupManagerFixedRateParams_t* fixedRateMsg = (GroupManagerFixedRateParams_t*)pK_MSG_DATA(groupManagerMessage);
	GroupManagerGlobalParametersDb.fixedRateFlag = fixedRateMsg->fixedRateMuFlag;

#ifdef GROUP_MANAGER_DEBUG
	ILOG2_D("groupManagerSetMuFixedRate: isFixedRate %d", GroupManagerGlobalParametersDb.fixedRateFlag);
#endif
}

/**********************************************************************************

groupManagerStationEventRemoveGroupDueToStation 


Description:
------------
handle the event of remove group due to group failure.
The group failure can be either bad performence indicated by LA 
or notification process failure before probing because of this station.
this function might cause station to enter HALT state


Input: 
-----
stationIndex - the station index	
probingGroupIndex - the removed group index

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventRemoveGroupDueToStation(StaId stationIndex, uint8 probingGroupIndex)
{
	groupManagerRunStationStateMachine(stationIndex, GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP_DUE_TO_STATION, &probingGroupIndex);
}

/**********************************************************************************

groupManagerStationEventRemoveGroup 


Description:
------------
handle the event of remove group for a specific station
This function won't cause station to enter HALT state, the remove
group was initiated either because one of the stations in the group was 
disconnected or single user validation process caused to remove group because 
the station operates better in SU mode and there is no other primary
for this group



Input:  
-----
stationIndex - the station index	
probingGroupIndex - the removed group index

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventRemoveGroup(StaId stationIndex, uint8 probingGroupIndex)
{
	groupManagerRunStationStateMachine(stationIndex, GROUP_MANAGER_STATION_EVENT_REMOVE_GROUP, &probingGroupIndex);
}

/**********************************************************************************

groupManagerStationEventTrainingFailed 


Description:
------------
handle the event of training failure for a specific station. 
The function will be invoked inside function which is provided as API to training manager
to handle training results. This function is invoked for those stations that
their training has failed (Station's training has failed if all tested combinations
with this station didn't provide score above the threshold)


Input: 
-----
stationIndex - the station index	

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventTrainingFailed(StaId stationIndex) 
{
	groupManagerRunStationStateMachine(stationIndex, GROUP_MANAGER_STATION_EVENT_TRAINING_FAILED, NULL);
}

/**********************************************************************************

groupManagerStationEventNotificationStarted 


Description:
------------
handle the event of notification started for a specific station
each station in group is notified that notification process has started


Input: 
-----
stationIndex - the station index	
probingGroupIndex - the probing group index

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventNotificationStarted(StaId stationIndex, uint8 probingGroupIndex)
{
	groupManagerRunStationStateMachine(stationIndex, GROUP_MANAGER_STATION_EVENT_NOTIFICATION_STARTED, &probingGroupIndex);
}

/**********************************************************************************

groupManagerStationEventProbingStarted 


Description:
------------
handle the event of probing started for a specific station
all stations in group notified that their groups started probing process


Input: 
-----
stationIndex - the station index	
probingGroupIndex - the probing group index

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventProbingStarted(StaId stationIndex, uint8 probingGroupIndex)
{
	groupManagerRunStationStateMachine(stationIndex, GROUP_MANAGER_STATION_EVENT_PROBING_STARTED, &probingGroupIndex);
}

/**********************************************************************************

groupManagerStationEventMoveToTempActive 


Description:
------------
change active station to temp active - will have a temporary primary group which is
probing group

Input: 
-----
stationIndex - the station index	
probingGroupIndex - the probing group index
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventMoveToTempActive(StaId stationIndex, uint8 probingGroupIndex)
{
	GroupManagerStationParameters_t *groupManagerStationDataBaseEntry = NULL;

	groupManagerStationDataBaseEntry = &GroupManagerStationDb[stationIndex];
	
	/* This event should happen only for active stations */
	ASSERT(groupManagerStationDataBaseEntry->state == GROUP_MANAGER_STATION_STATE_ACTIVE);
	/* in case the station chosen for TempActive has primary group, temporary remove it from single user validation process*/
	if(GROUP_MANAGER_INVALID_GROUP_INDEX != GroupManagerStationDb[stationIndex].primaryGroup)
	{
		Utils_ZeroBitInBitmap(GroupManagerGlobalParametersDb.singleUserValidationBitmap, stationIndex);
	}
	groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_TEMP_ACTIVE);
	/* configure Selector with new primary for probing group*/
	groupManagerConfigureStationForSelector(stationIndex, TRUE, probingGroupIndex, TRUE);
}

/**********************************************************************************

groupManagerStationEventActivateGroup 


Description:
------------
handle the event that a probing group is activated (the activation might fail in case
at the end of the process there is no primary for the probing group) for a specific
station.
The function is invoked from function which handles probing results of a group.

Input: 
-----
stationIndex - the station index	
groupManagerProbingResultsParameters - the message that contains the probing results
and groups' marks
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventActivateGroup(StaId stationIndex, GroupManagerProbingResultsMessage_t *groupManagerProbingResultsParameters)
{
	groupManagerRunStationStateMachine(stationIndex, GROUP_MANAGER_STATION_EVENT_ACTIVATE_GROUP, groupManagerProbingResultsParameters);
}

/**********************************************************************************

groupManagerStationEventSingleUserValidationStartProcess 


Description:
------------
start single user validation process for a specific station - at this point single user
validation process of last station just finished and the next active station in bitmap
of active stations (with primary group) will be added to calendar wheel
Input: 
-----
stationIndex - the station index	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventSingleUserValidationStartProcess(StaId stationIndex)
{
	GroupManagerStationParameters_t *groupManagerStationDataBaseEntry = NULL;

	groupManagerStationDataBaseEntry = &GroupManagerStationDb[stationIndex];
	
	/* This event should happen only for active stations */
	ASSERT(groupManagerStationDataBaseEntry->state == GROUP_MANAGER_STATION_STATE_ACTIVE);
	/* adds the station to calendar wheel , GROUP_MANAGER_SINGLE_USER_VALIDATION_TIMER - represents number of slots*/
	CalendarWheel_AddEntry(&GroupManagerStationsCalendarWheel, (CalendarWheelEntry_t *)groupManagerStationDataBaseEntry, GROUP_MANAGER_SINGLE_USER_VALIDATION_TIMER);
	/* this is the point in code where the bit of single user validation must be set to zero because
	the station already in calendar wheel waiting for user validation to begin*/
	Utils_ZeroBitInBitmap(GroupManagerGlobalParametersDb.singleUserValidationBitmap, stationIndex);

	groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_PENDING);
}


/**********************************************************************************

groupManagerStationEventRemoveStationInNotAllowed 


Description:
------------
remove a station in the state of 'not allowed' 

Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventRemoveStationInNotAllowed(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{
	UNUSED_PARAM(parameters);	
	groupManagerSendConfirmToStationManager(groupManagerStationDataBaseEntry->stationIndex);
	groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_IDLE);
}
	
/**********************************************************************************

groupManagerStationEventRemoveStationInSingleUser 


Description:
------------
remove a station in the state of 'single user' 
station is candidate , might be in virtual group but not in active/probing group

Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventRemoveStationInSingleUser(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{
	StaId stationIndex = 0;
	UNUSED_PARAM(parameters);
	stationIndex = groupManagerStationDataBaseEntry->stationIndex;
	/* remove all virtual groups that this station is member of*/
	groupManagerRemoveAllStationVirtualGroups(groupManagerStationDataBaseEntry);
	groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_TRAINING_CONFIRMATION);
	/* stop training if this station is part of the ongoing training*/
	groupManagerTriggerRemoveStationInTraining(stationIndex);
}

/**********************************************************************************

groupManagerStationEventRemoveStationInNotifyingOrProbing 


Description:
------------
remove a station in the state of 'notifying' or 'probing' state 
The function is invoked through SM table in case "remove station" will be sebt by BSS

Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventRemoveStationInNotifyingOrProbing(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{
	StaId stationIndex = 0;
	UNUSED_PARAM(parameters);
	stationIndex = groupManagerStationDataBaseEntry->stationIndex;

	/* remove all virtual groups this stations is member of*/
	groupManagerRemoveAllStationVirtualGroups(groupManagerStationDataBaseEntry);
	/* this functins removes all groups in probing state this station is member of*/
	groupManagerRemoveStationActiveGroups(groupManagerStationDataBaseEntry);
	groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_TRAINING_CONFIRMATION);
	groupManagerTriggerRemoveStationInTraining(stationIndex);
}

/**********************************************************************************

groupManagerStationEventRemoveStationInActiveOrTempActive


Description:
------------
remove a station in the state of 'active' or 'temp active' 
station in tempActive - when it is used temporary  as primary for some probing group because all stations in this
probing group are primary for other groups and some member had to be primary for this group
the function is invoked from SM table in case "remove station" arrives from BSS
Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	remove all virtual groups this stations is member of

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventRemoveStationInActiveOrTempActive(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{
	StaId stationIndex = 0;
	UNUSED_PARAM(parameters);
	stationIndex = groupManagerStationDataBaseEntry->stationIndex;
	groupManagerRemoveAllStationVirtualGroups(groupManagerStationDataBaseEntry);
	groupManagerRemoveStationActiveGroups(groupManagerStationDataBaseEntry);
	Utils_ZeroBitInBitmap(GroupManagerGlobalParametersDb.waitingGroupsMarksBitmap, stationIndex);
	Utils_ZeroBitInBitmap(GroupManagerGlobalParametersDb.singleUserValidationBitmap, stationIndex);
#ifdef GROUP_MANAGER_DEBUG				
	ILOG0_D("groupManagerStationEventRemoveStationInActiveOrTempActive: on station index %d", stationIndex);
#endif
	// disable multi user mode of station in Selector mu bitmap
	groupManagerConfigureStationForSelector(stationIndex,FALSE,0, TRUE);
	groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_TRAINING_CONFIRMATION);
	groupManagerTriggerRemoveStationInTraining(stationIndex);
}

/**********************************************************************************

groupManagerStationEventRemoveStationInSingleUserValidationPending


Description:
------------
remove a station in the state of 'single user validation pending' 

Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventRemoveStationInSingleUserValidationPending(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{
	StaId stationIndex = 0;
	UNUSED_PARAM(parameters);
	stationIndex = groupManagerStationDataBaseEntry->stationIndex;

	CalendarWheel_RemoveEntry(&GroupManagerStationsCalendarWheel, (CalendarWheelEntry_t*)groupManagerStationDataBaseEntry);
	groupManagerRemoveAllStationVirtualGroups(groupManagerStationDataBaseEntry);
	groupManagerRemoveStationActiveGroups(groupManagerStationDataBaseEntry);
	groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_TRAINING_CONFIRMATION);

	groupManagerTriggerRemoveStationInTraining(stationIndex);
	groupManagerStartNewSingleUserValidation();
}

/**********************************************************************************

groupManagerStationEventRemoveStationInSuValWaitingLa


Description:
------------
remove a station in the state of 'single user validation waiting for link adaptation' - 
the station is waiting for confirmation from LA on "start single uer validation"

Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventRemoveStationInSuValWaitingLa(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{
	StaId stationIndex = 0;
	UNUSED_PARAM(parameters);

	stationIndex = groupManagerStationDataBaseEntry->stationIndex;

	groupManagerRemoveAllStationVirtualGroups(groupManagerStationDataBaseEntry);
	groupManagerRemoveStationActiveGroups(groupManagerStationDataBaseEntry);
	groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_LINK_ADAPTATION_AND_TRAINING_CONFIRMATION);

	groupManagerTriggerRemoveStationInTraining(stationIndex);
    groupManagerSendEndSingleUserValidationMessage(stationIndex);
}

/**********************************************************************************

groupManagerStationEventRemoveStationInSingleUserValidation


Description:
------------
remove a station in the state of 'single user validation' - station is tested
as single user in LA

Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventRemoveStationInSingleUserValidation(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{
	StaId stationIndex = 0;
	UNUSED_PARAM(parameters);

	stationIndex = groupManagerStationDataBaseEntry->stationIndex;
	/* remove "single user validation results timeout" from calendar wheel because the process is being stopped*/
	CalendarWheel_RemoveEntry(&GroupManagerStationsCalendarWheel, (CalendarWheelEntry_t*)groupManagerStationDataBaseEntry);
	groupManagerRemoveAllStationVirtualGroups(groupManagerStationDataBaseEntry);
	groupManagerRemoveStationActiveGroups(groupManagerStationDataBaseEntry);
	groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_LINK_ADAPTATION_AND_TRAINING_CONFIRMATION);

	groupManagerTriggerRemoveStationInTraining(stationIndex);
	groupManagerSendEndSingleUserValidationMessage(stationIndex);
}

/**********************************************************************************

groupManagerStationEventRemoveStationAfterSuValEnded


Description:
------------
remove a station in the state of 'single user validation succeeded', 'single user
validation failed', 'single user validation canceled - group removed' and 'single
user validation canceled - group failed'

Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventRemoveStationAfterSuValEnded(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{
	StaId stationIndex = 0;
	UNUSED_PARAM(parameters);

	stationIndex = groupManagerStationDataBaseEntry->stationIndex;

	groupManagerRemoveAllStationVirtualGroups(groupManagerStationDataBaseEntry);
	groupManagerRemoveStationActiveGroups(groupManagerStationDataBaseEntry);
	groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_LINK_ADAPTATION_AND_TRAINING_CONFIRMATION);

	groupManagerTriggerRemoveStationInTraining(stationIndex);
}

/**********************************************************************************

groupManagerStationEventRemoveStationInHaltingWaitingForTrainingCfm


Description:
------------
remove a station in the state of 'halting - waiting for training confirmation'
Station started a halt process and stop training request was sent to Training mng.
Station is waiting for CFM from Training mng
For station in halting process all bits except halting bit already set to zero


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventRemoveStationInHaltingWaitingForTrainingCfm(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{
	StaId stationIndex = 0;
	UNUSED_PARAM(parameters);

	stationIndex = groupManagerStationDataBaseEntry->stationIndex;
	/* set stations halt bit to 0 and change state to wait for CFM on stop training which was sent in the beginning of halt process*/
	Utils_ZeroBitInBitmap(GroupManagerGlobalParametersDb.trainingBitmaps.haltStationsBitmap.stationBitmap, stationIndex);
	groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_TRAINING_CONFIRMATION);
	/* This function is invoke in order to update to training about the removal of the station */
	groupManagerTriggerRemoveStationInTraining(stationIndex);
}


/**********************************************************************************

groupManagerStationEventRemoveStationInHalt

Description:
------------
remove a station in the state of 'halt' - in this state Training already notified that
station in halt state and all relevant groups of this station were removed. Not waiting for any CFM
from other modules

Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventRemoveStationInHalt(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{
	StaId stationIndex = 0;
	UNUSED_PARAM(parameters);

	stationIndex = groupManagerStationDataBaseEntry->stationIndex;
	
	Utils_ZeroBitInBitmap(GroupManagerGlobalParametersDb.trainingBitmaps.haltStationsBitmap.stationBitmap, stationIndex);
	/* remove halt timeout event - station is being removed*/
	CalendarWheel_RemoveEntry(&GroupManagerStationsCalendarWheel, (CalendarWheelEntry_t*)groupManagerStationDataBaseEntry);
	/* station that is removed moved to IDLE state to indicate that this slot in DB is free*/
	groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_TRAINING_CONFIRMATION);
	
	/* This function is invoke in order to update to training about the removal of the station */
	groupManagerTriggerRemoveStationInTraining(stationIndex);
}

/**********************************************************************************

groupManagerStationEventRemoveGroupDueToStationInNotifying


Description:
------------
group is removed due to group failure in the state of 'notifying'
the failure was caused by station passed as the parameter

Note - the station can be part of only single group in notify state
The station hasn't answered to group id management frame and it is not part of any other notifying group so it will be halted


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database tp station which didn't reply to group id management frame during notify
parameters - group index	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventRemoveGroupDueToStationInNotifying(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{
	StaId stationIndex = groupManagerStationDataBaseEntry->stationIndex;	
    uint8 groupIndex = *(uint8 *)parameters;
#ifdef GROUP_MANAGER_DEBUG				
	ILOG0_DD("groupManagerStationEventRemoveGroupDueToStationInNotifying: group index %d, station index %d", groupIndex, stationIndex);
#endif
	
	/* turn of bit of this group in stations db */
	groupManagerClearGroupBitInStationBitmap(groupManagerStationDataBaseEntry, groupIndex);

	groupManagerRemoveAllStationVirtualGroups(groupManagerStationDataBaseEntry);
	Utils_SetBitInBitmap(GroupManagerGlobalParametersDb.trainingBitmaps.haltStationsBitmap.stationBitmap, stationIndex);

	groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_HALTING_WAITING_FOR_TRAINING_CONFIRMATION);

	groupManagerStationDataBaseEntry->numberOfHalts ++;
	if(groupManagerStationDataBaseEntry->numberOfHalts == GROUP_MANAGER_MAX_HALT_THRESHOLD)
	{
		groupManagerStationDataBaseEntry->numberOfHalts = 0;
		groupManagerStationDataBaseEntry->timeout = GROUP_MANAGER_MAX_HALT_PERIOD;
	}
	else
	{		
		groupManagerStationDataBaseEntry->timeout = groupManagerStationDataBaseEntry->numberOfHalts + GROUP_MANAGER_HALT_PENALTY_FROM_PROBING;
	}
	groupManagerTriggerHaltStationInTraining(stationIndex);
	/* note - timeout field is set here but station will be added to calendar wheel only when training CFM will be returned in
			  function groupManagerStationEventTrainingCfmInHaltingWaitingForTrainingCfm */
		
}

/**********************************************************************************

groupManagerStationEventRemoveGroupDueToStationInProbing


Description:
------------
group is removed due to group failure in the state of 'probing'
the station passed as parameter must be in probing state and not in active state


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - index of group which failed probing	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventRemoveGroupDueToStationInProbing(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{
	uint16 notifyingGroupsBitmap = 0;
	uint16 probingGroupsBitmap = 0;
	StaId stationIndex = groupManagerStationDataBaseEntry->stationIndex;
	uint8 groupIndex = *(uint8 *)parameters;

	groupManagerClearGroupBitInStationBitmap(groupManagerStationDataBaseEntry, groupIndex);
	/*if the station acts as primary for this probing group but this station is not temp active. This station participates
	only in probing group and doesn't have any active groups.*/
    if(groupIndex == groupManagerStationDataBaseEntry->tempPrimaryGroup)
    {
		groupManagerStationDataBaseEntry->tempPrimaryGroup = GROUP_MANAGER_INVALID_GROUP_INDEX;	
		/* configure selector registers - this station is no longer primary in any group because the group
		is removed */
#ifdef GROUP_MANAGER_DEBUG						
		ILOG0_DD("groupManagerStationEventRemoveGroupDueToStationInProbing: group index %d, station index %d. Configure station for selector as SU.", groupIndex, stationIndex); 
#endif
		groupManagerConfigureStationForSelector(stationIndex, FALSE, 0, TRUE);
    }	
	/* find aditional probing groups this station is part of:
		if there are such groups station will remain in PROBING state and there is nothing else to do
		if there is no such groups check if this stations is part of notifying groups. If TRUE move stations state to NOTIFY
	*/
   	probingGroupsBitmap = groupManagerStationDataBaseEntry->groupsBitmap & GroupManagerGlobalParametersDb.probingGroupsBitmap;
	if(0 == probingGroupsBitmap)
	{
		notifyingGroupsBitmap = groupManagerStationDataBaseEntry->groupsBitmap & GroupManagerGlobalParametersDb.notifyingGroupsBitmap;
		if(notifyingGroupsBitmap)
		{
#ifdef GROUP_MANAGER_DEBUG							
			ILOG0_DD("groupManagerStationEventRemoveGroupDueToStationInProbing: group index %d, station index %d. Station is only a member in notifying groups",
				groupIndex, stationIndex);
#endif			
			groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_NOTIFY_STATION);
		}
		/* handle the case when this station has no other probing or notifying groups */
		else
		{
			if(groupManagerStationDataBaseEntry->virtualGroupsBitmap)
   			{
#ifdef GROUP_MANAGER_DEBUG								
   				ILOG0_DD("groupManagerStationEventRemoveGroupDueToStationInProbing: group index %d, station index %d. Station is only a member in virtual groups",
				groupIndex, stationIndex);
#endif
				/* The station is member in virtual groups so it wont start halt process, will be moved to SINGLE USER */
				groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_SINGLE_USER);
    		}
				/* the station is not a memeber of active, probing, notify or virtual group - so it must start HALT process */
			else
			{
#ifdef GROUP_MANAGER_DEBUG								
				ILOG0_DD("groupManagerStationEventRemoveGroupDueToStationInProbing: group index %d, station index %d. Station is not a member of any group. Moving to HALT",
				groupIndex, stationIndex);
#endif				
				Utils_SetBitInBitmap(GroupManagerGlobalParametersDb.trainingBitmaps.haltStationsBitmap.stationBitmap, stationIndex);
				groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_HALTING_WAITING_FOR_TRAINING_CONFIRMATION);
				
				groupManagerStationDataBaseEntry->numberOfHalts ++;
				if(groupManagerStationDataBaseEntry->numberOfHalts == GROUP_MANAGER_MAX_HALT_THRESHOLD)
				{
					groupManagerStationDataBaseEntry->numberOfHalts = 0;
					groupManagerStationDataBaseEntry->timeout = GROUP_MANAGER_MAX_HALT_PERIOD;
				}
				else
				{		
					groupManagerStationDataBaseEntry->timeout = groupManagerStationDataBaseEntry->numberOfHalts + GROUP_MANAGER_HALT_PENALTY_FROM_PROBING;
				}
					/* this will cause training manager to not involve the station in upcoming trainings but the
					   station will still stay in Training dB. Only removeStationFromTraining will remove the station*/
				groupManagerTriggerHaltStationInTraining(stationIndex);
			}
		}
	}
}

/**********************************************************************************

groupManagerStationEventRemoveGroupInTempActive


Description:
------------
group is removed due to group failure and station in the state of 'temp active'
The function handles


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - the group index	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventRemoveGroupInTempActive(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{
	uint16 activeGroupsBitmap = 0;
	StaId stationIndex = groupManagerStationDataBaseEntry->stationIndex;
    uint8 groupIndex = *(uint8 *)parameters;
	uint8 primaryGroupIndex = 0;

	groupManagerClearGroupBitInStationBitmap(groupManagerStationDataBaseEntry, groupIndex);

	/* handle remove case for group for which this station is tempActive*/
    if(groupIndex == groupManagerStationDataBaseEntry->tempPrimaryGroup)
    {
#ifdef GROUP_MANAGER_DEBUG						
    	ILOG0_DD("groupManagerStationEventRemoveGroupInTempActive: Removing groupIndex %d. stationIndex %d is tempActive of the group", groupIndex, stationIndex);
#endif
		groupManagerStationDataBaseEntry->tempPrimaryGroup = GROUP_MANAGER_INVALID_GROUP_INDEX;
		/* return the groups state to ACTIVE because it's assurred that this station is a member of at least
		one active group - it's previous primary group. If this primary group was removed earlier and no other
		groups in which this station is active are found the station's state would have change to "PROBING",
		and we couldn't reach this point in code*/
		groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_ACTIVE);
		primaryGroupIndex = groupManagerStationDataBaseEntry->primaryGroup;
		/* if the group for which this station was primary before was not removed configure in selector this
		station to be it's primary again*/
		if(GROUP_MANAGER_INVALID_GROUP_INDEX != primaryGroupIndex)
		{
			// no need to change txop and mdpu in staDb because station was already primary
			groupManagerConfigureStationForSelector(stationIndex, TRUE, primaryGroupIndex, FALSE);
		}		
		else
		{
			groupManagerSetStationSuTxopAndAmpduLimit(stationIndex, TRUE, SU_NUM_OF_MPDUS_IN_A_MPDU);
			//  possible fix for Selector's primary station mapping bug
			//groupManagerConfigureStationForSelector(stationIndex, FALSE, 0);
		}
    }
	/* handle remove case for station in temp active state and it's previous primary group is removed*/
	else if(groupManagerStationDataBaseEntry->primaryGroup == groupIndex)
	{
		// set to invalid primary group index of this station
		groupManagerStationDataBaseEntry->primaryGroup = GROUP_MANAGER_INVALID_GROUP_INDEX;
		groupManagerSetStationSuTxopAndAmpduLimit(stationIndex, TRUE , SU_NUM_OF_MPDUS_IN_A_MPDU);
		/* check if the station is part of additional active groups*/
		activeGroupsBitmap = groupManagerStationDataBaseEntry->groupsBitmap & GroupManagerGlobalParametersDb.activeGroupsBitmap;
		if(activeGroupsBitmap)
		{	
#ifdef GROUP_MANAGER_DEBUG							
			ILOG0_DD("groupManagerStationEventRemoveGroupInTempActive: Removing groupIndex %d which is primary group of stationIndex %d. station is tempActive of another group and part of more active groups", 
					groupIndex, stationIndex);
#endif			
			/* station lost it's primary group - send request for group marks to trigger evaluation process for this station.
			when group marks will be returned  Group manager will go over only on groups with this station
			and try to find a new primary group for this station*/
			groupManagerSendGroupsMarksMessage(stationIndex);
		}
		/* if the station is not a member of additional active groups but it's for sure 
			it is a member of at least one probing group otherwise we couldn't enter this function*/
		else
		{
#ifdef GROUP_MANAGER_DEBUG							
			ILOG0_DD("groupManagerStationEventRemoveGroupInTempActive: Removing groupIndex %d which is primary group of stationIndex %d. station is tempActive and member of only probing groups", groupIndex, stationIndex);
#endif			
			/* There is at least one probing group - the group index != temp primary  so no need to check */
			Utils_ZeroBitInBitmap(GroupManagerGlobalParametersDb.waitingGroupsMarksBitmap, stationIndex);
			/* in this case the state must be changed to probing*/
			groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_PROBING);
		}
	}		
}

/**********************************************************************************

groupManagerStationEventRemoveGroupDueToStationInActive


Description:
------------
group is removed due to group failure in the state of 'active' 


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - the group index	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventRemoveGroupDueToStationInActive(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{
	groupManagerRemoveGroupInActive(groupManagerStationDataBaseEntry, parameters, TRUE);	
}

/**********************************************************************************

groupManagerStationEventRemoveGroupDueToStationInSingleUserValidationPending


Description:
------------
group is removed due to group failure in the state of 'single user validation pending'
The function is called after group was removed and now group members are
updated about the removal in order to change their DB. The state of the station might be updated 
and teh final result can be HALT because group removal occurerd due to failure


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - the group index	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventRemoveGroupDueToStationInSingleUserValidationPending(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{
	groupManagerRemoveGroupInSingleUserValidationPending(groupManagerStationDataBaseEntry, parameters, TRUE);
}

/**********************************************************************************

groupManagerStationEventRemoveGroupDueToStationInSuValWaitingLa


Description:
------------
group is removed due to group failure in the state of 'single user validation waiting
for link adaptation'. 
The group is already removed and now the station which is waiting 
for LA CFM on "start single user validation" is notified. In this function station DB
will be updated. "End single user validation" is sent to LA. As the result LA will 
send "start single user validation CFM" on previous request and "end single user val CFM" on this
request. Group mng will ignore that CFM on start

Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - the group index	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventRemoveGroupDueToStationInSuValWaitingLa(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{
    uint8 groupIndex = *(uint8 *)parameters;
	
	groupManagerClearGroupBitInStationBitmap(groupManagerStationDataBaseEntry, groupIndex);
#ifdef GROUP_MANAGER_DEBUG								
	ILOG0_DD("groupManagerStationEventRemoveGroupDueToStationInSuValWaitingLa: groupIndex %d, station index %d",
		groupIndex, groupManagerStationDataBaseEntry->stationIndex);
#endif	
	if(groupManagerStationDataBaseEntry->primaryGroup == groupIndex)
	{
#ifdef GROUP_MANAGER_DEBUG									
		ILOG0_V("groupManagerStationEventRemoveGroupDueToStationInSuValWaitingLa: the primary group of station waiting to start SU val is removed, cancel SU val");
#endif
		groupManagerSendEndSingleUserValidationMessage(groupManagerStationDataBaseEntry->stationIndex);
		groupManagerStationDataBaseEntry->primaryGroup = GROUP_MANAGER_INVALID_GROUP_INDEX;
		groupManagerSetStationSuTxopAndAmpduLimit(groupManagerStationDataBaseEntry->stationIndex, TRUE , SU_NUM_OF_MPDUS_IN_A_MPDU);
		// possible fix for Selector's primary station mapping bug
		//groupManagerConfigureStationForSelector(groupManagerStationDataBaseEntry->stationIndex, FALSE, 0);
		groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_CANCELED_GROUP_FAILED);	
	}		
}

/**********************************************************************************

groupManagerStationEventRemoveGroupDueToStationInSingleUserValidation


Description:
------------
group is removed due to group failure in the state of 'single user validation'
The group already added by LA and single user validation for this station is on-going


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - the group index	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventRemoveGroupDueToStationInSingleUserValidation(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{
    uint8 groupIndex = *(uint8 *)parameters;
	
	groupManagerClearGroupBitInStationBitmap(groupManagerStationDataBaseEntry, groupIndex);
#ifdef GROUP_MANAGER_DEBUG								
	ILOG0_DDD("groupManagerStationEventRemoveGroupDueToStationInSingleUserValidation: group index %d, station index %d, primary group index = %d", groupIndex, groupManagerStationDataBaseEntry->stationIndex, groupManagerStationDataBaseEntry->primaryGroup);
#endif
	if(groupManagerStationDataBaseEntry->primaryGroup == groupIndex)
	{   
		/* remove timeout event on single user validation because the process is aborted*/
		CalendarWheel_RemoveEntry(&GroupManagerStationsCalendarWheel, (CalendarWheelEntry_t*)groupManagerStationDataBaseEntry);
		/* send end single user validation and move to state which waits for CFM from LA*/

		//primary group of station which is in process of SU val is removed. Cancel Su val
		groupManagerSendEndSingleUserValidationMessage(groupManagerStationDataBaseEntry->stationIndex);
		groupManagerStationDataBaseEntry->primaryGroup = GROUP_MANAGER_INVALID_GROUP_INDEX;
		groupManagerSetStationSuTxopAndAmpduLimit(groupManagerStationDataBaseEntry->stationIndex, TRUE, SU_NUM_OF_MPDUS_IN_A_MPDU);
		groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_CANCELED_GROUP_FAILED);	
	}		
}

/**********************************************************************************

groupManagerStationEventRemoveGroupDueToStationInSingleUserValidationEnded


Description:
------------
group is removed due to group failure in the states of 'single user validation failed'
or 'single user validation succeeded'. In these states Group manager is waiting for 
"single user validation end cfm" from LA and group removed event occurrs.
Only the state of the station is changed and there is nothing to send to LA because 
end message was already sent

Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - the group index	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventRemoveGroupDueToStationInSingleUserValidationEnded(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{
    uint8 groupIndex = *(uint8 *)parameters;
	
	groupManagerClearGroupBitInStationBitmap(groupManagerStationDataBaseEntry, groupIndex);

	if(groupManagerStationDataBaseEntry->primaryGroup == groupIndex)
	{   
		groupManagerStationDataBaseEntry->primaryGroup = GROUP_MANAGER_INVALID_GROUP_INDEX;
		groupManagerSetStationSuTxopAndAmpduLimit(groupManagerStationDataBaseEntry->stationIndex, TRUE, SU_NUM_OF_MPDUS_IN_A_MPDU);
		groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_CANCELED_GROUP_FAILED);	
	}		
}

/**********************************************************************************

groupManagerStationEventRemoveGroupInNotifying


Description:
------------
group is removed in the state of 'notifying'. Group is removed not due to group failure
note: station can be member of only one group in notify state, can't participate 
in multiple groups which are found in notify state


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - the group index	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventRemoveGroupInNotifying(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 
	StaId stationIndex = groupManagerStationDataBaseEntry->stationIndex;
    uint8 groupIndex = *(uint8 *)parameters;;
#ifdef GROUP_MANAGER_DEBUG							
	ILOG0_DD("groupManagerStationEventRemoveGroupInNotifying: groupIndex %d, stationIndex %d", groupIndex, stationIndex);
#else
	UNUSED_PARAM(stationIndex);
#endif

	groupManagerClearGroupBitInStationBitmap(groupManagerStationDataBaseEntry, groupIndex);
	
	/* move to single user validation state even if there are no virtual groups this station is member of.
	because group is removed not because of failure so the station is not punished*/
	groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_SINGLE_USER);
			
}

/**********************************************************************************

groupManagerStationEventRemoveGroupInProbing


Description:
------------
group is removed in the state of 'probing'. Removal process is complete and this function is invoked
for all group members which where in probing state


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - the group index	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventRemoveGroupInProbing(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 
	uint16 notifyingGroupsBitmap = 0;
	uint16 probingGroupsBitmap = 0;
	StaId stationIndex = groupManagerStationDataBaseEntry->stationIndex;
	uint8 groupIndex = *(uint8 *)parameters;
	
	groupManagerClearGroupBitInStationBitmap(groupManagerStationDataBaseEntry, groupIndex);
	/* if this station acted as primary for the probing group which is in fact was removed
	update in Selector that this station is no longer primary of this group*/
	if(groupIndex == groupManagerStationDataBaseEntry->tempPrimaryGroup)
    {
#ifdef GROUP_MANAGER_DEBUG									
    	ILOG0_DD("groupManagerStationEventRemoveGroupInProbing: Station %d is primary of probing group %d which is beeing removed. configure station in selector to SU",
			stationIndex, groupIndex);
#endif    	
		groupManagerStationDataBaseEntry->tempPrimaryGroup = GROUP_MANAGER_INVALID_GROUP_INDEX;	
		groupManagerConfigureStationForSelector(stationIndex, FALSE, 0, TRUE);
	}
	
    probingGroupsBitmap = groupManagerStationDataBaseEntry->groupsBitmap & GroupManagerGlobalParametersDb.probingGroupsBitmap;
	/* if this station is still member of additional probing groups - it remains in probing state and there is nothing to be done,
	otherwise check to which state this station must move depending on which groups it is member of*/
	if(0 == probingGroupsBitmap)
	{
		notifyingGroupsBitmap = groupManagerStationDataBaseEntry->groupsBitmap & GroupManagerGlobalParametersDb.notifyingGroupsBitmap;
		if(notifyingGroupsBitmap)
		{
#ifdef GROUP_MANAGER_DEBUG										
			ILOG0_D("groupManagerStationEventRemoveGroupInProbing: Station %d remains member of only notify groups",
			stationIndex);
#endif			
			groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_NOTIFY_STATION);
		}
		else
		{
#ifdef GROUP_MANAGER_DEBUG										
			ILOG0_D("groupManagerStationEventRemoveGroupInProbing: Station %d is not a member of any active/probing/notify group", stationIndex);
#endif
			/* change state to single user validation - station is not punished because group removed not because of failure */
			groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_SINGLE_USER);
		}
	}
}

/**********************************************************************************

groupManagerStationEventRemoveGroupInActive


Description:
------------
group is removed for station in the state of 'active'. 
Group removal process is complete and now station is notified. As the result 
station's state might change depending on which groups it's member of. But the final
state won't be HALT because group was removed not because of failure


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - the group index	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventRemoveGroupInActive(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 
	groupManagerRemoveGroupInActive(groupManagerStationDataBaseEntry, parameters, FALSE);

}

/**********************************************************************************

groupManagerStationEventRemoveGroupInSingleUserValidationPending


Description:
------------
group is removed in the state of 'single user validation pending'


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - the group index	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventRemoveGroupInSingleUserValidationPending(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 
	groupManagerRemoveGroupInSingleUserValidationPending(groupManagerStationDataBaseEntry, parameters, FALSE);
}

/**********************************************************************************

groupManagerStationEventRemoveGroupInSuValWaitingLa


Description:
------------
group is removed in the state of 'single user validation - waiting for link adaptation'
The function is invoked when station is waiting for LA confirmation
on "Start single user validation" on station and remove group event occurred. 
Regular "remove group" not because of failure

Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - the group index	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventRemoveGroupInSuValWaitingLa(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 
    uint8 groupIndex = 0;

	groupIndex = *(uint8 *)parameters;
	groupManagerClearGroupBitInStationBitmap(groupManagerStationDataBaseEntry, groupIndex);
	/*if removed group was my primary group*/
	if(groupManagerStationDataBaseEntry->primaryGroup == groupIndex)
	{
		groupManagerSendEndSingleUserValidationMessage(groupManagerStationDataBaseEntry->stationIndex);
		groupManagerStationDataBaseEntry->primaryGroup = GROUP_MANAGER_INVALID_GROUP_INDEX;
		groupManagerSetStationSuTxopAndAmpduLimit(groupManagerStationDataBaseEntry->stationIndex, TRUE, SU_NUM_OF_MPDUS_IN_A_MPDU);
		// possible fix for Selector's primary station mapping bug
		// groupManagerConfigureStationForSelector(groupManagerStationDataBaseEntry->stationIndex, FALSE, 0);
		groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_CANCELED_GROUP_REMOVED);	
	}
	/* when EndSingleUserValidation LA cfm will arrive and station in state SINGLE_USER_VALIDATION_CANCELED_GROUP_REMOVED
	   group manager will perform the process which evaluates new state of the station based on its membership in groups*/
}

/**********************************************************************************

groupManagerStationEventRemoveGroupInSingleUserValidation


Description:
------------
group is removed in the state of 'single user validation'


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - the group index	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventRemoveGroupInSingleUserValidation(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 
    uint8 groupIndex = 0;

	groupIndex = *(uint8 *)parameters;
	groupManagerClearGroupBitInStationBitmap(groupManagerStationDataBaseEntry, groupIndex);

	if(groupManagerStationDataBaseEntry->primaryGroup == groupIndex)
	{   
		CalendarWheel_RemoveEntry(&GroupManagerStationsCalendarWheel, (CalendarWheelEntry_t*)groupManagerStationDataBaseEntry);
		groupManagerSendEndSingleUserValidationMessage(groupManagerStationDataBaseEntry->stationIndex);
		groupManagerStationDataBaseEntry->primaryGroup = GROUP_MANAGER_INVALID_GROUP_INDEX;
		groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_CANCELED_GROUP_REMOVED);	
		/* when EndSingleUserValidation LA cfm will arrive and station in state _INGLE_USER_VALIDATION_CANCELED_GROUP_REMOVED
	   group manager will perform the process which evaluates new state of teh station based on its membership in groups*/
	}		
}

/**********************************************************************************

groupManagerStationEventRemoveGroupInSingleUserValidationEnded


Description:
------------
group is removed in the states 'single user validation failed' or 'single user 
validation succeeded'. In these states station already processed results and only waiting 
for cfm on end single user validation


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - the group index	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventRemoveGroupInSingleUserValidationEnded(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 
    uint8 groupIndex = 0;

	groupIndex = *(uint8 *)parameters;
	groupManagerClearGroupBitInStationBitmap(groupManagerStationDataBaseEntry, groupIndex);

	if(groupManagerStationDataBaseEntry->primaryGroup == groupIndex)
	{   
		groupManagerStationDataBaseEntry->primaryGroup = GROUP_MANAGER_INVALID_GROUP_INDEX;
		groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_CANCELED_GROUP_REMOVED);	
	}			
}

/**********************************************************************************

groupManagerStationEventTrainingFailedInSingleUser


Description:
------------
training failed in the state of 'single user'. The station participated in training
and received bad results for all its combinations. The station will be halted if
not a member of any virtual group


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventTrainingFailedInSingleUser(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 
	StaId stationIndex = 0;
	UNUSED_PARAM(parameters);

	stationIndex = groupManagerStationDataBaseEntry->stationIndex;
	if(0 == groupManagerStationDataBaseEntry->virtualGroupsBitmap)
	{   
		Utils_SetBitInBitmap(GroupManagerGlobalParametersDb.trainingBitmaps.haltStationsBitmap.stationBitmap, stationIndex);	
		
		groupManagerStationDataBaseEntry->numberOfHalts ++;
		if(groupManagerStationDataBaseEntry->numberOfHalts == GROUP_MANAGER_MAX_HALT_THRESHOLD)
		{
			groupManagerStationDataBaseEntry->numberOfHalts = 0;
			groupManagerStationDataBaseEntry->timeout = GROUP_MANAGER_MAX_HALT_PERIOD;
		}
		else
		{		
			groupManagerStationDataBaseEntry->timeout = groupManagerStationDataBaseEntry->numberOfHalts;
		}
        CalendarWheel_AddEntry(&GroupManagerStationsCalendarWheel, (CalendarWheelEntry_t *)groupManagerStationDataBaseEntry, groupManagerStationDataBaseEntry->timeout);

		groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_HALT);	
		Utils_ZeroBitInBitmap(GroupManagerGlobalParametersDb.waitingGroupsMarksBitmap, stationIndex);
	}			
}

/**********************************************************************************

groupManagerStationEventNotificationStartedInSingleUser


Description:
------------
notificaiton started in the state of 'single user' 


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - the group index	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventNotificationStartedInSingleUser(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 
	uint8 groupIndex = 0;

	groupIndex = *(uint8 *)parameters;
	groupManagerSetGroupBitInStationBitmap(groupManagerStationDataBaseEntry, groupIndex);
	
	groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_NOTIFY_STATION);	
}

/**********************************************************************************

groupManagerStationEventProbingStartedInNotifying


Description:
------------
probing started in the state of 'notifying' 


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventProbingStartedInNotifying(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 
	UNUSED_PARAM(parameters);
	/* change state to probing*/
	groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_PROBING);	
}

/**********************************************************************************

groupManagerStationEventActivateGroupInProbing


Description:
------------
activate group in the state of 'probing'. probing ended with good results this group is activated.
The function is invoked on each member of the probing group to notify it that the group will become active


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventActivateGroupInProbing(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 
	StaId stationIndex = groupManagerStationDataBaseEntry->stationIndex;
	uint8 groupIndex = 0;
	GroupManagerProbingResultsMessage_t *groupManagerProbingResultsParameters = NULL;
	groupManagerProbingResultsParameters = (GroupManagerProbingResultsMessage_t *)parameters;
	groupIndex = groupManagerProbingResultsParameters->probingGroupIndex;
#ifdef GROUP_MANAGER_DEBUG							
	ILOG0_DD("groupManagerStationEventActivateGroupInProbing: group idx %d, SID %d ",groupIndex, stationIndex);
#endif
	
	/* if the station in probing state meaning it has no other group in which it will be a primary so
	 * this station will be a primary for the new active group*/
	groupManagerSetStationPrimaryBitInGroupsDb(stationIndex, groupIndex);
	/* set new primary group index in station db*/
	groupManagerStationDataBaseEntry->primaryGroup = groupIndex;
	
	/* handle case in which station participates in multiple probing groups.
	   if tempPrimaryGroup contains index of this group meaning this station was primary for this group or if
	   tempPrimaryGroup contains invalid index meaning this station is a member of at least one probing group and 
	   it doesn't acts as primary for any probing group. In both cases above this station will be set as primary 
	   for the new active group in Selector*/   
	if((groupIndex == groupManagerStationDataBaseEntry->tempPrimaryGroup) || (GROUP_MANAGER_INVALID_GROUP_INDEX == groupManagerStationDataBaseEntry->tempPrimaryGroup))
	{
#ifdef GROUP_MANAGER_DEBUG									
		ILOG0_D("groupManagerStationEventActivateGroupInProbing: tempPrimaryGroup before activation is %d ",groupManagerStationDataBaseEntry->tempPrimaryGroup);	
#endif
		groupManagerStationDataBaseEntry->tempPrimaryGroup = GROUP_MANAGER_INVALID_GROUP_INDEX;
		groupManagerConfigureStationForSelector(stationIndex, TRUE, groupIndex, TRUE);
	
		groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_ACTIVE);	
		groupManagerTryToStartNewSingleUserValidation(stationIndex);
	}
	/* this station is not primary member of this probing group but it acts as primary member of additional probing group
		(because its tempPrimaryGroup field is not invalid and not index of this group). The additional probing group
		is still in process of probing */ 
	else
	{
		/* move to state tempActive because current probing group is activated and the station became primary in this group
		so it must be tempActive for additional probing group*/
		groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_TEMP_ACTIVE);
		Utils_ZeroBitInBitmap(GroupManagerGlobalParametersDb.singleUserValidationBitmap, stationIndex);		
	}		
}

/**********************************************************************************

groupManagerStationEventActivateGroupInTempActive


Description:
------------
activate group in the state of 'temp active'. 
This function is called on tempActive station when probing results are good. It is possible
that this station is tempActive for some other probing group and it is a member of this probing group.


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventActivateGroupInTempActive(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 
	StaId stationIndex = groupManagerStationDataBaseEntry->stationIndex;
	uint8 primaryGroupIndex = 0;
	uint8 groupIndex = 0;
	uint8 bestGroupIndex = 0;
	GroupManagerProbingResultsMessage_t *groupManagerProbingResultsParameters = NULL;
	
	groupManagerProbingResultsParameters = (GroupManagerProbingResultsMessage_t *)parameters;
	groupIndex = groupManagerProbingResultsParameters->probingGroupIndex;

	/* find index of the group with the best score among all groups this station is member of including
	the probing group. the latest scores provided as parameter to the function*/
	bestGroupIndex = groupManagerChooseBestGroupFromProbingMessage(groupManagerStationDataBaseEntry, groupManagerProbingResultsParameters);
	
	/* no need to wait for group marks , just received all group marks in probing results*/
	Utils_ZeroBitInBitmap(GroupManagerGlobalParametersDb.waitingGroupsMarksBitmap, stationIndex);
	
	primaryGroupIndex = groupManagerStationDataBaseEntry->primaryGroup;
	/* if this station has a previous primary group which is not the one with the best score the station won't act as  
	   primary for its previous group any longer, set to 0 its primary bit in previous primary group db*/
	if((GROUP_MANAGER_INVALID_GROUP_INDEX != primaryGroupIndex) && (bestGroupIndex != primaryGroupIndex))
	{	
		groupManagerClearStationPrimaryInGroup(stationIndex, primaryGroupIndex);
	}	
	/* turn on new primary bit in the probing group db*/
	groupManagerSetStationPrimaryBitInGroupsDb(stationIndex, bestGroupIndex);
	/* configure new primary index in stations db*/ 
	groupManagerStationDataBaseEntry->primaryGroup = bestGroupIndex;

	/* At this point not allowed to configure Selectors group db: as long as this station acts as tempActive for probing group
	   this station written in Selector db as primary for the group. Until probing for that group is not complete
	   it is not allowed to change Selector's configuration*/
	
	/* handle case in which this station is tempActive for this probing group and probing is complete*/
	if(groupIndex == groupManagerStationDataBaseEntry->tempPrimaryGroup)
	{
		/* if this probing group is not the one with the best score for this station */
		if(bestGroupIndex != groupManagerStationDataBaseEntry->tempPrimaryGroup)
		{
			/* turn off the primary bit of the station in Probing group db*/
			groupManagerClearStationPrimaryInProbingGroup(stationIndex, groupIndex);
		}
		/* this station is no longer tempActive, configure the best group index for this station in Selectors db*/
		groupManagerStationDataBaseEntry->tempPrimaryGroup = GROUP_MANAGER_INVALID_GROUP_INDEX;
		/*note: no need to change TXOP and #MPDUs in AMPDU in StaDb in SU because this was already changed when station became 
		        primary in the beginning of the probing */
		groupManagerConfigureStationForSelector(stationIndex, TRUE, bestGroupIndex, FALSE);
		
		/* New primary added, trigger single user validation*/
		groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_ACTIVE);
		groupManagerTryToStartNewSingleUserValidation(stationIndex);
	}
}

/**********************************************************************************

groupManagerStationEventActivateGroupInActive


Description:
------------
activate group in the state of 'active' 


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventActivateGroupInActive(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 
	StaId stationIndex =  groupManagerStationDataBaseEntry->stationIndex;
	uint8 primaryGroupIndex = 0;
	uint8 bestGroupIndex = 0;
	uint8 indexInArray = 0;
	GroupManagerProbingResultsMessage_t *probingResults = (GroupManagerProbingResultsMessage_t*)parameters;

	/* no need to wait for group marks because the updated group marks were returned in probing results*/
	Utils_ZeroBitInBitmap(GroupManagerGlobalParametersDb.waitingGroupsMarksBitmap, stationIndex);
	
	bestGroupIndex = groupManagerChooseBestGroupFromProbingMessage(groupManagerStationDataBaseEntry, (GroupManagerProbingResultsMessage_t *)parameters);

	primaryGroupIndex = groupManagerStationDataBaseEntry->primaryGroup;
	/* if new best group is not the previous primary group*/
	if(bestGroupIndex != primaryGroupIndex)
	{	
		/* if the station was a primary for some group, clear stations bit in this previous group DB
		 no need to trigger SU val process the station is already part of it*/
		if(GROUP_MANAGER_INVALID_GROUP_INDEX != primaryGroupIndex)
		{
			groupManagerClearStationPrimaryInGroup(stationIndex, primaryGroupIndex);
			groupManagerConfigureStationAsPrimaryInGroup(stationIndex, bestGroupIndex);
			Utils_SetBitInBitmap(GroupManagerGlobalParametersDb.singleUserValidationBitmap, stationIndex);
		}
	
		/* the station was secondary for all active groups (operates better in SU mode) 
			before turning it to primary of some group check stationGroupMark field in probingResults message and compare it to the score
			of new best group. If SU score of the station is higher do not turn this station to primary of new group and do not 
			turn on its bit in GroupManagerGlobalParametersDb.singleUserValidationBitmap and do not trigger groupManagerTryToStartNewSingleUserValidation
			*/
			
		else
		{
			//ASSERT_IN_LOOP-FW
			ASSERT(bestGroupIndex < GROUP_MANAGER_MAX_NUMBER_OF_GROUPS);
			for(indexInArray = 0; indexInArray < GroupManagerGroupDb[probingResults->probingGroupIndex].numberOfParticipatingStations; indexInArray++)
			{
				if(stationIndex == GroupManagerGroupDb[probingResults->probingGroupIndex].participatingStations[indexInArray])
				{
					if((probingResults->groupMarks[bestGroupIndex] + GROUP_MANAGER_PROBING_GROUP_MARK_MARGIN) > probingResults->stationSuMark[indexInArray])
					{
						groupManagerConfigureStationAsPrimaryInGroup(stationIndex, bestGroupIndex);
						Utils_SetBitInBitmap(GroupManagerGlobalParametersDb.singleUserValidationBitmap, stationIndex);
						groupManagerTryToStartNewSingleUserValidation(stationIndex);
					}
				}
			}	
		}
	}	
}

/**********************************************************************************

groupManagerStationEventActivateGroupInSingleUserValidationPending


Description:
------------
activate group in the state of 'single user validation pending' 


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventActivateGroupInSingleUserValidationPending(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 
	StaId stationIndex = groupManagerStationDataBaseEntry->stationIndex;
	uint8 primaryGroupIndex = 0;
	uint8 bestGroupIndex = 0; 

	bestGroupIndex = groupManagerChooseBestGroupFromProbingMessage(groupManagerStationDataBaseEntry, (GroupManagerProbingResultsMessage_t *)parameters);

	primaryGroupIndex = groupManagerStationDataBaseEntry->primaryGroup;
	/* if new best group is not the current primary group of the station and the station in state SingleUserValidationPending
	  (meaning it is already in calendar wheel waiting for SU val) Abort the pending SU val process and return this station to be primary active for new
	group. The reason to abort SU val pending is : if this station is the only primary for this group the su val process will
	disable this group for a while and we would like to give this group some time to transmit*/
	if(bestGroupIndex != primaryGroupIndex)
	{	
		CalendarWheel_RemoveEntry(&GroupManagerStationsCalendarWheel, (CalendarWheelEntry_t*)groupManagerStationDataBaseEntry);
		groupManagerClearStationPrimaryInGroup(stationIndex, primaryGroupIndex);
	
		groupManagerConfigureStationAsPrimaryInGroup(stationIndex, bestGroupIndex);

		groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_ACTIVE);
		// TODO: check if this station is primary for static group and if so do not add to single user validation candidates
		Utils_SetBitInBitmap(GroupManagerGlobalParametersDb.singleUserValidationBitmap, stationIndex);
		 
		groupManagerStartNewSingleUserValidation();	
	}
}

/**********************************************************************************

groupManagerStationEventActivateGroupInSuValWaitingLa


Description:
------------
activate group in the state of 'single user validation waiting for link adaptation' 
Probing results are ready and this function is invoked on station (member of probing group)
which is waiting for LA cfm on start user validation request


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventActivateGroupInSuValWaitingLa(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 
	StaId stationIndex = groupManagerStationDataBaseEntry->stationIndex;
	uint8 primaryGroupIndex = 0;
	uint8 bestGroupIndex = 0;

	bestGroupIndex = groupManagerChooseBestGroupFromProbingMessage(groupManagerStationDataBaseEntry, (GroupManagerProbingResultsMessage_t *)parameters);
	
	primaryGroupIndex = groupManagerStationDataBaseEntry->primaryGroup;
	/* if new best group is not current primary group of the station and primary group can be changed - abort single user validation
	   of the station and change primary group for the station */
	if(bestGroupIndex != primaryGroupIndex)
	{	
		groupManagerClearStationPrimaryInGroup(stationIndex, primaryGroupIndex);
		groupManagerConfigureStationAsPrimaryInGroup(stationIndex, bestGroupIndex);

		/* notify LA that SU val process is ended. Group mng will receive CFM on previously sent "start" and CFM on end.
		   it will ignore the cfm on start and will change to state active on cfm on end*/ 
		groupManagerSendEndSingleUserValidationMessage(stationIndex);
		/* in state SINGLE_USER_VALIDATION_SUCCEEDED wait for LA cfm*/
		groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_SUCCEEDED);	
	}	
}

/**********************************************************************************

groupManagerStationEventActivateGroupInSingleUserValidation


Description:
------------
activate group in the state of 'single user validation'. Station waits for single user 
validation results from LA but it also participated in probing group and now this 
probing group is activated
The function performs the same actions as "groupManagerStationEventActivateGroupInSuValWaitingLa"
above but in addition removes this station from calendar wheel to avoid SU val timeout in the future

Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 

	// TODO: can be one function for this one and above with boolean parameter remove_calendar_wheel
**********************************************************************************/
static void groupManagerStationEventActivateGroupInSingleUserValidation(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 
	StaId stationIndex = groupManagerStationDataBaseEntry->stationIndex;
	uint8 primaryGroupIndex = groupManagerStationDataBaseEntry->primaryGroup;
	uint8 bestGroupIndex = groupManagerChooseBestGroupFromProbingMessage(groupManagerStationDataBaseEntry, (GroupManagerProbingResultsMessage_t *)parameters);
	
	if(bestGroupIndex != primaryGroupIndex)
	{	
		CalendarWheel_RemoveEntry(&GroupManagerStationsCalendarWheel, (CalendarWheelEntry_t*)groupManagerStationDataBaseEntry);
		groupManagerClearStationPrimaryInGroup(stationIndex, primaryGroupIndex);

		groupManagerConfigureStationAsPrimaryInGroup(stationIndex, bestGroupIndex);
		
		groupManagerSendEndSingleUserValidationMessage(stationIndex);
		groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_SUCCEEDED);	
	}	
}

/**********************************************************************************

groupManagerStationEventActivateGroupInSingleUserValidationEnded


Description:
------------
activate group in the states 'single user validation failed' or 'single user 
validation succeeded'
'singlue user validation failed' - station is better as SU
'single user validation succeeded' - station is better in group


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventActivateGroupInSingleUserValidationEnded(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 
	StaId stationIndex = groupManagerStationDataBaseEntry->stationIndex;
	uint8 primaryGroupIndex = groupManagerStationDataBaseEntry->primaryGroup;
	uint8 bestGroupIndex = groupManagerChooseBestGroupFromProbingMessage(groupManagerStationDataBaseEntry, (GroupManagerProbingResultsMessage_t *)parameters);

	/* if primary group is about to change , ignore single user validation outcome (even if station is better in SU mode)
	   change primary group to the best group found.
	   No need to send SU val end to LA because it was already sent and station is waiting for 'end cfm'. */
	if(bestGroupIndex != primaryGroupIndex)
	{	
		groupManagerClearStationPrimaryInGroup(stationIndex, primaryGroupIndex);
#ifdef GROUP_MANAGER_DEBUG									
		ILOG0_DD("groupManagerStationEventActivateGroupInSingleUserValidationEnded: SID %d, configured to be primary of group %d", stationIndex, bestGroupIndex);
#endif
		groupManagerConfigureStationAsPrimaryInGroup(stationIndex, bestGroupIndex);
		
		groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_SUCCEEDED);	
	}	
}

/**********************************************************************************

groupManagerStationEventActivateGroupInSingleUserValidationCanceled


Description:
------------
activate group in the states 'single user validation canceled - group removed' or 
'single user validation canceled -  group failed'
In both cases it is decided to go back to operate in MU mode with the best group found 
and move to GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_SUCCEEDED state in which 
station is waiting for LA cfm on signals sent previously


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventActivateGroupInSingleUserValidationCanceled(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 
	StaId stationIndex = groupManagerStationDataBaseEntry->stationIndex;
	uint8 bestGroupIndex = groupManagerChooseBestGroupFromProbingMessage(groupManagerStationDataBaseEntry, (GroupManagerProbingResultsMessage_t *)parameters);
		
	groupManagerConfigureStationAsPrimaryInGroup(stationIndex, bestGroupIndex);
	/* no need to set priority bit, station which starts single user validation it's priority bit stil set to active*/
	groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_SUCCEEDED);	
}

/**********************************************************************************

groupManagerStationEventTimerInSingleUserValidationPending


Description:
------------
timer event in the state 'single user validation pending'.
Message is sent to LA to trigger single user validation on the station which timer expired in waiting for SU val.


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventTimerInSingleUserValidationPending(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 
	LaStartSingleUserValidationParams_t *startSingleUserValidationParameters = NULL;
	K_MSG *linkAdaptationMessage	= NULL;
	uint8 vapId = 0;
	UNUSED_PARAM(parameters);
	linkAdaptationMessage = OSAL_GET_MESSAGE(sizeof(LaStartSingleUserValidationParams_t));
	startSingleUserValidationParameters = ((LaStartSingleUserValidationParams_t *)linkAdaptationMessage->abData);
	startSingleUserValidationParameters->stationIndex = groupManagerStationDataBaseEntry->stationIndex;
	StaDb_GetVapId(groupManagerStationDataBaseEntry->stationIndex, &vapId);
	
	CalendarWheel_RemoveEntry(&GroupManagerStationsCalendarWheel, (CalendarWheelEntry_t*)groupManagerStationDataBaseEntry);

	groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_WAITING_FOR_LINK_ADAPTATION);	
	OSAL_SEND_MESSAGE(LINK_ADAPTATION_START_SINGLE_USER_VALIDATION, TASK_LINK_ADAPTATION, linkAdaptationMessage, vapId);
}

/**********************************************************************************

groupManagerStationEventTimerInSingleUserValidation


Description:
------------
timer event in the state 'single user validation'
timeout received while waiting for single user validation process to end
send 'single user validation end' and remove timeout event from calendar wheel.

Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventTimerInSingleUserValidation(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 
	UNUSED_PARAM(parameters);
	CalendarWheel_RemoveEntry(&GroupManagerStationsCalendarWheel, (CalendarWheelEntry_t*)groupManagerStationDataBaseEntry);
    groupManagerSendEndSingleUserValidationMessage(groupManagerStationDataBaseEntry->stationIndex);
	
	groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_SUCCEEDED);	
}

/**********************************************************************************

groupManagerStationEventTimerInHalt


Description:
------------
timer event in the state 'halt'.
Timeout on halt state has expired, station can move to 'candidate' state. 
turn off halt bit of the station


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventTimerInHalt(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 
	StaId stationIndex = 0;
	UNUSED_PARAM(parameters);

	stationIndex = groupManagerStationDataBaseEntry->stationIndex;
	CalendarWheel_RemoveEntry(&GroupManagerStationsCalendarWheel, (CalendarWheelEntry_t*)groupManagerStationDataBaseEntry);
	Utils_ZeroBitInBitmap(GroupManagerGlobalParametersDb.trainingBitmaps.haltStationsBitmap.stationBitmap, stationIndex);
	groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_SINGLE_USER);	
}

/**********************************************************************************

groupManagerStationEventSuValStartCfmInSuValWaitingLa


Description:
------------
handle the event of single user validation start confirmation in the state of 
'single user validation - waiting for link adaptation'


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventSuValStartCfmInSuValWaitingLa(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 
	StaId stationIndex = groupManagerStationDataBaseEntry->stationIndex;
	UNUSED_PARAM(parameters);

	/* station will operate in SU val mode - configure group of the station to be without this primary*/
#ifdef GROUP_MANAGER_DEBUG								
	ILOG0_D("groupManagerStationEventSuValStartCfmInSuValWaitingLa: SID %d configured to SU mode", stationIndex);
#endif
	groupManagerConfigureStationForSelector(stationIndex, FALSE, 0, TRUE);
	/* timeout of su validation process*/
	CalendarWheel_AddEntry(&GroupManagerStationsCalendarWheel, (CalendarWheelEntry_t *)groupManagerStationDataBaseEntry, GROUP_MANAGER_SINGLE_USER_VALIDATION_RESULTS_TIMER);
	groupManagerChangeStationState(&GroupManagerStationDb[stationIndex], GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION);
}

/**********************************************************************************

groupManagerStationEventSuValResultsReceivedInSuVal


Description:
------------
handle the event of single user validation results received in the state of 'single
user validation'


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - pointer to single user validation results message parameters	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventSuValResultsReceivedInSuVal(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 
	uint8 bestGroupIndex = 0;
	uint8 bestGroupMark = 0;
	uint16 stationActiveGroupsBitmap = 0;
	StaId stationIndex = groupManagerStationDataBaseEntry->stationIndex;
	GroupManagerSingleUserResultsMessage_t *groupManagerMessageParameters = NULL;	
	
	/* remove timeout event - results arrived on time*/
    CalendarWheel_RemoveEntry(&GroupManagerStationsCalendarWheel, (CalendarWheelEntry_t*)groupManagerStationDataBaseEntry);
    groupManagerSendEndSingleUserValidationMessage(stationIndex);
	groupManagerMessageParameters = (GroupManagerSingleUserResultsMessage_t *)parameters;
	stationActiveGroupsBitmap = groupManagerStationDataBaseEntry->groupsBitmap & GroupManagerGlobalParametersDb.activeGroupsBitmap;

	bestGroupIndex = groupManagerFindBestGroup(stationActiveGroupsBitmap, groupManagerMessageParameters->groupMarks);
	bestGroupMark = groupManagerMessageParameters->groupMarks[bestGroupIndex];

	if(bestGroupMark + GroupManagerGlobalParametersDb.muConfigParams.singleUserPerformenceMargin < groupManagerMessageParameters->stationMark)
	{
		/* Single user validation failed - the station is better as single user*/
		groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_FAILED);
		// no need to change txopMode and mpdusin StaDb because these parameters were changed in the beginning of su validation process of the station
	}
	else
	{
		/* Single user validation succeeded */
		groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_SUCCEEDED);
		if(bestGroupMark != groupManagerStationDataBaseEntry->primaryGroup)
		{
			/* Best group is better than primary - switch groups */
					
			groupManagerClearStationPrimaryInGroup(stationIndex, groupManagerStationDataBaseEntry->primaryGroup);

			groupManagerConfigureStationAsPrimaryInGroup(stationIndex, bestGroupIndex);
		}
	}
}

/**********************************************************************************

groupManagerStationEventSuValEndCfmInSuValFailed


Description:
------------
handle the event of single user validation end confirmation received in the state of 
'single user validation failed'


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------groupManagerStationEventSuValEndCfmInSuValFailed
	void - 
	
**********************************************************************************/
static void groupManagerStationEventSuValEndCfmInSuValFailed(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 
	uint8 primaryGroupIndex = 0;
	StaId stationIndex = 0;
	UNUSED_PARAM(parameters);
    stationIndex = groupManagerStationDataBaseEntry->stationIndex;
	primaryGroupIndex = groupManagerStationDataBaseEntry->primaryGroup;

	groupManagerClearStationPrimaryInGroup(stationIndex, primaryGroupIndex);
	
	groupManagerStationDataBaseEntry->primaryGroup = GROUP_MANAGER_INVALID_GROUP_INDEX;
	groupManagerStartNewSingleUserValidation();
	groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_ACTIVE);
}

/**********************************************************************************

groupManagerStationEventSuValEndCfmInSuValSucceeded


Description:
------------
handle the event of single user validation end confirmation received in the state of 
'single user validation succeeded'


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventSuValEndCfmInSuValSucceeded(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 
	StaId stationIndex = groupManagerStationDataBaseEntry->stationIndex;
	UNUSED_PARAM(parameters);

#ifdef GROUP_MANAGER_DEBUG							   
	ILOG0_DD("groupManagerStationEventSuValEndCfmInSuValSucceeded: SID %d, configured to work in MU mode, primary for group %d", stationIndex, groupManagerStationDataBaseEntry->primaryGroup);
#endif
	groupManagerConfigureStationForSelector(stationIndex, TRUE, groupManagerStationDataBaseEntry->primaryGroup, TRUE);	
		
	Utils_SetBitInBitmap(GroupManagerGlobalParametersDb.singleUserValidationBitmap, stationIndex);
	groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_ACTIVE);
	groupManagerStartNewSingleUserValidation();
}

/**********************************************************************************

groupManagerStationEventSuValEndCfmInSuValCanceledGroupRemoved


Description:
------------
handle the event of single user validation end confirmation received in the state of 
'single user validation canceled - group removed'


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventSuValEndCfmInSuValCanceledGroupRemoved(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 
	UNUSED_PARAM(parameters);
	groupManagerSuValEndCfmInSuValCanceled(groupManagerStationDataBaseEntry, parameters, FALSE);
}

/**********************************************************************************

groupManagerStationEventSuValEndCfmInSuValCanceledGroupFailed


Description:
------------
handle the event of single user validation end confirmation received in the state of 
'single user validation canceled - group failed'


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventSuValEndCfmInSuValCanceledGroupFailed(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 
	UNUSED_PARAM(parameters);
	groupManagerSuValEndCfmInSuValCanceled(groupManagerStationDataBaseEntry, parameters, TRUE);
}
	

/**********************************************************************************

groupManagerStationEventSuValEndCfmInRemovingWaitingForTrainingAndLaCfm


Description:
------------
handle the event of single user validation end confirmation received in the state of 
'removing - waiting for training and link adaptation confirmation'
LA cfm arrived need to wait for training cfm. In this state the current station was removed
and at this point we have additional free spot for some station to start single user validation


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventSuValEndCfmInRemovingWaitingForTrainingAndLaCfm(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 	
	UNUSED_PARAM(parameters);
	groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_TRAINING_CONFIRMATION);
	/* this station was rmeoved, free spot available for single user validation. Trigger a process to find new candidate*/
	groupManagerStartNewSingleUserValidation();
}

/**********************************************************************************

groupManagerStationEventSuValEndCfmInRemovingWaitingForLaCfm


Description:
------------
handle the event of single user validation end confirmation received in the state of 
'removing - waiting for link adaptation confirmation'.
This happens after Training confirmation during station removal has arrived earlier.
Can trigger process which will remove all groups that this removed station was member of.


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventSuValEndCfmInRemovingWaitingForLaCfm(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 	
	StaId stationIndex = 0;
	UNUSED_PARAM(parameters);

	stationIndex = groupManagerStationDataBaseEntry->stationIndex;
	
	if(0 == groupManagerStationDataBaseEntry->groupsBitmap)
	{
		/* if this  station didn't participate in any active group move directly to idle state*/
		groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_IDLE);	
		/* station removal is complete, all Cfms received. Send cfm on removal to BSS manager*/
		groupManagerSendConfirmToStationManager(stationIndex);
	}
	else
	{
		/* is the removed station was a member of some active groups - remove all these groups*/
		groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_ALL_GROUP_REMOVAL);
	}
	/* free spot is available - new candidates can be found for su. val process*/	
	groupManagerStartNewSingleUserValidation();
}

/**********************************************************************************

groupManagerStationEventTrainingCfmInHaltingWaitingForTrainingCfm


Description:
------------
handle the event of training confirmation received in the state 'halting - waiting
for training confirmation'.
The station will enter HALT state if:
	1. it's group failed in probing and it's not a member of any additional group
	2. LA provided bad result on it's active group and it's not a member of any additional groups
	3. Notify process of the group has failed because this station didn't reply

On HALT need to notify Training Mng to remove this station from training and to wait for CFM from Training mng.
At this point Training CFM HALT was received - can move to HALT state and start HALT timeout.
The timeout value was set in groupManagerStationEventRemoveGroupDueToStationInProbing or
groupManagerStationEventRemoveGroupDueToStationInNotifying (functions invoked when starting halt process)



Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventTrainingCfmInHaltingWaitingForTrainingCfm(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 	
	UNUSED_PARAM(parameters);
	groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_HALT);
	Utils_ZeroBitInBitmap(GroupManagerGlobalParametersDb.waitingGroupsMarksBitmap, groupManagerStationDataBaseEntry->stationIndex);
	CalendarWheel_AddEntry(&GroupManagerStationsCalendarWheel, (CalendarWheelEntry_t *)groupManagerStationDataBaseEntry, groupManagerStationDataBaseEntry->timeout);
}

/**********************************************************************************

groupManagerStationEventTrainingCfmInRemovingWaitingForTrainingAndLaCfm


Description:
------------
handle the event of training confirmation received in the state 'removing - waiting
for training and link adaptation confirmation'


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventTrainingCfmInRemovingWaitingForTrainingAndLaCfm(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 
	UNUSED_PARAM(parameters);
	/* nothing to do, only wait for training cfm*/
	groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_LINK_ADAPTATION_CONFIRMATION);
}

/**********************************************************************************

groupManagerStationEventTrainingCfmInRemovingWaitingForTrainingCfm


Description:
------------
handle the event of training confirmation received in the state 'removing - waiting
for training confirmation'


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventTrainingCfmInRemovingWaitingForTrainingCfm(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{ 	
	StaId stationIndex = 0;
	UNUSED_PARAM(parameters);
	stationIndex = groupManagerStationDataBaseEntry->stationIndex;
	/* if station is not a member of any group, move to idle send cfm to BSS*/
	if(0 == groupManagerStationDataBaseEntry->groupsBitmap)
	{
		groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_IDLE);	
		groupManagerSendConfirmToStationManager(stationIndex);
	}
	else
	{
		/* if station was a member of some groups - trigger groups removal process*/
		groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_REMOVING_WAITING_FOR_ALL_GROUP_REMOVAL);
	}
}


/**********************************************************************************

groupManagerStationEventToStateAssert


Description:
------------
create assert in case of illegal event. 
This function is invoked in a case illegal event arrives in some state of station

Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventToStateAssert(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{
	UNUSED_PARAM(parameters);
	UNUSED_PARAM(groupManagerStationDataBaseEntry);	
#ifdef GROUP_MANAGER_DEBUG							
	ILOG0_DD("Assert station %d state: %d", groupManagerStationDataBaseEntry->stationIndex, groupManagerStationDataBaseEntry->state);
#endif
	FATAL("Assert in group manager station event");
}

/**********************************************************************************

groupManagerStationEventToStateIgnore


Description:
------------
ignore event in a specific state of station


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventToStateIgnore(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{
	UNUSED_PARAM(parameters);
	UNUSED_PARAM(groupManagerStationDataBaseEntry);	
	ILOG2_DD("Ingore event station %d state: %d", groupManagerStationDataBaseEntry->stationIndex, groupManagerStationDataBaseEntry->state);
}

/**********************************************************************************

groupManagerStationEventRemoveGroupUpdateStationBitmap


Description:
------------
only update the matching group bitmap in station DB when a group is removed

Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - the group index		
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventRemoveGroupUpdateStationBitmap(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{
	uint8 groupIndex = 0;

	groupIndex = *(uint8 *)parameters;
	ILOG2_DDD("Remove Group event update station bitmap station %d state: %d group %d", groupManagerStationDataBaseEntry->stationIndex, groupManagerStationDataBaseEntry->state, groupIndex);

	groupManagerClearGroupBitInStationBitmap(groupManagerStationDataBaseEntry, groupIndex);
}

/**********************************************************************************

groupManagerStationEventAddGroupUpdateStationBitmap


Description:
------------
only update the matching group bitmap in station DB when a group is added


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - the group index		
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventAddGroupUpdateStationBitmap(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{
	uint8 groupIndex = 0;

	groupIndex = *(uint8 *)parameters;
	ILOG2_DDD("Add Group event update station bitmap station %d state: %d group %d", groupManagerStationDataBaseEntry->stationIndex, groupManagerStationDataBaseEntry->state, groupIndex);

	groupManagerSetGroupBitInStationBitmap(groupManagerStationDataBaseEntry, groupIndex);
}

/**********************************************************************************

groupManagerStationEventRemoveGroupInRemovingWaitingForAllGroupRemoval



Description:
------------
remove group (due to station or not) in the state of 'removing waiting for all group 
removal' 

In this state station is waiting for groups removal during groups removal process.


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
parameters - the index of removed group
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerStationEventRemoveGroupInRemovingWaitingForAllGroupRemoval(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters)
{
	uint8 groupIndex = 0;
	StaId stationIndex = 0;

	stationIndex = groupManagerStationDataBaseEntry->stationIndex;
	groupIndex = *(uint8 *)parameters;
	//ILOG0_DD("groupManagerStationEventRemoveGroupInRemovingWaitingForAllGroupRemoval station %d group %d", stationIndex, groupIndex);
	groupManagerClearGroupBitInStationBitmap(groupManagerStationDataBaseEntry, groupIndex);
	if(0 == groupManagerStationDataBaseEntry->groupsBitmap)
	{
		/* if all groups were removed move to IDLE, send cfm to BSS mng*/
		groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_IDLE);	
		groupManagerSendConfirmToStationManager(stationIndex);
	}
}

/**********************************************************************************

groupManagerRunStationStateMachine 


Description:
------------
run the group manager state machine


Input: 
-----
stationIndex - the station for which this event was received
event - the event that occured
parameters - address to needed parameters	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerRunStationStateMachine(StaId stationIndex, GroupManagerStationEvent_e event, void *parameters)
{
	GroupManagerStationEventToStateFunction_e functionIndex = GROUP_MANAGER_STATION_EVENT_TO_STATE_FIRST;
	GroupManagerStationState_e state = GROUP_MANAGER_STATION_STATE_IDLE;  
#ifdef GROUP_MANAGER_DEBUG								
    ILOG0_DDD("groupManagerRunStationStateMachine station %d state: %d event %d", stationIndex, GroupManagerStationDb[stationIndex].state, event);
#endif
	
	state = GroupManagerStationDb[stationIndex].state;
    functionIndex = GroupManagerStationStateMachine[state][event];
	GroupManagerStaionEventToStateFunctionsTable[functionIndex](&GroupManagerStationDb[stationIndex], parameters);
}


static void groupManagerUpdateTrainingPriorityBitmaps(GroupManagerStationState_e currentState, GroupManagerStationState_e newState, StaId stationIndex)
{
	groupManagerStationPriorities_e currentPriority = GroupManagerStationStateToPriorityConversionTable[currentState];
	groupManagerStationPriorities_e newPriority = GroupManagerStationStateToPriorityConversionTable[newState];

	DEBUG_ASSERT((currentPriority < GROUP_MANAGER_TOTAL_NUM_OF_PRIORITIES) && (newPriority < GROUP_MANAGER_TOTAL_NUM_OF_PRIORITIES));

	if (currentPriority != GROUP_MANAGER_STATION_PRIORITY_NONE)
	{
		Utils_ZeroBitInBitmap(GroupManagerGlobalParametersDb.trainingBitmaps.stationPriorityBitmap[currentPriority].stationBitmap, stationIndex); 
	}
	if (newPriority != GROUP_MANAGER_STATION_PRIORITY_NONE)
	{
		Utils_SetBitInBitmap(GroupManagerGlobalParametersDb.trainingBitmaps.stationPriorityBitmap[newPriority].stationBitmap, stationIndex);
	}
}

/**********************************************************************************

groupManagerChangeStationState 


Description:
------------
change the state of a station


Input: 
-----
groupManagerStationDataBaseEntry - a pointer to the station entry
newState - the new state to change to
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerChangeStationState(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, GroupManagerStationState_e newState)
{
#ifdef GROUP_MANAGER_DEBUG							
    ILOG0_DDD("groupManagerChangeStationState: Change station %d state: old state %d new state %d", groupManagerStationDataBaseEntry->stationIndex, groupManagerStationDataBaseEntry->state, newState);
#endif

	//update training bitmaps
	groupManagerUpdateTrainingPriorityBitmaps(groupManagerStationDataBaseEntry->state, newState, groupManagerStationDataBaseEntry->stationIndex);

	//update new state in database
	groupManagerStationDataBaseEntry->state = newState;
}	


/**********************************************************************************

groupManagerSendConfirmToStationManager 


Description:
------------
Sends Confirmation message on removal of the station to BSS manager.


Input: 
-----
clientId -  the id that the group manager got when registered
stationIndex - the index of the station
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerSendConfirmToStationManager(StaId stationIndex)
{
	K_MSG *pMsg;
	BssManagerStaManagerCfm_t *confirmMessage;

	/*Allocate message*/
	pMsg = OSAL_GET_MESSAGE(sizeof(BssManagerStaManagerCfm_t));
	confirmMessage = (BssManagerStaManagerCfm_t*) pK_MSG_DATA(pMsg);
	/*Set Client ID to the registered ID*/
	confirmMessage->clientId = BSS_MANAGER_STA_MANAGER_GM_CLIENT;
	/*Set STA ID*/
	confirmMessage->sid = stationIndex;
	/*Send confirmation message*/
#ifdef GROUP_MANAGER_DEBUG								
	ILOG0_DD("group manager, Send Confirmation, Client %d, stationIndex %d", confirmMessage->clientId, stationIndex);
#endif
	OSAL_SEND_MESSAGE(BSS_MANAGER_STA_MANAGER_REG_CFM, TASK_BSS_MANAGER, pMsg, VAP_ID_DO_NOT_CARE);
}	
	

/**********************************************************************************

groupManagerRemoveAllStationVirtualGroups 


Description:
------------
remove all virtual groups that this station is a member of 


Input: 
-----
groupManagerStationDataBaseEntry - a pointer to the station entry

	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerRemoveAllStationVirtualGroups(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry)
{
	uint32 virtualGroupsBitmap = 0;
	uint8 virtualGroupIndex = 0;
	
	virtualGroupsBitmap = groupManagerStationDataBaseEntry->virtualGroupsBitmap;
#ifdef GROUP_MANAGER_DEBUG								
	ILOG0_DD("groupManagerRemoveAllStationVirtualGroups: SID %d, virtualGroupsBitmap 0x%x",
			groupManagerStationDataBaseEntry->stationIndex, virtualGroupsBitmap);
#endif
	
	while(virtualGroupsBitmap)
	{
		virtualGroupIndex = Utils_FindFirstSetAndClear(&virtualGroupsBitmap);
		groupManagerRemoveVirtualGroup(virtualGroupIndex);	
	}
}	

/**********************************************************************************

groupManagerRemoveStationActiveGroups 


Description:
------------
start removal process of all active groups (including those in probing states) the station is member of 


Input: 
-----
groupManagerStationDataBaseEntry - a pointer to the station entry

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerRemoveStationActiveGroups(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry)
{
	uint16 activeGroupsBitmap = 0;
	uint8 activeGroupIndex = 0;
	StaId stationIndex = 0;

	stationIndex = groupManagerStationDataBaseEntry->stationIndex;

	activeGroupsBitmap = groupManagerStationDataBaseEntry->groupsBitmap;
	/* for each active group (probing groups included) start process of removal*/
	while(activeGroupsBitmap)
	{
		activeGroupIndex = Utils_CountTrailingZeros(activeGroupsBitmap);
		activeGroupsBitmap &= ~(0x1 << activeGroupIndex);
		/* trigger internal event of remove group - active group must be removed when old Nss is too high*/
		/* Note : station index relevant when removing group in Notify state. In case of remove group not
		due to timeout of notification process - GIDM Mng won't try to lock the group (Locker will do it)
		(in the second case GIDM will lock the group)*/
		groupManagerGroupEventRemoveGroup(activeGroupIndex, &stationIndex);
		/* remove group with this index - the relevant bit in groupsBitmap will be set to zero only when Lock CFM on
		this specific group will arrive. Only at this point it can be assurred that the group was removed and it's 
		transmissions were aborted*/
	}
}

/**********************************************************************************

groupManagerFindStationNssInVirtualGroup 


Description:
------------
returns the number of nss a station has in a virtual group


Input: 
-----
stationIndex -  the station index
virtualGroupIndex -  the index of the virtual group
	
		
Output:
-------
	

Returns:
--------
	the number of nss the station has in the virtual group 
	
**********************************************************************************/
static uint8 groupManagerFindStationNssInVirtualGroup(StaId stationIndex, uint8 virtualGroupIndex)
{
    uint8 nss = 0;
	uint8 indexInArray = 0;
	uint8 shift = 0; /* number of shifts to perform to nssBitmap */

	/* Find entry index in the virtual group in order to find the previous BW */
		
	for(indexInArray = 0; indexInArray < GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP; indexInArray++)
	{
		if(GroupManagerVirtualGroupDatabase[virtualGroupIndex].participatingStations[indexInArray] == stationIndex)
		{
			/* shift is done to multiply stationIndex by 2 in order to calculate number of shifts required to reach 
			  nss offset of this specific station*/
			shift = indexInArray << GROUP_MANAGER_NSS_SIZE_SHIFT;
			/* shift nss value*/
			nss = GroupManagerVirtualGroupDatabase[virtualGroupIndex].nssBitmap >> shift;
			/* apply mask of "11" on shifted nss value to extract the actual nss of this specific station*/
			nss &= GROUP_MANAGER_NSS_MASK;
			return nss;
		}		
	}
	FATAL("Station was not found in virtual group");
	return 0;
}

/**********************************************************************************

groupManagerFindStationNssInActiveGroup 


Description:
------------
returns the number of nss a station has in a active group 


Input: 
-----
stationIndex - the station index
groupIndex - the index of the group
	
		
Output:
-------
	

Returns:
--------
	the number of nss the station has in the virtual group 
	
**********************************************************************************/
static uint8 groupManagerFindStationNssInActiveGroup(StaId stationIndex, uint8 groupIndex)
{
    uint8 nss = 0;
	uint8 indexInArray = 0;
	uint8 shift = 0; /* number of shifts to perform to nssBitmap */

	/* Find entry index in the virtual group in order to find the previous BW */
		
	for(indexInArray = 0; indexInArray < GroupManagerGroupDb[groupIndex].numberOfParticipatingStations; indexInArray++)
	{
		if(GroupManagerGroupDb[groupIndex].participatingStations[indexInArray] == stationIndex)
		{
			shift = indexInArray << GROUP_MANAGER_NSS_SIZE_SHIFT;
			nss = GroupManagerGroupDb[groupIndex].nssBitmap >> shift;
			nss &= GROUP_MANAGER_NSS_MASK;
			return nss;
		}		
	}
	FATAL("Station was not found in group");
	return 0;
}


/**********************************************************************************

groupManagerSetStationPrimaryBitInGroupsDb 


Description:
------------
configure a new primary station in the group entry. 
each group entry contains a bitmap of primary stations - this function sets to 1 
the bit corresponding to station index in this bitmap

Input: 
-----
stationIndex - the station index
groupIndex - the index of the group
	
		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
static void groupManagerSetStationPrimaryBitInGroupsDb(StaId stationIndex, uint8 groupIndex)
{
	uint8 indexInArray = 0;

	for(indexInArray = 0; indexInArray < GroupManagerGroupDb[groupIndex].numberOfParticipatingStations; indexInArray++)
	{
		/* find stationIndex in the group Db and set it's primary bit to 1*/
		if(GroupManagerGroupDb[groupIndex].participatingStations[indexInArray] == stationIndex)
		{
			/* turn on relevant bit in primary stations bitmap */
			GroupManagerGroupDb[groupIndex].primaryStationsBitmap |= (0x1 << indexInArray);
			break;
		}		
	}
#ifdef GROUP_MANAGER_DEBUG							
	ILOG0_DDD("groupManagerSetStationPrimaryBitInGroupsDb: stationIndex %d, groupIndex %d, primary stations bitmap after adding new primary 0x%x",
			stationIndex, groupIndex, GroupManagerGroupDb[groupIndex].primaryStationsBitmap);
#endif
}

/**********************************************************************************

groupManagerConfigureStationAsPrimaryInGroup 


Description:
------------
set station as primary in group Db and in Selector

Input: 
-----
stationIndex - the station index
groupIndex - the index of the group
	
		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/

static void groupManagerConfigureStationAsPrimaryInGroup(StaId stationIndex, uint8 groupIndex)
{
	groupManagerSetStationPrimaryBitInGroupsDb(stationIndex, groupIndex);
	GroupManagerStationDb[stationIndex].primaryGroup = groupIndex;
#ifdef GROUP_MANAGER_DEBUG							
		ILOG0_DD("groupManagerConfigureStationAsPrimaryInGroup: station index %d configured primary for selector with group index %d",
			stationIndex, groupIndex);
#endif
	groupManagerConfigureStationForSelector(stationIndex, TRUE, groupIndex, TRUE);
}

/**********************************************************************************

groupManagerClearStationPrimaryInProbingGroup 


Description:
------------
stop station from being primary in PROBING group.
Find stationIndex in group DB and turn off its primary bit


Input: 
-----
stationIndex - the station index
groupIndex - the index of the group
	
		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
static void groupManagerClearStationPrimaryInProbingGroup(StaId stationIndex, uint8 groupIndex)
{
	uint8 indexInArray = 0;

	for(indexInArray = 0; indexInArray < GroupManagerGroupDb[groupIndex].numberOfParticipatingStations; indexInArray++)
	{
		if(GroupManagerGroupDb[groupIndex].participatingStations[indexInArray] == stationIndex)
		{
			GroupManagerGroupDb[groupIndex].primaryStationsBitmap &= ~(0x1 << indexInArray);
			break;
		}		
	}
}

/**********************************************************************************

groupManagerClearStationPrimaryInGroup 


Description:
------------
stop station from being primary in ACTIVE group - might cause to trigger remove group event 
if this primary was the last primary for this group


Input: 
-----
stationIndex - the station index
groupIndex - the index of the group
	
		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
static void groupManagerClearStationPrimaryInGroup(StaId stationIndex, uint8 groupIndex)
{
	uint8 indexInArray = 0;
#ifdef GROUP_MANAGER_DEBUG							
	ILOG0_DDD("groupManagerClearStationPrimaryInGroup: station index %d, group index %d, primaryStationsbitmap for this group 0x%x",
		stationIndex, groupIndex, GroupManagerGroupDb[groupIndex].primaryStationsBitmap);
#endif
	for(indexInArray = 0; indexInArray < GroupManagerGroupDb[groupIndex].numberOfParticipatingStations; indexInArray++)
	{
		if(GroupManagerGroupDb[groupIndex].participatingStations[indexInArray] == stationIndex)
		{
			GroupManagerGroupDb[groupIndex].primaryStationsBitmap &= ~(0x1 << indexInArray);
			break;
		}		
	}

	if(0 == GroupManagerGroupDb[groupIndex].primaryStationsBitmap)
	{
		/* No primary for the group - trigger remove group */
		/* No need for station index since the group is not in notifying state */
		if(groupIndex != GroupManagerGlobalParametersDb.staticOccupiedGroupIndex)
		{
			groupManagerGroupEventRemoveGroup(groupIndex, NULL);
		}
		else
		{
			FATAL("Trying to delete staticGroup !");
		}
	}
}

/**********************************************************************************

groupManagerSendEndSingleUserValidationMessage 


Description:
------------
send single validation end message to the link adaptation


Input: 
-----
stationIndex - the station index
	
		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
static void groupManagerSendEndSingleUserValidationMessage(StaId stationIndex)
{
	LaEndSingleUserValidationParams_t *endSingleUserValidationParameters = NULL;
	K_MSG *linkAdaptationMessage	= NULL;
	uint8 vapId = 0;

	StaDb_GetVapId(stationIndex, &vapId);
	linkAdaptationMessage = OSAL_GET_MESSAGE( sizeof(LaEndSingleUserValidationParams_t));
	endSingleUserValidationParameters = ((LaEndSingleUserValidationParams_t *)linkAdaptationMessage->abData);
	endSingleUserValidationParameters->stationIndex = stationIndex;
#ifdef GROUP_MANAGER_DEBUG								
	ILOG0_D("groupManagerSendEndSingleUserValidationMessage: to LA on SID %d", stationIndex);
#endif
	OSAL_SEND_MESSAGE(LINK_ADAPTATION_STOP_SINGLE_USER_VALIDATION, TASK_LINK_ADAPTATION, linkAdaptationMessage, vapId);
}

/**********************************************************************************

groupManagerTriggerRemoveStationInTraining 


Description:
------------
invoke remove station in the training module. Send removeStaReq to Training Mng

Input: 
-----
stationIndex - the station index to remove
	
		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
static void groupManagerTriggerRemoveStationInTraining(StaId stationIndex)
{
	TrainingManagerRemoveStaParams_t removeStationParameters;
	
	memset(&removeStationParameters, 0, sizeof(TrainingManagerRemoveStaParams_t));
	
	removeStationParameters.staId = stationIndex;
#ifdef GROUP_MANAGER_DEBUG								
	ILOG0_D("groupManagerTriggerRemoveStationInTraining: sent to Training mng on SID %d", stationIndex);
#endif
	TrainingManagerRemoveStaReq(&removeStationParameters);
}

/**********************************************************************************

groupManagerTriggerHaltStationInTraining 


Description:
------------
invoke halt station in the Training manager. 

Input: 
-----
stationIndex - the station index to halt
	
		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
static void groupManagerTriggerHaltStationInTraining(StaId stationIndex)
{
	TrainingManagerHaltStaParams_t haltStationParameters;
	/* sends HALT on station to training mng*/
	memset(&haltStationParameters, 0, sizeof(TrainingManagerHaltStaParams_t));
	
	haltStationParameters.staId = stationIndex;

	TrainingManagerHaltStaReq(&haltStationParameters);

}

/**********************************************************************************

groupManagerChooseBestGroupFromProbingMessage 


Description:
------------
choose group with best mark from a station active groups


Input: 
-----
groupManagerStationDataBaseEntry - - pointer to the entry in the group manager station 
database	
groupManagerProbingResultsParameters - the pointer to the message of probing results
	
		
Output:
-------
	

Returns:
--------
	the group index with the best mark for a specific station
	
**********************************************************************************/
static uint8 groupManagerChooseBestGroupFromProbingMessage(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, GroupManagerProbingResultsMessage_t *groupManagerProbingResultsParameters)
{
	uint16 stationActiveGroupsBitmap = 0;
	uint8 bestGroupIndex = GROUP_MANAGER_INVALID_GROUP_INDEX;

	stationActiveGroupsBitmap  = groupManagerStationDataBaseEntry->groupsBitmap & GroupManagerGlobalParametersDb.activeGroupsBitmap;

	/* Include the probing group in the search of best group */
	stationActiveGroupsBitmap |= (0x1 << groupManagerProbingResultsParameters->probingGroupIndex);

	bestGroupIndex =  groupManagerFindBestGroup(stationActiveGroupsBitmap, groupManagerProbingResultsParameters->groupMarks);

	return bestGroupIndex;
}

/**********************************************************************************

groupManagerFindBestGroup 


Description:
------------
find best group in groups bitmap
WARNING: the bitmap must be != 0 and only 16 LSB might be set

Input: 
-----
groupsBitmap - the groups to choose from
groupMarks - an array of groups' marks
	
		
Output:
-------
	

Returns:
--------
	the group index with the best mark for a specific station
	
**********************************************************************************/
static uint8 groupManagerFindBestGroup(uint16 groupsBitmap, uint8 *groupMarks)
{
	uint8 bestGroupIndex = GROUP_MANAGER_INVALID_GROUP_INDEX;
	uint8 bestGroupMark = 0;
	uint8 groupIndex = 0;

	while(groupsBitmap)
	{
		/* There should be at least one group otherwise the station was not in active or temp active state */
		groupIndex = Utils_CountTrailingZeros(groupsBitmap);
		groupsBitmap &= ~(0x1 << groupIndex);
		if(groupMarks[groupIndex] >= bestGroupMark)
		{
			bestGroupIndex = groupIndex;
			bestGroupMark = groupMarks[groupIndex];
		}
	}

#ifdef GROUP_MANAGER_DEBUG								
	ILOG0_D("groupManagerFindBestGroup: group with best mark  is %d", bestGroupIndex);
#endif
    return bestGroupIndex;
}


/**********************************************************************************

groupManagerPrimaryGroupWasRemoved 


Description:
------------
handle the event that primary group of station was removed in the state of 'active' and 'single
user validation pending'


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database	
	
		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
static void groupManagerPrimaryGroupWasRemoved(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry)
{
	StaId stationIndex = 0;

	stationIndex = groupManagerStationDataBaseEntry->stationIndex;
	
	groupManagerStationDataBaseEntry->primaryGroup = GROUP_MANAGER_INVALID_GROUP_INDEX;
	// possible fix for Selector's primary station mapping bug
	//groupManagerConfigureStationForSelector(stationIndex, FALSE, 0);
	Utils_ZeroBitInBitmap(GroupManagerGlobalParametersDb.singleUserValidationBitmap, stationIndex);
	/* if this station is part of additional active groups request groups mark to trigger evaluation process in 
	   which new primary group might be found for this station, the station remains in active state
	   because it is a member of more active groups*/
	groupManagerSendGroupsMarksMessage(stationIndex);
}

/**********************************************************************************

groupManagerRemoveGroupInActive 


Description:
------------
handle the event that group was removed in 'active' state 


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database
parameters - pointer to the group index that was removed
isHaltNeeded - a flag indicates if halt is needed in case the station is not a member
of any virtual or HW group

		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
static void groupManagerRemoveGroupInActive(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters, uint8 isHaltNeeded)
{
	uint16 activeGroupsBitmap = 0;
    uint8 groupIndex = *(uint8 *)parameters;
#ifdef GROUP_MANAGER_DEBUG							
	ILOG0_DD("groupManagerRemoveGroupInActive: group index: %d, haltNeeded: %d ", groupIndex, isHaltNeeded);
#endif
	/* delete the group from station dB*/
	groupManagerClearGroupBitInStationBitmap(groupManagerStationDataBaseEntry, groupIndex);

	activeGroupsBitmap = groupManagerStationDataBaseEntry->groupsBitmap & GroupManagerGlobalParametersDb.activeGroupsBitmap;
	if(activeGroupsBitmap)
	{
#ifdef GROUP_MANAGER_DEBUG									
		ILOG0_D("groupManagerRemoveGroupInActive: station has more active groups. bitmap of groups 0x%x", activeGroupsBitmap);
#endif
		/* if station has more active groups and the removed group is primary group*/
		if(groupManagerStationDataBaseEntry->primaryGroup == groupIndex)
		{
			groupManagerSetStationSuTxopAndAmpduLimit(groupManagerStationDataBaseEntry->stationIndex, TRUE, SU_NUM_OF_MPDUS_IN_A_MPDU);
			groupManagerPrimaryGroupWasRemoved(groupManagerStationDataBaseEntry);
		}	
	}
	/* if there is no more active groups for this station  - exit active state*/
	else
	{	
		groupManagerStationExitActiveState(groupManagerStationDataBaseEntry, isHaltNeeded);		
	}
}

/**********************************************************************************

groupManagerRemoveGroupInSingleUserValidationPending 


Description:
------------
handle the event that a group was removed in 'single user validation pending' state


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database
parameters - pointer to the group index
isHaltNeeded - a flag indicates if halt is needed in case the station is not a member
of any virtual or HW group

		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
static void groupManagerRemoveGroupInSingleUserValidationPending(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters, uint8 isHaltNeeded)
{	
	uint16 activeGroupsBitmap = 0;
    uint8 groupIndex = 0;

	groupIndex = *(uint8 *)parameters;
	groupManagerClearGroupBitInStationBitmap(groupManagerStationDataBaseEntry, groupIndex);

	/* if removed group was primary of this station*/
	if(groupManagerStationDataBaseEntry->primaryGroup == groupIndex)
	{
		/* cancel upcoming single user validation by removing from calendar wheel*/
		CalendarWheel_RemoveEntry(&GroupManagerStationsCalendarWheel, (CalendarWheelEntry_t*)groupManagerStationDataBaseEntry);
		groupManagerStationDataBaseEntry->primaryGroup = GROUP_MANAGER_INVALID_GROUP_INDEX;
		// possible fix for Selector's primary station mapping bug
		//groupManagerConfigureStationForSelector(groupManagerStationDataBaseEntry->stationIndex, FALSE, 0);
		activeGroupsBitmap = groupManagerStationDataBaseEntry->groupsBitmap & GroupManagerGlobalParametersDb.activeGroupsBitmap;
		/* update new state of the station depending in which groups this station participates*/
		if(activeGroupsBitmap)
		{	
			// station is no longer in single user pending and still member of more active groups - returns it's state to active
			groupManagerChangeStationState(groupManagerStationDataBaseEntry,GROUP_MANAGER_STATION_STATE_ACTIVE);
			groupManagerPrimaryGroupWasRemoved(groupManagerStationDataBaseEntry);
		}
		else
		{
			groupManagerStationExitActiveState(groupManagerStationDataBaseEntry, isHaltNeeded);
		}
		groupManagerStartNewSingleUserValidation();
	}	
	/*if removed group is not a primary group of station but some other group it is member of, 
	  no need to check if station is a member of more active groups and if not to trigger 
	  groupManagerStationExitActiveState because in state of su val pending it is assurred that 
	  station has at least one more active primary group*/
}


/**********************************************************************************

groupManagerSuValEndCfmInSuValCanceled 


Description:
------------
handle the event that a single user validation end confirmation has received in the
states of 'single user validation canceled - group removed' and 'single user validation
canceled - group failed'
Note: State machine described in Figure 23 in Group manager doc.


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database
parameters - pointer to the group index
isHaltNeeded - a flag indicates if halt is needed in case the station is not a member
of any virtual or HW group

		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
static void groupManagerSuValEndCfmInSuValCanceled(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, void *parameters, uint8 isHaltNeeded)
{	
	uint16 activeGroupsBitmap = 0;
	StaId stationIndex = 0;
	UNUSED_PARAM(parameters);

	stationIndex = groupManagerStationDataBaseEntry->stationIndex;
	/* if member of more active gorups*/		
	activeGroupsBitmap = groupManagerStationDataBaseEntry->groupsBitmap & GroupManagerGlobalParametersDb.activeGroupsBitmap;
	if(activeGroupsBitmap)
	{	
		/* try to send request for group marks*/
		groupManagerSendGroupsMarksMessage(stationIndex);
		/* return to active state*/
		groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_ACTIVE);
	}
	else
	{
	/* exit active state*/
		groupManagerStationExitActiveState(groupManagerStationDataBaseEntry, isHaltNeeded);
	}
	/* free spot now available - find candidate for su val process*/
	groupManagerStartNewSingleUserValidation();
}

/**********************************************************************************

groupManagerTryToStartNewSingleUserValidation 


Description:
------------
try to start new single user validation - this function is called when a new station
becomes a primary member.
checks if there is still place for more single user validation processes and if so
finds in Round Robin next candidate and triggers on it SU validation


Input: 
-----
stationIndex - the station that became primary
	
		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
static void groupManagerTryToStartNewSingleUserValidation(StaId stationIndex)
{
	uint8 newSingleUserValidationStationIndex = 0;
	
	/* do not add station member in static group to single user validation group*/	
	if( (GroupManagerGlobalParametersDb.staticOccupiedGroupIndex != GROUP_MANAGER_INVALID_GROUP_INDEX) &&
		(GroupManagerStationDb[stationIndex].primaryGroup == GroupManagerGlobalParametersDb.staticOccupiedGroupIndex))
	{ 
		return;
	}

	/* add new primary station to singlue user validation candidate bits only if this station is not primary for static group */
	ILOG2_DD("groupManagerTryToStartNewSingleUserValidation: Single User Validation bitmap %d, SID %d",
			GroupManagerGlobalParametersDb.singleUserValidationBitmap[0] , stationIndex);
	
	Utils_SetBitInBitmap(GroupManagerGlobalParametersDb.singleUserValidationBitmap, stationIndex);
#ifdef GROUP_MANAGER_DEBUG								
	ILOG0_DD("groupManagerTryToStartNewSingleUserValidation: Single User Validation bitmap after adding new one %d, SID %d",
			GroupManagerGlobalParametersDb.singleUserValidationBitmap[0] , stationIndex);
#endif
	
	/* if number of single user validation done in parallel didn't reach the threshold*/	
	if(GroupManagerGlobalParametersDb.muConfigParams.maxSingleUserValidationStations > GroupManagerGlobalParametersDb.numberOfCurrentSingleUserValidation)
	{
		/* find next single user validation index in Round Robin and trigger the process on this new station (station added to calendar wheel) */
		newSingleUserValidationStationIndex = Utils_RoundRobinBitmap(GroupManagerGlobalParametersDb.singleUserValidationBitmap, GroupManagerGlobalParametersDb.lastSingleUserValidationStation, GROUP_MANAGER_STATIONS_BITMAP_SIZE_IN_BYTES); 
#ifdef GROUP_MANAGER_DEBUG									
		ILOG0_D("groupManagerTryToStartNewSingleUserValidation: start single user val on SID %d", newSingleUserValidationStationIndex);
#endif
		GroupManagerGlobalParametersDb.numberOfCurrentSingleUserValidation++;
		GroupManagerGlobalParametersDb.lastSingleUserValidationStation = newSingleUserValidationStationIndex;
		groupManagerStationEventSingleUserValidationStartProcess(newSingleUserValidationStationIndex);
	}
}

/**********************************************************************************

groupManagerSendGroupsMarksMessage 


Description:
------------
send groups' marks message to the link adaptation


Input: 
-----
stationIndex - the station that needs the groups' marks
	
		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
static void groupManagerSendGroupsMarksMessage(StaId stationIndex)
{
	uint8 vapId = 0;

	StaDb_GetVapId(stationIndex, &vapId);
	Utils_SetBitInBitmap(GroupManagerGlobalParametersDb.waitingGroupsMarksBitmap, stationIndex);
#ifdef GROUP_MANAGER_DEBUG								
	ILOG0_V("groupManagerSendGroupsMarksMessage");
#endif
	/* send request for group marks to LA only if was not sent before */
	if(!GroupManagerGlobalParametersDb.wasGroupsMarksMessageSent)
	{
		GroupManagerGlobalParametersDb.wasGroupsMarksMessageSent = TRUE;
		OSAL_SEND_NO_DATA_MESSAGE(LINK_ADAPTATION_GET_GROUPS_MARKS, TASK_LINK_ADAPTATION, vapId);
	}
}


/**********************************************************************************

groupManagerStationExitActiveState 


Description:
------------
Handle the scenario when a station exits an active state


Input: 
-----
groupManagerStationDataBaseEntry - pointer to the entry in the group manager station 
database
isHaltNeeded - a flag that indicates whether a halt needed in case the station is not 
part of any virtual/HW group
		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
static void groupManagerStationExitActiveState(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, uint8 isHaltNeeded)
{	
	StaId stationIndex = 0;
	uint16 probingGroupsBitmap = 0;
	uint16 notifyingGroupsBitmap= 0;

	stationIndex = groupManagerStationDataBaseEntry->stationIndex;
	
	Utils_ZeroBitInBitmap(GroupManagerGlobalParametersDb.waitingGroupsMarksBitmap, stationIndex);
	// remove the station from single user validation candidates if was in the list
	Utils_ZeroBitInBitmap(GroupManagerGlobalParametersDb.singleUserValidationBitmap, stationIndex);
	/* check if this station is member of probing groups*/
	probingGroupsBitmap = groupManagerStationDataBaseEntry->groupsBitmap & GroupManagerGlobalParametersDb.probingGroupsBitmap;
	
	if(probingGroupsBitmap)
	{
#ifdef GROUP_MANAGER_DEBUG									
		ILOG0_V("groupManagerStationExitActiveState: station has more probing groups");
#endif
		// if station is not primary for any probing group it is member of 
		if(groupManagerStationDataBaseEntry->tempPrimaryGroup == GROUP_MANAGER_INVALID_GROUP_INDEX)
		{
			groupManagerSetStationSuTxopAndAmpduLimit(stationIndex, TRUE , SU_NUM_OF_MPDUS_IN_A_MPDU);
		}
		/*if found probing groups - change station state to probing*/
		groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_PROBING);
	}
	else
	{
		groupManagerSetStationSuTxopAndAmpduLimit(stationIndex, TRUE , SU_NUM_OF_MPDUS_IN_A_MPDU);
		/* if station is not a member of active or probing groups check if it is a member of notify groups if so change 
		it's state to "NOTIFY" */
		notifyingGroupsBitmap = groupManagerStationDataBaseEntry->groupsBitmap & GroupManagerGlobalParametersDb.notifyingGroupsBitmap;
		if(notifyingGroupsBitmap)
		{
#ifdef GROUP_MANAGER_DEBUG										
			ILOG0_V("groupManagerStationExitActiveState: station has more notifying groups");
#endif
			groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_NOTIFY_STATION);
		}
		else
		{	
			/* here need to check if the station is at least a member of virtual groups*/
			if(!isHaltNeeded || groupManagerStationDataBaseEntry->virtualGroupsBitmap)
			{
				/* The station is member in virtual groups so it wont start halt process */	
#ifdef GROUP_MANAGER_DEBUG											
				ILOG0_DD("groupManagerStationExitActiveState: can remain in single user state. isHaltNeeded %d, virtual groups bitmap 0x%x", isHaltNeeded, groupManagerStationDataBaseEntry->virtualGroupsBitmap);
#endif
				groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_SINGLE_USER);
			}
			else
			{
				/* the station which groups is removed is not a member of additional groups of any type - must start halt process*/
#ifdef GROUP_MANAGER_DEBUG											
				ILOG0_V("groupManagerStationExitActiveState: station is moving to HALT");
#endif
				Utils_SetBitInBitmap(GroupManagerGlobalParametersDb.trainingBitmaps.haltStationsBitmap.stationBitmap, stationIndex);
				groupManagerChangeStationState(groupManagerStationDataBaseEntry, GROUP_MANAGER_STATION_STATE_HALTING_WAITING_FOR_TRAINING_CONFIRMATION);

				/* Number of halts is always cleared in active(according to SAS state so no need to calculate */   
				groupManagerStationDataBaseEntry->numberOfHalts = 1;

	            /* Halt period in active is always penalty since the number of halts in active is always 0 */
				groupManagerStationDataBaseEntry->timeout =  GROUP_MANAGER_HALT_PENALTY_FROM_ACTIVE;
						   
				groupManagerTriggerHaltStationInTraining(stationIndex);
				/* station will be added to calendar wheel (for halt timeout) upon reception of Training confirmation  */
			}				
		}		
	}		
}


/**********************************************************************************

groupManagerStartNewSingleUserValidation 


Description:
------------
start new single user validation - choose next station
this function is called when a single user validation process has ended for a station. 
Similar to the function 'groupManagerTryToStartNewSingleUserValidation' but the 
difference is that at least one free spot is found


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
static void groupManagerStartNewSingleUserValidation(void)
{
	StaId stationIndex = 0;
	StaId nextCandidateForSingleUserValidationStationIndex = GroupManagerGlobalParametersDb.lastSingleUserValidationStation + 1;

	/*Max number of stations may not be 2^*/
	if (nextCandidateForSingleUserValidationStationIndex == HW_NUM_OF_STATIONS)
	{
		nextCandidateForSingleUserValidationStationIndex = 0;
	}
	stationIndex = Utils_RoundRobinBitmap(GroupManagerGlobalParametersDb.singleUserValidationBitmap, nextCandidateForSingleUserValidationStationIndex, GROUP_MANAGER_STATIONS_BITMAP_SIZE_IN_BYTES);
#ifdef GROUP_MANAGER_DEBUG								
	ILOG0_D("groupManagerStartNewSingleUserValidation: all stations candidates for su validation bitmap 0x%x", GroupManagerGlobalParametersDb.singleUserValidationBitmap[0]);
#endif
	if(CONVERT_WORDS_TO_BIT_INDEX(GROUP_MANAGER_SIZE_OF_STATION_BITMAP_IN_WORDS) != stationIndex)
	{
		GroupManagerGlobalParametersDb.lastSingleUserValidationStation = stationIndex;
#ifdef GROUP_MANAGER_DEBUG									
		ILOG0_D("groupManagerStartNewSingleUserValidation: SID %d chosen next for SU validation, moving to Su val pending", stationIndex);
#endif
		groupManagerStationEventSingleUserValidationStartProcess(stationIndex);
	}
	else
	{
#ifdef GROUP_MANAGER_DEBUG									
		ILOG0_V("groupManagerStartNewSingleUserValidation: no candidates for su val found");
#endif
		GroupManagerGlobalParametersDb.numberOfCurrentSingleUserValidation--;
	}
}

/**********************************************************************************

groupManagerConfigureStationForSelector 


Description:
------------
configure station entry in Selectors MU bitmap


Input: stationIndex - entry of station to configute
-----  isPrimary if station is primary for some group
	   groupIndex - index of grop for which this station is primary	
	
		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/

static void groupManagerConfigureStationForSelector(StaId stationIndex, bool isPrimary, uint8 groupIndex, bool changeInStaDbRequired)
{
	TxSelectorPrimaryStaInfo_u stationSelectorInformation;

	stationSelectorInformation.val = 0;
	stationSelectorInformation.bitFields.groupIdx = groupIndex;
	stationSelectorInformation.bitFields.isPrimaryInGroup = isPrimary;
	TxSelector_ModifyPrimaryStaInfo(stationIndex, stationSelectorInformation);

	if(changeInStaDbRequired)
	{
		uint8 txopMode = !(isPrimary);
		uint8 numOfMpdus = (isPrimary == TRUE) ? GROUP_MANAGER_SU_AGGREGATION_THRESH : SU_NUM_OF_MPDUS_IN_A_MPDU;
		groupManagerSetStationSuTxopAndAmpduLimit(stationIndex, txopMode, numOfMpdus);
	}
}


/**********************************************************************************

groupManagerConfigureNewGroupForSelector 


Description:
------------
configure new group and primary station for this group in Selector's DB


Input: groupIndex
-----  primaryStationIndex 
	
		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
static void groupManagerConfigureNewGroupForSelector(uint8 groupIndex, StaId primaryStationIndex)
{
	uint8 i;
	RegTxSelectorAllStaBitmapUpdate_u allStaInfo;
	GroupManagerGroupParameters_t *groupDbEntry = &GroupManagerGroupDb[groupIndex];
	
#ifdef ENET_INC_ARCH_WAVE600
	TxSelectorAllGroupValidUspBitmap_t allGroupValidUspBitmap;

	allGroupValidUspBitmap.validUspBitmap = 0;
#endif

	allStaInfo.val = 0;
#ifdef GROUP_MANAGER_DEBUG								
	ILOG0_D("groupManagerConfigureNewGroupForSelector: numOfStations %d", groupDbEntry->numberOfParticipatingStations);
#endif
	for(i=0; i < groupDbEntry->numberOfParticipatingStations; i ++)
	{
#ifdef GROUP_MANAGER_DEBUG								
		ILOG0_D("groupManagerConfigureNewGroupForSelector: sid %d", groupDbEntry->participatingStations[i]);
#endif
		// fill in StaId per user position of the station in a group
#ifdef ENET_INC_ARCH_WAVE600
      // in GEN6 each 8 bits represent staId, valid bit of station in group is set in another register
		allStaInfo.val |= (groupDbEntry->participatingStations[i] & GROUP_MANAGER_STAID_LENGTH_MASK) << (i*GROUP_MANAGER_USER_POSITION_INCREASE_SHIFT_SIZE);
		// set valid bit of station in its USP in a group
		allGroupValidUspBitmap.validUspBitmap |= (1 << i);
#else
	// in GEN5 each 8 bits represent: LSB 7 bits is staId + MSB bit Sta Valid bit in group
		allStaInfo.val |= (((uint32)(GROUP_MANAGER_VALID_USER_POSITION_MASK | groupDbEntry->participatingStations[i])) << (i*GROUP_MANAGER_USER_POSITION_INCREASE_SHIFT_SIZE));
#endif
	}

	/* configure all group members and Tx mode in Selector*/
#ifdef ENET_INC_ARCH_WAVE600
	TxSelector_MapAllStasToGroup(groupIndex, allStaInfo, allGroupValidUspBitmap);
#else
	TxSelector_MapAllStasToGroup(groupIndex, allStaInfo);
#endif
	TxSelector_UpdateGroupTxMode(groupIndex, TX_SELECTOR_BF_WITH_DATA);
	
#ifdef GROUP_MANAGER_DEBUG								
	ILOG0_DD("groupManagerConfigureNewGroupForSelector: Configure primary SID %d for group %d in Selector",
			primaryStationIndex, groupIndex);
#endif
	// For station configured as primary for group - disable TXOP for each enabled TID, reduce #mpdus in AMPDU to 8
	groupManagerConfigureStationForSelector(primaryStationIndex, TRUE, groupIndex, TRUE);
	/* The group was locked in init or previous remove group. Here unlock the group because from now on it will be used for transmission*/
	TxSelector_UnlockGroup(groupIndex);
		
}


/**********************************************************************************

groupManagerTriggerStaticGroupGeneration 


Description:
------------
this fucntion starts generation of static group by invoking ProcessTrainingResults
function on combination vector with static group members which were requested by user


Input: 	
		
Output:
-------

Returns:
--------
	
**********************************************************************************/
static void groupManagerTriggerStaticGroupGeneration(void)
{
	uint8 i,j,k = 0;
	GroupCombinationTableEntry_t staticGroupCombination[GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP];
	uint8 nssPerUsp[GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP];	

	memset(&staticGroupCombination, 0xFF, sizeof(staticGroupCombination));
	
	/* generate combinations vector*/
	for(k=0; k < GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP; k++)
	{	
		j = 0;
				
		staticGroupCombination[k].staId = GroupManagerGlobalParametersDb.staticGroupParams.staIds[k];
		staticGroupCombination[k].groupMark = GROUP_MANAGER_BEST_STATIC_GROUP_MARK;
		staticGroupCombination[k].groupCombinationPhyRates[0] = GroupManagerGlobalParametersDb.staticGroupInitialMcs;
		staticGroupCombination[k].groupCombinationPhyRates[1] = GroupManagerGlobalParametersDb.staticGroupInitialMcs;
		staticGroupCombination[k].groupCombinationPhyRates[2] = GroupManagerGlobalParametersDb.staticGroupInitialMcs;
		staticGroupCombination[k].groupCombinationPhyRates[3] = GroupManagerGlobalParametersDb.staticGroupInitialMcs;
		staticGroupCombination[k].singleNstsPhyRate = GROUP_MANAGER_NSTS_PHY_RATE_SINGLE;
		staticGroupCombination[k].doubleNstsPhyRate = GROUP_MANAGER_INVALID_RATE_VALUE;

		MEMCPY(nssPerUsp, GroupManagerGlobalParametersDb.staticGroupParams.nssPerUsp, GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP);
		for(i=0; ((i < GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP) && (GroupManagerGlobalParametersDb.staticGroupParams.staIds[i]!= INVALID_STA_INDEX)); i++)
		{
			while((nssPerUsp[i] > 0) && (j < GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP))
			{
				staticGroupCombination[k].groupCombinationSIDs[j] = GroupManagerGlobalParametersDb.staticGroupParams.staIds[i];
				nssPerUsp[i]--;
				j++;
			}
		}
	}	
	groupManagerProcessTrainingResults(GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP, staticGroupCombination, TRUE);
}

/**********************************************************************************

groupManagerClearGroupBitInStationBitmap 


Description:
------------
clear the bit of a group in the station bitmap

Input: 
-----
groupManagerStationDataBaseEntry - a pointer to the station entry
groupIndex -  the group index
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerClearGroupBitInStationBitmap(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, uint8 groupIndex)
{
	groupManagerStationDataBaseEntry->groupsBitmap &= ~(0x1 << groupIndex);
}

/**********************************************************************************

groupManagerSetGroupBitInStationBitmap 


Description:
------------
set bit of a group in the station bitmap

Input: 
-----
groupManagerStationDataBaseEntry - a pointer to the station entry
groupIndex -  the group index
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerSetGroupBitInStationBitmap(GroupManagerStationParameters_t *groupManagerStationDataBaseEntry, uint8 groupIndex)
{
	groupManagerStationDataBaseEntry->groupsBitmap |= (0x1 << groupIndex);
}

/**********************************************************************************

groupManagerGroupEventRemoveGroup


Description:
------------
handle the event of remove group


Input: 
-----
groupIndex - the group index to remove
stationIndex - the  address of the station index that is removed (it is used only in the case of 
remove station during notifying state)

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerGroupEventRemoveGroup(uint8 groupIndex, void *parameters)
{
	uint8 i;
	GroupManagerGroupParameters_t *groupEntry = &GroupManagerGroupDb[groupIndex];
	
	/* for all the primary members of this group restore StaDb txop and amsdu parameters*/
	for(i=0; i < groupEntry->numberOfParticipatingStations; i++)
	{
		if((groupEntry->primaryStationsBitmap & (0x1 << i)) !=0 )
		{
			groupManagerSetStationSuTxopAndAmpduLimit(groupEntry->participatingStations[i], TRUE, SU_NUM_OF_MPDUS_IN_A_MPDU);
		}
	}
	
	/*depending on group state the relevant remove function will be called */
	groupManagerRunGroupStateMachine(groupIndex, GROUP_MANAGER_GROUP_EVENT_REMOVE_GROUP, parameters);
}

/**********************************************************************************

groupManagerGroupEventStartNotifying 


Description:
------------
handle the event of starting notification to group (start activation process for
a group). Invoked from 'groupManagerTryToFindNewGroupForProbing'

Input: 
-----
groupIndex - the group index to activate
virtualGroupIndex - the virtual group index to take the parameters from

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerGroupEventStartNotifying(uint8 groupIndex, uint8 virtualGroupIndex)
{
	uint8 indexInArray = 0;
	StaId stationIndex = 0;
	GroupManagerGroupParameters_t *newGroup = &GroupManagerGroupDb[groupIndex];
	GroupManagerVirtualGroupParameters_t *virtualGroup = &GroupManagerVirtualGroupDatabase[virtualGroupIndex];
	ASSERT(GROUP_MANAGER_GROUP_STATE_IDLE == newGroup->state);
#ifdef GROUP_MANAGER_DEBUG								
	ILOG0_DD("groupManagerGroupEventStartNotifying: group index for notification %d, virtual group index %d", groupIndex, virtualGroupIndex);
#endif
	/* Reset only needed fields - some fields should stay permanent */
	newGroup->groupFailed = FALSE;
	newGroup->primaryStationsBitmap = 0;
	newGroup->nssBitmap = virtualGroup->nssBitmap;
	newGroup->numberOfParticipatingStations = virtualGroup->numberOfParticipatingStations;
	newGroup->groupIndex = groupIndex;

	
	// update group info in statistics db
	groupDbStatistics.groupInfoDb[groupIndex][STATISTICS_MU_GROUP_ID_INDEX] = groupIndex;
	groupDbStatistics.groupInfoDb[groupIndex][STATISTICS_NUM_OF_MU_GROUP_MEMBERS_INDEX] = virtualGroup->numberOfParticipatingStations;
		
	// set to invalid value all SIDs of group members - this db will be used in function GroupDb_FillGroupParams
	// and the function will check those stations which SID is set to legal value
	for (indexInArray = 0; indexInArray < GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP; indexInArray++)
	{
		newGroup->participatingStations[indexInArray] = INVALID_STA_INDEX;
	}
	
	ASSERT(virtualGroup->numberOfParticipatingStations <= GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP); //KW_FIX_FW_G Range Check
	
	for(indexInArray = 0; indexInArray < virtualGroup->numberOfParticipatingStations; indexInArray++)
	{
		stationIndex = virtualGroup->participatingStations[indexInArray];
		
		if(INVALID_STA_INDEX != stationIndex)
		{
			// for each member in new notifying group - turn off the bit of this group from virtual bitmap in station DB
			GroupManagerStationDb[stationIndex].virtualGroupsBitmap &= ~(0x1<<virtualGroupIndex);

			groupDbStatistics.groupInfoDb[groupIndex][STATISTICS_MU_GROUP_MEMBERS_ONE + indexInArray] = stationIndex;
			
			newGroup->participatingStations[indexInArray] = stationIndex;
			newGroup->notificationsArray[indexInArray] = GROUP_MANAGER_NOTIFICATION_NOT_RECEIVED;
			newGroup->mcs[indexInArray] = virtualGroup->mcs[indexInArray];
			newGroup->singleNstsPhyRates[indexInArray] = virtualGroup->ratePerStationSingleUserOneNsts[indexInArray];
			newGroup->doubleNstsPhyRates[indexInArray] = virtualGroup->ratePerStationSingleUserTwoNsts[indexInArray];
			/* update the userPositionBits of the station in the offset of new gorup with correct user position value of this station in this group
*/
			/* first turn off the relevant uspBits of the station*/
			GroupManagerStationDb[stationIndex].userPositionVector &= ~( GROUP_MANAGER_USP_BITS_MASK << (groupIndex * GROUP_MANAGER_USP_SIZE_SHIFT));
			/* write a new usp value in these bits*/
			GroupManagerStationDb[stationIndex].userPositionVector |= ((indexInArray & GROUP_MANAGER_USP_BITS_MASK) << (groupIndex * GROUP_MANAGER_USP_SIZE_SHIFT));
#ifdef GROUP_MANAGER_DEBUG
			ILOG0_DDD("groupManagerGroupEventStartNotifying: station index %d, group index %d, uspVector 0x%x",
						stationIndex, groupIndex, GroupManagerStationDb[stationIndex].userPositionVector);
#endif
			/* notify member that notification started
			   this function is ignored in all states of station except single user validation state. Because only from this state the station can move to 'NOTIFY' state*/
			groupManagerStationEventNotificationStarted(stationIndex, groupIndex);
			GidmManagerStartTxReq(&stationIndex);
		}
	}
	/* trigger timer of probing results of the group*/
	CalendarWheel_AddEntry(&GroupManagerGroupsCalendarWheel, (CalendarWheelEntry_t *)(&GroupManagerGroupDb[groupIndex]), GROUP_MANAGER_PROBING_RESULTS_TIMER);
	
	GroupManagerGlobalParametersDb.notifyingGroupsBitmap |= (0x1 << groupIndex);
	GroupManagerGlobalParametersDb.occupiedGroupsBitmap |= (0x1 << groupIndex);
	GroupManagerGlobalParametersDb.numberOfOccupiedGroups++;
	GroupManagerGlobalParametersDb.numberOfProbingGroups++;
	groupManagerChangeGroupState(&GroupManagerGroupDb[groupIndex], GROUP_MANAGER_GROUP_STATE_PROBING_NOTIFYING);
}

/**********************************************************************************

groupManagerGroupEventRemoveGroupInProbingNotifying


Description:
------------
handle the event remove group in the state of 'probing notifying' , group is in notify process


Input: 
-----
groupManagerGroupDataBaseEntry - pointer to the entry in the group manager group 
database	
parameters - station index which is used to indicate to GIDM manager which station Queue must be locked and which not
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerGroupEventRemoveGroupInProbingNotifying(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters)
{ 	
	/* remove notification timeout of the group - to avoid timeout*/
	CalendarWheel_RemoveEntry(&GroupManagerGroupsCalendarWheel, (CalendarWheelEntry_t*)groupManagerGroupDataBaseEntry);
	
	groupManagerGroupRemoveGroupInProbingNotifying(groupManagerGroupDataBaseEntry, parameters);
}

/**********************************************************************************

groupManagerGroupEventRemoveGroupInProbingWaitingForLinkAdaptation


Description:
------------
handle the event remove group in the state of 'probing waiting for link adaptation'
At this state "Add group" was sent previously to LA but "Add CFM" wasn't received yet and
due to some reason we need to remove this group. "Remove group " sent to LA.
LA will see "add group" message first and will send Add CFM" and possibly even probing results as well 
but Group manager will ignore "Add group CFM" and probing results from this point and on
because group state will be GROUP_MANAGER_GROUP_STATE_REMOVING_WAITING_FOR_LINK_ADAPTATION_CONFIRMATION.
Group manager will wait for "Remove group CFM"

Input: 
-----
groupManagerGroupDataBaseEntry - pointer to the entry in the group manager group 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerGroupEventRemoveGroupInProbingWaitingForLinkAdaptation(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters)
{ 	
	uint8 groupIndex = 0;
	UNUSED_PARAM(parameters);
	UNUSED_PARAM(groupManagerGroupDataBaseEntry);	
	/* set removing group bit for this group*/
	groupIndex =  groupManagerGroupDataBaseEntry->groupIndex;
	GroupManagerGlobalParametersDb.removingGroupsBitmap |= (0x1 << groupIndex);
	
	groupManagerChangeGroupState(groupManagerGroupDataBaseEntry, GROUP_MANAGER_GROUP_STATE_REMOVING_WAITING_FOR_LINK_ADAPTATION_CONFIRMATION);
	/* send to LA remove group request*/
	groupManagerSendRemoveGroupMessage(groupIndex);
}

/**********************************************************************************

groupManagerGroupEventRemoveGroupInProbing


Description:
------------
handle the event remove group in the state of 'probing'


Input: 
-----
groupManagerGroupDataBaseEntry - pointer to the entry in the group manager group 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerGroupEventRemoveGroupInProbing(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters)
{ 	
	uint8 groupIndex = 0;
	UNUSED_PARAM(parameters);
	groupIndex =  groupManagerGroupDataBaseEntry->groupIndex;
	
	GroupManagerGlobalParametersDb.removingGroupsBitmap |= (0x1 << groupIndex);
	/* remove probing timeout - because the group is removed */
	CalendarWheel_RemoveEntry(&GroupManagerGroupsCalendarWheel, (CalendarWheelEntry_t*)groupManagerGroupDataBaseEntry);
	/* on group removal - lock the group to stop transmissions in multi user for this group*/
#ifdef GROUP_MANAGER_DEBUG
	ILOG0_D("groupManagerGroupEventRemoveGroupInProbing: group %d", groupIndex);
#endif
	groupManagerLockGroup(groupIndex);
	
	groupManagerChangeGroupState(groupManagerGroupDataBaseEntry, GROUP_MANAGER_GROUP_STATE_REMOVING_WAITING_FOR_LOCK_AND_LINK_ADAPTATION_CONFIRMATIONS);
	/* send group remove request to LA*/												
	groupManagerSendRemoveGroupMessage(groupIndex);
}

/**********************************************************************************

groupManagerGroupEventRemoveGroupInActive


Description:
------------
handle the event remove group in the state of 'active'


Input: 
-----
groupManagerGroupDataBaseEntry - pointer to the entry in the group manager group 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerGroupEventRemoveGroupInActive(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters)
{ 	
	uint8 groupIndex = 0;
	UNUSED_PARAM(parameters);

	groupIndex = groupManagerGroupDataBaseEntry->groupIndex;
#ifdef GROUP_MANAGER_DEBUG
		ILOG0_D("groupManagerGroupEventRemoveGroupInActive: group %d", groupIndex);
#endif

	groupManagerLockGroup(groupIndex);
	
	GroupManagerGlobalParametersDb.removingGroupsBitmap |= (0x1 << groupManagerGroupDataBaseEntry->groupIndex);
	GroupManagerGlobalParametersDb.numberOfRemovingGroupsInActive++;
	groupManagerChangeGroupState(groupManagerGroupDataBaseEntry, GROUP_MANAGER_GROUP_STATE_REMOVING_WAITING_FOR_LOCK_AND_LINK_ADAPTATION_CONFIRMATIONS);
	groupManagerSendRemoveGroupMessage(groupIndex);
}

/**********************************************************************************

groupManagerGroupEventNotificationCfmInProbingNotifying


Description:
------------
handle the event of notification confirmation in the state of 'probing notifying'
during notification stage, reply received from one of the stations in group to which 
GIDM sent group id message


Input: 
-----
groupManagerGroupDataBaseEntry - pointer to the entry in the group manager group 
database	
parameters - notification parameters	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerGroupEventNotificationCfmInProbingNotifying(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters)
{ 	
	uint8 indexInArray = 0;
	uint8 numberOfSucceededNotifications = 0;
	StaId stationIndex = 0;
	GroupManagerNotificationParameters_t *notificationParameters = NULL;
	GroupManagerNotificationStatus_e notificationStatus = GROUP_MANAGER_NOTIFICATION_FAILED;
	uint8 groupIndex = 0;

	notificationParameters = (GroupManagerNotificationParameters_t *)parameters;
	stationIndex = notificationParameters->stationIndex;
	groupIndex = groupManagerGroupDataBaseEntry->groupIndex;

	notificationStatus = notificationParameters->notificationStatus;
#ifdef GROUP_MANAGER_DEBUG								
	ILOG0_DDD("groupManagerGroupEventNotificationCfmInProbingNotifying: on SID %d, groupIdx %d, notifStatus %d", stationIndex, groupIndex, notificationStatus);
#endif
	for(indexInArray = 0; indexInArray < groupManagerGroupDataBaseEntry->numberOfParticipatingStations; indexInArray++)
	{
		/* found index of station which sent CFM to group id message*/	
		if(groupManagerGroupDataBaseEntry->participatingStations[indexInArray] == stationIndex)
		{
			/* save notification status for this station*/
			groupManagerGroupDataBaseEntry->notificationsArray[indexInArray] = notificationStatus;
		}
		if(GROUP_MANAGER_NOTIFICATION_SUCCEEDED == groupManagerGroupDataBaseEntry->notificationsArray[indexInArray])
		{
			numberOfSucceededNotifications++;
		}
	}	
	
	if(GROUP_MANAGER_NOTIFICATION_SUCCEEDED == notificationStatus)
	{
		/* if received CFMs from all stations in notifcation process of the group*/
		if(numberOfSucceededNotifications == groupManagerGroupDataBaseEntry->numberOfParticipatingStations)
		{
#ifdef GROUP_MANAGER_DEBUG							
			ILOG0_V("groupManagerGroupEventNotificationCfmInProbingNotifying: All notifications arrived successfully");
#endif
			/* remove notification timeout event*/
			CalendarWheel_RemoveEntry(&GroupManagerGroupsCalendarWheel, (CalendarWheelEntry_t*)groupManagerGroupDataBaseEntry);
			
			for(indexInArray = 0; indexInArray < groupManagerGroupDataBaseEntry->numberOfParticipatingStations; indexInArray++)
			{
				/* this function has effect only on group members which are in 'notify' state and they will move to 'probing' state.
				in other states the function is ignored because state remains unchanged*/
				groupManagerStationEventProbingStarted(groupManagerGroupDataBaseEntry->participatingStations[indexInArray], groupIndex);
			}

			// turn off notifying bit and set probing bit for this group
			GroupManagerGlobalParametersDb.notifyingGroupsBitmap &= ~(0x1 << groupIndex);
			GroupManagerGlobalParametersDb.probingGroupsBitmap|= (0x1 << groupIndex);
			/* change state to wait for cfm on add group*/
			groupManagerChangeGroupState(groupManagerGroupDataBaseEntry, GROUP_MANAGER_GROUP_STATE_PROBING_WAITING_FOR_LINK_ADAPTATION);

			/* before activating group in HW fill in group Selector and Aggregation DB*/
			groupManagerConfigureHwGroupDbs(groupIndex);
			/* send add group request*/
			groupManagerSendAddGroupMessage(groupIndex);
			
			/* try to find next candidate for probing*/
			groupManagerTryToFindNewGroupForProbing(FALSE);
			/* trigger timer mechanism again if finished creating static group*/
			if(GroupManagerGlobalParametersDb.globalState == GROUP_MANAGER_GLOBAL_STATE_CREATING_STATIC_GROUP)
			{
				
#ifdef GROUP_MANAGER_DEBUG											
				ILOG0_V("groupManagerGroupEventNotificationCfmInProbingNotifying: timer started after static group generation");
#endif
				OSAL_SET_TIMER_EXPLICIT(GROUP_MANAGER_TRAINING_TIMER_EXPIRED, OSAL_TIMERS_MS_TO_K_TICKS(GROUP_MANAGER_CALENDAR_WHEEL_TIMER_IN_MILLISECONDS), TASK_GROUP_MANAGER);
			
				groupManagerChangeGlobalState(GROUP_MANAGER_GLOBAL_STATE_ACTIVE);
			}
		}
	}
	else
	{
		/* even if Notification has failed for one of the group members - move to removing state */
		// if notification failed for static group notify user by printing to cmd
		if((GroupManagerGlobalParametersDb.staticOccupiedGroupIndex != GROUP_MANAGER_INVALID_GROUP_INDEX) && 
				(GroupManagerGlobalParametersDb.staticOccupiedGroupIndex == groupIndex))
		{
			SERIAL_TRACE("Static group generation failed because notification stage has failed",0,0,0);
			ILOG0_V("[groupManagerGroupEventNotificationCfmInProbingNotifying], Static group generation failed because notification stage has failed");
			GroupManagerGlobalParametersDb.staticOccupiedGroupIndex = GROUP_MANAGER_INVALID_GROUP_INDEX;
			GroupManagerGlobalParametersDb.staticVirtualGroupIndex = GROUP_MANAGER_INVALID_GROUP_INDEX;
			OSAL_SET_TIMER_EXPLICIT(GROUP_MANAGER_TRAINING_TIMER_EXPIRED, OSAL_TIMERS_MS_TO_K_TICKS(GROUP_MANAGER_CALENDAR_WHEEL_TIMER_IN_MILLISECONDS), TASK_GROUP_MANAGER);
		}
		CalendarWheel_RemoveEntry(&GroupManagerGroupsCalendarWheel, (CalendarWheelEntry_t*)groupManagerGroupDataBaseEntry);
		if(numberOfSucceededNotifications == (groupManagerGroupDataBaseEntry->numberOfParticipatingStations - 1))
		{
#ifdef GROUP_MANAGER_DEBUG										
			ILOG0_V("groupManagerGroupEventNotificationCfmInProbingNotifying: All notifications arrived - last FAILED");
#endif
			/* In this case all other notifications have already ended and the last one failed */
			for(indexInArray = 0; indexInArray < groupManagerGroupDataBaseEntry->numberOfParticipatingStations; indexInArray++)
			{
				if(GROUP_MANAGER_NOTIFICATION_FAILED == groupManagerGroupDataBaseEntry->notificationsArray[indexInArray])
				{
					groupManagerStationEventRemoveGroupDueToStation(groupManagerGroupDataBaseEntry->participatingStations[indexInArray], groupIndex);					
				}
				else
				{
					/* Station whose notification succeeded or not transmitted wont be "punished" */
					groupManagerStationEventRemoveGroup(groupManagerGroupDataBaseEntry->participatingStations[indexInArray], groupIndex);
				}
			}
			GroupManagerGlobalParametersDb.notifyingGroupsBitmap &= ~(0x1 << groupIndex);
			GroupManagerGlobalParametersDb.occupiedGroupsBitmap &= ~(0x1 << groupIndex);
			GroupManagerGlobalParametersDb.numberOfOccupiedGroups --;
			GroupManagerGlobalParametersDb.numberOfProbingGroups --;
			
			/* all replies were received , move to idle immediately*/
			groupManagerChangeGroupState(groupManagerGroupDataBaseEntry, GROUP_MANAGER_GROUP_STATE_IDLE);

			// group was removed update statistics info
			GroupManagerResetGroupMembersStatistic(groupIndex);

			groupManagerTryToFindNewGroupForProbing(FALSE);
		}
		/* not all stations replied to group id message*/
		else
		{
#ifdef GROUP_MANAGER_DEBUG										
			ILOG0_V("groupManagerGroupEventNotificationCfmInProbingNotifying: one of notification failed, more notifications will arrive - start remove group now");
#endif
			/* the station has failed but still have more stations which didn't return reply yet and it might be that we need to wait for their reply if 
			   group id message could not be extracted from queue*/
			groupManagerGroupRemoveGroupInProbingNotifying(groupManagerGroupDataBaseEntry, NULL);
		}
	}
}

/**********************************************************************************

groupManagerGroupEventNotificationCfmInRemovingWaitingForNotifications


Description:
------------
handle the event of notification confirmation in the state of 'removing - waiting
for notifications'. Group manager requested to Abort notification process on group due
to some reason (for eaxample one of the stations disconnected) , remove group process 
is triggered. But GIDM manager couldn't abort notification message for some of the stations
in group and notification CFM for one of those stations arrived.
Group manager will use the status returned to check if station failed and if TRUE
will move it to HALT state
This function can be invoked in the following cases:
a. Notification timeout occurred- for all group members status in notificationsArray 
	will be set to GROUP_MANAGER_NOTIFICATION_TIMEOUT and this function is called afterwards
b. the first member that returned notification status FAILED - set for all other members for which notification
   didn't arrive yet status "GROUP_MANAGER_NOTIFICATION_CANCELLED" , abort notification process 
   for entire group and remove it,

Input: 
-----
groupManagerGroupDataBaseEntry - pointer to the entry in the group manager group 
database	
parameters - notification parameters	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerGroupEventNotificationCfmInRemovingWaitingForNotifications(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters)
{ 	
	uint8 indexInArray = 0;
	uint8 numberOfNotificationsConfirmed = 1; /* At least one notification has confirmed -  the current one */
	StaId stationIndex = 0;
	GroupManagerNotificationParameters_t *notificationParameters = NULL;
	uint8 groupIndex = 0;

	notificationParameters = (GroupManagerNotificationParameters_t *)parameters;
	stationIndex = notificationParameters->stationIndex;
	groupIndex = groupManagerGroupDataBaseEntry->groupIndex;

	for(indexInArray = 0; indexInArray < groupManagerGroupDataBaseEntry->numberOfParticipatingStations; indexInArray++)
	{
		/* found index of station which sent cfm.
		 If requested notification through GIDM but notification wasn't received yet 
		 send stopTx request to GIDM Mng with reason = GIDM_STOP_REASON_REMOVE_STA. This will indicate to GIDM that
		 it doesn't need to lock station QUEUE, this will be done through BSS API which removes the station*/
		if(groupManagerGroupDataBaseEntry->participatingStations[indexInArray] == stationIndex)
		{
			
			/* in this case remove was originally invoked because eirlier some other member failed during 
			notification and group manager changed it's notificationsArray value to GROUP_MANAGER_NOTIFICATION_TIMEOUT*/
			if(GROUP_MANAGER_NOTIFICATION_TIMEOUT == groupManagerGroupDataBaseEntry->notificationsArray[indexInArray])
			{
				/* even if cfm has arrived we still consider it as failure in this scenario*/ 
				groupManagerGroupDataBaseEntry->notificationsArray[indexInArray] = GROUP_MANAGER_NOTIFICATION_FAILED;
			}
			/* in this case group removal triggered due to some other reason not notification failure*/
			else
			{
				/* store real notification status if remove done in anothwr case in order to know how the station will be punished later*/
				groupManagerGroupDataBaseEntry->notificationsArray[indexInArray] = notificationParameters->notificationStatus;
			}
		}
		/*in the same loop count the number of stations which returned notification CFM before abort of Notification process was triggered*/
		else if(GROUP_MANAGER_NOTIFICATION_CANCELED != groupManagerGroupDataBaseEntry->notificationsArray[indexInArray])
		{
			/* When the notification status is different from GROUP_MANAGER_NOTIFICATION_CANCELED in this state it means that it was confirmed */
			numberOfNotificationsConfirmed ++;
		}
	}
#ifdef GROUP_MANAGER_DEBUG							
/*	ILOG0_DDDDDDD("groupManagerGroupEventNotificationCfmInRemovingWaitingForNotifications: station index %d, status received %d, numOfConfirmations till now = %d, notification array: (%d, %d, %d, %d)",
		stationIndex,  notificationParameters->notificationStatus, numberOfNotificationsConfirmed,
		groupManagerGroupDataBaseEntry->notificationsArray[0], groupManagerGroupDataBaseEntry->notificationsArray[1],
		groupManagerGroupDataBaseEntry->notificationsArray[2], groupManagerGroupDataBaseEntry->notificationsArray[3]);*/
#endif
	
	/* if all notifications group waited for have arrived*/
	if(numberOfNotificationsConfirmed == groupManagerGroupDataBaseEntry->numberOfParticipatingStations)
	{
		/* The group has finished its removal process */
		for(indexInArray = 0; indexInArray < groupManagerGroupDataBaseEntry->numberOfParticipatingStations; indexInArray++)
		{
			if(GROUP_MANAGER_NOTIFICATION_FAILED == groupManagerGroupDataBaseEntry->notificationsArray[indexInArray])
			{
				/* Notify member which failed in notification process that group remove process is done. 
				 This station will probably be punished*/
				groupManagerStationEventRemoveGroupDueToStation(groupManagerGroupDataBaseEntry->participatingStations[indexInArray], groupIndex);					
			}
			else
			{
				/* Station whose notification succeeded or not transmitted wont be "punished" */
				groupManagerStationEventRemoveGroup(groupManagerGroupDataBaseEntry->participatingStations[indexInArray], groupIndex);
			}
		}
		GroupManagerGlobalParametersDb.removingGroupsBitmap &= ~(0x1 << groupIndex);
		GroupManagerGlobalParametersDb.notifyingGroupsBitmap &= ~(0x1 << groupIndex);
		GroupManagerGlobalParametersDb.occupiedGroupsBitmap &= ~(0x1 << groupIndex);
		GroupManagerGlobalParametersDb.numberOfOccupiedGroups --;
		GroupManagerGlobalParametersDb.numberOfProbingGroups --;

		groupManagerChangeGroupState(groupManagerGroupDataBaseEntry, GROUP_MANAGER_GROUP_STATE_IDLE);
		GroupManagerResetGroupMembersStatistic(groupIndex);
		groupManagerFinishGlobalRemovalProcess();
		/* if group manager will be in state idle, virtual group bitmap will be 0, so new probing group won't be found*/
		groupManagerTryToFindNewGroupForProbing(FALSE);
	}
}

/**********************************************************************************

groupManagerGroupEventTimerInProbingNotifying


Description:
------------
handle the event of timer in the state of 'probing notifying' -
timeout during notify process before probing

Input: 
-----
groupManagerGroupDataBaseEntry - pointer to the entry in the group manager group 
database	
parameters - not used
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerGroupEventTimerInProbingNotifying(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters)
{ 	
	uint8 indexInArray = 0;
	GidmManagerStopTxReq_t stopTxParams;
	UNUSED_PARAM(parameters);
	memset(&stopTxParams, 0, sizeof(GidmManagerStopTxReq_t));
	/* for all stations which didn't send cfm to GIDM manager send StopTx*/
	for(indexInArray = 0; indexInArray < groupManagerGroupDataBaseEntry->numberOfParticipatingStations; indexInArray++)
	{	
		if(GROUP_MANAGER_NOTIFICATION_NOT_RECEIVED == groupManagerGroupDataBaseEntry->notificationsArray[indexInArray])
		{
			groupManagerGroupDataBaseEntry->notificationsArray[indexInArray] = GROUP_MANAGER_NOTIFICATION_TIMEOUT;
			stopTxParams.staIndex = groupManagerGroupDataBaseEntry->participatingStations[indexInArray];
			stopTxParams.stopReason = GIDM_STOP_REASON_TIMEOUT;
			GidmManagerStopTxReq(&stopTxParams);	
		}		
	}
	GroupManagerGlobalParametersDb.removingGroupsBitmap |= (0x1 << groupManagerGroupDataBaseEntry->groupIndex);
	groupManagerChangeGroupState(groupManagerGroupDataBaseEntry, GROUP_MANAGER_GROUP_STATE_REMOVING_WAITING_FOR_NOTIFICATIONS);
}

/**********************************************************************************

groupManagerGroupEventTimerInProbing


Description:
------------
handle the event of timer in the state of 'probing' - timeout during probing process of the group


Input: 
-----
groupManagerGroupDataBaseEntry - pointer to the entry in the group manager group 
database	
parameters - not used
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerGroupEventTimerInProbing(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters)
{ 	
	uint8 groupIndex = 0;
	uint8 i = 0;
	GroupManagerProbingResultsMessage_t* probingResults = NULL; 
	K_MSG *groupManagerProbingResultsMessage = NULL;
	UNUSED_PARAM(parameters);

	groupIndex = groupManagerGroupDataBaseEntry->groupIndex;
	// even if probing results didn't arrive on time - static group is not deleted
	if(GroupManagerGlobalParametersDb.staticOccupiedGroupIndex != groupIndex)
	{
		groupManagerGroupDataBaseEntry->groupFailed = TRUE;
		/* lock group in HW DB*/
		groupManagerLockGroup(groupIndex);
		/* turn on bit of the goupr in removing bitmap*/
		GroupManagerGlobalParametersDb.removingGroupsBitmap |= (0x1 << groupManagerGroupDataBaseEntry->groupIndex);
		/* in state of probing need to wait for Lock cfm and LA cfm on remove group request*/
		groupManagerChangeGroupState(groupManagerGroupDataBaseEntry, GROUP_MANAGER_GROUP_STATE_REMOVING_WAITING_FOR_LOCK_AND_LINK_ADAPTATION_CONFIRMATIONS);
		/* send remove group request to LA*/
		groupManagerSendRemoveGroupMessage(groupIndex);
	}
	else
	{
		// in case of static group send probing results to turn the probing static group to active static group
		groupManagerProbingResultsMessage = OSAL_GET_MESSAGE(sizeof(GroupManagerProbingResultsMessage_t));
		probingResults = ((GroupManagerProbingResultsMessage_t *)pK_MSG_DATA(groupManagerProbingResultsMessage));
		probingResults->probingGroupIndex = groupIndex;
		memset(probingResults->groupMarks, INVALID_GROUP_MARKS, sizeof(probingResults->groupMarks));
		probingResults->groupMarks[groupIndex] = GROUP_MANAGER_BEST_STATIC_GROUP_MARK;
		for (i=0; i < GroupManagerGroupDb[groupIndex].numberOfParticipatingStations; i++)
		{
			probingResults->stationMuMark[i] = (GroupManagerGlobalParametersDb.muConfigParams.stationGradeInProbingGroupThreshold+1);
			probingResults->stationSuMark[i] = 0;
		}
		
		OSAL_SEND_MESSAGE(GROUP_MANAGER_PROBING_RESULTS, TASK_GROUP_MANAGER, groupManagerProbingResultsMessage, VAP_ID_DO_NOT_CARE);
	}
}

/**********************************************************************************

groupManagerGroupEventAddGroupCfmInProbingWaitingForLa


Description:
------------
handle the event of add group confirmation in the state of 'probing - waiting for
link adaptation'. In this state group finished notify process before the probing 
and sent "add group" event to LA before starting the actual probing. Add Group Cfm arrived and 
group will be moving to "Probing state"


Input: 
-----
groupManagerGroupDataBaseEntry - pointer to the entry in the group manager group 
database	
parameters - not used
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerGroupEventAddGroupCfmInProbingWaitingForLa(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters)
{ 	
	uint8 indexInArray = 0;
	StaId primaryStationIndex = INVALID_STA_INDEX; /* this index will hold index of primary member in probing state*/
	StaId primaryActiveStationIndex = INVALID_STA_INDEX;/* this index will hold index of primary member in active state*/
	StaId stationIndex = 0;
	uint8 groupIndex = groupManagerGroupDataBaseEntry->groupIndex; 
	UNUSED_PARAM(parameters);


	/* The next loop is to find a temporary primary station for the group. the priority is:
	   1)probing and is not referencing to any other probing group
	   2)active with primary
	   3)active with no primary (secondary)
	   a station in a state different from the 3 above cant be primary for this group */

	/* The goal is to find probing member if such member exists otherwise active member will be chosen*/
	for(indexInArray = 0; indexInArray < groupManagerGroupDataBaseEntry->numberOfParticipatingStations; indexInArray++)
	{	
		stationIndex = groupManagerGroupDataBaseEntry->participatingStations[indexInArray];
		/* if in state probing and is not a primary of any other probing group*/
		if((GROUP_MANAGER_STATION_STATE_PROBING == GroupManagerStationDb[stationIndex].state) && (GROUP_MANAGER_INVALID_GROUP_INDEX == GroupManagerStationDb[stationIndex].tempPrimaryGroup))
		{
			// probing member found no need to continue the search, will be primary for this probing group
			primaryStationIndex = stationIndex;
			break;
		}
		else if(GROUP_MANAGER_STATION_STATE_ACTIVE == GroupManagerStationDb[stationIndex].state)
		{
			/* if active with some primary group */
			if(GROUP_MANAGER_INVALID_GROUP_INDEX != GroupManagerStationDb[stationIndex].primaryGroup)
			{
				primaryActiveStationIndex = stationIndex;
			}
			/* if secondary */
			else if(INVALID_STA_INDEX == primaryActiveStationIndex)
			{
				primaryActiveStationIndex = stationIndex;
			}
		}		
	}
	/* if found primary for this group*/
	if((INVALID_STA_INDEX != primaryStationIndex) || (INVALID_STA_INDEX != primaryActiveStationIndex))
	{
		
		/* if primary is not in probing state but in active state */
		if(INVALID_STA_INDEX == primaryStationIndex)
		{
			/* move to temp active*/
			groupManagerStationEventMoveToTempActive(primaryActiveStationIndex, groupIndex);
			/* overwrite primary index with the found index*/
			primaryStationIndex = primaryActiveStationIndex;
		}
		
		groupManagerSetStationPrimaryBitInGroupsDb(primaryStationIndex, groupIndex);
		/* store index of probing group in TempPrimaryGroup of the chosen station*/
		GroupManagerStationDb[primaryStationIndex].tempPrimaryGroup = groupIndex;	
		
		CalendarWheel_AddEntry(&GroupManagerGroupsCalendarWheel, (CalendarWheelEntry_t *)groupManagerGroupDataBaseEntry, GROUP_MANAGER_PROBING_RESULTS_TIMER); 

		groupManagerConfigureNewGroupForSelector(groupIndex, primaryStationIndex);
		
		groupManagerChangeGroupState(groupManagerGroupDataBaseEntry, GROUP_MANAGER_GROUP_STATE_PROBING);
	}
	else
	{
		/* No primary for this group - remove group */
		GroupManagerGlobalParametersDb.removingGroupsBitmap |= (0x1 << groupIndex);
		/* wait for LA cfm on remove group*/
		groupManagerChangeGroupState(groupManagerGroupDataBaseEntry, GROUP_MANAGER_GROUP_STATE_REMOVING_WAITING_FOR_LINK_ADAPTATION_CONFIRMATION);
		groupManagerSendRemoveGroupMessage(groupIndex);
		// if failure occurred in static group generation
		if(GroupManagerGlobalParametersDb.staticOccupiedGroupIndex == groupIndex)
		{
			GroupManagerGlobalParametersDb.staticOccupiedGroupIndex = GROUP_MANAGER_INVALID_GROUP_INDEX;
			SERIAL_TRACE("Static group couldn't be created, no available primary found for this group",0,0,0);
			ILOG0_V("[groupManagerGroupEventAddGroupCfmInProbingWaitingForLa], Static group couldn't be created, no available primary found for this group");
		}
	}
}

/**********************************************************************************

groupManagerGroupEventRemoveGroupCfmInRemovingWaitingForLockAndLaCfms


Description:
------------
handle the event of remove group confirmation in the state of 'removing - waiting for
lock and link adaptation confirmations'. RemoveGroup CFM arrived from LA


Input: 
-----
groupManagerGroupDataBaseEntry - pointer to the entry in the group manager group 
database	
parameters - not used
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerGroupEventRemoveGroupCfmInRemovingWaitingForLockAndLaCfms(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters)
{ 	
	UNUSED_PARAM(parameters);
	/* keep waiting for Lock cfm*/
	groupManagerChangeGroupState(groupManagerGroupDataBaseEntry, GROUP_MANAGER_GROUP_STATE_REMOVING_WAITING_FOR_LOCK_CONFIRMATION);
}

/**********************************************************************************

groupManagerGroupEventRemoveGroupCfmInRemovingWaitingForLaCfm


Description:
------------
handle the event of remove group confirmation in the state of 'removing - waiting for
link adaptation confirmation'. Lock CFM arrived earlier now waiting only for remove group cfm form LA
and it arrived. Finish group removal process


Input: 
-----
groupManagerGroupDataBaseEntry - pointer to the entry in the group manager group 
database	
parameters - not used
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerGroupEventRemoveGroupCfmInRemovingWaitingForLaCfm(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters)
{ 
	UNUSED_PARAM(parameters);
#ifdef GROUP_MANAGER_DEBUG								
	ILOG0_V("groupManagerGroupEventRemoveGroupCfmInRemovingWaitingForLaCfm");
#endif
	groupManagerFinishGroupRemovalProcess(groupManagerGroupDataBaseEntry);
}

/**********************************************************************************

groupManagerGroupEventProbingResultsAreReadyInProbing


Description:
------------
handle the event of probing results received in the state of 'probing'
If the score of this group is above the threshold for all stations in the group check if
the station can be primary of the group or some other group has a higher score for this station
this process is done by invoking groupManagerStationEventActivateGroup on station


Input: 
-----
groupManagerGroupDataBaseEntry - pointer to the entry in the group manager group 
database	
parameters - the address of the message that contains all the parameters
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerGroupEventProbingResultsAreReadyInProbing(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters)
{ 	
	uint8 isGroupValid = TRUE;
	GroupManagerProbingResultsMessage_t *groupManagerProbingResultsParameters = NULL;
	uint8 indexInArray = 0;
	uint8 probingGroupMark = 0;
	uint8 lowestMark = GROUP_MANAGER_BEST_GROUP_MARK;
	uint16 tempBitmap = 0;
	uint8 probingGroupIndex = GROUP_MANAGER_INVALID_GROUP_INDEX;
	uint8 groupIndex = GROUP_MANAGER_INVALID_GROUP_INDEX;
	uint8 lowestMarkGroupIndex = GROUP_MANAGER_INVALID_GROUP_INDEX;
	StaId primaryStationIndex = INVALID_STA_INDEX;
	StaId stationIndex = INVALID_STA_INDEX;
	uint16 probingGroupsBitmap = 0;

	groupManagerProbingResultsParameters = (GroupManagerProbingResultsMessage_t *)parameters;
	probingGroupIndex = groupManagerProbingResultsParameters->probingGroupIndex;
	probingGroupMark = groupManagerProbingResultsParameters->groupMarks[probingGroupIndex];
#ifdef GROUP_MANAGER_DEBUG							
	ILOG0_DD("groupManagerGroupEventProbingResultsAreReadyInProbing: group Idx %d, group mark %d", probingGroupIndex, probingGroupMark);
	ILOG0_DDDD("groupManagerGroupEventProbingResultsAreReadyInProbing: states (%d, %d, %d, %d)",
		GroupManagerStationDb[groupManagerGroupDataBaseEntry->participatingStations[0]].state,
		GroupManagerStationDb[groupManagerGroupDataBaseEntry->participatingStations[1]].state,
		GroupManagerStationDb[groupManagerGroupDataBaseEntry->participatingStations[2]].state,
		GroupManagerStationDb[groupManagerGroupDataBaseEntry->participatingStations[3]].state);
#endif

	/* remove timeout event of probing*/
	CalendarWheel_RemoveEntry(&GroupManagerGroupsCalendarWheel, (CalendarWheelEntry_t*)groupManagerGroupDataBaseEntry); 

	/* Check if group passes validation criteria */
	if(probingGroupMark >= GroupManagerGlobalParametersDb.muConfigParams.probingGroupGradeThreshold)
	{
		for(indexInArray = 0; indexInArray < groupManagerGroupDataBaseEntry->numberOfParticipatingStations; indexInArray++)
		{
			probingGroupsBitmap = (GroupManagerStationDb[groupManagerGroupDataBaseEntry->participatingStations[indexInArray]].groupsBitmap & GroupManagerGlobalParametersDb.probingGroupsBitmap);
			if((GroupManagerStationDb[groupManagerGroupDataBaseEntry->participatingStations[indexInArray]].state == GROUP_MANAGER_STATION_STATE_PROBING) &&
				(probingGroupsBitmap & (0x1 << probingGroupIndex)) && 
				(groupManagerProbingResultsParameters->stationMuMark[indexInArray] < GroupManagerGlobalParametersDb.muConfigParams.stationGradeInProbingGroupThreshold))
			{
				isGroupValid = FALSE; 
				groupManagerGroupDataBaseEntry->groupFailed = TRUE;
#ifdef GROUP_MANAGER_DEBUG											
				ILOG0_DD("groupManagerGroupEventProbingResultsAreReadyInProbing: for group %d station %d mark is below threshold!",
					probingGroupIndex, groupManagerGroupDataBaseEntry->participatingStations[indexInArray]);

				ILOG0_DDDD("groupManagerGroupEventProbingResultsAreReadyInProbing: (%d, %d, %d, %d)",
					groupManagerProbingResultsParameters->stationMuMark[0],
					groupManagerProbingResultsParameters->stationMuMark[1],
					groupManagerProbingResultsParameters->stationMuMark[2],
					groupManagerProbingResultsParameters->stationMuMark[3]);
#endif
				break;
			}
		}	
	}
	else
	{
		isGroupValid = FALSE;
		groupManagerGroupDataBaseEntry->groupFailed = TRUE;
#ifdef GROUP_MANAGER_DEBUG									
		ILOG0_V("groupManagerGroupEventProbingResultsAreReadyInProbing: Group mark is below threshold!!!");
#endif
	}		

	if(isGroupValid)
	{
#ifdef GROUP_MANAGER_DEBUG									
		ILOG0_DD("groupManagerGroupEventProbingResultsAreReadyInProbing: num of active groups %d, num of removing active groups %d",
			GroupManagerGlobalParametersDb.numberOfActiveGroups, GroupManagerGlobalParametersDb.numberOfRemovingGroupsInActive);
#endif
		
		/* number of active groups includes also groups in remove process, hence we need to decrease this number from active groups to find out the actual
		number of active groups. If after remove is over the actual number of active groups equal to MAX_ACTIVE_THRESH, no free space for new active group,
		need to find a group with lower score to overwrite it*/
		if(GROUP_MANAGER_MAX_NUMBER_OF_ACTIVE_GROUPS == (GroupManagerGlobalParametersDb.numberOfActiveGroups - GroupManagerGlobalParametersDb.numberOfRemovingGroupsInActive))
		{
			/* Active groups are full - find the minimum group for comparison from active groups that are not in removal process */
			tempBitmap = GroupManagerGlobalParametersDb.activeGroupsBitmap & (~GroupManagerGlobalParametersDb.removingGroupsBitmap);
			/* find the group with lowest score among actual active groups*/
			while(tempBitmap)
			{
				groupIndex = Utils_CountTrailingZeros(tempBitmap);
				tempBitmap &= ~(0x1 << groupIndex);
				if(groupManagerProbingResultsParameters->groupMarks[groupIndex] <= lowestMark)
				{
					lowestMark = groupManagerProbingResultsParameters->groupMarks[groupIndex];
					lowestMarkGroupIndex = groupIndex;
				}
			}	
#ifdef GROUP_MANAGER_DEBUG							
			ILOG0_DDD("groupManagerGroupEventProbingResultsAreReadyInProbing: active groups are full. Worth active group index %d, group mark %d, probing grop mark %d", 
				lowestMarkGroupIndex, lowestMark, probingGroupMark);
#endif
			
			// TODO: check if the mark of the probing should be higher than the minimum group + threshold in the SAS
			if(probingGroupMark < lowestMark)
			{
				isGroupValid = FALSE;
			}
		}
	}

	if(isGroupValid)
	{
		/* Trigger activate group on all probing group members - as the result some group members migth have different primary group at the end*/
		for(indexInArray = 0; indexInArray < groupManagerGroupDataBaseEntry->numberOfParticipatingStations; indexInArray++)
		{
#ifdef GROUP_MANAGER_DEBUG										
			ILOG0_D("groupManagerGroupEventProbingResultsAreReadyInProbing: Send activateGroup event to SID %d", groupManagerGroupDataBaseEntry->participatingStations[indexInArray]);
#endif
			groupManagerStationEventActivateGroup(groupManagerGroupDataBaseEntry->participatingStations[indexInArray], groupManagerProbingResultsParameters);
		}

		/* Check if there is at least one primary for the probing group */
		for(indexInArray = 0; indexInArray < groupManagerGroupDataBaseEntry->numberOfParticipatingStations; indexInArray++)
		{
			stationIndex = groupManagerGroupDataBaseEntry->participatingStations[indexInArray];
			if(probingGroupIndex == GroupManagerStationDb[stationIndex].primaryGroup)
			{
				primaryStationIndex = stationIndex;
				break;
			}
		}
		/* if the probing group has no primary (all group members have a groups with better score) the group will start a remove process
		   Remove process required because in HW and LA this group is active and can be chosen for transmission so it must be locked*/
		
		if(INVALID_STA_INDEX == primaryStationIndex)
		{
			isGroupValid = FALSE;
		}
	}

	if(isGroupValid)
	{
#ifdef GROUP_MANAGER_DEBUG									
		ILOG0_D("groupManagerGroupEventProbingResultsAreReadyInProbing: Valid probing group found. Group idx %d. will be active",probingGroupIndex);
#endif

		/* check again this condition because as the result of invocking groupManagerStationEventActivateGroup on each group member some groups could be removed and 
		the number of actual active groups could decrease. If so than the condition is false so no need to remove from DB group with lowest rank previously found 
		instead new probing group can be added to free space*/
		if(GROUP_MANAGER_MAX_NUMBER_OF_ACTIVE_GROUPS == (GroupManagerGlobalParametersDb.numberOfActiveGroups - GroupManagerGlobalParametersDb.numberOfRemovingGroupsInActive))
		{
			/* The probing group is valid and has a mark better than the minimum and the active groups are full even after activating the group - remove the minimum group */
			groupManagerGroupEventRemoveGroup(lowestMarkGroupIndex, NULL);
		}

		/* activate the probing group*/
		GroupManagerGlobalParametersDb.probingGroupsBitmap &= ~(0x1 << probingGroupIndex);
		GroupManagerGlobalParametersDb.numberOfProbingGroups--;
		GroupManagerGlobalParametersDb.activeGroupsBitmap |= (0x1 << probingGroupIndex);
		GroupManagerGlobalParametersDb.numberOfActiveGroups ++;

		groupManagerChangeGroupState(groupManagerGroupDataBaseEntry, GROUP_MANAGER_GROUP_STATE_ACTIVE);
		if((GroupManagerGlobalParametersDb.staticOccupiedGroupIndex != GROUP_MANAGER_INVALID_GROUP_INDEX) && 
			(GroupManagerGlobalParametersDb.staticOccupiedGroupIndex == probingGroupIndex))
		{
			SERIAL_TRACE("Static group was created",0,0,0);
			ILOG0_V("[groupManagerGroupEventProbingResultsAreReadyInProbing], Static group was created");
		}
		groupManagerTryToFindNewGroupForProbing(FALSE);
		
	}
	else
	{
#ifdef GROUP_MANAGER_DEBUG									
		ILOG0_D("groupManagerGroupEventProbingResultsAreReadyInProbing: Probing group idx %d is not valid. No primary station availabe for it. Will be removed", probingGroupIndex);
#endif
		// if probing group found as not valid Lock this group
		groupManagerLockGroup(probingGroupIndex);
		// send remove group msg to LA - because in LA this group is active
		groupManagerSendRemoveGroupMessage(probingGroupIndex);
		GroupManagerGlobalParametersDb.removingGroupsBitmap |= (0x1 << probingGroupIndex);
		groupManagerChangeGroupState(groupManagerGroupDataBaseEntry, GROUP_MANAGER_GROUP_STATE_REMOVING_WAITING_FOR_LOCK_AND_LINK_ADAPTATION_CONFIRMATIONS);	
		if((GroupManagerGlobalParametersDb.staticOccupiedGroupIndex != GROUP_MANAGER_INVALID_GROUP_INDEX) && 
			(GroupManagerGlobalParametersDb.staticOccupiedGroupIndex == probingGroupIndex))
		{
			SERIAL_TRACE("Failed to generate static group, probing results failed",0,0,0);
			ILOG0_V("[groupManagerGroupEventProbingResultsAreReadyInProbing], Failed to generate static group, probing results failed");
			GroupManagerGlobalParametersDb.staticOccupiedGroupIndex = GROUP_MANAGER_INVALID_GROUP_INDEX;
		}
	}
}

/**********************************************************************************

groupManagerGroupEventPropingResultsReadyInActive



Description:
------------
handle the event of probing results received in the state of 'active'
this might happen in case timeout on probing results occurred for static group and we sent 
fake probing results to move static group to active state even without probing results
After group is in active state probing results for the group arrived and we need to ignore them.
For non static grops this event must cause an ASSERT


Input: 
-----
groupManagerGroupDataBaseEntry - pointer to the entry in the group manager group 
database	
parameters - the address of the message that contains all the parameters
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerGroupEventPropingResultsReadyInActive(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters)
{
	UNUSED_PARAM(parameters);
	if(groupManagerGroupDataBaseEntry->groupIndex == GroupManagerGlobalParametersDb.staticOccupiedGroupIndex)
	{
#ifdef GROUP_MANAGER_DEBUG									
		ILOG0_DD("Ignore 'probing results ready event' for group %d (static group) in state: %d", groupManagerGroupDataBaseEntry->groupIndex, groupManagerGroupDataBaseEntry->state);
#endif
	}
	else
	{
#ifdef GROUP_MANAGER_DEBUG									
		ILOG0_DD("Assert group %d state: %d on probing results ready event", groupManagerGroupDataBaseEntry->groupIndex, groupManagerGroupDataBaseEntry->state);
#endif
		FATAL("Assert in group manager group event");
	}
}

/**********************************************************************************

groupManagerGroupEventGroupHasFailedInActive


Description:
------------
handle the event of group failure in the state of 'active'


Input: 
-----
groupManagerGroupDataBaseEntry - pointer to the entry in the group manager group 
database	
parameters - the address of the message that contains all the parameters
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerGroupEventGroupHasFailedInActive(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters)
{ 	
	groupManagerGroupDataBaseEntry->groupFailed = TRUE;
	groupManagerGroupEventRemoveGroupInActive(groupManagerGroupDataBaseEntry, parameters);
}

/**********************************************************************************

groupManagerGroupEventLockCfmInRemovingWaitingForLockAndLaCfm


Description:
------------
handle the event of lock confirmation in the state of 'removing - waiting for lock
and link adaptation confirmations'


Input: 
-----
groupManagerGroupDataBaseEntry - pointer to the entry in the group manager group 
database	
parameters - the address of the message that contains all the parameters
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerGroupEventLockCfmInRemovingWaitingForLockAndLaCfm(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters)
{ 	
	UNUSED_PARAM(parameters);
	groupManagerChangeGroupState(groupManagerGroupDataBaseEntry, GROUP_MANAGER_GROUP_STATE_REMOVING_WAITING_FOR_LINK_ADAPTATION_CONFIRMATION);
	groupManagerHandleNextGroupWaitingForLock(groupManagerGroupDataBaseEntry->groupIndex);
}


/**********************************************************************************

groupManagerGroupEventLockConfirmationInRemovingWaitingForLockConfirmation


Description:
------------
handle the event of lock confirmation in the state of 'removing - waiting for lock
confirmation'


Input: 
-----
groupManagerGroupDataBaseEntry - pointer to the entry in the group manager group 
database	
parameters - the address of the message that contains all the parameters
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerGroupEventLockConfirmationInRemovingWaitingForLockConfirmation(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters)
{ 	
	UNUSED_PARAM(parameters);
	groupManagerFinishGroupRemovalProcess(groupManagerGroupDataBaseEntry);
	groupManagerHandleNextGroupWaitingForLock(groupManagerGroupDataBaseEntry->groupIndex);
}


/**********************************************************************************

groupManagerGroupEventToStateAssert


Description:
------------
create assert in case of illegal event

Input: 
-----
groupManagerGroupDataBaseEntry - pointer to the entry in the group manager group 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerGroupEventToStateAssert(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters)
{
	UNUSED_PARAM(parameters);
	UNUSED_PARAM(groupManagerGroupDataBaseEntry);	
#ifdef GROUP_MANAGER_DEBUG								
	ILOG0_DD("Assert group %d state: %d", groupManagerGroupDataBaseEntry->groupIndex, groupManagerGroupDataBaseEntry->state);
#endif
	FATAL("Assert in group manager group event");
}

/**********************************************************************************

groupManagerGroupEventToStateIgnore


Description:
------------
ignore event in a specific state


Input: 
-----
groupManagerGroupDataBaseEntry - pointer to the entry in the group manager group 
database	
parameters - not used	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerGroupEventToStateIgnore(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters)
{
	UNUSED_PARAM(parameters);
	UNUSED_PARAM(groupManagerGroupDataBaseEntry);	
#ifdef GROUP_MANAGER_DEBUG								
	ILOG0_DD("Ignore event group %d state: %d", groupManagerGroupDataBaseEntry->groupIndex, groupManagerGroupDataBaseEntry->state);
#endif
}

/**********************************************************************************

groupManagerRunGroupStateMachine 


Description:
------------
run the group manager group state machine

Input: 
-----
groupIndex - a pointer to the group entry
event - the event that occured
parameters - the address of the needed parameters

	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerRunGroupStateMachine(uint8 groupIndex, GroupManagerGroupEvent_e event, void *parameters)
{
	GroupManagerGroupEventToStateFunction_e functionIndex = GROUP_MANAGER_GROUP_EVENT_TO_STATE_FIRST;
	GroupManagerGroupState_e state = GROUP_MANAGER_GROUP_STATE_IDLE; 
   
	state = GroupManagerGroupDb[groupIndex].state;
    functionIndex = GroupManagerGroupStateMachine[state][event];
#ifdef GROUP_MANAGER_DEBUG								
	ILOG0_DDD("groupManagerRunGroupStateMachine group %d state: %d event %d", groupIndex, state, event);
#endif
	GroupManagerGroupEventToStateFunctionsTable[functionIndex](&GroupManagerGroupDb[groupIndex], parameters);
}


/**********************************************************************************
groupManagerChangeGlobalState


Description:
------------
change the global state

newState - the new state to change to
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerChangeGlobalState(GroupManagerGlobalState_e newState)
{
#ifdef GROUP_MANAGER_DEBUG								
    ILOG0_DD("groupManagerChangeGlobalState: current state: %d, new state %d", GroupManagerGlobalParametersDb.globalState , newState);
#endif

	GroupManagerGlobalParametersDb.globalState = newState;
}


/**********************************************************************************

groupManagerChangeGroupState 


Description:
------------
change the state of a station

Input: 
-----
groupManagerGroupDataBaseEntry - a pointer to the group entry
newState - the new state to change to
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerChangeGroupState(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, GroupManagerGroupState_e newState)
{
#ifdef GROUP_MANAGER_DEBUG								
    ILOG0_DDD("groupManagerChangeGroupState: Change group %d state: old state %d new state %d", groupManagerGroupDataBaseEntry->groupIndex, groupManagerGroupDataBaseEntry->state, newState);
#endif
	groupManagerGroupDataBaseEntry->state = newState;

	groupDbStatistics.groupInfoDb[groupManagerGroupDataBaseEntry->groupIndex][STATISTICS_MU_GROUP_STATE_INDEX] = newState;
}

/**********************************************************************************

groupManagerSendRemoveGroupMessage 


Description:
------------
send remove group message to the link adaptation

Input: 
-----
groupIndex - the group index
	
		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
static void groupManagerSendRemoveGroupMessage(uint8 groupIndex)
{
	LaRemoveGroupParams_t *removeGroupParameters = NULL;
	K_MSG *linkAdaptationMessage	= NULL;
	uint8 vapId = 0;

	linkAdaptationMessage = OSAL_GET_MESSAGE( sizeof(LaRemoveGroupParams_t));
	removeGroupParameters = ((LaRemoveGroupParams_t *)linkAdaptationMessage->abData);
	removeGroupParameters->groupIndex = groupIndex;
	MEMCPY(removeGroupParameters->participatingStations, GroupManagerGroupDb[groupIndex].participatingStations, sizeof(removeGroupParameters->participatingStations));
	StaDb_GetVapId(removeGroupParameters->participatingStations[0], &vapId);
#ifdef GROUP_MANAGER_DEBUG								
	ILOG0_D("groupManagerSendRemoveGroupMessage: groupIndex %d", groupIndex);
#endif
	OSAL_SEND_MESSAGE(LINK_ADAPTATION_REMOVE_GROUP, TASK_LINK_ADAPTATION, linkAdaptationMessage, vapId);
}


/**********************************************************************************

groupManagerSendAddGroupMessage 


Description:
------------
send add group message to the link adaptation

Input: 
-----
groupIndex - the group index
	
		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
static void groupManagerSendAddGroupMessage(uint8 groupIndex)
{
	LaAddGroupParams_t *addGroupParameters 	= NULL;
	K_MSG *linkAdaptationMessage			= NULL;
	uint8 i = 0;
	uint8 vapId = 0;
	GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry = &GroupManagerGroupDb[groupIndex];

	StaDb_GetVapId(groupManagerGroupDataBaseEntry->participatingStations[0], &vapId);
	
	linkAdaptationMessage = OSAL_GET_MESSAGE( sizeof(LaAddGroupParams_t));
	addGroupParameters = ((LaAddGroupParams_t *)linkAdaptationMessage->abData);
	addGroupParameters->groupIndex = groupIndex;
	addGroupParameters->nssPerUsp = groupManagerGroupDataBaseEntry->nssBitmap;
	for( i = 0; i < GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP; i++)
	{
		addGroupParameters->participatingStations[i] = groupManagerGroupDataBaseEntry->participatingStations[i];
	}
#ifdef GROUP_MANAGER_DEBUG								
	ILOG0_V("groupManagerSendAddGroupMessage: send add group to LA");
#endif
	/* send add group request*/
	OSAL_SEND_MESSAGE(LINK_ADAPTATION_ADD_GROUP, TASK_LINK_ADAPTATION, linkAdaptationMessage, vapId);
}

/**********************************************************************************

groupManagerLockGroup 


Description:
------------
send lock request if no requests are pending otherwise set bit in groups waiting for
lock bitmap


Input: 
-----
groupIndex - the group index to lock
	
		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
static void groupManagerLockGroup(uint8 groupIndex)
{
	RequesterLockParams_t lockRequestParameters; 

    if(!GroupManagerGlobalParametersDb.isLockPending)
    {
#ifdef GROUP_MANAGER_DEBUG									
    	ILOG0_D("groupManagerLockGroup: sending lock request to Locker on group %d", groupIndex);
#endif
    	memset(&lockRequestParameters, 0, sizeof(RequesterLockParams_t));

	 	GroupManagerGlobalParametersDb.isLockPending = TRUE;
		lockRequestParameters.returnMsg = GROUP_MANAGER_GROUP_LOCK_CONFIRM; 
		lockRequestParameters.returnTask = TASK_GROUP_MANAGER; 
		Locker_LockGroup(groupIndex ,&lockRequestParameters); 
    }
	else
	{
		GroupManagerGlobalParametersDb.waitingGroupsForLockBitmap |= (0x1 << groupIndex);
#ifdef GROUP_MANAGER_DEBUG									
		ILOG0_DD("groupManagerLockGroup: lock was already sent. Adding group %d to waiting list. waitingLock bitmap 0x%x", 
				groupIndex, GroupManagerGlobalParametersDb.waitingGroupsForLockBitmap);
#endif
	}
}

/**********************************************************************************

groupManagerGroupRemoveGroupInProbingNotifying


Description:
------------
start removal process in the state of 'probing notifying' of the group.
The group is in notification process before probing


Input: 
-----
groupManagerGroupDataBaseEntry - pointer to the entry in the group manager group 
database	
parameters - station index (if used)
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerGroupRemoveGroupInProbingNotifying(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry, void *parameters)
{ 	
	uint8 indexInArray = 0;
	StaId stationIndex = INVALID_STA_INDEX;
	GidmManagerStopTxReq_t stopTxParams;
	
	memset(&stopTxParams, 0, sizeof(GidmManagerStopTxReq_t));

	if(NULL != parameters)
	{
		stationIndex = *(StaId *)parameters;
	}

	/*change group state to GROUP_STATE_REMOVING_WAITING_FOR_NOTIFICATIONS*/
	groupManagerChangeGroupState(groupManagerGroupDataBaseEntry, GROUP_MANAGER_GROUP_STATE_REMOVING_WAITING_FOR_NOTIFICATIONS);
	
	/* go over all SIDs in this group */
	for(indexInArray = 0; indexInArray < groupManagerGroupDataBaseEntry->numberOfParticipatingStations; indexInArray++)
	{
		/* found index of station which is being removed from the group. 
		 If requested notification through GIDM but notification wasn't received yet 
		 send stopTx request to GIDM Mng with reason = GIDM_STOP_REASON_REMOVE_STA. This will indicate to GIDM that
		 it doesn't need to lock station QUEUE, this will be done through BSS API which removes the station*/
		if((INVALID_STA_INDEX != stationIndex) && (groupManagerGroupDataBaseEntry->participatingStations[indexInArray] == stationIndex))
		{
			if(GROUP_MANAGER_NOTIFICATION_NOT_RECEIVED == groupManagerGroupDataBaseEntry->notificationsArray[indexInArray])
			{
#ifdef GROUP_MANAGER_DEBUG											
				ILOG0_D("groupManagerGroupRemoveGroupInProbingNotifying: station %d is removed during probing, send stopTx to GIDM Mng", stationIndex);
#endif
				stopTxParams.staIndex = stationIndex;
				stopTxParams.stopReason = GIDM_STOP_REASON_REMOVE_STA;
				groupManagerGroupDataBaseEntry->notificationsArray[indexInArray] = GROUP_MANAGER_NOTIFICATION_CANCELED;
				GidmManagerStopTxReq(&stopTxParams);	
			}
		}	
		/* for any other station in this group for which notification was sent but not received cfm send
		stopTx request to GIDM Mng with reason = GIDM_STOP_REASON_TIMEOUT. This will indicate to GIDM that it has to lock
		stations's queue and try to extract notification message for station*/
		else if(GROUP_MANAGER_NOTIFICATION_NOT_RECEIVED == groupManagerGroupDataBaseEntry->notificationsArray[indexInArray])
		{
#ifdef GROUP_MANAGER_DEBUG										
			ILOG0_D("groupManagerGroupRemoveGroupInProbingNotifying: send stopTx on station %d to GIDM Mng, notification cancelled", 
				groupManagerGroupDataBaseEntry->participatingStations[indexInArray]);
#endif
			stopTxParams.staIndex = groupManagerGroupDataBaseEntry->participatingStations[indexInArray];
			stopTxParams.stopReason = GIDM_STOP_REASON_TIMEOUT;
			// NOTE: the order of the following 2 rows must not be changed. notificationsArray value must be set before invocation
			// of GidmManagerStopTxReq that also changes the values in this array
			groupManagerGroupDataBaseEntry->notificationsArray[indexInArray] = GROUP_MANAGER_NOTIFICATION_CANCELED;
			GidmManagerStopTxReq(&stopTxParams);
		}	
		/* NOTE: stations which notificationArray status is already GROUP_MANAGER_NOTIFICATION_CANCELED are not handled because either
		notification for these stations was already cancelled either notification cfm arrived and therefore their status is SUCCESS or FAIL
		so there is nothing to send to GIDM mng*/

	}
#ifdef GROUP_MANAGER_DEBUG							
	ILOG0_D("groupManagerGroupRemoveGroupInProbingNotifying: station index received %d",
		stationIndex);
	ILOG0_DDDD("groupManagerGroupRemoveGroupInProbingNotifying: notification status of the group (%d, %d, %d, %d)",
			groupManagerGroupDataBaseEntry->notificationsArray[0], groupManagerGroupDataBaseEntry->notificationsArray[1],
			groupManagerGroupDataBaseEntry->notificationsArray[2], groupManagerGroupDataBaseEntry->notificationsArray[3]);
#endif
	
	/* set removing group bitmap for this group*/
	GroupManagerGlobalParametersDb.removingGroupsBitmap |= (0x1 << groupManagerGroupDataBaseEntry->groupIndex);
}

/**********************************************************************************

groupManagerHandleNextGroupWaitingForLock 


Description:
------------
choose next group to lock in a round robin method 


Input: 
-----
groupIndex - the last group index that was locked
	
		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
static void groupManagerHandleNextGroupWaitingForLock(uint8 groupIndex)
{
	uint32 tempBitmap = 0;
	uint8 nextGroupToLock = GROUP_MANAGER_INVALID_GROUP_INDEX;
	uint16 mask = 0;

	// this function is invoked in case GroupLockCfm arrived in group state waiting_for_lock_cfm/ waiting_for_lock_cfm&LaCfm
	// therefore flag isLockPending can be set to FALSe because Lock was just confirmed
	GroupManagerGlobalParametersDb.isLockPending = FALSE;
	if(GroupManagerGlobalParametersDb.waitingGroupsForLockBitmap)
	{
		/* Take only upper bits from index */
		mask = GROUP_MANAGER_ALL_GROUPS_MASK << groupIndex;
		tempBitmap = GroupManagerGlobalParametersDb.waitingGroupsForLockBitmap & mask;
		if(0 == tempBitmap)
		{
			/* No group was found in upper bits - take all bits */
			tempBitmap = GroupManagerGlobalParametersDb.waitingGroupsForLockBitmap;
		}
		nextGroupToLock = Utils_FindFirstSet(tempBitmap);
		GroupManagerGlobalParametersDb.waitingGroupsForLockBitmap &= ~(0x1 << nextGroupToLock);
#ifdef GROUP_MANAGER_DEBUG									
		ILOG0_D("groupManagerHandleNextGroupWaitingForLock: group  %d", nextGroupToLock);
#endif
		groupManagerLockGroup(nextGroupToLock);
	}
}

/**********************************************************************************

groupManagerFinishGroupRemovalProcess 


Description:
------------
Finish group removal process (no more events are needed) 


Input: 
-----
groupManagerGroupDataBaseEntry - pointer to the entry in the group manager group 
database

		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
static void groupManagerFinishGroupRemovalProcess(GroupManagerGroupParameters_t *groupManagerGroupDataBaseEntry)
{
	uint8 groupIndex = 0;
	uint8 indexInArray = 0;

	groupIndex = groupManagerGroupDataBaseEntry->groupIndex;
#ifdef GROUP_MANAGER_DEBUG							
	ILOG0_DDDD("groupManagerFinishGroupRemovalProcess: group idx %d, removingGroupsBitmap 0x%x, occupiedGroupsBitmap 0x%x, numberOfOccupiedGroups %d",
			groupIndex, GroupManagerGlobalParametersDb.removingGroupsBitmap, GroupManagerGlobalParametersDb.occupiedGroupsBitmap, GroupManagerGlobalParametersDb.numberOfOccupiedGroups);
	ILOG0_DD("groupManagerFinishGroupRemovalProcess: activeGroupsBitmap 0x%x, probingGroupsBitmap 0x%x",
		GroupManagerGlobalParametersDb.activeGroupsBitmap, GroupManagerGlobalParametersDb.probingGroupsBitmap);
#endif
	
    /* Clear bit in all bitmaps besides notifyingGroupsBitmap - all states besides notifying can enter this state */
	GroupManagerGlobalParametersDb.removingGroupsBitmap &= ~(0x1 << groupIndex);
	GroupManagerGlobalParametersDb.occupiedGroupsBitmap &= ~(0x1 << groupIndex);
	GroupManagerGlobalParametersDb.numberOfOccupiedGroups--;
	
	/* if this was active group - turn of the bit of the group*/
    if(GroupManagerGlobalParametersDb.activeGroupsBitmap & (0x1 << groupIndex))
    {
		GroupManagerGlobalParametersDb.activeGroupsBitmap &= ~(0x1 << groupIndex);
		GroupManagerGlobalParametersDb.numberOfActiveGroups--;
		GroupManagerGlobalParametersDb.numberOfRemovingGroupsInActive--;
		ILOG0_DDD("groupManagerFinishGroupRemovalProcess: group was in active state. After clear bits: activeGroupsBitmap 0x%x, numberOfActiveGroups %d, num of removing groups in active %d",
			GroupManagerGlobalParametersDb.activeGroupsBitmap, GroupManagerGlobalParametersDb.numberOfActiveGroups, GroupManagerGlobalParametersDb.numberOfRemovingGroupsInActive);
    }
	/* if was probing group*/
	else if(GroupManagerGlobalParametersDb.probingGroupsBitmap & (0x1 << groupIndex)) 
	{
		GroupManagerGlobalParametersDb.probingGroupsBitmap &= ~(0x1 << groupIndex);
		GroupManagerGlobalParametersDb.numberOfProbingGroups--;
#ifdef GROUP_MANAGER_DEBUG									
		ILOG0_DD("groupManagerFinishGroupRemovalProcess: group was in probing state. After clear bits: probingGroupsBitmap 0x%x, numberOfProbingGroups %d",
			GroupManagerGlobalParametersDb.probingGroupsBitmap, GroupManagerGlobalParametersDb.numberOfProbingGroups);
#endif
	}

	//TBD REVIEW THIS OPTIONAL FEATURE
	// for member of the group for which the removed group is primary - turn off primary bit in Selector
/*	for(indexInArray = 0; indexInArray < groupManagerGroupDataBaseEntry->numberOfParticipatingStations; indexInArray++)
	{
		if(GroupManagerStationDb[groupManagerGroupDataBaseEntry->participatingStations[indexInArray]].primaryGroup == groupIndex)
		{
			groupManagerConfigureStationForSelector(groupManagerGroupDataBaseEntry->participatingStations[indexInArray],FALSE,0);
		}
	}
*/	
	/* if group removed due to group failure , punish all stations in the group, by running 'groupManagerStationEventRemoveGroupDueToStation' function 
	on each station*/
	if(groupManagerGroupDataBaseEntry->groupFailed)
	{
#ifdef GROUP_MANAGER_DEBUG									
		ILOG0_V("groupManagerFinishGroupRemovalProcess: group failed");
#endif
		for(indexInArray = 0; indexInArray < groupManagerGroupDataBaseEntry->numberOfParticipatingStations; indexInArray++)
		{	
			groupManagerStationEventRemoveGroupDueToStation(groupManagerGroupDataBaseEntry->participatingStations[indexInArray], groupIndex);
		}
		groupManagerGroupDataBaseEntry->groupFailed = FALSE;
	}
	/* if group removed because of another reason no need to punish it's members*/
	else
	{
#ifdef GROUP_MANAGER_DEBUG									
		ILOG0_V("groupManagerFinishGroupRemovalProcess: group not failed");
#endif
		for(indexInArray = 0; indexInArray < groupManagerGroupDataBaseEntry->numberOfParticipatingStations; indexInArray++)
		{	
			groupManagerStationEventRemoveGroup(groupManagerGroupDataBaseEntry->participatingStations[indexInArray], groupIndex);
		}
	}

	groupManagerChangeGroupState(groupManagerGroupDataBaseEntry, GROUP_MANAGER_GROUP_STATE_IDLE);
	GroupManagerResetGroupMembersStatistic(groupIndex);
	/* check if disable was required*/
	groupManagerFinishGlobalRemovalProcess();
	groupManagerTryToFindNewGroupForProbing(FALSE);
}

/**********************************************************************************

groupManagerFinishGlobalRemovalProcess 


Description:
------------
Finish global removal process - check if the group manager is in disabling process 


Input: 
-----
groupManagerGroupDataBaseEntry - pointer to the entry in the group manager group 
database

		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
static void groupManagerFinishGlobalRemovalProcess(void)
{
	if(GROUP_MANAGER_GLOBAL_STATE_DISABLING_WAITING_FOR_ALL_GROUP_REMOVAL == GroupManagerGlobalParametersDb.globalState)
	{
		// if we have no groups at all (no notifying, no probing and no active) can move to IDLE state
		if(0 == GroupManagerGlobalParametersDb.occupiedGroupsBitmap)
		{
			groupManagerChangeGlobalState(GROUP_MANAGER_GLOBAL_STATE_IDLE);
			//send CFM to HDK
#ifdef GROUP_MANAGER_DEBUG										
			ILOG0_V("groupManagerFinishGlobalRemovalProcess: sending multi user operation setting cfm");
#endif
			groupManagerSetMultiUserOperationCfm();
		}
	}
	else if(GROUP_MANAGER_GLOBAL_STATE_CREATE_STATIC_GROUP_WAITING_FOR_ALL_GROUP_REMOVAL == GroupManagerGlobalParametersDb.globalState)
	{
		if(0 == GroupManagerGlobalParametersDb.occupiedGroupsBitmap)
		{
			groupManagerChangeGlobalState(GROUP_MANAGER_GLOBAL_STATE_CREATING_STATIC_GROUP);
#ifdef GROUP_MANAGER_DEBUG										
			ILOG0_V("groupManagerFinishGlobalRemovalProcess: starting static group generation");
#endif
			groupManagerTriggerStaticGroupGeneration();
		}
	}
}

/**********************************************************************************

groupManagerTryToFindNewGroupForProbing 


Description:
------------
Try to find new group for probing from the virtual groups


Input: 
-----

		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
static void groupManagerTryToFindNewGroupForProbing(bool isStaticGroup)
{
	uint32 validVirtualGroupsBitmap = 0;
	uint32 tempBitmap = 0;
	uint8 virtualGroupIndex = 0;
	uint8 indexInArray = 0;
	uint8 isMemberInNotifying = FALSE;
	uint8 isTherePrimaryForGroup = FALSE;
	StaId stationIndex = 0;
	uint16 notifyingBitmap = 0;
	uint8 bestVirtualGroupIndex = 0;
	uint8 bestMark = 0;
	uint8 newGroupIndex = 0;
#ifdef GROUP_MANAGER_DEBUG							
	ILOG0_DD("groupManagerTryToFindNewGroupForProbing: probing groups %d, occupied groups %d", GroupManagerGlobalParametersDb.numberOfProbingGroups, GroupManagerGlobalParametersDb.numberOfOccupiedGroups);
#endif
	/* new group can be added to probing only if number of such groups didn't reach it's threshold*/
	if(GroupManagerGlobalParametersDb.muConfigParams.maxNumProbingGroups > GroupManagerGlobalParametersDb.numberOfProbingGroups)
	{
		// and if total number of operating groups (notify, probing, active) is not above max DB size
		if(GROUP_MANAGER_MAX_NUMBER_OF_GROUPS != GroupManagerGlobalParametersDb.numberOfOccupiedGroups)
		{
			/* if there are some candidates in virtual table*/
			if(GroupManagerGlobalParametersDb.virtualGroupsBitmap)
			{
				tempBitmap = GroupManagerGlobalParametersDb.virtualGroupsBitmap;
#ifdef GROUP_MANAGER_DEBUG											
				ILOG0_D("groupManagerTryToFindNewGroupForProbing: virtuals db bitmap 0x%x", tempBitmap);
#endif
				while(tempBitmap)
				{
					virtualGroupIndex = Utils_FindFirstSetAndClear(&tempBitmap);
					
					isMemberInNotifying = FALSE; /* if one of members of the group in Notify stage*/
					isTherePrimaryForGroup = FALSE;
					
					/* Check if virtual group is valid for probing */
					for(indexInArray = 0; indexInArray < GroupManagerVirtualGroupDatabase[virtualGroupIndex].numberOfParticipatingStations; indexInArray++)
					{
						stationIndex = GroupManagerVirtualGroupDatabase[virtualGroupIndex].participatingStations[indexInArray];
						/* check if the member of this group already pariticpates in probing of the group which is in notifying stage*/
						/*We do not allow to a member to participate in more than one Notification process - because of limitation in GIDM design*/
						notifyingBitmap = GroupManagerStationDb[stationIndex].groupsBitmap & GroupManagerGlobalParametersDb.notifyingGroupsBitmap;
						if(0 != notifyingBitmap)
						{
							isMemberInNotifying = TRUE;
							// no need to continue this group can't be chosen for probing
							break;
						}
						if((GROUP_MANAGER_STATION_STATE_ACTIVE == GroupManagerStationDb[stationIndex].state) ||
						   (GROUP_MANAGER_STATION_STATE_SINGLE_USER == GroupManagerStationDb[stationIndex].state)||
						   /* if  member's state is probing but it is not a primary for the other probing group*/
						   ((GROUP_MANAGER_STATION_STATE_PROBING == GroupManagerStationDb[stationIndex].state) && (GROUP_MANAGER_INVALID_GROUP_INDEX == GroupManagerStationDb[stationIndex].tempPrimaryGroup)))
						{
							// can be primary for this probing group
							isTherePrimaryForGroup = TRUE;
						}	
					}
					/* only if both conditions apply we can consider this group for probing*/
					if(!isMemberInNotifying && isTherePrimaryForGroup)
					{
						/* There is at least one station for primary member and no member is in notifying */
						validVirtualGroupsBitmap |= (0x1 << virtualGroupIndex);
					}
					else
					{
#ifdef GROUP_MANAGER_DEBUG													
/*						ILOG0_DDDDDDDDD("groupManagerTryToFindNewGroupForProbing: virtual group %d not good for probing. states:%d- %d,%d- %d,%d- %d,%d- %d",
								virtualGroupIndex,
								GroupManagerVirtualGroupDatabase[virtualGroupIndex].participatingStations[0], GroupManagerStationDb[GroupManagerVirtualGroupDatabase[virtualGroupIndex].participatingStations[0]].state,
								GroupManagerVirtualGroupDatabase[virtualGroupIndex].participatingStations[1], GroupManagerStationDb[GroupManagerVirtualGroupDatabase[virtualGroupIndex].participatingStations[1]].state,
								GroupManagerVirtualGroupDatabase[virtualGroupIndex].participatingStations[2], GroupManagerStationDb[GroupManagerVirtualGroupDatabase[virtualGroupIndex].participatingStations[2]].state,
								GroupManagerVirtualGroupDatabase[virtualGroupIndex].participatingStations[3], GroupManagerStationDb[GroupManagerVirtualGroupDatabase[virtualGroupIndex].participatingStations[3]].state);*/
#endif
						// if function was invoked to trigger static group generation and it can't be done, delete static group from virtual groups
						if(isStaticGroup)
						{
							groupManagerRemoveVirtualGroup(GroupManagerGlobalParametersDb.staticVirtualGroupIndex);
							GroupManagerGlobalParametersDb.staticVirtualGroupIndex = GROUP_MANAGER_INVALID_GROUP_INDEX;
							if(isMemberInNotifying)
							{
								SERIAL_TRACE("Static group can't be created - one in notify stage.",0,0,0);
								ILOG0_V("[groupManagerTryToFindNewGroupForProbing], Static group can't be created - one in notify stage");
							}
							if(isTherePrimaryForGroup == 0)
							{
								SERIAL_TRACE("Static group can't be created - available primary can't be found",0,0,0);
								ILOG0_V("[groupManagerTryToFindNewGroupForProbing], Static group can't be created - available primary can't be found");
							}
							
						}
					}
					
				}
				/* if found at least one group*/
				if(validVirtualGroupsBitmap)
				{
					ILOG2_D("groupManagerTryToFindNewGroupForProbing: virtual groups bitmap candidates for probing 0x%x", validVirtualGroupsBitmap);
					/* Find virtual group with best mark (if marks are equal compare TSF) */
					bestVirtualGroupIndex = Utils_FindFirstSetAndClear(&validVirtualGroupsBitmap);
					bestMark = GroupManagerVirtualGroupDatabase[bestVirtualGroupIndex].groupMark;
						
					while(validVirtualGroupsBitmap)
					{
						virtualGroupIndex = Utils_FindFirstSetAndClear(&validVirtualGroupsBitmap);
						if((GroupManagerVirtualGroupDatabase[virtualGroupIndex].groupMark > bestMark) ||
						   ((GroupManagerVirtualGroupDatabase[virtualGroupIndex].groupMark == bestMark) && (GroupManagerVirtualGroupDatabase[virtualGroupIndex].timeStamp > GroupManagerVirtualGroupDatabase[bestVirtualGroupIndex].timeStamp)))
						{
							bestVirtualGroupIndex = virtualGroupIndex;
							bestMark = GroupManagerVirtualGroupDatabase[bestVirtualGroupIndex].groupMark;
						}
					}
#ifdef GROUP_MANAGER_DEBUG							
					ILOG0_D("groupManagerTryToFindNewGroupForProbing: found best virtual group index for probing %d", bestVirtualGroupIndex);
#endif
					/* the chosen group bit is set to 0, because it is moved to probing state*/
					GroupManagerGlobalParametersDb.virtualGroupsBitmap &= ~(0x1 << bestVirtualGroupIndex);
					/* find not occupied slot in DB*/
					tempBitmap = (~GroupManagerGlobalParametersDb.occupiedGroupsBitmap) & GROUP_MANAGER_ALL_GROUPS_MASK;
					/* find first free index in DB among all not occupied*/
					newGroupIndex = Utils_CountTrailingZeros(tempBitmap);
					// store the index of static group if trying to start probing on this group
					if((isStaticGroup == TRUE) && (GroupManagerGlobalParametersDb.staticVirtualGroupIndex == bestVirtualGroupIndex))
					{
						GroupManagerGlobalParametersDb.staticOccupiedGroupIndex = newGroupIndex;
						GroupManagerGlobalParametersDb.staticVirtualGroupIndex = GROUP_MANAGER_INVALID_GROUP_INDEX;
					}
					/* Start notify process of the group*/
					groupManagerGroupEventStartNotifying(newGroupIndex, bestVirtualGroupIndex);	
				}
			}
		}
	}
}

/**********************************************************************************

groupManagerConfigureHwGroupDbs 


Description:
Configuration of group DB in SHRAM before beginning of MAC probing on the group

Input: index of group in DB
-----

		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/

static void groupManagerConfigureHwGroupDbs(uint8 groupIndex)
{
	uint8 i = 0;
	uint8 bwIndex = 0;
	uint8 shift = 0; 
	// matrix of calculated NSTC values extracted from PHY params as provided in combination vector after training. Same value will be used for 20/40/80 Mhz
	uint8 macNssPerUsp[GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP] = {0,0,0,0};
	// matrix of calculated MAC MU MCS value per each supported band. Index 0 for 20 Mhz, index 1 for 40 Mhz, index 80 for 80 Mhz
#ifdef ENET_INC_ARCH_WAVE600
	int8 macMcsPerBwPerUsp[GROUP_MANAGER_NUM_OF_SUPPORTED_BW][GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP] = {{0}, {0}, {0}, {0}};
#else
	int8 macMcsPerBwPerUsp[GROUP_MANAGER_NUM_OF_SUPPORTED_BW][GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP] = {{0}, {0}, {0}};
#endif
	GroupManagerGroupParameters_t *groupDbEntry = &GroupManagerGroupDb[groupIndex];
	int32 nstsPhyCalibFactor;
	uint8* wpRatePerBwPtr = NULL;
	uint8 wpRateForBw = 0;


	for(i=0; i < groupDbEntry->numberOfParticipatingStations; i ++)
	{
		// fill in nss value per user position of the station in a group
		shift = i << GROUP_MANAGER_NSS_SIZE_SHIFT;
		macNssPerUsp[i] = ((groupDbEntry->nssBitmap >> shift) & GROUP_MANAGER_NSS_MASK); // nss 0 - for value 1, nss 1 - for value 2 , etc	
		// calculate WP mcs value per USP per BW
		for (bwIndex=0; bwIndex < GROUP_MANAGER_NUM_OF_SUPPORTED_BW; bwIndex++)
		{
#ifdef ENET_INC_ARCH_WAVE600

#ifdef PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400
			wpRatePerBwPtr = (uint8*)&(StaDbHwEntries[groupDbEntry->participatingStations[i]].wp.word6) + bwIndex;
#else
			wpRatePerBwPtr = (uint8*)&(StaDbHwEntries[groupDbEntry->participatingStations[i]].wp.word4) + bwIndex;
#endif
			wpRateForBw = *wpRatePerBwPtr;

#else
			wpRatePerBwPtr = (uint8*)&(StaDbHwEntries[groupDbEntry->participatingStations[i]].common.word5 /*tcr320MhzData*/) +
									bwIndex * (FM_STRUCT_OFFSET(StaDbCommon_t,word7/*tcr340MhzData*/) - FM_STRUCT_OFFSET(StaDbCommon_t,word5/*tcr320MhzData*/));
			wpRateForBw = ((Tcr3_t*)wpRatePerBwPtr)->wpRate;
#endif
			nstsPhyCalibFactor = 0;

			if((groupDbEntry->singleNstsPhyRates[i]!=GROUP_MANAGER_INVALID_RATE_VALUE) && (EXTRACT_NSS_FROM_VHT_HE_RATE(wpRateForBw) == SPATIAL_STREAM_1))
			{
				nstsPhyCalibFactor = EXTRACT_MCS_FROM_VHT_HE_RATE(wpRateForBw) - groupDbEntry->singleNstsPhyRates[i];
			}
			else if((groupDbEntry->doubleNstsPhyRates[i]!=GROUP_MANAGER_INVALID_RATE_VALUE) && (EXTRACT_NSS_FROM_VHT_HE_RATE(wpRateForBw) == SPATIAL_STREAM_2))
			{
			 	nstsPhyCalibFactor = EXTRACT_MCS_FROM_VHT_HE_RATE(wpRateForBw) - groupDbEntry->doubleNstsPhyRates[i];
			}
			
			// in case fixed rate is used do not apply formula, use the supplied rate as is
			if(GroupManagerGlobalParametersDb.fixedRateFlag == TRUE)
			{
				macMcsPerBwPerUsp[bwIndex][i] = groupDbEntry->mcs[i];
			}
			else
			{
				if(nstsPhyCalibFactor >= 0)
				{
					macMcsPerBwPerUsp[bwIndex][i] = groupDbEntry->mcs[i] + MAX(2,nstsPhyCalibFactor) - GROUP_MANAGER_MCS_DEGRADATION_FACTOR(groupDbEntry->mcs[i]);
				}
				else
				{
					macMcsPerBwPerUsp[bwIndex][i] = groupDbEntry->mcs[i] + nstsPhyCalibFactor - GROUP_MANAGER_MCS_DEGRADATION_FACTOR(groupDbEntry->mcs[i]);
				}
			}

			// THIS is W/A for the formula above which sometimes provides illegal MCS values
			if(macMcsPerBwPerUsp[bwIndex][i] > MCS_9)
			{
/*				ILOG2_DDDDDDD("groupManagerConfigureHwGroupDbs: i %d, bw %d, nss 0x%x, mcs 0x%x, calibrator %d, SU mcs 0x%x, wpRate 0x%x",i, bwIndex, macNssPerUsp[i], 
					macMcsPerBwPerUsp[bwIndex][i], nstsPhyCalibFactor, groupDbEntry->mcs[i], pTcr3ForCurrentBw->wpRate);*/
				macMcsPerBwPerUsp[bwIndex][i] = MCS_9;
			}
			else if(macMcsPerBwPerUsp[bwIndex][i] < 0) 
			{
/*				ILOG2_DDDDDDD("groupManagerConfigureHwGroupDbs: i %d, bw %d, nss 0x%x, mcs 0x%x, calibrator %d, SU mcs 0x%x, wpRate 0x%x",i, bwIndex, macNssPerUsp[i], 
					macMcsPerBwPerUsp[bwIndex][i], nstsPhyCalibFactor, groupDbEntry->mcs[i], pTcr3ForCurrentBw->wpRate);*/
				macMcsPerBwPerUsp[bwIndex][i] = MCS_1;
			}
		}

	}

	/* configure group params in HW DB*/
	GroupDb_FillGroupParams(groupIndex, groupDbEntry->numberOfParticipatingStations, groupDbEntry->participatingStations, macNssPerUsp, macMcsPerBwPerUsp);

	
}

/**********************************************************************************

groupManagerRemoveAllActiveGroups 


Description:
------------
start removal process of all active groups 


Input: 
-----


		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerRemoveAllActiveGroups(void)
{
	uint16 activeGroupsBitmap = 0;
	uint8 activeGroupIndex = 0;

	activeGroupsBitmap = GroupManagerGlobalParametersDb.occupiedGroupsBitmap;
	/* trigger remove group event on all active groups*/
	while(activeGroupsBitmap)
	{
		activeGroupIndex = Utils_CountTrailingZeros(activeGroupsBitmap);
		activeGroupsBitmap &= ~(0x1 << activeGroupIndex);
		groupManagerGroupEventRemoveGroup(activeGroupIndex, NULL);
	}
}

/**********************************************************************************

groupManagerRemoveAllVirtualGroups 


Description:
------------
remove all virtual groups 


Input: 
-----


		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerRemoveAllVirtualGroups(void)
{
	uint32 virtualGroupsBitmap = 0;
	uint8 virtualGroupIndex = 0;
	uint8 indexInArray = 0;
	StaId stationIndex = 0;

	virtualGroupsBitmap = GroupManagerGlobalParametersDb.virtualGroupsBitmap;
	while(virtualGroupsBitmap)
	{
		virtualGroupIndex = Utils_FindFirstSetAndClear(&virtualGroupsBitmap);
		for(indexInArray = 0; indexInArray < GroupManagerVirtualGroupDatabase[virtualGroupIndex].numberOfParticipatingStations; indexInArray++)
		{
			stationIndex = GroupManagerVirtualGroupDatabase[virtualGroupIndex].participatingStations[indexInArray];
			GroupManagerStationDb[stationIndex].virtualGroupsBitmap = 0;
		}
	}
	GroupManagerGlobalParametersDb.virtualGroupsBitmap = 0;
}


/**********************************************************************************

groupManagerRemoveVirtualGroup 


Description:
------------
remove virtual group from DB


Input: 
-----

virtualGroupIndex
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerRemoveVirtualGroup(uint8 virtualGroupIndex)
{
	uint8 indexInArray = 0;
	StaId stationIndex = 0;

	/* for all stations that participate in this group , turn of bitmap of this virtual group in it's station dB*/
	for(indexInArray = 0; indexInArray < GroupManagerVirtualGroupDatabase[virtualGroupIndex].numberOfParticipatingStations; indexInArray++)
	{
		stationIndex = GroupManagerVirtualGroupDatabase[virtualGroupIndex].participatingStations[indexInArray];
		GroupManagerStationDb[stationIndex].virtualGroupsBitmap &= ~(0x1<<virtualGroupIndex);
	}
	
	/* delete the group from global virtual groups bitmap*/
	GroupManagerGlobalParametersDb.virtualGroupsBitmap &= ~(0x1 << virtualGroupIndex);
}


/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

GroupManager_TaskEntry 


Description:
------------
the entry point of the group manager task

Input: 
-----
groupManagerMessage - pointer to the message to handle	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void GroupManager_TaskEntry(K_MSG *groupManagerMessage)
{
	/* Use common task switching and Table */ 
	vTaskDispatcher(groupManagerMessage, afpTaskTable, TASK_GROUP_MANAGER_START, TASK_GROUP_MANAGER_END);
}

/**********************************************************************************

GroupManager_StationIsNotInTraining 


Description:
------------
confirmation from the training that a specific station is not participating in the
current training

Input: 
-----
stationIndex - the station for which the confirmation was sent
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void GroupManager_StationIsNotInTraining(StaId stationIndex)
{ 
	groupManagerRunStationStateMachine(stationIndex, GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION, NULL);
}

/**********************************************************************************

GroupManager_TrainingWasStoppedConfirmation 


Description:
------------
confirmation from the training that the current training was stopped

Input: 
-----
stationBitmap - a pointer to a bitmap of stations which were waiting for confirmation 
from the training module
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void GroupManager_TrainingWasStoppedConfirmation(uint32 *stationBitmap) 
{ 	
	StaId stationIndex = 0;
	uint8 wordInBitmap = 0;
	uint32 tempBitmap = 0;
	uint8 indexOffset = 0;	
	
#ifdef GROUP_MANAGER_DEBUG							
	ILOG0_V("GroupManager_TrainingWasStoppedConfirmation");
#endif
	//for each word process all stations which bits are set to 1*/
    for(wordInBitmap = 0; wordInBitmap < GROUP_MANAGER_SIZE_OF_STATION_BITMAP_IN_WORDS; wordInBitmap++)
    {
		tempBitmap = stationBitmap[wordInBitmap];
		while(tempBitmap)
		{
			stationIndex = Utils_FindFirstSetAndClear(&tempBitmap);
			stationIndex += indexOffset;
			/* training cfm arrived for station -trigger handler based on station state*/
			groupManagerRunStationStateMachine(stationIndex, GROUP_MANAGER_STATION_EVENT_TRAINING_CONFIRMATION, NULL);
		}
		// move correctly to the next word
		indexOffset += GROUP_MANAGER_NUMBER_OF_STATIONS_BITS_IN_WORD;
    }
	
	if(GROUP_MANAGER_GLOBAL_STATE_DISABLING_WAITING_FOR_TRAINING == GroupManagerGlobalParametersDb.globalState)
	{
		if(0 == GroupManagerGlobalParametersDb.occupiedGroupsBitmap)
		{
			groupManagerChangeGlobalState(GROUP_MANAGER_GLOBAL_STATE_IDLE);
#ifdef GROUP_MANAGER_DEBUG										
			ILOG0_V("GroupManager_TrainingWasStoppedConfirmation: send multi user operation setting cfm");
#endif
			groupManagerSetMultiUserOperationCfm();
		}
		else
		{
			groupManagerChangeGlobalState(GROUP_MANAGER_GLOBAL_STATE_DISABLING_WAITING_FOR_ALL_GROUP_REMOVAL);
		}
		/*trigger training timer after training was stopped. this is done regardless of disable state because if there are 
		stations in HALT state and we might return from disabled state before their HALT timeout expired */
		OSAL_SET_TIMER_EXPLICIT(GROUP_MANAGER_TRAINING_TIMER_EXPIRED, OSAL_TIMERS_MS_TO_K_TICKS(GROUP_MANAGER_CALENDAR_WHEEL_TIMER_IN_MILLISECONDS), TASK_GROUP_MANAGER);
	}
	else if (GROUP_MANAGER_GLOBAL_STATE_CREATE_STATIC_GROUP_WAITING_FOR_TRAINING == GroupManagerGlobalParametersDb.globalState)
	{
		if(0 == GroupManagerGlobalParametersDb.occupiedGroupsBitmap)
		{
			groupManagerChangeGlobalState(GROUP_MANAGER_GLOBAL_STATE_CREATING_STATIC_GROUP);
#ifdef GROUP_MANAGER_DEBUG										
			ILOG0_V("GroupManager_TrainingWasStoppedConfirmation: Starting To create static group");
#endif
			groupManagerTriggerStaticGroupGeneration();
		}
		else
		{
			groupManagerChangeGlobalState(GROUP_MANAGER_GLOBAL_STATE_CREATE_STATIC_GROUP_WAITING_FOR_ALL_GROUP_REMOVAL);
		}
		/* Do not trigger timer at this point, will be triggered after static group will finish notification stage .
		   This is in order to make sure that training manager will provide training results only after group is created and
		   won't interfere*/
	}
	else if (GROUP_MANAGER_GLOBAL_STATE_ACTIVE == GroupManagerGlobalParametersDb.globalState)
	{
		/* this function can be invoked in global state = active when training manager returns CFM on StopTraining on HALTED station.
		in this case timer must be triggered again to enable trainign in teh future*/
		OSAL_SET_TIMER_EXPLICIT(GROUP_MANAGER_TRAINING_TIMER_EXPIRED, OSAL_TIMERS_MS_TO_K_TICKS(GROUP_MANAGER_CALENDAR_WHEEL_TIMER_IN_MILLISECONDS), TASK_GROUP_MANAGER);
	}
	else
	{
		FATAL("Assert in GroupManager_TrainingWasStoppedConfirmation");
	}
	
}

/**********************************************************************************

GroupManager_TrainingFailed 


Description:
------------
handle the event that training failed, trigger training timer again by using threadX timer


Input: 
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void GroupManager_TrainingFailed(void) 
{ 
	OSAL_SET_TIMER_EXPLICIT(GROUP_MANAGER_TRAINING_TIMER_EXPIRED, OSAL_TIMERS_MS_TO_K_TICKS(GROUP_MANAGER_CALENDAR_WHEEL_TIMER_IN_MILLISECONDS), TASK_GROUP_MANAGER);
}


/**********************************************************************************

GroupManager_ProcessTrainingResults


Description:
------------
handle the results (combinations) received from the training module or static group request


Input: 
-----
numOfCombinations	- number of valid entries in trainingCombinations array
trainingCombinations - a pointer to an array that contains the combinations
isStaticGroup		 - TRUE if called because of "create static group" CLI, FALSe otherwise
calculated by the training module

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerProcessTrainingResults(uint8 numOfCombinations, GroupCombinationTableEntry_t *trainingCombinations, bool isStaticGroup) 
{
	uint8 combinationNumber = 0;
	StaId sortedMembers[GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP] = {INVALID_STA_INDEX, INVALID_STA_INDEX, INVALID_STA_INDEX, INVALID_STA_INDEX};
	StaId participatingStations[GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP] = {INVALID_STA_INDEX, INVALID_STA_INDEX, INVALID_STA_INDEX, INVALID_STA_INDEX};
	uint8 numberOfNss[GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP] = {0, 0, 0, 0};
	uint8 phyRates[GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP] = {0xFF, 0xFF, 0xFF, 0xFF}; // used to store rates (MCS) that were received for group in this training
	uint8 nssBitmap = 0;
	uint8 indexInCombination = 0;
	uint8 indexInArray = 0;
	uint8 indexInArrayOfLowestStationIndex = 0;
	StaId stationIndex = INVALID_STA_INDEX;
	uint8 numberOfParticipatingStations = 0;
	uint8 shift = 0;
	uint8 alreadyExist = FALSE;
	uint32 tempBitmap = 0;
	uint16 occupiedBitmap = 0;
	uint8 groupIndex = 0;
	uint8 virtualGroupIndex = 0;
	uint8 virtualGroupWithLowestMark = 0;
	uint8 lowestGroupMark = 0;
	StaId currentStationInVector = 0;
	uint8 virtualGroupMark = 0;
	uint8 copyCombination = FALSE;
	uint32 currentTimeStamp = 0;
	uint32 invalidVal = GROUP_MANAGER_INVALID_COMBINATION_VALUE;
	
	/* run on entire combination vector*/
	for(combinationNumber = 0; combinationNumber < numOfCombinations; combinationNumber++)
	{
		// for each entry processing set all parameters to initial values
		for (indexInArray = 0; indexInArray < GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP; indexInArray++)
		{
			sortedMembers[indexInArray] = INVALID_STA_INDEX;
			participatingStations[indexInArray] = INVALID_STA_INDEX;
		}
		memset(phyRates, 0xFF, GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP);
		memset(numberOfNss, 0, GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP);
		numberOfParticipatingStations = 0;
		nssBitmap = 0;
#ifdef GROUP_MANAGER_DEBUG												
		ILOG0_D("GroupManager_TrainingResultsWereReceived: entry num %d", combinationNumber);
#endif
		currentStationInVector = trainingCombinations[combinationNumber].staId;
		// the following logic valid only for valid stations in combinations array
		if(currentStationInVector == INVALID_STA_INDEX)
		{
			ILOG2_D("GroupManager_TrainingResultsWereReceived: invalid	StaIdx %d", currentStationInVector);
			continue;
		}
	
		// if station index is valid and it's combination value is 0xFFFFFFFF meaning there is no potential group found for this station
		if(memcmp( trainingCombinations[combinationNumber].groupCombinationSIDs, &invalidVal, sizeof(uint32)) == 0)
		{
#ifdef GROUP_MANAGER_DEBUG													
			ILOG0_D("GroupManager_TrainingResultsWereReceived: staIdx %d, invalid combination. ", currentStationInVector);
#endif
			// training failed for this station
			groupManagerStationEventTrainingFailed(currentStationInVector); 	
		}
		else
		{
			/* Sort the stations by their index */
			for(indexInArray = 0; indexInArray < GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP; indexInArray++)
			{
				// to sort the SIDs each time run on entire vectore of SIDs from training
				for(indexInCombination = 0; indexInCombination < GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP; indexInCombination++)
				{
					if(trainingCombinations[combinationNumber].groupCombinationSIDs[indexInCombination] <= stationIndex)
					{
						stationIndex = trainingCombinations[combinationNumber].groupCombinationSIDs[indexInCombination];
						indexInArrayOfLowestStationIndex = indexInCombination;
					}
				}
				// store the lowest found SID in new db 
				sortedMembers[indexInArray] = stationIndex;
				// set to invalid high value the copied SID
				trainingCombinations[combinationNumber].groupCombinationSIDs[indexInArrayOfLowestStationIndex] = INVALID_STA_INDEX;
				// store provided MCS of station 
				phyRates[indexInArray] = trainingCombinations[combinationNumber].groupCombinationPhyRates[indexInArrayOfLowestStationIndex];
				// set station index to high to start looking for next lowest SID 
				stationIndex = INVALID_STA_INDEX;
			}
#ifdef GROUP_MANAGER_DEBUG										
			ILOG0_D("GroupManager_TrainingResultsWereReceived: staId %d", 
					currentStationInVector);
			ILOG0_DDDD("GroupManager_TrainingResultsWereReceived: (%d, %d, %d, %d).", 
					sortedMembers[0], sortedMembers[1],sortedMembers[2], sortedMembers[3]);
			ILOG0_DDDD("GroupManager_TrainingResultsWereReceived: MCS - (%d, %d, %d, %d)",
					phyRates[0],phyRates[1], phyRates[2],phyRates[3]);
			ILOG2_DDD("GroupManager_TrainingResultsWereReceived: groupGrade - %d, singleNsts %d, doubleNsts %d",
				   trainingCombinations[combinationNumber].groupMark, trainingCombinations[combinationNumber].singleNstsPhyRate, 
				   trainingCombinations[combinationNumber].doubleNstsPhyRate);
				
	
#endif
	
			/* Remove duplicate from sorted members and calculate NSS */
			stationIndex = INVALID_STA_INDEX;

			for(indexInArray = 0; indexInArray < GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP; indexInArray++)
			{
				if(INVALID_STA_INDEX != sortedMembers[indexInArray])
				{
					if(stationIndex != sortedMembers[indexInArray])
					{
						stationIndex = sortedMembers[indexInArray];
						/* participatingStations final members of group without duplicates*/
						participatingStations[numberOfParticipatingStations] = stationIndex;
						numberOfParticipatingStations++;
					}
					/* contains final NSS results for group in participatingStations array*/
                    ASSERT(numberOfParticipatingStations != 0);
					numberOfNss[(uint8)(numberOfParticipatingStations - 1)]++;
				}
			}
				
				/* build nss bitmap for new virtual group*/
			for(indexInArray = 0; indexInArray < numberOfParticipatingStations; indexInArray++)
			{
				shift = indexInArray << GROUP_MANAGER_NSS_SIZE_SHIFT;
				nssBitmap |= ((numberOfNss[indexInArray] - 1) << shift);
			}
#ifdef GROUP_MANAGER_DEBUG													
			ILOG0_D("GroupManager_TrainingResultsWereReceived: Nss bitmap: 0x%x", nssBitmap);
#endif
						
			/* check if new group already exists in occupying groups*/
			occupiedBitmap = GroupManagerGlobalParametersDb.occupiedGroupsBitmap;
			alreadyExist = FALSE;
			while(occupiedBitmap)
			{
				/* Check if group already exists in group database */
				groupIndex = Utils_CountTrailingZeros(occupiedBitmap);
				occupiedBitmap &= ~(0x1 << groupIndex);
				// if all members are equal and nss for each member is equal found the same group in our db

				ILOG2_D("GroupManager_TrainingResultsWereReceived:occupied groupId %d",	groupIndex);
	
				ILOG2_DDDD("GroupManager_TrainingResultsWereReceived:occupied groupStations (%d,%d,%d,%d)",
						GroupManagerGroupDb[groupIndex].participatingStations[0],
						GroupManagerGroupDb[groupIndex].participatingStations[1],
						GroupManagerGroupDb[groupIndex].participatingStations[2],
						GroupManagerGroupDb[groupIndex].participatingStations[3]);
		
				if((memcmp(participatingStations, GroupManagerGroupDb[groupIndex].participatingStations,
					sizeof(participatingStations)) == 0) &&
					(nssBitmap == GroupManagerGroupDb[groupIndex].nssBitmap))
				{
					alreadyExist = TRUE;
#ifdef GROUP_MANAGER_DEBUG															
					ILOG0_DDD("GroupManager_TrainingResultsWereReceived: Group Exists in occupied DB: EntryIdx %d, staId %d, groupIndex %d", 
						combinationNumber, currentStationInVector, groupIndex);
#endif
					if(isStaticGroup)
					{
						SERIAL_TRACE("The group already exists.",0,0,0);
						ILOG0_V("[groupManagerProcessTrainingResults], The group already exists");
					}
					break;
				}
			}
			// if group doesn't exist in occupying groups
			if(!alreadyExist)
			{
				/* Check if group already exist in virtual groups database */
				virtualGroupMark = trainingCombinations[combinationNumber].groupMark; 
				tempBitmap = GroupManagerGlobalParametersDb.virtualGroupsBitmap;
				currentTimeStamp = GET_TSF_TIMER_LOW();
				
				while(tempBitmap)
				{
					virtualGroupIndex = Utils_FindFirstSetAndClear(&tempBitmap);
#ifdef GROUP_MANAGER_DEBUG															
					ILOG0_D("GroupManager_TrainingResultsWereReceived: virtualGroupId %d",
						virtualGroupIndex);
					ILOG0_DDDD("GroupManager_TrainingResultsWereReceived: groupStations (%d,%d,%d,%d)",
						GroupManagerVirtualGroupDatabase[virtualGroupIndex].participatingStations[0],
						GroupManagerVirtualGroupDatabase[virtualGroupIndex].participatingStations[1],
						GroupManagerVirtualGroupDatabase[virtualGroupIndex].participatingStations[2],
						GroupManagerVirtualGroupDatabase[virtualGroupIndex].participatingStations[3]);
#endif
						
					if((memcmp(participatingStations,GroupManagerVirtualGroupDatabase[virtualGroupIndex].participatingStations, sizeof(participatingStations))==0) &&
						(nssBitmap == GroupManagerVirtualGroupDatabase[virtualGroupIndex].nssBitmap))
					{
						/* Group already exists in virtual bitmap - update mark and timestamp */
						GroupManagerVirtualGroupDatabase[virtualGroupIndex].groupMark = virtualGroupMark;
						GroupManagerVirtualGroupDatabase[virtualGroupIndex].timeStamp = currentTimeStamp;
						alreadyExist = TRUE;
						if(isStaticGroup)
						{
							// update index of staticVirtualGroup in DB
							GroupManagerGlobalParametersDb.staticVirtualGroupIndex = virtualGroupIndex;
						}
#ifdef GROUP_MANAGER_DEBUG																
						ILOG0_DDD("GroupManager_TrainingResultsWereReceived: Group Exists in VIRTUAL DB: EntryIdx %d, staId %d, virtualGroupIdx %d", 
									combinationNumber, currentStationInVector, virtualGroupIndex);
#endif
						break;
					}
				}
				/* if group not part of virtual DB*/
				if(!alreadyExist)
				{
					tempBitmap = ~GroupManagerGlobalParametersDb.virtualGroupsBitmap;
					copyCombination = FALSE; // this flag indicates whether new group will enter virtual DB
					// if virtual db is full the tempBitmap value is 0
					if(0 != tempBitmap) 
					{		
						// store new group in first place found in virtual db
						virtualGroupWithLowestMark = Utils_FindFirstSet(tempBitmap);
						GroupManagerGlobalParametersDb.virtualGroupsBitmap |= (0x1 << virtualGroupWithLowestMark);
#ifdef GROUP_MANAGER_DEBUG																
						ILOG0_D("GroupManager_TrainingResultsWereReceived: Found free spot %d in virtual DB", virtualGroupWithLowestMark);
#endif
						copyCombination = TRUE;
					}
					else
					{
						/* Virtual groups database is full find group with minimum mark to compare */
						tempBitmap = GroupManagerGlobalParametersDb.virtualGroupsBitmap;
						virtualGroupWithLowestMark = Utils_FindFirstSetAndClear(&tempBitmap);
						lowestGroupMark = GroupManagerVirtualGroupDatabase[virtualGroupWithLowestMark].groupMark;
#ifdef GROUP_MANAGER_DEBUG																
						ILOG0_V("GroupManager_TrainingResultsWereReceived: Virtual DB is full!");
#endif
						while(tempBitmap)
						{
							virtualGroupIndex = Utils_FindFirstSetAndClear(&tempBitmap);
							if((GroupManagerVirtualGroupDatabase[virtualGroupIndex].groupMark < lowestGroupMark) ||
							   ((GroupManagerVirtualGroupDatabase[virtualGroupIndex].groupMark == lowestGroupMark)) && (GroupManagerVirtualGroupDatabase[virtualGroupIndex].timeStamp < GroupManagerVirtualGroupDatabase[virtualGroupWithLowestMark].timeStamp))
							{
								virtualGroupWithLowestMark = virtualGroupIndex;
								lowestGroupMark = GroupManagerVirtualGroupDatabase[virtualGroupIndex].groupMark;
							}
						}
						// if virtulGroupMark (mark of the new candidate group from training) is higgher than the mark of lowest found in DB 
						if(virtualGroupMark >= lowestGroupMark)
						{
#ifdef GROUP_MANAGER_DEBUG																	
							ILOG0_DDD("GroupManager_TrainingResultsWereReceived: new group will overwrite groupIdx %d in virtual DB. Prev group grade %d, new group grade %d", 
							virtualGroupWithLowestMark, lowestGroupMark, virtualGroupMark);
#endif
							/* In case the mark is equal the timestamp is equal or newer so replace the virtual group */
							copyCombination = TRUE;
						}
					}
					// if static group not found in virtual DB store the virtual static group index
					if(isStaticGroup)
					{
						GroupManagerGlobalParametersDb.staticVirtualGroupIndex = virtualGroupWithLowestMark;
					}
					if(copyCombination)
					{
						/* Fill the virtual group entry */
						GroupManagerVirtualGroupDatabase[virtualGroupWithLowestMark].groupMark = virtualGroupMark;
						GroupManagerVirtualGroupDatabase[virtualGroupWithLowestMark].timeStamp = currentTimeStamp;
						GroupManagerVirtualGroupDatabase[virtualGroupWithLowestMark].nssBitmap = nssBitmap;
						GroupManagerVirtualGroupDatabase[virtualGroupWithLowestMark].numberOfParticipatingStations = numberOfParticipatingStations;
						MEMCPY(GroupManagerVirtualGroupDatabase[virtualGroupWithLowestMark].participatingStations, participatingStations, sizeof(participatingStations));
						MEMCPY(GroupManagerVirtualGroupDatabase[virtualGroupWithLowestMark].mcs, phyRates, sizeof(phyRates));
						/* Find the single user NSS in the structure for each message */
						for(indexInArray = 0; indexInArray < numberOfParticipatingStations; indexInArray++)
						{
							currentStationInVector = participatingStations[indexInArray];
							GroupManagerStationDb[currentStationInVector].virtualGroupsBitmap|=(0x1 << virtualGroupWithLowestMark);

							/* run on entire combination vector for all group members and find entry of the group member to extract its NSS value*/
							for(indexInCombination = 0; indexInCombination < GROUP_MANAGER_MAX_NUM_OF_STA_FOR_MU_TRAINING; indexInCombination++)
							{
								if(trainingCombinations[indexInCombination].staId == currentStationInVector)
								{
									// copy even values are invalid, handle them in add group on La CFM in Probing state
									GroupManagerVirtualGroupDatabase[virtualGroupWithLowestMark].ratePerStationSingleUserOneNsts[indexInArray] = trainingCombinations[indexInCombination].singleNstsPhyRate;
									GroupManagerVirtualGroupDatabase[virtualGroupWithLowestMark].ratePerStationSingleUserTwoNsts[indexInArray] = trainingCombinations[indexInCombination].doubleNstsPhyRate;
									break;
								}
							}	
						}
					}
				}
			}
		}
	}
	if(isStaticGroup == FALSE)
	{
		// trigger training timer after processing training results
		OSAL_SET_TIMER_EXPLICIT(GROUP_MANAGER_TRAINING_TIMER_EXPIRED, OSAL_TIMERS_MS_TO_K_TICKS(GROUP_MANAGER_CALENDAR_WHEEL_TIMER_IN_MILLISECONDS), TASK_GROUP_MANAGER);
	}
	
	// check if probing can be triggered immediately
	groupManagerTryToFindNewGroupForProbing(isStaticGroup);
	
}

/**********************************************************************************

GroupManager_TrainingResultsWereReceived 


Description:
------------
handle the results (combinations) received from the training module


Input: 
-----
trainingCombinations - a pointer to an array that contains the combinations
calculated by the training module

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

void GroupManager_TrainingResultsWereReceived(GroupManagerTrainingResults_t *trainingresults) 
{ 	
	groupManagerProcessTrainingResults(GROUP_MANAGER_MAX_NUM_OF_STA_FOR_MU_TRAINING, trainingresults->pGroupCombinationTable, FALSE);
}

/**********************************************************************************

GroupManager_GetTrainingBitmaps 


Description:
------------
return the address of all training bitmaps


Input: 
-----

		
Output:
-------
	

Returns:
--------
	  the address of the training bitmaps
	
**********************************************************************************/
GroupManagerTrainingBitmap_t* GroupManager_GetTrainingBitmaps(void) 
{ 	
	return &GroupManagerGlobalParametersDb.trainingBitmaps;
}

/**********************************************************************************

GroupManager_GidmNotificationConfirmation 


Description:
------------
handle the event of notification confirmation from the GIDM manager

Input: 
-----
notificationParameters - a pointer to the structure that contians the status and
the station index
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void GroupManager_GidmNotificationConfirmation(GroupManagerNotificationParameters_t *notificationParameters)
{
	uint8 groupIndex = 0;
	uint16 stationNotifyingBitmap = 0;
	
	stationNotifyingBitmap = GroupManagerStationDb[notificationParameters->stationIndex].groupsBitmap & GroupManagerGlobalParametersDb.notifyingGroupsBitmap;

	/* A station can be only in one notifying group */
	groupIndex = Utils_CountTrailingZeros(stationNotifyingBitmap);
#ifdef GROUP_MANAGER_DEBUG											
	ILOG0_DD("GroupManager_GidmNotificationConfirmation: SID %d, notify group idx %d", 
		notificationParameters->stationIndex, groupIndex);
#endif	
	groupManagerRunGroupStateMachine(groupIndex, GROUP_MANAGER_GROUP_EVENT_NOTIFICATION_HAS_BEEN_CONFIRMED, notificationParameters);
}

/**********************************************************************************

GroupManager_FillGidmPacketPayload 


Description:
------------
fill the payload of gidm manager for a specific station


Input: 
-----
stationIndex - the station index for which to fill the payload
gidmPayload - pointer to the payload to fill
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void GroupManager_FillGidmPacketPayload(StaId stationIndex, GroupIdManagementPayload_t *gidmPayload)
{
	uint16 groupsBitmap = 0;
	uint32 userPositionVector = 0;
	GroupManagerStationParameters_t *groupManagerStationDataBaseEntry = NULL;
	uint8 shiftUserPosition = 0;
	uint8 userPositionByteIndex = 0;
	
    memset(gidmPayload, 0, sizeof(GroupIdManagementPayload_t));

	groupManagerStationDataBaseEntry = &GroupManagerStationDb[stationIndex];

    groupsBitmap = groupManagerStationDataBaseEntry->groupsBitmap;
	userPositionVector = groupManagerStationDataBaseEntry->userPositionVector; // contains user position of this station in each group for which message is sent

    /* Group 0 is reserved - all groups are shifted by 1 */
	/* Fill membership status array */
	gidmPayload->MemberShipStatus.MembershipStatusBitmap[0] = groupsBitmap << 1;
	gidmPayload->MemberShipStatus.MembershipStatusBitmap[1] = groupsBitmap >> 7;
	gidmPayload->MemberShipStatus.MembershipStatusBitmap[2] = groupsBitmap >> 15;

    /* Fill user position array */
	gidmPayload->UserPosition.UserPositionBitmap[0] = ((userPositionVector << 2) & 0xFF);

	shiftUserPosition = 6;
	for(userPositionByteIndex = 1; userPositionByteIndex < GROUP_MANAGER_NUMBER_OF_USER_POSITION_BYTES_IN_USE; userPositionByteIndex++)
	{
		gidmPayload->UserPosition.UserPositionBitmap[userPositionByteIndex] = ((userPositionVector >> shiftUserPosition) & 0xFF);
		shiftUserPosition += GROUP_MANAGER_USER_POSITION_INCREASE_SHIFT_SIZE;
	}
	
#ifdef MU_MIMO_STATIC_GROUP_SEND_AID_TO_STATION_ENABLED
	if ((StaDB_GetNegotiatedWds(stationIndex) == TRUE) && (StaDB_GetWdsClientType(stationIndex) == PEER_AP))
	{
		gidmPayload->UserPosition.UserPositionBitmap[sizeof(UserPositionArray_t) - 1] = stationIndex + 1;
	}
#endif /* #ifdef MU_MIMO_STATIC_GROUP_SEND_AID_TO_STATION_ENABLED */

	gidmPayload->CategoryCode = VHT_CATEGORY;
	gidmPayload->ActionCode = VHT_CATEGORY_GROUP_ID_MANAGEMENT;
}

#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif

/**********************************************************************************

GroupManager_PostInit 


Description:
------------
Post Initialization of Group Manager

Input:
-----


Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void GroupManager_PostInit(void)
{	
	/* trigger here periodic timer - only after threadX init is complete*/
	if (GROUP_MANAGER_GLOBAL_STATE_ACTIVE == GroupManagerGlobalParametersDb.globalState)
	{
		OSAL_SET_TIMER_EXPLICIT(GROUP_MANAGER_TRAINING_TIMER_EXPIRED, OSAL_TIMERS_MS_TO_K_TICKS(GROUP_MANAGER_CALENDAR_WHEEL_TIMER_IN_MILLISECONDS), TASK_GROUP_MANAGER);
	}
}

/**********************************************************************************

GroupManager_Initialize 


Description:
------------
initialize the group manager initiator and TS manager recipient internal databases 
and structures


Input:
-----


Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void GroupManager_Initialize(void)
{
	StaId indexInArray = 0;

	/* Initialize Calendar wheel of stations- timer will be set later */
	CalendarWheel_Initialize(&GroupManagerStationsCalendarWheel, 
							 GROUP_MANAGER_CALENDAR_WHEEL_SIZE, 
							 GroupManagerStasCalendarWheelSlots, 
							 (uint32)(&GroupManagerStationDb), 
							 sizeof(GroupManagerStationParameters_t));
	
	/* Initialize Calendar wheel of groups- timer will be set later */
	CalendarWheel_Initialize(&GroupManagerGroupsCalendarWheel, 
							 GROUP_MANAGER_CALENDAR_WHEEL_SIZE, 
							 GroupManagerGrpsCalendarWheelSlots, 
							 (uint32)(&GroupManagerGroupDb), 
							 sizeof(GroupManagerGroupParameters_t));

	memset(&GroupManagerGlobalParametersDb, 0, sizeof(GroupManagerGlobalParameters_t));
	GroupManagerGlobalParametersDb.staticVirtualGroupIndex = GROUP_MANAGER_INVALID_GROUP_INDEX;
	GroupManagerGlobalParametersDb.staticOccupiedGroupIndex = GROUP_MANAGER_INVALID_GROUP_INDEX;
	GroupManagerGlobalParametersDb.staticGroupInitialMcs = GROUP_MANAGER_STATIC_GROUP_INITIAL_MCS;
	memset(GroupManagerStationDb, 0, sizeof(GroupManagerStationDb));
	memset(GroupManagerGroupDb, 0, sizeof(GroupManagerGroupDb));
	memset(GroupManagerVirtualGroupDatabase, 0, sizeof(GroupManagerVirtualGroupDatabase));
	
	groupManagerChangeGlobalState(GROUP_MANAGER_GLOBAL_STATE_IDLE);

	GroupManagerGlobalParametersDb.setOperationRequestPhase = GROUP_MANAGER_SET_OPERATION_REQUEST_PHASE_IDLE;
	GroupManagerGlobalParametersDb.returnMsg = NULL;

	GroupManagerGlobalParametersDb.muConfigParams.probingGroupGradeThreshold = GROUP_MANAGER_PROBING_GROUP_MARK_THRESHOLD;
	GroupManagerGlobalParametersDb.muConfigParams.stationGradeInProbingGroupThreshold = GROUP_MANAGER_PROBING_GROUP_STATION_MARK_THRESHOLD;
	GroupManagerGlobalParametersDb.muConfigParams.maxSingleUserValidationStations = GROUP_MANAGER_MAX_NUMBER_OF_SINGLE_USER_VALIDATION;
	GroupManagerGlobalParametersDb.muConfigParams.singleUserPerformenceMargin = GROUP_MANAGER_SINGLE_USER_PERFORMENCE_MARGIN;
	GroupManagerGlobalParametersDb.muConfigParams.maxNumProbingGroups = GROUP_MANAGER_MAX_NUMBER_OF_PROBING_GROUPS;
		
	for(indexInArray = 0; indexInArray < HW_NUM_OF_STATIONS; indexInArray++)
	{
		GroupManagerStationDb[indexInArray].stationIndex = indexInArray;
		GroupManagerStationDb[indexInArray].primaryGroup = GROUP_MANAGER_INVALID_GROUP_INDEX;
		
	}
	
	for(indexInArray = 0; indexInArray < GROUP_MANAGER_MAX_NUMBER_OF_GROUPS; indexInArray++)
	{
		GroupManagerGroupDb[indexInArray].groupIndex = indexInArray;
	}

	// initialize statistics db
	memset(&groupDbStatistics, GROUP_MANAGER_INVALID_GROUP_INDEX, sizeof(groupDbStatistics)); 
	
	TrainingManager_Init();
	// initialize HW group DB
	GroupDB_Init();
	gidmManager_Init();
	
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif


/**********************************************************************************

GroupManagerGetGroupDbAddress 


Description:
------------
get address of statistics db


Input:
-----


Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
uint8* GroupManagerGetGroupDbAddress()
{
	return (uint8*)&groupDbStatistics;
}

#if defined (MU_MIMO_STATIC_GROUP_ENABLED)

void GroupManager_UpdateMuStaticGroupTxMode(TxSelectorGroupTxMode_e txMode)
{
	if (!((GroupManagerGlobalParametersDb.staticOccupiedGroupIndex == GROUP_MANAGER_INVALID_GROUP_INDEX) ||
		(GroupManagerGlobalParametersDb.globalState != GROUP_MANAGER_GLOBAL_STATE_ACTIVE)))
	{
		TxSelector_UpdateGroupTxMode(GroupManagerGlobalParametersDb.staticOccupiedGroupIndex, txMode);
	}
}

void GroupManager_UpdateMuStaticGroupInitialMcs(mcs_e mcs)
{
	GroupManagerGlobalParametersDb.staticGroupInitialMcs = mcs;
}

#endif // MU_MIMO_STATIC_GROUP_ENABLED

/**********************************************************************************

GroupManagerResetGroupMembersStatistic 


Description:
------------
resets members info of provided group index


Input:
-----


Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void GroupManagerResetGroupMembersStatistic(uint8 groupIndex)
{
	memset(groupDbStatistics.groupInfoDb[groupIndex], GROUP_MANAGER_INVALID_GROUP_INDEX, sizeof(groupDbStatistics.groupInfoDb[groupIndex][0]) * MU_NUM_OF_STATS);
}


/**********************************************************************************

GroupManager_UpdateMuGroupTxopMode



Description:
------------
overwrite MU TXOP mode


Input:
-----


Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

void GroupManager_UpdateMuGroupTxopMode(bool txopMode)
{
	
	uint8 groupId = 0;
	
	uint16 tempBitmap = GroupManagerGlobalParametersDb.activeGroupsBitmap & (~GroupManagerGlobalParametersDb.removingGroupsBitmap);

	//First set global flag - all new Groups will use this value
	GroupDb_TxopEnable = txopMode;

	/* go over active groups */
	while(tempBitmap)
	{
		groupId = Utils_CountTrailingZeros(tempBitmap);
		tempBitmap &= ~(0x1 << groupId);
		GroupDb_UpdateMuGroupTxopMode(groupId, txopMode);	
	}
}

GroupManagerStationState_e GroupManager_GetStaState(StaId staId)
{
	return GroupManagerStationDb[staId].state;
}
#ifdef ENET_INC_ARCH_WAVE600D2
uint8 GroupManager_GetStaMbssRxCtrlSupport(StaId staId)
{
	return GroupManagerStationDb[staId].mbssRxControlSupport;
}
#endif
#ifndef ENET_INC_ARCH_WAVE600
void GroupManager_SetPpduTxModePerStation(StaId stationIndex)
{
	GroupManagerStationParameters_t *pGroupManagerStationParameters = NULL;
	uint32 groupIndex;
	uint32 tempBitmap;
	uint32 usp;
	GroupDbPpduTxMode_e ppduTxMode;

	DEBUG_ASSERT(stationIndex < HW_NUM_OF_STATIONS);

	pGroupManagerStationParameters = &(GroupManagerStationDb[stationIndex]);

	ppduTxMode = (GroupDbPpduTxMode_e)StaDb_GetMinPpduTxModePerStation(stationIndex);

	tempBitmap = pGroupManagerStationParameters->groupsBitmap;
	
	while(tempBitmap)
	{
		groupIndex = Utils_CountTrailingZeros(tempBitmap);
		tempBitmap &= ~(0x1 << groupIndex);

		usp = (pGroupManagerStationParameters->userPositionVector >> (groupIndex * GROUP_MANAGER_USP_SIZE_SHIFT)) & GROUP_MANAGER_USP_BITS_MASK;

		GroupDb_SetTxPpduModePerUspInGroup(groupIndex, usp, ppduTxMode);
	}
}
#endif
/**********************************************************************************

GroupManager_UpdateMuConfigParams


Description:
------------
overwrites the value of specific configuration param


Input:
	paramType- parameter to overwrite
	paramVal - new value
-----


Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

void GroupManager_UpdateMuConfigParams(GroupManagerConfigurationParamType_e paramType, uint8 paramVal)
{
	switch(paramType)

	{
		
		case GROUP_MANAGER_PROBING_GROUP_GRADE_PARAM:
			GroupManagerGlobalParametersDb.muConfigParams.probingGroupGradeThreshold = paramVal;
			break;

		case GROUP_MANAGER_MAX_SU_VAL_STATIONS_PARAM:
			GroupManagerGlobalParametersDb.muConfigParams.maxSingleUserValidationStations = paramVal;
			break;

		case GROUP_MANAGER_MIN_STATION_GRADE_IN_GROUP_PARAM:
			GroupManagerGlobalParametersDb.muConfigParams.stationGradeInProbingGroupThreshold = paramVal;
			break;

		case GROUP_MANAGER_SU_PERFORMENCE_PARAM:
			GroupManagerGlobalParametersDb.muConfigParams.singleUserPerformenceMargin = paramVal;
			break;

		case GROUP_MANAGER_MAX_PROBING_GROUPS_PARAM:
			GroupManagerGlobalParametersDb.muConfigParams.maxNumProbingGroups = paramVal;
			break;

		default:
			
			break;
	}
}


/**********************************************************************************

groupManagerSetStationSuTxopAndAmpduLimit



Description:
------------
overwrites the value of txop and ampduLimit for specific station in staDb


Input:

-----


Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void groupManagerSetStationSuTxopAndAmpduLimit(StaId stationIndex, uint8 txopMode, uint8 numOfAmpdus)
{
	K_MSG *kMsg_p = PNULL;
	LaSetTxopForMuMember_t *setTxop_msg = PNULL;
	uint8 vapId = 0;
	
	kMsg_p = OSAL_GET_MESSAGE(sizeof(LaSetTxopForMuMember_t));
	setTxop_msg = (LaSetTxopForMuMember_t *)pK_MSG_DATA(kMsg_p);
	setTxop_msg->staId = stationIndex;
	setTxop_msg->txopMode = txopMode;
	StaDb_setTxopMode(stationIndex, txopMode);
	StaDB_SetStationAmpduLimit(stationIndex, numOfAmpdus);
	StaDb_GetVapId(stationIndex, &vapId);
	OSAL_SEND_MESSAGE(LINK_ADAPTATION_SET_TXOP_FOR_MU_MEMBER, TASK_LINK_ADAPTATION, kMsg_p, vapId);
}


/**********************************************************************************

GroupManager_EnableDynamicModeWithStaticMode



Description:
------------
changes mode of operation from static gorup pnly to static grop and dynamic group. Triggers timer
in order to enable training mechanism again


Input:
-----


Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/

void GroupManager_EnableDynamicModeWithStaticMode()
{
	GroupManagerGlobalParametersDb.staticGroupOnlyMode = FALSE;
	OSAL_SET_TIMER_EXPLICIT(GROUP_MANAGER_TRAINING_TIMER_EXPIRED, OSAL_TIMERS_MS_TO_K_TICKS(GROUP_MANAGER_CALENDAR_WHEEL_TIMER_IN_MILLISECONDS), TASK_GROUP_MANAGER);
}


#if defined (MU_MIMO_STATIC_GROUP_ENABLED)

/**********************************************************************************

groupManagerCreateMuStaticGroup 


Description:
------------
functions which supports creation of static MU group

Input:
-----
groupManagerMessage - message from CLI command to generate static MU group


Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerCreateMuStaticGroup(K_MSG* groupManagerMessage)
{
	GroupManager_CreateStaticGroup_t *staticGroup_msg = (GroupManager_CreateStaticGroup_t *)pK_MSG_DATA(groupManagerMessage);
	uint8 i = 0;
	uint8 totalNssNum = 0;

	while((i!=GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP) && (staticGroup_msg->staIds[i]!= INVALID_STA_INDEX))
	{
		totalNssNum+=staticGroup_msg->nssPerUsp[i];
		i++;
	}

	if ((totalNssNum > 4) || (totalNssNum > GroupManagerGlobalParametersDb.cocTxAntsNum))
	{
		SERIAL_TRACE("Error: Can't create group!! Num of Requested streams is bigger than num of active Tx antennas",0,0,0);
		ILOG0_V("[groupManagerCreateMuStaticGroup], Error: Can't create group!! Num of Requested streams is bigger than num of active Tx antennas");
	}
	else if(GroupManagerGlobalParametersDb.staticOccupiedGroupIndex != GROUP_MANAGER_INVALID_GROUP_INDEX)
	{
		SERIAL_TRACE("One static group already exists. Can't create additional one",0,0,0);
		ILOG0_V("[groupManagerCreateMuStaticGroup], One static group already exists. Can't create additional one");
	}
	else if((GroupManagerGlobalParametersDb.globalState == GROUP_MANAGER_GLOBAL_STATE_CREATE_STATIC_GROUP_WAITING_FOR_TRAINING) ||
		(GroupManagerGlobalParametersDb.globalState == GROUP_MANAGER_GLOBAL_STATE_CREATE_STATIC_GROUP_WAITING_FOR_ALL_GROUP_REMOVAL))
	{
		SERIAL_TRACE("Processing previous static group request. Discarding this one",0,0,0);
		ILOG0_V("[groupManagerCreateMuStaticGroup], Processing previous static group request. Discarding this one");
	}
	else if((GroupManagerGlobalParametersDb.globalState == GROUP_MANAGER_GLOBAL_STATE_DISABLING_WAITING_FOR_TRAINING) ||
		(GroupManagerGlobalParametersDb.globalState == GROUP_MANAGER_GLOBAL_STATE_DISABLING_WAITING_FOR_ALL_GROUP_REMOVAL))
	{
		//ILOG0_V("Can't create static group - disabling MU mode");
	}
	else
	{
		groupManagerChangeGlobalState(GROUP_MANAGER_GLOBAL_STATE_CREATE_STATIC_GROUP_WAITING_FOR_TRAINING);
		GroupManagerGlobalParametersDb.staticGroupOnlyMode = TRUE;
		/* each station in HALT state will be returned to candidate state and station in SU validation pending will be returned to active state*/
		for(i=0; (i < GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP) && (staticGroup_msg->staIds[i]!= INVALID_STA_INDEX); i++)
		{
			if(GroupManagerStationDb[staticGroup_msg->staIds[i]].state == GROUP_MANAGER_STATION_STATE_HALT)
			{
		//		ILOG0_D("groupManagerCreateMuStaticGroup: station %d moved from HALT to SINGLE_USER, will be added to static group", 
		//			staticGroup_msg->staIds[i]);
				CalendarWheel_RemoveEntry(&GroupManagerStationsCalendarWheel, (CalendarWheelEntry_t*)&GroupManagerStationDb[staticGroup_msg->staIds[i]]);
				Utils_ZeroBitInBitmap(GroupManagerGlobalParametersDb.trainingBitmaps.haltStationsBitmap.stationBitmap, staticGroup_msg->staIds[i]);
				groupManagerChangeStationState(&GroupManagerStationDb[staticGroup_msg->staIds[i]], GROUP_MANAGER_STATION_STATE_SINGLE_USER);	
			}
			else if(GroupManagerStationDb[staticGroup_msg->staIds[i]].state == GROUP_MANAGER_STATION_STATE_SINGLE_USER_VALIDATION_PENDING)
			{
		//		ILOG0_D("groupManagerCreateMuStaticGroup: station %d moved from SU_VAL_PENDING to ACTIVE, will be added to static group", 
		//			staticGroup_msg->staIds[i]);
				CalendarWheel_RemoveEntry(&GroupManagerStationsCalendarWheel, (CalendarWheelEntry_t*)&GroupManagerStationDb[staticGroup_msg->staIds[i]]);
				groupManagerChangeStationState(&GroupManagerStationDb[staticGroup_msg->staIds[i]], GROUP_MANAGER_STATION_STATE_ACTIVE);
			}
		}
		/* store request params*/
		MEMCPY(&GroupManagerGlobalParametersDb.staticGroupParams, staticGroup_msg, sizeof(GroupManager_CreateStaticGroup_t));
		/* delete all groups in Dbs and stop training*/
#ifdef GROUP_MANAGER_DEBUG												
		ILOG0_V("groupManagerCreateMuStaticGroup: preparing for static group generation");
#endif
		groupManagerRemoveAllVirtualGroups();
		groupManagerRemoveAllActiveGroups();
		TrainingManagerStopTrainingReq();
		
	}
}

/**********************************************************************************

groupManagerDeleteMuStaticGroup 


Description:
------------
functions which supports deletion of static MU group

Input:
-----
groupManagerMessage - message from CLI command with details of static group to delete


Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void groupManagerDeleteMuStaticGroup(K_MSG* groupManagerMessage)
{ 
	UNUSED_PARAM(groupManagerMessage);	
#ifdef GROUP_MANAGER_DEBUG											
	ILOG0_V("groupManagerDeleteMuStaticGroup");
#endif
	GroupManagerGlobalParametersDb.staticGroupOnlyMode = FALSE;
	if((GroupManagerGlobalParametersDb.staticOccupiedGroupIndex == GROUP_MANAGER_INVALID_GROUP_INDEX) ||
		(GroupManagerGlobalParametersDb.globalState != GROUP_MANAGER_GLOBAL_STATE_ACTIVE))
	{
		SERIAL_TRACE("There is no static group to delete!!!",0,0,0);
		ILOG0_V("[groupManagerDeleteMuStaticGroup], There is no static group to delete!!!");
	}
	else
	{
		groupManagerGroupEventRemoveGroup(GroupManagerGlobalParametersDb.staticOccupiedGroupIndex, NULL);
	}
}


#endif //MU_MIMO_STATIC_GROUP_ENABLED

#ifdef TRAINING_WAVE600_Z0
static void GroupManagerIllegalEvent(K_MSG* groupManagerMessage)
{
	UNUSED_PARAM(groupManagerMessage);	
	FATAL("GroupManagerIllegalEvent");
}
#endif // TRAINING_WAVE600_Z0

/**********************************************************************************
groupManagerUpdateCoCInfo



Description:
------------

	
Input: 
-----	
	
Returns: none
--------
	 
**********************************************************************************/
 
void groupManagerUpdateCoCInfo(K_MSG *pCoCInfoMsg)
{
	cocUpdateAntMaskMsg_t*	pCoCInfo = (cocUpdateAntMaskMsg_t*)pK_MSG_DATA(pCoCInfoMsg);
	GroupManagerGlobalParametersDb.cocTxAntsNum = pCoCInfo->currentTxAntsNum;
}
uint8 groupManagerGetCoCTxAntsNum(void)
{
	return GroupManagerGlobalParametersDb.cocTxAntsNum;
}
#if defined (ENET_INC_ARCH_WAVE600)
uint8 GroupManager_GetVapforGroup(uint8 groupIndex)
{
	uint8 vapId;
	GroupManagerGroupParameters_t*	groupManagerGroupDataBaseEntry = &GroupManagerGroupDb[groupIndex];
	StaDb_GetVapId(groupManagerGroupDataBaseEntry->participatingStations[0], &vapId);

	return vapId;
}
#endif

