#ifndef MEM_ACCESS_H_
#define MEM_ACCESS_H_

#define HW_REGISTER            volatile uint32
#define SHADOW_REGISTER        uint32

#define HW_16_BIT_REGISTER     volatile uint16
#define SHADOW_16_BIT_REGISTER uint16

/*********************************************************************************
Method:       GENERATE_MASK
Description:  Create 32 bit GENERATE_MASK 1xbits.
Parameter:    bits - the number of '1's starting from bit #0.
*********************************************************************************/
#define GENERATE_MASK( bits )  ((uint32) (((uint32)1 << (bits)) - 1))

/*********************************************************************************
Method:       GENERATE_FIELD
Description:  Create 32 bit number in which val is placed at field_shift as start
              bit and field_bits is the bit count
Parameter:    val - the value
              field_shift - start bit
              field_bits - GENERATE_MASK limit
*********************************************************************************/
#define GENERATE_FIELD( val, field_shift, field_bits )         \
  ( (((uint32) (val)) & GENERATE_MASK( field_bits )) << (field_shift) )

/*********************************************************************************
Method:       RETRIEVE_FIELD
Description:  Extract the field value starting from field_shift and limit the
              number of bits by field_bits from val
Parameter:    val - the value from which we will extract the field value
              field_shift - start bit
              field_bits - GENERATE_MASK limit
*********************************************************************************/
#define RETRIEVE_FIELD( val, field_shift, field_bits )         \
  ( (((uint32) (val)) >> (field_shift)) & GENERATE_MASK( field_bits ) )

/*********************************************************************************
Method:       GENERATE_BIT
Description:  Set 1/0 (val) in a 32 bit size value
Parameter:    val - 1/0
              bit - the bit number
*********************************************************************************/
#define GENERATE_BIT( val, bit )  GENERATE_FIELD( val, bit, 1 )

/*********************************************************************************
Method:       RETRIEVE_BIT
Description:  Get the value of a specific bit
Parameter:    val - the value from which we will extract the bit
              bit - the bit number
*********************************************************************************/
#define RETRIEVE_BIT( val, bit )  RETRIEVE_FIELD( val, bit, 1 )

/*********************************************************************************
Method:       READ
Description:  Get the value from a specific location
Parameter:    addr - the memory location
              type - address type (32/16/8 volatile ...)
*********************************************************************************/
#define READ( addr, type )        ( (*((type *)(addr))) )

/*********************************************************************************
Method:       WRITE
Description:  Set the value to a specific location
Parameter:    addr - the memory location
              type - address type (32/16/8 volatile ...)
              val - the value to set
*********************************************************************************/
#define WRITE( addr, type, val )  *((type *) (addr)) = (val)

/*********************************************************************************
Method:       READ_FIELD
Description:  Extract a field from specific memory location starting from
              field_shift and limit the number of bits by field_bits
Parameter:    addr - the memory location
              type - address type (32/16/8 volatile ...)
              field_shift - start bit
              field_bits - GENERATE_MASK limit
*********************************************************************************/
#define READ_FIELD( addr, type, field_shift, field_bits )       \
  ( (READ( addr, type ) >> (field_shift)) & GENERATE_MASK( field_bits ) )

/*********************************************************************************
Method:       WRITE_FIELD
Description:  Set a field to specific memory location, from field_shift and limit the
              number of bits by field_bits
Parameter:    addr - the memory location
              type - address type (32/16/8 volatile ...)
              field_shift - start bit
              field_bits - GENERATE_MASK limit
*********************************************************************************/
#define WRITE_FIELD( addr, type, val, field_shift, field_bits ) \
  WRITE( addr, type,                                            \
  ( READ( addr, type ) &                                        \
  (~ (GENERATE_MASK( field_bits ) << (field_shift)) ) ) |       \
  GENERATE_FIELD( val, field_shift, field_bits ) )

/*********************************************************************************
Method:       READ_BIT
Description:  Get the value of a specific bit from specific memory location
Parameter:    addr - the memory location
              type - address type (32/16/8 volatile ...)
              bit - bit index
*********************************************************************************/
#define READ_BIT( addr, type, bit )       READ_FIELD( addr, type, bit, 1 )

/*********************************************************************************
Method:       WRITE_BIT
Description:  Write 1/0 to specific bit from specific memory location
Parameter:    addr - the memory location
              type - address type (32/16/8 volatile ...)
              val - 1/0
              bit - bit index
*********************************************************************************/
#define WRITE_BIT( addr, type, val, bit ) WRITE_FIELD( addr, type, val, bit, 1 )

/*********************************************************************************
Method:       CLEAR_BIT
Description:  Clear a specific bit from specific memory location
Parameter:    addr - the memory location
              type - address type (32/16/8 volatile ...)
              bit - bit index
*********************************************************************************/
#define CLEAR_BIT( addr, type, bit )      WRITE_BIT( addr, type, 0, bit )

/*********************************************************************************
Method:       SET_BIT
Description:  Set a specific bit from specific memory location
Parameter:    addr - the memory location
              type - address type (32/16/8 volatile ...)
              bit - bit index
*********************************************************************************/
#define SET_BIT( addr, type, bit )        WRITE_BIT( addr, type, 1, bit )

#endif /* MEM_ACCESS_H_ */
