#ifndef __LpfClbrUtils_h__
#define __LpfClbrUtils_h__

#include "HdkGlobalDefs.h"
#include "LpfClbrHndlr.h"
#include "PhyCalDriver_API.h"
#include "RficDriver_API.h"

/**********************************************************************/
/***					Type Definitions							***/
/**********************************************************************/
// Definition of setter functions for the LPF words (Rx, Partial Rx, Tx) - used in the generic min search function
typedef void (*setLpfWordFuncPtr_t)(AntennaBitmaps_e inAnts, RficPathTypeIndex_e inPathType, uint8* inLpfWord);
typedef void (*upperBoundErrorMarginAdjustmentFuncPtr_t)(int64 inOrigEnergy, int64 *outMarginEnergy);
typedef void (*lowerBoundErrorMarginAdjustmentFuncPtr_t)(int64 inOrigEnergy, int64 *outMarginEnergy);

/**********************************************************************/
/***				Public Functions Definitions					***/
/**********************************************************************/
//************************************
// Method:    LpfUtils_measureBinEnergyFull
// Purpose:   Measure the energy at the supplied bin (generates its own tone)
// Parameter: LpfElements_t * pElements - local scratchpad
// Parameter: int8 bin - The bin on which to measure the energy
// Parameter: ToneGenShiftMode_e shiftMode - The tone generator mode
// Parameter: int64 * measuredEnergy - The measured energy
// Returns:   void
//************************************
void LpfUtils_measureBinEnergyFull(IN LpfElements_t* pElements, IN int8 bin, IN ToneGenShiftMode_e shiftMode, OUT int64* measuredEnergy);
//************************************
// Method:    LpfUtils_measureRefAvgEnergy
// Purpose:   Measure the average energy of the pass band (using several adjacent bins)
// Parameter: LpfElements_t * pElements - local scratchpad
// Parameter: int8 bin - The first bin of the pass band for the average
// Parameter: int64 * pRefAvgEnergy - results
// Returns:   void
//************************************
void LpfUtils_measureRefAvgEnergy(IN LpfElements_t* pElements, IN int8 bin, OUT int64* pRefAvgEnergy);
//************************************
// Method:    LpfUtils_findBestWord
// Purpose:   Min search for the LPF word that best fits the desired energy level. This is a generic function serving all LPF calibration options.
// Parameter: IN LpfElements_t * pElements - local scratchpad
// Parameter: IN setLpfWordFuncPtr_t setLpfWordFunc - Pointer to the LPF word setter function to use
// Parameter: IN upperBoundErrorMarginAdjustmentFuncPtr_t upperBoundErrFunc - Method used to adjust upper error margin in verifyPassBandEnergy
// Parameter: IN lowerBoundErrorMarginAdjustmentFuncPtr_t lowerBoundErrFunc - Method used to adjust lower error margin in verifyPassBandEnergy
// Parameter: IN int8 bin - The bn on which the min search is conducted
// Parameter: IN ToneGenShiftMode_e shiftMode - Tone generator shift mode
// Parameter: IN int64 * energyOffset - Energy offsets to apply on measured energies (optional)
// Parameter: IN bool enforcePairing - Enforce measured bin level lower than adjacent bin (to avoid glitches)
// Parameter: IN bool enforcePositive - Enforce an LPF word where the measured energy is higher than the desired (even though may not be the closest)
// Parameter: IN int8 highCorrectionUnit - LPF word correction unit to attain highe level of energy
// Parameter: OUT uint8 * resLpfWords - Results words best matcing the desired outcome.
// Returns:   RetVal_e  - Did the best results found reside within the configured margin from the desired energy.
//************************************
RetVal_e LpfUtils_findBestWord(IN LpfElements_t* pElements, IN setLpfWordFuncPtr_t setLpfWordFunc, IN upperBoundErrorMarginAdjustmentFuncPtr_t upperBoundErrFunc, IN lowerBoundErrorMarginAdjustmentFuncPtr_t lowerBoundErrFunc, IN int8 bin, IN ToneGenShiftMode_e shiftMode, IN int64* energyOffset, IN bool enforcePairing, IN bool enforcePositive, IN int8 highCorrectionUnit, OUT uint8* resLpfWords);
//************************************
// Method:    LpfUtils_verifyBinEnergy
// Purpose:   Measure the energy at the supplied bin and compare to the supplied reference
// Parameter: LpfElements_t * pElements - local scratchpad
// Parameter: int8 bin - Bin to measure the energy over
// Parameter: ToneGenShiftMode_e shiftMode - Tone generator mode
// Parameter: uint8 fixedPointBitShift - Fixed point offset bit shift
// Parameter: int64 * energyOffset - Energy offsets to apply on measured energies (optional)
// Parameter: int64 * refEnergy - Reference energy to test against
// Returns:   RetVal_e - success iff the energy is lower than the reference
//************************************
RetVal_e LpfUtils_verifyBinEnergy(IN LpfElements_t* pElements, IN int8 bin, IN ToneGenShiftMode_e shiftMode, IN uint8 fixedPointBitShift, IN int64* energyOffset, IN int64* refEnergy);
//************************************
// Method:    LpfUtils_getRxStopBinRefEnergy
// Purpose:   Adjust the Rx reference energy to test against the stop bin
// Parameter: LpfElements_t * pElements - local scratchpad
// Parameter: int64 * res - reference energy
// Returns:   void
//************************************
void LpfUtils_getRxStopBinRefEnergy(IN LpfElements_t* pElements, OUT int64* res);
//************************************
// Method:    LpfUtils_getTxStopBinRefEnergy
// Purpose:   Adjust the Tx reference energy to test against the stop bin
// Parameter: LpfElements_t * pElements - local scratchpad
// Parameter: int64 * res - reference energy
// Returns:   void
//************************************
void LpfUtils_getTxStopBinRefEnergy(IN LpfElements_t* pElements, OUT int64* res);

#endif // LpfClbrUtils_h__
