#include "LpfClbrHndlr.h"
#include "LpfClbrHndlr_API.h"
#include "LpfDefinitions.h"
#include "RegAccess_Api.h"
#include "mt_sysrst.h"
#include "stringLibApi.h"
#include "PhyDriver_API.h"
#include "PhyCalDriver_API.h"
#include "RficDriver_API.h"
#include "Afe_API.h"
#include "CalibrationHandlerUtils.h"
#include "ErrorHandler_Api.h"
#include "CalibrationHandler.h"
#include "Shram_ClbrDataBuffer.h"

#include "loggerAPI.h"

#define LOG_LOCAL_GID   GLOBAL_GID_CALIBRATIONS
#define LOG_LOCAL_FID 3


////////////////////////
// Definitions
////////////////////////


extern Calibration_params_otf_t Calibration_params_otf;	

////////////////////////
// Forward Declarations
////////////////////////
static RetVal_e lpfCalibrateAntenna(IN LpfElements_t* pElements);
static void LPF_StoreResults(void);


 //************************************
 // Method:    lpfInitAntennaCalibration
 // Purpose:   Init SW & HW for antenna calibration - only the shared logics are here
 // Parameter: IN LpfElements_t * pElements
 // Returns:   void
 //************************************
static void lpfInitAntennaCalibration(IN LpfElements_t* pElements)
{
	// Arrange the antennas used for this calibration process
	RficDriver_GetDigitalLoopBackAnt(&pElements->dataSet.antInfo);
	pElements->dataSet.antBitmap = (AntennaBitmaps_e)((1 << pElements->dataSet.antInfo.TxLoopbackAnt)| 
													  (1 << pElements->dataSet.antInfo.RxLoopbackAnt));

	// Initialize RF
	SetRxGains((AntennaBitmaps_e)(1<<pElements->dataSet.antInfo.RfCalibAnt),HDK_LNA_INDEX_HIGH_GAIN,pElements->params.pgc1,pElements->params.pgc2,pElements->params.pgc3); // setting low values to both LNA & MIXER,( LNA=0,MIXER=1).both are not in use

	// Initialize PHY
	PhyCalDrv_SetLoopBackMode(CLBR_PROC_TYPE_LPF,pElements->dataSet.antInfo.RfCalibAnt);
	AFE_SetLoopBack(pElements->dataSet.antInfo.RfCalibAnt, pElements->dataSet.antInfo.RxLoopbackAnt, AFE_MODE_LOOPBACK,0);
	PhyCalDrv_SetToneGenParams(0, 0, ANTENNA_BITMAP_ALL_ANTENNAS_GEN5, 0);
	PhyCalDrv_ToneGenInit(pElements->dataSet.currToneGenAmp);//amplitude
}

 //************************************
 // Method:    lpfFinishAntennaCalibration
 // Purpose:   TBD
 // Parameter: IN LpfElements_t * pElements
 // Returns:   void
 //************************************
static void lpfFinishAntennaCalibration(IN LpfElements_t* pElements)
{
	// Restore HW status 
	PhyCalDrv_ToneGenRestore();
	PhyCalDrv_RestoreOperationalTx();
	AFE_RestoreFromLoopBack();
	RficDriver_LoadRegDB();
}

 //************************************
 // Method:    lpfInitCalibrationDataSet
 // Purpose:   Init the LPF calibration data set
 // Parameter: IN LpfElements_t * pElements
 // Returns:   void
 //************************************
static void lpfInitCalibrationDataSet(IN LpfElements_t* pElements)
{
	// Reset the data set
	memset(&pElements->dataSet, 0, sizeof(pElements->dataSet));

	// Initialize data set values for this run
	pElements->dataSet.isIqSwap = FALSE;

	pElements->dataSet.isCB = (HDK_getChannelWidth() == 0) ? FALSE:TRUE;

	// Set whether calibration is needed for both I & Q paths 
	pElements->dataSet.numOfRfPathsToCalibrate = LPF_NUM_OF_RF_PATHS_TO_CALIB;

	// choosing the bands that we want to transmit & measure in according to the mode : CB/NCB
	pElements->dataSet.currToneGenAmp = pElements->params.toneGenAmp;  // values are from 7-11 each value is double from the previous;
	if (pElements->dataSet.isCB == FALSE)
	{
		pElements->dataSet.currToneGenAmp -= 1; // we need to reduce the Power at the ToneGen due to compression issues.
	}
}

static RetVal_e lpfPerformCalibration(IN LpfElements_t* pElements)
{
	RetVal_e retVal = RET_VAL_SUCCESS;
	uint8 antMask,numOfAnts;

	HDK_GetMaxActiveAntennasNumAndMask(&numOfAnts, &antMask);

	// Actual calibration, per antenna
	for (pElements->dataSet.antInfo.RfCalibAnt=ANTENNA_0;pElements->dataSet.antInfo.RfCalibAnt<TOTAL_ANTENNAS;pElements->dataSet.antInfo.RfCalibAnt++)
	{
		if(ANTENNA_MASK_IS_BIT_ON(antMask, pElements->dataSet.antInfo.RfCalibAnt))
		{	
			retVal = lpfCalibrateAntenna(pElements);
			if (SINGLE_ANTENNA_MODE == TRUE && retVal == RET_VAL_SUCCESS)
			{
				break;
			}
		}
		else
		{
			pElements->results.status[pElements->dataSet.antInfo.RfCalibAnt] = CAL_STATUS_INIT;
		}
	}

	return retVal;
}

static void lpfFinishCalibration(IN LpfElements_t* pElements, IN uint32 calStartTime)
{
	// Set the words found
	LpfDep_SetAntCalibResults(pElements);

	// Save run time
	pElements->results.lastRunTime = TIME_STAMP(END_TIME,calStartTime);
}

static RetVal_e lpfCalibrateAntenna(IN LpfElements_t* pElements)
{
	RetVal_e retVal;

	// Init RF and PHY
	lpfInitAntennaCalibration(pElements);

	// Actual calibration process per antenna
	retVal = LpfDep_PerformAntennaCalibration(pElements);

	// Update status
	pElements->results.status[pElements->dataSet.antInfo.RfCalibAnt] = ((retVal == RET_VAL_SUCCESS) ? CAL_STATUS_PASS : CAL_STATUS_FAIL);

	// Restore (for all Antennas)
	lpfFinishAntennaCalibration(pElements);

	return retVal;
}

static RetVal_e lpfRun(IN LpfElements_t* pElements)
{
	RetVal_e retVal;
	uint32 calStartTime = TIME_STAMP(START_TIME,0);
	CalDataState_e calStoreState;

#if LM_CALIBRATION_LOGS_ON
	ILOG2_V("LPF: Executing Calibration.");
#endif

	lpfInitCalibrationDataSet(pElements);

	retVal = lpfPerformCalibration(pElements);

	lpfFinishCalibration(pElements, calStartTime);

#if LM_CALIBRATION_LOGS_ON
	ILOG2_DD("LPF: Finished Calibration. Ret Val = %d, Run Time = %d", retVal, pElements->results.lastRunTime);
#endif
	
	/* Store calibration results */
	ClbrHndlr_GetCalibrationDataState(&calStoreState);
	if((calStoreState == CAL_DATA_STATE_STORE) && (CLBR_TYPE_OFFLINE == ClbrHndlr_GetClbrType()))
	{
		LPF_StoreResults();
	}		
	return retVal;
}

////////////////////////
// Public Methods
////////////////////////
void LPF_LoadResults(void)
{
	Antenna_e iAntenna;

	for(iAntenna = ANTENNA_0; iAntenna< TOTAL_ANTENNAS; iAntenna++)
	{ 
		RficDriver_SetRxAllPGCsTuneWord((AntennaBitmaps_e)iAntenna ,RFIC_PATH_TYPE_I_AND_Q, ShramClbrDataBufferStoreChanData.lpfData.lpfWord[iAntenna]); 

#if LM_CALIBRATION_LOGS_ON
		ILOG2_DDD("LPF_LoadResults(): Ant %u lpfI=%u lpfQ=%u",iAntenna, ShramClbrDataBufferStoreChanData.lpfData.lpfWord[iAntenna][RFIC_PATH_TYPE_I], ShramClbrDataBufferStoreChanData.lpfData.lpfWord[iAntenna][RFIC_PATH_TYPE_Q]);
#endif

		LPF_SetStatus(iAntenna, ShramClbrDataBufferStoreChanData.lpfData.calStatus[iAntenna]);
	}
}

//************************************
// Method:     LPF_StoreResults
// Purpose:    Stores LPF calibration data to a buffer
// Parameter: None
// Returns:     None
//************************************
static void LPF_StoreResults(void)
{
	bool CalState = TRUE;
	CalStatus_e antStatus;
	Antenna_e ant;

    for(ant = ANTENNA_0; ant< TOTAL_ANTENNAS; ant++)
    { 
        antStatus = LPF_GetStatus(ant);
        ShramClbrDataBufferStoreChanData.lpfData.calStatus[ant] = antStatus;
        if ((antStatus == CAL_STATUS_FAIL) || (antStatus == CAL_STATUS_IN_PROCESS))
        {
            CalState = FALSE;
        }
        RficDriver_GetRxPGCTuneWord(ant, RFIC_PATH_TYPE_I_AND_Q, ShramClbrDataBufferStoreChanData.lpfData.lpfWord[ant]);       
    }
    ClbrHndlr_UpdateCalState(CLBR_PROC_TYPE_LPF, CalState);
}

RetVal_e LPF_Calibrate(IN ClbrType_e calibType)
{
	return lpfRun(&Calibration_params_otf.Lpfparams);
}

void LPF_SetStatus(IN uint32 antenna, IN CalStatus_e status)
{
	Calibration_params_otf.Lpfparams.results.status[antenna] = status;
}

CalStatus_e LPF_GetStatus(IN uint32 antenna)
{
	return Calibration_params_otf.Lpfparams.results.status[antenna];
}

uint32 LPF_GetLastRunTime(void)
{
	return Calibration_params_otf.Lpfparams.results.lastRunTime;
}

uint32* LPF_GetElement(void)
{
	return (uint32*)&Calibration_params_otf.Lpfparams;
}




