#include "LpfClbrUtils.h"
#include "LpfDefinitions.h"
#include "stringLibApi.h"
#include "MT_Math.h"
#include "ErrorHandler_Api.h"

#include "loggerAPI.h"

#define LOG_LOCAL_GID   GLOBAL_GID_CALIBRATIONS
#define LOG_LOCAL_FID 5

/******************************************************************************/
/***					Static Function Declaration							***/
/******************************************************************************/
static int64 measureBinEnergy(IN LpfElements_t* pElements, IN int8 bin, IN bool shouldGenTone, IN ToneGenShiftMode_e shiftMode);
static RetVal_e verifyPassBandEnergy(int64 energy[], IN LpfElements_t* pElements, IN upperBoundErrorMarginAdjustmentFuncPtr_t upperBoundErrFunc, IN lowerBoundErrorMarginAdjustmentFuncPtr_t lowerBoundErrFunc);
static void enforcePositiveDelta(IN LpfElements_t* pElements, IN setLpfWordFuncPtr_t setLpfWordFunc, IN int8 bin, IN ToneGenShiftMode_e shiftMode, IN int8 higherEnergyUnit, OUT int64 energy[], OUT uint8 lpfWord[]);

/******************************************************************************/
/***						Public Functions Definitions					***/
/******************************************************************************/
RetVal_e LpfUtils_findBestWord(	IN LpfElements_t* pElements, // General scratch-pad
								IN setLpfWordFuncPtr_t setLpfWordFunc, // Method used to set LPF words
								IN upperBoundErrorMarginAdjustmentFuncPtr_t upperBoundErrFunc, // Method used to adjust upper error margin in verifyPassBandEnergy
								IN lowerBoundErrorMarginAdjustmentFuncPtr_t lowerBoundErrFunc, // Method used to adjust lower error margin in verifyPassBandEnergy
								IN int8 bin, // The bin on which the minimum search is conducted
								IN ToneGenShiftMode_e shiftMode, // Tone generator shift mode for the supplied bin
								IN int64* energyOffset, // Should offsets be applied to measured values
								IN bool enforcePairing, // Must the calibrated value be lower than a designated adjacent bin
								IN bool enforcePositive, // Must the calibrated value be greater than the pass band reference
								IN int8 highCorrectionUnit, // Unit to increase measured energy
								OUT uint8* resLpfWords) // Results
{
	uint8 stepSize =  pElements->params.initialWord/2;
	uint8 NumOfSteps = 0;
	int8 signCorrection;
	int64 energy[RFIC_PATH_TYPE_MAX];
	int64 energyDelta[RFIC_PATH_TYPE_MAX];
	int64 bestDelta[RFIC_PATH_TYPE_MAX] = {MAX_INT64, MAX_INT64};
	uint8 bestWords[RFIC_PATH_TYPE_MAX];
	int64 bestEnergy[RFIC_PATH_TYPE_MAX];
	RficPathTypeIndex_e branch;
	RetVal_e retVal;
	int8 pairedBin = bin+ENFORCED_PAIRING_BIN_OFFSET;
	bool pairFlag = FALSE;
	bool positivePair = TRUE;

	resLpfWords[RFIC_PATH_TYPE_I] = pElements->params.initialWord;
	resLpfWords[RFIC_PATH_TYPE_Q] = pElements->params.initialWord;

	// Transmit the tone
	PhyCalDrv_ToggleTxToneExtended(bin, TONE_GEN_TOGGLE_ON, shiftMode);
	
	do 
	{
		setLpfWordFunc(pElements->dataSet.antBitmap ,RFIC_PATH_TYPE_I_AND_Q , resLpfWords);

		for (branch=RFIC_PATH_TYPE_I;branch<pElements->dataSet.numOfRfPathsToCalibrate;branch++)
		{
			PhyCalDrv_GoertzelInit(GOERTZEL_0, pElements->dataSet.antInfo.RxLoopbackAnt, branch, pElements->dataSet.isIqSwap, pElements->params.numOfCycles);
			energy[branch] = measureBinEnergy(pElements, bin, FALSE, shiftMode);// in linear scale, shift usage requires adjustments

			// Apply offset, if applicable
			if (energyOffset != NULL)
			{
				energy[branch] <<= TX_FIXED_POINT_BITS_SHIFT;
				energy[branch] /= energyOffset[branch];
			}
			 
			// Some implementations need a reference to avoid glitch
			if (enforcePairing)
			{
				// The calibration bin is too adjacent to the reference, we thus toggle it off while measuring the reference
				PhyCalDrv_ToggleTxToneExtended(bin, TONE_GEN_TOGGLE_OFF, shiftMode);

				if (RX_CB_PAIRING_CAL_BIN_ADJUST(energy[branch]) > measureBinEnergy(pElements, pairedBin, TRUE, shiftMode)) // Greater than reference, must correct back
				{
					positivePair = FALSE;
				}
				else
				{
					pairFlag = TRUE; // Mark that at least one config was OK
					positivePair = TRUE;
				}

				PhyCalDrv_ToggleTxToneExtended(bin, TONE_GEN_TOGGLE_ON, shiftMode);
			}

			if (positivePair == TRUE) // Either no pairing enforced or pairing OK
			{
				energyDelta[branch] = pElements->dataSet.desiredPowerLevel[branch]-energy[branch]; 
				signCorrection = (energyDelta[branch]>0)?highCorrectionUnit:highCorrectionUnit*(-1);

				// If this is the best delta so far - keep it
				if (Abs64(energyDelta[branch]) < Abs64(bestDelta[branch]))
				{
					bestDelta[branch] = energyDelta[branch];
					bestWords[branch] = resLpfWords[branch];
					bestEnergy[branch] = energy[branch];
				}
			}
			else // Pairing enforced and it is not OK, must lower energy
			{
				signCorrection = highCorrectionUnit*(-1);
			}

			// Set next word to test
			resLpfWords[branch] += signCorrection*stepSize; 
		}


		// Set next step size and update number
		stepSize/=2;
		NumOfSteps++;	

		// Due to SW ease reasons, if we are at lpf word =1 and we need a correction down to '0'  
		// add assertion since shuold never happen (extreme HW assembly)
		if (((resLpfWords[RFIC_PATH_TYPE_I]==1) && (energyDelta[RFIC_PATH_TYPE_I]<0)) || ((resLpfWords[RFIC_PATH_TYPE_Q]==1) && (energyDelta[RFIC_PATH_TYPE_Q]<0)))
		{
			DEBUG_ASSERT(FALSE);
		}
	}while(NumOfSteps < pElements->params.maxNumOfEdgeBandSteps);

	// Save the results to be set upon finish (it is otherwise run over during the calibration)
	memcpy(resLpfWords, bestWords, sizeof(resLpfWords));
	
	// Some implementations enforce that the result will be greater than the reference
	if (enforcePositive)
	{
		enforcePositiveDelta(pElements, setLpfWordFunc, bin, shiftMode, highCorrectionUnit, energy, resLpfWords);
	}

	// Set the best configuration (may not be the last configuration tested)
	setLpfWordFunc(pElements->dataSet.antBitmap ,RFIC_PATH_TYPE_I_AND_Q , resLpfWords);

	// check if we failed in the search 
	retVal = verifyPassBandEnergy(bestEnergy, pElements, upperBoundErrFunc, lowerBoundErrFunc);
	
	// If pairing was enforced, make sure that at least one pair was OK
	if(enforcePairing && (pairFlag == FALSE)) 
	{
		retVal = RET_VAL_FAIL;
	}

	// Cancel the tone transmission
	PhyCalDrv_ToggleTxToneExtended(bin, TONE_GEN_TOGGLE_OFF, shiftMode);

	return retVal;
}

void LpfUtils_measureRefAvgEnergy(LpfElements_t* pElements, int8 bin, int64* pRefAvgEnergy)
{
	int32 i;
	int64 binEnergy;
	int64 energySum;
	RficPathTypeIndex_e branch;

	for (branch=RFIC_PATH_TYPE_I;branch<pElements->dataSet.numOfRfPathsToCalibrate;branch++)
	{
		energySum = 0;

		PhyCalDrv_GoertzelInit(GOERTZEL_0, pElements->dataSet.antInfo.RxLoopbackAnt, branch, pElements->dataSet.isIqSwap,pElements->params.numOfCycles); 

		for (i=0;i<PASS_BAND_REF_BIN_NUM;i++)
		{
			binEnergy = measureBinEnergy(pElements, bin, TRUE, TONE_GEN_SHIFT_NONE);
			energySum += binEnergy; // in dB
			bin++;
		}

		pRefAvgEnergy[branch] = (energySum/PASS_BAND_REF_BIN_NUM);
	}

	return;
}

void LpfUtils_measureBinEnergyFull(LpfElements_t* pElements, int8 bin, ToneGenShiftMode_e shiftMode, int64* measuredEnergy)
{
	RficPathTypeIndex_e branch;

	for (branch=RFIC_PATH_TYPE_I;branch<pElements->dataSet.numOfRfPathsToCalibrate;branch++)
	{
		PhyCalDrv_GoertzelInit(GOERTZEL_0, pElements->dataSet.antInfo.RxLoopbackAnt, branch, pElements->dataSet.isIqSwap,pElements->params.numOfCycles);
		measuredEnergy[branch] = measureBinEnergy(pElements, bin, TRUE, shiftMode);
	}
}

void LpfUtils_getRxStopBinRefEnergy(LpfElements_t* pElements, int64* res)
{
	RficPathTypeIndex_e branch;

	for (branch=RFIC_PATH_TYPE_I;branch<pElements->dataSet.numOfRfPathsToCalibrate;branch++)
	{
		// set the upper bound stop band
		if (pElements->dataSet.isCB == FALSE) // we are at NON CB mode
		{
			res[branch] = RX_NCB_UPPER_BOUND_STOP_BAND_ADJUST(pElements->dataSet.rxRefAvgEnergy[branch]);
			res[branch] = RX_NCB_UPPER_BOUND_STOP_BAND_ADJUST_SINC_ATTEN(res[branch]);// SINC attenuation
		} 
		else
		{
			res[branch] = RX_CB_UPPER_BOUND_STOP_BAND_ADJUST(pElements->dataSet.rxRefAvgEnergy[branch]);
			res[branch] = RX_CB_UPPER_BOUND_STOP_BAND_ADJUST_SINC_ATTEN(res[branch]); // SINC attenuation
		}

		res[branch] = UPPER_BOUND_STOP_BAND_MARGIN_ADJUST(res[branch]);
	}
}

void LpfUtils_getTxStopBinRefEnergy(LpfElements_t* pElements, int64* res)
{
	RficPathTypeIndex_e branch;

	for (branch=RFIC_PATH_TYPE_I;branch<pElements->dataSet.numOfRfPathsToCalibrate;branch++)
	{
		// eliminate the Rx loop-back contribution
		res[branch] = ((pElements->dataSet.txRefAvgEnergy[branch])<<TX_FIXED_POINT_BITS_SHIFT) / pElements->dataSet.rxRefAvgEnergy[branch];

		// set the upper bound stop band
		if (pElements->dataSet.isCB == FALSE) // we are at NON CB mode
		{
			res[branch] = TX_NCB_UPPER_BOUND_STOP_BAND_ADJUST(res[branch]);
		}
		else
		{
			res[branch] = TX_CB_UPPER_BOUND_STOP_BAND_ADJUST(res[branch]);
		}

		res[branch] = UPPER_BOUND_STOP_BAND_MARGIN_ADJUST(res[branch]);
	}
}

RetVal_e LpfUtils_verifyBinEnergy(LpfElements_t* pElements, int8 bin, ToneGenShiftMode_e shiftMode, uint8 fixedPointBitShift, int64* energyOffset, int64* refEnergy)
{
	int64 psEnergyLin[RFIC_PATH_TYPE_MAX];
	RficPathTypeIndex_e branch;

	PhyCalDrv_SetDifiBypass(PHY_CAL_DRV_DIFI_BP_ON);

	LpfUtils_measureBinEnergyFull(pElements, bin, shiftMode, psEnergyLin);

	PhyCalDrv_SetDifiBypass(PHY_CAL_DRV_DIFI_BP_OFF);

	for (branch=RFIC_PATH_TYPE_I;branch<pElements->dataSet.numOfRfPathsToCalibrate;branch++)
	{
		/* Adjust the energy we just measured */
		if (energyOffset != NULL)
		{
			psEnergyLin[branch] = ((psEnergyLin[branch])<<fixedPointBitShift) / energyOffset[branch];
		}

		/* Verify the energy of at stop bin */
		if (psEnergyLin[branch] > refEnergy[branch])
		{
			return RET_VAL_FAIL;
		}
	}
	
	return RET_VAL_SUCCESS;
}

/******************************************************************************/
/***						Staticic Functions Definitions					***/
/******************************************************************************/
//************************************
// Method:    measureBinEnergy
// Purpose:   this function transmits one tone and measures the Power using the Goertzel
//            block ,and returns the result [using 10log10(avik) who returns a fix point (4bit)]
// Parameter: IN LpfElements_t * pElements - local scratchpad
// Parameter: IN int8 bin - the transmitted bin we want to measure using the goertzel algorithm  (Uint)
// Parameter: IN bool shouldGenTone - Should the method invoke tone
// Parameter: IN ToneGenShiftMode_e shiftMode - Tone generator mode
// Returns:   int64 - Measured linear energy
//************************************
static int64 measureBinEnergy(IN LpfElements_t* pElements, IN int8 bin, IN bool shouldGenTone, IN ToneGenShiftMode_e shiftMode)
{
	int64 i64ResultI, i64ResultQ, res;
	GoertzelResults_t goertzelRes;
	int16 goertzelBin = bin; 

	if(shouldGenTone)444
	{
		PhyCalDrv_ToggleTxToneExtended(bin,TONE_GEN_TOGGLE_ON,shiftMode);
	}

	if (shiftMode != TONE_GEN_SHIFT_NONE)
	{
		goertzelBin+=(pElements->dataSet.isCB?64:32);
	}

	PhyCalDrv_GoertzelMeasure(1, &goertzelRes, goertzelBin, DELAY_AFTER_PHY_TX_CHANGE);

	if(shouldGenTone)
	{
		PhyCalDrv_ToggleTxToneExtended(bin,TONE_GEN_TOGGLE_OFF,shiftMode);
	}

	i64ResultI = goertzelRes.Result_I;
	i64ResultQ = goertzelRes.Result_Q;

	res = (i64ResultI*i64ResultI + i64ResultQ*i64ResultQ);

	if (shiftMode != TONE_GEN_SHIFT_NONE)
	{
		res = SHIFTER_ENERGY_LEVEL_ADJUST(res); // due to the reduction using the shift in the ToneGen
	}
	
	return res;
}

//************************************
// Method:    verifyPassBandEnergy
// Purpose:   Verify that the supplied energy resides within the defined margin
// Parameter: int64 energy[] - Energy to test
// Parameter: IN LpfElements_t * pElements - local scratchpad
// Returns:   RetVal_e
//************************************
static RetVal_e verifyPassBandEnergy(int64 energy[], IN LpfElements_t* pElements,
									IN upperBoundErrorMarginAdjustmentFuncPtr_t upperBoundErrFunc,
									IN lowerBoundErrorMarginAdjustmentFuncPtr_t lowerBoundErrFunc)
{
	int64 lowerBoundErrorMargin[RFIC_PATH_TYPE_MAX];
	int64 upperBoundErrorMargin[RFIC_PATH_TYPE_MAX];
	RficPathTypeIndex_e branch;
	RetVal_e retVal = RET_VAL_SUCCESS;

	for (branch=RFIC_PATH_TYPE_I;branch<pElements->dataSet.numOfRfPathsToCalibrate;branch++)
	{
		upperBoundErrFunc(pElements->dataSet.desiredPowerLevel[branch], &(upperBoundErrorMargin[branch]));
		lowerBoundErrFunc(pElements->dataSet.desiredPowerLevel[branch], &(lowerBoundErrorMargin[branch]));

		if (energy[branch] <= lowerBoundErrorMargin[branch])
		{
			pElements->results.passBandError[pElements->dataSet.antInfo.RfCalibAnt] = -1;
			retVal = RET_VAL_FAIL;
			break;
		}
		else if(energy[branch] >= upperBoundErrorMargin[branch])
		{
			pElements->results.passBandError[pElements->dataSet.antInfo.RfCalibAnt] = 1;
			retVal = RET_VAL_FAIL;
			break;
		}
	}

	return retVal;
}

//************************************
// Method:    enforcePositiveDelta
// Purpose:   Enforce the choosing of the LPF word with the least POSITIVE delta
// Parameter: IN LpfElements_t * pElements - local scratchpad
// Parameter: IN setLpfWordFuncPtr_t setLpfWordFunc - LPF word setter function pointer
// Parameter: IN int8 bin - Bin used
// Parameter: IN ToneGenShiftMode_e shiftMode - Tone generator mode
// Parameter: IN int8 higherEnergyUnit - Unit of a higher energy change in the Goertzel block measurements
// Parameter: OUT int64 energy[] - The new energy measured over the bin (may not change)
// Parameter: OUT uint8 lpfWord[] - The new LPF word to use (may not change)
// Returns:   void
//************************************
static void enforcePositiveDelta(	IN LpfElements_t* pElements, 
									IN setLpfWordFuncPtr_t setLpfWordFunc, 
									IN int8 bin, 
									IN ToneGenShiftMode_e shiftMode, 
									IN int8 higherEnergyUnit, 
									OUT int64 energy[], 
									OUT uint8 lpfWord[])
{
	bool shouldCorrectWord = FALSE;
	RficPathTypeIndex_e branch;

	for (branch=RFIC_PATH_TYPE_I;branch<pElements->dataSet.numOfRfPathsToCalibrate;branch++)
	{
		if (energy[branch] < pElements->dataSet.desiredPowerLevel[branch])
		{
			if(lpfWord[branch] + higherEnergyUnit < MAX_LPF_WORD) // Due to the unsigned nature of the LPF word, this is good for both wraparound on the minimum and max
			{
				shouldCorrectWord = TRUE;
				lpfWord[branch] += higherEnergyUnit; // we need to correct the word toward down
			}
		}
	}

	// We need to always remain at least in the desired energy
	if(shouldCorrectWord == TRUE)
	{
		setLpfWordFunc((AntennaBitmaps_e)pElements->dataSet.antInfo.RfCalibAnt ,RFIC_PATH_TYPE_I_AND_Q , lpfWord);
		for (branch=RFIC_PATH_TYPE_I;branch<pElements->dataSet.numOfRfPathsToCalibrate;branch++)
		{
			PhyCalDrv_GoertzelInit(GOERTZEL_0, pElements->dataSet.antInfo.RxLoopbackAnt, branch, pElements->dataSet.isIqSwap,pElements->params.numOfCycles); 
			energy[branch] = measureBinEnergy(pElements, bin, FALSE, shiftMode); // in dB
		}
	}
}

