#include "LpfClbrHndlr.h"
#include "LpfClbrUtils.h"
#include "PhyDriver_API.h"
#include "PhyCalDriver_API.h"

#include "loggerAPI.h"

#define LOG_LOCAL_GID   GLOBAL_GID_CALIBRATIONS
#define LOG_LOCAL_FID 8

/******************************************************************************/
/***					Static Function Declaration							***/
/******************************************************************************/
static RetVal_e performCbRxCalibration(IN LpfElements_t* pElements);
static RetVal_e performNcbRxCalibration(IN LpfElements_t* pElements);
static RetVal_e performTxCalibration(IN LpfElements_t* pElements);

static inline void LpfDep_RxCbUpperBoundErrorMarginAdjustmentFunc(int64 inOrigEnergy, int64 *outMarginEnergy);
static inline void LpfDep_RxCbLowerBoundErrorMarginAdjustmentFunc(int64 inOrigEnergy, int64 *outMarginEnergy);
static inline void LpfDep_RxNcbUpperBoundErrorMarginAdjustmentFunc(int64 inOrigEnergy, int64 *outMarginEnergy);
static inline void LpfDep_RxNcbLowerBoundErrorMarginAdjustmentFunc(int64 inOrigEnergy, int64 *outMarginEnergy);
static inline void LpfDep_TxCbUpperBoundErrorMarginAdjustmentFunc(int64 inOrigEnergy, int64 *outMarginEnergy);
static inline void LpfDep_TxCbLowerBoundErrorMarginAdjustmentFunc(int64 inOrigEnergy, int64 *outMarginEnergy);
static inline void LpfDep_TxNcbUpperBoundErrorMarginAdjustmentFunc(int64 inOrigEnergy, int64 *outMarginEnergy);
static inline void LpfDep_TxNcbLowerBoundErrorMarginAdjustmentFunc(int64 inOrigEnergy, int64 *outMarginEnergy);

/******************************************************************************/
/***						Public Functions Definitions					***/
/******************************************************************************/
//************************************
// Method:    LpfDep_PerformAntennaCalibration
// Purpose:   Performs the LPF calibration for the antenna specified in pElements
//            Platform Dependant!!
// Parameter: LpfElements_t* pElements - A structure containing all parameters,
//                                       results and scratch-pad needed for the LPF calibration
// Returns:   RET_VAL_SUCCESS or RET_VAL_FAIL
//************************************
RetVal_e LpfDep_PerformAntennaCalibration(IN LpfElements_t* pElements)
{
	int8 rxStopBin = pElements->dataSet.isCB ? pElements->params.cbStopBandTone:pElements->params.ncbStopBandTone;
	//int8 txStopBin = pElements->dataSet.isCB ? TX_CB_STOP_BIN : TX_NCB_STOP_BIN;
	int64 stopBinEnergy[RFIC_PATH_TYPE_MAX];
	uint8 lpf[RFIC_PATH_TYPE_MAX];
	RetVal_e retVal;

	lpf[RFIC_PATH_TYPE_I] = pElements->params.initialWord;
	lpf[RFIC_PATH_TYPE_Q] = pElements->params.initialWord;

	do 
	{
		/////////////////////
		// Rx Part         //
		/////////////////////

		// Configure RF
		RficDriver_SetLoopback(pElements->dataSet.antInfo.RfCalibAnt,pElements->dataSet.antInfo.RxLoopbackAnt, RFIC_LOOPBACK_TYPE_RX_LPF); 

		// Measure the average reference energy 
		RficDriver_SetRxAllPGCsTuneWord((AntennaBitmaps_e)pElements->dataSet.antBitmap ,RFIC_PATH_TYPE_I_AND_Q , lpf);
		LpfUtils_measureRefAvgEnergy(pElements, pElements->params.passBandTone, pElements->dataSet.rxRefAvgEnergy);

		// Perform needed Rx LPF calibration (CB or NCB)
		retVal = (pElements->dataSet.isCB == TRUE)? performCbRxCalibration(pElements) : performNcbRxCalibration(pElements);

		// If failed, no value in continuing with Tx LPF calibration - abort job
		if (retVal != RET_VAL_SUCCESS)
		{
			break;
		} 

		// Verify stop bin
		LpfUtils_getRxStopBinRefEnergy(pElements, stopBinEnergy);
		if(LpfUtils_verifyBinEnergy(pElements, rxStopBin, TONE_GEN_SHIFT_UP, RX_FIXED_POINT_BITS_SHIFT, NULL, stopBinEnergy) == RET_VAL_FAIL)
		{
			pElements->results.stopBandError[pElements->dataSet.antInfo.RfCalibAnt] = 1;
			retVal = RET_VAL_FAIL;
			break;
		}

		/////////////////////
		// Tx Part         //
		/////////////////////

		if (performTxCalibration(pElements) != RET_VAL_SUCCESS)
		{
			retVal = RET_VAL_FAIL;
			break;
		} 

// TODO (Ariel Groentma and Amir.F) - find the right paramenters for Tx LPF verify and start using this part
// 		// Verify stop bin
// 		LpfUtils_getTxStopBinRefEnergy(pElements, stopBinEnergy);
// 		if(LpfUtils_verifyBinEnergy(pElements, txStopBin, TONE_GEN_SHIFT_UP, TX_FIXED_POINT_BITS_SHIFT, pElements->dataSet.txStopBinRxEnergy, stopBinEnergy) == RET_VAL_FAIL)
// 		{
// 			pElements->results.stopBandError[pElements->dataSet.antInfo.RfCalibAnt] = 1;
// 			retVal = RET_VAL_FAIL;
// 			break;
// 		}

	} while (FALSE);

	// If any failure was encountered, reset the LPF words to default value
	if (retVal != RET_VAL_SUCCESS)
	{
		RficDriver_SetRxAllPGCsTuneWord((AntennaBitmaps_e)pElements->dataSet.antInfo.RfCalibAnt ,RFIC_PATH_TYPE_I_AND_Q , lpf);
	}

	return retVal;
}

//************************************
// Method:    LpfDep_SetAntCalibResults
// Purpose:   Set the calibration results. Cannot be done during the calibration since the registers are used repeatedly
// Parameter: IN LpfElements_t * pElements - A data structure containg the calibration results
// Returns:   void
//************************************
void LpfDep_SetAntCalibResults(IN LpfElements_t* pElements)
{
	Antenna_e ant;
	uint8 antMask, numOfAnts;
	HDK_GetMaxActiveAntennasNumAndMask(&numOfAnts, &antMask);

	for(ant=ANTENNA_0;ant<=TOTAL_ANTENNAS;ant++)
	{
		if(ANTENNA_MASK_IS_BIT_ON(antMask, ant))
		{
			RficDriver_SetRxAllPGCsTuneWord((AntennaBitmaps_e)ant ,RFIC_PATH_TYPE_I_AND_Q , pElements->results.antRxLpfWordsMtx[ant]);

			/* We want to set specific values to PGC3, only in nCB mode */
			if (pElements->dataSet.isCB == FALSE)
			{
				RficDriver_SetRxPGC3TuneWord((AntennaBitmaps_e)ant ,RFIC_PATH_TYPE_I_AND_Q , pElements->results.antRxLpfPartialWordsMtx[ant]);
			}

			RficDriver_SetTxPGCTuneWord((AntennaBitmaps_e)ant ,RFIC_PATH_TYPE_I_AND_Q , pElements->results.antTxLpfWordsMtx[ant]);
		}
	}
}

/******************************************************************************/
/***						Staticic Functions Definitions					***/
/******************************************************************************/
//************************************
// Method:    performCbRxCalibration
// Purpose:   Rx calibration in CB mode. Conducts min search in pairing mode and verifies also an adjacent bin.
// Parameter: IN LpfElements_t * pElements
// Returns:   RetVal_e
//************************************
static RetVal_e performCbRxCalibration(IN LpfElements_t* pElements)
{
	int64 adjacentBinEnergy[RFIC_PATH_TYPE_MAX];
	RficPathTypeIndex_e branch;

	// Set the target desired level
	for (branch=RFIC_PATH_TYPE_I;branch<pElements->dataSet.numOfRfPathsToCalibrate;branch++)
	{
		pElements->dataSet.desiredPowerLevel[branch] = RX_CB_DESIRED_POWER_LEVEL_ADJUST(pElements->dataSet.rxRefAvgEnergy[branch]);
		pElements->dataSet.desiredPowerLevel[branch] = RX_CB_DESIRED_POWER_LEVEL_ADJUST_SINC_ATTEN(pElements->dataSet.desiredPowerLevel[branch]);	// due to attenuation at the edge (sinc) we need to decrease 0.81dB or multiply in 0.83 
	}

	// Perform MinSearch for the best trim
	if (LpfUtils_findBestWord(pElements, RficDriver_SetRxAllPGCsTuneWord, LpfDep_RxCbUpperBoundErrorMarginAdjustmentFunc, LpfDep_RxCbLowerBoundErrorMarginAdjustmentFunc, RX_CB_EDGE_BIN, TONE_GEN_SHIFT_NONE, NULL, TRUE, FALSE, RX_CB_HIGHER_ENERGY_UNIT, pElements->results.antRxLpfWordsMtx[pElements->dataSet.antInfo.RfCalibAnt]) == RET_VAL_FAIL) 
	{
		return RET_VAL_FAIL;
	}

	// Verify adjacent bin
	for (branch=RFIC_PATH_TYPE_I;branch<pElements->dataSet.numOfRfPathsToCalibrate;branch++)
	{
		adjacentBinEnergy[branch] = RX_CB_ADJACENT_BIN_ADJUST(pElements->dataSet.rxRefAvgEnergy[branch]);
	}
	if (LpfUtils_verifyBinEnergy(pElements, RX_CB_ADJACENT_BIN, TONE_GEN_SHIFT_UP, RX_FIXED_POINT_BITS_SHIFT, NULL, adjacentBinEnergy))
	{
		return RET_VAL_FAIL;
	}

	return RET_VAL_SUCCESS;
}

//************************************
// Method:    performNcbRxCalibration
// Purpose:   Rx calibration in NCB mode. Conducts two "clean" min searches. the first configures both PGC12 and PGC3 and the second 
//            fine-tunes PGC3 over another bin.
// Parameter: IN LpfElements_t * pElements
// Returns:   RetVal_e
//************************************
static RetVal_e performNcbRxCalibration(IN LpfElements_t* pElements)
{
	RficPathTypeIndex_e branch;

	// Set the target desired level
	for (branch=RFIC_PATH_TYPE_I;branch<pElements->dataSet.numOfRfPathsToCalibrate;branch++)
	{
		pElements->dataSet.desiredPowerLevel[branch] = RX_NCB_OUT_OF_BAND_BIN_LEVEL_ADJUST(pElements->dataSet.rxRefAvgEnergy[branch]);
		pElements->dataSet.desiredPowerLevel[branch] = RX_NCB_OUT_OF_BAND_BIN_LEVEL_ADJUST_ADC_DAC(pElements->dataSet.desiredPowerLevel[branch]);	// due to attenuation at the edge (sinc) we need to decrease 0.81dB or multiply in 0.83 
	}

	// Perform MinSearch for the best trim - searches for best PGC12 values
	if (LpfUtils_findBestWord(pElements, RficDriver_SetRxAllPGCsTuneWord, LpfDep_RxNcbUpperBoundErrorMarginAdjustmentFunc, LpfDep_RxNcbLowerBoundErrorMarginAdjustmentFunc, RX_NCB_EDGE_BIN, TONE_GEN_SHIFT_UP, NULL, FALSE, FALSE, RX_NCB_HIGHER_ENERGY_UNIT, pElements->results.antRxLpfWordsMtx[pElements->dataSet.antInfo.RfCalibAnt]) == RET_VAL_FAIL)
	{
		return RET_VAL_FAIL;
	}

	// Set the target desired level
	for (branch=RFIC_PATH_TYPE_I;branch<pElements->dataSet.numOfRfPathsToCalibrate;branch++)
	{
		pElements->dataSet.desiredPowerLevel[branch] = RX_NCB_IN_BAND_BIN_LEVEL_ADJUST_ADC_DAC(pElements->dataSet.rxRefAvgEnergy[branch]);
	}

	// Perform MinSearch to reduce the ripple - searches for the best PGC3 value
	if (LpfUtils_findBestWord(pElements, RficDriver_SetRxPGC3TuneWord, LpfDep_RxNcbUpperBoundErrorMarginAdjustmentFunc, LpfDep_RxNcbLowerBoundErrorMarginAdjustmentFunc, RX_NCB_RIPPLE_BIN, TONE_GEN_SHIFT_NONE, NULL, FALSE, FALSE, RX_NCB_HIGHER_ENERGY_UNIT, pElements->results.antRxLpfPartialWordsMtx[pElements->dataSet.antInfo.RfCalibAnt]) == RET_VAL_FAIL)
	{
		return RET_VAL_FAIL;
	}
	
	return RET_VAL_SUCCESS;
}

//************************************
// Method:    performTxCalibration
// Purpose:   Tx calibration. Conducts a min search with cancellation of measurements done in Rx loop (both the reference pass band energy and each measurement of the edge bin)
// Parameter: IN LpfElements_t * pElements
// Returns:   RetVal_e
//************************************
static RetVal_e performTxCalibration(IN LpfElements_t* pElements)
{
	int64 txCalBinEnergy[RFIC_PATH_TYPE_MAX];
	int8 txBin = (pElements->dataSet.isCB?TX_CB_CAL_BIN:TX_NCB_CAL_BIN);
	RficPathTypeIndex_e branch;
	upperBoundErrorMarginAdjustmentFuncPtr_t upperBoundErrFunc = (pElements->dataSet.isCB?LpfDep_TxCbUpperBoundErrorMarginAdjustmentFunc:LpfDep_TxNcbUpperBoundErrorMarginAdjustmentFunc);
	lowerBoundErrorMarginAdjustmentFuncPtr_t lowerBoundErrFunc = (pElements->dataSet.isCB?LpfDep_TxCbLowerBoundErrorMarginAdjustmentFunc:LpfDep_TxNcbLowerBoundErrorMarginAdjustmentFunc);

	// Measure reference bins (still with Rx loop-back)
	LpfUtils_measureBinEnergyFull(pElements, txBin, TONE_GEN_SHIFT_NONE, txCalBinEnergy);

	// Measure stop bin energy in Rx loop-back for later reference
	PhyCalDrv_SetDifiBypass(PHY_CAL_DRV_DIFI_BP_ON);
	LpfUtils_measureBinEnergyFull(pElements, pElements->params.cbStopBandTone, TONE_GEN_SHIFT_UP, pElements->dataSet.txStopBinRxEnergy);
	PhyCalDrv_SetDifiBypass(PHY_CAL_DRV_DIFI_BP_OFF);

	// Prepare for Tx LPF calibration
	RficDriver_SetLoopback(pElements->dataSet.antInfo.RfCalibAnt,pElements->dataSet.antInfo.RxLoopbackAnt, RFIC_LOOPBACK_TYPE_TX_LPF); 
	LpfUtils_measureRefAvgEnergy(pElements, pElements->params.passBandTone, pElements->dataSet.txRefAvgEnergy);

	// Set the target desired level
	for (branch=RFIC_PATH_TYPE_I;branch<pElements->dataSet.numOfRfPathsToCalibrate;branch++)
	{
		pElements->dataSet.desiredPowerLevel[branch] = ((pElements->dataSet.txRefAvgEnergy[branch])<<TX_FIXED_POINT_BITS_SHIFT)/pElements->dataSet.rxRefAvgEnergy[branch];
		pElements->dataSet.desiredPowerLevel[branch] = TX_DESIRED_POWER_LEVEL_ADJUST(pElements->dataSet.desiredPowerLevel[branch]);
	}

	// Perform MinSearch for the best trim 
	if (LpfUtils_findBestWord(pElements, RficDriver_SetTxPGCTuneWord, upperBoundErrFunc, lowerBoundErrFunc, txBin, TONE_GEN_SHIFT_NONE, txCalBinEnergy, FALSE, FALSE, TX_HIGHER_ENERGY_UNIT, pElements->results.antTxLpfWordsMtx[pElements->dataSet.antInfo.RfCalibAnt]) == RET_VAL_FAIL) 
	{
		return RET_VAL_FAIL;
	}

	return RET_VAL_SUCCESS;
}

//************************************
// Method:    LpfDep_Rx(Tx)Cb(Ncb)Upper(Lower)BoundErrorMarginAdjustmentFunc
// Purpose:   Adjusting the original energy up or down, to use during Rx or Tx LPF calibration,
//            according to the name of the function
// Parameter: int64 inOrigEnergy - The original energy we want to adjust
// Parameter: int64 *outMarginEnergy - The energy after the adjustment
// Returns:   None
//************************************
static inline void LpfDep_RxCbUpperBoundErrorMarginAdjustmentFunc(int64 inOrigEnergy, int64 *outMarginEnergy)
{
    *outMarginEnergy = RX_CB_UPPER_BOUND_ERROR_MARGIN_ADJUST(inOrigEnergy);
}

static inline void LpfDep_RxCbLowerBoundErrorMarginAdjustmentFunc(int64 inOrigEnergy, int64 *outMarginEnergy)
{
    *outMarginEnergy = RX_CB_LOWER_BOUND_ERROR_MARGIN_ADJUST(inOrigEnergy);
}

static inline void LpfDep_RxNcbUpperBoundErrorMarginAdjustmentFunc(int64 inOrigEnergy, int64 *outMarginEnergy)
{
    *outMarginEnergy = RX_NCB_UPPER_BOUND_ERROR_MARGIN_ADJUST(inOrigEnergy);
}

static inline void LpfDep_RxNcbLowerBoundErrorMarginAdjustmentFunc(int64 inOrigEnergy, int64 *outMarginEnergy)
{
    *outMarginEnergy = RX_NCB_LOWER_BOUND_ERROR_MARGIN_ADJUST(inOrigEnergy);
}

static inline void LpfDep_TxCbUpperBoundErrorMarginAdjustmentFunc(int64 inOrigEnergy, int64 *outMarginEnergy)
{
    *outMarginEnergy = TX_CB_UPPER_BOUND_ERROR_MARGIN_ADJUST(inOrigEnergy);
}

static inline void LpfDep_TxCbLowerBoundErrorMarginAdjustmentFunc(int64 inOrigEnergy, int64 *outMarginEnergy)
{
    *outMarginEnergy = TX_CB_LOWER_BOUND_ERROR_MARGIN_ADJUST(inOrigEnergy);
}

static inline void LpfDep_TxNcbUpperBoundErrorMarginAdjustmentFunc(int64 inOrigEnergy, int64 *outMarginEnergy)
{
    *outMarginEnergy = TX_NCB_UPPER_BOUND_ERROR_MARGIN_ADJUST(inOrigEnergy);
}

static inline void LpfDep_TxNcbLowerBoundErrorMarginAdjustmentFunc(int64 inOrigEnergy, int64 *outMarginEnergy)
{
    *outMarginEnergy = TX_NCB_LOWER_BOUND_ERROR_MARGIN_ADJUST(inOrigEnergy);
}
